#pragma once

#include <wangle/bootstrap/ServerBootstrap.h>
#include <wangle/channel/Handler.h>
#include <wangle/channel/Pipeline.h>

#include "controlHandler.hpp"
#include "service.hpp"

namespace rtransfer {
namespace server {

/**
 * @c RoutingDataHandler is a Wangle handler that determines whether a
 * connection is meant to be handled by a control or by a data pipeline.
 * If the ctrlId sent on the connection is 0000000000000000 (ACII '0'),
 * a control pipeline is created to handle it; otherwise it is a data
 * connection and a corresponding control connection is found for it.
 * When a control connection is created, a new ctrlId is generated for it
 * and sent back to the client.
 */
class RoutingDataHandler : public wangle::HandlerAdapter<folly::IOBufQueue &,
                               std::unique_ptr<folly::IOBuf>> {
public:
    RoutingDataHandler(ServerSideLinkFactory &factory,
        ServiceFactory &serviceFactory, SecretManager &secretManager)
        : linkFactory_{factory}
        , serviceFactory_{serviceFactory}
        , secretManager_{secretManager}
    {
    }

    void read(Context *ctx, folly::IOBufQueue &bufQueue) override;

    folly::Future<folly::Unit> write(
        Context *ctx, std::unique_ptr<folly::IOBuf> buf) override
    {
        return ctx->fireWrite(std::move(buf));
    }

private:
    void closeNow(Context *ctx, folly::StringPiece description);

    void newControlPipeline(Context *ctx, wangle::PipelineBase *pipeline,
        folly::StringPiece ctrlId, folly::fbstring providerId,
        folly::StringPiece mySecret);

    void newDataPipeline(Context *ctx, wangle::PipelineBase *pipeline,
        const folly::fbstring &ctrlId, folly::StringPiece mySecret);

    bool isDone_ = false;
    ServerSideLinkFactory &linkFactory_;
    ServiceFactory &serviceFactory_;
    SecretManager &secretManager_;
};

/**
 * A Wangle PipelineFactory that creates pipelines with @c RoutingDataHandler .
 */
class PipelineFactory
    : public wangle::PipelineFactory<wangle::DefaultPipeline> {
public:
    using MsgPtr = std::unique_ptr<proto::LinkMessage>;

    PipelineFactory(ServerSideLinkFactory factory,
        ServiceFactory &serviceFactory, SecretManager &secretManager)
        : linkFactory_{factory}
        , serviceFactory_{serviceFactory}
        , secretManager_{secretManager}
    {
    }

    wangle::DefaultPipeline::Ptr newPipeline(
        std::shared_ptr<folly::AsyncTransportWrapper> sock) override;

private:
    ServerSideLinkFactory linkFactory_;
    ServiceFactory &serviceFactory_;
    SecretManager &secretManager_;
};

std::unique_ptr<wangle::ServerBootstrap<>> makeServer(std::uint16_t port,
    ServerSideLinkFactory linkFactory, ServiceFactory &serviceFactory,
    SecretManager &secretManager,
    wangle::ServerSocketConfig &serverSocketConfig);

}  // namespace server
}  // namespace rtransfer
