#include "shaperMap.hpp"

#include <gflags/gflags.h>

#include <functional>

namespace rtransfer {

ShaperMap::ShaperMap(Reader &reader)
    : reader_{reader}
{
}

folly::Future<std::size_t> ShaperMap::fetch(MsgPtr msg)
{
    const auto &fetch = msg->fetch();
    const auto reqId = fetch.req_id();

    Shaper::ID key{fetch.src(), fetch.dest(), fetch.priority()};

    auto res = shapers_.emplace(
        std::move(key), std::make_shared<Shaper>(key, reader_, shaperTimer_));

    const bool newShaperAdded = res.second;
    auto shaper = res.first->second;

    reqToShaper_.emplace(reqId, shaper);

    VLOG(2) << "Stored request " << reqId << " in shaper map";

    if (!newShaperAdded)
        VLOG(2) << "Reusing existing shaper for reqId: " << reqId;

    return shaper->read(std::move(msg));
}

folly::Future<folly::Unit> ShaperMap::cancel(MsgPtr msg)
{
    VLOG(2) << "Cancelling shaperMap request " << msg->req_id();

    const auto &cancel = msg->cancel();
    auto it = reqToShaper_.find(cancel.req_id());
    if (it == reqToShaper_.cend())
        return folly::makeFuture();

    if (it == reqToShaper_.cend()) {
        reqToShaper_.erase(it);
    }

    auto shaperPtr = it->second;

    assert(shaperPtr.get() != nullptr);

    return shaperPtr->cancel(std::move(msg)).via(folly::getCPUExecutor().get());
}

void ShaperMap::ack(MsgPtr msg)
{
    VLOG(2) << "Ack for request " << msg->req_id() << " received";

    for (const auto &ack : msg->acks().acks()) {
        folly::fbvector<std::uint64_t> offsets;
        offsets.insert(
            offsets.end(), ack.offsets().begin(), ack.offsets().end());

        auto it = reqToShaper_.find(ack.req_id());  // NOLINT
        if (it == reqToShaper_.cend()) {
            LOG(WARNING) << "Cannot find shaper for an ack reqId: "
                         << ack.req_id();
            return;
        }
        it->second->ack(ack.req_id(), std::move(offsets))
            .via(folly::getCPUExecutor().get())
            .thenValue([this, reqId = ack.req_id()](bool &&ackedAll) {
                if (ackedAll)
                    reqToShaper_.erase(reqId);
            });
    }
}

}  // namespace rtransfer
