#pragma once

#include <glog/logging.h>
#include <wangle/channel/Handler.h>

#include <array>

#include "periodicHandler.hpp"

constexpr const unsigned char pingMsg_ = 0b10101010;

/**
 * A handler that sends a ping message on a pipeline every 10 seconds.
 * @see pingMsg_
 */
class PingSender : public wangle::HandlerAdapter<folly::IOBufQueue &,
                       std::unique_ptr<folly::IOBuf>> {
public:
    void attachPipeline(Context *ctx) override
    {
        if (ctx->getTransport())
            startHandler(ctx);
    }

    void transportActive(Context *ctx) override
    {
        startHandler(ctx);
        ctx->fireTransportActive();
    }

    void transportInactive(Context *ctx) override
    {
        periodicHandler_.reset();
        ctx->fireTransportInactive();
    }

private:
    void startHandler(Context *ctx)
    {
        periodicHandler_ =
            std::make_unique<PeriodicHandler>(std::chrono::seconds{20}, [ctx] {
                VLOG(4) << "Sending ping over data connection";
                ctx->getTransport()->getEventBase()->runInEventBaseThread(
                    [ctx] {
                        ctx->fireWrite(folly::IOBuf::wrapBuffer(&pingMsg_, 1));
                    });
            });
    }

    std::unique_ptr<PeriodicHandler> periodicHandler_;
};

/**
 * A handler that receives a ping message and drops it from the incoming stream.
 */
class PingReceiver : public wangle::HandlerAdapter<folly::IOBufQueue &,
                         std::unique_ptr<folly::IOBuf>> {
public:
    void read(Context *ctx, folly::IOBufQueue &buf) override
    {
        if (buf.chainLength() < 1)
            return;

        buf.gather(1);
        if (UNLIKELY(*buf.front()->data() == pingMsg_)) {
            VLOG(4) << "Received ping over data connection";
            buf.trimStart(1);
        }

        if (LIKELY(buf.chainLength() > 0))
            ctx->fireRead(buf);
    }
};
