%%%-------------------------------------------------------------------
%%% @author Konrad Zemek
%%% @copyright (C) 2017 ACK CYFRONET AGH
%%% This software is released under the MIT license
%%% cited in 'LICENSE.txt'.
%%% @end
%%%-------------------------------------------------------------------
%%% @doc
%%% @end
%%%-------------------------------------------------------------------
-module(port_mock).

-behaviour(gen_server).

-compile([export_all]).

%%%===================================================================
%%% API
%%%===================================================================

start_link(Opts) ->
    gen_server:start_link({local, ?MODULE}, ?MODULE, Opts, []).

request(Req) ->
    ReqId = uuid(),
    gen_server:cast(?MODULE, {request, self(), Req#{req_id => ReqId}}),
    ReqId.

sync_request(Req) ->
    sync_request(Req, 5000).

sync_request(Req, _Timeout) ->
    ReqId = request(Req),
    receive
        {response, ReqId, Response} -> Response
    end.

fulfill_fetch(ReqId, Size) ->
    ok = gen_server:call(?MODULE, {fulfill_fetch, ReqId, Size}).

%%%===================================================================
%%% gen_server callbacks
%%%===================================================================

init(Opts) ->
    {ok, #{opts => Opts}}.

handle_call({fulfill_fetch, FetchReqId, Size0}, _From,
            #{fetches := Fetches, opts := Opts} = State) ->
    case maps:get(FetchReqId, Fetches, undefined) of
        undefined -> {reply, ok, State};
        {From, ReqId, #{offset := Offset, size := FetchSize}} = FetchDetails ->
            Size = case is_integer(Size0) of true -> Size0; _ -> FetchSize end,
            BlockSize = proplists:get_value(block_size, Opts, 1024 * 1024),
            send_updates(From, ReqId, Offset, Size, BlockSize),
            respond(From, ReqId, #{<<"wrote">> => integer_to_binary(Size)}),
            NewState = maps:update_with(
                          fetches,
                          fun(Fetches) -> maps:remove(FetchReqId, Fetches) end,
                          State),
            {reply, ok, NewState}
    end;
handle_call(_Message, _From, State) ->
    {reply, ok, State}.

handle_cast({request, From, Req}, State) ->
    NewState = handle_request(Req, From, State),
    {noreply, NewState}.

handle_info(_Message, State) ->
    {noreply, State}.

terminate(_Reason, _State) ->
    ok.

code_change(_Vsn, State, _Extra) ->
    {ok, State}.

%%%===================================================================
%%% Helpers
%%%===================================================================

handle_request(#{req_id := ReqId,
                 create_helper := StorageDetails =
                     #{storage_id := StorageId,
                       name := HelperName,
                       params := HelperArgs,
                       io_buffered := IOBuffered}}, From, State) ->
    respond(From, ReqId, #{<<"done">> => true}),
    NewElem = #{StorageId => StorageDetails},
    maps:update_with(storages,
                     fun(Storages) -> maps:merge(Storages, NewElem) end,
                     NewElem, State);

handle_request(#{req_id := ReqId,
                 connect := ConnDetails =
                     #{peer_host := Host,
                       peer_port := Port}}, From, State) ->
    ConnId = uuid(),
    respond(From, ReqId, #{<<"connectionId">> => ConnId}),
    NewElem = #{ConnId => ConnDetails},
    maps:update_with(connections,
                     fun(Conns) -> maps:merge(Conns, NewElem) end,
                     NewElem, State);

handle_request(#{req_id := ReqId,
                 fetch := FetchDetails =
                     #{connection_id := ConnId,
                       src_storage_id := SrcStorageId,
                       src_file_id := SrcFileId,
                       dest_storage_id := DestStorageId,
                       dest_file_id := DestFileId,
                       offset := Offset,
                       size := Size,
                       priority := Priority,
                       req_id := FetchReqId}}, From,
               #{connections := Conns, opts := Opts} = State) ->
    coordinator ! {new_fetch, FetchReqId},
    case maps:is_key(ConnId, Conns) of
        false ->
            respond(From, ReqId, {error, <<"no such connection ", ConnId/binary>>}),
            State;
        true ->
            NewElem = #{FetchReqId => {From, ReqId, FetchDetails}},
            maps:update_with(fetches,
                             fun(Fetches) -> maps:merge(Fetches, NewElem) end,
                             NewElem, State)
    end;

handle_request(#{req_id := ReqId,
                 cancel :=
                     #{connection_id := ConnId,
                       src_storage_id := SrcStorageId,
                       dest_storage_id := DestStorageId,
                       req_id := FetchReqId}} = Req, From,
               #{connections := Conns} = State) ->
    Fetches = maps:get(fetches, State, #{}),
    case maps:is_key(ConnId, Conns) of
        false ->
            respond(From, ReqId, {error, <<"no such connection ", ConnId/binary>>}),
            State;
        true ->
            case maps:get(FetchReqId, Fetches, undefined) of
                {Fr, RID, _Details} ->
                    respond(Fr, RID, {error, <<"canceled">>}),
                    respond(From, ReqId, #{<<"done">> => true}),
                    maps:update_with(fetches,
                                     fun(Fetches) -> maps:remove(FetchReqId, Fetches) end,
                                     State);
                undefined ->
                    %% Fetch got queued after cancel; retry
                    gen_server:cast(?MODULE, {request, From, Req}),
                    State
            end
    end.

send_updates(_, _, _, 0, _) -> ok;
send_updates(From, ReqId, Offset, Size, BlockSize) ->
    UpdateSize = min(Size, BlockSize),
    respond(From, ReqId, #{<<"isUpdate">> => true,
                           <<"progress">> => #{<<"offset">> => integer_to_binary(Offset),
                                               <<"size">> => integer_to_binary(UpdateSize)}}),
    send_updates(From, ReqId, Offset + UpdateSize, Size - UpdateSize, BlockSize).

respond(From, ReqId, {error, Reason}) ->
    From ! {response, ReqId, {error, Reason}};
respond(From, ReqId, Response) ->
    From ! {response, ReqId, Response#{<<"reqId">> => ReqId}}.

uuid() ->
    uuid:uuid_to_string(uuid:get_v4(), binary_nodash).
