/*
 * Copyright 2017-present Facebook, Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
#pragma once

#include <folly/executors/IOThreadPoolExecutor.h>
#include <folly/futures/Promise.h>
#include <folly/io/SocketOptionMap.h>
#include <folly/io/async/AsyncSSLSocket.h>
#include <folly/io/async/AsyncSocket.h>
#include <folly/io/async/DestructorCheck.h>
#include <folly/io/async/EventBaseManager.h>
#include <wangle/bootstrap/BaseClientBootstrap.h>
#include <wangle/channel/Pipeline.h>

namespace rtransfer {

/*
 * A thin wrapper around Pipeline and AsyncSocket to match
 * ServerBootstrap.  On connect() a new pipeline is created.
 */
/**
 * This is an exact copy of @link
 * https://github.com/facebook/wangle/blob/v2017.10.30.00/wangle/bootstrap/ClientBootstrap.h
 * The single difference is setSocketOptions that allows us to set socket
 * options for the client socket *before* connection. This is used to set
 * receive buf size - man socket states that this option only has effect when
 * set before connection, so @c ClientBootstrap is the only place we can do this
 * without ditching bootstrap object alltogether.
 */
template <typename Pipeline>
class ClientBootstrap : public wangle::BaseClientBootstrap<Pipeline>,
                        public folly::DestructorCheck {
    class ConnectCallback : public folly::AsyncSocket::ConnectCallback {
    public:
        ConnectCallback(folly::Promise<Pipeline*> promise,
            ClientBootstrap* bootstrap,
            std::shared_ptr<folly::AsyncSocket> socket)
            : promise_{std::move(promise)}
            , bootstrap_{bootstrap}
            , socket_{std::move(socket)}
            , safety_{*bootstrap}
        {
        }

        void connectSuccess() noexcept override
        {
            if (!safety_.destroyed()) {
                bootstrap_->makePipeline(std::move(socket_));
                if (bootstrap_->getPipeline()) {
                    bootstrap_->getPipeline()->transportActive();
                }
                try {
                    promise_.setValue(bootstrap_->getPipeline());
                }
                catch (folly::PromiseAlreadySatisfied& /*e*/) {
                }
            }
            delete this;
        }

        void connectErr(const folly::AsyncSocketException& ex) noexcept override
        {
            try {
                promise_.setException(
                    folly::make_exception_wrapper<folly::AsyncSocketException>(
                        ex));
            }
            catch (folly::PromiseAlreadySatisfied& /*e*/) {
            }
            delete this;
        }

    private:
        folly::Promise<Pipeline*> promise_;
        ClientBootstrap* bootstrap_;
        std::shared_ptr<folly::AsyncSocket> socket_;
        folly::DestructorCheck::Safety safety_;
    };

public:
    ClientBootstrap() = default;

    ClientBootstrap* group(std::shared_ptr<folly::IOThreadPoolExecutor> group)
    {
        group_ = group;
        return this;
    }

    ClientBootstrap* bind(int port)
    {
        port_ = port;
        return this;
    }

    ClientBootstrap* setSocketOptions(folly::SocketOptionMap opts)
    {
        options_ = std::move(opts);
        return this;
    }

    folly::Future<Pipeline*> connect(const folly::SocketAddress& address,
        std::chrono::milliseconds timeout) override
    {
        auto* base = (group_) ? group_->getEventBase()
                              : folly::EventBaseManager::get()->getEventBase();
        folly::Future<Pipeline*> retval(static_cast<Pipeline*>(nullptr));
        base->runImmediatelyOrRunInEventBaseThreadAndWait([&]() {
            std::shared_ptr<folly::AsyncSocket> socket;
            if (this->sslContext_) {
                auto sslSocket = folly::AsyncSSLSocket::newSocket(
                    this->sslContext_, base, this->deferSecurityNegotiation_);
                if (this->sslSession_) {
                    sslSocket->setSSLSession(this->sslSession_);
                }
                socket = sslSocket;
            }
            else {
                socket = folly::AsyncSocket::newSocket(base);
            }
            folly::Promise<Pipeline*> promise;
            retval = promise.getFuture();
            socket->connect(
                // NOLINTNEXTLINE(cppcoreguidelines-owning-memory)
                new ConnectCallback(std::move(promise), this, socket), address,
                timeout.count(), options_);
        });
        return retval;
    }

    ~ClientBootstrap() override = default;

private:
    int port_{};
    std::shared_ptr<folly::IOThreadPoolExecutor> group_;
    folly::SocketOptionMap options_;
};

}  // namespace rtransfer
