#pragma once

#include <folly/FBString.h>
#include <folly/FBVector.h>
#include <folly/Random.h>
#include <folly/SpinLock.h>
#include <folly/concurrency/ConcurrentHashMap.h>
#include <glog/logging.h>
#include <sys/ioctl.h>
#include <wangle/channel/Handler.h>
#include <wangle/channel/Pipeline.h>

#include <algorithm>

#include "common.hpp"
#include "proto/rtransfer_messages.pb.h"

namespace rtransfer {

class DataConn;

/**
 * @c ServerSideLink is a Wangle handler that collects a control pipeline and
 * data pipelines into a single "link".
 * The link is registered in a global concurrent hash map that maps ctrlId to a
 * @c ServerSideLink object, so new data pipelines can be connected into the
 * link.
 */
class ServerSideLink
    : public wangle::HandlerAdapter<std::unique_ptr<proto::LinkMessage>> {
public:
    using Conn = std::pair<wangle::DefaultPipeline::Ptr, wangle::TransportInfo>;

    /**
     * Constructor.
     * @param peerAddr Address of the peer on the other side of the link, used
     * for logging and metrics purposes.
     */
    explicit ServerSideLink(std::string peerAddr);

    std::string getPeerAddr();

    void attachPipeline(Context *ctx) override;

    /**
     * Adds a data pipeline to this link.
     */
    void addDataConnection(wangle::DefaultPipeline::Ptr pipeline);

    /**
     * Removes a data pipeline from this link.
     */
    void removeDataConnection(wangle::DefaultPipeline::Ptr pipeline);

    /**
     * Returns data pipelines registered with this link, along with @c
     * wangle::TransportInfo for the associated connections.
     * The transport info is used by @c Reader to estimate bandwidth over a
     * specific pipeline.
     */
    folly::fbvector<Conn> dataConnections(
        std::chrono::steady_clock::time_point now);

    void readEOF(Context *ctx) override;

private:
    void maybeProbe(std::chrono::steady_clock::time_point now);
    wangle::TransportInfo probe(const wangle::DefaultPipeline::Ptr &conn);

    std::string peerAddr_;
    Context *ctrlCtx_ = nullptr;
    folly::fbvector<Conn> dataConns_;
    folly::SpinLock lock_;

    std::chrono::steady_clock::time_point lastProbe_;
};

/**
 * A Wangle handler that's a part of every data pipeline, the sole purpose of
 * which is to register the pipeline with the given @c ServerSideLink .
 */
class DataConn : public wangle::HandlerAdapter<folly::IOBufQueue &,
                     std::unique_ptr<folly::IOBuf>> {
public:
    explicit DataConn(std::shared_ptr<ServerSideLink> link);

    void attachPipeline(Context *ctx) override;

    void readEOF(Context *ctx) override;

private:
    wangle::DefaultPipeline::Ptr getPipeline(Context *ctx);

    std::shared_ptr<ServerSideLink> link_;
};

/**
 * Handles @c ServerSideLink registration in a given map of links, and
 * associates data connection with the right link for their ctrlId.
 */
class ServerSideLinkFactory {
public:
    explicit ServerSideLinkFactory(
        folly::ConcurrentHashMap<folly::fbstring, std::weak_ptr<ServerSideLink>>
            &map);

    std::shared_ptr<ServerSideLink> operator()(
        folly::StringPiece ctrlId, std::string peerAddr);

    std::shared_ptr<DataConn> createDataConn(const folly::fbstring &ctrlId);

private:
    folly::ConcurrentHashMap<folly::fbstring, std::weak_ptr<ServerSideLink>>
        &map_;
};

}  // namespace rtransfer
