%%%-------------------------------------------------------------------
%%% @author Konrad Zemek
%%% @copyright (C) 2017 ACK CYFRONET AGH
%%% This software is released under the MIT license
%%% cited in 'LICENSE.txt'.
%%% @end
%%%-------------------------------------------------------------------
%%% @doc
%%% {@module} provides API for fetching data from a remote provider.
%%% @end
%%%-------------------------------------------------------------------
-module(rtransfer_link).
-author("Konrad Zemek").

-behaviour(gen_server).

%%%===================================================================
%%% Type definitions
%%%===================================================================

-type hostname() :: inet:ip_address() | string() | binary().

-type address() :: {hostname(), inet:port_number()}.

-type connection() :: pid().

-type request() :: rtransfer_link_request:t().

-type error_reason() ::
        canceled | {connection | storage | other, Reason :: any()}.

-type notify_fun() ::
        fun((Ref :: reference(), Offset :: non_neg_integer(),
             Size :: pos_integer()) -> ok).

-type on_complete_fun() ::
        fun((Ref :: reference(),
             {ok, Size :: non_neg_integer()} | {error, error_reason()}) -> ok).

-type state() :: #{
             conns := #{address() => connection()},
             requests := #{connection() => #{reference() => {}}},
             storages := #{binary() => [{binary(), binary()}]}
            }.

-export_type([address/0, hostname/0]).

%%%===================================================================
%%% Exports
%%%===================================================================

-export_type([notify_fun/0, on_complete_fun/0]).
-export([start_link/0, fetch/4, fetch/5, cancel/1, add_storage/3,
         set_provider_nodes/2, allow_connection/4, get_memory_stats/0]).
-export([init/1, handle_info/2, handle_cast/2, code_change/3, terminate/2, handle_call/3]).

%%%===================================================================
%%% API
%%%===================================================================

-spec start_link() -> {ok, pid()} | {error, any()}.
start_link() ->
    gen_server2:start_link({local, ?MODULE}, ?MODULE, {}, []).

-spec fetch(request(), TransferData :: binary(), Notify :: notify_fun(),
            OnComplete :: on_complete_fun()) ->
                   {ok, reference()} | {error, Reason :: any()}.
fetch(Request, TransferData, Notify, OnComplete) ->
    fetch(?MODULE, Request, TransferData, Notify, OnComplete).

-spec fetch(term(), request(), TransferData :: binary(), Notify :: notify_fun(),
            OnComplete :: on_complete_fun()) ->
                   {ok, reference()} | {error, Reason :: any()}.
fetch(Server, Request0, TransferData, Notify, OnComplete) ->
    try
        check_request(Request0),
        Ref = make_ref(),
        Request = Request0#{ref => Ref, transfer_data => TransferData,
                            on_complete => OnComplete, notify => Notify},
        ok = gen_server2:call(Server, {fetch, Request}, infinity),
        {ok, Ref}
    catch
        {?MODULE, Reason} ->
            {error, Reason}
    end.

-spec cancel(reference()) -> ok.
cancel(Ref) ->
    gen_server2:call(?MODULE, {cancel, Ref}, infinity).

-spec set_provider_nodes(Nodes :: [node()], CallbackModule :: module()) -> any().
set_provider_nodes(Nodes, CallbackModule) ->
    rtransfer_link_callback:set_provider_nodes(Nodes, CallbackModule).

-spec add_storage(StorageId :: binary(), HelperName :: binary(),
                  HelperArgs :: [{binary(), binary()}]) -> ok.
add_storage(StorageId, HelperName, HelperArgs) ->
    % this procedure may take more than 10 seconds on slower machines
    gen_server2:call(?MODULE, {add_storage, StorageId, HelperName, HelperArgs}, timer:seconds(30)).

-spec allow_connection(ProviderId :: binary(), MySecret :: binary(),
                       PeerSecret :: binary(), Expiration :: non_neg_integer()) -> ok.
allow_connection(ProviderId, MySecret, PeerSecret, Expiration) ->
    gen_server2:call(?MODULE, {allow_connection, ProviderId, MySecret,
                               PeerSecret, Expiration}).

-spec get_memory_stats() -> iodata() | {error, timeout}.
get_memory_stats() ->
    rtransfer_link_port:sync_request(#{get_memory_stats => #{}}).

%%%===================================================================
%%% gen_server callbacks
%%%===================================================================

-spec init(term()) -> {ok, state()}.
init(_) ->
    %% Add all previously added storages in case of a restart.
    [gen_server2:cast(self(), R) || R <- ets:tab2list(rtransfer_link_storages)],
    [gen_server2:cast(self(), {fetch, Req}) ||
        {_Ref, Req} <- ets:tab2list(rtransfer_link_requests)],
    {ok, #{
       requests => #{},
       conns => #{},
       storages => #{}
      }}.

-spec handle_call(term(), {pid(), term()}, state()) ->
                         {reply, term(), state()}.
handle_call({allow_connection, ProviderId, MySecret, PeerSecret, Expiration}, _From, State) ->
    Req = #{allow_connection => #{my_secret => base64:encode(MySecret),
                                  peer_secret => base64:encode(PeerSecret),
                                  provider_id => base64:encode(ProviderId),
                                  expiration => Expiration}},
    Reply = rtransfer_link_port:sync_request(Req),
    {reply, Reply, State};

handle_call({fetch, Req}, _From, State) ->
    NewState = do_fetch(Req, State),
    {reply, ok, NewState};

handle_call({cancel, Ref}, _From, State) ->
    case ets:lookup(rtransfer_link_requests, Ref) of
        [{_, #{conn := Conn} = Req}] -> rtransfer_link_connection:cancel(Conn, Req);
        _ -> ok
    end,
    ets:delete(rtransfer_link_requests, Ref),
    {reply, ok, State};

handle_call({add_storage, _, _, _} = Request, _From, State) ->
    {Ans, State2} = check_and_add_new_storage(Request, State),
    {reply, Ans, State2};

handle_call(_, _From, State) ->
    {reply, not_implemented, State}.

-spec handle_info(term(), state()) -> {noreply, state()}.
handle_info({'DOWN', _, process, Connection, _Reason}, State) ->
    NewState = conn_down(Connection, State),
    {noreply, NewState};

handle_info(Message, State) ->
    lager:debug("Ignoring message: ~p", [Message]),
    {noreply, State}.

handle_cast({fetch, Req}, State) ->
    NewState = do_fetch(Req, State),
    {noreply, NewState};

handle_cast({done, Conn, Ref}, State) ->
    NewState = forget_request(Conn, Ref, State),
    {noreply, NewState};

handle_cast({add_storage, _, _, _} = Request, State) ->
    {_, State2} = check_and_add_new_storage(Request, State),
    {noreply, State2}.

-spec code_change(term(), state(), term()) -> {ok, state()}.
code_change(_, State, _Extra) ->
    {ok, State}.

-spec terminate(term(), state()) -> ok.
terminate(_, _) ->
    ok.

%%%===================================================================
%%% Helpers
%%%===================================================================

-spec conn_down(connection(), state()) -> state().
conn_down(Conn, #{conns := Conns, requests := Requests} = State) ->
    AbandonedReqs = maps:get(Conn, Requests, #{}),
    NewRequests = maps:remove(Conn, Requests),
    NewConns = maps:filter(fun(_K, V) -> V =/= Conn end, Conns),
    NewState = State#{conns := NewConns, requests := NewRequests},
    maps:fold(
      fun(Ref, _, S) ->
              #{on_complete := OnComplete} = ets:lookup_element(rtransfer_link_requests, Ref, 2),
              ets:delete(rtransfer_link_requests, Ref),
              erlang:apply(OnComplete, [Ref, {error, disconnected}]),
              S
      end,
      NewState, AbandonedReqs).

-spec do_fetch(map(), state()) -> state().
do_fetch(#{ref := Reference} = Req0, #{requests := Requests} = State) ->
    ReqId = make_req_id(),
    case rtransfer_link_quota_manager:register_request(Req0) of
        false ->
            #{on_complete := OnComplete0} = Req0,
            erlang:apply(OnComplete0, [Reference, {error, <<"quota exceeded">>}]),
            State;

        OnComplete1 ->
            {Conn, NewState} = get_conn_for_req(Req0, State),

            %% Note that Req#OnComplete is not modified and instead Req is saved
            %% with its original OnComplete0 - allows us to easily restart the
            %% operation, wrapping OnComplete again
            Req = Req0#{req_id => ReqId, conn => Conn},
            Self = self(),
            OnComplete = fun(Ref, Status) ->
                                 gen_server2:cast(Self, {done, Conn, Ref}),
                                 erlang:apply(OnComplete1, [Ref, Status])
                         end,

            ets:insert(rtransfer_link_requests, {Reference, Req}),
            rtransfer_link_connection:fetch(Conn, Req#{on_complete => OnComplete}),

            ReqsForConn = maps:get(Conn, Requests, #{}),
            NewReqsForConn = maps:put(Reference, {}, ReqsForConn),
            NewRequests = maps:put(Conn, NewReqsForConn, Requests),
            NewState#{requests := NewRequests}
    end.

-spec get_conn_for_req(request(), state()) -> {connection(), state()}.
get_conn_for_req(#{provider_id := ProvId}, #{conns := Conns} = State) ->
    Addrs = rtransfer_link_callback:get_nodes(ProvId),
    {Host0, Port} = rtransfer_link_utils:random_element(Addrs),
    Addr = {Host, Port} = {rtransfer_link_utils:hostname_to_binary(Host0), Port},
    case maps:get(Addr, Conns, false) of
        false ->
            {ok, NewConn} = supervisor:start_child(rtransfer_link_connection_sup,
                                                   [ProvId, Host, Port]),
            erlang:monitor(process, NewConn),
            NewConns = maps:put(Addr, NewConn, Conns),
            {NewConn, State#{conns := NewConns}};

        Conn ->
            {Conn, State}
    end.

-spec do_add_storage(StorageId :: binary(), HelperName :: binary(),
                     HelperArgs :: [{binary(), binary()}]) ->
                            {error, any()} | term().
do_add_storage(StorageId, HelperName, HelperArgs0) ->
    HelperArgs = [#{key => K, value => base64:encode(V)} || {K, V} <- HelperArgs0],
    Req = #{create_helper => #{storage_id => base64:encode(StorageId),
                               name => HelperName,
                               params => HelperArgs,
                               io_buffered => false}},
    try_add_storage(StorageId, Req).

-spec try_add_storage(StorageId :: binary(), Req :: #{create_helper := map()}) ->
                             {error, any()} | term().
try_add_storage(StorageId, Req) ->
    case rtransfer_link_port:sync_request(Req) of
        {error, timeout} ->
            lager:error("error adding storage ~p: timeout; retrying", [StorageId]),
            timer:sleep(100),
            try_add_storage(StorageId, Req);
        {error, Reason} ->
            lager:error("failed to add storage ~p due to ~p", [StorageId, Reason]),
            {error, Reason};
        Result ->
            Result
    end.

-spec check_request(request()) -> ok | no_return().
check_request(R) ->
    maps:is_key(ref, R) andalso throw({?MODULE, {bad_key, ref, R}}),
    lists:foreach(
      fun({Key, Validator}) ->
              maps:is_key(Key, R) orelse throw({?MODULE, {no_key, Key, R}}),
              Val = maps:get(Key, R),
              Validator(Val) orelse throw({?MODULE, {bad_val, Key, Val}})
      end,
      [{provider_id,     fun is_binary/1},
       {file_guid,       fun is_binary/1},
       {src_storage_id,  fun is_binary/1},
       {src_file_id,     fun is_binary/1},
       {dest_storage_id, fun is_binary/1},
       {dest_file_id,    fun is_binary/1},
       {space_id,        fun is_binary/1},
       {offset,          fun(O) -> is_integer(O) andalso O >= 0 end},
       {size,            fun(S) -> is_integer(S) andalso S > 0 end},
       {priority,        fun(P) -> is_integer(P) andalso P >= 0 andalso P =< 255 end}]).

-spec forget_request(connection(), reference(), state()) -> state().
forget_request(Conn, Ref, #{requests := Requests} = State) ->
    ets:delete(rtransfer_link_requests, Ref),
    ReqsForConn = maps:get(Conn, Requests, #{}),
    NewReqsForConn = maps:remove(Ref, ReqsForConn),
    NewRequests =
        case map_size(NewReqsForConn) of
            0 -> maps:remove(Conn, Requests);
            _ -> maps:put(Conn, NewReqsForConn, Requests)
        end,
    State#{requests := NewRequests}.

-spec make_req_id() -> integer().
make_req_id() ->
    Bytes = crypto:strong_rand_bytes(8),
    binary:decode_unsigned(Bytes, big).

-spec check_and_add_new_storage({add_storage, StorageId :: binary(), HelperName :: binary(),
    HelperArgs :: [{binary(), binary()}]}, state()) -> {ok | {error, term()}, state()}.
check_and_add_new_storage({add_storage, StorageId, HelperName, HelperArgs} = Request, #{storages := Storages} = State) ->
    case maps:get(StorageId, Storages, undefined) of
        {HelperName, HelperArgs} ->
            {ok, State};
        _ ->
            ets:insert(rtransfer_link_storages, Request),
            Ans = do_add_storage(StorageId, HelperName, HelperArgs),
            FinalAns = case Ans of
                ok -> ok;
                #{<<"done">> := true} -> ok;
                _ -> {error, Ans}
            end,

            State2 = case FinalAns of
                ok ->
                    NewStorages = Storages#{StorageId => {HelperName, HelperArgs}},
                    State#{storages => NewStorages};
                _  ->
                    State
            end,

            {FinalAns, State2}
    end.