#pragma once

#include <google/protobuf/util/json_util.h>
#include <wangle/channel/Handler.h>

#include <string>

namespace rtransfer {

/**
 * A Wangle handler that translates JSON text into protobuf messages.
 */
template <typename Request, typename Response>
class JSONMessageHandler
    : public wangle::Handler<std::string, std::unique_ptr<Request>,
          std::unique_ptr<Response>, std::string> {
public:
    using Context =
        typename wangle::Handler<std::string, std::unique_ptr<Request>,
            std::unique_ptr<Response>, std::string>::Context;

    void read(Context *ctx, std::string data) override
    {
        auto msg = std::make_unique<Request>();
        auto status =
            google::protobuf::util::JsonStringToMessage(data, msg.get());
        if (status.ok())
            ctx->fireRead(std::move(msg));
        else
            readException(
                ctx, folly::make_exception_wrapper<std::runtime_error>(
                         status.error_message()));
    }

    folly::Future<folly::Unit> write(
        Context *ctx, std::unique_ptr<Response> msg) override
    {
        if (!msg)
            return folly::makeFuture();

        std::string data;
        auto status = google::protobuf::util::MessageToJsonString(*msg, &data);
        if (!status.ok())
            return ctx->fireWriteException(
                folly::make_exception_wrapper<std::runtime_error>(
                    status.error_message()));

        data.push_back('\n');
        return ctx->fireWrite(std::move(data));
    }

    void readException(Context *ctx, folly::exception_wrapper e) override
    {
        auto msg = std::make_unique<Response>();
        const auto &what = e.what();
        msg->mutable_error()->set_description(what.data(), what.size());
        write(ctx, std::move(msg));
    }
};

}  // namespace rtransfer
