%%%-------------------------------------------------------------------
%%% @author Konrad Zemek
%%% @copyright (C) 2018 ACK CYFRONET AGH
%%% This software is released under the MIT license
%%% cited in 'LICENSE.txt'.
%%% @end
%%%-------------------------------------------------------------------
%%% @doc
%%% @end
%%%-------------------------------------------------------------------
-module(rtransfer_link_callback).
-author("Konrad Zemek").

%%%===================================================================
%%% Exports
%%%===================================================================

-export([set_provider_nodes/2, get_connection_secret/2]).
-export([get_nodes/1, open/2, fsync/1, close/1, auth_request/2]).

%%%===================================================================
%%% Callback definitions
%%%===================================================================

-callback get_nodes(ProviderId :: binary()) ->
    [rtransfer_link:address()].

-callback open(FileUUID :: binary(), read | write) ->
    {ok, Handle :: term()} | {error, Reason :: any()}.

-callback fsync(Handle :: term()) -> any().

-callback close(Handle :: term()) -> any().

-callback auth_request(TransferData :: binary(), ProviderId :: binary()) ->
    false | {StorageId :: binary(), FileId :: binary(), FileGuid :: binary()}.

-callback get_connection_secret(ProviderId :: binary(), rtransfer_link:address()) ->
    {MySecret :: binary(), PeerSecret :: binary()}.

%%%===================================================================
%%% API
%%%===================================================================

set_provider_nodes(Nodes, CBMod) ->
    [ets:insert(rtransfer_link_provider_nodes, E)
     || E <- [{nodes, Nodes}, {cb_mod, CBMod}]].

get_connection_secret(ProviderId, Addr) ->
    {Node, CBMod} = get_node(),
    rpc_call(Node, CBMod, get_connection_secret, [ProviderId, Addr]).

get_nodes(ProviderId) ->
    {Node, CBMod} = get_node(),
    rpc_call(Node, CBMod, get_nodes, [ProviderId]).

open(FileUUID, OpenType) ->
    {Node, CBMod} = get_node(),
    case rpc_call(Node, CBMod, open, [FileUUID, OpenType]) of
        {ok, Handle} ->
            {ok, {Node, CBMod, Handle}};
        Other ->
            Other
    end.

fsync({CBNode, CBMod, Handle}) ->
    rpc_call(CBNode, CBMod, fsync, [Handle]).

close({CBNode, CBMod, Handle}) ->
    rpc_call(CBNode, CBMod, close, [Handle]).

auth_request(TransferData, ProviderId) ->
    {Node, CBMod} = get_node(),
    rpc_call(Node, CBMod, auth_request, [TransferData, ProviderId]).

%%%===================================================================
%%% Helpers
%%%===================================================================

get_node() ->
    Nodes = ets:lookup_element(rtransfer_link_provider_nodes, nodes, 2),
    CBMod = ets:lookup_element(rtransfer_link_provider_nodes, cb_mod, 2),
    Node = rtransfer_link_utils:random_element(Nodes),
    {Node, CBMod}.

rpc_call(Node, Module, Fun, Args) when Node =:= node() ->
    erlang:apply(Module, Fun, Args);
rpc_call(Node, Module, Fun, Args) ->
    rpc:call(Node, Module, Fun, Args).
