#pragma once

#include <folly/FBVector.h>
#include <folly/SpinLock.h>
#include <folly/executors/SerialExecutor.h>
#include <wangle/acceptor/ConnectionManager.h>
#include <wangle/acceptor/ManagedConnection.h>
#include <wangle/channel/StaticPipeline.h>
#include <wangle/codec/LengthFieldBasedFrameDecoder.h>
#include <wangle/codec/LengthFieldPrepender.h>

#include <atomic>
#include <memory>

#include "clientBootstrap.h"
#include "clientDispatcher.hpp"
#include "fetchManager.hpp"
#include "handleCache.hpp"
#include "periodicHandler.hpp"
#include "proto/rtransfer_messages.pb.h"
#include "protoHandler.hpp"

namespace rtransfer {
namespace client {

using MsgPtr = std::unique_ptr<proto::LinkMessage>;

using Pipeline = wangle::Pipeline<folly::IOBufQueue &, MsgPtr>;
using DataPipeline =
    wangle::Pipeline<folly::IOBufQueue &, std::unique_ptr<folly::IOBuf>>;
using DataBootstrapPtr =
    std::shared_ptr<rtransfer::ClientBootstrap<DataPipeline>>;

class Client;
class ConnectionCloseHandler;

/**
 * A Wangle handler adapter that reads ctrlId out of byte stream and passes any
 * further bytes downstream.
 */
class CtrlIdHandler : public wangle::HandlerAdapter<folly::IOBufQueue &,
                          std::unique_ptr<folly::IOBuf>> {
public:
    explicit CtrlIdHandler(folly::fbstring addr);

    void read(Context *ctx, folly::IOBufQueue &buf) override;

    /**
     * @returns A future object to a ctrlId read from the stream.
     */
    folly::Future<folly::fbstring> ctrlId();

private:
    folly::fbstring peerAddr_;
    folly::Promise<folly::fbstring> ctrlIdPromise_;
};

/**
 * A Wangle handler adapter that reads HTTP response out of a byte stream and
 * passes any further bytes downstream.
 */
class HttpResponseHandler : public wangle::HandlerAdapter<folly::IOBufQueue &,
                                std::unique_ptr<folly::IOBuf>> {
public:
    explicit HttpResponseHandler(folly::fbstring addr);

    void read(Context *ctx, folly::IOBufQueue &buf) override;

    /**
     * @returns A future object signalizing that the HTTP response has been
     * read.
     */
    folly::Future<folly::Unit> done();

private:
    folly::fbstring peerAddr_;
    folly::Promise<folly::Unit> promise_;
};

/**
 * A Wangle handler adapter that reads peer's secret out of a byte stream and
 * passes any further bytes downstream.
 */
class PeerSecretHandler : public wangle::HandlerAdapter<folly::IOBufQueue &,
                              std::unique_ptr<folly::IOBuf>> {
public:
    PeerSecretHandler(folly::fbstring addr, folly::fbstring peerSecret)
        : peerSecret_{std::move(peerSecret)}
        , peerAddr_{std::move(addr)}
    {
    }

    void read(Context *ctx, folly::IOBufQueue &buf) override;

    /**
     * @returns A future object signalizing the peer's secret has been read and
     * matches the expected secret.
     */
    folly::Future<folly::Unit> done();

private:
    folly::fbstring peerSecret_;
    folly::fbstring peerAddr_;
    folly::Promise<folly::Unit> promise_;
};

/**
 * @c PipelineFactory creates client control pipelines.
 * Client control pipeline most importantly contains a @c HttpResponseHandler ,
 * @c PeerSecretHandler and @c CtrlIdHandler - these handlers read and validate,
 * in order, a HTTP response to HTTP request that initializes the connection,
 * peer secret that authenticates the remote provider, and ctrlId that is used
 * to bundle the control connection and data connections into a single "link".
 * After these handlers are satisfied, data on the pipeline is decoded into @c
 * proto::LinkMessage by a @c ProtoHandler .
 */
class PipelineFactory : public wangle::PipelineFactory<Pipeline> {
public:
    PipelineFactory(
        std::shared_ptr<ConnectionCloseHandler> connectionCloseHandler,
        folly::fbstring peerSecret, const folly::SocketAddress &addr);

    Pipeline::Ptr newPipeline(
        std::shared_ptr<folly::AsyncTransportWrapper> sock) override;

private:
    std::shared_ptr<ConnectionCloseHandler> connectionCloseHandler_;
    folly::fbstring peerSecret_;
    folly::fbstring peerAddr_;
};

/**
 * @c ConnectionCloseHandler is a shared handler between client's data and
 * control pipelines that form a single link. The handler returns a future that
 * is fulfilled by any of the connection closing or encountering an exception.
 */
class ConnectionCloseHandler
    : public wangle::HandlerAdapter<folly::IOBufQueue &,
          std::unique_ptr<folly::IOBuf>> {
public:
    folly::Future<folly::Unit> getCloseFuture() { return promise_.getFuture(); }

    void readEOF(Context * /*ctx*/) override { setPromise(); }

    void readException(
        Context * /*ctx*/, folly::exception_wrapper /*e*/) override
    {
        setPromise();
    }

private:
    void setPromise()
    {
        if (!promiseFulfilled_.test_and_set())
            promise_.setValue();
    }

    std::atomic_flag promiseFulfilled_ = ATOMIC_FLAG_INIT;
    folly::Promise<folly::Unit> promise_;
};

/**
 * @c PipelineFactory creates client data pipelines.
 * Client data pipeline contains a @c HttpResponseHandler and @c
 * PeerSecretHandler and @c CtrlIdHandler - these handlers read and validate, in
 * order, a HTTP response to HTTP request that initializes the connection and
 * peer secret that authenticates the remote provider. A @c PingSender handler
 * ensures that the connection is not closed from lack of network activity.
 * After these handlers are satisfied, data is split into frames and sent to a
 * @c FetchManager instance shared between data pipelines that belong to a
 * single "link"
 * @note Sockets' receive buffer size will be set to `FLAGS_recv_buf_size`.
 * @note The ping sender sends data from client to server, and it is the only
 * data sent in this direction so it does not interfere with actually useful
 * data.
 */
class DataPipelineFactory : public wangle::PipelineFactory<DataPipeline> {
public:
    DataPipelineFactory(std::shared_ptr<FetchManager> fetchManager,
        std::shared_ptr<ConnectionCloseHandler> connectionCloseHandler,
        folly::fbstring peerSecret, const folly::SocketAddress &addr);

    DataPipeline::Ptr newPipeline(
        std::shared_ptr<folly::AsyncTransportWrapper> sock) override;

private:
    std::shared_ptr<FetchManager> fetchManager_;
    std::shared_ptr<ConnectionCloseHandler> connectionCloseHandler_;
    folly::fbstring peerSecret_;
    folly::fbstring peerAddr_;
};

/**
 * A @c Client instance manages a control and data pipelines that form a single
 * "link" between providers.
 * This object exposes an API for this logical link, e.g. a `fetch` operation
 * sends a protocol message over the control pipeline and returns a future that
 * will be fulfilled by data received on the data pipelines.
 * The client will send a ping message through its control channel every 10s
 * via @c PeriodicHandler .
 */
class Client : public std::enable_shared_from_this<Client> {
public:
    using ClientBootstrapPtr =
        std::shared_ptr<rtransfer::ClientBootstrap<Pipeline>>;

    /**
     * @param addr Address to which to connect to establish a link.
     * @param executor Async operations of the @c Client instance, mainly
     * sockets, will use this executor.
     * @param storages A map of available @c Storage instances.
     * @param mySecret A previously-established secret we'll use to authenticate
     * with remote provider.
     * @param peerSecret A previously-established secret the peer will use to
     * authenticate itself with us.
     */
    Client(folly::SocketAddress addr,
        std::shared_ptr<folly::IOThreadPoolExecutor> executor,
        StoragesMap &storages, folly::fbstring mySecret,
        folly::fbstring peerSecret);

    /**
     * Closes connections that belong to this instance.
     */
    ~Client();

    Client(const Client &) = delete;
    Client(Client &&) = delete;
    Client &operator=(const Client &) = delete;
    Client &operator=(Client &&) = delete;

    /**
     * Establish a link.
     * A control pipeline is created first and if successful,
     * `FLAGS_number_of_data_conns` data pipelines will be created in parallel.
     * @returns a future that will be fulfilled once the link is established.
     */
    folly::Future<folly::Unit> connect();

    /**
     * Ping the remote endpoint.
     * @returns a future that will be fulfilled when a "pong" message is
     * received.
     */
    folly::Future<folly::Unit> ping();

    /**
     * Fetch data from the remote endpoint.
     * @note `srcStorageId`, `srcFileId` and `fileGuid` are no longer relevant
     * since the remote endpoint will use `transferData` to authorize our
     * request and fetch storage and file ids for the file we're interested in.
     * These parameters were left in only for backwards compatibility but can
     * probably be removed at the moment I'm writing this doc.
     * @param destStorageId Id of the storage the data will be written to.
     * @param destFileId Id of the file the data will be written to.
     * @param offset Where are we reading the data from.
     * @param size How much data are we reading.
     * @param priority How important is getting this data to us.
     * @param reqId An integer that will be used to identify this fetch request.
     * @param transferData Opaque data that will be used on the provider side to
     * authorize and direct our request.
     * @param notifyCb A callback that will be called (possibly multiple times)
     * with an offset and size as pieces of the request start coming in.
     * @returns A future that will be fulfilled once all data is read.
     */
    folly::Future<std::size_t> fetch(folly::StringPiece srcStorageId,
        folly::StringPiece srcFileId, folly::StringPiece destStorageId,
        folly::StringPiece destFileId, folly::StringPiece fileGuid,
        std::uint64_t offset, std::size_t size, std::uint8_t priority,
        std::uint64_t reqId, folly::StringPiece transferData,
        folly::Function<void(std::uint64_t, std::size_t)> notifyCb);

    /**
     * Cancels an ongoing fetch request.
     * @note `srcStorageId` and `destStorageId` are no longer relevant since
     * `reqId` is enough for the remote endpoint to find data structures that
     * contain the request.
     * @param reqId ID of the request to cancel, as given in `fetch`.
     * @returns A future that says the cancel went through, though it doesn't
     * contain information whether there was anything to cancel.
     */
    folly::Future<folly::Unit> cancel(std::uint64_t reqId,
        folly::StringPiece srcStorageId, folly::StringPiece destStorageId);

    /**
     * Sends an ack message for received data.
     * The acks are bundled and sent to the remote provider.
     * This API function is used by @c FetchManager , and is called once a piece
     * of data we received has been written to the local storage.
     * @param reqId ID of the request the data has been written for.
     * @param offset offset of the data we received.
     * @note We write whole blocks received from the remote, so if we ack data
     * on offset X the remote can immediately correlate that to a specific block
     * it sent starting at offset X.
     */
    void ack(std::uint64_t reqId, std::uint64_t offset);

    /**
     * @returns A future that will be fulfilled once any connection is closed or
     * encounters an error.
     */
    folly::Future<folly::Unit> getCloseFuture()
    {
        return connectionCloseHandler_->getCloseFuture();
    };

    /**
     * Closes this connection right now.
     */
    void closeNow();

    HandleCache &handleCache() { return handleCache_; }

private:
    static folly::Future<MsgPtr> ensureResponse(
        MsgPtr msg, proto::LinkMessage::PayloadCase expected);

    folly::Future<folly::Unit> newDataConn();

    ClientBootstrapPtr ctrl();

    folly::fbvector<DataBootstrapPtr> &data();

    std::unique_ptr<folly::IOBuf> makeHandshake(folly::StringPiece ctrlId);
    void ackLoop();

    std::unique_ptr<PeriodicHandler> pinger_;

    StoragesMap &storages_;
    folly::fbstring mySecret_;
    folly::fbstring peerSecret_;
    folly::SocketAddress addr_;
    HandleCache handleCache_;
    std::shared_ptr<FetchManager> fetchManager_ =
        std::make_shared<FetchManager>(*this, storages_, handleCache_);
    folly::fbstring ctrlId_;
    ClientBootstrapPtr ctrl_{
        std::make_shared<rtransfer::ClientBootstrap<Pipeline>>()};
    folly::fbvector<DataBootstrapPtr> data_;
    ClientDispatcher dispatcher_;
    wangle::Service<MsgPtr, MsgPtr> &service_{dispatcher_};

    std::shared_ptr<ConnectionCloseHandler> connectionCloseHandler_ =
        std::make_shared<ConnectionCloseHandler>();

    std::shared_ptr<folly::IOThreadPoolExecutor> clientExecutor_;
    folly::Executor::KeepAlive<folly::SerialExecutor> executor_;

    folly::SpinLock acksLock_;
    folly::fbvector<std::pair<std::uint64_t, std::uint64_t>> pendingAcks_;
    folly::fbvector<std::pair<std::uint64_t, std::uint64_t>> usedAcks_;
    bool someoneHandlingAcks_ = false;

    std::atomic_flag closeNowFlag_ = ATOMIC_FLAG_INIT;
};

}  // namespace client
}  // namespace rtransfer
