%%%-------------------------------------------------------------------
%%% @author Michał Wrzeszcz
%%% @copyright (C) 2018 ACK CYFRONET AGH
%%% This software is released under the MIT license
%%% cited in 'LICENSE.txt'.
%%% @end
%%%-------------------------------------------------------------------
%%% @doc
%%% @end
%%%-------------------------------------------------------------------
-module(rtransfer_link_monitor).
-author("Michał Wrzeszcz").

-behaviour(gen_server).

%%%===================================================================
%%% Type definitions
%%%===================================================================

-type state() :: #{reference() => rtransfer_link:on_complete_fun()}.

%%%===================================================================
%%% Exports
%%%===================================================================

-export([start_link/1, monitor/3, cancel_ref/2, terminate/1]).
-export([init/1, handle_info/2, handle_cast/2, code_change/3,
         terminate/2, handle_call/3]).

%%%===================================================================
%%% API
%%%===================================================================

-spec start_link(pid()) -> {ok, pid()} | {error, any()}.
start_link(Pid) ->
    gen_server2:start_link(?MODULE, Pid, []).

-spec monitor(pid(), reference(), rtransfer_link:on_complete_fun()) -> ok.
monitor(Pid, Ref, OnCompleteFun) ->
    gen_server2:call(Pid, {guard, Ref, OnCompleteFun}, infinity).

-spec cancel_ref(pid(), reference()) -> ok.
cancel_ref(Pid, Ref) ->
    gen_server2:call(Pid, {cancel, Ref}, infinity).

-spec terminate(pid()) -> ok.
terminate(Pid) ->
    gen_server2:call(Pid, terminate, infinity).

%%%===================================================================
%%% gen_server callbacks
%%%===================================================================

-spec init(term()) -> {ok, state()}.
init(Pid) ->
    erlang:monitor(process, Pid),
    {ok, #{}}.

-spec handle_call(term(), {pid(), term()}, state()) ->
                         {reply, term(), state()}.
handle_call({guard, Ref, OnCompleteFun}, _, State) ->
    State2 = maps:put(Ref, OnCompleteFun, State),
    {reply, ok, State2};
handle_call({cancel, Ref}, _, State) ->
    State2 = case maps:get(Ref, State, undefined) of
        undefined ->
            lager:error("Unknown reference ~p", [Ref]),
            State;
        _ ->
            maps:remove(Ref, State)
    end,
    {reply, ok, State2};
handle_call(terminate, From, State) ->
    gen_server:reply(From, ok),
    {stop, normal, State}.

-spec handle_info(term(), state()) -> {stop, normal, state()}.
handle_info({'DOWN', _, process, _, Reason}, State) ->
    lists:foreach(fun({Ref, OnCompleteFun}) ->
        lager:warning("Fetch ~p failed due to ~p~n", [Ref, Reason]),
        OnCompleteFun(Ref, {error, {other, Reason}})
    end, maps:to_list(State)),
    {stop, normal, #{}}.

-spec handle_cast(term(), state()) -> {noreply, state()}.
handle_cast(_Request, State) ->
    lager:warning("Unknown message: ~p", [_Request]),
    {noreply, State}.

-spec code_change(term(), state(), term()) -> {ok, state()}.
code_change(_, State, _Extra) ->
    {ok, State}.

-spec terminate(term(), state()) -> ok.
terminate(_, _) ->
    ok.
