#pragma once

#include <folly/FBString.h>
#include <folly/Optional.h>
#include <folly/concurrency/ConcurrentHashMap.h>

#include <chrono>

namespace rtransfer {

/**
 * @c SecretManager stores authorization secrets of providers.
 * @c ControlService will store the secrets here when handling
 * `allow_connection` proto message.
 * The expired secrets are discarded on read.
 */
class SecretManager {
public:
    void allowConnection(folly::fbstring mySecret,
        const folly::fbstring& peerSecret, folly::fbstring providerId,
        std::chrono::steady_clock::time_point expirationTime)
    {
        secrets_.insert(peerSecret,
            Value{std::move(providerId), std::move(mySecret), expirationTime});
    }

    folly::Optional<std::pair<folly::fbstring, folly::fbstring>>
    secretToProviderId(const folly::fbstring& secret)
    {
        auto it = secrets_.find(secret);  // NOLINT
        if (it == secrets_.cend())
            return {};

        if (it->second.expirationTime <= std::chrono::steady_clock::now()) {
            secrets_.erase(it);
            return {};
        }

        return {std::make_pair(it->second.providerId, it->second.mySecret)};
    }

private:
    struct Value {
        folly::fbstring providerId;
        folly::fbstring mySecret;
        std::chrono::steady_clock::time_point expirationTime;
    };

    folly::ConcurrentHashMap<folly::fbstring, Value> secrets_;
};

}  // namespace rtransfer
