#pragma once

#include <folly/FBString.h>
#include <folly/Hash.h>
#include <folly/Optional.h>
#include <folly/executors/SerialExecutor.h>
#include <folly/futures/SharedPromise.h>
#include <helpers/storageHelper.h>

#include <list>
#include <unordered_map>

#include "openType.hpp"
#include "periodicHandler.hpp"

namespace rtransfer {

class Storage;

/**
 * @c HandleCache is a wrapper for @c Storage that keeps storage handles (@c
 * one::helpers::FileHandle ) open for reuse.
 * The unused handles are kept for `FLAGS_descriptor_cache_duration` time. The
 * cache will attempt to keep no more than `FLAGS_max_open_descriptors` most
 * recently used unused handles. The expired handles are cleaned every
 * `FLAGS_descriptor_cache_tick` milliseconds via @c PeriodicHandler .
 *
 * @note This cache is very important for performance with some storage helpers.
 */
class HandleCache {
public:
    HandleCache();

    folly::Future<folly::IOBufQueue> read(Storage &storage, std::uint64_t reqId,
        const folly::fbstring &fileId, const folly::fbstring &fileGuid,
        std::uint64_t offset, std::size_t size, std::uint8_t priority);

    folly::Future<folly::Unit> write(Storage &storage, std::uint64_t reqId,
        const folly::fbstring &fileId, const folly::fbstring &fileGuid,
        std::uint64_t offset, folly::IOBufQueue buf, std::uint8_t priority);

    folly::Future<folly::Unit> sync(Storage &storage,
        const folly::fbstring &fileId, const folly::fbstring &fileGuid);

    folly::Future<folly::Unit> invalidateStorage(
        const folly::fbstring &storageId);

private:
    using Key = std::tuple<folly::fbstring /*storageId*/,
        folly::fbstring /*fileGuid*/, OpenType>;

    struct LRUNode {
        Key key;
        std::chrono::steady_clock::time_point addTimePoint;
        // Set to true if the handle is no longer valid (e.g. after helper
        // parameters have changed)
        bool invalid{false};
    };

    struct Node {
        std::shared_ptr<folly::SharedPromise<one::helpers::FileHandlePtr>>
            promise;
        std::list<LRUNode>::iterator lruIt;
        std::size_t refCount = 0;
    };

    template <typename Cb>
    auto withHandle(Storage &storage, const folly::fbstring &fileId,
        const folly::fbstring &fileGuid, OpenType openType, Cb &&cb)
        -> decltype(cb(std::declval<one::helpers::FileHandlePtr>()));

    folly::Future<one::helpers::FileHandlePtr> checkout(Storage &storage,
        const folly::fbstring &fileId, const folly::fbstring &fileGuid,
        OpenType openType);

    void checkin(const folly::fbstring &storageId,
        const folly::fbstring &fileGuid, OpenType openType);

    void trigger();
    void doTrigger();

    folly::Executor::KeepAlive<folly::SerialExecutor> executor_;
    std::unordered_map<Key, Node> handles_;
    std::list<LRUNode> lru_;

    PeriodicHandler periodicHandler_;
};

}  // namespace rtransfer
