#pragma once

#include <folly/FBString.h>
#include <folly/FBVector.h>
#include <folly/Function.h>
#include <folly/SpinLock.h>

#include <unordered_map>
#include <utility>

#include "periodicHandler.hpp"

namespace rtransfer {

/**
 * NotifyAggregator aggregates notifications to reduce their number.
 * Without an explicit flush, the notifications are flushed every second via
 * @c PeriodicHandler .
 */
class NotifyAggregator {
public:
    using Fun =
        folly::Function<void(folly::fbstring, std::uint64_t, std::size_t)>;

    using Notif = std::pair<std::uint64_t, std::size_t>;

    /**
     * @param fun The callback to call to send an aggregated notification.
     */
    explicit NotifyAggregator(Fun fun)
        : notifyCb_{std::move(fun)}
    {
    }

    /**
     * Puts a notification into a buffer that will be later read for
     * aggregation.
     */
    void notify(
        const folly::fbstring &reqId, std::uint64_t offset, std::size_t size)
    {
        folly::SpinLockGuard guard{lock_};
        notifs_[reqId].emplace_back(offset, size);
    }

    /**
     * Aggregates and sends notification in the buffer for a specific request.
     */
    void flush(const folly::fbstring &reqId)
    {
        folly::fbvector<Notif> notifs;
        {
            folly::SpinLockGuard guard{lock_};
            auto it = notifs_.find(reqId);
            if (it == notifs_.end())
                return;
            notifs.swap(it->second);
            notifs_.erase(it);
        }
        doFlush(reqId, std::move(notifs));
    }

private:
    void doFlush(const folly::fbstring &reqId, folly::fbvector<Notif> notifs)
    {
        std::sort(notifs.begin(), notifs.end());
        folly::fbvector<Notif> condensed;

        for (auto &notif : notifs) {
            if (!condensed.empty() &&
                condensed.back().first + condensed.back().second == notif.first)
                condensed.back().second += notif.second;
            else
                condensed.push_back(notif);
        }

        for (auto &notif : condensed)
            notifyCb_(reqId, notif.first, notif.second);
    }

    void flushAll()
    {
        decltype(notifs_) notifsMap;
        {
            folly::SpinLockGuard guard{lock_};
            notifsMap.swap(notifs_);
        }

        for (auto &el : notifsMap)
            doFlush(el.first, std::move(el.second));
    }

    folly::SpinLock lock_;
    Fun notifyCb_;
    std::unordered_map<folly::fbstring, folly::fbvector<Notif>> notifs_;

    // TODO: flag for aggregation period
    // TODO: since notifications are aggregated, if storage read/write would be
    // aggregated too, there would probably be no need for splitting requests on
    // write to sockets and shaper could have a small quantum (and actually use
    // that value)
    PeriodicHandler periodicHandler_{std::chrono::milliseconds{1000},
        std::bind(&NotifyAggregator::flushAll, this)};
};

}  // namespace rtransfer
