"""This module contains test cases for transfer to ceph storage"""

__author__ = "Bartek Kryza"
__copyright__ = """(C) 2023 ACK CYFRONET AGH,
This software is released under the MIT license cited in 'LICENSE.txt'."""

import base64
import os

from test_common import *

def test_slow_null_to_ceph_storage_full_transfer(rtransfer, request, ceph_server, src_dir):
    # Create temporary file with random data
    random_block = os.urandom(1234*1234)
    file_block_count = 250

    file_size = file_block_count * len(random_block)

    with open(os.path.join(src_dir, "file1.dat"), 'wb') as file1:
        for _ in range(0, file_block_count):
            file1.write(random_block)

    # Start rtransfer instances
    (dest, src) = rtransfer

    dest_storage_id = create_ceph_helper("dest_ceph", dest,
                                         ceph_server.mon_host,
                                         ceph_server.username,
                                         ceph_server.key,
                                         ceph_server.pool_name,
                                         ceph_server.block_size)
    src_storage_id = create_slow_null_helper("src_slow_null", src)
    conns = connect_links(dest, src)

    wait_for_connection()

    fetch_offset = 0
    fetch_size = file_size

    srcfd = to_b64("file1.dat")
    destfd = to_b64("file1.dat")

    priority = 1

    do_fetch(dest, src, conns[0], src_storage_id, srcfd, dest_storage_id,
             destfd, fetch_offset, fetch_size, priority)

    while True:
        resp = dest.get_response(skip_updates=False)
        if "isUpdate" in resp and resp["isUpdate"]:
            # Check that blocks are transferred at fixed size
            # matching the destination storage size
            assert int(resp["progress"].get("offset", 0)) % ceph_server.block_size == 0
        else:
            assert fetch_size == int(resp["wrote"])
            break


def test_posix_to_ceph_storage_full_transfer(rtransfer, ceph_server, src_dir):
    # Create temporary file with random data
    random_block = os.urandom(1000*1000)
    file_block_count = 100

    file_size = file_block_count * len(random_block)

    with open(os.path.join(src_dir, "file1.dat"), 'wb') as file1:
        for _ in range(0, file_block_count):
            file1.write(random_block)

    # Start rtransfer instances
    (dest, src) = rtransfer
    dest_storage_id = create_ceph_helper("dest_ceph", dest,
                                         ceph_server.mon_host,
                                         ceph_server.username,
                                         ceph_server.key,
                                         ceph_server.pool_name,
                                         ceph_server.block_size)
    src_storage_id = create_posix_helper("src_posix",
                                         src,
                                         str(src_dir))
    conns = connect_links(dest, src)

    wait_for_connection()

    fetch_offset = 0
    fetch_size = file_size

    srcfd = to_b64("file1.dat")
    destfd = to_b64("file1.dat")

    priority = 1

    do_fetch(dest, src, conns[0], src_storage_id, srcfd, dest_storage_id,
             destfd, fetch_offset, fetch_size, priority)

    while True:
        resp = dest.get_response(skip_updates=False)
        if "isUpdate" in resp and resp["isUpdate"]:
            # Check that blocks are transferred at fixed size
            # matching the destination storage size
            assert int(resp["progress"].get("offset", 0)) % ceph_server.block_size == 0
        else:
            assert fetch_size == int(resp["wrote"])
            break

    # TODO
    #
    # source_md5 = hashlib.md5(open(os.path.join(src_dir, "file1.dat"),'rb').read()).hexdigest()
    #
    # dest_md5 = hashlib.md5(server.read_file_contents("file1.dat", file_size)).hexdigest()
    #
    # assert source_md5  == dest_md5
