-module(proper_SUITE).
-author("Konrad Zemek").

-include_lib("proper/include/proper.hrl").
-include_lib("eunit/include/eunit.hrl").

-behaviour(proper_statem).

-compile([export_all]).

all() -> [proper_test].

fulfill(#{fetches := Fetches, awaiting := Awaiting} = State) ->
    {Fulfilled, Awaiting1} = maps:fold(
                               fun(Ref, From, {F, A}) ->
                                       case maps:get(Ref, Fetches, undefined) of
                                           undefined -> {F, maps:put(Ref, From, A)};
                                           ReqId ->
                                               From ! {fetch_is_on, Ref, ReqId},
                                               {[Ref | F], A}
                                       end
                               end, {[], #{}}, Awaiting),
    Fetches1 = maps:without(Fulfilled, Fetches),
    State#{fetches := Fetches1, awaiting := Awaiting1}.

coordinator() ->
    coordinator(#{fetches => #{}, req_to_ref => #{}, awaiting => #{}}).

coordinator(#{fetches := Fetches, req_to_ref := ReqToRef, awaiting := Awaiting} = State) ->
    State1 =
        receive
            {new_ref, From, Ref, ReqId} ->
                From ! ok,
                State#{req_to_ref := maps:put(ReqId, Ref, ReqToRef)};
            {new_fetch, ReqId} ->
                Ref = maps:get(ReqId, ReqToRef),
                ReqToRef1 = maps:remove(ReqId, ReqToRef),
                Fetches1 = maps:put(Ref, ReqId, Fetches),
                State#{req_to_ref := ReqToRef1, fetches := Fetches1};
            {wait, From, Ref} ->
                State#{awaiting := maps:put(Ref, From, Awaiting)};
            stop ->
                stop
        end,
    case State1 of
        stop -> ok;
        _ -> coordinator(fulfill(State1))
    end.

init_per_testcase(_, Config) ->
    meck:new(rtransfer_link_port, [no_history, unlink]),
    meck:new(rtransfer_link_request, [passthrough, no_history, unlink]),
    meck:expect(rtransfer_link_port, start_link,
                fun(Opts) -> port_mock:start_link(Opts) end),
    meck:expect(rtransfer_link_port, request,
                fun(Req) -> port_mock:request(Req) end),
    meck:expect(rtransfer_link_port, sync_request,
                fun(Req) -> port_mock:sync_request(Req) end),
    meck:expect(rtransfer_link_port, sync_request,
                fun(Req, Timeout) -> port_mock:sync_request(Req, Timeout) end),
    meck:expect(rtransfer_link_request, fetch,
                fun(ConnId, #{ref := Ref, req_id := ReqId} = Req, Opts) ->
                        coordinator ! {new_ref, self(), Ref, ReqId},
                        receive ok -> ok after 1000 -> error(timeout) end,
                        meck:passthrough([ConnId, Req, Opts])
                end),
    Config.

end_per_testcase(_) ->
    meck:unload(rtransfer_link_port),
    meck:unload(rtransfer_link_request).

proper_test(_) ->
    ?assert(proper:quickcheck(?MODULE:aproper_test(), [10000, {to_file, user}])).

start_rtransfer() ->
    register(coordinator, spawn(fun coordinator/0)),
    rtransfer_link_sup:start_link([{port_mod, port_mock},
                                   {get_nodes_fun, fun(_) -> [{{127, 0, 0, 1}, 5555}] end}]).

cleanup_rtransfer(Pid) ->
    monitor(process, Pid),
    exit(Pid, normal),
    Ref = monitor(process, Pid),
    receive
        {'DOWN', Ref, process, Pid, _Reason} -> ok
    after 1000 ->
            error(exit_timeout)
    end,
    Coordinator = whereis(coordinator),
    unregister(coordinator),
    Coordinator ! stop.

aproper_test() ->
    ?FORALL(Cmds, commands(?MODULE),
            ?TRAPEXIT(
               begin
                   {ok, Sup} = start_rtransfer(),
                   {History, State, Result} = run_commands(?MODULE, Cmds),
                   cleanup_rtransfer(Sup),
                   ?WHENFAIL(io:format("History: ~w~nState: ~w\nResult: ~w~n",
                                       [History,State,Result]),
                             aggregate(command_names(Cmds), Result =:= ok))
               end)).

command(#{prepared_reqs := PreparedReqs, running_reqs := RunningReqs,
          storages := Storages} = S) ->
    frequency(
      [{1, {call, rtransfer_link, add_storage, [ascii(), ascii(), list({ascii(), utf8()})]}}] ++
          [{5, {call, rtransfer_link, prepare_request,
            [ascii(), ascii(), ascii(), ascii(), storage(S), ascii(), ascii(),
             non_neg_integer(), pos_integer(), priority()]}} || Storages =/= []] ++
          [{10, {call, ?MODULE, fetch, [prepared_req(S)]}} || PreparedReqs =/= []] ++
          [{20, {call, ?MODULE, fulfill_fetch, [running_req(S), exactly(all)]}} || RunningReqs =/= []] ++
          [{20, {call, ?MODULE, cancel, [running_req(S)]}} || RunningReqs =/= []]).

initial_state() ->
    #{prepared_reqs => [], running_reqs => [], storages => []}.

next_state(S, _V, {call, _, add_storage, [StorageId, _, _]}) ->
    maps:update_with(storages, fun(Storages) -> [StorageId | Storages] end, S);
next_state(S, V, {call, _, prepare_request, [_, _, _, _, _, _, _, _O, Size, _Prio]}) ->
    maps:update_with(prepared_reqs, fun(PR) -> [V | PR] end, S);
next_state(S, V, {call, _, fetch, [Req]}) ->
    S1 = maps:update_with(prepared_reqs, fun(PR) -> lists:delete(Req, PR) end, S),
    maps:update_with(running_reqs, fun(RR) -> [V | RR] end, S1);
next_state(S, _V, {call, _, fulfill_fetch, [Req, _]}) ->
    maps:update_with(running_reqs, fun(RR) -> lists:delete(Req, RR) end, S);
next_state(S, _V, {call, _, cancel, [Req]}) ->
    maps:update_with(running_reqs, fun(RR) -> lists:delete(Req, RR) end, S).

precondition(_, _) -> true.

postcondition(_S, {call, _, add_storage, [_, _, _]}, Result) ->
    Result =:= ok;
postcondition(_S, {call, _, prepare_request, [_, _, _, _, DestStorageId | _]}, Result) ->
    is_map(Result) andalso DestStorageId =:= maps:get(dest_storage_id, Result, undefined);
postcondition(_S, {call, _, fetch, [_]}, Result) ->
    case Result of
        {Ref, _ReqId} when is_reference(Ref) -> true;
        _ -> false
    end;
postcondition(_S, {call, _, fulfill_fetch, [_, _]}, Result) ->
    case Result of
        {ok, _} -> true;
        _ -> false
    end;
postcondition(_S, {call, _, cancel, [_]}, Result) ->
    case Result of
        {error, {_, <<"canceled">>}} -> true;
        {error, <<"canceled">>} -> true;
        _ -> false
    end.

fetch(Req) ->
    Self = self(),
    NotifyFun = fun(Ref, Offset, Size) -> Self ! {Ref, Offset, Size} end,
    OnCompleteFun = fun(Ref, Result) -> Self ! {Ref, Result} end,
    Ref = rtransfer_link:fetch(Req, NotifyFun, OnCompleteFun),
    coordinator ! {wait, self(), Ref},
    receive {fetch_is_on, Ref, ReqId} -> {Ref, ReqId}
    after 1000 -> {error, timeout} end.

fulfill_fetch({Ref, ReqId}, all) ->
    ok = port_mock:fulfill_fetch(ReqId, all),
    receive {Ref, Result} -> Result
    after 1000 -> {error, timeout} end.

cancel({Ref, _ReqId}) ->
    ok = rtransfer_link:cancel(Ref),
    receive {Ref, Result} -> Result
    after 1000 -> {error, timeout} end.

%% Types

priority() ->
    elements([low_priority, medium_priority, high_priority, very_high_priority]).

prepared_req(#{prepared_reqs := Reqs}) ->
    elements(Reqs).

running_req(#{running_reqs := Reqs}) ->
    elements(Reqs).

storage(#{storages := Storages}) ->
    elements(Storages).

ascii() ->
    ?LET(L, non_empty(list(integer(32, 126))), list_to_binary(L)).
