%%%-------------------------------------------------------------------
%%% @author Konrad Zemek
%%% @copyright (C) 2018 ACK CYFRONET AGH
%%% This software is released under the MIT license
%%% cited in 'LICENSE.txt'.
%%% @end
%%%-------------------------------------------------------------------
%%% @doc
%%% @end
%%%-------------------------------------------------------------------
-module(rtransfer_link_questions).
-author("Konrad Zemek").

%%%===================================================================
%%% Exports
%%%===================================================================

-export([handle/1]).

%%%===================================================================
%%% API
%%%===================================================================

handle(#{<<"authRequest">> := AuthReq} = Response) ->
    spawn_link(
      fun() ->
              ReqId = maps:get(<<"reqId">>, Response),
              TransferData = base64:decode(maps:get(<<"transferData">>, AuthReq)),
              ProviderId = base64:decode(maps:get(<<"providerId">>, AuthReq)),
              AuthResp =
                  case auth_request(TransferData, ProviderId) of
                      false -> #{is_authorized => false};
                      {StorageId, FileId, FileGuid} ->
                          #{is_authorized => true,
                            storage_id => base64:encode(StorageId),
                            file_id => base64:encode(FileId),
                            file_guid => base64:encode(FileGuid)}
                  end,
              Answer = #{req_id => ReqId, auth_response => AuthResp},
              rtransfer_link_port:answer(Answer)
      end).

auth_request(TransferData, ProviderId) ->
    {MinBf, MaxBf} = application:get_env(rtransfer_link, auth_request_backoff, {1000, 10000}),
    Backoff = backoff:init(MinBf, MaxBf),
    Retries = application:get_env(rtransfer_link, auth_request_retries, 5),
    auth_request(Backoff, Retries, TransferData, ProviderId).

auth_request(_Backoff, 0, TransferData, ProviderId) ->
    lager:warning("All auth requests with failed, args: ~p ~p", [TransferData, ProviderId]),
    false;

auth_request(Backoff, Retries, TransferData, ProviderId) ->
    try rtransfer_link_callback:auth_request(TransferData, ProviderId) of
        {<<_/binary>>, <<_/binary>>, <<_/binary>>} = Response -> Response;
        _ -> false
    catch
        Class:Error:Stacktrace ->
            NextTryIn = backoff:get(Backoff),
            {_, NewBackoff} = backoff:fail(Backoff),
            lager:warning("Validating auth request failed with ~p:~p. Will retry in ~p",
                          [Class, Error, NextTryIn]),
            lager:debug("Args: ~p ~p~nStacktrace: ~p", [TransferData, ProviderId, Stacktrace]),
            timer:sleep(NextTryIn),
            auth_request(NewBackoff, Retries - 1, TransferData, ProviderId)
    end.
