#pragma once

#include <folly/FBString.h>
#include <folly/SpinLock.h>
#include <folly/String.h>
#include <folly/concurrency/ConcurrentHashMap.h>
#include <folly/futures/Future.h>
#include <monitoring/monitoring.h>
#include <wangle/channel/Handler.h>

#include <cstdint>
#include <unordered_map>

#include "common.hpp"
#include "handleCache.hpp"
#include "storage.hpp"

namespace rtransfer {

namespace client {
class Client;
}  // namespace client

/**
 * A @c FetchManager handles incoming data on data pipelines.
 * A single instance hooks itself as a handler for all data pipelines in a
 * single "link". Every fetch request going through the control pipeline is
 * recorded in the fetch manager, and all incoming data is correlated with its
 * fetch info and passed on to a proper storage.
 */
class FetchManager
    : public wangle::InboundHandler<std::unique_ptr<folly::IOBuf>,
          folly::Unit> {
public:
    /**
     * @param client A reference to @c Client instance used to send ACKs
     * through.
     * @see Client::ack
     */
    FetchManager(client::Client &client, StoragesMap &storages,
        HandleCache &handleCache);

    /**
     * Notifies `this` of a new fetch so that incoming data can be correlated
     * with fetch details.
     * @param notifyCb Callback to be called when a full block has been
     * persisted on the destination storage.
     * @returns a future that will be fulfilled once the fetch is completed.
     */
    folly::Future<std::size_t> newFetch(std::uint64_t requestId,
        const folly::fbstring &destStorageId, const folly::fbstring &destFileId,
        const folly::fbstring &fileGuid, std::uint64_t offset, std::size_t size,
        folly::Function<void(std::uint64_t, std::size_t)> notifyCb);

    void read(Context *ctx, std::unique_ptr<folly::IOBuf> buf) override;

    /**
     * Cancel a fetch.
     */
    void cancelFetch(std::uint64_t requestId);

    /**
     * Cancel a fetch with a given exception.
     */
    void cancelFetch(
        std::uint64_t requestId, folly::exception_wrapper exception);

    /**
     * Sets a size for a fetch.
     * The initial size given in `newFetch` might not always be the actual size
     * available to download from the remote. In case the actual size is
     * smaller, the remote will notify the parent @c Client instance by a
     * message on the control chanel. The client will then call `setTotalSize`
     * on the fetch manager to let it know not to wait on data that's not
     * coming.
     */
    void setTotalSize(std::uint64_t reqId, std::size_t size);

private:
    struct PartsInfo {
        std::uint8_t totalParts{255};
        folly::fbvector<std::pair<std::uint8_t, std::unique_ptr<folly::IOBuf>>>
            bufs;
    };

    struct FetchInfo {
        mutable folly::SpinLock lock;
        folly::fbstring storageId;
        folly::fbstring fileId;
        folly::fbstring fileGuid;
        std::unordered_map<std::uint64_t, PartsInfo> partsInfo;
        std::size_t size{0};
        std::size_t wrote{0};
        folly::Optional<folly::Promise<std::size_t>> promise;
        folly::Function<void(std::uint64_t, std::size_t)> notifyCb;
    };

    struct Header {
        explicit Header(std::unique_ptr<folly::IOBuf> & /*buf*/);
        std::uint64_t reqId{0};
        bool isLastPart{false};
        std::uint8_t part{0};
        std::uint64_t offset{0};
    };

    using Map =
        folly::ConcurrentHashMap<std::uint64_t, std::shared_ptr<FetchInfo>>;

    void doRead(std::unique_ptr<folly::IOBuf> buf);

    template <typename Ref>
    void cancelFetchStr(Ref ref, folly::StringPiece reason);

    void cancelFetch(
        Map::const_iterator it, folly::exception_wrapper exception);

    void maybeFulfillPromise(Map::const_iterator it);

    folly::Optional<folly::IOBufQueue> tallyParts(
        FetchInfo &req, Header &header, std::unique_ptr<folly::IOBuf> buf);

    client::Client &client_;
    StoragesMap &storages_;
    HandleCache &handleCache_;

    // Map of request ids and FetchInfo requests
    Map fetchInfo_;
};

}  // namespace rtransfer
