#pragma once

#include <folly/executors/SerialExecutor.h>
#include <folly/futures/Future.h>
#include <folly/futures/SharedPromise.h>

#include <queue>
#include <unordered_map>

#include "controlService.hpp"
#include "periodicHandler.hpp"

namespace rtransfer {

/**
 * AuthCache is a proxy for request authorization requests.
 * Responses to the requests are cached for FLAGS_auth_cache_duration
 * milliseconds and the expired entries are pruned every FLAGS_auth_cache_ticks
 * milliseconds.
 */
class AuthCache {
public:
    /**
     * Constructor.
     * @param controlService A service used to request authorizations from the
     * local provider.
     */
    explicit AuthCache(link_control::ControlService& controlService);

    /**
     * Authorize a transfer request from a remote provider.
     * @param providerId ID of the peer provider that sent us the request.
     * @param transferData opaque data describing the transfer request.
     * @returns A message containing details of the accessed file, most
     * importantly a storage id and file id.
     */
    folly::Future<std::shared_ptr<link_control::proto::Request>> authorize(
        folly::fbstring providerId, folly::fbstring transferData);

private:
    using Key = std::pair<folly::fbstring, folly::fbstring>;
    using Promise =
        folly::SharedPromise<std::shared_ptr<link_control::proto::Request>>;

    struct ExpNode {
        Key key;
        std::chrono::steady_clock::time_point addTimePoint;
    };

    folly::Future<std::shared_ptr<link_control::proto::Request>> doAuthorize(
        const folly::fbstring& providerId, const folly::fbstring& transferData);

    std::shared_ptr<Promise> authorizeWithServer(Key key);

    void trigger();
    void doTrigger();

    folly::Executor::KeepAlive<folly::SerialExecutor> executor_;
    std::unordered_map<Key, std::shared_ptr<Promise>> responses_;
    std::queue<ExpNode> expirations_;

    link_control::ControlService& controlService_;
    PeriodicHandler periodicHandler_;
};

}  // namespace rtransfer
