%%%-------------------------------------------------------------------
%%% @author Piotr Duleba
%%% @copyright (C) 2020 ACK CYFRONET AGH
%%% This software is released under the MIT license
%%% cited in 'LICENSE.txt'.
%%% @end
%%%-------------------------------------------------------------------
%%% @doc
%%% This file provides functions for making rpc calls to Onepanel Nodes for use during tests.
%%% @end
%%%-------------------------------------------------------------------
-module(panel_test_rpc).
-author("Piotr Duleba").

-include_lib("ctool/include/test/test_utils.hrl").

%% API
-export([
    call/2, call/3,
    insecure_call/2, insecure_call/3,

    call/4, call/5,
    insecure_call/4, insecure_call/5,

    get_service_hosts/2,

    get_op_worker_nodes/1,
    get_cert_chain_ders/1,
    get_cluster_id/1,
    create_invite_token/1,
    is_emergency_passphrase_set/1,
    set_emergency_passphrase/2,
    unset_emergency_passphrase/1,
    verify_emergency_passphrase/2,
    list_onepanel_deployment/1,
    is_onepanel_initialized/1,

    add_storage/2,
    support_space/4
]).

-type panel_service() ::
    cluster_manager |
    op_worker |
    oz_worker.

-export_type([panel_service/0]).

-define(CALL(NodeSelector, Args), test_rpc:call(onepanel, NodeSelector, test_rpc_api, ?FUNCTION_NAME, Args)).


%%%===================================================================
%%% API functions
%%%===================================================================


-spec call(oct_background:node_selector(), test_rpc:routine()) ->
    term() | no_return().
call(NodeSelector, Routine) ->
    call(NodeSelector, Routine, infinity).


-spec call(oct_background:node_selector(), test_rpc:routine(), timeout()) ->
    term() | no_return().
call(NodeSelector, Routine, Timeout) ->
    test_rpc:call(onepanel, NodeSelector, Routine, Timeout).


-spec insecure_call(oct_background:node_selector(), test_rpc:routine()) ->
    term() | no_return().
insecure_call(NodeSelector, Routine) ->
    insecure_call(NodeSelector, Routine, infinity).


-spec insecure_call(oct_background:node_selector(), test_rpc:routine(), timeout()) ->
    term() | no_return().
insecure_call(NodeSelector, Routine, Timeout) ->
    test_rpc:insecure_call(onepanel, NodeSelector, Routine, Timeout).


-spec call(oct_background:node_selector(), module(), atom(), [term()]) ->
    term() | no_return().
call(NodeSelector, Module, FunctionName, Args) ->
    call(NodeSelector, Module, FunctionName, Args, infinity).


-spec call(oct_background:node_selector(), module(), atom(), [term()], timeout()) ->
    term() | no_return().
call(NodeSelector, Module, FunctionName, Args, Timeout) ->
    test_rpc:call(onepanel, NodeSelector, Module, FunctionName, Args, Timeout).


-spec insecure_call(oct_background:node_selector(), module(), atom(), [term()]) ->
    term() | no_return().
insecure_call(NodeSelector, Module, FunctionName, Args) ->
    insecure_call(NodeSelector, Module, FunctionName, Args, infinity).


-spec insecure_call(oct_background:node_selector(), module(), atom(), [term()], timeout()) ->
    term() | no_return().
insecure_call(NodeSelector, Module, FunctionName, Args, Timeout) ->
    test_rpc:insecure_call(onepanel, NodeSelector, Module, FunctionName, Args, Timeout).


-spec get_service_hosts(oct_background:node_selector(), panel_service()) -> [string()].
get_service_hosts(NodeSelector, PanelService) ->
    ?CALL(NodeSelector, [PanelService]).


-spec get_op_worker_nodes(oct_background:node_selector()) -> [node()].
get_op_worker_nodes(NodeSelector) ->
    ?CALL(NodeSelector, []).


-spec get_cert_chain_ders(oct_background:node_selector()) -> [public_key:der_encoded()].
get_cert_chain_ders(NodeSelector) ->
    ?CALL(NodeSelector, []).


-spec get_cluster_id(oct_background:node_selector()) -> binary().
get_cluster_id(NodeSelector) ->
    ?CALL(NodeSelector, []).


-spec create_invite_token(oct_background:node_selector()) -> binary().
create_invite_token(NodeSelector) ->
    {ok, Token} = ?assertMatch({ok, _}, ?CALL(NodeSelector, [])),
    Token.


-spec is_emergency_passphrase_set(oct_background:node_selector()) -> boolean().
is_emergency_passphrase_set(NodeSelector) ->
    ?CALL(NodeSelector, []).


-spec set_emergency_passphrase(oct_background:node_selector(), binary()) ->
    ok | od_error:error().
set_emergency_passphrase(Nodeselector, NewPassphrase) ->
    ?CALL(Nodeselector, [NewPassphrase]).


-spec unset_emergency_passphrase(oct_background:node_selector()) -> ok | no_return().
unset_emergency_passphrase(Nodeselector) ->
    ?CALL(Nodeselector, []).


-spec verify_emergency_passphrase(oct_background:node_selector(), binary()) -> boolean().
verify_emergency_passphrase(NodeSelector, Passphrase) ->
    ?CALL(NodeSelector, [Passphrase]).


-spec list_onepanel_deployment(oct_background:node_selector()) -> list().
list_onepanel_deployment(NodeSelector) ->
    ?CALL(NodeSelector, []).


-spec is_onepanel_initialized(oct_background:node_selector()) -> boolean().
is_onepanel_initialized(NodeSelector) ->
    ?CALL(NodeSelector, []).


-spec add_storage(oct_background:node_selector(), map()) -> binary().
add_storage(NodeSelector, Data) ->
    {ok, StorageId} = ?assertMatch({ok, _}, ?CALL(NodeSelector, [Data])),
    StorageId.


-spec support_space(oct_background:node_selector(), binary(), binary(), binary()) -> binary().
support_space(NodeSelector, StorageId, SerializedToken, SupportSize) ->
    ?CALL(NodeSelector, [StorageId, SerializedToken, SupportSize]).
