%%%-------------------------------------------------------------------
%%% @author Piotr Duleba
%%% @copyright (C) 2021 ACK CYFRONET AGH
%%% This software is released under the MIT license
%%% cited in 'LICENSE.txt'.
%%% @end
%%%-------------------------------------------------------------------
%%% @doc
%%% This file provides functions for rpc test modules.
%%% @end
%%%-------------------------------------------------------------------
-module(test_rpc).
-author("Piotr Duleba").

-export([
    call/3, call/4,
    insecure_call/4,

    call/5, call/6,
    insecure_call/6
]).

-type service() :: op_worker | oz_worker | onepanel.
-type routine() :: fun(() -> term()).

-export_type([service/0, routine/0]).


%%%===================================================================
%%% API functions
%%%===================================================================


-spec call(service(), oct_background:node_selector(), routine()) ->
    term() | no_return().
call(Service, NodeSelector, Routine) ->
    call(Service, NodeSelector, Routine, infinity).


-spec call(service(), oct_background:node_selector(), routine(), timeout()) ->
    term() | no_return().
call(Service, NodeSelector, Routine, Timeout) ->
    Node = select_node(Service, NodeSelector),

    try
        insecure_call(Service, Node, Routine, Timeout)
    catch Type:Reason:Stacktrace ->
        ct:pal(
            "Test RPC on node ~tp failed!~n"
            "Stacktrace: ~ts~n",
            [Node, iolist_to_binary(lager:pr_stacktrace(Stacktrace, {Type, Reason}))]
        ),
        error({test_rpc_failed, Node, Reason})
    end.


-spec insecure_call(service(), oct_background:node_selector(), routine(), timeout()) ->
    term() | no_return().
insecure_call(Service, NodeSelector, Routine, Timeout) ->
    erpc:call(select_node(Service, NodeSelector), Routine, Timeout).


-spec call(service(), oct_background:node_selector(), module(), atom(), [term()]) ->
    term() | no_return().
call(Service, NodeSelector, Module, FunctionName, Args) ->
    call(Service, NodeSelector, Module, FunctionName, Args, infinity).


-spec call(service(), oct_background:node_selector(), module(), atom(), [term()], timeout()) ->
    term() | no_return().
call(Service, NodeSelector, Module, FunctionName, Args, Timeout) ->
    Node = select_node(Service, NodeSelector),

    try
        insecure_call(Service, Node, Module, FunctionName, Args, Timeout)
    catch Type:Reason ->
        ct:pal(
            "Test RPC on node ~tp failed!~n"
            "MFA: ~tp:~tp(~tp)~n"
            "Reason: ~tp",
            [Node, Module, FunctionName, Args, Reason]
        ),
        error({test_rpc_failed, Node, Module, FunctionName, Type, Reason})
    end.


-spec insecure_call(service(), oct_background:node_selector(), module(), atom(), [term()], timeout()) ->
    term() | no_return().
insecure_call(Service, NodeSelector, Module, FunctionName, Args, Timeout) ->
    erpc:call(select_node(Service, NodeSelector), Module, FunctionName, Args, Timeout).


%%%===================================================================
%%% Helpers
%%%===================================================================


%% @private
-spec select_node(service(), oct_background:node_selector()) -> node().
select_node(Service, NodeSelector) ->
    case is_known_node(NodeSelector) of
        true ->
            NodeSelector;
        false ->
            lists_utils:random_element(get_service_nodes(Service, NodeSelector))
    end.


%% @private
-spec is_known_node(oct_background:node_selector()) -> boolean().
is_known_node(NodeSelector) ->
    lists:member(NodeSelector, nodes(known)).


%% @private
-spec get_service_nodes(service(), oct_background:node_selector()) -> [node()].
get_service_nodes(Service, NodeSelector) ->
    case Service of
        op_worker -> oct_background:get_provider_nodes(NodeSelector);
        oz_worker -> oct_background:get_zone_nodes();
        onepanel -> oct_background:get_service_panels(NodeSelector)
    end.
