%%%-------------------------------------------------------------------
%%% @author Wojciech Geisler
%%% @copyright (C) 2018 ACK CYFRONET AGH
%%% This software is released under the MIT license cited in 'LICENSE.txt'.
%%% @end
%%%-------------------------------------------------------------------
%%% @doc
%%% This module handles parsing and detecing IP address.
%%% @end
%%%-------------------------------------------------------------------
-module(onepanel_ip).
-author("Wojciech Geisler").

-include("names.hrl").
-include_lib("ctool/include/logging.hrl").

-export([determine_ip/1, ip4_to_binary/1, is_ip/1, hostname_ips/0]).

-type service_with_ip() :: ?SERVICE_OZW | ?SERVICE_OPW | ?SERVICE_ONES3.


%%%===================================================================
%%% API
%%%===================================================================

%%--------------------------------------------------------------------
%% @doc Attempts to determine IP of current node.
%% @end
%%--------------------------------------------------------------------
-spec determine_ip(service_with_ip()) -> inet:ip4_address().
determine_ip(ServiceName) ->
    % use first working method of getting IP
    lists_utils:foldl_while(fun(IpSupplier, PrevResult) ->
        try
            {ok, IP} = IpSupplier(ServiceName),
            {halt, IP}
        catch _:_ ->
            {cont, PrevResult}
        end
    end, undefined, [
        fun determine_ip_by_oz/1,
        fun determine_ip_by_domain/1,
        fun determine_ip_by_external_service/1,
        fun determine_ip_by_shell/1,
        fun(_) -> {ok, {127, 0, 0, 1}} end
    ]).


%%--------------------------------------------------------------------
%% @doc Converts IP tuple to binary.
%% @end
%%--------------------------------------------------------------------
-spec ip4_to_binary(ip_utils:ip()) -> binary().
ip4_to_binary(IPTuple) ->
    {ok, Binary} = ip_utils:to_binary(IPTuple),
    Binary.


%%--------------------------------------------------------------------
%% @doc Detects IP address in a string, binary, or tuple form.
%% @end
%%--------------------------------------------------------------------
-spec is_ip(term()) -> boolean().
is_ip(Value) ->
    try ip_utils:to_ip4_address(Value) of
        {ok, _} -> true;
        _ -> false
    catch _:_ -> false end.


%%--------------------------------------------------------------------
%% @doc
%% Returns list of IPv4 addresses returned by shell command 'hostname i'.
%% @end
%%--------------------------------------------------------------------
-spec hostname_ips() -> [inet:ip4_address()].
hostname_ips() ->
    {_, Result, _} = shell_utils:execute(["hostname", "-i"]),
    Words = string:split(Result, " ", all),
    % filter out IPv6 addresses
    lists:filtermap(fun(Word) ->
        case ip_utils:to_ip4_address(Word) of
            {ok, IP} -> {true, IP};
            _ -> false
        end
    end, Words).


%%%===================================================================
%%% Internal functions
%%%===================================================================

%%--------------------------------------------------------------------
%% @private
%% @doc
%% Attempts to use Onezone endpoint to determine the public IP of this node.
%% Will work only in onepanel of registered oneprovider.
%% @end
%%--------------------------------------------------------------------
-spec determine_ip_by_oz(service_with_ip()) ->
    {ok, inet:ip4_address()} | {error, no_address | not_registered}.
determine_ip_by_oz(?SERVICE_OZW) ->
    {error, no_address};
determine_ip_by_oz(ServiceName) when
    ServiceName =:= ?SERVICE_OPW;
    ServiceName =:= ?SERVICE_ONES3
->
    case service_oneprovider:is_registered() of
        true ->
            {ok, IPBin} = oz_providers:check_ip_address(none),
            ip_utils:to_ip4_address(IPBin);
        _ ->
            {error, not_registered}
    end.


%%--------------------------------------------------------------------
%% @private
%% @doc
%% Attempts to resolve cluster's domain.
%% If the query returns single IP address, it is used as the determined IP.
%% If more than one value is returned, attempts to match it to
%% hostname -i in order to differentiate between nodes.
%% If none match, the first IP returned by DNS is used.
%% @end
%%--------------------------------------------------------------------
-spec determine_ip_by_domain(service_with_ip()) ->
    {ok, inet:ip4_address()} | {error, no_address}.
determine_ip_by_domain(ServiceName) ->
    Domain = case ServiceName of
        ?SERVICE_OZW -> service_oz_worker:get_domain();
        ?SERVICE_OPW -> service_op_worker:get_domain();
        ?SERVICE_ONES3 -> service_ones3:get_domain()
    end,
    case inet_res:lookup(binary_to_list(Domain), in, a) of
        [] ->
            {error, no_address};
        [{_, _, _, _} = IP] ->
            {ok, IP};
        ManyIPs ->
            HostnameIPs = hostname_ips(),
            case lists_utils:intersect(HostnameIPs, ManyIPs) of
                [] -> {ok, hd(ManyIPs)};
                [IP | _] -> {ok, IP}
            end
    end.


%%--------------------------------------------------------------------
%% @private
%% @doc
%% Attempts to use external service to determine the public IP of current node.
%% @end
%%--------------------------------------------------------------------
-spec determine_ip_by_external_service(service_with_ip()) ->
    {ok, inet:ip4_address()} | {error, term()}.
determine_ip_by_external_service(_ServiceName) ->
    URL = onepanel_env:get(ip_check_url),
    case http_client:get(URL) of
        {ok, _, _, Body} ->
            Trimmed = onepanel_utils:trim(Body, both),
            ip_utils:to_ip4_address(Trimmed);
        Error ->
            Error
    end.


%%--------------------------------------------------------------------
%% @private
%% @doc
%% Uses shell command "hostname" to determine current IP.
%% @end
%%--------------------------------------------------------------------
-spec determine_ip_by_shell(service_with_ip()) ->
    {ok, inet:ip4_address()} | {error, no_address}.
determine_ip_by_shell(_ServiceName) ->
    case hostname_ips() of
        [Head | _] -> {ok, Head};
        [] -> {error, no_address}
    end.
