%%%-------------------------------------------------------------------
%%% @author Lukasz Opiola
%%% @copyright (C) 2019 ACK CYFRONET AGH
%%% This software is released under the MIT license
%%% cited in 'LICENSE.txt'.
%%% @end
%%%-------------------------------------------------------------------
%%% @doc
%%% This module encapsulates logic related to users' linked accounts.
%%% @end
%%%-------------------------------------------------------------------
-module(linked_accounts).
-author("Lukasz Opiola").

-include("auth/entitlement_mapping.hrl").
-include("datastore/oz_datastore_models.hrl").
-include_lib("ctool/include/errors.hrl").
-include_lib("ctool/include/logging.hrl").

%% API
-export([to_map/2, to_maps/2]).
-export([gen_user_id/1, gen_user_id/2]).
-export([find_user/1, acquire_user/2]).
-export([merge/2]).
-export([build_test_user_info/1]).

%%%===================================================================
%%% API functions
%%%===================================================================

%%--------------------------------------------------------------------
%% @doc
%% Converts a linked account into a serializable map. Scope can be one of:
%%  * all_fields - returns all fields of the linked account; intended for
%%                 the owner user as it includes private data
%%  * luma_payload - returns the fields used for user mapping in LUMA, stripped
%%                   of some unnecessary / private data.
%% @end
%%--------------------------------------------------------------------
-spec to_map(od_user:linked_account(), Scope :: all_fields | luma_payload) -> map().
to_map(LinkedAccount, Scope) ->
    #linked_account{
        idp = IdP,
        subject_id = SubjectId,
        full_name = FullName,
        username = Username,
        emails = Emails,
        entitlements = Entitlements,
        custom = Custom
    } = LinkedAccount,

    %% @TODO VFS-4506 fullName and entitlements are no longer
    % sent to LUMA as they are ambiguous and inconclusive for user mapping
    {FullNameValue, EntitlementsValue} = case Scope of
        all_fields -> {FullName, Entitlements};
        luma_payload -> {undefined, []}
    end,

    #{
        <<"idp">> => IdP,
        <<"subjectId">> => SubjectId,
        <<"fullName">> => utils:undefined_to_null(FullNameValue),
        <<"username">> => utils:undefined_to_null(Username),
        <<"emails">> => Emails,
        <<"entitlements">> => EntitlementsValue,
        <<"custom">> => Custom,

        %% @TODO VFS-4506 deprecated, included for backward compatibility
        <<"name">> => utils:undefined_to_null(FullNameValue),
        <<"login">> => utils:undefined_to_null(Username),
        <<"alias">> => utils:undefined_to_null(Username),
        <<"emailList">> => Emails,
        <<"groups">> => EntitlementsValue
    }.


%%--------------------------------------------------------------------
%% @doc
%% Converts a list of linked_account records into a serializable list of maps.
%% @end
%%--------------------------------------------------------------------
-spec to_maps([od_user:linked_account()], Scope :: all_fields | luma_payload) -> [map()].
to_maps(LinkedAccounts, Scope) ->
    [to_map(L, Scope) || L <- LinkedAccounts].


%%--------------------------------------------------------------------
%% @doc
%% @equiv gen_user_id(IdP, SubjectId)
%% @end
%%--------------------------------------------------------------------
-spec gen_user_id(od_user:linked_account()) -> od_user:id().
gen_user_id(#linked_account{idp = IdP, subject_id = SubjectId}) ->
    gen_user_id(IdP, SubjectId).


%%--------------------------------------------------------------------
%% @doc
%% Constructs user id based on IdP name and user's subjectId in that IdP.
%% Onezone versions pre 19.02.1 used legacy key mapping - checks if such user
%% is present and if so, reuses the legacy id to retain the user mapping after
%% upgrade. Otherwise, returns an id constructed using the new procedure.
%% @end
%%--------------------------------------------------------------------
-spec gen_user_id(auth_config:idp(), SubjectId :: binary()) -> od_user:id().
gen_user_id(IdP, SubjectId) ->
    LegacyUserId = datastore_key:build_adjacent(<<"">>, str_utils:format_bin("~ts:~ts", [IdP, SubjectId])),
    case user_logic:exists(LegacyUserId) of
        true -> LegacyUserId;
        false -> datastore_key:new_from_digest([atom_to_binary(IdP, utf8), SubjectId])
    end.


%%--------------------------------------------------------------------
%% @doc
%% Returns the user doc linked to given account or {error, not_found}.
%% @end
%%--------------------------------------------------------------------
-spec find_user(od_user:linked_account()) ->
    {ok, od_user:doc()} | {error, not_found}.
find_user(LinkedAccount) ->
    od_user:get_by_linked_account(LinkedAccount).


%%--------------------------------------------------------------------
%% @doc
%% Retrieves a user by given linked account and merges the carried information.
%% If such user does not exist, creates a new user based on that linked account.
%% Checks if the user is blocked and returns an error if so.
%% @end
%%--------------------------------------------------------------------
-spec acquire_user(od_user:linked_account(), idp_auth:flow_type()) ->
    {ok, od_user:doc()} | errors:error().
acquire_user(LinkedAccount, FlowType) ->
    case find_user(LinkedAccount) of
        {ok, #document{value = #od_user{blocked = true}}} ->
            ?ERR_USER_BLOCKED(?err_ctx());
        {ok, #document{key = UserId}} ->
            merge(UserId, LinkedAccount);
        {error, not_found} ->
            create_user(LinkedAccount, FlowType)
    end.


%%--------------------------------------------------------------------
%% @doc
%% Adds a linked account to user's account or replaces the old one (if
%% present). Gathers emails into user's account in the process. Blocks until
%% user's effective relations have been fully synchronized.
%% @end
%%--------------------------------------------------------------------
-spec merge(od_user:id(), od_user:linked_account()) -> {ok, od_user:doc()} | errors:error().
merge(UserId, LinkedAccount) ->
    % The update cannot be done in one transaction, because linked account
    % merging causes adding/removing the user from groups, which modifies user
    % doc and would cause a deadlock. Instead, use a critical section to make
    % sure that merging accounts is sequential.
    Result = critical_section:run({merge_acc, UserId}, fun() ->
        ?catch_exceptions(merge_unsafe(UserId, LinkedAccount))
    end),
    case Result of
        {ok, Doc} ->
            entity_graph:ensure_up_to_date(),
            {ok, Doc};
        {error, not_found} ->
            ?ERROR_NOT_FOUND;
        ?ERR = Error ->
            Error
    end.


%%--------------------------------------------------------------------
%% @doc
%% Build a JSON compatible user info based on a linked account for test page
%% purposes. The info expresses what user data would be gathered during an
%% analogous production login process.
%% @end
%%--------------------------------------------------------------------
-spec build_test_user_info(od_user:linked_account()) ->
    {od_user:id(), json_utils:json_term()}.
build_test_user_info(LinkedAccount) ->
    #linked_account{
        idp = IdP,
        full_name = FullName,
        username = Username,
        emails = Emails,
        entitlements = Entitlements
    } = LinkedAccount,
    MappedEntitlements = entitlement_mapping:map_entitlements(IdP, Entitlements),
    UserId = gen_user_id(LinkedAccount),
    {UserId, #{
        <<"userId">> => UserId,
        <<"fullName">> => user_logic:normalize_full_name(FullName),
        <<"username">> => user_logic:normalize_username(Username),
        <<"emails">> => normalize_emails(Emails),
        <<"linkedAccounts">> => [to_map(LinkedAccount, all_fields)],
        <<"groups">> => maps:from_list(
            lists:map(fun({GroupId, #idp_entitlement{privileges = Privileges}}) ->
                {GroupId, Privileges}
            end, MappedEntitlements)
        )
    }}.

%%%===================================================================
%%% Internal functions
%%%===================================================================

%%--------------------------------------------------------------------
%% @private
%% @doc
%% Creates a new user based on given linked account. Before creating such user,
%% it must be ensured that a user with such linked account does not exist.
%% @end
%%--------------------------------------------------------------------
-spec create_user(od_user:linked_account(), idp_auth:flow_type()) ->
    {ok, od_user:doc()} | errors:error().
create_user(LinkedAccount = #linked_account{full_name = FullName, username = Username}, FlowType) ->
    ProposedUserId = gen_user_id(LinkedAccount),
    {ok, UserId} = user_logic:create(?ROOT, ProposedUserId, #{
        <<"fullName">> => user_logic:normalize_full_name(FullName)
    }),
    ?notice(
        "New user account has been created:~n"
        "> reason:    ~ts~n"
        "> userId:    ~ts~n"
        "> fullName:  ~ts~n"
        "> username:  ~ts~n"
        "> IdP:       ~ts~n"
        "> subjectId: ~ts", [
            case FlowType of
                gui_login -> "first login via GUI";
                access_token -> "first authentication using a delegated IdP access token"
            end,
            UserId,
            FullName,
            Username,
            LinkedAccount#linked_account.idp,
            LinkedAccount#linked_account.subject_id
        ]
    ),
    % Setting the username might fail (if it's not unique) - it's not considered a failure.
    user_logic:update_username(?ROOT, UserId, user_logic:normalize_username(Username)),
    merge(UserId, LinkedAccount).


%%--------------------------------------------------------------------
%% @private
%% @doc
%% Adds a linked account to user's account or replaces the old one (if
%% present). Gathers emails and entitlements into user's account in the process.
%% This code must not be run in parallel.
%% @end
%%--------------------------------------------------------------------
-spec merge_unsafe(od_user:id(), od_user:linked_account()) ->
    {ok, od_user:doc()}.
merge_unsafe(UserId, LinkedAccount) ->
    {ok, #document{value = #od_user{
        emails = Emails, linked_accounts = LinkedAccounts, entitlements = PreviousEntitlements
    } = UserInfo}} = od_user:get(UserId),
    #linked_account{
        idp = IdP, subject_id = SubjectId, emails = LinkedEmails,
        access_token = NewAccessT, refresh_token = NewRefreshT
    } = LinkedAccount,
    % Add (normalized), valid emails from the IdP that are not yet added to the account
    NewEmails = lists:usort(Emails ++ normalize_emails(LinkedEmails)),

    % Replace existing linked account, if present
    NewLinkedAccs = case find_linked_account(UserInfo, IdP, SubjectId) of
        OldLinkedAcc = #linked_account{access_token = OldAccessT, refresh_token = OldRefreshT} ->
            LinkedAccCoalescedTokens = LinkedAccount#linked_account{
                access_token = case NewAccessT of {undefined, _} -> OldAccessT; _ -> NewAccessT end,
                refresh_token = case NewRefreshT of undefined -> OldRefreshT; _ -> NewRefreshT end
            },
            lists:delete(OldLinkedAcc, LinkedAccounts) ++ [LinkedAccCoalescedTokens];
        undefined ->
            LinkedAccounts ++ [LinkedAccount]
    end,

    NewEntitlements = entitlement_mapping:coalesce_entitlements(
        UserId, NewLinkedAccs, PreviousEntitlements
    ),

    % Return updated user info
    od_user:update(UserId, fun(User = #od_user{}) ->
        {ok, User#od_user{
            emails = NewEmails,
            linked_accounts = NewLinkedAccs,
            entitlements = NewEntitlements
        }}
    end).


%%--------------------------------------------------------------------
%% @private
%% @doc
%% Finds a linked account in user doc based on IdP and user id in that IdP.
%% Returns undefined upon failure.
%% @end
%%--------------------------------------------------------------------
-spec find_linked_account(od_user:record(), auth_config:idp(),
    SubjectId :: binary()) -> undefined | od_user:linked_account().
find_linked_account(#od_user{linked_accounts = LinkedAccounts}, IdP, SubjectId) ->
    lists:foldl(fun(LinkedAccount, Acc) ->
        case LinkedAccount of
            #linked_account{idp = IdP, subject_id = SubjectId} ->
                LinkedAccount;
            _ ->
                Acc
        end
    end, undefined, LinkedAccounts).


%% @private
-spec normalize_emails([binary()]) -> [binary()].
normalize_emails(Emails) ->
    lists:filtermap(fun(Email) ->
        Normalized = http_utils:normalize_email(Email),
        case http_utils:validate_email(Normalized) of
            true -> {true, Normalized};
            false -> false
        end
    end, Emails).