%%%-------------------------------------------------------------------
%%% @author Bartosz Walkowicz
%%% @copyright (C) 2024 ACK CYFRONET AGH
%%% This software is released under the MIT license
%%% cited in 'LICENSE.txt'.
%%% @end
%%%-------------------------------------------------------------------
%%% @doc
%%% Module responsible for managing test nodes in Onedata environment.
%%% @end
%%%-------------------------------------------------------------------
-module(oct_nodes).
-author("Bartosz Walkowicz").

-include_lib("ctool/include/test/test_utils.hrl").

%% Node configuration
-export([
    refresh_config/1
]).

%% Node connection
-export([
    connect_with_nodes/1
]).

%% Node management
-export([
    kill_node/2,
    start_node/2,
    disable_panel_healthcheck/1,
    disable_panel_healthcheck/2
]).

-define(ATTEMPTS, 120).
-define(DEFAULT_COOKIE, cluster_node).

%% Types for better code clarity
-type node_type() :: worker | onepanel | cluster_manager.
-type service_type() :: op | oz.
-type service() :: onedata:service() | cluster_manager.
-type node_key() :: test_config:key().
-type node_and_key() :: {node(), node_key()}.

-export_type([
    node_type/0,
    service_type/0
]).


%%====================================================================
%% Node configuration
%%====================================================================


%% @doc Refresh nodes configuration based on deployment status
-spec refresh_config(test_config:config()) -> test_config:config().
refresh_config(Config) ->
    ensure_default_cookie(),

    lists:foldl(fun({PodName, PodAttrs}, ConfigAcc) ->
        Hostname = proplists:get_value("hostname", PodAttrs),
        ServiceType = get_service_type(PodAttrs),
        maybe_add_service_nodes(ServiceType, Hostname, PodName, ConfigAcc)
    end, Config, proplists:get_value("pods", oct_onenv_cli:status(Config))).


%%====================================================================
%% Node connection
%%====================================================================


%% @doc Connect to all nodes in environment
-spec connect_with_nodes(test_config:config()) -> test_config:config().
connect_with_nodes(Config) ->
    ensure_default_cookie(),

    DeployedHosts = connect_to_panel_nodes(Config),
    connect_to_service_nodes(Config, DeployedHosts).


%%====================================================================
%% Node management
%%====================================================================


%% @doc Kill specific node in environment
-spec kill_node(test_config:config(), node()) -> ok.
kill_node(Config, Node) ->
    Pod = test_config:get_custom(Config, [pods, Node]),
    Service = get_service(Node),
    kill_beam_process(Config, Pod, Service).


%% @doc Start specific node in environment
-spec start_node(test_config:config(), node()) -> ok.
start_node(Config, Node) ->
    Service = get_service(Node),
    Pod = test_config:get_custom(Config, [pods, Node]),
    start_service(Config, Pod, Service).


%% @doc Disable healthcheck for all services
-spec disable_panel_healthcheck(test_config:config()) -> ok.
disable_panel_healthcheck(Config) ->
    disable_panel_healthcheck(Config, all).


%% @doc Disable healthcheck for specific service
-spec disable_panel_healthcheck(test_config:config(), service() | all) -> ok.
disable_panel_healthcheck(Config, all) ->
    lists:foreach(fun(Service) ->
        disable_panel_healthcheck(Config, Service)
    end, [op_worker, oz_worker, cluster_manager]);
disable_panel_healthcheck(Config, Service) ->
    lists:foreach(fun(PanelNode) ->
        Ctx = rpc:call(PanelNode, service, get_ctx, [Service]),
        ok = rpc:call(PanelNode, service, deregister_healthcheck, [Service, Ctx])
    end, test_config:get_custom(Config, op_panel_nodes)).


%%====================================================================
%% Internal functions
%%====================================================================


%% @private
-spec ensure_default_cookie() -> ok.
ensure_default_cookie() ->
    erlang:set_cookie(node(), ?DEFAULT_COOKIE),
    ok.


%% @private
-spec get_service_type(proplists:proplist()) -> service_type() | undefined.
get_service_type(PodAttrs) ->
    case proplists:get_value("service-type", PodAttrs) of
        "oneprovider" -> op;
        "onezone" -> oz;
        _ -> undefined
    end.


%% @private
-spec maybe_add_service_nodes(service_type() | undefined, string(), string(), test_config:config()) ->
    test_config:config().
maybe_add_service_nodes(undefined, _Hostname, _PodName, Config) ->
    Config;
maybe_add_service_nodes(ServiceType, Hostname, PodName, Config) ->
    NodesAndKeys = prepare_nodes(ServiceType, Hostname),
    add_nodes_to_config(NodesAndKeys, PodName, Config).


%% @private
-spec prepare_nodes(service_type(), string()) -> [node_and_key()].
prepare_nodes(ServiceType, Hostname) ->
    PanelNode = list_to_atom("onepanel@" ++ Hostname),
    connect_with_node(PanelNode),

    lists:foldl(fun(NodeType, Acc) ->
        PanelServiceList = node_name_prefix(NodeType, ServiceType),
        PanelService = list_to_atom(PanelServiceList),
        case lists:member(Hostname, panel_test_rpc:get_service_hosts(PanelNode, PanelService)) of
            true ->
                NodeName = list_to_atom(PanelServiceList ++ "@" ++ Hostname),
                Key = service_to_config_key(NodeType, ServiceType),
                [{NodeName, Key} | Acc];
            false ->
                Acc
        end
    end, [{PanelNode, service_to_config_key(onepanel, ServiceType)}], [cluster_manager, worker]).


%% @private
-spec add_nodes_to_config([node_and_key()], string(), test_config:config()) -> test_config:config().
add_nodes_to_config(NodesAndKeys, PodName, Config) ->
    lists:foldl(fun({Node, Key}, TmpConfig) ->
        PrevNodes = test_config:get_custom(TmpConfig, Key, []),
        PrevPods = test_config:get_custom(TmpConfig, pods, #{}),
        test_config:set_many(TmpConfig, [
            [Key, lists:uniq([Node | PrevNodes])],
            [pods, PrevPods#{Node => PodName}]
        ])
    end, Config, NodesAndKeys).


%% @private
-spec node_name_prefix(node_type(), service_type()) -> string().
node_name_prefix(worker, Type) -> atom_to_list(Type) ++ "_worker";
node_name_prefix(Component, _) -> atom_to_list(Component).


%% @private
-spec service_to_config_key(node_type(), service_type() | undefined) -> node_key().
service_to_config_key(worker, op) -> op_worker_nodes;
service_to_config_key(worker, oz) -> oz_worker_nodes;
service_to_config_key(onepanel, op) -> op_panel_nodes;
service_to_config_key(onepanel, oz) -> oz_panel_nodes;
service_to_config_key(cluster_manager, _) -> cm_nodes.


%% @private
-spec connect_to_panel_nodes(test_config:config()) -> [Host :: string()].
connect_to_panel_nodes(Config) ->
    AllPanelNodes = test_config:get_all_op_panel_nodes(Config) ++ test_config:get_all_oz_panel_nodes(Config),

    lists:filtermap(fun(PanelNode) ->
        connect_with_node(PanelNode),
        wait_for_panel_init(PanelNode)
    end, AllPanelNodes).


%% @private
-spec wait_for_panel_init(node()) -> {true, Host :: string()} | false.
wait_for_panel_init(PanelNode) ->
    % NOTE: Until onepanel is initialized rpc:call/4 returns badrpc, 
    % therefore we have to catch an error thrown by test_rpc:call/4.
    ?assertMatch(true, catch panel_test_rpc:is_onepanel_initialized(PanelNode), ?ATTEMPTS),

    case is_panel_deployed(PanelNode) of
        true -> {true, utils:get_host(PanelNode)};
        false -> false
    end.


%% @private
-spec connect_to_service_nodes(test_config:config(), [Host :: string()]) -> test_config:config().
connect_to_service_nodes(Config, DeployedHosts) ->
    AllOzWorkerNodes = test_config:get_all_oz_worker_nodes(Config),
    AllOpWorkerNodes = test_config:get_all_op_worker_nodes(Config),
    AllCMNodes = test_config:get_custom(Config, cm_nodes, []),

    ConnectedOzWorkerNodes = connect_nodes_on_hosts(AllOzWorkerNodes, DeployedHosts),
    ConnectedOpWorkerNodes = connect_nodes_on_hosts(AllOpWorkerNodes, DeployedHosts),
    ConnectedCMNodes = connect_nodes_on_hosts(AllCMNodes, DeployedHosts),

    test_config:set_many(Config, [
        [oz_worker_nodes, ConnectedOzWorkerNodes],
        [op_worker_nodes, ConnectedOpWorkerNodes],
        [cm_nodes, ConnectedCMNodes]
    ]).


%% @private
-spec connect_nodes_on_hosts([node()], [Host :: string()]) -> [node()].
connect_nodes_on_hosts(Nodes, DeployedHosts) ->
    lists:filter(fun(Node) ->
        Host = utils:get_host(Node),
        case lists:member(Host, DeployedHosts) of
            true ->
                connect_with_node(Node),
                true;
            false ->
                false
        end
    end, Nodes).


%% @private
-spec connect_with_node(node()) -> ok.
connect_with_node(Node) ->
    ?assertMatch(true, net_kernel:connect_node(Node), ?ATTEMPTS).


%% @private
-spec is_panel_deployed(node()) -> boolean().
is_panel_deployed(PanelNode) ->
    case panel_test_rpc:list_onepanel_deployment(PanelNode) of
        [{onepanel_deployment, onepanel_deployment, {_, nil}}] -> false;
        [{onepanel_deployment, onepanel_deployment, {_, _}}] -> true;
        _ -> false
    end.


%% @private
-spec get_service(node()) -> test_config:service_as_list().
get_service(Node) ->
    [Service, Host] = string:split(atom_to_list(Node), "@"),
    case Service of
        "onepanel" -> case string:find(Host, "onezone") of
            nomatch -> "op_panel";
            _ -> "oz_panel"
        end;
        _ -> Service
    end.


%% @private
-spec kill_beam_process(test_config:config(), string(), string()) -> ok.
kill_beam_process(Config, Pod, Service) ->
    GetPidCommand = [
        "ps", "-eo", "'%p,%a'",
        "|", "grep", "beam.*rel/" ++ Service,
        "|", "head", "-n", "1",
        "|", "cut", "-d", "','", "-f", "1"
    ],
    PidToKill = ?assertMatch([_ | _], string:trim(oct_onenv_cli:exec(Config, [Pod] ++ GetPidCommand))),
    [] = oct_onenv_cli:exec(Config, [Pod, "kill", "-s", "SIGKILL", PidToKill]),
    ok.


%% @private
-spec start_service(test_config:config(), string(), string()) -> ok.
start_service(Config, Pod, Service) ->
    ScriptPath = test_config:get_custom(Config, list_to_atom(Service ++ "_script")),
    [] = oct_onenv_cli:exec(Config, [Pod, ScriptPath, "start"]),
    ok.
