/**
 * @file main.cc
 * @author Rafal Slota
 * @copyright (C) 2016 ACK CYFRONET AGH
 * @copyright This software is released under the MIT license cited in
 * 'LICENSE.txt'
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#ifdef linux
/* For pread()/pwrite()/utimensat() */
#define _XOPEN_SOURCE 700
#endif

#include "../s3/onezoneRestClient.h"
#include "auth/authException.h"
#include "auth/authManager.h"
#include "communication/exception.h"
#include "configuration.h"
#include "context.h"
#include "errors/handshakeErrors.h"
#include "fsOperations.h"
#include "fslogic/composite.h"
#include "fuseOperations.h"
#include "helpers/init.h"
#include "helpers/logging.h"
#include "logging.h"
#include "messages/configuration.h"
#include "messages/getConfiguration.h"
#include "messages/handshakeResponse.h"
#include "monitoring/monitoring.h"
#include "monitoring/monitoringConfiguration.h"
#include "options/options.h"
#include "scheduler.h"
#include "scopeExit.h"
#include "version.h"

#include <Poco/Net/SSLManager.h>
#include <folly/Singleton.h>
#include <fuse3/fuse_lowlevel.h>
#include <fuse3/fuse_opt.h>
#include <macaroons.hpp>
#include <syslog.h>

#include <sys/mount.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <unistd.h>

#include <csignal>
#include <exception>
#include <future>
#include <iostream>
#include <memory>
#include <random>
#include <regex>
#include <string>

#ifdef ENABLE_BACKWARD_CPP
#define BACKWARD_HAS_DW 1
#define BACKWARD_HAS_UNWIND 1
#include <backward.hpp>
#endif

using namespace one;                  // NOLINT
using namespace one::client;          // NOLINT
using namespace one::client::logging; // NOLINT
using namespace one::monitoring;      // NOLINT

namespace {
std::shared_ptr<options::Options> _options{};
} // namespace

static void syslogCallback(
    enum fuse_log_level level, const char *fmt, va_list ap)
{
    char localfmt[1024]; // NOLINT

    auto current_log_level = FUSE_LOG_DEBUG;
    bool use_syslog = true;

    if (current_log_level < level) {
        return;
    }

    sprintf(localfmt, "[ID: %08ld] %s", syscall(__NR_gettid), fmt);

    if (use_syslog) {
        int priority = LOG_ERR;
        switch (level) {
            case FUSE_LOG_EMERG:
                priority = LOG_EMERG;
                break;
            case FUSE_LOG_ALERT:
                priority = LOG_ALERT;
                break;
            case FUSE_LOG_CRIT:
                priority = LOG_CRIT;
                break;
            case FUSE_LOG_ERR:
                priority = LOG_ERR;
                break;
            case FUSE_LOG_WARNING:
                priority = LOG_WARNING;
                break;
            case FUSE_LOG_NOTICE:
                priority = LOG_NOTICE;
                break;
            case FUSE_LOG_INFO:
                priority = LOG_INFO;
                break;
            case FUSE_LOG_DEBUG:
                priority = LOG_DEBUG;
                break;
        }
        vsyslog(priority, fmt, ap);
    }
    else {
        vfprintf(stderr, fmt, ap);
    }
}

std::shared_ptr<options::Options> getOptions(int argc, char *argv[])
{
    auto options = std::make_shared<options::Options>();
    try {
        options->parse(argc, argv);
        return options;
    }
    catch (const boost::program_options::error &e) {
        fmt::print(stderr, "{}\nSee '{} --help'\n",
            std::regex_replace(e.what(), std::regex("--"), ""), argv[0]);
        exit(EXIT_FAILURE);
    }
}

void sigtermHandler(int signum)
{
    if (!_options)
        exit(signum);

#ifdef ENABLE_BACKWARD_CPP
    const auto crashDumpPath = _options->getLogDirPath() /
        fmt::format("crash-{}.log", std::time(nullptr));
    constexpr auto kStackTraceSizeMax = 48U;
    std::ofstream crashDumpStream(crashDumpPath.c_str(), std::ios::trunc);
    if (crashDumpStream.is_open()) {
        backward::StackTrace st;
        st.load_here(kStackTraceSizeMax);
        backward::Printer p;
        p.print(st, crashDumpStream);
    }
    crashDumpStream.flush();
    crashDumpStream.close();
#endif

    fmt::print(stderr,
        "Oneclient received ({}) signal - releasing mountpoint: {}\n", signum,
        _options->getMountpoint().c_str());

    const auto *exec = "/bin/fusermount3";

    // NOLINTNEXTLINE(hicpp-vararg,cppcoreguidelines-pro-type-vararg)
    execl(exec, exec, "-uz", _options->getMountpoint().c_str(), NULL);

    // Raise signal again. Should usually terminate the program.
    std::raise(signum);
}

void unmountFuse(std::shared_ptr<options::Options> options)
{
    int status = 0;
    int pid = fork();

    if (pid != 0) {
        waitpid(pid, &status, 0);
    }
    else {
#if defined(__APPLE__)
        auto exec = "/usr/sbin/diskutil";
        // NOLINTNEXTLINE(hicpp-vararg,cppcoreguidelines-pro-type-vararg)
        execl(exec, exec, "unmount", options->getMountpoint().c_str(), nullptr);
#else
        const auto *exec = "/bin/fusermount3";
        // NOLINTNEXTLINE(hicpp-vararg,cppcoreguidelines-pro-type-vararg)
        execl(exec, exec, "-uz", options->getMountpoint().c_str(), nullptr);
#endif
    }
    if (status == 0) {
        std::cout << "Oneclient has been successfully unmounted." << std::endl;
    }
    exit(status);
}

class InsecureCertificateHandler : public Poco::Net::InvalidCertificateHandler {
    using Poco::Net::InvalidCertificateHandler::InvalidCertificateHandler;

    void onInvalidCertificate(const void * /*pSender*/,
        Poco::Net::VerificationErrorArgs &errorCert) override
    {
        errorCert.setIgnoreError(true);
    }
};

bool verifyOnezoneConnection(
    const std::string &onezoneHost, std::shared_ptr<options::Options> options)
{
    auto onezoneRestClient =
        std::make_unique<one::rest::onezone::OnezoneClient>(onezoneHost);

    auto accessScope =
        onezoneRestClient->inferAccessTokenScope(*options->getAccessToken());

    return !accessScope.spaces.empty();
}

int main(int argc, char *argv[])
{
    helpers::init();

    auto context = std::make_shared<OneclientContext>();
    auto options = getOptions(argc, argv);
    _options = options;
    boost::optional<std::string> onezoneHost;
    context->setOptions(options);

    context->setScheduler(
        std::make_shared<Scheduler>(_options->getSchedulerThreadCount()));

    if (options->getHelp()) {
        std::cout << options->formatHelp(argv[0]);
        return EXIT_SUCCESS;
    }
    if (options->getVersion()) {
        fmt::print("Oneclient: {}\nFUSE library: {}.{}\n", ONECLIENT_VERSION,
            FUSE_MAJOR_VERSION, FUSE_MINOR_VERSION);
        return EXIT_SUCCESS;
    }
    if (options->getUnmount()) {
        unmountFuse(options);
    }
    if (!options->getOnezoneHost() && options->getAccessToken()) {
        try {
            auto deserialized =
                one::client::auth::deserialize(*options->getAccessToken());
            onezoneHost = deserialized.location();
        }
        catch (const std::exception &e) {
            fmt::print(stderr,
                "ERROR: Failed to extract Onezone host name from access "
                "token.\n");
            return EXIT_FAILURE;
        }
    }
    else {
        onezoneHost = options->getOnezoneHost();
    }

    if (!onezoneHost) {
        fmt::print(stderr,
            "ERROR: Cannot determine Onezone host name.\nSee "
            "'{} "
            "--help'.\n",
            argv[0]);
        return EXIT_FAILURE;
    }

    if (options->hasDeprecated()) {
        std::cout << options->formatDeprecated();
    }
    if (options->isInsecure()) {
        constexpr auto kVerificationDepth{9};

        // Initialize insecure access to Onedata REST services
        Poco::Net::Context::Ptr pContext =
            new Poco::Net::Context(Poco::Net::Context::CLIENT_USE, "", "", "",
                Poco::Net::Context::VERIFY_NONE, kVerificationDepth, true,
                "ALL:!ADH:!LOW:!EXP:!MD5:@STRENGTH");
        Poco::Net::SSLManager::instance().initializeClient({},
            Poco::SharedPtr<InsecureCertificateHandler>(
                new InsecureCertificateHandler(true)),
            pContext);
    }
    startLogging(argv[0], options);

    int res{};

    try {
        auto fuse_oper = fuseOperations(options);
        auto args = options->getFuseArgs(argv[0]);
        char *mountpoint{nullptr};
        int multithreaded{0};
        int foreground{0};
        struct fuse_session *fuse{nullptr};
        bool use_syslog = false;

        struct fuse_cmdline_opts opts {
        };
        res = fuse_parse_cmdline(&args, &opts);
        if (res == -1)
            return EXIT_FAILURE;

        if (use_syslog) {
            fuse_set_log_func(syslogCallback);
            openlog("oneclient", LOG_PID, LOG_DAEMON);
        }

        multithreaded = !opts.singlethread; // NOLINT
        foreground = opts.foreground;
        mountpoint = opts.mountpoint;

        if (foreground == 0) {
            FLAGS_stderrthreshold = 3;
        }
        else {
            FLAGS_stderrthreshold = options->getDebug() ? 0 : 1;
        }

        ScopeExit freeMountpoint{[=] {
            free(mountpoint); // NOLINT
        }};

        fmt::print(
            stderr, "Connecting to Onezone at: {}\n", onezoneHost.value());

        auto tokenAccessHasSpaces =
            verifyOnezoneConnection(onezoneHost.value(), options);

        if (!tokenAccessHasSpaces) {
            fmt::print(stderr,
                "Access token does not give access to any data spaces in {}\n",
                onezoneHost.value());
            return EXIT_FAILURE;
        }

        std::unique_ptr<fslogic::Composite> fsLogic;

        fuse = fuse_session_new(&args, &fuse_oper, sizeof(fuse_oper), &fsLogic);
        if (fuse == nullptr)
            return EXIT_FAILURE;

        res = fuse_set_signal_handlers(fuse);
        if (res == -1)
            return EXIT_FAILURE;

        ScopeExit removeHandlers{[=] { fuse_remove_signal_handlers(fuse); }};

        res = fuse_session_mount(fuse, mountpoint);
        if (res != 0)
            return EXIT_FAILURE;

        ScopeExit unmountFuse{[=] { fuse_session_unmount(fuse); }};
        ScopeExit destroyFuse{[=] { fuse_session_destroy(fuse); }, unmountFuse};

        std::signal(SIGINT, sigtermHandler);
        std::signal(SIGTERM, sigtermHandler);
        std::signal(SIGSEGV, sigtermHandler);

        std::cout << "Oneclient has been successfully mounted in '"
                  << options->getMountpoint().c_str() << "'." << std::endl;

        if (foreground == 0) {
            assert(context.get() != nullptr);
            assert(context->scheduler().get() != nullptr);

            context.reset();

            folly::SingletonVault::singleton()->destroyInstances();

            fuse_remove_signal_handlers(fuse);
            res = fuse_daemonize(foreground);

            if (res != -1)
                res = fuse_set_signal_handlers(fuse);

            if (res == -1)
                return EXIT_FAILURE;

            folly::SingletonVault::singleton()->reenableInstances();

            auto onezoneRestClient =
                std::make_unique<one::rest::onezone::OnezoneClient>(
                    onezoneHost.value());

            fsLogic = std::make_unique<fslogic::Composite>(
                options, std::move(onezoneRestClient));
        }
        else {
            FLAGS_stderrthreshold = options->getDebug() ? 0 : 1;

            auto onezoneRestClient =
                std::make_unique<one::rest::onezone::OnezoneClient>(
                    onezoneHost.value());

            fsLogic = std::make_unique<fslogic::Composite>(
                options, std::move(onezoneRestClient));
        }

        if (startPerformanceMonitoring(options) != EXIT_SUCCESS)
            return EXIT_FAILURE;

#if FUSE_USE_VERSION > 31
        struct fuse_loop_config config {
        };
        config.clone_fd = opts.clone_fd;
        config.max_idle_threads = opts.max_idle_threads;
        res = (multithreaded != 0) ? fuse_session_loop_mt(fuse, &config)
                                   : fuse_session_loop(fuse);
#elif FUSE_VERSION == 31
        res = (multithreaded != 0) ? fuse_session_loop_mt(fuse, opts.clone_fd)
                                   : fuse_session_loop(fuse);
#else
        res = (multithreaded != 0) ? fuse_session_loop_mt(fuse)
                                   : fuse_session_loop(fuse);
#endif
    }
    catch (const macaroons::exception::Invalid &e) {
        fmt::print(stderr,
            "ERROR: Cannot parse token - please make sure that the access "
            "token has been copied correctly.\n");
        return EXIT_FAILURE;
    }
    catch (const macaroons::exception::NotAuthorized &e) {
        fmt::print(stderr,
            "ERROR: Invalid token - please make sure that the access token is "
            "valid for Oneclient in Onezone: {}\n",
            *onezoneHost);
        return EXIT_FAILURE;
    }
    catch (const std::system_error &e) {
        using one::errors::handshake::ErrorCode;

        if (e.code() == ErrorCode::macaroon_expired)
            fmt::print(stderr,
                "ERROR: Expired token - the provided token is "
                "expired, please create a new one.\n");
        else if (e.code() == ErrorCode::invalid_macaroon ||
            e.code() == ErrorCode::invalid_provider ||
            e.code() == ErrorCode::macaroon_not_found)
            fmt::print(stderr,
                "ERROR: Invalid token - the provided token is not valid for "
                "Oneclient access");
        else if (e.code() == ErrorCode::incompatible_version)
            fmt::print(stderr,
                "ERROR: This Oneclient version ({}) is not compatible with "
                "this Onezone, see: "
                "https://{}/api/v3/onezone/configuration\nPlease also "
                "consult the current Onedata compatibility matrix: "
                "https://onedata.org/#/home/versions\n",
                ONECLIENT_VERSION, *onezoneHost);
        else {
            fmt::print(
                stderr, "ERROR: Cannot connect to Oneprovider: {}\n", e.what());
        }

        return EXIT_FAILURE;
    }
    catch (const folly::AsyncSocketException &e) {
        std::string message;
        using fas = folly::AsyncSocketException;
        if (e.getType() == fas::AsyncSocketExceptionType::SSL_ERROR) {
            message = "SSL socket creation failed, if the Oneprovider has "
                      "self-hosted certificate add option '-i'";
        }
        else if (e.getType() == fas::AsyncSocketExceptionType::TIMED_OUT) {
            message = "Connection timed out, please make sure the Oneprovider "
                      "hostname is correct and reachable from this host.";
        }
        else {
            message = e.what();
        }

        fmt::print(
            stderr, "ERROR: Cannot connect to Oneprovider - {}\n", message);

        return EXIT_FAILURE;
    }
    catch (const Poco::Net::HostNotFoundException &e) {
        fmt::print(stderr, "ERROR: Cannot connect to Onezone {} - {}\n",
            *onezoneHost, e.what());
        return EXIT_FAILURE;
    }
    catch (const Poco::Net::HTTPException &e) {
        fmt::print(stderr, "ERROR: Onezone {} cannot handle request - {}\n",
            *onezoneHost, e.what());
        return EXIT_FAILURE;
    }
    catch (const std::exception &e) {
        fmt::print(stderr, "ERROR: Unknown error {}\n", e.what());
        return EXIT_FAILURE;
    }

    return res == -1 ? EXIT_FAILURE : EXIT_SUCCESS;
}
