%%%-------------------------------------------------------------------
%%% @author Piotr Duleba
%%% @copyright (C) 2020 ACK CYFRONET AGH
%%% This software is released under the MIT license
%%% cited in 'LICENSE.txt'.
%%% @end
%%%-------------------------------------------------------------------
%%% @doc
%%% This module contains functions that create background configuration
%%% and provide access to configuration info.
%%% @end
%%%-------------------------------------------------------------------
-module(oct_background).

-include("oct_background.hrl").
-include_lib("ctool/include/aai/aai.hrl").
-include_lib("ctool/include/aai/caveats.hrl").
-include_lib("ctool/include/test/test_utils.hrl").

-type entity_id() :: binary().
-type entity_placeholder() :: atom().
-type entity_name() :: binary().
-type entity_selector() :: entity_id() | entity_placeholder().
-type node_selector() :: entity_id() | entity_placeholder() | node().
-type onenv_test_config() :: #onenv_test_config{}.

-export_type([
    entity_id/0,
    entity_placeholder/0,
    entity_selector/0,
    node_selector/0,
    onenv_test_config/0
]).

%% API
-export([
    init_per_suite/2,
    end_per_suite/0
]).
-export([
    update_environment/1,
    update_background_config/1,

    to_entity_placeholder/1,
    to_entity_id/1,
    get_service_panels/1,
    get_all_panels/0,

    get_provider_id/1,
    get_provider_panels/1,
    get_provider_domain/1,
    get_provider_name/1,
    get_random_provider_node/1,
    get_provider_nodes/1,
    get_all_providers_nodes/0,
    get_provider_eff_users/1,
    get_provider_supported_spaces/1,
    get_provider_ids/0,

    get_zone_panels/0,
    get_zone_nodes/0,
    get_zone_domain/0,

    get_user_id/1,
    get_user_name/1,
    get_user_fullname/1,
    get_user_access_token/1,
    get_user_session_id/2,

    get_space_id/1,
    get_space_name/1,
    get_space_supporting_providers/1,

    get_group_id/1,
    get_group_name/1
]).

-define(DEFAULT_TEMP_CAVEAT_TTL, 360000).

%%%===================================================================
%%% API
%%%===================================================================


-spec init_per_suite(test_config:config(), onenv_test_config()) ->
    test_config:config().
init_per_suite(Config, #onenv_test_config{
    onenv_scenario = Scenario,
    envs = Envs,
    posthook = CustomPostHook
}) ->
    application:start(ssl),
    application:ensure_all_started(hackney),

    AddEnvs = lists:map(fun({Service, Application, CustomEnv}) ->
        {add_envs, [Service, Application, CustomEnv]}
    end, Envs),

    test_config:set_many(Config, AddEnvs ++ [
        {set_onenv_scenario, [Scenario]},
        {set_posthook, fun(NewConfig) ->
            CustomPostHook(
                prepare_base_test_config(NewConfig)
            )
        end}
    ]).


-spec end_per_suite() -> ok.
end_per_suite() ->
    application:stop(hackney),
    application:stop(ssl).


-spec update_environment(test_config:config()) -> test_config:config().
update_environment(Config) ->
    TestModule = test_config:get_custom(Config, test_module),
    ConfigWithConnectedNodes = oct_environment:connect_with_nodes(Config),
    oct_environment:load_modules(TestModule, ConfigWithConnectedNodes),
    update_background_config(ConfigWithConnectedNodes).


-spec update_background_config(test_config:config()) -> test_config:config().
update_background_config(Config) ->
    prepare_base_test_config(Config).


-spec to_entity_placeholder(binary() | atom()) -> entity_placeholder().
to_entity_placeholder(Placeholder) when is_atom(Placeholder) ->
    Placeholder;
to_entity_placeholder(EntityId) when is_binary(EntityId) ->
    kv_utils:get([id_to_placeholder, EntityId], get_oct_mappings()).


-spec to_entity_id(binary() | atom()) -> entity_id().
to_entity_id(Placeholder) when is_binary(Placeholder) ->
    Placeholder;
to_entity_id(EntityId) when is_atom(EntityId) ->
    kv_utils:get([placeholder_to_id, EntityId], get_oct_mappings()).


get_service_panels(EntitySelector) ->
    EntityPlaceholder = to_entity_placeholder(EntitySelector),
    case lists:member(EntityPlaceholder, maps:keys(kv_utils:get([oneproviders], get_oct_mappings()))) of
        true ->
            get_provider_panels(EntityPlaceholder);
        false ->
            get_zone_panels()
    end.


-spec get_all_panels() -> [node()].
get_all_panels() ->
    ProviderPanels = lists:flatmap(fun(ProviderPlaceholder) ->
        oct_background:get_provider_panels(ProviderPlaceholder)
    end, maps:keys(kv_utils:get([oneproviders], get_oct_mappings()))),
    ProviderPanels ++ oct_background:get_zone_panels().


-spec get_provider_id(entity_selector()) -> entity_id().
get_provider_id(EntitySelector) ->
    ProviderPlaceholder = to_entity_placeholder(EntitySelector),
    kv_utils:get([oneproviders, ProviderPlaceholder, id], get_oct_mappings()).


-spec get_provider_panels(entity_selector()) -> [node()].
get_provider_panels(EntitySelector) ->
    ProviderPlaceholder = to_entity_placeholder(EntitySelector),
    kv_utils:get([oneproviders, ProviderPlaceholder, panels], get_oct_mappings()).


-spec get_provider_domain(entity_selector()) -> binary().
get_provider_domain(EntitySelector) ->
    ProviderPlaceholder = to_entity_placeholder(EntitySelector),
    kv_utils:get([oneproviders, ProviderPlaceholder, domain], get_oct_mappings()).


-spec get_provider_name(entity_selector()) -> binary().
get_provider_name(EntitySelector) ->
    ProviderPlaceholder = to_entity_placeholder(EntitySelector),
    kv_utils:get([oneproviders, ProviderPlaceholder, name], get_oct_mappings()).


-spec get_random_provider_node(entity_selector()) -> node().
get_random_provider_node(ProviderSelector) ->
    lists_utils:random_element(get_provider_nodes(ProviderSelector)).


-spec get_provider_nodes(entity_selector()) -> [node()].
get_provider_nodes(EntitySelector) ->
    ProviderPlaceholder = to_entity_placeholder(EntitySelector),
    kv_utils:get([oneproviders, ProviderPlaceholder, nodes], get_oct_mappings()).


-spec get_all_providers_nodes() -> [node()].
get_all_providers_nodes() ->
    lists:foldl(fun(Provider, NodeAcc) ->
        oct_background:get_provider_nodes(Provider) ++ NodeAcc
    end, [], maps:keys(kv_utils:get([oneproviders], get_oct_mappings()))).


-spec get_provider_eff_users(entity_selector()) -> [entity_id()].
get_provider_eff_users(EntitySelector) ->
    ProviderPlaceholder = to_entity_placeholder(EntitySelector),
    kv_utils:get([oneproviders, ProviderPlaceholder, users], get_oct_mappings()).


-spec get_provider_supported_spaces(entity_selector()) -> [entity_id()].
get_provider_supported_spaces(EntitySelector) ->
    ProviderPlaceholder = to_entity_placeholder(EntitySelector),
    kv_utils:get([oneproviders, ProviderPlaceholder, supported_spaces], get_oct_mappings()).


-spec get_provider_ids() -> [entity_id()].
get_provider_ids() ->
    maps:fold(fun(_, ProviderDetails, IdsAcc) ->
        [kv_utils:get([id], ProviderDetails) | IdsAcc]
    end, [], kv_utils:get([oneproviders], get_oct_mappings())).


-spec get_zone_nodes() -> [node()].
get_zone_nodes() ->
    kv_utils:get([onezones, zone, nodes], get_oct_mappings()).


-spec get_zone_panels() -> [node()].
get_zone_panels() ->
    kv_utils:get([onezones, zone, panels], get_oct_mappings()).


-spec get_zone_domain() -> binary().
get_zone_domain() ->
    kv_utils:get([onezones, zone, domain], get_oct_mappings()).


-spec get_user_id(entity_selector()) -> entity_id().
get_user_id(EntitySelector) ->
    UserPlaceholder = to_entity_placeholder(EntitySelector),
    kv_utils:get([users, UserPlaceholder, id], get_oct_mappings()).


-spec get_user_name(entity_selector()) -> entity_id().
get_user_name(EntitySelector) ->
    UserPlaceholder = to_entity_placeholder(EntitySelector),
    kv_utils:get([users, UserPlaceholder, name], get_oct_mappings()).


-spec get_user_fullname(entity_selector()) -> entity_id().
get_user_fullname(EntitySelector) ->
    UserPlaceholder = to_entity_placeholder(EntitySelector),
    kv_utils:get([users, UserPlaceholder, fullname], get_oct_mappings()).


-spec get_user_access_token(entity_selector()) -> binary().
get_user_access_token(EntitySelector) ->
    UserPlaceholder = to_entity_placeholder(EntitySelector),
    kv_utils:get([users, UserPlaceholder, access_token], get_oct_mappings()).


-spec get_user_session_id(
    UserSelector :: entity_selector(),
    ProviderSelector :: entity_selector()
) ->
    entity_id().
get_user_session_id(UserSelector, ProviderSelector) ->
    UserPlaceholder = to_entity_placeholder(UserSelector),
    ProviderPlaceholder = to_entity_placeholder(ProviderSelector),
    kv_utils:get([users, UserPlaceholder, sessions, ProviderPlaceholder], get_oct_mappings()).


-spec get_space_id(entity_selector()) -> entity_id().
get_space_id(EntitySelector) ->
    SpacePlaceholder = to_entity_placeholder(EntitySelector),
    kv_utils:get([spaces, SpacePlaceholder, id], get_oct_mappings()).


-spec get_space_name(entity_id() | entity_placeholder()) -> entity_name().
get_space_name(SpaceIdOrPlaceholder) ->
    SpacePlaceholder = to_entity_placeholder(SpaceIdOrPlaceholder),
    kv_utils:get([spaces, SpacePlaceholder, name], get_oct_mappings()).


-spec get_space_supporting_providers(entity_id() | entity_placeholder()) -> [entity_id()].
get_space_supporting_providers(SpaceIdOrPlaceholder) ->
    SpacePlaceholder = to_entity_placeholder(SpaceIdOrPlaceholder),
    kv_utils:get([spaces, SpacePlaceholder, supporting_providers], get_oct_mappings()).


-spec get_group_id(entity_selector()) -> entity_id().
get_group_id(EntitySelector) ->
    GroupPlaceholder = to_entity_placeholder(EntitySelector),
    kv_utils:get([groups, GroupPlaceholder, id], get_oct_mappings()).


-spec get_group_name(entity_selector()) -> entity_name().
get_group_name(EntitySelector) ->
    GroupPlaceholder = to_entity_placeholder(EntitySelector),
    kv_utils:get([groups, GroupPlaceholder, name], get_oct_mappings()).


%%%===================================================================
%%% Internal functions
%%%===================================================================


%% @private
-spec prepare_base_test_config(test_config:config()) -> test_config:config().
prepare_base_test_config(Config) ->
    OneProvidersMap = describe_providers(Config),
    OneZoneMap = describe_onezone(Config),

    {UsersMap, GroupsMap, SpacesMap} = case test_config:get_all_oz_worker_nodes(Config) of
        [] ->
            {#{}, #{}, #{}};
        _ ->
            {
                describe_users(OneZoneMap, OneProvidersMap),
                describe_groups(OneZoneMap),
                describe_spaces(OneZoneMap)
            }
    end,

    {IdToPlaceholderMap, PlaceholderToIdMap} = maps:fold(
        fun(Placeholder, EntityDesc, {IdToPlaceholderAcc, PlaceholderToIdAcc}) ->
            EntityId = maps:get(id, EntityDesc),
            {IdToPlaceholderAcc#{EntityId => Placeholder}, PlaceholderToIdAcc#{Placeholder => EntityId}}
        end,
        {#{}, #{}},
        maps_utils:merge([OneProvidersMap, OneZoneMap, UsersMap, GroupsMap, SpacesMap])
    ),

    store_oct_mappings(#{
        oneproviders => OneProvidersMap,
        onezones => OneZoneMap,
        spaces => SpacesMap,
        groups => GroupsMap,
        users => UsersMap,
        id_to_placeholder => IdToPlaceholderMap,
        placeholder_to_id => PlaceholderToIdMap
    }),

    Config.


%% @private
-spec describe_providers(test_config:config()) -> map().
describe_providers(Config) ->
    OpWorkerNodes = test_config:get_all_op_worker_nodes(Config),
    lists:foldl(fun(OpWorkerNode, ProvidersDescAcc) ->
        ProviderId = opw_test_rpc:get_provider_id(OpWorkerNode),
        ProviderName = opw_test_rpc:get_provider_name(OpWorkerNode),
        Placeholder = binary_to_atom(lists:last(binary:split(ProviderName, <<"-">>, [global])), utf8),
        case kv_utils:find([Placeholder, id], ProvidersDescAcc) of
            {ok, ProviderId} ->
                UpdatedWorkerNodesDescAcc = kv_utils:update_with([Placeholder, nodes], fun(Nodes) ->
                    [OpWorkerNode | Nodes]
                end, ProvidersDescAcc),
                kv_utils:update_with([Placeholder, panels], fun(Panels) ->
                    Panel = get_provider_panel(Config, OpWorkerNode),
                    [Panel | Panels]
                end, UpdatedWorkerNodesDescAcc);
            error ->
                Storages = opw_test_rpc:get_storages(OpWorkerNode),
                Users = opw_test_rpc:get_provider_eff_users(OpWorkerNode),
                Panel = get_provider_panel(Config, OpWorkerNode),
                Spaces = opw_test_rpc:get_spaces(OpWorkerNode),
                Domain = ?GET_DOMAIN_BIN(OpWorkerNode),
                ProviderDetails = #{
                    id => ProviderId,
                    nodes => [OpWorkerNode],
                    domain => Domain,
                    name => ProviderName,
                    panels => [Panel],
                    supported_spaces => Spaces,
                    storages => Storages,
                    users => Users
                },
                ProvidersDescAcc#{Placeholder => ProviderDetails};
            {ok, _OtherProviderId} ->
                ct:fail("Wrong enviroment setup. There are multiple providers with the same name: ~tp.", [ProviderName])
        end
    end, #{}, OpWorkerNodes).


%% @private
-spec describe_onezone(test_config:config()) -> map().
describe_onezone(Config) ->
    OzWorkerNodes = test_config:get_all_oz_worker_nodes(Config),
    ZoneDetails = #{
        id => <<"onezone">>,
        nodes => OzWorkerNodes,
        panels => test_config:get_all_oz_panel_nodes(Config)
    },
    ZoneDetailsWithDomain = case OzWorkerNodes of
        [] -> ZoneDetails;
        [OzWorkerNode | _] -> ZoneDetails#{domain => ?GET_DOMAIN_BIN(OzWorkerNode)}
    end,
    #{zone => ZoneDetailsWithDomain}.


%% @private
-spec describe_users(map(), map()) -> map().
describe_users(OnezoneMap, OneprovidersMap) ->
    [OzNode] = kv_utils:get([zone, nodes], OnezoneMap),
    UserIds = ozw_test_rpc:list_users(OzNode),

    lists:foldl(fun(UserId, Acc) ->
        UserProtectedData = ozw_test_rpc:get_user_protected_data(OzNode, UserId),
        UserName = maps:get(<<"username">>, UserProtectedData),
        FullName = maps:get(<<"fullName">>, UserProtectedData),
        Placeholder = case is_binary(UserName) of
            true -> binary_to_atom(UserName, utf8);
            false -> UserName
        end,
        AccessToken = create_oz_temp_access_token(OzNode, UserId),

        Sessions = maps:fold(fun(ProviderPlaceholder, ProviderDesc, SessionAcc) ->
            case lists:member(UserId, kv_utils:get([users], ProviderDesc)) of
                true ->
                    [ProviderNode | _] = kv_utils:get([nodes], ProviderDesc),
                    SessionAcc#{ProviderPlaceholder => create_session(ProviderNode, UserId, AccessToken)};
                false ->
                    SessionAcc
            end
        end, #{}, OneprovidersMap),
        Acc#{Placeholder => #{
            id => UserId,
            name => UserName,
            fullname => FullName,
            access_token => AccessToken,
            sessions => Sessions
        }}
    end, #{}, UserIds).


%% @private
-spec describe_groups(map()) -> map().
describe_groups(OnezoneMap) ->
    [OzNode] = kv_utils:get([zone, nodes], OnezoneMap),

    lists:foldl(fun(GroupId, GroupDescAcc) ->

        GroupProtectedData = ozw_test_rpc:get_group_protected_data(OzNode, aai:root_auth(), GroupId),
        GroupName = maps:get(<<"name">>, GroupProtectedData),
        GroupPlaceholder = binary_to_atom(GroupName, utf8),

        GroupDescAcc#{GroupPlaceholder => #{
            id => GroupId,
            name => GroupName
        }}
    end, #{}, ozw_test_rpc:list_groups(OzNode)).


%% @private
-spec describe_spaces(map()) -> map().
describe_spaces(OnezoneMap) ->
    [OzNode] = kv_utils:get([zone, nodes], OnezoneMap),
    SpaceIds = ozw_test_rpc:list_spaces(OzNode),

    lists:foldl(fun(SpaceId, SpaceDescAcc) ->

        SpaceInfo = ozw_test_rpc:get_space_protected_data(OzNode, aai:root_auth(), SpaceId),
        SpaceName = maps:get(<<"name">>, SpaceInfo),
        SpacePlaceholder = binary_to_atom(SpaceName, utf8),

        SpaceDescAcc#{SpacePlaceholder => #{
            id => SpaceId,
            name => SpaceName,
            supporting_providers => maps:keys(maps:get(<<"providers">>, SpaceInfo))
        }}

    end, #{}, SpaceIds).


%% @private
-spec get_oct_mappings() -> map().
get_oct_mappings() ->
    node_cache:get(oct_mapping).


%% @private
-spec store_oct_mappings(map()) -> ok.
store_oct_mappings(OctMapping) ->
    node_cache:put(oct_mapping, OctMapping).


%% @private
-spec create_oz_temp_access_token(node(), UserId :: binary()) -> tokens:serialized().
create_oz_temp_access_token(OzNode, UserId) ->
    Auth = ?USER(UserId),
    Now = ozw_test_rpc:timestamp_seconds(OzNode),
    Token = ozw_test_rpc:create_user_temporary_token(OzNode, Auth, UserId, #{
        <<"type">> => ?ACCESS_TOKEN,
        <<"caveats">> => [#cv_time{valid_until = Now + ?DEFAULT_TEMP_CAVEAT_TTL}]
    }),

    {ok, SerializedToken} = tokens:serialize(Token),
    SerializedToken.


%% @private
-spec create_session(node(), entity_id(), tokens:serialized()) -> entity_id().
create_session(Node, UserId, AccessToken) ->
    Nonce = crypto:strong_rand_bytes(10),
    Identity = ?SUB(user, UserId),
    TokenCredentials = opw_test_rpc:build_token_credentials(Node, AccessToken, undefined, local_ip_v4(), oneclient, allow_data_access_caveats),
    SessionId = opw_test_rpc:create_fuse_session(Node, Nonce, Identity, TokenCredentials),
    SessionId.


%% @private
-spec local_ip_v4() -> inet:ip_address().
local_ip_v4() ->
    {ok, Addrs} = inet:getifaddrs(),
    hd([
        Addr || {_, Opts} <- Addrs, {addr, Addr} <- Opts,
        size(Addr) == 4, Addr =/= {127, 0, 0, 1}
    ]).


%% @private
-spec get_provider_panel(test_config:config(), node()) -> node().
get_provider_panel(Config, OpWorkerNode) ->
    ProviderHost = ?GET_HOSTNAME(OpWorkerNode),
    [Panel] = lists:filter(fun(PanelNode) ->
        ?GET_HOSTNAME(PanelNode) == ProviderHost
    end, test_config:get_all_op_panel_nodes(Config)),
    Panel.
