%%%--------------------------------------------------------------------
%%% @author Bartosz Walkowicz
%%% @copyright (C) 2020 ACK CYFRONET AGH
%%% This software is released under the MIT license
%%% cited in 'LICENSE.txt'.
%%% @end
%%%--------------------------------------------------------------------
%%% @doc
%%% Provides functions to manage authorization nonce records that holds
%%% random nonces used for inviting nodes to cluster.
%%% @end
%%%--------------------------------------------------------------------
-module(authorization_nonce).
-author("Bartosz Walkowicz").

-behaviour(model_behaviour).

-include("modules/errors.hrl").
-include("modules/models.hrl").
-include("names.hrl").

%% API
-export([create/0, verify/1, delete_expired_nonces/0]).

%% Model behaviour callbacks
-export([
    get_fields/0, get_record_version/0, upgrade/2, seed/0,
    list/0, create/1, save/1, exists/1, get/1, update/2, delete/1
]).

-type nonce() :: binary().
-type timestamp() :: time:seconds().
-type record() :: #authorization_nonce{}.

-export_type([nonce/0, timestamp/0, record/0]).

-define(NONCE_TTL, onepanel_env:get(
    panel_authorization_nonce_ttl_sec, ?APP_NAME, 3600
)).
-define(NOW(), global_clock:timestamp_seconds()).


%%%===================================================================
%%% API functions
%%%===================================================================


%%--------------------------------------------------------------------
%% @doc
%% Creates a new, random authorization nonce.
%% @end
%%--------------------------------------------------------------------
-spec create() -> {ok, nonce()} | {error, term()}.
create() ->
    Record = #authorization_nonce{
        nonce = onepanel_utils:gen_uuid(),
        expires = ?NOW() + ?NONCE_TTL
    },
    case create(Record) of
        {ok, Nonce} -> {ok, Nonce};
        {error, _} = Error -> Error
    end.


%%--------------------------------------------------------------------
%% @doc
%% Verifies if given nonce was generated by this panel/cluster.
%% Expired nonces are considered invalid.
%% @end
%%--------------------------------------------------------------------
-spec verify(nonce()) -> boolean().
verify(Nonce) ->
    Now = ?NOW(),
    case ?MODULE:get(Nonce) of
        {ok, #authorization_nonce{expires = Expires}} when Expires > Now ->
            true;
        _ ->
            false
    end.


-spec delete_expired_nonces() -> ok.
delete_expired_nonces() ->
    Now = ?NOW(),
    lists:foreach(fun(#authorization_nonce{nonce = Nonce, expires = Expires}) ->
        case Expires < Now of
            true -> delete(Nonce);
            false -> ok
        end
    end, list()).


%%%===================================================================
%%% Model behaviour callbacks
%%%===================================================================


%%--------------------------------------------------------------------
%% @doc
%% {@link model_behaviour:get_fields/0}
%% @end
%%--------------------------------------------------------------------
-spec get_fields() -> list(atom()).
get_fields() ->
    record_info(fields, ?MODULE).


%%--------------------------------------------------------------------
%% @doc
%% {@link model_behaviour:get_record_version/0}
%% @end
%%--------------------------------------------------------------------
-spec get_record_version() -> model_behaviour:version().
get_record_version() ->
    1.


%%--------------------------------------------------------------------
%% @doc
%% {@link model_behaviour:upgrade/2}
%% @end
%%--------------------------------------------------------------------
-spec upgrade(PreviousVsn :: model_behaviour:version(), PreviousRecord :: tuple()) ->
    no_return().
upgrade(1, _Record) ->
    error(?ERROR_NOT_SUPPORTED).


%%--------------------------------------------------------------------
%% @doc
%% {@link model_behaviour:seed/0}
%% @end
%%--------------------------------------------------------------------
-spec seed() -> any().
seed() ->
    ok.


%%--------------------------------------------------------------------
%% @doc
%% {@link model_behaviour:list/0}
%% @end
%%--------------------------------------------------------------------
-spec list() -> Records :: [record()] | no_return().
list() ->
    model:list(?MODULE).


%%--------------------------------------------------------------------
%% @doc
%% {@link model_behaviour:create/1}
%% @end
%%--------------------------------------------------------------------
-spec create(Record :: record()) ->
    {ok, nonce()} | {error, _} | no_return().
create(Record) ->
    model:create(?MODULE, Record).


%%--------------------------------------------------------------------
%% @doc
%% {@link model_behaviour:save/1}
%% @end
%%--------------------------------------------------------------------
-spec save(Record :: record()) -> ok | no_return().
save(Record) ->
    model:save(?MODULE, Record).


%%--------------------------------------------------------------------
%% @doc
%% {@link model_behaviour:exists/1}
%% @end
%%--------------------------------------------------------------------
-spec exists(Key :: model_behaviour:key()) ->
    boolean() | no_return().
exists(Key) ->
    model:exists(?MODULE, Key).


%%--------------------------------------------------------------------
%% @doc
%% {@link model_behaviour:get/1}
%% @end
%%--------------------------------------------------------------------
-spec get(Key :: model_behaviour:key()) ->
    {ok, Record :: record()} | {error, _} | no_return().
get(Key) ->
    model:get(?MODULE, Key).


%%--------------------------------------------------------------------
%% @doc
%% {@link model_behaviour:update/2}
%% @end
%%--------------------------------------------------------------------
-spec update(Key :: model_behaviour:key(), Diff :: model_behaviour:diff()) ->
    no_return().
update(_Key, _Diff) ->
    error(?ERROR_NOT_SUPPORTED).


%%--------------------------------------------------------------------
%% @doc
%% {@link model_behaviour:delete/1}
%% @end
%%--------------------------------------------------------------------
-spec delete(Key :: model_behaviour:key()) -> ok | no_return().
delete(Key) ->
    model:delete(?MODULE, Key).
