%%%-------------------------------------------------------------------
%%% @author Michal Stanisz
%%% @copyright (C) 2020 ACK CYFRONET AGH
%%% This software is released under the MIT license
%%% cited in 'LICENSE.txt'.
%%% @end
%%%-------------------------------------------------------------------
%%% @doc
%%% Functions used by ct test to start and manage environment 
%%% using onenv for testing.
%%% @end
%%%-------------------------------------------------------------------
-module(oct_environment).
-author("Michal Stanisz").

-include_lib("ctool/include/logging.hrl").
-include_lib("ctool/include/test/test_utils.hrl").

%% API
-export([
    setup_environment/2,
    teardown_environment/1,
    disable_panel_healthcheck/1,
    connect_with_nodes/1,
    load_modules/2,
    kill_node/2,
    start_node/2
]).

-define(ATTEMPTS, 120).
-define(TIMEOUT, timer:seconds(60)).
-define(DEFAULT_COOKIE, cluster_node).

-define(ALL_NODES_TYPES, [cm_nodes, op_worker_nodes, op_panel_nodes, oz_worker_nodes, oz_panel_nodes]).

-type path() :: string().
-type component() :: worker | onepanel | cluster_manager.
-type service() :: onedata:service() | cluster_manager.


%%%===================================================================
%%% API
%%%===================================================================


-spec setup_environment(test_config:config(), TestModule :: module()) ->
    test_config:config().
setup_environment(Config0, TestModule) ->
    application:start(yamerl),
    ProjectRoot = test_utils:project_root_dir(Config0),
    OnenvScript = filename:join([ProjectRoot, "one-env", "onenv"]),
    PathToSources = os:getenv("path_to_sources"),
    AbsPathToSources = filename:join([ProjectRoot, PathToSources]),

    % strings "true" or "false"
    CleanEnv = os:getenv("clean_env"),
    CoverEnabled = os:getenv("cover") == "true",

    % Dummy first call to onenv to setup configs.
    % Path to sources must be proivided in first onenv call that creates one-env docker
    utils:cmd([OnenvScript, "status", "--path-to-sources", AbsPathToSources]),
    Sources = utils:cmd(["cd", ProjectRoot, "&&", OnenvScript, "find_sources"]),
    ct:pal("~nUsing sources from:~n~n~ts", [Sources]),

    Config = test_config:set_many(read_optional_args(Config0), [
        {set_onenv_script_path, [OnenvScript]},
        {set_project_root_path, [ProjectRoot]}
    ]),

    ConfigWithCover = case CoverEnabled of
        true ->
            {ok, CoverSpec} = file:consult(filename:join(ProjectRoot, "test_distributed/cover_tmp.spec")),
            CoveredDirs = lists:map(fun(DirRelPath) ->
                list_to_atom(filename:join(ProjectRoot, DirRelPath))
            end, kv_utils:get(incl_dirs_r, CoverSpec)),

            ExcludedModules = kv_utils:get(excl_mods, CoverSpec),
            RepoName = string:trim(os:cmd("basename -s .git `git config --get remote.origin.url`")),
            case RepoName of
                "onepanel" ->
                    ConfigWithOzPanel = test_config:add_envs(Config, oz_panel, oz_panel,
                        [{covered_dirs, CoveredDirs}, {covered_excluded_modules, ExcludedModules}]),
                    test_config:add_envs(ConfigWithOzPanel, oz_panel, oz_panel,
                        [{covered_dirs, CoveredDirs}, {covered_excluded_modules, ExcludedModules}]);
                "op-worker" ->
                    test_config:add_envs(Config, op_worker, op_worker,
                        [{covered_dirs, CoveredDirs}, {covered_excluded_modules, ExcludedModules}]);
                "oz-worker" ->
                    test_config:add_envs(Config, oz_worker, oz_worker,
                        [{covered_dirs, CoveredDirs}, {covered_excluded_modules, ExcludedModules}]);
                "cluster-manager" ->
                    test_config:add_envs(Config, cluster_manager, cluster_manager,
                    [{covered_dirs, CoveredDirs}, {covered_excluded_modules, ExcludedModules}])
            end;

        false ->
            Config
    end,

    PodsProplist = start_environment(ConfigWithCover),
    add_entries_to_etc_hosts(PodsProplist),
    NodesConfig = prepare_nodes_config(ConfigWithCover, PodsProplist),
    ConnectedNodesConfig = connect_with_nodes(NodesConfig),

    load_modules(TestModule, ConnectedNodesConfig),

    % when cover is enabled all modules are recompiled resulting in custom configs being lost
    % so setting custom envs manually is necessary
    CoverEnabled andalso set_custom_envs(NodesConfig),

    test_config:set_many(ConnectedNodesConfig, [
        [clean_env, CleanEnv == "true"],
        [op_worker_script, script_path(NodesConfig, "op_worker")],
        [cluster_manager_script, script_path(NodesConfig, "cluster_manager")],
        [test_module, TestModule]
    ]).


-spec read_optional_args(test_config:config()) -> test_config:config().
read_optional_args(Config) ->
    SourcesFilters = case os:getenv("sources_filters") of
        false -> [];
        SourcesFiltersString -> string:split(SourcesFiltersString, ";")
    end,
    % use utils:ensure_defined as os:getenv accepts only string as default value, so undefined there breaks dialyzer
    OnezoneImage = utils:ensure_defined(os:getenv("onezone_image"), false, undefined),
    OneproviderImage = utils:ensure_defined(os:getenv("oneprovider_image"), false, undefined),
    test_config:set_many(Config, [
        [sources_filters, SourcesFilters],
        [onezone_image, OnezoneImage],
        [oneprovider_image, OneproviderImage]
    ]).


-spec teardown_environment(test_config:config()) -> ok.
teardown_environment(Config) ->
    OnenvScript = test_config:get_onenv_script_path(Config),
    PrivDir = test_config:get_custom(Config, priv_dir),
    CleanEnv = test_config:get_custom(Config, clean_env, true),

    ExportResult = shell_utils:get_success_output([OnenvScript, "export", PrivDir]),
    file:write_file(filename:join(PrivDir, "onenv_export.log"), ExportResult),
    CleanEnv andalso test_node_starter:finalize(Config),
    CleanEnv andalso utils:cmd([OnenvScript, "clean", "--all", "--persistent-volumes"]),
    ok.


-spec disable_panel_healthcheck(test_config:config()) -> ok.
disable_panel_healthcheck(Config) ->
    disable_panel_healthcheck(Config, all).


-spec disable_panel_healthcheck(test_config:config(), service() | all) -> ok.
disable_panel_healthcheck(Config, all) ->
    lists:foreach(fun(Service) ->
        disable_panel_healthcheck(Config, Service)
    end, [op_worker, oz_worker, cluster_manager]);
disable_panel_healthcheck(Config, Service) ->
    lists:foreach(fun(PanelNode) ->
        Ctx = rpc:call(PanelNode, service, get_ctx, [Service]),
        ok = rpc:call(PanelNode, service, deregister_healthcheck, [Service, Ctx])
    end, test_config:get_custom(Config, op_panel_nodes)).


-spec connect_with_nodes(test_config:config()) -> test_config:config().
connect_with_nodes(Config) ->
    erlang:set_cookie(node(), ?DEFAULT_COOKIE),

    AllPanelNodes = test_config:get_all_op_panel_nodes(Config) ++ test_config:get_all_oz_panel_nodes(Config),
    DeployedHosts = lists:filtermap(fun(PanelNode) ->
        connect_with_node(PanelNode),

        % NOTE: Until onepanel is initialized rpc:call/4 returns badrpc, therefore we have to catch an error thrown by test_rpc:call/4.
        ?assertMatch(true, catch panel_test_rpc:is_onepanel_initialized(PanelNode), ?ATTEMPTS),
        case is_panel_deployed(PanelNode) of
            true -> {true, utils:get_host(PanelNode)};
            false -> false
        end
    end, AllPanelNodes),

    AllOzWorkerNodes = test_config:get_all_oz_worker_nodes(Config),
    AllOpWorkerNodes = test_config:get_all_op_worker_nodes(Config),
    AllCMNodes = test_config:get_custom(Config, cm_nodes),

    ConnectedOzWorkerNodes = connect_with_nodes_on_deployed_hosts(AllOzWorkerNodes, DeployedHosts),
    ConnectedOpWorkerNodes = connect_with_nodes_on_deployed_hosts(AllOpWorkerNodes, DeployedHosts),
    ConnectedCMNodes = connect_with_nodes_on_deployed_hosts(AllCMNodes, DeployedHosts),

    test_config:set_many(Config, [
        [oz_worker_nodes, ConnectedOzWorkerNodes],
        [op_worker_nodes, ConnectedOpWorkerNodes],
        [cm_nodes, ConnectedCMNodes]]
    ).


-spec load_modules(TestModule :: module(), Modules :: [module()]) -> ok.
load_modules(TestModule, Config) ->
    Modules = [TestModule, test_utils | ?config(load_modules, Config, [])],

    lists:foreach(fun(NodeType) ->
        Nodes = test_config:get_custom(Config, NodeType, []),
        lists:foreach(fun(Node) ->
            lists:foreach(fun(Module) ->
                {Module, Binary, Filename} = code:get_object_code(Module),
                rpc:call(Node, code, delete, [Module], ?TIMEOUT),
                rpc:call(Node, code, purge, [Module], ?TIMEOUT),
                ?assertEqual({module, Module}, rpc:call(
                    Node, code, load_binary, [Module, Filename, Binary], ?TIMEOUT
                ))
            end, Modules)
        end, Nodes)
    end, ?ALL_NODES_TYPES).


-spec kill_node(test_config:config(), node()) -> ok.
kill_node(Config, Node) ->
    OnenvScript = test_config:get_onenv_script_path(Config),
    Pod = test_config:get_custom(Config, [pods, Node]),
    Service = get_service(Node),
    GetPidCommand = [
        "ps", "-eo", "'%p,%a'",
        "|", "grep", "beam.*rel/" ++ Service,
        "|", "head", "-n", "1",
        "|", "cut", "-d", "','", "-f", "1"
    ],
    PidToKill = ?assertMatch([_ | _], string:trim(utils:cmd([OnenvScript, "exec", Pod] ++ GetPidCommand))),
    [] = utils:cmd([OnenvScript, "exec", Pod, "kill", "-s", "SIGKILL", PidToKill]),
    ok.


-spec start_node(test_config:config(), node()) -> ok.
start_node(Config, Node) ->
    OnenvScript = test_config:get_onenv_script_path(Config),
    Service = get_service(Node),
    ScriptPath = test_config:get_custom(Config, list_to_atom(Service ++ "_script")),
    Pod = test_config:get_custom(Config, [pods, Node]),
    [] = utils:cmd([OnenvScript, "exec", Pod, ScriptPath, "start"]),
    ok.


%%%===================================================================
%%% Internal functions
%%%===================================================================


%% @private
-spec start_environment(test_config:config()) -> proplists:proplist().
start_environment(Config) ->
    ScenarioName = test_config:get_scenario(Config),
    OnenvScript = test_config:get_onenv_script_path(Config),
    ProjectRoot = test_config:get_project_root_path(Config),
    CustomEnvs = test_config:get_custom_envs(Config),
    
    CustomConfigsPaths = add_custom_configs(Config, CustomEnvs),

    OptionalArgs = build_onenv_up_optional_args(Config),
    ?assert(is_list(ScenarioName)),
    ScenarioPath = filename:join([ProjectRoot, "test_distributed", "onenv_scenarios", ScenarioName ++ ".yaml"]),
    ct:pal("Starting onenv scenario ~tp~n~n~tp", [ScenarioName, ScenarioPath]),
    BaseStartCmd = ["cd", ProjectRoot, "&&", OnenvScript, "up"] ++ OptionalArgs ++ [ScenarioPath],
    StartCmd = case os:getenv("rsync") of
        "true" -> BaseStartCmd ++ ["--rsync"];
        _ -> BaseStartCmd
    end,
    OnenvStartLogs = utils:cmd(StartCmd),
    ct:pal("~ts", [OnenvStartLogs]),

    lists:foreach(fun file:delete/1, CustomConfigsPaths),
    utils:cmd([OnenvScript, "wait", "--timeout", "1200"]),

    Status = utils:cmd([OnenvScript, "status"]),
    [StatusProplist] = yamerl:decode(Status),
    ct:pal("~ts", [Status]),
    case proplists:get_value("ready", StatusProplist) of
        false ->
            ok = teardown_environment(Config),
            throw(environment_not_ready);
        true -> ok
    end,
    proplists:get_value("pods", StatusProplist).


%% @private
-spec build_onenv_up_optional_args(test_config:config()) -> [string()].
build_onenv_up_optional_args(Config) ->
    ZoneImageArgs = case test_config:get_custom(Config, onezone_image) of
        undefined  -> [];
        OnezoneImage -> ["-zi", OnezoneImage]
    end,
    ProviderImageArgs = case test_config:get_custom(Config, oneprovider_image) of
        undefined  -> [];
        OneproviderImage -> ["-pi", OneproviderImage]
    end,
    SourcesFilters = test_config:get_custom(Config, sources_filters),
    SourcesFiltersArgs = lists:flatmap(fun(Filter) ->
        ["-sf", Filter]
    end, SourcesFilters),
    ZoneImageArgs ++ ProviderImageArgs ++ SourcesFiltersArgs.


%% @private
-spec prepare_nodes_config(test_config:config(), proplists:proplist()) -> test_config:config().
prepare_nodes_config(Config, PodsProplist) ->
    lists:foldl(fun({PodName, X}, TmpConfig) ->
        Hostname = proplists:get_value("hostname", X),
        case proplists:get_value("service-type", X) of
            "oneprovider" ->
                NodesAndKeys = prepare_nodes(op, Hostname),
                add_nodes_to_config(NodesAndKeys, PodName, TmpConfig);
            "onezone" ->
                NodesAndKeys = prepare_nodes(oz, Hostname),
                add_nodes_to_config(NodesAndKeys, PodName, TmpConfig);
            _ ->
                TmpConfig
        end
    end, Config, PodsProplist).


%% @private
-spec prepare_nodes(oz | op, string()) -> [{node(), test_config:key()}].
prepare_nodes(ServiceType, Hostname) ->
    lists:map(fun(NodeType) ->
        NodeName = list_to_atom(node_name_prefix(NodeType, ServiceType) ++ "@" ++ Hostname),
        Key = service_to_config_key(NodeType, ServiceType),
        {NodeName, Key}
    end, [worker, onepanel, cluster_manager]).


%% @private
-spec add_nodes_to_config([{node(), test_config:key()}], PodName :: string(), test_config:config()) ->
    test_config:config().
add_nodes_to_config(NodesAndKeys, PodName, Config) ->
    lists:foldl(fun({Node, Key}, TmpConfig) ->
        PrevNodes = test_config:get_custom(TmpConfig, Key, []),
        PrevPods = test_config:get_custom(TmpConfig, pods, #{}),
        test_config:set_many(TmpConfig, [
            [Key, [Node | PrevNodes]],
            [pods, PrevPods#{Node => PodName}]
        ])
    end, Config, NodesAndKeys).


%% @private
-spec add_entries_to_etc_hosts(proplists:proplist()) -> ok.
add_entries_to_etc_hosts(PodsConfig) ->
    HostsEntries = lists:foldl(fun({_ServiceName, ServiceConfig}, Acc0) ->
        ServiceType = proplists:get_value("service-type", ServiceConfig),
        case lists:member(ServiceType, ["onezone", "oneprovider"]) of
            true ->
                Ip = proplists:get_value("ip", ServiceConfig),

                Acc1 = case proplists:get_value("domain", ServiceConfig, undefined) of
                    undefined -> Acc0;
                    Domain -> [{Domain, Ip} | Acc0]
                end,
                case proplists:get_value("hostname", ServiceConfig, undefined) of
                    undefined -> Acc1;
                    Hostname -> [{Hostname, Ip} | Acc1]
                end;
            false ->
                Acc0
        end
    end, [], PodsConfig),
    
    {ok, RawFile} = file:read_file("/etc/hosts"),
    
    EntriesToPreserve = lists:filtermap(fun
        (<<>>) ->
            false;
        (Entry) ->
            [Ip, Domain] = binary:split(Entry, [<<"\t">>, <<" ">>]),
            case Domain of
                <<"dev-one", _Tail/binary>> ->
                    false;
                _ ->
                    {true, {Domain, Ip}}
            end
    end, binary:split(RawFile, <<"\n">>, [global])),
    {ok, File} = file:open("/etc/hosts", [write]),
    
    lists:foreach(fun({DomainOrHostname, Ip}) ->
        io:fwrite(File, "~ts\t~ts~n", [Ip, DomainOrHostname])
    end, EntriesToPreserve ++ HostsEntries),

    file:close(File).


%% @private
-spec add_custom_configs(test_config:config(), [{Component :: atom(), proplists:proplist()}]) -> [path()].
add_custom_configs(Config, CustomEnvs) ->
    lists:foldl(fun({Component, Envs}, Acc) ->
        Path = test_custom_config_path(Config, atom_to_list(Component)),
        file:write_file(Path, io_lib:format("~tp.", [Envs])),
        [Path | Acc]
    end, [], CustomEnvs).


%% @private
-spec script_path(test_config:config(), test_config:service_as_list()) -> path().
script_path(Config, Service) ->
    SourcesRelPath = sources_rel_path(Config, Service),
    filename:join([SourcesRelPath, Service, "bin", Service]).


%% @private
-spec test_custom_config_path(test_config:config(), test_config:service_as_list()) -> path().
test_custom_config_path(Config, Service) ->
    SourcesRelPath = sources_rel_path(Config, Service),
    filename:join([SourcesRelPath, Service, "etc", "config.d", "ct_test_custom.config"]).


%% @private
-spec sources_rel_path(test_config:config(), test_config:service_as_list()) -> path().
sources_rel_path(Config, Service) ->
    Service1 = re:replace(Service, "_", "-", [{return, list}]),

    OnenvScript = test_config:get_onenv_script_path(Config),
    ProjectRoot = test_config:get_project_root_path(Config),
    SourcesYaml = utils:cmd(["cd", ProjectRoot, "&&", OnenvScript, "find_sources"]),
    [Sources] = yamerl:decode(SourcesYaml),

    filename:join([kv_utils:get([Service1], Sources), "_build", "default", "rel"]).


%% @private
-spec set_custom_envs(test_config:config()) -> ok.
set_custom_envs(Config) ->
    CustomEnvs = test_config:get_custom_envs(Config),
    lists:foreach(fun({Component, Envs}) ->
        ConfigKey = service_to_config_key(Component),
        Nodes = test_config:get_custom(Config, ConfigKey),
        lists:foreach(fun(Node) -> set_custom_envs_on_node(Node, Envs) end, Nodes)
    end, CustomEnvs).


%% @private
-spec set_custom_envs_on_node(node(), [{test_config:service(), proplists:proplist()}]) -> ok.
set_custom_envs_on_node(Node, CustomEnvs) ->
    lists:foreach(fun({Application, EnvList}) ->
        lists:foreach(fun({EnvKey, EnvValue}) ->
            ok = rpc:call(Node, application, set_env, [Application, EnvKey, EnvValue])
        end, EnvList)
    end, CustomEnvs).


%% @private
-spec service_to_config_key(component(), op | oz | undefined) -> test_config:key().
service_to_config_key(worker, op) -> op_worker_nodes;
service_to_config_key(worker, oz) -> oz_worker_nodes;
service_to_config_key(onepanel, op) -> op_panel_nodes;
service_to_config_key(onepanel, oz) -> oz_panel_nodes;
service_to_config_key(cluster_manager, _) -> cm_nodes.


%% @private
-spec service_to_config_key(test_config:service()) -> test_config:key().
service_to_config_key(op_worker) -> service_to_config_key(worker, op);
service_to_config_key(oz_worker) -> service_to_config_key(worker, oz);
service_to_config_key(op_panel) -> service_to_config_key(onepanel, op);
service_to_config_key(oz_panel) -> service_to_config_key(onepanel, oz);
service_to_config_key(cluster_manager) -> service_to_config_key(cluster_manager, undefined).


%% @private
-spec node_name_prefix(component(), op | oz) -> string().
node_name_prefix(worker, Type) -> atom_to_list(Type) ++ "_worker";
node_name_prefix(Component, _) -> atom_to_list(Component).


%% @private
-spec get_service(node()) -> test_config:service_as_list().
get_service(Node) ->
    [Service, Host] = string:split(atom_to_list(Node), "@"),
    case Service of
        "onepanel" -> case string:find(Host, "onezone") of
            nomatch -> "op_panel";
            _ -> "oz_panel"
        end;
        _ -> Service
    end.


%% @private
-spec connect_with_nodes_on_deployed_hosts([node()], [Host :: string()]) -> [node()].
connect_with_nodes_on_deployed_hosts(Nodes, DeployedHosts) ->
    lists:filter(fun(Node) ->
        Host = utils:get_host(Node),
        case lists:member(Host, DeployedHosts) of
            true ->
                connect_with_node(Node),
                true;
            false ->
                false
        end
    end, Nodes).


%% @private
-spec connect_with_node(node()) -> ok.
connect_with_node(Node) ->
    ?assertMatch(true, net_kernel:connect_node(Node), ?ATTEMPTS).


%% @private
-spec is_panel_deployed(node()) -> boolean().
is_panel_deployed(PanelNode) ->
    case panel_test_rpc:list_onepanel_deployment(PanelNode) of
        [{onepanel_deployment, onepanel_deployment, {_, nil}}] -> false;
        [{onepanel_deployment, onepanel_deployment, {_, _}}] -> true;
        _ -> false
    end.