%%%--------------------------------------------------------------------
%%% @author Piotr Duleba
%%% @copyright (C) 2020 ACK CYFRONET AGH
%%% This software is released under the MIT license
%%% cited in 'LICENSE.txt'.
%%% @end
%%%--------------------------------------------------------------------
%%% @doc
%%% Unit tests for oct_background module.
%%% @end
%%%--------------------------------------------------------------------
-module(oct_background_test).

-ifdef(TEST).

-include_lib("eunit/include/eunit.hrl").
-include_lib("onenv_ct/include/oct_background.hrl").

-define(KRAKOW_WORKER, 'op_worker@dev-oneprovider-krakow-0.dev-oneprovider-krakow.default.svc.cluster.local').
-define(PARIS_WORKER, 'op_worker@dev-oneprovider-paris-0.dev-oneprovider-paris.default.svc.cluster.local').
-define(KRAKOW_PANEL, 'onepanel@dev-oneprovider-krakow-0.dev-oneprovider-krakow.default.svc.cluster.local').
-define(PARIS_PANEL, 'onepanel@dev-oneprovider-paris-0.dev-oneprovider-paris.default.svc.cluster.local').
-define(ZONE_WORKER, 'oz_worker@dev-onezone-0.dev-onezone.default.svc.cluster.local').
-define(ZONE_PANEL, 'onepanel@dev-onezone-0.dev-onezone.default.svc.cluster.local').

-define(ONEZONE_ID, <<"onezone">>).
-define(ONEZONE_PLACEHOLDER, zone).

-define(PROVIDER_KRAKOW_ID, <<"provider_krakow_id">>).
-define(PROVIDER_KRAKOW_PLACEHOLDER, krakow).
-define(PROVIDER_KRAKOW_NAME, <<"dev-oneprovider-krakow">>).
-define(PROVIDER_KRAKOW_DOMAIN, <<"dev-oneprovider-krakow.default.svc.cluster.local">>).
-define(USER_1_KRAKOW_SESSION, <<"user_1_krakow_session">>).
-define(USER_2_KRAKOW_SESSION, <<"user_2_krakow_session">>).

-define(PROVIDER_PARIS_ID, <<"provider_paris_id">>).
-define(PROVIDER_PARIS_PLACEHOLDER, paris).
-define(PROVIDER_PARIS_NAME, <<"dev-oneprovider-paris">>).
-define(PROVIDER_PARIS_DOMAIN, <<"dev-oneprovider-paris.default.svc.cluster.local">>).
-define(USER_2_PARIS_SESSION, <<"user_2_paris_session">>).
-define(USER_3_PARIS_SESSION, <<"user_3_paris_session">>).

-define(USER_1_ID, <<"user_1_id">>).
-define(USER_1_PLACEHOLDER, user_1).
-define(USER_1_NAME, <<"user_1">>).
-define(USER_1_FULLNAME, <<"user_1_fullname">>).
-define(USER_1_ACCESS_TOKEN, <<"user_1_access_token">>).
-define(USER_2_ID, <<"user_2_id">>).
-define(USER_2_PLACEHOLDER, user_2).
-define(USER_2_NAME, <<"user_2">>).
-define(USER_2_FULLNAME, <<"user_2_fullname">>).
-define(USER_2_ACCESS_TOKEN, <<"user_2_access_token">>).
-define(USER_3_ID, <<"user_3_id">>).
-define(USER_3_PLACEHOLDER, user_3).
-define(USER_3_NAME, <<"user_3">>).
-define(USER_3_FULLNAME, <<"user_3_fullname">>).
-define(USER_3_ACCESS_TOKEN, <<"user_3_access_token">>).

-define(GROUP_1_ID, <<"group_1_id">>).
-define(GROUP_1_PLACEHOLDER, group_1_name).
-define(GROUP_1_NAME, <<"group_1_name">>).
-define(GROUP_2_ID, <<"group_2_id">>).
-define(GROUP_2_PLACEHOLDER, group_2_name).
-define(GROUP_2_NAME, <<"group_2_name">>).

-define(SPACE_1_ID, <<"space_1_id">>).
-define(SPACE_1_PLACEHOLDER, space_1).
-define(SPACE_1_NAME, <<"space_1">>).
-define(SPACE_2_ID, <<"space_2_id">>).
-define(SPACE_2_PLACEHOLDER, space_2).
-define(SPACE_2_NAME, <<"space_2">>).
-define(SPACE_3_ID, <<"space_3_id">>).
-define(SPACE_3_PLACEHOLDER, space_3).
-define(SPACE_3_NAME, <<"space_3">>).

-define(STORAGE_1_ID, <<"storage_1_id">>).
-define(STORAGE_2_ID, <<"storage_2_id">>).
-define(STORAGE_3_ID, <<"storage_3_id">>).
-define(STORAGE_4_ID, <<"storage_4_id">>).

-define(DUMMY_SUPPORT_SIZE, 100).

-define(CT_CONFIG_1_OP, [
    {oz_panel_nodes, [?ZONE_PANEL]},
    {oz_worker_nodes, [?ZONE_WORKER]},
    {op_panel_nodes, [?KRAKOW_PANEL]},
    {op_worker_nodes, [?KRAKOW_WORKER]}
]).

-define(CT_CONFIG_2_OP, [
    {oz_panel_nodes, [?ZONE_PANEL]},
    {oz_worker_nodes, [?ZONE_WORKER]},
    {op_panel_nodes, [?KRAKOW_PANEL, ?PARIS_PANEL]},
    {op_worker_nodes, [?KRAKOW_WORKER, ?PARIS_WORKER]}
]).


%%%===================================================================
%%% Test functions
%%%===================================================================


to_entity_placeholder_test_() -> foreach_ct_config_(fun(Scenario) ->
    F = fun oct_background:to_entity_placeholder/1,
    build_instantiator(Scenario, [
        ?_assertEqual(?ONEZONE_PLACEHOLDER, F(?ONEZONE_ID)),
        ?_assertEqual(?ONEZONE_PLACEHOLDER, F(?ONEZONE_PLACEHOLDER)),
        ?_assertEqual(?PROVIDER_KRAKOW_PLACEHOLDER, F(?PROVIDER_KRAKOW_ID)),
        ?_assertEqual(?PROVIDER_KRAKOW_PLACEHOLDER, F(?PROVIDER_KRAKOW_PLACEHOLDER)),
        ?_assertEqual(?USER_1_PLACEHOLDER, F(?USER_1_ID)),
        ?_assertEqual(?USER_1_PLACEHOLDER, F(?USER_1_PLACEHOLDER)),
        ?_assertEqual(?GROUP_1_PLACEHOLDER, F(?GROUP_1_ID)),
        ?_assertEqual(?GROUP_1_PLACEHOLDER, F(?GROUP_1_PLACEHOLDER)),
        ?_assertEqual(?SPACE_1_PLACEHOLDER, F(?SPACE_1_ID)),
        ?_assertEqual(?SPACE_1_PLACEHOLDER, F(?SPACE_1_PLACEHOLDER))

    ], [], [
        ?_assertEqual(?PROVIDER_PARIS_PLACEHOLDER, F(?PROVIDER_PARIS_ID)),
        ?_assertEqual(?PROVIDER_PARIS_PLACEHOLDER, F(?PROVIDER_PARIS_PLACEHOLDER)),
        ?_assertEqual(?GROUP_2_PLACEHOLDER, F(?GROUP_2_ID)),
        ?_assertEqual(?GROUP_2_PLACEHOLDER, F(?GROUP_2_PLACEHOLDER)),
        ?_assertEqual(?SPACE_3_PLACEHOLDER, F(?SPACE_3_ID)),
        ?_assertEqual(?SPACE_3_PLACEHOLDER, F(?SPACE_3_PLACEHOLDER)),
        ?_assertEqual(?USER_3_PLACEHOLDER, F(?USER_3_ID)),
        ?_assertEqual(?USER_3_PLACEHOLDER, F(?USER_3_PLACEHOLDER))
    ])
end).


to_entity_id_test_() -> foreach_ct_config_(fun(Scenario) ->
    F = fun oct_background:to_entity_id/1,
    build_instantiator(Scenario, [
        ?_assertEqual(?ONEZONE_ID, F(?ONEZONE_ID)),
        ?_assertEqual(?ONEZONE_ID, F(?ONEZONE_PLACEHOLDER)),
        ?_assertEqual(?PROVIDER_KRAKOW_ID, F(?PROVIDER_KRAKOW_ID)),
        ?_assertEqual(?PROVIDER_KRAKOW_ID, F(?PROVIDER_KRAKOW_PLACEHOLDER)),
        ?_assertEqual(?USER_1_ID, F(?USER_1_ID)),
        ?_assertEqual(?USER_1_ID, F(?USER_1_PLACEHOLDER)),
        ?_assertEqual(?GROUP_1_ID, F(?GROUP_1_ID)),
        ?_assertEqual(?GROUP_1_ID, F(?GROUP_1_PLACEHOLDER)),
        ?_assertEqual(?SPACE_1_ID, F(?SPACE_1_ID)),
        ?_assertEqual(?SPACE_1_ID, F(?SPACE_1_PLACEHOLDER))

    ], [], [
        ?_assertEqual(?PROVIDER_PARIS_ID, F(?PROVIDER_PARIS_ID)),
        ?_assertEqual(?PROVIDER_PARIS_ID, F(?PROVIDER_PARIS_PLACEHOLDER)),
        ?_assertEqual(?GROUP_2_ID, F(?GROUP_2_ID)),
        ?_assertEqual(?GROUP_2_ID, F(?GROUP_2_PLACEHOLDER)),
        ?_assertEqual(?SPACE_3_ID, F(?SPACE_3_ID)),
        ?_assertEqual(?SPACE_3_ID, F(?SPACE_3_PLACEHOLDER)),
        ?_assertEqual(?USER_3_ID, F(?USER_3_ID)),
        ?_assertEqual(?USER_3_ID, F(?USER_3_PLACEHOLDER))
    ])
end).


get_service_panels_test_() -> foreach_ct_config_(fun(Scenario) ->
    F = fun oct_background:get_service_panels/1,
    build_instantiator(Scenario, [
        ?_assertEqual([?KRAKOW_PANEL], F(?PROVIDER_KRAKOW_PLACEHOLDER)),
        ?_assertEqual([?KRAKOW_PANEL], F(?PROVIDER_KRAKOW_ID)),
        ?_assertEqual([?ZONE_PANEL], F(?ONEZONE_ID)),
        ?_assertEqual([?ZONE_PANEL], F(?ONEZONE_PLACEHOLDER))
    ], [], [
        ?_assertEqual([?PARIS_PANEL], F(?PROVIDER_PARIS_PLACEHOLDER))
    ])
end).


get_all_panels_test_() -> foreach_ct_config_(fun(Scenario) ->
    F = fun oct_background:get_all_panels/0,
    build_instantiator(Scenario, [], [
        ?_assertEqual([?KRAKOW_PANEL, ?ZONE_PANEL], F())
    ], [
        ?_assertEqual([?KRAKOW_PANEL, ?PARIS_PANEL, ?ZONE_PANEL], F())
    ])
end).


get_provider_id_test_() -> foreach_ct_config_(fun(Scenario) ->
    F = fun oct_background:get_provider_id/1,
    build_instantiator(Scenario, [
        ?_assertEqual(?PROVIDER_KRAKOW_ID, F(?PROVIDER_KRAKOW_PLACEHOLDER)),
        ?_assertEqual(?PROVIDER_KRAKOW_ID, F(?PROVIDER_KRAKOW_ID))
    ], [], [
        ?_assertEqual(?PROVIDER_PARIS_ID, F(?PROVIDER_PARIS_PLACEHOLDER)),
        ?_assertEqual(?PROVIDER_PARIS_ID, F(?PROVIDER_PARIS_ID))
    ])
end).


get_provider_panel_test_() -> foreach_ct_config_(fun(Scenario) ->
    F = fun oct_background:get_provider_panels/1,
    build_instantiator(Scenario, [
        ?_assertEqual([?KRAKOW_PANEL], F(?PROVIDER_KRAKOW_PLACEHOLDER)),
        ?_assertEqual([?KRAKOW_PANEL], F(?PROVIDER_KRAKOW_ID))
    ], [], [
        ?_assertEqual([?PARIS_PANEL], F(?PROVIDER_PARIS_PLACEHOLDER)),
        ?_assertEqual([?PARIS_PANEL], F(?PROVIDER_PARIS_ID))
    ])
end).


get_provider_domain_test_() -> foreach_ct_config_(fun(Scenario) ->
    F = fun oct_background:get_provider_domain/1,
    build_instantiator(Scenario, [
        ?_assertEqual(?PROVIDER_KRAKOW_DOMAIN, F(?PROVIDER_KRAKOW_PLACEHOLDER)),
        ?_assertEqual(?PROVIDER_KRAKOW_DOMAIN, F(?PROVIDER_KRAKOW_ID))
    ], [], [
        ?_assertEqual(?PROVIDER_PARIS_DOMAIN, F(?PROVIDER_PARIS_PLACEHOLDER)),
        ?_assertEqual(?PROVIDER_PARIS_DOMAIN, F(?PROVIDER_PARIS_ID))
    ])
end).


get_provider_name_test_() -> foreach_ct_config_(fun(Scenario) ->
    F = fun oct_background:get_provider_name/1,
    build_instantiator(Scenario, [
        ?_assertEqual(?PROVIDER_KRAKOW_NAME, F(?PROVIDER_KRAKOW_PLACEHOLDER)),
        ?_assertEqual(?PROVIDER_KRAKOW_NAME, F(?PROVIDER_KRAKOW_ID))
    ], [], [
        ?_assertEqual(?PROVIDER_PARIS_NAME, F(?PROVIDER_PARIS_PLACEHOLDER)),
        ?_assertEqual(?PROVIDER_PARIS_NAME, F(?PROVIDER_PARIS_ID))
    ])
end).


get_provider_nodes_test_() -> foreach_ct_config_(fun(Scenario) ->
    F = fun oct_background:get_provider_nodes/1,
    build_instantiator(Scenario, [
        ?_assertEqual([?KRAKOW_WORKER], F(?PROVIDER_KRAKOW_PLACEHOLDER)),
        ?_assertEqual([?KRAKOW_WORKER], F(?PROVIDER_KRAKOW_ID))
    ], [], [
        ?_assertEqual([?PARIS_WORKER], F(?PROVIDER_PARIS_PLACEHOLDER)),
        ?_assertEqual([?PARIS_WORKER], F(?PROVIDER_PARIS_ID))
    ])
end).


get_provider_eff_users_test_() -> foreach_ct_config_(fun(Scenario) ->
    F = fun oct_background:get_provider_eff_users/1,
    build_instantiator(Scenario, [
        ?_assertEqual([?USER_1_ID, ?USER_2_ID], F(?PROVIDER_KRAKOW_PLACEHOLDER)),
        ?_assertEqual([?USER_1_ID, ?USER_2_ID], F(?PROVIDER_KRAKOW_ID))
    ], [], [
        ?_assertEqual([?USER_2_ID, ?USER_3_ID], F(?PROVIDER_PARIS_PLACEHOLDER)),
        ?_assertEqual([?USER_2_ID, ?USER_3_ID], F(?PROVIDER_PARIS_ID))
    ])
end).


get_provider_supported_spaces_test_() -> foreach_ct_config_(fun(Scenario) ->
    F = fun oct_background:get_provider_supported_spaces/1,
    build_instantiator(Scenario, [
        ?_assertEqual([?SPACE_1_ID, ?SPACE_2_ID], F(?PROVIDER_KRAKOW_PLACEHOLDER)),
        ?_assertEqual([?SPACE_1_ID, ?SPACE_2_ID], F(?PROVIDER_KRAKOW_ID))
    ], [], [
        ?_assertEqual([?SPACE_2_ID, ?SPACE_3_ID], F(?PROVIDER_PARIS_PLACEHOLDER)),
        ?_assertEqual([?SPACE_2_ID, ?SPACE_3_ID], F(?PROVIDER_PARIS_ID))
    ])
end).


get_provider_ids_test_() -> foreach_ct_config_(fun(Scenario) ->
    F = fun oct_background:get_provider_ids/0,
    build_instantiator(Scenario, [], [
        ?_assertEqual([?PROVIDER_KRAKOW_ID], F())
    ], [
        ?_assertEqual([?PROVIDER_PARIS_ID, ?PROVIDER_KRAKOW_ID], F())
    ])
end).


get_zone_panels_test_() -> foreach_ct_config_(fun(_) ->
    F = fun oct_background:get_zone_panels/0,
    ?_assertEqual([?ZONE_PANEL], F())
end).


get_zone_nodes_test_() -> foreach_ct_config_(fun(_) ->
    F = fun oct_background:get_zone_nodes/0,
    ?_assertEqual([?ZONE_WORKER], F())
end).


get_user_id_test_() -> foreach_ct_config_(fun(Scenario) ->
    F = fun oct_background:get_user_id/1,
    build_instantiator(Scenario, [
        ?_assertEqual(?USER_1_ID, F(?USER_1_PLACEHOLDER)),
        ?_assertEqual(?USER_2_ID, F(?USER_2_ID))
    ], [], [
        ?_assertEqual(?USER_3_ID, F(?USER_3_PLACEHOLDER))
    ])
end).


get_user_name_test_() -> foreach_ct_config_(fun(Scenario) ->
    F = fun oct_background:get_user_name/1,
    build_instantiator(Scenario, [
        ?_assertEqual(?USER_1_NAME, F(?USER_1_PLACEHOLDER)),
        ?_assertEqual(?USER_2_NAME, F(?USER_2_ID))
    ], [], [
        ?_assertEqual(?USER_3_NAME, F(?USER_3_PLACEHOLDER))
    ])
end).


get_user_fullname_test_() -> foreach_ct_config_(fun(Scenario) ->
    F = fun oct_background:get_user_fullname/1,
    build_instantiator(Scenario, [
        ?_assertEqual(?USER_1_FULLNAME, F(?USER_1_PLACEHOLDER)),
        ?_assertEqual(?USER_2_FULLNAME, F(?USER_2_ID))
    ], [], [
        ?_assertEqual(?USER_3_FULLNAME, F(?USER_3_PLACEHOLDER))
    ])
end).


get_user_access_token_test_() -> foreach_ct_config_(fun(Scenario) ->
    F = fun oct_background:get_user_access_token/1,
    build_instantiator(Scenario, [
        ?_assertEqual(?USER_1_ACCESS_TOKEN, F(?USER_1_PLACEHOLDER)),
        ?_assertEqual(?USER_2_ACCESS_TOKEN, F(?USER_2_ID))
    ], [], [
        ?_assertEqual(?USER_3_ACCESS_TOKEN, F(?USER_3_PLACEHOLDER))
    ])
end).


get_user_user_session_id_test_() -> foreach_ct_config_(fun(Scenario) ->
    F = fun oct_background:get_user_session_id/2,
    build_instantiator(Scenario, [
        ?_assertEqual(?USER_1_KRAKOW_SESSION, F(?USER_1_PLACEHOLDER, ?PROVIDER_KRAKOW_PLACEHOLDER)),
        ?_assertEqual(?USER_2_KRAKOW_SESSION, F(?USER_2_ID, ?PROVIDER_KRAKOW_ID))
    ], [], [
        ?_assertEqual(?USER_2_PARIS_SESSION, F(?USER_2_ID, ?PROVIDER_PARIS_PLACEHOLDER)),
        ?_assertEqual(?USER_3_PARIS_SESSION, F(?USER_3_PLACEHOLDER, ?PROVIDER_PARIS_ID))
    ])
end).


get_space_id_test_() -> foreach_ct_config_(fun(Scenario) ->
    F = fun oct_background:get_space_id/1,
    build_instantiator(Scenario, [
        ?_assertEqual(?SPACE_1_ID, F(?SPACE_1_PLACEHOLDER)),
        ?_assertEqual(?SPACE_1_ID, F(?SPACE_1_ID)),
        ?_assertEqual(?SPACE_2_ID, F(?SPACE_2_PLACEHOLDER)),
        ?_assertEqual(?SPACE_2_ID, F(?SPACE_2_ID))
    ], [], [
        ?_assertEqual(?SPACE_3_ID, F(?SPACE_3_PLACEHOLDER)),
        ?_assertEqual(?SPACE_3_ID, F(?SPACE_3_ID))
    ])
end).


get_space_supporting_providers_test_() -> foreach_ct_config_(fun(Scenario) ->
    F = fun oct_background:get_space_supporting_providers/1,
    build_instantiator(Scenario, [
        ?_assertEqual([?PROVIDER_KRAKOW_ID], F(?SPACE_1_PLACEHOLDER)),
        ?_assertEqual([?PROVIDER_KRAKOW_ID], F(?SPACE_1_ID))
    ], [
        ?_assertEqual([?PROVIDER_KRAKOW_ID], F(?SPACE_2_PLACEHOLDER)),
        ?_assertEqual([?PROVIDER_KRAKOW_ID], F(?SPACE_2_ID))
    ], [
        ?_assertEqual([?PROVIDER_KRAKOW_ID, ?PROVIDER_PARIS_ID], F(?SPACE_2_PLACEHOLDER)),
        ?_assertEqual([?PROVIDER_PARIS_ID], F(?SPACE_3_ID))
    ])
end).


get_group_id_test_() -> foreach_ct_config_(fun(Scenario) ->
    F = fun oct_background:get_group_id/1,
    build_instantiator(Scenario, [
        ?_assertEqual(?GROUP_1_ID, F(?GROUP_1_PLACEHOLDER)),
        ?_assertEqual(?GROUP_1_ID, F(?GROUP_1_ID))
    ], [], [
        ?_assertEqual(?GROUP_2_ID, F(?GROUP_2_PLACEHOLDER)),
        ?_assertEqual(?GROUP_2_ID, F(?GROUP_2_ID))
    ])
end).


get_group_name_test_() -> foreach_ct_config_(fun(Scenario) ->
    F = fun oct_background:get_group_name/1,
    build_instantiator(Scenario, [
        ?_assertEqual(?GROUP_1_NAME, F(?GROUP_1_PLACEHOLDER)),
        ?_assertEqual(?GROUP_1_NAME, F(?GROUP_1_ID))
    ], [], [
        ?_assertEqual(?GROUP_2_NAME, F(?GROUP_2_PLACEHOLDER)),
        ?_assertEqual(?GROUP_2_NAME, F(?GROUP_2_ID))
    ])
end).


%%%===================================================================
%%% Helper functions
%%%===================================================================


%% @private
-spec foreach_ct_config_(list()) -> list().
foreach_ct_config_(TestsGenerator) ->
    [[{setup, build_setup_fun(Scenario), fun cleanup/1, TestsGenerator}] || Scenario <- ["1op", "2op"]].


%% @private
-spec build_setup_fun(string()) -> string().
build_setup_fun(Scenario) ->
    fun() ->
        Config = case Scenario of
            "1op" ->
                ?CT_CONFIG_1_OP;
            "2op" ->
                ?CT_CONFIG_2_OP
        end,
        mock_rpc_logic(Scenario),
        mock_token_logic(),
        node_cache:init(),
        ResultConfig = oct_background:init_per_suite(Config, #onenv_test_config{
            onenv_scenario = Scenario
        }),
        Posthook = test_config:get_custom(ResultConfig, env_up_posthook),
        Posthook(Config),
        Scenario
    end.


-spec cleanup(term()) -> ok.
%% @private
cleanup(_) ->
    meck:unload(opw_test_rpc),
    meck:unload(ozw_test_rpc),
    meck:unload(panel_test_rpc),
    meck:unload(tokens),
    node_cache:destroy().


%% @private
-spec build_instantiator(string(), list(), list(), list()) -> list().
build_instantiator("1op", CommonTestCases, CtConfig1OpTests, _CtConfig2OpTestCases) ->
    CommonTestCases ++ CtConfig1OpTests;
build_instantiator("2op", CommonTestCases, _CtConfig1OpTests, CtConfig2OpTestCases) ->
    CommonTestCases ++ CtConfig2OpTestCases.


%% @private
-spec mock_token_logic() -> ok.
mock_token_logic() ->
    meck:new(tokens, [unstick]),
    meck:expect(tokens, serialize, fun(Token) -> {ok, Token} end).


%% @private
-spec mock_rpc_logic(list()) -> ok.
mock_rpc_logic(Scenario) ->
    meck:new(opw_test_rpc, [unstick]),
    meck:new(ozw_test_rpc, [unstick]),
    meck:new(panel_test_rpc, [unstick]),

    meck:expect(opw_test_rpc, get_provider_id, fun(NodeSelector) ->
        case NodeSelector of
            ?KRAKOW_WORKER -> ?PROVIDER_KRAKOW_ID;
            ?PARIS_WORKER -> ?PROVIDER_PARIS_ID
        end
    end),

    meck:expect(opw_test_rpc, get_provider_name, fun(NodeSelector) ->
        case NodeSelector of
            ?KRAKOW_WORKER -> ?PROVIDER_KRAKOW_NAME;
            ?PARIS_WORKER -> ?PROVIDER_PARIS_NAME
        end
    end),

    meck:expect(opw_test_rpc, get_storages, fun(NodeSelector) ->
        case NodeSelector of
            ?KRAKOW_WORKER -> [?STORAGE_1_ID, ?STORAGE_2_ID];
            ?PARIS_WORKER -> [?STORAGE_3_ID, ?STORAGE_4_ID]
        end
    end),

    meck:expect(opw_test_rpc, get_provider_eff_users, fun(NodeSelector) ->
        case NodeSelector of
            ?KRAKOW_WORKER -> [?USER_1_ID, ?USER_2_ID];
            ?PARIS_WORKER -> [?USER_2_ID, ?USER_3_ID]
        end
    end),

    meck:expect(opw_test_rpc, get_spaces, fun(NodeSelector) ->
        case NodeSelector of
            ?KRAKOW_WORKER -> [?SPACE_1_ID, ?SPACE_2_ID];
            ?PARIS_WORKER -> [?SPACE_2_ID, ?SPACE_3_ID]
        end
    end),

    meck:expect(opw_test_rpc, build_token_credentials, fun(_, _, _, _, _, _) ->
        <<"someTokenCredentials">>
    end),

    meck:expect(opw_test_rpc, create_fuse_session, fun(NodeSelector, _, {_, _, _, UserId}, _) ->
        case {NodeSelector, UserId} of
            {?KRAKOW_WORKER, ?USER_1_ID} -> ?USER_1_KRAKOW_SESSION;
            {?KRAKOW_WORKER, ?USER_2_ID} -> ?USER_2_KRAKOW_SESSION;
            {?PARIS_WORKER, ?USER_2_ID} -> ?USER_2_PARIS_SESSION;
            {?PARIS_WORKER, ?USER_3_ID} -> ?USER_3_PARIS_SESSION
        end
    end),

    meck:expect(ozw_test_rpc, list_users, fun(_) ->
        case Scenario of
            "1op" -> [?USER_1_ID, ?USER_2_ID];
            "2op" -> [?USER_1_ID, ?USER_2_ID, ?USER_3_ID]
        end
    end),

    meck:expect(ozw_test_rpc, get_user_protected_data, fun(_, UserId) ->
        case UserId of
            ?USER_1_ID -> #{<<"username">> => ?USER_1_NAME, <<"fullName">> => ?USER_1_FULLNAME};
            ?USER_2_ID -> #{<<"username">> => ?USER_2_NAME, <<"fullName">> => ?USER_2_FULLNAME};
            ?USER_3_ID -> #{<<"username">> => ?USER_3_NAME, <<"fullName">> => ?USER_3_FULLNAME}
        end
    end),

    meck:expect(ozw_test_rpc, list_spaces, fun(_) ->
        case Scenario of
            "1op" -> [?SPACE_1_ID, ?SPACE_2_ID];
            "2op" -> [?SPACE_1_ID, ?SPACE_2_ID, ?SPACE_3_ID]
        end
    end),

    meck:expect(ozw_test_rpc, get_space_protected_data, fun(_, _, SpaceId) ->
        case SpaceId of
            ?SPACE_1_ID ->
                #{
                    <<"name">> => ?SPACE_1_NAME,
                    <<"providers">> => #{?PROVIDER_KRAKOW_ID => ?DUMMY_SUPPORT_SIZE}
                };
            ?SPACE_2_ID ->
                BasicSpace2Info = #{<<"name">> => ?SPACE_2_NAME},

                case Scenario of
                    "1op" ->
                        BasicSpace2Info#{<<"providers">> => #{
                            ?PROVIDER_KRAKOW_ID => ?DUMMY_SUPPORT_SIZE
                        }};
                    "2op" ->
                        BasicSpace2Info#{<<"providers">> => #{
                            ?PROVIDER_KRAKOW_ID => ?DUMMY_SUPPORT_SIZE,
                            ?PROVIDER_PARIS_ID => ?DUMMY_SUPPORT_SIZE
                        }}
                end;
            ?SPACE_3_ID ->
                #{
                    <<"name">> => ?SPACE_3_NAME,
                    <<"providers">> => #{?PROVIDER_PARIS_ID => ?DUMMY_SUPPORT_SIZE}
                }
        end
    end),

    meck:expect(ozw_test_rpc, list_groups, fun(_) ->
        case Scenario of
            "1op" -> [?GROUP_1_ID];
            "2op" -> [?GROUP_1_ID, ?GROUP_2_ID]
        end
    end),

    meck:expect(ozw_test_rpc, get_group_protected_data, fun(_, _, GroupId) ->
        case GroupId of
            ?GROUP_1_ID -> #{<<"name">> => ?GROUP_1_NAME};
            ?GROUP_2_ID -> #{<<"name">> => ?GROUP_2_NAME}
        end
    end),

    meck:expect(ozw_test_rpc, timestamp_seconds, fun(_) ->
        1000000
    end),

    meck:expect(ozw_test_rpc, create_user_temporary_token, fun(_, _, UserId, _) ->
        case UserId of
            ?USER_1_ID -> ?USER_1_ACCESS_TOKEN;
            ?USER_2_ID -> ?USER_2_ACCESS_TOKEN;
            ?USER_3_ID -> ?USER_3_ACCESS_TOKEN
        end
    end),

    meck:expect(panel_test_rpc, get_op_worker_nodes, fun(NodeSelector) ->
        case NodeSelector of
            ?KRAKOW_PANEL -> ['op_worker@dev-oneprovider-krakow-0.dev-oneprovider-krakow.default.svc.cluster.local'];
            ?PARIS_PANEL -> ['op_worker@dev-oneprovider-paris-0.dev-oneprovider-paris.default.svc.cluster.local']
        end
    end).


-endif.
