#pragma once

#include <folly/concurrency/ConcurrentHashMap.h>
#include <folly/synchronization/Baton.h>
#include <wangle/channel/Handler.h>
#include <wangle/channel/Pipeline.h>
#include <wangle/service/Service.h>

#include <thread>
#include <utility>

#include "client.hpp"
#include "common.hpp"
#include "notifyAggregator.hpp"
#include "proto/control.pb.h"
#include "storageHelperFactory.hpp"

namespace rtransfer {
constexpr int kMaxPortNumber = 65535;
class Service;

namespace link_control {
namespace proto {
using RequestPtr = std::unique_ptr<Request>;
using ResponsePtr = std::unique_ptr<Response>;
}  // namespace proto

using Pipeline = wangle::Pipeline<folly::IOBufQueue &, proto::ResponsePtr>;

/**
 * @c ControlService is a controlling service of the whole executable.
 * The assumption is that there is only one ControlService instance. As of
 * writing this documentation, the instance is driven by a pipeline connected to
 * stdin/stdout.
 * The @c ControlService handles protocol messages in package
 * `rtransfer::link_control::proto`.
 */
class ControlService
    : public wangle::Service<proto::RequestPtr, proto::ResponsePtr> {
public:
    ControlService(StorageHelperFactory &factory, ClientLinksMap &clientLinks,
        StoragesMap &storages, SecretManager &secretManager)
        : helperFactory_{factory}
        , clientLinks_{clientLinks}
        , storages_{storages}
        , secretManager_{secretManager}
    {
    }

    /**
     * Closes all client links associated with this service.
     */
    ~ControlService() override;

    ControlService(const ControlService &) = delete;
    ControlService(ControlService &&) = delete;
    ControlService &operator=(const ControlService &) = delete;
    ControlService &operator=(ControlService &&) = delete;

    folly::Future<proto::ResponsePtr> operator()(
        proto::RequestPtr request) override;

    /**
     * Sets a controlling pipeline.
     * Setting the controlling pipeline is needed so that the service can output
     * data that is not a response to a previous request.
     * This cannot/shouldn't be done in constructor, as a @c
     * wangle::StaticPipeline would only be constructed after the @c
     * ControlService object.
     */
    void setPipeline(Pipeline *pipeline) { pipeline_ = pipeline; }

    /**
     * Provides a mechanism to "ask" something of the connected provider, i.e.
     * to send a request on stdout and wait for a reply that will come on stdin.
     * The main use of this mechanism is authorization of incoming fetch
     * requests.
     */
    folly::Future<proto::RequestPtr> ask(proto::ResponsePtr request);

    void registerService(std::weak_ptr<rtransfer::Service> srvc);

private:
    folly::Future<proto::ResponsePtr> doAction(proto::RequestPtr request);

    static folly::Future<proto::ResponsePtr> error(
        folly::StringPiece description, const std::string &reqId = {});

    static folly::Future<proto::ResponsePtr> done(folly::Unit &&unit = {});

    folly::Future<proto::ResponsePtr> createHelper(proto::RequestPtr req);

    folly::Future<proto::ResponsePtr> connect(proto::RequestPtr req);

    folly::Future<proto::ResponsePtr> fetch(proto::RequestPtr req);

    folly::Future<proto::ResponsePtr> cancel(proto::RequestPtr req);

    folly::Future<proto::ResponsePtr> allowConnection(proto::RequestPtr req);

    folly::Future<proto::ResponsePtr> memoryStats(proto::RequestPtr req);

    void notify(folly::fbstring reqId, std::uint64_t offset, std::size_t size);

    StorageHelperFactory &helperFactory_;
    ClientLinksMap &clientLinks_;
    std::vector<std::weak_ptr<rtransfer::Service>> services_;
    StoragesMap &storages_;
    folly::ConcurrentHashMap<folly::fbstring,
        std::shared_ptr<folly::Promise<proto::RequestPtr>>>
        questions_;
    std::shared_ptr<folly::IOThreadPoolExecutor> clientExecutor_ =
        std::make_shared<folly::IOThreadPoolExecutor>(
            std::thread::hardware_concurrency(),
            std::make_shared<folly::NamedThreadFactory>("ClientPool"));
    SecretManager &secretManager_;
    Pipeline *pipeline_ = nullptr;

    NotifyAggregator notifyAggregator_{
        [this](folly::fbstring reqId, std::uint64_t offset, std::size_t size) {
            notify(std::move(reqId), offset, size);
        }};
};

Pipeline::Ptr newPipeline(ControlService &service, folly::Baton<> &stopBaton);

}  // namespace link_control
}  // namespace rtransfer
