#pragma once

#include <folly/FBString.h>
#include <folly/Function.h>
#include <folly/Synchronized.h>
#include <folly/executors/CPUThreadPoolExecutor.h>
#include <folly/executors/SerialExecutor.h>
#include <folly/executors/thread_factory/NamedThreadFactory.h>
#include <folly/io/IOBufQueue.h>
#include <helpers/storageHelper.h>

#include <atomic>
#include <chrono>
#include <memory>

#include "chainQueue.hpp"
#include "openType.hpp"

namespace rtransfer {

/**
 * @c Storage maintains two queues - read and write queues. The queues are
 * filled with @c Storage::WriteRequest and @c Storage::ReadRequest .
 * There are multiple queues per storage - a pair per a "bucket".
 *
 * A bucket is just a unit of concurrency here - for storage helpers that don't
 * support concurrent operations on their file handles (think old implementation
 * of `PosixHelper`), there are at most as many operations in progress as there
 * are buckets. Inside a bucket, a next operation from a queue is only processed
 * when the previous one ends. This obviously introduces a lot of
 * synchronization overhead, which is why it's best if the filehandles support
 * concurrent operations (`isConcurrencyEnabled() == true`).
 *
 * Operations are hashed to a specific bucket based on their file handle.
 */
class Storage {
public:
    struct ReadRequest {
        std::uint64_t reqId;
        folly::Promise<folly::IOBufQueue> promise;
        one::helpers::FileHandlePtr handle;
        std::uint64_t offset;
        std::size_t size;

        std::size_t getSize() const { return size; }
    };

    struct WriteRequest {
        std::uint64_t reqId;
        folly::Promise<folly::Unit> promise;
        one::helpers::FileHandlePtr handle;
        std::uint64_t offset;
        folly::IOBufQueue buf;

        std::size_t getSize() const { return buf.chainLength(); }
    };

    struct Bucket {
        template <typename Request>
        struct Queue {
            std::mutex mutex;
            bool isUsed = false;
            ChainQueue<Request> impl;
        };

        explicit Bucket(std::shared_ptr<folly::Executor> executor_)
            : executor{folly::SerialExecutor::create(
                  folly::getKeepAliveToken(executor_.get()))}
        {
        }

        folly::Executor::KeepAlive<folly::SerialExecutor> executor;
        Queue<ReadRequest> readQueue;
        Queue<WriteRequest> writeQueue;
    };

    Storage(const folly::fbstring &storageId,
        one::helpers::StorageHelperPtr storageHelper);

    folly::Future<folly::IOBufQueue> read(
        one::helpers::FileHandlePtr /*handle*/, std::uint64_t reqId,
        std::uint64_t offset, std::size_t size, std::uint8_t priority);

    folly::Future<folly::Unit> write(one::helpers::FileHandlePtr /*handle*/,
        std::uint64_t reqId, std::uint64_t offset, folly::IOBufQueue buf,
        std::uint8_t priority);

    folly::Future<one::helpers::FileHandlePtr> open(
        const folly::fbstring &fileId, OpenType openType);

    one::helpers::StorageHelperPtr helper() { return storageHelper_; }

    const folly::fbstring &storageId() const { return storageId_; }

private:
    std::string m(const char * /*name*/);

    template <typename Req, typename Cb>
    auto inBucket(Req &&req, Cb &&cb);

    folly::fbstring storageId_;
    folly::fbstring storageIdHex_;
    one::helpers::StorageHelperPtr storageHelper_;

    constexpr static const std::size_t maxInProgress_{100};

    folly::fbvector<std::unique_ptr<Bucket>> buckets_;
};

}  // namespace rtransfer
