#pragma once

#include <folly/Hash.h>
#include <folly/concurrency/ConcurrentHashMap.h>
#include <folly/executors/SerialExecutor.h>
#include <folly/futures/Future.h>
#include <wangle/channel/Handler.h>
#include <wangle/service/Service.h>

#include <chrono>
#include <unordered_map>

#include "common.hpp"
#include "periodicHandler.hpp"
#include "shaper.hpp"
#include "shaperTimer.hpp"

namespace rtransfer {

class Reader;

/**
 * \uml{note[top]
 * ShaperMap is just a collection of two tightly coupled hashmaps - a map of
 * shapers themselves, and a map of reqId -> shaper so that we can find which
 * shaper to pester about our new info about a request (most importantly: ack)
 * without sending extra information.}
 */
class ShaperMap {
public:
    explicit ShaperMap(Reader &reader);

    folly::Future<std::size_t> fetch(MsgPtr msg);

    folly::Future<folly::Unit> cancel(MsgPtr msg);

    void ack(MsgPtr msg);

private:
    /**
     * Maps shapers to keys generated from (src_id, dest_id, priority)
     */
    folly::ConcurrentHashMap<Shaper::ID, std::shared_ptr<Shaper>> shapers_;

    /**
     * Maps requests to shapers stored in shapers_. It is used to efficiently
     * retrieve a shaper for a specific request.
     */
    folly::ConcurrentHashMap<std::uint64_t, std::shared_ptr<Shaper>>
        reqToShaper_;

    Reader &reader_;

    ShaperTimer<std::chrono::steady_clock> shaperTimer_;
};

}  // namespace rtransfer
