#include "ssl.hpp"

#include <gflags/gflags.h>

DEFINE_bool(use_ssl, false, "use TLSv1.2 for client and server connections");
DEFINE_bool(ssl_no_verify_cert, false, "disable certificate verification");
DEFINE_string(ssl_cert_path, "", "path to SSL certificate used by server");
DEFINE_string(ssl_key_path, "", "path to SSL key file used by server");
DEFINE_string(ssl_password_path, "", "path to password file for SSL key file");
DEFINE_string(ssl_ca_path, "", "path to CA file for SSL clients");
DEFINE_string(ssl_ecc_curve_name, "", "name of ECC curve used by server");
DEFINE_string(
    ssl_cipher_list, "", "colon-separated list of ciphers accepted by server");

namespace rtransfer {

bool useSSL() { return FLAGS_use_ssl; }

wangle::SSLContextConfig sslContextConfig()
{
    wangle::SSLContextConfig sslCtx;
    sslCtx.isDefault = true;
    sslCtx.sslVersion = folly::SSLContext::TLSv1_2;
    sslCtx.clientVerification =
        folly::SSLContext::VerifyClientCertificate::DO_NOT_REQUEST;
    sslCtx.setCertificate(
        FLAGS_ssl_cert_path, FLAGS_ssl_key_path, FLAGS_ssl_password_path);
    if (!FLAGS_ssl_cipher_list.empty())
        sslCtx.sslCiphers = FLAGS_ssl_cipher_list;
    if (!FLAGS_ssl_ecc_curve_name.empty())
        sslCtx.eccCurveName = FLAGS_ssl_ecc_curve_name;
    return sslCtx;
}

folly::SSLContextPtr sslContext()
{
    auto sslCtx =
        std::make_shared<folly::SSLContext>(folly::SSLContext::TLSv1_2);

    if (!FLAGS_ssl_ca_path.empty())
        sslCtx->loadTrustedCertificates(FLAGS_ssl_ca_path.c_str());

    LOG_IF(WARNING, !SSL_CTX_set_default_verify_paths(sslCtx->getSSLCtx()))
        << "Failed to load default CA paths";

    if (FLAGS_ssl_no_verify_cert) {
        LOG(WARNING) << "Certificate verification is disabled";
        sslCtx->setVerificationOption(
            folly::SSLContext::SSLVerifyPeerEnum::NO_VERIFY);
    }
    else
        sslCtx->setVerificationOption(
            folly::SSLContext::SSLVerifyPeerEnum::VERIFY);

    return sslCtx;
}

}  // namespace rtransfer
