#include "authCache.hpp"

#include <folly/executors/GlobalExecutor.h>
#include <gflags/gflags.h>

#include <chrono>

DEFINE_uint64(auth_cache_duration, 60000,
    "duration after which a valid authorization will be expired");
DEFINE_uint64(auth_cache_tick, 1000,
    "cache will be pruned every auth_cache_tick milliseconds");

namespace rtransfer {

AuthCache::AuthCache(link_control::ControlService &controlService)
    : executor_{folly::SerialExecutor::create(folly::getKeepAliveToken(
          folly::getUnsafeMutableGlobalCPUExecutor().get()))}
    , controlService_{controlService}
    , periodicHandler_{std::chrono::milliseconds{FLAGS_auth_cache_tick},
          std::bind(&AuthCache::trigger, this)}
{
}

folly::Future<std::shared_ptr<link_control::proto::Request>>
AuthCache::authorize(folly::fbstring providerId, folly::fbstring transferData)
{
    VLOG(2) << "Scheduling authorization request with provider " << providerId;

    return via(executor_.get())
        .thenValue([this, providerId = std::move(providerId),
                       transferData = std::move(transferData)](
                       auto && /*unit*/) mutable {
            return doAuthorize(providerId, transferData);
        });
}

folly::Future<std::shared_ptr<link_control::proto::Request>>
AuthCache::doAuthorize(
    const folly::fbstring &providerId, const folly::fbstring &transferData)
{
    VLOG(2) << "Performing authorization request with provider " << providerId;

    auto key = std::make_pair(providerId, transferData);
    auto res = responses_.emplace(key, std::shared_ptr<Promise>{});
    if (res.second)
        res.first->second = authorizeWithServer(std::move(key));

    return res.first->second->getFuture();
}

std::shared_ptr<AuthCache::Promise> AuthCache::authorizeWithServer(Key key)
{
    auto promise = std::make_shared<Promise>();

    auto ask = std::make_unique<link_control::proto::Response>();
    auto *authRequest = ask->mutable_auth_request();
    authRequest->set_provider_id(key.first.data(), key.first.size());
    authRequest->set_transfer_data(key.second.data(), key.second.size());
    controlService_.ask(std::move(ask))
        .via(executor_.get())
        .thenTry([this, key, promise](
                     folly::Try<link_control::proto::RequestPtr>
                         &&maybeAuthResponse) {
            if (maybeAuthResponse.hasException()) {
                promise->setWith(
                    [&maybeAuthResponse]()
                        -> std::shared_ptr<link_control::proto::Request> {
                        throw std::runtime_error(
                            fmt::format("no authorization received: {}",
                                maybeAuthResponse.exception().what()));
                    });
                responses_.erase(key);
                return folly::makeFuture();
            }

            auto &authResponse = maybeAuthResponse.value();

            if (authResponse->payload_case() !=
                link_control::proto::Request::kAuthResponse) {
                LOG(ERROR) << "Internal authorization error";
                promise->setWith(
                    []() -> std::shared_ptr<link_control::proto::Request> {
                        throw std::runtime_error(
                            "invalid authorization message return by provider");
                    });
                responses_.erase(key);
                return folly::makeFuture();
            }

            if (!authResponse->auth_response().is_authorized()) {
                VLOG(2) << "Request unauthorized";
                promise->setWith(
                    []() -> std::shared_ptr<link_control::proto::Request> {
                        throw std::runtime_error("unauthorized");
                    });
                responses_.erase(key);
                return folly::makeFuture();
            }

            expirations_.emplace(
                ExpNode{key, std::chrono::steady_clock::now()});

            promise->setValue(std::make_shared<link_control::proto::Request>(
                std::move(*authResponse)));

            return folly::makeFuture();
        });

    return promise;
}

void AuthCache::trigger()
{
    executor_->add([this] { doTrigger(); });
}

void AuthCache::doTrigger()
{
    auto exp = std::chrono::steady_clock::now() -
               std::chrono::milliseconds{FLAGS_auth_cache_duration};

    while (!expirations_.empty() && expirations_.front().addTimePoint < exp) {
        responses_.erase(expirations_.front().key);
        expirations_.pop();
    }
}

}  // namespace rtransfer
