#include <folly/ConcurrentSkipList.h>
#include <folly/concurrency/ConcurrentHashMap.h>
#include <folly/executors/CPUThreadPoolExecutor.h>
#include <folly/executors/GlobalExecutor.h>
#include <folly/init/Init.h>
#include <folly/synchronization/Baton.h>
#include <gflags/gflags.h>
#include <glog/logging.h>
#include <google/protobuf/util/json_util.h>
#include <helpers/init.h>
#include <monitoring/monitoring.h>

#include <csignal>
#include <iostream>

#include "client.hpp"
#include "common.hpp"
#include "controlService.hpp"
#include "secretManager.hpp"
#include "server.hpp"
#include "serverSideLink.hpp"
#include "service.hpp"
#include "ssl.hpp"
#include "storageHelperFactory.hpp"

static bool validatePort(const char *flagname, std::int32_t value)
{
    if (value > 0 && value < rtransfer::kMaxPortNumber)
        return true;

    std::cerr << "Invalid value for --" << flagname << ": " << value
              << std::endl;
    return false;
}

DEFINE_int32(server_port, 5555, "data port");
DEFINE_validator(server_port, &validatePort);
DEFINE_int32(helper_workers, 100, "number of threads handling helpers' IO");

DEFINE_string(graphite_url, "", "url to Graphite monitoring server");
DEFINE_string(graphite_namespace_prefix, "rtransfer_link",
    "prefix for monitoring metrics");
DEFINE_uint64(
    graphite_reporting_period, 30, "reporting period for metrics (in seconds)");
DEFINE_bool(graphite_reporting_full, false, "enable full metrics reporting");
DEFINE_bool(log_read_write_perf, false, "enable read/write performance logger");

// NOLINTNEXTLINE(bugprone-exception-escape)
int main(int argc, char **argv)
{
    gflags::ParseCommandLineFlags(&argc, &argv, true);
    folly::Init follyInit{&argc, &argv};

    auto executor = std::make_shared<folly::CPUThreadPoolExecutor>(
        std::thread::hardware_concurrency());

    folly::setUnsafeMutableGlobalCPUExecutor(executor);

    one::helpers::init();

    if (FLAGS_log_read_write_perf) {
        one::helpers::startReadWritePerfLogger(FLAGS_log_dir);
    }

    if (!FLAGS_graphite_url.empty()) {
        auto config = std::make_shared<
            one::monitoring::GraphiteMonitoringConfiguration>();
        config->fromGraphiteURL(FLAGS_graphite_url);
        config->namespacePrefix = FLAGS_graphite_namespace_prefix;
        config->reportingPeriod = FLAGS_graphite_reporting_period;
        config->reportingLevel = FLAGS_graphite_reporting_full
                                     ? cppmetrics::core::ReportingLevel::Full
                                     : cppmetrics::core::ReportingLevel::Basic;

        one::helpers::configureMonitoring(config, true);
    }

    rtransfer::SecretManager secretManager;
    rtransfer::ServerLinksMap serverLinks;
    rtransfer::ClientLinksMap clientLinks;
    rtransfer::StoragesMap storages;
    rtransfer::StorageHelperFactory helperFactory{
        FLAGS_helper_workers, clientLinks, serverLinks};

    // TODO: have object responsible for maintaining storagemap using
    //       helperFactory
    rtransfer::link_control::ControlService controlService{
        helperFactory, clientLinks, storages, secretManager};

    rtransfer::ServiceFactory serviceFactory{storages, controlService};
    rtransfer::ServerSideLinkFactory serverSideLinkFactory{serverLinks};

    auto serverSocketConfig = std::make_shared<wangle::ServerSocketConfig>();
    if (rtransfer::useSSL())
        serverSocketConfig->sslContextConfigs.emplace_back(
            rtransfer::sslContextConfig());

    auto server =
        rtransfer::server::makeServer(FLAGS_server_port, serverSideLinkFactory,
            serviceFactory, secretManager, serverSocketConfig);

    folly::Baton<> stopBaton;
    auto controlPipeline =
        rtransfer::link_control::newPipeline(controlService, stopBaton);
    stopBaton.wait();
}
