#pragma once

#include <folly/io/IOBufQueue.h>
#include <folly/io/async/EventBaseThread.h>
#include <folly/io/async/ScopedEventBaseThread.h>
#include <wangle/channel/Handler.h>

#include <iostream>

namespace rtransfer {

/**
 * A Wangle handler translating stdio/stdout into Wangle framework.
 * We use that so that @c link_control::ControlService can be a @c
 * wangle::Service and we don't have to come up with too much custom machinery.
 * Reading from stdin is blocking which is why it has to have its own thread.
 * Concidentally, this thread also drives @c link_control::ControlService
 * operations - but they are quite short, quickly scheduling more work on thread
 * pools.
 */
class StdIOHandler : public wangle::BytesToBytesHandler {
public:
    StdIOHandler()
    {
        reader_.getEventBase()->runInEventBaseThread(
            [this] { scheduleRead(); });
    }

    ~StdIOHandler() override { reader_.getEventBase()->terminateLoopSoon(); }

    StdIOHandler(const StdIOHandler &) = delete;
    StdIOHandler(StdIOHandler &&) = delete;
    StdIOHandler &operator=(const StdIOHandler &) = delete;
    StdIOHandler &operator=(StdIOHandler &&) = delete;

    folly::EventBase *getEventBase() { return reader_.getEventBase(); }

    folly::Future<folly::Unit> write(
        Context * /*ctx*/, std::unique_ptr<folly::IOBuf> buf) override
    {
        return via(writer_.getEventBase())
            .thenValue([buf = std::move(buf)](auto && /*unit*/) mutable {
                for (auto b : *buf)
                    std::cout.write(
                        // NOLINTNEXTLINE
                        reinterpret_cast<const char *>(b.data()), b.size());
                std::cout.flush();
            });
    }

private:
    void readFromStdin()
    {
        auto alloc = buf_.preallocate(1024, 2048);
        std::cin.getline(static_cast<char *>(alloc.first), alloc.second);
        if (std::cin.eof()) {
            getContext()->fireReadEOF();
            reader_.getEventBase()->terminateLoopSoon();
            return;
        }
        if (std::cin.bad()) {
            getContext()->fireReadException(
                folly::make_exception_wrapper<std::runtime_error>(
                    "failed to read data from stdin"));
            reader_.getEventBase()->terminateLoopSoon();
            return;
        }

        std::size_t readBytes = std::cin.gcount();

        if (std::cin.fail()) {
            // Read full buffer without encountering a newline
            std::cin.clear();
        }
        else {
            // getline does not save the newline character
            // NOLINTNEXTLINE
            static_cast<char *>(alloc.first)[readBytes - 1] = '\n';
        }

        if (readBytes > 0) {
            buf_.postallocate(readBytes);
            getContext()->fireRead(buf_);
        }

        scheduleRead();
    }

    void scheduleRead()
    {
        reader_.getEventBase()->runInLoop(
            std::bind(&StdIOHandler::readFromStdin, this));
    }

    folly::IOBufQueue buf_{folly::IOBufQueue::cacheChainLength()};
    folly::ScopedEventBaseThread reader_{"stdIOread"};
    folly::ScopedEventBaseThread writer_{"stdIOwrite"};
};

}  // namespace rtransfer
