#pragma once

#include <folly/FBString.h>
#include <folly/FBVector.h>
#include <folly/executors/IOThreadPoolExecutor.h>
#include <folly/system/ThreadName.h>
#include <helpers/storageHelper.h>
#include <helpers/storageHelperCreator.h>

#include <thread>

#include "common.hpp"
#include "proto/control.pb.h"
#include "serverSideLink.hpp"
#include "storage.hpp"

namespace rtransfer {

/**
 * @c StorageHelperFactory creates new @c Storage objects
 * using locally managed thread pool as workers for the underlying @c
 * one::helpers::StorageHelper objects.
 */
class StorageHelperFactory {
public:
    /**
     * Constructor.
     * @param threads Number of local threads used for created storage helpers.
     */
    StorageHelperFactory(
        int threads, ClientLinksMap &clientLinks, ServerLinksMap &serverLinks)
        : helperIOExecutor_{std::make_shared<folly::IOThreadPoolExecutor>(
              threads)}
        , clientLinks_{clientLinks}
        , serverLinks_{serverLinks}
    {
    }

    /**
     * Destructor.
     * Stops the worker threads used for storage helpers.
     */
    ~StorageHelperFactory() { helperIOExecutor_->stop(); }

    StorageHelperFactory(const StorageHelperFactory &) = delete;

    StorageHelperFactory(StorageHelperFactory &&) = delete;

    StorageHelperFactory &operator=(const StorageHelperFactory &) = delete;

    StorageHelperFactory &operator=(StorageHelperFactory &&) = delete;

    /**
     * Creates a new @c Storage instance.
     * @param req A message containing parameters used to initialize the @c
     * one::helpers::StorageHelper object underlying the Storage instance.
     */
    std::shared_ptr<Storage> operator()(
        const link_control::proto::CreateHelper &req)
    {
        std::unordered_map<folly::fbstring, folly::fbstring> params;
        for (const auto &param : req.params())
            params.emplace(
                folly::fbstring{param.key()}, folly::fbstring{param.value()});

        auto helper = helperCreator_.getStorageHelper(
            req.name(), params, req.io_buffered());

        return std::make_shared<Storage>(req.storage_id(), std::move(helper));
    }

private:
    std::shared_ptr<folly::IOThreadPoolExecutor> helperIOExecutor_;

    ClientLinksMap &clientLinks_;
    ServerLinksMap &serverLinks_;

    one::helpers::StorageHelperCreator<void> helperCreator_{helperIOExecutor_,
        helperIOExecutor_, helperIOExecutor_, helperIOExecutor_,
        helperIOExecutor_, helperIOExecutor_, helperIOExecutor_,
        helperIOExecutor_, helperIOExecutor_, helperIOExecutor_};
};

}  // namespace rtransfer
