"""This module contains test cases for transfer to s3 storage"""

__author__ = "Bartek Kryza"
__copyright__ = """(C) 2023 ACK CYFRONET AGH,
This software is released under the MIT license cited in 'LICENSE.txt'."""

import os
import time

from test_common import *


def test_refresh_helper_params_null_to_s3(rtransfer, request, s3_server, src_dir):
    # Create temporary file with random data
    random_block = os.urandom(1234*1234)
    file_block_count = 250

    file_size = file_block_count * len(random_block)

    # Start rtransfer instances
    (dest, src) = rtransfer

    dest_storage_id = create_s3_helper("dest_s3", dest,
                                       s3_server.scheme,
                                       s3_server.hostname,
                                       s3_server.access_key,
                                       s3_server.secret_key,
                                       'no_such_bucket',
                                       s3_server.prefix,
                                       s3_server.block_size)

    src_storage_id = create_null_helper("src_null", src)
    conns = connect_links(dest, src)

    wait_for_connection()

    fetch_offset = 0
    fetch_size = file_size

    srcfd = to_b64("file1.dat")
    destfd = to_b64("file1.dat")

    priority = 1

    req_id = do_fetch(dest, src, conns[0], src_storage_id, srcfd, dest_storage_id,
                      destfd, fetch_offset, fetch_size, priority)
    retries = 3
    parameters_changed = False
    while True:
        resp = dest.get_response(skip_updates=False)
        if "isUpdate" in resp and resp["isUpdate"]:
            # Check that blocks are transferred at fixed size
            # matching the destination storage size
            assert int(resp["progress"].get("offset", 0)) % s3_server.block_size == 0
        elif "cancel"in resp:
            continue
        elif "error" in resp and retries >= 0 and \
                ("The specified bucket does not exist" in resp["error"]["description"]):
                if not parameters_changed:
                    # Update helper params with correct bucket name
                    create_s3_helper("dest_s3", dest,
                                     s3_server.scheme,
                                     s3_server.hostname,
                                     s3_server.access_key,
                                     s3_server.secret_key,
                                     s3_server.bucket,
                                     s3_server.prefix,
                                     s3_server.block_size)
                    parameters_changed = True
                retries = retries - 1
                time.sleep(5)
                req_id = do_fetch(dest, src, conns[0], src_storage_id, srcfd, dest_storage_id,
                                  destfd, fetch_offset, fetch_size, priority, True)
        else:
            assert fetch_size == int(resp["wrote"])
            break


def test_refresh_helper_params_s3_to_null(rtransfer, request, s3_server, src_dir):
    # Create temporary file with random data
    file_block_count = 2

    file_size = file_block_count * s3_server.block_size

    file_id = random_str()

    # Create source file on S3
    s3_server.create_file(file_id, file_block_count)

    # Start rtransfer instances
    (dest, src) = rtransfer

    src_storage_id = create_s3_helper("src_s3", src,
                                       s3_server.scheme,
                                       s3_server.hostname,
                                       s3_server.access_key,
                                       s3_server.secret_key,
                                       s3_server.bucket,
                                       s3_server.prefix,
                                       s3_server.block_size)

    dest_storage_id = create_null_helper("dest_null", dest)
    conns = connect_links(dest, src)

    wait_for_connection()

    fetch_offset = 0
    fetch_size = file_size

    srcfd = to_b64("file1.dat")
    destfd = to_b64("file1.dat")

    priority = 1

    req_id = do_fetch(dest, src, conns[0], src_storage_id, srcfd, dest_storage_id,
                      destfd, fetch_offset, fetch_size, priority)

    retries = 2
    parameters_changed = False
    while True:
        resp = dest.get_response(skip_updates=False)
        if "isUpdate" in resp and resp["isUpdate"]:
            continue
        elif "cancel" in resp:
            continue
        elif "error" in resp and retries >= 0 and \
                ("The specified bucket does not exist" in resp["error"]["description"]):
                if not parameters_changed:
                    # Update helper params with correct bucket name
                    create_s3_helper("src_s3", src,
                                     s3_server.scheme,
                                     s3_server.hostname,
                                     s3_server.access_key,
                                     s3_server.secret_key,
                                     s3_server.bucket,
                                     s3_server.prefix,
                                     s3_server.block_size)
                    parameters_changed = True
                retries = retries - 1
                time.sleep(5)
                req_id = do_fetch(dest, src, conns[0], src_storage_id, srcfd, dest_storage_id,
                                  destfd, fetch_offset, fetch_size, priority, True)
        else:
            assert fetch_size == int(resp["wrote"])
            break


def test_slow_null_to_s3_storage_full_transfer(rtransfer, request, s3_server, src_dir):
    # Create temporary file with random data
    random_block = os.urandom(1234*1234)
    file_block_count = 250

    file_size = file_block_count * len(random_block)

    file_id = random_str()

    with open(os.path.join(src_dir, file_id), 'wb') as file1:
        for _ in range(0, file_block_count):
            file1.write(random_block)

    # Start rtransfer instances
    (dest, src) = rtransfer

    dest_storage_id = create_s3_helper("dest_s3", dest,
                                       s3_server.scheme,
                                       s3_server.hostname,
                                       s3_server.access_key,
                                       s3_server.secret_key,
                                       s3_server.bucket,
                                       s3_server.prefix,
                                       s3_server.block_size)
    src_storage_id = create_slow_null_helper("src_slow_null", src)
    conns = connect_links(dest, src)

    wait_for_connection()

    fetch_offset = 0
    fetch_size = file_size

    srcfd = to_b64("file1.dat")
    destfd = to_b64("file1.dat")

    priority = 1

    do_fetch(dest, src, conns[0], src_storage_id, srcfd, dest_storage_id,
             destfd, fetch_offset, fetch_size, priority)

    while True:
        resp = dest.get_response(skip_updates=False)
        if "isUpdate" in resp and resp["isUpdate"]:
            # Check that blocks are transferred at fixed size
            # matching the destination storage size
            assert int(resp["progress"].get("offset", 0)) % s3_server.block_size == 0
        else:
            assert fetch_size == int(resp["wrote"])
            break


def test_posix_to_s3_storage_full_transfer(rtransfer, s3_server, src_dir):
    #
    # Create temporary file with random data
    #
    random_block = os.urandom(1000*1000)
    file_block_count = 100

    file_size = file_block_count * len(random_block)

    file_id = random_str()

    with open(os.path.join(src_dir, file_id), 'wb') as file1:
        for _ in range(0, file_block_count):
            file1.write(random_block)

    # Start rtransfer instances
    (dest, src) = rtransfer
    dest_storage_id = create_s3_helper("dest_s3", dest,
                                       s3_server.scheme,
                                       s3_server.hostname,
                                       s3_server.access_key,
                                       s3_server.secret_key,
                                       s3_server.bucket,
                                       s3_server.prefix,
                                       s3_server.block_size)
    src_storage_id = create_posix_helper("src_posix",
                                         src,
                                         str(src_dir))
    conns = connect_links(dest, src)

    wait_for_connection()

    fetch_offset = 0
    fetch_size = file_size

    srcfd = to_b64(file_id)
    destfd = to_b64(file_id)

    priority = 1

    do_fetch(dest, src, conns[0], src_storage_id, srcfd, dest_storage_id,
             destfd, fetch_offset, fetch_size, priority)

    while True:
        resp = dest.get_response(skip_updates=False)
        if "isUpdate" in resp and resp["isUpdate"]:
            # Check that blocks are transferred at fixed size
            # matching the destination storage size
            assert int(resp["progress"].get("offset", 0)) % s3_server.block_size == 0
        else:
            assert fetch_size == int(resp["wrote"])
            break

    #
    # Make sure that transferred file contents are the same as original
    #
    src_contents = open(os.path.join(src_dir, file_id),'rb').read()
    source_md5 = hashlib.md5(src_contents).hexdigest()

    dest_contents = s3_server.read_file_contents(file_id, file_size)
    dest_md5 = hashlib.md5(dest_contents).hexdigest()

    assert source_md5 == dest_md5
