%%%-------------------------------------------------------------------
%%% @author Michał Wrzeszcz
%%% @copyright (C) 2018 ACK CYFRONET AGH
%%% This software is released under the MIT license
%%% cited in 'LICENSE.txt'.
%%% @end
%%%-------------------------------------------------------------------
%%% @doc
%%% @end
%%%-------------------------------------------------------------------
-module(rtransfer_link_request_server).
-author("Michał Wrzeszcz").

-behaviour(gen_server).

%%%===================================================================
%%% Definitions
%%%===================================================================

-type state() :: #{
    monitor_pid := pid(),
    ref_map := #{rtransfer_link_port:req_id() => {reference(),
        rtransfer_link:notify_fun(), rtransfer_link:on_complete_fun(),
        fun(() -> any())}}
}.

%% The process is supposed to die after ?DIE_AFTER time of idling
%% (no requests in flight)
-define(DIE_AFTER, 300000).

%%%===================================================================
%%% Exports
%%%===================================================================

-export([start/0, fetch/3]).
-export([init/1, handle_info/2, handle_cast/2, code_change/3,
         terminate/2, handle_call/3]).

%%%===================================================================
%%% API
%%%===================================================================

-spec start() -> {ok, pid()} | {error, any()}.
start() ->
    gen_server2:start(?MODULE, {}, []).

-spec fetch(Pid :: pid(), ConnectionId :: binary(),
    Req :: rtransfer_link_request:t()) -> ok.
fetch(Pid, ConnectionId, Req) ->
    gen_server2:call(Pid, {fetch, ConnectionId, Req}, infinity).

%%%===================================================================
%%% gen_server callbacks
%%%===================================================================

-spec init(term()) -> {ok, state(), non_neg_integer()}.
init(_) ->
    {ok, Pid} = rtransfer_link_monitor:start_link(self()),
    {ok, #{monitor_pid => Pid, ref_map => #{}}, ?DIE_AFTER}.

-spec handle_call(term(), {pid(), term()}, state()) ->
                         {noreply, state(), non_neg_integer()}.
handle_call({fetch, ConnectionId, Req}, From,
    #{monitor_pid := MonitorPid, ref_map := RefMap} = State) ->
    gen_server:reply(From, ok),
    case rtransfer_link_request:do_fetch(ConnectionId, Req, MonitorPid) of
        {ok, ReqId, Ref, NotifyFun, OnCompleteFun} ->
            RefMap2 = maps:put(ReqId, {Ref, NotifyFun, OnCompleteFun},
                RefMap),
            {noreply, State#{ref_map := RefMap2}, ?DIE_AFTER};
        _ ->
            {noreply, State, ?DIE_AFTER}
    end.

-spec handle_info(term(), state()) -> {noreply, state(), non_neg_integer()} |
    {stop, {shutdown, timeout}, state()}.
handle_info({response, ReqId, Resp},
    #{monitor_pid := MonitorPid, ref_map := RefMap} = State) ->
    case maps:get(ReqId, RefMap, undefined) of
        undefined ->
            lager:error("Unknown request id: ~p", [ReqId]),
            {noreply, State, ?DIE_AFTER};
        {Ref, NotifyFun, OnCompleteFun} ->
            try
                case rtransfer_link_request:handle_fetch_updates(Resp, NotifyFun) of
                    #{<<"wrote">> := Wrote} ->
                        OnCompleteFun(Ref, {ok, to_integer(Wrote)}),
                        rtransfer_link_monitor:cancel_ref(MonitorPid, Ref),
                        erlang:send_after(15000, self(), {end_req, ReqId}),
                        {noreply, State, ?DIE_AFTER};
                    #{<<"progress">> := _Progress} ->
                        {noreply, State, ?DIE_AFTER}
                end
            catch
                Reason ->
                    OnCompleteFun(Ref, {error, Reason}),
                    rtransfer_link_monitor:cancel_ref(MonitorPid, Ref),
                    {noreply, State#{ref_map := maps:remove(ReqId, RefMap)},
                        ?DIE_AFTER}
            end
    end;

handle_info({end_req, ReqId}, #{ref_map := RefMap} = State) ->
    NewRefMap = maps:remove(ReqId, RefMap),
    {noreply, State#{ref_map := NewRefMap}, ?DIE_AFTER};

handle_info(timeout, #{ref_map := RefMap} = State) ->
    case maps:size(RefMap) of
        0 ->
            {stop, normal, State};
        _ ->
            {noreply, State, ?DIE_AFTER}
    end.

-spec handle_cast(term(), state()) -> {noreply, state(), non_neg_integer()}.
handle_cast(_Request, State) ->
    lager:warning("Unknown message: ~p", [_Request]),
    {noreply, State, ?DIE_AFTER}.

-spec code_change(term(), state(), term()) -> {ok, state()}.
code_change(_, State, _Extra) ->
    {ok, State}.

-spec terminate(term(), state()) -> ok.
terminate(_, #{monitor_pid := MonitorPid}) ->
    rtransfer_link_monitor:terminate(MonitorPid),
    ok.

%%%===================================================================
%%% Helpers
%%%===================================================================

-spec to_integer(integer() | binary()) -> integer().
to_integer(Val) when is_integer(Val) -> Val;
to_integer(Val) when is_binary(Val) -> binary_to_integer(Val).