%%%-------------------------------------------------------------------
%%% @author Jakub Kudzia
%%% @copyright (C) 2021 ACK CYFRONET AGH
%%% This software is released under the MIT license
%%% cited in 'LICENSE.txt'.
%%% @end
%%%-------------------------------------------------------------------
%%% @doc
%%% This module is responsible for calculating paths that identify datasets.
%%% These paths are of type file_meta:uuid_based_path()
%%% with only one difference: SpaceDirUuid is used instead
%%% of SpaceId on the first element of the path
%%% (which is basically a bug in paths_cache).
%%% @end
%%%-------------------------------------------------------------------
-module(dataset_path).
-author("Jakub Kudzia").

%% API
-export([get/2, get_space_path/1, to_id/1]).

-compile({no_auto_import,[get/1]}).

%%%===================================================================
%%% API functions
%%%===================================================================

-spec get(od_space:id(), file_meta:uuid()) -> {ok, dataset:path()}.
get(SpaceId, Uuid) ->
    {ok, DatasetPath} = paths_cache:get_uuid_based(SpaceId, Uuid),
    [Sep, SpaceId | Tail] = filename:split(DatasetPath),
    {ok, filename:join([Sep, space_dir:uuid(SpaceId) | Tail])}.


-spec get_space_path(od_space:id()) -> {ok, dataset:path()}.
get_space_path(SpaceId) ->
    get(SpaceId, space_dir:uuid(SpaceId)).


-spec to_id(dataset:path()) -> dataset:id().
to_id(DatasetPath) ->
    lists:last(filepath_utils:split(DatasetPath)).