% This file is part of Jiffy released under the MIT license.
% See the LICENSE file for more information.

-module(jiffy_03_number_tests).


-include_lib("eunit/include/eunit.hrl").
-include("jiffy_util.hrl").


number_success_test_() ->
    [gen(ok, Case) || Case <- cases(ok)].


number_failure_test_() ->
    [gen(error, Case) || Case <- cases(error)].


number_double_test_() ->
    [gen(floats, Case) || Case <- cases(floats)].


gen(ok, {J, E}) ->
    gen(ok, {J, E, J});
gen(ok, {J1, E, J2}) ->
    {msg("~s", [J1]), [
        {"Decode", ?_assertEqual(E, dec(J1))},
        {"Encode", ?_assertEqual(J2, enc(E))}
    ]};

gen(error, J) ->
    {msg("Error: ~s", [J]), [
        ?_assertError(_, dec(J))
    ]};

gen(floats, F) ->
    NegF = -1.0 * F,
    {msg("float round trip - ~p", [F]), [
        {"Pos", ?_assertEqual(F, dec(enc(F)))},
        {"Neg", ?_assertEqual(NegF, dec(enc(NegF)))}
    ]}.


cases(ok) ->
    [
        {<<"0">>, 0},
        {<<"-0">>, 0, <<"0">>},
        {<<"1">>, 1},
        {<<"12">>, 12},
        {<<"-3">>, -3},
        {<<"{\"key\":9223372036854775808}">>,{[{<<"key">>,1 bsl 63}]}},
        {<<"1234567890123456789012345">>, 1234567890123456789012345},
        {<<"1310050760199">>, 1310050760199},
        {
            <<"1234567890123456789012345.0">>,
            1.23456789012345678e24,
            <<"1.2345678901234568e+24">>
        },
        {
            <<"1234567890123456789012345.0E3">>,
            1.2345678901234569e27,
            <<"1.2345678901234569e+27">>
        },
        {
            <<"1234567890123456789012345012">>,
            1234567890123456789012345012,
            <<"1234567890123456789012345012">>
        },
        {<<"1.0">>, 1.0},
        {
            <<"0.000000000000000000000000000000000001">>,
            1.0E-36,
            <<"1e-36">>
        },
        {<<"0.75">>, 0.75},
        {<<"2.0123456789">>, 2.0123456789, <<"2.0123456789">>},
        {<<"2.4234324E24">>, 2.4234324E24, <<"2.4234324e+24">>},
        {<<"-3.1416">>, -3.1416, <<"-3.1416">>},
        {<<"1E4">>, 10000.0, <<"10000.0">>},
        {<<"1.0E+01">>, 10.0, <<"10.0">>},
        {<<"1e1">>, 10.0, <<"10.0">>},
        {<<"3.0E2">>, 300.0, <<"300.0">>},
        {<<"0E3">>, 0.0, <<"0.0">>},
        {<<"1.5E3">>, 1500.0, <<"1500.0">>},
        {<<"2.5E-1">>, 0.25, <<"0.25">>},
        {<<"-0.325E+2">>, -32.5, <<"-32.5">>}
    ];

cases(error) ->
    [
        <<"02">>,
        <<"-01">>,
        <<"+12">>,
        <<"-">>,
        <<"1.">>,
        <<".1">>,
        <<"1.-1">>,
        <<"1E">>,
        <<"1-E2">>,
        <<"2E +3">>,
        <<"1E3000">>,
        <<"0.1E30000">>,
        <<"1EA">>
    ];

cases(floats) ->
    [
        0.0,
        0.00000001,
        0.000000012,
        0.0000000123,
        0.0000001,
        0.00000012,
        0.000000123,
        0.000001,
        0.00001,
        0.01,
        0.0123,
        0.1,
        0.3,
        1.0,
        1.0e20,
        1.0e21,
        9.0,
        10.0,
        90.0,
        90.12,
        10000.0,
        12345.0,
        12345.0e23,
        100000.0,
        100000000000000000000.0,
        111111111111111111111.0,
        1111111111111111111111.0,
        11111111111111111111111.0
    ].
