# Onenv Common Test (OCT)

Framework for managing test environments for Onedata system using onenv tool and Kubernetes.

## Overview

OCT provides infrastructure for running Common Test (CT) suites in Onedata distributed environment. It handles:
- Environment setup and teardown
- Node management and connectivity
- Code coverage
- Custom configurations
- Test background cache

## Usage

### Basic Test Suite
```erlang
-module(my_test_suite).
-include_lib("onenv_ct/include/oct_background.hrl").

init_per_suite(Config) ->
    oct_background:init_per_suite(Config, #onenv_test_config{
        onenv_scenario = "1op",
        envs = [{op_worker, my_app, [{key, value}]}]
    }).

end_per_suite(Config) ->
    oct_background:end_per_suite().
```

### Configuration Types
```erlang
-record(onenv_test_config, {
    onenv_scenario :: string(),
    envs = [] :: [{service(), app(), [{key(), value()}]}],
    posthook = fun(Config) -> Config end :: fun()
}).
```

## Dependencies

- Common Test (CT)
- Onenv tool
- Kubernetes
- Erlang/OTP 25+

## Architecture

### Test Flow

```mermaid
sequenceDiagram
    participant CT as Common Test
    participant Hook as cth_onenv_up
    participant Env as oct_environment
    participant CLI as oct_onenv_cli
    participant Nodes as oct_nodes
    participant K8s as Kubernetes

    CT->>Hook: init/2
    Hook->>Env: setup_environment/2
    Env->>CLI: up/2
    CLI->>K8s: deploy environment
    Env->>Nodes: refresh_config/1
    Nodes->>Env: return config
    Env->>Nodes: connect_with_nodes/1
    Nodes->>Env: return connected config
    Env->>Hook: return final config
    Hook->>CT: start tests
```

### Module Dependencies

```mermaid
graph TD
    CT[Common Test] --> Hook[cth_onenv_up]
    Hook --> Env[oct_environment]
    Env --> CLI[oct_onenv_cli]
    Env --> Nodes[oct_nodes]
    Env --> Coverage[oct_coverage]
    Env --> Background[oct_background]
    CLI --> K8s[Kubernetes/onenv]
    Nodes --> CLI
```

## Core Modules

### oct_environment
Main coordinator module that:
- Sets up test environment
- Manages test configuration
- Coordinates other modules

### oct_onenv_cli
Interface to onenv CLI tool:
- Core commands (up, clean, status)
- Logs & diagnostics
- Sources management
- Direct mapping to onenv commands

### oct_nodes
Node management module handling:
- Node configuration
- Node connections
- Node lifecycle (kill/start)
- Health checks

### oct_coverage
Coverage management module for:
- Coverage configuration
- Service-specific settings

### oct_background
Background configuration module for:
- Test data preparation (cache, etc.)
