/*
 *  Copyright (c) 2018-present, Facebook, Inc.
 *  All rights reserved.
 *
 *  This source code is licensed under the BSD-style license found in the
 *  LICENSE file in the root directory of this source tree.
 */

#pragma once

#include <fizz/crypto/aead/Aead.h>
#include <fizz/protocol/Types.h>
#include <folly/String.h>
#include <folly/io/IOBuf.h>
#include <folly/ssl/OpenSSLPtrTypes.h>

// @lint-ignore-every PRIVATEKEY

namespace fizz {
namespace test {

constexpr folly::StringPiece kEd25519Key = R"(
-----BEGIN PRIVATE KEY-----
MC4CAQAwBQYDK2VwBCIEINDCVMS1CMCgqf2SUdOQl/1ilCd67zNqlImeOBP8DJFr
-----END PRIVATE KEY-----
)";

constexpr folly::StringPiece kEd25519PublicKey = R"(
-----BEGIN PUBLIC KEY-----
MCowBQYDK2VwAyEAvXS5jZQpYCHV3DziV9F6PrH7zmd44/jO793mczej1+o=
-----END PUBLIC KEY-----
)";

constexpr folly::StringPiece kEd25519Certificate = R"(
-----BEGIN CERTIFICATE-----
MIIBXTCCAQ8CFFOx7Unm3rmX57I6phuWsecBc9mpMAUGAytlcDBRMQswCQYDVQQG
EwJYWDEVMBMGA1UEBwwMRGVmYXVsdCBDaXR5MRwwGgYDVQQKDBNEZWZhdWx0IENv
bXBhbnkgTHRkMQ0wCwYDVQQDDARGaXp6MB4XDTIwMDUxNjAwMzAxMVoXDTQwMDUx
MTAwMzAxMVowUTELMAkGA1UEBhMCWFgxFTATBgNVBAcMDERlZmF1bHQgQ2l0eTEc
MBoGA1UECgwTRGVmYXVsdCBDb21wYW55IEx0ZDENMAsGA1UEAwwERml6ejAqMAUG
AytlcAMhAL10uY2UKWAh1dw84lfRej6x+85neOP4zu/d5nM3o9fqMAUGAytlcANB
AONhPabQCW2kBi7kZ5bX8kQgrxSDZusESh/02gMel9b2CpW+24UH1ZDfmGY5mJy1
NAI85uRq0GsoJsSad8CJ4wk=
-----END CERTIFICATE-----
)";

constexpr folly::StringPiece kEd448Key = R"(
-----BEGIN PRIVATE KEY-----
MEcCAQAwBQYDK2VxBDsEOWdYg9tVktUUJwpDTtti6D2MuDb1M7rZ+uIhZSddLhu0
375Dom/0xZsdKhEuisrgYNJa9OH7ST7YOA==
-----END PRIVATE KEY-----
)";

constexpr folly::StringPiece kEd448PublicKey = R"(
-----BEGIN PUBLIC KEY-----
MEMwBQYDK2VxAzoAHIz+4aGo1599po9r6UXFle3rwi/CnyESbACf4SwNLQX+fIZL
j/XXTLbAqyQUskY6VdUc1olg4ssA
-----END PUBLIC KEY-----
)";

constexpr folly::StringPiece kP256Key = R"(
-----BEGIN EC PRIVATE KEY-----
MHcCAQEEIHMPeLV/nP/gkcgU2weiXl198mEX8RbFjPRoXuGcpxMXoAoGCCqGSM49
AwEHoUQDQgAEnYe8rdtl2Nz234sUipZ5tbcQ2xnJWput//E0aMs1i04h0kpcgmES
ZY67ltZOKYXftBwZSDNDkaSqgbZ4N+Lb8A==
-----END EC PRIVATE KEY-----
)";

constexpr folly::StringPiece kP256PublicKey = R"(
-----BEGIN PUBLIC KEY-----
MFkwEwYHKoZIzj0CAQYIKoZIzj0DAQcDQgAEnYe8rdtl2Nz234sUipZ5tbcQ2xnJ
Wput//E0aMs1i04h0kpcgmESZY67ltZOKYXftBwZSDNDkaSqgbZ4N+Lb8A==
-----END PUBLIC KEY-----
)";

constexpr folly::StringPiece kP256K1Key = R"(
-----BEGIN EC PRIVATE KEY-----
MHQCAQEEIO/faBJKYzuLCTg58AIyWbeu2c6VVuPToCrINFj4y8wToAcGBSuBBAAK
oUQDQgAEW5bWEQR9xaAT0Zaev8QmA/lzz1a9+z1cphweKH8ynBYmWiWivaqGcVAU
Lt/SKc3HlcKGQhblkaGHEt3pCQeZUA==
-----END EC PRIVATE KEY-----
)";

constexpr folly::StringPiece kP256Certificate = R"(
-----BEGIN CERTIFICATE-----
MIIB7jCCAZWgAwIBAgIJAMVp7skBzobZMAoGCCqGSM49BAMCMFQxCzAJBgNVBAYT
AlVTMQswCQYDVQQIDAJOWTELMAkGA1UEBwwCTlkxDTALBgNVBAoMBEZpenoxDTAL
BgNVBAsMBEZpenoxDTALBgNVBAMMBEZpenowHhcNMTcwNDA0MTgyOTA5WhcNNDEx
MTI0MTgyOTA5WjBUMQswCQYDVQQGEwJVUzELMAkGA1UECAwCTlkxCzAJBgNVBAcM
Ak5ZMQ0wCwYDVQQKDARGaXp6MQ0wCwYDVQQLDARGaXp6MQ0wCwYDVQQDDARGaXp6
MFkwEwYHKoZIzj0CAQYIKoZIzj0DAQcDQgAEnYe8rdtl2Nz234sUipZ5tbcQ2xnJ
Wput//E0aMs1i04h0kpcgmESZY67ltZOKYXftBwZSDNDkaSqgbZ4N+Lb8KNQME4w
HQYDVR0OBBYEFDxbi6lU2XUvrzyK1tGmJEncyqhQMB8GA1UdIwQYMBaAFDxbi6lU
2XUvrzyK1tGmJEncyqhQMAwGA1UdEwQFMAMBAf8wCgYIKoZIzj0EAwIDRwAwRAIg
NJt9NNcTL7J1ZXbgv6NsvhcjM3p6b175yNO/GqfvpKUCICXFCpHgqkJy8fUsPVWD
p9fO4UsXiDUnOgvYFDA+YtcU
-----END CERTIFICATE-----
)";

constexpr folly::StringPiece kP384Key = R"(
-----BEGIN EC PRIVATE KEY-----
MIGkAgEBBDCq2Al9+ZmWvvKXeKF4x+P0D3BvbFD9uxxgixjp2pVrGiQeybNLsmgz
aCjz+ETbBEKgBwYFK4EEACKhZANiAASBEEGWK7rJhZ6yqhmkdS5XOUnT8euKvLEx
wWdOdQYfXhe2okck7rcyaO2C/T0r/JPbiMq7+MnankwUef/bmlS1IpkK70AcGrAE
qh/fDiaqhpLKWGCtb2z46ju5Xb7LgqM=
-----END EC PRIVATE KEY-----
)";

constexpr folly::StringPiece kP384PublicKey = R"(
-----BEGIN PUBLIC KEY-----
MHYwEAYHKoZIzj0CAQYFK4EEACIDYgAEgRBBliu6yYWesqoZpHUuVzlJ0/Hriryx
McFnTnUGH14XtqJHJO63Mmjtgv09K/yT24jKu/jJ2p5MFHn/25pUtSKZCu9AHBqw
BKof3w4mqoaSylhgrW9s+Oo7uV2+y4Kj
-----END PUBLIC KEY-----
)";

constexpr folly::StringPiece kP384Certificate = R"(
-----BEGIN CERTIFICATE-----
MIICJzCCAaygAwIBAgIJAOzjkDsjaGgLMAoGCCqGSM49BAMCMFExCzAJBgNVBAYT
AlVTMRUwEwYDVQQHDAxEZWZhdWx0IENpdHkxHDAaBgNVBAoME0RlZmF1bHQgQ29t
cGFueSBMdGQxDTALBgNVBAMMBGZpenowHhcNMTgwNzI3MjIwNTE2WhcNMTgwODI2
MjIwNTE2WjBRMQswCQYDVQQGEwJVUzEVMBMGA1UEBwwMRGVmYXVsdCBDaXR5MRww
GgYDVQQKDBNEZWZhdWx0IENvbXBhbnkgTHRkMQ0wCwYDVQQDDARmaXp6MHYwEAYH
KoZIzj0CAQYFK4EEACIDYgAEgRBBliu6yYWesqoZpHUuVzlJ0/HriryxMcFnTnUG
H14XtqJHJO63Mmjtgv09K/yT24jKu/jJ2p5MFHn/25pUtSKZCu9AHBqwBKof3w4m
qoaSylhgrW9s+Oo7uV2+y4Kjo1AwTjAdBgNVHQ4EFgQUn8WNTodJ81ytFl+3BWKI
I9X784wwHwYDVR0jBBgwFoAUn8WNTodJ81ytFl+3BWKII9X784wwDAYDVR0TBAUw
AwEB/zAKBggqhkjOPQQDAgNpADBmAjEA6S4jXP5fruwrmszizG854snSHl5LVEuQ
8F6RIEsOnWSbKBBoteaSXwA6Tv+jYZnmAjEA8YU0LnO+eA75Aku0mPDQsVJOjCvX
OMc6lsBbEEu0yLF89FcMYbxH2JCRxa528qd+
-----END CERTIFICATE-----
)";

constexpr folly::StringPiece kP521Key = R"(
-----BEGIN EC PRIVATE KEY-----
MIHcAgEBBEIBPYFDNkfM6VuyQKVaQigvLTeKXcPWr4qMLHtN4V81QZLIE9HJvL29
hbqwRG/kw2XM8ZQSnsqd1XTevQSfEPK45+CgBwYFK4EEACOhgYkDgYYABAGnbVsM
QvVX5Bi/mCs1bpA/k3udUwzKglDgfJjKAyk5WLL6jaIPnE6fdoTfPocrRw2sU2hp
e2ZM5MZ3P+uIYnB7XwFDQYLrlio6qKcHyGjKpu3uYt2EVhcu8L1zj6w6uwToe9Pp
6pwb0J+HIKmUxhYstCVG+G4yjekwOH6OzCxo7EG88g==
-----END EC PRIVATE KEY-----
)";

constexpr folly::StringPiece kP521PublicKey = R"(
-----BEGIN PUBLIC KEY-----
MIGbMBAGByqGSM49AgEGBSuBBAAjA4GGAAQBp21bDEL1V+QYv5grNW6QP5N7nVMM
yoJQ4HyYygMpOViy+o2iD5xOn3aE3z6HK0cNrFNoaXtmTOTGdz/riGJwe18BQ0GC
65YqOqinB8hoyqbt7mLdhFYXLvC9c4+sOrsE6HvT6eqcG9CfhyCplMYWLLQlRvhu
Mo3pMDh+jswsaOxBvPI=
-----END PUBLIC KEY-----
)";

constexpr folly::StringPiece kP521Certificate = R"(
-----BEGIN CERTIFICATE-----
MIICcTCCAdKgAwIBAgIJAMaW6GnzHQdaMAoGCCqGSM49BAMCMFExCzAJBgNVBAYT
AlhYMRUwEwYDVQQHDAxEZWZhdWx0IENpdHkxHDAaBgNVBAoME0RlZmF1bHQgQ29t
cGFueSBMdGQxDTALBgNVBAMMBGZpenowHhcNMTgwNzI3MjIwNjI2WhcNMTgwODI2
MjIwNjI2WjBRMQswCQYDVQQGEwJYWDEVMBMGA1UEBwwMRGVmYXVsdCBDaXR5MRww
GgYDVQQKDBNEZWZhdWx0IENvbXBhbnkgTHRkMQ0wCwYDVQQDDARmaXp6MIGbMBAG
ByqGSM49AgEGBSuBBAAjA4GGAAQBp21bDEL1V+QYv5grNW6QP5N7nVMMyoJQ4HyY
ygMpOViy+o2iD5xOn3aE3z6HK0cNrFNoaXtmTOTGdz/riGJwe18BQ0GC65YqOqin
B8hoyqbt7mLdhFYXLvC9c4+sOrsE6HvT6eqcG9CfhyCplMYWLLQlRvhuMo3pMDh+
jswsaOxBvPKjUDBOMB0GA1UdDgQWBBRqn91E64NUNowqgdlYhJCerng6bTAfBgNV
HSMEGDAWgBRqn91E64NUNowqgdlYhJCerng6bTAMBgNVHRMEBTADAQH/MAoGCCqG
SM49BAMCA4GMADCBiAJCAaCZdl3sxO15gKzpODf/y591WTtaXWEz5MdlHP2y3m1X
4ae1xJMdmB6Qw5/bYHAus9GCwLx37WxaMxYcSG2mfIaGAkIBnIE2i1fX3LZCfkKx
SR7JD8cautCXMGtmivUW8TItC72Gt8DWD1hVfR9OnjnuKFotlVZ1FF/HbG2Hwa4/
Skcvpq0=
-----END CERTIFICATE-----
)";

constexpr folly::StringPiece kRSAKey = R"(
-----BEGIN RSA PRIVATE KEY-----
MIIEowIBAAKCAQEAxWSZkGZoflV+hnNKZVuCUr0eOedYtFIEU16mARPP3KPqHFrc
EXudA5/46hoogfSb2WYqEaCeltY3GiPG+WPdhhDEi5h4hImvL+g/iTU7LLmIhmkx
4hKzAYx012012Hxy7pvEJJy6S7wEcJj0AxNsWFo8SysIfO5Rw57CTHolxwcbuCsf
q6Cca3PE0gc9UXZ2KaTJNuph8gWPDdio8Au5YnYpvIYy0QXt6eUFAH8huNRBOUK+
XHng+/zAIXQAtGJEW/rx/vKDUWm0nzZKlIUXPIdCSMCTO6qj+UFvypd0SN4PXW/6
DUJeGi3bXFql9XF8ysy6ZgheHKsvgPDlSkOO5QIDAQABAoIBAH36IE+aLCJEqgVg
2yAo8NmhPQMzdNdQaNZUV5nkbQqhyi3ihwTZYXIIzLGLlEKsPDLZEwxA+DXH7U7P
epN7esuvM+DrpC7iOF5Y2QPWZze6CTnPNTjVOk1IuOEsH+tUMX27B8leBT+EdUjv
NPwpj0IsBMLsafRhnOk+Dz8Dzcxgyee6r0bGy4f7LZ4/e/mQzR2y5IrePmpRJH/8
NF6OfLpAxrVb57VVDaKEGjR0i0xOR8/nuNKgUVeAzkFNytJl5oxzHR3qHBg/mYmQ
Y4eu0tCZcczennWm06jt6P3qCD7xGtlj8RwFZeQ7UAwaHAPDd60LDb2Kt7WO87OY
wWYz2ZkCgYEA/hKBLyJsgvsGL25jTrcqdChJURiBl+KGQj3giSNy+WgHnQuSpX/G
6s46tgvuLpWzS3E4eX/ymdjWPe0S0N8ZVUAzr5Ne4/EdDTPmaZsLXEsJ+19YRfLY
/E9v0ao4Rliwk0lC4ieeNY79PKZ4XBP5eNg9c7T2dHFEadJHvB13P2sCgYEAxuQB
LShZzOPoN967I9VOkwYLrf5k0LdgKmuDgzFo7Y9WMiqy0GdcfsiPunde+HHXmRDE
k1Am14w3NQLQlLYs9QTUBHYVllPmpkLqozFauGNw5hHcq7YZtfZqIZLL0aueZhDt
aV/nNDMYTqgclJbpMEtQ6itLonLX1htJS0LGju8CgYAUIdE8s+SXSF7IGV4jGZ5y
j9Kna2Kxb1CmJRMKqtVsb9LSnd0j4gkYB/lc/XsFE/WeD72i2wGVMTpgs81e4Z63
Q5wU2H+XXj6oEim6yn9O8985nOd4u3J4I68/lNZatKJhOVfkZbFCmy7zxFkbKrYE
3lWYOomg1mzOWcw3WrpdOwKBgBzQtdbsDK6hqmhFzckwQ1ULwC9yNvsCRl1+/CvK
wdABP1S/WO7WutDnj+NNHiZSoBDeHAzXxPjfqLP2d2/EQRkynOEUlqYT+8vE0Cfa
o7EpPxcmKVl7ZZg2XkQE96CmA6n3892bdmzapj/nHVTt1FUZ5Fw475lWmCy+AU11
HD1jAoGBAOIA6kfwN6Z3mWRuivfruzk50NeqV2Xsmaa+Sk18n0Cz2cOynUWNl7qX
/8ZTeGj9UFARNbQ5aM+gRzgEPigpAiWK3qoedBfEUjKUyDZlfWlDu8Ds8T+HvZ0i
h8YfdM73KnIU0O8s+bvCkg/sFvqDmQ9ocdTSo1aVkNY4avhSgstK
-----END RSA PRIVATE KEY-----
)";

constexpr folly::StringPiece kRSACertificate = R"(
-----BEGIN CERTIFICATE-----
MIIDYTCCAkmgAwIBAgIJAMNCCDasHKJvMA0GCSqGSIb3DQEBCwUAMEsxCzAJBgNV
BAYTAlVTMQswCQYDVQQIDAJOWTERMA8GA1UEBwwITmV3IFlvcmsxDTALBgNVBAsM
BEZpenoxDTALBgNVBAMMBEZpenowHhcNMTYxMjI5MDYyNDE4WhcNNDEwODIwMDYy
NDE4WjBLMQswCQYDVQQGEwJVUzELMAkGA1UECAwCTlkxETAPBgNVBAcMCE5ldyBZ
b3JrMQ0wCwYDVQQLDARGaXp6MQ0wCwYDVQQDDARGaXp6MIIBIjANBgkqhkiG9w0B
AQEFAAOCAQ8AMIIBCgKCAQEAxWSZkGZoflV+hnNKZVuCUr0eOedYtFIEU16mARPP
3KPqHFrcEXudA5/46hoogfSb2WYqEaCeltY3GiPG+WPdhhDEi5h4hImvL+g/iTU7
LLmIhmkx4hKzAYx012012Hxy7pvEJJy6S7wEcJj0AxNsWFo8SysIfO5Rw57CTHol
xwcbuCsfq6Cca3PE0gc9UXZ2KaTJNuph8gWPDdio8Au5YnYpvIYy0QXt6eUFAH8h
uNRBOUK+XHng+/zAIXQAtGJEW/rx/vKDUWm0nzZKlIUXPIdCSMCTO6qj+UFvypd0
SN4PXW/6DUJeGi3bXFql9XF8ysy6ZgheHKsvgPDlSkOO5QIDAQABo0gwRjAJBgNV
HRMEAjAAMAsGA1UdDwQEAwIF4DAsBgNVHREEJTAjggoqLmZpenouY29tgghmaXp6
LmNvbYILZXhhbXBsZS5uZXQwDQYJKoZIhvcNAQELBQADggEBAIpIvwxxSJrLGW8I
rz4PpKLoeKetLCW3HYVrrMF8nGLKwlzeWLa0BpQN638DuDLOsaGZX0OshsOsPCc9
FWub8Vdu5pA1zuDLS03aL2F4DBMyvKvDmqa0+Jsj+SuIk054oF1Q4jvx9VE0JBmx
1FfHZ5Ug+f8DLWYvLMN6G9P6YYzoENn589oa//JHYWDoJimt2IB80R1k47gIvGde
W4CnlNH1jYO1/mr1yVHNrpdkOab2ItdEycdTw8zi/QOGRhFevjcR+p6c+NKr2/OS
iv9+LfvbaFlvdxkkKGr3mrsb2EgzB1LiSHTllA+yS88QzxcSRhzSeSg/bvL+xsWT
xcGg1w0=
-----END CERTIFICATE-----
)";

constexpr folly::StringPiece kClientAuthCACert = R"(
-----BEGIN CERTIFICATE-----
MIIFGjCCAwKgAwIBAgIJAL3Ohy9z0aLMMA0GCSqGSIb3DQEBCwUAMBoxCzAJBgNV
BAYTAlVTMQswCQYDVQQIDAJDQTAeFw0xNzA5MDEyMDI5NDJaFw0zNzA4MjcyMDI5
NDJaMBoxCzAJBgNVBAYTAlVTMQswCQYDVQQIDAJDQTCCAiIwDQYJKoZIhvcNAQEB
BQADggIPADCCAgoCggIBAMU+/6+YdKaBvzSWOcN5XJjw8GuFs5hzKkZm9Yh7TCpu
05jJ9LaXqYUMfwUqTAtUVjk/LVb9SdL4h6nAZE8ZnB4YDEPy2ymBkkN1JQ+nlOHS
z5HpLbPu5fJiZjvqmnrM2J29kqFN0s781hlkab4XrYiIjkvwUYSgNwJe2pnYtEwd
RQg417EKJmCzP973zfsCnkUzazVI4Dqr5+tRBibZfycxUEzbGRuhCNiAFxRXJNsl
5y1Iva9dfe8V7SHlB7wcXfqbaS9v5ytqfxbmhoVGcj4VIHHsvD4W8l3EZJiMm4/k
h1v/INRdsa3jOFxYtxglkMbEtGseD1cTcqnphHane5zMmm+5iqhw4hlWLoCnz64G
8Y4RQkVlLlWhYnrbH84pYWVotuXzBz9EiAddOimlPWQ8B8VJseH01QNXYbBuqaZo
UHpB4tVbA+8aDk7UPpvBgtYkbZlJezRNQCxX2dM94CUzXyNOiGLrNAzSCrurbc3m
Bk53fUCfQAxbd3NYIM9vf69hzdhGpwuLs7FoWTJ1E1EivrpXLVHx0MPuZ8G5OpYB
4rRQioCtqcC53XFQoGUzLxD2k/6/jl7/BHmvZym1C59HCZK6zSNQ1UCMylEC1ZBH
BqUH1ThXRzGJhVw83mRnvczUElj+YE4fSrpm/cNjJlFUbSmgQeaMzCpDQiZ6Uz0N
AgMBAAGjYzBhMB0GA1UdDgQWBBQfGqX5QugIVrqaC9dENMsysX5eGTAfBgNVHSME
GDAWgBQfGqX5QugIVrqaC9dENMsysX5eGTAPBgNVHRMBAf8EBTADAQH/MA4GA1Ud
DwEB/wQEAwIBhjANBgkqhkiG9w0BAQsFAAOCAgEADtD+hggO5YSCsTJN6EwzttJ+
cs8baCm1n9ElxHbNplaHsoRR0xxUgV7FcKkEPqQaEY81FuU34t8tcBmnoAmmt2K1
aNyGeMCH/BJRzwTtzUsayv/OdYOsndXoSflSTPAcAQvipwLEYWVfMkyfLtKXOkcc
Zk1/uxEeT2CtMoPXePd9fNs80kRAXPnuNgg9Hyz9lAA3/HAWCqjr25NEenEJm+Jn
v5A73iAKwLGabCwMu8231SGTVe1kw94n3EMGYZUJ3Cuf+Wd76EEmri/CqL6iFFAq
Jm0+jo2kqacTIxaDWvxpZx5dUDy7U/4+F0Tn4i+4PmAPMXlmUqdhXiXrIAcvwN/x
Y5flgUR63wjwnCWqTgv54mxm5MvILxr66b75qx1ORtYms/zksOb0hUUNBNsuo6rb
r3Wd6FQ7SXI4UoplpaHZknw7pGNk/U7tqznz1KjfYYBhbg9TAa/e9zSPmm5BLMZB
+lKCyOAi3qNRfUs80AzJd+MSGw2QSD+wGMzCRwb9kdjtYTbxfzSR9HQLj3iMWiIf
2qt6iPt3YSAMphBOi+DrStrUpKsFeYzD6gGMgjfYtCDlrmjNoObfjkP5n1viRn8g
1Ur2kDbFrm8vyRCnVJUfkJwot3F9gbv0bj22HA46sDisEc3D380tiR55mjiLKa8r
wt3Vj1V7ondA7QQEDuE=
-----END CERTIFICATE-----
)";

constexpr folly::StringPiece kClientAuthClientCert = R"(
-----BEGIN CERTIFICATE-----
MIIErzCCApegAwIBAgICEAAwDQYJKoZIhvcNAQELBQAwGjELMAkGA1UEBhMCVVMx
CzAJBgNVBAgMAkNBMCAXDTE3MDkwMTIxMDU1MloYDzIxMTcwODA4MjEwNTUyWjBj
MQswCQYDVQQGEwJVUzELMAkGA1UECAwCQ0ExDTALBgNVBAsMBEZpenoxFDASBgNV
BAMMC0ZpenogQ2xpZW50MSIwIAYJKoZIhvcNAQkBFhNmaXp6Y2xpZW50QGZpenou
Y29tMIIBIjANBgkqhkiG9w0BAQEFAAOCAQ8AMIIBCgKCAQEAzU4RuUU3QjX9rMeC
aVYPX1G8ITkGkvRW1Fbo5P9Irmw/uOeXnTKFtJ3h+r+omA607aqymHfjk1uDtxwH
2GI1VEIYce1u+qYSiLdfWUOQ1za5abeGhT/PJSONVg02maaMZmxSi5L7f/9x5B5B
fIecgurN+7W381ENyp06akg3CSAtNITL1+Au8wOh4woAJ1qKcgrhHeSG5qOqe2cw
vkcfy1Pahmp5sQtebI+1VaLXSW2XS4i/59oM7jfWAs2ZZW2EYrT3z0q3bEiL9OVz
Qjumv5fi9kJR7Pol/Q3p0MD4/8TPA6eEFf9FYPoguF6f2RKvN7uqCjIhH9r679Na
UBQcZQIDAQABo4GzMIGwMAkGA1UdEwQCMAAwEQYJYIZIAYb4QgEBBAQDAgWgMCEG
CWCGSAGG+EIBDQQUFhJDbGllbnQgQ2VydGlmaWNhdGUwHQYDVR0OBBYEFE/TN4DI
NrYzGanBqw9MD+gmmkkDMB8GA1UdIwQYMBaAFB8apflC6AhWupoL10Q0yzKxfl4Z
MA4GA1UdDwEB/wQEAwIF4DAdBgNVHSUEFjAUBggrBgEFBQcDAgYIKwYBBQUHAwQw
DQYJKoZIhvcNAQELBQADggIBAHBGcdKXTFQoZOWdt/XR3B3tLJDPoo9anFc+w+/C
tNMUCOT1/ILdDtjkXQ9+U6czkgOQbcUzUkUF9sfPJ2aBf95azvU51xXH19CPyLc2
H59CHdUUGD1CiJ8H/EH1kjb6sBIX17OvYYYj47qT1us2ZM45w86zhLZkmDkIcdMy
A9ZhNtQDk/oWGE1Q6YV2CT78RSNke8RqNe7XrDIJh9ZOGyfYqXuDsdGE8md0nCAC
kqWje0HeHkWTJTkdNLSp6DNpXmMyUg8wjcOvj3dTNOEtqqTAHx8wzL9YlCsUTLot
HGXpVhMv2X+GMKQsXTBVUVrWaFj6x4owXmRK8le315AgpsIGhTqFlGLpbs7dCm6T
upufJ6ZoY4gLxcwPD691PQznjqC45j4Zi+KxY8ptsIVJbcK9kbcBIb/T7HnQHCOI
vTUQFPPhST0fRhpsiOGaJxPOn6Ht8lSf3A1vS+TfD4E5cyojZzXe2H7ao3sCB4a4
X8tTiQTUHQVFNQqiPsErBOX1ybr1uwRFcX+uIwiVtcxRi5otMZNIFpulvmwb517R
NprWAWb0X4RYqPoTNW/OGk421nI0Bc5Rhi4+6RO93BrkPY0HHDnge7LB5A7WynbG
8gdVGTDuTniJ7Z4xIz6/YMxEM8F+BCQzq1f/2dnlSfEgIk2AdsW1verYIfc6xwqr
V1Xd
-----END CERTIFICATE-----
)";

constexpr folly::StringPiece kClientAuthClientKey = R"(
-----BEGIN RSA PRIVATE KEY-----
MIIEpAIBAAKCAQEAzU4RuUU3QjX9rMeCaVYPX1G8ITkGkvRW1Fbo5P9Irmw/uOeX
nTKFtJ3h+r+omA607aqymHfjk1uDtxwH2GI1VEIYce1u+qYSiLdfWUOQ1za5abeG
hT/PJSONVg02maaMZmxSi5L7f/9x5B5BfIecgurN+7W381ENyp06akg3CSAtNITL
1+Au8wOh4woAJ1qKcgrhHeSG5qOqe2cwvkcfy1Pahmp5sQtebI+1VaLXSW2XS4i/
59oM7jfWAs2ZZW2EYrT3z0q3bEiL9OVzQjumv5fi9kJR7Pol/Q3p0MD4/8TPA6eE
Ff9FYPoguF6f2RKvN7uqCjIhH9r679NaUBQcZQIDAQABAoIBABJURPrQdiWAIny5
dygmWcr9xhkFOVQHYnUkqThAnmGMIohEjOGo8Az3B4TWdZvVvEt3sOgx0+yFABZp
9z1XT0GNgRcyxnS0PPVdwbJ7eMPuC9GnY+gjkT/Uk7q6BUTkfKZsR1ZUsgV0bdua
4+fVR6Dci7VMPPcJJn0PWnbAqcEzqBUI3+HqUpx1LACqbMYtlTpFxHO811ycbTkB
SJMlEvzH5Ry5gptJur7kepyA4o6hz9C/KPjuCcjXUwjGDggqQJkj6ulUdQIZek8R
9ckeHETrQiuSnUngJbgG83kHVGISdX9kgqVfo+t2cIsEGO9KJOQ4HNL9iXvh+Coj
Hsc1ZKECgYEA/Ian9dD27mkgYz5z1oO4R2t0Hjr18tI5WUd4Smc6sDAX0xjBsVui
sBm0apE+QvgHrD+JWxrBDTB1SZmfgd9OjjCveILo8eUTk91jYgOov9+GLpL+k2rY
dsBdsLu+x9+w6d+N+ySVRJU4MfweEYrah13IV+nr+gLd7/uxpsP2owkCgYEA0CEc
SCUBHkoa4O4/jkn3yBXnGq1SSaLA9LCQXhLlfWUd7nZAX3dNjXAS7JE5QtOt1C7p
7Fg7fS/dEYD6VyGmscq9yJL7xKD0zx1nKlr5ZhCGTZM6QXvmSRDRf06Ne+z7Gxhf
oDmM8vclHu5z7J7aYnb7BPwTBz4W/IKqyKptuX0CgYEAvEBGBeJ5stF764ArKseX
IObh8YdSwmolhoDZE95QyfeQSzYxbt45Mb532Nnd6F6umaHj657OO97cELoNRuwJ
y8ZPiS9OKMX1Ck4IQJEpw6AGj8HVNmB4bk6Jia7sUmRXpoGZfpePD0mOUvT4FWvF
7RPkvmUMHZz1hmnDzeZYfsECgYBFrD/eetzD+gTDdZ47ENHs3kAuMwVP9zbm8tBx
XZwLfTUm5dko3DRO+8yp+veTQm93j3jpkKPhhYtY17d2QaKe9ALSSToVGdi6GNq3
WBsTKsb64yVmVIbDwOiQ2IgOi0eIS4F/jqR7lXaMw0B1Xrw2qCPiCEyKaSOixk9L
u5ndtQKBgQC0CLFhehVr4AOx65l9VGNW9lx8XlUXTMaElv6DKqQhTOGIe48pzHkQ
VxjmpoOD8HC2xqb+HfofiwVTR5Q1khz/BqcjdrT8TRxFD4bKjVePOxyrIcTiGJzx
V0495bcyl/ghl1VyoKOzjcc1pMQCLUaj4MmbSmW0A9wxP1eVPTOhEg==
-----END RSA PRIVATE KEY-----
)";

constexpr folly::StringPiece kCertWithNoCNButWithSANs = R"(
-----BEGIN CERTIFICATE-----
MIIC4DCCAcigAwIBAgIJALTMdz3uPpCKMA0GCSqGSIb3DQEBCwUAMC8xLTArBgNV
BAoMJGludGVyb3AgcnVubmVyIENlcnRpZmljYXRlIEF1dGhvcml0eTAeFw0yMDA5
MjMxNjQyMzVaFw0yMTA5MjMxNjQyMzVaMBkxFzAVBgNVBAoMDmludGVyb3AgcnVu
bmVyMIIBIjANBgkqhkiG9w0BAQEFAAOCAQ8AMIIBCgKCAQEA1mRHbCeMxpws+sHC
OFAALZpm0Lme6a2rmtAQDdNl+ZHj9jENaNpyqhtgmADQ4xe4eQ/3b+9Dzp5ZZttq
XQuJwSitVQkzk3C8fwEFxbU9lXmn2eU07EbNjc6yp+TNtfFiae5NaivuryeBGFAW
tZ7DfJW6u2nD4iGcCz8DGiWQ3A66yZr2F7vyww7L3IJgZJHLtNEJqyqNp/V6437y
e/av1MVptvyHF/x1G2C5sg7OmoL8Sh2eI9KSJ7cP8GK7AKUTGUNn86CUe1lshGof
sFxiOHUgOqGcWEC24G18rgt1vCKaxWDfTGO3pY+55bzFpnVdnBlK7kPG2+j+Bj8W
NwUEwQIDAQABoxUwEzARBgNVHREECjAIggZzZXJ2ZXIwDQYJKoZIhvcNAQELBQAD
ggEBAKawb0cI0Ieu19mk5LFkK74Hw7s4+58SucdC93zNBvitm1fnMDmCpYfFTsEV
LuuheCQGQSCwL5IMDXdhaHMefL1VTvkws/QOQWYcHB3M3SvC+nApkiSy7Z77+Q1i
oetv9zLnPX1l9wV2GozxygbLrFAICBNbfCBoES06ZrAVI5WKpRyH1iBN0O2oky8g
J6zWLo7/3QdkDXqxjPzAhe0QiLZj3gVbmb0qCDOQOb59WxfoXqH+0LYYWcz046p7
Qc0UmyO6jI4gPtR+E3C/U7qHpD0fkz2aVJp+vleOBtzinGfAhYPDgTCwQ55xGU72
MPZDaUDgJLPG+bgt0iK1AV7EBec=
-----END CERTIFICATE-----
)";

constexpr folly::StringPiece kCertWithNoCNButWithSANsKey = R"(
-----BEGIN PRIVATE KEY-----
MIIEwAIBADANBgkqhkiG9w0BAQEFAASCBKowggSmAgEAAoIBAQDWZEdsJ4zGnCz6
wcI4UAAtmmbQuZ7praua0BAN02X5keP2MQ1o2nKqG2CYANDjF7h5D/dv70POnllm
22pdC4nBKK1VCTOTcLx/AQXFtT2VeafZ5TTsRs2NzrKn5M218WJp7k1qK+6vJ4EY
UBa1nsN8lbq7acPiIZwLPwMaJZDcDrrJmvYXu/LDDsvcgmBkkcu00QmrKo2n9Xrj
fvJ79q/UxWm2/IcX/HUbYLmyDs6agvxKHZ4j0pIntw/wYrsApRMZQ2fzoJR7WWyE
ah+wXGI4dSA6oZxYQLbgbXyuC3W8IprFYN9MY7elj7nlvMWmdV2cGUruQ8bb6P4G
PxY3BQTBAgMBAAECggEBAJsCzGVVv0KG/zqbR6thpI9UeQxneY/pww7fawwkEjI9
mr6RvulWMNvviYq95EqeBwJ5WeWz8Kn+8hMdiC0YP5TKrXCzg3gSZifJ/HtzzMA7
wvIX+IjxtIPYtHISS+5GRmrjI1QlyaEZBg0nMxREY4G73NTO5xOkS2gSOlL4YGHK
bNdzQpLyZj6Goosr0g1J+l2/EVKO+/xBvZv6Fj6im+PmAimb+xe9iul7QDQHlYtI
T9J+2lvgzAof+uGYkMLwIO4R6xnjfd+Ey91rrQXgLfu1H63IT5qD0cOQ8sp+6XOW
slX6n1UxCjGYAJeSXqqu+5fX03XZt4h2zej2xtwLzeUCgYEA/RcnrLjzW2K1fnZq
WYuLcl9YwG4gUqqAzPsVOBPsWpnU1DMuC/L24u/CIlGLEEPRXgt8r6nGqfSg6Hs4
EOve8px3So5Nvt8EhugbANuQJlBPb3rkbV6cnCAbzpS6KYwCMNRZImO2K1bsab17
GBNagzKy5wqp6TFY0+CMySxytJ8CgYEA2Ns70YY3iYDFLLxrVpxHQG9bR4chATE2
tnePihfGgaW0oa744cMhJF/JIfpepB2JG/aOD6vmgtDO/z+vMqkO9Dvt/6wHa+S4
5AVkJnIkApNWXYdPM53szLx9fiuiQrQszsVpBSA20ytG9JjZ8/8i4T80KTgV0Il+
BrfzYYzjap8CgYEApCg8p59e2UtxBRGxcVs9m3WUj1vewz+sQ0goPzNM/ocAsJJx
r3ZsBE4W0UOqu2YBispQmW+5V8tAAwrJFtCmzx4FkeozKzZkLUynFytSmEdG/rvr
JbVURz/cSWXWSdRyJ1HUbqXWJs4+kWdBTCBheO+NcqZMBuDnCKaBKosV62kCgYEA
mvx9ES+Cy9Rrl6CGep0w012C+GkcbpS0zM5AmWqKpig/I/tAx2HFcxC+WHlvYI33
azPYrlymX8JK3uSuG1/2Xxnh7IQPvc33UoiD3nJfSDPzWt8U/QgWsPDaI+2dh1zs
VU+D0nUGmf/pM3F2/ErRq/iXGAqMlMFff//Cg2rFMnMCgYEA7De3L3aNpHFWhtUi
7c3DGtP3qTC6FAS9GJlA1TVsjgHt4nL3ZAKWOqXwVa1kXNDeYgN/Br39dfYW/KRE
Liz9kBryc02/2YSjg4YFDgrCqzeERPmwj8AKM3sPyL+nLzZOvJ3EbhyiQiOYGh+U
osLtL9tFEp8IrKHnoNVDuMPaZPM=
-----END PRIVATE KEY-----
)";


std::unique_ptr<folly::IOBuf> toIOBuf(folly::StringPiece hexData);

folly::ssl::EvpPkeyUniquePtr getPrivateKey(folly::StringPiece key);

folly::ssl::EvpPkeyUniquePtr getPublicKey(folly::StringPiece key);

folly::ssl::X509UniquePtr getCert(folly::StringPiece cert);

std::unique_ptr<folly::IOBuf> getCertData(folly::StringPiece cert);

void useMockRandom();

std::unique_ptr<Aead> getCipher(CipherSuite cipher);

} // namespace test
} // namespace fizz
