%%%-------------------------------------------------------------------
%%% @author Lukasz Opiola
%%% @copyright (C) 2017 ACK CYFRONET AGH
%%% This software is released under the MIT license
%%% cited in 'LICENSE.txt'.
%%% @end
%%%-------------------------------------------------------------------
%%% @doc
%%% This module tests the Graph Sync server behaviour and Graph Sync channel by
%%% testing interaction between Graph Sync client and server.
%%% @end
%%%-------------------------------------------------------------------
-module(graph_sync_test_SUITE).
-author("Lukasz Opiola").

-include("global_definitions.hrl").
-include("graph_sync/graph_sync.hrl").
-include("graph_sync_mocks.hrl").
-include("modules/datastore/datastore_models.hrl").
-include("performance_test_utils.hrl").
-include_lib("ctool/include/aai/aai.hrl").
-include_lib("ctool/include/errors.hrl").
-include_lib("ctool/include/test/test_utils.hrl").
-include_lib("ctool/include/logging.hrl").
-include_lib("ctool/include/test/assertions.hrl").
-include_lib("ctool/include/test/performance.hrl").
-include_lib("gui/include/gui.hrl").

%% API
-export([all/0]).
-export([init_per_suite/1, end_per_suite/1]).
-export([init_per_testcase/2, end_per_testcase/2]).

-export([
    handshake_test/1,
    rpc_req_test/1,
    async_req_test/1,
    graph_req_test/1,
    batch_req_test/1,
    subscribe_test/1,
    parallel_requests_test/1,
    unsubscribe_test/1,
    nosub_test/1,
    auth_override_test/1,
    nobody_auth_override_test/1,
    auto_scope_test/1,

    bad_entity_type_test/1,
    bad_message_test/1,
    timed_out_request_test/1,
    crashed_request_test/1,
    stale_request_pruning_test/1,
    throttling_test/1,

    session_persistence_test/1,
    subscriptions_persistence_test/1,
    gs_server_session_clearing_test_api_level/1,
    gs_server_session_clearing_test_connection_level/1,
    service_availability_rpc_test/1,
    service_availability_graph_test/1,
    service_availability_handshake_test/1
]).

-define(TEST_CASES, [
    handshake_test,
    rpc_req_test,
    async_req_test,
    graph_req_test,
    batch_req_test,
    subscribe_test,
    parallel_requests_test,
    unsubscribe_test,
    nosub_test,
    auth_override_test,
    nobody_auth_override_test,
    auto_scope_test,

    bad_entity_type_test,
    bad_message_test,
    timed_out_request_test,
    crashed_request_test,
    stale_request_pruning_test,
    throttling_test,

    session_persistence_test,
    subscriptions_persistence_test,
    gs_server_session_clearing_test_api_level,
    gs_server_session_clearing_test_connection_level,
    service_availability_rpc_test,
    service_availability_graph_test,
    service_availability_handshake_test
]).

-define(KEY_FILE, ?TEST_RELEASE_ETC_DIR("certs/web_key.pem")).
-define(CERT_FILE, ?TEST_RELEASE_ETC_DIR("certs/web_cert.pem")).
-define(CHAIN_FILE, ?TEST_RELEASE_ETC_DIR("certs/web_chain.pem")).
-define(TRUSTED_CACERTS_FILE, ?TEST_RELEASE_ETC_DIR("cacerts/OneDataTestWebServerCa.pem")).

-define(SSL_OPTS(Config), [{secure, only_verify_peercert}, {cacerts, get_trusted_cacerts(Config)}]).
-define(DUMMY_IP, {13, 190, 241, 56}).

-define(wait_until_true(Term), ?assertEqual(true, Term, 50)).
-define(ATTEMPTS, 20).


%%%===================================================================
%%% API functions
%%%===================================================================


all() ->
    ?ALL(?TEST_CASES).


handshake_test(Config) ->
    [handshake_test_base(Config, ProtoVersion) || ProtoVersion <- ?SUPPORTED_PROTO_VERSIONS].

handshake_test_base(Config, ProtoVersion) ->
    % Try to connect with no cookie - should be treated as anonymous
    spawn_client(Config, ProtoVersion, undefined, ?SUB(nobody)),

    % Try to connect with user 1 token
    spawn_client(Config, ProtoVersion, {token, ?USER_1_TOKEN}, ?SUB(user, ?USER_1)),

    % Try to connect with user 2 token
    spawn_client(Config, ProtoVersion, {token, ?USER_2_TOKEN}, ?SUB(user, ?USER_2)),

    % Try to connect with user 1 token requiring session cookies...
    spawn_client(
        Config, ProtoVersion, {with_http_cookies, {token, ?USER_1_TOKEN_REQUIRING_COOKIES}, ?DUMMY_COOKIES},
        ?SUB(user, ?USER_1)
    ),
    % ... it should not succeed if there are no cookies provided
    spawn_client(
        Config, ProtoVersion, {with_http_cookies, {token, ?USER_1_TOKEN_REQUIRING_COOKIES}, []},
        ?ERR_UNAUTHORIZED(undefined)
    ),
    spawn_client(
        Config, ProtoVersion, {token, ?USER_1_TOKEN_REQUIRING_COOKIES},
        ?ERR_UNAUTHORIZED(undefined)
    ),

    % Try to connect with bad token
    spawn_client(Config, ProtoVersion, {token, <<"bkkwksdf">>}, ?ERR_UNAUTHORIZED(undefined)),

    % Try to connect with provider token
    spawn_client(Config, ProtoVersion, {token, ?PROVIDER_1_TOKEN}, ?SUB(?ONEPROVIDER, ?PROVIDER_1)),

    % Try to connect with bad protocol version
    SuppVersions = gs_protocol:supported_versions(),
    spawn_client(Config, [lists:max(SuppVersions) + 1], undefined, ?ERR_BAD_VERSION(SuppVersions)).


rpc_req_test(Config) ->
    [rpc_req_test_base(Config, ProtoVersion) || ProtoVersion <- ?SUPPORTED_PROTO_VERSIONS].

rpc_req_test_base(Config, ProtoVersion) ->
    Client1 = spawn_client(Config, ProtoVersion, {token, ?USER_1_TOKEN}, ?SUB(user, ?USER_1)),
    Client2 = spawn_client(Config, ProtoVersion, {token, ?USER_2_TOKEN}, ?SUB(user, ?USER_2)),

    ?assertMatch(
        {ok, #gs_resp_rpc{result = #{<<"a">> := <<"b">>}}},
        gs_client:rpc_request(Client1, <<"user1Fun">>, #{<<"a">> => <<"b">>})
    ),
    ?assertMatch(
        ?ERR_FORBIDDEN(_),
        gs_client:rpc_request(Client1, <<"user2Fun">>, #{<<"a">> => <<"b">>})
    ),
    ?assertMatch(
        {ok, #gs_resp_rpc{result = #{<<"a">> := <<"b">>}}},
        gs_client:rpc_request(Client2, <<"user2Fun">>, #{<<"a">> => <<"b">>})
    ),
    ?assertMatch(
        ?ERR_FORBIDDEN(_),
        gs_client:rpc_request(Client2, <<"user1Fun">>, #{<<"a">> => <<"b">>})
    ),
    ?assertMatch(
        ?ERR_RPC_UNDEFINED,
        gs_client:rpc_request(Client1, <<"nonExistentFun">>, #{<<"a">> => <<"b">>})
    ).


async_req_test(Config) ->
    [async_req_test_base(Config, ProtoVersion) || ProtoVersion <- ?SUPPORTED_PROTO_VERSIONS].

async_req_test_base(Config, ProtoVersion) ->
    Client1 = spawn_client(Config, ProtoVersion, {token, ?USER_1_TOKEN}, ?SUB(user, ?USER_1)),

    ?assertEqual(
        {ok, #gs_resp_rpc{result = #{<<"someDummy">> => <<"arguments127">>}}},
        await_result(async_request_long_operation(Client1))
    ).


graph_req_test(Config) ->
    [graph_req_test_base(Config, ProtoVersion) || ProtoVersion <- ?SUPPORTED_PROTO_VERSIONS].

graph_req_test_base(Config, ProtoVersion) ->
    User1Data = (?USER_DATA_WITHOUT_GRI(?USER_1))#{
        <<"gri">> => gri:serialize(#gri{type = od_user, id = ?USER_1, aspect = instance}),
        <<"revision">> => 1
    },

    Client1 = spawn_client(Config, ProtoVersion, {token, ?USER_1_TOKEN}, ?SUB(user, ?USER_1)),
    Client2 = spawn_client(Config, ProtoVersion, {token, ?USER_2_TOKEN}, ?SUB(user, ?USER_2)),

    ?assertMatch(
        {ok, #gs_resp_graph{data_format = resource, data = User1Data}},
        gs_client:graph_request(Client1, #gri{
            type = od_user, id = ?USER_1, aspect = instance
        }, get)
    ),

    ?assertMatch(
        ?ERR_FORBIDDEN(_),
        gs_client:graph_request(Client2, #gri{
            type = od_user, id = ?USER_1, aspect = instance
        }, get)
    ),

    % User 2 should be able to get user 1 data through space ?SPACE_1
    ?assertMatch(
        {ok, #gs_resp_graph{data_format = resource, data = User1Data}},
        gs_client:graph_request(Client2, #gri{
            type = od_user, id = ?USER_1, aspect = instance
        }, get, #{}, false, ?THROUGH_SPACE(?SPACE_1))
    ),

    % User should be able to get it's own data using "self" as id
    ?assertMatch(
        {ok, #gs_resp_graph{data_format = resource, data = User1Data}},
        gs_client:graph_request(Client1, #gri{
            type = od_user, id = ?SELF, aspect = instance
        }, get)
    ),

    ?assertMatch(
        {ok, #gs_resp_graph{}},
        gs_client:graph_request(Client1, #gri{
            type = od_user, id = ?USER_1, aspect = instance
        }, update, #{
            <<"name">> => <<"newName">>
        })
    ),

    ?assertMatch(
        ?ERR_BAD_VALUE_STRING(<<"name">>),
        gs_client:graph_request(Client1, #gri{
            type = od_user, id = ?USER_1, aspect = instance
        }, update, #{
            <<"name">> => 1234
        })
    ),

    ?assertMatch(
        ?ERR_MISSING_REQUIRED_VALUE(<<"name">>),
        gs_client:graph_request(Client1, #gri{
            type = od_user, id = ?USER_1, aspect = instance
        }, update, #{})
    ),

    ?assertMatch(
        ?ERR_FORBIDDEN(_),
        gs_client:graph_request(Client2, #gri{
            type = od_user, id = ?USER_1, aspect = instance
        }, update)
    ),

    ?assertMatch(
        ?ERR_FORBIDDEN(_),
        gs_client:graph_request(Client2, #gri{
            type = od_user, id = ?USER_1, aspect = instance
        }, delete)
    ),

    ?assertMatch(
        {ok, #gs_resp_graph{}},
        gs_client:graph_request(Client1, #gri{
            type = od_user, id = ?USER_1, aspect = instance
        }, delete)
    ),

    NewSpaceGRI = gri:serialize(
        #gri{type = od_space, id = ?SPACE_1, aspect = instance}
    ),
    ?assertMatch(
        {ok, #gs_resp_graph{data_format = resource, data = #{
            <<"gri">> := NewSpaceGRI,
            <<"name">> := ?SPACE_1_NAME,
            <<"revision">> := 1
        }}},
        gs_client:graph_request(Client1, #gri{
            type = od_space, id = undefined, aspect = instance
        }, create, #{<<"name">> => ?SPACE_1_NAME}, false, ?AS_USER(?USER_1))
    ),

    % Make sure "self" works in auth hints
    NewGroupGRI = gri:serialize(
        #gri{type = od_group, id = ?GROUP_1, aspect = instance}
    ),
    ?assertMatch(
        {ok, #gs_resp_graph{data_format = resource, data = #{
            <<"gri">> := NewGroupGRI,
            <<"name">> := ?GROUP_1_NAME,
            <<"revision">> := 1
        }}},
        gs_client:graph_request(Client1, #gri{
            type = od_group, id = undefined, aspect = instance
        }, create, #{<<"name">> => ?GROUP_1_NAME}, false, ?AS_USER(?SELF))
    ),

    % Test creating a value rather than resource
    Value = 1293462394,
    ?assertMatch(
        {ok, #gs_resp_graph{data_format = value, data = Value}},
        gs_client:graph_request(Client1, #gri{
            type = od_group, id = ?GROUP_1, aspect = int_value
        }, create, #{<<"value">> => integer_to_binary(Value)})
    ).


batch_req_test(Config) ->
    [batch_req_test_base(Config, ProtoVersion) || ProtoVersion <- ?SUPPORTED_PROTO_VERSIONS].

batch_req_test_base(Config, ProtoVersion) ->
    User1Data = (?USER_DATA_WITHOUT_GRI(?USER_1))#{
        <<"gri">> => gri:serialize(#gri{type = od_user, id = ?USER_1, aspect = instance}),
        <<"revision">> => 1
    },
    User2Data = (?USER_DATA_WITHOUT_GRI(?USER_2))#{
        <<"gri">> => gri:serialize(#gri{type = od_user, id = ?USER_2, aspect = instance}),
        <<"revision">> => 1
    },

    Client1 = spawn_client(Config, ProtoVersion, {token, ?USER_1_TOKEN}, ?SUB(user, ?USER_1)),
    Client2 = spawn_client(Config, ProtoVersion, {token, ?USER_2_TOKEN}, ?SUB(user, ?USER_2)),

    ?assertMatch(
        {ok, #gs_resp_batch{responses = [
            #gs_resp{subtype = graph, id = <<"1">>, response = #gs_resp_graph{
                data_format = resource, data = User1Data
            }},
            #gs_resp{subtype = graph, id = <<"2">>, response = #gs_resp_graph{
                data_format = resource, data = User1Data
            }},
            #gs_resp{subtype = graph, id = <<"3">>, response = #gs_resp_graph{
                data_format = resource, data = User1Data
            }}
        ]}},
        gs_client:batch_request(Client1, [
            #gs_req{subtype = graph, id = <<"1">>, request = #gs_req_graph{
                gri = #gri{type = od_user, id = ?USER_1, aspect = instance},
                operation = get
            }},
            #gs_req{subtype = graph, id = <<"2">>, request = #gs_req_graph{
                gri = #gri{type = od_user, id = ?USER_1, aspect = instance},
                operation = get
            }},
            #gs_req{subtype = graph, id = <<"3">>, request = #gs_req_graph{
                gri = #gri{type = od_user, id = ?USER_1, aspect = instance},
                operation = get
            }}
        ])
    ),

    ?assertMatch(
        {ok, #gs_resp_batch{responses = [
            #gs_resp{subtype = graph, id = <<"1">>, error = ?ERR_FORBIDDEN},
            #gs_resp{subtype = graph, id = <<"2">>, response = #gs_resp_graph{
                data_format = resource, data = User2Data
            }},
            #gs_resp{subtype = graph, id = <<"3">>, error = ?ERR_FORBIDDEN}
        ]}},
        gs_client:batch_request(Client2, [
            #gs_req{subtype = graph, id = <<"1">>, request = #gs_req_graph{
                gri = #gri{type = od_user, id = ?USER_1, aspect = instance},
                operation = get
            }},
            #gs_req{subtype = graph, id = <<"2">>, request = #gs_req_graph{
                gri = #gri{type = od_user, id = ?USER_2, aspect = instance},
                operation = get,
                subscribe = true
            }},
            #gs_req{subtype = graph, id = <<"3">>, request = #gs_req_graph{
                gri = #gri{type = od_user, id = ?USER_1, aspect = instance},
                operation = get
            }}
        ])
    ),

    RpcArgs = #{<<"x">> => 13},
    ?assertMatch(
        {ok, #gs_resp_batch{responses = [
            #gs_resp{subtype = batch, id = <<"2">>, response = #gs_resp_batch{responses = [
                #gs_resp{id = <<"2.1">>, response = #gs_resp_unsub{}},
                #gs_resp{subtype = batch, id = <<"2.2">>, response = #gs_resp_batch{responses = [
                    #gs_resp{subtype = graph, id = <<"2.2.1">>, response = #gs_resp_graph{
                        data_format = resource, data = User2Data
                    }},
                    #gs_resp{subtype = rpc, id = <<"2.2.2">>, error = ?ERR_FORBIDDEN}
                ]}},
                #gs_resp{subtype = graph, id = <<"2.3">>, error = ?ERR_FORBIDDEN}
            ]}},
            #gs_resp{subtype = graph, id = <<"1">>, error = ?ERR_FORBIDDEN},
            #gs_resp{subtype = rpc, id = <<"3">>, response = #gs_resp_rpc{result = RpcArgs}}
        ]}},
        gs_client:batch_request(Client2, [
            #gs_req{subtype = batch, id = <<"2">>, request = #gs_req_batch{
                requests = [
                    % Client2 has subscribed for that record in the previous request
                    #gs_req{subtype = unsub, id = <<"2.1">>, request = #gs_req_unsub{
                        gri = #gri{type = od_user, id = ?USER_2, aspect = instance}
                    }},
                    #gs_req{subtype = batch, id = <<"2.2">>, request = #gs_req_batch{
                        requests = [
                            #gs_req{subtype = graph, id = <<"2.2.1">>, request = #gs_req_graph{
                                gri = #gri{type = od_user, id = ?USER_2, aspect = instance},
                                operation = get
                            }},
                            #gs_req{subtype = rpc, id = <<"2.2.2">>, request = #gs_req_rpc{
                                function = <<"user1Fun">>,
                                args = RpcArgs
                            }}
                        ]
                    }},
                    #gs_req{subtype = graph, id = <<"2.3">>, request = #gs_req_graph{
                        gri = #gri{type = od_user, id = ?USER_1, aspect = instance},
                        operation = get
                    }}
                ]
            }},
            #gs_req{subtype = graph, id = <<"1">>, request = #gs_req_graph{
                gri = #gri{type = od_user, id = ?USER_1, aspect = instance},
                operation = get
            }},
            #gs_req{subtype = rpc, id = <<"3">>, request = #gs_req_rpc{
                function = <<"user2Fun">>,
                args = RpcArgs
            }}
        ])
    ).


subscribe_test(Config) ->
    [subscribe_test_base(Config, ProtoVersion) || ProtoVersion <- ?SUPPORTED_PROTO_VERSIONS].

subscribe_test_base(Config, ProtoVersion) ->
    GathererPid = spawn(fun() ->
        gatherer_loop(#{})
    end),

    User1Data = (?USER_DATA_WITHOUT_GRI(?USER_1))#{
        <<"gri">> => gri:serialize(#gri{type = od_user, id = ?USER_1, aspect = instance}),
        <<"revision">> => 1
    },
    User2Data = (?USER_DATA_WITHOUT_GRI(?USER_2))#{
        <<"gri">> => gri:serialize(#gri{type = od_user, id = ?USER_2, aspect = instance}),
        <<"revision">> => 1
    },

    Client1 = spawn_client(Config, ProtoVersion, {token, ?USER_1_TOKEN}, ?SUB(user, ?USER_1), fun(Push) ->
        GathererPid ! {gather_message, client1, Push}
    end),

    Client2 = spawn_client(Config, ProtoVersion, {token, ?USER_2_TOKEN}, ?SUB(user, ?USER_2), fun(Push) ->
        GathererPid ! {gather_message, client2, Push}
    end),

    ?assertMatch(
        {ok, #gs_resp_graph{data_format = resource, data = User1Data}},
        gs_client:graph_request(Client1, #gri{
            type = od_user, id = ?USER_1, aspect = instance
        }, get, #{}, true)
    ),

    User1NameSubstring = binary:part(maps:get(<<"name">>, User1Data), 0, 4),

    ?assertMatch(
        {ok, #gs_resp_graph{data_format = resource, data = #{
            <<"nameSubstring">> := User1NameSubstring,
            <<"revision">> := 1
        }}},
        gs_client:graph_request(Client1, #gri{
            type = od_user, id = ?USER_1, aspect = {name_substring, <<"4">>}
        }, get, #{}, true)
    ),

    ?assertMatch(
        {ok, #gs_resp_graph{data_format = resource, data = User2Data}},
        gs_client:graph_request(Client2, #gri{
            type = od_user, id = ?USER_2, aspect = instance
        }, get, #{}, true)
    ),

    User2NameSubstring = binary:part(maps:get(<<"name">>, User2Data), 0, 6),

    ?assertMatch(
        {ok, #gs_resp_graph{data_format = resource, data = #{
            <<"nameSubstring">> := User2NameSubstring,
            <<"revision">> := 1
        }}},
        gs_client:graph_request(Client2, #gri{
            type = od_user, id = ?USER_2, aspect = {name_substring, <<"6">>}
        }, get, #{}, true)
    ),

    ?assertMatch(
        {ok, #gs_resp_graph{}},
        gs_client:graph_request(Client1, #gri{
            type = od_user, id = ?USER_1, aspect = instance
        }, update, #{<<"name">> => <<"newName1">>})
    ),

    ?assertMatch(
        {ok, #gs_resp_graph{}},
        gs_client:graph_request(Client2, #gri{
            type = od_user, id = ?USER_2, aspect = instance
        }, update, #{<<"name">> => <<"newName2">>})
    ),

    ?assertMatch(
        {ok, #gs_resp_graph{}},
        gs_client:graph_request(Client1, #gri{
            type = od_user, id = ?USER_1, aspect = instance
        }, delete)
    ),

    NewUser1Data = User1Data#{
        <<"name">> => <<"newName1">>,
        <<"revision">> => 2
    },
    NewUser2Data = User2Data#{
        <<"name">> => <<"newName2">>,
        <<"revision">> => 2
    },

    NewUser1NameSubstring = binary:part(maps:get(<<"name">>, NewUser1Data), 0, 4),
    NewUser2NameSubstring = binary:part(maps:get(<<"name">>, NewUser2Data), 0, 6),

    ?wait_until_true(verify_message_present(GathererPid, client1, fun(Msg) ->
        case Msg of
            #gs_push_graph{gri = #gri{
                type = od_user, id = ?USER_1, aspect = instance
            }, change_type = updated, data = NewUser1Data} ->
                true;
            _ ->
                false
        end
    end)),

    ?wait_until_true(verify_message_present(GathererPid, client1, fun(Msg) ->
        case Msg of
            #gs_push_graph{gri = #gri{
                type = od_user, id = ?USER_1, aspect = {name_substring, <<"4">>}
            }, change_type = updated, data = #{
                <<"nameSubstring">> := NewUser1NameSubstring,
                <<"revision">> := 2
            }} ->
                true;
            _ ->
                false
        end
    end)),

    ?wait_until_true(verify_message_present(GathererPid, client1, fun(Msg) ->
        case Msg of
            #gs_push_graph{gri = #gri{
                type = od_user, id = ?USER_1, aspect = instance
            }, change_type = deleted, data = undefined} ->
                true;
            _ ->
                false
        end
    end)),

    ?wait_until_true(verify_message_present(GathererPid, client2, fun(Msg) ->
        case Msg of
            #gs_push_graph{gri = #gri{
                type = od_user, id = ?USER_2, aspect = instance
            }, change_type = updated, data = NewUser2Data} ->
                true;
            _ ->
                false
        end
    end)),

    ?wait_until_true(verify_message_present(GathererPid, client2, fun(Msg) ->
        case Msg of
            #gs_push_graph{gri = #gri{
                type = od_user, id = ?USER_2, aspect = {name_substring, <<"6">>}
            }, change_type = updated, data = #{
                <<"nameSubstring">> := NewUser2NameSubstring,
                <<"revision">> := 2
            }} ->
                true;
            _ ->
                false
        end
    end)).


parallel_requests_test(Config) ->
    [parallel_requests_test_base(Config, ProtoVersion) || ProtoVersion <- ?SUPPORTED_PROTO_VERSIONS].

parallel_requests_test_base(Config, ProtoVersion) ->
    ClientCount = 20,
    RequestCountPerClient = 1000,

    Clients = lists_utils:generate(fun(_) ->
        spawn_client(Config, ProtoVersion, {token, ?USER_1_TOKEN}, ?SUB(user, ?USER_1))
    end, ClientCount),

    MeasurementsMillis = lists:map(fun(Client) ->
        Stopwatch = stopwatch:start(),
        lists_utils:pmap(fun(_) ->
            RequestId = gs_client:async_request(Client, #gs_req{
                subtype = graph,
                request = #gs_req_graph{
                    gri = #gri{type = od_user, id = ?USER_1, aspect = instance},
                    operation = get,
                    subscribe = true
                }
            }),
            receive
                {result, RequestId, Result} ->
                    ?assertMatch({ok, _}, Result)
            after
                timer:seconds(60) ->
                    error(gather_timeout)
            end
        end, lists:seq(1, RequestCountPerClient), 100),
        stopwatch:read_millis(Stopwatch)
    end, Clients),

    MeasurementDump = str_utils:join_binary([<<"">>] ++ lists:map(fun({Ordinal, Millis}) ->
        str_utils:format_bin("#~2..0b -> ~B ms", [Ordinal, Millis])
    end, lists:enumerate(MeasurementsMillis)), <<"\n">>),
    ct:pal("Parallel requests: ~B clients, ~B requests each, total times: (avg: ~tp ms)~ts", [
        ClientCount,
        RequestCountPerClient,
        lists:sum(MeasurementsMillis) div length(MeasurementsMillis),
        MeasurementDump
    ]).


unsubscribe_test(Config) ->
    [unsubscribe_test_base(Config, ProtoVersion) || ProtoVersion <- ?SUPPORTED_PROTO_VERSIONS].

unsubscribe_test_base(Config, ProtoVersion) ->
    GathererPid = spawn(fun() ->
        gatherer_loop(#{})
    end),

    User1Data = (?USER_DATA_WITHOUT_GRI(?USER_1))#{
        <<"gri">> => gri:serialize(#gri{type = od_user, id = ?USER_1, aspect = instance}),
        <<"revision">> => 1
    },

    Client1 = spawn_client(Config, ProtoVersion, {token, ?USER_1_TOKEN}, ?SUB(user, ?USER_1), fun(Push) ->
        GathererPid ! {gather_message, client1, Push}
    end),

    ?assertMatch(
        {ok, #gs_resp_graph{data_format = resource, data = User1Data}},
        gs_client:graph_request(Client1, #gri{
            type = od_user, id = ?USER_1, aspect = instance
        }, get, #{}, true)
    ),

    ?assertMatch(
        {ok, #gs_resp_graph{}},
        gs_client:graph_request(Client1, #gri{
            type = od_user, id = ?USER_1, aspect = instance
        }, update, #{<<"name">> => <<"newName1">>})
    ),

    NewUser1Data = User1Data#{
        <<"name">> => <<"newName1">>,
        <<"revision">> => 2
    },

    ?wait_until_true(verify_message_present(GathererPid, client1, fun(Msg) ->
        case Msg of
            #gs_push_graph{gri = #gri{
                type = od_user, id = ?USER_1, aspect = instance
            }, change_type = updated, data = NewUser1Data} ->
                true;
            _ ->
                false
        end
    end)),

    ?assertMatch(
        {ok, #gs_resp_unsub{}},
        gs_client:unsub_request(Client1, #gri{
            type = od_user, id = ?USER_1, aspect = instance
        })
    ),

    ?assertMatch(
        {ok, #gs_resp_graph{}},
        gs_client:graph_request(Client1, #gri{
            type = od_user, id = ?USER_1, aspect = instance
        }, update, #{<<"name">> => <<"newName2">>})
    ),

    NewestUser1Data = User1Data#{
        <<"name">> => <<"newName2">>,
        <<"revision">> => 2
    },

    ?assert(verify_message_absent(GathererPid, client1, fun(Msg) ->
        case Msg of
            #gs_push_graph{gri = #gri{
                type = od_user, id = ?USER_1, aspect = instance
            }, change_type = updated, data = NewestUser1Data} ->
                true;
            _ ->
                false
        end
    end, 20)).


nosub_test(Config) ->
    [nosub_test_base(Config, ProtoVersion) || ProtoVersion <- ?SUPPORTED_PROTO_VERSIONS].

nosub_test_base(Config, ProtoVersion) ->
    GathererPid = spawn(fun() ->
        gatherer_loop(#{})
    end),

    User2Data = (?USER_DATA_WITHOUT_GRI(?USER_2))#{
        <<"gri">> => gri:serialize(#gri{type = od_user, id = ?USER_2, aspect = instance}),
        <<"revision">> => 1
    },

    Client1 = spawn_client(Config, ProtoVersion, {token, ?USER_1_TOKEN}, ?SUB(user, ?USER_1), fun(Push) ->
        GathererPid ! {gather_message, client1, Push}
    end),

    Client2 = spawn_client(Config, ProtoVersion, {token, ?USER_2_TOKEN}, ?SUB(user, ?USER_2), fun(Push) ->
        GathererPid ! {gather_message, client2, Push}
    end),

    ?assertMatch(
        ?ERR_FORBIDDEN(_),
        gs_client:graph_request(Client1, #gri{
            type = od_user, id = ?USER_2, aspect = instance
        }, get, #{}, true)
    ),

    ?assertMatch(
        {ok, #gs_resp_graph{data_format = resource, data = User2Data}},
        gs_client:graph_request(Client1, #gri{
            type = od_user, id = ?USER_2, aspect = instance
        }, get, #{}, true, ?THROUGH_SPACE(?SPACE_1))
    ),

    ?assertMatch(
        {ok, #gs_resp_graph{}},
        gs_client:graph_request(Client2, #gri{
            type = od_user, id = ?USER_2, aspect = instance
        }, update, #{<<"name">> => <<"newName1">>})
    ),

    NewUser2Data = User2Data#{
        <<"name">> => <<"newName1">>,
        <<"revision">> => 2
    },

    ?wait_until_true(verify_message_present(GathererPid, client1, fun(Msg) ->
        case Msg of
            #gs_push_graph{gri = #gri{
                type = od_user, id = ?USER_2, aspect = instance
            }, change_type = updated, data = NewUser2Data} ->
                true;
            _ ->
                false
        end
    end)),

    ?assertMatch(
        {ok, #gs_resp_graph{}},
        gs_client:graph_request(Client2, #gri{
            type = od_user, id = ?USER_2, aspect = instance
        }, update, #{<<"name">> => ?USER_NAME_THAT_CAUSES_NO_ACCESS_THROUGH_SPACE})
    ),

    NewestUser2Data = User2Data#{
        <<"name">> => ?USER_NAME_THAT_CAUSES_NO_ACCESS_THROUGH_SPACE,
        <<"revision">> => 2
    },

    ?wait_until_true(verify_message_present(GathererPid, client1, fun(Msg) ->
        case Msg of
            #gs_push_nosub{gri = #gri{
                type = od_user, id = ?USER_2, aspect = instance
            }, reason = forbidden} ->
                true;
            _ ->
                false
        end
    end)),

    ?assert(verify_message_absent(GathererPid, client1, fun(Msg) ->
        case Msg of
            #gs_push_graph{gri = #gri{
                type = od_user, id = ?USER_2, aspect = instance
            }, change_type = updated, data = NewestUser2Data} ->
                true;
            _ ->
                false
        end
    end, 20)).


auth_override_test(Config) ->
    [auth_override_test_base(Config, ProtoVersion) || ProtoVersion <- ?SUPPORTED_PROTO_VERSIONS].

auth_override_test_base(Config, ProtoVersion) ->
    GathererPid = spawn(fun() ->
        gatherer_loop(#{})
    end),

    UserClient = spawn_client(Config, ProtoVersion, {token, ?USER_1_TOKEN}, ?SUB(user, ?USER_1), fun(Push) ->
        GathererPid ! {gather_message, client1, Push}
    end),

    ProviderClient = spawn_client(Config, ProtoVersion, {token, ?PROVIDER_1_TOKEN}, ?SUB(?ONEPROVIDER, ?PROVIDER_1), fun(Push) ->
        GathererPid ! {gather_message, provider_client, Push}
    end),

    User2Data = (?USER_DATA_WITHOUT_GRI(?USER_2))#{
        <<"gri">> => gri:serialize(#gri{type = od_user, id = ?USER_2, aspect = instance}),
        <<"revision">> => 1
    },

    GetUserReq = #gs_req{
        subtype = graph,
        auth_override = #auth_override{
            client_auth = {token, ?USER_2_TOKEN},
            peer_ip = ?WHITELISTED_IP,
            interface = ?WHITELISTED_INTERFACE,
            consumer_token = ?WHITELISTED_CONSUMER_TOKEN
        },
        request = #gs_req_graph{
            gri = #gri{type = od_user, id = ?SELF, aspect = instance},
            operation = get,
            subscribe = true
        }
    },

    % Provider should be able to get user's 2 data by possessing his token and
    % using it as auth override during the request.
    ?assertMatch(
        {ok, #gs_resp_graph{data_format = resource, data = User2Data}},
        gs_client:sync_request(ProviderClient, GetUserReq)
    ),

    % Auth override is allowed only for providers
    ?assertMatch(
        ?ERR_FORBIDDEN(_),
        gs_client:sync_request(UserClient, GetUserReq)
    ),

    % Subscribing should work too - provider should be receiving future changes of
    % user's 2 record.
    NewUser2Name = <<"newName2">>,
    NewUser2Data = User2Data#{
        <<"name">> => NewUser2Name,
        <<"revision">> => 2
    },

    ?assertMatch(
        {ok, #gs_resp_graph{}},
        gs_client:sync_request(ProviderClient, GetUserReq#gs_req{
            request = #gs_req_graph{
                gri = #gri{type = od_user, id = ?SELF, aspect = instance},
                operation = update,
                data = #{<<"name">> => NewUser2Name}
            }
        })
    ),

    ?wait_until_true(verify_message_present(GathererPid, provider_client, fun(Msg) ->
        case Msg of
            #gs_push_graph{
                gri = #gri{type = od_user, id = ?USER_2, aspect = instance},
                change_type = updated,
                data = NewUser2Data
            } ->
                true;
            _ ->
                false
        end
    end)),

    % Check if auth override data that is not whitelisted causes an error.
    % For test purposes, there are defined blacklisted ip, interface and consumer token.
    ReqWithOverrideData = fun(PeerIp, Interface, ConsumerToken) ->
        GetUserReq#gs_req{
            auth_override = #auth_override{
                client_auth = {token, ?USER_2_TOKEN},
                peer_ip = PeerIp,
                interface = Interface,
                consumer_token = ConsumerToken
            }
        } end,

    % Additional auth override options are supported since version 4
    case ProtoVersion > 3 of
        false ->
            ok;
        true ->
            ?assertMatch(?ERR_UNAUTHORIZED(_), gs_client:sync_request(ProviderClient, ReqWithOverrideData(
                ?BLACKLISTED_IP, ?WHITELISTED_INTERFACE, ?WHITELISTED_CONSUMER_TOKEN
            ))),
            ?assertMatch(?ERR_UNAUTHORIZED(_), gs_client:sync_request(ProviderClient, ReqWithOverrideData(
                ?WHITELISTED_IP, ?BLACKLISTED_INTERFACE, ?WHITELISTED_CONSUMER_TOKEN
            ))),
            ?assertMatch(?ERR_UNAUTHORIZED(_), gs_client:sync_request(ProviderClient, ReqWithOverrideData(
                ?WHITELISTED_IP, ?WHITELISTED_INTERFACE, ?BLACKLISTED_CONSUMER_TOKEN
            )))
    end.


nobody_auth_override_test(Config) ->
    [nobody_auth_override_test_base(Config, ProtoVersion) || ProtoVersion <- ?SUPPORTED_PROTO_VERSIONS].

nobody_auth_override_test_base(Config, ProtoVersion) ->
    Client1 = spawn_client(Config, ProtoVersion, {token, ?PROVIDER_1_TOKEN}, ?SUB(?ONEPROVIDER, ?PROVIDER_1)),

    % Request with auth based on connection owner
    ?assertMatch(
        {ok, #gs_resp_graph{data_format = resource, data = ?SHARE_DATA_MATCHER(<<"private">>)}},
        gs_client:sync_request(Client1, #gs_req{
            subtype = graph,
            request = #gs_req_graph{
                gri = #gri{type = od_share, id = ?SHARE, aspect = instance, scope = auto},
                operation = get
            }
        })
    ),

    % Request with nobody auth override
    ?assertMatch(
        {ok, #gs_resp_graph{data_format = resource, data = ?SHARE_DATA_MATCHER(<<"public">>)}},
        gs_client:sync_request(Client1, #gs_req{
            subtype = graph,
            auth_override = #auth_override{
                client_auth = nobody,
                peer_ip = ?WHITELISTED_IP,
                interface = ?WHITELISTED_INTERFACE,
                consumer_token = ?WHITELISTED_CONSUMER_TOKEN
            },
            request = #gs_req_graph{
                gri = #gri{type = od_share, id = ?SHARE, aspect = instance, scope = auto},
                operation = get
            }
        })
    ).


auto_scope_test(Config) ->
    [auto_scope_test_base(Config, ProtoVersion) || ProtoVersion <- ?SUPPORTED_PROTO_VERSIONS].

auto_scope_test_base(Config, ProtoVersion) ->
    [Node | _] = ?config(cluster_worker_nodes, Config),

    GathererPid = spawn(fun() ->
        gatherer_loop(#{})
    end),

    graph_sync_mocks:mock_max_scope_towards_handle_service(Config, ?USER_1, none),
    graph_sync_mocks:mock_max_scope_towards_handle_service(Config, ?USER_2, public),

    Client1 = spawn_client(Config, ProtoVersion, {token, ?USER_1_TOKEN}, ?SUB(user, ?USER_1), fun(Push) ->
        GathererPid ! {gather_message, client1, Push}
    end),

    Client2 = spawn_client(Config, ProtoVersion, {token, ?USER_2_TOKEN}, ?SUB(user, ?USER_2), fun(Push) ->
        GathererPid ! {gather_message, client2, Push}
    end),

    HsGRI = #gri{type = od_handle_service, id = ?HANDLE_SERVICE, aspect = instance},
    HsGRIAuto = HsGRI#gri{scope = auto},
    HsGRIAutoStr = gri:serialize(HsGRIAuto),

    ?assertEqual(
        ?ERR_FORBIDDEN(undefined),
        gs_client:graph_request(Client1, HsGRI#gri{scope = auto}, get, #{}, true)
    ),

    ?assertEqual(
        ?ERR_FORBIDDEN(undefined),
        gs_client:graph_request(Client2, HsGRI#gri{scope = shared}, get, #{}, true)
    ),

    ?assertEqual(
        {ok, #gs_resp_graph{data_format = resource, data = #{
            <<"gri">> => HsGRIAutoStr, <<"public">> => <<"pub1">>,
            <<"revision">> => 1
        }}},
        gs_client:graph_request(Client2, HsGRI#gri{scope = auto}, get, #{}, true)
    ),

    graph_sync_mocks:mock_max_scope_towards_handle_service(Config, ?USER_1, shared),

    ?assertEqual(
        {ok, #gs_resp_graph{data_format = resource, data = #{
            <<"gri">> => HsGRIAutoStr, <<"public">> => <<"pub1">>, <<"shared">> => <<"sha1">>,
            <<"revision">> => 1
        }}},
        gs_client:graph_request(Client1, HsGRI#gri{scope = auto}, get, #{}, true)
    ),

    graph_sync_mocks:mock_max_scope_towards_handle_service(Config, ?USER_2, private),

    Revision2 = 5,
    HServiceData2 = ?HANDLE_SERVICE_DATA(<<"pub2">>, <<"sha2">>, <<"pro2">>, <<"pri2">>),
    rpc:call(Node, gs_server, updated, [od_handle_service, ?HANDLE_SERVICE, {HServiceData2, Revision2}]),

    ?wait_until_true(verify_message_present(GathererPid, client1, fun(Msg) ->
        Expected = ?LIMIT_HANDLE_SERVICE_DATA(shared, HServiceData2)#{
            <<"gri">> => HsGRIAutoStr,
            <<"revision">> => Revision2
        },
        case Msg of
            #gs_push_graph{gri = HsGRIAuto, change_type = updated, data = Expected} ->
                true;
            _ ->
                false
        end
    end)),

    ?wait_until_true(verify_message_present(GathererPid, client2, fun(Msg) ->
        Expected = ?LIMIT_HANDLE_SERVICE_DATA(private, HServiceData2)#{
            <<"gri">> => HsGRIAutoStr,
            <<"revision">> => Revision2
        },
        case Msg of
            #gs_push_graph{gri = HsGRIAuto, change_type = updated, data = Expected} ->
                true;
            _ ->
                false
        end
    end)),

    graph_sync_mocks:mock_max_scope_towards_handle_service(Config, ?USER_1, protected),
    graph_sync_mocks:mock_max_scope_towards_handle_service(Config, ?USER_2, none),

    Revision3 = 12,
    HServiceData3 = ?HANDLE_SERVICE_DATA(<<"pub3">>, <<"sha3">>, <<"pro3">>, <<"pri3">>),
    rpc:call(Node, gs_server, updated, [od_handle_service, ?HANDLE_SERVICE, {HServiceData3, Revision3}]),

    ?wait_until_true(verify_message_present(GathererPid, client1, fun(Msg) ->
        Expected = ?LIMIT_HANDLE_SERVICE_DATA(protected, HServiceData3)#{
            <<"gri">> => HsGRIAutoStr,
            <<"revision">> => Revision3
        },
        case Msg of
            #gs_push_graph{gri = HsGRIAuto, change_type = updated, data = Expected} ->
                true;
            _ ->
                false
        end
    end)),

    ?wait_until_true(verify_message_present(GathererPid, client2, fun(Msg) ->
        case Msg of
            #gs_push_nosub{gri = HsGRIAuto, reason = forbidden} ->
                true;
            _ ->
                false
        end
    end)),

    % Check if create with auto scope works as expected
    graph_sync_mocks:mock_max_scope_towards_handle_service(Config, ?USER_2, protected),
    ?assertEqual(
        {ok, #gs_resp_graph{data_format = resource, data = #{
            <<"gri">> => HsGRIAutoStr, <<"public">> => <<"pub1">>,
            <<"shared">> => <<"sha1">>, <<"protected">> => <<"pro1">>,
            <<"revision">> => 1
        }}},
        gs_client:graph_request(Client2, HsGRI#gri{scope = auto}, create, #{}, true)
    ),

    Revision4 = 14,
    HServiceData4 = ?HANDLE_SERVICE_DATA(<<"pub4">>, <<"sha4">>, <<"pro4">>, <<"pri4">>),
    rpc:call(Node, gs_server, updated, [od_handle_service, ?HANDLE_SERVICE, {HServiceData4, Revision4}]),

    ?wait_until_true(verify_message_present(GathererPid, client2, fun(Msg) ->
        Expected = ?LIMIT_HANDLE_SERVICE_DATA(protected, HServiceData4)#{
            <<"gri">> => HsGRIAutoStr,
            <<"revision">> => Revision4
        },
        case Msg of
            #gs_push_graph{gri = HsGRIAuto, change_type = updated, data = Expected} ->
                true;
            _ ->
                false
        end
    end)).


bad_entity_type_test(Config) ->
    [bad_entity_type_test_base(Config, ProtoVersion) || ProtoVersion <- ?SUPPORTED_PROTO_VERSIONS].

bad_entity_type_test_base(Config, ProtoVersion) ->
    Client1 = spawn_client(Config, ProtoVersion, {token, ?USER_1_TOKEN}, ?SUB(user, ?USER_1)),

    ?assertMatch(
        ?ERR_BAD_GRI,
        % op_file entity type is not supported (as per gs_logic_plugin)
        gs_client:graph_request(Client1, #gri{
            type = op_file, id = <<"123">>, aspect = instance
        }, get, #{}, false)
    ).


bad_message_test(Config) ->
    [bad_message_test_base(Config, ProtoVersion) || ProtoVersion <- ?SUPPORTED_PROTO_VERSIONS].

bad_message_test_base(Config, ProtoVersion) ->
    GathererPid = spawn(fun() ->
        gatherer_loop(#{})
    end),

    Client1 = spawn_client(Config, ProtoVersion, {token, ?USER_1_TOKEN}, ?SUB(user, ?USER_1), fun(Push) ->
        GathererPid ! {gather_message, client1, Push}
    end),

    BadMessageData = <<"nonsense">>,
    % uses the internals of the gs_client to send data directly to the server
    Client1 ! {push, BadMessageData},

    ?wait_until_true(verify_message_present(GathererPid, client1, fun(Msg) ->
        case Msg of
            #gs_push_error{error = ?ERR_BAD_MESSAGE(BadMessageData)} ->
                true;
            _ ->
                false
        end
    end)).


timed_out_request_test(Config) ->
    [timed_out_request_test_base(Config, ProtoVersion) || ProtoVersion <- ?SUPPORTED_PROTO_VERSIONS].

timed_out_request_test_base(Config, ProtoVersion) ->
    Nodes = ?config(cluster_worker_nodes, Config),
    % set a very low threshold - the long lasting operation in graph_sync_mocks takes 20 seconds
    test_utils:set_env(Nodes, cluster_worker, graph_sync_request_processing_timeout_sec, 5),

    Client1 = spawn_client(Config, ProtoVersion, {token, ?USER_1_TOKEN}, ?SUB(user, ?USER_1)),

    ?assertEqual(?ERROR_TIMEOUT, await_result(async_request_long_operation(Client1))).


crashed_request_test(Config) ->
    [crashed_request_test_base(Config, ProtoVersion) || ProtoVersion <- ?SUPPORTED_PROTO_VERSIONS].

crashed_request_test_base(Config, ProtoVersion) ->
    Client1 = spawn_client(Config, ProtoVersion, {token, ?USER_1_TOKEN}, ?SUB(user, ?USER_1)),

    ?assertMatch(
        ?ERR_INTERNAL_SERVER_ERROR(_),
        gs_client:rpc_request(Client1, <<"crashingOperation">>, #{})
    ).


stale_request_pruning_test(Config) ->
    [stale_request_pruning_test_base(Config, ProtoVersion) || ProtoVersion <- ?SUPPORTED_PROTO_VERSIONS].

stale_request_pruning_test_base(Config, ProtoVersion) ->
    Nodes = ?config(cluster_worker_nodes, Config),
    % pruning is done on heartbeats
    test_utils:set_env(Nodes, cluster_worker, graph_sync_websocket_keepalive, timer:seconds(5)),
    % set a very low threshold - the long lasting operation in graph_sync_mocks takes 20 seconds
    test_utils:set_env(Nodes, cluster_worker, graph_sync_stale_request_threshold_sec, 5),
    % test that identity filtering does not cause any errors
    test_utils:set_env(Nodes, cluster_worker, graph_sync_verbose_logs_identity_filter, [
        <<"usr-", (?USER_1)/binary>>
    ]),

    NumberOfRequestsOfEachType = 10,

    Client = spawn_client(Config, ProtoVersion, {token, ?USER_1_TOKEN}, ?SUB(user, ?USER_1)),
    User1Data = (?USER_DATA_WITHOUT_GRI(?USER_1))#{
        <<"gri">> => gri:serialize(#gri{type = od_user, id = ?USER_1, aspect = instance}),
        <<"revision">> => 1
    },

    LongRequests = lists_utils:generate(fun(_) ->
        async_request_long_operation(Client)
    end, NumberOfRequestsOfEachType),

    GraphRequests = lists_utils:generate(fun(_) ->
        gs_client:async_request(Client, #gs_req{subtype = graph, request = #gs_req_graph{
            operation = get,
            gri = #gri{type = od_user, id = ?USER_1, aspect = instance}
        }})
    end, NumberOfRequestsOfEachType),

    % stale requests should be pruned and the ERROR_TIMEOUT error should be returned
    lists:foreach(fun(LongReqId) ->
        ?assertEqual(?ERROR_TIMEOUT, await_result(LongReqId))
    end, LongRequests),

    % regular (quick) graph requests should not be affected
    lists:foreach(fun(GraphReqId) ->
        ?assertMatch(
            {ok, #gs_resp_graph{data_format = resource, data = User1Data}},
            await_result(GraphReqId)
        )
    end, GraphRequests).


throttling_test(Config) ->
    [throttling_test_base(Config, ProtoVersion) || ProtoVersion <- ?SUPPORTED_PROTO_VERSIONS].

throttling_test_base(Config, ProtoVersion) ->
    Nodes = ?config(cluster_worker_nodes, Config),

    WorkerPoolSize = 6,
    test_utils:set_env(Nodes, cluster_worker, graph_sync_max_pool_usage_per_connection, 0.8),

    {_, []} = utils:rpc_multicall(Nodes, gs_worker_pool, stop, []),
    {_, []} = utils:rpc_multicall(Nodes, gs_worker_pool, init, [WorkerPoolSize]),

    ThrottledClient = spawn_client(Config, ProtoVersion, {token, ?USER_1_TOKEN}, ?SUB(user, ?USER_1)),
    WellBehavedClient = spawn_client(Config, ProtoVersion, {token, ?USER_1_TOKEN}, ?SUB(user, ?USER_1)),
    User1Data = (?USER_DATA_WITHOUT_GRI(?USER_1))#{
        <<"gri">> => gri:serialize(#gri{type = od_user, id = ?USER_1, aspect = instance}),
        <<"revision">> => 1
    },

    % these take ~20 seconds; send (2 * pool size) requests to test that they are properly throttled
    LongRequests = lists_utils:generate(fun(_) ->
        async_request_long_operation(ThrottledClient)
    end, WorkerPoolSize * 2),

    % wait to make sure all above requests have been sent
    timer:sleep(5000),

    % below requests are quick and should make it to the pool (the throttled client must not
    % be able to take all the processing slots, as we set graph_sync_max_pool_usage_per_connection
    % to the 80% of the pool)
    GraphRequests = lists_utils:generate(fun(_) ->
        gs_client:async_request(WellBehavedClient, #gs_req{subtype = graph, request = #gs_req_graph{
            operation = get,
            gri = #gri{type = od_user, id = ?USER_1, aspect = instance}
        }})
    end, WorkerPoolSize * 2),

    Stopwatch = stopwatch:start(),
    lists:foreach(fun(GraphReqId) ->
        ?assertMatch(
            {ok, #gs_resp_graph{data_format = resource, data = User1Data}},
            await_result(GraphReqId)
        )
    end, GraphRequests),
    % make sure the requests were processed before the long lasting ones have ended,
    % which proves that the throttling and queueing works
    ?assert(stopwatch:read_seconds(Stopwatch) < 8),

    % at some point, the long lasting ones should finish too
    lists:foreach(fun(LongReqId) ->
        ?assertEqual({ok, #gs_resp_rpc{result = #{<<"someDummy">> => <<"arguments127">>}}}, await_result(LongReqId))
    end, LongRequests).



session_persistence_test(Config) ->
    [Node | _] = ?config(cluster_worker_nodes, Config),
    Session = ?assertMatch(
        #gs_session{
            auth = dummyAuth,
            conn_ref = dummyConnRef,
            translator = dummyTranslator
        },
        rpc:call(Node, gs_persistence, create_session, [dummyAuth, dummyConnRef, 7, dummyTranslator])
    ),

    SessionId = Session#gs_session.id,

    ?assertMatch({ok, Session}, rpc:call(Node, gs_persistence, get_session, [SessionId])),

    ?assertMatch(ok, rpc:call(Node, gs_persistence, delete_session, [SessionId])),

    ?assertMatch({error, not_found}, rpc:call(Node, gs_persistence, get_session, [SessionId])).


subscriptions_persistence_test(Config) ->
    [Node | _] = ?config(cluster_worker_nodes, Config),
    #gs_session{id = Sess1} = rpc:call(Node, gs_persistence, create_session, [auth, self(), 7, dummyTranslator]),
    #gs_session{id = Sess2} = rpc:call(Node, gs_persistence, create_session, [auth, self(), 7, dummyTranslator]),
    #gs_session{id = Sess3} = rpc:call(Node, gs_persistence, create_session, [auth, self(), 7, dummyTranslator]),
    Auth1 = dummyAuth1,
    Auth2 = dummyAuth2,
    Auth3 = dummyAuth3,
    AuthHint1 = dummyAuthHint1,
    AuthHint2 = dummyAuthHint2,
    AuthHint3 = dummyAuthHint3,
    Sub1 = {Sess1, {Auth1, AuthHint1}},
    Sub2 = {Sess2, {Auth2, AuthHint2}},
    Sub3 = {Sess3, {Auth3, AuthHint3}},

    UserId = <<"dummyId">>,
    GRIPriv = #gri{type = od_user, id = UserId, aspect = instance, scope = private},
    GRIProt = #gri{type = od_user, id = UserId, aspect = instance, scope = protected},
    GRIShrd = #gri{type = od_user, id = UserId, aspect = instance, scope = shared},

    ?assertEqual(#{}, rpc:call(Node, gs_persistence, get_entity_subscribers, [od_user, UserId])),
    ?assertEqual([], rpc:call(Node, gs_subscriber, get_subscriptions, [Sess1])),
    ?assertEqual([], rpc:call(Node, gs_subscriber, get_subscriptions, [Sess2])),
    ?assertEqual([], rpc:call(Node, gs_subscriber, get_subscriptions, [Sess3])),

    ?assertEqual(ok, rpc:call(Node, gs_persistence, subscribe, [Sess1, GRIPriv, Auth1, AuthHint1])),
    ?assertEqual(ok, rpc:call(Node, gs_persistence, subscribe, [Sess2, GRIShrd, Auth2, AuthHint2])),
    ?assertEqual(#{
        {instance, private} => [Sub1],
        {instance, shared} => [Sub2]
    }, rpc:call(Node, gs_persistence, get_entity_subscribers, [od_user, UserId])),
    ?assertEqual([GRIPriv], rpc:call(Node, gs_subscriber, get_subscriptions, [Sess1])),
    ?assertEqual([GRIShrd], rpc:call(Node, gs_subscriber, get_subscriptions, [Sess2])),
    ?assertEqual([], rpc:call(Node, gs_subscriber, get_subscriptions, [Sess3])),

    % subscribing should be idempotent
    ?assertEqual(ok, rpc:call(Node, gs_persistence, subscribe, [Sess1, GRIPriv, Auth1, AuthHint1])),
    ?assertEqual(ok, rpc:call(Node, gs_persistence, subscribe, [Sess2, GRIShrd, Auth2, AuthHint2])),
    ?assertEqual(#{
        {instance, private} => [Sub1],
        {instance, shared} => [Sub2]
    }, rpc:call(Node, gs_persistence, get_entity_subscribers, [od_user, UserId])),
    ?assertEqual([GRIPriv], rpc:call(Node, gs_subscriber, get_subscriptions, [Sess1])),
    ?assertEqual([GRIShrd], rpc:call(Node, gs_subscriber, get_subscriptions, [Sess2])),
    ?assertEqual([], rpc:call(Node, gs_subscriber, get_subscriptions, [Sess3])),

    % subscribe for different scopes
    ?assertEqual(ok, rpc:call(Node, gs_persistence, subscribe, [Sess1, GRIProt, Auth1, AuthHint1])),
    ?assertEqual(ok, rpc:call(Node, gs_persistence, subscribe, [Sess2, GRIProt, Auth2, AuthHint2])),
    ?assertEqual(#{
        {instance, private} => [Sub1],
        {instance, protected} => ordsets:from_list([Sub1, Sub2]),
        {instance, shared} => [Sub2]
    }, rpc:call(Node, gs_persistence, get_entity_subscribers, [od_user, UserId])),
    ?assertEqual(ordsets:from_list([GRIPriv, GRIProt]), rpc:call(Node, gs_subscriber, get_subscriptions, [Sess1])),
    ?assertEqual(ordsets:from_list([GRIProt, GRIShrd]), rpc:call(Node, gs_subscriber, get_subscriptions, [Sess2])),
    ?assertEqual([], rpc:call(Node, gs_subscriber, get_subscriptions, [Sess3])),

    ?assertEqual(ok, rpc:call(Node, gs_persistence, subscribe, [Sess3, GRIShrd, Auth3, AuthHint3])),
    ?assertEqual(#{
        {instance, private} => [Sub1],
        {instance, protected} => ordsets:from_list([Sub1, Sub2]),
        {instance, shared} => ordsets:from_list([Sub2, Sub3])
    }, rpc:call(Node, gs_persistence, get_entity_subscribers, [od_user, UserId])),
    ?assertEqual(ordsets:from_list([GRIPriv, GRIProt]), rpc:call(Node, gs_subscriber, get_subscriptions, [Sess1])),
    ?assertEqual(ordsets:from_list([GRIProt, GRIShrd]), rpc:call(Node, gs_subscriber, get_subscriptions, [Sess2])),
    ?assertEqual([GRIShrd], rpc:call(Node, gs_subscriber, get_subscriptions, [Sess3])),

    ?assertEqual(ok, rpc:call(Node, gs_persistence, unsubscribe, [Sess2, GRIProt])),
    ?assertEqual(#{
        {instance, private} => [Sub1],
        {instance, protected} => [Sub1],
        {instance, shared} => ordsets:from_list([Sub2, Sub3])
    }, rpc:call(Node, gs_persistence, get_entity_subscribers, [od_user, UserId])),
    ?assertEqual(ordsets:from_list([GRIPriv, GRIProt]), rpc:call(Node, gs_subscriber, get_subscriptions, [Sess1])),
    ?assertEqual([GRIShrd], rpc:call(Node, gs_subscriber, get_subscriptions, [Sess2])),
    ?assertEqual([GRIShrd], rpc:call(Node, gs_subscriber, get_subscriptions, [Sess3])),

    ?assertEqual(ok, rpc:call(Node, gs_persistence, delete_session, [Sess1])),
    ?assertEqual({error, not_found}, rpc:call(Node, gs_persistence, get_session, [Sess1])),
    ?assertEqual(#{
        {instance, shared} => ordsets:from_list([Sub2, Sub3])
    }, rpc:call(Node, gs_persistence, get_entity_subscribers, [od_user, UserId])),
    ?assertEqual([], rpc:call(Node, gs_subscriber, get_subscriptions, [Sess1])),
    ?assertEqual([GRIShrd], rpc:call(Node, gs_subscriber, get_subscriptions, [Sess2])),
    ?assertEqual([GRIShrd], rpc:call(Node, gs_subscriber, get_subscriptions, [Sess3])),

    ?assertEqual(ok, rpc:call(Node, gs_persistence, delete_session, [Sess2])),
    ?assertEqual(ok, rpc:call(Node, gs_persistence, delete_session, [Sess3])).


gs_server_session_clearing_test_api_level(Config) ->
    [Node | _] = ?config(cluster_worker_nodes, Config),
    Auth = {token, ?USER_1_TOKEN},
    ConnRef = self(),
    Translator = ?GS_EXAMPLE_TRANSLATOR,
    HandshakeReq = #gs_req_handshake{
        auth = Auth,
        supported_versions = gs_protocol:supported_versions()
    },
    {ok, SessionData, #gs_resp_handshake{
        identity = ?SUB(user, ?USER_1)
    }} = ?assertMatch(
        {ok, _, _},
        rpc:call(Node, gs_server, handshake, [ConnRef, Translator, ?DUMMY_IP, _Cookies = [], HandshakeReq])
    ),

    GRI1 = #gri{type = od_user, id = ?USER_1, aspect = instance},
    % Make sure there are no leftovers from previous tests
    ?assertMatch(ok, rpc:call(Node, gs_persistence, remove_all_subscribers, [GRI1])),
    ?assertMatch(
        {ok, _},
        rpc:call(Node, gs_server, handle_request, [SessionData, #gs_req_graph{
            gri = GRI1,
            operation = get,
            subscribe = true
        }])
    ),

    GRI2 = #gri{type = od_user, id = ?USER_2, aspect = instance},
    % Make sure there are no leftovers from previous tests
    ?assertMatch(ok, rpc:call(Node, gs_persistence, remove_all_subscribers, [GRI2])),
    ?assertMatch(
        {ok, _},
        rpc:call(Node, gs_server, handle_request, [SessionData, #gs_req_graph{
            gri = GRI2,
            operation = get,
            auth_hint = ?THROUGH_SPACE(?SPACE_1),
            subscribe = true
        }])
    ),

    % Make sure that client disconnect removes all subscriptions
    ?assertEqual(ok, rpc:call(Node, gs_server, cleanup_session, [SessionData])),
    ?assertEqual([], rpc:call(Node, gs_subscriber, get_subscriptions, [SessionData#gs_session.id])),
    ?assertEqual(#{}, rpc:call(Node, gs_subscription, get_entity_subscribers, [od_user, ?USER_1])),
    ?assertEqual(#{}, rpc:call(Node, gs_subscription, get_entity_subscribers, [od_user, ?USER_2])).


gs_server_session_clearing_test_connection_level(Config) ->
    [gs_server_session_clearing_test_connection_level_base(Config, ProtoVersion) || ProtoVersion <- ?SUPPORTED_PROTO_VERSIONS].

gs_server_session_clearing_test_connection_level_base(Config, ProtoVersion) ->
    [Node | _] = ?config(cluster_worker_nodes, Config),

    {ok, Client1, #gs_resp_handshake{session_id = SessionId}} = gs_client:start_link(
        get_gs_ws_url(Config),
        {token, ?USER_1_TOKEN},
        [ProtoVersion],
        fun(_) -> ok end,
        ?SSL_OPTS(Config)
    ),

    GRI1 = #gri{type = od_user, id = ?USER_1, aspect = instance},
    % Make sure there are no leftovers from previous tests
    ?assertMatch(ok, rpc:call(Node, gs_persistence, remove_all_subscribers, [GRI1])),
    ?assertMatch(
        {ok, _},
        gs_client:graph_request(Client1, #gri{
            type = od_user, id = ?USER_1, aspect = instance
        }, get, #{}, true)
    ),

    GRI2 = #gri{type = od_user, id = ?USER_2, aspect = instance},
    % Make sure there are no leftovers from previous tests
    ?assertMatch(ok, rpc:call(Node, gs_persistence, remove_all_subscribers, [GRI2])),
    ?assertMatch(
        {ok, _},
        gs_client:graph_request(Client1, #gri{
            type = od_user, id = ?USER_2, aspect = instance
        }, get, #{}, true, ?THROUGH_SPACE(?SPACE_1))
    ),

    process_flag(trap_exit, true),
    exit(Client1, kill),
    ?assertMatch([], rpc:call(Node, gs_subscriber, get_subscriptions, [SessionId]), ?ATTEMPTS),
    ?assertEqual(#{}, rpc:call(Node, gs_subscription, get_entity_subscribers, [od_user, ?USER_1]), ?ATTEMPTS),
    ?assertEqual(#{}, rpc:call(Node, gs_subscription, get_entity_subscribers, [od_user, ?USER_2]), ?ATTEMPTS).


service_availability_rpc_test(Config) ->
    [service_availability_rpc_test(Config, ProtoVersion) || ProtoVersion <- ?SUPPORTED_PROTO_VERSIONS].

service_availability_rpc_test(Config, ProtoVersion) ->
    Nodes = ?config(cluster_worker_nodes, Config),
    Self = self(),
    RpcArgs = #{<<"a">> => <<"b">>},

    Client1 = spawn_client(Config, ProtoVersion, {token, ?USER_1_TOKEN}, ?SUB(user, ?USER_1), fun(Push) ->
        Self ! Push
    end),

    graph_sync_mocks:simulate_service_availability(Nodes, true),
    ?assertMatch(
        {ok, #gs_resp_rpc{result = RpcArgs}},
        gs_client:rpc_request(Client1, <<"user1Fun">>, RpcArgs)
    ),

    graph_sync_mocks:simulate_service_availability(Nodes, false),
    ?assertMatch(
        ?ERR_SERVICE_UNAVAILABLE,
        gs_client:rpc_request(Client1, <<"user1Fun">>, RpcArgs)
    ),

    graph_sync_mocks:simulate_service_availability(Nodes, true),
    ?assertMatch(
        {ok, #gs_resp_rpc{result = RpcArgs}},
        gs_client:rpc_request(Client1, <<"user1Fun">>, RpcArgs)
    ).


service_availability_graph_test(Config) ->
    [service_availability_graph_test(Config, ProtoVersion) || ProtoVersion <- ?SUPPORTED_PROTO_VERSIONS].

service_availability_graph_test(Config, ProtoVersion) ->
    Nodes = ?config(cluster_worker_nodes, Config),
    User1Data = (?USER_DATA_WITHOUT_GRI(?USER_1))#{
        <<"gri">> => gri:serialize(#gri{type = od_user, id = ?USER_1, aspect = instance}),
        <<"revision">> => 1
    },
    Client1 = spawn_client(Config, ProtoVersion, {token, ?USER_1_TOKEN}, ?SUB(user, ?USER_1)),

    graph_sync_mocks:simulate_service_availability(Nodes, true),
    ?assertMatch(
        {ok, #gs_resp_graph{data_format = resource, data = User1Data}},
        gs_client:graph_request(Client1, #gri{
            type = od_user, id = ?USER_1, aspect = instance
        }, get)
    ),

    graph_sync_mocks:simulate_service_availability(Nodes, false),
    ?assertMatch(
        ?ERR_SERVICE_UNAVAILABLE,
        gs_client:graph_request(Client1, #gri{
            type = od_user, id = ?USER_1, aspect = instance
        }, get)
    ),

    graph_sync_mocks:simulate_service_availability(Nodes, true),
    ?assertMatch(
        {ok, #gs_resp_graph{data_format = resource, data = User1Data}},
        gs_client:graph_request(Client1, #gri{
            type = od_user, id = ?USER_1, aspect = instance
        }, get)
    ).


service_availability_handshake_test(Config) ->
    [service_availability_handshake_test(Config, ProtoVersion) || ProtoVersion <- ?SUPPORTED_PROTO_VERSIONS].

service_availability_handshake_test(Config, ProtoVersion) ->
    Nodes = ?config(cluster_worker_nodes, Config),

    graph_sync_mocks:simulate_service_availability(Nodes, true),
    spawn_client(Config, ProtoVersion, {token, ?USER_1_TOKEN}, ?SUB(user, ?USER_1)),

    graph_sync_mocks:simulate_service_availability(Nodes, false),

    spawn_client(Config, ProtoVersion, {token, ?USER_1_TOKEN}, ?ERR_SERVICE_UNAVAILABLE),

    graph_sync_mocks:simulate_service_availability(Nodes, true),
    spawn_client(Config, ProtoVersion, {token, ?USER_1_TOKEN}, ?SUB(user, ?USER_1)).


%%%===================================================================
%%% Helper functions related to asynchronous subscriptions messages
%%%===================================================================

gatherer_loop(MessagesMap) ->
    NewMap = receive
        {gather_message, ClientRef, Message} ->
            ClientMessages = maps:get(ClientRef, MessagesMap, []),
            maps:put(ClientRef, [Message | ClientMessages], MessagesMap);
        {get_messages, ClientRef, Pid} ->
            ClientMessages = maps:get(ClientRef, MessagesMap, []),
            Pid ! ClientMessages,
            MessagesMap
    end,
    gatherer_loop(NewMap).


verify_message_present(GathererPid, ClientRef, MessageMatcherFun) ->
    GathererPid ! {get_messages, ClientRef, self()},
    AllMessages = receive
        M when is_list(M) -> M
    end,
    lists:any(fun(Message) ->
        MessageMatcherFun(Message)
    end, AllMessages).


verify_message_absent(_, _, _, 0) ->
    true;
verify_message_absent(GathererPid, ClientRef, MessageMatcherFun, Retries) ->
    case verify_message_present(GathererPid, ClientRef, MessageMatcherFun) of
        true ->
            false;
        false ->
            timer:sleep(1000),
            verify_message_absent(GathererPid, ClientRef, MessageMatcherFun, Retries - 1)
    end.

%%%===================================================================
%%% Internal functions
%%%===================================================================

% ExpResult :: {error, term()} | aai:subject().
spawn_client(Config, ProtoVersion, Auth, ExpResult) ->
    spawn_client(Config, ProtoVersion, Auth, ExpResult, fun(_) -> ok end).

spawn_client(Config, ProtoVersion, Auth, ExpResult, PushCallback) when is_integer(ProtoVersion) ->
    spawn_client(Config, [ProtoVersion], Auth, ExpResult, PushCallback);
spawn_client(Config, ProtoVersions, Auth, ExpResult, PushCallback) ->
    Result = gs_client:start_link(
        get_gs_ws_url(Config),
        Auth,
        ProtoVersions,
        PushCallback,
        ?SSL_OPTS(Config)
    ),
    case ExpResult of
        {error, _} ->
            ?assertEqual(ExpResult, Result),
            connection_error;
        ExpIdentity ->
            ?assertMatch({ok, _, #gs_resp_handshake{identity = ExpIdentity}}, Result),
            {ok, Client, _} = Result,
            store_client(Client),
            Client
    end.


store_client(Client) ->
    node_cache:update(clients, fun(Clients) -> {ok, [Client | Clients], infinity} end, []).


kill_and_clear_clients() ->
    Clients = node_cache:get(clients, []),
    % clients spawned in the test are linked to this process
    process_flag(trap_exit, true),
    lists:foreach(fun(Client) ->
        exit(Client, kill)
    end, Clients),
    node_cache:put(clients, []).


async_request_long_operation(Client) ->
    gs_client:async_request(Client, #gs_req{
        subtype = rpc,
        request = #gs_req_rpc{
            function = <<"veryLongOperation">>,
            args = #{<<"someDummy">> => <<"arguments127">>}
        }
    }).


await_result(Id) ->
    receive
        {result, Id, Result} ->
            Result
    after
        timer:seconds(90) ->
            error(receive_timeout)
    end.


get_gs_ws_url(Config) ->
    Node = ?RAND_ELEMENT(?config(cluster_worker_nodes, Config)),
    NodeIP = test_utils:get_docker_ip(Node),
    str_utils:format_bin("wss://~ts:~B/", [NodeIP, ?GS_PORT]).


start_gs_listener(Node) ->
    ok = rpc:call(Node, application, ensure_started, [cowboy]),
    ?assertMatch(ok, rpc:call(Node, gui, start, [#gui_config{
        port = ?GS_PORT,
        key_file = ?KEY_FILE,
        cert_file = ?CERT_FILE,
        chain_file = ?CHAIN_FILE,
        number_of_acceptors = ?GS_HTTPS_ACCEPTORS,
        custom_cowboy_routes = [
            {"/[...]", gs_ws_handler, [?GS_EXAMPLE_TRANSLATOR]}
        ]
    }])).


stop_gs_listener(Node) ->
    rpc:call(Node, gui, stop, []).


get_trusted_cacerts(Config) ->
    [Node | _] = ?config(cluster_worker_nodes, Config),
    rpc:call(Node, cert_utils, load_ders, [?TRUSTED_CACERTS_FILE]).


%%%===================================================================
%%% Setup/teardown functions
%%%===================================================================

init_per_suite(Config) ->
    ssl:start(),

    PostHook = fun(UpdatedConfig) ->
        Nodes = ?config(cluster_worker_nodes, UpdatedConfig),
        [start_gs_listener(N) || N <- Nodes],
        graph_sync_mocks:mock_callbacks(UpdatedConfig),

        test_utils:set_env(Nodes, cluster_worker, graph_sync_verbose_logs_severity, regular),
        test_utils:set_env(Nodes, cluster_worker, graph_sync_verbose_logs_print_credentials, true),
        test_utils:set_env(Nodes, cluster_worker, graph_sync_verbose_logs_print_errors, true),
        test_utils:set_env(Nodes, cluster_worker, graph_sync_verbose_logs_print_data, true),

        UpdatedConfig
    end,
    [{?LOAD_MODULES, [graph_sync_mocks]}, {?ENV_UP_POSTHOOK, PostHook} | Config].


init_per_testcase(_, Config) ->
    Nodes = ?config(cluster_worker_nodes, Config),
    % set the defaults (some tests manipulate this config)
    test_utils:set_env(Nodes, cluster_worker, graph_sync_max_pool_usage_per_connection, 0.05),
    test_utils:set_env(Nodes, cluster_worker, graph_sync_stale_request_threshold_sec, 120),
    test_utils:set_env(Nodes, cluster_worker, graph_sync_websocket_keepalive, timer:seconds(15)),
    test_utils:set_env(Nodes, cluster_worker, graph_sync_request_processing_timeout_sec, 60),
    test_utils:set_env(Nodes, cluster_worker, graph_sync_verbose_logs_identity_filter, undefined),

    {_, []} = utils:rpc_multicall(Nodes, gs_worker_pool, init, [200]),

    Config.


end_per_testcase(_, Config) ->
    Nodes = ?config(cluster_worker_nodes, Config),
    kill_and_clear_clients(),
    {_, []} = utils:rpc_multicall(Nodes, gs_worker_pool, stop, []),
    ok.


end_per_suite(Config) ->
    ssl:stop(),
    Nodes = ?config(cluster_worker_nodes, Config),
    [stop_gs_listener(N) || N <- Nodes],
    graph_sync_mocks:unmock_callbacks(Config),
    ok.
