-module(idna_unicode_data).

-export([combining_class/1,
         compat/1,
         composition/2,
         lowercase/1]).

-define(COMBINING_CLASS, 1).
-define(DECOMPOSITION, 2).
-define(LOWERCASE_MAPPING, 3).

combining_class(C) ->
    case lookup(C) of
        false -> 0;
        Props ->
            erlang:list_to_integer(element(?COMBINING_CLASS, Props))
    end.

compat(C) ->
    lookup(C, fun(Props) ->
                case element(?DECOMPOSITION, Props) of
                    [] -> undefined;
                    Val ->
                        Tokens = string:tokens(Val, " "),
                        CodePoints = dehex(case hd(Val) of
                                    $< -> tl(Tokens);
                                    _ -> Tokens
                                end),
                        CodePoints
                end
        end).

composition(A, B) ->
    Key = lists:flatten([hex(A), " ", hex(B)]),
    case decomposition(Key) of
        false -> undefined;
        Val -> erlang:list_to_integer(Val, 16)
    end.


lowercase(C) ->
    lookup(C, fun(Props) ->
                case element(?LOWERCASE_MAPPING, Props) of
                    [] -> C;
                    Hex -> erlang:list_to_integer(Hex, 16)
                end
        end).

%% --------------------
%% helper functions
%% --------------------

hex(Codepoint) ->
  string:right(erlang:integer_to_list(Codepoint, 16), 4, $0).

dehex(Strings) ->
  [erlang:list_to_integer(String, 16) || String <- Strings].

lookup(Codepoint) ->
    lookup1(hex(Codepoint)).

lookup(Codepoint, Fun) ->
    case lookup(Codepoint) of
        false -> {error, bad_codepoint};
        Props -> Fun(Props)
    end.


%% GENERATED
decomposition("003B") -> 	"037E";
decomposition("003C 0338") -> 	"226E";
decomposition("003D 0338") -> 	"2260";
decomposition("003E 0338") -> 	"226F";
decomposition("0041 0300") -> 	"00C0";
decomposition("0041 0301") -> 	"00C1";
decomposition("0041 0302") -> 	"00C2";
decomposition("0041 0303") -> 	"00C3";
decomposition("0041 0304") -> 	"0100";
decomposition("0041 0306") -> 	"0102";
decomposition("0041 0307") -> 	"0226";
decomposition("0041 0308") -> 	"00C4";
decomposition("0041 0309") -> 	"1EA2";
decomposition("0041 030A") -> 	"00C5";
decomposition("0041 030C") -> 	"01CD";
decomposition("0041 030F") -> 	"0200";
decomposition("0041 0311") -> 	"0202";
decomposition("0041 0323") -> 	"1EA0";
decomposition("0041 0325") -> 	"1E00";
decomposition("0041 0328") -> 	"0104";
decomposition("0042 0307") -> 	"1E02";
decomposition("0042 0323") -> 	"1E04";
decomposition("0042 0331") -> 	"1E06";
decomposition("0043 0301") -> 	"0106";
decomposition("0043 0302") -> 	"0108";
decomposition("0043 0307") -> 	"010A";
decomposition("0043 030C") -> 	"010C";
decomposition("0043 0327") -> 	"00C7";
decomposition("0044 0307") -> 	"1E0A";
decomposition("0044 030C") -> 	"010E";
decomposition("0044 0323") -> 	"1E0C";
decomposition("0044 0327") -> 	"1E10";
decomposition("0044 032D") -> 	"1E12";
decomposition("0044 0331") -> 	"1E0E";
decomposition("0045 0300") -> 	"00C8";
decomposition("0045 0301") -> 	"00C9";
decomposition("0045 0302") -> 	"00CA";
decomposition("0045 0303") -> 	"1EBC";
decomposition("0045 0304") -> 	"0112";
decomposition("0045 0306") -> 	"0114";
decomposition("0045 0307") -> 	"0116";
decomposition("0045 0308") -> 	"00CB";
decomposition("0045 0309") -> 	"1EBA";
decomposition("0045 030C") -> 	"011A";
decomposition("0045 030F") -> 	"0204";
decomposition("0045 0311") -> 	"0206";
decomposition("0045 0323") -> 	"1EB8";
decomposition("0045 0327") -> 	"0228";
decomposition("0045 0328") -> 	"0118";
decomposition("0045 032D") -> 	"1E18";
decomposition("0045 0330") -> 	"1E1A";
decomposition("0046 0307") -> 	"1E1E";
decomposition("0047 0301") -> 	"01F4";
decomposition("0047 0302") -> 	"011C";
decomposition("0047 0304") -> 	"1E20";
decomposition("0047 0306") -> 	"011E";
decomposition("0047 0307") -> 	"0120";
decomposition("0047 030C") -> 	"01E6";
decomposition("0047 0327") -> 	"0122";
decomposition("0048 0302") -> 	"0124";
decomposition("0048 0307") -> 	"1E22";
decomposition("0048 0308") -> 	"1E26";
decomposition("0048 030C") -> 	"021E";
decomposition("0048 0323") -> 	"1E24";
decomposition("0048 0327") -> 	"1E28";
decomposition("0048 032E") -> 	"1E2A";
decomposition("0049 0300") -> 	"00CC";
decomposition("0049 0301") -> 	"00CD";
decomposition("0049 0302") -> 	"00CE";
decomposition("0049 0303") -> 	"0128";
decomposition("0049 0304") -> 	"012A";
decomposition("0049 0306") -> 	"012C";
decomposition("0049 0307") -> 	"0130";
decomposition("0049 0308") -> 	"00CF";
decomposition("0049 0309") -> 	"1EC8";
decomposition("0049 030C") -> 	"01CF";
decomposition("0049 030F") -> 	"0208";
decomposition("0049 0311") -> 	"020A";
decomposition("0049 0323") -> 	"1ECA";
decomposition("0049 0328") -> 	"012E";
decomposition("0049 0330") -> 	"1E2C";
decomposition("004A 0302") -> 	"0134";
decomposition("004B 0301") -> 	"1E30";
decomposition("004B 030C") -> 	"01E8";
decomposition("004B 0323") -> 	"1E32";
decomposition("004B 0327") -> 	"0136";
decomposition("004B 0331") -> 	"1E34";
decomposition("004B") -> 	"212A";
decomposition("004C 0301") -> 	"0139";
decomposition("004C 030C") -> 	"013D";
decomposition("004C 0323") -> 	"1E36";
decomposition("004C 0327") -> 	"013B";
decomposition("004C 032D") -> 	"1E3C";
decomposition("004C 0331") -> 	"1E3A";
decomposition("004D 0301") -> 	"1E3E";
decomposition("004D 0307") -> 	"1E40";
decomposition("004D 0323") -> 	"1E42";
decomposition("004E 0300") -> 	"01F8";
decomposition("004E 0301") -> 	"0143";
decomposition("004E 0303") -> 	"00D1";
decomposition("004E 0307") -> 	"1E44";
decomposition("004E 030C") -> 	"0147";
decomposition("004E 0323") -> 	"1E46";
decomposition("004E 0327") -> 	"0145";
decomposition("004E 032D") -> 	"1E4A";
decomposition("004E 0331") -> 	"1E48";
decomposition("004F 0300") -> 	"00D2";
decomposition("004F 0301") -> 	"00D3";
decomposition("004F 0302") -> 	"00D4";
decomposition("004F 0303") -> 	"00D5";
decomposition("004F 0304") -> 	"014C";
decomposition("004F 0306") -> 	"014E";
decomposition("004F 0307") -> 	"022E";
decomposition("004F 0308") -> 	"00D6";
decomposition("004F 0309") -> 	"1ECE";
decomposition("004F 030B") -> 	"0150";
decomposition("004F 030C") -> 	"01D1";
decomposition("004F 030F") -> 	"020C";
decomposition("004F 0311") -> 	"020E";
decomposition("004F 031B") -> 	"01A0";
decomposition("004F 0323") -> 	"1ECC";
decomposition("004F 0328") -> 	"01EA";
decomposition("0050 0301") -> 	"1E54";
decomposition("0050 0307") -> 	"1E56";
decomposition("0052 0301") -> 	"0154";
decomposition("0052 0307") -> 	"1E58";
decomposition("0052 030C") -> 	"0158";
decomposition("0052 030F") -> 	"0210";
decomposition("0052 0311") -> 	"0212";
decomposition("0052 0323") -> 	"1E5A";
decomposition("0052 0327") -> 	"0156";
decomposition("0052 0331") -> 	"1E5E";
decomposition("0053 0301") -> 	"015A";
decomposition("0053 0302") -> 	"015C";
decomposition("0053 0307") -> 	"1E60";
decomposition("0053 030C") -> 	"0160";
decomposition("0053 0323") -> 	"1E62";
decomposition("0053 0326") -> 	"0218";
decomposition("0053 0327") -> 	"015E";
decomposition("0054 0307") -> 	"1E6A";
decomposition("0054 030C") -> 	"0164";
decomposition("0054 0323") -> 	"1E6C";
decomposition("0054 0326") -> 	"021A";
decomposition("0054 0327") -> 	"0162";
decomposition("0054 032D") -> 	"1E70";
decomposition("0054 0331") -> 	"1E6E";
decomposition("0055 0300") -> 	"00D9";
decomposition("0055 0301") -> 	"00DA";
decomposition("0055 0302") -> 	"00DB";
decomposition("0055 0303") -> 	"0168";
decomposition("0055 0304") -> 	"016A";
decomposition("0055 0306") -> 	"016C";
decomposition("0055 0308") -> 	"00DC";
decomposition("0055 0309") -> 	"1EE6";
decomposition("0055 030A") -> 	"016E";
decomposition("0055 030B") -> 	"0170";
decomposition("0055 030C") -> 	"01D3";
decomposition("0055 030F") -> 	"0214";
decomposition("0055 0311") -> 	"0216";
decomposition("0055 031B") -> 	"01AF";
decomposition("0055 0323") -> 	"1EE4";
decomposition("0055 0324") -> 	"1E72";
decomposition("0055 0328") -> 	"0172";
decomposition("0055 032D") -> 	"1E76";
decomposition("0055 0330") -> 	"1E74";
decomposition("0056 0303") -> 	"1E7C";
decomposition("0056 0323") -> 	"1E7E";
decomposition("0057 0300") -> 	"1E80";
decomposition("0057 0301") -> 	"1E82";
decomposition("0057 0302") -> 	"0174";
decomposition("0057 0307") -> 	"1E86";
decomposition("0057 0308") -> 	"1E84";
decomposition("0057 0323") -> 	"1E88";
decomposition("0058 0307") -> 	"1E8A";
decomposition("0058 0308") -> 	"1E8C";
decomposition("0059 0300") -> 	"1EF2";
decomposition("0059 0301") -> 	"00DD";
decomposition("0059 0302") -> 	"0176";
decomposition("0059 0303") -> 	"1EF8";
decomposition("0059 0304") -> 	"0232";
decomposition("0059 0307") -> 	"1E8E";
decomposition("0059 0308") -> 	"0178";
decomposition("0059 0309") -> 	"1EF6";
decomposition("0059 0323") -> 	"1EF4";
decomposition("005A 0301") -> 	"0179";
decomposition("005A 0302") -> 	"1E90";
decomposition("005A 0307") -> 	"017B";
decomposition("005A 030C") -> 	"017D";
decomposition("005A 0323") -> 	"1E92";
decomposition("005A 0331") -> 	"1E94";
decomposition("0060") -> 	"1FEF";
decomposition("0061 0300") -> 	"00E0";
decomposition("0061 0301") -> 	"00E1";
decomposition("0061 0302") -> 	"00E2";
decomposition("0061 0303") -> 	"00E3";
decomposition("0061 0304") -> 	"0101";
decomposition("0061 0306") -> 	"0103";
decomposition("0061 0307") -> 	"0227";
decomposition("0061 0308") -> 	"00E4";
decomposition("0061 0309") -> 	"1EA3";
decomposition("0061 030A") -> 	"00E5";
decomposition("0061 030C") -> 	"01CE";
decomposition("0061 030F") -> 	"0201";
decomposition("0061 0311") -> 	"0203";
decomposition("0061 0323") -> 	"1EA1";
decomposition("0061 0325") -> 	"1E01";
decomposition("0061 0328") -> 	"0105";
decomposition("0062 0307") -> 	"1E03";
decomposition("0062 0323") -> 	"1E05";
decomposition("0062 0331") -> 	"1E07";
decomposition("0063 0301") -> 	"0107";
decomposition("0063 0302") -> 	"0109";
decomposition("0063 0307") -> 	"010B";
decomposition("0063 030C") -> 	"010D";
decomposition("0063 0327") -> 	"00E7";
decomposition("0064 0307") -> 	"1E0B";
decomposition("0064 030C") -> 	"010F";
decomposition("0064 0323") -> 	"1E0D";
decomposition("0064 0327") -> 	"1E11";
decomposition("0064 032D") -> 	"1E13";
decomposition("0064 0331") -> 	"1E0F";
decomposition("0065 0300") -> 	"00E8";
decomposition("0065 0301") -> 	"00E9";
decomposition("0065 0302") -> 	"00EA";
decomposition("0065 0303") -> 	"1EBD";
decomposition("0065 0304") -> 	"0113";
decomposition("0065 0306") -> 	"0115";
decomposition("0065 0307") -> 	"0117";
decomposition("0065 0308") -> 	"00EB";
decomposition("0065 0309") -> 	"1EBB";
decomposition("0065 030C") -> 	"011B";
decomposition("0065 030F") -> 	"0205";
decomposition("0065 0311") -> 	"0207";
decomposition("0065 0323") -> 	"1EB9";
decomposition("0065 0327") -> 	"0229";
decomposition("0065 0328") -> 	"0119";
decomposition("0065 032D") -> 	"1E19";
decomposition("0065 0330") -> 	"1E1B";
decomposition("0066 0307") -> 	"1E1F";
decomposition("0067 0301") -> 	"01F5";
decomposition("0067 0302") -> 	"011D";
decomposition("0067 0304") -> 	"1E21";
decomposition("0067 0306") -> 	"011F";
decomposition("0067 0307") -> 	"0121";
decomposition("0067 030C") -> 	"01E7";
decomposition("0067 0327") -> 	"0123";
decomposition("0068 0302") -> 	"0125";
decomposition("0068 0307") -> 	"1E23";
decomposition("0068 0308") -> 	"1E27";
decomposition("0068 030C") -> 	"021F";
decomposition("0068 0323") -> 	"1E25";
decomposition("0068 0327") -> 	"1E29";
decomposition("0068 032E") -> 	"1E2B";
decomposition("0068 0331") -> 	"1E96";
decomposition("0069 0300") -> 	"00EC";
decomposition("0069 0301") -> 	"00ED";
decomposition("0069 0302") -> 	"00EE";
decomposition("0069 0303") -> 	"0129";
decomposition("0069 0304") -> 	"012B";
decomposition("0069 0306") -> 	"012D";
decomposition("0069 0308") -> 	"00EF";
decomposition("0069 0309") -> 	"1EC9";
decomposition("0069 030C") -> 	"01D0";
decomposition("0069 030F") -> 	"0209";
decomposition("0069 0311") -> 	"020B";
decomposition("0069 0323") -> 	"1ECB";
decomposition("0069 0328") -> 	"012F";
decomposition("0069 0330") -> 	"1E2D";
decomposition("006A 0302") -> 	"0135";
decomposition("006A 030C") -> 	"01F0";
decomposition("006B 0301") -> 	"1E31";
decomposition("006B 030C") -> 	"01E9";
decomposition("006B 0323") -> 	"1E33";
decomposition("006B 0327") -> 	"0137";
decomposition("006B 0331") -> 	"1E35";
decomposition("006C 0301") -> 	"013A";
decomposition("006C 030C") -> 	"013E";
decomposition("006C 0323") -> 	"1E37";
decomposition("006C 0327") -> 	"013C";
decomposition("006C 032D") -> 	"1E3D";
decomposition("006C 0331") -> 	"1E3B";
decomposition("006D 0301") -> 	"1E3F";
decomposition("006D 0307") -> 	"1E41";
decomposition("006D 0323") -> 	"1E43";
decomposition("006E 0300") -> 	"01F9";
decomposition("006E 0301") -> 	"0144";
decomposition("006E 0303") -> 	"00F1";
decomposition("006E 0307") -> 	"1E45";
decomposition("006E 030C") -> 	"0148";
decomposition("006E 0323") -> 	"1E47";
decomposition("006E 0327") -> 	"0146";
decomposition("006E 032D") -> 	"1E4B";
decomposition("006E 0331") -> 	"1E49";
decomposition("006F 0300") -> 	"00F2";
decomposition("006F 0301") -> 	"00F3";
decomposition("006F 0302") -> 	"00F4";
decomposition("006F 0303") -> 	"00F5";
decomposition("006F 0304") -> 	"014D";
decomposition("006F 0306") -> 	"014F";
decomposition("006F 0307") -> 	"022F";
decomposition("006F 0308") -> 	"00F6";
decomposition("006F 0309") -> 	"1ECF";
decomposition("006F 030B") -> 	"0151";
decomposition("006F 030C") -> 	"01D2";
decomposition("006F 030F") -> 	"020D";
decomposition("006F 0311") -> 	"020F";
decomposition("006F 031B") -> 	"01A1";
decomposition("006F 0323") -> 	"1ECD";
decomposition("006F 0328") -> 	"01EB";
decomposition("0070 0301") -> 	"1E55";
decomposition("0070 0307") -> 	"1E57";
decomposition("0072 0301") -> 	"0155";
decomposition("0072 0307") -> 	"1E59";
decomposition("0072 030C") -> 	"0159";
decomposition("0072 030F") -> 	"0211";
decomposition("0072 0311") -> 	"0213";
decomposition("0072 0323") -> 	"1E5B";
decomposition("0072 0327") -> 	"0157";
decomposition("0072 0331") -> 	"1E5F";
decomposition("0073 0301") -> 	"015B";
decomposition("0073 0302") -> 	"015D";
decomposition("0073 0307") -> 	"1E61";
decomposition("0073 030C") -> 	"0161";
decomposition("0073 0323") -> 	"1E63";
decomposition("0073 0326") -> 	"0219";
decomposition("0073 0327") -> 	"015F";
decomposition("0074 0307") -> 	"1E6B";
decomposition("0074 0308") -> 	"1E97";
decomposition("0074 030C") -> 	"0165";
decomposition("0074 0323") -> 	"1E6D";
decomposition("0074 0326") -> 	"021B";
decomposition("0074 0327") -> 	"0163";
decomposition("0074 032D") -> 	"1E71";
decomposition("0074 0331") -> 	"1E6F";
decomposition("0075 0300") -> 	"00F9";
decomposition("0075 0301") -> 	"00FA";
decomposition("0075 0302") -> 	"00FB";
decomposition("0075 0303") -> 	"0169";
decomposition("0075 0304") -> 	"016B";
decomposition("0075 0306") -> 	"016D";
decomposition("0075 0308") -> 	"00FC";
decomposition("0075 0309") -> 	"1EE7";
decomposition("0075 030A") -> 	"016F";
decomposition("0075 030B") -> 	"0171";
decomposition("0075 030C") -> 	"01D4";
decomposition("0075 030F") -> 	"0215";
decomposition("0075 0311") -> 	"0217";
decomposition("0075 031B") -> 	"01B0";
decomposition("0075 0323") -> 	"1EE5";
decomposition("0075 0324") -> 	"1E73";
decomposition("0075 0328") -> 	"0173";
decomposition("0075 032D") -> 	"1E77";
decomposition("0075 0330") -> 	"1E75";
decomposition("0076 0303") -> 	"1E7D";
decomposition("0076 0323") -> 	"1E7F";
decomposition("0077 0300") -> 	"1E81";
decomposition("0077 0301") -> 	"1E83";
decomposition("0077 0302") -> 	"0175";
decomposition("0077 0307") -> 	"1E87";
decomposition("0077 0308") -> 	"1E85";
decomposition("0077 030A") -> 	"1E98";
decomposition("0077 0323") -> 	"1E89";
decomposition("0078 0307") -> 	"1E8B";
decomposition("0078 0308") -> 	"1E8D";
decomposition("0079 0300") -> 	"1EF3";
decomposition("0079 0301") -> 	"00FD";
decomposition("0079 0302") -> 	"0177";
decomposition("0079 0303") -> 	"1EF9";
decomposition("0079 0304") -> 	"0233";
decomposition("0079 0307") -> 	"1E8F";
decomposition("0079 0308") -> 	"00FF";
decomposition("0079 0309") -> 	"1EF7";
decomposition("0079 030A") -> 	"1E99";
decomposition("0079 0323") -> 	"1EF5";
decomposition("007A 0301") -> 	"017A";
decomposition("007A 0302") -> 	"1E91";
decomposition("007A 0307") -> 	"017C";
decomposition("007A 030C") -> 	"017E";
decomposition("007A 0323") -> 	"1E93";
decomposition("007A 0331") -> 	"1E95";
decomposition("00A8 0300") -> 	"1FED";
decomposition("00A8 0301") -> 	"0385";
decomposition("00A8 0342") -> 	"1FC1";
decomposition("00B4") -> 	"1FFD";
decomposition("00B7") -> 	"0387";
decomposition("00C2 0300") -> 	"1EA6";
decomposition("00C2 0301") -> 	"1EA4";
decomposition("00C2 0303") -> 	"1EAA";
decomposition("00C2 0309") -> 	"1EA8";
decomposition("00C4 0304") -> 	"01DE";
decomposition("00C5 0301") -> 	"01FA";
decomposition("00C5") -> 	"212B";
decomposition("00C6 0301") -> 	"01FC";
decomposition("00C6 0304") -> 	"01E2";
decomposition("00C7 0301") -> 	"1E08";
decomposition("00CA 0300") -> 	"1EC0";
decomposition("00CA 0301") -> 	"1EBE";
decomposition("00CA 0303") -> 	"1EC4";
decomposition("00CA 0309") -> 	"1EC2";
decomposition("00CF 0301") -> 	"1E2E";
decomposition("00D4 0300") -> 	"1ED2";
decomposition("00D4 0301") -> 	"1ED0";
decomposition("00D4 0303") -> 	"1ED6";
decomposition("00D4 0309") -> 	"1ED4";
decomposition("00D5 0301") -> 	"1E4C";
decomposition("00D5 0304") -> 	"022C";
decomposition("00D5 0308") -> 	"1E4E";
decomposition("00D6 0304") -> 	"022A";
decomposition("00D8 0301") -> 	"01FE";
decomposition("00DC 0300") -> 	"01DB";
decomposition("00DC 0301") -> 	"01D7";
decomposition("00DC 0304") -> 	"01D5";
decomposition("00DC 030C") -> 	"01D9";
decomposition("00E2 0300") -> 	"1EA7";
decomposition("00E2 0301") -> 	"1EA5";
decomposition("00E2 0303") -> 	"1EAB";
decomposition("00E2 0309") -> 	"1EA9";
decomposition("00E4 0304") -> 	"01DF";
decomposition("00E5 0301") -> 	"01FB";
decomposition("00E6 0301") -> 	"01FD";
decomposition("00E6 0304") -> 	"01E3";
decomposition("00E7 0301") -> 	"1E09";
decomposition("00EA 0300") -> 	"1EC1";
decomposition("00EA 0301") -> 	"1EBF";
decomposition("00EA 0303") -> 	"1EC5";
decomposition("00EA 0309") -> 	"1EC3";
decomposition("00EF 0301") -> 	"1E2F";
decomposition("00F4 0300") -> 	"1ED3";
decomposition("00F4 0301") -> 	"1ED1";
decomposition("00F4 0303") -> 	"1ED7";
decomposition("00F4 0309") -> 	"1ED5";
decomposition("00F5 0301") -> 	"1E4D";
decomposition("00F5 0304") -> 	"022D";
decomposition("00F5 0308") -> 	"1E4F";
decomposition("00F6 0304") -> 	"022B";
decomposition("00F8 0301") -> 	"01FF";
decomposition("00FC 0300") -> 	"01DC";
decomposition("00FC 0301") -> 	"01D8";
decomposition("00FC 0304") -> 	"01D6";
decomposition("00FC 030C") -> 	"01DA";
decomposition("0102 0300") -> 	"1EB0";
decomposition("0102 0301") -> 	"1EAE";
decomposition("0102 0303") -> 	"1EB4";
decomposition("0102 0309") -> 	"1EB2";
decomposition("0103 0300") -> 	"1EB1";
decomposition("0103 0301") -> 	"1EAF";
decomposition("0103 0303") -> 	"1EB5";
decomposition("0103 0309") -> 	"1EB3";
decomposition("0112 0300") -> 	"1E14";
decomposition("0112 0301") -> 	"1E16";
decomposition("0113 0300") -> 	"1E15";
decomposition("0113 0301") -> 	"1E17";
decomposition("014C 0300") -> 	"1E50";
decomposition("014C 0301") -> 	"1E52";
decomposition("014D 0300") -> 	"1E51";
decomposition("014D 0301") -> 	"1E53";
decomposition("015A 0307") -> 	"1E64";
decomposition("015B 0307") -> 	"1E65";
decomposition("0160 0307") -> 	"1E66";
decomposition("0161 0307") -> 	"1E67";
decomposition("0168 0301") -> 	"1E78";
decomposition("0169 0301") -> 	"1E79";
decomposition("016A 0308") -> 	"1E7A";
decomposition("016B 0308") -> 	"1E7B";
decomposition("017F 0307") -> 	"1E9B";
decomposition("01A0 0300") -> 	"1EDC";
decomposition("01A0 0301") -> 	"1EDA";
decomposition("01A0 0303") -> 	"1EE0";
decomposition("01A0 0309") -> 	"1EDE";
decomposition("01A0 0323") -> 	"1EE2";
decomposition("01A1 0300") -> 	"1EDD";
decomposition("01A1 0301") -> 	"1EDB";
decomposition("01A1 0303") -> 	"1EE1";
decomposition("01A1 0309") -> 	"1EDF";
decomposition("01A1 0323") -> 	"1EE3";
decomposition("01AF 0300") -> 	"1EEA";
decomposition("01AF 0301") -> 	"1EE8";
decomposition("01AF 0303") -> 	"1EEE";
decomposition("01AF 0309") -> 	"1EEC";
decomposition("01AF 0323") -> 	"1EF0";
decomposition("01B0 0300") -> 	"1EEB";
decomposition("01B0 0301") -> 	"1EE9";
decomposition("01B0 0303") -> 	"1EEF";
decomposition("01B0 0309") -> 	"1EED";
decomposition("01B0 0323") -> 	"1EF1";
decomposition("01B7 030C") -> 	"01EE";
decomposition("01EA 0304") -> 	"01EC";
decomposition("01EB 0304") -> 	"01ED";
decomposition("0226 0304") -> 	"01E0";
decomposition("0227 0304") -> 	"01E1";
decomposition("0228 0306") -> 	"1E1C";
decomposition("0229 0306") -> 	"1E1D";
decomposition("022E 0304") -> 	"0230";
decomposition("022F 0304") -> 	"0231";
decomposition("0292 030C") -> 	"01EF";
decomposition("02B9") -> 	"0374";
decomposition("0300") -> 	"0340";
decomposition("0301") -> 	"0341";
decomposition("0308 0301") -> 	"0344";
decomposition("0313") -> 	"0343";
decomposition("0385") -> 	"1FEE";
decomposition("0386") -> 	"1FBB";
decomposition("0388") -> 	"1FC9";
decomposition("0389") -> 	"1FCB";
decomposition("038A") -> 	"1FDB";
decomposition("038C") -> 	"1FF9";
decomposition("038E") -> 	"1FEB";
decomposition("038F") -> 	"1FFB";
decomposition("0390") -> 	"1FD3";
decomposition("0391 0300") -> 	"1FBA";
decomposition("0391 0301") -> 	"0386";
decomposition("0391 0304") -> 	"1FB9";
decomposition("0391 0306") -> 	"1FB8";
decomposition("0391 0313") -> 	"1F08";
decomposition("0391 0314") -> 	"1F09";
decomposition("0391 0345") -> 	"1FBC";
decomposition("0395 0300") -> 	"1FC8";
decomposition("0395 0301") -> 	"0388";
decomposition("0395 0313") -> 	"1F18";
decomposition("0395 0314") -> 	"1F19";
decomposition("0397 0300") -> 	"1FCA";
decomposition("0397 0301") -> 	"0389";
decomposition("0397 0313") -> 	"1F28";
decomposition("0397 0314") -> 	"1F29";
decomposition("0397 0345") -> 	"1FCC";
decomposition("0399 0300") -> 	"1FDA";
decomposition("0399 0301") -> 	"038A";
decomposition("0399 0304") -> 	"1FD9";
decomposition("0399 0306") -> 	"1FD8";
decomposition("0399 0308") -> 	"03AA";
decomposition("0399 0313") -> 	"1F38";
decomposition("0399 0314") -> 	"1F39";
decomposition("039F 0300") -> 	"1FF8";
decomposition("039F 0301") -> 	"038C";
decomposition("039F 0313") -> 	"1F48";
decomposition("039F 0314") -> 	"1F49";
decomposition("03A1 0314") -> 	"1FEC";
decomposition("03A5 0300") -> 	"1FEA";
decomposition("03A5 0301") -> 	"038E";
decomposition("03A5 0304") -> 	"1FE9";
decomposition("03A5 0306") -> 	"1FE8";
decomposition("03A5 0308") -> 	"03AB";
decomposition("03A5 0314") -> 	"1F59";
decomposition("03A9 0300") -> 	"1FFA";
decomposition("03A9 0301") -> 	"038F";
decomposition("03A9 0313") -> 	"1F68";
decomposition("03A9 0314") -> 	"1F69";
decomposition("03A9 0345") -> 	"1FFC";
decomposition("03A9") -> 	"2126";
decomposition("03AC 0345") -> 	"1FB4";
decomposition("03AC") -> 	"1F71";
decomposition("03AD") -> 	"1F73";
decomposition("03AE 0345") -> 	"1FC4";
decomposition("03AE") -> 	"1F75";
decomposition("03AF") -> 	"1F77";
decomposition("03B0") -> 	"1FE3";
decomposition("03B1 0300") -> 	"1F70";
decomposition("03B1 0301") -> 	"03AC";
decomposition("03B1 0304") -> 	"1FB1";
decomposition("03B1 0306") -> 	"1FB0";
decomposition("03B1 0313") -> 	"1F00";
decomposition("03B1 0314") -> 	"1F01";
decomposition("03B1 0342") -> 	"1FB6";
decomposition("03B1 0345") -> 	"1FB3";
decomposition("03B5 0300") -> 	"1F72";
decomposition("03B5 0301") -> 	"03AD";
decomposition("03B5 0313") -> 	"1F10";
decomposition("03B5 0314") -> 	"1F11";
decomposition("03B7 0300") -> 	"1F74";
decomposition("03B7 0301") -> 	"03AE";
decomposition("03B7 0313") -> 	"1F20";
decomposition("03B7 0314") -> 	"1F21";
decomposition("03B7 0342") -> 	"1FC6";
decomposition("03B7 0345") -> 	"1FC3";
decomposition("03B9 0300") -> 	"1F76";
decomposition("03B9 0301") -> 	"03AF";
decomposition("03B9 0304") -> 	"1FD1";
decomposition("03B9 0306") -> 	"1FD0";
decomposition("03B9 0308") -> 	"03CA";
decomposition("03B9 0313") -> 	"1F30";
decomposition("03B9 0314") -> 	"1F31";
decomposition("03B9 0342") -> 	"1FD6";
decomposition("03B9") -> 	"1FBE";
decomposition("03BF 0300") -> 	"1F78";
decomposition("03BF 0301") -> 	"03CC";
decomposition("03BF 0313") -> 	"1F40";
decomposition("03BF 0314") -> 	"1F41";
decomposition("03C1 0313") -> 	"1FE4";
decomposition("03C1 0314") -> 	"1FE5";
decomposition("03C5 0300") -> 	"1F7A";
decomposition("03C5 0301") -> 	"03CD";
decomposition("03C5 0304") -> 	"1FE1";
decomposition("03C5 0306") -> 	"1FE0";
decomposition("03C5 0308") -> 	"03CB";
decomposition("03C5 0313") -> 	"1F50";
decomposition("03C5 0314") -> 	"1F51";
decomposition("03C5 0342") -> 	"1FE6";
decomposition("03C9 0300") -> 	"1F7C";
decomposition("03C9 0301") -> 	"03CE";
decomposition("03C9 0313") -> 	"1F60";
decomposition("03C9 0314") -> 	"1F61";
decomposition("03C9 0342") -> 	"1FF6";
decomposition("03C9 0345") -> 	"1FF3";
decomposition("03CA 0300") -> 	"1FD2";
decomposition("03CA 0301") -> 	"0390";
decomposition("03CA 0342") -> 	"1FD7";
decomposition("03CB 0300") -> 	"1FE2";
decomposition("03CB 0301") -> 	"03B0";
decomposition("03CB 0342") -> 	"1FE7";
decomposition("03CC") -> 	"1F79";
decomposition("03CD") -> 	"1F7B";
decomposition("03CE 0345") -> 	"1FF4";
decomposition("03CE") -> 	"1F7D";
decomposition("03D2 0301") -> 	"03D3";
decomposition("03D2 0308") -> 	"03D4";
decomposition("0406 0308") -> 	"0407";
decomposition("0410 0306") -> 	"04D0";
decomposition("0410 0308") -> 	"04D2";
decomposition("0413 0301") -> 	"0403";
decomposition("0415 0300") -> 	"0400";
decomposition("0415 0306") -> 	"04D6";
decomposition("0415 0308") -> 	"0401";
decomposition("0416 0306") -> 	"04C1";
decomposition("0416 0308") -> 	"04DC";
decomposition("0417 0308") -> 	"04DE";
decomposition("0418 0300") -> 	"040D";
decomposition("0418 0304") -> 	"04E2";
decomposition("0418 0306") -> 	"0419";
decomposition("0418 0308") -> 	"04E4";
decomposition("041A 0301") -> 	"040C";
decomposition("041E 0308") -> 	"04E6";
decomposition("0423 0304") -> 	"04EE";
decomposition("0423 0306") -> 	"040E";
decomposition("0423 0308") -> 	"04F0";
decomposition("0423 030B") -> 	"04F2";
decomposition("0427 0308") -> 	"04F4";
decomposition("042B 0308") -> 	"04F8";
decomposition("042D 0308") -> 	"04EC";
decomposition("0430 0306") -> 	"04D1";
decomposition("0430 0308") -> 	"04D3";
decomposition("0433 0301") -> 	"0453";
decomposition("0435 0300") -> 	"0450";
decomposition("0435 0306") -> 	"04D7";
decomposition("0435 0308") -> 	"0451";
decomposition("0436 0306") -> 	"04C2";
decomposition("0436 0308") -> 	"04DD";
decomposition("0437 0308") -> 	"04DF";
decomposition("0438 0300") -> 	"045D";
decomposition("0438 0304") -> 	"04E3";
decomposition("0438 0306") -> 	"0439";
decomposition("0438 0308") -> 	"04E5";
decomposition("043A 0301") -> 	"045C";
decomposition("043E 0308") -> 	"04E7";
decomposition("0443 0304") -> 	"04EF";
decomposition("0443 0306") -> 	"045E";
decomposition("0443 0308") -> 	"04F1";
decomposition("0443 030B") -> 	"04F3";
decomposition("0447 0308") -> 	"04F5";
decomposition("044B 0308") -> 	"04F9";
decomposition("044D 0308") -> 	"04ED";
decomposition("0456 0308") -> 	"0457";
decomposition("0474 030F") -> 	"0476";
decomposition("0475 030F") -> 	"0477";
decomposition("04D8 0308") -> 	"04DA";
decomposition("04D9 0308") -> 	"04DB";
decomposition("04E8 0308") -> 	"04EA";
decomposition("04E9 0308") -> 	"04EB";
decomposition("05D0 05B7") -> 	"FB2E";
decomposition("05D0 05B8") -> 	"FB2F";
decomposition("05D0 05BC") -> 	"FB30";
decomposition("05D1 05BC") -> 	"FB31";
decomposition("05D1 05BF") -> 	"FB4C";
decomposition("05D2 05BC") -> 	"FB32";
decomposition("05D3 05BC") -> 	"FB33";
decomposition("05D4 05BC") -> 	"FB34";
decomposition("05D5 05B9") -> 	"FB4B";
decomposition("05D5 05BC") -> 	"FB35";
decomposition("05D6 05BC") -> 	"FB36";
decomposition("05D8 05BC") -> 	"FB38";
decomposition("05D9 05B4") -> 	"FB1D";
decomposition("05D9 05BC") -> 	"FB39";
decomposition("05DA 05BC") -> 	"FB3A";
decomposition("05DB 05BC") -> 	"FB3B";
decomposition("05DB 05BF") -> 	"FB4D";
decomposition("05DC 05BC") -> 	"FB3C";
decomposition("05DE 05BC") -> 	"FB3E";
decomposition("05E0 05BC") -> 	"FB40";
decomposition("05E1 05BC") -> 	"FB41";
decomposition("05E3 05BC") -> 	"FB43";
decomposition("05E4 05BC") -> 	"FB44";
decomposition("05E4 05BF") -> 	"FB4E";
decomposition("05E6 05BC") -> 	"FB46";
decomposition("05E7 05BC") -> 	"FB47";
decomposition("05E8 05BC") -> 	"FB48";
decomposition("05E9 05BC") -> 	"FB49";
decomposition("05E9 05C1") -> 	"FB2A";
decomposition("05E9 05C2") -> 	"FB2B";
decomposition("05EA 05BC") -> 	"FB4A";
decomposition("05F2 05B7") -> 	"FB1F";
decomposition("0627 0653") -> 	"0622";
decomposition("0627 0654") -> 	"0623";
decomposition("0627 0655") -> 	"0625";
decomposition("0648 0654") -> 	"0624";
decomposition("064A 0654") -> 	"0626";
decomposition("06C1 0654") -> 	"06C2";
decomposition("06D2 0654") -> 	"06D3";
decomposition("06D5 0654") -> 	"06C0";
decomposition("0915 093C") -> 	"0958";
decomposition("0916 093C") -> 	"0959";
decomposition("0917 093C") -> 	"095A";
decomposition("091C 093C") -> 	"095B";
decomposition("0921 093C") -> 	"095C";
decomposition("0922 093C") -> 	"095D";
decomposition("0928 093C") -> 	"0929";
decomposition("092B 093C") -> 	"095E";
decomposition("092F 093C") -> 	"095F";
decomposition("0930 093C") -> 	"0931";
decomposition("0933 093C") -> 	"0934";
decomposition("09A1 09BC") -> 	"09DC";
decomposition("09A2 09BC") -> 	"09DD";
decomposition("09AF 09BC") -> 	"09DF";
decomposition("09C7 09BE") -> 	"09CB";
decomposition("09C7 09D7") -> 	"09CC";
decomposition("0A16 0A3C") -> 	"0A59";
decomposition("0A17 0A3C") -> 	"0A5A";
decomposition("0A1C 0A3C") -> 	"0A5B";
decomposition("0A2B 0A3C") -> 	"0A5E";
decomposition("0A32 0A3C") -> 	"0A33";
decomposition("0A38 0A3C") -> 	"0A36";
decomposition("0B21 0B3C") -> 	"0B5C";
decomposition("0B22 0B3C") -> 	"0B5D";
decomposition("0B47 0B3E") -> 	"0B4B";
decomposition("0B47 0B56") -> 	"0B48";
decomposition("0B47 0B57") -> 	"0B4C";
decomposition("0B92 0BD7") -> 	"0B94";
decomposition("0BC6 0BBE") -> 	"0BCA";
decomposition("0BC6 0BD7") -> 	"0BCC";
decomposition("0BC7 0BBE") -> 	"0BCB";
decomposition("0C46 0C56") -> 	"0C48";
decomposition("0CBF 0CD5") -> 	"0CC0";
decomposition("0CC6 0CC2") -> 	"0CCA";
decomposition("0CC6 0CD5") -> 	"0CC7";
decomposition("0CC6 0CD6") -> 	"0CC8";
decomposition("0CCA 0CD5") -> 	"0CCB";
decomposition("0D46 0D3E") -> 	"0D4A";
decomposition("0D46 0D57") -> 	"0D4C";
decomposition("0D47 0D3E") -> 	"0D4B";
decomposition("0DD9 0DCA") -> 	"0DDA";
decomposition("0DD9 0DCF") -> 	"0DDC";
decomposition("0DD9 0DDF") -> 	"0DDE";
decomposition("0DDC 0DCA") -> 	"0DDD";
decomposition("0F40 0FB5") -> 	"0F69";
decomposition("0F42 0FB7") -> 	"0F43";
decomposition("0F4C 0FB7") -> 	"0F4D";
decomposition("0F51 0FB7") -> 	"0F52";
decomposition("0F56 0FB7") -> 	"0F57";
decomposition("0F5B 0FB7") -> 	"0F5C";
decomposition("0F71 0F72") -> 	"0F73";
decomposition("0F71 0F74") -> 	"0F75";
decomposition("0F71 0F80") -> 	"0F81";
decomposition("0F90 0FB5") -> 	"0FB9";
decomposition("0F92 0FB7") -> 	"0F93";
decomposition("0F9C 0FB7") -> 	"0F9D";
decomposition("0FA1 0FB7") -> 	"0FA2";
decomposition("0FA6 0FB7") -> 	"0FA7";
decomposition("0FAB 0FB7") -> 	"0FAC";
decomposition("0FB2 0F80") -> 	"0F76";
decomposition("0FB3 0F80") -> 	"0F78";
decomposition("1025 102E") -> 	"1026";
decomposition("11099 110BA") -> 	"1109A";
decomposition("1109B 110BA") -> 	"1109C";
decomposition("110A5 110BA") -> 	"110AB";
decomposition("11131 11127") -> 	"1112E";
decomposition("11132 11127") -> 	"1112F";
decomposition("11347 1133E") -> 	"1134B";
decomposition("11347 11357") -> 	"1134C";
decomposition("114B9 114B0") -> 	"114BC";
decomposition("115B8 115AF") -> 	"115BA";
decomposition("115B9 115AF") -> 	"115BB";
decomposition("1B05 1B35") -> 	"1B06";
decomposition("1B07 1B35") -> 	"1B08";
decomposition("1B09 1B35") -> 	"1B0A";
decomposition("1B0B 1B35") -> 	"1B0C";
decomposition("1B0D 1B35") -> 	"1B0E";
decomposition("1B11 1B35") -> 	"1B12";
decomposition("1B3A 1B35") -> 	"1B3B";
decomposition("1B3C 1B35") -> 	"1B3D";
decomposition("1B3E 1B35") -> 	"1B40";
decomposition("1B3F 1B35") -> 	"1B41";
decomposition("1B42 1B35") -> 	"1B43";
decomposition("1D157 1D165") -> 	"1D15E";
decomposition("1D158 1D165") -> 	"1D15F";
decomposition("1D15F 1D16E") -> 	"1D160";
decomposition("1D15F 1D170") -> 	"1D162";
decomposition("1D1B9 1D165") -> 	"1D1BB";
decomposition("1D1BA 1D165") -> 	"1D1BC";
decomposition("1D1BB 1D16E") -> 	"1D1BD";
decomposition("1D1BC 1D16E") -> 	"1D1BE";
decomposition("1E36 0304") -> 	"1E38";
decomposition("1E37 0304") -> 	"1E39";
decomposition("1E5A 0304") -> 	"1E5C";
decomposition("1E5B 0304") -> 	"1E5D";
decomposition("1E62 0307") -> 	"1E68";
decomposition("1E63 0307") -> 	"1E69";
decomposition("1EA0 0302") -> 	"1EAC";
decomposition("1EA0 0306") -> 	"1EB6";
decomposition("1EA1 0302") -> 	"1EAD";
decomposition("1EA1 0306") -> 	"1EB7";
decomposition("1EB8 0302") -> 	"1EC6";
decomposition("1EB9 0302") -> 	"1EC7";
decomposition("1ECC 0302") -> 	"1ED8";
decomposition("1ECD 0302") -> 	"1ED9";
decomposition("1F00 0300") -> 	"1F02";
decomposition("1F00 0301") -> 	"1F04";
decomposition("1F00 0342") -> 	"1F06";
decomposition("1F00 0345") -> 	"1F80";
decomposition("1F01 0300") -> 	"1F03";
decomposition("1F01 0301") -> 	"1F05";
decomposition("1F01 0342") -> 	"1F07";
decomposition("1F01 0345") -> 	"1F81";
decomposition("1F02 0345") -> 	"1F82";
decomposition("1F03 0345") -> 	"1F83";
decomposition("1F04 0345") -> 	"1F84";
decomposition("1F05 0345") -> 	"1F85";
decomposition("1F06 0345") -> 	"1F86";
decomposition("1F07 0345") -> 	"1F87";
decomposition("1F08 0300") -> 	"1F0A";
decomposition("1F08 0301") -> 	"1F0C";
decomposition("1F08 0342") -> 	"1F0E";
decomposition("1F08 0345") -> 	"1F88";
decomposition("1F09 0300") -> 	"1F0B";
decomposition("1F09 0301") -> 	"1F0D";
decomposition("1F09 0342") -> 	"1F0F";
decomposition("1F09 0345") -> 	"1F89";
decomposition("1F0A 0345") -> 	"1F8A";
decomposition("1F0B 0345") -> 	"1F8B";
decomposition("1F0C 0345") -> 	"1F8C";
decomposition("1F0D 0345") -> 	"1F8D";
decomposition("1F0E 0345") -> 	"1F8E";
decomposition("1F0F 0345") -> 	"1F8F";
decomposition("1F10 0300") -> 	"1F12";
decomposition("1F10 0301") -> 	"1F14";
decomposition("1F11 0300") -> 	"1F13";
decomposition("1F11 0301") -> 	"1F15";
decomposition("1F18 0300") -> 	"1F1A";
decomposition("1F18 0301") -> 	"1F1C";
decomposition("1F19 0300") -> 	"1F1B";
decomposition("1F19 0301") -> 	"1F1D";
decomposition("1F20 0300") -> 	"1F22";
decomposition("1F20 0301") -> 	"1F24";
decomposition("1F20 0342") -> 	"1F26";
decomposition("1F20 0345") -> 	"1F90";
decomposition("1F21 0300") -> 	"1F23";
decomposition("1F21 0301") -> 	"1F25";
decomposition("1F21 0342") -> 	"1F27";
decomposition("1F21 0345") -> 	"1F91";
decomposition("1F22 0345") -> 	"1F92";
decomposition("1F23 0345") -> 	"1F93";
decomposition("1F24 0345") -> 	"1F94";
decomposition("1F25 0345") -> 	"1F95";
decomposition("1F26 0345") -> 	"1F96";
decomposition("1F27 0345") -> 	"1F97";
decomposition("1F28 0300") -> 	"1F2A";
decomposition("1F28 0301") -> 	"1F2C";
decomposition("1F28 0342") -> 	"1F2E";
decomposition("1F28 0345") -> 	"1F98";
decomposition("1F29 0300") -> 	"1F2B";
decomposition("1F29 0301") -> 	"1F2D";
decomposition("1F29 0342") -> 	"1F2F";
decomposition("1F29 0345") -> 	"1F99";
decomposition("1F2A 0345") -> 	"1F9A";
decomposition("1F2B 0345") -> 	"1F9B";
decomposition("1F2C 0345") -> 	"1F9C";
decomposition("1F2D 0345") -> 	"1F9D";
decomposition("1F2E 0345") -> 	"1F9E";
decomposition("1F2F 0345") -> 	"1F9F";
decomposition("1F30 0300") -> 	"1F32";
decomposition("1F30 0301") -> 	"1F34";
decomposition("1F30 0342") -> 	"1F36";
decomposition("1F31 0300") -> 	"1F33";
decomposition("1F31 0301") -> 	"1F35";
decomposition("1F31 0342") -> 	"1F37";
decomposition("1F38 0300") -> 	"1F3A";
decomposition("1F38 0301") -> 	"1F3C";
decomposition("1F38 0342") -> 	"1F3E";
decomposition("1F39 0300") -> 	"1F3B";
decomposition("1F39 0301") -> 	"1F3D";
decomposition("1F39 0342") -> 	"1F3F";
decomposition("1F40 0300") -> 	"1F42";
decomposition("1F40 0301") -> 	"1F44";
decomposition("1F41 0300") -> 	"1F43";
decomposition("1F41 0301") -> 	"1F45";
decomposition("1F48 0300") -> 	"1F4A";
decomposition("1F48 0301") -> 	"1F4C";
decomposition("1F49 0300") -> 	"1F4B";
decomposition("1F49 0301") -> 	"1F4D";
decomposition("1F50 0300") -> 	"1F52";
decomposition("1F50 0301") -> 	"1F54";
decomposition("1F50 0342") -> 	"1F56";
decomposition("1F51 0300") -> 	"1F53";
decomposition("1F51 0301") -> 	"1F55";
decomposition("1F51 0342") -> 	"1F57";
decomposition("1F59 0300") -> 	"1F5B";
decomposition("1F59 0301") -> 	"1F5D";
decomposition("1F59 0342") -> 	"1F5F";
decomposition("1F60 0300") -> 	"1F62";
decomposition("1F60 0301") -> 	"1F64";
decomposition("1F60 0342") -> 	"1F66";
decomposition("1F60 0345") -> 	"1FA0";
decomposition("1F61 0300") -> 	"1F63";
decomposition("1F61 0301") -> 	"1F65";
decomposition("1F61 0342") -> 	"1F67";
decomposition("1F61 0345") -> 	"1FA1";
decomposition("1F62 0345") -> 	"1FA2";
decomposition("1F63 0345") -> 	"1FA3";
decomposition("1F64 0345") -> 	"1FA4";
decomposition("1F65 0345") -> 	"1FA5";
decomposition("1F66 0345") -> 	"1FA6";
decomposition("1F67 0345") -> 	"1FA7";
decomposition("1F68 0300") -> 	"1F6A";
decomposition("1F68 0301") -> 	"1F6C";
decomposition("1F68 0342") -> 	"1F6E";
decomposition("1F68 0345") -> 	"1FA8";
decomposition("1F69 0300") -> 	"1F6B";
decomposition("1F69 0301") -> 	"1F6D";
decomposition("1F69 0342") -> 	"1F6F";
decomposition("1F69 0345") -> 	"1FA9";
decomposition("1F6A 0345") -> 	"1FAA";
decomposition("1F6B 0345") -> 	"1FAB";
decomposition("1F6C 0345") -> 	"1FAC";
decomposition("1F6D 0345") -> 	"1FAD";
decomposition("1F6E 0345") -> 	"1FAE";
decomposition("1F6F 0345") -> 	"1FAF";
decomposition("1F70 0345") -> 	"1FB2";
decomposition("1F74 0345") -> 	"1FC2";
decomposition("1F7C 0345") -> 	"1FF2";
decomposition("1FB6 0345") -> 	"1FB7";
decomposition("1FBF 0300") -> 	"1FCD";
decomposition("1FBF 0301") -> 	"1FCE";
decomposition("1FBF 0342") -> 	"1FCF";
decomposition("1FC6 0345") -> 	"1FC7";
decomposition("1FF6 0345") -> 	"1FF7";
decomposition("1FFE 0300") -> 	"1FDD";
decomposition("1FFE 0301") -> 	"1FDE";
decomposition("1FFE 0342") -> 	"1FDF";
decomposition("2002") -> 	"2000";
decomposition("2003") -> 	"2001";
decomposition("20122") -> 	"2F803";
decomposition("2051C") -> 	"2F812";
decomposition("20525") -> 	"2F91B";
decomposition("2054B") -> 	"2F816";
decomposition("2063A") -> 	"2F80D";
decomposition("20804") -> 	"2F9D9";
decomposition("208DE") -> 	"2F9DD";
decomposition("20A2C") -> 	"2F834";
decomposition("20B63") -> 	"2F838";
decomposition("214E4") -> 	"2F859";
decomposition("216A8") -> 	"2F860";
decomposition("216EA") -> 	"2F861";
decomposition("2190 0338") -> 	"219A";
decomposition("2192 0338") -> 	"219B";
decomposition("2194 0338") -> 	"21AE";
decomposition("219C8") -> 	"2F86C";
decomposition("21B18") -> 	"2F871";
decomposition("21D0 0338") -> 	"21CD";
decomposition("21D0B") -> 	"2F8F8";
decomposition("21D2 0338") -> 	"21CF";
decomposition("21D4 0338") -> 	"21CE";
decomposition("21DE4") -> 	"2F87B";
decomposition("21DE6") -> 	"2F87D";
decomposition("2203 0338") -> 	"2204";
decomposition("2208 0338") -> 	"2209";
decomposition("220B 0338") -> 	"220C";
decomposition("22183") -> 	"2F889";
decomposition("2219F") -> 	"2F939";
decomposition("2223 0338") -> 	"2224";
decomposition("2225 0338") -> 	"2226";
decomposition("22331") -> 	"2F891";
decomposition("223C 0338") -> 	"2241";
decomposition("2243 0338") -> 	"2244";
decomposition("2245 0338") -> 	"2247";
decomposition("2248 0338") -> 	"2249";
decomposition("224D 0338") -> 	"226D";
decomposition("2261 0338") -> 	"2262";
decomposition("2264 0338") -> 	"2270";
decomposition("2265 0338") -> 	"2271";
decomposition("226D4") -> 	"2F8A4";
decomposition("2272 0338") -> 	"2274";
decomposition("2273 0338") -> 	"2275";
decomposition("2276 0338") -> 	"2278";
decomposition("2277 0338") -> 	"2279";
decomposition("227A 0338") -> 	"2280";
decomposition("227B 0338") -> 	"2281";
decomposition("227C 0338") -> 	"22E0";
decomposition("227D 0338") -> 	"22E1";
decomposition("2282 0338") -> 	"2284";
decomposition("2283 0338") -> 	"2285";
decomposition("22844") -> 	"FAD0";
decomposition("2284A") -> 	"FACF";
decomposition("2286 0338") -> 	"2288";
decomposition("2287 0338") -> 	"2289";
decomposition("2291 0338") -> 	"22E2";
decomposition("2292 0338") -> 	"22E3";
decomposition("22A2 0338") -> 	"22AC";
decomposition("22A8 0338") -> 	"22AD";
decomposition("22A9 0338") -> 	"22AE";
decomposition("22AB 0338") -> 	"22AF";
decomposition("22B0C") -> 	"2F8B8";
decomposition("22B2 0338") -> 	"22EA";
decomposition("22B3 0338") -> 	"22EB";
decomposition("22B4 0338") -> 	"22EC";
decomposition("22B5 0338") -> 	"22ED";
decomposition("22BF1") -> 	"2F8BE";
decomposition("2300A") -> 	"2F8CA";
decomposition("232B8") -> 	"2F897";
decomposition("2335F") -> 	"2F980";
decomposition("23393") -> 	"2F989";
decomposition("2339C") -> 	"2F98A";
decomposition("233C3") -> 	"2F8DD";
decomposition("233D5") -> 	"FAD1";
decomposition("2346D") -> 	"2F8E3";
decomposition("236A3") -> 	"2F8EC";
decomposition("238A7") -> 	"2F8F0";
decomposition("23A8D") -> 	"2F8F7";
decomposition("23AFA") -> 	"2F8F9";
decomposition("23CBC") -> 	"2F8FB";
decomposition("23D1E") -> 	"2F906";
decomposition("23ED1") -> 	"2F90D";
decomposition("23F5E") -> 	"2F910";
decomposition("23F8E") -> 	"2F911";
decomposition("24263") -> 	"2F91D";
decomposition("242EE") -> 	"FA6C";
decomposition("243AB") -> 	"2F91F";
decomposition("24608") -> 	"2F923";
decomposition("24735") -> 	"2F926";
decomposition("24814") -> 	"2F927";
decomposition("24C36") -> 	"2F935";
decomposition("24C92") -> 	"2F937";
decomposition("24FA1") -> 	"2F93B";
decomposition("24FB8") -> 	"2F93C";
decomposition("25044") -> 	"2F93D";
decomposition("250F2") -> 	"2F942";
decomposition("250F3") -> 	"2F941";
decomposition("25119") -> 	"2F943";
decomposition("25133") -> 	"2F944";
decomposition("25249") -> 	"FAD5";
decomposition("2541D") -> 	"2F94D";
decomposition("25626") -> 	"2F952";
decomposition("2569A") -> 	"2F954";
decomposition("256C5") -> 	"2F955";
decomposition("2597C") -> 	"2F95C";
decomposition("25AA7") -> 	"2F95D";
decomposition("25BAB") -> 	"2F961";
decomposition("25C80") -> 	"2F965";
decomposition("25CD0") -> 	"FAD6";
decomposition("25F86") -> 	"2F96B";
decomposition("261DA") -> 	"2F898";
decomposition("26228") -> 	"2F972";
decomposition("26247") -> 	"2F973";
decomposition("262D9") -> 	"2F975";
decomposition("2633E") -> 	"2F977";
decomposition("264DA") -> 	"2F97B";
decomposition("26523") -> 	"2F97C";
decomposition("265A8") -> 	"2F97E";
decomposition("267A7") -> 	"2F987";
decomposition("267B5") -> 	"2F988";
decomposition("26B3C") -> 	"2F997";
decomposition("26C36") -> 	"2F9A4";
decomposition("26CD5") -> 	"2F9A6";
decomposition("26D6B") -> 	"2F9A5";
decomposition("26F2C") -> 	"2F9AD";
decomposition("26FB1") -> 	"2F9B0";
decomposition("270D2") -> 	"2F9B1";
decomposition("273CA") -> 	"2F9AB";
decomposition("27667") -> 	"2F9C5";
decomposition("278AE") -> 	"2F9CB";
decomposition("27966") -> 	"2F9CC";
decomposition("27CA8") -> 	"2F9D3";
decomposition("27ED3") -> 	"FAD7";
decomposition("27F2F") -> 	"2F9D8";
decomposition("285D2") -> 	"2F9E0";
decomposition("285ED") -> 	"2F9E1";
decomposition("2872E") -> 	"2F9E5";
decomposition("28BFA") -> 	"2F9ED";
decomposition("28D77") -> 	"2F9F1";
decomposition("29145") -> 	"2F9F6";
decomposition("291DF") -> 	"2F81C";
decomposition("2921A") -> 	"2F9F7";
decomposition("2940A") -> 	"2F9FB";
decomposition("29496") -> 	"2F9FD";
decomposition("295B6") -> 	"2FA01";
decomposition("29B30") -> 	"2FA09";
decomposition("2A0CE") -> 	"2FA10";
decomposition("2A105") -> 	"2FA12";
decomposition("2A20E") -> 	"2FA13";
decomposition("2A291") -> 	"2FA14";
decomposition("2A392") -> 	"2F88F";
decomposition("2A600") -> 	"2FA1D";
decomposition("2ADD 0338") -> 	"2ADC";
decomposition("3008") -> 	"2329";
decomposition("3009") -> 	"232A";
decomposition("3046 3099") -> 	"3094";
decomposition("304B 3099") -> 	"304C";
decomposition("304D 3099") -> 	"304E";
decomposition("304F 3099") -> 	"3050";
decomposition("3051 3099") -> 	"3052";
decomposition("3053 3099") -> 	"3054";
decomposition("3055 3099") -> 	"3056";
decomposition("3057 3099") -> 	"3058";
decomposition("3059 3099") -> 	"305A";
decomposition("305B 3099") -> 	"305C";
decomposition("305D 3099") -> 	"305E";
decomposition("305F 3099") -> 	"3060";
decomposition("3061 3099") -> 	"3062";
decomposition("3064 3099") -> 	"3065";
decomposition("3066 3099") -> 	"3067";
decomposition("3068 3099") -> 	"3069";
decomposition("306F 3099") -> 	"3070";
decomposition("306F 309A") -> 	"3071";
decomposition("3072 3099") -> 	"3073";
decomposition("3072 309A") -> 	"3074";
decomposition("3075 3099") -> 	"3076";
decomposition("3075 309A") -> 	"3077";
decomposition("3078 3099") -> 	"3079";
decomposition("3078 309A") -> 	"307A";
decomposition("307B 3099") -> 	"307C";
decomposition("307B 309A") -> 	"307D";
decomposition("309D 3099") -> 	"309E";
decomposition("30A6 3099") -> 	"30F4";
decomposition("30AB 3099") -> 	"30AC";
decomposition("30AD 3099") -> 	"30AE";
decomposition("30AF 3099") -> 	"30B0";
decomposition("30B1 3099") -> 	"30B2";
decomposition("30B3 3099") -> 	"30B4";
decomposition("30B5 3099") -> 	"30B6";
decomposition("30B7 3099") -> 	"30B8";
decomposition("30B9 3099") -> 	"30BA";
decomposition("30BB 3099") -> 	"30BC";
decomposition("30BD 3099") -> 	"30BE";
decomposition("30BF 3099") -> 	"30C0";
decomposition("30C1 3099") -> 	"30C2";
decomposition("30C4 3099") -> 	"30C5";
decomposition("30C6 3099") -> 	"30C7";
decomposition("30C8 3099") -> 	"30C9";
decomposition("30CF 3099") -> 	"30D0";
decomposition("30CF 309A") -> 	"30D1";
decomposition("30D2 3099") -> 	"30D3";
decomposition("30D2 309A") -> 	"30D4";
decomposition("30D5 3099") -> 	"30D6";
decomposition("30D5 309A") -> 	"30D7";
decomposition("30D8 3099") -> 	"30D9";
decomposition("30D8 309A") -> 	"30DA";
decomposition("30DB 3099") -> 	"30DC";
decomposition("30DB 309A") -> 	"30DD";
decomposition("30EF 3099") -> 	"30F7";
decomposition("30F0 3099") -> 	"30F8";
decomposition("30F1 3099") -> 	"30F9";
decomposition("30F2 3099") -> 	"30FA";
decomposition("30FD 3099") -> 	"30FE";
decomposition("349E") -> 	"2F80C";
decomposition("34B9") -> 	"2F813";
decomposition("34BB") -> 	"2F9CA";
decomposition("34DF") -> 	"2F81F";
decomposition("3515") -> 	"2F824";
decomposition("36EE") -> 	"2F867";
decomposition("36FC") -> 	"2F868";
decomposition("3781") -> 	"2F876";
decomposition("382F") -> 	"2F883";
decomposition("3862") -> 	"2F888";
decomposition("387C") -> 	"2F88A";
decomposition("38C7") -> 	"2F896";
decomposition("38E3") -> 	"2F89B";
decomposition("391C") -> 	"2F8A2";
decomposition("393A") -> 	"2F8A1";
decomposition("3A2E") -> 	"2F8C2";
decomposition("3A6C") -> 	"2F8C7";
decomposition("3AE4") -> 	"2F8D1";
decomposition("3B08") -> 	"2F8D0";
decomposition("3B19") -> 	"2F8CE";
decomposition("3B49") -> 	"2F8DE";
decomposition("3B9D") -> 	"2F8E7";
decomposition("3C18") -> 	"2F8EE";
decomposition("3C4E") -> 	"2F8F2";
decomposition("3D33") -> 	"2F90A";
decomposition("3D96") -> 	"2F916";
decomposition("3EAC") -> 	"2F92A";
decomposition("3EB8") -> 	"2F92C";
decomposition("3F1B") -> 	"2F933";
decomposition("3FFC") -> 	"2F93E";
decomposition("4008") -> 	"2F93F";
decomposition("4018") -> 	"FAD3";
decomposition("4039") -> 	"2F949";
decomposition("4046") -> 	"2F94B";
decomposition("4096") -> 	"2F94C";
decomposition("40E3") -> 	"2F951";
decomposition("412F") -> 	"2F958";
decomposition("4202") -> 	"2F960";
decomposition("4227") -> 	"2F964";
decomposition("42A0") -> 	"2F967";
decomposition("4301") -> 	"2F96D";
decomposition("4334") -> 	"2F971";
decomposition("4359") -> 	"2F974";
decomposition("43D5") -> 	"2F981";
decomposition("43D9") -> 	"2F8D7";
decomposition("440B") -> 	"2F984";
decomposition("446B") -> 	"2F98E";
decomposition("452B") -> 	"2F9A7";
decomposition("455D") -> 	"2F9AE";
decomposition("4561") -> 	"2F9AF";
decomposition("456B") -> 	"2F9B2";
decomposition("45D7") -> 	"2F9BF";
decomposition("45F9") -> 	"2F9C2";
decomposition("4635") -> 	"2F9C8";
decomposition("46BE") -> 	"2F9CD";
decomposition("46C7") -> 	"2F9CE";
decomposition("4995") -> 	"2F9EF";
decomposition("49E6") -> 	"2F9F2";
decomposition("4A6E") -> 	"2F9F8";
decomposition("4A76") -> 	"2F9F9";
decomposition("4AB2") -> 	"2F9FC";
decomposition("4B33") -> 	"2FA03";
decomposition("4BCE") -> 	"2FA08";
decomposition("4CCE") -> 	"2FA0D";
decomposition("4CED") -> 	"2FA0E";
decomposition("4CF8") -> 	"2FA11";
decomposition("4D56") -> 	"2FA16";
decomposition("4E0D") -> 	"F967";
decomposition("4E26") -> 	"FA70";
decomposition("4E32") -> 	"F905";
decomposition("4E38") -> 	"2F801";
decomposition("4E39") -> 	"F95E";
decomposition("4E3D") -> 	"2F800";
decomposition("4E41") -> 	"2F802";
decomposition("4E82") -> 	"F91B";
decomposition("4E86") -> 	"F9BA";
decomposition("4EAE") -> 	"F977";
decomposition("4EC0") -> 	"F9FD";
decomposition("4ECC") -> 	"2F819";
decomposition("4EE4") -> 	"F9A8";
decomposition("4F60") -> 	"2F804";
decomposition("4F80") -> 	"FA73";
decomposition("4F86") -> 	"F92D";
decomposition("4F8B") -> 	"F9B5";
decomposition("4FAE") -> 	"2F805";
decomposition("4FBB") -> 	"2F806";
decomposition("4FBF") -> 	"F965";
decomposition("5002") -> 	"2F807";
decomposition("502B") -> 	"F9D4";
decomposition("507A") -> 	"2F808";
decomposition("5099") -> 	"2F809";
decomposition("50CF") -> 	"2F80B";
decomposition("50DA") -> 	"F9BB";
decomposition("50E7") -> 	"2F80A";
decomposition("5140") -> 	"FA0C";
decomposition("5145") -> 	"FA74";
decomposition("514D") -> 	"2F80E";
decomposition("5154") -> 	"2F80F";
decomposition("5164") -> 	"2F810";
decomposition("5167") -> 	"2F814";
decomposition("5168") -> 	"FA72";
decomposition("5169") -> 	"F978";
decomposition("516D") -> 	"F9D1";
decomposition("5177") -> 	"2F811";
decomposition("5180") -> 	"FA75";
decomposition("518D") -> 	"2F815";
decomposition("5192") -> 	"2F8D2";
decomposition("5195") -> 	"2F8D3";
decomposition("5197") -> 	"2F817";
decomposition("51A4") -> 	"2F818";
decomposition("51AC") -> 	"2F81A";
decomposition("51B5") -> 	"2F81B";
decomposition("51B7") -> 	"F92E";
decomposition("51C9") -> 	"F979";
decomposition("51CC") -> 	"F955";
decomposition("51DC") -> 	"F954";
decomposition("51DE") -> 	"FA15";
decomposition("51F5") -> 	"2F81D";
decomposition("5203") -> 	"2F81E";
decomposition("5207") -> 	"2F850";
decomposition("5217") -> 	"F99C";
decomposition("5229") -> 	"F9DD";
decomposition("523A") -> 	"F9FF";
decomposition("523B") -> 	"2F820";
decomposition("5246") -> 	"2F821";
decomposition("5272") -> 	"2F822";
decomposition("5277") -> 	"2F823";
decomposition("5289") -> 	"F9C7";
decomposition("529B") -> 	"F98A";
decomposition("52A3") -> 	"F99D";
decomposition("52B3") -> 	"2F992";
decomposition("52C7") -> 	"2F825";
decomposition("52C9") -> 	"2F826";
decomposition("52D2") -> 	"F952";
decomposition("52DE") -> 	"F92F";
decomposition("52E4") -> 	"2F827";
decomposition("52F5") -> 	"F97F";
decomposition("52FA") -> 	"2F828";
decomposition("5305") -> 	"2F829";
decomposition("5306") -> 	"2F82A";
decomposition("5317") -> 	"2F82B";
decomposition("533F") -> 	"F9EB";
decomposition("5349") -> 	"2F82C";
decomposition("5351") -> 	"2F82D";
decomposition("535A") -> 	"2F82E";
decomposition("5373") -> 	"2F82F";
decomposition("5375") -> 	"F91C";
decomposition("537D") -> 	"2F830";
decomposition("537F") -> 	"2F831";
decomposition("53C3") -> 	"F96B";
decomposition("53CA") -> 	"2F836";
decomposition("53DF") -> 	"2F837";
decomposition("53E5") -> 	"F906";
decomposition("53EB") -> 	"2F839";
decomposition("53F1") -> 	"2F83A";
decomposition("5406") -> 	"2F83B";
decomposition("540F") -> 	"F9DE";
decomposition("541D") -> 	"F9ED";
decomposition("5438") -> 	"2F83D";
decomposition("5442") -> 	"F980";
decomposition("5448") -> 	"2F83E";
decomposition("5468") -> 	"2F83F";
decomposition("549E") -> 	"2F83C";
decomposition("54A2") -> 	"2F840";
decomposition("54BD") -> 	"F99E";
decomposition("54F6") -> 	"2F841";
decomposition("5510") -> 	"2F842";
decomposition("5553") -> 	"2F843";
decomposition("5555") -> 	"FA79";
decomposition("5563") -> 	"2F844";
decomposition("5584") -> 	"2F845";
decomposition("5587") -> 	"F90B";
decomposition("5599") -> 	"2F847";
decomposition("559D") -> 	"FA36";
decomposition("55AB") -> 	"2F848";
decomposition("55B3") -> 	"2F849";
decomposition("55C0") -> 	"FA0D";
decomposition("55C2") -> 	"2F84A";
decomposition("55E2") -> 	"FA7B";
decomposition("5606") -> 	"2F84C";
decomposition("5651") -> 	"2F84E";
decomposition("5668") -> 	"FA38";
decomposition("5674") -> 	"2F84F";
decomposition("56F9") -> 	"F9A9";
decomposition("5716") -> 	"2F84B";
decomposition("5717") -> 	"2F84D";
decomposition("578B") -> 	"2F855";
decomposition("57CE") -> 	"2F852";
decomposition("57F4") -> 	"2F853";
decomposition("580D") -> 	"2F854";
decomposition("5831") -> 	"2F857";
decomposition("5832") -> 	"2F856";
decomposition("5840") -> 	"FA39";
decomposition("585A") -> 	"FA10";
decomposition("585E") -> 	"F96C";
decomposition("58A8") -> 	"FA3A";
decomposition("58AC") -> 	"2F858";
decomposition("58B3") -> 	"FA7D";
decomposition("58D8") -> 	"F94A";
decomposition("58DF") -> 	"F942";
decomposition("58EE") -> 	"2F851";
decomposition("58F2") -> 	"2F85A";
decomposition("58F7") -> 	"2F85B";
decomposition("5906") -> 	"2F85C";
decomposition("591A") -> 	"2F85D";
decomposition("5922") -> 	"2F85E";
decomposition("5944") -> 	"FA7E";
decomposition("5948") -> 	"F90C";
decomposition("5951") -> 	"F909";
decomposition("5954") -> 	"FA7F";
decomposition("5962") -> 	"2F85F";
decomposition("5973") -> 	"F981";
decomposition("59D8") -> 	"2F865";
decomposition("59EC") -> 	"2F862";
decomposition("5A1B") -> 	"2F863";
decomposition("5A27") -> 	"2F864";
decomposition("5A62") -> 	"FA80";
decomposition("5A66") -> 	"2F866";
decomposition("5AB5") -> 	"2F986";
decomposition("5B08") -> 	"2F869";
decomposition("5B28") -> 	"FA81";
decomposition("5B3E") -> 	"2F86A";
decomposition("5B85") -> 	"FA04";
decomposition("5BC3") -> 	"2F86D";
decomposition("5BD8") -> 	"2F86E";
decomposition("5BE7") -> 	"2F86F";
decomposition("5BEE") -> 	"F9BC";
decomposition("5BF3") -> 	"2F870";
decomposition("5BFF") -> 	"2F872";
decomposition("5C06") -> 	"2F873";
decomposition("5C22") -> 	"2F875";
decomposition("5C3F") -> 	"F9BD";
decomposition("5C60") -> 	"2F877";
decomposition("5C62") -> 	"F94B";
decomposition("5C64") -> 	"FA3B";
decomposition("5C65") -> 	"F9DF";
decomposition("5C6E") -> 	"2F878";
decomposition("5C8D") -> 	"2F87A";
decomposition("5CC0") -> 	"2F879";
decomposition("5D19") -> 	"F9D5";
decomposition("5D43") -> 	"2F87C";
decomposition("5D50") -> 	"F921";
decomposition("5D6B") -> 	"2F87F";
decomposition("5D6E") -> 	"2F87E";
decomposition("5D7C") -> 	"2F880";
decomposition("5DB2") -> 	"2F9F4";
decomposition("5DBA") -> 	"F9AB";
decomposition("5DE1") -> 	"2F881";
decomposition("5DE2") -> 	"2F882";
decomposition("5DFD") -> 	"2F884";
decomposition("5E28") -> 	"2F885";
decomposition("5E3D") -> 	"2F886";
decomposition("5E69") -> 	"2F887";
decomposition("5E74") -> 	"F98E";
decomposition("5EA6") -> 	"FA01";
decomposition("5EB0") -> 	"2F88B";
decomposition("5EB3") -> 	"2F88C";
decomposition("5EB6") -> 	"2F88D";
decomposition("5EC9") -> 	"F9A2";
decomposition("5ECA") -> 	"2F88E";
decomposition("5ED2") -> 	"FA82";
decomposition("5ED3") -> 	"FA0B";
decomposition("5ED9") -> 	"FA83";
decomposition("5EEC") -> 	"F982";
decomposition("5EFE") -> 	"2F890";
decomposition("5F04") -> 	"F943";
decomposition("5F22") -> 	"2F894";
decomposition("5F53") -> 	"2F874";
decomposition("5F62") -> 	"2F899";
decomposition("5F69") -> 	"FA84";
decomposition("5F6B") -> 	"2F89A";
decomposition("5F8B") -> 	"F9D8";
decomposition("5F9A") -> 	"2F89C";
decomposition("5FA9") -> 	"F966";
decomposition("5FAD") -> 	"FA85";
decomposition("5FCD") -> 	"2F89D";
decomposition("5FD7") -> 	"2F89E";
decomposition("5FF5") -> 	"F9A3";
decomposition("5FF9") -> 	"2F89F";
decomposition("6012") -> 	"F960";
decomposition("601C") -> 	"F9AC";
decomposition("6075") -> 	"FA6B";
decomposition("6081") -> 	"2F8A0";
decomposition("6094") -> 	"2F8A3";
decomposition("60C7") -> 	"2F8A5";
decomposition("60D8") -> 	"FA86";
decomposition("60E1") -> 	"F9B9";
decomposition("6108") -> 	"FA88";
decomposition("6144") -> 	"F9D9";
decomposition("6148") -> 	"2F8A6";
decomposition("614C") -> 	"2F8A7";
decomposition("614E") -> 	"2F8A8";
decomposition("6160") -> 	"FA8A";
decomposition("6168") -> 	"FA3E";
decomposition("617A") -> 	"2F8AA";
decomposition("618E") -> 	"2F8AB";
decomposition("6190") -> 	"F98F";
decomposition("61A4") -> 	"2F8AD";
decomposition("61AF") -> 	"2F8AE";
decomposition("61B2") -> 	"2F8AC";
decomposition("61DE") -> 	"2F8AF";
decomposition("61F2") -> 	"2F8B0";
decomposition("61F6") -> 	"2F8B1";
decomposition("6200") -> 	"F990";
decomposition("6210") -> 	"2F8B2";
decomposition("621B") -> 	"2F8B3";
decomposition("622E") -> 	"F9D2";
decomposition("6234") -> 	"FA8C";
decomposition("625D") -> 	"2F8B4";
decomposition("62B1") -> 	"2F8B5";
decomposition("62C9") -> 	"F925";
decomposition("62CF") -> 	"F95B";
decomposition("62D3") -> 	"FA02";
decomposition("62D4") -> 	"2F8B6";
decomposition("62FC") -> 	"2F8BA";
decomposition("62FE") -> 	"F973";
decomposition("633D") -> 	"2F8B9";
decomposition("6350") -> 	"2F8B7";
decomposition("6368") -> 	"2F8BB";
decomposition("637B") -> 	"F9A4";
decomposition("6383") -> 	"2F8BC";
decomposition("63A0") -> 	"F975";
decomposition("63A9") -> 	"2F8C1";
decomposition("63C4") -> 	"FA8D";
decomposition("63C5") -> 	"2F8C0";
decomposition("63E4") -> 	"2F8BD";
decomposition("641C") -> 	"FA8E";
decomposition("6422") -> 	"2F8BF";
decomposition("6452") -> 	"FA8F";
decomposition("6469") -> 	"2F8C3";
decomposition("6477") -> 	"2F8C6";
decomposition("647E") -> 	"2F8C4";
decomposition("649A") -> 	"F991";
decomposition("649D") -> 	"2F8C5";
decomposition("64C4") -> 	"F930";
decomposition("654F") -> 	"2F8C8";
decomposition("6556") -> 	"FA90";
decomposition("656C") -> 	"2F8C9";
decomposition("6578") -> 	"F969";
decomposition("6599") -> 	"F9BE";
decomposition("65C5") -> 	"F983";
decomposition("65E2") -> 	"FA42";
decomposition("65E3") -> 	"2F8CB";
decomposition("6613") -> 	"F9E0";
decomposition("6649") -> 	"2F8CD";
decomposition("6674") -> 	"FA12";
decomposition("6688") -> 	"F9C5";
decomposition("6691") -> 	"2F8CF";
decomposition("669C") -> 	"2F8D5";
decomposition("66B4") -> 	"FA06";
decomposition("66C6") -> 	"F98B";
decomposition("66F4") -> 	"F901";
decomposition("66F8") -> 	"2F8CC";
decomposition("6700") -> 	"2F8D4";
decomposition("6717") -> 	"2F8D8";
decomposition("671B") -> 	"2F8D9";
decomposition("6721") -> 	"2F8DA";
decomposition("674E") -> 	"F9E1";
decomposition("6753") -> 	"2F8DC";
decomposition("6756") -> 	"FA94";
decomposition("675E") -> 	"2F8DB";
decomposition("677B") -> 	"F9C8";
decomposition("6785") -> 	"2F8E0";
decomposition("6797") -> 	"F9F4";
decomposition("67F3") -> 	"F9C9";
decomposition("67FA") -> 	"2F8DF";
decomposition("6817") -> 	"F9DA";
decomposition("681F") -> 	"2F8E5";
decomposition("6852") -> 	"2F8E1";
decomposition("6881") -> 	"F97A";
decomposition("6885") -> 	"2F8E2";
decomposition("688E") -> 	"2F8E4";
decomposition("68A8") -> 	"F9E2";
decomposition("6914") -> 	"2F8E6";
decomposition("6942") -> 	"2F8E8";
decomposition("69A3") -> 	"2F8E9";
decomposition("69EA") -> 	"2F8EA";
decomposition("6A02") -> 	"F914";
decomposition("6A13") -> 	"F94C";
decomposition("6AA8") -> 	"2F8EB";
decomposition("6AD3") -> 	"F931";
decomposition("6ADB") -> 	"2F8ED";
decomposition("6B04") -> 	"F91D";
decomposition("6B21") -> 	"2F8EF";
decomposition("6B54") -> 	"2F8F1";
decomposition("6B72") -> 	"2F8F3";
decomposition("6B77") -> 	"F98C";
decomposition("6B79") -> 	"FA95";
decomposition("6B9F") -> 	"2F8F4";
decomposition("6BAE") -> 	"F9A5";
decomposition("6BBA") -> 	"2F8F5";
decomposition("6BBB") -> 	"2F8F6";
decomposition("6C4E") -> 	"2F8FA";
decomposition("6C67") -> 	"2F8FE";
decomposition("6C88") -> 	"F972";
decomposition("6CBF") -> 	"2F8FC";
decomposition("6CCC") -> 	"F968";
decomposition("6CCD") -> 	"2F8FD";
decomposition("6CE5") -> 	"F9E3";
decomposition("6D16") -> 	"2F8FF";
decomposition("6D1B") -> 	"F915";
decomposition("6D1E") -> 	"FA05";
decomposition("6D34") -> 	"2F907";
decomposition("6D3E") -> 	"2F900";
decomposition("6D41") -> 	"2F902";
decomposition("6D69") -> 	"2F903";
decomposition("6D6A") -> 	"F92A";
decomposition("6D77") -> 	"2F901";
decomposition("6D78") -> 	"2F904";
decomposition("6D85") -> 	"2F905";
decomposition("6DCB") -> 	"F9F5";
decomposition("6DDA") -> 	"F94D";
decomposition("6DEA") -> 	"F9D6";
decomposition("6DF9") -> 	"2F90E";
decomposition("6E1A") -> 	"FA46";
decomposition("6E2F") -> 	"2F908";
decomposition("6E6E") -> 	"2F909";
decomposition("6E9C") -> 	"F9CB";
decomposition("6EBA") -> 	"F9EC";
decomposition("6EC7") -> 	"2F90C";
decomposition("6ECB") -> 	"2F90B";
decomposition("6ED1") -> 	"F904";
decomposition("6EDB") -> 	"FA98";
decomposition("6F0F") -> 	"F94E";
decomposition("6F22") -> 	"FA47";
decomposition("6F23") -> 	"F992";
decomposition("6F6E") -> 	"2F90F";
decomposition("6FC6") -> 	"2F912";
decomposition("6FEB") -> 	"F922";
decomposition("6FFE") -> 	"F984";
decomposition("701B") -> 	"2F915";
decomposition("701E") -> 	"2F914";
decomposition("7039") -> 	"2F913";
decomposition("704A") -> 	"2F917";
decomposition("7070") -> 	"2F835";
decomposition("7077") -> 	"2F919";
decomposition("707D") -> 	"2F918";
decomposition("7099") -> 	"F9FB";
decomposition("70AD") -> 	"2F91A";
decomposition("70C8") -> 	"F99F";
decomposition("70D9") -> 	"F916";
decomposition("7145") -> 	"2F91C";
decomposition("7149") -> 	"F993";
decomposition("716E") -> 	"FA48";
decomposition("719C") -> 	"2F91E";
decomposition("71CE") -> 	"F9C0";
decomposition("71D0") -> 	"F9EE";
decomposition("7210") -> 	"F932";
decomposition("721B") -> 	"F91E";
decomposition("7228") -> 	"2F920";
decomposition("722B") -> 	"FA49";
decomposition("7235") -> 	"2F921";
decomposition("7250") -> 	"2F922";
decomposition("7262") -> 	"F946";
decomposition("7280") -> 	"2F924";
decomposition("7295") -> 	"2F925";
decomposition("72AF") -> 	"FA9F";
decomposition("72C0") -> 	"F9FA";
decomposition("72FC") -> 	"F92B";
decomposition("732A") -> 	"FA16";
decomposition("7375") -> 	"F9A7";
decomposition("737A") -> 	"2F928";
decomposition("7387") -> 	"F961";
decomposition("738B") -> 	"2F929";
decomposition("73A5") -> 	"2F92B";
decomposition("73B2") -> 	"F9AD";
decomposition("73DE") -> 	"F917";
decomposition("7406") -> 	"F9E4";
decomposition("7409") -> 	"F9CC";
decomposition("7422") -> 	"FA4A";
decomposition("7447") -> 	"2F92E";
decomposition("745C") -> 	"2F92F";
decomposition("7469") -> 	"F9AE";
decomposition("7471") -> 	"2F930";
decomposition("7485") -> 	"2F931";
decomposition("7489") -> 	"F994";
decomposition("7498") -> 	"F9EF";
decomposition("74CA") -> 	"2F932";
decomposition("7506") -> 	"FAA2";
decomposition("7524") -> 	"2F934";
decomposition("753B") -> 	"FAA3";
decomposition("753E") -> 	"2F936";
decomposition("7559") -> 	"F9CD";
decomposition("7565") -> 	"F976";
decomposition("7570") -> 	"2F938";
decomposition("75E2") -> 	"F9E5";
decomposition("7610") -> 	"2F93A";
decomposition("761D") -> 	"FAA4";
decomposition("761F") -> 	"FAA5";
decomposition("7642") -> 	"F9C1";
decomposition("7669") -> 	"F90E";
decomposition("76CA") -> 	"FA17";
decomposition("76DB") -> 	"FAA7";
decomposition("76E7") -> 	"F933";
decomposition("76F4") -> 	"2F940";
decomposition("7701") -> 	"F96D";
decomposition("771E") -> 	"2F945";
decomposition("771F") -> 	"2F946";
decomposition("7740") -> 	"FAAA";
decomposition("774A") -> 	"2F948";
decomposition("778B") -> 	"2F94A";
decomposition("77A7") -> 	"FA9D";
decomposition("784E") -> 	"2F94E";
decomposition("786B") -> 	"F9CE";
decomposition("788C") -> 	"2F94F";
decomposition("7891") -> 	"FA4B";
decomposition("78CA") -> 	"F947";
decomposition("78CC") -> 	"2F950";
decomposition("78FB") -> 	"F964";
decomposition("792A") -> 	"F985";
decomposition("793C") -> 	"FA18";
decomposition("793E") -> 	"FA4C";
decomposition("7948") -> 	"FA4E";
decomposition("7949") -> 	"FA4D";
decomposition("7950") -> 	"FA4F";
decomposition("7956") -> 	"2F953";
decomposition("795D") -> 	"FA51";
decomposition("795E") -> 	"FA19";
decomposition("7965") -> 	"FA1A";
decomposition("797F") -> 	"F93C";
decomposition("798D") -> 	"FA52";
decomposition("798E") -> 	"FA53";
decomposition("798F") -> 	"2F956";
decomposition("79AE") -> 	"F9B6";
decomposition("79CA") -> 	"F995";
decomposition("79EB") -> 	"2F957";
decomposition("7A1C") -> 	"F956";
decomposition("7A40") -> 	"2F959";
decomposition("7A4A") -> 	"2F95A";
decomposition("7A4F") -> 	"2F95B";
decomposition("7A81") -> 	"FA55";
decomposition("7AB1") -> 	"FAAC";
decomposition("7ACB") -> 	"F9F7";
decomposition("7AEE") -> 	"2F95F";
decomposition("7B20") -> 	"F9F8";
decomposition("7BC0") -> 	"FA56";
decomposition("7BC6") -> 	"2F962";
decomposition("7BC9") -> 	"2F963";
decomposition("7C3E") -> 	"F9A6";
decomposition("7C60") -> 	"F944";
decomposition("7C7B") -> 	"FAAE";
decomposition("7C92") -> 	"F9F9";
decomposition("7CBE") -> 	"FA1D";
decomposition("7CD2") -> 	"2F966";
decomposition("7CD6") -> 	"FA03";
decomposition("7CE3") -> 	"2F969";
decomposition("7CE7") -> 	"F97B";
decomposition("7CE8") -> 	"2F968";
decomposition("7D00") -> 	"2F96A";
decomposition("7D10") -> 	"F9CF";
decomposition("7D22") -> 	"F96A";
decomposition("7D2F") -> 	"F94F";
decomposition("7D5B") -> 	"FAAF";
decomposition("7D63") -> 	"2F96C";
decomposition("7DA0") -> 	"F93D";
decomposition("7DBE") -> 	"F957";
decomposition("7DC7") -> 	"2F96E";
decomposition("7DF4") -> 	"F996";
decomposition("7E02") -> 	"2F96F";
decomposition("7E09") -> 	"FA58";
decomposition("7E37") -> 	"F950";
decomposition("7E41") -> 	"FA59";
decomposition("7E45") -> 	"2F970";
decomposition("7F3E") -> 	"FAB1";
decomposition("7F72") -> 	"FA5A";
decomposition("7F79") -> 	"F9E6";
decomposition("7F7A") -> 	"2F976";
decomposition("7F85") -> 	"F90F";
decomposition("7F95") -> 	"2F978";
decomposition("7F9A") -> 	"F9AF";
decomposition("7FBD") -> 	"FA1E";
decomposition("7FFA") -> 	"2F979";
decomposition("8001") -> 	"F934";
decomposition("8005") -> 	"2F97A";
decomposition("8046") -> 	"F9B0";
decomposition("8060") -> 	"2F97D";
decomposition("806F") -> 	"F997";
decomposition("8070") -> 	"2F97F";
decomposition("807E") -> 	"F945";
decomposition("808B") -> 	"F953";
decomposition("80AD") -> 	"2F8D6";
decomposition("80B2") -> 	"2F982";
decomposition("8103") -> 	"2F983";
decomposition("813E") -> 	"2F985";
decomposition("81D8") -> 	"F926";
decomposition("81E8") -> 	"F9F6";
decomposition("81ED") -> 	"FA5C";
decomposition("8201") -> 	"2F893";
decomposition("8204") -> 	"2F98C";
decomposition("8218") -> 	"FA6D";
decomposition("826F") -> 	"F97C";
decomposition("8279") -> 	"FA5D";
decomposition("828B") -> 	"2F990";
decomposition("8291") -> 	"2F98F";
decomposition("829D") -> 	"2F991";
decomposition("82B1") -> 	"2F993";
decomposition("82B3") -> 	"2F994";
decomposition("82BD") -> 	"2F995";
decomposition("82E5") -> 	"2F998";
decomposition("82E6") -> 	"2F996";
decomposition("831D") -> 	"2F999";
decomposition("8323") -> 	"2F99C";
decomposition("8336") -> 	"F9FE";
decomposition("8352") -> 	"FAB3";
decomposition("8353") -> 	"2F9A0";
decomposition("8363") -> 	"2F99A";
decomposition("83AD") -> 	"2F99B";
decomposition("83BD") -> 	"2F99D";
decomposition("83C9") -> 	"F93E";
decomposition("83CA") -> 	"2F9A1";
decomposition("83CC") -> 	"2F9A2";
decomposition("83DC") -> 	"2F9A3";
decomposition("83E7") -> 	"2F99E";
decomposition("83EF") -> 	"FAB4";
decomposition("83F1") -> 	"F958";
decomposition("843D") -> 	"F918";
decomposition("8449") -> 	"F96E";
decomposition("8457") -> 	"2F99F";
decomposition("84EE") -> 	"F999";
decomposition("84F1") -> 	"2F9A8";
decomposition("84F3") -> 	"2F9A9";
decomposition("84FC") -> 	"F9C2";
decomposition("8516") -> 	"2F9AA";
decomposition("8564") -> 	"2F9AC";
decomposition("85CD") -> 	"F923";
decomposition("85FA") -> 	"F9F0";
decomposition("8606") -> 	"F935";
decomposition("8612") -> 	"FA20";
decomposition("862D") -> 	"F91F";
decomposition("863F") -> 	"F910";
decomposition("8650") -> 	"2F9B3";
decomposition("865C") -> 	"2F9B4";
decomposition("8667") -> 	"2F9B5";
decomposition("8669") -> 	"2F9B6";
decomposition("8688") -> 	"2F9B8";
decomposition("86A9") -> 	"2F9B7";
decomposition("86E2") -> 	"2F9BA";
decomposition("870E") -> 	"2F9B9";
decomposition("8728") -> 	"2F9BC";
decomposition("876B") -> 	"2F9BD";
decomposition("8779") -> 	"2F9BB";
decomposition("8786") -> 	"2F9BE";
decomposition("87BA") -> 	"F911";
decomposition("87E1") -> 	"2F9C0";
decomposition("8801") -> 	"2F9C1";
decomposition("881F") -> 	"F927";
decomposition("884C") -> 	"FA08";
decomposition("8860") -> 	"2F9C3";
decomposition("8863") -> 	"2F9C4";
decomposition("88C2") -> 	"F9A0";
decomposition("88CF") -> 	"F9E7";
decomposition("88D7") -> 	"2F9C6";
decomposition("88DE") -> 	"2F9C7";
decomposition("88E1") -> 	"F9E8";
decomposition("88F8") -> 	"F912";
decomposition("88FA") -> 	"2F9C9";
decomposition("8910") -> 	"FA60";
decomposition("8941") -> 	"FAB6";
decomposition("8964") -> 	"F924";
decomposition("8986") -> 	"FAB7";
decomposition("898B") -> 	"FA0A";
decomposition("8996") -> 	"FA61";
decomposition("8AA0") -> 	"2F9CF";
decomposition("8AAA") -> 	"F96F";
decomposition("8ABF") -> 	"FAB9";
decomposition("8ACB") -> 	"FABB";
decomposition("8AD2") -> 	"F97D";
decomposition("8AD6") -> 	"F941";
decomposition("8AED") -> 	"2F9D0";
decomposition("8AF8") -> 	"FA22";
decomposition("8AFE") -> 	"F95D";
decomposition("8B01") -> 	"FA62";
decomposition("8B39") -> 	"FA63";
decomposition("8B58") -> 	"F9FC";
decomposition("8B80") -> 	"F95A";
decomposition("8B8A") -> 	"2F9D1";
decomposition("8C48") -> 	"F900";
decomposition("8C55") -> 	"2F9D2";
decomposition("8CAB") -> 	"2F9D4";
decomposition("8CC1") -> 	"2F9D5";
decomposition("8CC2") -> 	"F948";
decomposition("8CC8") -> 	"F903";
decomposition("8CD3") -> 	"FA64";
decomposition("8D08") -> 	"FA65";
decomposition("8D1B") -> 	"2F9D6";
decomposition("8D77") -> 	"2F9D7";
decomposition("8DBC") -> 	"2F9DB";
decomposition("8DCB") -> 	"2F9DA";
decomposition("8DEF") -> 	"F937";
decomposition("8DF0") -> 	"2F9DC";
decomposition("8ECA") -> 	"F902";
decomposition("8ED4") -> 	"2F9DE";
decomposition("8F26") -> 	"F998";
decomposition("8F2A") -> 	"F9D7";
decomposition("8F38") -> 	"2F9DF";
decomposition("8F3B") -> 	"FA07";
decomposition("8F62") -> 	"F98D";
decomposition("8F9E") -> 	"2F98D";
decomposition("8FB0") -> 	"F971";
decomposition("8FB6") -> 	"FA66";
decomposition("9023") -> 	"F99A";
decomposition("9038") -> 	"FA25";
decomposition("9072") -> 	"FAC3";
decomposition("907C") -> 	"F9C3";
decomposition("908F") -> 	"F913";
decomposition("9094") -> 	"2F9E2";
decomposition("90CE") -> 	"F92C";
decomposition("90DE") -> 	"FA2E";
decomposition("90F1") -> 	"2F9E3";
decomposition("90FD") -> 	"FA26";
decomposition("9111") -> 	"2F9E4";
decomposition("911B") -> 	"2F9E6";
decomposition("916A") -> 	"F919";
decomposition("9199") -> 	"FAC4";
decomposition("91B4") -> 	"F9B7";
decomposition("91CC") -> 	"F9E9";
decomposition("91CF") -> 	"F97E";
decomposition("91D1") -> 	"F90A";
decomposition("9234") -> 	"F9B1";
decomposition("9238") -> 	"2F9E7";
decomposition("9276") -> 	"FAC5";
decomposition("927C") -> 	"2F9EA";
decomposition("92D7") -> 	"2F9E8";
decomposition("92D8") -> 	"2F9E9";
decomposition("9304") -> 	"F93F";
decomposition("934A") -> 	"F99B";
decomposition("93F9") -> 	"2F9EB";
decomposition("9415") -> 	"2F9EC";
decomposition("958B") -> 	"2F9EE";
decomposition("95AD") -> 	"F986";
decomposition("95B7") -> 	"2F9F0";
decomposition("962E") -> 	"F9C6";
decomposition("964B") -> 	"F951";
decomposition("964D") -> 	"FA09";
decomposition("9675") -> 	"F959";
decomposition("9678") -> 	"F9D3";
decomposition("967C") -> 	"FAC6";
decomposition("9686") -> 	"F9DC";
decomposition("96A3") -> 	"F9F1";
decomposition("96B7") -> 	"FA2F";
decomposition("96B8") -> 	"F9B8";
decomposition("96C3") -> 	"2F9F3";
decomposition("96E2") -> 	"F9EA";
decomposition("96E3") -> 	"FA68";
decomposition("96F6") -> 	"F9B2";
decomposition("96F7") -> 	"F949";
decomposition("9723") -> 	"2F9F5";
decomposition("9732") -> 	"F938";
decomposition("9748") -> 	"F9B3";
decomposition("9756") -> 	"FA1C";
decomposition("97DB") -> 	"FAC9";
decomposition("97E0") -> 	"2F9FA";
decomposition("97FF") -> 	"FA69";
decomposition("980B") -> 	"2F9FE";
decomposition("9818") -> 	"F9B4";
decomposition("9829") -> 	"2FA00";
decomposition("983B") -> 	"FA6A";
decomposition("985E") -> 	"F9D0";
decomposition("98E2") -> 	"2FA02";
decomposition("98EF") -> 	"FA2A";
decomposition("98FC") -> 	"FA2B";
decomposition("9928") -> 	"FA2C";
decomposition("9929") -> 	"2FA04";
decomposition("99A7") -> 	"2FA05";
decomposition("99C2") -> 	"2FA06";
decomposition("99F1") -> 	"F91A";
decomposition("99FE") -> 	"2FA07";
decomposition("9A6A") -> 	"F987";
decomposition("9B12") -> 	"2FA0A";
decomposition("9B6F") -> 	"F939";
decomposition("9C40") -> 	"2FA0B";
decomposition("9C57") -> 	"F9F2";
decomposition("9CFD") -> 	"2FA0C";
decomposition("9D67") -> 	"2FA0F";
decomposition("9DB4") -> 	"FA2D";
decomposition("9DFA") -> 	"F93A";
decomposition("9E1E") -> 	"F920";
decomposition("9E7F") -> 	"F940";
decomposition("9E97") -> 	"F988";
decomposition("9E9F") -> 	"F9F3";
decomposition("9EBB") -> 	"2FA15";
decomposition("9ECE") -> 	"F989";
decomposition("9EF9") -> 	"2FA17";
decomposition("9EFE") -> 	"2FA18";
decomposition("9F05") -> 	"2FA19";
decomposition("9F0F") -> 	"2FA1A";
decomposition("9F16") -> 	"2FA1B";
decomposition("9F3B") -> 	"2FA1C";
decomposition("9F43") -> 	"FAD8";
decomposition("9F8D") -> 	"F9C4";
decomposition("9F8E") -> 	"FAD9";
decomposition("9F9C") -> 	"F907";
decomposition("<circle> 0030") -> 	"24EA";
decomposition("<circle> 1100 1161") -> 	"326E";
decomposition("<circle> 30A2") -> 	"32D0";
decomposition("<circle> 4E00") -> 	"3280";
decomposition("<circle> 512A") -> 	"329D";
decomposition("<circle> 6587") -> 	"3246";
decomposition("<circle> 706B") -> 	"328B";
decomposition("<circle> 8CA1") -> 	"3296";
decomposition("<circle> 9069") -> 	"329C";
decomposition("<compat> 0020 0301") -> 	"00B4";
decomposition("<compat> 1100") -> 	"3131";
decomposition("<compat> 2032 2032 2032 2032") -> 	"2057";
decomposition("<compat> 3012") -> 	"3036";
decomposition("<compat> 4E00") -> 	"2F00";
decomposition("<compat> 513F") -> 	"2F09";
decomposition("<compat> 6208") -> 	"2F3D";
decomposition("<compat> 706B") -> 	"2F55";
decomposition("<compat> 8001") -> 	"2F7C";
decomposition("<compat> 9091") -> 	"2FA2";
decomposition("<final> 0622") -> 	"FE82";
decomposition("<font> 002B") -> 	"FB29";
decomposition("<font> 0030") -> 	"1D7CE";
decomposition("<font> 0041") -> 	"1D400";
decomposition("<font> 0050") -> 	"1D40F";
decomposition("<font> 0061") -> 	"1D41A";
decomposition("<font> 0070") -> 	"1D429";
decomposition("<font> 0127") -> 	"210F";
decomposition("<font> 0131") -> 	"1D6A4";
decomposition("<font> 0237") -> 	"1D6A5";
decomposition("<font> 0391") -> 	"1D6A8";
decomposition("<font> 03A0") -> 	"1D6B7";
decomposition("<font> 03B1") -> 	"1D6C2";
decomposition("<font> 03C0") -> 	"1D6D1";
decomposition("<font> 03D1") -> 	"1D6DD";
decomposition("<font> 03F0") -> 	"1D6DE";
decomposition("<font> 05D0") -> 	"FB21";
decomposition("<font> 05E2") -> 	"FB20";
decomposition("<font> 0627") -> 	"1EE00";
decomposition("<font> 0630") -> 	"1EE18";
decomposition("<font> 0641") -> 	"1EE10";
decomposition("<font> 066E") -> 	"1EE1C";
decomposition("<font> 06A1") -> 	"1EE1E";
decomposition("<font> 06BA") -> 	"1EE1D";
decomposition("<font> 2202") -> 	"1D6DB";
decomposition("<font> 2211") -> 	"2140";
decomposition("<fraction> 0030 2044 0033") -> 	"2189";
decomposition("<initial> 0626 062C") -> 	"FC97";
decomposition("<isolated> 0020 064B") -> 	"FE70";
decomposition("<medial> 0626 0645") -> 	"FCDF";
decomposition("<narrow> 2190") -> 	"FFE9";
decomposition("<narrow> 3001") -> 	"FF64";
decomposition("<noBreak> 0020") -> 	"00A0";
decomposition("<small> 0021") -> 	"FE57";
decomposition("<small> 2014") -> 	"FE58";
decomposition("<small> 3001") -> 	"FE51";
decomposition("<square> 0041 0055") -> 	"3373";
decomposition("<square> 307B 304B") -> 	"1F200";
decomposition("<square> 4E00") -> 	"1F229";
decomposition("<square> 518D") -> 	"1F21E";
decomposition("<square> 624B") -> 	"1F210";
decomposition("<square> 7121") -> 	"1F21A";
decomposition("<square> 89E3") -> 	"1F216";
decomposition("<square> 904A") -> 	"1F22B";
decomposition("<sub> 0028") -> 	"208D";
decomposition("<sub> 0029") -> 	"208E";
decomposition("<sub> 002B") -> 	"208A";
decomposition("<sub> 0030") -> 	"2080";
decomposition("<sub> 0031") -> 	"2081";
decomposition("<sub> 0032") -> 	"2082";
decomposition("<sub> 0033") -> 	"2083";
decomposition("<sub> 0034") -> 	"2084";
decomposition("<sub> 0035") -> 	"2085";
decomposition("<sub> 0036") -> 	"2086";
decomposition("<sub> 0037") -> 	"2087";
decomposition("<sub> 0038") -> 	"2088";
decomposition("<sub> 0039") -> 	"2089";
decomposition("<sub> 003D") -> 	"208C";
decomposition("<sub> 0061") -> 	"2090";
decomposition("<sub> 0065") -> 	"2091";
decomposition("<sub> 0068") -> 	"2095";
decomposition("<sub> 0069") -> 	"1D62";
decomposition("<sub> 006A") -> 	"2C7C";
decomposition("<sub> 006B") -> 	"2096";
decomposition("<sub> 006C") -> 	"2097";
decomposition("<sub> 006D") -> 	"2098";
decomposition("<sub> 006E") -> 	"2099";
decomposition("<sub> 006F") -> 	"2092";
decomposition("<sub> 0070") -> 	"209A";
decomposition("<sub> 0072") -> 	"1D63";
decomposition("<sub> 0073") -> 	"209B";
decomposition("<sub> 0074") -> 	"209C";
decomposition("<sub> 0075") -> 	"1D64";
decomposition("<sub> 0076") -> 	"1D65";
decomposition("<sub> 0078") -> 	"2093";
decomposition("<sub> 0259") -> 	"2094";
decomposition("<sub> 03B2") -> 	"1D66";
decomposition("<sub> 03B3") -> 	"1D67";
decomposition("<sub> 03C1") -> 	"1D68";
decomposition("<sub> 03C6") -> 	"1D69";
decomposition("<sub> 03C7") -> 	"1D6A";
decomposition("<sub> 2212") -> 	"208B";
decomposition("<super> 0028") -> 	"207D";
decomposition("<super> 0126") -> 	"A7F8";
decomposition("<super> 0222") -> 	"1D3D";
decomposition("<super> 03B2") -> 	"1D5D";
decomposition("<super> 043D") -> 	"1D78";
decomposition("<super> 10DC") -> 	"10FC";
decomposition("<super> 1D02") -> 	"1D46";
decomposition("<super> 2212") -> 	"207B";
decomposition("<super> 2D61") -> 	"2D6F";
decomposition("<super> 4E00") -> 	"3192";
decomposition("<super> 56DB") -> 	"3195";
decomposition("<super> 5730") -> 	"319E";
decomposition("<super> 5929") -> 	"319D";
decomposition("<super> 7532") -> 	"3199";
decomposition("<super> A727") -> 	"AB5C";
decomposition("<super> AB37") -> 	"AB5D";
decomposition("<vertical> 0021") -> 	"FE15";
decomposition("<wide> 0020") -> 	"3000";
decomposition("<wide> 0030") -> 	"FF10";
decomposition("<wide> 0040") -> 	"FF20";
decomposition("<wide> 0050") -> 	"FF30";
decomposition("<wide> 0060") -> 	"FF40";
decomposition("<wide> 0070") -> 	"FF50";
decomposition("<wide> 00A2") -> 	"FFE0";
decomposition("<wide> 20A9") -> 	"FFE6";
decomposition("<wide> 2985") -> 	"FF5F";
decomposition("FB49 05C1") -> 	"FB2C";
decomposition("FB49 05C2") -> 	"FB2D";
decomposition(_) -> false.

lookup1("0000") ->	{"0","",""};
lookup1("0001") ->	{"0","",""};
lookup1("0002") ->	{"0","",""};
lookup1("0003") ->	{"0","",""};
lookup1("0004") ->	{"0","",""};
lookup1("0005") ->	{"0","",""};
lookup1("0006") ->	{"0","",""};
lookup1("0007") ->	{"0","",""};
lookup1("0008") ->	{"0","",""};
lookup1("0009") ->	{"0","",""};
lookup1("000A") ->	{"0","",""};
lookup1("000B") ->	{"0","",""};
lookup1("000C") ->	{"0","",""};
lookup1("000D") ->	{"0","",""};
lookup1("000E") ->	{"0","",""};
lookup1("000F") ->	{"0","",""};
lookup1("0010") ->	{"0","",""};
lookup1("0011") ->	{"0","",""};
lookup1("0012") ->	{"0","",""};
lookup1("0013") ->	{"0","",""};
lookup1("0014") ->	{"0","",""};
lookup1("0015") ->	{"0","",""};
lookup1("0016") ->	{"0","",""};
lookup1("0017") ->	{"0","",""};
lookup1("0018") ->	{"0","",""};
lookup1("0019") ->	{"0","",""};
lookup1("001A") ->	{"0","",""};
lookup1("001B") ->	{"0","",""};
lookup1("001C") ->	{"0","",""};
lookup1("001D") ->	{"0","",""};
lookup1("001E") ->	{"0","",""};
lookup1("001F") ->	{"0","",""};
lookup1("0020") ->	{"0","",""};
lookup1("0021") ->	{"0","",""};
lookup1("0022") ->	{"0","",""};
lookup1("0023") ->	{"0","",""};
lookup1("0024") ->	{"0","",""};
lookup1("0025") ->	{"0","",""};
lookup1("0026") ->	{"0","",""};
lookup1("0027") ->	{"0","",""};
lookup1("0028") ->	{"0","",""};
lookup1("0029") ->	{"0","",""};
lookup1("002A") ->	{"0","",""};
lookup1("002B") ->	{"0","",""};
lookup1("002C") ->	{"0","",""};
lookup1("002D") ->	{"0","",""};
lookup1("002E") ->	{"0","",""};
lookup1("002F") ->	{"0","",""};
lookup1("0030") ->	{"0","",""};
lookup1("0031") ->	{"0","",""};
lookup1("0032") ->	{"0","",""};
lookup1("0033") ->	{"0","",""};
lookup1("0034") ->	{"0","",""};
lookup1("0035") ->	{"0","",""};
lookup1("0036") ->	{"0","",""};
lookup1("0037") ->	{"0","",""};
lookup1("0038") ->	{"0","",""};
lookup1("0039") ->	{"0","",""};
lookup1("003A") ->	{"0","",""};
lookup1("003B") ->	{"0","",""};
lookup1("003C") ->	{"0","",""};
lookup1("003D") ->	{"0","",""};
lookup1("003E") ->	{"0","",""};
lookup1("003F") ->	{"0","",""};
lookup1("0040") ->	{"0","",""};
lookup1("0041") ->	{"0","","0061"};
lookup1("0042") ->	{"0","","0062"};
lookup1("0043") ->	{"0","","0063"};
lookup1("0044") ->	{"0","","0064"};
lookup1("0045") ->	{"0","","0065"};
lookup1("0046") ->	{"0","","0066"};
lookup1("0047") ->	{"0","","0067"};
lookup1("0048") ->	{"0","","0068"};
lookup1("0049") ->	{"0","","0069"};
lookup1("004A") ->	{"0","","006A"};
lookup1("004B") ->	{"0","","006B"};
lookup1("004C") ->	{"0","","006C"};
lookup1("004D") ->	{"0","","006D"};
lookup1("004E") ->	{"0","","006E"};
lookup1("004F") ->	{"0","","006F"};
lookup1("0050") ->	{"0","","0070"};
lookup1("0051") ->	{"0","","0071"};
lookup1("0052") ->	{"0","","0072"};
lookup1("0053") ->	{"0","","0073"};
lookup1("0054") ->	{"0","","0074"};
lookup1("0055") ->	{"0","","0075"};
lookup1("0056") ->	{"0","","0076"};
lookup1("0057") ->	{"0","","0077"};
lookup1("0058") ->	{"0","","0078"};
lookup1("0059") ->	{"0","","0079"};
lookup1("005A") ->	{"0","","007A"};
lookup1("005B") ->	{"0","",""};
lookup1("005C") ->	{"0","",""};
lookup1("005D") ->	{"0","",""};
lookup1("005E") ->	{"0","",""};
lookup1("005F") ->	{"0","",""};
lookup1("0060") ->	{"0","",""};
lookup1("0061") ->	{"0","",""};
lookup1("0062") ->	{"0","",""};
lookup1("0063") ->	{"0","",""};
lookup1("0064") ->	{"0","",""};
lookup1("0065") ->	{"0","",""};
lookup1("0066") ->	{"0","",""};
lookup1("0067") ->	{"0","",""};
lookup1("0068") ->	{"0","",""};
lookup1("0069") ->	{"0","",""};
lookup1("006A") ->	{"0","",""};
lookup1("006B") ->	{"0","",""};
lookup1("006C") ->	{"0","",""};
lookup1("006D") ->	{"0","",""};
lookup1("006E") ->	{"0","",""};
lookup1("006F") ->	{"0","",""};
lookup1("0070") ->	{"0","",""};
lookup1("0071") ->	{"0","",""};
lookup1("0072") ->	{"0","",""};
lookup1("0073") ->	{"0","",""};
lookup1("0074") ->	{"0","",""};
lookup1("0075") ->	{"0","",""};
lookup1("0076") ->	{"0","",""};
lookup1("0077") ->	{"0","",""};
lookup1("0078") ->	{"0","",""};
lookup1("0079") ->	{"0","",""};
lookup1("007A") ->	{"0","",""};
lookup1("007B") ->	{"0","",""};
lookup1("007C") ->	{"0","",""};
lookup1("007D") ->	{"0","",""};
lookup1("007E") ->	{"0","",""};
lookup1("007F") ->	{"0","",""};
lookup1("0080") ->	{"0","",""};
lookup1("0081") ->	{"0","",""};
lookup1("0082") ->	{"0","",""};
lookup1("0083") ->	{"0","",""};
lookup1("0084") ->	{"0","",""};
lookup1("0085") ->	{"0","",""};
lookup1("0086") ->	{"0","",""};
lookup1("0087") ->	{"0","",""};
lookup1("0088") ->	{"0","",""};
lookup1("0089") ->	{"0","",""};
lookup1("008A") ->	{"0","",""};
lookup1("008B") ->	{"0","",""};
lookup1("008C") ->	{"0","",""};
lookup1("008D") ->	{"0","",""};
lookup1("008E") ->	{"0","",""};
lookup1("008F") ->	{"0","",""};
lookup1("0090") ->	{"0","",""};
lookup1("0091") ->	{"0","",""};
lookup1("0092") ->	{"0","",""};
lookup1("0093") ->	{"0","",""};
lookup1("0094") ->	{"0","",""};
lookup1("0095") ->	{"0","",""};
lookup1("0096") ->	{"0","",""};
lookup1("0097") ->	{"0","",""};
lookup1("0098") ->	{"0","",""};
lookup1("0099") ->	{"0","",""};
lookup1("009A") ->	{"0","",""};
lookup1("009B") ->	{"0","",""};
lookup1("009C") ->	{"0","",""};
lookup1("009D") ->	{"0","",""};
lookup1("009E") ->	{"0","",""};
lookup1("009F") ->	{"0","",""};
lookup1("00A0") ->	{"0","<noBreak> 0020",""};
lookup1("00A1") ->	{"0","",""};
lookup1("00A2") ->	{"0","",""};
lookup1("00A3") ->	{"0","",""};
lookup1("00A4") ->	{"0","",""};
lookup1("00A5") ->	{"0","",""};
lookup1("00A6") ->	{"0","",""};
lookup1("00A7") ->	{"0","",""};
lookup1("00A8") ->	{"0","<compat> 0020 0308",""};
lookup1("00A9") ->	{"0","",""};
lookup1("00AA") ->	{"0","<super> 0061",""};
lookup1("00AB") ->	{"0","",""};
lookup1("00AC") ->	{"0","",""};
lookup1("00AD") ->	{"0","",""};
lookup1("00AE") ->	{"0","",""};
lookup1("00AF") ->	{"0","<compat> 0020 0304",""};
lookup1("00B0") ->	{"0","",""};
lookup1("00B1") ->	{"0","",""};
lookup1("00B2") ->	{"0","<super> 0032",""};
lookup1("00B3") ->	{"0","<super> 0033",""};
lookup1("00B4") ->	{"0","<compat> 0020 0301",""};
lookup1("00B5") ->	{"0","<compat> 03BC",""};
lookup1("00B6") ->	{"0","",""};
lookup1("00B7") ->	{"0","",""};
lookup1("00B8") ->	{"0","<compat> 0020 0327",""};
lookup1("00B9") ->	{"0","<super> 0031",""};
lookup1("00BA") ->	{"0","<super> 006F",""};
lookup1("00BB") ->	{"0","",""};
lookup1("00BC") ->	{"0","<fraction> 0031 2044 0034",""};
lookup1("00BD") ->	{"0","<fraction> 0031 2044 0032",""};
lookup1("00BE") ->	{"0","<fraction> 0033 2044 0034",""};
lookup1("00BF") ->	{"0","",""};
lookup1("00C0") ->	{"0","0041 0300","00E0"};
lookup1("00C1") ->	{"0","0041 0301","00E1"};
lookup1("00C2") ->	{"0","0041 0302","00E2"};
lookup1("00C3") ->	{"0","0041 0303","00E3"};
lookup1("00C4") ->	{"0","0041 0308","00E4"};
lookup1("00C5") ->	{"0","0041 030A","00E5"};
lookup1("00C6") ->	{"0","","00E6"};
lookup1("00C7") ->	{"0","0043 0327","00E7"};
lookup1("00C8") ->	{"0","0045 0300","00E8"};
lookup1("00C9") ->	{"0","0045 0301","00E9"};
lookup1("00CA") ->	{"0","0045 0302","00EA"};
lookup1("00CB") ->	{"0","0045 0308","00EB"};
lookup1("00CC") ->	{"0","0049 0300","00EC"};
lookup1("00CD") ->	{"0","0049 0301","00ED"};
lookup1("00CE") ->	{"0","0049 0302","00EE"};
lookup1("00CF") ->	{"0","0049 0308","00EF"};
lookup1("00D0") ->	{"0","","00F0"};
lookup1("00D1") ->	{"0","004E 0303","00F1"};
lookup1("00D2") ->	{"0","004F 0300","00F2"};
lookup1("00D3") ->	{"0","004F 0301","00F3"};
lookup1("00D4") ->	{"0","004F 0302","00F4"};
lookup1("00D5") ->	{"0","004F 0303","00F5"};
lookup1("00D6") ->	{"0","004F 0308","00F6"};
lookup1("00D7") ->	{"0","",""};
lookup1("00D8") ->	{"0","","00F8"};
lookup1("00D9") ->	{"0","0055 0300","00F9"};
lookup1("00DA") ->	{"0","0055 0301","00FA"};
lookup1("00DB") ->	{"0","0055 0302","00FB"};
lookup1("00DC") ->	{"0","0055 0308","00FC"};
lookup1("00DD") ->	{"0","0059 0301","00FD"};
lookup1("00DE") ->	{"0","","00FE"};
lookup1("00DF") ->	{"0","",""};
lookup1("00E0") ->	{"0","0061 0300",""};
lookup1("00E1") ->	{"0","0061 0301",""};
lookup1("00E2") ->	{"0","0061 0302",""};
lookup1("00E3") ->	{"0","0061 0303",""};
lookup1("00E4") ->	{"0","0061 0308",""};
lookup1("00E5") ->	{"0","0061 030A",""};
lookup1("00E6") ->	{"0","",""};
lookup1("00E7") ->	{"0","0063 0327",""};
lookup1("00E8") ->	{"0","0065 0300",""};
lookup1("00E9") ->	{"0","0065 0301",""};
lookup1("00EA") ->	{"0","0065 0302",""};
lookup1("00EB") ->	{"0","0065 0308",""};
lookup1("00EC") ->	{"0","0069 0300",""};
lookup1("00ED") ->	{"0","0069 0301",""};
lookup1("00EE") ->	{"0","0069 0302",""};
lookup1("00EF") ->	{"0","0069 0308",""};
lookup1("00F0") ->	{"0","",""};
lookup1("00F1") ->	{"0","006E 0303",""};
lookup1("00F2") ->	{"0","006F 0300",""};
lookup1("00F3") ->	{"0","006F 0301",""};
lookup1("00F4") ->	{"0","006F 0302",""};
lookup1("00F5") ->	{"0","006F 0303",""};
lookup1("00F6") ->	{"0","006F 0308",""};
lookup1("00F7") ->	{"0","",""};
lookup1("00F8") ->	{"0","",""};
lookup1("00F9") ->	{"0","0075 0300",""};
lookup1("00FA") ->	{"0","0075 0301",""};
lookup1("00FB") ->	{"0","0075 0302",""};
lookup1("00FC") ->	{"0","0075 0308",""};
lookup1("00FD") ->	{"0","0079 0301",""};
lookup1("00FE") ->	{"0","",""};
lookup1("00FF") ->	{"0","0079 0308",""};
lookup1("0100") ->	{"0","0041 0304","0101"};
lookup1("0101") ->	{"0","0061 0304",""};
lookup1("0102") ->	{"0","0041 0306","0103"};
lookup1("0103") ->	{"0","0061 0306",""};
lookup1("0104") ->	{"0","0041 0328","0105"};
lookup1("0105") ->	{"0","0061 0328",""};
lookup1("0106") ->	{"0","0043 0301","0107"};
lookup1("0107") ->	{"0","0063 0301",""};
lookup1("0108") ->	{"0","0043 0302","0109"};
lookup1("0109") ->	{"0","0063 0302",""};
lookup1("010A") ->	{"0","0043 0307","010B"};
lookup1("010B") ->	{"0","0063 0307",""};
lookup1("010C") ->	{"0","0043 030C","010D"};
lookup1("010D") ->	{"0","0063 030C",""};
lookup1("010E") ->	{"0","0044 030C","010F"};
lookup1("010F") ->	{"0","0064 030C",""};
lookup1("0110") ->	{"0","","0111"};
lookup1("0111") ->	{"0","",""};
lookup1("0112") ->	{"0","0045 0304","0113"};
lookup1("0113") ->	{"0","0065 0304",""};
lookup1("0114") ->	{"0","0045 0306","0115"};
lookup1("0115") ->	{"0","0065 0306",""};
lookup1("0116") ->	{"0","0045 0307","0117"};
lookup1("0117") ->	{"0","0065 0307",""};
lookup1("0118") ->	{"0","0045 0328","0119"};
lookup1("0119") ->	{"0","0065 0328",""};
lookup1("011A") ->	{"0","0045 030C","011B"};
lookup1("011B") ->	{"0","0065 030C",""};
lookup1("011C") ->	{"0","0047 0302","011D"};
lookup1("011D") ->	{"0","0067 0302",""};
lookup1("011E") ->	{"0","0047 0306","011F"};
lookup1("011F") ->	{"0","0067 0306",""};
lookup1("0120") ->	{"0","0047 0307","0121"};
lookup1("0121") ->	{"0","0067 0307",""};
lookup1("0122") ->	{"0","0047 0327","0123"};
lookup1("0123") ->	{"0","0067 0327",""};
lookup1("0124") ->	{"0","0048 0302","0125"};
lookup1("0125") ->	{"0","0068 0302",""};
lookup1("0126") ->	{"0","","0127"};
lookup1("0127") ->	{"0","",""};
lookup1("0128") ->	{"0","0049 0303","0129"};
lookup1("0129") ->	{"0","0069 0303",""};
lookup1("012A") ->	{"0","0049 0304","012B"};
lookup1("012B") ->	{"0","0069 0304",""};
lookup1("012C") ->	{"0","0049 0306","012D"};
lookup1("012D") ->	{"0","0069 0306",""};
lookup1("012E") ->	{"0","0049 0328","012F"};
lookup1("012F") ->	{"0","0069 0328",""};
lookup1("0130") ->	{"0","0049 0307","0069"};
lookup1("0131") ->	{"0","",""};
lookup1("0132") ->	{"0","<compat> 0049 004A","0133"};
lookup1("0133") ->	{"0","<compat> 0069 006A",""};
lookup1("0134") ->	{"0","004A 0302","0135"};
lookup1("0135") ->	{"0","006A 0302",""};
lookup1("0136") ->	{"0","004B 0327","0137"};
lookup1("0137") ->	{"0","006B 0327",""};
lookup1("0138") ->	{"0","",""};
lookup1("0139") ->	{"0","004C 0301","013A"};
lookup1("013A") ->	{"0","006C 0301",""};
lookup1("013B") ->	{"0","004C 0327","013C"};
lookup1("013C") ->	{"0","006C 0327",""};
lookup1("013D") ->	{"0","004C 030C","013E"};
lookup1("013E") ->	{"0","006C 030C",""};
lookup1("013F") ->	{"0","<compat> 004C 00B7","0140"};
lookup1("0140") ->	{"0","<compat> 006C 00B7",""};
lookup1("0141") ->	{"0","","0142"};
lookup1("0142") ->	{"0","",""};
lookup1("0143") ->	{"0","004E 0301","0144"};
lookup1("0144") ->	{"0","006E 0301",""};
lookup1("0145") ->	{"0","004E 0327","0146"};
lookup1("0146") ->	{"0","006E 0327",""};
lookup1("0147") ->	{"0","004E 030C","0148"};
lookup1("0148") ->	{"0","006E 030C",""};
lookup1("0149") ->	{"0","<compat> 02BC 006E",""};
lookup1("014A") ->	{"0","","014B"};
lookup1("014B") ->	{"0","",""};
lookup1("014C") ->	{"0","004F 0304","014D"};
lookup1("014D") ->	{"0","006F 0304",""};
lookup1("014E") ->	{"0","004F 0306","014F"};
lookup1("014F") ->	{"0","006F 0306",""};
lookup1("0150") ->	{"0","004F 030B","0151"};
lookup1("0151") ->	{"0","006F 030B",""};
lookup1("0152") ->	{"0","","0153"};
lookup1("0153") ->	{"0","",""};
lookup1("0154") ->	{"0","0052 0301","0155"};
lookup1("0155") ->	{"0","0072 0301",""};
lookup1("0156") ->	{"0","0052 0327","0157"};
lookup1("0157") ->	{"0","0072 0327",""};
lookup1("0158") ->	{"0","0052 030C","0159"};
lookup1("0159") ->	{"0","0072 030C",""};
lookup1("015A") ->	{"0","0053 0301","015B"};
lookup1("015B") ->	{"0","0073 0301",""};
lookup1("015C") ->	{"0","0053 0302","015D"};
lookup1("015D") ->	{"0","0073 0302",""};
lookup1("015E") ->	{"0","0053 0327","015F"};
lookup1("015F") ->	{"0","0073 0327",""};
lookup1("0160") ->	{"0","0053 030C","0161"};
lookup1("0161") ->	{"0","0073 030C",""};
lookup1("0162") ->	{"0","0054 0327","0163"};
lookup1("0163") ->	{"0","0074 0327",""};
lookup1("0164") ->	{"0","0054 030C","0165"};
lookup1("0165") ->	{"0","0074 030C",""};
lookup1("0166") ->	{"0","","0167"};
lookup1("0167") ->	{"0","",""};
lookup1("0168") ->	{"0","0055 0303","0169"};
lookup1("0169") ->	{"0","0075 0303",""};
lookup1("016A") ->	{"0","0055 0304","016B"};
lookup1("016B") ->	{"0","0075 0304",""};
lookup1("016C") ->	{"0","0055 0306","016D"};
lookup1("016D") ->	{"0","0075 0306",""};
lookup1("016E") ->	{"0","0055 030A","016F"};
lookup1("016F") ->	{"0","0075 030A",""};
lookup1("0170") ->	{"0","0055 030B","0171"};
lookup1("0171") ->	{"0","0075 030B",""};
lookup1("0172") ->	{"0","0055 0328","0173"};
lookup1("0173") ->	{"0","0075 0328",""};
lookup1("0174") ->	{"0","0057 0302","0175"};
lookup1("0175") ->	{"0","0077 0302",""};
lookup1("0176") ->	{"0","0059 0302","0177"};
lookup1("0177") ->	{"0","0079 0302",""};
lookup1("0178") ->	{"0","0059 0308","00FF"};
lookup1("0179") ->	{"0","005A 0301","017A"};
lookup1("017A") ->	{"0","007A 0301",""};
lookup1("017B") ->	{"0","005A 0307","017C"};
lookup1("017C") ->	{"0","007A 0307",""};
lookup1("017D") ->	{"0","005A 030C","017E"};
lookup1("017E") ->	{"0","007A 030C",""};
lookup1("017F") ->	{"0","<compat> 0073",""};
lookup1("0180") ->	{"0","",""};
lookup1("0181") ->	{"0","","0253"};
lookup1("0182") ->	{"0","","0183"};
lookup1("0183") ->	{"0","",""};
lookup1("0184") ->	{"0","","0185"};
lookup1("0185") ->	{"0","",""};
lookup1("0186") ->	{"0","","0254"};
lookup1("0187") ->	{"0","","0188"};
lookup1("0188") ->	{"0","",""};
lookup1("0189") ->	{"0","","0256"};
lookup1("018A") ->	{"0","","0257"};
lookup1("018B") ->	{"0","","018C"};
lookup1("018C") ->	{"0","",""};
lookup1("018D") ->	{"0","",""};
lookup1("018E") ->	{"0","","01DD"};
lookup1("018F") ->	{"0","","0259"};
lookup1("0190") ->	{"0","","025B"};
lookup1("0191") ->	{"0","","0192"};
lookup1("0192") ->	{"0","",""};
lookup1("0193") ->	{"0","","0260"};
lookup1("0194") ->	{"0","","0263"};
lookup1("0195") ->	{"0","",""};
lookup1("0196") ->	{"0","","0269"};
lookup1("0197") ->	{"0","","0268"};
lookup1("0198") ->	{"0","","0199"};
lookup1("0199") ->	{"0","",""};
lookup1("019A") ->	{"0","",""};
lookup1("019B") ->	{"0","",""};
lookup1("019C") ->	{"0","","026F"};
lookup1("019D") ->	{"0","","0272"};
lookup1("019E") ->	{"0","",""};
lookup1("019F") ->	{"0","","0275"};
lookup1("01A0") ->	{"0","004F 031B","01A1"};
lookup1("01A1") ->	{"0","006F 031B",""};
lookup1("01A2") ->	{"0","","01A3"};
lookup1("01A3") ->	{"0","",""};
lookup1("01A4") ->	{"0","","01A5"};
lookup1("01A5") ->	{"0","",""};
lookup1("01A6") ->	{"0","","0280"};
lookup1("01A7") ->	{"0","","01A8"};
lookup1("01A8") ->	{"0","",""};
lookup1("01A9") ->	{"0","","0283"};
lookup1("01AA") ->	{"0","",""};
lookup1("01AB") ->	{"0","",""};
lookup1("01AC") ->	{"0","","01AD"};
lookup1("01AD") ->	{"0","",""};
lookup1("01AE") ->	{"0","","0288"};
lookup1("01AF") ->	{"0","0055 031B","01B0"};
lookup1("01B0") ->	{"0","0075 031B",""};
lookup1("01B1") ->	{"0","","028A"};
lookup1("01B2") ->	{"0","","028B"};
lookup1("01B3") ->	{"0","","01B4"};
lookup1("01B4") ->	{"0","",""};
lookup1("01B5") ->	{"0","","01B6"};
lookup1("01B6") ->	{"0","",""};
lookup1("01B7") ->	{"0","","0292"};
lookup1("01B8") ->	{"0","","01B9"};
lookup1("01B9") ->	{"0","",""};
lookup1("01BA") ->	{"0","",""};
lookup1("01BB") ->	{"0","",""};
lookup1("01BC") ->	{"0","","01BD"};
lookup1("01BD") ->	{"0","",""};
lookup1("01BE") ->	{"0","",""};
lookup1("01BF") ->	{"0","",""};
lookup1("01C0") ->	{"0","",""};
lookup1("01C1") ->	{"0","",""};
lookup1("01C2") ->	{"0","",""};
lookup1("01C3") ->	{"0","",""};
lookup1("01C4") ->	{"0","<compat> 0044 017D","01C6"};
lookup1("01C5") ->	{"0","<compat> 0044 017E","01C6"};
lookup1("01C6") ->	{"0","<compat> 0064 017E",""};
lookup1("01C7") ->	{"0","<compat> 004C 004A","01C9"};
lookup1("01C8") ->	{"0","<compat> 004C 006A","01C9"};
lookup1("01C9") ->	{"0","<compat> 006C 006A",""};
lookup1("01CA") ->	{"0","<compat> 004E 004A","01CC"};
lookup1("01CB") ->	{"0","<compat> 004E 006A","01CC"};
lookup1("01CC") ->	{"0","<compat> 006E 006A",""};
lookup1("01CD") ->	{"0","0041 030C","01CE"};
lookup1("01CE") ->	{"0","0061 030C",""};
lookup1("01CF") ->	{"0","0049 030C","01D0"};
lookup1("01D0") ->	{"0","0069 030C",""};
lookup1("01D1") ->	{"0","004F 030C","01D2"};
lookup1("01D2") ->	{"0","006F 030C",""};
lookup1("01D3") ->	{"0","0055 030C","01D4"};
lookup1("01D4") ->	{"0","0075 030C",""};
lookup1("01D5") ->	{"0","00DC 0304","01D6"};
lookup1("01D6") ->	{"0","00FC 0304",""};
lookup1("01D7") ->	{"0","00DC 0301","01D8"};
lookup1("01D8") ->	{"0","00FC 0301",""};
lookup1("01D9") ->	{"0","00DC 030C","01DA"};
lookup1("01DA") ->	{"0","00FC 030C",""};
lookup1("01DB") ->	{"0","00DC 0300","01DC"};
lookup1("01DC") ->	{"0","00FC 0300",""};
lookup1("01DD") ->	{"0","",""};
lookup1("01DE") ->	{"0","00C4 0304","01DF"};
lookup1("01DF") ->	{"0","00E4 0304",""};
lookup1("01E0") ->	{"0","0226 0304","01E1"};
lookup1("01E1") ->	{"0","0227 0304",""};
lookup1("01E2") ->	{"0","00C6 0304","01E3"};
lookup1("01E3") ->	{"0","00E6 0304",""};
lookup1("01E4") ->	{"0","","01E5"};
lookup1("01E5") ->	{"0","",""};
lookup1("01E6") ->	{"0","0047 030C","01E7"};
lookup1("01E7") ->	{"0","0067 030C",""};
lookup1("01E8") ->	{"0","004B 030C","01E9"};
lookup1("01E9") ->	{"0","006B 030C",""};
lookup1("01EA") ->	{"0","004F 0328","01EB"};
lookup1("01EB") ->	{"0","006F 0328",""};
lookup1("01EC") ->	{"0","01EA 0304","01ED"};
lookup1("01ED") ->	{"0","01EB 0304",""};
lookup1("01EE") ->	{"0","01B7 030C","01EF"};
lookup1("01EF") ->	{"0","0292 030C",""};
lookup1("01F0") ->	{"0","006A 030C",""};
lookup1("01F1") ->	{"0","<compat> 0044 005A","01F3"};
lookup1("01F2") ->	{"0","<compat> 0044 007A","01F3"};
lookup1("01F3") ->	{"0","<compat> 0064 007A",""};
lookup1("01F4") ->	{"0","0047 0301","01F5"};
lookup1("01F5") ->	{"0","0067 0301",""};
lookup1("01F6") ->	{"0","","0195"};
lookup1("01F7") ->	{"0","","01BF"};
lookup1("01F8") ->	{"0","004E 0300","01F9"};
lookup1("01F9") ->	{"0","006E 0300",""};
lookup1("01FA") ->	{"0","00C5 0301","01FB"};
lookup1("01FB") ->	{"0","00E5 0301",""};
lookup1("01FC") ->	{"0","00C6 0301","01FD"};
lookup1("01FD") ->	{"0","00E6 0301",""};
lookup1("01FE") ->	{"0","00D8 0301","01FF"};
lookup1("01FF") ->	{"0","00F8 0301",""};
lookup1("0200") ->	{"0","0041 030F","0201"};
lookup1("0201") ->	{"0","0061 030F",""};
lookup1("0202") ->	{"0","0041 0311","0203"};
lookup1("0203") ->	{"0","0061 0311",""};
lookup1("0204") ->	{"0","0045 030F","0205"};
lookup1("0205") ->	{"0","0065 030F",""};
lookup1("0206") ->	{"0","0045 0311","0207"};
lookup1("0207") ->	{"0","0065 0311",""};
lookup1("0208") ->	{"0","0049 030F","0209"};
lookup1("0209") ->	{"0","0069 030F",""};
lookup1("020A") ->	{"0","0049 0311","020B"};
lookup1("020B") ->	{"0","0069 0311",""};
lookup1("020C") ->	{"0","004F 030F","020D"};
lookup1("020D") ->	{"0","006F 030F",""};
lookup1("020E") ->	{"0","004F 0311","020F"};
lookup1("020F") ->	{"0","006F 0311",""};
lookup1("0210") ->	{"0","0052 030F","0211"};
lookup1("0211") ->	{"0","0072 030F",""};
lookup1("0212") ->	{"0","0052 0311","0213"};
lookup1("0213") ->	{"0","0072 0311",""};
lookup1("0214") ->	{"0","0055 030F","0215"};
lookup1("0215") ->	{"0","0075 030F",""};
lookup1("0216") ->	{"0","0055 0311","0217"};
lookup1("0217") ->	{"0","0075 0311",""};
lookup1("0218") ->	{"0","0053 0326","0219"};
lookup1("0219") ->	{"0","0073 0326",""};
lookup1("021A") ->	{"0","0054 0326","021B"};
lookup1("021B") ->	{"0","0074 0326",""};
lookup1("021C") ->	{"0","","021D"};
lookup1("021D") ->	{"0","",""};
lookup1("021E") ->	{"0","0048 030C","021F"};
lookup1("021F") ->	{"0","0068 030C",""};
lookup1("0220") ->	{"0","","019E"};
lookup1("0221") ->	{"0","",""};
lookup1("0222") ->	{"0","","0223"};
lookup1("0223") ->	{"0","",""};
lookup1("0224") ->	{"0","","0225"};
lookup1("0225") ->	{"0","",""};
lookup1("0226") ->	{"0","0041 0307","0227"};
lookup1("0227") ->	{"0","0061 0307",""};
lookup1("0228") ->	{"0","0045 0327","0229"};
lookup1("0229") ->	{"0","0065 0327",""};
lookup1("022A") ->	{"0","00D6 0304","022B"};
lookup1("022B") ->	{"0","00F6 0304",""};
lookup1("022C") ->	{"0","00D5 0304","022D"};
lookup1("022D") ->	{"0","00F5 0304",""};
lookup1("022E") ->	{"0","004F 0307","022F"};
lookup1("022F") ->	{"0","006F 0307",""};
lookup1("0230") ->	{"0","022E 0304","0231"};
lookup1("0231") ->	{"0","022F 0304",""};
lookup1("0232") ->	{"0","0059 0304","0233"};
lookup1("0233") ->	{"0","0079 0304",""};
lookup1("0234") ->	{"0","",""};
lookup1("0235") ->	{"0","",""};
lookup1("0236") ->	{"0","",""};
lookup1("0237") ->	{"0","",""};
lookup1("0238") ->	{"0","",""};
lookup1("0239") ->	{"0","",""};
lookup1("023A") ->	{"0","","2C65"};
lookup1("023B") ->	{"0","","023C"};
lookup1("023C") ->	{"0","",""};
lookup1("023D") ->	{"0","","019A"};
lookup1("023E") ->	{"0","","2C66"};
lookup1("023F") ->	{"0","",""};
lookup1("0240") ->	{"0","",""};
lookup1("0241") ->	{"0","","0242"};
lookup1("0242") ->	{"0","",""};
lookup1("0243") ->	{"0","","0180"};
lookup1("0244") ->	{"0","","0289"};
lookup1("0245") ->	{"0","","028C"};
lookup1("0246") ->	{"0","","0247"};
lookup1("0247") ->	{"0","",""};
lookup1("0248") ->	{"0","","0249"};
lookup1("0249") ->	{"0","",""};
lookup1("024A") ->	{"0","","024B"};
lookup1("024B") ->	{"0","",""};
lookup1("024C") ->	{"0","","024D"};
lookup1("024D") ->	{"0","",""};
lookup1("024E") ->	{"0","","024F"};
lookup1("024F") ->	{"0","",""};
lookup1("0250") ->	{"0","",""};
lookup1("0251") ->	{"0","",""};
lookup1("0252") ->	{"0","",""};
lookup1("0253") ->	{"0","",""};
lookup1("0254") ->	{"0","",""};
lookup1("0255") ->	{"0","",""};
lookup1("0256") ->	{"0","",""};
lookup1("0257") ->	{"0","",""};
lookup1("0258") ->	{"0","",""};
lookup1("0259") ->	{"0","",""};
lookup1("025A") ->	{"0","",""};
lookup1("025B") ->	{"0","",""};
lookup1("025C") ->	{"0","",""};
lookup1("025D") ->	{"0","",""};
lookup1("025E") ->	{"0","",""};
lookup1("025F") ->	{"0","",""};
lookup1("0260") ->	{"0","",""};
lookup1("0261") ->	{"0","",""};
lookup1("0262") ->	{"0","",""};
lookup1("0263") ->	{"0","",""};
lookup1("0264") ->	{"0","",""};
lookup1("0265") ->	{"0","",""};
lookup1("0266") ->	{"0","",""};
lookup1("0267") ->	{"0","",""};
lookup1("0268") ->	{"0","",""};
lookup1("0269") ->	{"0","",""};
lookup1("026A") ->	{"0","",""};
lookup1("026B") ->	{"0","",""};
lookup1("026C") ->	{"0","",""};
lookup1("026D") ->	{"0","",""};
lookup1("026E") ->	{"0","",""};
lookup1("026F") ->	{"0","",""};
lookup1("0270") ->	{"0","",""};
lookup1("0271") ->	{"0","",""};
lookup1("0272") ->	{"0","",""};
lookup1("0273") ->	{"0","",""};
lookup1("0274") ->	{"0","",""};
lookup1("0275") ->	{"0","",""};
lookup1("0276") ->	{"0","",""};
lookup1("0277") ->	{"0","",""};
lookup1("0278") ->	{"0","",""};
lookup1("0279") ->	{"0","",""};
lookup1("027A") ->	{"0","",""};
lookup1("027B") ->	{"0","",""};
lookup1("027C") ->	{"0","",""};
lookup1("027D") ->	{"0","",""};
lookup1("027E") ->	{"0","",""};
lookup1("027F") ->	{"0","",""};
lookup1("0280") ->	{"0","",""};
lookup1("0281") ->	{"0","",""};
lookup1("0282") ->	{"0","",""};
lookup1("0283") ->	{"0","",""};
lookup1("0284") ->	{"0","",""};
lookup1("0285") ->	{"0","",""};
lookup1("0286") ->	{"0","",""};
lookup1("0287") ->	{"0","",""};
lookup1("0288") ->	{"0","",""};
lookup1("0289") ->	{"0","",""};
lookup1("028A") ->	{"0","",""};
lookup1("028B") ->	{"0","",""};
lookup1("028C") ->	{"0","",""};
lookup1("028D") ->	{"0","",""};
lookup1("028E") ->	{"0","",""};
lookup1("028F") ->	{"0","",""};
lookup1("0290") ->	{"0","",""};
lookup1("0291") ->	{"0","",""};
lookup1("0292") ->	{"0","",""};
lookup1("0293") ->	{"0","",""};
lookup1("0294") ->	{"0","",""};
lookup1("0295") ->	{"0","",""};
lookup1("0296") ->	{"0","",""};
lookup1("0297") ->	{"0","",""};
lookup1("0298") ->	{"0","",""};
lookup1("0299") ->	{"0","",""};
lookup1("029A") ->	{"0","",""};
lookup1("029B") ->	{"0","",""};
lookup1("029C") ->	{"0","",""};
lookup1("029D") ->	{"0","",""};
lookup1("029E") ->	{"0","",""};
lookup1("029F") ->	{"0","",""};
lookup1("02A0") ->	{"0","",""};
lookup1("02A1") ->	{"0","",""};
lookup1("02A2") ->	{"0","",""};
lookup1("02A3") ->	{"0","",""};
lookup1("02A4") ->	{"0","",""};
lookup1("02A5") ->	{"0","",""};
lookup1("02A6") ->	{"0","",""};
lookup1("02A7") ->	{"0","",""};
lookup1("02A8") ->	{"0","",""};
lookup1("02A9") ->	{"0","",""};
lookup1("02AA") ->	{"0","",""};
lookup1("02AB") ->	{"0","",""};
lookup1("02AC") ->	{"0","",""};
lookup1("02AD") ->	{"0","",""};
lookup1("02AE") ->	{"0","",""};
lookup1("02AF") ->	{"0","",""};
lookup1("02B0") ->	{"0","<super> 0068",""};
lookup1("02B1") ->	{"0","<super> 0266",""};
lookup1("02B2") ->	{"0","<super> 006A",""};
lookup1("02B3") ->	{"0","<super> 0072",""};
lookup1("02B4") ->	{"0","<super> 0279",""};
lookup1("02B5") ->	{"0","<super> 027B",""};
lookup1("02B6") ->	{"0","<super> 0281",""};
lookup1("02B7") ->	{"0","<super> 0077",""};
lookup1("02B8") ->	{"0","<super> 0079",""};
lookup1("02B9") ->	{"0","",""};
lookup1("02BA") ->	{"0","",""};
lookup1("02BB") ->	{"0","",""};
lookup1("02BC") ->	{"0","",""};
lookup1("02BD") ->	{"0","",""};
lookup1("02BE") ->	{"0","",""};
lookup1("02BF") ->	{"0","",""};
lookup1("02C0") ->	{"0","",""};
lookup1("02C1") ->	{"0","",""};
lookup1("02C2") ->	{"0","",""};
lookup1("02C3") ->	{"0","",""};
lookup1("02C4") ->	{"0","",""};
lookup1("02C5") ->	{"0","",""};
lookup1("02C6") ->	{"0","",""};
lookup1("02C7") ->	{"0","",""};
lookup1("02C8") ->	{"0","",""};
lookup1("02C9") ->	{"0","",""};
lookup1("02CA") ->	{"0","",""};
lookup1("02CB") ->	{"0","",""};
lookup1("02CC") ->	{"0","",""};
lookup1("02CD") ->	{"0","",""};
lookup1("02CE") ->	{"0","",""};
lookup1("02CF") ->	{"0","",""};
lookup1("02D0") ->	{"0","",""};
lookup1("02D1") ->	{"0","",""};
lookup1("02D2") ->	{"0","",""};
lookup1("02D3") ->	{"0","",""};
lookup1("02D4") ->	{"0","",""};
lookup1("02D5") ->	{"0","",""};
lookup1("02D6") ->	{"0","",""};
lookup1("02D7") ->	{"0","",""};
lookup1("02D8") ->	{"0","<compat> 0020 0306",""};
lookup1("02D9") ->	{"0","<compat> 0020 0307",""};
lookup1("02DA") ->	{"0","<compat> 0020 030A",""};
lookup1("02DB") ->	{"0","<compat> 0020 0328",""};
lookup1("02DC") ->	{"0","<compat> 0020 0303",""};
lookup1("02DD") ->	{"0","<compat> 0020 030B",""};
lookup1("02DE") ->	{"0","",""};
lookup1("02DF") ->	{"0","",""};
lookup1("02E0") ->	{"0","<super> 0263",""};
lookup1("02E1") ->	{"0","<super> 006C",""};
lookup1("02E2") ->	{"0","<super> 0073",""};
lookup1("02E3") ->	{"0","<super> 0078",""};
lookup1("02E4") ->	{"0","<super> 0295",""};
lookup1("02E5") ->	{"0","",""};
lookup1("02E6") ->	{"0","",""};
lookup1("02E7") ->	{"0","",""};
lookup1("02E8") ->	{"0","",""};
lookup1("02E9") ->	{"0","",""};
lookup1("02EA") ->	{"0","",""};
lookup1("02EB") ->	{"0","",""};
lookup1("02EC") ->	{"0","",""};
lookup1("02ED") ->	{"0","",""};
lookup1("02EE") ->	{"0","",""};
lookup1("02EF") ->	{"0","",""};
lookup1("02F0") ->	{"0","",""};
lookup1("02F1") ->	{"0","",""};
lookup1("02F2") ->	{"0","",""};
lookup1("02F3") ->	{"0","",""};
lookup1("02F4") ->	{"0","",""};
lookup1("02F5") ->	{"0","",""};
lookup1("02F6") ->	{"0","",""};
lookup1("02F7") ->	{"0","",""};
lookup1("02F8") ->	{"0","",""};
lookup1("02F9") ->	{"0","",""};
lookup1("02FA") ->	{"0","",""};
lookup1("02FB") ->	{"0","",""};
lookup1("02FC") ->	{"0","",""};
lookup1("02FD") ->	{"0","",""};
lookup1("02FE") ->	{"0","",""};
lookup1("02FF") ->	{"0","",""};
lookup1("0300") ->	{"230","",""};
lookup1("0301") ->	{"230","",""};
lookup1("0302") ->	{"230","",""};
lookup1("0303") ->	{"230","",""};
lookup1("0304") ->	{"230","",""};
lookup1("0305") ->	{"230","",""};
lookup1("0306") ->	{"230","",""};
lookup1("0307") ->	{"230","",""};
lookup1("0308") ->	{"230","",""};
lookup1("0309") ->	{"230","",""};
lookup1("030A") ->	{"230","",""};
lookup1("030B") ->	{"230","",""};
lookup1("030C") ->	{"230","",""};
lookup1("030D") ->	{"230","",""};
lookup1("030E") ->	{"230","",""};
lookup1("030F") ->	{"230","",""};
lookup1("0310") ->	{"230","",""};
lookup1("0311") ->	{"230","",""};
lookup1("0312") ->	{"230","",""};
lookup1("0313") ->	{"230","",""};
lookup1("0314") ->	{"230","",""};
lookup1("0315") ->	{"232","",""};
lookup1("0316") ->	{"220","",""};
lookup1("0317") ->	{"220","",""};
lookup1("0318") ->	{"220","",""};
lookup1("0319") ->	{"220","",""};
lookup1("031A") ->	{"232","",""};
lookup1("031B") ->	{"216","",""};
lookup1("031C") ->	{"220","",""};
lookup1("031D") ->	{"220","",""};
lookup1("031E") ->	{"220","",""};
lookup1("031F") ->	{"220","",""};
lookup1("0320") ->	{"220","",""};
lookup1("0321") ->	{"202","",""};
lookup1("0322") ->	{"202","",""};
lookup1("0323") ->	{"220","",""};
lookup1("0324") ->	{"220","",""};
lookup1("0325") ->	{"220","",""};
lookup1("0326") ->	{"220","",""};
lookup1("0327") ->	{"202","",""};
lookup1("0328") ->	{"202","",""};
lookup1("0329") ->	{"220","",""};
lookup1("032A") ->	{"220","",""};
lookup1("032B") ->	{"220","",""};
lookup1("032C") ->	{"220","",""};
lookup1("032D") ->	{"220","",""};
lookup1("032E") ->	{"220","",""};
lookup1("032F") ->	{"220","",""};
lookup1("0330") ->	{"220","",""};
lookup1("0331") ->	{"220","",""};
lookup1("0332") ->	{"220","",""};
lookup1("0333") ->	{"220","",""};
lookup1("0334") ->	{"1","",""};
lookup1("0335") ->	{"1","",""};
lookup1("0336") ->	{"1","",""};
lookup1("0337") ->	{"1","",""};
lookup1("0338") ->	{"1","",""};
lookup1("0339") ->	{"220","",""};
lookup1("033A") ->	{"220","",""};
lookup1("033B") ->	{"220","",""};
lookup1("033C") ->	{"220","",""};
lookup1("033D") ->	{"230","",""};
lookup1("033E") ->	{"230","",""};
lookup1("033F") ->	{"230","",""};
lookup1("0340") ->	{"230","0300",""};
lookup1("0341") ->	{"230","0301",""};
lookup1("0342") ->	{"230","",""};
lookup1("0343") ->	{"230","0313",""};
lookup1("0344") ->	{"230","0308 0301",""};
lookup1("0345") ->	{"240","",""};
lookup1("0346") ->	{"230","",""};
lookup1("0347") ->	{"220","",""};
lookup1("0348") ->	{"220","",""};
lookup1("0349") ->	{"220","",""};
lookup1("034A") ->	{"230","",""};
lookup1("034B") ->	{"230","",""};
lookup1("034C") ->	{"230","",""};
lookup1("034D") ->	{"220","",""};
lookup1("034E") ->	{"220","",""};
lookup1("034F") ->	{"0","",""};
lookup1("0350") ->	{"230","",""};
lookup1("0351") ->	{"230","",""};
lookup1("0352") ->	{"230","",""};
lookup1("0353") ->	{"220","",""};
lookup1("0354") ->	{"220","",""};
lookup1("0355") ->	{"220","",""};
lookup1("0356") ->	{"220","",""};
lookup1("0357") ->	{"230","",""};
lookup1("0358") ->	{"232","",""};
lookup1("0359") ->	{"220","",""};
lookup1("035A") ->	{"220","",""};
lookup1("035B") ->	{"230","",""};
lookup1("035C") ->	{"233","",""};
lookup1("035D") ->	{"234","",""};
lookup1("035E") ->	{"234","",""};
lookup1("035F") ->	{"233","",""};
lookup1("0360") ->	{"234","",""};
lookup1("0361") ->	{"234","",""};
lookup1("0362") ->	{"233","",""};
lookup1("0363") ->	{"230","",""};
lookup1("0364") ->	{"230","",""};
lookup1("0365") ->	{"230","",""};
lookup1("0366") ->	{"230","",""};
lookup1("0367") ->	{"230","",""};
lookup1("0368") ->	{"230","",""};
lookup1("0369") ->	{"230","",""};
lookup1("036A") ->	{"230","",""};
lookup1("036B") ->	{"230","",""};
lookup1("036C") ->	{"230","",""};
lookup1("036D") ->	{"230","",""};
lookup1("036E") ->	{"230","",""};
lookup1("036F") ->	{"230","",""};
lookup1("0370") ->	{"0","","0371"};
lookup1("0371") ->	{"0","",""};
lookup1("0372") ->	{"0","","0373"};
lookup1("0373") ->	{"0","",""};
lookup1("0374") ->	{"0","02B9",""};
lookup1("0375") ->	{"0","",""};
lookup1("0376") ->	{"0","","0377"};
lookup1("0377") ->	{"0","",""};
lookup1("037A") ->	{"0","<compat> 0020 0345",""};
lookup1("037B") ->	{"0","",""};
lookup1("037C") ->	{"0","",""};
lookup1("037D") ->	{"0","",""};
lookup1("037E") ->	{"0","003B",""};
lookup1("037F") ->	{"0","","03F3"};
lookup1("0384") ->	{"0","<compat> 0020 0301",""};
lookup1("0385") ->	{"0","00A8 0301",""};
lookup1("0386") ->	{"0","0391 0301","03AC"};
lookup1("0387") ->	{"0","00B7",""};
lookup1("0388") ->	{"0","0395 0301","03AD"};
lookup1("0389") ->	{"0","0397 0301","03AE"};
lookup1("038A") ->	{"0","0399 0301","03AF"};
lookup1("038C") ->	{"0","039F 0301","03CC"};
lookup1("038E") ->	{"0","03A5 0301","03CD"};
lookup1("038F") ->	{"0","03A9 0301","03CE"};
lookup1("0390") ->	{"0","03CA 0301",""};
lookup1("0391") ->	{"0","","03B1"};
lookup1("0392") ->	{"0","","03B2"};
lookup1("0393") ->	{"0","","03B3"};
lookup1("0394") ->	{"0","","03B4"};
lookup1("0395") ->	{"0","","03B5"};
lookup1("0396") ->	{"0","","03B6"};
lookup1("0397") ->	{"0","","03B7"};
lookup1("0398") ->	{"0","","03B8"};
lookup1("0399") ->	{"0","","03B9"};
lookup1("039A") ->	{"0","","03BA"};
lookup1("039B") ->	{"0","","03BB"};
lookup1("039C") ->	{"0","","03BC"};
lookup1("039D") ->	{"0","","03BD"};
lookup1("039E") ->	{"0","","03BE"};
lookup1("039F") ->	{"0","","03BF"};
lookup1("03A0") ->	{"0","","03C0"};
lookup1("03A1") ->	{"0","","03C1"};
lookup1("03A3") ->	{"0","","03C3"};
lookup1("03A4") ->	{"0","","03C4"};
lookup1("03A5") ->	{"0","","03C5"};
lookup1("03A6") ->	{"0","","03C6"};
lookup1("03A7") ->	{"0","","03C7"};
lookup1("03A8") ->	{"0","","03C8"};
lookup1("03A9") ->	{"0","","03C9"};
lookup1("03AA") ->	{"0","0399 0308","03CA"};
lookup1("03AB") ->	{"0","03A5 0308","03CB"};
lookup1("03AC") ->	{"0","03B1 0301",""};
lookup1("03AD") ->	{"0","03B5 0301",""};
lookup1("03AE") ->	{"0","03B7 0301",""};
lookup1("03AF") ->	{"0","03B9 0301",""};
lookup1("03B0") ->	{"0","03CB 0301",""};
lookup1("03B1") ->	{"0","",""};
lookup1("03B2") ->	{"0","",""};
lookup1("03B3") ->	{"0","",""};
lookup1("03B4") ->	{"0","",""};
lookup1("03B5") ->	{"0","",""};
lookup1("03B6") ->	{"0","",""};
lookup1("03B7") ->	{"0","",""};
lookup1("03B8") ->	{"0","",""};
lookup1("03B9") ->	{"0","",""};
lookup1("03BA") ->	{"0","",""};
lookup1("03BB") ->	{"0","",""};
lookup1("03BC") ->	{"0","",""};
lookup1("03BD") ->	{"0","",""};
lookup1("03BE") ->	{"0","",""};
lookup1("03BF") ->	{"0","",""};
lookup1("03C0") ->	{"0","",""};
lookup1("03C1") ->	{"0","",""};
lookup1("03C2") ->	{"0","",""};
lookup1("03C3") ->	{"0","",""};
lookup1("03C4") ->	{"0","",""};
lookup1("03C5") ->	{"0","",""};
lookup1("03C6") ->	{"0","",""};
lookup1("03C7") ->	{"0","",""};
lookup1("03C8") ->	{"0","",""};
lookup1("03C9") ->	{"0","",""};
lookup1("03CA") ->	{"0","03B9 0308",""};
lookup1("03CB") ->	{"0","03C5 0308",""};
lookup1("03CC") ->	{"0","03BF 0301",""};
lookup1("03CD") ->	{"0","03C5 0301",""};
lookup1("03CE") ->	{"0","03C9 0301",""};
lookup1("03CF") ->	{"0","","03D7"};
lookup1("03D0") ->	{"0","<compat> 03B2",""};
lookup1("03D1") ->	{"0","<compat> 03B8",""};
lookup1("03D2") ->	{"0","<compat> 03A5",""};
lookup1("03D3") ->	{"0","03D2 0301",""};
lookup1("03D4") ->	{"0","03D2 0308",""};
lookup1("03D5") ->	{"0","<compat> 03C6",""};
lookup1("03D6") ->	{"0","<compat> 03C0",""};
lookup1("03D7") ->	{"0","",""};
lookup1("03D8") ->	{"0","","03D9"};
lookup1("03D9") ->	{"0","",""};
lookup1("03DA") ->	{"0","","03DB"};
lookup1("03DB") ->	{"0","",""};
lookup1("03DC") ->	{"0","","03DD"};
lookup1("03DD") ->	{"0","",""};
lookup1("03DE") ->	{"0","","03DF"};
lookup1("03DF") ->	{"0","",""};
lookup1("03E0") ->	{"0","","03E1"};
lookup1("03E1") ->	{"0","",""};
lookup1("03E2") ->	{"0","","03E3"};
lookup1("03E3") ->	{"0","",""};
lookup1("03E4") ->	{"0","","03E5"};
lookup1("03E5") ->	{"0","",""};
lookup1("03E6") ->	{"0","","03E7"};
lookup1("03E7") ->	{"0","",""};
lookup1("03E8") ->	{"0","","03E9"};
lookup1("03E9") ->	{"0","",""};
lookup1("03EA") ->	{"0","","03EB"};
lookup1("03EB") ->	{"0","",""};
lookup1("03EC") ->	{"0","","03ED"};
lookup1("03ED") ->	{"0","",""};
lookup1("03EE") ->	{"0","","03EF"};
lookup1("03EF") ->	{"0","",""};
lookup1("03F0") ->	{"0","<compat> 03BA",""};
lookup1("03F1") ->	{"0","<compat> 03C1",""};
lookup1("03F2") ->	{"0","<compat> 03C2",""};
lookup1("03F3") ->	{"0","",""};
lookup1("03F4") ->	{"0","<compat> 0398","03B8"};
lookup1("03F5") ->	{"0","<compat> 03B5",""};
lookup1("03F6") ->	{"0","",""};
lookup1("03F7") ->	{"0","","03F8"};
lookup1("03F8") ->	{"0","",""};
lookup1("03F9") ->	{"0","<compat> 03A3","03F2"};
lookup1("03FA") ->	{"0","","03FB"};
lookup1("03FB") ->	{"0","",""};
lookup1("03FC") ->	{"0","",""};
lookup1("03FD") ->	{"0","","037B"};
lookup1("03FE") ->	{"0","","037C"};
lookup1("03FF") ->	{"0","","037D"};
lookup1("0400") ->	{"0","0415 0300","0450"};
lookup1("0401") ->	{"0","0415 0308","0451"};
lookup1("0402") ->	{"0","","0452"};
lookup1("0403") ->	{"0","0413 0301","0453"};
lookup1("0404") ->	{"0","","0454"};
lookup1("0405") ->	{"0","","0455"};
lookup1("0406") ->	{"0","","0456"};
lookup1("0407") ->	{"0","0406 0308","0457"};
lookup1("0408") ->	{"0","","0458"};
lookup1("0409") ->	{"0","","0459"};
lookup1("040A") ->	{"0","","045A"};
lookup1("040B") ->	{"0","","045B"};
lookup1("040C") ->	{"0","041A 0301","045C"};
lookup1("040D") ->	{"0","0418 0300","045D"};
lookup1("040E") ->	{"0","0423 0306","045E"};
lookup1("040F") ->	{"0","","045F"};
lookup1("0410") ->	{"0","","0430"};
lookup1("0411") ->	{"0","","0431"};
lookup1("0412") ->	{"0","","0432"};
lookup1("0413") ->	{"0","","0433"};
lookup1("0414") ->	{"0","","0434"};
lookup1("0415") ->	{"0","","0435"};
lookup1("0416") ->	{"0","","0436"};
lookup1("0417") ->	{"0","","0437"};
lookup1("0418") ->	{"0","","0438"};
lookup1("0419") ->	{"0","0418 0306","0439"};
lookup1("041A") ->	{"0","","043A"};
lookup1("041B") ->	{"0","","043B"};
lookup1("041C") ->	{"0","","043C"};
lookup1("041D") ->	{"0","","043D"};
lookup1("041E") ->	{"0","","043E"};
lookup1("041F") ->	{"0","","043F"};
lookup1("0420") ->	{"0","","0440"};
lookup1("0421") ->	{"0","","0441"};
lookup1("0422") ->	{"0","","0442"};
lookup1("0423") ->	{"0","","0443"};
lookup1("0424") ->	{"0","","0444"};
lookup1("0425") ->	{"0","","0445"};
lookup1("0426") ->	{"0","","0446"};
lookup1("0427") ->	{"0","","0447"};
lookup1("0428") ->	{"0","","0448"};
lookup1("0429") ->	{"0","","0449"};
lookup1("042A") ->	{"0","","044A"};
lookup1("042B") ->	{"0","","044B"};
lookup1("042C") ->	{"0","","044C"};
lookup1("042D") ->	{"0","","044D"};
lookup1("042E") ->	{"0","","044E"};
lookup1("042F") ->	{"0","","044F"};
lookup1("0430") ->	{"0","",""};
lookup1("0431") ->	{"0","",""};
lookup1("0432") ->	{"0","",""};
lookup1("0433") ->	{"0","",""};
lookup1("0434") ->	{"0","",""};
lookup1("0435") ->	{"0","",""};
lookup1("0436") ->	{"0","",""};
lookup1("0437") ->	{"0","",""};
lookup1("0438") ->	{"0","",""};
lookup1("0439") ->	{"0","0438 0306",""};
lookup1("043A") ->	{"0","",""};
lookup1("043B") ->	{"0","",""};
lookup1("043C") ->	{"0","",""};
lookup1("043D") ->	{"0","",""};
lookup1("043E") ->	{"0","",""};
lookup1("043F") ->	{"0","",""};
lookup1("0440") ->	{"0","",""};
lookup1("0441") ->	{"0","",""};
lookup1("0442") ->	{"0","",""};
lookup1("0443") ->	{"0","",""};
lookup1("0444") ->	{"0","",""};
lookup1("0445") ->	{"0","",""};
lookup1("0446") ->	{"0","",""};
lookup1("0447") ->	{"0","",""};
lookup1("0448") ->	{"0","",""};
lookup1("0449") ->	{"0","",""};
lookup1("044A") ->	{"0","",""};
lookup1("044B") ->	{"0","",""};
lookup1("044C") ->	{"0","",""};
lookup1("044D") ->	{"0","",""};
lookup1("044E") ->	{"0","",""};
lookup1("044F") ->	{"0","",""};
lookup1("0450") ->	{"0","0435 0300",""};
lookup1("0451") ->	{"0","0435 0308",""};
lookup1("0452") ->	{"0","",""};
lookup1("0453") ->	{"0","0433 0301",""};
lookup1("0454") ->	{"0","",""};
lookup1("0455") ->	{"0","",""};
lookup1("0456") ->	{"0","",""};
lookup1("0457") ->	{"0","0456 0308",""};
lookup1("0458") ->	{"0","",""};
lookup1("0459") ->	{"0","",""};
lookup1("045A") ->	{"0","",""};
lookup1("045B") ->	{"0","",""};
lookup1("045C") ->	{"0","043A 0301",""};
lookup1("045D") ->	{"0","0438 0300",""};
lookup1("045E") ->	{"0","0443 0306",""};
lookup1("045F") ->	{"0","",""};
lookup1("0460") ->	{"0","","0461"};
lookup1("0461") ->	{"0","",""};
lookup1("0462") ->	{"0","","0463"};
lookup1("0463") ->	{"0","",""};
lookup1("0464") ->	{"0","","0465"};
lookup1("0465") ->	{"0","",""};
lookup1("0466") ->	{"0","","0467"};
lookup1("0467") ->	{"0","",""};
lookup1("0468") ->	{"0","","0469"};
lookup1("0469") ->	{"0","",""};
lookup1("046A") ->	{"0","","046B"};
lookup1("046B") ->	{"0","",""};
lookup1("046C") ->	{"0","","046D"};
lookup1("046D") ->	{"0","",""};
lookup1("046E") ->	{"0","","046F"};
lookup1("046F") ->	{"0","",""};
lookup1("0470") ->	{"0","","0471"};
lookup1("0471") ->	{"0","",""};
lookup1("0472") ->	{"0","","0473"};
lookup1("0473") ->	{"0","",""};
lookup1("0474") ->	{"0","","0475"};
lookup1("0475") ->	{"0","",""};
lookup1("0476") ->	{"0","0474 030F","0477"};
lookup1("0477") ->	{"0","0475 030F",""};
lookup1("0478") ->	{"0","","0479"};
lookup1("0479") ->	{"0","",""};
lookup1("047A") ->	{"0","","047B"};
lookup1("047B") ->	{"0","",""};
lookup1("047C") ->	{"0","","047D"};
lookup1("047D") ->	{"0","",""};
lookup1("047E") ->	{"0","","047F"};
lookup1("047F") ->	{"0","",""};
lookup1("0480") ->	{"0","","0481"};
lookup1("0481") ->	{"0","",""};
lookup1("0482") ->	{"0","",""};
lookup1("0483") ->	{"230","",""};
lookup1("0484") ->	{"230","",""};
lookup1("0485") ->	{"230","",""};
lookup1("0486") ->	{"230","",""};
lookup1("0487") ->	{"230","",""};
lookup1("0488") ->	{"0","",""};
lookup1("0489") ->	{"0","",""};
lookup1("048A") ->	{"0","","048B"};
lookup1("048B") ->	{"0","",""};
lookup1("048C") ->	{"0","","048D"};
lookup1("048D") ->	{"0","",""};
lookup1("048E") ->	{"0","","048F"};
lookup1("048F") ->	{"0","",""};
lookup1("0490") ->	{"0","","0491"};
lookup1("0491") ->	{"0","",""};
lookup1("0492") ->	{"0","","0493"};
lookup1("0493") ->	{"0","",""};
lookup1("0494") ->	{"0","","0495"};
lookup1("0495") ->	{"0","",""};
lookup1("0496") ->	{"0","","0497"};
lookup1("0497") ->	{"0","",""};
lookup1("0498") ->	{"0","","0499"};
lookup1("0499") ->	{"0","",""};
lookup1("049A") ->	{"0","","049B"};
lookup1("049B") ->	{"0","",""};
lookup1("049C") ->	{"0","","049D"};
lookup1("049D") ->	{"0","",""};
lookup1("049E") ->	{"0","","049F"};
lookup1("049F") ->	{"0","",""};
lookup1("04A0") ->	{"0","","04A1"};
lookup1("04A1") ->	{"0","",""};
lookup1("04A2") ->	{"0","","04A3"};
lookup1("04A3") ->	{"0","",""};
lookup1("04A4") ->	{"0","","04A5"};
lookup1("04A5") ->	{"0","",""};
lookup1("04A6") ->	{"0","","04A7"};
lookup1("04A7") ->	{"0","",""};
lookup1("04A8") ->	{"0","","04A9"};
lookup1("04A9") ->	{"0","",""};
lookup1("04AA") ->	{"0","","04AB"};
lookup1("04AB") ->	{"0","",""};
lookup1("04AC") ->	{"0","","04AD"};
lookup1("04AD") ->	{"0","",""};
lookup1("04AE") ->	{"0","","04AF"};
lookup1("04AF") ->	{"0","",""};
lookup1("04B0") ->	{"0","","04B1"};
lookup1("04B1") ->	{"0","",""};
lookup1("04B2") ->	{"0","","04B3"};
lookup1("04B3") ->	{"0","",""};
lookup1("04B4") ->	{"0","","04B5"};
lookup1("04B5") ->	{"0","",""};
lookup1("04B6") ->	{"0","","04B7"};
lookup1("04B7") ->	{"0","",""};
lookup1("04B8") ->	{"0","","04B9"};
lookup1("04B9") ->	{"0","",""};
lookup1("04BA") ->	{"0","","04BB"};
lookup1("04BB") ->	{"0","",""};
lookup1("04BC") ->	{"0","","04BD"};
lookup1("04BD") ->	{"0","",""};
lookup1("04BE") ->	{"0","","04BF"};
lookup1("04BF") ->	{"0","",""};
lookup1("04C0") ->	{"0","","04CF"};
lookup1("04C1") ->	{"0","0416 0306","04C2"};
lookup1("04C2") ->	{"0","0436 0306",""};
lookup1("04C3") ->	{"0","","04C4"};
lookup1("04C4") ->	{"0","",""};
lookup1("04C5") ->	{"0","","04C6"};
lookup1("04C6") ->	{"0","",""};
lookup1("04C7") ->	{"0","","04C8"};
lookup1("04C8") ->	{"0","",""};
lookup1("04C9") ->	{"0","","04CA"};
lookup1("04CA") ->	{"0","",""};
lookup1("04CB") ->	{"0","","04CC"};
lookup1("04CC") ->	{"0","",""};
lookup1("04CD") ->	{"0","","04CE"};
lookup1("04CE") ->	{"0","",""};
lookup1("04CF") ->	{"0","",""};
lookup1("04D0") ->	{"0","0410 0306","04D1"};
lookup1("04D1") ->	{"0","0430 0306",""};
lookup1("04D2") ->	{"0","0410 0308","04D3"};
lookup1("04D3") ->	{"0","0430 0308",""};
lookup1("04D4") ->	{"0","","04D5"};
lookup1("04D5") ->	{"0","",""};
lookup1("04D6") ->	{"0","0415 0306","04D7"};
lookup1("04D7") ->	{"0","0435 0306",""};
lookup1("04D8") ->	{"0","","04D9"};
lookup1("04D9") ->	{"0","",""};
lookup1("04DA") ->	{"0","04D8 0308","04DB"};
lookup1("04DB") ->	{"0","04D9 0308",""};
lookup1("04DC") ->	{"0","0416 0308","04DD"};
lookup1("04DD") ->	{"0","0436 0308",""};
lookup1("04DE") ->	{"0","0417 0308","04DF"};
lookup1("04DF") ->	{"0","0437 0308",""};
lookup1("04E0") ->	{"0","","04E1"};
lookup1("04E1") ->	{"0","",""};
lookup1("04E2") ->	{"0","0418 0304","04E3"};
lookup1("04E3") ->	{"0","0438 0304",""};
lookup1("04E4") ->	{"0","0418 0308","04E5"};
lookup1("04E5") ->	{"0","0438 0308",""};
lookup1("04E6") ->	{"0","041E 0308","04E7"};
lookup1("04E7") ->	{"0","043E 0308",""};
lookup1("04E8") ->	{"0","","04E9"};
lookup1("04E9") ->	{"0","",""};
lookup1("04EA") ->	{"0","04E8 0308","04EB"};
lookup1("04EB") ->	{"0","04E9 0308",""};
lookup1("04EC") ->	{"0","042D 0308","04ED"};
lookup1("04ED") ->	{"0","044D 0308",""};
lookup1("04EE") ->	{"0","0423 0304","04EF"};
lookup1("04EF") ->	{"0","0443 0304",""};
lookup1("04F0") ->	{"0","0423 0308","04F1"};
lookup1("04F1") ->	{"0","0443 0308",""};
lookup1("04F2") ->	{"0","0423 030B","04F3"};
lookup1("04F3") ->	{"0","0443 030B",""};
lookup1("04F4") ->	{"0","0427 0308","04F5"};
lookup1("04F5") ->	{"0","0447 0308",""};
lookup1("04F6") ->	{"0","","04F7"};
lookup1("04F7") ->	{"0","",""};
lookup1("04F8") ->	{"0","042B 0308","04F9"};
lookup1("04F9") ->	{"0","044B 0308",""};
lookup1("04FA") ->	{"0","","04FB"};
lookup1("04FB") ->	{"0","",""};
lookup1("04FC") ->	{"0","","04FD"};
lookup1("04FD") ->	{"0","",""};
lookup1("04FE") ->	{"0","","04FF"};
lookup1("04FF") ->	{"0","",""};
lookup1("0500") ->	{"0","","0501"};
lookup1("0501") ->	{"0","",""};
lookup1("0502") ->	{"0","","0503"};
lookup1("0503") ->	{"0","",""};
lookup1("0504") ->	{"0","","0505"};
lookup1("0505") ->	{"0","",""};
lookup1("0506") ->	{"0","","0507"};
lookup1("0507") ->	{"0","",""};
lookup1("0508") ->	{"0","","0509"};
lookup1("0509") ->	{"0","",""};
lookup1("050A") ->	{"0","","050B"};
lookup1("050B") ->	{"0","",""};
lookup1("050C") ->	{"0","","050D"};
lookup1("050D") ->	{"0","",""};
lookup1("050E") ->	{"0","","050F"};
lookup1("050F") ->	{"0","",""};
lookup1("0510") ->	{"0","","0511"};
lookup1("0511") ->	{"0","",""};
lookup1("0512") ->	{"0","","0513"};
lookup1("0513") ->	{"0","",""};
lookup1("0514") ->	{"0","","0515"};
lookup1("0515") ->	{"0","",""};
lookup1("0516") ->	{"0","","0517"};
lookup1("0517") ->	{"0","",""};
lookup1("0518") ->	{"0","","0519"};
lookup1("0519") ->	{"0","",""};
lookup1("051A") ->	{"0","","051B"};
lookup1("051B") ->	{"0","",""};
lookup1("051C") ->	{"0","","051D"};
lookup1("051D") ->	{"0","",""};
lookup1("051E") ->	{"0","","051F"};
lookup1("051F") ->	{"0","",""};
lookup1("0520") ->	{"0","","0521"};
lookup1("0521") ->	{"0","",""};
lookup1("0522") ->	{"0","","0523"};
lookup1("0523") ->	{"0","",""};
lookup1("0524") ->	{"0","","0525"};
lookup1("0525") ->	{"0","",""};
lookup1("0526") ->	{"0","","0527"};
lookup1("0527") ->	{"0","",""};
lookup1("0528") ->	{"0","","0529"};
lookup1("0529") ->	{"0","",""};
lookup1("052A") ->	{"0","","052B"};
lookup1("052B") ->	{"0","",""};
lookup1("052C") ->	{"0","","052D"};
lookup1("052D") ->	{"0","",""};
lookup1("052E") ->	{"0","","052F"};
lookup1("052F") ->	{"0","",""};
lookup1("0531") ->	{"0","","0561"};
lookup1("0532") ->	{"0","","0562"};
lookup1("0533") ->	{"0","","0563"};
lookup1("0534") ->	{"0","","0564"};
lookup1("0535") ->	{"0","","0565"};
lookup1("0536") ->	{"0","","0566"};
lookup1("0537") ->	{"0","","0567"};
lookup1("0538") ->	{"0","","0568"};
lookup1("0539") ->	{"0","","0569"};
lookup1("053A") ->	{"0","","056A"};
lookup1("053B") ->	{"0","","056B"};
lookup1("053C") ->	{"0","","056C"};
lookup1("053D") ->	{"0","","056D"};
lookup1("053E") ->	{"0","","056E"};
lookup1("053F") ->	{"0","","056F"};
lookup1("0540") ->	{"0","","0570"};
lookup1("0541") ->	{"0","","0571"};
lookup1("0542") ->	{"0","","0572"};
lookup1("0543") ->	{"0","","0573"};
lookup1("0544") ->	{"0","","0574"};
lookup1("0545") ->	{"0","","0575"};
lookup1("0546") ->	{"0","","0576"};
lookup1("0547") ->	{"0","","0577"};
lookup1("0548") ->	{"0","","0578"};
lookup1("0549") ->	{"0","","0579"};
lookup1("054A") ->	{"0","","057A"};
lookup1("054B") ->	{"0","","057B"};
lookup1("054C") ->	{"0","","057C"};
lookup1("054D") ->	{"0","","057D"};
lookup1("054E") ->	{"0","","057E"};
lookup1("054F") ->	{"0","","057F"};
lookup1("0550") ->	{"0","","0580"};
lookup1("0551") ->	{"0","","0581"};
lookup1("0552") ->	{"0","","0582"};
lookup1("0553") ->	{"0","","0583"};
lookup1("0554") ->	{"0","","0584"};
lookup1("0555") ->	{"0","","0585"};
lookup1("0556") ->	{"0","","0586"};
lookup1("0559") ->	{"0","",""};
lookup1("055A") ->	{"0","",""};
lookup1("055B") ->	{"0","",""};
lookup1("055C") ->	{"0","",""};
lookup1("055D") ->	{"0","",""};
lookup1("055E") ->	{"0","",""};
lookup1("055F") ->	{"0","",""};
lookup1("0561") ->	{"0","",""};
lookup1("0562") ->	{"0","",""};
lookup1("0563") ->	{"0","",""};
lookup1("0564") ->	{"0","",""};
lookup1("0565") ->	{"0","",""};
lookup1("0566") ->	{"0","",""};
lookup1("0567") ->	{"0","",""};
lookup1("0568") ->	{"0","",""};
lookup1("0569") ->	{"0","",""};
lookup1("056A") ->	{"0","",""};
lookup1("056B") ->	{"0","",""};
lookup1("056C") ->	{"0","",""};
lookup1("056D") ->	{"0","",""};
lookup1("056E") ->	{"0","",""};
lookup1("056F") ->	{"0","",""};
lookup1("0570") ->	{"0","",""};
lookup1("0571") ->	{"0","",""};
lookup1("0572") ->	{"0","",""};
lookup1("0573") ->	{"0","",""};
lookup1("0574") ->	{"0","",""};
lookup1("0575") ->	{"0","",""};
lookup1("0576") ->	{"0","",""};
lookup1("0577") ->	{"0","",""};
lookup1("0578") ->	{"0","",""};
lookup1("0579") ->	{"0","",""};
lookup1("057A") ->	{"0","",""};
lookup1("057B") ->	{"0","",""};
lookup1("057C") ->	{"0","",""};
lookup1("057D") ->	{"0","",""};
lookup1("057E") ->	{"0","",""};
lookup1("057F") ->	{"0","",""};
lookup1("0580") ->	{"0","",""};
lookup1("0581") ->	{"0","",""};
lookup1("0582") ->	{"0","",""};
lookup1("0583") ->	{"0","",""};
lookup1("0584") ->	{"0","",""};
lookup1("0585") ->	{"0","",""};
lookup1("0586") ->	{"0","",""};
lookup1("0587") ->	{"0","<compat> 0565 0582",""};
lookup1("0589") ->	{"0","",""};
lookup1("058A") ->	{"0","",""};
lookup1("058D") ->	{"0","",""};
lookup1("058E") ->	{"0","",""};
lookup1("058F") ->	{"0","",""};
lookup1("0591") ->	{"220","",""};
lookup1("0592") ->	{"230","",""};
lookup1("0593") ->	{"230","",""};
lookup1("0594") ->	{"230","",""};
lookup1("0595") ->	{"230","",""};
lookup1("0596") ->	{"220","",""};
lookup1("0597") ->	{"230","",""};
lookup1("0598") ->	{"230","",""};
lookup1("0599") ->	{"230","",""};
lookup1("059A") ->	{"222","",""};
lookup1("059B") ->	{"220","",""};
lookup1("059C") ->	{"230","",""};
lookup1("059D") ->	{"230","",""};
lookup1("059E") ->	{"230","",""};
lookup1("059F") ->	{"230","",""};
lookup1("05A0") ->	{"230","",""};
lookup1("05A1") ->	{"230","",""};
lookup1("05A2") ->	{"220","",""};
lookup1("05A3") ->	{"220","",""};
lookup1("05A4") ->	{"220","",""};
lookup1("05A5") ->	{"220","",""};
lookup1("05A6") ->	{"220","",""};
lookup1("05A7") ->	{"220","",""};
lookup1("05A8") ->	{"230","",""};
lookup1("05A9") ->	{"230","",""};
lookup1("05AA") ->	{"220","",""};
lookup1("05AB") ->	{"230","",""};
lookup1("05AC") ->	{"230","",""};
lookup1("05AD") ->	{"222","",""};
lookup1("05AE") ->	{"228","",""};
lookup1("05AF") ->	{"230","",""};
lookup1("05B0") ->	{"10","",""};
lookup1("05B1") ->	{"11","",""};
lookup1("05B2") ->	{"12","",""};
lookup1("05B3") ->	{"13","",""};
lookup1("05B4") ->	{"14","",""};
lookup1("05B5") ->	{"15","",""};
lookup1("05B6") ->	{"16","",""};
lookup1("05B7") ->	{"17","",""};
lookup1("05B8") ->	{"18","",""};
lookup1("05B9") ->	{"19","",""};
lookup1("05BA") ->	{"19","",""};
lookup1("05BB") ->	{"20","",""};
lookup1("05BC") ->	{"21","",""};
lookup1("05BD") ->	{"22","",""};
lookup1("05BE") ->	{"0","",""};
lookup1("05BF") ->	{"23","",""};
lookup1("05C0") ->	{"0","",""};
lookup1("05C1") ->	{"24","",""};
lookup1("05C2") ->	{"25","",""};
lookup1("05C3") ->	{"0","",""};
lookup1("05C4") ->	{"230","",""};
lookup1("05C5") ->	{"220","",""};
lookup1("05C6") ->	{"0","",""};
lookup1("05C7") ->	{"18","",""};
lookup1("05D0") ->	{"0","",""};
lookup1("05D1") ->	{"0","",""};
lookup1("05D2") ->	{"0","",""};
lookup1("05D3") ->	{"0","",""};
lookup1("05D4") ->	{"0","",""};
lookup1("05D5") ->	{"0","",""};
lookup1("05D6") ->	{"0","",""};
lookup1("05D7") ->	{"0","",""};
lookup1("05D8") ->	{"0","",""};
lookup1("05D9") ->	{"0","",""};
lookup1("05DA") ->	{"0","",""};
lookup1("05DB") ->	{"0","",""};
lookup1("05DC") ->	{"0","",""};
lookup1("05DD") ->	{"0","",""};
lookup1("05DE") ->	{"0","",""};
lookup1("05DF") ->	{"0","",""};
lookup1("05E0") ->	{"0","",""};
lookup1("05E1") ->	{"0","",""};
lookup1("05E2") ->	{"0","",""};
lookup1("05E3") ->	{"0","",""};
lookup1("05E4") ->	{"0","",""};
lookup1("05E5") ->	{"0","",""};
lookup1("05E6") ->	{"0","",""};
lookup1("05E7") ->	{"0","",""};
lookup1("05E8") ->	{"0","",""};
lookup1("05E9") ->	{"0","",""};
lookup1("05EA") ->	{"0","",""};
lookup1("05F0") ->	{"0","",""};
lookup1("05F1") ->	{"0","",""};
lookup1("05F2") ->	{"0","",""};
lookup1("05F3") ->	{"0","",""};
lookup1("05F4") ->	{"0","",""};
lookup1("0600") ->	{"0","",""};
lookup1("0601") ->	{"0","",""};
lookup1("0602") ->	{"0","",""};
lookup1("0603") ->	{"0","",""};
lookup1("0604") ->	{"0","",""};
lookup1("0605") ->	{"0","",""};
lookup1("0606") ->	{"0","",""};
lookup1("0607") ->	{"0","",""};
lookup1("0608") ->	{"0","",""};
lookup1("0609") ->	{"0","",""};
lookup1("060A") ->	{"0","",""};
lookup1("060B") ->	{"0","",""};
lookup1("060C") ->	{"0","",""};
lookup1("060D") ->	{"0","",""};
lookup1("060E") ->	{"0","",""};
lookup1("060F") ->	{"0","",""};
lookup1("0610") ->	{"230","",""};
lookup1("0611") ->	{"230","",""};
lookup1("0612") ->	{"230","",""};
lookup1("0613") ->	{"230","",""};
lookup1("0614") ->	{"230","",""};
lookup1("0615") ->	{"230","",""};
lookup1("0616") ->	{"230","",""};
lookup1("0617") ->	{"230","",""};
lookup1("0618") ->	{"30","",""};
lookup1("0619") ->	{"31","",""};
lookup1("061A") ->	{"32","",""};
lookup1("061B") ->	{"0","",""};
lookup1("061C") ->	{"0","",""};
lookup1("061E") ->	{"0","",""};
lookup1("061F") ->	{"0","",""};
lookup1("0620") ->	{"0","",""};
lookup1("0621") ->	{"0","",""};
lookup1("0622") ->	{"0","0627 0653",""};
lookup1("0623") ->	{"0","0627 0654",""};
lookup1("0624") ->	{"0","0648 0654",""};
lookup1("0625") ->	{"0","0627 0655",""};
lookup1("0626") ->	{"0","064A 0654",""};
lookup1("0627") ->	{"0","",""};
lookup1("0628") ->	{"0","",""};
lookup1("0629") ->	{"0","",""};
lookup1("062A") ->	{"0","",""};
lookup1("062B") ->	{"0","",""};
lookup1("062C") ->	{"0","",""};
lookup1("062D") ->	{"0","",""};
lookup1("062E") ->	{"0","",""};
lookup1("062F") ->	{"0","",""};
lookup1("0630") ->	{"0","",""};
lookup1("0631") ->	{"0","",""};
lookup1("0632") ->	{"0","",""};
lookup1("0633") ->	{"0","",""};
lookup1("0634") ->	{"0","",""};
lookup1("0635") ->	{"0","",""};
lookup1("0636") ->	{"0","",""};
lookup1("0637") ->	{"0","",""};
lookup1("0638") ->	{"0","",""};
lookup1("0639") ->	{"0","",""};
lookup1("063A") ->	{"0","",""};
lookup1("063B") ->	{"0","",""};
lookup1("063C") ->	{"0","",""};
lookup1("063D") ->	{"0","",""};
lookup1("063E") ->	{"0","",""};
lookup1("063F") ->	{"0","",""};
lookup1("0640") ->	{"0","",""};
lookup1("0641") ->	{"0","",""};
lookup1("0642") ->	{"0","",""};
lookup1("0643") ->	{"0","",""};
lookup1("0644") ->	{"0","",""};
lookup1("0645") ->	{"0","",""};
lookup1("0646") ->	{"0","",""};
lookup1("0647") ->	{"0","",""};
lookup1("0648") ->	{"0","",""};
lookup1("0649") ->	{"0","",""};
lookup1("064A") ->	{"0","",""};
lookup1("064B") ->	{"27","",""};
lookup1("064C") ->	{"28","",""};
lookup1("064D") ->	{"29","",""};
lookup1("064E") ->	{"30","",""};
lookup1("064F") ->	{"31","",""};
lookup1("0650") ->	{"32","",""};
lookup1("0651") ->	{"33","",""};
lookup1("0652") ->	{"34","",""};
lookup1("0653") ->	{"230","",""};
lookup1("0654") ->	{"230","",""};
lookup1("0655") ->	{"220","",""};
lookup1("0656") ->	{"220","",""};
lookup1("0657") ->	{"230","",""};
lookup1("0658") ->	{"230","",""};
lookup1("0659") ->	{"230","",""};
lookup1("065A") ->	{"230","",""};
lookup1("065B") ->	{"230","",""};
lookup1("065C") ->	{"220","",""};
lookup1("065D") ->	{"230","",""};
lookup1("065E") ->	{"230","",""};
lookup1("065F") ->	{"220","",""};
lookup1("0660") ->	{"0","",""};
lookup1("0661") ->	{"0","",""};
lookup1("0662") ->	{"0","",""};
lookup1("0663") ->	{"0","",""};
lookup1("0664") ->	{"0","",""};
lookup1("0665") ->	{"0","",""};
lookup1("0666") ->	{"0","",""};
lookup1("0667") ->	{"0","",""};
lookup1("0668") ->	{"0","",""};
lookup1("0669") ->	{"0","",""};
lookup1("066A") ->	{"0","",""};
lookup1("066B") ->	{"0","",""};
lookup1("066C") ->	{"0","",""};
lookup1("066D") ->	{"0","",""};
lookup1("066E") ->	{"0","",""};
lookup1("066F") ->	{"0","",""};
lookup1("0670") ->	{"35","",""};
lookup1("0671") ->	{"0","",""};
lookup1("0672") ->	{"0","",""};
lookup1("0673") ->	{"0","",""};
lookup1("0674") ->	{"0","",""};
lookup1("0675") ->	{"0","<compat> 0627 0674",""};
lookup1("0676") ->	{"0","<compat> 0648 0674",""};
lookup1("0677") ->	{"0","<compat> 06C7 0674",""};
lookup1("0678") ->	{"0","<compat> 064A 0674",""};
lookup1("0679") ->	{"0","",""};
lookup1("067A") ->	{"0","",""};
lookup1("067B") ->	{"0","",""};
lookup1("067C") ->	{"0","",""};
lookup1("067D") ->	{"0","",""};
lookup1("067E") ->	{"0","",""};
lookup1("067F") ->	{"0","",""};
lookup1("0680") ->	{"0","",""};
lookup1("0681") ->	{"0","",""};
lookup1("0682") ->	{"0","",""};
lookup1("0683") ->	{"0","",""};
lookup1("0684") ->	{"0","",""};
lookup1("0685") ->	{"0","",""};
lookup1("0686") ->	{"0","",""};
lookup1("0687") ->	{"0","",""};
lookup1("0688") ->	{"0","",""};
lookup1("0689") ->	{"0","",""};
lookup1("068A") ->	{"0","",""};
lookup1("068B") ->	{"0","",""};
lookup1("068C") ->	{"0","",""};
lookup1("068D") ->	{"0","",""};
lookup1("068E") ->	{"0","",""};
lookup1("068F") ->	{"0","",""};
lookup1("0690") ->	{"0","",""};
lookup1("0691") ->	{"0","",""};
lookup1("0692") ->	{"0","",""};
lookup1("0693") ->	{"0","",""};
lookup1("0694") ->	{"0","",""};
lookup1("0695") ->	{"0","",""};
lookup1("0696") ->	{"0","",""};
lookup1("0697") ->	{"0","",""};
lookup1("0698") ->	{"0","",""};
lookup1("0699") ->	{"0","",""};
lookup1("069A") ->	{"0","",""};
lookup1("069B") ->	{"0","",""};
lookup1("069C") ->	{"0","",""};
lookup1("069D") ->	{"0","",""};
lookup1("069E") ->	{"0","",""};
lookup1("069F") ->	{"0","",""};
lookup1("06A0") ->	{"0","",""};
lookup1("06A1") ->	{"0","",""};
lookup1("06A2") ->	{"0","",""};
lookup1("06A3") ->	{"0","",""};
lookup1("06A4") ->	{"0","",""};
lookup1("06A5") ->	{"0","",""};
lookup1("06A6") ->	{"0","",""};
lookup1("06A7") ->	{"0","",""};
lookup1("06A8") ->	{"0","",""};
lookup1("06A9") ->	{"0","",""};
lookup1("06AA") ->	{"0","",""};
lookup1("06AB") ->	{"0","",""};
lookup1("06AC") ->	{"0","",""};
lookup1("06AD") ->	{"0","",""};
lookup1("06AE") ->	{"0","",""};
lookup1("06AF") ->	{"0","",""};
lookup1("06B0") ->	{"0","",""};
lookup1("06B1") ->	{"0","",""};
lookup1("06B2") ->	{"0","",""};
lookup1("06B3") ->	{"0","",""};
lookup1("06B4") ->	{"0","",""};
lookup1("06B5") ->	{"0","",""};
lookup1("06B6") ->	{"0","",""};
lookup1("06B7") ->	{"0","",""};
lookup1("06B8") ->	{"0","",""};
lookup1("06B9") ->	{"0","",""};
lookup1("06BA") ->	{"0","",""};
lookup1("06BB") ->	{"0","",""};
lookup1("06BC") ->	{"0","",""};
lookup1("06BD") ->	{"0","",""};
lookup1("06BE") ->	{"0","",""};
lookup1("06BF") ->	{"0","",""};
lookup1("06C0") ->	{"0","06D5 0654",""};
lookup1("06C1") ->	{"0","",""};
lookup1("06C2") ->	{"0","06C1 0654",""};
lookup1("06C3") ->	{"0","",""};
lookup1("06C4") ->	{"0","",""};
lookup1("06C5") ->	{"0","",""};
lookup1("06C6") ->	{"0","",""};
lookup1("06C7") ->	{"0","",""};
lookup1("06C8") ->	{"0","",""};
lookup1("06C9") ->	{"0","",""};
lookup1("06CA") ->	{"0","",""};
lookup1("06CB") ->	{"0","",""};
lookup1("06CC") ->	{"0","",""};
lookup1("06CD") ->	{"0","",""};
lookup1("06CE") ->	{"0","",""};
lookup1("06CF") ->	{"0","",""};
lookup1("06D0") ->	{"0","",""};
lookup1("06D1") ->	{"0","",""};
lookup1("06D2") ->	{"0","",""};
lookup1("06D3") ->	{"0","06D2 0654",""};
lookup1("06D4") ->	{"0","",""};
lookup1("06D5") ->	{"0","",""};
lookup1("06D6") ->	{"230","",""};
lookup1("06D7") ->	{"230","",""};
lookup1("06D8") ->	{"230","",""};
lookup1("06D9") ->	{"230","",""};
lookup1("06DA") ->	{"230","",""};
lookup1("06DB") ->	{"230","",""};
lookup1("06DC") ->	{"230","",""};
lookup1("06DD") ->	{"0","",""};
lookup1("06DE") ->	{"0","",""};
lookup1("06DF") ->	{"230","",""};
lookup1("06E0") ->	{"230","",""};
lookup1("06E1") ->	{"230","",""};
lookup1("06E2") ->	{"230","",""};
lookup1("06E3") ->	{"220","",""};
lookup1("06E4") ->	{"230","",""};
lookup1("06E5") ->	{"0","",""};
lookup1("06E6") ->	{"0","",""};
lookup1("06E7") ->	{"230","",""};
lookup1("06E8") ->	{"230","",""};
lookup1("06E9") ->	{"0","",""};
lookup1("06EA") ->	{"220","",""};
lookup1("06EB") ->	{"230","",""};
lookup1("06EC") ->	{"230","",""};
lookup1("06ED") ->	{"220","",""};
lookup1("06EE") ->	{"0","",""};
lookup1("06EF") ->	{"0","",""};
lookup1("06F0") ->	{"0","",""};
lookup1("06F1") ->	{"0","",""};
lookup1("06F2") ->	{"0","",""};
lookup1("06F3") ->	{"0","",""};
lookup1("06F4") ->	{"0","",""};
lookup1("06F5") ->	{"0","",""};
lookup1("06F6") ->	{"0","",""};
lookup1("06F7") ->	{"0","",""};
lookup1("06F8") ->	{"0","",""};
lookup1("06F9") ->	{"0","",""};
lookup1("06FA") ->	{"0","",""};
lookup1("06FB") ->	{"0","",""};
lookup1("06FC") ->	{"0","",""};
lookup1("06FD") ->	{"0","",""};
lookup1("06FE") ->	{"0","",""};
lookup1("06FF") ->	{"0","",""};
lookup1("0700") ->	{"0","",""};
lookup1("0701") ->	{"0","",""};
lookup1("0702") ->	{"0","",""};
lookup1("0703") ->	{"0","",""};
lookup1("0704") ->	{"0","",""};
lookup1("0705") ->	{"0","",""};
lookup1("0706") ->	{"0","",""};
lookup1("0707") ->	{"0","",""};
lookup1("0708") ->	{"0","",""};
lookup1("0709") ->	{"0","",""};
lookup1("070A") ->	{"0","",""};
lookup1("070B") ->	{"0","",""};
lookup1("070C") ->	{"0","",""};
lookup1("070D") ->	{"0","",""};
lookup1("070F") ->	{"0","",""};
lookup1("0710") ->	{"0","",""};
lookup1("0711") ->	{"36","",""};
lookup1("0712") ->	{"0","",""};
lookup1("0713") ->	{"0","",""};
lookup1("0714") ->	{"0","",""};
lookup1("0715") ->	{"0","",""};
lookup1("0716") ->	{"0","",""};
lookup1("0717") ->	{"0","",""};
lookup1("0718") ->	{"0","",""};
lookup1("0719") ->	{"0","",""};
lookup1("071A") ->	{"0","",""};
lookup1("071B") ->	{"0","",""};
lookup1("071C") ->	{"0","",""};
lookup1("071D") ->	{"0","",""};
lookup1("071E") ->	{"0","",""};
lookup1("071F") ->	{"0","",""};
lookup1("0720") ->	{"0","",""};
lookup1("0721") ->	{"0","",""};
lookup1("0722") ->	{"0","",""};
lookup1("0723") ->	{"0","",""};
lookup1("0724") ->	{"0","",""};
lookup1("0725") ->	{"0","",""};
lookup1("0726") ->	{"0","",""};
lookup1("0727") ->	{"0","",""};
lookup1("0728") ->	{"0","",""};
lookup1("0729") ->	{"0","",""};
lookup1("072A") ->	{"0","",""};
lookup1("072B") ->	{"0","",""};
lookup1("072C") ->	{"0","",""};
lookup1("072D") ->	{"0","",""};
lookup1("072E") ->	{"0","",""};
lookup1("072F") ->	{"0","",""};
lookup1("0730") ->	{"230","",""};
lookup1("0731") ->	{"220","",""};
lookup1("0732") ->	{"230","",""};
lookup1("0733") ->	{"230","",""};
lookup1("0734") ->	{"220","",""};
lookup1("0735") ->	{"230","",""};
lookup1("0736") ->	{"230","",""};
lookup1("0737") ->	{"220","",""};
lookup1("0738") ->	{"220","",""};
lookup1("0739") ->	{"220","",""};
lookup1("073A") ->	{"230","",""};
lookup1("073B") ->	{"220","",""};
lookup1("073C") ->	{"220","",""};
lookup1("073D") ->	{"230","",""};
lookup1("073E") ->	{"220","",""};
lookup1("073F") ->	{"230","",""};
lookup1("0740") ->	{"230","",""};
lookup1("0741") ->	{"230","",""};
lookup1("0742") ->	{"220","",""};
lookup1("0743") ->	{"230","",""};
lookup1("0744") ->	{"220","",""};
lookup1("0745") ->	{"230","",""};
lookup1("0746") ->	{"220","",""};
lookup1("0747") ->	{"230","",""};
lookup1("0748") ->	{"220","",""};
lookup1("0749") ->	{"230","",""};
lookup1("074A") ->	{"230","",""};
lookup1("074D") ->	{"0","",""};
lookup1("074E") ->	{"0","",""};
lookup1("074F") ->	{"0","",""};
lookup1("0750") ->	{"0","",""};
lookup1("0751") ->	{"0","",""};
lookup1("0752") ->	{"0","",""};
lookup1("0753") ->	{"0","",""};
lookup1("0754") ->	{"0","",""};
lookup1("0755") ->	{"0","",""};
lookup1("0756") ->	{"0","",""};
lookup1("0757") ->	{"0","",""};
lookup1("0758") ->	{"0","",""};
lookup1("0759") ->	{"0","",""};
lookup1("075A") ->	{"0","",""};
lookup1("075B") ->	{"0","",""};
lookup1("075C") ->	{"0","",""};
lookup1("075D") ->	{"0","",""};
lookup1("075E") ->	{"0","",""};
lookup1("075F") ->	{"0","",""};
lookup1("0760") ->	{"0","",""};
lookup1("0761") ->	{"0","",""};
lookup1("0762") ->	{"0","",""};
lookup1("0763") ->	{"0","",""};
lookup1("0764") ->	{"0","",""};
lookup1("0765") ->	{"0","",""};
lookup1("0766") ->	{"0","",""};
lookup1("0767") ->	{"0","",""};
lookup1("0768") ->	{"0","",""};
lookup1("0769") ->	{"0","",""};
lookup1("076A") ->	{"0","",""};
lookup1("076B") ->	{"0","",""};
lookup1("076C") ->	{"0","",""};
lookup1("076D") ->	{"0","",""};
lookup1("076E") ->	{"0","",""};
lookup1("076F") ->	{"0","",""};
lookup1("0770") ->	{"0","",""};
lookup1("0771") ->	{"0","",""};
lookup1("0772") ->	{"0","",""};
lookup1("0773") ->	{"0","",""};
lookup1("0774") ->	{"0","",""};
lookup1("0775") ->	{"0","",""};
lookup1("0776") ->	{"0","",""};
lookup1("0777") ->	{"0","",""};
lookup1("0778") ->	{"0","",""};
lookup1("0779") ->	{"0","",""};
lookup1("077A") ->	{"0","",""};
lookup1("077B") ->	{"0","",""};
lookup1("077C") ->	{"0","",""};
lookup1("077D") ->	{"0","",""};
lookup1("077E") ->	{"0","",""};
lookup1("077F") ->	{"0","",""};
lookup1("0780") ->	{"0","",""};
lookup1("0781") ->	{"0","",""};
lookup1("0782") ->	{"0","",""};
lookup1("0783") ->	{"0","",""};
lookup1("0784") ->	{"0","",""};
lookup1("0785") ->	{"0","",""};
lookup1("0786") ->	{"0","",""};
lookup1("0787") ->	{"0","",""};
lookup1("0788") ->	{"0","",""};
lookup1("0789") ->	{"0","",""};
lookup1("078A") ->	{"0","",""};
lookup1("078B") ->	{"0","",""};
lookup1("078C") ->	{"0","",""};
lookup1("078D") ->	{"0","",""};
lookup1("078E") ->	{"0","",""};
lookup1("078F") ->	{"0","",""};
lookup1("0790") ->	{"0","",""};
lookup1("0791") ->	{"0","",""};
lookup1("0792") ->	{"0","",""};
lookup1("0793") ->	{"0","",""};
lookup1("0794") ->	{"0","",""};
lookup1("0795") ->	{"0","",""};
lookup1("0796") ->	{"0","",""};
lookup1("0797") ->	{"0","",""};
lookup1("0798") ->	{"0","",""};
lookup1("0799") ->	{"0","",""};
lookup1("079A") ->	{"0","",""};
lookup1("079B") ->	{"0","",""};
lookup1("079C") ->	{"0","",""};
lookup1("079D") ->	{"0","",""};
lookup1("079E") ->	{"0","",""};
lookup1("079F") ->	{"0","",""};
lookup1("07A0") ->	{"0","",""};
lookup1("07A1") ->	{"0","",""};
lookup1("07A2") ->	{"0","",""};
lookup1("07A3") ->	{"0","",""};
lookup1("07A4") ->	{"0","",""};
lookup1("07A5") ->	{"0","",""};
lookup1("07A6") ->	{"0","",""};
lookup1("07A7") ->	{"0","",""};
lookup1("07A8") ->	{"0","",""};
lookup1("07A9") ->	{"0","",""};
lookup1("07AA") ->	{"0","",""};
lookup1("07AB") ->	{"0","",""};
lookup1("07AC") ->	{"0","",""};
lookup1("07AD") ->	{"0","",""};
lookup1("07AE") ->	{"0","",""};
lookup1("07AF") ->	{"0","",""};
lookup1("07B0") ->	{"0","",""};
lookup1("07B1") ->	{"0","",""};
lookup1("07C0") ->	{"0","",""};
lookup1("07C1") ->	{"0","",""};
lookup1("07C2") ->	{"0","",""};
lookup1("07C3") ->	{"0","",""};
lookup1("07C4") ->	{"0","",""};
lookup1("07C5") ->	{"0","",""};
lookup1("07C6") ->	{"0","",""};
lookup1("07C7") ->	{"0","",""};
lookup1("07C8") ->	{"0","",""};
lookup1("07C9") ->	{"0","",""};
lookup1("07CA") ->	{"0","",""};
lookup1("07CB") ->	{"0","",""};
lookup1("07CC") ->	{"0","",""};
lookup1("07CD") ->	{"0","",""};
lookup1("07CE") ->	{"0","",""};
lookup1("07CF") ->	{"0","",""};
lookup1("07D0") ->	{"0","",""};
lookup1("07D1") ->	{"0","",""};
lookup1("07D2") ->	{"0","",""};
lookup1("07D3") ->	{"0","",""};
lookup1("07D4") ->	{"0","",""};
lookup1("07D5") ->	{"0","",""};
lookup1("07D6") ->	{"0","",""};
lookup1("07D7") ->	{"0","",""};
lookup1("07D8") ->	{"0","",""};
lookup1("07D9") ->	{"0","",""};
lookup1("07DA") ->	{"0","",""};
lookup1("07DB") ->	{"0","",""};
lookup1("07DC") ->	{"0","",""};
lookup1("07DD") ->	{"0","",""};
lookup1("07DE") ->	{"0","",""};
lookup1("07DF") ->	{"0","",""};
lookup1("07E0") ->	{"0","",""};
lookup1("07E1") ->	{"0","",""};
lookup1("07E2") ->	{"0","",""};
lookup1("07E3") ->	{"0","",""};
lookup1("07E4") ->	{"0","",""};
lookup1("07E5") ->	{"0","",""};
lookup1("07E6") ->	{"0","",""};
lookup1("07E7") ->	{"0","",""};
lookup1("07E8") ->	{"0","",""};
lookup1("07E9") ->	{"0","",""};
lookup1("07EA") ->	{"0","",""};
lookup1("07EB") ->	{"230","",""};
lookup1("07EC") ->	{"230","",""};
lookup1("07ED") ->	{"230","",""};
lookup1("07EE") ->	{"230","",""};
lookup1("07EF") ->	{"230","",""};
lookup1("07F0") ->	{"230","",""};
lookup1("07F1") ->	{"230","",""};
lookup1("07F2") ->	{"220","",""};
lookup1("07F3") ->	{"230","",""};
lookup1("07F4") ->	{"0","",""};
lookup1("07F5") ->	{"0","",""};
lookup1("07F6") ->	{"0","",""};
lookup1("07F7") ->	{"0","",""};
lookup1("07F8") ->	{"0","",""};
lookup1("07F9") ->	{"0","",""};
lookup1("07FA") ->	{"0","",""};
lookup1("0800") ->	{"0","",""};
lookup1("0801") ->	{"0","",""};
lookup1("0802") ->	{"0","",""};
lookup1("0803") ->	{"0","",""};
lookup1("0804") ->	{"0","",""};
lookup1("0805") ->	{"0","",""};
lookup1("0806") ->	{"0","",""};
lookup1("0807") ->	{"0","",""};
lookup1("0808") ->	{"0","",""};
lookup1("0809") ->	{"0","",""};
lookup1("080A") ->	{"0","",""};
lookup1("080B") ->	{"0","",""};
lookup1("080C") ->	{"0","",""};
lookup1("080D") ->	{"0","",""};
lookup1("080E") ->	{"0","",""};
lookup1("080F") ->	{"0","",""};
lookup1("0810") ->	{"0","",""};
lookup1("0811") ->	{"0","",""};
lookup1("0812") ->	{"0","",""};
lookup1("0813") ->	{"0","",""};
lookup1("0814") ->	{"0","",""};
lookup1("0815") ->	{"0","",""};
lookup1("0816") ->	{"230","",""};
lookup1("0817") ->	{"230","",""};
lookup1("0818") ->	{"230","",""};
lookup1("0819") ->	{"230","",""};
lookup1("081A") ->	{"0","",""};
lookup1("081B") ->	{"230","",""};
lookup1("081C") ->	{"230","",""};
lookup1("081D") ->	{"230","",""};
lookup1("081E") ->	{"230","",""};
lookup1("081F") ->	{"230","",""};
lookup1("0820") ->	{"230","",""};
lookup1("0821") ->	{"230","",""};
lookup1("0822") ->	{"230","",""};
lookup1("0823") ->	{"230","",""};
lookup1("0824") ->	{"0","",""};
lookup1("0825") ->	{"230","",""};
lookup1("0826") ->	{"230","",""};
lookup1("0827") ->	{"230","",""};
lookup1("0828") ->	{"0","",""};
lookup1("0829") ->	{"230","",""};
lookup1("082A") ->	{"230","",""};
lookup1("082B") ->	{"230","",""};
lookup1("082C") ->	{"230","",""};
lookup1("082D") ->	{"230","",""};
lookup1("0830") ->	{"0","",""};
lookup1("0831") ->	{"0","",""};
lookup1("0832") ->	{"0","",""};
lookup1("0833") ->	{"0","",""};
lookup1("0834") ->	{"0","",""};
lookup1("0835") ->	{"0","",""};
lookup1("0836") ->	{"0","",""};
lookup1("0837") ->	{"0","",""};
lookup1("0838") ->	{"0","",""};
lookup1("0839") ->	{"0","",""};
lookup1("083A") ->	{"0","",""};
lookup1("083B") ->	{"0","",""};
lookup1("083C") ->	{"0","",""};
lookup1("083D") ->	{"0","",""};
lookup1("083E") ->	{"0","",""};
lookup1("0840") ->	{"0","",""};
lookup1("0841") ->	{"0","",""};
lookup1("0842") ->	{"0","",""};
lookup1("0843") ->	{"0","",""};
lookup1("0844") ->	{"0","",""};
lookup1("0845") ->	{"0","",""};
lookup1("0846") ->	{"0","",""};
lookup1("0847") ->	{"0","",""};
lookup1("0848") ->	{"0","",""};
lookup1("0849") ->	{"0","",""};
lookup1("084A") ->	{"0","",""};
lookup1("084B") ->	{"0","",""};
lookup1("084C") ->	{"0","",""};
lookup1("084D") ->	{"0","",""};
lookup1("084E") ->	{"0","",""};
lookup1("084F") ->	{"0","",""};
lookup1("0850") ->	{"0","",""};
lookup1("0851") ->	{"0","",""};
lookup1("0852") ->	{"0","",""};
lookup1("0853") ->	{"0","",""};
lookup1("0854") ->	{"0","",""};
lookup1("0855") ->	{"0","",""};
lookup1("0856") ->	{"0","",""};
lookup1("0857") ->	{"0","",""};
lookup1("0858") ->	{"0","",""};
lookup1("0859") ->	{"220","",""};
lookup1("085A") ->	{"220","",""};
lookup1("085B") ->	{"220","",""};
lookup1("085E") ->	{"0","",""};
lookup1("08A0") ->	{"0","",""};
lookup1("08A1") ->	{"0","",""};
lookup1("08A2") ->	{"0","",""};
lookup1("08A3") ->	{"0","",""};
lookup1("08A4") ->	{"0","",""};
lookup1("08A5") ->	{"0","",""};
lookup1("08A6") ->	{"0","",""};
lookup1("08A7") ->	{"0","",""};
lookup1("08A8") ->	{"0","",""};
lookup1("08A9") ->	{"0","",""};
lookup1("08AA") ->	{"0","",""};
lookup1("08AB") ->	{"0","",""};
lookup1("08AC") ->	{"0","",""};
lookup1("08AD") ->	{"0","",""};
lookup1("08AE") ->	{"0","",""};
lookup1("08AF") ->	{"0","",""};
lookup1("08B0") ->	{"0","",""};
lookup1("08B1") ->	{"0","",""};
lookup1("08B2") ->	{"0","",""};
lookup1("08E4") ->	{"230","",""};
lookup1("08E5") ->	{"230","",""};
lookup1("08E6") ->	{"220","",""};
lookup1("08E7") ->	{"230","",""};
lookup1("08E8") ->	{"230","",""};
lookup1("08E9") ->	{"220","",""};
lookup1("08EA") ->	{"230","",""};
lookup1("08EB") ->	{"230","",""};
lookup1("08EC") ->	{"230","",""};
lookup1("08ED") ->	{"220","",""};
lookup1("08EE") ->	{"220","",""};
lookup1("08EF") ->	{"220","",""};
lookup1("08F0") ->	{"27","",""};
lookup1("08F1") ->	{"28","",""};
lookup1("08F2") ->	{"29","",""};
lookup1("08F3") ->	{"230","",""};
lookup1("08F4") ->	{"230","",""};
lookup1("08F5") ->	{"230","",""};
lookup1("08F6") ->	{"220","",""};
lookup1("08F7") ->	{"230","",""};
lookup1("08F8") ->	{"230","",""};
lookup1("08F9") ->	{"220","",""};
lookup1("08FA") ->	{"220","",""};
lookup1("08FB") ->	{"230","",""};
lookup1("08FC") ->	{"230","",""};
lookup1("08FD") ->	{"230","",""};
lookup1("08FE") ->	{"230","",""};
lookup1("08FF") ->	{"230","",""};
lookup1("0900") ->	{"0","",""};
lookup1("0901") ->	{"0","",""};
lookup1("0902") ->	{"0","",""};
lookup1("0903") ->	{"0","",""};
lookup1("0904") ->	{"0","",""};
lookup1("0905") ->	{"0","",""};
lookup1("0906") ->	{"0","",""};
lookup1("0907") ->	{"0","",""};
lookup1("0908") ->	{"0","",""};
lookup1("0909") ->	{"0","",""};
lookup1("090A") ->	{"0","",""};
lookup1("090B") ->	{"0","",""};
lookup1("090C") ->	{"0","",""};
lookup1("090D") ->	{"0","",""};
lookup1("090E") ->	{"0","",""};
lookup1("090F") ->	{"0","",""};
lookup1("0910") ->	{"0","",""};
lookup1("0911") ->	{"0","",""};
lookup1("0912") ->	{"0","",""};
lookup1("0913") ->	{"0","",""};
lookup1("0914") ->	{"0","",""};
lookup1("0915") ->	{"0","",""};
lookup1("0916") ->	{"0","",""};
lookup1("0917") ->	{"0","",""};
lookup1("0918") ->	{"0","",""};
lookup1("0919") ->	{"0","",""};
lookup1("091A") ->	{"0","",""};
lookup1("091B") ->	{"0","",""};
lookup1("091C") ->	{"0","",""};
lookup1("091D") ->	{"0","",""};
lookup1("091E") ->	{"0","",""};
lookup1("091F") ->	{"0","",""};
lookup1("0920") ->	{"0","",""};
lookup1("0921") ->	{"0","",""};
lookup1("0922") ->	{"0","",""};
lookup1("0923") ->	{"0","",""};
lookup1("0924") ->	{"0","",""};
lookup1("0925") ->	{"0","",""};
lookup1("0926") ->	{"0","",""};
lookup1("0927") ->	{"0","",""};
lookup1("0928") ->	{"0","",""};
lookup1("0929") ->	{"0","0928 093C",""};
lookup1("092A") ->	{"0","",""};
lookup1("092B") ->	{"0","",""};
lookup1("092C") ->	{"0","",""};
lookup1("092D") ->	{"0","",""};
lookup1("092E") ->	{"0","",""};
lookup1("092F") ->	{"0","",""};
lookup1("0930") ->	{"0","",""};
lookup1("0931") ->	{"0","0930 093C",""};
lookup1("0932") ->	{"0","",""};
lookup1("0933") ->	{"0","",""};
lookup1("0934") ->	{"0","0933 093C",""};
lookup1("0935") ->	{"0","",""};
lookup1("0936") ->	{"0","",""};
lookup1("0937") ->	{"0","",""};
lookup1("0938") ->	{"0","",""};
lookup1("0939") ->	{"0","",""};
lookup1("093A") ->	{"0","",""};
lookup1("093B") ->	{"0","",""};
lookup1("093C") ->	{"7","",""};
lookup1("093D") ->	{"0","",""};
lookup1("093E") ->	{"0","",""};
lookup1("093F") ->	{"0","",""};
lookup1("0940") ->	{"0","",""};
lookup1("0941") ->	{"0","",""};
lookup1("0942") ->	{"0","",""};
lookup1("0943") ->	{"0","",""};
lookup1("0944") ->	{"0","",""};
lookup1("0945") ->	{"0","",""};
lookup1("0946") ->	{"0","",""};
lookup1("0947") ->	{"0","",""};
lookup1("0948") ->	{"0","",""};
lookup1("0949") ->	{"0","",""};
lookup1("094A") ->	{"0","",""};
lookup1("094B") ->	{"0","",""};
lookup1("094C") ->	{"0","",""};
lookup1("094D") ->	{"9","",""};
lookup1("094E") ->	{"0","",""};
lookup1("094F") ->	{"0","",""};
lookup1("0950") ->	{"0","",""};
lookup1("0951") ->	{"230","",""};
lookup1("0952") ->	{"220","",""};
lookup1("0953") ->	{"230","",""};
lookup1("0954") ->	{"230","",""};
lookup1("0955") ->	{"0","",""};
lookup1("0956") ->	{"0","",""};
lookup1("0957") ->	{"0","",""};
lookup1("0958") ->	{"0","0915 093C",""};
lookup1("0959") ->	{"0","0916 093C",""};
lookup1("095A") ->	{"0","0917 093C",""};
lookup1("095B") ->	{"0","091C 093C",""};
lookup1("095C") ->	{"0","0921 093C",""};
lookup1("095D") ->	{"0","0922 093C",""};
lookup1("095E") ->	{"0","092B 093C",""};
lookup1("095F") ->	{"0","092F 093C",""};
lookup1("0960") ->	{"0","",""};
lookup1("0961") ->	{"0","",""};
lookup1("0962") ->	{"0","",""};
lookup1("0963") ->	{"0","",""};
lookup1("0964") ->	{"0","",""};
lookup1("0965") ->	{"0","",""};
lookup1("0966") ->	{"0","",""};
lookup1("0967") ->	{"0","",""};
lookup1("0968") ->	{"0","",""};
lookup1("0969") ->	{"0","",""};
lookup1("096A") ->	{"0","",""};
lookup1("096B") ->	{"0","",""};
lookup1("096C") ->	{"0","",""};
lookup1("096D") ->	{"0","",""};
lookup1("096E") ->	{"0","",""};
lookup1("096F") ->	{"0","",""};
lookup1("0970") ->	{"0","",""};
lookup1("0971") ->	{"0","",""};
lookup1("0972") ->	{"0","",""};
lookup1("0973") ->	{"0","",""};
lookup1("0974") ->	{"0","",""};
lookup1("0975") ->	{"0","",""};
lookup1("0976") ->	{"0","",""};
lookup1("0977") ->	{"0","",""};
lookup1("0978") ->	{"0","",""};
lookup1("0979") ->	{"0","",""};
lookup1("097A") ->	{"0","",""};
lookup1("097B") ->	{"0","",""};
lookup1("097C") ->	{"0","",""};
lookup1("097D") ->	{"0","",""};
lookup1("097E") ->	{"0","",""};
lookup1("097F") ->	{"0","",""};
lookup1("0980") ->	{"0","",""};
lookup1("0981") ->	{"0","",""};
lookup1("0982") ->	{"0","",""};
lookup1("0983") ->	{"0","",""};
lookup1("0985") ->	{"0","",""};
lookup1("0986") ->	{"0","",""};
lookup1("0987") ->	{"0","",""};
lookup1("0988") ->	{"0","",""};
lookup1("0989") ->	{"0","",""};
lookup1("098A") ->	{"0","",""};
lookup1("098B") ->	{"0","",""};
lookup1("098C") ->	{"0","",""};
lookup1("098F") ->	{"0","",""};
lookup1("0990") ->	{"0","",""};
lookup1("0993") ->	{"0","",""};
lookup1("0994") ->	{"0","",""};
lookup1("0995") ->	{"0","",""};
lookup1("0996") ->	{"0","",""};
lookup1("0997") ->	{"0","",""};
lookup1("0998") ->	{"0","",""};
lookup1("0999") ->	{"0","",""};
lookup1("099A") ->	{"0","",""};
lookup1("099B") ->	{"0","",""};
lookup1("099C") ->	{"0","",""};
lookup1("099D") ->	{"0","",""};
lookup1("099E") ->	{"0","",""};
lookup1("099F") ->	{"0","",""};
lookup1("09A0") ->	{"0","",""};
lookup1("09A1") ->	{"0","",""};
lookup1("09A2") ->	{"0","",""};
lookup1("09A3") ->	{"0","",""};
lookup1("09A4") ->	{"0","",""};
lookup1("09A5") ->	{"0","",""};
lookup1("09A6") ->	{"0","",""};
lookup1("09A7") ->	{"0","",""};
lookup1("09A8") ->	{"0","",""};
lookup1("09AA") ->	{"0","",""};
lookup1("09AB") ->	{"0","",""};
lookup1("09AC") ->	{"0","",""};
lookup1("09AD") ->	{"0","",""};
lookup1("09AE") ->	{"0","",""};
lookup1("09AF") ->	{"0","",""};
lookup1("09B0") ->	{"0","",""};
lookup1("09B2") ->	{"0","",""};
lookup1("09B6") ->	{"0","",""};
lookup1("09B7") ->	{"0","",""};
lookup1("09B8") ->	{"0","",""};
lookup1("09B9") ->	{"0","",""};
lookup1("09BC") ->	{"7","",""};
lookup1("09BD") ->	{"0","",""};
lookup1("09BE") ->	{"0","",""};
lookup1("09BF") ->	{"0","",""};
lookup1("09C0") ->	{"0","",""};
lookup1("09C1") ->	{"0","",""};
lookup1("09C2") ->	{"0","",""};
lookup1("09C3") ->	{"0","",""};
lookup1("09C4") ->	{"0","",""};
lookup1("09C7") ->	{"0","",""};
lookup1("09C8") ->	{"0","",""};
lookup1("09CB") ->	{"0","09C7 09BE",""};
lookup1("09CC") ->	{"0","09C7 09D7",""};
lookup1("09CD") ->	{"9","",""};
lookup1("09CE") ->	{"0","",""};
lookup1("09D7") ->	{"0","",""};
lookup1("09DC") ->	{"0","09A1 09BC",""};
lookup1("09DD") ->	{"0","09A2 09BC",""};
lookup1("09DF") ->	{"0","09AF 09BC",""};
lookup1("09E0") ->	{"0","",""};
lookup1("09E1") ->	{"0","",""};
lookup1("09E2") ->	{"0","",""};
lookup1("09E3") ->	{"0","",""};
lookup1("09E6") ->	{"0","",""};
lookup1("09E7") ->	{"0","",""};
lookup1("09E8") ->	{"0","",""};
lookup1("09E9") ->	{"0","",""};
lookup1("09EA") ->	{"0","",""};
lookup1("09EB") ->	{"0","",""};
lookup1("09EC") ->	{"0","",""};
lookup1("09ED") ->	{"0","",""};
lookup1("09EE") ->	{"0","",""};
lookup1("09EF") ->	{"0","",""};
lookup1("09F0") ->	{"0","",""};
lookup1("09F1") ->	{"0","",""};
lookup1("09F2") ->	{"0","",""};
lookup1("09F3") ->	{"0","",""};
lookup1("09F4") ->	{"0","",""};
lookup1("09F5") ->	{"0","",""};
lookup1("09F6") ->	{"0","",""};
lookup1("09F7") ->	{"0","",""};
lookup1("09F8") ->	{"0","",""};
lookup1("09F9") ->	{"0","",""};
lookup1("09FA") ->	{"0","",""};
lookup1("09FB") ->	{"0","",""};
lookup1("0A01") ->	{"0","",""};
lookup1("0A02") ->	{"0","",""};
lookup1("0A03") ->	{"0","",""};
lookup1("0A05") ->	{"0","",""};
lookup1("0A06") ->	{"0","",""};
lookup1("0A07") ->	{"0","",""};
lookup1("0A08") ->	{"0","",""};
lookup1("0A09") ->	{"0","",""};
lookup1("0A0A") ->	{"0","",""};
lookup1("0A0F") ->	{"0","",""};
lookup1("0A10") ->	{"0","",""};
lookup1("0A13") ->	{"0","",""};
lookup1("0A14") ->	{"0","",""};
lookup1("0A15") ->	{"0","",""};
lookup1("0A16") ->	{"0","",""};
lookup1("0A17") ->	{"0","",""};
lookup1("0A18") ->	{"0","",""};
lookup1("0A19") ->	{"0","",""};
lookup1("0A1A") ->	{"0","",""};
lookup1("0A1B") ->	{"0","",""};
lookup1("0A1C") ->	{"0","",""};
lookup1("0A1D") ->	{"0","",""};
lookup1("0A1E") ->	{"0","",""};
lookup1("0A1F") ->	{"0","",""};
lookup1("0A20") ->	{"0","",""};
lookup1("0A21") ->	{"0","",""};
lookup1("0A22") ->	{"0","",""};
lookup1("0A23") ->	{"0","",""};
lookup1("0A24") ->	{"0","",""};
lookup1("0A25") ->	{"0","",""};
lookup1("0A26") ->	{"0","",""};
lookup1("0A27") ->	{"0","",""};
lookup1("0A28") ->	{"0","",""};
lookup1("0A2A") ->	{"0","",""};
lookup1("0A2B") ->	{"0","",""};
lookup1("0A2C") ->	{"0","",""};
lookup1("0A2D") ->	{"0","",""};
lookup1("0A2E") ->	{"0","",""};
lookup1("0A2F") ->	{"0","",""};
lookup1("0A30") ->	{"0","",""};
lookup1("0A32") ->	{"0","",""};
lookup1("0A33") ->	{"0","0A32 0A3C",""};
lookup1("0A35") ->	{"0","",""};
lookup1("0A36") ->	{"0","0A38 0A3C",""};
lookup1("0A38") ->	{"0","",""};
lookup1("0A39") ->	{"0","",""};
lookup1("0A3C") ->	{"7","",""};
lookup1("0A3E") ->	{"0","",""};
lookup1("0A3F") ->	{"0","",""};
lookup1("0A40") ->	{"0","",""};
lookup1("0A41") ->	{"0","",""};
lookup1("0A42") ->	{"0","",""};
lookup1("0A47") ->	{"0","",""};
lookup1("0A48") ->	{"0","",""};
lookup1("0A4B") ->	{"0","",""};
lookup1("0A4C") ->	{"0","",""};
lookup1("0A4D") ->	{"9","",""};
lookup1("0A51") ->	{"0","",""};
lookup1("0A59") ->	{"0","0A16 0A3C",""};
lookup1("0A5A") ->	{"0","0A17 0A3C",""};
lookup1("0A5B") ->	{"0","0A1C 0A3C",""};
lookup1("0A5C") ->	{"0","",""};
lookup1("0A5E") ->	{"0","0A2B 0A3C",""};
lookup1("0A66") ->	{"0","",""};
lookup1("0A67") ->	{"0","",""};
lookup1("0A68") ->	{"0","",""};
lookup1("0A69") ->	{"0","",""};
lookup1("0A6A") ->	{"0","",""};
lookup1("0A6B") ->	{"0","",""};
lookup1("0A6C") ->	{"0","",""};
lookup1("0A6D") ->	{"0","",""};
lookup1("0A6E") ->	{"0","",""};
lookup1("0A6F") ->	{"0","",""};
lookup1("0A70") ->	{"0","",""};
lookup1("0A71") ->	{"0","",""};
lookup1("0A72") ->	{"0","",""};
lookup1("0A73") ->	{"0","",""};
lookup1("0A74") ->	{"0","",""};
lookup1("0A75") ->	{"0","",""};
lookup1("0A81") ->	{"0","",""};
lookup1("0A82") ->	{"0","",""};
lookup1("0A83") ->	{"0","",""};
lookup1("0A85") ->	{"0","",""};
lookup1("0A86") ->	{"0","",""};
lookup1("0A87") ->	{"0","",""};
lookup1("0A88") ->	{"0","",""};
lookup1("0A89") ->	{"0","",""};
lookup1("0A8A") ->	{"0","",""};
lookup1("0A8B") ->	{"0","",""};
lookup1("0A8C") ->	{"0","",""};
lookup1("0A8D") ->	{"0","",""};
lookup1("0A8F") ->	{"0","",""};
lookup1("0A90") ->	{"0","",""};
lookup1("0A91") ->	{"0","",""};
lookup1("0A93") ->	{"0","",""};
lookup1("0A94") ->	{"0","",""};
lookup1("0A95") ->	{"0","",""};
lookup1("0A96") ->	{"0","",""};
lookup1("0A97") ->	{"0","",""};
lookup1("0A98") ->	{"0","",""};
lookup1("0A99") ->	{"0","",""};
lookup1("0A9A") ->	{"0","",""};
lookup1("0A9B") ->	{"0","",""};
lookup1("0A9C") ->	{"0","",""};
lookup1("0A9D") ->	{"0","",""};
lookup1("0A9E") ->	{"0","",""};
lookup1("0A9F") ->	{"0","",""};
lookup1("0AA0") ->	{"0","",""};
lookup1("0AA1") ->	{"0","",""};
lookup1("0AA2") ->	{"0","",""};
lookup1("0AA3") ->	{"0","",""};
lookup1("0AA4") ->	{"0","",""};
lookup1("0AA5") ->	{"0","",""};
lookup1("0AA6") ->	{"0","",""};
lookup1("0AA7") ->	{"0","",""};
lookup1("0AA8") ->	{"0","",""};
lookup1("0AAA") ->	{"0","",""};
lookup1("0AAB") ->	{"0","",""};
lookup1("0AAC") ->	{"0","",""};
lookup1("0AAD") ->	{"0","",""};
lookup1("0AAE") ->	{"0","",""};
lookup1("0AAF") ->	{"0","",""};
lookup1("0AB0") ->	{"0","",""};
lookup1("0AB2") ->	{"0","",""};
lookup1("0AB3") ->	{"0","",""};
lookup1("0AB5") ->	{"0","",""};
lookup1("0AB6") ->	{"0","",""};
lookup1("0AB7") ->	{"0","",""};
lookup1("0AB8") ->	{"0","",""};
lookup1("0AB9") ->	{"0","",""};
lookup1("0ABC") ->	{"7","",""};
lookup1("0ABD") ->	{"0","",""};
lookup1("0ABE") ->	{"0","",""};
lookup1("0ABF") ->	{"0","",""};
lookup1("0AC0") ->	{"0","",""};
lookup1("0AC1") ->	{"0","",""};
lookup1("0AC2") ->	{"0","",""};
lookup1("0AC3") ->	{"0","",""};
lookup1("0AC4") ->	{"0","",""};
lookup1("0AC5") ->	{"0","",""};
lookup1("0AC7") ->	{"0","",""};
lookup1("0AC8") ->	{"0","",""};
lookup1("0AC9") ->	{"0","",""};
lookup1("0ACB") ->	{"0","",""};
lookup1("0ACC") ->	{"0","",""};
lookup1("0ACD") ->	{"9","",""};
lookup1("0AD0") ->	{"0","",""};
lookup1("0AE0") ->	{"0","",""};
lookup1("0AE1") ->	{"0","",""};
lookup1("0AE2") ->	{"0","",""};
lookup1("0AE3") ->	{"0","",""};
lookup1("0AE6") ->	{"0","",""};
lookup1("0AE7") ->	{"0","",""};
lookup1("0AE8") ->	{"0","",""};
lookup1("0AE9") ->	{"0","",""};
lookup1("0AEA") ->	{"0","",""};
lookup1("0AEB") ->	{"0","",""};
lookup1("0AEC") ->	{"0","",""};
lookup1("0AED") ->	{"0","",""};
lookup1("0AEE") ->	{"0","",""};
lookup1("0AEF") ->	{"0","",""};
lookup1("0AF0") ->	{"0","",""};
lookup1("0AF1") ->	{"0","",""};
lookup1("0B01") ->	{"0","",""};
lookup1("0B02") ->	{"0","",""};
lookup1("0B03") ->	{"0","",""};
lookup1("0B05") ->	{"0","",""};
lookup1("0B06") ->	{"0","",""};
lookup1("0B07") ->	{"0","",""};
lookup1("0B08") ->	{"0","",""};
lookup1("0B09") ->	{"0","",""};
lookup1("0B0A") ->	{"0","",""};
lookup1("0B0B") ->	{"0","",""};
lookup1("0B0C") ->	{"0","",""};
lookup1("0B0F") ->	{"0","",""};
lookup1("0B10") ->	{"0","",""};
lookup1("0B13") ->	{"0","",""};
lookup1("0B14") ->	{"0","",""};
lookup1("0B15") ->	{"0","",""};
lookup1("0B16") ->	{"0","",""};
lookup1("0B17") ->	{"0","",""};
lookup1("0B18") ->	{"0","",""};
lookup1("0B19") ->	{"0","",""};
lookup1("0B1A") ->	{"0","",""};
lookup1("0B1B") ->	{"0","",""};
lookup1("0B1C") ->	{"0","",""};
lookup1("0B1D") ->	{"0","",""};
lookup1("0B1E") ->	{"0","",""};
lookup1("0B1F") ->	{"0","",""};
lookup1("0B20") ->	{"0","",""};
lookup1("0B21") ->	{"0","",""};
lookup1("0B22") ->	{"0","",""};
lookup1("0B23") ->	{"0","",""};
lookup1("0B24") ->	{"0","",""};
lookup1("0B25") ->	{"0","",""};
lookup1("0B26") ->	{"0","",""};
lookup1("0B27") ->	{"0","",""};
lookup1("0B28") ->	{"0","",""};
lookup1("0B2A") ->	{"0","",""};
lookup1("0B2B") ->	{"0","",""};
lookup1("0B2C") ->	{"0","",""};
lookup1("0B2D") ->	{"0","",""};
lookup1("0B2E") ->	{"0","",""};
lookup1("0B2F") ->	{"0","",""};
lookup1("0B30") ->	{"0","",""};
lookup1("0B32") ->	{"0","",""};
lookup1("0B33") ->	{"0","",""};
lookup1("0B35") ->	{"0","",""};
lookup1("0B36") ->	{"0","",""};
lookup1("0B37") ->	{"0","",""};
lookup1("0B38") ->	{"0","",""};
lookup1("0B39") ->	{"0","",""};
lookup1("0B3C") ->	{"7","",""};
lookup1("0B3D") ->	{"0","",""};
lookup1("0B3E") ->	{"0","",""};
lookup1("0B3F") ->	{"0","",""};
lookup1("0B40") ->	{"0","",""};
lookup1("0B41") ->	{"0","",""};
lookup1("0B42") ->	{"0","",""};
lookup1("0B43") ->	{"0","",""};
lookup1("0B44") ->	{"0","",""};
lookup1("0B47") ->	{"0","",""};
lookup1("0B48") ->	{"0","0B47 0B56",""};
lookup1("0B4B") ->	{"0","0B47 0B3E",""};
lookup1("0B4C") ->	{"0","0B47 0B57",""};
lookup1("0B4D") ->	{"9","",""};
lookup1("0B56") ->	{"0","",""};
lookup1("0B57") ->	{"0","",""};
lookup1("0B5C") ->	{"0","0B21 0B3C",""};
lookup1("0B5D") ->	{"0","0B22 0B3C",""};
lookup1("0B5F") ->	{"0","",""};
lookup1("0B60") ->	{"0","",""};
lookup1("0B61") ->	{"0","",""};
lookup1("0B62") ->	{"0","",""};
lookup1("0B63") ->	{"0","",""};
lookup1("0B66") ->	{"0","",""};
lookup1("0B67") ->	{"0","",""};
lookup1("0B68") ->	{"0","",""};
lookup1("0B69") ->	{"0","",""};
lookup1("0B6A") ->	{"0","",""};
lookup1("0B6B") ->	{"0","",""};
lookup1("0B6C") ->	{"0","",""};
lookup1("0B6D") ->	{"0","",""};
lookup1("0B6E") ->	{"0","",""};
lookup1("0B6F") ->	{"0","",""};
lookup1("0B70") ->	{"0","",""};
lookup1("0B71") ->	{"0","",""};
lookup1("0B72") ->	{"0","",""};
lookup1("0B73") ->	{"0","",""};
lookup1("0B74") ->	{"0","",""};
lookup1("0B75") ->	{"0","",""};
lookup1("0B76") ->	{"0","",""};
lookup1("0B77") ->	{"0","",""};
lookup1("0B82") ->	{"0","",""};
lookup1("0B83") ->	{"0","",""};
lookup1("0B85") ->	{"0","",""};
lookup1("0B86") ->	{"0","",""};
lookup1("0B87") ->	{"0","",""};
lookup1("0B88") ->	{"0","",""};
lookup1("0B89") ->	{"0","",""};
lookup1("0B8A") ->	{"0","",""};
lookup1("0B8E") ->	{"0","",""};
lookup1("0B8F") ->	{"0","",""};
lookup1("0B90") ->	{"0","",""};
lookup1("0B92") ->	{"0","",""};
lookup1("0B93") ->	{"0","",""};
lookup1("0B94") ->	{"0","0B92 0BD7",""};
lookup1("0B95") ->	{"0","",""};
lookup1("0B99") ->	{"0","",""};
lookup1("0B9A") ->	{"0","",""};
lookup1("0B9C") ->	{"0","",""};
lookup1("0B9E") ->	{"0","",""};
lookup1("0B9F") ->	{"0","",""};
lookup1("0BA3") ->	{"0","",""};
lookup1("0BA4") ->	{"0","",""};
lookup1("0BA8") ->	{"0","",""};
lookup1("0BA9") ->	{"0","",""};
lookup1("0BAA") ->	{"0","",""};
lookup1("0BAE") ->	{"0","",""};
lookup1("0BAF") ->	{"0","",""};
lookup1("0BB0") ->	{"0","",""};
lookup1("0BB1") ->	{"0","",""};
lookup1("0BB2") ->	{"0","",""};
lookup1("0BB3") ->	{"0","",""};
lookup1("0BB4") ->	{"0","",""};
lookup1("0BB5") ->	{"0","",""};
lookup1("0BB6") ->	{"0","",""};
lookup1("0BB7") ->	{"0","",""};
lookup1("0BB8") ->	{"0","",""};
lookup1("0BB9") ->	{"0","",""};
lookup1("0BBE") ->	{"0","",""};
lookup1("0BBF") ->	{"0","",""};
lookup1("0BC0") ->	{"0","",""};
lookup1("0BC1") ->	{"0","",""};
lookup1("0BC2") ->	{"0","",""};
lookup1("0BC6") ->	{"0","",""};
lookup1("0BC7") ->	{"0","",""};
lookup1("0BC8") ->	{"0","",""};
lookup1("0BCA") ->	{"0","0BC6 0BBE",""};
lookup1("0BCB") ->	{"0","0BC7 0BBE",""};
lookup1("0BCC") ->	{"0","0BC6 0BD7",""};
lookup1("0BCD") ->	{"9","",""};
lookup1("0BD0") ->	{"0","",""};
lookup1("0BD7") ->	{"0","",""};
lookup1("0BE6") ->	{"0","",""};
lookup1("0BE7") ->	{"0","",""};
lookup1("0BE8") ->	{"0","",""};
lookup1("0BE9") ->	{"0","",""};
lookup1("0BEA") ->	{"0","",""};
lookup1("0BEB") ->	{"0","",""};
lookup1("0BEC") ->	{"0","",""};
lookup1("0BED") ->	{"0","",""};
lookup1("0BEE") ->	{"0","",""};
lookup1("0BEF") ->	{"0","",""};
lookup1("0BF0") ->	{"0","",""};
lookup1("0BF1") ->	{"0","",""};
lookup1("0BF2") ->	{"0","",""};
lookup1("0BF3") ->	{"0","",""};
lookup1("0BF4") ->	{"0","",""};
lookup1("0BF5") ->	{"0","",""};
lookup1("0BF6") ->	{"0","",""};
lookup1("0BF7") ->	{"0","",""};
lookup1("0BF8") ->	{"0","",""};
lookup1("0BF9") ->	{"0","",""};
lookup1("0BFA") ->	{"0","",""};
lookup1("0C00") ->	{"0","",""};
lookup1("0C01") ->	{"0","",""};
lookup1("0C02") ->	{"0","",""};
lookup1("0C03") ->	{"0","",""};
lookup1("0C05") ->	{"0","",""};
lookup1("0C06") ->	{"0","",""};
lookup1("0C07") ->	{"0","",""};
lookup1("0C08") ->	{"0","",""};
lookup1("0C09") ->	{"0","",""};
lookup1("0C0A") ->	{"0","",""};
lookup1("0C0B") ->	{"0","",""};
lookup1("0C0C") ->	{"0","",""};
lookup1("0C0E") ->	{"0","",""};
lookup1("0C0F") ->	{"0","",""};
lookup1("0C10") ->	{"0","",""};
lookup1("0C12") ->	{"0","",""};
lookup1("0C13") ->	{"0","",""};
lookup1("0C14") ->	{"0","",""};
lookup1("0C15") ->	{"0","",""};
lookup1("0C16") ->	{"0","",""};
lookup1("0C17") ->	{"0","",""};
lookup1("0C18") ->	{"0","",""};
lookup1("0C19") ->	{"0","",""};
lookup1("0C1A") ->	{"0","",""};
lookup1("0C1B") ->	{"0","",""};
lookup1("0C1C") ->	{"0","",""};
lookup1("0C1D") ->	{"0","",""};
lookup1("0C1E") ->	{"0","",""};
lookup1("0C1F") ->	{"0","",""};
lookup1("0C20") ->	{"0","",""};
lookup1("0C21") ->	{"0","",""};
lookup1("0C22") ->	{"0","",""};
lookup1("0C23") ->	{"0","",""};
lookup1("0C24") ->	{"0","",""};
lookup1("0C25") ->	{"0","",""};
lookup1("0C26") ->	{"0","",""};
lookup1("0C27") ->	{"0","",""};
lookup1("0C28") ->	{"0","",""};
lookup1("0C2A") ->	{"0","",""};
lookup1("0C2B") ->	{"0","",""};
lookup1("0C2C") ->	{"0","",""};
lookup1("0C2D") ->	{"0","",""};
lookup1("0C2E") ->	{"0","",""};
lookup1("0C2F") ->	{"0","",""};
lookup1("0C30") ->	{"0","",""};
lookup1("0C31") ->	{"0","",""};
lookup1("0C32") ->	{"0","",""};
lookup1("0C33") ->	{"0","",""};
lookup1("0C34") ->	{"0","",""};
lookup1("0C35") ->	{"0","",""};
lookup1("0C36") ->	{"0","",""};
lookup1("0C37") ->	{"0","",""};
lookup1("0C38") ->	{"0","",""};
lookup1("0C39") ->	{"0","",""};
lookup1("0C3D") ->	{"0","",""};
lookup1("0C3E") ->	{"0","",""};
lookup1("0C3F") ->	{"0","",""};
lookup1("0C40") ->	{"0","",""};
lookup1("0C41") ->	{"0","",""};
lookup1("0C42") ->	{"0","",""};
lookup1("0C43") ->	{"0","",""};
lookup1("0C44") ->	{"0","",""};
lookup1("0C46") ->	{"0","",""};
lookup1("0C47") ->	{"0","",""};
lookup1("0C48") ->	{"0","0C46 0C56",""};
lookup1("0C4A") ->	{"0","",""};
lookup1("0C4B") ->	{"0","",""};
lookup1("0C4C") ->	{"0","",""};
lookup1("0C4D") ->	{"9","",""};
lookup1("0C55") ->	{"84","",""};
lookup1("0C56") ->	{"91","",""};
lookup1("0C58") ->	{"0","",""};
lookup1("0C59") ->	{"0","",""};
lookup1("0C60") ->	{"0","",""};
lookup1("0C61") ->	{"0","",""};
lookup1("0C62") ->	{"0","",""};
lookup1("0C63") ->	{"0","",""};
lookup1("0C66") ->	{"0","",""};
lookup1("0C67") ->	{"0","",""};
lookup1("0C68") ->	{"0","",""};
lookup1("0C69") ->	{"0","",""};
lookup1("0C6A") ->	{"0","",""};
lookup1("0C6B") ->	{"0","",""};
lookup1("0C6C") ->	{"0","",""};
lookup1("0C6D") ->	{"0","",""};
lookup1("0C6E") ->	{"0","",""};
lookup1("0C6F") ->	{"0","",""};
lookup1("0C78") ->	{"0","",""};
lookup1("0C79") ->	{"0","",""};
lookup1("0C7A") ->	{"0","",""};
lookup1("0C7B") ->	{"0","",""};
lookup1("0C7C") ->	{"0","",""};
lookup1("0C7D") ->	{"0","",""};
lookup1("0C7E") ->	{"0","",""};
lookup1("0C7F") ->	{"0","",""};
lookup1("0C81") ->	{"0","",""};
lookup1("0C82") ->	{"0","",""};
lookup1("0C83") ->	{"0","",""};
lookup1("0C85") ->	{"0","",""};
lookup1("0C86") ->	{"0","",""};
lookup1("0C87") ->	{"0","",""};
lookup1("0C88") ->	{"0","",""};
lookup1("0C89") ->	{"0","",""};
lookup1("0C8A") ->	{"0","",""};
lookup1("0C8B") ->	{"0","",""};
lookup1("0C8C") ->	{"0","",""};
lookup1("0C8E") ->	{"0","",""};
lookup1("0C8F") ->	{"0","",""};
lookup1("0C90") ->	{"0","",""};
lookup1("0C92") ->	{"0","",""};
lookup1("0C93") ->	{"0","",""};
lookup1("0C94") ->	{"0","",""};
lookup1("0C95") ->	{"0","",""};
lookup1("0C96") ->	{"0","",""};
lookup1("0C97") ->	{"0","",""};
lookup1("0C98") ->	{"0","",""};
lookup1("0C99") ->	{"0","",""};
lookup1("0C9A") ->	{"0","",""};
lookup1("0C9B") ->	{"0","",""};
lookup1("0C9C") ->	{"0","",""};
lookup1("0C9D") ->	{"0","",""};
lookup1("0C9E") ->	{"0","",""};
lookup1("0C9F") ->	{"0","",""};
lookup1("0CA0") ->	{"0","",""};
lookup1("0CA1") ->	{"0","",""};
lookup1("0CA2") ->	{"0","",""};
lookup1("0CA3") ->	{"0","",""};
lookup1("0CA4") ->	{"0","",""};
lookup1("0CA5") ->	{"0","",""};
lookup1("0CA6") ->	{"0","",""};
lookup1("0CA7") ->	{"0","",""};
lookup1("0CA8") ->	{"0","",""};
lookup1("0CAA") ->	{"0","",""};
lookup1("0CAB") ->	{"0","",""};
lookup1("0CAC") ->	{"0","",""};
lookup1("0CAD") ->	{"0","",""};
lookup1("0CAE") ->	{"0","",""};
lookup1("0CAF") ->	{"0","",""};
lookup1("0CB0") ->	{"0","",""};
lookup1("0CB1") ->	{"0","",""};
lookup1("0CB2") ->	{"0","",""};
lookup1("0CB3") ->	{"0","",""};
lookup1("0CB5") ->	{"0","",""};
lookup1("0CB6") ->	{"0","",""};
lookup1("0CB7") ->	{"0","",""};
lookup1("0CB8") ->	{"0","",""};
lookup1("0CB9") ->	{"0","",""};
lookup1("0CBC") ->	{"7","",""};
lookup1("0CBD") ->	{"0","",""};
lookup1("0CBE") ->	{"0","",""};
lookup1("0CBF") ->	{"0","",""};
lookup1("0CC0") ->	{"0","0CBF 0CD5",""};
lookup1("0CC1") ->	{"0","",""};
lookup1("0CC2") ->	{"0","",""};
lookup1("0CC3") ->	{"0","",""};
lookup1("0CC4") ->	{"0","",""};
lookup1("0CC6") ->	{"0","",""};
lookup1("0CC7") ->	{"0","0CC6 0CD5",""};
lookup1("0CC8") ->	{"0","0CC6 0CD6",""};
lookup1("0CCA") ->	{"0","0CC6 0CC2",""};
lookup1("0CCB") ->	{"0","0CCA 0CD5",""};
lookup1("0CCC") ->	{"0","",""};
lookup1("0CCD") ->	{"9","",""};
lookup1("0CD5") ->	{"0","",""};
lookup1("0CD6") ->	{"0","",""};
lookup1("0CDE") ->	{"0","",""};
lookup1("0CE0") ->	{"0","",""};
lookup1("0CE1") ->	{"0","",""};
lookup1("0CE2") ->	{"0","",""};
lookup1("0CE3") ->	{"0","",""};
lookup1("0CE6") ->	{"0","",""};
lookup1("0CE7") ->	{"0","",""};
lookup1("0CE8") ->	{"0","",""};
lookup1("0CE9") ->	{"0","",""};
lookup1("0CEA") ->	{"0","",""};
lookup1("0CEB") ->	{"0","",""};
lookup1("0CEC") ->	{"0","",""};
lookup1("0CED") ->	{"0","",""};
lookup1("0CEE") ->	{"0","",""};
lookup1("0CEF") ->	{"0","",""};
lookup1("0CF1") ->	{"0","",""};
lookup1("0CF2") ->	{"0","",""};
lookup1("0D01") ->	{"0","",""};
lookup1("0D02") ->	{"0","",""};
lookup1("0D03") ->	{"0","",""};
lookup1("0D05") ->	{"0","",""};
lookup1("0D06") ->	{"0","",""};
lookup1("0D07") ->	{"0","",""};
lookup1("0D08") ->	{"0","",""};
lookup1("0D09") ->	{"0","",""};
lookup1("0D0A") ->	{"0","",""};
lookup1("0D0B") ->	{"0","",""};
lookup1("0D0C") ->	{"0","",""};
lookup1("0D0E") ->	{"0","",""};
lookup1("0D0F") ->	{"0","",""};
lookup1("0D10") ->	{"0","",""};
lookup1("0D12") ->	{"0","",""};
lookup1("0D13") ->	{"0","",""};
lookup1("0D14") ->	{"0","",""};
lookup1("0D15") ->	{"0","",""};
lookup1("0D16") ->	{"0","",""};
lookup1("0D17") ->	{"0","",""};
lookup1("0D18") ->	{"0","",""};
lookup1("0D19") ->	{"0","",""};
lookup1("0D1A") ->	{"0","",""};
lookup1("0D1B") ->	{"0","",""};
lookup1("0D1C") ->	{"0","",""};
lookup1("0D1D") ->	{"0","",""};
lookup1("0D1E") ->	{"0","",""};
lookup1("0D1F") ->	{"0","",""};
lookup1("0D20") ->	{"0","",""};
lookup1("0D21") ->	{"0","",""};
lookup1("0D22") ->	{"0","",""};
lookup1("0D23") ->	{"0","",""};
lookup1("0D24") ->	{"0","",""};
lookup1("0D25") ->	{"0","",""};
lookup1("0D26") ->	{"0","",""};
lookup1("0D27") ->	{"0","",""};
lookup1("0D28") ->	{"0","",""};
lookup1("0D29") ->	{"0","",""};
lookup1("0D2A") ->	{"0","",""};
lookup1("0D2B") ->	{"0","",""};
lookup1("0D2C") ->	{"0","",""};
lookup1("0D2D") ->	{"0","",""};
lookup1("0D2E") ->	{"0","",""};
lookup1("0D2F") ->	{"0","",""};
lookup1("0D30") ->	{"0","",""};
lookup1("0D31") ->	{"0","",""};
lookup1("0D32") ->	{"0","",""};
lookup1("0D33") ->	{"0","",""};
lookup1("0D34") ->	{"0","",""};
lookup1("0D35") ->	{"0","",""};
lookup1("0D36") ->	{"0","",""};
lookup1("0D37") ->	{"0","",""};
lookup1("0D38") ->	{"0","",""};
lookup1("0D39") ->	{"0","",""};
lookup1("0D3A") ->	{"0","",""};
lookup1("0D3D") ->	{"0","",""};
lookup1("0D3E") ->	{"0","",""};
lookup1("0D3F") ->	{"0","",""};
lookup1("0D40") ->	{"0","",""};
lookup1("0D41") ->	{"0","",""};
lookup1("0D42") ->	{"0","",""};
lookup1("0D43") ->	{"0","",""};
lookup1("0D44") ->	{"0","",""};
lookup1("0D46") ->	{"0","",""};
lookup1("0D47") ->	{"0","",""};
lookup1("0D48") ->	{"0","",""};
lookup1("0D4A") ->	{"0","0D46 0D3E",""};
lookup1("0D4B") ->	{"0","0D47 0D3E",""};
lookup1("0D4C") ->	{"0","0D46 0D57",""};
lookup1("0D4D") ->	{"9","",""};
lookup1("0D4E") ->	{"0","",""};
lookup1("0D57") ->	{"0","",""};
lookup1("0D60") ->	{"0","",""};
lookup1("0D61") ->	{"0","",""};
lookup1("0D62") ->	{"0","",""};
lookup1("0D63") ->	{"0","",""};
lookup1("0D66") ->	{"0","",""};
lookup1("0D67") ->	{"0","",""};
lookup1("0D68") ->	{"0","",""};
lookup1("0D69") ->	{"0","",""};
lookup1("0D6A") ->	{"0","",""};
lookup1("0D6B") ->	{"0","",""};
lookup1("0D6C") ->	{"0","",""};
lookup1("0D6D") ->	{"0","",""};
lookup1("0D6E") ->	{"0","",""};
lookup1("0D6F") ->	{"0","",""};
lookup1("0D70") ->	{"0","",""};
lookup1("0D71") ->	{"0","",""};
lookup1("0D72") ->	{"0","",""};
lookup1("0D73") ->	{"0","",""};
lookup1("0D74") ->	{"0","",""};
lookup1("0D75") ->	{"0","",""};
lookup1("0D79") ->	{"0","",""};
lookup1("0D7A") ->	{"0","",""};
lookup1("0D7B") ->	{"0","",""};
lookup1("0D7C") ->	{"0","",""};
lookup1("0D7D") ->	{"0","",""};
lookup1("0D7E") ->	{"0","",""};
lookup1("0D7F") ->	{"0","",""};
lookup1("0D82") ->	{"0","",""};
lookup1("0D83") ->	{"0","",""};
lookup1("0D85") ->	{"0","",""};
lookup1("0D86") ->	{"0","",""};
lookup1("0D87") ->	{"0","",""};
lookup1("0D88") ->	{"0","",""};
lookup1("0D89") ->	{"0","",""};
lookup1("0D8A") ->	{"0","",""};
lookup1("0D8B") ->	{"0","",""};
lookup1("0D8C") ->	{"0","",""};
lookup1("0D8D") ->	{"0","",""};
lookup1("0D8E") ->	{"0","",""};
lookup1("0D8F") ->	{"0","",""};
lookup1("0D90") ->	{"0","",""};
lookup1("0D91") ->	{"0","",""};
lookup1("0D92") ->	{"0","",""};
lookup1("0D93") ->	{"0","",""};
lookup1("0D94") ->	{"0","",""};
lookup1("0D95") ->	{"0","",""};
lookup1("0D96") ->	{"0","",""};
lookup1("0D9A") ->	{"0","",""};
lookup1("0D9B") ->	{"0","",""};
lookup1("0D9C") ->	{"0","",""};
lookup1("0D9D") ->	{"0","",""};
lookup1("0D9E") ->	{"0","",""};
lookup1("0D9F") ->	{"0","",""};
lookup1("0DA0") ->	{"0","",""};
lookup1("0DA1") ->	{"0","",""};
lookup1("0DA2") ->	{"0","",""};
lookup1("0DA3") ->	{"0","",""};
lookup1("0DA4") ->	{"0","",""};
lookup1("0DA5") ->	{"0","",""};
lookup1("0DA6") ->	{"0","",""};
lookup1("0DA7") ->	{"0","",""};
lookup1("0DA8") ->	{"0","",""};
lookup1("0DA9") ->	{"0","",""};
lookup1("0DAA") ->	{"0","",""};
lookup1("0DAB") ->	{"0","",""};
lookup1("0DAC") ->	{"0","",""};
lookup1("0DAD") ->	{"0","",""};
lookup1("0DAE") ->	{"0","",""};
lookup1("0DAF") ->	{"0","",""};
lookup1("0DB0") ->	{"0","",""};
lookup1("0DB1") ->	{"0","",""};
lookup1("0DB3") ->	{"0","",""};
lookup1("0DB4") ->	{"0","",""};
lookup1("0DB5") ->	{"0","",""};
lookup1("0DB6") ->	{"0","",""};
lookup1("0DB7") ->	{"0","",""};
lookup1("0DB8") ->	{"0","",""};
lookup1("0DB9") ->	{"0","",""};
lookup1("0DBA") ->	{"0","",""};
lookup1("0DBB") ->	{"0","",""};
lookup1("0DBD") ->	{"0","",""};
lookup1("0DC0") ->	{"0","",""};
lookup1("0DC1") ->	{"0","",""};
lookup1("0DC2") ->	{"0","",""};
lookup1("0DC3") ->	{"0","",""};
lookup1("0DC4") ->	{"0","",""};
lookup1("0DC5") ->	{"0","",""};
lookup1("0DC6") ->	{"0","",""};
lookup1("0DCA") ->	{"9","",""};
lookup1("0DCF") ->	{"0","",""};
lookup1("0DD0") ->	{"0","",""};
lookup1("0DD1") ->	{"0","",""};
lookup1("0DD2") ->	{"0","",""};
lookup1("0DD3") ->	{"0","",""};
lookup1("0DD4") ->	{"0","",""};
lookup1("0DD6") ->	{"0","",""};
lookup1("0DD8") ->	{"0","",""};
lookup1("0DD9") ->	{"0","",""};
lookup1("0DDA") ->	{"0","0DD9 0DCA",""};
lookup1("0DDB") ->	{"0","",""};
lookup1("0DDC") ->	{"0","0DD9 0DCF",""};
lookup1("0DDD") ->	{"0","0DDC 0DCA",""};
lookup1("0DDE") ->	{"0","0DD9 0DDF",""};
lookup1("0DDF") ->	{"0","",""};
lookup1("0DE6") ->	{"0","",""};
lookup1("0DE7") ->	{"0","",""};
lookup1("0DE8") ->	{"0","",""};
lookup1("0DE9") ->	{"0","",""};
lookup1("0DEA") ->	{"0","",""};
lookup1("0DEB") ->	{"0","",""};
lookup1("0DEC") ->	{"0","",""};
lookup1("0DED") ->	{"0","",""};
lookup1("0DEE") ->	{"0","",""};
lookup1("0DEF") ->	{"0","",""};
lookup1("0DF2") ->	{"0","",""};
lookup1("0DF3") ->	{"0","",""};
lookup1("0DF4") ->	{"0","",""};
lookup1("0E01") ->	{"0","",""};
lookup1("0E02") ->	{"0","",""};
lookup1("0E03") ->	{"0","",""};
lookup1("0E04") ->	{"0","",""};
lookup1("0E05") ->	{"0","",""};
lookup1("0E06") ->	{"0","",""};
lookup1("0E07") ->	{"0","",""};
lookup1("0E08") ->	{"0","",""};
lookup1("0E09") ->	{"0","",""};
lookup1("0E0A") ->	{"0","",""};
lookup1("0E0B") ->	{"0","",""};
lookup1("0E0C") ->	{"0","",""};
lookup1("0E0D") ->	{"0","",""};
lookup1("0E0E") ->	{"0","",""};
lookup1("0E0F") ->	{"0","",""};
lookup1("0E10") ->	{"0","",""};
lookup1("0E11") ->	{"0","",""};
lookup1("0E12") ->	{"0","",""};
lookup1("0E13") ->	{"0","",""};
lookup1("0E14") ->	{"0","",""};
lookup1("0E15") ->	{"0","",""};
lookup1("0E16") ->	{"0","",""};
lookup1("0E17") ->	{"0","",""};
lookup1("0E18") ->	{"0","",""};
lookup1("0E19") ->	{"0","",""};
lookup1("0E1A") ->	{"0","",""};
lookup1("0E1B") ->	{"0","",""};
lookup1("0E1C") ->	{"0","",""};
lookup1("0E1D") ->	{"0","",""};
lookup1("0E1E") ->	{"0","",""};
lookup1("0E1F") ->	{"0","",""};
lookup1("0E20") ->	{"0","",""};
lookup1("0E21") ->	{"0","",""};
lookup1("0E22") ->	{"0","",""};
lookup1("0E23") ->	{"0","",""};
lookup1("0E24") ->	{"0","",""};
lookup1("0E25") ->	{"0","",""};
lookup1("0E26") ->	{"0","",""};
lookup1("0E27") ->	{"0","",""};
lookup1("0E28") ->	{"0","",""};
lookup1("0E29") ->	{"0","",""};
lookup1("0E2A") ->	{"0","",""};
lookup1("0E2B") ->	{"0","",""};
lookup1("0E2C") ->	{"0","",""};
lookup1("0E2D") ->	{"0","",""};
lookup1("0E2E") ->	{"0","",""};
lookup1("0E2F") ->	{"0","",""};
lookup1("0E30") ->	{"0","",""};
lookup1("0E31") ->	{"0","",""};
lookup1("0E32") ->	{"0","",""};
lookup1("0E33") ->	{"0","<compat> 0E4D 0E32",""};
lookup1("0E34") ->	{"0","",""};
lookup1("0E35") ->	{"0","",""};
lookup1("0E36") ->	{"0","",""};
lookup1("0E37") ->	{"0","",""};
lookup1("0E38") ->	{"103","",""};
lookup1("0E39") ->	{"103","",""};
lookup1("0E3A") ->	{"9","",""};
lookup1("0E3F") ->	{"0","",""};
lookup1("0E40") ->	{"0","",""};
lookup1("0E41") ->	{"0","",""};
lookup1("0E42") ->	{"0","",""};
lookup1("0E43") ->	{"0","",""};
lookup1("0E44") ->	{"0","",""};
lookup1("0E45") ->	{"0","",""};
lookup1("0E46") ->	{"0","",""};
lookup1("0E47") ->	{"0","",""};
lookup1("0E48") ->	{"107","",""};
lookup1("0E49") ->	{"107","",""};
lookup1("0E4A") ->	{"107","",""};
lookup1("0E4B") ->	{"107","",""};
lookup1("0E4C") ->	{"0","",""};
lookup1("0E4D") ->	{"0","",""};
lookup1("0E4E") ->	{"0","",""};
lookup1("0E4F") ->	{"0","",""};
lookup1("0E50") ->	{"0","",""};
lookup1("0E51") ->	{"0","",""};
lookup1("0E52") ->	{"0","",""};
lookup1("0E53") ->	{"0","",""};
lookup1("0E54") ->	{"0","",""};
lookup1("0E55") ->	{"0","",""};
lookup1("0E56") ->	{"0","",""};
lookup1("0E57") ->	{"0","",""};
lookup1("0E58") ->	{"0","",""};
lookup1("0E59") ->	{"0","",""};
lookup1("0E5A") ->	{"0","",""};
lookup1("0E5B") ->	{"0","",""};
lookup1("0E81") ->	{"0","",""};
lookup1("0E82") ->	{"0","",""};
lookup1("0E84") ->	{"0","",""};
lookup1("0E87") ->	{"0","",""};
lookup1("0E88") ->	{"0","",""};
lookup1("0E8A") ->	{"0","",""};
lookup1("0E8D") ->	{"0","",""};
lookup1("0E94") ->	{"0","",""};
lookup1("0E95") ->	{"0","",""};
lookup1("0E96") ->	{"0","",""};
lookup1("0E97") ->	{"0","",""};
lookup1("0E99") ->	{"0","",""};
lookup1("0E9A") ->	{"0","",""};
lookup1("0E9B") ->	{"0","",""};
lookup1("0E9C") ->	{"0","",""};
lookup1("0E9D") ->	{"0","",""};
lookup1("0E9E") ->	{"0","",""};
lookup1("0E9F") ->	{"0","",""};
lookup1("0EA1") ->	{"0","",""};
lookup1("0EA2") ->	{"0","",""};
lookup1("0EA3") ->	{"0","",""};
lookup1("0EA5") ->	{"0","",""};
lookup1("0EA7") ->	{"0","",""};
lookup1("0EAA") ->	{"0","",""};
lookup1("0EAB") ->	{"0","",""};
lookup1("0EAD") ->	{"0","",""};
lookup1("0EAE") ->	{"0","",""};
lookup1("0EAF") ->	{"0","",""};
lookup1("0EB0") ->	{"0","",""};
lookup1("0EB1") ->	{"0","",""};
lookup1("0EB2") ->	{"0","",""};
lookup1("0EB3") ->	{"0","<compat> 0ECD 0EB2",""};
lookup1("0EB4") ->	{"0","",""};
lookup1("0EB5") ->	{"0","",""};
lookup1("0EB6") ->	{"0","",""};
lookup1("0EB7") ->	{"0","",""};
lookup1("0EB8") ->	{"118","",""};
lookup1("0EB9") ->	{"118","",""};
lookup1("0EBB") ->	{"0","",""};
lookup1("0EBC") ->	{"0","",""};
lookup1("0EBD") ->	{"0","",""};
lookup1("0EC0") ->	{"0","",""};
lookup1("0EC1") ->	{"0","",""};
lookup1("0EC2") ->	{"0","",""};
lookup1("0EC3") ->	{"0","",""};
lookup1("0EC4") ->	{"0","",""};
lookup1("0EC6") ->	{"0","",""};
lookup1("0EC8") ->	{"122","",""};
lookup1("0EC9") ->	{"122","",""};
lookup1("0ECA") ->	{"122","",""};
lookup1("0ECB") ->	{"122","",""};
lookup1("0ECC") ->	{"0","",""};
lookup1("0ECD") ->	{"0","",""};
lookup1("0ED0") ->	{"0","",""};
lookup1("0ED1") ->	{"0","",""};
lookup1("0ED2") ->	{"0","",""};
lookup1("0ED3") ->	{"0","",""};
lookup1("0ED4") ->	{"0","",""};
lookup1("0ED5") ->	{"0","",""};
lookup1("0ED6") ->	{"0","",""};
lookup1("0ED7") ->	{"0","",""};
lookup1("0ED8") ->	{"0","",""};
lookup1("0ED9") ->	{"0","",""};
lookup1("0EDC") ->	{"0","<compat> 0EAB 0E99",""};
lookup1("0EDD") ->	{"0","<compat> 0EAB 0EA1",""};
lookup1("0EDE") ->	{"0","",""};
lookup1("0EDF") ->	{"0","",""};
lookup1("0F00") ->	{"0","",""};
lookup1("0F01") ->	{"0","",""};
lookup1("0F02") ->	{"0","",""};
lookup1("0F03") ->	{"0","",""};
lookup1("0F04") ->	{"0","",""};
lookup1("0F05") ->	{"0","",""};
lookup1("0F06") ->	{"0","",""};
lookup1("0F07") ->	{"0","",""};
lookup1("0F08") ->	{"0","",""};
lookup1("0F09") ->	{"0","",""};
lookup1("0F0A") ->	{"0","",""};
lookup1("0F0B") ->	{"0","",""};
lookup1("0F0C") ->	{"0","<noBreak> 0F0B",""};
lookup1("0F0D") ->	{"0","",""};
lookup1("0F0E") ->	{"0","",""};
lookup1("0F0F") ->	{"0","",""};
lookup1("0F10") ->	{"0","",""};
lookup1("0F11") ->	{"0","",""};
lookup1("0F12") ->	{"0","",""};
lookup1("0F13") ->	{"0","",""};
lookup1("0F14") ->	{"0","",""};
lookup1("0F15") ->	{"0","",""};
lookup1("0F16") ->	{"0","",""};
lookup1("0F17") ->	{"0","",""};
lookup1("0F18") ->	{"220","",""};
lookup1("0F19") ->	{"220","",""};
lookup1("0F1A") ->	{"0","",""};
lookup1("0F1B") ->	{"0","",""};
lookup1("0F1C") ->	{"0","",""};
lookup1("0F1D") ->	{"0","",""};
lookup1("0F1E") ->	{"0","",""};
lookup1("0F1F") ->	{"0","",""};
lookup1("0F20") ->	{"0","",""};
lookup1("0F21") ->	{"0","",""};
lookup1("0F22") ->	{"0","",""};
lookup1("0F23") ->	{"0","",""};
lookup1("0F24") ->	{"0","",""};
lookup1("0F25") ->	{"0","",""};
lookup1("0F26") ->	{"0","",""};
lookup1("0F27") ->	{"0","",""};
lookup1("0F28") ->	{"0","",""};
lookup1("0F29") ->	{"0","",""};
lookup1("0F2A") ->	{"0","",""};
lookup1("0F2B") ->	{"0","",""};
lookup1("0F2C") ->	{"0","",""};
lookup1("0F2D") ->	{"0","",""};
lookup1("0F2E") ->	{"0","",""};
lookup1("0F2F") ->	{"0","",""};
lookup1("0F30") ->	{"0","",""};
lookup1("0F31") ->	{"0","",""};
lookup1("0F32") ->	{"0","",""};
lookup1("0F33") ->	{"0","",""};
lookup1("0F34") ->	{"0","",""};
lookup1("0F35") ->	{"220","",""};
lookup1("0F36") ->	{"0","",""};
lookup1("0F37") ->	{"220","",""};
lookup1("0F38") ->	{"0","",""};
lookup1("0F39") ->	{"216","",""};
lookup1("0F3A") ->	{"0","",""};
lookup1("0F3B") ->	{"0","",""};
lookup1("0F3C") ->	{"0","",""};
lookup1("0F3D") ->	{"0","",""};
lookup1("0F3E") ->	{"0","",""};
lookup1("0F3F") ->	{"0","",""};
lookup1("0F40") ->	{"0","",""};
lookup1("0F41") ->	{"0","",""};
lookup1("0F42") ->	{"0","",""};
lookup1("0F43") ->	{"0","0F42 0FB7",""};
lookup1("0F44") ->	{"0","",""};
lookup1("0F45") ->	{"0","",""};
lookup1("0F46") ->	{"0","",""};
lookup1("0F47") ->	{"0","",""};
lookup1("0F49") ->	{"0","",""};
lookup1("0F4A") ->	{"0","",""};
lookup1("0F4B") ->	{"0","",""};
lookup1("0F4C") ->	{"0","",""};
lookup1("0F4D") ->	{"0","0F4C 0FB7",""};
lookup1("0F4E") ->	{"0","",""};
lookup1("0F4F") ->	{"0","",""};
lookup1("0F50") ->	{"0","",""};
lookup1("0F51") ->	{"0","",""};
lookup1("0F52") ->	{"0","0F51 0FB7",""};
lookup1("0F53") ->	{"0","",""};
lookup1("0F54") ->	{"0","",""};
lookup1("0F55") ->	{"0","",""};
lookup1("0F56") ->	{"0","",""};
lookup1("0F57") ->	{"0","0F56 0FB7",""};
lookup1("0F58") ->	{"0","",""};
lookup1("0F59") ->	{"0","",""};
lookup1("0F5A") ->	{"0","",""};
lookup1("0F5B") ->	{"0","",""};
lookup1("0F5C") ->	{"0","0F5B 0FB7",""};
lookup1("0F5D") ->	{"0","",""};
lookup1("0F5E") ->	{"0","",""};
lookup1("0F5F") ->	{"0","",""};
lookup1("0F60") ->	{"0","",""};
lookup1("0F61") ->	{"0","",""};
lookup1("0F62") ->	{"0","",""};
lookup1("0F63") ->	{"0","",""};
lookup1("0F64") ->	{"0","",""};
lookup1("0F65") ->	{"0","",""};
lookup1("0F66") ->	{"0","",""};
lookup1("0F67") ->	{"0","",""};
lookup1("0F68") ->	{"0","",""};
lookup1("0F69") ->	{"0","0F40 0FB5",""};
lookup1("0F6A") ->	{"0","",""};
lookup1("0F6B") ->	{"0","",""};
lookup1("0F6C") ->	{"0","",""};
lookup1("0F71") ->	{"129","",""};
lookup1("0F72") ->	{"130","",""};
lookup1("0F73") ->	{"0","0F71 0F72",""};
lookup1("0F74") ->	{"132","",""};
lookup1("0F75") ->	{"0","0F71 0F74",""};
lookup1("0F76") ->	{"0","0FB2 0F80",""};
lookup1("0F77") ->	{"0","<compat> 0FB2 0F81",""};
lookup1("0F78") ->	{"0","0FB3 0F80",""};
lookup1("0F79") ->	{"0","<compat> 0FB3 0F81",""};
lookup1("0F7A") ->	{"130","",""};
lookup1("0F7B") ->	{"130","",""};
lookup1("0F7C") ->	{"130","",""};
lookup1("0F7D") ->	{"130","",""};
lookup1("0F7E") ->	{"0","",""};
lookup1("0F7F") ->	{"0","",""};
lookup1("0F80") ->	{"130","",""};
lookup1("0F81") ->	{"0","0F71 0F80",""};
lookup1("0F82") ->	{"230","",""};
lookup1("0F83") ->	{"230","",""};
lookup1("0F84") ->	{"9","",""};
lookup1("0F85") ->	{"0","",""};
lookup1("0F86") ->	{"230","",""};
lookup1("0F87") ->	{"230","",""};
lookup1("0F88") ->	{"0","",""};
lookup1("0F89") ->	{"0","",""};
lookup1("0F8A") ->	{"0","",""};
lookup1("0F8B") ->	{"0","",""};
lookup1("0F8C") ->	{"0","",""};
lookup1("0F8D") ->	{"0","",""};
lookup1("0F8E") ->	{"0","",""};
lookup1("0F8F") ->	{"0","",""};
lookup1("0F90") ->	{"0","",""};
lookup1("0F91") ->	{"0","",""};
lookup1("0F92") ->	{"0","",""};
lookup1("0F93") ->	{"0","0F92 0FB7",""};
lookup1("0F94") ->	{"0","",""};
lookup1("0F95") ->	{"0","",""};
lookup1("0F96") ->	{"0","",""};
lookup1("0F97") ->	{"0","",""};
lookup1("0F99") ->	{"0","",""};
lookup1("0F9A") ->	{"0","",""};
lookup1("0F9B") ->	{"0","",""};
lookup1("0F9C") ->	{"0","",""};
lookup1("0F9D") ->	{"0","0F9C 0FB7",""};
lookup1("0F9E") ->	{"0","",""};
lookup1("0F9F") ->	{"0","",""};
lookup1("0FA0") ->	{"0","",""};
lookup1("0FA1") ->	{"0","",""};
lookup1("0FA2") ->	{"0","0FA1 0FB7",""};
lookup1("0FA3") ->	{"0","",""};
lookup1("0FA4") ->	{"0","",""};
lookup1("0FA5") ->	{"0","",""};
lookup1("0FA6") ->	{"0","",""};
lookup1("0FA7") ->	{"0","0FA6 0FB7",""};
lookup1("0FA8") ->	{"0","",""};
lookup1("0FA9") ->	{"0","",""};
lookup1("0FAA") ->	{"0","",""};
lookup1("0FAB") ->	{"0","",""};
lookup1("0FAC") ->	{"0","0FAB 0FB7",""};
lookup1("0FAD") ->	{"0","",""};
lookup1("0FAE") ->	{"0","",""};
lookup1("0FAF") ->	{"0","",""};
lookup1("0FB0") ->	{"0","",""};
lookup1("0FB1") ->	{"0","",""};
lookup1("0FB2") ->	{"0","",""};
lookup1("0FB3") ->	{"0","",""};
lookup1("0FB4") ->	{"0","",""};
lookup1("0FB5") ->	{"0","",""};
lookup1("0FB6") ->	{"0","",""};
lookup1("0FB7") ->	{"0","",""};
lookup1("0FB8") ->	{"0","",""};
lookup1("0FB9") ->	{"0","0F90 0FB5",""};
lookup1("0FBA") ->	{"0","",""};
lookup1("0FBB") ->	{"0","",""};
lookup1("0FBC") ->	{"0","",""};
lookup1("0FBE") ->	{"0","",""};
lookup1("0FBF") ->	{"0","",""};
lookup1("0FC0") ->	{"0","",""};
lookup1("0FC1") ->	{"0","",""};
lookup1("0FC2") ->	{"0","",""};
lookup1("0FC3") ->	{"0","",""};
lookup1("0FC4") ->	{"0","",""};
lookup1("0FC5") ->	{"0","",""};
lookup1("0FC6") ->	{"220","",""};
lookup1("0FC7") ->	{"0","",""};
lookup1("0FC8") ->	{"0","",""};
lookup1("0FC9") ->	{"0","",""};
lookup1("0FCA") ->	{"0","",""};
lookup1("0FCB") ->	{"0","",""};
lookup1("0FCC") ->	{"0","",""};
lookup1("0FCE") ->	{"0","",""};
lookup1("0FCF") ->	{"0","",""};
lookup1("0FD0") ->	{"0","",""};
lookup1("0FD1") ->	{"0","",""};
lookup1("0FD2") ->	{"0","",""};
lookup1("0FD3") ->	{"0","",""};
lookup1("0FD4") ->	{"0","",""};
lookup1("0FD5") ->	{"0","",""};
lookup1("0FD6") ->	{"0","",""};
lookup1("0FD7") ->	{"0","",""};
lookup1("0FD8") ->	{"0","",""};
lookup1("0FD9") ->	{"0","",""};
lookup1("0FDA") ->	{"0","",""};
lookup1("1000") ->	{"0","",""};
lookup1("10000") ->	{"0","",""};
lookup1("100000") ->	{"0","",""};
lookup1("10001") ->	{"0","",""};
lookup1("10002") ->	{"0","",""};
lookup1("10003") ->	{"0","",""};
lookup1("10004") ->	{"0","",""};
lookup1("10005") ->	{"0","",""};
lookup1("10006") ->	{"0","",""};
lookup1("10007") ->	{"0","",""};
lookup1("10008") ->	{"0","",""};
lookup1("10009") ->	{"0","",""};
lookup1("1000A") ->	{"0","",""};
lookup1("1000B") ->	{"0","",""};
lookup1("1000D") ->	{"0","",""};
lookup1("1000E") ->	{"0","",""};
lookup1("1000F") ->	{"0","",""};
lookup1("1001") ->	{"0","",""};
lookup1("10010") ->	{"0","",""};
lookup1("10011") ->	{"0","",""};
lookup1("10012") ->	{"0","",""};
lookup1("10013") ->	{"0","",""};
lookup1("10014") ->	{"0","",""};
lookup1("10015") ->	{"0","",""};
lookup1("10016") ->	{"0","",""};
lookup1("10017") ->	{"0","",""};
lookup1("10018") ->	{"0","",""};
lookup1("10019") ->	{"0","",""};
lookup1("1001A") ->	{"0","",""};
lookup1("1001B") ->	{"0","",""};
lookup1("1001C") ->	{"0","",""};
lookup1("1001D") ->	{"0","",""};
lookup1("1001E") ->	{"0","",""};
lookup1("1001F") ->	{"0","",""};
lookup1("1002") ->	{"0","",""};
lookup1("10020") ->	{"0","",""};
lookup1("10021") ->	{"0","",""};
lookup1("10022") ->	{"0","",""};
lookup1("10023") ->	{"0","",""};
lookup1("10024") ->	{"0","",""};
lookup1("10025") ->	{"0","",""};
lookup1("10026") ->	{"0","",""};
lookup1("10028") ->	{"0","",""};
lookup1("10029") ->	{"0","",""};
lookup1("1002A") ->	{"0","",""};
lookup1("1002B") ->	{"0","",""};
lookup1("1002C") ->	{"0","",""};
lookup1("1002D") ->	{"0","",""};
lookup1("1002E") ->	{"0","",""};
lookup1("1002F") ->	{"0","",""};
lookup1("1003") ->	{"0","",""};
lookup1("10030") ->	{"0","",""};
lookup1("10031") ->	{"0","",""};
lookup1("10032") ->	{"0","",""};
lookup1("10033") ->	{"0","",""};
lookup1("10034") ->	{"0","",""};
lookup1("10035") ->	{"0","",""};
lookup1("10036") ->	{"0","",""};
lookup1("10037") ->	{"0","",""};
lookup1("10038") ->	{"0","",""};
lookup1("10039") ->	{"0","",""};
lookup1("1003A") ->	{"0","",""};
lookup1("1003C") ->	{"0","",""};
lookup1("1003D") ->	{"0","",""};
lookup1("1003F") ->	{"0","",""};
lookup1("1004") ->	{"0","",""};
lookup1("10040") ->	{"0","",""};
lookup1("10041") ->	{"0","",""};
lookup1("10042") ->	{"0","",""};
lookup1("10043") ->	{"0","",""};
lookup1("10044") ->	{"0","",""};
lookup1("10045") ->	{"0","",""};
lookup1("10046") ->	{"0","",""};
lookup1("10047") ->	{"0","",""};
lookup1("10048") ->	{"0","",""};
lookup1("10049") ->	{"0","",""};
lookup1("1004A") ->	{"0","",""};
lookup1("1004B") ->	{"0","",""};
lookup1("1004C") ->	{"0","",""};
lookup1("1004D") ->	{"0","",""};
lookup1("1005") ->	{"0","",""};
lookup1("10050") ->	{"0","",""};
lookup1("10051") ->	{"0","",""};
lookup1("10052") ->	{"0","",""};
lookup1("10053") ->	{"0","",""};
lookup1("10054") ->	{"0","",""};
lookup1("10055") ->	{"0","",""};
lookup1("10056") ->	{"0","",""};
lookup1("10057") ->	{"0","",""};
lookup1("10058") ->	{"0","",""};
lookup1("10059") ->	{"0","",""};
lookup1("1005A") ->	{"0","",""};
lookup1("1005B") ->	{"0","",""};
lookup1("1005C") ->	{"0","",""};
lookup1("1005D") ->	{"0","",""};
lookup1("1006") ->	{"0","",""};
lookup1("1007") ->	{"0","",""};
lookup1("1008") ->	{"0","",""};
lookup1("10080") ->	{"0","",""};
lookup1("10081") ->	{"0","",""};
lookup1("10082") ->	{"0","",""};
lookup1("10083") ->	{"0","",""};
lookup1("10084") ->	{"0","",""};
lookup1("10085") ->	{"0","",""};
lookup1("10086") ->	{"0","",""};
lookup1("10087") ->	{"0","",""};
lookup1("10088") ->	{"0","",""};
lookup1("10089") ->	{"0","",""};
lookup1("1008A") ->	{"0","",""};
lookup1("1008B") ->	{"0","",""};
lookup1("1008C") ->	{"0","",""};
lookup1("1008D") ->	{"0","",""};
lookup1("1008E") ->	{"0","",""};
lookup1("1008F") ->	{"0","",""};
lookup1("1009") ->	{"0","",""};
lookup1("10090") ->	{"0","",""};
lookup1("10091") ->	{"0","",""};
lookup1("10092") ->	{"0","",""};
lookup1("10093") ->	{"0","",""};
lookup1("10094") ->	{"0","",""};
lookup1("10095") ->	{"0","",""};
lookup1("10096") ->	{"0","",""};
lookup1("10097") ->	{"0","",""};
lookup1("10098") ->	{"0","",""};
lookup1("10099") ->	{"0","",""};
lookup1("1009A") ->	{"0","",""};
lookup1("1009B") ->	{"0","",""};
lookup1("1009C") ->	{"0","",""};
lookup1("1009D") ->	{"0","",""};
lookup1("1009E") ->	{"0","",""};
lookup1("1009F") ->	{"0","",""};
lookup1("100A") ->	{"0","",""};
lookup1("100A0") ->	{"0","",""};
lookup1("100A1") ->	{"0","",""};
lookup1("100A2") ->	{"0","",""};
lookup1("100A3") ->	{"0","",""};
lookup1("100A4") ->	{"0","",""};
lookup1("100A5") ->	{"0","",""};
lookup1("100A6") ->	{"0","",""};
lookup1("100A7") ->	{"0","",""};
lookup1("100A8") ->	{"0","",""};
lookup1("100A9") ->	{"0","",""};
lookup1("100AA") ->	{"0","",""};
lookup1("100AB") ->	{"0","",""};
lookup1("100AC") ->	{"0","",""};
lookup1("100AD") ->	{"0","",""};
lookup1("100AE") ->	{"0","",""};
lookup1("100AF") ->	{"0","",""};
lookup1("100B") ->	{"0","",""};
lookup1("100B0") ->	{"0","",""};
lookup1("100B1") ->	{"0","",""};
lookup1("100B2") ->	{"0","",""};
lookup1("100B3") ->	{"0","",""};
lookup1("100B4") ->	{"0","",""};
lookup1("100B5") ->	{"0","",""};
lookup1("100B6") ->	{"0","",""};
lookup1("100B7") ->	{"0","",""};
lookup1("100B8") ->	{"0","",""};
lookup1("100B9") ->	{"0","",""};
lookup1("100BA") ->	{"0","",""};
lookup1("100BB") ->	{"0","",""};
lookup1("100BC") ->	{"0","",""};
lookup1("100BD") ->	{"0","",""};
lookup1("100BE") ->	{"0","",""};
lookup1("100BF") ->	{"0","",""};
lookup1("100C") ->	{"0","",""};
lookup1("100C0") ->	{"0","",""};
lookup1("100C1") ->	{"0","",""};
lookup1("100C2") ->	{"0","",""};
lookup1("100C3") ->	{"0","",""};
lookup1("100C4") ->	{"0","",""};
lookup1("100C5") ->	{"0","",""};
lookup1("100C6") ->	{"0","",""};
lookup1("100C7") ->	{"0","",""};
lookup1("100C8") ->	{"0","",""};
lookup1("100C9") ->	{"0","",""};
lookup1("100CA") ->	{"0","",""};
lookup1("100CB") ->	{"0","",""};
lookup1("100CC") ->	{"0","",""};
lookup1("100CD") ->	{"0","",""};
lookup1("100CE") ->	{"0","",""};
lookup1("100CF") ->	{"0","",""};
lookup1("100D") ->	{"0","",""};
lookup1("100D0") ->	{"0","",""};
lookup1("100D1") ->	{"0","",""};
lookup1("100D2") ->	{"0","",""};
lookup1("100D3") ->	{"0","",""};
lookup1("100D4") ->	{"0","",""};
lookup1("100D5") ->	{"0","",""};
lookup1("100D6") ->	{"0","",""};
lookup1("100D7") ->	{"0","",""};
lookup1("100D8") ->	{"0","",""};
lookup1("100D9") ->	{"0","",""};
lookup1("100DA") ->	{"0","",""};
lookup1("100DB") ->	{"0","",""};
lookup1("100DC") ->	{"0","",""};
lookup1("100DD") ->	{"0","",""};
lookup1("100DE") ->	{"0","",""};
lookup1("100DF") ->	{"0","",""};
lookup1("100E") ->	{"0","",""};
lookup1("100E0") ->	{"0","",""};
lookup1("100E1") ->	{"0","",""};
lookup1("100E2") ->	{"0","",""};
lookup1("100E3") ->	{"0","",""};
lookup1("100E4") ->	{"0","",""};
lookup1("100E5") ->	{"0","",""};
lookup1("100E6") ->	{"0","",""};
lookup1("100E7") ->	{"0","",""};
lookup1("100E8") ->	{"0","",""};
lookup1("100E9") ->	{"0","",""};
lookup1("100EA") ->	{"0","",""};
lookup1("100EB") ->	{"0","",""};
lookup1("100EC") ->	{"0","",""};
lookup1("100ED") ->	{"0","",""};
lookup1("100EE") ->	{"0","",""};
lookup1("100EF") ->	{"0","",""};
lookup1("100F") ->	{"0","",""};
lookup1("100F0") ->	{"0","",""};
lookup1("100F1") ->	{"0","",""};
lookup1("100F2") ->	{"0","",""};
lookup1("100F3") ->	{"0","",""};
lookup1("100F4") ->	{"0","",""};
lookup1("100F5") ->	{"0","",""};
lookup1("100F6") ->	{"0","",""};
lookup1("100F7") ->	{"0","",""};
lookup1("100F8") ->	{"0","",""};
lookup1("100F9") ->	{"0","",""};
lookup1("100FA") ->	{"0","",""};
lookup1("1010") ->	{"0","",""};
lookup1("10100") ->	{"0","",""};
lookup1("10101") ->	{"0","",""};
lookup1("10102") ->	{"0","",""};
lookup1("10107") ->	{"0","",""};
lookup1("10108") ->	{"0","",""};
lookup1("10109") ->	{"0","",""};
lookup1("1010A") ->	{"0","",""};
lookup1("1010B") ->	{"0","",""};
lookup1("1010C") ->	{"0","",""};
lookup1("1010D") ->	{"0","",""};
lookup1("1010E") ->	{"0","",""};
lookup1("1010F") ->	{"0","",""};
lookup1("1011") ->	{"0","",""};
lookup1("10110") ->	{"0","",""};
lookup1("10111") ->	{"0","",""};
lookup1("10112") ->	{"0","",""};
lookup1("10113") ->	{"0","",""};
lookup1("10114") ->	{"0","",""};
lookup1("10115") ->	{"0","",""};
lookup1("10116") ->	{"0","",""};
lookup1("10117") ->	{"0","",""};
lookup1("10118") ->	{"0","",""};
lookup1("10119") ->	{"0","",""};
lookup1("1011A") ->	{"0","",""};
lookup1("1011B") ->	{"0","",""};
lookup1("1011C") ->	{"0","",""};
lookup1("1011D") ->	{"0","",""};
lookup1("1011E") ->	{"0","",""};
lookup1("1011F") ->	{"0","",""};
lookup1("1012") ->	{"0","",""};
lookup1("10120") ->	{"0","",""};
lookup1("10121") ->	{"0","",""};
lookup1("10122") ->	{"0","",""};
lookup1("10123") ->	{"0","",""};
lookup1("10124") ->	{"0","",""};
lookup1("10125") ->	{"0","",""};
lookup1("10126") ->	{"0","",""};
lookup1("10127") ->	{"0","",""};
lookup1("10128") ->	{"0","",""};
lookup1("10129") ->	{"0","",""};
lookup1("1012A") ->	{"0","",""};
lookup1("1012B") ->	{"0","",""};
lookup1("1012C") ->	{"0","",""};
lookup1("1012D") ->	{"0","",""};
lookup1("1012E") ->	{"0","",""};
lookup1("1012F") ->	{"0","",""};
lookup1("1013") ->	{"0","",""};
lookup1("10130") ->	{"0","",""};
lookup1("10131") ->	{"0","",""};
lookup1("10132") ->	{"0","",""};
lookup1("10133") ->	{"0","",""};
lookup1("10137") ->	{"0","",""};
lookup1("10138") ->	{"0","",""};
lookup1("10139") ->	{"0","",""};
lookup1("1013A") ->	{"0","",""};
lookup1("1013B") ->	{"0","",""};
lookup1("1013C") ->	{"0","",""};
lookup1("1013D") ->	{"0","",""};
lookup1("1013E") ->	{"0","",""};
lookup1("1013F") ->	{"0","",""};
lookup1("1014") ->	{"0","",""};
lookup1("10140") ->	{"0","",""};
lookup1("10141") ->	{"0","",""};
lookup1("10142") ->	{"0","",""};
lookup1("10143") ->	{"0","",""};
lookup1("10144") ->	{"0","",""};
lookup1("10145") ->	{"0","",""};
lookup1("10146") ->	{"0","",""};
lookup1("10147") ->	{"0","",""};
lookup1("10148") ->	{"0","",""};
lookup1("10149") ->	{"0","",""};
lookup1("1014A") ->	{"0","",""};
lookup1("1014B") ->	{"0","",""};
lookup1("1014C") ->	{"0","",""};
lookup1("1014D") ->	{"0","",""};
lookup1("1014E") ->	{"0","",""};
lookup1("1014F") ->	{"0","",""};
lookup1("1015") ->	{"0","",""};
lookup1("10150") ->	{"0","",""};
lookup1("10151") ->	{"0","",""};
lookup1("10152") ->	{"0","",""};
lookup1("10153") ->	{"0","",""};
lookup1("10154") ->	{"0","",""};
lookup1("10155") ->	{"0","",""};
lookup1("10156") ->	{"0","",""};
lookup1("10157") ->	{"0","",""};
lookup1("10158") ->	{"0","",""};
lookup1("10159") ->	{"0","",""};
lookup1("1015A") ->	{"0","",""};
lookup1("1015B") ->	{"0","",""};
lookup1("1015C") ->	{"0","",""};
lookup1("1015D") ->	{"0","",""};
lookup1("1015E") ->	{"0","",""};
lookup1("1015F") ->	{"0","",""};
lookup1("1016") ->	{"0","",""};
lookup1("10160") ->	{"0","",""};
lookup1("10161") ->	{"0","",""};
lookup1("10162") ->	{"0","",""};
lookup1("10163") ->	{"0","",""};
lookup1("10164") ->	{"0","",""};
lookup1("10165") ->	{"0","",""};
lookup1("10166") ->	{"0","",""};
lookup1("10167") ->	{"0","",""};
lookup1("10168") ->	{"0","",""};
lookup1("10169") ->	{"0","",""};
lookup1("1016A") ->	{"0","",""};
lookup1("1016B") ->	{"0","",""};
lookup1("1016C") ->	{"0","",""};
lookup1("1016D") ->	{"0","",""};
lookup1("1016E") ->	{"0","",""};
lookup1("1016F") ->	{"0","",""};
lookup1("1017") ->	{"0","",""};
lookup1("10170") ->	{"0","",""};
lookup1("10171") ->	{"0","",""};
lookup1("10172") ->	{"0","",""};
lookup1("10173") ->	{"0","",""};
lookup1("10174") ->	{"0","",""};
lookup1("10175") ->	{"0","",""};
lookup1("10176") ->	{"0","",""};
lookup1("10177") ->	{"0","",""};
lookup1("10178") ->	{"0","",""};
lookup1("10179") ->	{"0","",""};
lookup1("1017A") ->	{"0","",""};
lookup1("1017B") ->	{"0","",""};
lookup1("1017C") ->	{"0","",""};
lookup1("1017D") ->	{"0","",""};
lookup1("1017E") ->	{"0","",""};
lookup1("1017F") ->	{"0","",""};
lookup1("1018") ->	{"0","",""};
lookup1("10180") ->	{"0","",""};
lookup1("10181") ->	{"0","",""};
lookup1("10182") ->	{"0","",""};
lookup1("10183") ->	{"0","",""};
lookup1("10184") ->	{"0","",""};
lookup1("10185") ->	{"0","",""};
lookup1("10186") ->	{"0","",""};
lookup1("10187") ->	{"0","",""};
lookup1("10188") ->	{"0","",""};
lookup1("10189") ->	{"0","",""};
lookup1("1018A") ->	{"0","",""};
lookup1("1018B") ->	{"0","",""};
lookup1("1018C") ->	{"0","",""};
lookup1("1019") ->	{"0","",""};
lookup1("10190") ->	{"0","",""};
lookup1("10191") ->	{"0","",""};
lookup1("10192") ->	{"0","",""};
lookup1("10193") ->	{"0","",""};
lookup1("10194") ->	{"0","",""};
lookup1("10195") ->	{"0","",""};
lookup1("10196") ->	{"0","",""};
lookup1("10197") ->	{"0","",""};
lookup1("10198") ->	{"0","",""};
lookup1("10199") ->	{"0","",""};
lookup1("1019A") ->	{"0","",""};
lookup1("1019B") ->	{"0","",""};
lookup1("101A") ->	{"0","",""};
lookup1("101A0") ->	{"0","",""};
lookup1("101B") ->	{"0","",""};
lookup1("101C") ->	{"0","",""};
lookup1("101D") ->	{"0","",""};
lookup1("101D0") ->	{"0","",""};
lookup1("101D1") ->	{"0","",""};
lookup1("101D2") ->	{"0","",""};
lookup1("101D3") ->	{"0","",""};
lookup1("101D4") ->	{"0","",""};
lookup1("101D5") ->	{"0","",""};
lookup1("101D6") ->	{"0","",""};
lookup1("101D7") ->	{"0","",""};
lookup1("101D8") ->	{"0","",""};
lookup1("101D9") ->	{"0","",""};
lookup1("101DA") ->	{"0","",""};
lookup1("101DB") ->	{"0","",""};
lookup1("101DC") ->	{"0","",""};
lookup1("101DD") ->	{"0","",""};
lookup1("101DE") ->	{"0","",""};
lookup1("101DF") ->	{"0","",""};
lookup1("101E") ->	{"0","",""};
lookup1("101E0") ->	{"0","",""};
lookup1("101E1") ->	{"0","",""};
lookup1("101E2") ->	{"0","",""};
lookup1("101E3") ->	{"0","",""};
lookup1("101E4") ->	{"0","",""};
lookup1("101E5") ->	{"0","",""};
lookup1("101E6") ->	{"0","",""};
lookup1("101E7") ->	{"0","",""};
lookup1("101E8") ->	{"0","",""};
lookup1("101E9") ->	{"0","",""};
lookup1("101EA") ->	{"0","",""};
lookup1("101EB") ->	{"0","",""};
lookup1("101EC") ->	{"0","",""};
lookup1("101ED") ->	{"0","",""};
lookup1("101EE") ->	{"0","",""};
lookup1("101EF") ->	{"0","",""};
lookup1("101F") ->	{"0","",""};
lookup1("101F0") ->	{"0","",""};
lookup1("101F1") ->	{"0","",""};
lookup1("101F2") ->	{"0","",""};
lookup1("101F3") ->	{"0","",""};
lookup1("101F4") ->	{"0","",""};
lookup1("101F5") ->	{"0","",""};
lookup1("101F6") ->	{"0","",""};
lookup1("101F7") ->	{"0","",""};
lookup1("101F8") ->	{"0","",""};
lookup1("101F9") ->	{"0","",""};
lookup1("101FA") ->	{"0","",""};
lookup1("101FB") ->	{"0","",""};
lookup1("101FC") ->	{"0","",""};
lookup1("101FD") ->	{"220","",""};
lookup1("1020") ->	{"0","",""};
lookup1("1021") ->	{"0","",""};
lookup1("1022") ->	{"0","",""};
lookup1("1023") ->	{"0","",""};
lookup1("1024") ->	{"0","",""};
lookup1("1025") ->	{"0","",""};
lookup1("1026") ->	{"0","1025 102E",""};
lookup1("1027") ->	{"0","",""};
lookup1("1028") ->	{"0","",""};
lookup1("10280") ->	{"0","",""};
lookup1("10281") ->	{"0","",""};
lookup1("10282") ->	{"0","",""};
lookup1("10283") ->	{"0","",""};
lookup1("10284") ->	{"0","",""};
lookup1("10285") ->	{"0","",""};
lookup1("10286") ->	{"0","",""};
lookup1("10287") ->	{"0","",""};
lookup1("10288") ->	{"0","",""};
lookup1("10289") ->	{"0","",""};
lookup1("1028A") ->	{"0","",""};
lookup1("1028B") ->	{"0","",""};
lookup1("1028C") ->	{"0","",""};
lookup1("1028D") ->	{"0","",""};
lookup1("1028E") ->	{"0","",""};
lookup1("1028F") ->	{"0","",""};
lookup1("1029") ->	{"0","",""};
lookup1("10290") ->	{"0","",""};
lookup1("10291") ->	{"0","",""};
lookup1("10292") ->	{"0","",""};
lookup1("10293") ->	{"0","",""};
lookup1("10294") ->	{"0","",""};
lookup1("10295") ->	{"0","",""};
lookup1("10296") ->	{"0","",""};
lookup1("10297") ->	{"0","",""};
lookup1("10298") ->	{"0","",""};
lookup1("10299") ->	{"0","",""};
lookup1("1029A") ->	{"0","",""};
lookup1("1029B") ->	{"0","",""};
lookup1("1029C") ->	{"0","",""};
lookup1("102A") ->	{"0","",""};
lookup1("102A0") ->	{"0","",""};
lookup1("102A1") ->	{"0","",""};
lookup1("102A2") ->	{"0","",""};
lookup1("102A3") ->	{"0","",""};
lookup1("102A4") ->	{"0","",""};
lookup1("102A5") ->	{"0","",""};
lookup1("102A6") ->	{"0","",""};
lookup1("102A7") ->	{"0","",""};
lookup1("102A8") ->	{"0","",""};
lookup1("102A9") ->	{"0","",""};
lookup1("102AA") ->	{"0","",""};
lookup1("102AB") ->	{"0","",""};
lookup1("102AC") ->	{"0","",""};
lookup1("102AD") ->	{"0","",""};
lookup1("102AE") ->	{"0","",""};
lookup1("102AF") ->	{"0","",""};
lookup1("102B") ->	{"0","",""};
lookup1("102B0") ->	{"0","",""};
lookup1("102B1") ->	{"0","",""};
lookup1("102B2") ->	{"0","",""};
lookup1("102B3") ->	{"0","",""};
lookup1("102B4") ->	{"0","",""};
lookup1("102B5") ->	{"0","",""};
lookup1("102B6") ->	{"0","",""};
lookup1("102B7") ->	{"0","",""};
lookup1("102B8") ->	{"0","",""};
lookup1("102B9") ->	{"0","",""};
lookup1("102BA") ->	{"0","",""};
lookup1("102BB") ->	{"0","",""};
lookup1("102BC") ->	{"0","",""};
lookup1("102BD") ->	{"0","",""};
lookup1("102BE") ->	{"0","",""};
lookup1("102BF") ->	{"0","",""};
lookup1("102C") ->	{"0","",""};
lookup1("102C0") ->	{"0","",""};
lookup1("102C1") ->	{"0","",""};
lookup1("102C2") ->	{"0","",""};
lookup1("102C3") ->	{"0","",""};
lookup1("102C4") ->	{"0","",""};
lookup1("102C5") ->	{"0","",""};
lookup1("102C6") ->	{"0","",""};
lookup1("102C7") ->	{"0","",""};
lookup1("102C8") ->	{"0","",""};
lookup1("102C9") ->	{"0","",""};
lookup1("102CA") ->	{"0","",""};
lookup1("102CB") ->	{"0","",""};
lookup1("102CC") ->	{"0","",""};
lookup1("102CD") ->	{"0","",""};
lookup1("102CE") ->	{"0","",""};
lookup1("102CF") ->	{"0","",""};
lookup1("102D") ->	{"0","",""};
lookup1("102D0") ->	{"0","",""};
lookup1("102E") ->	{"0","",""};
lookup1("102E0") ->	{"220","",""};
lookup1("102E1") ->	{"0","",""};
lookup1("102E2") ->	{"0","",""};
lookup1("102E3") ->	{"0","",""};
lookup1("102E4") ->	{"0","",""};
lookup1("102E5") ->	{"0","",""};
lookup1("102E6") ->	{"0","",""};
lookup1("102E7") ->	{"0","",""};
lookup1("102E8") ->	{"0","",""};
lookup1("102E9") ->	{"0","",""};
lookup1("102EA") ->	{"0","",""};
lookup1("102EB") ->	{"0","",""};
lookup1("102EC") ->	{"0","",""};
lookup1("102ED") ->	{"0","",""};
lookup1("102EE") ->	{"0","",""};
lookup1("102EF") ->	{"0","",""};
lookup1("102F") ->	{"0","",""};
lookup1("102F0") ->	{"0","",""};
lookup1("102F1") ->	{"0","",""};
lookup1("102F2") ->	{"0","",""};
lookup1("102F3") ->	{"0","",""};
lookup1("102F4") ->	{"0","",""};
lookup1("102F5") ->	{"0","",""};
lookup1("102F6") ->	{"0","",""};
lookup1("102F7") ->	{"0","",""};
lookup1("102F8") ->	{"0","",""};
lookup1("102F9") ->	{"0","",""};
lookup1("102FA") ->	{"0","",""};
lookup1("102FB") ->	{"0","",""};
lookup1("1030") ->	{"0","",""};
lookup1("10300") ->	{"0","",""};
lookup1("10301") ->	{"0","",""};
lookup1("10302") ->	{"0","",""};
lookup1("10303") ->	{"0","",""};
lookup1("10304") ->	{"0","",""};
lookup1("10305") ->	{"0","",""};
lookup1("10306") ->	{"0","",""};
lookup1("10307") ->	{"0","",""};
lookup1("10308") ->	{"0","",""};
lookup1("10309") ->	{"0","",""};
lookup1("1030A") ->	{"0","",""};
lookup1("1030B") ->	{"0","",""};
lookup1("1030C") ->	{"0","",""};
lookup1("1030D") ->	{"0","",""};
lookup1("1030E") ->	{"0","",""};
lookup1("1030F") ->	{"0","",""};
lookup1("1031") ->	{"0","",""};
lookup1("10310") ->	{"0","",""};
lookup1("10311") ->	{"0","",""};
lookup1("10312") ->	{"0","",""};
lookup1("10313") ->	{"0","",""};
lookup1("10314") ->	{"0","",""};
lookup1("10315") ->	{"0","",""};
lookup1("10316") ->	{"0","",""};
lookup1("10317") ->	{"0","",""};
lookup1("10318") ->	{"0","",""};
lookup1("10319") ->	{"0","",""};
lookup1("1031A") ->	{"0","",""};
lookup1("1031B") ->	{"0","",""};
lookup1("1031C") ->	{"0","",""};
lookup1("1031D") ->	{"0","",""};
lookup1("1031E") ->	{"0","",""};
lookup1("1031F") ->	{"0","",""};
lookup1("1032") ->	{"0","",""};
lookup1("10320") ->	{"0","",""};
lookup1("10321") ->	{"0","",""};
lookup1("10322") ->	{"0","",""};
lookup1("10323") ->	{"0","",""};
lookup1("1033") ->	{"0","",""};
lookup1("10330") ->	{"0","",""};
lookup1("10331") ->	{"0","",""};
lookup1("10332") ->	{"0","",""};
lookup1("10333") ->	{"0","",""};
lookup1("10334") ->	{"0","",""};
lookup1("10335") ->	{"0","",""};
lookup1("10336") ->	{"0","",""};
lookup1("10337") ->	{"0","",""};
lookup1("10338") ->	{"0","",""};
lookup1("10339") ->	{"0","",""};
lookup1("1033A") ->	{"0","",""};
lookup1("1033B") ->	{"0","",""};
lookup1("1033C") ->	{"0","",""};
lookup1("1033D") ->	{"0","",""};
lookup1("1033E") ->	{"0","",""};
lookup1("1033F") ->	{"0","",""};
lookup1("1034") ->	{"0","",""};
lookup1("10340") ->	{"0","",""};
lookup1("10341") ->	{"0","",""};
lookup1("10342") ->	{"0","",""};
lookup1("10343") ->	{"0","",""};
lookup1("10344") ->	{"0","",""};
lookup1("10345") ->	{"0","",""};
lookup1("10346") ->	{"0","",""};
lookup1("10347") ->	{"0","",""};
lookup1("10348") ->	{"0","",""};
lookup1("10349") ->	{"0","",""};
lookup1("1034A") ->	{"0","",""};
lookup1("1035") ->	{"0","",""};
lookup1("10350") ->	{"0","",""};
lookup1("10351") ->	{"0","",""};
lookup1("10352") ->	{"0","",""};
lookup1("10353") ->	{"0","",""};
lookup1("10354") ->	{"0","",""};
lookup1("10355") ->	{"0","",""};
lookup1("10356") ->	{"0","",""};
lookup1("10357") ->	{"0","",""};
lookup1("10358") ->	{"0","",""};
lookup1("10359") ->	{"0","",""};
lookup1("1035A") ->	{"0","",""};
lookup1("1035B") ->	{"0","",""};
lookup1("1035C") ->	{"0","",""};
lookup1("1035D") ->	{"0","",""};
lookup1("1035E") ->	{"0","",""};
lookup1("1035F") ->	{"0","",""};
lookup1("1036") ->	{"0","",""};
lookup1("10360") ->	{"0","",""};
lookup1("10361") ->	{"0","",""};
lookup1("10362") ->	{"0","",""};
lookup1("10363") ->	{"0","",""};
lookup1("10364") ->	{"0","",""};
lookup1("10365") ->	{"0","",""};
lookup1("10366") ->	{"0","",""};
lookup1("10367") ->	{"0","",""};
lookup1("10368") ->	{"0","",""};
lookup1("10369") ->	{"0","",""};
lookup1("1036A") ->	{"0","",""};
lookup1("1036B") ->	{"0","",""};
lookup1("1036C") ->	{"0","",""};
lookup1("1036D") ->	{"0","",""};
lookup1("1036E") ->	{"0","",""};
lookup1("1036F") ->	{"0","",""};
lookup1("1037") ->	{"7","",""};
lookup1("10370") ->	{"0","",""};
lookup1("10371") ->	{"0","",""};
lookup1("10372") ->	{"0","",""};
lookup1("10373") ->	{"0","",""};
lookup1("10374") ->	{"0","",""};
lookup1("10375") ->	{"0","",""};
lookup1("10376") ->	{"230","",""};
lookup1("10377") ->	{"230","",""};
lookup1("10378") ->	{"230","",""};
lookup1("10379") ->	{"230","",""};
lookup1("1037A") ->	{"230","",""};
lookup1("1038") ->	{"0","",""};
lookup1("10380") ->	{"0","",""};
lookup1("10381") ->	{"0","",""};
lookup1("10382") ->	{"0","",""};
lookup1("10383") ->	{"0","",""};
lookup1("10384") ->	{"0","",""};
lookup1("10385") ->	{"0","",""};
lookup1("10386") ->	{"0","",""};
lookup1("10387") ->	{"0","",""};
lookup1("10388") ->	{"0","",""};
lookup1("10389") ->	{"0","",""};
lookup1("1038A") ->	{"0","",""};
lookup1("1038B") ->	{"0","",""};
lookup1("1038C") ->	{"0","",""};
lookup1("1038D") ->	{"0","",""};
lookup1("1038E") ->	{"0","",""};
lookup1("1038F") ->	{"0","",""};
lookup1("1039") ->	{"9","",""};
lookup1("10390") ->	{"0","",""};
lookup1("10391") ->	{"0","",""};
lookup1("10392") ->	{"0","",""};
lookup1("10393") ->	{"0","",""};
lookup1("10394") ->	{"0","",""};
lookup1("10395") ->	{"0","",""};
lookup1("10396") ->	{"0","",""};
lookup1("10397") ->	{"0","",""};
lookup1("10398") ->	{"0","",""};
lookup1("10399") ->	{"0","",""};
lookup1("1039A") ->	{"0","",""};
lookup1("1039B") ->	{"0","",""};
lookup1("1039C") ->	{"0","",""};
lookup1("1039D") ->	{"0","",""};
lookup1("1039F") ->	{"0","",""};
lookup1("103A") ->	{"9","",""};
lookup1("103A0") ->	{"0","",""};
lookup1("103A1") ->	{"0","",""};
lookup1("103A2") ->	{"0","",""};
lookup1("103A3") ->	{"0","",""};
lookup1("103A4") ->	{"0","",""};
lookup1("103A5") ->	{"0","",""};
lookup1("103A6") ->	{"0","",""};
lookup1("103A7") ->	{"0","",""};
lookup1("103A8") ->	{"0","",""};
lookup1("103A9") ->	{"0","",""};
lookup1("103AA") ->	{"0","",""};
lookup1("103AB") ->	{"0","",""};
lookup1("103AC") ->	{"0","",""};
lookup1("103AD") ->	{"0","",""};
lookup1("103AE") ->	{"0","",""};
lookup1("103AF") ->	{"0","",""};
lookup1("103B") ->	{"0","",""};
lookup1("103B0") ->	{"0","",""};
lookup1("103B1") ->	{"0","",""};
lookup1("103B2") ->	{"0","",""};
lookup1("103B3") ->	{"0","",""};
lookup1("103B4") ->	{"0","",""};
lookup1("103B5") ->	{"0","",""};
lookup1("103B6") ->	{"0","",""};
lookup1("103B7") ->	{"0","",""};
lookup1("103B8") ->	{"0","",""};
lookup1("103B9") ->	{"0","",""};
lookup1("103BA") ->	{"0","",""};
lookup1("103BB") ->	{"0","",""};
lookup1("103BC") ->	{"0","",""};
lookup1("103BD") ->	{"0","",""};
lookup1("103BE") ->	{"0","",""};
lookup1("103BF") ->	{"0","",""};
lookup1("103C") ->	{"0","",""};
lookup1("103C0") ->	{"0","",""};
lookup1("103C1") ->	{"0","",""};
lookup1("103C2") ->	{"0","",""};
lookup1("103C3") ->	{"0","",""};
lookup1("103C8") ->	{"0","",""};
lookup1("103C9") ->	{"0","",""};
lookup1("103CA") ->	{"0","",""};
lookup1("103CB") ->	{"0","",""};
lookup1("103CC") ->	{"0","",""};
lookup1("103CD") ->	{"0","",""};
lookup1("103CE") ->	{"0","",""};
lookup1("103CF") ->	{"0","",""};
lookup1("103D") ->	{"0","",""};
lookup1("103D0") ->	{"0","",""};
lookup1("103D1") ->	{"0","",""};
lookup1("103D2") ->	{"0","",""};
lookup1("103D3") ->	{"0","",""};
lookup1("103D4") ->	{"0","",""};
lookup1("103D5") ->	{"0","",""};
lookup1("103E") ->	{"0","",""};
lookup1("103F") ->	{"0","",""};
lookup1("1040") ->	{"0","",""};
lookup1("10400") ->	{"0","","10428"};
lookup1("10401") ->	{"0","","10429"};
lookup1("10402") ->	{"0","","1042A"};
lookup1("10403") ->	{"0","","1042B"};
lookup1("10404") ->	{"0","","1042C"};
lookup1("10405") ->	{"0","","1042D"};
lookup1("10406") ->	{"0","","1042E"};
lookup1("10407") ->	{"0","","1042F"};
lookup1("10408") ->	{"0","","10430"};
lookup1("10409") ->	{"0","","10431"};
lookup1("1040A") ->	{"0","","10432"};
lookup1("1040B") ->	{"0","","10433"};
lookup1("1040C") ->	{"0","","10434"};
lookup1("1040D") ->	{"0","","10435"};
lookup1("1040E") ->	{"0","","10436"};
lookup1("1040F") ->	{"0","","10437"};
lookup1("1041") ->	{"0","",""};
lookup1("10410") ->	{"0","","10438"};
lookup1("10411") ->	{"0","","10439"};
lookup1("10412") ->	{"0","","1043A"};
lookup1("10413") ->	{"0","","1043B"};
lookup1("10414") ->	{"0","","1043C"};
lookup1("10415") ->	{"0","","1043D"};
lookup1("10416") ->	{"0","","1043E"};
lookup1("10417") ->	{"0","","1043F"};
lookup1("10418") ->	{"0","","10440"};
lookup1("10419") ->	{"0","","10441"};
lookup1("1041A") ->	{"0","","10442"};
lookup1("1041B") ->	{"0","","10443"};
lookup1("1041C") ->	{"0","","10444"};
lookup1("1041D") ->	{"0","","10445"};
lookup1("1041E") ->	{"0","","10446"};
lookup1("1041F") ->	{"0","","10447"};
lookup1("1042") ->	{"0","",""};
lookup1("10420") ->	{"0","","10448"};
lookup1("10421") ->	{"0","","10449"};
lookup1("10422") ->	{"0","","1044A"};
lookup1("10423") ->	{"0","","1044B"};
lookup1("10424") ->	{"0","","1044C"};
lookup1("10425") ->	{"0","","1044D"};
lookup1("10426") ->	{"0","","1044E"};
lookup1("10427") ->	{"0","","1044F"};
lookup1("10428") ->	{"0","",""};
lookup1("10429") ->	{"0","",""};
lookup1("1042A") ->	{"0","",""};
lookup1("1042B") ->	{"0","",""};
lookup1("1042C") ->	{"0","",""};
lookup1("1042D") ->	{"0","",""};
lookup1("1042E") ->	{"0","",""};
lookup1("1042F") ->	{"0","",""};
lookup1("1043") ->	{"0","",""};
lookup1("10430") ->	{"0","",""};
lookup1("10431") ->	{"0","",""};
lookup1("10432") ->	{"0","",""};
lookup1("10433") ->	{"0","",""};
lookup1("10434") ->	{"0","",""};
lookup1("10435") ->	{"0","",""};
lookup1("10436") ->	{"0","",""};
lookup1("10437") ->	{"0","",""};
lookup1("10438") ->	{"0","",""};
lookup1("10439") ->	{"0","",""};
lookup1("1043A") ->	{"0","",""};
lookup1("1043B") ->	{"0","",""};
lookup1("1043C") ->	{"0","",""};
lookup1("1043D") ->	{"0","",""};
lookup1("1043E") ->	{"0","",""};
lookup1("1043F") ->	{"0","",""};
lookup1("1044") ->	{"0","",""};
lookup1("10440") ->	{"0","",""};
lookup1("10441") ->	{"0","",""};
lookup1("10442") ->	{"0","",""};
lookup1("10443") ->	{"0","",""};
lookup1("10444") ->	{"0","",""};
lookup1("10445") ->	{"0","",""};
lookup1("10446") ->	{"0","",""};
lookup1("10447") ->	{"0","",""};
lookup1("10448") ->	{"0","",""};
lookup1("10449") ->	{"0","",""};
lookup1("1044A") ->	{"0","",""};
lookup1("1044B") ->	{"0","",""};
lookup1("1044C") ->	{"0","",""};
lookup1("1044D") ->	{"0","",""};
lookup1("1044E") ->	{"0","",""};
lookup1("1044F") ->	{"0","",""};
lookup1("1045") ->	{"0","",""};
lookup1("10450") ->	{"0","",""};
lookup1("10451") ->	{"0","",""};
lookup1("10452") ->	{"0","",""};
lookup1("10453") ->	{"0","",""};
lookup1("10454") ->	{"0","",""};
lookup1("10455") ->	{"0","",""};
lookup1("10456") ->	{"0","",""};
lookup1("10457") ->	{"0","",""};
lookup1("10458") ->	{"0","",""};
lookup1("10459") ->	{"0","",""};
lookup1("1045A") ->	{"0","",""};
lookup1("1045B") ->	{"0","",""};
lookup1("1045C") ->	{"0","",""};
lookup1("1045D") ->	{"0","",""};
lookup1("1045E") ->	{"0","",""};
lookup1("1045F") ->	{"0","",""};
lookup1("1046") ->	{"0","",""};
lookup1("10460") ->	{"0","",""};
lookup1("10461") ->	{"0","",""};
lookup1("10462") ->	{"0","",""};
lookup1("10463") ->	{"0","",""};
lookup1("10464") ->	{"0","",""};
lookup1("10465") ->	{"0","",""};
lookup1("10466") ->	{"0","",""};
lookup1("10467") ->	{"0","",""};
lookup1("10468") ->	{"0","",""};
lookup1("10469") ->	{"0","",""};
lookup1("1046A") ->	{"0","",""};
lookup1("1046B") ->	{"0","",""};
lookup1("1046C") ->	{"0","",""};
lookup1("1046D") ->	{"0","",""};
lookup1("1046E") ->	{"0","",""};
lookup1("1046F") ->	{"0","",""};
lookup1("1047") ->	{"0","",""};
lookup1("10470") ->	{"0","",""};
lookup1("10471") ->	{"0","",""};
lookup1("10472") ->	{"0","",""};
lookup1("10473") ->	{"0","",""};
lookup1("10474") ->	{"0","",""};
lookup1("10475") ->	{"0","",""};
lookup1("10476") ->	{"0","",""};
lookup1("10477") ->	{"0","",""};
lookup1("10478") ->	{"0","",""};
lookup1("10479") ->	{"0","",""};
lookup1("1047A") ->	{"0","",""};
lookup1("1047B") ->	{"0","",""};
lookup1("1047C") ->	{"0","",""};
lookup1("1047D") ->	{"0","",""};
lookup1("1047E") ->	{"0","",""};
lookup1("1047F") ->	{"0","",""};
lookup1("1048") ->	{"0","",""};
lookup1("10480") ->	{"0","",""};
lookup1("10481") ->	{"0","",""};
lookup1("10482") ->	{"0","",""};
lookup1("10483") ->	{"0","",""};
lookup1("10484") ->	{"0","",""};
lookup1("10485") ->	{"0","",""};
lookup1("10486") ->	{"0","",""};
lookup1("10487") ->	{"0","",""};
lookup1("10488") ->	{"0","",""};
lookup1("10489") ->	{"0","",""};
lookup1("1048A") ->	{"0","",""};
lookup1("1048B") ->	{"0","",""};
lookup1("1048C") ->	{"0","",""};
lookup1("1048D") ->	{"0","",""};
lookup1("1048E") ->	{"0","",""};
lookup1("1048F") ->	{"0","",""};
lookup1("1049") ->	{"0","",""};
lookup1("10490") ->	{"0","",""};
lookup1("10491") ->	{"0","",""};
lookup1("10492") ->	{"0","",""};
lookup1("10493") ->	{"0","",""};
lookup1("10494") ->	{"0","",""};
lookup1("10495") ->	{"0","",""};
lookup1("10496") ->	{"0","",""};
lookup1("10497") ->	{"0","",""};
lookup1("10498") ->	{"0","",""};
lookup1("10499") ->	{"0","",""};
lookup1("1049A") ->	{"0","",""};
lookup1("1049B") ->	{"0","",""};
lookup1("1049C") ->	{"0","",""};
lookup1("1049D") ->	{"0","",""};
lookup1("104A") ->	{"0","",""};
lookup1("104A0") ->	{"0","",""};
lookup1("104A1") ->	{"0","",""};
lookup1("104A2") ->	{"0","",""};
lookup1("104A3") ->	{"0","",""};
lookup1("104A4") ->	{"0","",""};
lookup1("104A5") ->	{"0","",""};
lookup1("104A6") ->	{"0","",""};
lookup1("104A7") ->	{"0","",""};
lookup1("104A8") ->	{"0","",""};
lookup1("104A9") ->	{"0","",""};
lookup1("104B") ->	{"0","",""};
lookup1("104C") ->	{"0","",""};
lookup1("104D") ->	{"0","",""};
lookup1("104E") ->	{"0","",""};
lookup1("104F") ->	{"0","",""};
lookup1("1050") ->	{"0","",""};
lookup1("10500") ->	{"0","",""};
lookup1("10501") ->	{"0","",""};
lookup1("10502") ->	{"0","",""};
lookup1("10503") ->	{"0","",""};
lookup1("10504") ->	{"0","",""};
lookup1("10505") ->	{"0","",""};
lookup1("10506") ->	{"0","",""};
lookup1("10507") ->	{"0","",""};
lookup1("10508") ->	{"0","",""};
lookup1("10509") ->	{"0","",""};
lookup1("1050A") ->	{"0","",""};
lookup1("1050B") ->	{"0","",""};
lookup1("1050C") ->	{"0","",""};
lookup1("1050D") ->	{"0","",""};
lookup1("1050E") ->	{"0","",""};
lookup1("1050F") ->	{"0","",""};
lookup1("1051") ->	{"0","",""};
lookup1("10510") ->	{"0","",""};
lookup1("10511") ->	{"0","",""};
lookup1("10512") ->	{"0","",""};
lookup1("10513") ->	{"0","",""};
lookup1("10514") ->	{"0","",""};
lookup1("10515") ->	{"0","",""};
lookup1("10516") ->	{"0","",""};
lookup1("10517") ->	{"0","",""};
lookup1("10518") ->	{"0","",""};
lookup1("10519") ->	{"0","",""};
lookup1("1051A") ->	{"0","",""};
lookup1("1051B") ->	{"0","",""};
lookup1("1051C") ->	{"0","",""};
lookup1("1051D") ->	{"0","",""};
lookup1("1051E") ->	{"0","",""};
lookup1("1051F") ->	{"0","",""};
lookup1("1052") ->	{"0","",""};
lookup1("10520") ->	{"0","",""};
lookup1("10521") ->	{"0","",""};
lookup1("10522") ->	{"0","",""};
lookup1("10523") ->	{"0","",""};
lookup1("10524") ->	{"0","",""};
lookup1("10525") ->	{"0","",""};
lookup1("10526") ->	{"0","",""};
lookup1("10527") ->	{"0","",""};
lookup1("1053") ->	{"0","",""};
lookup1("10530") ->	{"0","",""};
lookup1("10531") ->	{"0","",""};
lookup1("10532") ->	{"0","",""};
lookup1("10533") ->	{"0","",""};
lookup1("10534") ->	{"0","",""};
lookup1("10535") ->	{"0","",""};
lookup1("10536") ->	{"0","",""};
lookup1("10537") ->	{"0","",""};
lookup1("10538") ->	{"0","",""};
lookup1("10539") ->	{"0","",""};
lookup1("1053A") ->	{"0","",""};
lookup1("1053B") ->	{"0","",""};
lookup1("1053C") ->	{"0","",""};
lookup1("1053D") ->	{"0","",""};
lookup1("1053E") ->	{"0","",""};
lookup1("1053F") ->	{"0","",""};
lookup1("1054") ->	{"0","",""};
lookup1("10540") ->	{"0","",""};
lookup1("10541") ->	{"0","",""};
lookup1("10542") ->	{"0","",""};
lookup1("10543") ->	{"0","",""};
lookup1("10544") ->	{"0","",""};
lookup1("10545") ->	{"0","",""};
lookup1("10546") ->	{"0","",""};
lookup1("10547") ->	{"0","",""};
lookup1("10548") ->	{"0","",""};
lookup1("10549") ->	{"0","",""};
lookup1("1054A") ->	{"0","",""};
lookup1("1054B") ->	{"0","",""};
lookup1("1054C") ->	{"0","",""};
lookup1("1054D") ->	{"0","",""};
lookup1("1054E") ->	{"0","",""};
lookup1("1054F") ->	{"0","",""};
lookup1("1055") ->	{"0","",""};
lookup1("10550") ->	{"0","",""};
lookup1("10551") ->	{"0","",""};
lookup1("10552") ->	{"0","",""};
lookup1("10553") ->	{"0","",""};
lookup1("10554") ->	{"0","",""};
lookup1("10555") ->	{"0","",""};
lookup1("10556") ->	{"0","",""};
lookup1("10557") ->	{"0","",""};
lookup1("10558") ->	{"0","",""};
lookup1("10559") ->	{"0","",""};
lookup1("1055A") ->	{"0","",""};
lookup1("1055B") ->	{"0","",""};
lookup1("1055C") ->	{"0","",""};
lookup1("1055D") ->	{"0","",""};
lookup1("1055E") ->	{"0","",""};
lookup1("1055F") ->	{"0","",""};
lookup1("1056") ->	{"0","",""};
lookup1("10560") ->	{"0","",""};
lookup1("10561") ->	{"0","",""};
lookup1("10562") ->	{"0","",""};
lookup1("10563") ->	{"0","",""};
lookup1("1056F") ->	{"0","",""};
lookup1("1057") ->	{"0","",""};
lookup1("1058") ->	{"0","",""};
lookup1("1059") ->	{"0","",""};
lookup1("105A") ->	{"0","",""};
lookup1("105B") ->	{"0","",""};
lookup1("105C") ->	{"0","",""};
lookup1("105D") ->	{"0","",""};
lookup1("105E") ->	{"0","",""};
lookup1("105F") ->	{"0","",""};
lookup1("1060") ->	{"0","",""};
lookup1("10600") ->	{"0","",""};
lookup1("10601") ->	{"0","",""};
lookup1("10602") ->	{"0","",""};
lookup1("10603") ->	{"0","",""};
lookup1("10604") ->	{"0","",""};
lookup1("10605") ->	{"0","",""};
lookup1("10606") ->	{"0","",""};
lookup1("10607") ->	{"0","",""};
lookup1("10608") ->	{"0","",""};
lookup1("10609") ->	{"0","",""};
lookup1("1060A") ->	{"0","",""};
lookup1("1060B") ->	{"0","",""};
lookup1("1060C") ->	{"0","",""};
lookup1("1060D") ->	{"0","",""};
lookup1("1060E") ->	{"0","",""};
lookup1("1060F") ->	{"0","",""};
lookup1("1061") ->	{"0","",""};
lookup1("10610") ->	{"0","",""};
lookup1("10611") ->	{"0","",""};
lookup1("10612") ->	{"0","",""};
lookup1("10613") ->	{"0","",""};
lookup1("10614") ->	{"0","",""};
lookup1("10615") ->	{"0","",""};
lookup1("10616") ->	{"0","",""};
lookup1("10617") ->	{"0","",""};
lookup1("10618") ->	{"0","",""};
lookup1("10619") ->	{"0","",""};
lookup1("1061A") ->	{"0","",""};
lookup1("1061B") ->	{"0","",""};
lookup1("1061C") ->	{"0","",""};
lookup1("1061D") ->	{"0","",""};
lookup1("1061E") ->	{"0","",""};
lookup1("1061F") ->	{"0","",""};
lookup1("1062") ->	{"0","",""};
lookup1("10620") ->	{"0","",""};
lookup1("10621") ->	{"0","",""};
lookup1("10622") ->	{"0","",""};
lookup1("10623") ->	{"0","",""};
lookup1("10624") ->	{"0","",""};
lookup1("10625") ->	{"0","",""};
lookup1("10626") ->	{"0","",""};
lookup1("10627") ->	{"0","",""};
lookup1("10628") ->	{"0","",""};
lookup1("10629") ->	{"0","",""};
lookup1("1062A") ->	{"0","",""};
lookup1("1062B") ->	{"0","",""};
lookup1("1062C") ->	{"0","",""};
lookup1("1062D") ->	{"0","",""};
lookup1("1062E") ->	{"0","",""};
lookup1("1062F") ->	{"0","",""};
lookup1("1063") ->	{"0","",""};
lookup1("10630") ->	{"0","",""};
lookup1("10631") ->	{"0","",""};
lookup1("10632") ->	{"0","",""};
lookup1("10633") ->	{"0","",""};
lookup1("10634") ->	{"0","",""};
lookup1("10635") ->	{"0","",""};
lookup1("10636") ->	{"0","",""};
lookup1("10637") ->	{"0","",""};
lookup1("10638") ->	{"0","",""};
lookup1("10639") ->	{"0","",""};
lookup1("1063A") ->	{"0","",""};
lookup1("1063B") ->	{"0","",""};
lookup1("1063C") ->	{"0","",""};
lookup1("1063D") ->	{"0","",""};
lookup1("1063E") ->	{"0","",""};
lookup1("1063F") ->	{"0","",""};
lookup1("1064") ->	{"0","",""};
lookup1("10640") ->	{"0","",""};
lookup1("10641") ->	{"0","",""};
lookup1("10642") ->	{"0","",""};
lookup1("10643") ->	{"0","",""};
lookup1("10644") ->	{"0","",""};
lookup1("10645") ->	{"0","",""};
lookup1("10646") ->	{"0","",""};
lookup1("10647") ->	{"0","",""};
lookup1("10648") ->	{"0","",""};
lookup1("10649") ->	{"0","",""};
lookup1("1064A") ->	{"0","",""};
lookup1("1064B") ->	{"0","",""};
lookup1("1064C") ->	{"0","",""};
lookup1("1064D") ->	{"0","",""};
lookup1("1064E") ->	{"0","",""};
lookup1("1064F") ->	{"0","",""};
lookup1("1065") ->	{"0","",""};
lookup1("10650") ->	{"0","",""};
lookup1("10651") ->	{"0","",""};
lookup1("10652") ->	{"0","",""};
lookup1("10653") ->	{"0","",""};
lookup1("10654") ->	{"0","",""};
lookup1("10655") ->	{"0","",""};
lookup1("10656") ->	{"0","",""};
lookup1("10657") ->	{"0","",""};
lookup1("10658") ->	{"0","",""};
lookup1("10659") ->	{"0","",""};
lookup1("1065A") ->	{"0","",""};
lookup1("1065B") ->	{"0","",""};
lookup1("1065C") ->	{"0","",""};
lookup1("1065D") ->	{"0","",""};
lookup1("1065E") ->	{"0","",""};
lookup1("1065F") ->	{"0","",""};
lookup1("1066") ->	{"0","",""};
lookup1("10660") ->	{"0","",""};
lookup1("10661") ->	{"0","",""};
lookup1("10662") ->	{"0","",""};
lookup1("10663") ->	{"0","",""};
lookup1("10664") ->	{"0","",""};
lookup1("10665") ->	{"0","",""};
lookup1("10666") ->	{"0","",""};
lookup1("10667") ->	{"0","",""};
lookup1("10668") ->	{"0","",""};
lookup1("10669") ->	{"0","",""};
lookup1("1066A") ->	{"0","",""};
lookup1("1066B") ->	{"0","",""};
lookup1("1066C") ->	{"0","",""};
lookup1("1066D") ->	{"0","",""};
lookup1("1066E") ->	{"0","",""};
lookup1("1066F") ->	{"0","",""};
lookup1("1067") ->	{"0","",""};
lookup1("10670") ->	{"0","",""};
lookup1("10671") ->	{"0","",""};
lookup1("10672") ->	{"0","",""};
lookup1("10673") ->	{"0","",""};
lookup1("10674") ->	{"0","",""};
lookup1("10675") ->	{"0","",""};
lookup1("10676") ->	{"0","",""};
lookup1("10677") ->	{"0","",""};
lookup1("10678") ->	{"0","",""};
lookup1("10679") ->	{"0","",""};
lookup1("1067A") ->	{"0","",""};
lookup1("1067B") ->	{"0","",""};
lookup1("1067C") ->	{"0","",""};
lookup1("1067D") ->	{"0","",""};
lookup1("1067E") ->	{"0","",""};
lookup1("1067F") ->	{"0","",""};
lookup1("1068") ->	{"0","",""};
lookup1("10680") ->	{"0","",""};
lookup1("10681") ->	{"0","",""};
lookup1("10682") ->	{"0","",""};
lookup1("10683") ->	{"0","",""};
lookup1("10684") ->	{"0","",""};
lookup1("10685") ->	{"0","",""};
lookup1("10686") ->	{"0","",""};
lookup1("10687") ->	{"0","",""};
lookup1("10688") ->	{"0","",""};
lookup1("10689") ->	{"0","",""};
lookup1("1068A") ->	{"0","",""};
lookup1("1068B") ->	{"0","",""};
lookup1("1068C") ->	{"0","",""};
lookup1("1068D") ->	{"0","",""};
lookup1("1068E") ->	{"0","",""};
lookup1("1068F") ->	{"0","",""};
lookup1("1069") ->	{"0","",""};
lookup1("10690") ->	{"0","",""};
lookup1("10691") ->	{"0","",""};
lookup1("10692") ->	{"0","",""};
lookup1("10693") ->	{"0","",""};
lookup1("10694") ->	{"0","",""};
lookup1("10695") ->	{"0","",""};
lookup1("10696") ->	{"0","",""};
lookup1("10697") ->	{"0","",""};
lookup1("10698") ->	{"0","",""};
lookup1("10699") ->	{"0","",""};
lookup1("1069A") ->	{"0","",""};
lookup1("1069B") ->	{"0","",""};
lookup1("1069C") ->	{"0","",""};
lookup1("1069D") ->	{"0","",""};
lookup1("1069E") ->	{"0","",""};
lookup1("1069F") ->	{"0","",""};
lookup1("106A") ->	{"0","",""};
lookup1("106A0") ->	{"0","",""};
lookup1("106A1") ->	{"0","",""};
lookup1("106A2") ->	{"0","",""};
lookup1("106A3") ->	{"0","",""};
lookup1("106A4") ->	{"0","",""};
lookup1("106A5") ->	{"0","",""};
lookup1("106A6") ->	{"0","",""};
lookup1("106A7") ->	{"0","",""};
lookup1("106A8") ->	{"0","",""};
lookup1("106A9") ->	{"0","",""};
lookup1("106AA") ->	{"0","",""};
lookup1("106AB") ->	{"0","",""};
lookup1("106AC") ->	{"0","",""};
lookup1("106AD") ->	{"0","",""};
lookup1("106AE") ->	{"0","",""};
lookup1("106AF") ->	{"0","",""};
lookup1("106B") ->	{"0","",""};
lookup1("106B0") ->	{"0","",""};
lookup1("106B1") ->	{"0","",""};
lookup1("106B2") ->	{"0","",""};
lookup1("106B3") ->	{"0","",""};
lookup1("106B4") ->	{"0","",""};
lookup1("106B5") ->	{"0","",""};
lookup1("106B6") ->	{"0","",""};
lookup1("106B7") ->	{"0","",""};
lookup1("106B8") ->	{"0","",""};
lookup1("106B9") ->	{"0","",""};
lookup1("106BA") ->	{"0","",""};
lookup1("106BB") ->	{"0","",""};
lookup1("106BC") ->	{"0","",""};
lookup1("106BD") ->	{"0","",""};
lookup1("106BE") ->	{"0","",""};
lookup1("106BF") ->	{"0","",""};
lookup1("106C") ->	{"0","",""};
lookup1("106C0") ->	{"0","",""};
lookup1("106C1") ->	{"0","",""};
lookup1("106C2") ->	{"0","",""};
lookup1("106C3") ->	{"0","",""};
lookup1("106C4") ->	{"0","",""};
lookup1("106C5") ->	{"0","",""};
lookup1("106C6") ->	{"0","",""};
lookup1("106C7") ->	{"0","",""};
lookup1("106C8") ->	{"0","",""};
lookup1("106C9") ->	{"0","",""};
lookup1("106CA") ->	{"0","",""};
lookup1("106CB") ->	{"0","",""};
lookup1("106CC") ->	{"0","",""};
lookup1("106CD") ->	{"0","",""};
lookup1("106CE") ->	{"0","",""};
lookup1("106CF") ->	{"0","",""};
lookup1("106D") ->	{"0","",""};
lookup1("106D0") ->	{"0","",""};
lookup1("106D1") ->	{"0","",""};
lookup1("106D2") ->	{"0","",""};
lookup1("106D3") ->	{"0","",""};
lookup1("106D4") ->	{"0","",""};
lookup1("106D5") ->	{"0","",""};
lookup1("106D6") ->	{"0","",""};
lookup1("106D7") ->	{"0","",""};
lookup1("106D8") ->	{"0","",""};
lookup1("106D9") ->	{"0","",""};
lookup1("106DA") ->	{"0","",""};
lookup1("106DB") ->	{"0","",""};
lookup1("106DC") ->	{"0","",""};
lookup1("106DD") ->	{"0","",""};
lookup1("106DE") ->	{"0","",""};
lookup1("106DF") ->	{"0","",""};
lookup1("106E") ->	{"0","",""};
lookup1("106E0") ->	{"0","",""};
lookup1("106E1") ->	{"0","",""};
lookup1("106E2") ->	{"0","",""};
lookup1("106E3") ->	{"0","",""};
lookup1("106E4") ->	{"0","",""};
lookup1("106E5") ->	{"0","",""};
lookup1("106E6") ->	{"0","",""};
lookup1("106E7") ->	{"0","",""};
lookup1("106E8") ->	{"0","",""};
lookup1("106E9") ->	{"0","",""};
lookup1("106EA") ->	{"0","",""};
lookup1("106EB") ->	{"0","",""};
lookup1("106EC") ->	{"0","",""};
lookup1("106ED") ->	{"0","",""};
lookup1("106EE") ->	{"0","",""};
lookup1("106EF") ->	{"0","",""};
lookup1("106F") ->	{"0","",""};
lookup1("106F0") ->	{"0","",""};
lookup1("106F1") ->	{"0","",""};
lookup1("106F2") ->	{"0","",""};
lookup1("106F3") ->	{"0","",""};
lookup1("106F4") ->	{"0","",""};
lookup1("106F5") ->	{"0","",""};
lookup1("106F6") ->	{"0","",""};
lookup1("106F7") ->	{"0","",""};
lookup1("106F8") ->	{"0","",""};
lookup1("106F9") ->	{"0","",""};
lookup1("106FA") ->	{"0","",""};
lookup1("106FB") ->	{"0","",""};
lookup1("106FC") ->	{"0","",""};
lookup1("106FD") ->	{"0","",""};
lookup1("106FE") ->	{"0","",""};
lookup1("106FF") ->	{"0","",""};
lookup1("1070") ->	{"0","",""};
lookup1("10700") ->	{"0","",""};
lookup1("10701") ->	{"0","",""};
lookup1("10702") ->	{"0","",""};
lookup1("10703") ->	{"0","",""};
lookup1("10704") ->	{"0","",""};
lookup1("10705") ->	{"0","",""};
lookup1("10706") ->	{"0","",""};
lookup1("10707") ->	{"0","",""};
lookup1("10708") ->	{"0","",""};
lookup1("10709") ->	{"0","",""};
lookup1("1070A") ->	{"0","",""};
lookup1("1070B") ->	{"0","",""};
lookup1("1070C") ->	{"0","",""};
lookup1("1070D") ->	{"0","",""};
lookup1("1070E") ->	{"0","",""};
lookup1("1070F") ->	{"0","",""};
lookup1("1071") ->	{"0","",""};
lookup1("10710") ->	{"0","",""};
lookup1("10711") ->	{"0","",""};
lookup1("10712") ->	{"0","",""};
lookup1("10713") ->	{"0","",""};
lookup1("10714") ->	{"0","",""};
lookup1("10715") ->	{"0","",""};
lookup1("10716") ->	{"0","",""};
lookup1("10717") ->	{"0","",""};
lookup1("10718") ->	{"0","",""};
lookup1("10719") ->	{"0","",""};
lookup1("1071A") ->	{"0","",""};
lookup1("1071B") ->	{"0","",""};
lookup1("1071C") ->	{"0","",""};
lookup1("1071D") ->	{"0","",""};
lookup1("1071E") ->	{"0","",""};
lookup1("1071F") ->	{"0","",""};
lookup1("1072") ->	{"0","",""};
lookup1("10720") ->	{"0","",""};
lookup1("10721") ->	{"0","",""};
lookup1("10722") ->	{"0","",""};
lookup1("10723") ->	{"0","",""};
lookup1("10724") ->	{"0","",""};
lookup1("10725") ->	{"0","",""};
lookup1("10726") ->	{"0","",""};
lookup1("10727") ->	{"0","",""};
lookup1("10728") ->	{"0","",""};
lookup1("10729") ->	{"0","",""};
lookup1("1072A") ->	{"0","",""};
lookup1("1072B") ->	{"0","",""};
lookup1("1072C") ->	{"0","",""};
lookup1("1072D") ->	{"0","",""};
lookup1("1072E") ->	{"0","",""};
lookup1("1072F") ->	{"0","",""};
lookup1("1073") ->	{"0","",""};
lookup1("10730") ->	{"0","",""};
lookup1("10731") ->	{"0","",""};
lookup1("10732") ->	{"0","",""};
lookup1("10733") ->	{"0","",""};
lookup1("10734") ->	{"0","",""};
lookup1("10735") ->	{"0","",""};
lookup1("10736") ->	{"0","",""};
lookup1("1074") ->	{"0","",""};
lookup1("10740") ->	{"0","",""};
lookup1("10741") ->	{"0","",""};
lookup1("10742") ->	{"0","",""};
lookup1("10743") ->	{"0","",""};
lookup1("10744") ->	{"0","",""};
lookup1("10745") ->	{"0","",""};
lookup1("10746") ->	{"0","",""};
lookup1("10747") ->	{"0","",""};
lookup1("10748") ->	{"0","",""};
lookup1("10749") ->	{"0","",""};
lookup1("1074A") ->	{"0","",""};
lookup1("1074B") ->	{"0","",""};
lookup1("1074C") ->	{"0","",""};
lookup1("1074D") ->	{"0","",""};
lookup1("1074E") ->	{"0","",""};
lookup1("1074F") ->	{"0","",""};
lookup1("1075") ->	{"0","",""};
lookup1("10750") ->	{"0","",""};
lookup1("10751") ->	{"0","",""};
lookup1("10752") ->	{"0","",""};
lookup1("10753") ->	{"0","",""};
lookup1("10754") ->	{"0","",""};
lookup1("10755") ->	{"0","",""};
lookup1("1076") ->	{"0","",""};
lookup1("10760") ->	{"0","",""};
lookup1("10761") ->	{"0","",""};
lookup1("10762") ->	{"0","",""};
lookup1("10763") ->	{"0","",""};
lookup1("10764") ->	{"0","",""};
lookup1("10765") ->	{"0","",""};
lookup1("10766") ->	{"0","",""};
lookup1("10767") ->	{"0","",""};
lookup1("1077") ->	{"0","",""};
lookup1("1078") ->	{"0","",""};
lookup1("1079") ->	{"0","",""};
lookup1("107A") ->	{"0","",""};
lookup1("107B") ->	{"0","",""};
lookup1("107C") ->	{"0","",""};
lookup1("107D") ->	{"0","",""};
lookup1("107E") ->	{"0","",""};
lookup1("107F") ->	{"0","",""};
lookup1("1080") ->	{"0","",""};
lookup1("10800") ->	{"0","",""};
lookup1("10801") ->	{"0","",""};
lookup1("10802") ->	{"0","",""};
lookup1("10803") ->	{"0","",""};
lookup1("10804") ->	{"0","",""};
lookup1("10805") ->	{"0","",""};
lookup1("10808") ->	{"0","",""};
lookup1("1080A") ->	{"0","",""};
lookup1("1080B") ->	{"0","",""};
lookup1("1080C") ->	{"0","",""};
lookup1("1080D") ->	{"0","",""};
lookup1("1080E") ->	{"0","",""};
lookup1("1080F") ->	{"0","",""};
lookup1("1081") ->	{"0","",""};
lookup1("10810") ->	{"0","",""};
lookup1("10811") ->	{"0","",""};
lookup1("10812") ->	{"0","",""};
lookup1("10813") ->	{"0","",""};
lookup1("10814") ->	{"0","",""};
lookup1("10815") ->	{"0","",""};
lookup1("10816") ->	{"0","",""};
lookup1("10817") ->	{"0","",""};
lookup1("10818") ->	{"0","",""};
lookup1("10819") ->	{"0","",""};
lookup1("1081A") ->	{"0","",""};
lookup1("1081B") ->	{"0","",""};
lookup1("1081C") ->	{"0","",""};
lookup1("1081D") ->	{"0","",""};
lookup1("1081E") ->	{"0","",""};
lookup1("1081F") ->	{"0","",""};
lookup1("1082") ->	{"0","",""};
lookup1("10820") ->	{"0","",""};
lookup1("10821") ->	{"0","",""};
lookup1("10822") ->	{"0","",""};
lookup1("10823") ->	{"0","",""};
lookup1("10824") ->	{"0","",""};
lookup1("10825") ->	{"0","",""};
lookup1("10826") ->	{"0","",""};
lookup1("10827") ->	{"0","",""};
lookup1("10828") ->	{"0","",""};
lookup1("10829") ->	{"0","",""};
lookup1("1082A") ->	{"0","",""};
lookup1("1082B") ->	{"0","",""};
lookup1("1082C") ->	{"0","",""};
lookup1("1082D") ->	{"0","",""};
lookup1("1082E") ->	{"0","",""};
lookup1("1082F") ->	{"0","",""};
lookup1("1083") ->	{"0","",""};
lookup1("10830") ->	{"0","",""};
lookup1("10831") ->	{"0","",""};
lookup1("10832") ->	{"0","",""};
lookup1("10833") ->	{"0","",""};
lookup1("10834") ->	{"0","",""};
lookup1("10835") ->	{"0","",""};
lookup1("10837") ->	{"0","",""};
lookup1("10838") ->	{"0","",""};
lookup1("1083C") ->	{"0","",""};
lookup1("1083F") ->	{"0","",""};
lookup1("1084") ->	{"0","",""};
lookup1("10840") ->	{"0","",""};
lookup1("10841") ->	{"0","",""};
lookup1("10842") ->	{"0","",""};
lookup1("10843") ->	{"0","",""};
lookup1("10844") ->	{"0","",""};
lookup1("10845") ->	{"0","",""};
lookup1("10846") ->	{"0","",""};
lookup1("10847") ->	{"0","",""};
lookup1("10848") ->	{"0","",""};
lookup1("10849") ->	{"0","",""};
lookup1("1084A") ->	{"0","",""};
lookup1("1084B") ->	{"0","",""};
lookup1("1084C") ->	{"0","",""};
lookup1("1084D") ->	{"0","",""};
lookup1("1084E") ->	{"0","",""};
lookup1("1084F") ->	{"0","",""};
lookup1("1085") ->	{"0","",""};
lookup1("10850") ->	{"0","",""};
lookup1("10851") ->	{"0","",""};
lookup1("10852") ->	{"0","",""};
lookup1("10853") ->	{"0","",""};
lookup1("10854") ->	{"0","",""};
lookup1("10855") ->	{"0","",""};
lookup1("10857") ->	{"0","",""};
lookup1("10858") ->	{"0","",""};
lookup1("10859") ->	{"0","",""};
lookup1("1085A") ->	{"0","",""};
lookup1("1085B") ->	{"0","",""};
lookup1("1085C") ->	{"0","",""};
lookup1("1085D") ->	{"0","",""};
lookup1("1085E") ->	{"0","",""};
lookup1("1085F") ->	{"0","",""};
lookup1("1086") ->	{"0","",""};
lookup1("10860") ->	{"0","",""};
lookup1("10861") ->	{"0","",""};
lookup1("10862") ->	{"0","",""};
lookup1("10863") ->	{"0","",""};
lookup1("10864") ->	{"0","",""};
lookup1("10865") ->	{"0","",""};
lookup1("10866") ->	{"0","",""};
lookup1("10867") ->	{"0","",""};
lookup1("10868") ->	{"0","",""};
lookup1("10869") ->	{"0","",""};
lookup1("1086A") ->	{"0","",""};
lookup1("1086B") ->	{"0","",""};
lookup1("1086C") ->	{"0","",""};
lookup1("1086D") ->	{"0","",""};
lookup1("1086E") ->	{"0","",""};
lookup1("1086F") ->	{"0","",""};
lookup1("1087") ->	{"0","",""};
lookup1("10870") ->	{"0","",""};
lookup1("10871") ->	{"0","",""};
lookup1("10872") ->	{"0","",""};
lookup1("10873") ->	{"0","",""};
lookup1("10874") ->	{"0","",""};
lookup1("10875") ->	{"0","",""};
lookup1("10876") ->	{"0","",""};
lookup1("10877") ->	{"0","",""};
lookup1("10878") ->	{"0","",""};
lookup1("10879") ->	{"0","",""};
lookup1("1087A") ->	{"0","",""};
lookup1("1087B") ->	{"0","",""};
lookup1("1087C") ->	{"0","",""};
lookup1("1087D") ->	{"0","",""};
lookup1("1087E") ->	{"0","",""};
lookup1("1087F") ->	{"0","",""};
lookup1("1088") ->	{"0","",""};
lookup1("10880") ->	{"0","",""};
lookup1("10881") ->	{"0","",""};
lookup1("10882") ->	{"0","",""};
lookup1("10883") ->	{"0","",""};
lookup1("10884") ->	{"0","",""};
lookup1("10885") ->	{"0","",""};
lookup1("10886") ->	{"0","",""};
lookup1("10887") ->	{"0","",""};
lookup1("10888") ->	{"0","",""};
lookup1("10889") ->	{"0","",""};
lookup1("1088A") ->	{"0","",""};
lookup1("1088B") ->	{"0","",""};
lookup1("1088C") ->	{"0","",""};
lookup1("1088D") ->	{"0","",""};
lookup1("1088E") ->	{"0","",""};
lookup1("1088F") ->	{"0","",""};
lookup1("1089") ->	{"0","",""};
lookup1("10890") ->	{"0","",""};
lookup1("10891") ->	{"0","",""};
lookup1("10892") ->	{"0","",""};
lookup1("10893") ->	{"0","",""};
lookup1("10894") ->	{"0","",""};
lookup1("10895") ->	{"0","",""};
lookup1("10896") ->	{"0","",""};
lookup1("10897") ->	{"0","",""};
lookup1("10898") ->	{"0","",""};
lookup1("10899") ->	{"0","",""};
lookup1("1089A") ->	{"0","",""};
lookup1("1089B") ->	{"0","",""};
lookup1("1089C") ->	{"0","",""};
lookup1("1089D") ->	{"0","",""};
lookup1("1089E") ->	{"0","",""};
lookup1("108A") ->	{"0","",""};
lookup1("108A7") ->	{"0","",""};
lookup1("108A8") ->	{"0","",""};
lookup1("108A9") ->	{"0","",""};
lookup1("108AA") ->	{"0","",""};
lookup1("108AB") ->	{"0","",""};
lookup1("108AC") ->	{"0","",""};
lookup1("108AD") ->	{"0","",""};
lookup1("108AE") ->	{"0","",""};
lookup1("108AF") ->	{"0","",""};
lookup1("108B") ->	{"0","",""};
lookup1("108C") ->	{"0","",""};
lookup1("108D") ->	{"220","",""};
lookup1("108E") ->	{"0","",""};
lookup1("108F") ->	{"0","",""};
lookup1("1090") ->	{"0","",""};
lookup1("10900") ->	{"0","",""};
lookup1("10901") ->	{"0","",""};
lookup1("10902") ->	{"0","",""};
lookup1("10903") ->	{"0","",""};
lookup1("10904") ->	{"0","",""};
lookup1("10905") ->	{"0","",""};
lookup1("10906") ->	{"0","",""};
lookup1("10907") ->	{"0","",""};
lookup1("10908") ->	{"0","",""};
lookup1("10909") ->	{"0","",""};
lookup1("1090A") ->	{"0","",""};
lookup1("1090B") ->	{"0","",""};
lookup1("1090C") ->	{"0","",""};
lookup1("1090D") ->	{"0","",""};
lookup1("1090E") ->	{"0","",""};
lookup1("1090F") ->	{"0","",""};
lookup1("1091") ->	{"0","",""};
lookup1("10910") ->	{"0","",""};
lookup1("10911") ->	{"0","",""};
lookup1("10912") ->	{"0","",""};
lookup1("10913") ->	{"0","",""};
lookup1("10914") ->	{"0","",""};
lookup1("10915") ->	{"0","",""};
lookup1("10916") ->	{"0","",""};
lookup1("10917") ->	{"0","",""};
lookup1("10918") ->	{"0","",""};
lookup1("10919") ->	{"0","",""};
lookup1("1091A") ->	{"0","",""};
lookup1("1091B") ->	{"0","",""};
lookup1("1091F") ->	{"0","",""};
lookup1("1092") ->	{"0","",""};
lookup1("10920") ->	{"0","",""};
lookup1("10921") ->	{"0","",""};
lookup1("10922") ->	{"0","",""};
lookup1("10923") ->	{"0","",""};
lookup1("10924") ->	{"0","",""};
lookup1("10925") ->	{"0","",""};
lookup1("10926") ->	{"0","",""};
lookup1("10927") ->	{"0","",""};
lookup1("10928") ->	{"0","",""};
lookup1("10929") ->	{"0","",""};
lookup1("1092A") ->	{"0","",""};
lookup1("1092B") ->	{"0","",""};
lookup1("1092C") ->	{"0","",""};
lookup1("1092D") ->	{"0","",""};
lookup1("1092E") ->	{"0","",""};
lookup1("1092F") ->	{"0","",""};
lookup1("1093") ->	{"0","",""};
lookup1("10930") ->	{"0","",""};
lookup1("10931") ->	{"0","",""};
lookup1("10932") ->	{"0","",""};
lookup1("10933") ->	{"0","",""};
lookup1("10934") ->	{"0","",""};
lookup1("10935") ->	{"0","",""};
lookup1("10936") ->	{"0","",""};
lookup1("10937") ->	{"0","",""};
lookup1("10938") ->	{"0","",""};
lookup1("10939") ->	{"0","",""};
lookup1("1093F") ->	{"0","",""};
lookup1("1094") ->	{"0","",""};
lookup1("1095") ->	{"0","",""};
lookup1("1096") ->	{"0","",""};
lookup1("1097") ->	{"0","",""};
lookup1("1098") ->	{"0","",""};
lookup1("10980") ->	{"0","",""};
lookup1("10981") ->	{"0","",""};
lookup1("10982") ->	{"0","",""};
lookup1("10983") ->	{"0","",""};
lookup1("10984") ->	{"0","",""};
lookup1("10985") ->	{"0","",""};
lookup1("10986") ->	{"0","",""};
lookup1("10987") ->	{"0","",""};
lookup1("10988") ->	{"0","",""};
lookup1("10989") ->	{"0","",""};
lookup1("1098A") ->	{"0","",""};
lookup1("1098B") ->	{"0","",""};
lookup1("1098C") ->	{"0","",""};
lookup1("1098D") ->	{"0","",""};
lookup1("1098E") ->	{"0","",""};
lookup1("1098F") ->	{"0","",""};
lookup1("1099") ->	{"0","",""};
lookup1("10990") ->	{"0","",""};
lookup1("10991") ->	{"0","",""};
lookup1("10992") ->	{"0","",""};
lookup1("10993") ->	{"0","",""};
lookup1("10994") ->	{"0","",""};
lookup1("10995") ->	{"0","",""};
lookup1("10996") ->	{"0","",""};
lookup1("10997") ->	{"0","",""};
lookup1("10998") ->	{"0","",""};
lookup1("10999") ->	{"0","",""};
lookup1("1099A") ->	{"0","",""};
lookup1("1099B") ->	{"0","",""};
lookup1("1099C") ->	{"0","",""};
lookup1("1099D") ->	{"0","",""};
lookup1("1099E") ->	{"0","",""};
lookup1("1099F") ->	{"0","",""};
lookup1("109A") ->	{"0","",""};
lookup1("109A0") ->	{"0","",""};
lookup1("109A1") ->	{"0","",""};
lookup1("109A2") ->	{"0","",""};
lookup1("109A3") ->	{"0","",""};
lookup1("109A4") ->	{"0","",""};
lookup1("109A5") ->	{"0","",""};
lookup1("109A6") ->	{"0","",""};
lookup1("109A7") ->	{"0","",""};
lookup1("109A8") ->	{"0","",""};
lookup1("109A9") ->	{"0","",""};
lookup1("109AA") ->	{"0","",""};
lookup1("109AB") ->	{"0","",""};
lookup1("109AC") ->	{"0","",""};
lookup1("109AD") ->	{"0","",""};
lookup1("109AE") ->	{"0","",""};
lookup1("109AF") ->	{"0","",""};
lookup1("109B") ->	{"0","",""};
lookup1("109B0") ->	{"0","",""};
lookup1("109B1") ->	{"0","",""};
lookup1("109B2") ->	{"0","",""};
lookup1("109B3") ->	{"0","",""};
lookup1("109B4") ->	{"0","",""};
lookup1("109B5") ->	{"0","",""};
lookup1("109B6") ->	{"0","",""};
lookup1("109B7") ->	{"0","",""};
lookup1("109BE") ->	{"0","",""};
lookup1("109BF") ->	{"0","",""};
lookup1("109C") ->	{"0","",""};
lookup1("109D") ->	{"0","",""};
lookup1("109E") ->	{"0","",""};
lookup1("109F") ->	{"0","",""};
lookup1("10A0") ->	{"0","","2D00"};
lookup1("10A00") ->	{"0","",""};
lookup1("10A01") ->	{"0","",""};
lookup1("10A02") ->	{"0","",""};
lookup1("10A03") ->	{"0","",""};
lookup1("10A05") ->	{"0","",""};
lookup1("10A06") ->	{"0","",""};
lookup1("10A0C") ->	{"0","",""};
lookup1("10A0D") ->	{"220","",""};
lookup1("10A0E") ->	{"0","",""};
lookup1("10A0F") ->	{"230","",""};
lookup1("10A1") ->	{"0","","2D01"};
lookup1("10A10") ->	{"0","",""};
lookup1("10A11") ->	{"0","",""};
lookup1("10A12") ->	{"0","",""};
lookup1("10A13") ->	{"0","",""};
lookup1("10A15") ->	{"0","",""};
lookup1("10A16") ->	{"0","",""};
lookup1("10A17") ->	{"0","",""};
lookup1("10A19") ->	{"0","",""};
lookup1("10A1A") ->	{"0","",""};
lookup1("10A1B") ->	{"0","",""};
lookup1("10A1C") ->	{"0","",""};
lookup1("10A1D") ->	{"0","",""};
lookup1("10A1E") ->	{"0","",""};
lookup1("10A1F") ->	{"0","",""};
lookup1("10A2") ->	{"0","","2D02"};
lookup1("10A20") ->	{"0","",""};
lookup1("10A21") ->	{"0","",""};
lookup1("10A22") ->	{"0","",""};
lookup1("10A23") ->	{"0","",""};
lookup1("10A24") ->	{"0","",""};
lookup1("10A25") ->	{"0","",""};
lookup1("10A26") ->	{"0","",""};
lookup1("10A27") ->	{"0","",""};
lookup1("10A28") ->	{"0","",""};
lookup1("10A29") ->	{"0","",""};
lookup1("10A2A") ->	{"0","",""};
lookup1("10A2B") ->	{"0","",""};
lookup1("10A2C") ->	{"0","",""};
lookup1("10A2D") ->	{"0","",""};
lookup1("10A2E") ->	{"0","",""};
lookup1("10A2F") ->	{"0","",""};
lookup1("10A3") ->	{"0","","2D03"};
lookup1("10A30") ->	{"0","",""};
lookup1("10A31") ->	{"0","",""};
lookup1("10A32") ->	{"0","",""};
lookup1("10A33") ->	{"0","",""};
lookup1("10A38") ->	{"230","",""};
lookup1("10A39") ->	{"1","",""};
lookup1("10A3A") ->	{"220","",""};
lookup1("10A3F") ->	{"9","",""};
lookup1("10A4") ->	{"0","","2D04"};
lookup1("10A40") ->	{"0","",""};
lookup1("10A41") ->	{"0","",""};
lookup1("10A42") ->	{"0","",""};
lookup1("10A43") ->	{"0","",""};
lookup1("10A44") ->	{"0","",""};
lookup1("10A45") ->	{"0","",""};
lookup1("10A46") ->	{"0","",""};
lookup1("10A47") ->	{"0","",""};
lookup1("10A5") ->	{"0","","2D05"};
lookup1("10A50") ->	{"0","",""};
lookup1("10A51") ->	{"0","",""};
lookup1("10A52") ->	{"0","",""};
lookup1("10A53") ->	{"0","",""};
lookup1("10A54") ->	{"0","",""};
lookup1("10A55") ->	{"0","",""};
lookup1("10A56") ->	{"0","",""};
lookup1("10A57") ->	{"0","",""};
lookup1("10A58") ->	{"0","",""};
lookup1("10A6") ->	{"0","","2D06"};
lookup1("10A60") ->	{"0","",""};
lookup1("10A61") ->	{"0","",""};
lookup1("10A62") ->	{"0","",""};
lookup1("10A63") ->	{"0","",""};
lookup1("10A64") ->	{"0","",""};
lookup1("10A65") ->	{"0","",""};
lookup1("10A66") ->	{"0","",""};
lookup1("10A67") ->	{"0","",""};
lookup1("10A68") ->	{"0","",""};
lookup1("10A69") ->	{"0","",""};
lookup1("10A6A") ->	{"0","",""};
lookup1("10A6B") ->	{"0","",""};
lookup1("10A6C") ->	{"0","",""};
lookup1("10A6D") ->	{"0","",""};
lookup1("10A6E") ->	{"0","",""};
lookup1("10A6F") ->	{"0","",""};
lookup1("10A7") ->	{"0","","2D07"};
lookup1("10A70") ->	{"0","",""};
lookup1("10A71") ->	{"0","",""};
lookup1("10A72") ->	{"0","",""};
lookup1("10A73") ->	{"0","",""};
lookup1("10A74") ->	{"0","",""};
lookup1("10A75") ->	{"0","",""};
lookup1("10A76") ->	{"0","",""};
lookup1("10A77") ->	{"0","",""};
lookup1("10A78") ->	{"0","",""};
lookup1("10A79") ->	{"0","",""};
lookup1("10A7A") ->	{"0","",""};
lookup1("10A7B") ->	{"0","",""};
lookup1("10A7C") ->	{"0","",""};
lookup1("10A7D") ->	{"0","",""};
lookup1("10A7E") ->	{"0","",""};
lookup1("10A7F") ->	{"0","",""};
lookup1("10A8") ->	{"0","","2D08"};
lookup1("10A80") ->	{"0","",""};
lookup1("10A81") ->	{"0","",""};
lookup1("10A82") ->	{"0","",""};
lookup1("10A83") ->	{"0","",""};
lookup1("10A84") ->	{"0","",""};
lookup1("10A85") ->	{"0","",""};
lookup1("10A86") ->	{"0","",""};
lookup1("10A87") ->	{"0","",""};
lookup1("10A88") ->	{"0","",""};
lookup1("10A89") ->	{"0","",""};
lookup1("10A8A") ->	{"0","",""};
lookup1("10A8B") ->	{"0","",""};
lookup1("10A8C") ->	{"0","",""};
lookup1("10A8D") ->	{"0","",""};
lookup1("10A8E") ->	{"0","",""};
lookup1("10A8F") ->	{"0","",""};
lookup1("10A9") ->	{"0","","2D09"};
lookup1("10A90") ->	{"0","",""};
lookup1("10A91") ->	{"0","",""};
lookup1("10A92") ->	{"0","",""};
lookup1("10A93") ->	{"0","",""};
lookup1("10A94") ->	{"0","",""};
lookup1("10A95") ->	{"0","",""};
lookup1("10A96") ->	{"0","",""};
lookup1("10A97") ->	{"0","",""};
lookup1("10A98") ->	{"0","",""};
lookup1("10A99") ->	{"0","",""};
lookup1("10A9A") ->	{"0","",""};
lookup1("10A9B") ->	{"0","",""};
lookup1("10A9C") ->	{"0","",""};
lookup1("10A9D") ->	{"0","",""};
lookup1("10A9E") ->	{"0","",""};
lookup1("10A9F") ->	{"0","",""};
lookup1("10AA") ->	{"0","","2D0A"};
lookup1("10AB") ->	{"0","","2D0B"};
lookup1("10AC") ->	{"0","","2D0C"};
lookup1("10AC0") ->	{"0","",""};
lookup1("10AC1") ->	{"0","",""};
lookup1("10AC2") ->	{"0","",""};
lookup1("10AC3") ->	{"0","",""};
lookup1("10AC4") ->	{"0","",""};
lookup1("10AC5") ->	{"0","",""};
lookup1("10AC6") ->	{"0","",""};
lookup1("10AC7") ->	{"0","",""};
lookup1("10AC8") ->	{"0","",""};
lookup1("10AC9") ->	{"0","",""};
lookup1("10ACA") ->	{"0","",""};
lookup1("10ACB") ->	{"0","",""};
lookup1("10ACC") ->	{"0","",""};
lookup1("10ACD") ->	{"0","",""};
lookup1("10ACE") ->	{"0","",""};
lookup1("10ACF") ->	{"0","",""};
lookup1("10AD") ->	{"0","","2D0D"};
lookup1("10AD0") ->	{"0","",""};
lookup1("10AD1") ->	{"0","",""};
lookup1("10AD2") ->	{"0","",""};
lookup1("10AD3") ->	{"0","",""};
lookup1("10AD4") ->	{"0","",""};
lookup1("10AD5") ->	{"0","",""};
lookup1("10AD6") ->	{"0","",""};
lookup1("10AD7") ->	{"0","",""};
lookup1("10AD8") ->	{"0","",""};
lookup1("10AD9") ->	{"0","",""};
lookup1("10ADA") ->	{"0","",""};
lookup1("10ADB") ->	{"0","",""};
lookup1("10ADC") ->	{"0","",""};
lookup1("10ADD") ->	{"0","",""};
lookup1("10ADE") ->	{"0","",""};
lookup1("10ADF") ->	{"0","",""};
lookup1("10AE") ->	{"0","","2D0E"};
lookup1("10AE0") ->	{"0","",""};
lookup1("10AE1") ->	{"0","",""};
lookup1("10AE2") ->	{"0","",""};
lookup1("10AE3") ->	{"0","",""};
lookup1("10AE4") ->	{"0","",""};
lookup1("10AE5") ->	{"230","",""};
lookup1("10AE6") ->	{"220","",""};
lookup1("10AEB") ->	{"0","",""};
lookup1("10AEC") ->	{"0","",""};
lookup1("10AED") ->	{"0","",""};
lookup1("10AEE") ->	{"0","",""};
lookup1("10AEF") ->	{"0","",""};
lookup1("10AF") ->	{"0","","2D0F"};
lookup1("10AF0") ->	{"0","",""};
lookup1("10AF1") ->	{"0","",""};
lookup1("10AF2") ->	{"0","",""};
lookup1("10AF3") ->	{"0","",""};
lookup1("10AF4") ->	{"0","",""};
lookup1("10AF5") ->	{"0","",""};
lookup1("10AF6") ->	{"0","",""};
lookup1("10B0") ->	{"0","","2D10"};
lookup1("10B00") ->	{"0","",""};
lookup1("10B01") ->	{"0","",""};
lookup1("10B02") ->	{"0","",""};
lookup1("10B03") ->	{"0","",""};
lookup1("10B04") ->	{"0","",""};
lookup1("10B05") ->	{"0","",""};
lookup1("10B06") ->	{"0","",""};
lookup1("10B07") ->	{"0","",""};
lookup1("10B08") ->	{"0","",""};
lookup1("10B09") ->	{"0","",""};
lookup1("10B0A") ->	{"0","",""};
lookup1("10B0B") ->	{"0","",""};
lookup1("10B0C") ->	{"0","",""};
lookup1("10B0D") ->	{"0","",""};
lookup1("10B0E") ->	{"0","",""};
lookup1("10B0F") ->	{"0","",""};
lookup1("10B1") ->	{"0","","2D11"};
lookup1("10B10") ->	{"0","",""};
lookup1("10B11") ->	{"0","",""};
lookup1("10B12") ->	{"0","",""};
lookup1("10B13") ->	{"0","",""};
lookup1("10B14") ->	{"0","",""};
lookup1("10B15") ->	{"0","",""};
lookup1("10B16") ->	{"0","",""};
lookup1("10B17") ->	{"0","",""};
lookup1("10B18") ->	{"0","",""};
lookup1("10B19") ->	{"0","",""};
lookup1("10B1A") ->	{"0","",""};
lookup1("10B1B") ->	{"0","",""};
lookup1("10B1C") ->	{"0","",""};
lookup1("10B1D") ->	{"0","",""};
lookup1("10B1E") ->	{"0","",""};
lookup1("10B1F") ->	{"0","",""};
lookup1("10B2") ->	{"0","","2D12"};
lookup1("10B20") ->	{"0","",""};
lookup1("10B21") ->	{"0","",""};
lookup1("10B22") ->	{"0","",""};
lookup1("10B23") ->	{"0","",""};
lookup1("10B24") ->	{"0","",""};
lookup1("10B25") ->	{"0","",""};
lookup1("10B26") ->	{"0","",""};
lookup1("10B27") ->	{"0","",""};
lookup1("10B28") ->	{"0","",""};
lookup1("10B29") ->	{"0","",""};
lookup1("10B2A") ->	{"0","",""};
lookup1("10B2B") ->	{"0","",""};
lookup1("10B2C") ->	{"0","",""};
lookup1("10B2D") ->	{"0","",""};
lookup1("10B2E") ->	{"0","",""};
lookup1("10B2F") ->	{"0","",""};
lookup1("10B3") ->	{"0","","2D13"};
lookup1("10B30") ->	{"0","",""};
lookup1("10B31") ->	{"0","",""};
lookup1("10B32") ->	{"0","",""};
lookup1("10B33") ->	{"0","",""};
lookup1("10B34") ->	{"0","",""};
lookup1("10B35") ->	{"0","",""};
lookup1("10B39") ->	{"0","",""};
lookup1("10B3A") ->	{"0","",""};
lookup1("10B3B") ->	{"0","",""};
lookup1("10B3C") ->	{"0","",""};
lookup1("10B3D") ->	{"0","",""};
lookup1("10B3E") ->	{"0","",""};
lookup1("10B3F") ->	{"0","",""};
lookup1("10B4") ->	{"0","","2D14"};
lookup1("10B40") ->	{"0","",""};
lookup1("10B41") ->	{"0","",""};
lookup1("10B42") ->	{"0","",""};
lookup1("10B43") ->	{"0","",""};
lookup1("10B44") ->	{"0","",""};
lookup1("10B45") ->	{"0","",""};
lookup1("10B46") ->	{"0","",""};
lookup1("10B47") ->	{"0","",""};
lookup1("10B48") ->	{"0","",""};
lookup1("10B49") ->	{"0","",""};
lookup1("10B4A") ->	{"0","",""};
lookup1("10B4B") ->	{"0","",""};
lookup1("10B4C") ->	{"0","",""};
lookup1("10B4D") ->	{"0","",""};
lookup1("10B4E") ->	{"0","",""};
lookup1("10B4F") ->	{"0","",""};
lookup1("10B5") ->	{"0","","2D15"};
lookup1("10B50") ->	{"0","",""};
lookup1("10B51") ->	{"0","",""};
lookup1("10B52") ->	{"0","",""};
lookup1("10B53") ->	{"0","",""};
lookup1("10B54") ->	{"0","",""};
lookup1("10B55") ->	{"0","",""};
lookup1("10B58") ->	{"0","",""};
lookup1("10B59") ->	{"0","",""};
lookup1("10B5A") ->	{"0","",""};
lookup1("10B5B") ->	{"0","",""};
lookup1("10B5C") ->	{"0","",""};
lookup1("10B5D") ->	{"0","",""};
lookup1("10B5E") ->	{"0","",""};
lookup1("10B5F") ->	{"0","",""};
lookup1("10B6") ->	{"0","","2D16"};
lookup1("10B60") ->	{"0","",""};
lookup1("10B61") ->	{"0","",""};
lookup1("10B62") ->	{"0","",""};
lookup1("10B63") ->	{"0","",""};
lookup1("10B64") ->	{"0","",""};
lookup1("10B65") ->	{"0","",""};
lookup1("10B66") ->	{"0","",""};
lookup1("10B67") ->	{"0","",""};
lookup1("10B68") ->	{"0","",""};
lookup1("10B69") ->	{"0","",""};
lookup1("10B6A") ->	{"0","",""};
lookup1("10B6B") ->	{"0","",""};
lookup1("10B6C") ->	{"0","",""};
lookup1("10B6D") ->	{"0","",""};
lookup1("10B6E") ->	{"0","",""};
lookup1("10B6F") ->	{"0","",""};
lookup1("10B7") ->	{"0","","2D17"};
lookup1("10B70") ->	{"0","",""};
lookup1("10B71") ->	{"0","",""};
lookup1("10B72") ->	{"0","",""};
lookup1("10B78") ->	{"0","",""};
lookup1("10B79") ->	{"0","",""};
lookup1("10B7A") ->	{"0","",""};
lookup1("10B7B") ->	{"0","",""};
lookup1("10B7C") ->	{"0","",""};
lookup1("10B7D") ->	{"0","",""};
lookup1("10B7E") ->	{"0","",""};
lookup1("10B7F") ->	{"0","",""};
lookup1("10B8") ->	{"0","","2D18"};
lookup1("10B80") ->	{"0","",""};
lookup1("10B81") ->	{"0","",""};
lookup1("10B82") ->	{"0","",""};
lookup1("10B83") ->	{"0","",""};
lookup1("10B84") ->	{"0","",""};
lookup1("10B85") ->	{"0","",""};
lookup1("10B86") ->	{"0","",""};
lookup1("10B87") ->	{"0","",""};
lookup1("10B88") ->	{"0","",""};
lookup1("10B89") ->	{"0","",""};
lookup1("10B8A") ->	{"0","",""};
lookup1("10B8B") ->	{"0","",""};
lookup1("10B8C") ->	{"0","",""};
lookup1("10B8D") ->	{"0","",""};
lookup1("10B8E") ->	{"0","",""};
lookup1("10B8F") ->	{"0","",""};
lookup1("10B9") ->	{"0","","2D19"};
lookup1("10B90") ->	{"0","",""};
lookup1("10B91") ->	{"0","",""};
lookup1("10B99") ->	{"0","",""};
lookup1("10B9A") ->	{"0","",""};
lookup1("10B9B") ->	{"0","",""};
lookup1("10B9C") ->	{"0","",""};
lookup1("10BA") ->	{"0","","2D1A"};
lookup1("10BA9") ->	{"0","",""};
lookup1("10BAA") ->	{"0","",""};
lookup1("10BAB") ->	{"0","",""};
lookup1("10BAC") ->	{"0","",""};
lookup1("10BAD") ->	{"0","",""};
lookup1("10BAE") ->	{"0","",""};
lookup1("10BAF") ->	{"0","",""};
lookup1("10BB") ->	{"0","","2D1B"};
lookup1("10BC") ->	{"0","","2D1C"};
lookup1("10BD") ->	{"0","","2D1D"};
lookup1("10BE") ->	{"0","","2D1E"};
lookup1("10BF") ->	{"0","","2D1F"};
lookup1("10C0") ->	{"0","","2D20"};
lookup1("10C00") ->	{"0","",""};
lookup1("10C01") ->	{"0","",""};
lookup1("10C02") ->	{"0","",""};
lookup1("10C03") ->	{"0","",""};
lookup1("10C04") ->	{"0","",""};
lookup1("10C05") ->	{"0","",""};
lookup1("10C06") ->	{"0","",""};
lookup1("10C07") ->	{"0","",""};
lookup1("10C08") ->	{"0","",""};
lookup1("10C09") ->	{"0","",""};
lookup1("10C0A") ->	{"0","",""};
lookup1("10C0B") ->	{"0","",""};
lookup1("10C0C") ->	{"0","",""};
lookup1("10C0D") ->	{"0","",""};
lookup1("10C0E") ->	{"0","",""};
lookup1("10C0F") ->	{"0","",""};
lookup1("10C1") ->	{"0","","2D21"};
lookup1("10C10") ->	{"0","",""};
lookup1("10C11") ->	{"0","",""};
lookup1("10C12") ->	{"0","",""};
lookup1("10C13") ->	{"0","",""};
lookup1("10C14") ->	{"0","",""};
lookup1("10C15") ->	{"0","",""};
lookup1("10C16") ->	{"0","",""};
lookup1("10C17") ->	{"0","",""};
lookup1("10C18") ->	{"0","",""};
lookup1("10C19") ->	{"0","",""};
lookup1("10C1A") ->	{"0","",""};
lookup1("10C1B") ->	{"0","",""};
lookup1("10C1C") ->	{"0","",""};
lookup1("10C1D") ->	{"0","",""};
lookup1("10C1E") ->	{"0","",""};
lookup1("10C1F") ->	{"0","",""};
lookup1("10C2") ->	{"0","","2D22"};
lookup1("10C20") ->	{"0","",""};
lookup1("10C21") ->	{"0","",""};
lookup1("10C22") ->	{"0","",""};
lookup1("10C23") ->	{"0","",""};
lookup1("10C24") ->	{"0","",""};
lookup1("10C25") ->	{"0","",""};
lookup1("10C26") ->	{"0","",""};
lookup1("10C27") ->	{"0","",""};
lookup1("10C28") ->	{"0","",""};
lookup1("10C29") ->	{"0","",""};
lookup1("10C2A") ->	{"0","",""};
lookup1("10C2B") ->	{"0","",""};
lookup1("10C2C") ->	{"0","",""};
lookup1("10C2D") ->	{"0","",""};
lookup1("10C2E") ->	{"0","",""};
lookup1("10C2F") ->	{"0","",""};
lookup1("10C3") ->	{"0","","2D23"};
lookup1("10C30") ->	{"0","",""};
lookup1("10C31") ->	{"0","",""};
lookup1("10C32") ->	{"0","",""};
lookup1("10C33") ->	{"0","",""};
lookup1("10C34") ->	{"0","",""};
lookup1("10C35") ->	{"0","",""};
lookup1("10C36") ->	{"0","",""};
lookup1("10C37") ->	{"0","",""};
lookup1("10C38") ->	{"0","",""};
lookup1("10C39") ->	{"0","",""};
lookup1("10C3A") ->	{"0","",""};
lookup1("10C3B") ->	{"0","",""};
lookup1("10C3C") ->	{"0","",""};
lookup1("10C3D") ->	{"0","",""};
lookup1("10C3E") ->	{"0","",""};
lookup1("10C3F") ->	{"0","",""};
lookup1("10C4") ->	{"0","","2D24"};
lookup1("10C40") ->	{"0","",""};
lookup1("10C41") ->	{"0","",""};
lookup1("10C42") ->	{"0","",""};
lookup1("10C43") ->	{"0","",""};
lookup1("10C44") ->	{"0","",""};
lookup1("10C45") ->	{"0","",""};
lookup1("10C46") ->	{"0","",""};
lookup1("10C47") ->	{"0","",""};
lookup1("10C48") ->	{"0","",""};
lookup1("10C5") ->	{"0","","2D25"};
lookup1("10C7") ->	{"0","","2D27"};
lookup1("10CD") ->	{"0","","2D2D"};
lookup1("10D0") ->	{"0","",""};
lookup1("10D1") ->	{"0","",""};
lookup1("10D2") ->	{"0","",""};
lookup1("10D3") ->	{"0","",""};
lookup1("10D4") ->	{"0","",""};
lookup1("10D5") ->	{"0","",""};
lookup1("10D6") ->	{"0","",""};
lookup1("10D7") ->	{"0","",""};
lookup1("10D8") ->	{"0","",""};
lookup1("10D9") ->	{"0","",""};
lookup1("10DA") ->	{"0","",""};
lookup1("10DB") ->	{"0","",""};
lookup1("10DC") ->	{"0","",""};
lookup1("10DD") ->	{"0","",""};
lookup1("10DE") ->	{"0","",""};
lookup1("10DF") ->	{"0","",""};
lookup1("10E0") ->	{"0","",""};
lookup1("10E1") ->	{"0","",""};
lookup1("10E2") ->	{"0","",""};
lookup1("10E3") ->	{"0","",""};
lookup1("10E4") ->	{"0","",""};
lookup1("10E5") ->	{"0","",""};
lookup1("10E6") ->	{"0","",""};
lookup1("10E60") ->	{"0","",""};
lookup1("10E61") ->	{"0","",""};
lookup1("10E62") ->	{"0","",""};
lookup1("10E63") ->	{"0","",""};
lookup1("10E64") ->	{"0","",""};
lookup1("10E65") ->	{"0","",""};
lookup1("10E66") ->	{"0","",""};
lookup1("10E67") ->	{"0","",""};
lookup1("10E68") ->	{"0","",""};
lookup1("10E69") ->	{"0","",""};
lookup1("10E6A") ->	{"0","",""};
lookup1("10E6B") ->	{"0","",""};
lookup1("10E6C") ->	{"0","",""};
lookup1("10E6D") ->	{"0","",""};
lookup1("10E6E") ->	{"0","",""};
lookup1("10E6F") ->	{"0","",""};
lookup1("10E7") ->	{"0","",""};
lookup1("10E70") ->	{"0","",""};
lookup1("10E71") ->	{"0","",""};
lookup1("10E72") ->	{"0","",""};
lookup1("10E73") ->	{"0","",""};
lookup1("10E74") ->	{"0","",""};
lookup1("10E75") ->	{"0","",""};
lookup1("10E76") ->	{"0","",""};
lookup1("10E77") ->	{"0","",""};
lookup1("10E78") ->	{"0","",""};
lookup1("10E79") ->	{"0","",""};
lookup1("10E7A") ->	{"0","",""};
lookup1("10E7B") ->	{"0","",""};
lookup1("10E7C") ->	{"0","",""};
lookup1("10E7D") ->	{"0","",""};
lookup1("10E7E") ->	{"0","",""};
lookup1("10E8") ->	{"0","",""};
lookup1("10E9") ->	{"0","",""};
lookup1("10EA") ->	{"0","",""};
lookup1("10EB") ->	{"0","",""};
lookup1("10EC") ->	{"0","",""};
lookup1("10ED") ->	{"0","",""};
lookup1("10EE") ->	{"0","",""};
lookup1("10EF") ->	{"0","",""};
lookup1("10F0") ->	{"0","",""};
lookup1("10F1") ->	{"0","",""};
lookup1("10F2") ->	{"0","",""};
lookup1("10F3") ->	{"0","",""};
lookup1("10F4") ->	{"0","",""};
lookup1("10F5") ->	{"0","",""};
lookup1("10F6") ->	{"0","",""};
lookup1("10F7") ->	{"0","",""};
lookup1("10F8") ->	{"0","",""};
lookup1("10F9") ->	{"0","",""};
lookup1("10FA") ->	{"0","",""};
lookup1("10FB") ->	{"0","",""};
lookup1("10FC") ->	{"0","<super> 10DC",""};
lookup1("10FD") ->	{"0","",""};
lookup1("10FE") ->	{"0","",""};
lookup1("10FF") ->	{"0","",""};
lookup1("10FFFD") ->	{"0","",""};
lookup1("1100") ->	{"0","",""};
lookup1("11000") ->	{"0","",""};
lookup1("11001") ->	{"0","",""};
lookup1("11002") ->	{"0","",""};
lookup1("11003") ->	{"0","",""};
lookup1("11004") ->	{"0","",""};
lookup1("11005") ->	{"0","",""};
lookup1("11006") ->	{"0","",""};
lookup1("11007") ->	{"0","",""};
lookup1("11008") ->	{"0","",""};
lookup1("11009") ->	{"0","",""};
lookup1("1100A") ->	{"0","",""};
lookup1("1100B") ->	{"0","",""};
lookup1("1100C") ->	{"0","",""};
lookup1("1100D") ->	{"0","",""};
lookup1("1100E") ->	{"0","",""};
lookup1("1100F") ->	{"0","",""};
lookup1("1101") ->	{"0","",""};
lookup1("11010") ->	{"0","",""};
lookup1("11011") ->	{"0","",""};
lookup1("11012") ->	{"0","",""};
lookup1("11013") ->	{"0","",""};
lookup1("11014") ->	{"0","",""};
lookup1("11015") ->	{"0","",""};
lookup1("11016") ->	{"0","",""};
lookup1("11017") ->	{"0","",""};
lookup1("11018") ->	{"0","",""};
lookup1("11019") ->	{"0","",""};
lookup1("1101A") ->	{"0","",""};
lookup1("1101B") ->	{"0","",""};
lookup1("1101C") ->	{"0","",""};
lookup1("1101D") ->	{"0","",""};
lookup1("1101E") ->	{"0","",""};
lookup1("1101F") ->	{"0","",""};
lookup1("1102") ->	{"0","",""};
lookup1("11020") ->	{"0","",""};
lookup1("11021") ->	{"0","",""};
lookup1("11022") ->	{"0","",""};
lookup1("11023") ->	{"0","",""};
lookup1("11024") ->	{"0","",""};
lookup1("11025") ->	{"0","",""};
lookup1("11026") ->	{"0","",""};
lookup1("11027") ->	{"0","",""};
lookup1("11028") ->	{"0","",""};
lookup1("11029") ->	{"0","",""};
lookup1("1102A") ->	{"0","",""};
lookup1("1102B") ->	{"0","",""};
lookup1("1102C") ->	{"0","",""};
lookup1("1102D") ->	{"0","",""};
lookup1("1102E") ->	{"0","",""};
lookup1("1102F") ->	{"0","",""};
lookup1("1103") ->	{"0","",""};
lookup1("11030") ->	{"0","",""};
lookup1("11031") ->	{"0","",""};
lookup1("11032") ->	{"0","",""};
lookup1("11033") ->	{"0","",""};
lookup1("11034") ->	{"0","",""};
lookup1("11035") ->	{"0","",""};
lookup1("11036") ->	{"0","",""};
lookup1("11037") ->	{"0","",""};
lookup1("11038") ->	{"0","",""};
lookup1("11039") ->	{"0","",""};
lookup1("1103A") ->	{"0","",""};
lookup1("1103B") ->	{"0","",""};
lookup1("1103C") ->	{"0","",""};
lookup1("1103D") ->	{"0","",""};
lookup1("1103E") ->	{"0","",""};
lookup1("1103F") ->	{"0","",""};
lookup1("1104") ->	{"0","",""};
lookup1("11040") ->	{"0","",""};
lookup1("11041") ->	{"0","",""};
lookup1("11042") ->	{"0","",""};
lookup1("11043") ->	{"0","",""};
lookup1("11044") ->	{"0","",""};
lookup1("11045") ->	{"0","",""};
lookup1("11046") ->	{"9","",""};
lookup1("11047") ->	{"0","",""};
lookup1("11048") ->	{"0","",""};
lookup1("11049") ->	{"0","",""};
lookup1("1104A") ->	{"0","",""};
lookup1("1104B") ->	{"0","",""};
lookup1("1104C") ->	{"0","",""};
lookup1("1104D") ->	{"0","",""};
lookup1("1105") ->	{"0","",""};
lookup1("11052") ->	{"0","",""};
lookup1("11053") ->	{"0","",""};
lookup1("11054") ->	{"0","",""};
lookup1("11055") ->	{"0","",""};
lookup1("11056") ->	{"0","",""};
lookup1("11057") ->	{"0","",""};
lookup1("11058") ->	{"0","",""};
lookup1("11059") ->	{"0","",""};
lookup1("1105A") ->	{"0","",""};
lookup1("1105B") ->	{"0","",""};
lookup1("1105C") ->	{"0","",""};
lookup1("1105D") ->	{"0","",""};
lookup1("1105E") ->	{"0","",""};
lookup1("1105F") ->	{"0","",""};
lookup1("1106") ->	{"0","",""};
lookup1("11060") ->	{"0","",""};
lookup1("11061") ->	{"0","",""};
lookup1("11062") ->	{"0","",""};
lookup1("11063") ->	{"0","",""};
lookup1("11064") ->	{"0","",""};
lookup1("11065") ->	{"0","",""};
lookup1("11066") ->	{"0","",""};
lookup1("11067") ->	{"0","",""};
lookup1("11068") ->	{"0","",""};
lookup1("11069") ->	{"0","",""};
lookup1("1106A") ->	{"0","",""};
lookup1("1106B") ->	{"0","",""};
lookup1("1106C") ->	{"0","",""};
lookup1("1106D") ->	{"0","",""};
lookup1("1106E") ->	{"0","",""};
lookup1("1106F") ->	{"0","",""};
lookup1("1107") ->	{"0","",""};
lookup1("1107F") ->	{"9","",""};
lookup1("1108") ->	{"0","",""};
lookup1("11080") ->	{"0","",""};
lookup1("11081") ->	{"0","",""};
lookup1("11082") ->	{"0","",""};
lookup1("11083") ->	{"0","",""};
lookup1("11084") ->	{"0","",""};
lookup1("11085") ->	{"0","",""};
lookup1("11086") ->	{"0","",""};
lookup1("11087") ->	{"0","",""};
lookup1("11088") ->	{"0","",""};
lookup1("11089") ->	{"0","",""};
lookup1("1108A") ->	{"0","",""};
lookup1("1108B") ->	{"0","",""};
lookup1("1108C") ->	{"0","",""};
lookup1("1108D") ->	{"0","",""};
lookup1("1108E") ->	{"0","",""};
lookup1("1108F") ->	{"0","",""};
lookup1("1109") ->	{"0","",""};
lookup1("11090") ->	{"0","",""};
lookup1("11091") ->	{"0","",""};
lookup1("11092") ->	{"0","",""};
lookup1("11093") ->	{"0","",""};
lookup1("11094") ->	{"0","",""};
lookup1("11095") ->	{"0","",""};
lookup1("11096") ->	{"0","",""};
lookup1("11097") ->	{"0","",""};
lookup1("11098") ->	{"0","",""};
lookup1("11099") ->	{"0","",""};
lookup1("1109A") ->	{"0","11099 110BA",""};
lookup1("1109B") ->	{"0","",""};
lookup1("1109C") ->	{"0","1109B 110BA",""};
lookup1("1109D") ->	{"0","",""};
lookup1("1109E") ->	{"0","",""};
lookup1("1109F") ->	{"0","",""};
lookup1("110A") ->	{"0","",""};
lookup1("110A0") ->	{"0","",""};
lookup1("110A1") ->	{"0","",""};
lookup1("110A2") ->	{"0","",""};
lookup1("110A3") ->	{"0","",""};
lookup1("110A4") ->	{"0","",""};
lookup1("110A5") ->	{"0","",""};
lookup1("110A6") ->	{"0","",""};
lookup1("110A7") ->	{"0","",""};
lookup1("110A8") ->	{"0","",""};
lookup1("110A9") ->	{"0","",""};
lookup1("110AA") ->	{"0","",""};
lookup1("110AB") ->	{"0","110A5 110BA",""};
lookup1("110AC") ->	{"0","",""};
lookup1("110AD") ->	{"0","",""};
lookup1("110AE") ->	{"0","",""};
lookup1("110AF") ->	{"0","",""};
lookup1("110B") ->	{"0","",""};
lookup1("110B0") ->	{"0","",""};
lookup1("110B1") ->	{"0","",""};
lookup1("110B2") ->	{"0","",""};
lookup1("110B3") ->	{"0","",""};
lookup1("110B4") ->	{"0","",""};
lookup1("110B5") ->	{"0","",""};
lookup1("110B6") ->	{"0","",""};
lookup1("110B7") ->	{"0","",""};
lookup1("110B8") ->	{"0","",""};
lookup1("110B9") ->	{"9","",""};
lookup1("110BA") ->	{"7","",""};
lookup1("110BB") ->	{"0","",""};
lookup1("110BC") ->	{"0","",""};
lookup1("110BD") ->	{"0","",""};
lookup1("110BE") ->	{"0","",""};
lookup1("110BF") ->	{"0","",""};
lookup1("110C") ->	{"0","",""};
lookup1("110C0") ->	{"0","",""};
lookup1("110C1") ->	{"0","",""};
lookup1("110D") ->	{"0","",""};
lookup1("110D0") ->	{"0","",""};
lookup1("110D1") ->	{"0","",""};
lookup1("110D2") ->	{"0","",""};
lookup1("110D3") ->	{"0","",""};
lookup1("110D4") ->	{"0","",""};
lookup1("110D5") ->	{"0","",""};
lookup1("110D6") ->	{"0","",""};
lookup1("110D7") ->	{"0","",""};
lookup1("110D8") ->	{"0","",""};
lookup1("110D9") ->	{"0","",""};
lookup1("110DA") ->	{"0","",""};
lookup1("110DB") ->	{"0","",""};
lookup1("110DC") ->	{"0","",""};
lookup1("110DD") ->	{"0","",""};
lookup1("110DE") ->	{"0","",""};
lookup1("110DF") ->	{"0","",""};
lookup1("110E") ->	{"0","",""};
lookup1("110E0") ->	{"0","",""};
lookup1("110E1") ->	{"0","",""};
lookup1("110E2") ->	{"0","",""};
lookup1("110E3") ->	{"0","",""};
lookup1("110E4") ->	{"0","",""};
lookup1("110E5") ->	{"0","",""};
lookup1("110E6") ->	{"0","",""};
lookup1("110E7") ->	{"0","",""};
lookup1("110E8") ->	{"0","",""};
lookup1("110F") ->	{"0","",""};
lookup1("110F0") ->	{"0","",""};
lookup1("110F1") ->	{"0","",""};
lookup1("110F2") ->	{"0","",""};
lookup1("110F3") ->	{"0","",""};
lookup1("110F4") ->	{"0","",""};
lookup1("110F5") ->	{"0","",""};
lookup1("110F6") ->	{"0","",""};
lookup1("110F7") ->	{"0","",""};
lookup1("110F8") ->	{"0","",""};
lookup1("110F9") ->	{"0","",""};
lookup1("1110") ->	{"0","",""};
lookup1("11100") ->	{"230","",""};
lookup1("11101") ->	{"230","",""};
lookup1("11102") ->	{"230","",""};
lookup1("11103") ->	{"0","",""};
lookup1("11104") ->	{"0","",""};
lookup1("11105") ->	{"0","",""};
lookup1("11106") ->	{"0","",""};
lookup1("11107") ->	{"0","",""};
lookup1("11108") ->	{"0","",""};
lookup1("11109") ->	{"0","",""};
lookup1("1110A") ->	{"0","",""};
lookup1("1110B") ->	{"0","",""};
lookup1("1110C") ->	{"0","",""};
lookup1("1110D") ->	{"0","",""};
lookup1("1110E") ->	{"0","",""};
lookup1("1110F") ->	{"0","",""};
lookup1("1111") ->	{"0","",""};
lookup1("11110") ->	{"0","",""};
lookup1("11111") ->	{"0","",""};
lookup1("11112") ->	{"0","",""};
lookup1("11113") ->	{"0","",""};
lookup1("11114") ->	{"0","",""};
lookup1("11115") ->	{"0","",""};
lookup1("11116") ->	{"0","",""};
lookup1("11117") ->	{"0","",""};
lookup1("11118") ->	{"0","",""};
lookup1("11119") ->	{"0","",""};
lookup1("1111A") ->	{"0","",""};
lookup1("1111B") ->	{"0","",""};
lookup1("1111C") ->	{"0","",""};
lookup1("1111D") ->	{"0","",""};
lookup1("1111E") ->	{"0","",""};
lookup1("1111F") ->	{"0","",""};
lookup1("1112") ->	{"0","",""};
lookup1("11120") ->	{"0","",""};
lookup1("11121") ->	{"0","",""};
lookup1("11122") ->	{"0","",""};
lookup1("11123") ->	{"0","",""};
lookup1("11124") ->	{"0","",""};
lookup1("11125") ->	{"0","",""};
lookup1("11126") ->	{"0","",""};
lookup1("11127") ->	{"0","",""};
lookup1("11128") ->	{"0","",""};
lookup1("11129") ->	{"0","",""};
lookup1("1112A") ->	{"0","",""};
lookup1("1112B") ->	{"0","",""};
lookup1("1112C") ->	{"0","",""};
lookup1("1112D") ->	{"0","",""};
lookup1("1112E") ->	{"0","11131 11127",""};
lookup1("1112F") ->	{"0","11132 11127",""};
lookup1("1113") ->	{"0","",""};
lookup1("11130") ->	{"0","",""};
lookup1("11131") ->	{"0","",""};
lookup1("11132") ->	{"0","",""};
lookup1("11133") ->	{"9","",""};
lookup1("11134") ->	{"9","",""};
lookup1("11136") ->	{"0","",""};
lookup1("11137") ->	{"0","",""};
lookup1("11138") ->	{"0","",""};
lookup1("11139") ->	{"0","",""};
lookup1("1113A") ->	{"0","",""};
lookup1("1113B") ->	{"0","",""};
lookup1("1113C") ->	{"0","",""};
lookup1("1113D") ->	{"0","",""};
lookup1("1113E") ->	{"0","",""};
lookup1("1113F") ->	{"0","",""};
lookup1("1114") ->	{"0","",""};
lookup1("11140") ->	{"0","",""};
lookup1("11141") ->	{"0","",""};
lookup1("11142") ->	{"0","",""};
lookup1("11143") ->	{"0","",""};
lookup1("1115") ->	{"0","",""};
lookup1("11150") ->	{"0","",""};
lookup1("11151") ->	{"0","",""};
lookup1("11152") ->	{"0","",""};
lookup1("11153") ->	{"0","",""};
lookup1("11154") ->	{"0","",""};
lookup1("11155") ->	{"0","",""};
lookup1("11156") ->	{"0","",""};
lookup1("11157") ->	{"0","",""};
lookup1("11158") ->	{"0","",""};
lookup1("11159") ->	{"0","",""};
lookup1("1115A") ->	{"0","",""};
lookup1("1115B") ->	{"0","",""};
lookup1("1115C") ->	{"0","",""};
lookup1("1115D") ->	{"0","",""};
lookup1("1115E") ->	{"0","",""};
lookup1("1115F") ->	{"0","",""};
lookup1("1116") ->	{"0","",""};
lookup1("11160") ->	{"0","",""};
lookup1("11161") ->	{"0","",""};
lookup1("11162") ->	{"0","",""};
lookup1("11163") ->	{"0","",""};
lookup1("11164") ->	{"0","",""};
lookup1("11165") ->	{"0","",""};
lookup1("11166") ->	{"0","",""};
lookup1("11167") ->	{"0","",""};
lookup1("11168") ->	{"0","",""};
lookup1("11169") ->	{"0","",""};
lookup1("1116A") ->	{"0","",""};
lookup1("1116B") ->	{"0","",""};
lookup1("1116C") ->	{"0","",""};
lookup1("1116D") ->	{"0","",""};
lookup1("1116E") ->	{"0","",""};
lookup1("1116F") ->	{"0","",""};
lookup1("1117") ->	{"0","",""};
lookup1("11170") ->	{"0","",""};
lookup1("11171") ->	{"0","",""};
lookup1("11172") ->	{"0","",""};
lookup1("11173") ->	{"7","",""};
lookup1("11174") ->	{"0","",""};
lookup1("11175") ->	{"0","",""};
lookup1("11176") ->	{"0","",""};
lookup1("1118") ->	{"0","",""};
lookup1("11180") ->	{"0","",""};
lookup1("11181") ->	{"0","",""};
lookup1("11182") ->	{"0","",""};
lookup1("11183") ->	{"0","",""};
lookup1("11184") ->	{"0","",""};
lookup1("11185") ->	{"0","",""};
lookup1("11186") ->	{"0","",""};
lookup1("11187") ->	{"0","",""};
lookup1("11188") ->	{"0","",""};
lookup1("11189") ->	{"0","",""};
lookup1("1118A") ->	{"0","",""};
lookup1("1118B") ->	{"0","",""};
lookup1("1118C") ->	{"0","",""};
lookup1("1118D") ->	{"0","",""};
lookup1("1118E") ->	{"0","",""};
lookup1("1118F") ->	{"0","",""};
lookup1("1119") ->	{"0","",""};
lookup1("11190") ->	{"0","",""};
lookup1("11191") ->	{"0","",""};
lookup1("11192") ->	{"0","",""};
lookup1("11193") ->	{"0","",""};
lookup1("11194") ->	{"0","",""};
lookup1("11195") ->	{"0","",""};
lookup1("11196") ->	{"0","",""};
lookup1("11197") ->	{"0","",""};
lookup1("11198") ->	{"0","",""};
lookup1("11199") ->	{"0","",""};
lookup1("1119A") ->	{"0","",""};
lookup1("1119B") ->	{"0","",""};
lookup1("1119C") ->	{"0","",""};
lookup1("1119D") ->	{"0","",""};
lookup1("1119E") ->	{"0","",""};
lookup1("1119F") ->	{"0","",""};
lookup1("111A") ->	{"0","",""};
lookup1("111A0") ->	{"0","",""};
lookup1("111A1") ->	{"0","",""};
lookup1("111A2") ->	{"0","",""};
lookup1("111A3") ->	{"0","",""};
lookup1("111A4") ->	{"0","",""};
lookup1("111A5") ->	{"0","",""};
lookup1("111A6") ->	{"0","",""};
lookup1("111A7") ->	{"0","",""};
lookup1("111A8") ->	{"0","",""};
lookup1("111A9") ->	{"0","",""};
lookup1("111AA") ->	{"0","",""};
lookup1("111AB") ->	{"0","",""};
lookup1("111AC") ->	{"0","",""};
lookup1("111AD") ->	{"0","",""};
lookup1("111AE") ->	{"0","",""};
lookup1("111AF") ->	{"0","",""};
lookup1("111B") ->	{"0","",""};
lookup1("111B0") ->	{"0","",""};
lookup1("111B1") ->	{"0","",""};
lookup1("111B2") ->	{"0","",""};
lookup1("111B3") ->	{"0","",""};
lookup1("111B4") ->	{"0","",""};
lookup1("111B5") ->	{"0","",""};
lookup1("111B6") ->	{"0","",""};
lookup1("111B7") ->	{"0","",""};
lookup1("111B8") ->	{"0","",""};
lookup1("111B9") ->	{"0","",""};
lookup1("111BA") ->	{"0","",""};
lookup1("111BB") ->	{"0","",""};
lookup1("111BC") ->	{"0","",""};
lookup1("111BD") ->	{"0","",""};
lookup1("111BE") ->	{"0","",""};
lookup1("111BF") ->	{"0","",""};
lookup1("111C") ->	{"0","",""};
lookup1("111C0") ->	{"9","",""};
lookup1("111C1") ->	{"0","",""};
lookup1("111C2") ->	{"0","",""};
lookup1("111C3") ->	{"0","",""};
lookup1("111C4") ->	{"0","",""};
lookup1("111C5") ->	{"0","",""};
lookup1("111C6") ->	{"0","",""};
lookup1("111C7") ->	{"0","",""};
lookup1("111C8") ->	{"0","",""};
lookup1("111CD") ->	{"0","",""};
lookup1("111D") ->	{"0","",""};
lookup1("111D0") ->	{"0","",""};
lookup1("111D1") ->	{"0","",""};
lookup1("111D2") ->	{"0","",""};
lookup1("111D3") ->	{"0","",""};
lookup1("111D4") ->	{"0","",""};
lookup1("111D5") ->	{"0","",""};
lookup1("111D6") ->	{"0","",""};
lookup1("111D7") ->	{"0","",""};
lookup1("111D8") ->	{"0","",""};
lookup1("111D9") ->	{"0","",""};
lookup1("111DA") ->	{"0","",""};
lookup1("111E") ->	{"0","",""};
lookup1("111E1") ->	{"0","",""};
lookup1("111E2") ->	{"0","",""};
lookup1("111E3") ->	{"0","",""};
lookup1("111E4") ->	{"0","",""};
lookup1("111E5") ->	{"0","",""};
lookup1("111E6") ->	{"0","",""};
lookup1("111E7") ->	{"0","",""};
lookup1("111E8") ->	{"0","",""};
lookup1("111E9") ->	{"0","",""};
lookup1("111EA") ->	{"0","",""};
lookup1("111EB") ->	{"0","",""};
lookup1("111EC") ->	{"0","",""};
lookup1("111ED") ->	{"0","",""};
lookup1("111EE") ->	{"0","",""};
lookup1("111EF") ->	{"0","",""};
lookup1("111F") ->	{"0","",""};
lookup1("111F0") ->	{"0","",""};
lookup1("111F1") ->	{"0","",""};
lookup1("111F2") ->	{"0","",""};
lookup1("111F3") ->	{"0","",""};
lookup1("111F4") ->	{"0","",""};
lookup1("1120") ->	{"0","",""};
lookup1("11200") ->	{"0","",""};
lookup1("11201") ->	{"0","",""};
lookup1("11202") ->	{"0","",""};
lookup1("11203") ->	{"0","",""};
lookup1("11204") ->	{"0","",""};
lookup1("11205") ->	{"0","",""};
lookup1("11206") ->	{"0","",""};
lookup1("11207") ->	{"0","",""};
lookup1("11208") ->	{"0","",""};
lookup1("11209") ->	{"0","",""};
lookup1("1120A") ->	{"0","",""};
lookup1("1120B") ->	{"0","",""};
lookup1("1120C") ->	{"0","",""};
lookup1("1120D") ->	{"0","",""};
lookup1("1120E") ->	{"0","",""};
lookup1("1120F") ->	{"0","",""};
lookup1("1121") ->	{"0","",""};
lookup1("11210") ->	{"0","",""};
lookup1("11211") ->	{"0","",""};
lookup1("11213") ->	{"0","",""};
lookup1("11214") ->	{"0","",""};
lookup1("11215") ->	{"0","",""};
lookup1("11216") ->	{"0","",""};
lookup1("11217") ->	{"0","",""};
lookup1("11218") ->	{"0","",""};
lookup1("11219") ->	{"0","",""};
lookup1("1121A") ->	{"0","",""};
lookup1("1121B") ->	{"0","",""};
lookup1("1121C") ->	{"0","",""};
lookup1("1121D") ->	{"0","",""};
lookup1("1121E") ->	{"0","",""};
lookup1("1121F") ->	{"0","",""};
lookup1("1122") ->	{"0","",""};
lookup1("11220") ->	{"0","",""};
lookup1("11221") ->	{"0","",""};
lookup1("11222") ->	{"0","",""};
lookup1("11223") ->	{"0","",""};
lookup1("11224") ->	{"0","",""};
lookup1("11225") ->	{"0","",""};
lookup1("11226") ->	{"0","",""};
lookup1("11227") ->	{"0","",""};
lookup1("11228") ->	{"0","",""};
lookup1("11229") ->	{"0","",""};
lookup1("1122A") ->	{"0","",""};
lookup1("1122B") ->	{"0","",""};
lookup1("1122C") ->	{"0","",""};
lookup1("1122D") ->	{"0","",""};
lookup1("1122E") ->	{"0","",""};
lookup1("1122F") ->	{"0","",""};
lookup1("1123") ->	{"0","",""};
lookup1("11230") ->	{"0","",""};
lookup1("11231") ->	{"0","",""};
lookup1("11232") ->	{"0","",""};
lookup1("11233") ->	{"0","",""};
lookup1("11234") ->	{"0","",""};
lookup1("11235") ->	{"9","",""};
lookup1("11236") ->	{"7","",""};
lookup1("11237") ->	{"0","",""};
lookup1("11238") ->	{"0","",""};
lookup1("11239") ->	{"0","",""};
lookup1("1123A") ->	{"0","",""};
lookup1("1123B") ->	{"0","",""};
lookup1("1123C") ->	{"0","",""};
lookup1("1123D") ->	{"0","",""};
lookup1("1124") ->	{"0","",""};
lookup1("1125") ->	{"0","",""};
lookup1("1126") ->	{"0","",""};
lookup1("1127") ->	{"0","",""};
lookup1("1128") ->	{"0","",""};
lookup1("1129") ->	{"0","",""};
lookup1("112A") ->	{"0","",""};
lookup1("112B") ->	{"0","",""};
lookup1("112B0") ->	{"0","",""};
lookup1("112B1") ->	{"0","",""};
lookup1("112B2") ->	{"0","",""};
lookup1("112B3") ->	{"0","",""};
lookup1("112B4") ->	{"0","",""};
lookup1("112B5") ->	{"0","",""};
lookup1("112B6") ->	{"0","",""};
lookup1("112B7") ->	{"0","",""};
lookup1("112B8") ->	{"0","",""};
lookup1("112B9") ->	{"0","",""};
lookup1("112BA") ->	{"0","",""};
lookup1("112BB") ->	{"0","",""};
lookup1("112BC") ->	{"0","",""};
lookup1("112BD") ->	{"0","",""};
lookup1("112BE") ->	{"0","",""};
lookup1("112BF") ->	{"0","",""};
lookup1("112C") ->	{"0","",""};
lookup1("112C0") ->	{"0","",""};
lookup1("112C1") ->	{"0","",""};
lookup1("112C2") ->	{"0","",""};
lookup1("112C3") ->	{"0","",""};
lookup1("112C4") ->	{"0","",""};
lookup1("112C5") ->	{"0","",""};
lookup1("112C6") ->	{"0","",""};
lookup1("112C7") ->	{"0","",""};
lookup1("112C8") ->	{"0","",""};
lookup1("112C9") ->	{"0","",""};
lookup1("112CA") ->	{"0","",""};
lookup1("112CB") ->	{"0","",""};
lookup1("112CC") ->	{"0","",""};
lookup1("112CD") ->	{"0","",""};
lookup1("112CE") ->	{"0","",""};
lookup1("112CF") ->	{"0","",""};
lookup1("112D") ->	{"0","",""};
lookup1("112D0") ->	{"0","",""};
lookup1("112D1") ->	{"0","",""};
lookup1("112D2") ->	{"0","",""};
lookup1("112D3") ->	{"0","",""};
lookup1("112D4") ->	{"0","",""};
lookup1("112D5") ->	{"0","",""};
lookup1("112D6") ->	{"0","",""};
lookup1("112D7") ->	{"0","",""};
lookup1("112D8") ->	{"0","",""};
lookup1("112D9") ->	{"0","",""};
lookup1("112DA") ->	{"0","",""};
lookup1("112DB") ->	{"0","",""};
lookup1("112DC") ->	{"0","",""};
lookup1("112DD") ->	{"0","",""};
lookup1("112DE") ->	{"0","",""};
lookup1("112DF") ->	{"0","",""};
lookup1("112E") ->	{"0","",""};
lookup1("112E0") ->	{"0","",""};
lookup1("112E1") ->	{"0","",""};
lookup1("112E2") ->	{"0","",""};
lookup1("112E3") ->	{"0","",""};
lookup1("112E4") ->	{"0","",""};
lookup1("112E5") ->	{"0","",""};
lookup1("112E6") ->	{"0","",""};
lookup1("112E7") ->	{"0","",""};
lookup1("112E8") ->	{"0","",""};
lookup1("112E9") ->	{"7","",""};
lookup1("112EA") ->	{"9","",""};
lookup1("112F") ->	{"0","",""};
lookup1("112F0") ->	{"0","",""};
lookup1("112F1") ->	{"0","",""};
lookup1("112F2") ->	{"0","",""};
lookup1("112F3") ->	{"0","",""};
lookup1("112F4") ->	{"0","",""};
lookup1("112F5") ->	{"0","",""};
lookup1("112F6") ->	{"0","",""};
lookup1("112F7") ->	{"0","",""};
lookup1("112F8") ->	{"0","",""};
lookup1("112F9") ->	{"0","",""};
lookup1("1130") ->	{"0","",""};
lookup1("11301") ->	{"0","",""};
lookup1("11302") ->	{"0","",""};
lookup1("11303") ->	{"0","",""};
lookup1("11305") ->	{"0","",""};
lookup1("11306") ->	{"0","",""};
lookup1("11307") ->	{"0","",""};
lookup1("11308") ->	{"0","",""};
lookup1("11309") ->	{"0","",""};
lookup1("1130A") ->	{"0","",""};
lookup1("1130B") ->	{"0","",""};
lookup1("1130C") ->	{"0","",""};
lookup1("1130F") ->	{"0","",""};
lookup1("1131") ->	{"0","",""};
lookup1("11310") ->	{"0","",""};
lookup1("11313") ->	{"0","",""};
lookup1("11314") ->	{"0","",""};
lookup1("11315") ->	{"0","",""};
lookup1("11316") ->	{"0","",""};
lookup1("11317") ->	{"0","",""};
lookup1("11318") ->	{"0","",""};
lookup1("11319") ->	{"0","",""};
lookup1("1131A") ->	{"0","",""};
lookup1("1131B") ->	{"0","",""};
lookup1("1131C") ->	{"0","",""};
lookup1("1131D") ->	{"0","",""};
lookup1("1131E") ->	{"0","",""};
lookup1("1131F") ->	{"0","",""};
lookup1("1132") ->	{"0","",""};
lookup1("11320") ->	{"0","",""};
lookup1("11321") ->	{"0","",""};
lookup1("11322") ->	{"0","",""};
lookup1("11323") ->	{"0","",""};
lookup1("11324") ->	{"0","",""};
lookup1("11325") ->	{"0","",""};
lookup1("11326") ->	{"0","",""};
lookup1("11327") ->	{"0","",""};
lookup1("11328") ->	{"0","",""};
lookup1("1132A") ->	{"0","",""};
lookup1("1132B") ->	{"0","",""};
lookup1("1132C") ->	{"0","",""};
lookup1("1132D") ->	{"0","",""};
lookup1("1132E") ->	{"0","",""};
lookup1("1132F") ->	{"0","",""};
lookup1("1133") ->	{"0","",""};
lookup1("11330") ->	{"0","",""};
lookup1("11332") ->	{"0","",""};
lookup1("11333") ->	{"0","",""};
lookup1("11335") ->	{"0","",""};
lookup1("11336") ->	{"0","",""};
lookup1("11337") ->	{"0","",""};
lookup1("11338") ->	{"0","",""};
lookup1("11339") ->	{"0","",""};
lookup1("1133C") ->	{"7","",""};
lookup1("1133D") ->	{"0","",""};
lookup1("1133E") ->	{"0","",""};
lookup1("1133F") ->	{"0","",""};
lookup1("1134") ->	{"0","",""};
lookup1("11340") ->	{"0","",""};
lookup1("11341") ->	{"0","",""};
lookup1("11342") ->	{"0","",""};
lookup1("11343") ->	{"0","",""};
lookup1("11344") ->	{"0","",""};
lookup1("11347") ->	{"0","",""};
lookup1("11348") ->	{"0","",""};
lookup1("1134B") ->	{"0","11347 1133E",""};
lookup1("1134C") ->	{"0","11347 11357",""};
lookup1("1134D") ->	{"9","",""};
lookup1("1135") ->	{"0","",""};
lookup1("11357") ->	{"0","",""};
lookup1("1135D") ->	{"0","",""};
lookup1("1135E") ->	{"0","",""};
lookup1("1135F") ->	{"0","",""};
lookup1("1136") ->	{"0","",""};
lookup1("11360") ->	{"0","",""};
lookup1("11361") ->	{"0","",""};
lookup1("11362") ->	{"0","",""};
lookup1("11363") ->	{"0","",""};
lookup1("11366") ->	{"230","",""};
lookup1("11367") ->	{"230","",""};
lookup1("11368") ->	{"230","",""};
lookup1("11369") ->	{"230","",""};
lookup1("1136A") ->	{"230","",""};
lookup1("1136B") ->	{"230","",""};
lookup1("1136C") ->	{"230","",""};
lookup1("1137") ->	{"0","",""};
lookup1("11370") ->	{"230","",""};
lookup1("11371") ->	{"230","",""};
lookup1("11372") ->	{"230","",""};
lookup1("11373") ->	{"230","",""};
lookup1("11374") ->	{"230","",""};
lookup1("1138") ->	{"0","",""};
lookup1("1139") ->	{"0","",""};
lookup1("113A") ->	{"0","",""};
lookup1("113B") ->	{"0","",""};
lookup1("113C") ->	{"0","",""};
lookup1("113D") ->	{"0","",""};
lookup1("113E") ->	{"0","",""};
lookup1("113F") ->	{"0","",""};
lookup1("1140") ->	{"0","",""};
lookup1("1141") ->	{"0","",""};
lookup1("1142") ->	{"0","",""};
lookup1("1143") ->	{"0","",""};
lookup1("1144") ->	{"0","",""};
lookup1("1145") ->	{"0","",""};
lookup1("1146") ->	{"0","",""};
lookup1("1147") ->	{"0","",""};
lookup1("1148") ->	{"0","",""};
lookup1("11480") ->	{"0","",""};
lookup1("11481") ->	{"0","",""};
lookup1("11482") ->	{"0","",""};
lookup1("11483") ->	{"0","",""};
lookup1("11484") ->	{"0","",""};
lookup1("11485") ->	{"0","",""};
lookup1("11486") ->	{"0","",""};
lookup1("11487") ->	{"0","",""};
lookup1("11488") ->	{"0","",""};
lookup1("11489") ->	{"0","",""};
lookup1("1148A") ->	{"0","",""};
lookup1("1148B") ->	{"0","",""};
lookup1("1148C") ->	{"0","",""};
lookup1("1148D") ->	{"0","",""};
lookup1("1148E") ->	{"0","",""};
lookup1("1148F") ->	{"0","",""};
lookup1("1149") ->	{"0","",""};
lookup1("11490") ->	{"0","",""};
lookup1("11491") ->	{"0","",""};
lookup1("11492") ->	{"0","",""};
lookup1("11493") ->	{"0","",""};
lookup1("11494") ->	{"0","",""};
lookup1("11495") ->	{"0","",""};
lookup1("11496") ->	{"0","",""};
lookup1("11497") ->	{"0","",""};
lookup1("11498") ->	{"0","",""};
lookup1("11499") ->	{"0","",""};
lookup1("1149A") ->	{"0","",""};
lookup1("1149B") ->	{"0","",""};
lookup1("1149C") ->	{"0","",""};
lookup1("1149D") ->	{"0","",""};
lookup1("1149E") ->	{"0","",""};
lookup1("1149F") ->	{"0","",""};
lookup1("114A") ->	{"0","",""};
lookup1("114A0") ->	{"0","",""};
lookup1("114A1") ->	{"0","",""};
lookup1("114A2") ->	{"0","",""};
lookup1("114A3") ->	{"0","",""};
lookup1("114A4") ->	{"0","",""};
lookup1("114A5") ->	{"0","",""};
lookup1("114A6") ->	{"0","",""};
lookup1("114A7") ->	{"0","",""};
lookup1("114A8") ->	{"0","",""};
lookup1("114A9") ->	{"0","",""};
lookup1("114AA") ->	{"0","",""};
lookup1("114AB") ->	{"0","",""};
lookup1("114AC") ->	{"0","",""};
lookup1("114AD") ->	{"0","",""};
lookup1("114AE") ->	{"0","",""};
lookup1("114AF") ->	{"0","",""};
lookup1("114B") ->	{"0","",""};
lookup1("114B0") ->	{"0","",""};
lookup1("114B1") ->	{"0","",""};
lookup1("114B2") ->	{"0","",""};
lookup1("114B3") ->	{"0","",""};
lookup1("114B4") ->	{"0","",""};
lookup1("114B5") ->	{"0","",""};
lookup1("114B6") ->	{"0","",""};
lookup1("114B7") ->	{"0","",""};
lookup1("114B8") ->	{"0","",""};
lookup1("114B9") ->	{"0","",""};
lookup1("114BA") ->	{"0","",""};
lookup1("114BB") ->	{"0","114B9 114BA",""};
lookup1("114BC") ->	{"0","114B9 114B0",""};
lookup1("114BD") ->	{"0","",""};
lookup1("114BE") ->	{"0","114B9 114BD",""};
lookup1("114BF") ->	{"0","",""};
lookup1("114C") ->	{"0","",""};
lookup1("114C0") ->	{"0","",""};
lookup1("114C1") ->	{"0","",""};
lookup1("114C2") ->	{"9","",""};
lookup1("114C3") ->	{"7","",""};
lookup1("114C4") ->	{"0","",""};
lookup1("114C5") ->	{"0","",""};
lookup1("114C6") ->	{"0","",""};
lookup1("114C7") ->	{"0","",""};
lookup1("114D") ->	{"0","",""};
lookup1("114D0") ->	{"0","",""};
lookup1("114D1") ->	{"0","",""};
lookup1("114D2") ->	{"0","",""};
lookup1("114D3") ->	{"0","",""};
lookup1("114D4") ->	{"0","",""};
lookup1("114D5") ->	{"0","",""};
lookup1("114D6") ->	{"0","",""};
lookup1("114D7") ->	{"0","",""};
lookup1("114D8") ->	{"0","",""};
lookup1("114D9") ->	{"0","",""};
lookup1("114E") ->	{"0","",""};
lookup1("114F") ->	{"0","",""};
lookup1("1150") ->	{"0","",""};
lookup1("1151") ->	{"0","",""};
lookup1("1152") ->	{"0","",""};
lookup1("1153") ->	{"0","",""};
lookup1("1154") ->	{"0","",""};
lookup1("1155") ->	{"0","",""};
lookup1("1156") ->	{"0","",""};
lookup1("1157") ->	{"0","",""};
lookup1("1158") ->	{"0","",""};
lookup1("11580") ->	{"0","",""};
lookup1("11581") ->	{"0","",""};
lookup1("11582") ->	{"0","",""};
lookup1("11583") ->	{"0","",""};
lookup1("11584") ->	{"0","",""};
lookup1("11585") ->	{"0","",""};
lookup1("11586") ->	{"0","",""};
lookup1("11587") ->	{"0","",""};
lookup1("11588") ->	{"0","",""};
lookup1("11589") ->	{"0","",""};
lookup1("1158A") ->	{"0","",""};
lookup1("1158B") ->	{"0","",""};
lookup1("1158C") ->	{"0","",""};
lookup1("1158D") ->	{"0","",""};
lookup1("1158E") ->	{"0","",""};
lookup1("1158F") ->	{"0","",""};
lookup1("1159") ->	{"0","",""};
lookup1("11590") ->	{"0","",""};
lookup1("11591") ->	{"0","",""};
lookup1("11592") ->	{"0","",""};
lookup1("11593") ->	{"0","",""};
lookup1("11594") ->	{"0","",""};
lookup1("11595") ->	{"0","",""};
lookup1("11596") ->	{"0","",""};
lookup1("11597") ->	{"0","",""};
lookup1("11598") ->	{"0","",""};
lookup1("11599") ->	{"0","",""};
lookup1("1159A") ->	{"0","",""};
lookup1("1159B") ->	{"0","",""};
lookup1("1159C") ->	{"0","",""};
lookup1("1159D") ->	{"0","",""};
lookup1("1159E") ->	{"0","",""};
lookup1("1159F") ->	{"0","",""};
lookup1("115A") ->	{"0","",""};
lookup1("115A0") ->	{"0","",""};
lookup1("115A1") ->	{"0","",""};
lookup1("115A2") ->	{"0","",""};
lookup1("115A3") ->	{"0","",""};
lookup1("115A4") ->	{"0","",""};
lookup1("115A5") ->	{"0","",""};
lookup1("115A6") ->	{"0","",""};
lookup1("115A7") ->	{"0","",""};
lookup1("115A8") ->	{"0","",""};
lookup1("115A9") ->	{"0","",""};
lookup1("115AA") ->	{"0","",""};
lookup1("115AB") ->	{"0","",""};
lookup1("115AC") ->	{"0","",""};
lookup1("115AD") ->	{"0","",""};
lookup1("115AE") ->	{"0","",""};
lookup1("115AF") ->	{"0","",""};
lookup1("115B") ->	{"0","",""};
lookup1("115B0") ->	{"0","",""};
lookup1("115B1") ->	{"0","",""};
lookup1("115B2") ->	{"0","",""};
lookup1("115B3") ->	{"0","",""};
lookup1("115B4") ->	{"0","",""};
lookup1("115B5") ->	{"0","",""};
lookup1("115B8") ->	{"0","",""};
lookup1("115B9") ->	{"0","",""};
lookup1("115BA") ->	{"0","115B8 115AF",""};
lookup1("115BB") ->	{"0","115B9 115AF",""};
lookup1("115BC") ->	{"0","",""};
lookup1("115BD") ->	{"0","",""};
lookup1("115BE") ->	{"0","",""};
lookup1("115BF") ->	{"9","",""};
lookup1("115C") ->	{"0","",""};
lookup1("115C0") ->	{"7","",""};
lookup1("115C1") ->	{"0","",""};
lookup1("115C2") ->	{"0","",""};
lookup1("115C3") ->	{"0","",""};
lookup1("115C4") ->	{"0","",""};
lookup1("115C5") ->	{"0","",""};
lookup1("115C6") ->	{"0","",""};
lookup1("115C7") ->	{"0","",""};
lookup1("115C8") ->	{"0","",""};
lookup1("115C9") ->	{"0","",""};
lookup1("115D") ->	{"0","",""};
lookup1("115E") ->	{"0","",""};
lookup1("115F") ->	{"0","",""};
lookup1("1160") ->	{"0","",""};
lookup1("11600") ->	{"0","",""};
lookup1("11601") ->	{"0","",""};
lookup1("11602") ->	{"0","",""};
lookup1("11603") ->	{"0","",""};
lookup1("11604") ->	{"0","",""};
lookup1("11605") ->	{"0","",""};
lookup1("11606") ->	{"0","",""};
lookup1("11607") ->	{"0","",""};
lookup1("11608") ->	{"0","",""};
lookup1("11609") ->	{"0","",""};
lookup1("1160A") ->	{"0","",""};
lookup1("1160B") ->	{"0","",""};
lookup1("1160C") ->	{"0","",""};
lookup1("1160D") ->	{"0","",""};
lookup1("1160E") ->	{"0","",""};
lookup1("1160F") ->	{"0","",""};
lookup1("1161") ->	{"0","",""};
lookup1("11610") ->	{"0","",""};
lookup1("11611") ->	{"0","",""};
lookup1("11612") ->	{"0","",""};
lookup1("11613") ->	{"0","",""};
lookup1("11614") ->	{"0","",""};
lookup1("11615") ->	{"0","",""};
lookup1("11616") ->	{"0","",""};
lookup1("11617") ->	{"0","",""};
lookup1("11618") ->	{"0","",""};
lookup1("11619") ->	{"0","",""};
lookup1("1161A") ->	{"0","",""};
lookup1("1161B") ->	{"0","",""};
lookup1("1161C") ->	{"0","",""};
lookup1("1161D") ->	{"0","",""};
lookup1("1161E") ->	{"0","",""};
lookup1("1161F") ->	{"0","",""};
lookup1("1162") ->	{"0","",""};
lookup1("11620") ->	{"0","",""};
lookup1("11621") ->	{"0","",""};
lookup1("11622") ->	{"0","",""};
lookup1("11623") ->	{"0","",""};
lookup1("11624") ->	{"0","",""};
lookup1("11625") ->	{"0","",""};
lookup1("11626") ->	{"0","",""};
lookup1("11627") ->	{"0","",""};
lookup1("11628") ->	{"0","",""};
lookup1("11629") ->	{"0","",""};
lookup1("1162A") ->	{"0","",""};
lookup1("1162B") ->	{"0","",""};
lookup1("1162C") ->	{"0","",""};
lookup1("1162D") ->	{"0","",""};
lookup1("1162E") ->	{"0","",""};
lookup1("1162F") ->	{"0","",""};
lookup1("1163") ->	{"0","",""};
lookup1("11630") ->	{"0","",""};
lookup1("11631") ->	{"0","",""};
lookup1("11632") ->	{"0","",""};
lookup1("11633") ->	{"0","",""};
lookup1("11634") ->	{"0","",""};
lookup1("11635") ->	{"0","",""};
lookup1("11636") ->	{"0","",""};
lookup1("11637") ->	{"0","",""};
lookup1("11638") ->	{"0","",""};
lookup1("11639") ->	{"0","",""};
lookup1("1163A") ->	{"0","",""};
lookup1("1163B") ->	{"0","",""};
lookup1("1163C") ->	{"0","",""};
lookup1("1163D") ->	{"0","",""};
lookup1("1163E") ->	{"0","",""};
lookup1("1163F") ->	{"9","",""};
lookup1("1164") ->	{"0","",""};
lookup1("11640") ->	{"0","",""};
lookup1("11641") ->	{"0","",""};
lookup1("11642") ->	{"0","",""};
lookup1("11643") ->	{"0","",""};
lookup1("11644") ->	{"0","",""};
lookup1("1165") ->	{"0","",""};
lookup1("11650") ->	{"0","",""};
lookup1("11651") ->	{"0","",""};
lookup1("11652") ->	{"0","",""};
lookup1("11653") ->	{"0","",""};
lookup1("11654") ->	{"0","",""};
lookup1("11655") ->	{"0","",""};
lookup1("11656") ->	{"0","",""};
lookup1("11657") ->	{"0","",""};
lookup1("11658") ->	{"0","",""};
lookup1("11659") ->	{"0","",""};
lookup1("1166") ->	{"0","",""};
lookup1("1167") ->	{"0","",""};
lookup1("1168") ->	{"0","",""};
lookup1("11680") ->	{"0","",""};
lookup1("11681") ->	{"0","",""};
lookup1("11682") ->	{"0","",""};
lookup1("11683") ->	{"0","",""};
lookup1("11684") ->	{"0","",""};
lookup1("11685") ->	{"0","",""};
lookup1("11686") ->	{"0","",""};
lookup1("11687") ->	{"0","",""};
lookup1("11688") ->	{"0","",""};
lookup1("11689") ->	{"0","",""};
lookup1("1168A") ->	{"0","",""};
lookup1("1168B") ->	{"0","",""};
lookup1("1168C") ->	{"0","",""};
lookup1("1168D") ->	{"0","",""};
lookup1("1168E") ->	{"0","",""};
lookup1("1168F") ->	{"0","",""};
lookup1("1169") ->	{"0","",""};
lookup1("11690") ->	{"0","",""};
lookup1("11691") ->	{"0","",""};
lookup1("11692") ->	{"0","",""};
lookup1("11693") ->	{"0","",""};
lookup1("11694") ->	{"0","",""};
lookup1("11695") ->	{"0","",""};
lookup1("11696") ->	{"0","",""};
lookup1("11697") ->	{"0","",""};
lookup1("11698") ->	{"0","",""};
lookup1("11699") ->	{"0","",""};
lookup1("1169A") ->	{"0","",""};
lookup1("1169B") ->	{"0","",""};
lookup1("1169C") ->	{"0","",""};
lookup1("1169D") ->	{"0","",""};
lookup1("1169E") ->	{"0","",""};
lookup1("1169F") ->	{"0","",""};
lookup1("116A") ->	{"0","",""};
lookup1("116A0") ->	{"0","",""};
lookup1("116A1") ->	{"0","",""};
lookup1("116A2") ->	{"0","",""};
lookup1("116A3") ->	{"0","",""};
lookup1("116A4") ->	{"0","",""};
lookup1("116A5") ->	{"0","",""};
lookup1("116A6") ->	{"0","",""};
lookup1("116A7") ->	{"0","",""};
lookup1("116A8") ->	{"0","",""};
lookup1("116A9") ->	{"0","",""};
lookup1("116AA") ->	{"0","",""};
lookup1("116AB") ->	{"0","",""};
lookup1("116AC") ->	{"0","",""};
lookup1("116AD") ->	{"0","",""};
lookup1("116AE") ->	{"0","",""};
lookup1("116AF") ->	{"0","",""};
lookup1("116B") ->	{"0","",""};
lookup1("116B0") ->	{"0","",""};
lookup1("116B1") ->	{"0","",""};
lookup1("116B2") ->	{"0","",""};
lookup1("116B3") ->	{"0","",""};
lookup1("116B4") ->	{"0","",""};
lookup1("116B5") ->	{"0","",""};
lookup1("116B6") ->	{"9","",""};
lookup1("116B7") ->	{"7","",""};
lookup1("116C") ->	{"0","",""};
lookup1("116C0") ->	{"0","",""};
lookup1("116C1") ->	{"0","",""};
lookup1("116C2") ->	{"0","",""};
lookup1("116C3") ->	{"0","",""};
lookup1("116C4") ->	{"0","",""};
lookup1("116C5") ->	{"0","",""};
lookup1("116C6") ->	{"0","",""};
lookup1("116C7") ->	{"0","",""};
lookup1("116C8") ->	{"0","",""};
lookup1("116C9") ->	{"0","",""};
lookup1("116D") ->	{"0","",""};
lookup1("116E") ->	{"0","",""};
lookup1("116F") ->	{"0","",""};
lookup1("1170") ->	{"0","",""};
lookup1("1171") ->	{"0","",""};
lookup1("1172") ->	{"0","",""};
lookup1("1173") ->	{"0","",""};
lookup1("1174") ->	{"0","",""};
lookup1("1175") ->	{"0","",""};
lookup1("1176") ->	{"0","",""};
lookup1("1177") ->	{"0","",""};
lookup1("1178") ->	{"0","",""};
lookup1("1179") ->	{"0","",""};
lookup1("117A") ->	{"0","",""};
lookup1("117B") ->	{"0","",""};
lookup1("117C") ->	{"0","",""};
lookup1("117D") ->	{"0","",""};
lookup1("117E") ->	{"0","",""};
lookup1("117F") ->	{"0","",""};
lookup1("1180") ->	{"0","",""};
lookup1("1181") ->	{"0","",""};
lookup1("1182") ->	{"0","",""};
lookup1("1183") ->	{"0","",""};
lookup1("1184") ->	{"0","",""};
lookup1("1185") ->	{"0","",""};
lookup1("1186") ->	{"0","",""};
lookup1("1187") ->	{"0","",""};
lookup1("1188") ->	{"0","",""};
lookup1("1189") ->	{"0","",""};
lookup1("118A") ->	{"0","",""};
lookup1("118A0") ->	{"0","","118C0"};
lookup1("118A1") ->	{"0","","118C1"};
lookup1("118A2") ->	{"0","","118C2"};
lookup1("118A3") ->	{"0","","118C3"};
lookup1("118A4") ->	{"0","","118C4"};
lookup1("118A5") ->	{"0","","118C5"};
lookup1("118A6") ->	{"0","","118C6"};
lookup1("118A7") ->	{"0","","118C7"};
lookup1("118A8") ->	{"0","","118C8"};
lookup1("118A9") ->	{"0","","118C9"};
lookup1("118AA") ->	{"0","","118CA"};
lookup1("118AB") ->	{"0","","118CB"};
lookup1("118AC") ->	{"0","","118CC"};
lookup1("118AD") ->	{"0","","118CD"};
lookup1("118AE") ->	{"0","","118CE"};
lookup1("118AF") ->	{"0","","118CF"};
lookup1("118B") ->	{"0","",""};
lookup1("118B0") ->	{"0","","118D0"};
lookup1("118B1") ->	{"0","","118D1"};
lookup1("118B2") ->	{"0","","118D2"};
lookup1("118B3") ->	{"0","","118D3"};
lookup1("118B4") ->	{"0","","118D4"};
lookup1("118B5") ->	{"0","","118D5"};
lookup1("118B6") ->	{"0","","118D6"};
lookup1("118B7") ->	{"0","","118D7"};
lookup1("118B8") ->	{"0","","118D8"};
lookup1("118B9") ->	{"0","","118D9"};
lookup1("118BA") ->	{"0","","118DA"};
lookup1("118BB") ->	{"0","","118DB"};
lookup1("118BC") ->	{"0","","118DC"};
lookup1("118BD") ->	{"0","","118DD"};
lookup1("118BE") ->	{"0","","118DE"};
lookup1("118BF") ->	{"0","","118DF"};
lookup1("118C") ->	{"0","",""};
lookup1("118C0") ->	{"0","",""};
lookup1("118C1") ->	{"0","",""};
lookup1("118C2") ->	{"0","",""};
lookup1("118C3") ->	{"0","",""};
lookup1("118C4") ->	{"0","",""};
lookup1("118C5") ->	{"0","",""};
lookup1("118C6") ->	{"0","",""};
lookup1("118C7") ->	{"0","",""};
lookup1("118C8") ->	{"0","",""};
lookup1("118C9") ->	{"0","",""};
lookup1("118CA") ->	{"0","",""};
lookup1("118CB") ->	{"0","",""};
lookup1("118CC") ->	{"0","",""};
lookup1("118CD") ->	{"0","",""};
lookup1("118CE") ->	{"0","",""};
lookup1("118CF") ->	{"0","",""};
lookup1("118D") ->	{"0","",""};
lookup1("118D0") ->	{"0","",""};
lookup1("118D1") ->	{"0","",""};
lookup1("118D2") ->	{"0","",""};
lookup1("118D3") ->	{"0","",""};
lookup1("118D4") ->	{"0","",""};
lookup1("118D5") ->	{"0","",""};
lookup1("118D6") ->	{"0","",""};
lookup1("118D7") ->	{"0","",""};
lookup1("118D8") ->	{"0","",""};
lookup1("118D9") ->	{"0","",""};
lookup1("118DA") ->	{"0","",""};
lookup1("118DB") ->	{"0","",""};
lookup1("118DC") ->	{"0","",""};
lookup1("118DD") ->	{"0","",""};
lookup1("118DE") ->	{"0","",""};
lookup1("118DF") ->	{"0","",""};
lookup1("118E") ->	{"0","",""};
lookup1("118E0") ->	{"0","",""};
lookup1("118E1") ->	{"0","",""};
lookup1("118E2") ->	{"0","",""};
lookup1("118E3") ->	{"0","",""};
lookup1("118E4") ->	{"0","",""};
lookup1("118E5") ->	{"0","",""};
lookup1("118E6") ->	{"0","",""};
lookup1("118E7") ->	{"0","",""};
lookup1("118E8") ->	{"0","",""};
lookup1("118E9") ->	{"0","",""};
lookup1("118EA") ->	{"0","",""};
lookup1("118EB") ->	{"0","",""};
lookup1("118EC") ->	{"0","",""};
lookup1("118ED") ->	{"0","",""};
lookup1("118EE") ->	{"0","",""};
lookup1("118EF") ->	{"0","",""};
lookup1("118F") ->	{"0","",""};
lookup1("118F0") ->	{"0","",""};
lookup1("118F1") ->	{"0","",""};
lookup1("118F2") ->	{"0","",""};
lookup1("118FF") ->	{"0","",""};
lookup1("1190") ->	{"0","",""};
lookup1("1191") ->	{"0","",""};
lookup1("1192") ->	{"0","",""};
lookup1("1193") ->	{"0","",""};
lookup1("1194") ->	{"0","",""};
lookup1("1195") ->	{"0","",""};
lookup1("1196") ->	{"0","",""};
lookup1("1197") ->	{"0","",""};
lookup1("1198") ->	{"0","",""};
lookup1("1199") ->	{"0","",""};
lookup1("119A") ->	{"0","",""};
lookup1("119B") ->	{"0","",""};
lookup1("119C") ->	{"0","",""};
lookup1("119D") ->	{"0","",""};
lookup1("119E") ->	{"0","",""};
lookup1("119F") ->	{"0","",""};
lookup1("11A0") ->	{"0","",""};
lookup1("11A1") ->	{"0","",""};
lookup1("11A2") ->	{"0","",""};
lookup1("11A3") ->	{"0","",""};
lookup1("11A4") ->	{"0","",""};
lookup1("11A5") ->	{"0","",""};
lookup1("11A6") ->	{"0","",""};
lookup1("11A7") ->	{"0","",""};
lookup1("11A8") ->	{"0","",""};
lookup1("11A9") ->	{"0","",""};
lookup1("11AA") ->	{"0","",""};
lookup1("11AB") ->	{"0","",""};
lookup1("11AC") ->	{"0","",""};
lookup1("11AC0") ->	{"0","",""};
lookup1("11AC1") ->	{"0","",""};
lookup1("11AC2") ->	{"0","",""};
lookup1("11AC3") ->	{"0","",""};
lookup1("11AC4") ->	{"0","",""};
lookup1("11AC5") ->	{"0","",""};
lookup1("11AC6") ->	{"0","",""};
lookup1("11AC7") ->	{"0","",""};
lookup1("11AC8") ->	{"0","",""};
lookup1("11AC9") ->	{"0","",""};
lookup1("11ACA") ->	{"0","",""};
lookup1("11ACB") ->	{"0","",""};
lookup1("11ACC") ->	{"0","",""};
lookup1("11ACD") ->	{"0","",""};
lookup1("11ACE") ->	{"0","",""};
lookup1("11ACF") ->	{"0","",""};
lookup1("11AD") ->	{"0","",""};
lookup1("11AD0") ->	{"0","",""};
lookup1("11AD1") ->	{"0","",""};
lookup1("11AD2") ->	{"0","",""};
lookup1("11AD3") ->	{"0","",""};
lookup1("11AD4") ->	{"0","",""};
lookup1("11AD5") ->	{"0","",""};
lookup1("11AD6") ->	{"0","",""};
lookup1("11AD7") ->	{"0","",""};
lookup1("11AD8") ->	{"0","",""};
lookup1("11AD9") ->	{"0","",""};
lookup1("11ADA") ->	{"0","",""};
lookup1("11ADB") ->	{"0","",""};
lookup1("11ADC") ->	{"0","",""};
lookup1("11ADD") ->	{"0","",""};
lookup1("11ADE") ->	{"0","",""};
lookup1("11ADF") ->	{"0","",""};
lookup1("11AE") ->	{"0","",""};
lookup1("11AE0") ->	{"0","",""};
lookup1("11AE1") ->	{"0","",""};
lookup1("11AE2") ->	{"0","",""};
lookup1("11AE3") ->	{"0","",""};
lookup1("11AE4") ->	{"0","",""};
lookup1("11AE5") ->	{"0","",""};
lookup1("11AE6") ->	{"0","",""};
lookup1("11AE7") ->	{"0","",""};
lookup1("11AE8") ->	{"0","",""};
lookup1("11AE9") ->	{"0","",""};
lookup1("11AEA") ->	{"0","",""};
lookup1("11AEB") ->	{"0","",""};
lookup1("11AEC") ->	{"0","",""};
lookup1("11AED") ->	{"0","",""};
lookup1("11AEE") ->	{"0","",""};
lookup1("11AEF") ->	{"0","",""};
lookup1("11AF") ->	{"0","",""};
lookup1("11AF0") ->	{"0","",""};
lookup1("11AF1") ->	{"0","",""};
lookup1("11AF2") ->	{"0","",""};
lookup1("11AF3") ->	{"0","",""};
lookup1("11AF4") ->	{"0","",""};
lookup1("11AF5") ->	{"0","",""};
lookup1("11AF6") ->	{"0","",""};
lookup1("11AF7") ->	{"0","",""};
lookup1("11AF8") ->	{"0","",""};
lookup1("11B0") ->	{"0","",""};
lookup1("11B1") ->	{"0","",""};
lookup1("11B2") ->	{"0","",""};
lookup1("11B3") ->	{"0","",""};
lookup1("11B4") ->	{"0","",""};
lookup1("11B5") ->	{"0","",""};
lookup1("11B6") ->	{"0","",""};
lookup1("11B7") ->	{"0","",""};
lookup1("11B8") ->	{"0","",""};
lookup1("11B9") ->	{"0","",""};
lookup1("11BA") ->	{"0","",""};
lookup1("11BB") ->	{"0","",""};
lookup1("11BC") ->	{"0","",""};
lookup1("11BD") ->	{"0","",""};
lookup1("11BE") ->	{"0","",""};
lookup1("11BF") ->	{"0","",""};
lookup1("11C0") ->	{"0","",""};
lookup1("11C1") ->	{"0","",""};
lookup1("11C2") ->	{"0","",""};
lookup1("11C3") ->	{"0","",""};
lookup1("11C4") ->	{"0","",""};
lookup1("11C5") ->	{"0","",""};
lookup1("11C6") ->	{"0","",""};
lookup1("11C7") ->	{"0","",""};
lookup1("11C8") ->	{"0","",""};
lookup1("11C9") ->	{"0","",""};
lookup1("11CA") ->	{"0","",""};
lookup1("11CB") ->	{"0","",""};
lookup1("11CC") ->	{"0","",""};
lookup1("11CD") ->	{"0","",""};
lookup1("11CE") ->	{"0","",""};
lookup1("11CF") ->	{"0","",""};
lookup1("11D0") ->	{"0","",""};
lookup1("11D1") ->	{"0","",""};
lookup1("11D2") ->	{"0","",""};
lookup1("11D3") ->	{"0","",""};
lookup1("11D4") ->	{"0","",""};
lookup1("11D5") ->	{"0","",""};
lookup1("11D6") ->	{"0","",""};
lookup1("11D7") ->	{"0","",""};
lookup1("11D8") ->	{"0","",""};
lookup1("11D9") ->	{"0","",""};
lookup1("11DA") ->	{"0","",""};
lookup1("11DB") ->	{"0","",""};
lookup1("11DC") ->	{"0","",""};
lookup1("11DD") ->	{"0","",""};
lookup1("11DE") ->	{"0","",""};
lookup1("11DF") ->	{"0","",""};
lookup1("11E0") ->	{"0","",""};
lookup1("11E1") ->	{"0","",""};
lookup1("11E2") ->	{"0","",""};
lookup1("11E3") ->	{"0","",""};
lookup1("11E4") ->	{"0","",""};
lookup1("11E5") ->	{"0","",""};
lookup1("11E6") ->	{"0","",""};
lookup1("11E7") ->	{"0","",""};
lookup1("11E8") ->	{"0","",""};
lookup1("11E9") ->	{"0","",""};
lookup1("11EA") ->	{"0","",""};
lookup1("11EB") ->	{"0","",""};
lookup1("11EC") ->	{"0","",""};
lookup1("11ED") ->	{"0","",""};
lookup1("11EE") ->	{"0","",""};
lookup1("11EF") ->	{"0","",""};
lookup1("11F0") ->	{"0","",""};
lookup1("11F1") ->	{"0","",""};
lookup1("11F2") ->	{"0","",""};
lookup1("11F3") ->	{"0","",""};
lookup1("11F4") ->	{"0","",""};
lookup1("11F5") ->	{"0","",""};
lookup1("11F6") ->	{"0","",""};
lookup1("11F7") ->	{"0","",""};
lookup1("11F8") ->	{"0","",""};
lookup1("11F9") ->	{"0","",""};
lookup1("11FA") ->	{"0","",""};
lookup1("11FB") ->	{"0","",""};
lookup1("11FC") ->	{"0","",""};
lookup1("11FD") ->	{"0","",""};
lookup1("11FE") ->	{"0","",""};
lookup1("11FF") ->	{"0","",""};
lookup1("1200") ->	{"0","",""};
lookup1("12000") ->	{"0","",""};
lookup1("12001") ->	{"0","",""};
lookup1("12002") ->	{"0","",""};
lookup1("12003") ->	{"0","",""};
lookup1("12004") ->	{"0","",""};
lookup1("12005") ->	{"0","",""};
lookup1("12006") ->	{"0","",""};
lookup1("12007") ->	{"0","",""};
lookup1("12008") ->	{"0","",""};
lookup1("12009") ->	{"0","",""};
lookup1("1200A") ->	{"0","",""};
lookup1("1200B") ->	{"0","",""};
lookup1("1200C") ->	{"0","",""};
lookup1("1200D") ->	{"0","",""};
lookup1("1200E") ->	{"0","",""};
lookup1("1200F") ->	{"0","",""};
lookup1("1201") ->	{"0","",""};
lookup1("12010") ->	{"0","",""};
lookup1("12011") ->	{"0","",""};
lookup1("12012") ->	{"0","",""};
lookup1("12013") ->	{"0","",""};
lookup1("12014") ->	{"0","",""};
lookup1("12015") ->	{"0","",""};
lookup1("12016") ->	{"0","",""};
lookup1("12017") ->	{"0","",""};
lookup1("12018") ->	{"0","",""};
lookup1("12019") ->	{"0","",""};
lookup1("1201A") ->	{"0","",""};
lookup1("1201B") ->	{"0","",""};
lookup1("1201C") ->	{"0","",""};
lookup1("1201D") ->	{"0","",""};
lookup1("1201E") ->	{"0","",""};
lookup1("1201F") ->	{"0","",""};
lookup1("1202") ->	{"0","",""};
lookup1("12020") ->	{"0","",""};
lookup1("12021") ->	{"0","",""};
lookup1("12022") ->	{"0","",""};
lookup1("12023") ->	{"0","",""};
lookup1("12024") ->	{"0","",""};
lookup1("12025") ->	{"0","",""};
lookup1("12026") ->	{"0","",""};
lookup1("12027") ->	{"0","",""};
lookup1("12028") ->	{"0","",""};
lookup1("12029") ->	{"0","",""};
lookup1("1202A") ->	{"0","",""};
lookup1("1202B") ->	{"0","",""};
lookup1("1202C") ->	{"0","",""};
lookup1("1202D") ->	{"0","",""};
lookup1("1202E") ->	{"0","",""};
lookup1("1202F") ->	{"0","",""};
lookup1("1203") ->	{"0","",""};
lookup1("12030") ->	{"0","",""};
lookup1("12031") ->	{"0","",""};
lookup1("12032") ->	{"0","",""};
lookup1("12033") ->	{"0","",""};
lookup1("12034") ->	{"0","",""};
lookup1("12035") ->	{"0","",""};
lookup1("12036") ->	{"0","",""};
lookup1("12037") ->	{"0","",""};
lookup1("12038") ->	{"0","",""};
lookup1("12039") ->	{"0","",""};
lookup1("1203A") ->	{"0","",""};
lookup1("1203B") ->	{"0","",""};
lookup1("1203C") ->	{"0","",""};
lookup1("1203D") ->	{"0","",""};
lookup1("1203E") ->	{"0","",""};
lookup1("1203F") ->	{"0","",""};
lookup1("1204") ->	{"0","",""};
lookup1("12040") ->	{"0","",""};
lookup1("12041") ->	{"0","",""};
lookup1("12042") ->	{"0","",""};
lookup1("12043") ->	{"0","",""};
lookup1("12044") ->	{"0","",""};
lookup1("12045") ->	{"0","",""};
lookup1("12046") ->	{"0","",""};
lookup1("12047") ->	{"0","",""};
lookup1("12048") ->	{"0","",""};
lookup1("12049") ->	{"0","",""};
lookup1("1204A") ->	{"0","",""};
lookup1("1204B") ->	{"0","",""};
lookup1("1204C") ->	{"0","",""};
lookup1("1204D") ->	{"0","",""};
lookup1("1204E") ->	{"0","",""};
lookup1("1204F") ->	{"0","",""};
lookup1("1205") ->	{"0","",""};
lookup1("12050") ->	{"0","",""};
lookup1("12051") ->	{"0","",""};
lookup1("12052") ->	{"0","",""};
lookup1("12053") ->	{"0","",""};
lookup1("12054") ->	{"0","",""};
lookup1("12055") ->	{"0","",""};
lookup1("12056") ->	{"0","",""};
lookup1("12057") ->	{"0","",""};
lookup1("12058") ->	{"0","",""};
lookup1("12059") ->	{"0","",""};
lookup1("1205A") ->	{"0","",""};
lookup1("1205B") ->	{"0","",""};
lookup1("1205C") ->	{"0","",""};
lookup1("1205D") ->	{"0","",""};
lookup1("1205E") ->	{"0","",""};
lookup1("1205F") ->	{"0","",""};
lookup1("1206") ->	{"0","",""};
lookup1("12060") ->	{"0","",""};
lookup1("12061") ->	{"0","",""};
lookup1("12062") ->	{"0","",""};
lookup1("12063") ->	{"0","",""};
lookup1("12064") ->	{"0","",""};
lookup1("12065") ->	{"0","",""};
lookup1("12066") ->	{"0","",""};
lookup1("12067") ->	{"0","",""};
lookup1("12068") ->	{"0","",""};
lookup1("12069") ->	{"0","",""};
lookup1("1206A") ->	{"0","",""};
lookup1("1206B") ->	{"0","",""};
lookup1("1206C") ->	{"0","",""};
lookup1("1206D") ->	{"0","",""};
lookup1("1206E") ->	{"0","",""};
lookup1("1206F") ->	{"0","",""};
lookup1("1207") ->	{"0","",""};
lookup1("12070") ->	{"0","",""};
lookup1("12071") ->	{"0","",""};
lookup1("12072") ->	{"0","",""};
lookup1("12073") ->	{"0","",""};
lookup1("12074") ->	{"0","",""};
lookup1("12075") ->	{"0","",""};
lookup1("12076") ->	{"0","",""};
lookup1("12077") ->	{"0","",""};
lookup1("12078") ->	{"0","",""};
lookup1("12079") ->	{"0","",""};
lookup1("1207A") ->	{"0","",""};
lookup1("1207B") ->	{"0","",""};
lookup1("1207C") ->	{"0","",""};
lookup1("1207D") ->	{"0","",""};
lookup1("1207E") ->	{"0","",""};
lookup1("1207F") ->	{"0","",""};
lookup1("1208") ->	{"0","",""};
lookup1("12080") ->	{"0","",""};
lookup1("12081") ->	{"0","",""};
lookup1("12082") ->	{"0","",""};
lookup1("12083") ->	{"0","",""};
lookup1("12084") ->	{"0","",""};
lookup1("12085") ->	{"0","",""};
lookup1("12086") ->	{"0","",""};
lookup1("12087") ->	{"0","",""};
lookup1("12088") ->	{"0","",""};
lookup1("12089") ->	{"0","",""};
lookup1("1208A") ->	{"0","",""};
lookup1("1208B") ->	{"0","",""};
lookup1("1208C") ->	{"0","",""};
lookup1("1208D") ->	{"0","",""};
lookup1("1208E") ->	{"0","",""};
lookup1("1208F") ->	{"0","",""};
lookup1("1209") ->	{"0","",""};
lookup1("12090") ->	{"0","",""};
lookup1("12091") ->	{"0","",""};
lookup1("12092") ->	{"0","",""};
lookup1("12093") ->	{"0","",""};
lookup1("12094") ->	{"0","",""};
lookup1("12095") ->	{"0","",""};
lookup1("12096") ->	{"0","",""};
lookup1("12097") ->	{"0","",""};
lookup1("12098") ->	{"0","",""};
lookup1("12099") ->	{"0","",""};
lookup1("1209A") ->	{"0","",""};
lookup1("1209B") ->	{"0","",""};
lookup1("1209C") ->	{"0","",""};
lookup1("1209D") ->	{"0","",""};
lookup1("1209E") ->	{"0","",""};
lookup1("1209F") ->	{"0","",""};
lookup1("120A") ->	{"0","",""};
lookup1("120A0") ->	{"0","",""};
lookup1("120A1") ->	{"0","",""};
lookup1("120A2") ->	{"0","",""};
lookup1("120A3") ->	{"0","",""};
lookup1("120A4") ->	{"0","",""};
lookup1("120A5") ->	{"0","",""};
lookup1("120A6") ->	{"0","",""};
lookup1("120A7") ->	{"0","",""};
lookup1("120A8") ->	{"0","",""};
lookup1("120A9") ->	{"0","",""};
lookup1("120AA") ->	{"0","",""};
lookup1("120AB") ->	{"0","",""};
lookup1("120AC") ->	{"0","",""};
lookup1("120AD") ->	{"0","",""};
lookup1("120AE") ->	{"0","",""};
lookup1("120AF") ->	{"0","",""};
lookup1("120B") ->	{"0","",""};
lookup1("120B0") ->	{"0","",""};
lookup1("120B1") ->	{"0","",""};
lookup1("120B2") ->	{"0","",""};
lookup1("120B3") ->	{"0","",""};
lookup1("120B4") ->	{"0","",""};
lookup1("120B5") ->	{"0","",""};
lookup1("120B6") ->	{"0","",""};
lookup1("120B7") ->	{"0","",""};
lookup1("120B8") ->	{"0","",""};
lookup1("120B9") ->	{"0","",""};
lookup1("120BA") ->	{"0","",""};
lookup1("120BB") ->	{"0","",""};
lookup1("120BC") ->	{"0","",""};
lookup1("120BD") ->	{"0","",""};
lookup1("120BE") ->	{"0","",""};
lookup1("120BF") ->	{"0","",""};
lookup1("120C") ->	{"0","",""};
lookup1("120C0") ->	{"0","",""};
lookup1("120C1") ->	{"0","",""};
lookup1("120C2") ->	{"0","",""};
lookup1("120C3") ->	{"0","",""};
lookup1("120C4") ->	{"0","",""};
lookup1("120C5") ->	{"0","",""};
lookup1("120C6") ->	{"0","",""};
lookup1("120C7") ->	{"0","",""};
lookup1("120C8") ->	{"0","",""};
lookup1("120C9") ->	{"0","",""};
lookup1("120CA") ->	{"0","",""};
lookup1("120CB") ->	{"0","",""};
lookup1("120CC") ->	{"0","",""};
lookup1("120CD") ->	{"0","",""};
lookup1("120CE") ->	{"0","",""};
lookup1("120CF") ->	{"0","",""};
lookup1("120D") ->	{"0","",""};
lookup1("120D0") ->	{"0","",""};
lookup1("120D1") ->	{"0","",""};
lookup1("120D2") ->	{"0","",""};
lookup1("120D3") ->	{"0","",""};
lookup1("120D4") ->	{"0","",""};
lookup1("120D5") ->	{"0","",""};
lookup1("120D6") ->	{"0","",""};
lookup1("120D7") ->	{"0","",""};
lookup1("120D8") ->	{"0","",""};
lookup1("120D9") ->	{"0","",""};
lookup1("120DA") ->	{"0","",""};
lookup1("120DB") ->	{"0","",""};
lookup1("120DC") ->	{"0","",""};
lookup1("120DD") ->	{"0","",""};
lookup1("120DE") ->	{"0","",""};
lookup1("120DF") ->	{"0","",""};
lookup1("120E") ->	{"0","",""};
lookup1("120E0") ->	{"0","",""};
lookup1("120E1") ->	{"0","",""};
lookup1("120E2") ->	{"0","",""};
lookup1("120E3") ->	{"0","",""};
lookup1("120E4") ->	{"0","",""};
lookup1("120E5") ->	{"0","",""};
lookup1("120E6") ->	{"0","",""};
lookup1("120E7") ->	{"0","",""};
lookup1("120E8") ->	{"0","",""};
lookup1("120E9") ->	{"0","",""};
lookup1("120EA") ->	{"0","",""};
lookup1("120EB") ->	{"0","",""};
lookup1("120EC") ->	{"0","",""};
lookup1("120ED") ->	{"0","",""};
lookup1("120EE") ->	{"0","",""};
lookup1("120EF") ->	{"0","",""};
lookup1("120F") ->	{"0","",""};
lookup1("120F0") ->	{"0","",""};
lookup1("120F1") ->	{"0","",""};
lookup1("120F2") ->	{"0","",""};
lookup1("120F3") ->	{"0","",""};
lookup1("120F4") ->	{"0","",""};
lookup1("120F5") ->	{"0","",""};
lookup1("120F6") ->	{"0","",""};
lookup1("120F7") ->	{"0","",""};
lookup1("120F8") ->	{"0","",""};
lookup1("120F9") ->	{"0","",""};
lookup1("120FA") ->	{"0","",""};
lookup1("120FB") ->	{"0","",""};
lookup1("120FC") ->	{"0","",""};
lookup1("120FD") ->	{"0","",""};
lookup1("120FE") ->	{"0","",""};
lookup1("120FF") ->	{"0","",""};
lookup1("1210") ->	{"0","",""};
lookup1("12100") ->	{"0","",""};
lookup1("12101") ->	{"0","",""};
lookup1("12102") ->	{"0","",""};
lookup1("12103") ->	{"0","",""};
lookup1("12104") ->	{"0","",""};
lookup1("12105") ->	{"0","",""};
lookup1("12106") ->	{"0","",""};
lookup1("12107") ->	{"0","",""};
lookup1("12108") ->	{"0","",""};
lookup1("12109") ->	{"0","",""};
lookup1("1210A") ->	{"0","",""};
lookup1("1210B") ->	{"0","",""};
lookup1("1210C") ->	{"0","",""};
lookup1("1210D") ->	{"0","",""};
lookup1("1210E") ->	{"0","",""};
lookup1("1210F") ->	{"0","",""};
lookup1("1211") ->	{"0","",""};
lookup1("12110") ->	{"0","",""};
lookup1("12111") ->	{"0","",""};
lookup1("12112") ->	{"0","",""};
lookup1("12113") ->	{"0","",""};
lookup1("12114") ->	{"0","",""};
lookup1("12115") ->	{"0","",""};
lookup1("12116") ->	{"0","",""};
lookup1("12117") ->	{"0","",""};
lookup1("12118") ->	{"0","",""};
lookup1("12119") ->	{"0","",""};
lookup1("1211A") ->	{"0","",""};
lookup1("1211B") ->	{"0","",""};
lookup1("1211C") ->	{"0","",""};
lookup1("1211D") ->	{"0","",""};
lookup1("1211E") ->	{"0","",""};
lookup1("1211F") ->	{"0","",""};
lookup1("1212") ->	{"0","",""};
lookup1("12120") ->	{"0","",""};
lookup1("12121") ->	{"0","",""};
lookup1("12122") ->	{"0","",""};
lookup1("12123") ->	{"0","",""};
lookup1("12124") ->	{"0","",""};
lookup1("12125") ->	{"0","",""};
lookup1("12126") ->	{"0","",""};
lookup1("12127") ->	{"0","",""};
lookup1("12128") ->	{"0","",""};
lookup1("12129") ->	{"0","",""};
lookup1("1212A") ->	{"0","",""};
lookup1("1212B") ->	{"0","",""};
lookup1("1212C") ->	{"0","",""};
lookup1("1212D") ->	{"0","",""};
lookup1("1212E") ->	{"0","",""};
lookup1("1212F") ->	{"0","",""};
lookup1("1213") ->	{"0","",""};
lookup1("12130") ->	{"0","",""};
lookup1("12131") ->	{"0","",""};
lookup1("12132") ->	{"0","",""};
lookup1("12133") ->	{"0","",""};
lookup1("12134") ->	{"0","",""};
lookup1("12135") ->	{"0","",""};
lookup1("12136") ->	{"0","",""};
lookup1("12137") ->	{"0","",""};
lookup1("12138") ->	{"0","",""};
lookup1("12139") ->	{"0","",""};
lookup1("1213A") ->	{"0","",""};
lookup1("1213B") ->	{"0","",""};
lookup1("1213C") ->	{"0","",""};
lookup1("1213D") ->	{"0","",""};
lookup1("1213E") ->	{"0","",""};
lookup1("1213F") ->	{"0","",""};
lookup1("1214") ->	{"0","",""};
lookup1("12140") ->	{"0","",""};
lookup1("12141") ->	{"0","",""};
lookup1("12142") ->	{"0","",""};
lookup1("12143") ->	{"0","",""};
lookup1("12144") ->	{"0","",""};
lookup1("12145") ->	{"0","",""};
lookup1("12146") ->	{"0","",""};
lookup1("12147") ->	{"0","",""};
lookup1("12148") ->	{"0","",""};
lookup1("12149") ->	{"0","",""};
lookup1("1214A") ->	{"0","",""};
lookup1("1214B") ->	{"0","",""};
lookup1("1214C") ->	{"0","",""};
lookup1("1214D") ->	{"0","",""};
lookup1("1214E") ->	{"0","",""};
lookup1("1214F") ->	{"0","",""};
lookup1("1215") ->	{"0","",""};
lookup1("12150") ->	{"0","",""};
lookup1("12151") ->	{"0","",""};
lookup1("12152") ->	{"0","",""};
lookup1("12153") ->	{"0","",""};
lookup1("12154") ->	{"0","",""};
lookup1("12155") ->	{"0","",""};
lookup1("12156") ->	{"0","",""};
lookup1("12157") ->	{"0","",""};
lookup1("12158") ->	{"0","",""};
lookup1("12159") ->	{"0","",""};
lookup1("1215A") ->	{"0","",""};
lookup1("1215B") ->	{"0","",""};
lookup1("1215C") ->	{"0","",""};
lookup1("1215D") ->	{"0","",""};
lookup1("1215E") ->	{"0","",""};
lookup1("1215F") ->	{"0","",""};
lookup1("1216") ->	{"0","",""};
lookup1("12160") ->	{"0","",""};
lookup1("12161") ->	{"0","",""};
lookup1("12162") ->	{"0","",""};
lookup1("12163") ->	{"0","",""};
lookup1("12164") ->	{"0","",""};
lookup1("12165") ->	{"0","",""};
lookup1("12166") ->	{"0","",""};
lookup1("12167") ->	{"0","",""};
lookup1("12168") ->	{"0","",""};
lookup1("12169") ->	{"0","",""};
lookup1("1216A") ->	{"0","",""};
lookup1("1216B") ->	{"0","",""};
lookup1("1216C") ->	{"0","",""};
lookup1("1216D") ->	{"0","",""};
lookup1("1216E") ->	{"0","",""};
lookup1("1216F") ->	{"0","",""};
lookup1("1217") ->	{"0","",""};
lookup1("12170") ->	{"0","",""};
lookup1("12171") ->	{"0","",""};
lookup1("12172") ->	{"0","",""};
lookup1("12173") ->	{"0","",""};
lookup1("12174") ->	{"0","",""};
lookup1("12175") ->	{"0","",""};
lookup1("12176") ->	{"0","",""};
lookup1("12177") ->	{"0","",""};
lookup1("12178") ->	{"0","",""};
lookup1("12179") ->	{"0","",""};
lookup1("1217A") ->	{"0","",""};
lookup1("1217B") ->	{"0","",""};
lookup1("1217C") ->	{"0","",""};
lookup1("1217D") ->	{"0","",""};
lookup1("1217E") ->	{"0","",""};
lookup1("1217F") ->	{"0","",""};
lookup1("1218") ->	{"0","",""};
lookup1("12180") ->	{"0","",""};
lookup1("12181") ->	{"0","",""};
lookup1("12182") ->	{"0","",""};
lookup1("12183") ->	{"0","",""};
lookup1("12184") ->	{"0","",""};
lookup1("12185") ->	{"0","",""};
lookup1("12186") ->	{"0","",""};
lookup1("12187") ->	{"0","",""};
lookup1("12188") ->	{"0","",""};
lookup1("12189") ->	{"0","",""};
lookup1("1218A") ->	{"0","",""};
lookup1("1218B") ->	{"0","",""};
lookup1("1218C") ->	{"0","",""};
lookup1("1218D") ->	{"0","",""};
lookup1("1218E") ->	{"0","",""};
lookup1("1218F") ->	{"0","",""};
lookup1("1219") ->	{"0","",""};
lookup1("12190") ->	{"0","",""};
lookup1("12191") ->	{"0","",""};
lookup1("12192") ->	{"0","",""};
lookup1("12193") ->	{"0","",""};
lookup1("12194") ->	{"0","",""};
lookup1("12195") ->	{"0","",""};
lookup1("12196") ->	{"0","",""};
lookup1("12197") ->	{"0","",""};
lookup1("12198") ->	{"0","",""};
lookup1("12199") ->	{"0","",""};
lookup1("1219A") ->	{"0","",""};
lookup1("1219B") ->	{"0","",""};
lookup1("1219C") ->	{"0","",""};
lookup1("1219D") ->	{"0","",""};
lookup1("1219E") ->	{"0","",""};
lookup1("1219F") ->	{"0","",""};
lookup1("121A") ->	{"0","",""};
lookup1("121A0") ->	{"0","",""};
lookup1("121A1") ->	{"0","",""};
lookup1("121A2") ->	{"0","",""};
lookup1("121A3") ->	{"0","",""};
lookup1("121A4") ->	{"0","",""};
lookup1("121A5") ->	{"0","",""};
lookup1("121A6") ->	{"0","",""};
lookup1("121A7") ->	{"0","",""};
lookup1("121A8") ->	{"0","",""};
lookup1("121A9") ->	{"0","",""};
lookup1("121AA") ->	{"0","",""};
lookup1("121AB") ->	{"0","",""};
lookup1("121AC") ->	{"0","",""};
lookup1("121AD") ->	{"0","",""};
lookup1("121AE") ->	{"0","",""};
lookup1("121AF") ->	{"0","",""};
lookup1("121B") ->	{"0","",""};
lookup1("121B0") ->	{"0","",""};
lookup1("121B1") ->	{"0","",""};
lookup1("121B2") ->	{"0","",""};
lookup1("121B3") ->	{"0","",""};
lookup1("121B4") ->	{"0","",""};
lookup1("121B5") ->	{"0","",""};
lookup1("121B6") ->	{"0","",""};
lookup1("121B7") ->	{"0","",""};
lookup1("121B8") ->	{"0","",""};
lookup1("121B9") ->	{"0","",""};
lookup1("121BA") ->	{"0","",""};
lookup1("121BB") ->	{"0","",""};
lookup1("121BC") ->	{"0","",""};
lookup1("121BD") ->	{"0","",""};
lookup1("121BE") ->	{"0","",""};
lookup1("121BF") ->	{"0","",""};
lookup1("121C") ->	{"0","",""};
lookup1("121C0") ->	{"0","",""};
lookup1("121C1") ->	{"0","",""};
lookup1("121C2") ->	{"0","",""};
lookup1("121C3") ->	{"0","",""};
lookup1("121C4") ->	{"0","",""};
lookup1("121C5") ->	{"0","",""};
lookup1("121C6") ->	{"0","",""};
lookup1("121C7") ->	{"0","",""};
lookup1("121C8") ->	{"0","",""};
lookup1("121C9") ->	{"0","",""};
lookup1("121CA") ->	{"0","",""};
lookup1("121CB") ->	{"0","",""};
lookup1("121CC") ->	{"0","",""};
lookup1("121CD") ->	{"0","",""};
lookup1("121CE") ->	{"0","",""};
lookup1("121CF") ->	{"0","",""};
lookup1("121D") ->	{"0","",""};
lookup1("121D0") ->	{"0","",""};
lookup1("121D1") ->	{"0","",""};
lookup1("121D2") ->	{"0","",""};
lookup1("121D3") ->	{"0","",""};
lookup1("121D4") ->	{"0","",""};
lookup1("121D5") ->	{"0","",""};
lookup1("121D6") ->	{"0","",""};
lookup1("121D7") ->	{"0","",""};
lookup1("121D8") ->	{"0","",""};
lookup1("121D9") ->	{"0","",""};
lookup1("121DA") ->	{"0","",""};
lookup1("121DB") ->	{"0","",""};
lookup1("121DC") ->	{"0","",""};
lookup1("121DD") ->	{"0","",""};
lookup1("121DE") ->	{"0","",""};
lookup1("121DF") ->	{"0","",""};
lookup1("121E") ->	{"0","",""};
lookup1("121E0") ->	{"0","",""};
lookup1("121E1") ->	{"0","",""};
lookup1("121E2") ->	{"0","",""};
lookup1("121E3") ->	{"0","",""};
lookup1("121E4") ->	{"0","",""};
lookup1("121E5") ->	{"0","",""};
lookup1("121E6") ->	{"0","",""};
lookup1("121E7") ->	{"0","",""};
lookup1("121E8") ->	{"0","",""};
lookup1("121E9") ->	{"0","",""};
lookup1("121EA") ->	{"0","",""};
lookup1("121EB") ->	{"0","",""};
lookup1("121EC") ->	{"0","",""};
lookup1("121ED") ->	{"0","",""};
lookup1("121EE") ->	{"0","",""};
lookup1("121EF") ->	{"0","",""};
lookup1("121F") ->	{"0","",""};
lookup1("121F0") ->	{"0","",""};
lookup1("121F1") ->	{"0","",""};
lookup1("121F2") ->	{"0","",""};
lookup1("121F3") ->	{"0","",""};
lookup1("121F4") ->	{"0","",""};
lookup1("121F5") ->	{"0","",""};
lookup1("121F6") ->	{"0","",""};
lookup1("121F7") ->	{"0","",""};
lookup1("121F8") ->	{"0","",""};
lookup1("121F9") ->	{"0","",""};
lookup1("121FA") ->	{"0","",""};
lookup1("121FB") ->	{"0","",""};
lookup1("121FC") ->	{"0","",""};
lookup1("121FD") ->	{"0","",""};
lookup1("121FE") ->	{"0","",""};
lookup1("121FF") ->	{"0","",""};
lookup1("1220") ->	{"0","",""};
lookup1("12200") ->	{"0","",""};
lookup1("12201") ->	{"0","",""};
lookup1("12202") ->	{"0","",""};
lookup1("12203") ->	{"0","",""};
lookup1("12204") ->	{"0","",""};
lookup1("12205") ->	{"0","",""};
lookup1("12206") ->	{"0","",""};
lookup1("12207") ->	{"0","",""};
lookup1("12208") ->	{"0","",""};
lookup1("12209") ->	{"0","",""};
lookup1("1220A") ->	{"0","",""};
lookup1("1220B") ->	{"0","",""};
lookup1("1220C") ->	{"0","",""};
lookup1("1220D") ->	{"0","",""};
lookup1("1220E") ->	{"0","",""};
lookup1("1220F") ->	{"0","",""};
lookup1("1221") ->	{"0","",""};
lookup1("12210") ->	{"0","",""};
lookup1("12211") ->	{"0","",""};
lookup1("12212") ->	{"0","",""};
lookup1("12213") ->	{"0","",""};
lookup1("12214") ->	{"0","",""};
lookup1("12215") ->	{"0","",""};
lookup1("12216") ->	{"0","",""};
lookup1("12217") ->	{"0","",""};
lookup1("12218") ->	{"0","",""};
lookup1("12219") ->	{"0","",""};
lookup1("1221A") ->	{"0","",""};
lookup1("1221B") ->	{"0","",""};
lookup1("1221C") ->	{"0","",""};
lookup1("1221D") ->	{"0","",""};
lookup1("1221E") ->	{"0","",""};
lookup1("1221F") ->	{"0","",""};
lookup1("1222") ->	{"0","",""};
lookup1("12220") ->	{"0","",""};
lookup1("12221") ->	{"0","",""};
lookup1("12222") ->	{"0","",""};
lookup1("12223") ->	{"0","",""};
lookup1("12224") ->	{"0","",""};
lookup1("12225") ->	{"0","",""};
lookup1("12226") ->	{"0","",""};
lookup1("12227") ->	{"0","",""};
lookup1("12228") ->	{"0","",""};
lookup1("12229") ->	{"0","",""};
lookup1("1222A") ->	{"0","",""};
lookup1("1222B") ->	{"0","",""};
lookup1("1222C") ->	{"0","",""};
lookup1("1222D") ->	{"0","",""};
lookup1("1222E") ->	{"0","",""};
lookup1("1222F") ->	{"0","",""};
lookup1("1223") ->	{"0","",""};
lookup1("12230") ->	{"0","",""};
lookup1("12231") ->	{"0","",""};
lookup1("12232") ->	{"0","",""};
lookup1("12233") ->	{"0","",""};
lookup1("12234") ->	{"0","",""};
lookup1("12235") ->	{"0","",""};
lookup1("12236") ->	{"0","",""};
lookup1("12237") ->	{"0","",""};
lookup1("12238") ->	{"0","",""};
lookup1("12239") ->	{"0","",""};
lookup1("1223A") ->	{"0","",""};
lookup1("1223B") ->	{"0","",""};
lookup1("1223C") ->	{"0","",""};
lookup1("1223D") ->	{"0","",""};
lookup1("1223E") ->	{"0","",""};
lookup1("1223F") ->	{"0","",""};
lookup1("1224") ->	{"0","",""};
lookup1("12240") ->	{"0","",""};
lookup1("12241") ->	{"0","",""};
lookup1("12242") ->	{"0","",""};
lookup1("12243") ->	{"0","",""};
lookup1("12244") ->	{"0","",""};
lookup1("12245") ->	{"0","",""};
lookup1("12246") ->	{"0","",""};
lookup1("12247") ->	{"0","",""};
lookup1("12248") ->	{"0","",""};
lookup1("12249") ->	{"0","",""};
lookup1("1224A") ->	{"0","",""};
lookup1("1224B") ->	{"0","",""};
lookup1("1224C") ->	{"0","",""};
lookup1("1224D") ->	{"0","",""};
lookup1("1224E") ->	{"0","",""};
lookup1("1224F") ->	{"0","",""};
lookup1("1225") ->	{"0","",""};
lookup1("12250") ->	{"0","",""};
lookup1("12251") ->	{"0","",""};
lookup1("12252") ->	{"0","",""};
lookup1("12253") ->	{"0","",""};
lookup1("12254") ->	{"0","",""};
lookup1("12255") ->	{"0","",""};
lookup1("12256") ->	{"0","",""};
lookup1("12257") ->	{"0","",""};
lookup1("12258") ->	{"0","",""};
lookup1("12259") ->	{"0","",""};
lookup1("1225A") ->	{"0","",""};
lookup1("1225B") ->	{"0","",""};
lookup1("1225C") ->	{"0","",""};
lookup1("1225D") ->	{"0","",""};
lookup1("1225E") ->	{"0","",""};
lookup1("1225F") ->	{"0","",""};
lookup1("1226") ->	{"0","",""};
lookup1("12260") ->	{"0","",""};
lookup1("12261") ->	{"0","",""};
lookup1("12262") ->	{"0","",""};
lookup1("12263") ->	{"0","",""};
lookup1("12264") ->	{"0","",""};
lookup1("12265") ->	{"0","",""};
lookup1("12266") ->	{"0","",""};
lookup1("12267") ->	{"0","",""};
lookup1("12268") ->	{"0","",""};
lookup1("12269") ->	{"0","",""};
lookup1("1226A") ->	{"0","",""};
lookup1("1226B") ->	{"0","",""};
lookup1("1226C") ->	{"0","",""};
lookup1("1226D") ->	{"0","",""};
lookup1("1226E") ->	{"0","",""};
lookup1("1226F") ->	{"0","",""};
lookup1("1227") ->	{"0","",""};
lookup1("12270") ->	{"0","",""};
lookup1("12271") ->	{"0","",""};
lookup1("12272") ->	{"0","",""};
lookup1("12273") ->	{"0","",""};
lookup1("12274") ->	{"0","",""};
lookup1("12275") ->	{"0","",""};
lookup1("12276") ->	{"0","",""};
lookup1("12277") ->	{"0","",""};
lookup1("12278") ->	{"0","",""};
lookup1("12279") ->	{"0","",""};
lookup1("1227A") ->	{"0","",""};
lookup1("1227B") ->	{"0","",""};
lookup1("1227C") ->	{"0","",""};
lookup1("1227D") ->	{"0","",""};
lookup1("1227E") ->	{"0","",""};
lookup1("1227F") ->	{"0","",""};
lookup1("1228") ->	{"0","",""};
lookup1("12280") ->	{"0","",""};
lookup1("12281") ->	{"0","",""};
lookup1("12282") ->	{"0","",""};
lookup1("12283") ->	{"0","",""};
lookup1("12284") ->	{"0","",""};
lookup1("12285") ->	{"0","",""};
lookup1("12286") ->	{"0","",""};
lookup1("12287") ->	{"0","",""};
lookup1("12288") ->	{"0","",""};
lookup1("12289") ->	{"0","",""};
lookup1("1228A") ->	{"0","",""};
lookup1("1228B") ->	{"0","",""};
lookup1("1228C") ->	{"0","",""};
lookup1("1228D") ->	{"0","",""};
lookup1("1228E") ->	{"0","",""};
lookup1("1228F") ->	{"0","",""};
lookup1("1229") ->	{"0","",""};
lookup1("12290") ->	{"0","",""};
lookup1("12291") ->	{"0","",""};
lookup1("12292") ->	{"0","",""};
lookup1("12293") ->	{"0","",""};
lookup1("12294") ->	{"0","",""};
lookup1("12295") ->	{"0","",""};
lookup1("12296") ->	{"0","",""};
lookup1("12297") ->	{"0","",""};
lookup1("12298") ->	{"0","",""};
lookup1("12299") ->	{"0","",""};
lookup1("1229A") ->	{"0","",""};
lookup1("1229B") ->	{"0","",""};
lookup1("1229C") ->	{"0","",""};
lookup1("1229D") ->	{"0","",""};
lookup1("1229E") ->	{"0","",""};
lookup1("1229F") ->	{"0","",""};
lookup1("122A") ->	{"0","",""};
lookup1("122A0") ->	{"0","",""};
lookup1("122A1") ->	{"0","",""};
lookup1("122A2") ->	{"0","",""};
lookup1("122A3") ->	{"0","",""};
lookup1("122A4") ->	{"0","",""};
lookup1("122A5") ->	{"0","",""};
lookup1("122A6") ->	{"0","",""};
lookup1("122A7") ->	{"0","",""};
lookup1("122A8") ->	{"0","",""};
lookup1("122A9") ->	{"0","",""};
lookup1("122AA") ->	{"0","",""};
lookup1("122AB") ->	{"0","",""};
lookup1("122AC") ->	{"0","",""};
lookup1("122AD") ->	{"0","",""};
lookup1("122AE") ->	{"0","",""};
lookup1("122AF") ->	{"0","",""};
lookup1("122B") ->	{"0","",""};
lookup1("122B0") ->	{"0","",""};
lookup1("122B1") ->	{"0","",""};
lookup1("122B2") ->	{"0","",""};
lookup1("122B3") ->	{"0","",""};
lookup1("122B4") ->	{"0","",""};
lookup1("122B5") ->	{"0","",""};
lookup1("122B6") ->	{"0","",""};
lookup1("122B7") ->	{"0","",""};
lookup1("122B8") ->	{"0","",""};
lookup1("122B9") ->	{"0","",""};
lookup1("122BA") ->	{"0","",""};
lookup1("122BB") ->	{"0","",""};
lookup1("122BC") ->	{"0","",""};
lookup1("122BD") ->	{"0","",""};
lookup1("122BE") ->	{"0","",""};
lookup1("122BF") ->	{"0","",""};
lookup1("122C") ->	{"0","",""};
lookup1("122C0") ->	{"0","",""};
lookup1("122C1") ->	{"0","",""};
lookup1("122C2") ->	{"0","",""};
lookup1("122C3") ->	{"0","",""};
lookup1("122C4") ->	{"0","",""};
lookup1("122C5") ->	{"0","",""};
lookup1("122C6") ->	{"0","",""};
lookup1("122C7") ->	{"0","",""};
lookup1("122C8") ->	{"0","",""};
lookup1("122C9") ->	{"0","",""};
lookup1("122CA") ->	{"0","",""};
lookup1("122CB") ->	{"0","",""};
lookup1("122CC") ->	{"0","",""};
lookup1("122CD") ->	{"0","",""};
lookup1("122CE") ->	{"0","",""};
lookup1("122CF") ->	{"0","",""};
lookup1("122D") ->	{"0","",""};
lookup1("122D0") ->	{"0","",""};
lookup1("122D1") ->	{"0","",""};
lookup1("122D2") ->	{"0","",""};
lookup1("122D3") ->	{"0","",""};
lookup1("122D4") ->	{"0","",""};
lookup1("122D5") ->	{"0","",""};
lookup1("122D6") ->	{"0","",""};
lookup1("122D7") ->	{"0","",""};
lookup1("122D8") ->	{"0","",""};
lookup1("122D9") ->	{"0","",""};
lookup1("122DA") ->	{"0","",""};
lookup1("122DB") ->	{"0","",""};
lookup1("122DC") ->	{"0","",""};
lookup1("122DD") ->	{"0","",""};
lookup1("122DE") ->	{"0","",""};
lookup1("122DF") ->	{"0","",""};
lookup1("122E") ->	{"0","",""};
lookup1("122E0") ->	{"0","",""};
lookup1("122E1") ->	{"0","",""};
lookup1("122E2") ->	{"0","",""};
lookup1("122E3") ->	{"0","",""};
lookup1("122E4") ->	{"0","",""};
lookup1("122E5") ->	{"0","",""};
lookup1("122E6") ->	{"0","",""};
lookup1("122E7") ->	{"0","",""};
lookup1("122E8") ->	{"0","",""};
lookup1("122E9") ->	{"0","",""};
lookup1("122EA") ->	{"0","",""};
lookup1("122EB") ->	{"0","",""};
lookup1("122EC") ->	{"0","",""};
lookup1("122ED") ->	{"0","",""};
lookup1("122EE") ->	{"0","",""};
lookup1("122EF") ->	{"0","",""};
lookup1("122F") ->	{"0","",""};
lookup1("122F0") ->	{"0","",""};
lookup1("122F1") ->	{"0","",""};
lookup1("122F2") ->	{"0","",""};
lookup1("122F3") ->	{"0","",""};
lookup1("122F4") ->	{"0","",""};
lookup1("122F5") ->	{"0","",""};
lookup1("122F6") ->	{"0","",""};
lookup1("122F7") ->	{"0","",""};
lookup1("122F8") ->	{"0","",""};
lookup1("122F9") ->	{"0","",""};
lookup1("122FA") ->	{"0","",""};
lookup1("122FB") ->	{"0","",""};
lookup1("122FC") ->	{"0","",""};
lookup1("122FD") ->	{"0","",""};
lookup1("122FE") ->	{"0","",""};
lookup1("122FF") ->	{"0","",""};
lookup1("1230") ->	{"0","",""};
lookup1("12300") ->	{"0","",""};
lookup1("12301") ->	{"0","",""};
lookup1("12302") ->	{"0","",""};
lookup1("12303") ->	{"0","",""};
lookup1("12304") ->	{"0","",""};
lookup1("12305") ->	{"0","",""};
lookup1("12306") ->	{"0","",""};
lookup1("12307") ->	{"0","",""};
lookup1("12308") ->	{"0","",""};
lookup1("12309") ->	{"0","",""};
lookup1("1230A") ->	{"0","",""};
lookup1("1230B") ->	{"0","",""};
lookup1("1230C") ->	{"0","",""};
lookup1("1230D") ->	{"0","",""};
lookup1("1230E") ->	{"0","",""};
lookup1("1230F") ->	{"0","",""};
lookup1("1231") ->	{"0","",""};
lookup1("12310") ->	{"0","",""};
lookup1("12311") ->	{"0","",""};
lookup1("12312") ->	{"0","",""};
lookup1("12313") ->	{"0","",""};
lookup1("12314") ->	{"0","",""};
lookup1("12315") ->	{"0","",""};
lookup1("12316") ->	{"0","",""};
lookup1("12317") ->	{"0","",""};
lookup1("12318") ->	{"0","",""};
lookup1("12319") ->	{"0","",""};
lookup1("1231A") ->	{"0","",""};
lookup1("1231B") ->	{"0","",""};
lookup1("1231C") ->	{"0","",""};
lookup1("1231D") ->	{"0","",""};
lookup1("1231E") ->	{"0","",""};
lookup1("1231F") ->	{"0","",""};
lookup1("1232") ->	{"0","",""};
lookup1("12320") ->	{"0","",""};
lookup1("12321") ->	{"0","",""};
lookup1("12322") ->	{"0","",""};
lookup1("12323") ->	{"0","",""};
lookup1("12324") ->	{"0","",""};
lookup1("12325") ->	{"0","",""};
lookup1("12326") ->	{"0","",""};
lookup1("12327") ->	{"0","",""};
lookup1("12328") ->	{"0","",""};
lookup1("12329") ->	{"0","",""};
lookup1("1232A") ->	{"0","",""};
lookup1("1232B") ->	{"0","",""};
lookup1("1232C") ->	{"0","",""};
lookup1("1232D") ->	{"0","",""};
lookup1("1232E") ->	{"0","",""};
lookup1("1232F") ->	{"0","",""};
lookup1("1233") ->	{"0","",""};
lookup1("12330") ->	{"0","",""};
lookup1("12331") ->	{"0","",""};
lookup1("12332") ->	{"0","",""};
lookup1("12333") ->	{"0","",""};
lookup1("12334") ->	{"0","",""};
lookup1("12335") ->	{"0","",""};
lookup1("12336") ->	{"0","",""};
lookup1("12337") ->	{"0","",""};
lookup1("12338") ->	{"0","",""};
lookup1("12339") ->	{"0","",""};
lookup1("1233A") ->	{"0","",""};
lookup1("1233B") ->	{"0","",""};
lookup1("1233C") ->	{"0","",""};
lookup1("1233D") ->	{"0","",""};
lookup1("1233E") ->	{"0","",""};
lookup1("1233F") ->	{"0","",""};
lookup1("1234") ->	{"0","",""};
lookup1("12340") ->	{"0","",""};
lookup1("12341") ->	{"0","",""};
lookup1("12342") ->	{"0","",""};
lookup1("12343") ->	{"0","",""};
lookup1("12344") ->	{"0","",""};
lookup1("12345") ->	{"0","",""};
lookup1("12346") ->	{"0","",""};
lookup1("12347") ->	{"0","",""};
lookup1("12348") ->	{"0","",""};
lookup1("12349") ->	{"0","",""};
lookup1("1234A") ->	{"0","",""};
lookup1("1234B") ->	{"0","",""};
lookup1("1234C") ->	{"0","",""};
lookup1("1234D") ->	{"0","",""};
lookup1("1234E") ->	{"0","",""};
lookup1("1234F") ->	{"0","",""};
lookup1("1235") ->	{"0","",""};
lookup1("12350") ->	{"0","",""};
lookup1("12351") ->	{"0","",""};
lookup1("12352") ->	{"0","",""};
lookup1("12353") ->	{"0","",""};
lookup1("12354") ->	{"0","",""};
lookup1("12355") ->	{"0","",""};
lookup1("12356") ->	{"0","",""};
lookup1("12357") ->	{"0","",""};
lookup1("12358") ->	{"0","",""};
lookup1("12359") ->	{"0","",""};
lookup1("1235A") ->	{"0","",""};
lookup1("1235B") ->	{"0","",""};
lookup1("1235C") ->	{"0","",""};
lookup1("1235D") ->	{"0","",""};
lookup1("1235E") ->	{"0","",""};
lookup1("1235F") ->	{"0","",""};
lookup1("1236") ->	{"0","",""};
lookup1("12360") ->	{"0","",""};
lookup1("12361") ->	{"0","",""};
lookup1("12362") ->	{"0","",""};
lookup1("12363") ->	{"0","",""};
lookup1("12364") ->	{"0","",""};
lookup1("12365") ->	{"0","",""};
lookup1("12366") ->	{"0","",""};
lookup1("12367") ->	{"0","",""};
lookup1("12368") ->	{"0","",""};
lookup1("12369") ->	{"0","",""};
lookup1("1236A") ->	{"0","",""};
lookup1("1236B") ->	{"0","",""};
lookup1("1236C") ->	{"0","",""};
lookup1("1236D") ->	{"0","",""};
lookup1("1236E") ->	{"0","",""};
lookup1("1236F") ->	{"0","",""};
lookup1("1237") ->	{"0","",""};
lookup1("12370") ->	{"0","",""};
lookup1("12371") ->	{"0","",""};
lookup1("12372") ->	{"0","",""};
lookup1("12373") ->	{"0","",""};
lookup1("12374") ->	{"0","",""};
lookup1("12375") ->	{"0","",""};
lookup1("12376") ->	{"0","",""};
lookup1("12377") ->	{"0","",""};
lookup1("12378") ->	{"0","",""};
lookup1("12379") ->	{"0","",""};
lookup1("1237A") ->	{"0","",""};
lookup1("1237B") ->	{"0","",""};
lookup1("1237C") ->	{"0","",""};
lookup1("1237D") ->	{"0","",""};
lookup1("1237E") ->	{"0","",""};
lookup1("1237F") ->	{"0","",""};
lookup1("1238") ->	{"0","",""};
lookup1("12380") ->	{"0","",""};
lookup1("12381") ->	{"0","",""};
lookup1("12382") ->	{"0","",""};
lookup1("12383") ->	{"0","",""};
lookup1("12384") ->	{"0","",""};
lookup1("12385") ->	{"0","",""};
lookup1("12386") ->	{"0","",""};
lookup1("12387") ->	{"0","",""};
lookup1("12388") ->	{"0","",""};
lookup1("12389") ->	{"0","",""};
lookup1("1238A") ->	{"0","",""};
lookup1("1238B") ->	{"0","",""};
lookup1("1238C") ->	{"0","",""};
lookup1("1238D") ->	{"0","",""};
lookup1("1238E") ->	{"0","",""};
lookup1("1238F") ->	{"0","",""};
lookup1("1239") ->	{"0","",""};
lookup1("12390") ->	{"0","",""};
lookup1("12391") ->	{"0","",""};
lookup1("12392") ->	{"0","",""};
lookup1("12393") ->	{"0","",""};
lookup1("12394") ->	{"0","",""};
lookup1("12395") ->	{"0","",""};
lookup1("12396") ->	{"0","",""};
lookup1("12397") ->	{"0","",""};
lookup1("12398") ->	{"0","",""};
lookup1("123A") ->	{"0","",""};
lookup1("123B") ->	{"0","",""};
lookup1("123C") ->	{"0","",""};
lookup1("123D") ->	{"0","",""};
lookup1("123E") ->	{"0","",""};
lookup1("123F") ->	{"0","",""};
lookup1("1240") ->	{"0","",""};
lookup1("12400") ->	{"0","",""};
lookup1("12401") ->	{"0","",""};
lookup1("12402") ->	{"0","",""};
lookup1("12403") ->	{"0","",""};
lookup1("12404") ->	{"0","",""};
lookup1("12405") ->	{"0","",""};
lookup1("12406") ->	{"0","",""};
lookup1("12407") ->	{"0","",""};
lookup1("12408") ->	{"0","",""};
lookup1("12409") ->	{"0","",""};
lookup1("1240A") ->	{"0","",""};
lookup1("1240B") ->	{"0","",""};
lookup1("1240C") ->	{"0","",""};
lookup1("1240D") ->	{"0","",""};
lookup1("1240E") ->	{"0","",""};
lookup1("1240F") ->	{"0","",""};
lookup1("1241") ->	{"0","",""};
lookup1("12410") ->	{"0","",""};
lookup1("12411") ->	{"0","",""};
lookup1("12412") ->	{"0","",""};
lookup1("12413") ->	{"0","",""};
lookup1("12414") ->	{"0","",""};
lookup1("12415") ->	{"0","",""};
lookup1("12416") ->	{"0","",""};
lookup1("12417") ->	{"0","",""};
lookup1("12418") ->	{"0","",""};
lookup1("12419") ->	{"0","",""};
lookup1("1241A") ->	{"0","",""};
lookup1("1241B") ->	{"0","",""};
lookup1("1241C") ->	{"0","",""};
lookup1("1241D") ->	{"0","",""};
lookup1("1241E") ->	{"0","",""};
lookup1("1241F") ->	{"0","",""};
lookup1("1242") ->	{"0","",""};
lookup1("12420") ->	{"0","",""};
lookup1("12421") ->	{"0","",""};
lookup1("12422") ->	{"0","",""};
lookup1("12423") ->	{"0","",""};
lookup1("12424") ->	{"0","",""};
lookup1("12425") ->	{"0","",""};
lookup1("12426") ->	{"0","",""};
lookup1("12427") ->	{"0","",""};
lookup1("12428") ->	{"0","",""};
lookup1("12429") ->	{"0","",""};
lookup1("1242A") ->	{"0","",""};
lookup1("1242B") ->	{"0","",""};
lookup1("1242C") ->	{"0","",""};
lookup1("1242D") ->	{"0","",""};
lookup1("1242E") ->	{"0","",""};
lookup1("1242F") ->	{"0","",""};
lookup1("1243") ->	{"0","",""};
lookup1("12430") ->	{"0","",""};
lookup1("12431") ->	{"0","",""};
lookup1("12432") ->	{"0","",""};
lookup1("12433") ->	{"0","",""};
lookup1("12434") ->	{"0","",""};
lookup1("12435") ->	{"0","",""};
lookup1("12436") ->	{"0","",""};
lookup1("12437") ->	{"0","",""};
lookup1("12438") ->	{"0","",""};
lookup1("12439") ->	{"0","",""};
lookup1("1243A") ->	{"0","",""};
lookup1("1243B") ->	{"0","",""};
lookup1("1243C") ->	{"0","",""};
lookup1("1243D") ->	{"0","",""};
lookup1("1243E") ->	{"0","",""};
lookup1("1243F") ->	{"0","",""};
lookup1("1244") ->	{"0","",""};
lookup1("12440") ->	{"0","",""};
lookup1("12441") ->	{"0","",""};
lookup1("12442") ->	{"0","",""};
lookup1("12443") ->	{"0","",""};
lookup1("12444") ->	{"0","",""};
lookup1("12445") ->	{"0","",""};
lookup1("12446") ->	{"0","",""};
lookup1("12447") ->	{"0","",""};
lookup1("12448") ->	{"0","",""};
lookup1("12449") ->	{"0","",""};
lookup1("1244A") ->	{"0","",""};
lookup1("1244B") ->	{"0","",""};
lookup1("1244C") ->	{"0","",""};
lookup1("1244D") ->	{"0","",""};
lookup1("1244E") ->	{"0","",""};
lookup1("1244F") ->	{"0","",""};
lookup1("1245") ->	{"0","",""};
lookup1("12450") ->	{"0","",""};
lookup1("12451") ->	{"0","",""};
lookup1("12452") ->	{"0","",""};
lookup1("12453") ->	{"0","",""};
lookup1("12454") ->	{"0","",""};
lookup1("12455") ->	{"0","",""};
lookup1("12456") ->	{"0","",""};
lookup1("12457") ->	{"0","",""};
lookup1("12458") ->	{"0","",""};
lookup1("12459") ->	{"0","",""};
lookup1("1245A") ->	{"0","",""};
lookup1("1245B") ->	{"0","",""};
lookup1("1245C") ->	{"0","",""};
lookup1("1245D") ->	{"0","",""};
lookup1("1245E") ->	{"0","",""};
lookup1("1245F") ->	{"0","",""};
lookup1("1246") ->	{"0","",""};
lookup1("12460") ->	{"0","",""};
lookup1("12461") ->	{"0","",""};
lookup1("12462") ->	{"0","",""};
lookup1("12463") ->	{"0","",""};
lookup1("12464") ->	{"0","",""};
lookup1("12465") ->	{"0","",""};
lookup1("12466") ->	{"0","",""};
lookup1("12467") ->	{"0","",""};
lookup1("12468") ->	{"0","",""};
lookup1("12469") ->	{"0","",""};
lookup1("1246A") ->	{"0","",""};
lookup1("1246B") ->	{"0","",""};
lookup1("1246C") ->	{"0","",""};
lookup1("1246D") ->	{"0","",""};
lookup1("1246E") ->	{"0","",""};
lookup1("1247") ->	{"0","",""};
lookup1("12470") ->	{"0","",""};
lookup1("12471") ->	{"0","",""};
lookup1("12472") ->	{"0","",""};
lookup1("12473") ->	{"0","",""};
lookup1("12474") ->	{"0","",""};
lookup1("1248") ->	{"0","",""};
lookup1("124A") ->	{"0","",""};
lookup1("124B") ->	{"0","",""};
lookup1("124C") ->	{"0","",""};
lookup1("124D") ->	{"0","",""};
lookup1("1250") ->	{"0","",""};
lookup1("1251") ->	{"0","",""};
lookup1("1252") ->	{"0","",""};
lookup1("1253") ->	{"0","",""};
lookup1("1254") ->	{"0","",""};
lookup1("1255") ->	{"0","",""};
lookup1("1256") ->	{"0","",""};
lookup1("1258") ->	{"0","",""};
lookup1("125A") ->	{"0","",""};
lookup1("125B") ->	{"0","",""};
lookup1("125C") ->	{"0","",""};
lookup1("125D") ->	{"0","",""};
lookup1("1260") ->	{"0","",""};
lookup1("1261") ->	{"0","",""};
lookup1("1262") ->	{"0","",""};
lookup1("1263") ->	{"0","",""};
lookup1("1264") ->	{"0","",""};
lookup1("1265") ->	{"0","",""};
lookup1("1266") ->	{"0","",""};
lookup1("1267") ->	{"0","",""};
lookup1("1268") ->	{"0","",""};
lookup1("1269") ->	{"0","",""};
lookup1("126A") ->	{"0","",""};
lookup1("126B") ->	{"0","",""};
lookup1("126C") ->	{"0","",""};
lookup1("126D") ->	{"0","",""};
lookup1("126E") ->	{"0","",""};
lookup1("126F") ->	{"0","",""};
lookup1("1270") ->	{"0","",""};
lookup1("1271") ->	{"0","",""};
lookup1("1272") ->	{"0","",""};
lookup1("1273") ->	{"0","",""};
lookup1("1274") ->	{"0","",""};
lookup1("1275") ->	{"0","",""};
lookup1("1276") ->	{"0","",""};
lookup1("1277") ->	{"0","",""};
lookup1("1278") ->	{"0","",""};
lookup1("1279") ->	{"0","",""};
lookup1("127A") ->	{"0","",""};
lookup1("127B") ->	{"0","",""};
lookup1("127C") ->	{"0","",""};
lookup1("127D") ->	{"0","",""};
lookup1("127E") ->	{"0","",""};
lookup1("127F") ->	{"0","",""};
lookup1("1280") ->	{"0","",""};
lookup1("1281") ->	{"0","",""};
lookup1("1282") ->	{"0","",""};
lookup1("1283") ->	{"0","",""};
lookup1("1284") ->	{"0","",""};
lookup1("1285") ->	{"0","",""};
lookup1("1286") ->	{"0","",""};
lookup1("1287") ->	{"0","",""};
lookup1("1288") ->	{"0","",""};
lookup1("128A") ->	{"0","",""};
lookup1("128B") ->	{"0","",""};
lookup1("128C") ->	{"0","",""};
lookup1("128D") ->	{"0","",""};
lookup1("1290") ->	{"0","",""};
lookup1("1291") ->	{"0","",""};
lookup1("1292") ->	{"0","",""};
lookup1("1293") ->	{"0","",""};
lookup1("1294") ->	{"0","",""};
lookup1("1295") ->	{"0","",""};
lookup1("1296") ->	{"0","",""};
lookup1("1297") ->	{"0","",""};
lookup1("1298") ->	{"0","",""};
lookup1("1299") ->	{"0","",""};
lookup1("129A") ->	{"0","",""};
lookup1("129B") ->	{"0","",""};
lookup1("129C") ->	{"0","",""};
lookup1("129D") ->	{"0","",""};
lookup1("129E") ->	{"0","",""};
lookup1("129F") ->	{"0","",""};
lookup1("12A0") ->	{"0","",""};
lookup1("12A1") ->	{"0","",""};
lookup1("12A2") ->	{"0","",""};
lookup1("12A3") ->	{"0","",""};
lookup1("12A4") ->	{"0","",""};
lookup1("12A5") ->	{"0","",""};
lookup1("12A6") ->	{"0","",""};
lookup1("12A7") ->	{"0","",""};
lookup1("12A8") ->	{"0","",""};
lookup1("12A9") ->	{"0","",""};
lookup1("12AA") ->	{"0","",""};
lookup1("12AB") ->	{"0","",""};
lookup1("12AC") ->	{"0","",""};
lookup1("12AD") ->	{"0","",""};
lookup1("12AE") ->	{"0","",""};
lookup1("12AF") ->	{"0","",""};
lookup1("12B0") ->	{"0","",""};
lookup1("12B2") ->	{"0","",""};
lookup1("12B3") ->	{"0","",""};
lookup1("12B4") ->	{"0","",""};
lookup1("12B5") ->	{"0","",""};
lookup1("12B8") ->	{"0","",""};
lookup1("12B9") ->	{"0","",""};
lookup1("12BA") ->	{"0","",""};
lookup1("12BB") ->	{"0","",""};
lookup1("12BC") ->	{"0","",""};
lookup1("12BD") ->	{"0","",""};
lookup1("12BE") ->	{"0","",""};
lookup1("12C0") ->	{"0","",""};
lookup1("12C2") ->	{"0","",""};
lookup1("12C3") ->	{"0","",""};
lookup1("12C4") ->	{"0","",""};
lookup1("12C5") ->	{"0","",""};
lookup1("12C8") ->	{"0","",""};
lookup1("12C9") ->	{"0","",""};
lookup1("12CA") ->	{"0","",""};
lookup1("12CB") ->	{"0","",""};
lookup1("12CC") ->	{"0","",""};
lookup1("12CD") ->	{"0","",""};
lookup1("12CE") ->	{"0","",""};
lookup1("12CF") ->	{"0","",""};
lookup1("12D0") ->	{"0","",""};
lookup1("12D1") ->	{"0","",""};
lookup1("12D2") ->	{"0","",""};
lookup1("12D3") ->	{"0","",""};
lookup1("12D4") ->	{"0","",""};
lookup1("12D5") ->	{"0","",""};
lookup1("12D6") ->	{"0","",""};
lookup1("12D8") ->	{"0","",""};
lookup1("12D9") ->	{"0","",""};
lookup1("12DA") ->	{"0","",""};
lookup1("12DB") ->	{"0","",""};
lookup1("12DC") ->	{"0","",""};
lookup1("12DD") ->	{"0","",""};
lookup1("12DE") ->	{"0","",""};
lookup1("12DF") ->	{"0","",""};
lookup1("12E0") ->	{"0","",""};
lookup1("12E1") ->	{"0","",""};
lookup1("12E2") ->	{"0","",""};
lookup1("12E3") ->	{"0","",""};
lookup1("12E4") ->	{"0","",""};
lookup1("12E5") ->	{"0","",""};
lookup1("12E6") ->	{"0","",""};
lookup1("12E7") ->	{"0","",""};
lookup1("12E8") ->	{"0","",""};
lookup1("12E9") ->	{"0","",""};
lookup1("12EA") ->	{"0","",""};
lookup1("12EB") ->	{"0","",""};
lookup1("12EC") ->	{"0","",""};
lookup1("12ED") ->	{"0","",""};
lookup1("12EE") ->	{"0","",""};
lookup1("12EF") ->	{"0","",""};
lookup1("12F0") ->	{"0","",""};
lookup1("12F1") ->	{"0","",""};
lookup1("12F2") ->	{"0","",""};
lookup1("12F3") ->	{"0","",""};
lookup1("12F4") ->	{"0","",""};
lookup1("12F5") ->	{"0","",""};
lookup1("12F6") ->	{"0","",""};
lookup1("12F7") ->	{"0","",""};
lookup1("12F8") ->	{"0","",""};
lookup1("12F9") ->	{"0","",""};
lookup1("12FA") ->	{"0","",""};
lookup1("12FB") ->	{"0","",""};
lookup1("12FC") ->	{"0","",""};
lookup1("12FD") ->	{"0","",""};
lookup1("12FE") ->	{"0","",""};
lookup1("12FF") ->	{"0","",""};
lookup1("1300") ->	{"0","",""};
lookup1("13000") ->	{"0","",""};
lookup1("13001") ->	{"0","",""};
lookup1("13002") ->	{"0","",""};
lookup1("13003") ->	{"0","",""};
lookup1("13004") ->	{"0","",""};
lookup1("13005") ->	{"0","",""};
lookup1("13006") ->	{"0","",""};
lookup1("13007") ->	{"0","",""};
lookup1("13008") ->	{"0","",""};
lookup1("13009") ->	{"0","",""};
lookup1("1300A") ->	{"0","",""};
lookup1("1300B") ->	{"0","",""};
lookup1("1300C") ->	{"0","",""};
lookup1("1300D") ->	{"0","",""};
lookup1("1300E") ->	{"0","",""};
lookup1("1300F") ->	{"0","",""};
lookup1("1301") ->	{"0","",""};
lookup1("13010") ->	{"0","",""};
lookup1("13011") ->	{"0","",""};
lookup1("13012") ->	{"0","",""};
lookup1("13013") ->	{"0","",""};
lookup1("13014") ->	{"0","",""};
lookup1("13015") ->	{"0","",""};
lookup1("13016") ->	{"0","",""};
lookup1("13017") ->	{"0","",""};
lookup1("13018") ->	{"0","",""};
lookup1("13019") ->	{"0","",""};
lookup1("1301A") ->	{"0","",""};
lookup1("1301B") ->	{"0","",""};
lookup1("1301C") ->	{"0","",""};
lookup1("1301D") ->	{"0","",""};
lookup1("1301E") ->	{"0","",""};
lookup1("1301F") ->	{"0","",""};
lookup1("1302") ->	{"0","",""};
lookup1("13020") ->	{"0","",""};
lookup1("13021") ->	{"0","",""};
lookup1("13022") ->	{"0","",""};
lookup1("13023") ->	{"0","",""};
lookup1("13024") ->	{"0","",""};
lookup1("13025") ->	{"0","",""};
lookup1("13026") ->	{"0","",""};
lookup1("13027") ->	{"0","",""};
lookup1("13028") ->	{"0","",""};
lookup1("13029") ->	{"0","",""};
lookup1("1302A") ->	{"0","",""};
lookup1("1302B") ->	{"0","",""};
lookup1("1302C") ->	{"0","",""};
lookup1("1302D") ->	{"0","",""};
lookup1("1302E") ->	{"0","",""};
lookup1("1302F") ->	{"0","",""};
lookup1("1303") ->	{"0","",""};
lookup1("13030") ->	{"0","",""};
lookup1("13031") ->	{"0","",""};
lookup1("13032") ->	{"0","",""};
lookup1("13033") ->	{"0","",""};
lookup1("13034") ->	{"0","",""};
lookup1("13035") ->	{"0","",""};
lookup1("13036") ->	{"0","",""};
lookup1("13037") ->	{"0","",""};
lookup1("13038") ->	{"0","",""};
lookup1("13039") ->	{"0","",""};
lookup1("1303A") ->	{"0","",""};
lookup1("1303B") ->	{"0","",""};
lookup1("1303C") ->	{"0","",""};
lookup1("1303D") ->	{"0","",""};
lookup1("1303E") ->	{"0","",""};
lookup1("1303F") ->	{"0","",""};
lookup1("1304") ->	{"0","",""};
lookup1("13040") ->	{"0","",""};
lookup1("13041") ->	{"0","",""};
lookup1("13042") ->	{"0","",""};
lookup1("13043") ->	{"0","",""};
lookup1("13044") ->	{"0","",""};
lookup1("13045") ->	{"0","",""};
lookup1("13046") ->	{"0","",""};
lookup1("13047") ->	{"0","",""};
lookup1("13048") ->	{"0","",""};
lookup1("13049") ->	{"0","",""};
lookup1("1304A") ->	{"0","",""};
lookup1("1304B") ->	{"0","",""};
lookup1("1304C") ->	{"0","",""};
lookup1("1304D") ->	{"0","",""};
lookup1("1304E") ->	{"0","",""};
lookup1("1304F") ->	{"0","",""};
lookup1("1305") ->	{"0","",""};
lookup1("13050") ->	{"0","",""};
lookup1("13051") ->	{"0","",""};
lookup1("13052") ->	{"0","",""};
lookup1("13053") ->	{"0","",""};
lookup1("13054") ->	{"0","",""};
lookup1("13055") ->	{"0","",""};
lookup1("13056") ->	{"0","",""};
lookup1("13057") ->	{"0","",""};
lookup1("13058") ->	{"0","",""};
lookup1("13059") ->	{"0","",""};
lookup1("1305A") ->	{"0","",""};
lookup1("1305B") ->	{"0","",""};
lookup1("1305C") ->	{"0","",""};
lookup1("1305D") ->	{"0","",""};
lookup1("1305E") ->	{"0","",""};
lookup1("1305F") ->	{"0","",""};
lookup1("1306") ->	{"0","",""};
lookup1("13060") ->	{"0","",""};
lookup1("13061") ->	{"0","",""};
lookup1("13062") ->	{"0","",""};
lookup1("13063") ->	{"0","",""};
lookup1("13064") ->	{"0","",""};
lookup1("13065") ->	{"0","",""};
lookup1("13066") ->	{"0","",""};
lookup1("13067") ->	{"0","",""};
lookup1("13068") ->	{"0","",""};
lookup1("13069") ->	{"0","",""};
lookup1("1306A") ->	{"0","",""};
lookup1("1306B") ->	{"0","",""};
lookup1("1306C") ->	{"0","",""};
lookup1("1306D") ->	{"0","",""};
lookup1("1306E") ->	{"0","",""};
lookup1("1306F") ->	{"0","",""};
lookup1("1307") ->	{"0","",""};
lookup1("13070") ->	{"0","",""};
lookup1("13071") ->	{"0","",""};
lookup1("13072") ->	{"0","",""};
lookup1("13073") ->	{"0","",""};
lookup1("13074") ->	{"0","",""};
lookup1("13075") ->	{"0","",""};
lookup1("13076") ->	{"0","",""};
lookup1("13077") ->	{"0","",""};
lookup1("13078") ->	{"0","",""};
lookup1("13079") ->	{"0","",""};
lookup1("1307A") ->	{"0","",""};
lookup1("1307B") ->	{"0","",""};
lookup1("1307C") ->	{"0","",""};
lookup1("1307D") ->	{"0","",""};
lookup1("1307E") ->	{"0","",""};
lookup1("1307F") ->	{"0","",""};
lookup1("1308") ->	{"0","",""};
lookup1("13080") ->	{"0","",""};
lookup1("13081") ->	{"0","",""};
lookup1("13082") ->	{"0","",""};
lookup1("13083") ->	{"0","",""};
lookup1("13084") ->	{"0","",""};
lookup1("13085") ->	{"0","",""};
lookup1("13086") ->	{"0","",""};
lookup1("13087") ->	{"0","",""};
lookup1("13088") ->	{"0","",""};
lookup1("13089") ->	{"0","",""};
lookup1("1308A") ->	{"0","",""};
lookup1("1308B") ->	{"0","",""};
lookup1("1308C") ->	{"0","",""};
lookup1("1308D") ->	{"0","",""};
lookup1("1308E") ->	{"0","",""};
lookup1("1308F") ->	{"0","",""};
lookup1("1309") ->	{"0","",""};
lookup1("13090") ->	{"0","",""};
lookup1("13091") ->	{"0","",""};
lookup1("13092") ->	{"0","",""};
lookup1("13093") ->	{"0","",""};
lookup1("13094") ->	{"0","",""};
lookup1("13095") ->	{"0","",""};
lookup1("13096") ->	{"0","",""};
lookup1("13097") ->	{"0","",""};
lookup1("13098") ->	{"0","",""};
lookup1("13099") ->	{"0","",""};
lookup1("1309A") ->	{"0","",""};
lookup1("1309B") ->	{"0","",""};
lookup1("1309C") ->	{"0","",""};
lookup1("1309D") ->	{"0","",""};
lookup1("1309E") ->	{"0","",""};
lookup1("1309F") ->	{"0","",""};
lookup1("130A") ->	{"0","",""};
lookup1("130A0") ->	{"0","",""};
lookup1("130A1") ->	{"0","",""};
lookup1("130A2") ->	{"0","",""};
lookup1("130A3") ->	{"0","",""};
lookup1("130A4") ->	{"0","",""};
lookup1("130A5") ->	{"0","",""};
lookup1("130A6") ->	{"0","",""};
lookup1("130A7") ->	{"0","",""};
lookup1("130A8") ->	{"0","",""};
lookup1("130A9") ->	{"0","",""};
lookup1("130AA") ->	{"0","",""};
lookup1("130AB") ->	{"0","",""};
lookup1("130AC") ->	{"0","",""};
lookup1("130AD") ->	{"0","",""};
lookup1("130AE") ->	{"0","",""};
lookup1("130AF") ->	{"0","",""};
lookup1("130B") ->	{"0","",""};
lookup1("130B0") ->	{"0","",""};
lookup1("130B1") ->	{"0","",""};
lookup1("130B2") ->	{"0","",""};
lookup1("130B3") ->	{"0","",""};
lookup1("130B4") ->	{"0","",""};
lookup1("130B5") ->	{"0","",""};
lookup1("130B6") ->	{"0","",""};
lookup1("130B7") ->	{"0","",""};
lookup1("130B8") ->	{"0","",""};
lookup1("130B9") ->	{"0","",""};
lookup1("130BA") ->	{"0","",""};
lookup1("130BB") ->	{"0","",""};
lookup1("130BC") ->	{"0","",""};
lookup1("130BD") ->	{"0","",""};
lookup1("130BE") ->	{"0","",""};
lookup1("130BF") ->	{"0","",""};
lookup1("130C") ->	{"0","",""};
lookup1("130C0") ->	{"0","",""};
lookup1("130C1") ->	{"0","",""};
lookup1("130C2") ->	{"0","",""};
lookup1("130C3") ->	{"0","",""};
lookup1("130C4") ->	{"0","",""};
lookup1("130C5") ->	{"0","",""};
lookup1("130C6") ->	{"0","",""};
lookup1("130C7") ->	{"0","",""};
lookup1("130C8") ->	{"0","",""};
lookup1("130C9") ->	{"0","",""};
lookup1("130CA") ->	{"0","",""};
lookup1("130CB") ->	{"0","",""};
lookup1("130CC") ->	{"0","",""};
lookup1("130CD") ->	{"0","",""};
lookup1("130CE") ->	{"0","",""};
lookup1("130CF") ->	{"0","",""};
lookup1("130D") ->	{"0","",""};
lookup1("130D0") ->	{"0","",""};
lookup1("130D1") ->	{"0","",""};
lookup1("130D2") ->	{"0","",""};
lookup1("130D3") ->	{"0","",""};
lookup1("130D4") ->	{"0","",""};
lookup1("130D5") ->	{"0","",""};
lookup1("130D6") ->	{"0","",""};
lookup1("130D7") ->	{"0","",""};
lookup1("130D8") ->	{"0","",""};
lookup1("130D9") ->	{"0","",""};
lookup1("130DA") ->	{"0","",""};
lookup1("130DB") ->	{"0","",""};
lookup1("130DC") ->	{"0","",""};
lookup1("130DD") ->	{"0","",""};
lookup1("130DE") ->	{"0","",""};
lookup1("130DF") ->	{"0","",""};
lookup1("130E") ->	{"0","",""};
lookup1("130E0") ->	{"0","",""};
lookup1("130E1") ->	{"0","",""};
lookup1("130E2") ->	{"0","",""};
lookup1("130E3") ->	{"0","",""};
lookup1("130E4") ->	{"0","",""};
lookup1("130E5") ->	{"0","",""};
lookup1("130E6") ->	{"0","",""};
lookup1("130E7") ->	{"0","",""};
lookup1("130E8") ->	{"0","",""};
lookup1("130E9") ->	{"0","",""};
lookup1("130EA") ->	{"0","",""};
lookup1("130EB") ->	{"0","",""};
lookup1("130EC") ->	{"0","",""};
lookup1("130ED") ->	{"0","",""};
lookup1("130EE") ->	{"0","",""};
lookup1("130EF") ->	{"0","",""};
lookup1("130F") ->	{"0","",""};
lookup1("130F0") ->	{"0","",""};
lookup1("130F1") ->	{"0","",""};
lookup1("130F2") ->	{"0","",""};
lookup1("130F3") ->	{"0","",""};
lookup1("130F4") ->	{"0","",""};
lookup1("130F5") ->	{"0","",""};
lookup1("130F6") ->	{"0","",""};
lookup1("130F7") ->	{"0","",""};
lookup1("130F8") ->	{"0","",""};
lookup1("130F9") ->	{"0","",""};
lookup1("130FA") ->	{"0","",""};
lookup1("130FB") ->	{"0","",""};
lookup1("130FC") ->	{"0","",""};
lookup1("130FD") ->	{"0","",""};
lookup1("130FE") ->	{"0","",""};
lookup1("130FF") ->	{"0","",""};
lookup1("1310") ->	{"0","",""};
lookup1("13100") ->	{"0","",""};
lookup1("13101") ->	{"0","",""};
lookup1("13102") ->	{"0","",""};
lookup1("13103") ->	{"0","",""};
lookup1("13104") ->	{"0","",""};
lookup1("13105") ->	{"0","",""};
lookup1("13106") ->	{"0","",""};
lookup1("13107") ->	{"0","",""};
lookup1("13108") ->	{"0","",""};
lookup1("13109") ->	{"0","",""};
lookup1("1310A") ->	{"0","",""};
lookup1("1310B") ->	{"0","",""};
lookup1("1310C") ->	{"0","",""};
lookup1("1310D") ->	{"0","",""};
lookup1("1310E") ->	{"0","",""};
lookup1("1310F") ->	{"0","",""};
lookup1("13110") ->	{"0","",""};
lookup1("13111") ->	{"0","",""};
lookup1("13112") ->	{"0","",""};
lookup1("13113") ->	{"0","",""};
lookup1("13114") ->	{"0","",""};
lookup1("13115") ->	{"0","",""};
lookup1("13116") ->	{"0","",""};
lookup1("13117") ->	{"0","",""};
lookup1("13118") ->	{"0","",""};
lookup1("13119") ->	{"0","",""};
lookup1("1311A") ->	{"0","",""};
lookup1("1311B") ->	{"0","",""};
lookup1("1311C") ->	{"0","",""};
lookup1("1311D") ->	{"0","",""};
lookup1("1311E") ->	{"0","",""};
lookup1("1311F") ->	{"0","",""};
lookup1("1312") ->	{"0","",""};
lookup1("13120") ->	{"0","",""};
lookup1("13121") ->	{"0","",""};
lookup1("13122") ->	{"0","",""};
lookup1("13123") ->	{"0","",""};
lookup1("13124") ->	{"0","",""};
lookup1("13125") ->	{"0","",""};
lookup1("13126") ->	{"0","",""};
lookup1("13127") ->	{"0","",""};
lookup1("13128") ->	{"0","",""};
lookup1("13129") ->	{"0","",""};
lookup1("1312A") ->	{"0","",""};
lookup1("1312B") ->	{"0","",""};
lookup1("1312C") ->	{"0","",""};
lookup1("1312D") ->	{"0","",""};
lookup1("1312E") ->	{"0","",""};
lookup1("1312F") ->	{"0","",""};
lookup1("1313") ->	{"0","",""};
lookup1("13130") ->	{"0","",""};
lookup1("13131") ->	{"0","",""};
lookup1("13132") ->	{"0","",""};
lookup1("13133") ->	{"0","",""};
lookup1("13134") ->	{"0","",""};
lookup1("13135") ->	{"0","",""};
lookup1("13136") ->	{"0","",""};
lookup1("13137") ->	{"0","",""};
lookup1("13138") ->	{"0","",""};
lookup1("13139") ->	{"0","",""};
lookup1("1313A") ->	{"0","",""};
lookup1("1313B") ->	{"0","",""};
lookup1("1313C") ->	{"0","",""};
lookup1("1313D") ->	{"0","",""};
lookup1("1313E") ->	{"0","",""};
lookup1("1313F") ->	{"0","",""};
lookup1("1314") ->	{"0","",""};
lookup1("13140") ->	{"0","",""};
lookup1("13141") ->	{"0","",""};
lookup1("13142") ->	{"0","",""};
lookup1("13143") ->	{"0","",""};
lookup1("13144") ->	{"0","",""};
lookup1("13145") ->	{"0","",""};
lookup1("13146") ->	{"0","",""};
lookup1("13147") ->	{"0","",""};
lookup1("13148") ->	{"0","",""};
lookup1("13149") ->	{"0","",""};
lookup1("1314A") ->	{"0","",""};
lookup1("1314B") ->	{"0","",""};
lookup1("1314C") ->	{"0","",""};
lookup1("1314D") ->	{"0","",""};
lookup1("1314E") ->	{"0","",""};
lookup1("1314F") ->	{"0","",""};
lookup1("1315") ->	{"0","",""};
lookup1("13150") ->	{"0","",""};
lookup1("13151") ->	{"0","",""};
lookup1("13152") ->	{"0","",""};
lookup1("13153") ->	{"0","",""};
lookup1("13154") ->	{"0","",""};
lookup1("13155") ->	{"0","",""};
lookup1("13156") ->	{"0","",""};
lookup1("13157") ->	{"0","",""};
lookup1("13158") ->	{"0","",""};
lookup1("13159") ->	{"0","",""};
lookup1("1315A") ->	{"0","",""};
lookup1("1315B") ->	{"0","",""};
lookup1("1315C") ->	{"0","",""};
lookup1("1315D") ->	{"0","",""};
lookup1("1315E") ->	{"0","",""};
lookup1("1315F") ->	{"0","",""};
lookup1("13160") ->	{"0","",""};
lookup1("13161") ->	{"0","",""};
lookup1("13162") ->	{"0","",""};
lookup1("13163") ->	{"0","",""};
lookup1("13164") ->	{"0","",""};
lookup1("13165") ->	{"0","",""};
lookup1("13166") ->	{"0","",""};
lookup1("13167") ->	{"0","",""};
lookup1("13168") ->	{"0","",""};
lookup1("13169") ->	{"0","",""};
lookup1("1316A") ->	{"0","",""};
lookup1("1316B") ->	{"0","",""};
lookup1("1316C") ->	{"0","",""};
lookup1("1316D") ->	{"0","",""};
lookup1("1316E") ->	{"0","",""};
lookup1("1316F") ->	{"0","",""};
lookup1("13170") ->	{"0","",""};
lookup1("13171") ->	{"0","",""};
lookup1("13172") ->	{"0","",""};
lookup1("13173") ->	{"0","",""};
lookup1("13174") ->	{"0","",""};
lookup1("13175") ->	{"0","",""};
lookup1("13176") ->	{"0","",""};
lookup1("13177") ->	{"0","",""};
lookup1("13178") ->	{"0","",""};
lookup1("13179") ->	{"0","",""};
lookup1("1317A") ->	{"0","",""};
lookup1("1317B") ->	{"0","",""};
lookup1("1317C") ->	{"0","",""};
lookup1("1317D") ->	{"0","",""};
lookup1("1317E") ->	{"0","",""};
lookup1("1317F") ->	{"0","",""};
lookup1("1318") ->	{"0","",""};
lookup1("13180") ->	{"0","",""};
lookup1("13181") ->	{"0","",""};
lookup1("13182") ->	{"0","",""};
lookup1("13183") ->	{"0","",""};
lookup1("13184") ->	{"0","",""};
lookup1("13185") ->	{"0","",""};
lookup1("13186") ->	{"0","",""};
lookup1("13187") ->	{"0","",""};
lookup1("13188") ->	{"0","",""};
lookup1("13189") ->	{"0","",""};
lookup1("1318A") ->	{"0","",""};
lookup1("1318B") ->	{"0","",""};
lookup1("1318C") ->	{"0","",""};
lookup1("1318D") ->	{"0","",""};
lookup1("1318E") ->	{"0","",""};
lookup1("1318F") ->	{"0","",""};
lookup1("1319") ->	{"0","",""};
lookup1("13190") ->	{"0","",""};
lookup1("13191") ->	{"0","",""};
lookup1("13192") ->	{"0","",""};
lookup1("13193") ->	{"0","",""};
lookup1("13194") ->	{"0","",""};
lookup1("13195") ->	{"0","",""};
lookup1("13196") ->	{"0","",""};
lookup1("13197") ->	{"0","",""};
lookup1("13198") ->	{"0","",""};
lookup1("13199") ->	{"0","",""};
lookup1("1319A") ->	{"0","",""};
lookup1("1319B") ->	{"0","",""};
lookup1("1319C") ->	{"0","",""};
lookup1("1319D") ->	{"0","",""};
lookup1("1319E") ->	{"0","",""};
lookup1("1319F") ->	{"0","",""};
lookup1("131A") ->	{"0","",""};
lookup1("131A0") ->	{"0","",""};
lookup1("131A1") ->	{"0","",""};
lookup1("131A2") ->	{"0","",""};
lookup1("131A3") ->	{"0","",""};
lookup1("131A4") ->	{"0","",""};
lookup1("131A5") ->	{"0","",""};
lookup1("131A6") ->	{"0","",""};
lookup1("131A7") ->	{"0","",""};
lookup1("131A8") ->	{"0","",""};
lookup1("131A9") ->	{"0","",""};
lookup1("131AA") ->	{"0","",""};
lookup1("131AB") ->	{"0","",""};
lookup1("131AC") ->	{"0","",""};
lookup1("131AD") ->	{"0","",""};
lookup1("131AE") ->	{"0","",""};
lookup1("131AF") ->	{"0","",""};
lookup1("131B") ->	{"0","",""};
lookup1("131B0") ->	{"0","",""};
lookup1("131B1") ->	{"0","",""};
lookup1("131B2") ->	{"0","",""};
lookup1("131B3") ->	{"0","",""};
lookup1("131B4") ->	{"0","",""};
lookup1("131B5") ->	{"0","",""};
lookup1("131B6") ->	{"0","",""};
lookup1("131B7") ->	{"0","",""};
lookup1("131B8") ->	{"0","",""};
lookup1("131B9") ->	{"0","",""};
lookup1("131BA") ->	{"0","",""};
lookup1("131BB") ->	{"0","",""};
lookup1("131BC") ->	{"0","",""};
lookup1("131BD") ->	{"0","",""};
lookup1("131BE") ->	{"0","",""};
lookup1("131BF") ->	{"0","",""};
lookup1("131C") ->	{"0","",""};
lookup1("131C0") ->	{"0","",""};
lookup1("131C1") ->	{"0","",""};
lookup1("131C2") ->	{"0","",""};
lookup1("131C3") ->	{"0","",""};
lookup1("131C4") ->	{"0","",""};
lookup1("131C5") ->	{"0","",""};
lookup1("131C6") ->	{"0","",""};
lookup1("131C7") ->	{"0","",""};
lookup1("131C8") ->	{"0","",""};
lookup1("131C9") ->	{"0","",""};
lookup1("131CA") ->	{"0","",""};
lookup1("131CB") ->	{"0","",""};
lookup1("131CC") ->	{"0","",""};
lookup1("131CD") ->	{"0","",""};
lookup1("131CE") ->	{"0","",""};
lookup1("131CF") ->	{"0","",""};
lookup1("131D") ->	{"0","",""};
lookup1("131D0") ->	{"0","",""};
lookup1("131D1") ->	{"0","",""};
lookup1("131D2") ->	{"0","",""};
lookup1("131D3") ->	{"0","",""};
lookup1("131D4") ->	{"0","",""};
lookup1("131D5") ->	{"0","",""};
lookup1("131D6") ->	{"0","",""};
lookup1("131D7") ->	{"0","",""};
lookup1("131D8") ->	{"0","",""};
lookup1("131D9") ->	{"0","",""};
lookup1("131DA") ->	{"0","",""};
lookup1("131DB") ->	{"0","",""};
lookup1("131DC") ->	{"0","",""};
lookup1("131DD") ->	{"0","",""};
lookup1("131DE") ->	{"0","",""};
lookup1("131DF") ->	{"0","",""};
lookup1("131E") ->	{"0","",""};
lookup1("131E0") ->	{"0","",""};
lookup1("131E1") ->	{"0","",""};
lookup1("131E2") ->	{"0","",""};
lookup1("131E3") ->	{"0","",""};
lookup1("131E4") ->	{"0","",""};
lookup1("131E5") ->	{"0","",""};
lookup1("131E6") ->	{"0","",""};
lookup1("131E7") ->	{"0","",""};
lookup1("131E8") ->	{"0","",""};
lookup1("131E9") ->	{"0","",""};
lookup1("131EA") ->	{"0","",""};
lookup1("131EB") ->	{"0","",""};
lookup1("131EC") ->	{"0","",""};
lookup1("131ED") ->	{"0","",""};
lookup1("131EE") ->	{"0","",""};
lookup1("131EF") ->	{"0","",""};
lookup1("131F") ->	{"0","",""};
lookup1("131F0") ->	{"0","",""};
lookup1("131F1") ->	{"0","",""};
lookup1("131F2") ->	{"0","",""};
lookup1("131F3") ->	{"0","",""};
lookup1("131F4") ->	{"0","",""};
lookup1("131F5") ->	{"0","",""};
lookup1("131F6") ->	{"0","",""};
lookup1("131F7") ->	{"0","",""};
lookup1("131F8") ->	{"0","",""};
lookup1("131F9") ->	{"0","",""};
lookup1("131FA") ->	{"0","",""};
lookup1("131FB") ->	{"0","",""};
lookup1("131FC") ->	{"0","",""};
lookup1("131FD") ->	{"0","",""};
lookup1("131FE") ->	{"0","",""};
lookup1("131FF") ->	{"0","",""};
lookup1("1320") ->	{"0","",""};
lookup1("13200") ->	{"0","",""};
lookup1("13201") ->	{"0","",""};
lookup1("13202") ->	{"0","",""};
lookup1("13203") ->	{"0","",""};
lookup1("13204") ->	{"0","",""};
lookup1("13205") ->	{"0","",""};
lookup1("13206") ->	{"0","",""};
lookup1("13207") ->	{"0","",""};
lookup1("13208") ->	{"0","",""};
lookup1("13209") ->	{"0","",""};
lookup1("1320A") ->	{"0","",""};
lookup1("1320B") ->	{"0","",""};
lookup1("1320C") ->	{"0","",""};
lookup1("1320D") ->	{"0","",""};
lookup1("1320E") ->	{"0","",""};
lookup1("1320F") ->	{"0","",""};
lookup1("1321") ->	{"0","",""};
lookup1("13210") ->	{"0","",""};
lookup1("13211") ->	{"0","",""};
lookup1("13212") ->	{"0","",""};
lookup1("13213") ->	{"0","",""};
lookup1("13214") ->	{"0","",""};
lookup1("13215") ->	{"0","",""};
lookup1("13216") ->	{"0","",""};
lookup1("13217") ->	{"0","",""};
lookup1("13218") ->	{"0","",""};
lookup1("13219") ->	{"0","",""};
lookup1("1321A") ->	{"0","",""};
lookup1("1321B") ->	{"0","",""};
lookup1("1321C") ->	{"0","",""};
lookup1("1321D") ->	{"0","",""};
lookup1("1321E") ->	{"0","",""};
lookup1("1321F") ->	{"0","",""};
lookup1("1322") ->	{"0","",""};
lookup1("13220") ->	{"0","",""};
lookup1("13221") ->	{"0","",""};
lookup1("13222") ->	{"0","",""};
lookup1("13223") ->	{"0","",""};
lookup1("13224") ->	{"0","",""};
lookup1("13225") ->	{"0","",""};
lookup1("13226") ->	{"0","",""};
lookup1("13227") ->	{"0","",""};
lookup1("13228") ->	{"0","",""};
lookup1("13229") ->	{"0","",""};
lookup1("1322A") ->	{"0","",""};
lookup1("1322B") ->	{"0","",""};
lookup1("1322C") ->	{"0","",""};
lookup1("1322D") ->	{"0","",""};
lookup1("1322E") ->	{"0","",""};
lookup1("1322F") ->	{"0","",""};
lookup1("1323") ->	{"0","",""};
lookup1("13230") ->	{"0","",""};
lookup1("13231") ->	{"0","",""};
lookup1("13232") ->	{"0","",""};
lookup1("13233") ->	{"0","",""};
lookup1("13234") ->	{"0","",""};
lookup1("13235") ->	{"0","",""};
lookup1("13236") ->	{"0","",""};
lookup1("13237") ->	{"0","",""};
lookup1("13238") ->	{"0","",""};
lookup1("13239") ->	{"0","",""};
lookup1("1323A") ->	{"0","",""};
lookup1("1323B") ->	{"0","",""};
lookup1("1323C") ->	{"0","",""};
lookup1("1323D") ->	{"0","",""};
lookup1("1323E") ->	{"0","",""};
lookup1("1323F") ->	{"0","",""};
lookup1("1324") ->	{"0","",""};
lookup1("13240") ->	{"0","",""};
lookup1("13241") ->	{"0","",""};
lookup1("13242") ->	{"0","",""};
lookup1("13243") ->	{"0","",""};
lookup1("13244") ->	{"0","",""};
lookup1("13245") ->	{"0","",""};
lookup1("13246") ->	{"0","",""};
lookup1("13247") ->	{"0","",""};
lookup1("13248") ->	{"0","",""};
lookup1("13249") ->	{"0","",""};
lookup1("1324A") ->	{"0","",""};
lookup1("1324B") ->	{"0","",""};
lookup1("1324C") ->	{"0","",""};
lookup1("1324D") ->	{"0","",""};
lookup1("1324E") ->	{"0","",""};
lookup1("1324F") ->	{"0","",""};
lookup1("1325") ->	{"0","",""};
lookup1("13250") ->	{"0","",""};
lookup1("13251") ->	{"0","",""};
lookup1("13252") ->	{"0","",""};
lookup1("13253") ->	{"0","",""};
lookup1("13254") ->	{"0","",""};
lookup1("13255") ->	{"0","",""};
lookup1("13256") ->	{"0","",""};
lookup1("13257") ->	{"0","",""};
lookup1("13258") ->	{"0","",""};
lookup1("13259") ->	{"0","",""};
lookup1("1325A") ->	{"0","",""};
lookup1("1325B") ->	{"0","",""};
lookup1("1325C") ->	{"0","",""};
lookup1("1325D") ->	{"0","",""};
lookup1("1325E") ->	{"0","",""};
lookup1("1325F") ->	{"0","",""};
lookup1("1326") ->	{"0","",""};
lookup1("13260") ->	{"0","",""};
lookup1("13261") ->	{"0","",""};
lookup1("13262") ->	{"0","",""};
lookup1("13263") ->	{"0","",""};
lookup1("13264") ->	{"0","",""};
lookup1("13265") ->	{"0","",""};
lookup1("13266") ->	{"0","",""};
lookup1("13267") ->	{"0","",""};
lookup1("13268") ->	{"0","",""};
lookup1("13269") ->	{"0","",""};
lookup1("1326A") ->	{"0","",""};
lookup1("1326B") ->	{"0","",""};
lookup1("1326C") ->	{"0","",""};
lookup1("1326D") ->	{"0","",""};
lookup1("1326E") ->	{"0","",""};
lookup1("1326F") ->	{"0","",""};
lookup1("1327") ->	{"0","",""};
lookup1("13270") ->	{"0","",""};
lookup1("13271") ->	{"0","",""};
lookup1("13272") ->	{"0","",""};
lookup1("13273") ->	{"0","",""};
lookup1("13274") ->	{"0","",""};
lookup1("13275") ->	{"0","",""};
lookup1("13276") ->	{"0","",""};
lookup1("13277") ->	{"0","",""};
lookup1("13278") ->	{"0","",""};
lookup1("13279") ->	{"0","",""};
lookup1("1327A") ->	{"0","",""};
lookup1("1327B") ->	{"0","",""};
lookup1("1327C") ->	{"0","",""};
lookup1("1327D") ->	{"0","",""};
lookup1("1327E") ->	{"0","",""};
lookup1("1327F") ->	{"0","",""};
lookup1("1328") ->	{"0","",""};
lookup1("13280") ->	{"0","",""};
lookup1("13281") ->	{"0","",""};
lookup1("13282") ->	{"0","",""};
lookup1("13283") ->	{"0","",""};
lookup1("13284") ->	{"0","",""};
lookup1("13285") ->	{"0","",""};
lookup1("13286") ->	{"0","",""};
lookup1("13287") ->	{"0","",""};
lookup1("13288") ->	{"0","",""};
lookup1("13289") ->	{"0","",""};
lookup1("1328A") ->	{"0","",""};
lookup1("1328B") ->	{"0","",""};
lookup1("1328C") ->	{"0","",""};
lookup1("1328D") ->	{"0","",""};
lookup1("1328E") ->	{"0","",""};
lookup1("1328F") ->	{"0","",""};
lookup1("1329") ->	{"0","",""};
lookup1("13290") ->	{"0","",""};
lookup1("13291") ->	{"0","",""};
lookup1("13292") ->	{"0","",""};
lookup1("13293") ->	{"0","",""};
lookup1("13294") ->	{"0","",""};
lookup1("13295") ->	{"0","",""};
lookup1("13296") ->	{"0","",""};
lookup1("13297") ->	{"0","",""};
lookup1("13298") ->	{"0","",""};
lookup1("13299") ->	{"0","",""};
lookup1("1329A") ->	{"0","",""};
lookup1("1329B") ->	{"0","",""};
lookup1("1329C") ->	{"0","",""};
lookup1("1329D") ->	{"0","",""};
lookup1("1329E") ->	{"0","",""};
lookup1("1329F") ->	{"0","",""};
lookup1("132A") ->	{"0","",""};
lookup1("132A0") ->	{"0","",""};
lookup1("132A1") ->	{"0","",""};
lookup1("132A2") ->	{"0","",""};
lookup1("132A3") ->	{"0","",""};
lookup1("132A4") ->	{"0","",""};
lookup1("132A5") ->	{"0","",""};
lookup1("132A6") ->	{"0","",""};
lookup1("132A7") ->	{"0","",""};
lookup1("132A8") ->	{"0","",""};
lookup1("132A9") ->	{"0","",""};
lookup1("132AA") ->	{"0","",""};
lookup1("132AB") ->	{"0","",""};
lookup1("132AC") ->	{"0","",""};
lookup1("132AD") ->	{"0","",""};
lookup1("132AE") ->	{"0","",""};
lookup1("132AF") ->	{"0","",""};
lookup1("132B") ->	{"0","",""};
lookup1("132B0") ->	{"0","",""};
lookup1("132B1") ->	{"0","",""};
lookup1("132B2") ->	{"0","",""};
lookup1("132B3") ->	{"0","",""};
lookup1("132B4") ->	{"0","",""};
lookup1("132B5") ->	{"0","",""};
lookup1("132B6") ->	{"0","",""};
lookup1("132B7") ->	{"0","",""};
lookup1("132B8") ->	{"0","",""};
lookup1("132B9") ->	{"0","",""};
lookup1("132BA") ->	{"0","",""};
lookup1("132BB") ->	{"0","",""};
lookup1("132BC") ->	{"0","",""};
lookup1("132BD") ->	{"0","",""};
lookup1("132BE") ->	{"0","",""};
lookup1("132BF") ->	{"0","",""};
lookup1("132C") ->	{"0","",""};
lookup1("132C0") ->	{"0","",""};
lookup1("132C1") ->	{"0","",""};
lookup1("132C2") ->	{"0","",""};
lookup1("132C3") ->	{"0","",""};
lookup1("132C4") ->	{"0","",""};
lookup1("132C5") ->	{"0","",""};
lookup1("132C6") ->	{"0","",""};
lookup1("132C7") ->	{"0","",""};
lookup1("132C8") ->	{"0","",""};
lookup1("132C9") ->	{"0","",""};
lookup1("132CA") ->	{"0","",""};
lookup1("132CB") ->	{"0","",""};
lookup1("132CC") ->	{"0","",""};
lookup1("132CD") ->	{"0","",""};
lookup1("132CE") ->	{"0","",""};
lookup1("132CF") ->	{"0","",""};
lookup1("132D") ->	{"0","",""};
lookup1("132D0") ->	{"0","",""};
lookup1("132D1") ->	{"0","",""};
lookup1("132D2") ->	{"0","",""};
lookup1("132D3") ->	{"0","",""};
lookup1("132D4") ->	{"0","",""};
lookup1("132D5") ->	{"0","",""};
lookup1("132D6") ->	{"0","",""};
lookup1("132D7") ->	{"0","",""};
lookup1("132D8") ->	{"0","",""};
lookup1("132D9") ->	{"0","",""};
lookup1("132DA") ->	{"0","",""};
lookup1("132DB") ->	{"0","",""};
lookup1("132DC") ->	{"0","",""};
lookup1("132DD") ->	{"0","",""};
lookup1("132DE") ->	{"0","",""};
lookup1("132DF") ->	{"0","",""};
lookup1("132E") ->	{"0","",""};
lookup1("132E0") ->	{"0","",""};
lookup1("132E1") ->	{"0","",""};
lookup1("132E2") ->	{"0","",""};
lookup1("132E3") ->	{"0","",""};
lookup1("132E4") ->	{"0","",""};
lookup1("132E5") ->	{"0","",""};
lookup1("132E6") ->	{"0","",""};
lookup1("132E7") ->	{"0","",""};
lookup1("132E8") ->	{"0","",""};
lookup1("132E9") ->	{"0","",""};
lookup1("132EA") ->	{"0","",""};
lookup1("132EB") ->	{"0","",""};
lookup1("132EC") ->	{"0","",""};
lookup1("132ED") ->	{"0","",""};
lookup1("132EE") ->	{"0","",""};
lookup1("132EF") ->	{"0","",""};
lookup1("132F") ->	{"0","",""};
lookup1("132F0") ->	{"0","",""};
lookup1("132F1") ->	{"0","",""};
lookup1("132F2") ->	{"0","",""};
lookup1("132F3") ->	{"0","",""};
lookup1("132F4") ->	{"0","",""};
lookup1("132F5") ->	{"0","",""};
lookup1("132F6") ->	{"0","",""};
lookup1("132F7") ->	{"0","",""};
lookup1("132F8") ->	{"0","",""};
lookup1("132F9") ->	{"0","",""};
lookup1("132FA") ->	{"0","",""};
lookup1("132FB") ->	{"0","",""};
lookup1("132FC") ->	{"0","",""};
lookup1("132FD") ->	{"0","",""};
lookup1("132FE") ->	{"0","",""};
lookup1("132FF") ->	{"0","",""};
lookup1("1330") ->	{"0","",""};
lookup1("13300") ->	{"0","",""};
lookup1("13301") ->	{"0","",""};
lookup1("13302") ->	{"0","",""};
lookup1("13303") ->	{"0","",""};
lookup1("13304") ->	{"0","",""};
lookup1("13305") ->	{"0","",""};
lookup1("13306") ->	{"0","",""};
lookup1("13307") ->	{"0","",""};
lookup1("13308") ->	{"0","",""};
lookup1("13309") ->	{"0","",""};
lookup1("1330A") ->	{"0","",""};
lookup1("1330B") ->	{"0","",""};
lookup1("1330C") ->	{"0","",""};
lookup1("1330D") ->	{"0","",""};
lookup1("1330E") ->	{"0","",""};
lookup1("1330F") ->	{"0","",""};
lookup1("1331") ->	{"0","",""};
lookup1("13310") ->	{"0","",""};
lookup1("13311") ->	{"0","",""};
lookup1("13312") ->	{"0","",""};
lookup1("13313") ->	{"0","",""};
lookup1("13314") ->	{"0","",""};
lookup1("13315") ->	{"0","",""};
lookup1("13316") ->	{"0","",""};
lookup1("13317") ->	{"0","",""};
lookup1("13318") ->	{"0","",""};
lookup1("13319") ->	{"0","",""};
lookup1("1331A") ->	{"0","",""};
lookup1("1331B") ->	{"0","",""};
lookup1("1331C") ->	{"0","",""};
lookup1("1331D") ->	{"0","",""};
lookup1("1331E") ->	{"0","",""};
lookup1("1331F") ->	{"0","",""};
lookup1("1332") ->	{"0","",""};
lookup1("13320") ->	{"0","",""};
lookup1("13321") ->	{"0","",""};
lookup1("13322") ->	{"0","",""};
lookup1("13323") ->	{"0","",""};
lookup1("13324") ->	{"0","",""};
lookup1("13325") ->	{"0","",""};
lookup1("13326") ->	{"0","",""};
lookup1("13327") ->	{"0","",""};
lookup1("13328") ->	{"0","",""};
lookup1("13329") ->	{"0","",""};
lookup1("1332A") ->	{"0","",""};
lookup1("1332B") ->	{"0","",""};
lookup1("1332C") ->	{"0","",""};
lookup1("1332D") ->	{"0","",""};
lookup1("1332E") ->	{"0","",""};
lookup1("1332F") ->	{"0","",""};
lookup1("1333") ->	{"0","",""};
lookup1("13330") ->	{"0","",""};
lookup1("13331") ->	{"0","",""};
lookup1("13332") ->	{"0","",""};
lookup1("13333") ->	{"0","",""};
lookup1("13334") ->	{"0","",""};
lookup1("13335") ->	{"0","",""};
lookup1("13336") ->	{"0","",""};
lookup1("13337") ->	{"0","",""};
lookup1("13338") ->	{"0","",""};
lookup1("13339") ->	{"0","",""};
lookup1("1333A") ->	{"0","",""};
lookup1("1333B") ->	{"0","",""};
lookup1("1333C") ->	{"0","",""};
lookup1("1333D") ->	{"0","",""};
lookup1("1333E") ->	{"0","",""};
lookup1("1333F") ->	{"0","",""};
lookup1("1334") ->	{"0","",""};
lookup1("13340") ->	{"0","",""};
lookup1("13341") ->	{"0","",""};
lookup1("13342") ->	{"0","",""};
lookup1("13343") ->	{"0","",""};
lookup1("13344") ->	{"0","",""};
lookup1("13345") ->	{"0","",""};
lookup1("13346") ->	{"0","",""};
lookup1("13347") ->	{"0","",""};
lookup1("13348") ->	{"0","",""};
lookup1("13349") ->	{"0","",""};
lookup1("1334A") ->	{"0","",""};
lookup1("1334B") ->	{"0","",""};
lookup1("1334C") ->	{"0","",""};
lookup1("1334D") ->	{"0","",""};
lookup1("1334E") ->	{"0","",""};
lookup1("1334F") ->	{"0","",""};
lookup1("1335") ->	{"0","",""};
lookup1("13350") ->	{"0","",""};
lookup1("13351") ->	{"0","",""};
lookup1("13352") ->	{"0","",""};
lookup1("13353") ->	{"0","",""};
lookup1("13354") ->	{"0","",""};
lookup1("13355") ->	{"0","",""};
lookup1("13356") ->	{"0","",""};
lookup1("13357") ->	{"0","",""};
lookup1("13358") ->	{"0","",""};
lookup1("13359") ->	{"0","",""};
lookup1("1335A") ->	{"0","",""};
lookup1("1335B") ->	{"0","",""};
lookup1("1335C") ->	{"0","",""};
lookup1("1335D") ->	{"0","",""};
lookup1("1335E") ->	{"0","",""};
lookup1("1335F") ->	{"0","",""};
lookup1("1336") ->	{"0","",""};
lookup1("13360") ->	{"0","",""};
lookup1("13361") ->	{"0","",""};
lookup1("13362") ->	{"0","",""};
lookup1("13363") ->	{"0","",""};
lookup1("13364") ->	{"0","",""};
lookup1("13365") ->	{"0","",""};
lookup1("13366") ->	{"0","",""};
lookup1("13367") ->	{"0","",""};
lookup1("13368") ->	{"0","",""};
lookup1("13369") ->	{"0","",""};
lookup1("1336A") ->	{"0","",""};
lookup1("1336B") ->	{"0","",""};
lookup1("1336C") ->	{"0","",""};
lookup1("1336D") ->	{"0","",""};
lookup1("1336E") ->	{"0","",""};
lookup1("1336F") ->	{"0","",""};
lookup1("1337") ->	{"0","",""};
lookup1("13370") ->	{"0","",""};
lookup1("13371") ->	{"0","",""};
lookup1("13372") ->	{"0","",""};
lookup1("13373") ->	{"0","",""};
lookup1("13374") ->	{"0","",""};
lookup1("13375") ->	{"0","",""};
lookup1("13376") ->	{"0","",""};
lookup1("13377") ->	{"0","",""};
lookup1("13378") ->	{"0","",""};
lookup1("13379") ->	{"0","",""};
lookup1("1337A") ->	{"0","",""};
lookup1("1337B") ->	{"0","",""};
lookup1("1337C") ->	{"0","",""};
lookup1("1337D") ->	{"0","",""};
lookup1("1337E") ->	{"0","",""};
lookup1("1337F") ->	{"0","",""};
lookup1("1338") ->	{"0","",""};
lookup1("13380") ->	{"0","",""};
lookup1("13381") ->	{"0","",""};
lookup1("13382") ->	{"0","",""};
lookup1("13383") ->	{"0","",""};
lookup1("13384") ->	{"0","",""};
lookup1("13385") ->	{"0","",""};
lookup1("13386") ->	{"0","",""};
lookup1("13387") ->	{"0","",""};
lookup1("13388") ->	{"0","",""};
lookup1("13389") ->	{"0","",""};
lookup1("1338A") ->	{"0","",""};
lookup1("1338B") ->	{"0","",""};
lookup1("1338C") ->	{"0","",""};
lookup1("1338D") ->	{"0","",""};
lookup1("1338E") ->	{"0","",""};
lookup1("1338F") ->	{"0","",""};
lookup1("1339") ->	{"0","",""};
lookup1("13390") ->	{"0","",""};
lookup1("13391") ->	{"0","",""};
lookup1("13392") ->	{"0","",""};
lookup1("13393") ->	{"0","",""};
lookup1("13394") ->	{"0","",""};
lookup1("13395") ->	{"0","",""};
lookup1("13396") ->	{"0","",""};
lookup1("13397") ->	{"0","",""};
lookup1("13398") ->	{"0","",""};
lookup1("13399") ->	{"0","",""};
lookup1("1339A") ->	{"0","",""};
lookup1("1339B") ->	{"0","",""};
lookup1("1339C") ->	{"0","",""};
lookup1("1339D") ->	{"0","",""};
lookup1("1339E") ->	{"0","",""};
lookup1("1339F") ->	{"0","",""};
lookup1("133A") ->	{"0","",""};
lookup1("133A0") ->	{"0","",""};
lookup1("133A1") ->	{"0","",""};
lookup1("133A2") ->	{"0","",""};
lookup1("133A3") ->	{"0","",""};
lookup1("133A4") ->	{"0","",""};
lookup1("133A5") ->	{"0","",""};
lookup1("133A6") ->	{"0","",""};
lookup1("133A7") ->	{"0","",""};
lookup1("133A8") ->	{"0","",""};
lookup1("133A9") ->	{"0","",""};
lookup1("133AA") ->	{"0","",""};
lookup1("133AB") ->	{"0","",""};
lookup1("133AC") ->	{"0","",""};
lookup1("133AD") ->	{"0","",""};
lookup1("133AE") ->	{"0","",""};
lookup1("133AF") ->	{"0","",""};
lookup1("133B") ->	{"0","",""};
lookup1("133B0") ->	{"0","",""};
lookup1("133B1") ->	{"0","",""};
lookup1("133B2") ->	{"0","",""};
lookup1("133B3") ->	{"0","",""};
lookup1("133B4") ->	{"0","",""};
lookup1("133B5") ->	{"0","",""};
lookup1("133B6") ->	{"0","",""};
lookup1("133B7") ->	{"0","",""};
lookup1("133B8") ->	{"0","",""};
lookup1("133B9") ->	{"0","",""};
lookup1("133BA") ->	{"0","",""};
lookup1("133BB") ->	{"0","",""};
lookup1("133BC") ->	{"0","",""};
lookup1("133BD") ->	{"0","",""};
lookup1("133BE") ->	{"0","",""};
lookup1("133BF") ->	{"0","",""};
lookup1("133C") ->	{"0","",""};
lookup1("133C0") ->	{"0","",""};
lookup1("133C1") ->	{"0","",""};
lookup1("133C2") ->	{"0","",""};
lookup1("133C3") ->	{"0","",""};
lookup1("133C4") ->	{"0","",""};
lookup1("133C5") ->	{"0","",""};
lookup1("133C6") ->	{"0","",""};
lookup1("133C7") ->	{"0","",""};
lookup1("133C8") ->	{"0","",""};
lookup1("133C9") ->	{"0","",""};
lookup1("133CA") ->	{"0","",""};
lookup1("133CB") ->	{"0","",""};
lookup1("133CC") ->	{"0","",""};
lookup1("133CD") ->	{"0","",""};
lookup1("133CE") ->	{"0","",""};
lookup1("133CF") ->	{"0","",""};
lookup1("133D") ->	{"0","",""};
lookup1("133D0") ->	{"0","",""};
lookup1("133D1") ->	{"0","",""};
lookup1("133D2") ->	{"0","",""};
lookup1("133D3") ->	{"0","",""};
lookup1("133D4") ->	{"0","",""};
lookup1("133D5") ->	{"0","",""};
lookup1("133D6") ->	{"0","",""};
lookup1("133D7") ->	{"0","",""};
lookup1("133D8") ->	{"0","",""};
lookup1("133D9") ->	{"0","",""};
lookup1("133DA") ->	{"0","",""};
lookup1("133DB") ->	{"0","",""};
lookup1("133DC") ->	{"0","",""};
lookup1("133DD") ->	{"0","",""};
lookup1("133DE") ->	{"0","",""};
lookup1("133DF") ->	{"0","",""};
lookup1("133E") ->	{"0","",""};
lookup1("133E0") ->	{"0","",""};
lookup1("133E1") ->	{"0","",""};
lookup1("133E2") ->	{"0","",""};
lookup1("133E3") ->	{"0","",""};
lookup1("133E4") ->	{"0","",""};
lookup1("133E5") ->	{"0","",""};
lookup1("133E6") ->	{"0","",""};
lookup1("133E7") ->	{"0","",""};
lookup1("133E8") ->	{"0","",""};
lookup1("133E9") ->	{"0","",""};
lookup1("133EA") ->	{"0","",""};
lookup1("133EB") ->	{"0","",""};
lookup1("133EC") ->	{"0","",""};
lookup1("133ED") ->	{"0","",""};
lookup1("133EE") ->	{"0","",""};
lookup1("133EF") ->	{"0","",""};
lookup1("133F") ->	{"0","",""};
lookup1("133F0") ->	{"0","",""};
lookup1("133F1") ->	{"0","",""};
lookup1("133F2") ->	{"0","",""};
lookup1("133F3") ->	{"0","",""};
lookup1("133F4") ->	{"0","",""};
lookup1("133F5") ->	{"0","",""};
lookup1("133F6") ->	{"0","",""};
lookup1("133F7") ->	{"0","",""};
lookup1("133F8") ->	{"0","",""};
lookup1("133F9") ->	{"0","",""};
lookup1("133FA") ->	{"0","",""};
lookup1("133FB") ->	{"0","",""};
lookup1("133FC") ->	{"0","",""};
lookup1("133FD") ->	{"0","",""};
lookup1("133FE") ->	{"0","",""};
lookup1("133FF") ->	{"0","",""};
lookup1("1340") ->	{"0","",""};
lookup1("13400") ->	{"0","",""};
lookup1("13401") ->	{"0","",""};
lookup1("13402") ->	{"0","",""};
lookup1("13403") ->	{"0","",""};
lookup1("13404") ->	{"0","",""};
lookup1("13405") ->	{"0","",""};
lookup1("13406") ->	{"0","",""};
lookup1("13407") ->	{"0","",""};
lookup1("13408") ->	{"0","",""};
lookup1("13409") ->	{"0","",""};
lookup1("1340A") ->	{"0","",""};
lookup1("1340B") ->	{"0","",""};
lookup1("1340C") ->	{"0","",""};
lookup1("1340D") ->	{"0","",""};
lookup1("1340E") ->	{"0","",""};
lookup1("1340F") ->	{"0","",""};
lookup1("1341") ->	{"0","",""};
lookup1("13410") ->	{"0","",""};
lookup1("13411") ->	{"0","",""};
lookup1("13412") ->	{"0","",""};
lookup1("13413") ->	{"0","",""};
lookup1("13414") ->	{"0","",""};
lookup1("13415") ->	{"0","",""};
lookup1("13416") ->	{"0","",""};
lookup1("13417") ->	{"0","",""};
lookup1("13418") ->	{"0","",""};
lookup1("13419") ->	{"0","",""};
lookup1("1341A") ->	{"0","",""};
lookup1("1341B") ->	{"0","",""};
lookup1("1341C") ->	{"0","",""};
lookup1("1341D") ->	{"0","",""};
lookup1("1341E") ->	{"0","",""};
lookup1("1341F") ->	{"0","",""};
lookup1("1342") ->	{"0","",""};
lookup1("13420") ->	{"0","",""};
lookup1("13421") ->	{"0","",""};
lookup1("13422") ->	{"0","",""};
lookup1("13423") ->	{"0","",""};
lookup1("13424") ->	{"0","",""};
lookup1("13425") ->	{"0","",""};
lookup1("13426") ->	{"0","",""};
lookup1("13427") ->	{"0","",""};
lookup1("13428") ->	{"0","",""};
lookup1("13429") ->	{"0","",""};
lookup1("1342A") ->	{"0","",""};
lookup1("1342B") ->	{"0","",""};
lookup1("1342C") ->	{"0","",""};
lookup1("1342D") ->	{"0","",""};
lookup1("1342E") ->	{"0","",""};
lookup1("1343") ->	{"0","",""};
lookup1("1344") ->	{"0","",""};
lookup1("1345") ->	{"0","",""};
lookup1("1346") ->	{"0","",""};
lookup1("1347") ->	{"0","",""};
lookup1("1348") ->	{"0","",""};
lookup1("1349") ->	{"0","",""};
lookup1("134A") ->	{"0","",""};
lookup1("134B") ->	{"0","",""};
lookup1("134C") ->	{"0","",""};
lookup1("134D") ->	{"0","",""};
lookup1("134E") ->	{"0","",""};
lookup1("134F") ->	{"0","",""};
lookup1("1350") ->	{"0","",""};
lookup1("1351") ->	{"0","",""};
lookup1("1352") ->	{"0","",""};
lookup1("1353") ->	{"0","",""};
lookup1("1354") ->	{"0","",""};
lookup1("1355") ->	{"0","",""};
lookup1("1356") ->	{"0","",""};
lookup1("1357") ->	{"0","",""};
lookup1("1358") ->	{"0","",""};
lookup1("1359") ->	{"0","",""};
lookup1("135A") ->	{"0","",""};
lookup1("135D") ->	{"230","",""};
lookup1("135E") ->	{"230","",""};
lookup1("135F") ->	{"230","",""};
lookup1("1360") ->	{"0","",""};
lookup1("1361") ->	{"0","",""};
lookup1("1362") ->	{"0","",""};
lookup1("1363") ->	{"0","",""};
lookup1("1364") ->	{"0","",""};
lookup1("1365") ->	{"0","",""};
lookup1("1366") ->	{"0","",""};
lookup1("1367") ->	{"0","",""};
lookup1("1368") ->	{"0","",""};
lookup1("1369") ->	{"0","",""};
lookup1("136A") ->	{"0","",""};
lookup1("136B") ->	{"0","",""};
lookup1("136C") ->	{"0","",""};
lookup1("136D") ->	{"0","",""};
lookup1("136E") ->	{"0","",""};
lookup1("136F") ->	{"0","",""};
lookup1("1370") ->	{"0","",""};
lookup1("1371") ->	{"0","",""};
lookup1("1372") ->	{"0","",""};
lookup1("1373") ->	{"0","",""};
lookup1("1374") ->	{"0","",""};
lookup1("1375") ->	{"0","",""};
lookup1("1376") ->	{"0","",""};
lookup1("1377") ->	{"0","",""};
lookup1("1378") ->	{"0","",""};
lookup1("1379") ->	{"0","",""};
lookup1("137A") ->	{"0","",""};
lookup1("137B") ->	{"0","",""};
lookup1("137C") ->	{"0","",""};
lookup1("1380") ->	{"0","",""};
lookup1("1381") ->	{"0","",""};
lookup1("1382") ->	{"0","",""};
lookup1("1383") ->	{"0","",""};
lookup1("1384") ->	{"0","",""};
lookup1("1385") ->	{"0","",""};
lookup1("1386") ->	{"0","",""};
lookup1("1387") ->	{"0","",""};
lookup1("1388") ->	{"0","",""};
lookup1("1389") ->	{"0","",""};
lookup1("138A") ->	{"0","",""};
lookup1("138B") ->	{"0","",""};
lookup1("138C") ->	{"0","",""};
lookup1("138D") ->	{"0","",""};
lookup1("138E") ->	{"0","",""};
lookup1("138F") ->	{"0","",""};
lookup1("1390") ->	{"0","",""};
lookup1("1391") ->	{"0","",""};
lookup1("1392") ->	{"0","",""};
lookup1("1393") ->	{"0","",""};
lookup1("1394") ->	{"0","",""};
lookup1("1395") ->	{"0","",""};
lookup1("1396") ->	{"0","",""};
lookup1("1397") ->	{"0","",""};
lookup1("1398") ->	{"0","",""};
lookup1("1399") ->	{"0","",""};
lookup1("13A0") ->	{"0","",""};
lookup1("13A1") ->	{"0","",""};
lookup1("13A2") ->	{"0","",""};
lookup1("13A3") ->	{"0","",""};
lookup1("13A4") ->	{"0","",""};
lookup1("13A5") ->	{"0","",""};
lookup1("13A6") ->	{"0","",""};
lookup1("13A7") ->	{"0","",""};
lookup1("13A8") ->	{"0","",""};
lookup1("13A9") ->	{"0","",""};
lookup1("13AA") ->	{"0","",""};
lookup1("13AB") ->	{"0","",""};
lookup1("13AC") ->	{"0","",""};
lookup1("13AD") ->	{"0","",""};
lookup1("13AE") ->	{"0","",""};
lookup1("13AF") ->	{"0","",""};
lookup1("13B0") ->	{"0","",""};
lookup1("13B1") ->	{"0","",""};
lookup1("13B2") ->	{"0","",""};
lookup1("13B3") ->	{"0","",""};
lookup1("13B4") ->	{"0","",""};
lookup1("13B5") ->	{"0","",""};
lookup1("13B6") ->	{"0","",""};
lookup1("13B7") ->	{"0","",""};
lookup1("13B8") ->	{"0","",""};
lookup1("13B9") ->	{"0","",""};
lookup1("13BA") ->	{"0","",""};
lookup1("13BB") ->	{"0","",""};
lookup1("13BC") ->	{"0","",""};
lookup1("13BD") ->	{"0","",""};
lookup1("13BE") ->	{"0","",""};
lookup1("13BF") ->	{"0","",""};
lookup1("13C0") ->	{"0","",""};
lookup1("13C1") ->	{"0","",""};
lookup1("13C2") ->	{"0","",""};
lookup1("13C3") ->	{"0","",""};
lookup1("13C4") ->	{"0","",""};
lookup1("13C5") ->	{"0","",""};
lookup1("13C6") ->	{"0","",""};
lookup1("13C7") ->	{"0","",""};
lookup1("13C8") ->	{"0","",""};
lookup1("13C9") ->	{"0","",""};
lookup1("13CA") ->	{"0","",""};
lookup1("13CB") ->	{"0","",""};
lookup1("13CC") ->	{"0","",""};
lookup1("13CD") ->	{"0","",""};
lookup1("13CE") ->	{"0","",""};
lookup1("13CF") ->	{"0","",""};
lookup1("13D0") ->	{"0","",""};
lookup1("13D1") ->	{"0","",""};
lookup1("13D2") ->	{"0","",""};
lookup1("13D3") ->	{"0","",""};
lookup1("13D4") ->	{"0","",""};
lookup1("13D5") ->	{"0","",""};
lookup1("13D6") ->	{"0","",""};
lookup1("13D7") ->	{"0","",""};
lookup1("13D8") ->	{"0","",""};
lookup1("13D9") ->	{"0","",""};
lookup1("13DA") ->	{"0","",""};
lookup1("13DB") ->	{"0","",""};
lookup1("13DC") ->	{"0","",""};
lookup1("13DD") ->	{"0","",""};
lookup1("13DE") ->	{"0","",""};
lookup1("13DF") ->	{"0","",""};
lookup1("13E0") ->	{"0","",""};
lookup1("13E1") ->	{"0","",""};
lookup1("13E2") ->	{"0","",""};
lookup1("13E3") ->	{"0","",""};
lookup1("13E4") ->	{"0","",""};
lookup1("13E5") ->	{"0","",""};
lookup1("13E6") ->	{"0","",""};
lookup1("13E7") ->	{"0","",""};
lookup1("13E8") ->	{"0","",""};
lookup1("13E9") ->	{"0","",""};
lookup1("13EA") ->	{"0","",""};
lookup1("13EB") ->	{"0","",""};
lookup1("13EC") ->	{"0","",""};
lookup1("13ED") ->	{"0","",""};
lookup1("13EE") ->	{"0","",""};
lookup1("13EF") ->	{"0","",""};
lookup1("13F0") ->	{"0","",""};
lookup1("13F1") ->	{"0","",""};
lookup1("13F2") ->	{"0","",""};
lookup1("13F3") ->	{"0","",""};
lookup1("13F4") ->	{"0","",""};
lookup1("1400") ->	{"0","",""};
lookup1("1401") ->	{"0","",""};
lookup1("1402") ->	{"0","",""};
lookup1("1403") ->	{"0","",""};
lookup1("1404") ->	{"0","",""};
lookup1("1405") ->	{"0","",""};
lookup1("1406") ->	{"0","",""};
lookup1("1407") ->	{"0","",""};
lookup1("1408") ->	{"0","",""};
lookup1("1409") ->	{"0","",""};
lookup1("140A") ->	{"0","",""};
lookup1("140B") ->	{"0","",""};
lookup1("140C") ->	{"0","",""};
lookup1("140D") ->	{"0","",""};
lookup1("140E") ->	{"0","",""};
lookup1("140F") ->	{"0","",""};
lookup1("1410") ->	{"0","",""};
lookup1("1411") ->	{"0","",""};
lookup1("1412") ->	{"0","",""};
lookup1("1413") ->	{"0","",""};
lookup1("1414") ->	{"0","",""};
lookup1("1415") ->	{"0","",""};
lookup1("1416") ->	{"0","",""};
lookup1("1417") ->	{"0","",""};
lookup1("1418") ->	{"0","",""};
lookup1("1419") ->	{"0","",""};
lookup1("141A") ->	{"0","",""};
lookup1("141B") ->	{"0","",""};
lookup1("141C") ->	{"0","",""};
lookup1("141D") ->	{"0","",""};
lookup1("141E") ->	{"0","",""};
lookup1("141F") ->	{"0","",""};
lookup1("1420") ->	{"0","",""};
lookup1("1421") ->	{"0","",""};
lookup1("1422") ->	{"0","",""};
lookup1("1423") ->	{"0","",""};
lookup1("1424") ->	{"0","",""};
lookup1("1425") ->	{"0","",""};
lookup1("1426") ->	{"0","",""};
lookup1("1427") ->	{"0","",""};
lookup1("1428") ->	{"0","",""};
lookup1("1429") ->	{"0","",""};
lookup1("142A") ->	{"0","",""};
lookup1("142B") ->	{"0","",""};
lookup1("142C") ->	{"0","",""};
lookup1("142D") ->	{"0","",""};
lookup1("142E") ->	{"0","",""};
lookup1("142F") ->	{"0","",""};
lookup1("1430") ->	{"0","",""};
lookup1("1431") ->	{"0","",""};
lookup1("1432") ->	{"0","",""};
lookup1("1433") ->	{"0","",""};
lookup1("1434") ->	{"0","",""};
lookup1("1435") ->	{"0","",""};
lookup1("1436") ->	{"0","",""};
lookup1("1437") ->	{"0","",""};
lookup1("1438") ->	{"0","",""};
lookup1("1439") ->	{"0","",""};
lookup1("143A") ->	{"0","",""};
lookup1("143B") ->	{"0","",""};
lookup1("143C") ->	{"0","",""};
lookup1("143D") ->	{"0","",""};
lookup1("143E") ->	{"0","",""};
lookup1("143F") ->	{"0","",""};
lookup1("1440") ->	{"0","",""};
lookup1("1441") ->	{"0","",""};
lookup1("1442") ->	{"0","",""};
lookup1("1443") ->	{"0","",""};
lookup1("1444") ->	{"0","",""};
lookup1("1445") ->	{"0","",""};
lookup1("1446") ->	{"0","",""};
lookup1("1447") ->	{"0","",""};
lookup1("1448") ->	{"0","",""};
lookup1("1449") ->	{"0","",""};
lookup1("144A") ->	{"0","",""};
lookup1("144B") ->	{"0","",""};
lookup1("144C") ->	{"0","",""};
lookup1("144D") ->	{"0","",""};
lookup1("144E") ->	{"0","",""};
lookup1("144F") ->	{"0","",""};
lookup1("1450") ->	{"0","",""};
lookup1("1451") ->	{"0","",""};
lookup1("1452") ->	{"0","",""};
lookup1("1453") ->	{"0","",""};
lookup1("1454") ->	{"0","",""};
lookup1("1455") ->	{"0","",""};
lookup1("1456") ->	{"0","",""};
lookup1("1457") ->	{"0","",""};
lookup1("1458") ->	{"0","",""};
lookup1("1459") ->	{"0","",""};
lookup1("145A") ->	{"0","",""};
lookup1("145B") ->	{"0","",""};
lookup1("145C") ->	{"0","",""};
lookup1("145D") ->	{"0","",""};
lookup1("145E") ->	{"0","",""};
lookup1("145F") ->	{"0","",""};
lookup1("1460") ->	{"0","",""};
lookup1("1461") ->	{"0","",""};
lookup1("1462") ->	{"0","",""};
lookup1("1463") ->	{"0","",""};
lookup1("1464") ->	{"0","",""};
lookup1("1465") ->	{"0","",""};
lookup1("1466") ->	{"0","",""};
lookup1("1467") ->	{"0","",""};
lookup1("1468") ->	{"0","",""};
lookup1("1469") ->	{"0","",""};
lookup1("146A") ->	{"0","",""};
lookup1("146B") ->	{"0","",""};
lookup1("146C") ->	{"0","",""};
lookup1("146D") ->	{"0","",""};
lookup1("146E") ->	{"0","",""};
lookup1("146F") ->	{"0","",""};
lookup1("1470") ->	{"0","",""};
lookup1("1471") ->	{"0","",""};
lookup1("1472") ->	{"0","",""};
lookup1("1473") ->	{"0","",""};
lookup1("1474") ->	{"0","",""};
lookup1("1475") ->	{"0","",""};
lookup1("1476") ->	{"0","",""};
lookup1("1477") ->	{"0","",""};
lookup1("1478") ->	{"0","",""};
lookup1("1479") ->	{"0","",""};
lookup1("147A") ->	{"0","",""};
lookup1("147B") ->	{"0","",""};
lookup1("147C") ->	{"0","",""};
lookup1("147D") ->	{"0","",""};
lookup1("147E") ->	{"0","",""};
lookup1("147F") ->	{"0","",""};
lookup1("1480") ->	{"0","",""};
lookup1("1481") ->	{"0","",""};
lookup1("1482") ->	{"0","",""};
lookup1("1483") ->	{"0","",""};
lookup1("1484") ->	{"0","",""};
lookup1("1485") ->	{"0","",""};
lookup1("1486") ->	{"0","",""};
lookup1("1487") ->	{"0","",""};
lookup1("1488") ->	{"0","",""};
lookup1("1489") ->	{"0","",""};
lookup1("148A") ->	{"0","",""};
lookup1("148B") ->	{"0","",""};
lookup1("148C") ->	{"0","",""};
lookup1("148D") ->	{"0","",""};
lookup1("148E") ->	{"0","",""};
lookup1("148F") ->	{"0","",""};
lookup1("1490") ->	{"0","",""};
lookup1("1491") ->	{"0","",""};
lookup1("1492") ->	{"0","",""};
lookup1("1493") ->	{"0","",""};
lookup1("1494") ->	{"0","",""};
lookup1("1495") ->	{"0","",""};
lookup1("1496") ->	{"0","",""};
lookup1("1497") ->	{"0","",""};
lookup1("1498") ->	{"0","",""};
lookup1("1499") ->	{"0","",""};
lookup1("149A") ->	{"0","",""};
lookup1("149B") ->	{"0","",""};
lookup1("149C") ->	{"0","",""};
lookup1("149D") ->	{"0","",""};
lookup1("149E") ->	{"0","",""};
lookup1("149F") ->	{"0","",""};
lookup1("14A0") ->	{"0","",""};
lookup1("14A1") ->	{"0","",""};
lookup1("14A2") ->	{"0","",""};
lookup1("14A3") ->	{"0","",""};
lookup1("14A4") ->	{"0","",""};
lookup1("14A5") ->	{"0","",""};
lookup1("14A6") ->	{"0","",""};
lookup1("14A7") ->	{"0","",""};
lookup1("14A8") ->	{"0","",""};
lookup1("14A9") ->	{"0","",""};
lookup1("14AA") ->	{"0","",""};
lookup1("14AB") ->	{"0","",""};
lookup1("14AC") ->	{"0","",""};
lookup1("14AD") ->	{"0","",""};
lookup1("14AE") ->	{"0","",""};
lookup1("14AF") ->	{"0","",""};
lookup1("14B0") ->	{"0","",""};
lookup1("14B1") ->	{"0","",""};
lookup1("14B2") ->	{"0","",""};
lookup1("14B3") ->	{"0","",""};
lookup1("14B4") ->	{"0","",""};
lookup1("14B5") ->	{"0","",""};
lookup1("14B6") ->	{"0","",""};
lookup1("14B7") ->	{"0","",""};
lookup1("14B8") ->	{"0","",""};
lookup1("14B9") ->	{"0","",""};
lookup1("14BA") ->	{"0","",""};
lookup1("14BB") ->	{"0","",""};
lookup1("14BC") ->	{"0","",""};
lookup1("14BD") ->	{"0","",""};
lookup1("14BE") ->	{"0","",""};
lookup1("14BF") ->	{"0","",""};
lookup1("14C0") ->	{"0","",""};
lookup1("14C1") ->	{"0","",""};
lookup1("14C2") ->	{"0","",""};
lookup1("14C3") ->	{"0","",""};
lookup1("14C4") ->	{"0","",""};
lookup1("14C5") ->	{"0","",""};
lookup1("14C6") ->	{"0","",""};
lookup1("14C7") ->	{"0","",""};
lookup1("14C8") ->	{"0","",""};
lookup1("14C9") ->	{"0","",""};
lookup1("14CA") ->	{"0","",""};
lookup1("14CB") ->	{"0","",""};
lookup1("14CC") ->	{"0","",""};
lookup1("14CD") ->	{"0","",""};
lookup1("14CE") ->	{"0","",""};
lookup1("14CF") ->	{"0","",""};
lookup1("14D0") ->	{"0","",""};
lookup1("14D1") ->	{"0","",""};
lookup1("14D2") ->	{"0","",""};
lookup1("14D3") ->	{"0","",""};
lookup1("14D4") ->	{"0","",""};
lookup1("14D5") ->	{"0","",""};
lookup1("14D6") ->	{"0","",""};
lookup1("14D7") ->	{"0","",""};
lookup1("14D8") ->	{"0","",""};
lookup1("14D9") ->	{"0","",""};
lookup1("14DA") ->	{"0","",""};
lookup1("14DB") ->	{"0","",""};
lookup1("14DC") ->	{"0","",""};
lookup1("14DD") ->	{"0","",""};
lookup1("14DE") ->	{"0","",""};
lookup1("14DF") ->	{"0","",""};
lookup1("14E0") ->	{"0","",""};
lookup1("14E1") ->	{"0","",""};
lookup1("14E2") ->	{"0","",""};
lookup1("14E3") ->	{"0","",""};
lookup1("14E4") ->	{"0","",""};
lookup1("14E5") ->	{"0","",""};
lookup1("14E6") ->	{"0","",""};
lookup1("14E7") ->	{"0","",""};
lookup1("14E8") ->	{"0","",""};
lookup1("14E9") ->	{"0","",""};
lookup1("14EA") ->	{"0","",""};
lookup1("14EB") ->	{"0","",""};
lookup1("14EC") ->	{"0","",""};
lookup1("14ED") ->	{"0","",""};
lookup1("14EE") ->	{"0","",""};
lookup1("14EF") ->	{"0","",""};
lookup1("14F0") ->	{"0","",""};
lookup1("14F1") ->	{"0","",""};
lookup1("14F2") ->	{"0","",""};
lookup1("14F3") ->	{"0","",""};
lookup1("14F4") ->	{"0","",""};
lookup1("14F5") ->	{"0","",""};
lookup1("14F6") ->	{"0","",""};
lookup1("14F7") ->	{"0","",""};
lookup1("14F8") ->	{"0","",""};
lookup1("14F9") ->	{"0","",""};
lookup1("14FA") ->	{"0","",""};
lookup1("14FB") ->	{"0","",""};
lookup1("14FC") ->	{"0","",""};
lookup1("14FD") ->	{"0","",""};
lookup1("14FE") ->	{"0","",""};
lookup1("14FF") ->	{"0","",""};
lookup1("1500") ->	{"0","",""};
lookup1("1501") ->	{"0","",""};
lookup1("1502") ->	{"0","",""};
lookup1("1503") ->	{"0","",""};
lookup1("1504") ->	{"0","",""};
lookup1("1505") ->	{"0","",""};
lookup1("1506") ->	{"0","",""};
lookup1("1507") ->	{"0","",""};
lookup1("1508") ->	{"0","",""};
lookup1("1509") ->	{"0","",""};
lookup1("150A") ->	{"0","",""};
lookup1("150B") ->	{"0","",""};
lookup1("150C") ->	{"0","",""};
lookup1("150D") ->	{"0","",""};
lookup1("150E") ->	{"0","",""};
lookup1("150F") ->	{"0","",""};
lookup1("1510") ->	{"0","",""};
lookup1("1511") ->	{"0","",""};
lookup1("1512") ->	{"0","",""};
lookup1("1513") ->	{"0","",""};
lookup1("1514") ->	{"0","",""};
lookup1("1515") ->	{"0","",""};
lookup1("1516") ->	{"0","",""};
lookup1("1517") ->	{"0","",""};
lookup1("1518") ->	{"0","",""};
lookup1("1519") ->	{"0","",""};
lookup1("151A") ->	{"0","",""};
lookup1("151B") ->	{"0","",""};
lookup1("151C") ->	{"0","",""};
lookup1("151D") ->	{"0","",""};
lookup1("151E") ->	{"0","",""};
lookup1("151F") ->	{"0","",""};
lookup1("1520") ->	{"0","",""};
lookup1("1521") ->	{"0","",""};
lookup1("1522") ->	{"0","",""};
lookup1("1523") ->	{"0","",""};
lookup1("1524") ->	{"0","",""};
lookup1("1525") ->	{"0","",""};
lookup1("1526") ->	{"0","",""};
lookup1("1527") ->	{"0","",""};
lookup1("1528") ->	{"0","",""};
lookup1("1529") ->	{"0","",""};
lookup1("152A") ->	{"0","",""};
lookup1("152B") ->	{"0","",""};
lookup1("152C") ->	{"0","",""};
lookup1("152D") ->	{"0","",""};
lookup1("152E") ->	{"0","",""};
lookup1("152F") ->	{"0","",""};
lookup1("1530") ->	{"0","",""};
lookup1("1531") ->	{"0","",""};
lookup1("1532") ->	{"0","",""};
lookup1("1533") ->	{"0","",""};
lookup1("1534") ->	{"0","",""};
lookup1("1535") ->	{"0","",""};
lookup1("1536") ->	{"0","",""};
lookup1("1537") ->	{"0","",""};
lookup1("1538") ->	{"0","",""};
lookup1("1539") ->	{"0","",""};
lookup1("153A") ->	{"0","",""};
lookup1("153B") ->	{"0","",""};
lookup1("153C") ->	{"0","",""};
lookup1("153D") ->	{"0","",""};
lookup1("153E") ->	{"0","",""};
lookup1("153F") ->	{"0","",""};
lookup1("1540") ->	{"0","",""};
lookup1("1541") ->	{"0","",""};
lookup1("1542") ->	{"0","",""};
lookup1("1543") ->	{"0","",""};
lookup1("1544") ->	{"0","",""};
lookup1("1545") ->	{"0","",""};
lookup1("1546") ->	{"0","",""};
lookup1("1547") ->	{"0","",""};
lookup1("1548") ->	{"0","",""};
lookup1("1549") ->	{"0","",""};
lookup1("154A") ->	{"0","",""};
lookup1("154B") ->	{"0","",""};
lookup1("154C") ->	{"0","",""};
lookup1("154D") ->	{"0","",""};
lookup1("154E") ->	{"0","",""};
lookup1("154F") ->	{"0","",""};
lookup1("1550") ->	{"0","",""};
lookup1("1551") ->	{"0","",""};
lookup1("1552") ->	{"0","",""};
lookup1("1553") ->	{"0","",""};
lookup1("1554") ->	{"0","",""};
lookup1("1555") ->	{"0","",""};
lookup1("1556") ->	{"0","",""};
lookup1("1557") ->	{"0","",""};
lookup1("1558") ->	{"0","",""};
lookup1("1559") ->	{"0","",""};
lookup1("155A") ->	{"0","",""};
lookup1("155B") ->	{"0","",""};
lookup1("155C") ->	{"0","",""};
lookup1("155D") ->	{"0","",""};
lookup1("155E") ->	{"0","",""};
lookup1("155F") ->	{"0","",""};
lookup1("1560") ->	{"0","",""};
lookup1("1561") ->	{"0","",""};
lookup1("1562") ->	{"0","",""};
lookup1("1563") ->	{"0","",""};
lookup1("1564") ->	{"0","",""};
lookup1("1565") ->	{"0","",""};
lookup1("1566") ->	{"0","",""};
lookup1("1567") ->	{"0","",""};
lookup1("1568") ->	{"0","",""};
lookup1("1569") ->	{"0","",""};
lookup1("156A") ->	{"0","",""};
lookup1("156B") ->	{"0","",""};
lookup1("156C") ->	{"0","",""};
lookup1("156D") ->	{"0","",""};
lookup1("156E") ->	{"0","",""};
lookup1("156F") ->	{"0","",""};
lookup1("1570") ->	{"0","",""};
lookup1("1571") ->	{"0","",""};
lookup1("1572") ->	{"0","",""};
lookup1("1573") ->	{"0","",""};
lookup1("1574") ->	{"0","",""};
lookup1("1575") ->	{"0","",""};
lookup1("1576") ->	{"0","",""};
lookup1("1577") ->	{"0","",""};
lookup1("1578") ->	{"0","",""};
lookup1("1579") ->	{"0","",""};
lookup1("157A") ->	{"0","",""};
lookup1("157B") ->	{"0","",""};
lookup1("157C") ->	{"0","",""};
lookup1("157D") ->	{"0","",""};
lookup1("157E") ->	{"0","",""};
lookup1("157F") ->	{"0","",""};
lookup1("1580") ->	{"0","",""};
lookup1("1581") ->	{"0","",""};
lookup1("1582") ->	{"0","",""};
lookup1("1583") ->	{"0","",""};
lookup1("1584") ->	{"0","",""};
lookup1("1585") ->	{"0","",""};
lookup1("1586") ->	{"0","",""};
lookup1("1587") ->	{"0","",""};
lookup1("1588") ->	{"0","",""};
lookup1("1589") ->	{"0","",""};
lookup1("158A") ->	{"0","",""};
lookup1("158B") ->	{"0","",""};
lookup1("158C") ->	{"0","",""};
lookup1("158D") ->	{"0","",""};
lookup1("158E") ->	{"0","",""};
lookup1("158F") ->	{"0","",""};
lookup1("1590") ->	{"0","",""};
lookup1("1591") ->	{"0","",""};
lookup1("1592") ->	{"0","",""};
lookup1("1593") ->	{"0","",""};
lookup1("1594") ->	{"0","",""};
lookup1("1595") ->	{"0","",""};
lookup1("1596") ->	{"0","",""};
lookup1("1597") ->	{"0","",""};
lookup1("1598") ->	{"0","",""};
lookup1("1599") ->	{"0","",""};
lookup1("159A") ->	{"0","",""};
lookup1("159B") ->	{"0","",""};
lookup1("159C") ->	{"0","",""};
lookup1("159D") ->	{"0","",""};
lookup1("159E") ->	{"0","",""};
lookup1("159F") ->	{"0","",""};
lookup1("15A0") ->	{"0","",""};
lookup1("15A1") ->	{"0","",""};
lookup1("15A2") ->	{"0","",""};
lookup1("15A3") ->	{"0","",""};
lookup1("15A4") ->	{"0","",""};
lookup1("15A5") ->	{"0","",""};
lookup1("15A6") ->	{"0","",""};
lookup1("15A7") ->	{"0","",""};
lookup1("15A8") ->	{"0","",""};
lookup1("15A9") ->	{"0","",""};
lookup1("15AA") ->	{"0","",""};
lookup1("15AB") ->	{"0","",""};
lookup1("15AC") ->	{"0","",""};
lookup1("15AD") ->	{"0","",""};
lookup1("15AE") ->	{"0","",""};
lookup1("15AF") ->	{"0","",""};
lookup1("15B0") ->	{"0","",""};
lookup1("15B1") ->	{"0","",""};
lookup1("15B2") ->	{"0","",""};
lookup1("15B3") ->	{"0","",""};
lookup1("15B4") ->	{"0","",""};
lookup1("15B5") ->	{"0","",""};
lookup1("15B6") ->	{"0","",""};
lookup1("15B7") ->	{"0","",""};
lookup1("15B8") ->	{"0","",""};
lookup1("15B9") ->	{"0","",""};
lookup1("15BA") ->	{"0","",""};
lookup1("15BB") ->	{"0","",""};
lookup1("15BC") ->	{"0","",""};
lookup1("15BD") ->	{"0","",""};
lookup1("15BE") ->	{"0","",""};
lookup1("15BF") ->	{"0","",""};
lookup1("15C0") ->	{"0","",""};
lookup1("15C1") ->	{"0","",""};
lookup1("15C2") ->	{"0","",""};
lookup1("15C3") ->	{"0","",""};
lookup1("15C4") ->	{"0","",""};
lookup1("15C5") ->	{"0","",""};
lookup1("15C6") ->	{"0","",""};
lookup1("15C7") ->	{"0","",""};
lookup1("15C8") ->	{"0","",""};
lookup1("15C9") ->	{"0","",""};
lookup1("15CA") ->	{"0","",""};
lookup1("15CB") ->	{"0","",""};
lookup1("15CC") ->	{"0","",""};
lookup1("15CD") ->	{"0","",""};
lookup1("15CE") ->	{"0","",""};
lookup1("15CF") ->	{"0","",""};
lookup1("15D0") ->	{"0","",""};
lookup1("15D1") ->	{"0","",""};
lookup1("15D2") ->	{"0","",""};
lookup1("15D3") ->	{"0","",""};
lookup1("15D4") ->	{"0","",""};
lookup1("15D5") ->	{"0","",""};
lookup1("15D6") ->	{"0","",""};
lookup1("15D7") ->	{"0","",""};
lookup1("15D8") ->	{"0","",""};
lookup1("15D9") ->	{"0","",""};
lookup1("15DA") ->	{"0","",""};
lookup1("15DB") ->	{"0","",""};
lookup1("15DC") ->	{"0","",""};
lookup1("15DD") ->	{"0","",""};
lookup1("15DE") ->	{"0","",""};
lookup1("15DF") ->	{"0","",""};
lookup1("15E0") ->	{"0","",""};
lookup1("15E1") ->	{"0","",""};
lookup1("15E2") ->	{"0","",""};
lookup1("15E3") ->	{"0","",""};
lookup1("15E4") ->	{"0","",""};
lookup1("15E5") ->	{"0","",""};
lookup1("15E6") ->	{"0","",""};
lookup1("15E7") ->	{"0","",""};
lookup1("15E8") ->	{"0","",""};
lookup1("15E9") ->	{"0","",""};
lookup1("15EA") ->	{"0","",""};
lookup1("15EB") ->	{"0","",""};
lookup1("15EC") ->	{"0","",""};
lookup1("15ED") ->	{"0","",""};
lookup1("15EE") ->	{"0","",""};
lookup1("15EF") ->	{"0","",""};
lookup1("15F0") ->	{"0","",""};
lookup1("15F1") ->	{"0","",""};
lookup1("15F2") ->	{"0","",""};
lookup1("15F3") ->	{"0","",""};
lookup1("15F4") ->	{"0","",""};
lookup1("15F5") ->	{"0","",""};
lookup1("15F6") ->	{"0","",""};
lookup1("15F7") ->	{"0","",""};
lookup1("15F8") ->	{"0","",""};
lookup1("15F9") ->	{"0","",""};
lookup1("15FA") ->	{"0","",""};
lookup1("15FB") ->	{"0","",""};
lookup1("15FC") ->	{"0","",""};
lookup1("15FD") ->	{"0","",""};
lookup1("15FE") ->	{"0","",""};
lookup1("15FF") ->	{"0","",""};
lookup1("1600") ->	{"0","",""};
lookup1("1601") ->	{"0","",""};
lookup1("1602") ->	{"0","",""};
lookup1("1603") ->	{"0","",""};
lookup1("1604") ->	{"0","",""};
lookup1("1605") ->	{"0","",""};
lookup1("1606") ->	{"0","",""};
lookup1("1607") ->	{"0","",""};
lookup1("1608") ->	{"0","",""};
lookup1("1609") ->	{"0","",""};
lookup1("160A") ->	{"0","",""};
lookup1("160B") ->	{"0","",""};
lookup1("160C") ->	{"0","",""};
lookup1("160D") ->	{"0","",""};
lookup1("160E") ->	{"0","",""};
lookup1("160F") ->	{"0","",""};
lookup1("1610") ->	{"0","",""};
lookup1("1611") ->	{"0","",""};
lookup1("1612") ->	{"0","",""};
lookup1("1613") ->	{"0","",""};
lookup1("1614") ->	{"0","",""};
lookup1("1615") ->	{"0","",""};
lookup1("1616") ->	{"0","",""};
lookup1("1617") ->	{"0","",""};
lookup1("1618") ->	{"0","",""};
lookup1("1619") ->	{"0","",""};
lookup1("161A") ->	{"0","",""};
lookup1("161B") ->	{"0","",""};
lookup1("161C") ->	{"0","",""};
lookup1("161D") ->	{"0","",""};
lookup1("161E") ->	{"0","",""};
lookup1("161F") ->	{"0","",""};
lookup1("1620") ->	{"0","",""};
lookup1("1621") ->	{"0","",""};
lookup1("1622") ->	{"0","",""};
lookup1("1623") ->	{"0","",""};
lookup1("1624") ->	{"0","",""};
lookup1("1625") ->	{"0","",""};
lookup1("1626") ->	{"0","",""};
lookup1("1627") ->	{"0","",""};
lookup1("1628") ->	{"0","",""};
lookup1("1629") ->	{"0","",""};
lookup1("162A") ->	{"0","",""};
lookup1("162B") ->	{"0","",""};
lookup1("162C") ->	{"0","",""};
lookup1("162D") ->	{"0","",""};
lookup1("162E") ->	{"0","",""};
lookup1("162F") ->	{"0","",""};
lookup1("1630") ->	{"0","",""};
lookup1("1631") ->	{"0","",""};
lookup1("1632") ->	{"0","",""};
lookup1("1633") ->	{"0","",""};
lookup1("1634") ->	{"0","",""};
lookup1("1635") ->	{"0","",""};
lookup1("1636") ->	{"0","",""};
lookup1("1637") ->	{"0","",""};
lookup1("1638") ->	{"0","",""};
lookup1("1639") ->	{"0","",""};
lookup1("163A") ->	{"0","",""};
lookup1("163B") ->	{"0","",""};
lookup1("163C") ->	{"0","",""};
lookup1("163D") ->	{"0","",""};
lookup1("163E") ->	{"0","",""};
lookup1("163F") ->	{"0","",""};
lookup1("1640") ->	{"0","",""};
lookup1("1641") ->	{"0","",""};
lookup1("1642") ->	{"0","",""};
lookup1("1643") ->	{"0","",""};
lookup1("1644") ->	{"0","",""};
lookup1("1645") ->	{"0","",""};
lookup1("1646") ->	{"0","",""};
lookup1("1647") ->	{"0","",""};
lookup1("1648") ->	{"0","",""};
lookup1("1649") ->	{"0","",""};
lookup1("164A") ->	{"0","",""};
lookup1("164B") ->	{"0","",""};
lookup1("164C") ->	{"0","",""};
lookup1("164D") ->	{"0","",""};
lookup1("164E") ->	{"0","",""};
lookup1("164F") ->	{"0","",""};
lookup1("1650") ->	{"0","",""};
lookup1("1651") ->	{"0","",""};
lookup1("1652") ->	{"0","",""};
lookup1("1653") ->	{"0","",""};
lookup1("1654") ->	{"0","",""};
lookup1("1655") ->	{"0","",""};
lookup1("1656") ->	{"0","",""};
lookup1("1657") ->	{"0","",""};
lookup1("1658") ->	{"0","",""};
lookup1("1659") ->	{"0","",""};
lookup1("165A") ->	{"0","",""};
lookup1("165B") ->	{"0","",""};
lookup1("165C") ->	{"0","",""};
lookup1("165D") ->	{"0","",""};
lookup1("165E") ->	{"0","",""};
lookup1("165F") ->	{"0","",""};
lookup1("1660") ->	{"0","",""};
lookup1("1661") ->	{"0","",""};
lookup1("1662") ->	{"0","",""};
lookup1("1663") ->	{"0","",""};
lookup1("1664") ->	{"0","",""};
lookup1("1665") ->	{"0","",""};
lookup1("1666") ->	{"0","",""};
lookup1("1667") ->	{"0","",""};
lookup1("1668") ->	{"0","",""};
lookup1("1669") ->	{"0","",""};
lookup1("166A") ->	{"0","",""};
lookup1("166B") ->	{"0","",""};
lookup1("166C") ->	{"0","",""};
lookup1("166D") ->	{"0","",""};
lookup1("166E") ->	{"0","",""};
lookup1("166F") ->	{"0","",""};
lookup1("1670") ->	{"0","",""};
lookup1("1671") ->	{"0","",""};
lookup1("1672") ->	{"0","",""};
lookup1("1673") ->	{"0","",""};
lookup1("1674") ->	{"0","",""};
lookup1("1675") ->	{"0","",""};
lookup1("1676") ->	{"0","",""};
lookup1("1677") ->	{"0","",""};
lookup1("1678") ->	{"0","",""};
lookup1("1679") ->	{"0","",""};
lookup1("167A") ->	{"0","",""};
lookup1("167B") ->	{"0","",""};
lookup1("167C") ->	{"0","",""};
lookup1("167D") ->	{"0","",""};
lookup1("167E") ->	{"0","",""};
lookup1("167F") ->	{"0","",""};
lookup1("1680") ->	{"0","",""};
lookup1("16800") ->	{"0","",""};
lookup1("16801") ->	{"0","",""};
lookup1("16802") ->	{"0","",""};
lookup1("16803") ->	{"0","",""};
lookup1("16804") ->	{"0","",""};
lookup1("16805") ->	{"0","",""};
lookup1("16806") ->	{"0","",""};
lookup1("16807") ->	{"0","",""};
lookup1("16808") ->	{"0","",""};
lookup1("16809") ->	{"0","",""};
lookup1("1680A") ->	{"0","",""};
lookup1("1680B") ->	{"0","",""};
lookup1("1680C") ->	{"0","",""};
lookup1("1680D") ->	{"0","",""};
lookup1("1680E") ->	{"0","",""};
lookup1("1680F") ->	{"0","",""};
lookup1("1681") ->	{"0","",""};
lookup1("16810") ->	{"0","",""};
lookup1("16811") ->	{"0","",""};
lookup1("16812") ->	{"0","",""};
lookup1("16813") ->	{"0","",""};
lookup1("16814") ->	{"0","",""};
lookup1("16815") ->	{"0","",""};
lookup1("16816") ->	{"0","",""};
lookup1("16817") ->	{"0","",""};
lookup1("16818") ->	{"0","",""};
lookup1("16819") ->	{"0","",""};
lookup1("1681A") ->	{"0","",""};
lookup1("1681B") ->	{"0","",""};
lookup1("1681C") ->	{"0","",""};
lookup1("1681D") ->	{"0","",""};
lookup1("1681E") ->	{"0","",""};
lookup1("1681F") ->	{"0","",""};
lookup1("1682") ->	{"0","",""};
lookup1("16820") ->	{"0","",""};
lookup1("16821") ->	{"0","",""};
lookup1("16822") ->	{"0","",""};
lookup1("16823") ->	{"0","",""};
lookup1("16824") ->	{"0","",""};
lookup1("16825") ->	{"0","",""};
lookup1("16826") ->	{"0","",""};
lookup1("16827") ->	{"0","",""};
lookup1("16828") ->	{"0","",""};
lookup1("16829") ->	{"0","",""};
lookup1("1682A") ->	{"0","",""};
lookup1("1682B") ->	{"0","",""};
lookup1("1682C") ->	{"0","",""};
lookup1("1682D") ->	{"0","",""};
lookup1("1682E") ->	{"0","",""};
lookup1("1682F") ->	{"0","",""};
lookup1("1683") ->	{"0","",""};
lookup1("16830") ->	{"0","",""};
lookup1("16831") ->	{"0","",""};
lookup1("16832") ->	{"0","",""};
lookup1("16833") ->	{"0","",""};
lookup1("16834") ->	{"0","",""};
lookup1("16835") ->	{"0","",""};
lookup1("16836") ->	{"0","",""};
lookup1("16837") ->	{"0","",""};
lookup1("16838") ->	{"0","",""};
lookup1("16839") ->	{"0","",""};
lookup1("1683A") ->	{"0","",""};
lookup1("1683B") ->	{"0","",""};
lookup1("1683C") ->	{"0","",""};
lookup1("1683D") ->	{"0","",""};
lookup1("1683E") ->	{"0","",""};
lookup1("1683F") ->	{"0","",""};
lookup1("1684") ->	{"0","",""};
lookup1("16840") ->	{"0","",""};
lookup1("16841") ->	{"0","",""};
lookup1("16842") ->	{"0","",""};
lookup1("16843") ->	{"0","",""};
lookup1("16844") ->	{"0","",""};
lookup1("16845") ->	{"0","",""};
lookup1("16846") ->	{"0","",""};
lookup1("16847") ->	{"0","",""};
lookup1("16848") ->	{"0","",""};
lookup1("16849") ->	{"0","",""};
lookup1("1684A") ->	{"0","",""};
lookup1("1684B") ->	{"0","",""};
lookup1("1684C") ->	{"0","",""};
lookup1("1684D") ->	{"0","",""};
lookup1("1684E") ->	{"0","",""};
lookup1("1684F") ->	{"0","",""};
lookup1("1685") ->	{"0","",""};
lookup1("16850") ->	{"0","",""};
lookup1("16851") ->	{"0","",""};
lookup1("16852") ->	{"0","",""};
lookup1("16853") ->	{"0","",""};
lookup1("16854") ->	{"0","",""};
lookup1("16855") ->	{"0","",""};
lookup1("16856") ->	{"0","",""};
lookup1("16857") ->	{"0","",""};
lookup1("16858") ->	{"0","",""};
lookup1("16859") ->	{"0","",""};
lookup1("1685A") ->	{"0","",""};
lookup1("1685B") ->	{"0","",""};
lookup1("1685C") ->	{"0","",""};
lookup1("1685D") ->	{"0","",""};
lookup1("1685E") ->	{"0","",""};
lookup1("1685F") ->	{"0","",""};
lookup1("1686") ->	{"0","",""};
lookup1("16860") ->	{"0","",""};
lookup1("16861") ->	{"0","",""};
lookup1("16862") ->	{"0","",""};
lookup1("16863") ->	{"0","",""};
lookup1("16864") ->	{"0","",""};
lookup1("16865") ->	{"0","",""};
lookup1("16866") ->	{"0","",""};
lookup1("16867") ->	{"0","",""};
lookup1("16868") ->	{"0","",""};
lookup1("16869") ->	{"0","",""};
lookup1("1686A") ->	{"0","",""};
lookup1("1686B") ->	{"0","",""};
lookup1("1686C") ->	{"0","",""};
lookup1("1686D") ->	{"0","",""};
lookup1("1686E") ->	{"0","",""};
lookup1("1686F") ->	{"0","",""};
lookup1("1687") ->	{"0","",""};
lookup1("16870") ->	{"0","",""};
lookup1("16871") ->	{"0","",""};
lookup1("16872") ->	{"0","",""};
lookup1("16873") ->	{"0","",""};
lookup1("16874") ->	{"0","",""};
lookup1("16875") ->	{"0","",""};
lookup1("16876") ->	{"0","",""};
lookup1("16877") ->	{"0","",""};
lookup1("16878") ->	{"0","",""};
lookup1("16879") ->	{"0","",""};
lookup1("1687A") ->	{"0","",""};
lookup1("1687B") ->	{"0","",""};
lookup1("1687C") ->	{"0","",""};
lookup1("1687D") ->	{"0","",""};
lookup1("1687E") ->	{"0","",""};
lookup1("1687F") ->	{"0","",""};
lookup1("1688") ->	{"0","",""};
lookup1("16880") ->	{"0","",""};
lookup1("16881") ->	{"0","",""};
lookup1("16882") ->	{"0","",""};
lookup1("16883") ->	{"0","",""};
lookup1("16884") ->	{"0","",""};
lookup1("16885") ->	{"0","",""};
lookup1("16886") ->	{"0","",""};
lookup1("16887") ->	{"0","",""};
lookup1("16888") ->	{"0","",""};
lookup1("16889") ->	{"0","",""};
lookup1("1688A") ->	{"0","",""};
lookup1("1688B") ->	{"0","",""};
lookup1("1688C") ->	{"0","",""};
lookup1("1688D") ->	{"0","",""};
lookup1("1688E") ->	{"0","",""};
lookup1("1688F") ->	{"0","",""};
lookup1("1689") ->	{"0","",""};
lookup1("16890") ->	{"0","",""};
lookup1("16891") ->	{"0","",""};
lookup1("16892") ->	{"0","",""};
lookup1("16893") ->	{"0","",""};
lookup1("16894") ->	{"0","",""};
lookup1("16895") ->	{"0","",""};
lookup1("16896") ->	{"0","",""};
lookup1("16897") ->	{"0","",""};
lookup1("16898") ->	{"0","",""};
lookup1("16899") ->	{"0","",""};
lookup1("1689A") ->	{"0","",""};
lookup1("1689B") ->	{"0","",""};
lookup1("1689C") ->	{"0","",""};
lookup1("1689D") ->	{"0","",""};
lookup1("1689E") ->	{"0","",""};
lookup1("1689F") ->	{"0","",""};
lookup1("168A") ->	{"0","",""};
lookup1("168A0") ->	{"0","",""};
lookup1("168A1") ->	{"0","",""};
lookup1("168A2") ->	{"0","",""};
lookup1("168A3") ->	{"0","",""};
lookup1("168A4") ->	{"0","",""};
lookup1("168A5") ->	{"0","",""};
lookup1("168A6") ->	{"0","",""};
lookup1("168A7") ->	{"0","",""};
lookup1("168A8") ->	{"0","",""};
lookup1("168A9") ->	{"0","",""};
lookup1("168AA") ->	{"0","",""};
lookup1("168AB") ->	{"0","",""};
lookup1("168AC") ->	{"0","",""};
lookup1("168AD") ->	{"0","",""};
lookup1("168AE") ->	{"0","",""};
lookup1("168AF") ->	{"0","",""};
lookup1("168B") ->	{"0","",""};
lookup1("168B0") ->	{"0","",""};
lookup1("168B1") ->	{"0","",""};
lookup1("168B2") ->	{"0","",""};
lookup1("168B3") ->	{"0","",""};
lookup1("168B4") ->	{"0","",""};
lookup1("168B5") ->	{"0","",""};
lookup1("168B6") ->	{"0","",""};
lookup1("168B7") ->	{"0","",""};
lookup1("168B8") ->	{"0","",""};
lookup1("168B9") ->	{"0","",""};
lookup1("168BA") ->	{"0","",""};
lookup1("168BB") ->	{"0","",""};
lookup1("168BC") ->	{"0","",""};
lookup1("168BD") ->	{"0","",""};
lookup1("168BE") ->	{"0","",""};
lookup1("168BF") ->	{"0","",""};
lookup1("168C") ->	{"0","",""};
lookup1("168C0") ->	{"0","",""};
lookup1("168C1") ->	{"0","",""};
lookup1("168C2") ->	{"0","",""};
lookup1("168C3") ->	{"0","",""};
lookup1("168C4") ->	{"0","",""};
lookup1("168C5") ->	{"0","",""};
lookup1("168C6") ->	{"0","",""};
lookup1("168C7") ->	{"0","",""};
lookup1("168C8") ->	{"0","",""};
lookup1("168C9") ->	{"0","",""};
lookup1("168CA") ->	{"0","",""};
lookup1("168CB") ->	{"0","",""};
lookup1("168CC") ->	{"0","",""};
lookup1("168CD") ->	{"0","",""};
lookup1("168CE") ->	{"0","",""};
lookup1("168CF") ->	{"0","",""};
lookup1("168D") ->	{"0","",""};
lookup1("168D0") ->	{"0","",""};
lookup1("168D1") ->	{"0","",""};
lookup1("168D2") ->	{"0","",""};
lookup1("168D3") ->	{"0","",""};
lookup1("168D4") ->	{"0","",""};
lookup1("168D5") ->	{"0","",""};
lookup1("168D6") ->	{"0","",""};
lookup1("168D7") ->	{"0","",""};
lookup1("168D8") ->	{"0","",""};
lookup1("168D9") ->	{"0","",""};
lookup1("168DA") ->	{"0","",""};
lookup1("168DB") ->	{"0","",""};
lookup1("168DC") ->	{"0","",""};
lookup1("168DD") ->	{"0","",""};
lookup1("168DE") ->	{"0","",""};
lookup1("168DF") ->	{"0","",""};
lookup1("168E") ->	{"0","",""};
lookup1("168E0") ->	{"0","",""};
lookup1("168E1") ->	{"0","",""};
lookup1("168E2") ->	{"0","",""};
lookup1("168E3") ->	{"0","",""};
lookup1("168E4") ->	{"0","",""};
lookup1("168E5") ->	{"0","",""};
lookup1("168E6") ->	{"0","",""};
lookup1("168E7") ->	{"0","",""};
lookup1("168E8") ->	{"0","",""};
lookup1("168E9") ->	{"0","",""};
lookup1("168EA") ->	{"0","",""};
lookup1("168EB") ->	{"0","",""};
lookup1("168EC") ->	{"0","",""};
lookup1("168ED") ->	{"0","",""};
lookup1("168EE") ->	{"0","",""};
lookup1("168EF") ->	{"0","",""};
lookup1("168F") ->	{"0","",""};
lookup1("168F0") ->	{"0","",""};
lookup1("168F1") ->	{"0","",""};
lookup1("168F2") ->	{"0","",""};
lookup1("168F3") ->	{"0","",""};
lookup1("168F4") ->	{"0","",""};
lookup1("168F5") ->	{"0","",""};
lookup1("168F6") ->	{"0","",""};
lookup1("168F7") ->	{"0","",""};
lookup1("168F8") ->	{"0","",""};
lookup1("168F9") ->	{"0","",""};
lookup1("168FA") ->	{"0","",""};
lookup1("168FB") ->	{"0","",""};
lookup1("168FC") ->	{"0","",""};
lookup1("168FD") ->	{"0","",""};
lookup1("168FE") ->	{"0","",""};
lookup1("168FF") ->	{"0","",""};
lookup1("1690") ->	{"0","",""};
lookup1("16900") ->	{"0","",""};
lookup1("16901") ->	{"0","",""};
lookup1("16902") ->	{"0","",""};
lookup1("16903") ->	{"0","",""};
lookup1("16904") ->	{"0","",""};
lookup1("16905") ->	{"0","",""};
lookup1("16906") ->	{"0","",""};
lookup1("16907") ->	{"0","",""};
lookup1("16908") ->	{"0","",""};
lookup1("16909") ->	{"0","",""};
lookup1("1690A") ->	{"0","",""};
lookup1("1690B") ->	{"0","",""};
lookup1("1690C") ->	{"0","",""};
lookup1("1690D") ->	{"0","",""};
lookup1("1690E") ->	{"0","",""};
lookup1("1690F") ->	{"0","",""};
lookup1("1691") ->	{"0","",""};
lookup1("16910") ->	{"0","",""};
lookup1("16911") ->	{"0","",""};
lookup1("16912") ->	{"0","",""};
lookup1("16913") ->	{"0","",""};
lookup1("16914") ->	{"0","",""};
lookup1("16915") ->	{"0","",""};
lookup1("16916") ->	{"0","",""};
lookup1("16917") ->	{"0","",""};
lookup1("16918") ->	{"0","",""};
lookup1("16919") ->	{"0","",""};
lookup1("1691A") ->	{"0","",""};
lookup1("1691B") ->	{"0","",""};
lookup1("1691C") ->	{"0","",""};
lookup1("1691D") ->	{"0","",""};
lookup1("1691E") ->	{"0","",""};
lookup1("1691F") ->	{"0","",""};
lookup1("1692") ->	{"0","",""};
lookup1("16920") ->	{"0","",""};
lookup1("16921") ->	{"0","",""};
lookup1("16922") ->	{"0","",""};
lookup1("16923") ->	{"0","",""};
lookup1("16924") ->	{"0","",""};
lookup1("16925") ->	{"0","",""};
lookup1("16926") ->	{"0","",""};
lookup1("16927") ->	{"0","",""};
lookup1("16928") ->	{"0","",""};
lookup1("16929") ->	{"0","",""};
lookup1("1692A") ->	{"0","",""};
lookup1("1692B") ->	{"0","",""};
lookup1("1692C") ->	{"0","",""};
lookup1("1692D") ->	{"0","",""};
lookup1("1692E") ->	{"0","",""};
lookup1("1692F") ->	{"0","",""};
lookup1("1693") ->	{"0","",""};
lookup1("16930") ->	{"0","",""};
lookup1("16931") ->	{"0","",""};
lookup1("16932") ->	{"0","",""};
lookup1("16933") ->	{"0","",""};
lookup1("16934") ->	{"0","",""};
lookup1("16935") ->	{"0","",""};
lookup1("16936") ->	{"0","",""};
lookup1("16937") ->	{"0","",""};
lookup1("16938") ->	{"0","",""};
lookup1("16939") ->	{"0","",""};
lookup1("1693A") ->	{"0","",""};
lookup1("1693B") ->	{"0","",""};
lookup1("1693C") ->	{"0","",""};
lookup1("1693D") ->	{"0","",""};
lookup1("1693E") ->	{"0","",""};
lookup1("1693F") ->	{"0","",""};
lookup1("1694") ->	{"0","",""};
lookup1("16940") ->	{"0","",""};
lookup1("16941") ->	{"0","",""};
lookup1("16942") ->	{"0","",""};
lookup1("16943") ->	{"0","",""};
lookup1("16944") ->	{"0","",""};
lookup1("16945") ->	{"0","",""};
lookup1("16946") ->	{"0","",""};
lookup1("16947") ->	{"0","",""};
lookup1("16948") ->	{"0","",""};
lookup1("16949") ->	{"0","",""};
lookup1("1694A") ->	{"0","",""};
lookup1("1694B") ->	{"0","",""};
lookup1("1694C") ->	{"0","",""};
lookup1("1694D") ->	{"0","",""};
lookup1("1694E") ->	{"0","",""};
lookup1("1694F") ->	{"0","",""};
lookup1("1695") ->	{"0","",""};
lookup1("16950") ->	{"0","",""};
lookup1("16951") ->	{"0","",""};
lookup1("16952") ->	{"0","",""};
lookup1("16953") ->	{"0","",""};
lookup1("16954") ->	{"0","",""};
lookup1("16955") ->	{"0","",""};
lookup1("16956") ->	{"0","",""};
lookup1("16957") ->	{"0","",""};
lookup1("16958") ->	{"0","",""};
lookup1("16959") ->	{"0","",""};
lookup1("1695A") ->	{"0","",""};
lookup1("1695B") ->	{"0","",""};
lookup1("1695C") ->	{"0","",""};
lookup1("1695D") ->	{"0","",""};
lookup1("1695E") ->	{"0","",""};
lookup1("1695F") ->	{"0","",""};
lookup1("1696") ->	{"0","",""};
lookup1("16960") ->	{"0","",""};
lookup1("16961") ->	{"0","",""};
lookup1("16962") ->	{"0","",""};
lookup1("16963") ->	{"0","",""};
lookup1("16964") ->	{"0","",""};
lookup1("16965") ->	{"0","",""};
lookup1("16966") ->	{"0","",""};
lookup1("16967") ->	{"0","",""};
lookup1("16968") ->	{"0","",""};
lookup1("16969") ->	{"0","",""};
lookup1("1696A") ->	{"0","",""};
lookup1("1696B") ->	{"0","",""};
lookup1("1696C") ->	{"0","",""};
lookup1("1696D") ->	{"0","",""};
lookup1("1696E") ->	{"0","",""};
lookup1("1696F") ->	{"0","",""};
lookup1("1697") ->	{"0","",""};
lookup1("16970") ->	{"0","",""};
lookup1("16971") ->	{"0","",""};
lookup1("16972") ->	{"0","",""};
lookup1("16973") ->	{"0","",""};
lookup1("16974") ->	{"0","",""};
lookup1("16975") ->	{"0","",""};
lookup1("16976") ->	{"0","",""};
lookup1("16977") ->	{"0","",""};
lookup1("16978") ->	{"0","",""};
lookup1("16979") ->	{"0","",""};
lookup1("1697A") ->	{"0","",""};
lookup1("1697B") ->	{"0","",""};
lookup1("1697C") ->	{"0","",""};
lookup1("1697D") ->	{"0","",""};
lookup1("1697E") ->	{"0","",""};
lookup1("1697F") ->	{"0","",""};
lookup1("1698") ->	{"0","",""};
lookup1("16980") ->	{"0","",""};
lookup1("16981") ->	{"0","",""};
lookup1("16982") ->	{"0","",""};
lookup1("16983") ->	{"0","",""};
lookup1("16984") ->	{"0","",""};
lookup1("16985") ->	{"0","",""};
lookup1("16986") ->	{"0","",""};
lookup1("16987") ->	{"0","",""};
lookup1("16988") ->	{"0","",""};
lookup1("16989") ->	{"0","",""};
lookup1("1698A") ->	{"0","",""};
lookup1("1698B") ->	{"0","",""};
lookup1("1698C") ->	{"0","",""};
lookup1("1698D") ->	{"0","",""};
lookup1("1698E") ->	{"0","",""};
lookup1("1698F") ->	{"0","",""};
lookup1("1699") ->	{"0","",""};
lookup1("16990") ->	{"0","",""};
lookup1("16991") ->	{"0","",""};
lookup1("16992") ->	{"0","",""};
lookup1("16993") ->	{"0","",""};
lookup1("16994") ->	{"0","",""};
lookup1("16995") ->	{"0","",""};
lookup1("16996") ->	{"0","",""};
lookup1("16997") ->	{"0","",""};
lookup1("16998") ->	{"0","",""};
lookup1("16999") ->	{"0","",""};
lookup1("1699A") ->	{"0","",""};
lookup1("1699B") ->	{"0","",""};
lookup1("1699C") ->	{"0","",""};
lookup1("1699D") ->	{"0","",""};
lookup1("1699E") ->	{"0","",""};
lookup1("1699F") ->	{"0","",""};
lookup1("169A") ->	{"0","",""};
lookup1("169A0") ->	{"0","",""};
lookup1("169A1") ->	{"0","",""};
lookup1("169A2") ->	{"0","",""};
lookup1("169A3") ->	{"0","",""};
lookup1("169A4") ->	{"0","",""};
lookup1("169A5") ->	{"0","",""};
lookup1("169A6") ->	{"0","",""};
lookup1("169A7") ->	{"0","",""};
lookup1("169A8") ->	{"0","",""};
lookup1("169A9") ->	{"0","",""};
lookup1("169AA") ->	{"0","",""};
lookup1("169AB") ->	{"0","",""};
lookup1("169AC") ->	{"0","",""};
lookup1("169AD") ->	{"0","",""};
lookup1("169AE") ->	{"0","",""};
lookup1("169AF") ->	{"0","",""};
lookup1("169B") ->	{"0","",""};
lookup1("169B0") ->	{"0","",""};
lookup1("169B1") ->	{"0","",""};
lookup1("169B2") ->	{"0","",""};
lookup1("169B3") ->	{"0","",""};
lookup1("169B4") ->	{"0","",""};
lookup1("169B5") ->	{"0","",""};
lookup1("169B6") ->	{"0","",""};
lookup1("169B7") ->	{"0","",""};
lookup1("169B8") ->	{"0","",""};
lookup1("169B9") ->	{"0","",""};
lookup1("169BA") ->	{"0","",""};
lookup1("169BB") ->	{"0","",""};
lookup1("169BC") ->	{"0","",""};
lookup1("169BD") ->	{"0","",""};
lookup1("169BE") ->	{"0","",""};
lookup1("169BF") ->	{"0","",""};
lookup1("169C") ->	{"0","",""};
lookup1("169C0") ->	{"0","",""};
lookup1("169C1") ->	{"0","",""};
lookup1("169C2") ->	{"0","",""};
lookup1("169C3") ->	{"0","",""};
lookup1("169C4") ->	{"0","",""};
lookup1("169C5") ->	{"0","",""};
lookup1("169C6") ->	{"0","",""};
lookup1("169C7") ->	{"0","",""};
lookup1("169C8") ->	{"0","",""};
lookup1("169C9") ->	{"0","",""};
lookup1("169CA") ->	{"0","",""};
lookup1("169CB") ->	{"0","",""};
lookup1("169CC") ->	{"0","",""};
lookup1("169CD") ->	{"0","",""};
lookup1("169CE") ->	{"0","",""};
lookup1("169CF") ->	{"0","",""};
lookup1("169D0") ->	{"0","",""};
lookup1("169D1") ->	{"0","",""};
lookup1("169D2") ->	{"0","",""};
lookup1("169D3") ->	{"0","",""};
lookup1("169D4") ->	{"0","",""};
lookup1("169D5") ->	{"0","",""};
lookup1("169D6") ->	{"0","",""};
lookup1("169D7") ->	{"0","",""};
lookup1("169D8") ->	{"0","",""};
lookup1("169D9") ->	{"0","",""};
lookup1("169DA") ->	{"0","",""};
lookup1("169DB") ->	{"0","",""};
lookup1("169DC") ->	{"0","",""};
lookup1("169DD") ->	{"0","",""};
lookup1("169DE") ->	{"0","",""};
lookup1("169DF") ->	{"0","",""};
lookup1("169E0") ->	{"0","",""};
lookup1("169E1") ->	{"0","",""};
lookup1("169E2") ->	{"0","",""};
lookup1("169E3") ->	{"0","",""};
lookup1("169E4") ->	{"0","",""};
lookup1("169E5") ->	{"0","",""};
lookup1("169E6") ->	{"0","",""};
lookup1("169E7") ->	{"0","",""};
lookup1("169E8") ->	{"0","",""};
lookup1("169E9") ->	{"0","",""};
lookup1("169EA") ->	{"0","",""};
lookup1("169EB") ->	{"0","",""};
lookup1("169EC") ->	{"0","",""};
lookup1("169ED") ->	{"0","",""};
lookup1("169EE") ->	{"0","",""};
lookup1("169EF") ->	{"0","",""};
lookup1("169F0") ->	{"0","",""};
lookup1("169F1") ->	{"0","",""};
lookup1("169F2") ->	{"0","",""};
lookup1("169F3") ->	{"0","",""};
lookup1("169F4") ->	{"0","",""};
lookup1("169F5") ->	{"0","",""};
lookup1("169F6") ->	{"0","",""};
lookup1("169F7") ->	{"0","",""};
lookup1("169F8") ->	{"0","",""};
lookup1("169F9") ->	{"0","",""};
lookup1("169FA") ->	{"0","",""};
lookup1("169FB") ->	{"0","",""};
lookup1("169FC") ->	{"0","",""};
lookup1("169FD") ->	{"0","",""};
lookup1("169FE") ->	{"0","",""};
lookup1("169FF") ->	{"0","",""};
lookup1("16A0") ->	{"0","",""};
lookup1("16A00") ->	{"0","",""};
lookup1("16A01") ->	{"0","",""};
lookup1("16A02") ->	{"0","",""};
lookup1("16A03") ->	{"0","",""};
lookup1("16A04") ->	{"0","",""};
lookup1("16A05") ->	{"0","",""};
lookup1("16A06") ->	{"0","",""};
lookup1("16A07") ->	{"0","",""};
lookup1("16A08") ->	{"0","",""};
lookup1("16A09") ->	{"0","",""};
lookup1("16A0A") ->	{"0","",""};
lookup1("16A0B") ->	{"0","",""};
lookup1("16A0C") ->	{"0","",""};
lookup1("16A0D") ->	{"0","",""};
lookup1("16A0E") ->	{"0","",""};
lookup1("16A0F") ->	{"0","",""};
lookup1("16A1") ->	{"0","",""};
lookup1("16A10") ->	{"0","",""};
lookup1("16A11") ->	{"0","",""};
lookup1("16A12") ->	{"0","",""};
lookup1("16A13") ->	{"0","",""};
lookup1("16A14") ->	{"0","",""};
lookup1("16A15") ->	{"0","",""};
lookup1("16A16") ->	{"0","",""};
lookup1("16A17") ->	{"0","",""};
lookup1("16A18") ->	{"0","",""};
lookup1("16A19") ->	{"0","",""};
lookup1("16A1A") ->	{"0","",""};
lookup1("16A1B") ->	{"0","",""};
lookup1("16A1C") ->	{"0","",""};
lookup1("16A1D") ->	{"0","",""};
lookup1("16A1E") ->	{"0","",""};
lookup1("16A1F") ->	{"0","",""};
lookup1("16A2") ->	{"0","",""};
lookup1("16A20") ->	{"0","",""};
lookup1("16A21") ->	{"0","",""};
lookup1("16A22") ->	{"0","",""};
lookup1("16A23") ->	{"0","",""};
lookup1("16A24") ->	{"0","",""};
lookup1("16A25") ->	{"0","",""};
lookup1("16A26") ->	{"0","",""};
lookup1("16A27") ->	{"0","",""};
lookup1("16A28") ->	{"0","",""};
lookup1("16A29") ->	{"0","",""};
lookup1("16A2A") ->	{"0","",""};
lookup1("16A2B") ->	{"0","",""};
lookup1("16A2C") ->	{"0","",""};
lookup1("16A2D") ->	{"0","",""};
lookup1("16A2E") ->	{"0","",""};
lookup1("16A2F") ->	{"0","",""};
lookup1("16A3") ->	{"0","",""};
lookup1("16A30") ->	{"0","",""};
lookup1("16A31") ->	{"0","",""};
lookup1("16A32") ->	{"0","",""};
lookup1("16A33") ->	{"0","",""};
lookup1("16A34") ->	{"0","",""};
lookup1("16A35") ->	{"0","",""};
lookup1("16A36") ->	{"0","",""};
lookup1("16A37") ->	{"0","",""};
lookup1("16A38") ->	{"0","",""};
lookup1("16A4") ->	{"0","",""};
lookup1("16A40") ->	{"0","",""};
lookup1("16A41") ->	{"0","",""};
lookup1("16A42") ->	{"0","",""};
lookup1("16A43") ->	{"0","",""};
lookup1("16A44") ->	{"0","",""};
lookup1("16A45") ->	{"0","",""};
lookup1("16A46") ->	{"0","",""};
lookup1("16A47") ->	{"0","",""};
lookup1("16A48") ->	{"0","",""};
lookup1("16A49") ->	{"0","",""};
lookup1("16A4A") ->	{"0","",""};
lookup1("16A4B") ->	{"0","",""};
lookup1("16A4C") ->	{"0","",""};
lookup1("16A4D") ->	{"0","",""};
lookup1("16A4E") ->	{"0","",""};
lookup1("16A4F") ->	{"0","",""};
lookup1("16A5") ->	{"0","",""};
lookup1("16A50") ->	{"0","",""};
lookup1("16A51") ->	{"0","",""};
lookup1("16A52") ->	{"0","",""};
lookup1("16A53") ->	{"0","",""};
lookup1("16A54") ->	{"0","",""};
lookup1("16A55") ->	{"0","",""};
lookup1("16A56") ->	{"0","",""};
lookup1("16A57") ->	{"0","",""};
lookup1("16A58") ->	{"0","",""};
lookup1("16A59") ->	{"0","",""};
lookup1("16A5A") ->	{"0","",""};
lookup1("16A5B") ->	{"0","",""};
lookup1("16A5C") ->	{"0","",""};
lookup1("16A5D") ->	{"0","",""};
lookup1("16A5E") ->	{"0","",""};
lookup1("16A6") ->	{"0","",""};
lookup1("16A60") ->	{"0","",""};
lookup1("16A61") ->	{"0","",""};
lookup1("16A62") ->	{"0","",""};
lookup1("16A63") ->	{"0","",""};
lookup1("16A64") ->	{"0","",""};
lookup1("16A65") ->	{"0","",""};
lookup1("16A66") ->	{"0","",""};
lookup1("16A67") ->	{"0","",""};
lookup1("16A68") ->	{"0","",""};
lookup1("16A69") ->	{"0","",""};
lookup1("16A6E") ->	{"0","",""};
lookup1("16A6F") ->	{"0","",""};
lookup1("16A7") ->	{"0","",""};
lookup1("16A8") ->	{"0","",""};
lookup1("16A9") ->	{"0","",""};
lookup1("16AA") ->	{"0","",""};
lookup1("16AB") ->	{"0","",""};
lookup1("16AC") ->	{"0","",""};
lookup1("16AD") ->	{"0","",""};
lookup1("16AD0") ->	{"0","",""};
lookup1("16AD1") ->	{"0","",""};
lookup1("16AD2") ->	{"0","",""};
lookup1("16AD3") ->	{"0","",""};
lookup1("16AD4") ->	{"0","",""};
lookup1("16AD5") ->	{"0","",""};
lookup1("16AD6") ->	{"0","",""};
lookup1("16AD7") ->	{"0","",""};
lookup1("16AD8") ->	{"0","",""};
lookup1("16AD9") ->	{"0","",""};
lookup1("16ADA") ->	{"0","",""};
lookup1("16ADB") ->	{"0","",""};
lookup1("16ADC") ->	{"0","",""};
lookup1("16ADD") ->	{"0","",""};
lookup1("16ADE") ->	{"0","",""};
lookup1("16ADF") ->	{"0","",""};
lookup1("16AE") ->	{"0","",""};
lookup1("16AE0") ->	{"0","",""};
lookup1("16AE1") ->	{"0","",""};
lookup1("16AE2") ->	{"0","",""};
lookup1("16AE3") ->	{"0","",""};
lookup1("16AE4") ->	{"0","",""};
lookup1("16AE5") ->	{"0","",""};
lookup1("16AE6") ->	{"0","",""};
lookup1("16AE7") ->	{"0","",""};
lookup1("16AE8") ->	{"0","",""};
lookup1("16AE9") ->	{"0","",""};
lookup1("16AEA") ->	{"0","",""};
lookup1("16AEB") ->	{"0","",""};
lookup1("16AEC") ->	{"0","",""};
lookup1("16AED") ->	{"0","",""};
lookup1("16AF") ->	{"0","",""};
lookup1("16AF0") ->	{"1","",""};
lookup1("16AF1") ->	{"1","",""};
lookup1("16AF2") ->	{"1","",""};
lookup1("16AF3") ->	{"1","",""};
lookup1("16AF4") ->	{"1","",""};
lookup1("16AF5") ->	{"0","",""};
lookup1("16B0") ->	{"0","",""};
lookup1("16B00") ->	{"0","",""};
lookup1("16B01") ->	{"0","",""};
lookup1("16B02") ->	{"0","",""};
lookup1("16B03") ->	{"0","",""};
lookup1("16B04") ->	{"0","",""};
lookup1("16B05") ->	{"0","",""};
lookup1("16B06") ->	{"0","",""};
lookup1("16B07") ->	{"0","",""};
lookup1("16B08") ->	{"0","",""};
lookup1("16B09") ->	{"0","",""};
lookup1("16B0A") ->	{"0","",""};
lookup1("16B0B") ->	{"0","",""};
lookup1("16B0C") ->	{"0","",""};
lookup1("16B0D") ->	{"0","",""};
lookup1("16B0E") ->	{"0","",""};
lookup1("16B0F") ->	{"0","",""};
lookup1("16B1") ->	{"0","",""};
lookup1("16B10") ->	{"0","",""};
lookup1("16B11") ->	{"0","",""};
lookup1("16B12") ->	{"0","",""};
lookup1("16B13") ->	{"0","",""};
lookup1("16B14") ->	{"0","",""};
lookup1("16B15") ->	{"0","",""};
lookup1("16B16") ->	{"0","",""};
lookup1("16B17") ->	{"0","",""};
lookup1("16B18") ->	{"0","",""};
lookup1("16B19") ->	{"0","",""};
lookup1("16B1A") ->	{"0","",""};
lookup1("16B1B") ->	{"0","",""};
lookup1("16B1C") ->	{"0","",""};
lookup1("16B1D") ->	{"0","",""};
lookup1("16B1E") ->	{"0","",""};
lookup1("16B1F") ->	{"0","",""};
lookup1("16B2") ->	{"0","",""};
lookup1("16B20") ->	{"0","",""};
lookup1("16B21") ->	{"0","",""};
lookup1("16B22") ->	{"0","",""};
lookup1("16B23") ->	{"0","",""};
lookup1("16B24") ->	{"0","",""};
lookup1("16B25") ->	{"0","",""};
lookup1("16B26") ->	{"0","",""};
lookup1("16B27") ->	{"0","",""};
lookup1("16B28") ->	{"0","",""};
lookup1("16B29") ->	{"0","",""};
lookup1("16B2A") ->	{"0","",""};
lookup1("16B2B") ->	{"0","",""};
lookup1("16B2C") ->	{"0","",""};
lookup1("16B2D") ->	{"0","",""};
lookup1("16B2E") ->	{"0","",""};
lookup1("16B2F") ->	{"0","",""};
lookup1("16B3") ->	{"0","",""};
lookup1("16B30") ->	{"230","",""};
lookup1("16B31") ->	{"230","",""};
lookup1("16B32") ->	{"230","",""};
lookup1("16B33") ->	{"230","",""};
lookup1("16B34") ->	{"230","",""};
lookup1("16B35") ->	{"230","",""};
lookup1("16B36") ->	{"230","",""};
lookup1("16B37") ->	{"0","",""};
lookup1("16B38") ->	{"0","",""};
lookup1("16B39") ->	{"0","",""};
lookup1("16B3A") ->	{"0","",""};
lookup1("16B3B") ->	{"0","",""};
lookup1("16B3C") ->	{"0","",""};
lookup1("16B3D") ->	{"0","",""};
lookup1("16B3E") ->	{"0","",""};
lookup1("16B3F") ->	{"0","",""};
lookup1("16B4") ->	{"0","",""};
lookup1("16B40") ->	{"0","",""};
lookup1("16B41") ->	{"0","",""};
lookup1("16B42") ->	{"0","",""};
lookup1("16B43") ->	{"0","",""};
lookup1("16B44") ->	{"0","",""};
lookup1("16B45") ->	{"0","",""};
lookup1("16B5") ->	{"0","",""};
lookup1("16B50") ->	{"0","",""};
lookup1("16B51") ->	{"0","",""};
lookup1("16B52") ->	{"0","",""};
lookup1("16B53") ->	{"0","",""};
lookup1("16B54") ->	{"0","",""};
lookup1("16B55") ->	{"0","",""};
lookup1("16B56") ->	{"0","",""};
lookup1("16B57") ->	{"0","",""};
lookup1("16B58") ->	{"0","",""};
lookup1("16B59") ->	{"0","",""};
lookup1("16B5B") ->	{"0","",""};
lookup1("16B5C") ->	{"0","",""};
lookup1("16B5D") ->	{"0","",""};
lookup1("16B5E") ->	{"0","",""};
lookup1("16B5F") ->	{"0","",""};
lookup1("16B6") ->	{"0","",""};
lookup1("16B60") ->	{"0","",""};
lookup1("16B61") ->	{"0","",""};
lookup1("16B63") ->	{"0","",""};
lookup1("16B64") ->	{"0","",""};
lookup1("16B65") ->	{"0","",""};
lookup1("16B66") ->	{"0","",""};
lookup1("16B67") ->	{"0","",""};
lookup1("16B68") ->	{"0","",""};
lookup1("16B69") ->	{"0","",""};
lookup1("16B6A") ->	{"0","",""};
lookup1("16B6B") ->	{"0","",""};
lookup1("16B6C") ->	{"0","",""};
lookup1("16B6D") ->	{"0","",""};
lookup1("16B6E") ->	{"0","",""};
lookup1("16B6F") ->	{"0","",""};
lookup1("16B7") ->	{"0","",""};
lookup1("16B70") ->	{"0","",""};
lookup1("16B71") ->	{"0","",""};
lookup1("16B72") ->	{"0","",""};
lookup1("16B73") ->	{"0","",""};
lookup1("16B74") ->	{"0","",""};
lookup1("16B75") ->	{"0","",""};
lookup1("16B76") ->	{"0","",""};
lookup1("16B77") ->	{"0","",""};
lookup1("16B7D") ->	{"0","",""};
lookup1("16B7E") ->	{"0","",""};
lookup1("16B7F") ->	{"0","",""};
lookup1("16B8") ->	{"0","",""};
lookup1("16B80") ->	{"0","",""};
lookup1("16B81") ->	{"0","",""};
lookup1("16B82") ->	{"0","",""};
lookup1("16B83") ->	{"0","",""};
lookup1("16B84") ->	{"0","",""};
lookup1("16B85") ->	{"0","",""};
lookup1("16B86") ->	{"0","",""};
lookup1("16B87") ->	{"0","",""};
lookup1("16B88") ->	{"0","",""};
lookup1("16B89") ->	{"0","",""};
lookup1("16B8A") ->	{"0","",""};
lookup1("16B8B") ->	{"0","",""};
lookup1("16B8C") ->	{"0","",""};
lookup1("16B8D") ->	{"0","",""};
lookup1("16B8E") ->	{"0","",""};
lookup1("16B8F") ->	{"0","",""};
lookup1("16B9") ->	{"0","",""};
lookup1("16BA") ->	{"0","",""};
lookup1("16BB") ->	{"0","",""};
lookup1("16BC") ->	{"0","",""};
lookup1("16BD") ->	{"0","",""};
lookup1("16BE") ->	{"0","",""};
lookup1("16BF") ->	{"0","",""};
lookup1("16C0") ->	{"0","",""};
lookup1("16C1") ->	{"0","",""};
lookup1("16C2") ->	{"0","",""};
lookup1("16C3") ->	{"0","",""};
lookup1("16C4") ->	{"0","",""};
lookup1("16C5") ->	{"0","",""};
lookup1("16C6") ->	{"0","",""};
lookup1("16C7") ->	{"0","",""};
lookup1("16C8") ->	{"0","",""};
lookup1("16C9") ->	{"0","",""};
lookup1("16CA") ->	{"0","",""};
lookup1("16CB") ->	{"0","",""};
lookup1("16CC") ->	{"0","",""};
lookup1("16CD") ->	{"0","",""};
lookup1("16CE") ->	{"0","",""};
lookup1("16CF") ->	{"0","",""};
lookup1("16D0") ->	{"0","",""};
lookup1("16D1") ->	{"0","",""};
lookup1("16D2") ->	{"0","",""};
lookup1("16D3") ->	{"0","",""};
lookup1("16D4") ->	{"0","",""};
lookup1("16D5") ->	{"0","",""};
lookup1("16D6") ->	{"0","",""};
lookup1("16D7") ->	{"0","",""};
lookup1("16D8") ->	{"0","",""};
lookup1("16D9") ->	{"0","",""};
lookup1("16DA") ->	{"0","",""};
lookup1("16DB") ->	{"0","",""};
lookup1("16DC") ->	{"0","",""};
lookup1("16DD") ->	{"0","",""};
lookup1("16DE") ->	{"0","",""};
lookup1("16DF") ->	{"0","",""};
lookup1("16E0") ->	{"0","",""};
lookup1("16E1") ->	{"0","",""};
lookup1("16E2") ->	{"0","",""};
lookup1("16E3") ->	{"0","",""};
lookup1("16E4") ->	{"0","",""};
lookup1("16E5") ->	{"0","",""};
lookup1("16E6") ->	{"0","",""};
lookup1("16E7") ->	{"0","",""};
lookup1("16E8") ->	{"0","",""};
lookup1("16E9") ->	{"0","",""};
lookup1("16EA") ->	{"0","",""};
lookup1("16EB") ->	{"0","",""};
lookup1("16EC") ->	{"0","",""};
lookup1("16ED") ->	{"0","",""};
lookup1("16EE") ->	{"0","",""};
lookup1("16EF") ->	{"0","",""};
lookup1("16F0") ->	{"0","",""};
lookup1("16F00") ->	{"0","",""};
lookup1("16F01") ->	{"0","",""};
lookup1("16F02") ->	{"0","",""};
lookup1("16F03") ->	{"0","",""};
lookup1("16F04") ->	{"0","",""};
lookup1("16F05") ->	{"0","",""};
lookup1("16F06") ->	{"0","",""};
lookup1("16F07") ->	{"0","",""};
lookup1("16F08") ->	{"0","",""};
lookup1("16F09") ->	{"0","",""};
lookup1("16F0A") ->	{"0","",""};
lookup1("16F0B") ->	{"0","",""};
lookup1("16F0C") ->	{"0","",""};
lookup1("16F0D") ->	{"0","",""};
lookup1("16F0E") ->	{"0","",""};
lookup1("16F0F") ->	{"0","",""};
lookup1("16F1") ->	{"0","",""};
lookup1("16F10") ->	{"0","",""};
lookup1("16F11") ->	{"0","",""};
lookup1("16F12") ->	{"0","",""};
lookup1("16F13") ->	{"0","",""};
lookup1("16F14") ->	{"0","",""};
lookup1("16F15") ->	{"0","",""};
lookup1("16F16") ->	{"0","",""};
lookup1("16F17") ->	{"0","",""};
lookup1("16F18") ->	{"0","",""};
lookup1("16F19") ->	{"0","",""};
lookup1("16F1A") ->	{"0","",""};
lookup1("16F1B") ->	{"0","",""};
lookup1("16F1C") ->	{"0","",""};
lookup1("16F1D") ->	{"0","",""};
lookup1("16F1E") ->	{"0","",""};
lookup1("16F1F") ->	{"0","",""};
lookup1("16F2") ->	{"0","",""};
lookup1("16F20") ->	{"0","",""};
lookup1("16F21") ->	{"0","",""};
lookup1("16F22") ->	{"0","",""};
lookup1("16F23") ->	{"0","",""};
lookup1("16F24") ->	{"0","",""};
lookup1("16F25") ->	{"0","",""};
lookup1("16F26") ->	{"0","",""};
lookup1("16F27") ->	{"0","",""};
lookup1("16F28") ->	{"0","",""};
lookup1("16F29") ->	{"0","",""};
lookup1("16F2A") ->	{"0","",""};
lookup1("16F2B") ->	{"0","",""};
lookup1("16F2C") ->	{"0","",""};
lookup1("16F2D") ->	{"0","",""};
lookup1("16F2E") ->	{"0","",""};
lookup1("16F2F") ->	{"0","",""};
lookup1("16F3") ->	{"0","",""};
lookup1("16F30") ->	{"0","",""};
lookup1("16F31") ->	{"0","",""};
lookup1("16F32") ->	{"0","",""};
lookup1("16F33") ->	{"0","",""};
lookup1("16F34") ->	{"0","",""};
lookup1("16F35") ->	{"0","",""};
lookup1("16F36") ->	{"0","",""};
lookup1("16F37") ->	{"0","",""};
lookup1("16F38") ->	{"0","",""};
lookup1("16F39") ->	{"0","",""};
lookup1("16F3A") ->	{"0","",""};
lookup1("16F3B") ->	{"0","",""};
lookup1("16F3C") ->	{"0","",""};
lookup1("16F3D") ->	{"0","",""};
lookup1("16F3E") ->	{"0","",""};
lookup1("16F3F") ->	{"0","",""};
lookup1("16F4") ->	{"0","",""};
lookup1("16F40") ->	{"0","",""};
lookup1("16F41") ->	{"0","",""};
lookup1("16F42") ->	{"0","",""};
lookup1("16F43") ->	{"0","",""};
lookup1("16F44") ->	{"0","",""};
lookup1("16F5") ->	{"0","",""};
lookup1("16F50") ->	{"0","",""};
lookup1("16F51") ->	{"0","",""};
lookup1("16F52") ->	{"0","",""};
lookup1("16F53") ->	{"0","",""};
lookup1("16F54") ->	{"0","",""};
lookup1("16F55") ->	{"0","",""};
lookup1("16F56") ->	{"0","",""};
lookup1("16F57") ->	{"0","",""};
lookup1("16F58") ->	{"0","",""};
lookup1("16F59") ->	{"0","",""};
lookup1("16F5A") ->	{"0","",""};
lookup1("16F5B") ->	{"0","",""};
lookup1("16F5C") ->	{"0","",""};
lookup1("16F5D") ->	{"0","",""};
lookup1("16F5E") ->	{"0","",""};
lookup1("16F5F") ->	{"0","",""};
lookup1("16F6") ->	{"0","",""};
lookup1("16F60") ->	{"0","",""};
lookup1("16F61") ->	{"0","",""};
lookup1("16F62") ->	{"0","",""};
lookup1("16F63") ->	{"0","",""};
lookup1("16F64") ->	{"0","",""};
lookup1("16F65") ->	{"0","",""};
lookup1("16F66") ->	{"0","",""};
lookup1("16F67") ->	{"0","",""};
lookup1("16F68") ->	{"0","",""};
lookup1("16F69") ->	{"0","",""};
lookup1("16F6A") ->	{"0","",""};
lookup1("16F6B") ->	{"0","",""};
lookup1("16F6C") ->	{"0","",""};
lookup1("16F6D") ->	{"0","",""};
lookup1("16F6E") ->	{"0","",""};
lookup1("16F6F") ->	{"0","",""};
lookup1("16F7") ->	{"0","",""};
lookup1("16F70") ->	{"0","",""};
lookup1("16F71") ->	{"0","",""};
lookup1("16F72") ->	{"0","",""};
lookup1("16F73") ->	{"0","",""};
lookup1("16F74") ->	{"0","",""};
lookup1("16F75") ->	{"0","",""};
lookup1("16F76") ->	{"0","",""};
lookup1("16F77") ->	{"0","",""};
lookup1("16F78") ->	{"0","",""};
lookup1("16F79") ->	{"0","",""};
lookup1("16F7A") ->	{"0","",""};
lookup1("16F7B") ->	{"0","",""};
lookup1("16F7C") ->	{"0","",""};
lookup1("16F7D") ->	{"0","",""};
lookup1("16F7E") ->	{"0","",""};
lookup1("16F8") ->	{"0","",""};
lookup1("16F8F") ->	{"0","",""};
lookup1("16F90") ->	{"0","",""};
lookup1("16F91") ->	{"0","",""};
lookup1("16F92") ->	{"0","",""};
lookup1("16F93") ->	{"0","",""};
lookup1("16F94") ->	{"0","",""};
lookup1("16F95") ->	{"0","",""};
lookup1("16F96") ->	{"0","",""};
lookup1("16F97") ->	{"0","",""};
lookup1("16F98") ->	{"0","",""};
lookup1("16F99") ->	{"0","",""};
lookup1("16F9A") ->	{"0","",""};
lookup1("16F9B") ->	{"0","",""};
lookup1("16F9C") ->	{"0","",""};
lookup1("16F9D") ->	{"0","",""};
lookup1("16F9E") ->	{"0","",""};
lookup1("16F9F") ->	{"0","",""};
lookup1("1700") ->	{"0","",""};
lookup1("1701") ->	{"0","",""};
lookup1("1702") ->	{"0","",""};
lookup1("1703") ->	{"0","",""};
lookup1("1704") ->	{"0","",""};
lookup1("1705") ->	{"0","",""};
lookup1("1706") ->	{"0","",""};
lookup1("1707") ->	{"0","",""};
lookup1("1708") ->	{"0","",""};
lookup1("1709") ->	{"0","",""};
lookup1("170A") ->	{"0","",""};
lookup1("170B") ->	{"0","",""};
lookup1("170C") ->	{"0","",""};
lookup1("170E") ->	{"0","",""};
lookup1("170F") ->	{"0","",""};
lookup1("1710") ->	{"0","",""};
lookup1("1711") ->	{"0","",""};
lookup1("1712") ->	{"0","",""};
lookup1("1713") ->	{"0","",""};
lookup1("1714") ->	{"9","",""};
lookup1("1720") ->	{"0","",""};
lookup1("1721") ->	{"0","",""};
lookup1("1722") ->	{"0","",""};
lookup1("1723") ->	{"0","",""};
lookup1("1724") ->	{"0","",""};
lookup1("1725") ->	{"0","",""};
lookup1("1726") ->	{"0","",""};
lookup1("1727") ->	{"0","",""};
lookup1("1728") ->	{"0","",""};
lookup1("1729") ->	{"0","",""};
lookup1("172A") ->	{"0","",""};
lookup1("172B") ->	{"0","",""};
lookup1("172C") ->	{"0","",""};
lookup1("172D") ->	{"0","",""};
lookup1("172E") ->	{"0","",""};
lookup1("172F") ->	{"0","",""};
lookup1("1730") ->	{"0","",""};
lookup1("1731") ->	{"0","",""};
lookup1("1732") ->	{"0","",""};
lookup1("1733") ->	{"0","",""};
lookup1("1734") ->	{"9","",""};
lookup1("1735") ->	{"0","",""};
lookup1("1736") ->	{"0","",""};
lookup1("1740") ->	{"0","",""};
lookup1("1741") ->	{"0","",""};
lookup1("1742") ->	{"0","",""};
lookup1("1743") ->	{"0","",""};
lookup1("1744") ->	{"0","",""};
lookup1("1745") ->	{"0","",""};
lookup1("1746") ->	{"0","",""};
lookup1("1747") ->	{"0","",""};
lookup1("1748") ->	{"0","",""};
lookup1("1749") ->	{"0","",""};
lookup1("174A") ->	{"0","",""};
lookup1("174B") ->	{"0","",""};
lookup1("174C") ->	{"0","",""};
lookup1("174D") ->	{"0","",""};
lookup1("174E") ->	{"0","",""};
lookup1("174F") ->	{"0","",""};
lookup1("1750") ->	{"0","",""};
lookup1("1751") ->	{"0","",""};
lookup1("1752") ->	{"0","",""};
lookup1("1753") ->	{"0","",""};
lookup1("1760") ->	{"0","",""};
lookup1("1761") ->	{"0","",""};
lookup1("1762") ->	{"0","",""};
lookup1("1763") ->	{"0","",""};
lookup1("1764") ->	{"0","",""};
lookup1("1765") ->	{"0","",""};
lookup1("1766") ->	{"0","",""};
lookup1("1767") ->	{"0","",""};
lookup1("1768") ->	{"0","",""};
lookup1("1769") ->	{"0","",""};
lookup1("176A") ->	{"0","",""};
lookup1("176B") ->	{"0","",""};
lookup1("176C") ->	{"0","",""};
lookup1("176E") ->	{"0","",""};
lookup1("176F") ->	{"0","",""};
lookup1("1770") ->	{"0","",""};
lookup1("1772") ->	{"0","",""};
lookup1("1773") ->	{"0","",""};
lookup1("1780") ->	{"0","",""};
lookup1("1781") ->	{"0","",""};
lookup1("1782") ->	{"0","",""};
lookup1("1783") ->	{"0","",""};
lookup1("1784") ->	{"0","",""};
lookup1("1785") ->	{"0","",""};
lookup1("1786") ->	{"0","",""};
lookup1("1787") ->	{"0","",""};
lookup1("1788") ->	{"0","",""};
lookup1("1789") ->	{"0","",""};
lookup1("178A") ->	{"0","",""};
lookup1("178B") ->	{"0","",""};
lookup1("178C") ->	{"0","",""};
lookup1("178D") ->	{"0","",""};
lookup1("178E") ->	{"0","",""};
lookup1("178F") ->	{"0","",""};
lookup1("1790") ->	{"0","",""};
lookup1("1791") ->	{"0","",""};
lookup1("1792") ->	{"0","",""};
lookup1("1793") ->	{"0","",""};
lookup1("1794") ->	{"0","",""};
lookup1("1795") ->	{"0","",""};
lookup1("1796") ->	{"0","",""};
lookup1("1797") ->	{"0","",""};
lookup1("1798") ->	{"0","",""};
lookup1("1799") ->	{"0","",""};
lookup1("179A") ->	{"0","",""};
lookup1("179B") ->	{"0","",""};
lookup1("179C") ->	{"0","",""};
lookup1("179D") ->	{"0","",""};
lookup1("179E") ->	{"0","",""};
lookup1("179F") ->	{"0","",""};
lookup1("17A0") ->	{"0","",""};
lookup1("17A1") ->	{"0","",""};
lookup1("17A2") ->	{"0","",""};
lookup1("17A3") ->	{"0","",""};
lookup1("17A4") ->	{"0","",""};
lookup1("17A5") ->	{"0","",""};
lookup1("17A6") ->	{"0","",""};
lookup1("17A7") ->	{"0","",""};
lookup1("17A8") ->	{"0","",""};
lookup1("17A9") ->	{"0","",""};
lookup1("17AA") ->	{"0","",""};
lookup1("17AB") ->	{"0","",""};
lookup1("17AC") ->	{"0","",""};
lookup1("17AD") ->	{"0","",""};
lookup1("17AE") ->	{"0","",""};
lookup1("17AF") ->	{"0","",""};
lookup1("17B0") ->	{"0","",""};
lookup1("17B1") ->	{"0","",""};
lookup1("17B2") ->	{"0","",""};
lookup1("17B3") ->	{"0","",""};
lookup1("17B4") ->	{"0","",""};
lookup1("17B5") ->	{"0","",""};
lookup1("17B6") ->	{"0","",""};
lookup1("17B7") ->	{"0","",""};
lookup1("17B8") ->	{"0","",""};
lookup1("17B9") ->	{"0","",""};
lookup1("17BA") ->	{"0","",""};
lookup1("17BB") ->	{"0","",""};
lookup1("17BC") ->	{"0","",""};
lookup1("17BD") ->	{"0","",""};
lookup1("17BE") ->	{"0","",""};
lookup1("17BF") ->	{"0","",""};
lookup1("17C0") ->	{"0","",""};
lookup1("17C1") ->	{"0","",""};
lookup1("17C2") ->	{"0","",""};
lookup1("17C3") ->	{"0","",""};
lookup1("17C4") ->	{"0","",""};
lookup1("17C5") ->	{"0","",""};
lookup1("17C6") ->	{"0","",""};
lookup1("17C7") ->	{"0","",""};
lookup1("17C8") ->	{"0","",""};
lookup1("17C9") ->	{"0","",""};
lookup1("17CA") ->	{"0","",""};
lookup1("17CB") ->	{"0","",""};
lookup1("17CC") ->	{"0","",""};
lookup1("17CD") ->	{"0","",""};
lookup1("17CE") ->	{"0","",""};
lookup1("17CF") ->	{"0","",""};
lookup1("17D0") ->	{"0","",""};
lookup1("17D1") ->	{"0","",""};
lookup1("17D2") ->	{"9","",""};
lookup1("17D3") ->	{"0","",""};
lookup1("17D4") ->	{"0","",""};
lookup1("17D5") ->	{"0","",""};
lookup1("17D6") ->	{"0","",""};
lookup1("17D7") ->	{"0","",""};
lookup1("17D8") ->	{"0","",""};
lookup1("17D9") ->	{"0","",""};
lookup1("17DA") ->	{"0","",""};
lookup1("17DB") ->	{"0","",""};
lookup1("17DC") ->	{"0","",""};
lookup1("17DD") ->	{"230","",""};
lookup1("17E0") ->	{"0","",""};
lookup1("17E1") ->	{"0","",""};
lookup1("17E2") ->	{"0","",""};
lookup1("17E3") ->	{"0","",""};
lookup1("17E4") ->	{"0","",""};
lookup1("17E5") ->	{"0","",""};
lookup1("17E6") ->	{"0","",""};
lookup1("17E7") ->	{"0","",""};
lookup1("17E8") ->	{"0","",""};
lookup1("17E9") ->	{"0","",""};
lookup1("17F0") ->	{"0","",""};
lookup1("17F1") ->	{"0","",""};
lookup1("17F2") ->	{"0","",""};
lookup1("17F3") ->	{"0","",""};
lookup1("17F4") ->	{"0","",""};
lookup1("17F5") ->	{"0","",""};
lookup1("17F6") ->	{"0","",""};
lookup1("17F7") ->	{"0","",""};
lookup1("17F8") ->	{"0","",""};
lookup1("17F9") ->	{"0","",""};
lookup1("1800") ->	{"0","",""};
lookup1("1801") ->	{"0","",""};
lookup1("1802") ->	{"0","",""};
lookup1("1803") ->	{"0","",""};
lookup1("1804") ->	{"0","",""};
lookup1("1805") ->	{"0","",""};
lookup1("1806") ->	{"0","",""};
lookup1("1807") ->	{"0","",""};
lookup1("1808") ->	{"0","",""};
lookup1("1809") ->	{"0","",""};
lookup1("180A") ->	{"0","",""};
lookup1("180B") ->	{"0","",""};
lookup1("180C") ->	{"0","",""};
lookup1("180D") ->	{"0","",""};
lookup1("180E") ->	{"0","",""};
lookup1("1810") ->	{"0","",""};
lookup1("1811") ->	{"0","",""};
lookup1("1812") ->	{"0","",""};
lookup1("1813") ->	{"0","",""};
lookup1("1814") ->	{"0","",""};
lookup1("1815") ->	{"0","",""};
lookup1("1816") ->	{"0","",""};
lookup1("1817") ->	{"0","",""};
lookup1("1818") ->	{"0","",""};
lookup1("1819") ->	{"0","",""};
lookup1("1820") ->	{"0","",""};
lookup1("1821") ->	{"0","",""};
lookup1("1822") ->	{"0","",""};
lookup1("1823") ->	{"0","",""};
lookup1("1824") ->	{"0","",""};
lookup1("1825") ->	{"0","",""};
lookup1("1826") ->	{"0","",""};
lookup1("1827") ->	{"0","",""};
lookup1("1828") ->	{"0","",""};
lookup1("1829") ->	{"0","",""};
lookup1("182A") ->	{"0","",""};
lookup1("182B") ->	{"0","",""};
lookup1("182C") ->	{"0","",""};
lookup1("182D") ->	{"0","",""};
lookup1("182E") ->	{"0","",""};
lookup1("182F") ->	{"0","",""};
lookup1("1830") ->	{"0","",""};
lookup1("1831") ->	{"0","",""};
lookup1("1832") ->	{"0","",""};
lookup1("1833") ->	{"0","",""};
lookup1("1834") ->	{"0","",""};
lookup1("1835") ->	{"0","",""};
lookup1("1836") ->	{"0","",""};
lookup1("1837") ->	{"0","",""};
lookup1("1838") ->	{"0","",""};
lookup1("1839") ->	{"0","",""};
lookup1("183A") ->	{"0","",""};
lookup1("183B") ->	{"0","",""};
lookup1("183C") ->	{"0","",""};
lookup1("183D") ->	{"0","",""};
lookup1("183E") ->	{"0","",""};
lookup1("183F") ->	{"0","",""};
lookup1("1840") ->	{"0","",""};
lookup1("1841") ->	{"0","",""};
lookup1("1842") ->	{"0","",""};
lookup1("1843") ->	{"0","",""};
lookup1("1844") ->	{"0","",""};
lookup1("1845") ->	{"0","",""};
lookup1("1846") ->	{"0","",""};
lookup1("1847") ->	{"0","",""};
lookup1("1848") ->	{"0","",""};
lookup1("1849") ->	{"0","",""};
lookup1("184A") ->	{"0","",""};
lookup1("184B") ->	{"0","",""};
lookup1("184C") ->	{"0","",""};
lookup1("184D") ->	{"0","",""};
lookup1("184E") ->	{"0","",""};
lookup1("184F") ->	{"0","",""};
lookup1("1850") ->	{"0","",""};
lookup1("1851") ->	{"0","",""};
lookup1("1852") ->	{"0","",""};
lookup1("1853") ->	{"0","",""};
lookup1("1854") ->	{"0","",""};
lookup1("1855") ->	{"0","",""};
lookup1("1856") ->	{"0","",""};
lookup1("1857") ->	{"0","",""};
lookup1("1858") ->	{"0","",""};
lookup1("1859") ->	{"0","",""};
lookup1("185A") ->	{"0","",""};
lookup1("185B") ->	{"0","",""};
lookup1("185C") ->	{"0","",""};
lookup1("185D") ->	{"0","",""};
lookup1("185E") ->	{"0","",""};
lookup1("185F") ->	{"0","",""};
lookup1("1860") ->	{"0","",""};
lookup1("1861") ->	{"0","",""};
lookup1("1862") ->	{"0","",""};
lookup1("1863") ->	{"0","",""};
lookup1("1864") ->	{"0","",""};
lookup1("1865") ->	{"0","",""};
lookup1("1866") ->	{"0","",""};
lookup1("1867") ->	{"0","",""};
lookup1("1868") ->	{"0","",""};
lookup1("1869") ->	{"0","",""};
lookup1("186A") ->	{"0","",""};
lookup1("186B") ->	{"0","",""};
lookup1("186C") ->	{"0","",""};
lookup1("186D") ->	{"0","",""};
lookup1("186E") ->	{"0","",""};
lookup1("186F") ->	{"0","",""};
lookup1("1870") ->	{"0","",""};
lookup1("1871") ->	{"0","",""};
lookup1("1872") ->	{"0","",""};
lookup1("1873") ->	{"0","",""};
lookup1("1874") ->	{"0","",""};
lookup1("1875") ->	{"0","",""};
lookup1("1876") ->	{"0","",""};
lookup1("1877") ->	{"0","",""};
lookup1("1880") ->	{"0","",""};
lookup1("1881") ->	{"0","",""};
lookup1("1882") ->	{"0","",""};
lookup1("1883") ->	{"0","",""};
lookup1("1884") ->	{"0","",""};
lookup1("1885") ->	{"0","",""};
lookup1("1886") ->	{"0","",""};
lookup1("1887") ->	{"0","",""};
lookup1("1888") ->	{"0","",""};
lookup1("1889") ->	{"0","",""};
lookup1("188A") ->	{"0","",""};
lookup1("188B") ->	{"0","",""};
lookup1("188C") ->	{"0","",""};
lookup1("188D") ->	{"0","",""};
lookup1("188E") ->	{"0","",""};
lookup1("188F") ->	{"0","",""};
lookup1("1890") ->	{"0","",""};
lookup1("1891") ->	{"0","",""};
lookup1("1892") ->	{"0","",""};
lookup1("1893") ->	{"0","",""};
lookup1("1894") ->	{"0","",""};
lookup1("1895") ->	{"0","",""};
lookup1("1896") ->	{"0","",""};
lookup1("1897") ->	{"0","",""};
lookup1("1898") ->	{"0","",""};
lookup1("1899") ->	{"0","",""};
lookup1("189A") ->	{"0","",""};
lookup1("189B") ->	{"0","",""};
lookup1("189C") ->	{"0","",""};
lookup1("189D") ->	{"0","",""};
lookup1("189E") ->	{"0","",""};
lookup1("189F") ->	{"0","",""};
lookup1("18A0") ->	{"0","",""};
lookup1("18A1") ->	{"0","",""};
lookup1("18A2") ->	{"0","",""};
lookup1("18A3") ->	{"0","",""};
lookup1("18A4") ->	{"0","",""};
lookup1("18A5") ->	{"0","",""};
lookup1("18A6") ->	{"0","",""};
lookup1("18A7") ->	{"0","",""};
lookup1("18A8") ->	{"0","",""};
lookup1("18A9") ->	{"228","",""};
lookup1("18AA") ->	{"0","",""};
lookup1("18B0") ->	{"0","",""};
lookup1("18B1") ->	{"0","",""};
lookup1("18B2") ->	{"0","",""};
lookup1("18B3") ->	{"0","",""};
lookup1("18B4") ->	{"0","",""};
lookup1("18B5") ->	{"0","",""};
lookup1("18B6") ->	{"0","",""};
lookup1("18B7") ->	{"0","",""};
lookup1("18B8") ->	{"0","",""};
lookup1("18B9") ->	{"0","",""};
lookup1("18BA") ->	{"0","",""};
lookup1("18BB") ->	{"0","",""};
lookup1("18BC") ->	{"0","",""};
lookup1("18BD") ->	{"0","",""};
lookup1("18BE") ->	{"0","",""};
lookup1("18BF") ->	{"0","",""};
lookup1("18C0") ->	{"0","",""};
lookup1("18C1") ->	{"0","",""};
lookup1("18C2") ->	{"0","",""};
lookup1("18C3") ->	{"0","",""};
lookup1("18C4") ->	{"0","",""};
lookup1("18C5") ->	{"0","",""};
lookup1("18C6") ->	{"0","",""};
lookup1("18C7") ->	{"0","",""};
lookup1("18C8") ->	{"0","",""};
lookup1("18C9") ->	{"0","",""};
lookup1("18CA") ->	{"0","",""};
lookup1("18CB") ->	{"0","",""};
lookup1("18CC") ->	{"0","",""};
lookup1("18CD") ->	{"0","",""};
lookup1("18CE") ->	{"0","",""};
lookup1("18CF") ->	{"0","",""};
lookup1("18D0") ->	{"0","",""};
lookup1("18D1") ->	{"0","",""};
lookup1("18D2") ->	{"0","",""};
lookup1("18D3") ->	{"0","",""};
lookup1("18D4") ->	{"0","",""};
lookup1("18D5") ->	{"0","",""};
lookup1("18D6") ->	{"0","",""};
lookup1("18D7") ->	{"0","",""};
lookup1("18D8") ->	{"0","",""};
lookup1("18D9") ->	{"0","",""};
lookup1("18DA") ->	{"0","",""};
lookup1("18DB") ->	{"0","",""};
lookup1("18DC") ->	{"0","",""};
lookup1("18DD") ->	{"0","",""};
lookup1("18DE") ->	{"0","",""};
lookup1("18DF") ->	{"0","",""};
lookup1("18E0") ->	{"0","",""};
lookup1("18E1") ->	{"0","",""};
lookup1("18E2") ->	{"0","",""};
lookup1("18E3") ->	{"0","",""};
lookup1("18E4") ->	{"0","",""};
lookup1("18E5") ->	{"0","",""};
lookup1("18E6") ->	{"0","",""};
lookup1("18E7") ->	{"0","",""};
lookup1("18E8") ->	{"0","",""};
lookup1("18E9") ->	{"0","",""};
lookup1("18EA") ->	{"0","",""};
lookup1("18EB") ->	{"0","",""};
lookup1("18EC") ->	{"0","",""};
lookup1("18ED") ->	{"0","",""};
lookup1("18EE") ->	{"0","",""};
lookup1("18EF") ->	{"0","",""};
lookup1("18F0") ->	{"0","",""};
lookup1("18F1") ->	{"0","",""};
lookup1("18F2") ->	{"0","",""};
lookup1("18F3") ->	{"0","",""};
lookup1("18F4") ->	{"0","",""};
lookup1("18F5") ->	{"0","",""};
lookup1("1900") ->	{"0","",""};
lookup1("1901") ->	{"0","",""};
lookup1("1902") ->	{"0","",""};
lookup1("1903") ->	{"0","",""};
lookup1("1904") ->	{"0","",""};
lookup1("1905") ->	{"0","",""};
lookup1("1906") ->	{"0","",""};
lookup1("1907") ->	{"0","",""};
lookup1("1908") ->	{"0","",""};
lookup1("1909") ->	{"0","",""};
lookup1("190A") ->	{"0","",""};
lookup1("190B") ->	{"0","",""};
lookup1("190C") ->	{"0","",""};
lookup1("190D") ->	{"0","",""};
lookup1("190E") ->	{"0","",""};
lookup1("190F") ->	{"0","",""};
lookup1("1910") ->	{"0","",""};
lookup1("1911") ->	{"0","",""};
lookup1("1912") ->	{"0","",""};
lookup1("1913") ->	{"0","",""};
lookup1("1914") ->	{"0","",""};
lookup1("1915") ->	{"0","",""};
lookup1("1916") ->	{"0","",""};
lookup1("1917") ->	{"0","",""};
lookup1("1918") ->	{"0","",""};
lookup1("1919") ->	{"0","",""};
lookup1("191A") ->	{"0","",""};
lookup1("191B") ->	{"0","",""};
lookup1("191C") ->	{"0","",""};
lookup1("191D") ->	{"0","",""};
lookup1("191E") ->	{"0","",""};
lookup1("1920") ->	{"0","",""};
lookup1("1921") ->	{"0","",""};
lookup1("1922") ->	{"0","",""};
lookup1("1923") ->	{"0","",""};
lookup1("1924") ->	{"0","",""};
lookup1("1925") ->	{"0","",""};
lookup1("1926") ->	{"0","",""};
lookup1("1927") ->	{"0","",""};
lookup1("1928") ->	{"0","",""};
lookup1("1929") ->	{"0","",""};
lookup1("192A") ->	{"0","",""};
lookup1("192B") ->	{"0","",""};
lookup1("1930") ->	{"0","",""};
lookup1("1931") ->	{"0","",""};
lookup1("1932") ->	{"0","",""};
lookup1("1933") ->	{"0","",""};
lookup1("1934") ->	{"0","",""};
lookup1("1935") ->	{"0","",""};
lookup1("1936") ->	{"0","",""};
lookup1("1937") ->	{"0","",""};
lookup1("1938") ->	{"0","",""};
lookup1("1939") ->	{"222","",""};
lookup1("193A") ->	{"230","",""};
lookup1("193B") ->	{"220","",""};
lookup1("1940") ->	{"0","",""};
lookup1("1944") ->	{"0","",""};
lookup1("1945") ->	{"0","",""};
lookup1("1946") ->	{"0","",""};
lookup1("1947") ->	{"0","",""};
lookup1("1948") ->	{"0","",""};
lookup1("1949") ->	{"0","",""};
lookup1("194A") ->	{"0","",""};
lookup1("194B") ->	{"0","",""};
lookup1("194C") ->	{"0","",""};
lookup1("194D") ->	{"0","",""};
lookup1("194E") ->	{"0","",""};
lookup1("194F") ->	{"0","",""};
lookup1("1950") ->	{"0","",""};
lookup1("1951") ->	{"0","",""};
lookup1("1952") ->	{"0","",""};
lookup1("1953") ->	{"0","",""};
lookup1("1954") ->	{"0","",""};
lookup1("1955") ->	{"0","",""};
lookup1("1956") ->	{"0","",""};
lookup1("1957") ->	{"0","",""};
lookup1("1958") ->	{"0","",""};
lookup1("1959") ->	{"0","",""};
lookup1("195A") ->	{"0","",""};
lookup1("195B") ->	{"0","",""};
lookup1("195C") ->	{"0","",""};
lookup1("195D") ->	{"0","",""};
lookup1("195E") ->	{"0","",""};
lookup1("195F") ->	{"0","",""};
lookup1("1960") ->	{"0","",""};
lookup1("1961") ->	{"0","",""};
lookup1("1962") ->	{"0","",""};
lookup1("1963") ->	{"0","",""};
lookup1("1964") ->	{"0","",""};
lookup1("1965") ->	{"0","",""};
lookup1("1966") ->	{"0","",""};
lookup1("1967") ->	{"0","",""};
lookup1("1968") ->	{"0","",""};
lookup1("1969") ->	{"0","",""};
lookup1("196A") ->	{"0","",""};
lookup1("196B") ->	{"0","",""};
lookup1("196C") ->	{"0","",""};
lookup1("196D") ->	{"0","",""};
lookup1("1970") ->	{"0","",""};
lookup1("1971") ->	{"0","",""};
lookup1("1972") ->	{"0","",""};
lookup1("1973") ->	{"0","",""};
lookup1("1974") ->	{"0","",""};
lookup1("1980") ->	{"0","",""};
lookup1("1981") ->	{"0","",""};
lookup1("1982") ->	{"0","",""};
lookup1("1983") ->	{"0","",""};
lookup1("1984") ->	{"0","",""};
lookup1("1985") ->	{"0","",""};
lookup1("1986") ->	{"0","",""};
lookup1("1987") ->	{"0","",""};
lookup1("1988") ->	{"0","",""};
lookup1("1989") ->	{"0","",""};
lookup1("198A") ->	{"0","",""};
lookup1("198B") ->	{"0","",""};
lookup1("198C") ->	{"0","",""};
lookup1("198D") ->	{"0","",""};
lookup1("198E") ->	{"0","",""};
lookup1("198F") ->	{"0","",""};
lookup1("1990") ->	{"0","",""};
lookup1("1991") ->	{"0","",""};
lookup1("1992") ->	{"0","",""};
lookup1("1993") ->	{"0","",""};
lookup1("1994") ->	{"0","",""};
lookup1("1995") ->	{"0","",""};
lookup1("1996") ->	{"0","",""};
lookup1("1997") ->	{"0","",""};
lookup1("1998") ->	{"0","",""};
lookup1("1999") ->	{"0","",""};
lookup1("199A") ->	{"0","",""};
lookup1("199B") ->	{"0","",""};
lookup1("199C") ->	{"0","",""};
lookup1("199D") ->	{"0","",""};
lookup1("199E") ->	{"0","",""};
lookup1("199F") ->	{"0","",""};
lookup1("19A0") ->	{"0","",""};
lookup1("19A1") ->	{"0","",""};
lookup1("19A2") ->	{"0","",""};
lookup1("19A3") ->	{"0","",""};
lookup1("19A4") ->	{"0","",""};
lookup1("19A5") ->	{"0","",""};
lookup1("19A6") ->	{"0","",""};
lookup1("19A7") ->	{"0","",""};
lookup1("19A8") ->	{"0","",""};
lookup1("19A9") ->	{"0","",""};
lookup1("19AA") ->	{"0","",""};
lookup1("19AB") ->	{"0","",""};
lookup1("19B0") ->	{"0","",""};
lookup1("19B1") ->	{"0","",""};
lookup1("19B2") ->	{"0","",""};
lookup1("19B3") ->	{"0","",""};
lookup1("19B4") ->	{"0","",""};
lookup1("19B5") ->	{"0","",""};
lookup1("19B6") ->	{"0","",""};
lookup1("19B7") ->	{"0","",""};
lookup1("19B8") ->	{"0","",""};
lookup1("19B9") ->	{"0","",""};
lookup1("19BA") ->	{"0","",""};
lookup1("19BB") ->	{"0","",""};
lookup1("19BC") ->	{"0","",""};
lookup1("19BD") ->	{"0","",""};
lookup1("19BE") ->	{"0","",""};
lookup1("19BF") ->	{"0","",""};
lookup1("19C0") ->	{"0","",""};
lookup1("19C1") ->	{"0","",""};
lookup1("19C2") ->	{"0","",""};
lookup1("19C3") ->	{"0","",""};
lookup1("19C4") ->	{"0","",""};
lookup1("19C5") ->	{"0","",""};
lookup1("19C6") ->	{"0","",""};
lookup1("19C7") ->	{"0","",""};
lookup1("19C8") ->	{"0","",""};
lookup1("19C9") ->	{"0","",""};
lookup1("19D0") ->	{"0","",""};
lookup1("19D1") ->	{"0","",""};
lookup1("19D2") ->	{"0","",""};
lookup1("19D3") ->	{"0","",""};
lookup1("19D4") ->	{"0","",""};
lookup1("19D5") ->	{"0","",""};
lookup1("19D6") ->	{"0","",""};
lookup1("19D7") ->	{"0","",""};
lookup1("19D8") ->	{"0","",""};
lookup1("19D9") ->	{"0","",""};
lookup1("19DA") ->	{"0","",""};
lookup1("19DE") ->	{"0","",""};
lookup1("19DF") ->	{"0","",""};
lookup1("19E0") ->	{"0","",""};
lookup1("19E1") ->	{"0","",""};
lookup1("19E2") ->	{"0","",""};
lookup1("19E3") ->	{"0","",""};
lookup1("19E4") ->	{"0","",""};
lookup1("19E5") ->	{"0","",""};
lookup1("19E6") ->	{"0","",""};
lookup1("19E7") ->	{"0","",""};
lookup1("19E8") ->	{"0","",""};
lookup1("19E9") ->	{"0","",""};
lookup1("19EA") ->	{"0","",""};
lookup1("19EB") ->	{"0","",""};
lookup1("19EC") ->	{"0","",""};
lookup1("19ED") ->	{"0","",""};
lookup1("19EE") ->	{"0","",""};
lookup1("19EF") ->	{"0","",""};
lookup1("19F0") ->	{"0","",""};
lookup1("19F1") ->	{"0","",""};
lookup1("19F2") ->	{"0","",""};
lookup1("19F3") ->	{"0","",""};
lookup1("19F4") ->	{"0","",""};
lookup1("19F5") ->	{"0","",""};
lookup1("19F6") ->	{"0","",""};
lookup1("19F7") ->	{"0","",""};
lookup1("19F8") ->	{"0","",""};
lookup1("19F9") ->	{"0","",""};
lookup1("19FA") ->	{"0","",""};
lookup1("19FB") ->	{"0","",""};
lookup1("19FC") ->	{"0","",""};
lookup1("19FD") ->	{"0","",""};
lookup1("19FE") ->	{"0","",""};
lookup1("19FF") ->	{"0","",""};
lookup1("1A00") ->	{"0","",""};
lookup1("1A01") ->	{"0","",""};
lookup1("1A02") ->	{"0","",""};
lookup1("1A03") ->	{"0","",""};
lookup1("1A04") ->	{"0","",""};
lookup1("1A05") ->	{"0","",""};
lookup1("1A06") ->	{"0","",""};
lookup1("1A07") ->	{"0","",""};
lookup1("1A08") ->	{"0","",""};
lookup1("1A09") ->	{"0","",""};
lookup1("1A0A") ->	{"0","",""};
lookup1("1A0B") ->	{"0","",""};
lookup1("1A0C") ->	{"0","",""};
lookup1("1A0D") ->	{"0","",""};
lookup1("1A0E") ->	{"0","",""};
lookup1("1A0F") ->	{"0","",""};
lookup1("1A10") ->	{"0","",""};
lookup1("1A11") ->	{"0","",""};
lookup1("1A12") ->	{"0","",""};
lookup1("1A13") ->	{"0","",""};
lookup1("1A14") ->	{"0","",""};
lookup1("1A15") ->	{"0","",""};
lookup1("1A16") ->	{"0","",""};
lookup1("1A17") ->	{"230","",""};
lookup1("1A18") ->	{"220","",""};
lookup1("1A19") ->	{"0","",""};
lookup1("1A1A") ->	{"0","",""};
lookup1("1A1B") ->	{"0","",""};
lookup1("1A1E") ->	{"0","",""};
lookup1("1A1F") ->	{"0","",""};
lookup1("1A20") ->	{"0","",""};
lookup1("1A21") ->	{"0","",""};
lookup1("1A22") ->	{"0","",""};
lookup1("1A23") ->	{"0","",""};
lookup1("1A24") ->	{"0","",""};
lookup1("1A25") ->	{"0","",""};
lookup1("1A26") ->	{"0","",""};
lookup1("1A27") ->	{"0","",""};
lookup1("1A28") ->	{"0","",""};
lookup1("1A29") ->	{"0","",""};
lookup1("1A2A") ->	{"0","",""};
lookup1("1A2B") ->	{"0","",""};
lookup1("1A2C") ->	{"0","",""};
lookup1("1A2D") ->	{"0","",""};
lookup1("1A2E") ->	{"0","",""};
lookup1("1A2F") ->	{"0","",""};
lookup1("1A30") ->	{"0","",""};
lookup1("1A31") ->	{"0","",""};
lookup1("1A32") ->	{"0","",""};
lookup1("1A33") ->	{"0","",""};
lookup1("1A34") ->	{"0","",""};
lookup1("1A35") ->	{"0","",""};
lookup1("1A36") ->	{"0","",""};
lookup1("1A37") ->	{"0","",""};
lookup1("1A38") ->	{"0","",""};
lookup1("1A39") ->	{"0","",""};
lookup1("1A3A") ->	{"0","",""};
lookup1("1A3B") ->	{"0","",""};
lookup1("1A3C") ->	{"0","",""};
lookup1("1A3D") ->	{"0","",""};
lookup1("1A3E") ->	{"0","",""};
lookup1("1A3F") ->	{"0","",""};
lookup1("1A40") ->	{"0","",""};
lookup1("1A41") ->	{"0","",""};
lookup1("1A42") ->	{"0","",""};
lookup1("1A43") ->	{"0","",""};
lookup1("1A44") ->	{"0","",""};
lookup1("1A45") ->	{"0","",""};
lookup1("1A46") ->	{"0","",""};
lookup1("1A47") ->	{"0","",""};
lookup1("1A48") ->	{"0","",""};
lookup1("1A49") ->	{"0","",""};
lookup1("1A4A") ->	{"0","",""};
lookup1("1A4B") ->	{"0","",""};
lookup1("1A4C") ->	{"0","",""};
lookup1("1A4D") ->	{"0","",""};
lookup1("1A4E") ->	{"0","",""};
lookup1("1A4F") ->	{"0","",""};
lookup1("1A50") ->	{"0","",""};
lookup1("1A51") ->	{"0","",""};
lookup1("1A52") ->	{"0","",""};
lookup1("1A53") ->	{"0","",""};
lookup1("1A54") ->	{"0","",""};
lookup1("1A55") ->	{"0","",""};
lookup1("1A56") ->	{"0","",""};
lookup1("1A57") ->	{"0","",""};
lookup1("1A58") ->	{"0","",""};
lookup1("1A59") ->	{"0","",""};
lookup1("1A5A") ->	{"0","",""};
lookup1("1A5B") ->	{"0","",""};
lookup1("1A5C") ->	{"0","",""};
lookup1("1A5D") ->	{"0","",""};
lookup1("1A5E") ->	{"0","",""};
lookup1("1A60") ->	{"9","",""};
lookup1("1A61") ->	{"0","",""};
lookup1("1A62") ->	{"0","",""};
lookup1("1A63") ->	{"0","",""};
lookup1("1A64") ->	{"0","",""};
lookup1("1A65") ->	{"0","",""};
lookup1("1A66") ->	{"0","",""};
lookup1("1A67") ->	{"0","",""};
lookup1("1A68") ->	{"0","",""};
lookup1("1A69") ->	{"0","",""};
lookup1("1A6A") ->	{"0","",""};
lookup1("1A6B") ->	{"0","",""};
lookup1("1A6C") ->	{"0","",""};
lookup1("1A6D") ->	{"0","",""};
lookup1("1A6E") ->	{"0","",""};
lookup1("1A6F") ->	{"0","",""};
lookup1("1A70") ->	{"0","",""};
lookup1("1A71") ->	{"0","",""};
lookup1("1A72") ->	{"0","",""};
lookup1("1A73") ->	{"0","",""};
lookup1("1A74") ->	{"0","",""};
lookup1("1A75") ->	{"230","",""};
lookup1("1A76") ->	{"230","",""};
lookup1("1A77") ->	{"230","",""};
lookup1("1A78") ->	{"230","",""};
lookup1("1A79") ->	{"230","",""};
lookup1("1A7A") ->	{"230","",""};
lookup1("1A7B") ->	{"230","",""};
lookup1("1A7C") ->	{"230","",""};
lookup1("1A7F") ->	{"220","",""};
lookup1("1A80") ->	{"0","",""};
lookup1("1A81") ->	{"0","",""};
lookup1("1A82") ->	{"0","",""};
lookup1("1A83") ->	{"0","",""};
lookup1("1A84") ->	{"0","",""};
lookup1("1A85") ->	{"0","",""};
lookup1("1A86") ->	{"0","",""};
lookup1("1A87") ->	{"0","",""};
lookup1("1A88") ->	{"0","",""};
lookup1("1A89") ->	{"0","",""};
lookup1("1A90") ->	{"0","",""};
lookup1("1A91") ->	{"0","",""};
lookup1("1A92") ->	{"0","",""};
lookup1("1A93") ->	{"0","",""};
lookup1("1A94") ->	{"0","",""};
lookup1("1A95") ->	{"0","",""};
lookup1("1A96") ->	{"0","",""};
lookup1("1A97") ->	{"0","",""};
lookup1("1A98") ->	{"0","",""};
lookup1("1A99") ->	{"0","",""};
lookup1("1AA0") ->	{"0","",""};
lookup1("1AA1") ->	{"0","",""};
lookup1("1AA2") ->	{"0","",""};
lookup1("1AA3") ->	{"0","",""};
lookup1("1AA4") ->	{"0","",""};
lookup1("1AA5") ->	{"0","",""};
lookup1("1AA6") ->	{"0","",""};
lookup1("1AA7") ->	{"0","",""};
lookup1("1AA8") ->	{"0","",""};
lookup1("1AA9") ->	{"0","",""};
lookup1("1AAA") ->	{"0","",""};
lookup1("1AAB") ->	{"0","",""};
lookup1("1AAC") ->	{"0","",""};
lookup1("1AAD") ->	{"0","",""};
lookup1("1AB0") ->	{"230","",""};
lookup1("1AB1") ->	{"230","",""};
lookup1("1AB2") ->	{"230","",""};
lookup1("1AB3") ->	{"230","",""};
lookup1("1AB4") ->	{"230","",""};
lookup1("1AB5") ->	{"220","",""};
lookup1("1AB6") ->	{"220","",""};
lookup1("1AB7") ->	{"220","",""};
lookup1("1AB8") ->	{"220","",""};
lookup1("1AB9") ->	{"220","",""};
lookup1("1ABA") ->	{"220","",""};
lookup1("1ABB") ->	{"230","",""};
lookup1("1ABC") ->	{"230","",""};
lookup1("1ABD") ->	{"220","",""};
lookup1("1ABE") ->	{"0","",""};
lookup1("1B00") ->	{"0","",""};
lookup1("1B000") ->	{"0","",""};
lookup1("1B001") ->	{"0","",""};
lookup1("1B01") ->	{"0","",""};
lookup1("1B02") ->	{"0","",""};
lookup1("1B03") ->	{"0","",""};
lookup1("1B04") ->	{"0","",""};
lookup1("1B05") ->	{"0","",""};
lookup1("1B06") ->	{"0","1B05 1B35",""};
lookup1("1B07") ->	{"0","",""};
lookup1("1B08") ->	{"0","1B07 1B35",""};
lookup1("1B09") ->	{"0","",""};
lookup1("1B0A") ->	{"0","1B09 1B35",""};
lookup1("1B0B") ->	{"0","",""};
lookup1("1B0C") ->	{"0","1B0B 1B35",""};
lookup1("1B0D") ->	{"0","",""};
lookup1("1B0E") ->	{"0","1B0D 1B35",""};
lookup1("1B0F") ->	{"0","",""};
lookup1("1B10") ->	{"0","",""};
lookup1("1B11") ->	{"0","",""};
lookup1("1B12") ->	{"0","1B11 1B35",""};
lookup1("1B13") ->	{"0","",""};
lookup1("1B14") ->	{"0","",""};
lookup1("1B15") ->	{"0","",""};
lookup1("1B16") ->	{"0","",""};
lookup1("1B17") ->	{"0","",""};
lookup1("1B18") ->	{"0","",""};
lookup1("1B19") ->	{"0","",""};
lookup1("1B1A") ->	{"0","",""};
lookup1("1B1B") ->	{"0","",""};
lookup1("1B1C") ->	{"0","",""};
lookup1("1B1D") ->	{"0","",""};
lookup1("1B1E") ->	{"0","",""};
lookup1("1B1F") ->	{"0","",""};
lookup1("1B20") ->	{"0","",""};
lookup1("1B21") ->	{"0","",""};
lookup1("1B22") ->	{"0","",""};
lookup1("1B23") ->	{"0","",""};
lookup1("1B24") ->	{"0","",""};
lookup1("1B25") ->	{"0","",""};
lookup1("1B26") ->	{"0","",""};
lookup1("1B27") ->	{"0","",""};
lookup1("1B28") ->	{"0","",""};
lookup1("1B29") ->	{"0","",""};
lookup1("1B2A") ->	{"0","",""};
lookup1("1B2B") ->	{"0","",""};
lookup1("1B2C") ->	{"0","",""};
lookup1("1B2D") ->	{"0","",""};
lookup1("1B2E") ->	{"0","",""};
lookup1("1B2F") ->	{"0","",""};
lookup1("1B30") ->	{"0","",""};
lookup1("1B31") ->	{"0","",""};
lookup1("1B32") ->	{"0","",""};
lookup1("1B33") ->	{"0","",""};
lookup1("1B34") ->	{"7","",""};
lookup1("1B35") ->	{"0","",""};
lookup1("1B36") ->	{"0","",""};
lookup1("1B37") ->	{"0","",""};
lookup1("1B38") ->	{"0","",""};
lookup1("1B39") ->	{"0","",""};
lookup1("1B3A") ->	{"0","",""};
lookup1("1B3B") ->	{"0","1B3A 1B35",""};
lookup1("1B3C") ->	{"0","",""};
lookup1("1B3D") ->	{"0","1B3C 1B35",""};
lookup1("1B3E") ->	{"0","",""};
lookup1("1B3F") ->	{"0","",""};
lookup1("1B40") ->	{"0","1B3E 1B35",""};
lookup1("1B41") ->	{"0","1B3F 1B35",""};
lookup1("1B42") ->	{"0","",""};
lookup1("1B43") ->	{"0","1B42 1B35",""};
lookup1("1B44") ->	{"9","",""};
lookup1("1B45") ->	{"0","",""};
lookup1("1B46") ->	{"0","",""};
lookup1("1B47") ->	{"0","",""};
lookup1("1B48") ->	{"0","",""};
lookup1("1B49") ->	{"0","",""};
lookup1("1B4A") ->	{"0","",""};
lookup1("1B4B") ->	{"0","",""};
lookup1("1B50") ->	{"0","",""};
lookup1("1B51") ->	{"0","",""};
lookup1("1B52") ->	{"0","",""};
lookup1("1B53") ->	{"0","",""};
lookup1("1B54") ->	{"0","",""};
lookup1("1B55") ->	{"0","",""};
lookup1("1B56") ->	{"0","",""};
lookup1("1B57") ->	{"0","",""};
lookup1("1B58") ->	{"0","",""};
lookup1("1B59") ->	{"0","",""};
lookup1("1B5A") ->	{"0","",""};
lookup1("1B5B") ->	{"0","",""};
lookup1("1B5C") ->	{"0","",""};
lookup1("1B5D") ->	{"0","",""};
lookup1("1B5E") ->	{"0","",""};
lookup1("1B5F") ->	{"0","",""};
lookup1("1B60") ->	{"0","",""};
lookup1("1B61") ->	{"0","",""};
lookup1("1B62") ->	{"0","",""};
lookup1("1B63") ->	{"0","",""};
lookup1("1B64") ->	{"0","",""};
lookup1("1B65") ->	{"0","",""};
lookup1("1B66") ->	{"0","",""};
lookup1("1B67") ->	{"0","",""};
lookup1("1B68") ->	{"0","",""};
lookup1("1B69") ->	{"0","",""};
lookup1("1B6A") ->	{"0","",""};
lookup1("1B6B") ->	{"230","",""};
lookup1("1B6C") ->	{"220","",""};
lookup1("1B6D") ->	{"230","",""};
lookup1("1B6E") ->	{"230","",""};
lookup1("1B6F") ->	{"230","",""};
lookup1("1B70") ->	{"230","",""};
lookup1("1B71") ->	{"230","",""};
lookup1("1B72") ->	{"230","",""};
lookup1("1B73") ->	{"230","",""};
lookup1("1B74") ->	{"0","",""};
lookup1("1B75") ->	{"0","",""};
lookup1("1B76") ->	{"0","",""};
lookup1("1B77") ->	{"0","",""};
lookup1("1B78") ->	{"0","",""};
lookup1("1B79") ->	{"0","",""};
lookup1("1B7A") ->	{"0","",""};
lookup1("1B7B") ->	{"0","",""};
lookup1("1B7C") ->	{"0","",""};
lookup1("1B80") ->	{"0","",""};
lookup1("1B81") ->	{"0","",""};
lookup1("1B82") ->	{"0","",""};
lookup1("1B83") ->	{"0","",""};
lookup1("1B84") ->	{"0","",""};
lookup1("1B85") ->	{"0","",""};
lookup1("1B86") ->	{"0","",""};
lookup1("1B87") ->	{"0","",""};
lookup1("1B88") ->	{"0","",""};
lookup1("1B89") ->	{"0","",""};
lookup1("1B8A") ->	{"0","",""};
lookup1("1B8B") ->	{"0","",""};
lookup1("1B8C") ->	{"0","",""};
lookup1("1B8D") ->	{"0","",""};
lookup1("1B8E") ->	{"0","",""};
lookup1("1B8F") ->	{"0","",""};
lookup1("1B90") ->	{"0","",""};
lookup1("1B91") ->	{"0","",""};
lookup1("1B92") ->	{"0","",""};
lookup1("1B93") ->	{"0","",""};
lookup1("1B94") ->	{"0","",""};
lookup1("1B95") ->	{"0","",""};
lookup1("1B96") ->	{"0","",""};
lookup1("1B97") ->	{"0","",""};
lookup1("1B98") ->	{"0","",""};
lookup1("1B99") ->	{"0","",""};
lookup1("1B9A") ->	{"0","",""};
lookup1("1B9B") ->	{"0","",""};
lookup1("1B9C") ->	{"0","",""};
lookup1("1B9D") ->	{"0","",""};
lookup1("1B9E") ->	{"0","",""};
lookup1("1B9F") ->	{"0","",""};
lookup1("1BA0") ->	{"0","",""};
lookup1("1BA1") ->	{"0","",""};
lookup1("1BA2") ->	{"0","",""};
lookup1("1BA3") ->	{"0","",""};
lookup1("1BA4") ->	{"0","",""};
lookup1("1BA5") ->	{"0","",""};
lookup1("1BA6") ->	{"0","",""};
lookup1("1BA7") ->	{"0","",""};
lookup1("1BA8") ->	{"0","",""};
lookup1("1BA9") ->	{"0","",""};
lookup1("1BAA") ->	{"9","",""};
lookup1("1BAB") ->	{"9","",""};
lookup1("1BAC") ->	{"0","",""};
lookup1("1BAD") ->	{"0","",""};
lookup1("1BAE") ->	{"0","",""};
lookup1("1BAF") ->	{"0","",""};
lookup1("1BB0") ->	{"0","",""};
lookup1("1BB1") ->	{"0","",""};
lookup1("1BB2") ->	{"0","",""};
lookup1("1BB3") ->	{"0","",""};
lookup1("1BB4") ->	{"0","",""};
lookup1("1BB5") ->	{"0","",""};
lookup1("1BB6") ->	{"0","",""};
lookup1("1BB7") ->	{"0","",""};
lookup1("1BB8") ->	{"0","",""};
lookup1("1BB9") ->	{"0","",""};
lookup1("1BBA") ->	{"0","",""};
lookup1("1BBB") ->	{"0","",""};
lookup1("1BBC") ->	{"0","",""};
lookup1("1BBD") ->	{"0","",""};
lookup1("1BBE") ->	{"0","",""};
lookup1("1BBF") ->	{"0","",""};
lookup1("1BC0") ->	{"0","",""};
lookup1("1BC00") ->	{"0","",""};
lookup1("1BC01") ->	{"0","",""};
lookup1("1BC02") ->	{"0","",""};
lookup1("1BC03") ->	{"0","",""};
lookup1("1BC04") ->	{"0","",""};
lookup1("1BC05") ->	{"0","",""};
lookup1("1BC06") ->	{"0","",""};
lookup1("1BC07") ->	{"0","",""};
lookup1("1BC08") ->	{"0","",""};
lookup1("1BC09") ->	{"0","",""};
lookup1("1BC0A") ->	{"0","",""};
lookup1("1BC0B") ->	{"0","",""};
lookup1("1BC0C") ->	{"0","",""};
lookup1("1BC0D") ->	{"0","",""};
lookup1("1BC0E") ->	{"0","",""};
lookup1("1BC0F") ->	{"0","",""};
lookup1("1BC1") ->	{"0","",""};
lookup1("1BC10") ->	{"0","",""};
lookup1("1BC11") ->	{"0","",""};
lookup1("1BC12") ->	{"0","",""};
lookup1("1BC13") ->	{"0","",""};
lookup1("1BC14") ->	{"0","",""};
lookup1("1BC15") ->	{"0","",""};
lookup1("1BC16") ->	{"0","",""};
lookup1("1BC17") ->	{"0","",""};
lookup1("1BC18") ->	{"0","",""};
lookup1("1BC19") ->	{"0","",""};
lookup1("1BC1A") ->	{"0","",""};
lookup1("1BC1B") ->	{"0","",""};
lookup1("1BC1C") ->	{"0","",""};
lookup1("1BC1D") ->	{"0","",""};
lookup1("1BC1E") ->	{"0","",""};
lookup1("1BC1F") ->	{"0","",""};
lookup1("1BC2") ->	{"0","",""};
lookup1("1BC20") ->	{"0","",""};
lookup1("1BC21") ->	{"0","",""};
lookup1("1BC22") ->	{"0","",""};
lookup1("1BC23") ->	{"0","",""};
lookup1("1BC24") ->	{"0","",""};
lookup1("1BC25") ->	{"0","",""};
lookup1("1BC26") ->	{"0","",""};
lookup1("1BC27") ->	{"0","",""};
lookup1("1BC28") ->	{"0","",""};
lookup1("1BC29") ->	{"0","",""};
lookup1("1BC2A") ->	{"0","",""};
lookup1("1BC2B") ->	{"0","",""};
lookup1("1BC2C") ->	{"0","",""};
lookup1("1BC2D") ->	{"0","",""};
lookup1("1BC2E") ->	{"0","",""};
lookup1("1BC2F") ->	{"0","",""};
lookup1("1BC3") ->	{"0","",""};
lookup1("1BC30") ->	{"0","",""};
lookup1("1BC31") ->	{"0","",""};
lookup1("1BC32") ->	{"0","",""};
lookup1("1BC33") ->	{"0","",""};
lookup1("1BC34") ->	{"0","",""};
lookup1("1BC35") ->	{"0","",""};
lookup1("1BC36") ->	{"0","",""};
lookup1("1BC37") ->	{"0","",""};
lookup1("1BC38") ->	{"0","",""};
lookup1("1BC39") ->	{"0","",""};
lookup1("1BC3A") ->	{"0","",""};
lookup1("1BC3B") ->	{"0","",""};
lookup1("1BC3C") ->	{"0","",""};
lookup1("1BC3D") ->	{"0","",""};
lookup1("1BC3E") ->	{"0","",""};
lookup1("1BC3F") ->	{"0","",""};
lookup1("1BC4") ->	{"0","",""};
lookup1("1BC40") ->	{"0","",""};
lookup1("1BC41") ->	{"0","",""};
lookup1("1BC42") ->	{"0","",""};
lookup1("1BC43") ->	{"0","",""};
lookup1("1BC44") ->	{"0","",""};
lookup1("1BC45") ->	{"0","",""};
lookup1("1BC46") ->	{"0","",""};
lookup1("1BC47") ->	{"0","",""};
lookup1("1BC48") ->	{"0","",""};
lookup1("1BC49") ->	{"0","",""};
lookup1("1BC4A") ->	{"0","",""};
lookup1("1BC4B") ->	{"0","",""};
lookup1("1BC4C") ->	{"0","",""};
lookup1("1BC4D") ->	{"0","",""};
lookup1("1BC4E") ->	{"0","",""};
lookup1("1BC4F") ->	{"0","",""};
lookup1("1BC5") ->	{"0","",""};
lookup1("1BC50") ->	{"0","",""};
lookup1("1BC51") ->	{"0","",""};
lookup1("1BC52") ->	{"0","",""};
lookup1("1BC53") ->	{"0","",""};
lookup1("1BC54") ->	{"0","",""};
lookup1("1BC55") ->	{"0","",""};
lookup1("1BC56") ->	{"0","",""};
lookup1("1BC57") ->	{"0","",""};
lookup1("1BC58") ->	{"0","",""};
lookup1("1BC59") ->	{"0","",""};
lookup1("1BC5A") ->	{"0","",""};
lookup1("1BC5B") ->	{"0","",""};
lookup1("1BC5C") ->	{"0","",""};
lookup1("1BC5D") ->	{"0","",""};
lookup1("1BC5E") ->	{"0","",""};
lookup1("1BC5F") ->	{"0","",""};
lookup1("1BC6") ->	{"0","",""};
lookup1("1BC60") ->	{"0","",""};
lookup1("1BC61") ->	{"0","",""};
lookup1("1BC62") ->	{"0","",""};
lookup1("1BC63") ->	{"0","",""};
lookup1("1BC64") ->	{"0","",""};
lookup1("1BC65") ->	{"0","",""};
lookup1("1BC66") ->	{"0","",""};
lookup1("1BC67") ->	{"0","",""};
lookup1("1BC68") ->	{"0","",""};
lookup1("1BC69") ->	{"0","",""};
lookup1("1BC6A") ->	{"0","",""};
lookup1("1BC7") ->	{"0","",""};
lookup1("1BC70") ->	{"0","",""};
lookup1("1BC71") ->	{"0","",""};
lookup1("1BC72") ->	{"0","",""};
lookup1("1BC73") ->	{"0","",""};
lookup1("1BC74") ->	{"0","",""};
lookup1("1BC75") ->	{"0","",""};
lookup1("1BC76") ->	{"0","",""};
lookup1("1BC77") ->	{"0","",""};
lookup1("1BC78") ->	{"0","",""};
lookup1("1BC79") ->	{"0","",""};
lookup1("1BC7A") ->	{"0","",""};
lookup1("1BC7B") ->	{"0","",""};
lookup1("1BC7C") ->	{"0","",""};
lookup1("1BC8") ->	{"0","",""};
lookup1("1BC80") ->	{"0","",""};
lookup1("1BC81") ->	{"0","",""};
lookup1("1BC82") ->	{"0","",""};
lookup1("1BC83") ->	{"0","",""};
lookup1("1BC84") ->	{"0","",""};
lookup1("1BC85") ->	{"0","",""};
lookup1("1BC86") ->	{"0","",""};
lookup1("1BC87") ->	{"0","",""};
lookup1("1BC88") ->	{"0","",""};
lookup1("1BC9") ->	{"0","",""};
lookup1("1BC90") ->	{"0","",""};
lookup1("1BC91") ->	{"0","",""};
lookup1("1BC92") ->	{"0","",""};
lookup1("1BC93") ->	{"0","",""};
lookup1("1BC94") ->	{"0","",""};
lookup1("1BC95") ->	{"0","",""};
lookup1("1BC96") ->	{"0","",""};
lookup1("1BC97") ->	{"0","",""};
lookup1("1BC98") ->	{"0","",""};
lookup1("1BC99") ->	{"0","",""};
lookup1("1BC9C") ->	{"0","",""};
lookup1("1BC9D") ->	{"0","",""};
lookup1("1BC9E") ->	{"1","",""};
lookup1("1BC9F") ->	{"0","",""};
lookup1("1BCA") ->	{"0","",""};
lookup1("1BCA0") ->	{"0","",""};
lookup1("1BCA1") ->	{"0","",""};
lookup1("1BCA2") ->	{"0","",""};
lookup1("1BCA3") ->	{"0","",""};
lookup1("1BCB") ->	{"0","",""};
lookup1("1BCC") ->	{"0","",""};
lookup1("1BCD") ->	{"0","",""};
lookup1("1BCE") ->	{"0","",""};
lookup1("1BCF") ->	{"0","",""};
lookup1("1BD0") ->	{"0","",""};
lookup1("1BD1") ->	{"0","",""};
lookup1("1BD2") ->	{"0","",""};
lookup1("1BD3") ->	{"0","",""};
lookup1("1BD4") ->	{"0","",""};
lookup1("1BD5") ->	{"0","",""};
lookup1("1BD6") ->	{"0","",""};
lookup1("1BD7") ->	{"0","",""};
lookup1("1BD8") ->	{"0","",""};
lookup1("1BD9") ->	{"0","",""};
lookup1("1BDA") ->	{"0","",""};
lookup1("1BDB") ->	{"0","",""};
lookup1("1BDC") ->	{"0","",""};
lookup1("1BDD") ->	{"0","",""};
lookup1("1BDE") ->	{"0","",""};
lookup1("1BDF") ->	{"0","",""};
lookup1("1BE0") ->	{"0","",""};
lookup1("1BE1") ->	{"0","",""};
lookup1("1BE2") ->	{"0","",""};
lookup1("1BE3") ->	{"0","",""};
lookup1("1BE4") ->	{"0","",""};
lookup1("1BE5") ->	{"0","",""};
lookup1("1BE6") ->	{"7","",""};
lookup1("1BE7") ->	{"0","",""};
lookup1("1BE8") ->	{"0","",""};
lookup1("1BE9") ->	{"0","",""};
lookup1("1BEA") ->	{"0","",""};
lookup1("1BEB") ->	{"0","",""};
lookup1("1BEC") ->	{"0","",""};
lookup1("1BED") ->	{"0","",""};
lookup1("1BEE") ->	{"0","",""};
lookup1("1BEF") ->	{"0","",""};
lookup1("1BF0") ->	{"0","",""};
lookup1("1BF1") ->	{"0","",""};
lookup1("1BF2") ->	{"9","",""};
lookup1("1BF3") ->	{"9","",""};
lookup1("1BFC") ->	{"0","",""};
lookup1("1BFD") ->	{"0","",""};
lookup1("1BFE") ->	{"0","",""};
lookup1("1BFF") ->	{"0","",""};
lookup1("1C00") ->	{"0","",""};
lookup1("1C01") ->	{"0","",""};
lookup1("1C02") ->	{"0","",""};
lookup1("1C03") ->	{"0","",""};
lookup1("1C04") ->	{"0","",""};
lookup1("1C05") ->	{"0","",""};
lookup1("1C06") ->	{"0","",""};
lookup1("1C07") ->	{"0","",""};
lookup1("1C08") ->	{"0","",""};
lookup1("1C09") ->	{"0","",""};
lookup1("1C0A") ->	{"0","",""};
lookup1("1C0B") ->	{"0","",""};
lookup1("1C0C") ->	{"0","",""};
lookup1("1C0D") ->	{"0","",""};
lookup1("1C0E") ->	{"0","",""};
lookup1("1C0F") ->	{"0","",""};
lookup1("1C10") ->	{"0","",""};
lookup1("1C11") ->	{"0","",""};
lookup1("1C12") ->	{"0","",""};
lookup1("1C13") ->	{"0","",""};
lookup1("1C14") ->	{"0","",""};
lookup1("1C15") ->	{"0","",""};
lookup1("1C16") ->	{"0","",""};
lookup1("1C17") ->	{"0","",""};
lookup1("1C18") ->	{"0","",""};
lookup1("1C19") ->	{"0","",""};
lookup1("1C1A") ->	{"0","",""};
lookup1("1C1B") ->	{"0","",""};
lookup1("1C1C") ->	{"0","",""};
lookup1("1C1D") ->	{"0","",""};
lookup1("1C1E") ->	{"0","",""};
lookup1("1C1F") ->	{"0","",""};
lookup1("1C20") ->	{"0","",""};
lookup1("1C21") ->	{"0","",""};
lookup1("1C22") ->	{"0","",""};
lookup1("1C23") ->	{"0","",""};
lookup1("1C24") ->	{"0","",""};
lookup1("1C25") ->	{"0","",""};
lookup1("1C26") ->	{"0","",""};
lookup1("1C27") ->	{"0","",""};
lookup1("1C28") ->	{"0","",""};
lookup1("1C29") ->	{"0","",""};
lookup1("1C2A") ->	{"0","",""};
lookup1("1C2B") ->	{"0","",""};
lookup1("1C2C") ->	{"0","",""};
lookup1("1C2D") ->	{"0","",""};
lookup1("1C2E") ->	{"0","",""};
lookup1("1C2F") ->	{"0","",""};
lookup1("1C30") ->	{"0","",""};
lookup1("1C31") ->	{"0","",""};
lookup1("1C32") ->	{"0","",""};
lookup1("1C33") ->	{"0","",""};
lookup1("1C34") ->	{"0","",""};
lookup1("1C35") ->	{"0","",""};
lookup1("1C36") ->	{"0","",""};
lookup1("1C37") ->	{"7","",""};
lookup1("1C3B") ->	{"0","",""};
lookup1("1C3C") ->	{"0","",""};
lookup1("1C3D") ->	{"0","",""};
lookup1("1C3E") ->	{"0","",""};
lookup1("1C3F") ->	{"0","",""};
lookup1("1C40") ->	{"0","",""};
lookup1("1C41") ->	{"0","",""};
lookup1("1C42") ->	{"0","",""};
lookup1("1C43") ->	{"0","",""};
lookup1("1C44") ->	{"0","",""};
lookup1("1C45") ->	{"0","",""};
lookup1("1C46") ->	{"0","",""};
lookup1("1C47") ->	{"0","",""};
lookup1("1C48") ->	{"0","",""};
lookup1("1C49") ->	{"0","",""};
lookup1("1C4D") ->	{"0","",""};
lookup1("1C4E") ->	{"0","",""};
lookup1("1C4F") ->	{"0","",""};
lookup1("1C50") ->	{"0","",""};
lookup1("1C51") ->	{"0","",""};
lookup1("1C52") ->	{"0","",""};
lookup1("1C53") ->	{"0","",""};
lookup1("1C54") ->	{"0","",""};
lookup1("1C55") ->	{"0","",""};
lookup1("1C56") ->	{"0","",""};
lookup1("1C57") ->	{"0","",""};
lookup1("1C58") ->	{"0","",""};
lookup1("1C59") ->	{"0","",""};
lookup1("1C5A") ->	{"0","",""};
lookup1("1C5B") ->	{"0","",""};
lookup1("1C5C") ->	{"0","",""};
lookup1("1C5D") ->	{"0","",""};
lookup1("1C5E") ->	{"0","",""};
lookup1("1C5F") ->	{"0","",""};
lookup1("1C60") ->	{"0","",""};
lookup1("1C61") ->	{"0","",""};
lookup1("1C62") ->	{"0","",""};
lookup1("1C63") ->	{"0","",""};
lookup1("1C64") ->	{"0","",""};
lookup1("1C65") ->	{"0","",""};
lookup1("1C66") ->	{"0","",""};
lookup1("1C67") ->	{"0","",""};
lookup1("1C68") ->	{"0","",""};
lookup1("1C69") ->	{"0","",""};
lookup1("1C6A") ->	{"0","",""};
lookup1("1C6B") ->	{"0","",""};
lookup1("1C6C") ->	{"0","",""};
lookup1("1C6D") ->	{"0","",""};
lookup1("1C6E") ->	{"0","",""};
lookup1("1C6F") ->	{"0","",""};
lookup1("1C70") ->	{"0","",""};
lookup1("1C71") ->	{"0","",""};
lookup1("1C72") ->	{"0","",""};
lookup1("1C73") ->	{"0","",""};
lookup1("1C74") ->	{"0","",""};
lookup1("1C75") ->	{"0","",""};
lookup1("1C76") ->	{"0","",""};
lookup1("1C77") ->	{"0","",""};
lookup1("1C78") ->	{"0","",""};
lookup1("1C79") ->	{"0","",""};
lookup1("1C7A") ->	{"0","",""};
lookup1("1C7B") ->	{"0","",""};
lookup1("1C7C") ->	{"0","",""};
lookup1("1C7D") ->	{"0","",""};
lookup1("1C7E") ->	{"0","",""};
lookup1("1C7F") ->	{"0","",""};
lookup1("1CC0") ->	{"0","",""};
lookup1("1CC1") ->	{"0","",""};
lookup1("1CC2") ->	{"0","",""};
lookup1("1CC3") ->	{"0","",""};
lookup1("1CC4") ->	{"0","",""};
lookup1("1CC5") ->	{"0","",""};
lookup1("1CC6") ->	{"0","",""};
lookup1("1CC7") ->	{"0","",""};
lookup1("1CD0") ->	{"230","",""};
lookup1("1CD1") ->	{"230","",""};
lookup1("1CD2") ->	{"230","",""};
lookup1("1CD3") ->	{"0","",""};
lookup1("1CD4") ->	{"1","",""};
lookup1("1CD5") ->	{"220","",""};
lookup1("1CD6") ->	{"220","",""};
lookup1("1CD7") ->	{"220","",""};
lookup1("1CD8") ->	{"220","",""};
lookup1("1CD9") ->	{"220","",""};
lookup1("1CDA") ->	{"230","",""};
lookup1("1CDB") ->	{"230","",""};
lookup1("1CDC") ->	{"220","",""};
lookup1("1CDD") ->	{"220","",""};
lookup1("1CDE") ->	{"220","",""};
lookup1("1CDF") ->	{"220","",""};
lookup1("1CE0") ->	{"230","",""};
lookup1("1CE1") ->	{"0","",""};
lookup1("1CE2") ->	{"1","",""};
lookup1("1CE3") ->	{"1","",""};
lookup1("1CE4") ->	{"1","",""};
lookup1("1CE5") ->	{"1","",""};
lookup1("1CE6") ->	{"1","",""};
lookup1("1CE7") ->	{"1","",""};
lookup1("1CE8") ->	{"1","",""};
lookup1("1CE9") ->	{"0","",""};
lookup1("1CEA") ->	{"0","",""};
lookup1("1CEB") ->	{"0","",""};
lookup1("1CEC") ->	{"0","",""};
lookup1("1CED") ->	{"220","",""};
lookup1("1CEE") ->	{"0","",""};
lookup1("1CEF") ->	{"0","",""};
lookup1("1CF0") ->	{"0","",""};
lookup1("1CF1") ->	{"0","",""};
lookup1("1CF2") ->	{"0","",""};
lookup1("1CF3") ->	{"0","",""};
lookup1("1CF4") ->	{"230","",""};
lookup1("1CF5") ->	{"0","",""};
lookup1("1CF6") ->	{"0","",""};
lookup1("1CF8") ->	{"230","",""};
lookup1("1CF9") ->	{"230","",""};
lookup1("1D00") ->	{"0","",""};
lookup1("1D000") ->	{"0","",""};
lookup1("1D001") ->	{"0","",""};
lookup1("1D002") ->	{"0","",""};
lookup1("1D003") ->	{"0","",""};
lookup1("1D004") ->	{"0","",""};
lookup1("1D005") ->	{"0","",""};
lookup1("1D006") ->	{"0","",""};
lookup1("1D007") ->	{"0","",""};
lookup1("1D008") ->	{"0","",""};
lookup1("1D009") ->	{"0","",""};
lookup1("1D00A") ->	{"0","",""};
lookup1("1D00B") ->	{"0","",""};
lookup1("1D00C") ->	{"0","",""};
lookup1("1D00D") ->	{"0","",""};
lookup1("1D00E") ->	{"0","",""};
lookup1("1D00F") ->	{"0","",""};
lookup1("1D01") ->	{"0","",""};
lookup1("1D010") ->	{"0","",""};
lookup1("1D011") ->	{"0","",""};
lookup1("1D012") ->	{"0","",""};
lookup1("1D013") ->	{"0","",""};
lookup1("1D014") ->	{"0","",""};
lookup1("1D015") ->	{"0","",""};
lookup1("1D016") ->	{"0","",""};
lookup1("1D017") ->	{"0","",""};
lookup1("1D018") ->	{"0","",""};
lookup1("1D019") ->	{"0","",""};
lookup1("1D01A") ->	{"0","",""};
lookup1("1D01B") ->	{"0","",""};
lookup1("1D01C") ->	{"0","",""};
lookup1("1D01D") ->	{"0","",""};
lookup1("1D01E") ->	{"0","",""};
lookup1("1D01F") ->	{"0","",""};
lookup1("1D02") ->	{"0","",""};
lookup1("1D020") ->	{"0","",""};
lookup1("1D021") ->	{"0","",""};
lookup1("1D022") ->	{"0","",""};
lookup1("1D023") ->	{"0","",""};
lookup1("1D024") ->	{"0","",""};
lookup1("1D025") ->	{"0","",""};
lookup1("1D026") ->	{"0","",""};
lookup1("1D027") ->	{"0","",""};
lookup1("1D028") ->	{"0","",""};
lookup1("1D029") ->	{"0","",""};
lookup1("1D02A") ->	{"0","",""};
lookup1("1D02B") ->	{"0","",""};
lookup1("1D02C") ->	{"0","",""};
lookup1("1D02D") ->	{"0","",""};
lookup1("1D02E") ->	{"0","",""};
lookup1("1D02F") ->	{"0","",""};
lookup1("1D03") ->	{"0","",""};
lookup1("1D030") ->	{"0","",""};
lookup1("1D031") ->	{"0","",""};
lookup1("1D032") ->	{"0","",""};
lookup1("1D033") ->	{"0","",""};
lookup1("1D034") ->	{"0","",""};
lookup1("1D035") ->	{"0","",""};
lookup1("1D036") ->	{"0","",""};
lookup1("1D037") ->	{"0","",""};
lookup1("1D038") ->	{"0","",""};
lookup1("1D039") ->	{"0","",""};
lookup1("1D03A") ->	{"0","",""};
lookup1("1D03B") ->	{"0","",""};
lookup1("1D03C") ->	{"0","",""};
lookup1("1D03D") ->	{"0","",""};
lookup1("1D03E") ->	{"0","",""};
lookup1("1D03F") ->	{"0","",""};
lookup1("1D04") ->	{"0","",""};
lookup1("1D040") ->	{"0","",""};
lookup1("1D041") ->	{"0","",""};
lookup1("1D042") ->	{"0","",""};
lookup1("1D043") ->	{"0","",""};
lookup1("1D044") ->	{"0","",""};
lookup1("1D045") ->	{"0","",""};
lookup1("1D046") ->	{"0","",""};
lookup1("1D047") ->	{"0","",""};
lookup1("1D048") ->	{"0","",""};
lookup1("1D049") ->	{"0","",""};
lookup1("1D04A") ->	{"0","",""};
lookup1("1D04B") ->	{"0","",""};
lookup1("1D04C") ->	{"0","",""};
lookup1("1D04D") ->	{"0","",""};
lookup1("1D04E") ->	{"0","",""};
lookup1("1D04F") ->	{"0","",""};
lookup1("1D05") ->	{"0","",""};
lookup1("1D050") ->	{"0","",""};
lookup1("1D051") ->	{"0","",""};
lookup1("1D052") ->	{"0","",""};
lookup1("1D053") ->	{"0","",""};
lookup1("1D054") ->	{"0","",""};
lookup1("1D055") ->	{"0","",""};
lookup1("1D056") ->	{"0","",""};
lookup1("1D057") ->	{"0","",""};
lookup1("1D058") ->	{"0","",""};
lookup1("1D059") ->	{"0","",""};
lookup1("1D05A") ->	{"0","",""};
lookup1("1D05B") ->	{"0","",""};
lookup1("1D05C") ->	{"0","",""};
lookup1("1D05D") ->	{"0","",""};
lookup1("1D05E") ->	{"0","",""};
lookup1("1D05F") ->	{"0","",""};
lookup1("1D06") ->	{"0","",""};
lookup1("1D060") ->	{"0","",""};
lookup1("1D061") ->	{"0","",""};
lookup1("1D062") ->	{"0","",""};
lookup1("1D063") ->	{"0","",""};
lookup1("1D064") ->	{"0","",""};
lookup1("1D065") ->	{"0","",""};
lookup1("1D066") ->	{"0","",""};
lookup1("1D067") ->	{"0","",""};
lookup1("1D068") ->	{"0","",""};
lookup1("1D069") ->	{"0","",""};
lookup1("1D06A") ->	{"0","",""};
lookup1("1D06B") ->	{"0","",""};
lookup1("1D06C") ->	{"0","",""};
lookup1("1D06D") ->	{"0","",""};
lookup1("1D06E") ->	{"0","",""};
lookup1("1D06F") ->	{"0","",""};
lookup1("1D07") ->	{"0","",""};
lookup1("1D070") ->	{"0","",""};
lookup1("1D071") ->	{"0","",""};
lookup1("1D072") ->	{"0","",""};
lookup1("1D073") ->	{"0","",""};
lookup1("1D074") ->	{"0","",""};
lookup1("1D075") ->	{"0","",""};
lookup1("1D076") ->	{"0","",""};
lookup1("1D077") ->	{"0","",""};
lookup1("1D078") ->	{"0","",""};
lookup1("1D079") ->	{"0","",""};
lookup1("1D07A") ->	{"0","",""};
lookup1("1D07B") ->	{"0","",""};
lookup1("1D07C") ->	{"0","",""};
lookup1("1D07D") ->	{"0","",""};
lookup1("1D07E") ->	{"0","",""};
lookup1("1D07F") ->	{"0","",""};
lookup1("1D08") ->	{"0","",""};
lookup1("1D080") ->	{"0","",""};
lookup1("1D081") ->	{"0","",""};
lookup1("1D082") ->	{"0","",""};
lookup1("1D083") ->	{"0","",""};
lookup1("1D084") ->	{"0","",""};
lookup1("1D085") ->	{"0","",""};
lookup1("1D086") ->	{"0","",""};
lookup1("1D087") ->	{"0","",""};
lookup1("1D088") ->	{"0","",""};
lookup1("1D089") ->	{"0","",""};
lookup1("1D08A") ->	{"0","",""};
lookup1("1D08B") ->	{"0","",""};
lookup1("1D08C") ->	{"0","",""};
lookup1("1D08D") ->	{"0","",""};
lookup1("1D08E") ->	{"0","",""};
lookup1("1D08F") ->	{"0","",""};
lookup1("1D09") ->	{"0","",""};
lookup1("1D090") ->	{"0","",""};
lookup1("1D091") ->	{"0","",""};
lookup1("1D092") ->	{"0","",""};
lookup1("1D093") ->	{"0","",""};
lookup1("1D094") ->	{"0","",""};
lookup1("1D095") ->	{"0","",""};
lookup1("1D096") ->	{"0","",""};
lookup1("1D097") ->	{"0","",""};
lookup1("1D098") ->	{"0","",""};
lookup1("1D099") ->	{"0","",""};
lookup1("1D09A") ->	{"0","",""};
lookup1("1D09B") ->	{"0","",""};
lookup1("1D09C") ->	{"0","",""};
lookup1("1D09D") ->	{"0","",""};
lookup1("1D09E") ->	{"0","",""};
lookup1("1D09F") ->	{"0","",""};
lookup1("1D0A") ->	{"0","",""};
lookup1("1D0A0") ->	{"0","",""};
lookup1("1D0A1") ->	{"0","",""};
lookup1("1D0A2") ->	{"0","",""};
lookup1("1D0A3") ->	{"0","",""};
lookup1("1D0A4") ->	{"0","",""};
lookup1("1D0A5") ->	{"0","",""};
lookup1("1D0A6") ->	{"0","",""};
lookup1("1D0A7") ->	{"0","",""};
lookup1("1D0A8") ->	{"0","",""};
lookup1("1D0A9") ->	{"0","",""};
lookup1("1D0AA") ->	{"0","",""};
lookup1("1D0AB") ->	{"0","",""};
lookup1("1D0AC") ->	{"0","",""};
lookup1("1D0AD") ->	{"0","",""};
lookup1("1D0AE") ->	{"0","",""};
lookup1("1D0AF") ->	{"0","",""};
lookup1("1D0B") ->	{"0","",""};
lookup1("1D0B0") ->	{"0","",""};
lookup1("1D0B1") ->	{"0","",""};
lookup1("1D0B2") ->	{"0","",""};
lookup1("1D0B3") ->	{"0","",""};
lookup1("1D0B4") ->	{"0","",""};
lookup1("1D0B5") ->	{"0","",""};
lookup1("1D0B6") ->	{"0","",""};
lookup1("1D0B7") ->	{"0","",""};
lookup1("1D0B8") ->	{"0","",""};
lookup1("1D0B9") ->	{"0","",""};
lookup1("1D0BA") ->	{"0","",""};
lookup1("1D0BB") ->	{"0","",""};
lookup1("1D0BC") ->	{"0","",""};
lookup1("1D0BD") ->	{"0","",""};
lookup1("1D0BE") ->	{"0","",""};
lookup1("1D0BF") ->	{"0","",""};
lookup1("1D0C") ->	{"0","",""};
lookup1("1D0C0") ->	{"0","",""};
lookup1("1D0C1") ->	{"0","",""};
lookup1("1D0C2") ->	{"0","",""};
lookup1("1D0C3") ->	{"0","",""};
lookup1("1D0C4") ->	{"0","",""};
lookup1("1D0C5") ->	{"0","",""};
lookup1("1D0C6") ->	{"0","",""};
lookup1("1D0C7") ->	{"0","",""};
lookup1("1D0C8") ->	{"0","",""};
lookup1("1D0C9") ->	{"0","",""};
lookup1("1D0CA") ->	{"0","",""};
lookup1("1D0CB") ->	{"0","",""};
lookup1("1D0CC") ->	{"0","",""};
lookup1("1D0CD") ->	{"0","",""};
lookup1("1D0CE") ->	{"0","",""};
lookup1("1D0CF") ->	{"0","",""};
lookup1("1D0D") ->	{"0","",""};
lookup1("1D0D0") ->	{"0","",""};
lookup1("1D0D1") ->	{"0","",""};
lookup1("1D0D2") ->	{"0","",""};
lookup1("1D0D3") ->	{"0","",""};
lookup1("1D0D4") ->	{"0","",""};
lookup1("1D0D5") ->	{"0","",""};
lookup1("1D0D6") ->	{"0","",""};
lookup1("1D0D7") ->	{"0","",""};
lookup1("1D0D8") ->	{"0","",""};
lookup1("1D0D9") ->	{"0","",""};
lookup1("1D0DA") ->	{"0","",""};
lookup1("1D0DB") ->	{"0","",""};
lookup1("1D0DC") ->	{"0","",""};
lookup1("1D0DD") ->	{"0","",""};
lookup1("1D0DE") ->	{"0","",""};
lookup1("1D0DF") ->	{"0","",""};
lookup1("1D0E") ->	{"0","",""};
lookup1("1D0E0") ->	{"0","",""};
lookup1("1D0E1") ->	{"0","",""};
lookup1("1D0E2") ->	{"0","",""};
lookup1("1D0E3") ->	{"0","",""};
lookup1("1D0E4") ->	{"0","",""};
lookup1("1D0E5") ->	{"0","",""};
lookup1("1D0E6") ->	{"0","",""};
lookup1("1D0E7") ->	{"0","",""};
lookup1("1D0E8") ->	{"0","",""};
lookup1("1D0E9") ->	{"0","",""};
lookup1("1D0EA") ->	{"0","",""};
lookup1("1D0EB") ->	{"0","",""};
lookup1("1D0EC") ->	{"0","",""};
lookup1("1D0ED") ->	{"0","",""};
lookup1("1D0EE") ->	{"0","",""};
lookup1("1D0EF") ->	{"0","",""};
lookup1("1D0F") ->	{"0","",""};
lookup1("1D0F0") ->	{"0","",""};
lookup1("1D0F1") ->	{"0","",""};
lookup1("1D0F2") ->	{"0","",""};
lookup1("1D0F3") ->	{"0","",""};
lookup1("1D0F4") ->	{"0","",""};
lookup1("1D0F5") ->	{"0","",""};
lookup1("1D10") ->	{"0","",""};
lookup1("1D100") ->	{"0","",""};
lookup1("1D101") ->	{"0","",""};
lookup1("1D102") ->	{"0","",""};
lookup1("1D103") ->	{"0","",""};
lookup1("1D104") ->	{"0","",""};
lookup1("1D105") ->	{"0","",""};
lookup1("1D106") ->	{"0","",""};
lookup1("1D107") ->	{"0","",""};
lookup1("1D108") ->	{"0","",""};
lookup1("1D109") ->	{"0","",""};
lookup1("1D10A") ->	{"0","",""};
lookup1("1D10B") ->	{"0","",""};
lookup1("1D10C") ->	{"0","",""};
lookup1("1D10D") ->	{"0","",""};
lookup1("1D10E") ->	{"0","",""};
lookup1("1D10F") ->	{"0","",""};
lookup1("1D11") ->	{"0","",""};
lookup1("1D110") ->	{"0","",""};
lookup1("1D111") ->	{"0","",""};
lookup1("1D112") ->	{"0","",""};
lookup1("1D113") ->	{"0","",""};
lookup1("1D114") ->	{"0","",""};
lookup1("1D115") ->	{"0","",""};
lookup1("1D116") ->	{"0","",""};
lookup1("1D117") ->	{"0","",""};
lookup1("1D118") ->	{"0","",""};
lookup1("1D119") ->	{"0","",""};
lookup1("1D11A") ->	{"0","",""};
lookup1("1D11B") ->	{"0","",""};
lookup1("1D11C") ->	{"0","",""};
lookup1("1D11D") ->	{"0","",""};
lookup1("1D11E") ->	{"0","",""};
lookup1("1D11F") ->	{"0","",""};
lookup1("1D12") ->	{"0","",""};
lookup1("1D120") ->	{"0","",""};
lookup1("1D121") ->	{"0","",""};
lookup1("1D122") ->	{"0","",""};
lookup1("1D123") ->	{"0","",""};
lookup1("1D124") ->	{"0","",""};
lookup1("1D125") ->	{"0","",""};
lookup1("1D126") ->	{"0","",""};
lookup1("1D129") ->	{"0","",""};
lookup1("1D12A") ->	{"0","",""};
lookup1("1D12B") ->	{"0","",""};
lookup1("1D12C") ->	{"0","",""};
lookup1("1D12D") ->	{"0","",""};
lookup1("1D12E") ->	{"0","",""};
lookup1("1D12F") ->	{"0","",""};
lookup1("1D13") ->	{"0","",""};
lookup1("1D130") ->	{"0","",""};
lookup1("1D131") ->	{"0","",""};
lookup1("1D132") ->	{"0","",""};
lookup1("1D133") ->	{"0","",""};
lookup1("1D134") ->	{"0","",""};
lookup1("1D135") ->	{"0","",""};
lookup1("1D136") ->	{"0","",""};
lookup1("1D137") ->	{"0","",""};
lookup1("1D138") ->	{"0","",""};
lookup1("1D139") ->	{"0","",""};
lookup1("1D13A") ->	{"0","",""};
lookup1("1D13B") ->	{"0","",""};
lookup1("1D13C") ->	{"0","",""};
lookup1("1D13D") ->	{"0","",""};
lookup1("1D13E") ->	{"0","",""};
lookup1("1D13F") ->	{"0","",""};
lookup1("1D14") ->	{"0","",""};
lookup1("1D140") ->	{"0","",""};
lookup1("1D141") ->	{"0","",""};
lookup1("1D142") ->	{"0","",""};
lookup1("1D143") ->	{"0","",""};
lookup1("1D144") ->	{"0","",""};
lookup1("1D145") ->	{"0","",""};
lookup1("1D146") ->	{"0","",""};
lookup1("1D147") ->	{"0","",""};
lookup1("1D148") ->	{"0","",""};
lookup1("1D149") ->	{"0","",""};
lookup1("1D14A") ->	{"0","",""};
lookup1("1D14B") ->	{"0","",""};
lookup1("1D14C") ->	{"0","",""};
lookup1("1D14D") ->	{"0","",""};
lookup1("1D14E") ->	{"0","",""};
lookup1("1D14F") ->	{"0","",""};
lookup1("1D15") ->	{"0","",""};
lookup1("1D150") ->	{"0","",""};
lookup1("1D151") ->	{"0","",""};
lookup1("1D152") ->	{"0","",""};
lookup1("1D153") ->	{"0","",""};
lookup1("1D154") ->	{"0","",""};
lookup1("1D155") ->	{"0","",""};
lookup1("1D156") ->	{"0","",""};
lookup1("1D157") ->	{"0","",""};
lookup1("1D158") ->	{"0","",""};
lookup1("1D159") ->	{"0","",""};
lookup1("1D15A") ->	{"0","",""};
lookup1("1D15B") ->	{"0","",""};
lookup1("1D15C") ->	{"0","",""};
lookup1("1D15D") ->	{"0","",""};
lookup1("1D15E") ->	{"0","1D157 1D165",""};
lookup1("1D15F") ->	{"0","1D158 1D165",""};
lookup1("1D16") ->	{"0","",""};
lookup1("1D160") ->	{"0","1D15F 1D16E",""};
lookup1("1D161") ->	{"0","1D15F 1D16F",""};
lookup1("1D162") ->	{"0","1D15F 1D170",""};
lookup1("1D163") ->	{"0","1D15F 1D171",""};
lookup1("1D164") ->	{"0","1D15F 1D172",""};
lookup1("1D165") ->	{"216","",""};
lookup1("1D166") ->	{"216","",""};
lookup1("1D167") ->	{"1","",""};
lookup1("1D168") ->	{"1","",""};
lookup1("1D169") ->	{"1","",""};
lookup1("1D16A") ->	{"0","",""};
lookup1("1D16B") ->	{"0","",""};
lookup1("1D16C") ->	{"0","",""};
lookup1("1D16D") ->	{"226","",""};
lookup1("1D16E") ->	{"216","",""};
lookup1("1D16F") ->	{"216","",""};
lookup1("1D17") ->	{"0","",""};
lookup1("1D170") ->	{"216","",""};
lookup1("1D171") ->	{"216","",""};
lookup1("1D172") ->	{"216","",""};
lookup1("1D173") ->	{"0","",""};
lookup1("1D174") ->	{"0","",""};
lookup1("1D175") ->	{"0","",""};
lookup1("1D176") ->	{"0","",""};
lookup1("1D177") ->	{"0","",""};
lookup1("1D178") ->	{"0","",""};
lookup1("1D179") ->	{"0","",""};
lookup1("1D17A") ->	{"0","",""};
lookup1("1D17B") ->	{"220","",""};
lookup1("1D17C") ->	{"220","",""};
lookup1("1D17D") ->	{"220","",""};
lookup1("1D17E") ->	{"220","",""};
lookup1("1D17F") ->	{"220","",""};
lookup1("1D18") ->	{"0","",""};
lookup1("1D180") ->	{"220","",""};
lookup1("1D181") ->	{"220","",""};
lookup1("1D182") ->	{"220","",""};
lookup1("1D183") ->	{"0","",""};
lookup1("1D184") ->	{"0","",""};
lookup1("1D185") ->	{"230","",""};
lookup1("1D186") ->	{"230","",""};
lookup1("1D187") ->	{"230","",""};
lookup1("1D188") ->	{"230","",""};
lookup1("1D189") ->	{"230","",""};
lookup1("1D18A") ->	{"220","",""};
lookup1("1D18B") ->	{"220","",""};
lookup1("1D18C") ->	{"0","",""};
lookup1("1D18D") ->	{"0","",""};
lookup1("1D18E") ->	{"0","",""};
lookup1("1D18F") ->	{"0","",""};
lookup1("1D19") ->	{"0","",""};
lookup1("1D190") ->	{"0","",""};
lookup1("1D191") ->	{"0","",""};
lookup1("1D192") ->	{"0","",""};
lookup1("1D193") ->	{"0","",""};
lookup1("1D194") ->	{"0","",""};
lookup1("1D195") ->	{"0","",""};
lookup1("1D196") ->	{"0","",""};
lookup1("1D197") ->	{"0","",""};
lookup1("1D198") ->	{"0","",""};
lookup1("1D199") ->	{"0","",""};
lookup1("1D19A") ->	{"0","",""};
lookup1("1D19B") ->	{"0","",""};
lookup1("1D19C") ->	{"0","",""};
lookup1("1D19D") ->	{"0","",""};
lookup1("1D19E") ->	{"0","",""};
lookup1("1D19F") ->	{"0","",""};
lookup1("1D1A") ->	{"0","",""};
lookup1("1D1A0") ->	{"0","",""};
lookup1("1D1A1") ->	{"0","",""};
lookup1("1D1A2") ->	{"0","",""};
lookup1("1D1A3") ->	{"0","",""};
lookup1("1D1A4") ->	{"0","",""};
lookup1("1D1A5") ->	{"0","",""};
lookup1("1D1A6") ->	{"0","",""};
lookup1("1D1A7") ->	{"0","",""};
lookup1("1D1A8") ->	{"0","",""};
lookup1("1D1A9") ->	{"0","",""};
lookup1("1D1AA") ->	{"230","",""};
lookup1("1D1AB") ->	{"230","",""};
lookup1("1D1AC") ->	{"230","",""};
lookup1("1D1AD") ->	{"230","",""};
lookup1("1D1AE") ->	{"0","",""};
lookup1("1D1AF") ->	{"0","",""};
lookup1("1D1B") ->	{"0","",""};
lookup1("1D1B0") ->	{"0","",""};
lookup1("1D1B1") ->	{"0","",""};
lookup1("1D1B2") ->	{"0","",""};
lookup1("1D1B3") ->	{"0","",""};
lookup1("1D1B4") ->	{"0","",""};
lookup1("1D1B5") ->	{"0","",""};
lookup1("1D1B6") ->	{"0","",""};
lookup1("1D1B7") ->	{"0","",""};
lookup1("1D1B8") ->	{"0","",""};
lookup1("1D1B9") ->	{"0","",""};
lookup1("1D1BA") ->	{"0","",""};
lookup1("1D1BB") ->	{"0","1D1B9 1D165",""};
lookup1("1D1BC") ->	{"0","1D1BA 1D165",""};
lookup1("1D1BD") ->	{"0","1D1BB 1D16E",""};
lookup1("1D1BE") ->	{"0","1D1BC 1D16E",""};
lookup1("1D1BF") ->	{"0","1D1BB 1D16F",""};
lookup1("1D1C") ->	{"0","",""};
lookup1("1D1C0") ->	{"0","1D1BC 1D16F",""};
lookup1("1D1C1") ->	{"0","",""};
lookup1("1D1C2") ->	{"0","",""};
lookup1("1D1C3") ->	{"0","",""};
lookup1("1D1C4") ->	{"0","",""};
lookup1("1D1C5") ->	{"0","",""};
lookup1("1D1C6") ->	{"0","",""};
lookup1("1D1C7") ->	{"0","",""};
lookup1("1D1C8") ->	{"0","",""};
lookup1("1D1C9") ->	{"0","",""};
lookup1("1D1CA") ->	{"0","",""};
lookup1("1D1CB") ->	{"0","",""};
lookup1("1D1CC") ->	{"0","",""};
lookup1("1D1CD") ->	{"0","",""};
lookup1("1D1CE") ->	{"0","",""};
lookup1("1D1CF") ->	{"0","",""};
lookup1("1D1D") ->	{"0","",""};
lookup1("1D1D0") ->	{"0","",""};
lookup1("1D1D1") ->	{"0","",""};
lookup1("1D1D2") ->	{"0","",""};
lookup1("1D1D3") ->	{"0","",""};
lookup1("1D1D4") ->	{"0","",""};
lookup1("1D1D5") ->	{"0","",""};
lookup1("1D1D6") ->	{"0","",""};
lookup1("1D1D7") ->	{"0","",""};
lookup1("1D1D8") ->	{"0","",""};
lookup1("1D1D9") ->	{"0","",""};
lookup1("1D1DA") ->	{"0","",""};
lookup1("1D1DB") ->	{"0","",""};
lookup1("1D1DC") ->	{"0","",""};
lookup1("1D1DD") ->	{"0","",""};
lookup1("1D1E") ->	{"0","",""};
lookup1("1D1F") ->	{"0","",""};
lookup1("1D20") ->	{"0","",""};
lookup1("1D200") ->	{"0","",""};
lookup1("1D201") ->	{"0","",""};
lookup1("1D202") ->	{"0","",""};
lookup1("1D203") ->	{"0","",""};
lookup1("1D204") ->	{"0","",""};
lookup1("1D205") ->	{"0","",""};
lookup1("1D206") ->	{"0","",""};
lookup1("1D207") ->	{"0","",""};
lookup1("1D208") ->	{"0","",""};
lookup1("1D209") ->	{"0","",""};
lookup1("1D20A") ->	{"0","",""};
lookup1("1D20B") ->	{"0","",""};
lookup1("1D20C") ->	{"0","",""};
lookup1("1D20D") ->	{"0","",""};
lookup1("1D20E") ->	{"0","",""};
lookup1("1D20F") ->	{"0","",""};
lookup1("1D21") ->	{"0","",""};
lookup1("1D210") ->	{"0","",""};
lookup1("1D211") ->	{"0","",""};
lookup1("1D212") ->	{"0","",""};
lookup1("1D213") ->	{"0","",""};
lookup1("1D214") ->	{"0","",""};
lookup1("1D215") ->	{"0","",""};
lookup1("1D216") ->	{"0","",""};
lookup1("1D217") ->	{"0","",""};
lookup1("1D218") ->	{"0","",""};
lookup1("1D219") ->	{"0","",""};
lookup1("1D21A") ->	{"0","",""};
lookup1("1D21B") ->	{"0","",""};
lookup1("1D21C") ->	{"0","",""};
lookup1("1D21D") ->	{"0","",""};
lookup1("1D21E") ->	{"0","",""};
lookup1("1D21F") ->	{"0","",""};
lookup1("1D22") ->	{"0","",""};
lookup1("1D220") ->	{"0","",""};
lookup1("1D221") ->	{"0","",""};
lookup1("1D222") ->	{"0","",""};
lookup1("1D223") ->	{"0","",""};
lookup1("1D224") ->	{"0","",""};
lookup1("1D225") ->	{"0","",""};
lookup1("1D226") ->	{"0","",""};
lookup1("1D227") ->	{"0","",""};
lookup1("1D228") ->	{"0","",""};
lookup1("1D229") ->	{"0","",""};
lookup1("1D22A") ->	{"0","",""};
lookup1("1D22B") ->	{"0","",""};
lookup1("1D22C") ->	{"0","",""};
lookup1("1D22D") ->	{"0","",""};
lookup1("1D22E") ->	{"0","",""};
lookup1("1D22F") ->	{"0","",""};
lookup1("1D23") ->	{"0","",""};
lookup1("1D230") ->	{"0","",""};
lookup1("1D231") ->	{"0","",""};
lookup1("1D232") ->	{"0","",""};
lookup1("1D233") ->	{"0","",""};
lookup1("1D234") ->	{"0","",""};
lookup1("1D235") ->	{"0","",""};
lookup1("1D236") ->	{"0","",""};
lookup1("1D237") ->	{"0","",""};
lookup1("1D238") ->	{"0","",""};
lookup1("1D239") ->	{"0","",""};
lookup1("1D23A") ->	{"0","",""};
lookup1("1D23B") ->	{"0","",""};
lookup1("1D23C") ->	{"0","",""};
lookup1("1D23D") ->	{"0","",""};
lookup1("1D23E") ->	{"0","",""};
lookup1("1D23F") ->	{"0","",""};
lookup1("1D24") ->	{"0","",""};
lookup1("1D240") ->	{"0","",""};
lookup1("1D241") ->	{"0","",""};
lookup1("1D242") ->	{"230","",""};
lookup1("1D243") ->	{"230","",""};
lookup1("1D244") ->	{"230","",""};
lookup1("1D245") ->	{"0","",""};
lookup1("1D25") ->	{"0","",""};
lookup1("1D26") ->	{"0","",""};
lookup1("1D27") ->	{"0","",""};
lookup1("1D28") ->	{"0","",""};
lookup1("1D29") ->	{"0","",""};
lookup1("1D2A") ->	{"0","",""};
lookup1("1D2B") ->	{"0","",""};
lookup1("1D2C") ->	{"0","<super> 0041",""};
lookup1("1D2D") ->	{"0","<super> 00C6",""};
lookup1("1D2E") ->	{"0","<super> 0042",""};
lookup1("1D2F") ->	{"0","",""};
lookup1("1D30") ->	{"0","<super> 0044",""};
lookup1("1D300") ->	{"0","",""};
lookup1("1D301") ->	{"0","",""};
lookup1("1D302") ->	{"0","",""};
lookup1("1D303") ->	{"0","",""};
lookup1("1D304") ->	{"0","",""};
lookup1("1D305") ->	{"0","",""};
lookup1("1D306") ->	{"0","",""};
lookup1("1D307") ->	{"0","",""};
lookup1("1D308") ->	{"0","",""};
lookup1("1D309") ->	{"0","",""};
lookup1("1D30A") ->	{"0","",""};
lookup1("1D30B") ->	{"0","",""};
lookup1("1D30C") ->	{"0","",""};
lookup1("1D30D") ->	{"0","",""};
lookup1("1D30E") ->	{"0","",""};
lookup1("1D30F") ->	{"0","",""};
lookup1("1D31") ->	{"0","<super> 0045",""};
lookup1("1D310") ->	{"0","",""};
lookup1("1D311") ->	{"0","",""};
lookup1("1D312") ->	{"0","",""};
lookup1("1D313") ->	{"0","",""};
lookup1("1D314") ->	{"0","",""};
lookup1("1D315") ->	{"0","",""};
lookup1("1D316") ->	{"0","",""};
lookup1("1D317") ->	{"0","",""};
lookup1("1D318") ->	{"0","",""};
lookup1("1D319") ->	{"0","",""};
lookup1("1D31A") ->	{"0","",""};
lookup1("1D31B") ->	{"0","",""};
lookup1("1D31C") ->	{"0","",""};
lookup1("1D31D") ->	{"0","",""};
lookup1("1D31E") ->	{"0","",""};
lookup1("1D31F") ->	{"0","",""};
lookup1("1D32") ->	{"0","<super> 018E",""};
lookup1("1D320") ->	{"0","",""};
lookup1("1D321") ->	{"0","",""};
lookup1("1D322") ->	{"0","",""};
lookup1("1D323") ->	{"0","",""};
lookup1("1D324") ->	{"0","",""};
lookup1("1D325") ->	{"0","",""};
lookup1("1D326") ->	{"0","",""};
lookup1("1D327") ->	{"0","",""};
lookup1("1D328") ->	{"0","",""};
lookup1("1D329") ->	{"0","",""};
lookup1("1D32A") ->	{"0","",""};
lookup1("1D32B") ->	{"0","",""};
lookup1("1D32C") ->	{"0","",""};
lookup1("1D32D") ->	{"0","",""};
lookup1("1D32E") ->	{"0","",""};
lookup1("1D32F") ->	{"0","",""};
lookup1("1D33") ->	{"0","<super> 0047",""};
lookup1("1D330") ->	{"0","",""};
lookup1("1D331") ->	{"0","",""};
lookup1("1D332") ->	{"0","",""};
lookup1("1D333") ->	{"0","",""};
lookup1("1D334") ->	{"0","",""};
lookup1("1D335") ->	{"0","",""};
lookup1("1D336") ->	{"0","",""};
lookup1("1D337") ->	{"0","",""};
lookup1("1D338") ->	{"0","",""};
lookup1("1D339") ->	{"0","",""};
lookup1("1D33A") ->	{"0","",""};
lookup1("1D33B") ->	{"0","",""};
lookup1("1D33C") ->	{"0","",""};
lookup1("1D33D") ->	{"0","",""};
lookup1("1D33E") ->	{"0","",""};
lookup1("1D33F") ->	{"0","",""};
lookup1("1D34") ->	{"0","<super> 0048",""};
lookup1("1D340") ->	{"0","",""};
lookup1("1D341") ->	{"0","",""};
lookup1("1D342") ->	{"0","",""};
lookup1("1D343") ->	{"0","",""};
lookup1("1D344") ->	{"0","",""};
lookup1("1D345") ->	{"0","",""};
lookup1("1D346") ->	{"0","",""};
lookup1("1D347") ->	{"0","",""};
lookup1("1D348") ->	{"0","",""};
lookup1("1D349") ->	{"0","",""};
lookup1("1D34A") ->	{"0","",""};
lookup1("1D34B") ->	{"0","",""};
lookup1("1D34C") ->	{"0","",""};
lookup1("1D34D") ->	{"0","",""};
lookup1("1D34E") ->	{"0","",""};
lookup1("1D34F") ->	{"0","",""};
lookup1("1D35") ->	{"0","<super> 0049",""};
lookup1("1D350") ->	{"0","",""};
lookup1("1D351") ->	{"0","",""};
lookup1("1D352") ->	{"0","",""};
lookup1("1D353") ->	{"0","",""};
lookup1("1D354") ->	{"0","",""};
lookup1("1D355") ->	{"0","",""};
lookup1("1D356") ->	{"0","",""};
lookup1("1D36") ->	{"0","<super> 004A",""};
lookup1("1D360") ->	{"0","",""};
lookup1("1D361") ->	{"0","",""};
lookup1("1D362") ->	{"0","",""};
lookup1("1D363") ->	{"0","",""};
lookup1("1D364") ->	{"0","",""};
lookup1("1D365") ->	{"0","",""};
lookup1("1D366") ->	{"0","",""};
lookup1("1D367") ->	{"0","",""};
lookup1("1D368") ->	{"0","",""};
lookup1("1D369") ->	{"0","",""};
lookup1("1D36A") ->	{"0","",""};
lookup1("1D36B") ->	{"0","",""};
lookup1("1D36C") ->	{"0","",""};
lookup1("1D36D") ->	{"0","",""};
lookup1("1D36E") ->	{"0","",""};
lookup1("1D36F") ->	{"0","",""};
lookup1("1D37") ->	{"0","<super> 004B",""};
lookup1("1D370") ->	{"0","",""};
lookup1("1D371") ->	{"0","",""};
lookup1("1D38") ->	{"0","<super> 004C",""};
lookup1("1D39") ->	{"0","<super> 004D",""};
lookup1("1D3A") ->	{"0","<super> 004E",""};
lookup1("1D3B") ->	{"0","",""};
lookup1("1D3C") ->	{"0","<super> 004F",""};
lookup1("1D3D") ->	{"0","<super> 0222",""};
lookup1("1D3E") ->	{"0","<super> 0050",""};
lookup1("1D3F") ->	{"0","<super> 0052",""};
lookup1("1D40") ->	{"0","<super> 0054",""};
lookup1("1D400") ->	{"0","<font> 0041",""};
lookup1("1D401") ->	{"0","<font> 0042",""};
lookup1("1D402") ->	{"0","<font> 0043",""};
lookup1("1D403") ->	{"0","<font> 0044",""};
lookup1("1D404") ->	{"0","<font> 0045",""};
lookup1("1D405") ->	{"0","<font> 0046",""};
lookup1("1D406") ->	{"0","<font> 0047",""};
lookup1("1D407") ->	{"0","<font> 0048",""};
lookup1("1D408") ->	{"0","<font> 0049",""};
lookup1("1D409") ->	{"0","<font> 004A",""};
lookup1("1D40A") ->	{"0","<font> 004B",""};
lookup1("1D40B") ->	{"0","<font> 004C",""};
lookup1("1D40C") ->	{"0","<font> 004D",""};
lookup1("1D40D") ->	{"0","<font> 004E",""};
lookup1("1D40E") ->	{"0","<font> 004F",""};
lookup1("1D40F") ->	{"0","<font> 0050",""};
lookup1("1D41") ->	{"0","<super> 0055",""};
lookup1("1D410") ->	{"0","<font> 0051",""};
lookup1("1D411") ->	{"0","<font> 0052",""};
lookup1("1D412") ->	{"0","<font> 0053",""};
lookup1("1D413") ->	{"0","<font> 0054",""};
lookup1("1D414") ->	{"0","<font> 0055",""};
lookup1("1D415") ->	{"0","<font> 0056",""};
lookup1("1D416") ->	{"0","<font> 0057",""};
lookup1("1D417") ->	{"0","<font> 0058",""};
lookup1("1D418") ->	{"0","<font> 0059",""};
lookup1("1D419") ->	{"0","<font> 005A",""};
lookup1("1D41A") ->	{"0","<font> 0061",""};
lookup1("1D41B") ->	{"0","<font> 0062",""};
lookup1("1D41C") ->	{"0","<font> 0063",""};
lookup1("1D41D") ->	{"0","<font> 0064",""};
lookup1("1D41E") ->	{"0","<font> 0065",""};
lookup1("1D41F") ->	{"0","<font> 0066",""};
lookup1("1D42") ->	{"0","<super> 0057",""};
lookup1("1D420") ->	{"0","<font> 0067",""};
lookup1("1D421") ->	{"0","<font> 0068",""};
lookup1("1D422") ->	{"0","<font> 0069",""};
lookup1("1D423") ->	{"0","<font> 006A",""};
lookup1("1D424") ->	{"0","<font> 006B",""};
lookup1("1D425") ->	{"0","<font> 006C",""};
lookup1("1D426") ->	{"0","<font> 006D",""};
lookup1("1D427") ->	{"0","<font> 006E",""};
lookup1("1D428") ->	{"0","<font> 006F",""};
lookup1("1D429") ->	{"0","<font> 0070",""};
lookup1("1D42A") ->	{"0","<font> 0071",""};
lookup1("1D42B") ->	{"0","<font> 0072",""};
lookup1("1D42C") ->	{"0","<font> 0073",""};
lookup1("1D42D") ->	{"0","<font> 0074",""};
lookup1("1D42E") ->	{"0","<font> 0075",""};
lookup1("1D42F") ->	{"0","<font> 0076",""};
lookup1("1D43") ->	{"0","<super> 0061",""};
lookup1("1D430") ->	{"0","<font> 0077",""};
lookup1("1D431") ->	{"0","<font> 0078",""};
lookup1("1D432") ->	{"0","<font> 0079",""};
lookup1("1D433") ->	{"0","<font> 007A",""};
lookup1("1D434") ->	{"0","<font> 0041",""};
lookup1("1D435") ->	{"0","<font> 0042",""};
lookup1("1D436") ->	{"0","<font> 0043",""};
lookup1("1D437") ->	{"0","<font> 0044",""};
lookup1("1D438") ->	{"0","<font> 0045",""};
lookup1("1D439") ->	{"0","<font> 0046",""};
lookup1("1D43A") ->	{"0","<font> 0047",""};
lookup1("1D43B") ->	{"0","<font> 0048",""};
lookup1("1D43C") ->	{"0","<font> 0049",""};
lookup1("1D43D") ->	{"0","<font> 004A",""};
lookup1("1D43E") ->	{"0","<font> 004B",""};
lookup1("1D43F") ->	{"0","<font> 004C",""};
lookup1("1D44") ->	{"0","<super> 0250",""};
lookup1("1D440") ->	{"0","<font> 004D",""};
lookup1("1D441") ->	{"0","<font> 004E",""};
lookup1("1D442") ->	{"0","<font> 004F",""};
lookup1("1D443") ->	{"0","<font> 0050",""};
lookup1("1D444") ->	{"0","<font> 0051",""};
lookup1("1D445") ->	{"0","<font> 0052",""};
lookup1("1D446") ->	{"0","<font> 0053",""};
lookup1("1D447") ->	{"0","<font> 0054",""};
lookup1("1D448") ->	{"0","<font> 0055",""};
lookup1("1D449") ->	{"0","<font> 0056",""};
lookup1("1D44A") ->	{"0","<font> 0057",""};
lookup1("1D44B") ->	{"0","<font> 0058",""};
lookup1("1D44C") ->	{"0","<font> 0059",""};
lookup1("1D44D") ->	{"0","<font> 005A",""};
lookup1("1D44E") ->	{"0","<font> 0061",""};
lookup1("1D44F") ->	{"0","<font> 0062",""};
lookup1("1D45") ->	{"0","<super> 0251",""};
lookup1("1D450") ->	{"0","<font> 0063",""};
lookup1("1D451") ->	{"0","<font> 0064",""};
lookup1("1D452") ->	{"0","<font> 0065",""};
lookup1("1D453") ->	{"0","<font> 0066",""};
lookup1("1D454") ->	{"0","<font> 0067",""};
lookup1("1D456") ->	{"0","<font> 0069",""};
lookup1("1D457") ->	{"0","<font> 006A",""};
lookup1("1D458") ->	{"0","<font> 006B",""};
lookup1("1D459") ->	{"0","<font> 006C",""};
lookup1("1D45A") ->	{"0","<font> 006D",""};
lookup1("1D45B") ->	{"0","<font> 006E",""};
lookup1("1D45C") ->	{"0","<font> 006F",""};
lookup1("1D45D") ->	{"0","<font> 0070",""};
lookup1("1D45E") ->	{"0","<font> 0071",""};
lookup1("1D45F") ->	{"0","<font> 0072",""};
lookup1("1D46") ->	{"0","<super> 1D02",""};
lookup1("1D460") ->	{"0","<font> 0073",""};
lookup1("1D461") ->	{"0","<font> 0074",""};
lookup1("1D462") ->	{"0","<font> 0075",""};
lookup1("1D463") ->	{"0","<font> 0076",""};
lookup1("1D464") ->	{"0","<font> 0077",""};
lookup1("1D465") ->	{"0","<font> 0078",""};
lookup1("1D466") ->	{"0","<font> 0079",""};
lookup1("1D467") ->	{"0","<font> 007A",""};
lookup1("1D468") ->	{"0","<font> 0041",""};
lookup1("1D469") ->	{"0","<font> 0042",""};
lookup1("1D46A") ->	{"0","<font> 0043",""};
lookup1("1D46B") ->	{"0","<font> 0044",""};
lookup1("1D46C") ->	{"0","<font> 0045",""};
lookup1("1D46D") ->	{"0","<font> 0046",""};
lookup1("1D46E") ->	{"0","<font> 0047",""};
lookup1("1D46F") ->	{"0","<font> 0048",""};
lookup1("1D47") ->	{"0","<super> 0062",""};
lookup1("1D470") ->	{"0","<font> 0049",""};
lookup1("1D471") ->	{"0","<font> 004A",""};
lookup1("1D472") ->	{"0","<font> 004B",""};
lookup1("1D473") ->	{"0","<font> 004C",""};
lookup1("1D474") ->	{"0","<font> 004D",""};
lookup1("1D475") ->	{"0","<font> 004E",""};
lookup1("1D476") ->	{"0","<font> 004F",""};
lookup1("1D477") ->	{"0","<font> 0050",""};
lookup1("1D478") ->	{"0","<font> 0051",""};
lookup1("1D479") ->	{"0","<font> 0052",""};
lookup1("1D47A") ->	{"0","<font> 0053",""};
lookup1("1D47B") ->	{"0","<font> 0054",""};
lookup1("1D47C") ->	{"0","<font> 0055",""};
lookup1("1D47D") ->	{"0","<font> 0056",""};
lookup1("1D47E") ->	{"0","<font> 0057",""};
lookup1("1D47F") ->	{"0","<font> 0058",""};
lookup1("1D48") ->	{"0","<super> 0064",""};
lookup1("1D480") ->	{"0","<font> 0059",""};
lookup1("1D481") ->	{"0","<font> 005A",""};
lookup1("1D482") ->	{"0","<font> 0061",""};
lookup1("1D483") ->	{"0","<font> 0062",""};
lookup1("1D484") ->	{"0","<font> 0063",""};
lookup1("1D485") ->	{"0","<font> 0064",""};
lookup1("1D486") ->	{"0","<font> 0065",""};
lookup1("1D487") ->	{"0","<font> 0066",""};
lookup1("1D488") ->	{"0","<font> 0067",""};
lookup1("1D489") ->	{"0","<font> 0068",""};
lookup1("1D48A") ->	{"0","<font> 0069",""};
lookup1("1D48B") ->	{"0","<font> 006A",""};
lookup1("1D48C") ->	{"0","<font> 006B",""};
lookup1("1D48D") ->	{"0","<font> 006C",""};
lookup1("1D48E") ->	{"0","<font> 006D",""};
lookup1("1D48F") ->	{"0","<font> 006E",""};
lookup1("1D49") ->	{"0","<super> 0065",""};
lookup1("1D490") ->	{"0","<font> 006F",""};
lookup1("1D491") ->	{"0","<font> 0070",""};
lookup1("1D492") ->	{"0","<font> 0071",""};
lookup1("1D493") ->	{"0","<font> 0072",""};
lookup1("1D494") ->	{"0","<font> 0073",""};
lookup1("1D495") ->	{"0","<font> 0074",""};
lookup1("1D496") ->	{"0","<font> 0075",""};
lookup1("1D497") ->	{"0","<font> 0076",""};
lookup1("1D498") ->	{"0","<font> 0077",""};
lookup1("1D499") ->	{"0","<font> 0078",""};
lookup1("1D49A") ->	{"0","<font> 0079",""};
lookup1("1D49B") ->	{"0","<font> 007A",""};
lookup1("1D49C") ->	{"0","<font> 0041",""};
lookup1("1D49E") ->	{"0","<font> 0043",""};
lookup1("1D49F") ->	{"0","<font> 0044",""};
lookup1("1D4A") ->	{"0","<super> 0259",""};
lookup1("1D4A2") ->	{"0","<font> 0047",""};
lookup1("1D4A5") ->	{"0","<font> 004A",""};
lookup1("1D4A6") ->	{"0","<font> 004B",""};
lookup1("1D4A9") ->	{"0","<font> 004E",""};
lookup1("1D4AA") ->	{"0","<font> 004F",""};
lookup1("1D4AB") ->	{"0","<font> 0050",""};
lookup1("1D4AC") ->	{"0","<font> 0051",""};
lookup1("1D4AE") ->	{"0","<font> 0053",""};
lookup1("1D4AF") ->	{"0","<font> 0054",""};
lookup1("1D4B") ->	{"0","<super> 025B",""};
lookup1("1D4B0") ->	{"0","<font> 0055",""};
lookup1("1D4B1") ->	{"0","<font> 0056",""};
lookup1("1D4B2") ->	{"0","<font> 0057",""};
lookup1("1D4B3") ->	{"0","<font> 0058",""};
lookup1("1D4B4") ->	{"0","<font> 0059",""};
lookup1("1D4B5") ->	{"0","<font> 005A",""};
lookup1("1D4B6") ->	{"0","<font> 0061",""};
lookup1("1D4B7") ->	{"0","<font> 0062",""};
lookup1("1D4B8") ->	{"0","<font> 0063",""};
lookup1("1D4B9") ->	{"0","<font> 0064",""};
lookup1("1D4BB") ->	{"0","<font> 0066",""};
lookup1("1D4BD") ->	{"0","<font> 0068",""};
lookup1("1D4BE") ->	{"0","<font> 0069",""};
lookup1("1D4BF") ->	{"0","<font> 006A",""};
lookup1("1D4C") ->	{"0","<super> 025C",""};
lookup1("1D4C0") ->	{"0","<font> 006B",""};
lookup1("1D4C1") ->	{"0","<font> 006C",""};
lookup1("1D4C2") ->	{"0","<font> 006D",""};
lookup1("1D4C3") ->	{"0","<font> 006E",""};
lookup1("1D4C5") ->	{"0","<font> 0070",""};
lookup1("1D4C6") ->	{"0","<font> 0071",""};
lookup1("1D4C7") ->	{"0","<font> 0072",""};
lookup1("1D4C8") ->	{"0","<font> 0073",""};
lookup1("1D4C9") ->	{"0","<font> 0074",""};
lookup1("1D4CA") ->	{"0","<font> 0075",""};
lookup1("1D4CB") ->	{"0","<font> 0076",""};
lookup1("1D4CC") ->	{"0","<font> 0077",""};
lookup1("1D4CD") ->	{"0","<font> 0078",""};
lookup1("1D4CE") ->	{"0","<font> 0079",""};
lookup1("1D4CF") ->	{"0","<font> 007A",""};
lookup1("1D4D") ->	{"0","<super> 0067",""};
lookup1("1D4D0") ->	{"0","<font> 0041",""};
lookup1("1D4D1") ->	{"0","<font> 0042",""};
lookup1("1D4D2") ->	{"0","<font> 0043",""};
lookup1("1D4D3") ->	{"0","<font> 0044",""};
lookup1("1D4D4") ->	{"0","<font> 0045",""};
lookup1("1D4D5") ->	{"0","<font> 0046",""};
lookup1("1D4D6") ->	{"0","<font> 0047",""};
lookup1("1D4D7") ->	{"0","<font> 0048",""};
lookup1("1D4D8") ->	{"0","<font> 0049",""};
lookup1("1D4D9") ->	{"0","<font> 004A",""};
lookup1("1D4DA") ->	{"0","<font> 004B",""};
lookup1("1D4DB") ->	{"0","<font> 004C",""};
lookup1("1D4DC") ->	{"0","<font> 004D",""};
lookup1("1D4DD") ->	{"0","<font> 004E",""};
lookup1("1D4DE") ->	{"0","<font> 004F",""};
lookup1("1D4DF") ->	{"0","<font> 0050",""};
lookup1("1D4E") ->	{"0","",""};
lookup1("1D4E0") ->	{"0","<font> 0051",""};
lookup1("1D4E1") ->	{"0","<font> 0052",""};
lookup1("1D4E2") ->	{"0","<font> 0053",""};
lookup1("1D4E3") ->	{"0","<font> 0054",""};
lookup1("1D4E4") ->	{"0","<font> 0055",""};
lookup1("1D4E5") ->	{"0","<font> 0056",""};
lookup1("1D4E6") ->	{"0","<font> 0057",""};
lookup1("1D4E7") ->	{"0","<font> 0058",""};
lookup1("1D4E8") ->	{"0","<font> 0059",""};
lookup1("1D4E9") ->	{"0","<font> 005A",""};
lookup1("1D4EA") ->	{"0","<font> 0061",""};
lookup1("1D4EB") ->	{"0","<font> 0062",""};
lookup1("1D4EC") ->	{"0","<font> 0063",""};
lookup1("1D4ED") ->	{"0","<font> 0064",""};
lookup1("1D4EE") ->	{"0","<font> 0065",""};
lookup1("1D4EF") ->	{"0","<font> 0066",""};
lookup1("1D4F") ->	{"0","<super> 006B",""};
lookup1("1D4F0") ->	{"0","<font> 0067",""};
lookup1("1D4F1") ->	{"0","<font> 0068",""};
lookup1("1D4F2") ->	{"0","<font> 0069",""};
lookup1("1D4F3") ->	{"0","<font> 006A",""};
lookup1("1D4F4") ->	{"0","<font> 006B",""};
lookup1("1D4F5") ->	{"0","<font> 006C",""};
lookup1("1D4F6") ->	{"0","<font> 006D",""};
lookup1("1D4F7") ->	{"0","<font> 006E",""};
lookup1("1D4F8") ->	{"0","<font> 006F",""};
lookup1("1D4F9") ->	{"0","<font> 0070",""};
lookup1("1D4FA") ->	{"0","<font> 0071",""};
lookup1("1D4FB") ->	{"0","<font> 0072",""};
lookup1("1D4FC") ->	{"0","<font> 0073",""};
lookup1("1D4FD") ->	{"0","<font> 0074",""};
lookup1("1D4FE") ->	{"0","<font> 0075",""};
lookup1("1D4FF") ->	{"0","<font> 0076",""};
lookup1("1D50") ->	{"0","<super> 006D",""};
lookup1("1D500") ->	{"0","<font> 0077",""};
lookup1("1D501") ->	{"0","<font> 0078",""};
lookup1("1D502") ->	{"0","<font> 0079",""};
lookup1("1D503") ->	{"0","<font> 007A",""};
lookup1("1D504") ->	{"0","<font> 0041",""};
lookup1("1D505") ->	{"0","<font> 0042",""};
lookup1("1D507") ->	{"0","<font> 0044",""};
lookup1("1D508") ->	{"0","<font> 0045",""};
lookup1("1D509") ->	{"0","<font> 0046",""};
lookup1("1D50A") ->	{"0","<font> 0047",""};
lookup1("1D50D") ->	{"0","<font> 004A",""};
lookup1("1D50E") ->	{"0","<font> 004B",""};
lookup1("1D50F") ->	{"0","<font> 004C",""};
lookup1("1D51") ->	{"0","<super> 014B",""};
lookup1("1D510") ->	{"0","<font> 004D",""};
lookup1("1D511") ->	{"0","<font> 004E",""};
lookup1("1D512") ->	{"0","<font> 004F",""};
lookup1("1D513") ->	{"0","<font> 0050",""};
lookup1("1D514") ->	{"0","<font> 0051",""};
lookup1("1D516") ->	{"0","<font> 0053",""};
lookup1("1D517") ->	{"0","<font> 0054",""};
lookup1("1D518") ->	{"0","<font> 0055",""};
lookup1("1D519") ->	{"0","<font> 0056",""};
lookup1("1D51A") ->	{"0","<font> 0057",""};
lookup1("1D51B") ->	{"0","<font> 0058",""};
lookup1("1D51C") ->	{"0","<font> 0059",""};
lookup1("1D51E") ->	{"0","<font> 0061",""};
lookup1("1D51F") ->	{"0","<font> 0062",""};
lookup1("1D52") ->	{"0","<super> 006F",""};
lookup1("1D520") ->	{"0","<font> 0063",""};
lookup1("1D521") ->	{"0","<font> 0064",""};
lookup1("1D522") ->	{"0","<font> 0065",""};
lookup1("1D523") ->	{"0","<font> 0066",""};
lookup1("1D524") ->	{"0","<font> 0067",""};
lookup1("1D525") ->	{"0","<font> 0068",""};
lookup1("1D526") ->	{"0","<font> 0069",""};
lookup1("1D527") ->	{"0","<font> 006A",""};
lookup1("1D528") ->	{"0","<font> 006B",""};
lookup1("1D529") ->	{"0","<font> 006C",""};
lookup1("1D52A") ->	{"0","<font> 006D",""};
lookup1("1D52B") ->	{"0","<font> 006E",""};
lookup1("1D52C") ->	{"0","<font> 006F",""};
lookup1("1D52D") ->	{"0","<font> 0070",""};
lookup1("1D52E") ->	{"0","<font> 0071",""};
lookup1("1D52F") ->	{"0","<font> 0072",""};
lookup1("1D53") ->	{"0","<super> 0254",""};
lookup1("1D530") ->	{"0","<font> 0073",""};
lookup1("1D531") ->	{"0","<font> 0074",""};
lookup1("1D532") ->	{"0","<font> 0075",""};
lookup1("1D533") ->	{"0","<font> 0076",""};
lookup1("1D534") ->	{"0","<font> 0077",""};
lookup1("1D535") ->	{"0","<font> 0078",""};
lookup1("1D536") ->	{"0","<font> 0079",""};
lookup1("1D537") ->	{"0","<font> 007A",""};
lookup1("1D538") ->	{"0","<font> 0041",""};
lookup1("1D539") ->	{"0","<font> 0042",""};
lookup1("1D53B") ->	{"0","<font> 0044",""};
lookup1("1D53C") ->	{"0","<font> 0045",""};
lookup1("1D53D") ->	{"0","<font> 0046",""};
lookup1("1D53E") ->	{"0","<font> 0047",""};
lookup1("1D54") ->	{"0","<super> 1D16",""};
lookup1("1D540") ->	{"0","<font> 0049",""};
lookup1("1D541") ->	{"0","<font> 004A",""};
lookup1("1D542") ->	{"0","<font> 004B",""};
lookup1("1D543") ->	{"0","<font> 004C",""};
lookup1("1D544") ->	{"0","<font> 004D",""};
lookup1("1D546") ->	{"0","<font> 004F",""};
lookup1("1D54A") ->	{"0","<font> 0053",""};
lookup1("1D54B") ->	{"0","<font> 0054",""};
lookup1("1D54C") ->	{"0","<font> 0055",""};
lookup1("1D54D") ->	{"0","<font> 0056",""};
lookup1("1D54E") ->	{"0","<font> 0057",""};
lookup1("1D54F") ->	{"0","<font> 0058",""};
lookup1("1D55") ->	{"0","<super> 1D17",""};
lookup1("1D550") ->	{"0","<font> 0059",""};
lookup1("1D552") ->	{"0","<font> 0061",""};
lookup1("1D553") ->	{"0","<font> 0062",""};
lookup1("1D554") ->	{"0","<font> 0063",""};
lookup1("1D555") ->	{"0","<font> 0064",""};
lookup1("1D556") ->	{"0","<font> 0065",""};
lookup1("1D557") ->	{"0","<font> 0066",""};
lookup1("1D558") ->	{"0","<font> 0067",""};
lookup1("1D559") ->	{"0","<font> 0068",""};
lookup1("1D55A") ->	{"0","<font> 0069",""};
lookup1("1D55B") ->	{"0","<font> 006A",""};
lookup1("1D55C") ->	{"0","<font> 006B",""};
lookup1("1D55D") ->	{"0","<font> 006C",""};
lookup1("1D55E") ->	{"0","<font> 006D",""};
lookup1("1D55F") ->	{"0","<font> 006E",""};
lookup1("1D56") ->	{"0","<super> 0070",""};
lookup1("1D560") ->	{"0","<font> 006F",""};
lookup1("1D561") ->	{"0","<font> 0070",""};
lookup1("1D562") ->	{"0","<font> 0071",""};
lookup1("1D563") ->	{"0","<font> 0072",""};
lookup1("1D564") ->	{"0","<font> 0073",""};
lookup1("1D565") ->	{"0","<font> 0074",""};
lookup1("1D566") ->	{"0","<font> 0075",""};
lookup1("1D567") ->	{"0","<font> 0076",""};
lookup1("1D568") ->	{"0","<font> 0077",""};
lookup1("1D569") ->	{"0","<font> 0078",""};
lookup1("1D56A") ->	{"0","<font> 0079",""};
lookup1("1D56B") ->	{"0","<font> 007A",""};
lookup1("1D56C") ->	{"0","<font> 0041",""};
lookup1("1D56D") ->	{"0","<font> 0042",""};
lookup1("1D56E") ->	{"0","<font> 0043",""};
lookup1("1D56F") ->	{"0","<font> 0044",""};
lookup1("1D57") ->	{"0","<super> 0074",""};
lookup1("1D570") ->	{"0","<font> 0045",""};
lookup1("1D571") ->	{"0","<font> 0046",""};
lookup1("1D572") ->	{"0","<font> 0047",""};
lookup1("1D573") ->	{"0","<font> 0048",""};
lookup1("1D574") ->	{"0","<font> 0049",""};
lookup1("1D575") ->	{"0","<font> 004A",""};
lookup1("1D576") ->	{"0","<font> 004B",""};
lookup1("1D577") ->	{"0","<font> 004C",""};
lookup1("1D578") ->	{"0","<font> 004D",""};
lookup1("1D579") ->	{"0","<font> 004E",""};
lookup1("1D57A") ->	{"0","<font> 004F",""};
lookup1("1D57B") ->	{"0","<font> 0050",""};
lookup1("1D57C") ->	{"0","<font> 0051",""};
lookup1("1D57D") ->	{"0","<font> 0052",""};
lookup1("1D57E") ->	{"0","<font> 0053",""};
lookup1("1D57F") ->	{"0","<font> 0054",""};
lookup1("1D58") ->	{"0","<super> 0075",""};
lookup1("1D580") ->	{"0","<font> 0055",""};
lookup1("1D581") ->	{"0","<font> 0056",""};
lookup1("1D582") ->	{"0","<font> 0057",""};
lookup1("1D583") ->	{"0","<font> 0058",""};
lookup1("1D584") ->	{"0","<font> 0059",""};
lookup1("1D585") ->	{"0","<font> 005A",""};
lookup1("1D586") ->	{"0","<font> 0061",""};
lookup1("1D587") ->	{"0","<font> 0062",""};
lookup1("1D588") ->	{"0","<font> 0063",""};
lookup1("1D589") ->	{"0","<font> 0064",""};
lookup1("1D58A") ->	{"0","<font> 0065",""};
lookup1("1D58B") ->	{"0","<font> 0066",""};
lookup1("1D58C") ->	{"0","<font> 0067",""};
lookup1("1D58D") ->	{"0","<font> 0068",""};
lookup1("1D58E") ->	{"0","<font> 0069",""};
lookup1("1D58F") ->	{"0","<font> 006A",""};
lookup1("1D59") ->	{"0","<super> 1D1D",""};
lookup1("1D590") ->	{"0","<font> 006B",""};
lookup1("1D591") ->	{"0","<font> 006C",""};
lookup1("1D592") ->	{"0","<font> 006D",""};
lookup1("1D593") ->	{"0","<font> 006E",""};
lookup1("1D594") ->	{"0","<font> 006F",""};
lookup1("1D595") ->	{"0","<font> 0070",""};
lookup1("1D596") ->	{"0","<font> 0071",""};
lookup1("1D597") ->	{"0","<font> 0072",""};
lookup1("1D598") ->	{"0","<font> 0073",""};
lookup1("1D599") ->	{"0","<font> 0074",""};
lookup1("1D59A") ->	{"0","<font> 0075",""};
lookup1("1D59B") ->	{"0","<font> 0076",""};
lookup1("1D59C") ->	{"0","<font> 0077",""};
lookup1("1D59D") ->	{"0","<font> 0078",""};
lookup1("1D59E") ->	{"0","<font> 0079",""};
lookup1("1D59F") ->	{"0","<font> 007A",""};
lookup1("1D5A") ->	{"0","<super> 026F",""};
lookup1("1D5A0") ->	{"0","<font> 0041",""};
lookup1("1D5A1") ->	{"0","<font> 0042",""};
lookup1("1D5A2") ->	{"0","<font> 0043",""};
lookup1("1D5A3") ->	{"0","<font> 0044",""};
lookup1("1D5A4") ->	{"0","<font> 0045",""};
lookup1("1D5A5") ->	{"0","<font> 0046",""};
lookup1("1D5A6") ->	{"0","<font> 0047",""};
lookup1("1D5A7") ->	{"0","<font> 0048",""};
lookup1("1D5A8") ->	{"0","<font> 0049",""};
lookup1("1D5A9") ->	{"0","<font> 004A",""};
lookup1("1D5AA") ->	{"0","<font> 004B",""};
lookup1("1D5AB") ->	{"0","<font> 004C",""};
lookup1("1D5AC") ->	{"0","<font> 004D",""};
lookup1("1D5AD") ->	{"0","<font> 004E",""};
lookup1("1D5AE") ->	{"0","<font> 004F",""};
lookup1("1D5AF") ->	{"0","<font> 0050",""};
lookup1("1D5B") ->	{"0","<super> 0076",""};
lookup1("1D5B0") ->	{"0","<font> 0051",""};
lookup1("1D5B1") ->	{"0","<font> 0052",""};
lookup1("1D5B2") ->	{"0","<font> 0053",""};
lookup1("1D5B3") ->	{"0","<font> 0054",""};
lookup1("1D5B4") ->	{"0","<font> 0055",""};
lookup1("1D5B5") ->	{"0","<font> 0056",""};
lookup1("1D5B6") ->	{"0","<font> 0057",""};
lookup1("1D5B7") ->	{"0","<font> 0058",""};
lookup1("1D5B8") ->	{"0","<font> 0059",""};
lookup1("1D5B9") ->	{"0","<font> 005A",""};
lookup1("1D5BA") ->	{"0","<font> 0061",""};
lookup1("1D5BB") ->	{"0","<font> 0062",""};
lookup1("1D5BC") ->	{"0","<font> 0063",""};
lookup1("1D5BD") ->	{"0","<font> 0064",""};
lookup1("1D5BE") ->	{"0","<font> 0065",""};
lookup1("1D5BF") ->	{"0","<font> 0066",""};
lookup1("1D5C") ->	{"0","<super> 1D25",""};
lookup1("1D5C0") ->	{"0","<font> 0067",""};
lookup1("1D5C1") ->	{"0","<font> 0068",""};
lookup1("1D5C2") ->	{"0","<font> 0069",""};
lookup1("1D5C3") ->	{"0","<font> 006A",""};
lookup1("1D5C4") ->	{"0","<font> 006B",""};
lookup1("1D5C5") ->	{"0","<font> 006C",""};
lookup1("1D5C6") ->	{"0","<font> 006D",""};
lookup1("1D5C7") ->	{"0","<font> 006E",""};
lookup1("1D5C8") ->	{"0","<font> 006F",""};
lookup1("1D5C9") ->	{"0","<font> 0070",""};
lookup1("1D5CA") ->	{"0","<font> 0071",""};
lookup1("1D5CB") ->	{"0","<font> 0072",""};
lookup1("1D5CC") ->	{"0","<font> 0073",""};
lookup1("1D5CD") ->	{"0","<font> 0074",""};
lookup1("1D5CE") ->	{"0","<font> 0075",""};
lookup1("1D5CF") ->	{"0","<font> 0076",""};
lookup1("1D5D") ->	{"0","<super> 03B2",""};
lookup1("1D5D0") ->	{"0","<font> 0077",""};
lookup1("1D5D1") ->	{"0","<font> 0078",""};
lookup1("1D5D2") ->	{"0","<font> 0079",""};
lookup1("1D5D3") ->	{"0","<font> 007A",""};
lookup1("1D5D4") ->	{"0","<font> 0041",""};
lookup1("1D5D5") ->	{"0","<font> 0042",""};
lookup1("1D5D6") ->	{"0","<font> 0043",""};
lookup1("1D5D7") ->	{"0","<font> 0044",""};
lookup1("1D5D8") ->	{"0","<font> 0045",""};
lookup1("1D5D9") ->	{"0","<font> 0046",""};
lookup1("1D5DA") ->	{"0","<font> 0047",""};
lookup1("1D5DB") ->	{"0","<font> 0048",""};
lookup1("1D5DC") ->	{"0","<font> 0049",""};
lookup1("1D5DD") ->	{"0","<font> 004A",""};
lookup1("1D5DE") ->	{"0","<font> 004B",""};
lookup1("1D5DF") ->	{"0","<font> 004C",""};
lookup1("1D5E") ->	{"0","<super> 03B3",""};
lookup1("1D5E0") ->	{"0","<font> 004D",""};
lookup1("1D5E1") ->	{"0","<font> 004E",""};
lookup1("1D5E2") ->	{"0","<font> 004F",""};
lookup1("1D5E3") ->	{"0","<font> 0050",""};
lookup1("1D5E4") ->	{"0","<font> 0051",""};
lookup1("1D5E5") ->	{"0","<font> 0052",""};
lookup1("1D5E6") ->	{"0","<font> 0053",""};
lookup1("1D5E7") ->	{"0","<font> 0054",""};
lookup1("1D5E8") ->	{"0","<font> 0055",""};
lookup1("1D5E9") ->	{"0","<font> 0056",""};
lookup1("1D5EA") ->	{"0","<font> 0057",""};
lookup1("1D5EB") ->	{"0","<font> 0058",""};
lookup1("1D5EC") ->	{"0","<font> 0059",""};
lookup1("1D5ED") ->	{"0","<font> 005A",""};
lookup1("1D5EE") ->	{"0","<font> 0061",""};
lookup1("1D5EF") ->	{"0","<font> 0062",""};
lookup1("1D5F") ->	{"0","<super> 03B4",""};
lookup1("1D5F0") ->	{"0","<font> 0063",""};
lookup1("1D5F1") ->	{"0","<font> 0064",""};
lookup1("1D5F2") ->	{"0","<font> 0065",""};
lookup1("1D5F3") ->	{"0","<font> 0066",""};
lookup1("1D5F4") ->	{"0","<font> 0067",""};
lookup1("1D5F5") ->	{"0","<font> 0068",""};
lookup1("1D5F6") ->	{"0","<font> 0069",""};
lookup1("1D5F7") ->	{"0","<font> 006A",""};
lookup1("1D5F8") ->	{"0","<font> 006B",""};
lookup1("1D5F9") ->	{"0","<font> 006C",""};
lookup1("1D5FA") ->	{"0","<font> 006D",""};
lookup1("1D5FB") ->	{"0","<font> 006E",""};
lookup1("1D5FC") ->	{"0","<font> 006F",""};
lookup1("1D5FD") ->	{"0","<font> 0070",""};
lookup1("1D5FE") ->	{"0","<font> 0071",""};
lookup1("1D5FF") ->	{"0","<font> 0072",""};
lookup1("1D60") ->	{"0","<super> 03C6",""};
lookup1("1D600") ->	{"0","<font> 0073",""};
lookup1("1D601") ->	{"0","<font> 0074",""};
lookup1("1D602") ->	{"0","<font> 0075",""};
lookup1("1D603") ->	{"0","<font> 0076",""};
lookup1("1D604") ->	{"0","<font> 0077",""};
lookup1("1D605") ->	{"0","<font> 0078",""};
lookup1("1D606") ->	{"0","<font> 0079",""};
lookup1("1D607") ->	{"0","<font> 007A",""};
lookup1("1D608") ->	{"0","<font> 0041",""};
lookup1("1D609") ->	{"0","<font> 0042",""};
lookup1("1D60A") ->	{"0","<font> 0043",""};
lookup1("1D60B") ->	{"0","<font> 0044",""};
lookup1("1D60C") ->	{"0","<font> 0045",""};
lookup1("1D60D") ->	{"0","<font> 0046",""};
lookup1("1D60E") ->	{"0","<font> 0047",""};
lookup1("1D60F") ->	{"0","<font> 0048",""};
lookup1("1D61") ->	{"0","<super> 03C7",""};
lookup1("1D610") ->	{"0","<font> 0049",""};
lookup1("1D611") ->	{"0","<font> 004A",""};
lookup1("1D612") ->	{"0","<font> 004B",""};
lookup1("1D613") ->	{"0","<font> 004C",""};
lookup1("1D614") ->	{"0","<font> 004D",""};
lookup1("1D615") ->	{"0","<font> 004E",""};
lookup1("1D616") ->	{"0","<font> 004F",""};
lookup1("1D617") ->	{"0","<font> 0050",""};
lookup1("1D618") ->	{"0","<font> 0051",""};
lookup1("1D619") ->	{"0","<font> 0052",""};
lookup1("1D61A") ->	{"0","<font> 0053",""};
lookup1("1D61B") ->	{"0","<font> 0054",""};
lookup1("1D61C") ->	{"0","<font> 0055",""};
lookup1("1D61D") ->	{"0","<font> 0056",""};
lookup1("1D61E") ->	{"0","<font> 0057",""};
lookup1("1D61F") ->	{"0","<font> 0058",""};
lookup1("1D62") ->	{"0","<sub> 0069",""};
lookup1("1D620") ->	{"0","<font> 0059",""};
lookup1("1D621") ->	{"0","<font> 005A",""};
lookup1("1D622") ->	{"0","<font> 0061",""};
lookup1("1D623") ->	{"0","<font> 0062",""};
lookup1("1D624") ->	{"0","<font> 0063",""};
lookup1("1D625") ->	{"0","<font> 0064",""};
lookup1("1D626") ->	{"0","<font> 0065",""};
lookup1("1D627") ->	{"0","<font> 0066",""};
lookup1("1D628") ->	{"0","<font> 0067",""};
lookup1("1D629") ->	{"0","<font> 0068",""};
lookup1("1D62A") ->	{"0","<font> 0069",""};
lookup1("1D62B") ->	{"0","<font> 006A",""};
lookup1("1D62C") ->	{"0","<font> 006B",""};
lookup1("1D62D") ->	{"0","<font> 006C",""};
lookup1("1D62E") ->	{"0","<font> 006D",""};
lookup1("1D62F") ->	{"0","<font> 006E",""};
lookup1("1D63") ->	{"0","<sub> 0072",""};
lookup1("1D630") ->	{"0","<font> 006F",""};
lookup1("1D631") ->	{"0","<font> 0070",""};
lookup1("1D632") ->	{"0","<font> 0071",""};
lookup1("1D633") ->	{"0","<font> 0072",""};
lookup1("1D634") ->	{"0","<font> 0073",""};
lookup1("1D635") ->	{"0","<font> 0074",""};
lookup1("1D636") ->	{"0","<font> 0075",""};
lookup1("1D637") ->	{"0","<font> 0076",""};
lookup1("1D638") ->	{"0","<font> 0077",""};
lookup1("1D639") ->	{"0","<font> 0078",""};
lookup1("1D63A") ->	{"0","<font> 0079",""};
lookup1("1D63B") ->	{"0","<font> 007A",""};
lookup1("1D63C") ->	{"0","<font> 0041",""};
lookup1("1D63D") ->	{"0","<font> 0042",""};
lookup1("1D63E") ->	{"0","<font> 0043",""};
lookup1("1D63F") ->	{"0","<font> 0044",""};
lookup1("1D64") ->	{"0","<sub> 0075",""};
lookup1("1D640") ->	{"0","<font> 0045",""};
lookup1("1D641") ->	{"0","<font> 0046",""};
lookup1("1D642") ->	{"0","<font> 0047",""};
lookup1("1D643") ->	{"0","<font> 0048",""};
lookup1("1D644") ->	{"0","<font> 0049",""};
lookup1("1D645") ->	{"0","<font> 004A",""};
lookup1("1D646") ->	{"0","<font> 004B",""};
lookup1("1D647") ->	{"0","<font> 004C",""};
lookup1("1D648") ->	{"0","<font> 004D",""};
lookup1("1D649") ->	{"0","<font> 004E",""};
lookup1("1D64A") ->	{"0","<font> 004F",""};
lookup1("1D64B") ->	{"0","<font> 0050",""};
lookup1("1D64C") ->	{"0","<font> 0051",""};
lookup1("1D64D") ->	{"0","<font> 0052",""};
lookup1("1D64E") ->	{"0","<font> 0053",""};
lookup1("1D64F") ->	{"0","<font> 0054",""};
lookup1("1D65") ->	{"0","<sub> 0076",""};
lookup1("1D650") ->	{"0","<font> 0055",""};
lookup1("1D651") ->	{"0","<font> 0056",""};
lookup1("1D652") ->	{"0","<font> 0057",""};
lookup1("1D653") ->	{"0","<font> 0058",""};
lookup1("1D654") ->	{"0","<font> 0059",""};
lookup1("1D655") ->	{"0","<font> 005A",""};
lookup1("1D656") ->	{"0","<font> 0061",""};
lookup1("1D657") ->	{"0","<font> 0062",""};
lookup1("1D658") ->	{"0","<font> 0063",""};
lookup1("1D659") ->	{"0","<font> 0064",""};
lookup1("1D65A") ->	{"0","<font> 0065",""};
lookup1("1D65B") ->	{"0","<font> 0066",""};
lookup1("1D65C") ->	{"0","<font> 0067",""};
lookup1("1D65D") ->	{"0","<font> 0068",""};
lookup1("1D65E") ->	{"0","<font> 0069",""};
lookup1("1D65F") ->	{"0","<font> 006A",""};
lookup1("1D66") ->	{"0","<sub> 03B2",""};
lookup1("1D660") ->	{"0","<font> 006B",""};
lookup1("1D661") ->	{"0","<font> 006C",""};
lookup1("1D662") ->	{"0","<font> 006D",""};
lookup1("1D663") ->	{"0","<font> 006E",""};
lookup1("1D664") ->	{"0","<font> 006F",""};
lookup1("1D665") ->	{"0","<font> 0070",""};
lookup1("1D666") ->	{"0","<font> 0071",""};
lookup1("1D667") ->	{"0","<font> 0072",""};
lookup1("1D668") ->	{"0","<font> 0073",""};
lookup1("1D669") ->	{"0","<font> 0074",""};
lookup1("1D66A") ->	{"0","<font> 0075",""};
lookup1("1D66B") ->	{"0","<font> 0076",""};
lookup1("1D66C") ->	{"0","<font> 0077",""};
lookup1("1D66D") ->	{"0","<font> 0078",""};
lookup1("1D66E") ->	{"0","<font> 0079",""};
lookup1("1D66F") ->	{"0","<font> 007A",""};
lookup1("1D67") ->	{"0","<sub> 03B3",""};
lookup1("1D670") ->	{"0","<font> 0041",""};
lookup1("1D671") ->	{"0","<font> 0042",""};
lookup1("1D672") ->	{"0","<font> 0043",""};
lookup1("1D673") ->	{"0","<font> 0044",""};
lookup1("1D674") ->	{"0","<font> 0045",""};
lookup1("1D675") ->	{"0","<font> 0046",""};
lookup1("1D676") ->	{"0","<font> 0047",""};
lookup1("1D677") ->	{"0","<font> 0048",""};
lookup1("1D678") ->	{"0","<font> 0049",""};
lookup1("1D679") ->	{"0","<font> 004A",""};
lookup1("1D67A") ->	{"0","<font> 004B",""};
lookup1("1D67B") ->	{"0","<font> 004C",""};
lookup1("1D67C") ->	{"0","<font> 004D",""};
lookup1("1D67D") ->	{"0","<font> 004E",""};
lookup1("1D67E") ->	{"0","<font> 004F",""};
lookup1("1D67F") ->	{"0","<font> 0050",""};
lookup1("1D68") ->	{"0","<sub> 03C1",""};
lookup1("1D680") ->	{"0","<font> 0051",""};
lookup1("1D681") ->	{"0","<font> 0052",""};
lookup1("1D682") ->	{"0","<font> 0053",""};
lookup1("1D683") ->	{"0","<font> 0054",""};
lookup1("1D684") ->	{"0","<font> 0055",""};
lookup1("1D685") ->	{"0","<font> 0056",""};
lookup1("1D686") ->	{"0","<font> 0057",""};
lookup1("1D687") ->	{"0","<font> 0058",""};
lookup1("1D688") ->	{"0","<font> 0059",""};
lookup1("1D689") ->	{"0","<font> 005A",""};
lookup1("1D68A") ->	{"0","<font> 0061",""};
lookup1("1D68B") ->	{"0","<font> 0062",""};
lookup1("1D68C") ->	{"0","<font> 0063",""};
lookup1("1D68D") ->	{"0","<font> 0064",""};
lookup1("1D68E") ->	{"0","<font> 0065",""};
lookup1("1D68F") ->	{"0","<font> 0066",""};
lookup1("1D69") ->	{"0","<sub> 03C6",""};
lookup1("1D690") ->	{"0","<font> 0067",""};
lookup1("1D691") ->	{"0","<font> 0068",""};
lookup1("1D692") ->	{"0","<font> 0069",""};
lookup1("1D693") ->	{"0","<font> 006A",""};
lookup1("1D694") ->	{"0","<font> 006B",""};
lookup1("1D695") ->	{"0","<font> 006C",""};
lookup1("1D696") ->	{"0","<font> 006D",""};
lookup1("1D697") ->	{"0","<font> 006E",""};
lookup1("1D698") ->	{"0","<font> 006F",""};
lookup1("1D699") ->	{"0","<font> 0070",""};
lookup1("1D69A") ->	{"0","<font> 0071",""};
lookup1("1D69B") ->	{"0","<font> 0072",""};
lookup1("1D69C") ->	{"0","<font> 0073",""};
lookup1("1D69D") ->	{"0","<font> 0074",""};
lookup1("1D69E") ->	{"0","<font> 0075",""};
lookup1("1D69F") ->	{"0","<font> 0076",""};
lookup1("1D6A") ->	{"0","<sub> 03C7",""};
lookup1("1D6A0") ->	{"0","<font> 0077",""};
lookup1("1D6A1") ->	{"0","<font> 0078",""};
lookup1("1D6A2") ->	{"0","<font> 0079",""};
lookup1("1D6A3") ->	{"0","<font> 007A",""};
lookup1("1D6A4") ->	{"0","<font> 0131",""};
lookup1("1D6A5") ->	{"0","<font> 0237",""};
lookup1("1D6A8") ->	{"0","<font> 0391",""};
lookup1("1D6A9") ->	{"0","<font> 0392",""};
lookup1("1D6AA") ->	{"0","<font> 0393",""};
lookup1("1D6AB") ->	{"0","<font> 0394",""};
lookup1("1D6AC") ->	{"0","<font> 0395",""};
lookup1("1D6AD") ->	{"0","<font> 0396",""};
lookup1("1D6AE") ->	{"0","<font> 0397",""};
lookup1("1D6AF") ->	{"0","<font> 0398",""};
lookup1("1D6B") ->	{"0","",""};
lookup1("1D6B0") ->	{"0","<font> 0399",""};
lookup1("1D6B1") ->	{"0","<font> 039A",""};
lookup1("1D6B2") ->	{"0","<font> 039B",""};
lookup1("1D6B3") ->	{"0","<font> 039C",""};
lookup1("1D6B4") ->	{"0","<font> 039D",""};
lookup1("1D6B5") ->	{"0","<font> 039E",""};
lookup1("1D6B6") ->	{"0","<font> 039F",""};
lookup1("1D6B7") ->	{"0","<font> 03A0",""};
lookup1("1D6B8") ->	{"0","<font> 03A1",""};
lookup1("1D6B9") ->	{"0","<font> 03F4",""};
lookup1("1D6BA") ->	{"0","<font> 03A3",""};
lookup1("1D6BB") ->	{"0","<font> 03A4",""};
lookup1("1D6BC") ->	{"0","<font> 03A5",""};
lookup1("1D6BD") ->	{"0","<font> 03A6",""};
lookup1("1D6BE") ->	{"0","<font> 03A7",""};
lookup1("1D6BF") ->	{"0","<font> 03A8",""};
lookup1("1D6C") ->	{"0","",""};
lookup1("1D6C0") ->	{"0","<font> 03A9",""};
lookup1("1D6C1") ->	{"0","<font> 2207",""};
lookup1("1D6C2") ->	{"0","<font> 03B1",""};
lookup1("1D6C3") ->	{"0","<font> 03B2",""};
lookup1("1D6C4") ->	{"0","<font> 03B3",""};
lookup1("1D6C5") ->	{"0","<font> 03B4",""};
lookup1("1D6C6") ->	{"0","<font> 03B5",""};
lookup1("1D6C7") ->	{"0","<font> 03B6",""};
lookup1("1D6C8") ->	{"0","<font> 03B7",""};
lookup1("1D6C9") ->	{"0","<font> 03B8",""};
lookup1("1D6CA") ->	{"0","<font> 03B9",""};
lookup1("1D6CB") ->	{"0","<font> 03BA",""};
lookup1("1D6CC") ->	{"0","<font> 03BB",""};
lookup1("1D6CD") ->	{"0","<font> 03BC",""};
lookup1("1D6CE") ->	{"0","<font> 03BD",""};
lookup1("1D6CF") ->	{"0","<font> 03BE",""};
lookup1("1D6D") ->	{"0","",""};
lookup1("1D6D0") ->	{"0","<font> 03BF",""};
lookup1("1D6D1") ->	{"0","<font> 03C0",""};
lookup1("1D6D2") ->	{"0","<font> 03C1",""};
lookup1("1D6D3") ->	{"0","<font> 03C2",""};
lookup1("1D6D4") ->	{"0","<font> 03C3",""};
lookup1("1D6D5") ->	{"0","<font> 03C4",""};
lookup1("1D6D6") ->	{"0","<font> 03C5",""};
lookup1("1D6D7") ->	{"0","<font> 03C6",""};
lookup1("1D6D8") ->	{"0","<font> 03C7",""};
lookup1("1D6D9") ->	{"0","<font> 03C8",""};
lookup1("1D6DA") ->	{"0","<font> 03C9",""};
lookup1("1D6DB") ->	{"0","<font> 2202",""};
lookup1("1D6DC") ->	{"0","<font> 03F5",""};
lookup1("1D6DD") ->	{"0","<font> 03D1",""};
lookup1("1D6DE") ->	{"0","<font> 03F0",""};
lookup1("1D6DF") ->	{"0","<font> 03D5",""};
lookup1("1D6E") ->	{"0","",""};
lookup1("1D6E0") ->	{"0","<font> 03F1",""};
lookup1("1D6E1") ->	{"0","<font> 03D6",""};
lookup1("1D6E2") ->	{"0","<font> 0391",""};
lookup1("1D6E3") ->	{"0","<font> 0392",""};
lookup1("1D6E4") ->	{"0","<font> 0393",""};
lookup1("1D6E5") ->	{"0","<font> 0394",""};
lookup1("1D6E6") ->	{"0","<font> 0395",""};
lookup1("1D6E7") ->	{"0","<font> 0396",""};
lookup1("1D6E8") ->	{"0","<font> 0397",""};
lookup1("1D6E9") ->	{"0","<font> 0398",""};
lookup1("1D6EA") ->	{"0","<font> 0399",""};
lookup1("1D6EB") ->	{"0","<font> 039A",""};
lookup1("1D6EC") ->	{"0","<font> 039B",""};
lookup1("1D6ED") ->	{"0","<font> 039C",""};
lookup1("1D6EE") ->	{"0","<font> 039D",""};
lookup1("1D6EF") ->	{"0","<font> 039E",""};
lookup1("1D6F") ->	{"0","",""};
lookup1("1D6F0") ->	{"0","<font> 039F",""};
lookup1("1D6F1") ->	{"0","<font> 03A0",""};
lookup1("1D6F2") ->	{"0","<font> 03A1",""};
lookup1("1D6F3") ->	{"0","<font> 03F4",""};
lookup1("1D6F4") ->	{"0","<font> 03A3",""};
lookup1("1D6F5") ->	{"0","<font> 03A4",""};
lookup1("1D6F6") ->	{"0","<font> 03A5",""};
lookup1("1D6F7") ->	{"0","<font> 03A6",""};
lookup1("1D6F8") ->	{"0","<font> 03A7",""};
lookup1("1D6F9") ->	{"0","<font> 03A8",""};
lookup1("1D6FA") ->	{"0","<font> 03A9",""};
lookup1("1D6FB") ->	{"0","<font> 2207",""};
lookup1("1D6FC") ->	{"0","<font> 03B1",""};
lookup1("1D6FD") ->	{"0","<font> 03B2",""};
lookup1("1D6FE") ->	{"0","<font> 03B3",""};
lookup1("1D6FF") ->	{"0","<font> 03B4",""};
lookup1("1D70") ->	{"0","",""};
lookup1("1D700") ->	{"0","<font> 03B5",""};
lookup1("1D701") ->	{"0","<font> 03B6",""};
lookup1("1D702") ->	{"0","<font> 03B7",""};
lookup1("1D703") ->	{"0","<font> 03B8",""};
lookup1("1D704") ->	{"0","<font> 03B9",""};
lookup1("1D705") ->	{"0","<font> 03BA",""};
lookup1("1D706") ->	{"0","<font> 03BB",""};
lookup1("1D707") ->	{"0","<font> 03BC",""};
lookup1("1D708") ->	{"0","<font> 03BD",""};
lookup1("1D709") ->	{"0","<font> 03BE",""};
lookup1("1D70A") ->	{"0","<font> 03BF",""};
lookup1("1D70B") ->	{"0","<font> 03C0",""};
lookup1("1D70C") ->	{"0","<font> 03C1",""};
lookup1("1D70D") ->	{"0","<font> 03C2",""};
lookup1("1D70E") ->	{"0","<font> 03C3",""};
lookup1("1D70F") ->	{"0","<font> 03C4",""};
lookup1("1D71") ->	{"0","",""};
lookup1("1D710") ->	{"0","<font> 03C5",""};
lookup1("1D711") ->	{"0","<font> 03C6",""};
lookup1("1D712") ->	{"0","<font> 03C7",""};
lookup1("1D713") ->	{"0","<font> 03C8",""};
lookup1("1D714") ->	{"0","<font> 03C9",""};
lookup1("1D715") ->	{"0","<font> 2202",""};
lookup1("1D716") ->	{"0","<font> 03F5",""};
lookup1("1D717") ->	{"0","<font> 03D1",""};
lookup1("1D718") ->	{"0","<font> 03F0",""};
lookup1("1D719") ->	{"0","<font> 03D5",""};
lookup1("1D71A") ->	{"0","<font> 03F1",""};
lookup1("1D71B") ->	{"0","<font> 03D6",""};
lookup1("1D71C") ->	{"0","<font> 0391",""};
lookup1("1D71D") ->	{"0","<font> 0392",""};
lookup1("1D71E") ->	{"0","<font> 0393",""};
lookup1("1D71F") ->	{"0","<font> 0394",""};
lookup1("1D72") ->	{"0","",""};
lookup1("1D720") ->	{"0","<font> 0395",""};
lookup1("1D721") ->	{"0","<font> 0396",""};
lookup1("1D722") ->	{"0","<font> 0397",""};
lookup1("1D723") ->	{"0","<font> 0398",""};
lookup1("1D724") ->	{"0","<font> 0399",""};
lookup1("1D725") ->	{"0","<font> 039A",""};
lookup1("1D726") ->	{"0","<font> 039B",""};
lookup1("1D727") ->	{"0","<font> 039C",""};
lookup1("1D728") ->	{"0","<font> 039D",""};
lookup1("1D729") ->	{"0","<font> 039E",""};
lookup1("1D72A") ->	{"0","<font> 039F",""};
lookup1("1D72B") ->	{"0","<font> 03A0",""};
lookup1("1D72C") ->	{"0","<font> 03A1",""};
lookup1("1D72D") ->	{"0","<font> 03F4",""};
lookup1("1D72E") ->	{"0","<font> 03A3",""};
lookup1("1D72F") ->	{"0","<font> 03A4",""};
lookup1("1D73") ->	{"0","",""};
lookup1("1D730") ->	{"0","<font> 03A5",""};
lookup1("1D731") ->	{"0","<font> 03A6",""};
lookup1("1D732") ->	{"0","<font> 03A7",""};
lookup1("1D733") ->	{"0","<font> 03A8",""};
lookup1("1D734") ->	{"0","<font> 03A9",""};
lookup1("1D735") ->	{"0","<font> 2207",""};
lookup1("1D736") ->	{"0","<font> 03B1",""};
lookup1("1D737") ->	{"0","<font> 03B2",""};
lookup1("1D738") ->	{"0","<font> 03B3",""};
lookup1("1D739") ->	{"0","<font> 03B4",""};
lookup1("1D73A") ->	{"0","<font> 03B5",""};
lookup1("1D73B") ->	{"0","<font> 03B6",""};
lookup1("1D73C") ->	{"0","<font> 03B7",""};
lookup1("1D73D") ->	{"0","<font> 03B8",""};
lookup1("1D73E") ->	{"0","<font> 03B9",""};
lookup1("1D73F") ->	{"0","<font> 03BA",""};
lookup1("1D74") ->	{"0","",""};
lookup1("1D740") ->	{"0","<font> 03BB",""};
lookup1("1D741") ->	{"0","<font> 03BC",""};
lookup1("1D742") ->	{"0","<font> 03BD",""};
lookup1("1D743") ->	{"0","<font> 03BE",""};
lookup1("1D744") ->	{"0","<font> 03BF",""};
lookup1("1D745") ->	{"0","<font> 03C0",""};
lookup1("1D746") ->	{"0","<font> 03C1",""};
lookup1("1D747") ->	{"0","<font> 03C2",""};
lookup1("1D748") ->	{"0","<font> 03C3",""};
lookup1("1D749") ->	{"0","<font> 03C4",""};
lookup1("1D74A") ->	{"0","<font> 03C5",""};
lookup1("1D74B") ->	{"0","<font> 03C6",""};
lookup1("1D74C") ->	{"0","<font> 03C7",""};
lookup1("1D74D") ->	{"0","<font> 03C8",""};
lookup1("1D74E") ->	{"0","<font> 03C9",""};
lookup1("1D74F") ->	{"0","<font> 2202",""};
lookup1("1D75") ->	{"0","",""};
lookup1("1D750") ->	{"0","<font> 03F5",""};
lookup1("1D751") ->	{"0","<font> 03D1",""};
lookup1("1D752") ->	{"0","<font> 03F0",""};
lookup1("1D753") ->	{"0","<font> 03D5",""};
lookup1("1D754") ->	{"0","<font> 03F1",""};
lookup1("1D755") ->	{"0","<font> 03D6",""};
lookup1("1D756") ->	{"0","<font> 0391",""};
lookup1("1D757") ->	{"0","<font> 0392",""};
lookup1("1D758") ->	{"0","<font> 0393",""};
lookup1("1D759") ->	{"0","<font> 0394",""};
lookup1("1D75A") ->	{"0","<font> 0395",""};
lookup1("1D75B") ->	{"0","<font> 0396",""};
lookup1("1D75C") ->	{"0","<font> 0397",""};
lookup1("1D75D") ->	{"0","<font> 0398",""};
lookup1("1D75E") ->	{"0","<font> 0399",""};
lookup1("1D75F") ->	{"0","<font> 039A",""};
lookup1("1D76") ->	{"0","",""};
lookup1("1D760") ->	{"0","<font> 039B",""};
lookup1("1D761") ->	{"0","<font> 039C",""};
lookup1("1D762") ->	{"0","<font> 039D",""};
lookup1("1D763") ->	{"0","<font> 039E",""};
lookup1("1D764") ->	{"0","<font> 039F",""};
lookup1("1D765") ->	{"0","<font> 03A0",""};
lookup1("1D766") ->	{"0","<font> 03A1",""};
lookup1("1D767") ->	{"0","<font> 03F4",""};
lookup1("1D768") ->	{"0","<font> 03A3",""};
lookup1("1D769") ->	{"0","<font> 03A4",""};
lookup1("1D76A") ->	{"0","<font> 03A5",""};
lookup1("1D76B") ->	{"0","<font> 03A6",""};
lookup1("1D76C") ->	{"0","<font> 03A7",""};
lookup1("1D76D") ->	{"0","<font> 03A8",""};
lookup1("1D76E") ->	{"0","<font> 03A9",""};
lookup1("1D76F") ->	{"0","<font> 2207",""};
lookup1("1D77") ->	{"0","",""};
lookup1("1D770") ->	{"0","<font> 03B1",""};
lookup1("1D771") ->	{"0","<font> 03B2",""};
lookup1("1D772") ->	{"0","<font> 03B3",""};
lookup1("1D773") ->	{"0","<font> 03B4",""};
lookup1("1D774") ->	{"0","<font> 03B5",""};
lookup1("1D775") ->	{"0","<font> 03B6",""};
lookup1("1D776") ->	{"0","<font> 03B7",""};
lookup1("1D777") ->	{"0","<font> 03B8",""};
lookup1("1D778") ->	{"0","<font> 03B9",""};
lookup1("1D779") ->	{"0","<font> 03BA",""};
lookup1("1D77A") ->	{"0","<font> 03BB",""};
lookup1("1D77B") ->	{"0","<font> 03BC",""};
lookup1("1D77C") ->	{"0","<font> 03BD",""};
lookup1("1D77D") ->	{"0","<font> 03BE",""};
lookup1("1D77E") ->	{"0","<font> 03BF",""};
lookup1("1D77F") ->	{"0","<font> 03C0",""};
lookup1("1D78") ->	{"0","<super> 043D",""};
lookup1("1D780") ->	{"0","<font> 03C1",""};
lookup1("1D781") ->	{"0","<font> 03C2",""};
lookup1("1D782") ->	{"0","<font> 03C3",""};
lookup1("1D783") ->	{"0","<font> 03C4",""};
lookup1("1D784") ->	{"0","<font> 03C5",""};
lookup1("1D785") ->	{"0","<font> 03C6",""};
lookup1("1D786") ->	{"0","<font> 03C7",""};
lookup1("1D787") ->	{"0","<font> 03C8",""};
lookup1("1D788") ->	{"0","<font> 03C9",""};
lookup1("1D789") ->	{"0","<font> 2202",""};
lookup1("1D78A") ->	{"0","<font> 03F5",""};
lookup1("1D78B") ->	{"0","<font> 03D1",""};
lookup1("1D78C") ->	{"0","<font> 03F0",""};
lookup1("1D78D") ->	{"0","<font> 03D5",""};
lookup1("1D78E") ->	{"0","<font> 03F1",""};
lookup1("1D78F") ->	{"0","<font> 03D6",""};
lookup1("1D79") ->	{"0","",""};
lookup1("1D790") ->	{"0","<font> 0391",""};
lookup1("1D791") ->	{"0","<font> 0392",""};
lookup1("1D792") ->	{"0","<font> 0393",""};
lookup1("1D793") ->	{"0","<font> 0394",""};
lookup1("1D794") ->	{"0","<font> 0395",""};
lookup1("1D795") ->	{"0","<font> 0396",""};
lookup1("1D796") ->	{"0","<font> 0397",""};
lookup1("1D797") ->	{"0","<font> 0398",""};
lookup1("1D798") ->	{"0","<font> 0399",""};
lookup1("1D799") ->	{"0","<font> 039A",""};
lookup1("1D79A") ->	{"0","<font> 039B",""};
lookup1("1D79B") ->	{"0","<font> 039C",""};
lookup1("1D79C") ->	{"0","<font> 039D",""};
lookup1("1D79D") ->	{"0","<font> 039E",""};
lookup1("1D79E") ->	{"0","<font> 039F",""};
lookup1("1D79F") ->	{"0","<font> 03A0",""};
lookup1("1D7A") ->	{"0","",""};
lookup1("1D7A0") ->	{"0","<font> 03A1",""};
lookup1("1D7A1") ->	{"0","<font> 03F4",""};
lookup1("1D7A2") ->	{"0","<font> 03A3",""};
lookup1("1D7A3") ->	{"0","<font> 03A4",""};
lookup1("1D7A4") ->	{"0","<font> 03A5",""};
lookup1("1D7A5") ->	{"0","<font> 03A6",""};
lookup1("1D7A6") ->	{"0","<font> 03A7",""};
lookup1("1D7A7") ->	{"0","<font> 03A8",""};
lookup1("1D7A8") ->	{"0","<font> 03A9",""};
lookup1("1D7A9") ->	{"0","<font> 2207",""};
lookup1("1D7AA") ->	{"0","<font> 03B1",""};
lookup1("1D7AB") ->	{"0","<font> 03B2",""};
lookup1("1D7AC") ->	{"0","<font> 03B3",""};
lookup1("1D7AD") ->	{"0","<font> 03B4",""};
lookup1("1D7AE") ->	{"0","<font> 03B5",""};
lookup1("1D7AF") ->	{"0","<font> 03B6",""};
lookup1("1D7B") ->	{"0","",""};
lookup1("1D7B0") ->	{"0","<font> 03B7",""};
lookup1("1D7B1") ->	{"0","<font> 03B8",""};
lookup1("1D7B2") ->	{"0","<font> 03B9",""};
lookup1("1D7B3") ->	{"0","<font> 03BA",""};
lookup1("1D7B4") ->	{"0","<font> 03BB",""};
lookup1("1D7B5") ->	{"0","<font> 03BC",""};
lookup1("1D7B6") ->	{"0","<font> 03BD",""};
lookup1("1D7B7") ->	{"0","<font> 03BE",""};
lookup1("1D7B8") ->	{"0","<font> 03BF",""};
lookup1("1D7B9") ->	{"0","<font> 03C0",""};
lookup1("1D7BA") ->	{"0","<font> 03C1",""};
lookup1("1D7BB") ->	{"0","<font> 03C2",""};
lookup1("1D7BC") ->	{"0","<font> 03C3",""};
lookup1("1D7BD") ->	{"0","<font> 03C4",""};
lookup1("1D7BE") ->	{"0","<font> 03C5",""};
lookup1("1D7BF") ->	{"0","<font> 03C6",""};
lookup1("1D7C") ->	{"0","",""};
lookup1("1D7C0") ->	{"0","<font> 03C7",""};
lookup1("1D7C1") ->	{"0","<font> 03C8",""};
lookup1("1D7C2") ->	{"0","<font> 03C9",""};
lookup1("1D7C3") ->	{"0","<font> 2202",""};
lookup1("1D7C4") ->	{"0","<font> 03F5",""};
lookup1("1D7C5") ->	{"0","<font> 03D1",""};
lookup1("1D7C6") ->	{"0","<font> 03F0",""};
lookup1("1D7C7") ->	{"0","<font> 03D5",""};
lookup1("1D7C8") ->	{"0","<font> 03F1",""};
lookup1("1D7C9") ->	{"0","<font> 03D6",""};
lookup1("1D7CA") ->	{"0","<font> 03DC",""};
lookup1("1D7CB") ->	{"0","<font> 03DD",""};
lookup1("1D7CE") ->	{"0","<font> 0030",""};
lookup1("1D7CF") ->	{"0","<font> 0031",""};
lookup1("1D7D") ->	{"0","",""};
lookup1("1D7D0") ->	{"0","<font> 0032",""};
lookup1("1D7D1") ->	{"0","<font> 0033",""};
lookup1("1D7D2") ->	{"0","<font> 0034",""};
lookup1("1D7D3") ->	{"0","<font> 0035",""};
lookup1("1D7D4") ->	{"0","<font> 0036",""};
lookup1("1D7D5") ->	{"0","<font> 0037",""};
lookup1("1D7D6") ->	{"0","<font> 0038",""};
lookup1("1D7D7") ->	{"0","<font> 0039",""};
lookup1("1D7D8") ->	{"0","<font> 0030",""};
lookup1("1D7D9") ->	{"0","<font> 0031",""};
lookup1("1D7DA") ->	{"0","<font> 0032",""};
lookup1("1D7DB") ->	{"0","<font> 0033",""};
lookup1("1D7DC") ->	{"0","<font> 0034",""};
lookup1("1D7DD") ->	{"0","<font> 0035",""};
lookup1("1D7DE") ->	{"0","<font> 0036",""};
lookup1("1D7DF") ->	{"0","<font> 0037",""};
lookup1("1D7E") ->	{"0","",""};
lookup1("1D7E0") ->	{"0","<font> 0038",""};
lookup1("1D7E1") ->	{"0","<font> 0039",""};
lookup1("1D7E2") ->	{"0","<font> 0030",""};
lookup1("1D7E3") ->	{"0","<font> 0031",""};
lookup1("1D7E4") ->	{"0","<font> 0032",""};
lookup1("1D7E5") ->	{"0","<font> 0033",""};
lookup1("1D7E6") ->	{"0","<font> 0034",""};
lookup1("1D7E7") ->	{"0","<font> 0035",""};
lookup1("1D7E8") ->	{"0","<font> 0036",""};
lookup1("1D7E9") ->	{"0","<font> 0037",""};
lookup1("1D7EA") ->	{"0","<font> 0038",""};
lookup1("1D7EB") ->	{"0","<font> 0039",""};
lookup1("1D7EC") ->	{"0","<font> 0030",""};
lookup1("1D7ED") ->	{"0","<font> 0031",""};
lookup1("1D7EE") ->	{"0","<font> 0032",""};
lookup1("1D7EF") ->	{"0","<font> 0033",""};
lookup1("1D7F") ->	{"0","",""};
lookup1("1D7F0") ->	{"0","<font> 0034",""};
lookup1("1D7F1") ->	{"0","<font> 0035",""};
lookup1("1D7F2") ->	{"0","<font> 0036",""};
lookup1("1D7F3") ->	{"0","<font> 0037",""};
lookup1("1D7F4") ->	{"0","<font> 0038",""};
lookup1("1D7F5") ->	{"0","<font> 0039",""};
lookup1("1D7F6") ->	{"0","<font> 0030",""};
lookup1("1D7F7") ->	{"0","<font> 0031",""};
lookup1("1D7F8") ->	{"0","<font> 0032",""};
lookup1("1D7F9") ->	{"0","<font> 0033",""};
lookup1("1D7FA") ->	{"0","<font> 0034",""};
lookup1("1D7FB") ->	{"0","<font> 0035",""};
lookup1("1D7FC") ->	{"0","<font> 0036",""};
lookup1("1D7FD") ->	{"0","<font> 0037",""};
lookup1("1D7FE") ->	{"0","<font> 0038",""};
lookup1("1D7FF") ->	{"0","<font> 0039",""};
lookup1("1D80") ->	{"0","",""};
lookup1("1D81") ->	{"0","",""};
lookup1("1D82") ->	{"0","",""};
lookup1("1D83") ->	{"0","",""};
lookup1("1D84") ->	{"0","",""};
lookup1("1D85") ->	{"0","",""};
lookup1("1D86") ->	{"0","",""};
lookup1("1D87") ->	{"0","",""};
lookup1("1D88") ->	{"0","",""};
lookup1("1D89") ->	{"0","",""};
lookup1("1D8A") ->	{"0","",""};
lookup1("1D8B") ->	{"0","",""};
lookup1("1D8C") ->	{"0","",""};
lookup1("1D8D") ->	{"0","",""};
lookup1("1D8E") ->	{"0","",""};
lookup1("1D8F") ->	{"0","",""};
lookup1("1D90") ->	{"0","",""};
lookup1("1D91") ->	{"0","",""};
lookup1("1D92") ->	{"0","",""};
lookup1("1D93") ->	{"0","",""};
lookup1("1D94") ->	{"0","",""};
lookup1("1D95") ->	{"0","",""};
lookup1("1D96") ->	{"0","",""};
lookup1("1D97") ->	{"0","",""};
lookup1("1D98") ->	{"0","",""};
lookup1("1D99") ->	{"0","",""};
lookup1("1D9A") ->	{"0","",""};
lookup1("1D9B") ->	{"0","<super> 0252",""};
lookup1("1D9C") ->	{"0","<super> 0063",""};
lookup1("1D9D") ->	{"0","<super> 0255",""};
lookup1("1D9E") ->	{"0","<super> 00F0",""};
lookup1("1D9F") ->	{"0","<super> 025C",""};
lookup1("1DA0") ->	{"0","<super> 0066",""};
lookup1("1DA1") ->	{"0","<super> 025F",""};
lookup1("1DA2") ->	{"0","<super> 0261",""};
lookup1("1DA3") ->	{"0","<super> 0265",""};
lookup1("1DA4") ->	{"0","<super> 0268",""};
lookup1("1DA5") ->	{"0","<super> 0269",""};
lookup1("1DA6") ->	{"0","<super> 026A",""};
lookup1("1DA7") ->	{"0","<super> 1D7B",""};
lookup1("1DA8") ->	{"0","<super> 029D",""};
lookup1("1DA9") ->	{"0","<super> 026D",""};
lookup1("1DAA") ->	{"0","<super> 1D85",""};
lookup1("1DAB") ->	{"0","<super> 029F",""};
lookup1("1DAC") ->	{"0","<super> 0271",""};
lookup1("1DAD") ->	{"0","<super> 0270",""};
lookup1("1DAE") ->	{"0","<super> 0272",""};
lookup1("1DAF") ->	{"0","<super> 0273",""};
lookup1("1DB0") ->	{"0","<super> 0274",""};
lookup1("1DB1") ->	{"0","<super> 0275",""};
lookup1("1DB2") ->	{"0","<super> 0278",""};
lookup1("1DB3") ->	{"0","<super> 0282",""};
lookup1("1DB4") ->	{"0","<super> 0283",""};
lookup1("1DB5") ->	{"0","<super> 01AB",""};
lookup1("1DB6") ->	{"0","<super> 0289",""};
lookup1("1DB7") ->	{"0","<super> 028A",""};
lookup1("1DB8") ->	{"0","<super> 1D1C",""};
lookup1("1DB9") ->	{"0","<super> 028B",""};
lookup1("1DBA") ->	{"0","<super> 028C",""};
lookup1("1DBB") ->	{"0","<super> 007A",""};
lookup1("1DBC") ->	{"0","<super> 0290",""};
lookup1("1DBD") ->	{"0","<super> 0291",""};
lookup1("1DBE") ->	{"0","<super> 0292",""};
lookup1("1DBF") ->	{"0","<super> 03B8",""};
lookup1("1DC0") ->	{"230","",""};
lookup1("1DC1") ->	{"230","",""};
lookup1("1DC2") ->	{"220","",""};
lookup1("1DC3") ->	{"230","",""};
lookup1("1DC4") ->	{"230","",""};
lookup1("1DC5") ->	{"230","",""};
lookup1("1DC6") ->	{"230","",""};
lookup1("1DC7") ->	{"230","",""};
lookup1("1DC8") ->	{"230","",""};
lookup1("1DC9") ->	{"230","",""};
lookup1("1DCA") ->	{"220","",""};
lookup1("1DCB") ->	{"230","",""};
lookup1("1DCC") ->	{"230","",""};
lookup1("1DCD") ->	{"234","",""};
lookup1("1DCE") ->	{"214","",""};
lookup1("1DCF") ->	{"220","",""};
lookup1("1DD0") ->	{"202","",""};
lookup1("1DD1") ->	{"230","",""};
lookup1("1DD2") ->	{"230","",""};
lookup1("1DD3") ->	{"230","",""};
lookup1("1DD4") ->	{"230","",""};
lookup1("1DD5") ->	{"230","",""};
lookup1("1DD6") ->	{"230","",""};
lookup1("1DD7") ->	{"230","",""};
lookup1("1DD8") ->	{"230","",""};
lookup1("1DD9") ->	{"230","",""};
lookup1("1DDA") ->	{"230","",""};
lookup1("1DDB") ->	{"230","",""};
lookup1("1DDC") ->	{"230","",""};
lookup1("1DDD") ->	{"230","",""};
lookup1("1DDE") ->	{"230","",""};
lookup1("1DDF") ->	{"230","",""};
lookup1("1DE0") ->	{"230","",""};
lookup1("1DE1") ->	{"230","",""};
lookup1("1DE2") ->	{"230","",""};
lookup1("1DE3") ->	{"230","",""};
lookup1("1DE4") ->	{"230","",""};
lookup1("1DE5") ->	{"230","",""};
lookup1("1DE6") ->	{"230","",""};
lookup1("1DE7") ->	{"230","",""};
lookup1("1DE8") ->	{"230","",""};
lookup1("1DE9") ->	{"230","",""};
lookup1("1DEA") ->	{"230","",""};
lookup1("1DEB") ->	{"230","",""};
lookup1("1DEC") ->	{"230","",""};
lookup1("1DED") ->	{"230","",""};
lookup1("1DEE") ->	{"230","",""};
lookup1("1DEF") ->	{"230","",""};
lookup1("1DF0") ->	{"230","",""};
lookup1("1DF1") ->	{"230","",""};
lookup1("1DF2") ->	{"230","",""};
lookup1("1DF3") ->	{"230","",""};
lookup1("1DF4") ->	{"230","",""};
lookup1("1DF5") ->	{"230","",""};
lookup1("1DFC") ->	{"233","",""};
lookup1("1DFD") ->	{"220","",""};
lookup1("1DFE") ->	{"230","",""};
lookup1("1DFF") ->	{"220","",""};
lookup1("1E00") ->	{"0","0041 0325","1E01"};
lookup1("1E01") ->	{"0","0061 0325",""};
lookup1("1E02") ->	{"0","0042 0307","1E03"};
lookup1("1E03") ->	{"0","0062 0307",""};
lookup1("1E04") ->	{"0","0042 0323","1E05"};
lookup1("1E05") ->	{"0","0062 0323",""};
lookup1("1E06") ->	{"0","0042 0331","1E07"};
lookup1("1E07") ->	{"0","0062 0331",""};
lookup1("1E08") ->	{"0","00C7 0301","1E09"};
lookup1("1E09") ->	{"0","00E7 0301",""};
lookup1("1E0A") ->	{"0","0044 0307","1E0B"};
lookup1("1E0B") ->	{"0","0064 0307",""};
lookup1("1E0C") ->	{"0","0044 0323","1E0D"};
lookup1("1E0D") ->	{"0","0064 0323",""};
lookup1("1E0E") ->	{"0","0044 0331","1E0F"};
lookup1("1E0F") ->	{"0","0064 0331",""};
lookup1("1E10") ->	{"0","0044 0327","1E11"};
lookup1("1E11") ->	{"0","0064 0327",""};
lookup1("1E12") ->	{"0","0044 032D","1E13"};
lookup1("1E13") ->	{"0","0064 032D",""};
lookup1("1E14") ->	{"0","0112 0300","1E15"};
lookup1("1E15") ->	{"0","0113 0300",""};
lookup1("1E16") ->	{"0","0112 0301","1E17"};
lookup1("1E17") ->	{"0","0113 0301",""};
lookup1("1E18") ->	{"0","0045 032D","1E19"};
lookup1("1E19") ->	{"0","0065 032D",""};
lookup1("1E1A") ->	{"0","0045 0330","1E1B"};
lookup1("1E1B") ->	{"0","0065 0330",""};
lookup1("1E1C") ->	{"0","0228 0306","1E1D"};
lookup1("1E1D") ->	{"0","0229 0306",""};
lookup1("1E1E") ->	{"0","0046 0307","1E1F"};
lookup1("1E1F") ->	{"0","0066 0307",""};
lookup1("1E20") ->	{"0","0047 0304","1E21"};
lookup1("1E21") ->	{"0","0067 0304",""};
lookup1("1E22") ->	{"0","0048 0307","1E23"};
lookup1("1E23") ->	{"0","0068 0307",""};
lookup1("1E24") ->	{"0","0048 0323","1E25"};
lookup1("1E25") ->	{"0","0068 0323",""};
lookup1("1E26") ->	{"0","0048 0308","1E27"};
lookup1("1E27") ->	{"0","0068 0308",""};
lookup1("1E28") ->	{"0","0048 0327","1E29"};
lookup1("1E29") ->	{"0","0068 0327",""};
lookup1("1E2A") ->	{"0","0048 032E","1E2B"};
lookup1("1E2B") ->	{"0","0068 032E",""};
lookup1("1E2C") ->	{"0","0049 0330","1E2D"};
lookup1("1E2D") ->	{"0","0069 0330",""};
lookup1("1E2E") ->	{"0","00CF 0301","1E2F"};
lookup1("1E2F") ->	{"0","00EF 0301",""};
lookup1("1E30") ->	{"0","004B 0301","1E31"};
lookup1("1E31") ->	{"0","006B 0301",""};
lookup1("1E32") ->	{"0","004B 0323","1E33"};
lookup1("1E33") ->	{"0","006B 0323",""};
lookup1("1E34") ->	{"0","004B 0331","1E35"};
lookup1("1E35") ->	{"0","006B 0331",""};
lookup1("1E36") ->	{"0","004C 0323","1E37"};
lookup1("1E37") ->	{"0","006C 0323",""};
lookup1("1E38") ->	{"0","1E36 0304","1E39"};
lookup1("1E39") ->	{"0","1E37 0304",""};
lookup1("1E3A") ->	{"0","004C 0331","1E3B"};
lookup1("1E3B") ->	{"0","006C 0331",""};
lookup1("1E3C") ->	{"0","004C 032D","1E3D"};
lookup1("1E3D") ->	{"0","006C 032D",""};
lookup1("1E3E") ->	{"0","004D 0301","1E3F"};
lookup1("1E3F") ->	{"0","006D 0301",""};
lookup1("1E40") ->	{"0","004D 0307","1E41"};
lookup1("1E41") ->	{"0","006D 0307",""};
lookup1("1E42") ->	{"0","004D 0323","1E43"};
lookup1("1E43") ->	{"0","006D 0323",""};
lookup1("1E44") ->	{"0","004E 0307","1E45"};
lookup1("1E45") ->	{"0","006E 0307",""};
lookup1("1E46") ->	{"0","004E 0323","1E47"};
lookup1("1E47") ->	{"0","006E 0323",""};
lookup1("1E48") ->	{"0","004E 0331","1E49"};
lookup1("1E49") ->	{"0","006E 0331",""};
lookup1("1E4A") ->	{"0","004E 032D","1E4B"};
lookup1("1E4B") ->	{"0","006E 032D",""};
lookup1("1E4C") ->	{"0","00D5 0301","1E4D"};
lookup1("1E4D") ->	{"0","00F5 0301",""};
lookup1("1E4E") ->	{"0","00D5 0308","1E4F"};
lookup1("1E4F") ->	{"0","00F5 0308",""};
lookup1("1E50") ->	{"0","014C 0300","1E51"};
lookup1("1E51") ->	{"0","014D 0300",""};
lookup1("1E52") ->	{"0","014C 0301","1E53"};
lookup1("1E53") ->	{"0","014D 0301",""};
lookup1("1E54") ->	{"0","0050 0301","1E55"};
lookup1("1E55") ->	{"0","0070 0301",""};
lookup1("1E56") ->	{"0","0050 0307","1E57"};
lookup1("1E57") ->	{"0","0070 0307",""};
lookup1("1E58") ->	{"0","0052 0307","1E59"};
lookup1("1E59") ->	{"0","0072 0307",""};
lookup1("1E5A") ->	{"0","0052 0323","1E5B"};
lookup1("1E5B") ->	{"0","0072 0323",""};
lookup1("1E5C") ->	{"0","1E5A 0304","1E5D"};
lookup1("1E5D") ->	{"0","1E5B 0304",""};
lookup1("1E5E") ->	{"0","0052 0331","1E5F"};
lookup1("1E5F") ->	{"0","0072 0331",""};
lookup1("1E60") ->	{"0","0053 0307","1E61"};
lookup1("1E61") ->	{"0","0073 0307",""};
lookup1("1E62") ->	{"0","0053 0323","1E63"};
lookup1("1E63") ->	{"0","0073 0323",""};
lookup1("1E64") ->	{"0","015A 0307","1E65"};
lookup1("1E65") ->	{"0","015B 0307",""};
lookup1("1E66") ->	{"0","0160 0307","1E67"};
lookup1("1E67") ->	{"0","0161 0307",""};
lookup1("1E68") ->	{"0","1E62 0307","1E69"};
lookup1("1E69") ->	{"0","1E63 0307",""};
lookup1("1E6A") ->	{"0","0054 0307","1E6B"};
lookup1("1E6B") ->	{"0","0074 0307",""};
lookup1("1E6C") ->	{"0","0054 0323","1E6D"};
lookup1("1E6D") ->	{"0","0074 0323",""};
lookup1("1E6E") ->	{"0","0054 0331","1E6F"};
lookup1("1E6F") ->	{"0","0074 0331",""};
lookup1("1E70") ->	{"0","0054 032D","1E71"};
lookup1("1E71") ->	{"0","0074 032D",""};
lookup1("1E72") ->	{"0","0055 0324","1E73"};
lookup1("1E73") ->	{"0","0075 0324",""};
lookup1("1E74") ->	{"0","0055 0330","1E75"};
lookup1("1E75") ->	{"0","0075 0330",""};
lookup1("1E76") ->	{"0","0055 032D","1E77"};
lookup1("1E77") ->	{"0","0075 032D",""};
lookup1("1E78") ->	{"0","0168 0301","1E79"};
lookup1("1E79") ->	{"0","0169 0301",""};
lookup1("1E7A") ->	{"0","016A 0308","1E7B"};
lookup1("1E7B") ->	{"0","016B 0308",""};
lookup1("1E7C") ->	{"0","0056 0303","1E7D"};
lookup1("1E7D") ->	{"0","0076 0303",""};
lookup1("1E7E") ->	{"0","0056 0323","1E7F"};
lookup1("1E7F") ->	{"0","0076 0323",""};
lookup1("1E80") ->	{"0","0057 0300","1E81"};
lookup1("1E800") ->	{"0","",""};
lookup1("1E801") ->	{"0","",""};
lookup1("1E802") ->	{"0","",""};
lookup1("1E803") ->	{"0","",""};
lookup1("1E804") ->	{"0","",""};
lookup1("1E805") ->	{"0","",""};
lookup1("1E806") ->	{"0","",""};
lookup1("1E807") ->	{"0","",""};
lookup1("1E808") ->	{"0","",""};
lookup1("1E809") ->	{"0","",""};
lookup1("1E80A") ->	{"0","",""};
lookup1("1E80B") ->	{"0","",""};
lookup1("1E80C") ->	{"0","",""};
lookup1("1E80D") ->	{"0","",""};
lookup1("1E80E") ->	{"0","",""};
lookup1("1E80F") ->	{"0","",""};
lookup1("1E81") ->	{"0","0077 0300",""};
lookup1("1E810") ->	{"0","",""};
lookup1("1E811") ->	{"0","",""};
lookup1("1E812") ->	{"0","",""};
lookup1("1E813") ->	{"0","",""};
lookup1("1E814") ->	{"0","",""};
lookup1("1E815") ->	{"0","",""};
lookup1("1E816") ->	{"0","",""};
lookup1("1E817") ->	{"0","",""};
lookup1("1E818") ->	{"0","",""};
lookup1("1E819") ->	{"0","",""};
lookup1("1E81A") ->	{"0","",""};
lookup1("1E81B") ->	{"0","",""};
lookup1("1E81C") ->	{"0","",""};
lookup1("1E81D") ->	{"0","",""};
lookup1("1E81E") ->	{"0","",""};
lookup1("1E81F") ->	{"0","",""};
lookup1("1E82") ->	{"0","0057 0301","1E83"};
lookup1("1E820") ->	{"0","",""};
lookup1("1E821") ->	{"0","",""};
lookup1("1E822") ->	{"0","",""};
lookup1("1E823") ->	{"0","",""};
lookup1("1E824") ->	{"0","",""};
lookup1("1E825") ->	{"0","",""};
lookup1("1E826") ->	{"0","",""};
lookup1("1E827") ->	{"0","",""};
lookup1("1E828") ->	{"0","",""};
lookup1("1E829") ->	{"0","",""};
lookup1("1E82A") ->	{"0","",""};
lookup1("1E82B") ->	{"0","",""};
lookup1("1E82C") ->	{"0","",""};
lookup1("1E82D") ->	{"0","",""};
lookup1("1E82E") ->	{"0","",""};
lookup1("1E82F") ->	{"0","",""};
lookup1("1E83") ->	{"0","0077 0301",""};
lookup1("1E830") ->	{"0","",""};
lookup1("1E831") ->	{"0","",""};
lookup1("1E832") ->	{"0","",""};
lookup1("1E833") ->	{"0","",""};
lookup1("1E834") ->	{"0","",""};
lookup1("1E835") ->	{"0","",""};
lookup1("1E836") ->	{"0","",""};
lookup1("1E837") ->	{"0","",""};
lookup1("1E838") ->	{"0","",""};
lookup1("1E839") ->	{"0","",""};
lookup1("1E83A") ->	{"0","",""};
lookup1("1E83B") ->	{"0","",""};
lookup1("1E83C") ->	{"0","",""};
lookup1("1E83D") ->	{"0","",""};
lookup1("1E83E") ->	{"0","",""};
lookup1("1E83F") ->	{"0","",""};
lookup1("1E84") ->	{"0","0057 0308","1E85"};
lookup1("1E840") ->	{"0","",""};
lookup1("1E841") ->	{"0","",""};
lookup1("1E842") ->	{"0","",""};
lookup1("1E843") ->	{"0","",""};
lookup1("1E844") ->	{"0","",""};
lookup1("1E845") ->	{"0","",""};
lookup1("1E846") ->	{"0","",""};
lookup1("1E847") ->	{"0","",""};
lookup1("1E848") ->	{"0","",""};
lookup1("1E849") ->	{"0","",""};
lookup1("1E84A") ->	{"0","",""};
lookup1("1E84B") ->	{"0","",""};
lookup1("1E84C") ->	{"0","",""};
lookup1("1E84D") ->	{"0","",""};
lookup1("1E84E") ->	{"0","",""};
lookup1("1E84F") ->	{"0","",""};
lookup1("1E85") ->	{"0","0077 0308",""};
lookup1("1E850") ->	{"0","",""};
lookup1("1E851") ->	{"0","",""};
lookup1("1E852") ->	{"0","",""};
lookup1("1E853") ->	{"0","",""};
lookup1("1E854") ->	{"0","",""};
lookup1("1E855") ->	{"0","",""};
lookup1("1E856") ->	{"0","",""};
lookup1("1E857") ->	{"0","",""};
lookup1("1E858") ->	{"0","",""};
lookup1("1E859") ->	{"0","",""};
lookup1("1E85A") ->	{"0","",""};
lookup1("1E85B") ->	{"0","",""};
lookup1("1E85C") ->	{"0","",""};
lookup1("1E85D") ->	{"0","",""};
lookup1("1E85E") ->	{"0","",""};
lookup1("1E85F") ->	{"0","",""};
lookup1("1E86") ->	{"0","0057 0307","1E87"};
lookup1("1E860") ->	{"0","",""};
lookup1("1E861") ->	{"0","",""};
lookup1("1E862") ->	{"0","",""};
lookup1("1E863") ->	{"0","",""};
lookup1("1E864") ->	{"0","",""};
lookup1("1E865") ->	{"0","",""};
lookup1("1E866") ->	{"0","",""};
lookup1("1E867") ->	{"0","",""};
lookup1("1E868") ->	{"0","",""};
lookup1("1E869") ->	{"0","",""};
lookup1("1E86A") ->	{"0","",""};
lookup1("1E86B") ->	{"0","",""};
lookup1("1E86C") ->	{"0","",""};
lookup1("1E86D") ->	{"0","",""};
lookup1("1E86E") ->	{"0","",""};
lookup1("1E86F") ->	{"0","",""};
lookup1("1E87") ->	{"0","0077 0307",""};
lookup1("1E870") ->	{"0","",""};
lookup1("1E871") ->	{"0","",""};
lookup1("1E872") ->	{"0","",""};
lookup1("1E873") ->	{"0","",""};
lookup1("1E874") ->	{"0","",""};
lookup1("1E875") ->	{"0","",""};
lookup1("1E876") ->	{"0","",""};
lookup1("1E877") ->	{"0","",""};
lookup1("1E878") ->	{"0","",""};
lookup1("1E879") ->	{"0","",""};
lookup1("1E87A") ->	{"0","",""};
lookup1("1E87B") ->	{"0","",""};
lookup1("1E87C") ->	{"0","",""};
lookup1("1E87D") ->	{"0","",""};
lookup1("1E87E") ->	{"0","",""};
lookup1("1E87F") ->	{"0","",""};
lookup1("1E88") ->	{"0","0057 0323","1E89"};
lookup1("1E880") ->	{"0","",""};
lookup1("1E881") ->	{"0","",""};
lookup1("1E882") ->	{"0","",""};
lookup1("1E883") ->	{"0","",""};
lookup1("1E884") ->	{"0","",""};
lookup1("1E885") ->	{"0","",""};
lookup1("1E886") ->	{"0","",""};
lookup1("1E887") ->	{"0","",""};
lookup1("1E888") ->	{"0","",""};
lookup1("1E889") ->	{"0","",""};
lookup1("1E88A") ->	{"0","",""};
lookup1("1E88B") ->	{"0","",""};
lookup1("1E88C") ->	{"0","",""};
lookup1("1E88D") ->	{"0","",""};
lookup1("1E88E") ->	{"0","",""};
lookup1("1E88F") ->	{"0","",""};
lookup1("1E89") ->	{"0","0077 0323",""};
lookup1("1E890") ->	{"0","",""};
lookup1("1E891") ->	{"0","",""};
lookup1("1E892") ->	{"0","",""};
lookup1("1E893") ->	{"0","",""};
lookup1("1E894") ->	{"0","",""};
lookup1("1E895") ->	{"0","",""};
lookup1("1E896") ->	{"0","",""};
lookup1("1E897") ->	{"0","",""};
lookup1("1E898") ->	{"0","",""};
lookup1("1E899") ->	{"0","",""};
lookup1("1E89A") ->	{"0","",""};
lookup1("1E89B") ->	{"0","",""};
lookup1("1E89C") ->	{"0","",""};
lookup1("1E89D") ->	{"0","",""};
lookup1("1E89E") ->	{"0","",""};
lookup1("1E89F") ->	{"0","",""};
lookup1("1E8A") ->	{"0","0058 0307","1E8B"};
lookup1("1E8A0") ->	{"0","",""};
lookup1("1E8A1") ->	{"0","",""};
lookup1("1E8A2") ->	{"0","",""};
lookup1("1E8A3") ->	{"0","",""};
lookup1("1E8A4") ->	{"0","",""};
lookup1("1E8A5") ->	{"0","",""};
lookup1("1E8A6") ->	{"0","",""};
lookup1("1E8A7") ->	{"0","",""};
lookup1("1E8A8") ->	{"0","",""};
lookup1("1E8A9") ->	{"0","",""};
lookup1("1E8AA") ->	{"0","",""};
lookup1("1E8AB") ->	{"0","",""};
lookup1("1E8AC") ->	{"0","",""};
lookup1("1E8AD") ->	{"0","",""};
lookup1("1E8AE") ->	{"0","",""};
lookup1("1E8AF") ->	{"0","",""};
lookup1("1E8B") ->	{"0","0078 0307",""};
lookup1("1E8B0") ->	{"0","",""};
lookup1("1E8B1") ->	{"0","",""};
lookup1("1E8B2") ->	{"0","",""};
lookup1("1E8B3") ->	{"0","",""};
lookup1("1E8B4") ->	{"0","",""};
lookup1("1E8B5") ->	{"0","",""};
lookup1("1E8B6") ->	{"0","",""};
lookup1("1E8B7") ->	{"0","",""};
lookup1("1E8B8") ->	{"0","",""};
lookup1("1E8B9") ->	{"0","",""};
lookup1("1E8BA") ->	{"0","",""};
lookup1("1E8BB") ->	{"0","",""};
lookup1("1E8BC") ->	{"0","",""};
lookup1("1E8BD") ->	{"0","",""};
lookup1("1E8BE") ->	{"0","",""};
lookup1("1E8BF") ->	{"0","",""};
lookup1("1E8C") ->	{"0","0058 0308","1E8D"};
lookup1("1E8C0") ->	{"0","",""};
lookup1("1E8C1") ->	{"0","",""};
lookup1("1E8C2") ->	{"0","",""};
lookup1("1E8C3") ->	{"0","",""};
lookup1("1E8C4") ->	{"0","",""};
lookup1("1E8C7") ->	{"0","",""};
lookup1("1E8C8") ->	{"0","",""};
lookup1("1E8C9") ->	{"0","",""};
lookup1("1E8CA") ->	{"0","",""};
lookup1("1E8CB") ->	{"0","",""};
lookup1("1E8CC") ->	{"0","",""};
lookup1("1E8CD") ->	{"0","",""};
lookup1("1E8CE") ->	{"0","",""};
lookup1("1E8CF") ->	{"0","",""};
lookup1("1E8D") ->	{"0","0078 0308",""};
lookup1("1E8D0") ->	{"220","",""};
lookup1("1E8D1") ->	{"220","",""};
lookup1("1E8D2") ->	{"220","",""};
lookup1("1E8D3") ->	{"220","",""};
lookup1("1E8D4") ->	{"220","",""};
lookup1("1E8D5") ->	{"220","",""};
lookup1("1E8D6") ->	{"220","",""};
lookup1("1E8E") ->	{"0","0059 0307","1E8F"};
lookup1("1E8F") ->	{"0","0079 0307",""};
lookup1("1E90") ->	{"0","005A 0302","1E91"};
lookup1("1E91") ->	{"0","007A 0302",""};
lookup1("1E92") ->	{"0","005A 0323","1E93"};
lookup1("1E93") ->	{"0","007A 0323",""};
lookup1("1E94") ->	{"0","005A 0331","1E95"};
lookup1("1E95") ->	{"0","007A 0331",""};
lookup1("1E96") ->	{"0","0068 0331",""};
lookup1("1E97") ->	{"0","0074 0308",""};
lookup1("1E98") ->	{"0","0077 030A",""};
lookup1("1E99") ->	{"0","0079 030A",""};
lookup1("1E9A") ->	{"0","<compat> 0061 02BE",""};
lookup1("1E9B") ->	{"0","017F 0307",""};
lookup1("1E9C") ->	{"0","",""};
lookup1("1E9D") ->	{"0","",""};
lookup1("1E9E") ->	{"0","","00DF"};
lookup1("1E9F") ->	{"0","",""};
lookup1("1EA0") ->	{"0","0041 0323","1EA1"};
lookup1("1EA1") ->	{"0","0061 0323",""};
lookup1("1EA2") ->	{"0","0041 0309","1EA3"};
lookup1("1EA3") ->	{"0","0061 0309",""};
lookup1("1EA4") ->	{"0","00C2 0301","1EA5"};
lookup1("1EA5") ->	{"0","00E2 0301",""};
lookup1("1EA6") ->	{"0","00C2 0300","1EA7"};
lookup1("1EA7") ->	{"0","00E2 0300",""};
lookup1("1EA8") ->	{"0","00C2 0309","1EA9"};
lookup1("1EA9") ->	{"0","00E2 0309",""};
lookup1("1EAA") ->	{"0","00C2 0303","1EAB"};
lookup1("1EAB") ->	{"0","00E2 0303",""};
lookup1("1EAC") ->	{"0","1EA0 0302","1EAD"};
lookup1("1EAD") ->	{"0","1EA1 0302",""};
lookup1("1EAE") ->	{"0","0102 0301","1EAF"};
lookup1("1EAF") ->	{"0","0103 0301",""};
lookup1("1EB0") ->	{"0","0102 0300","1EB1"};
lookup1("1EB1") ->	{"0","0103 0300",""};
lookup1("1EB2") ->	{"0","0102 0309","1EB3"};
lookup1("1EB3") ->	{"0","0103 0309",""};
lookup1("1EB4") ->	{"0","0102 0303","1EB5"};
lookup1("1EB5") ->	{"0","0103 0303",""};
lookup1("1EB6") ->	{"0","1EA0 0306","1EB7"};
lookup1("1EB7") ->	{"0","1EA1 0306",""};
lookup1("1EB8") ->	{"0","0045 0323","1EB9"};
lookup1("1EB9") ->	{"0","0065 0323",""};
lookup1("1EBA") ->	{"0","0045 0309","1EBB"};
lookup1("1EBB") ->	{"0","0065 0309",""};
lookup1("1EBC") ->	{"0","0045 0303","1EBD"};
lookup1("1EBD") ->	{"0","0065 0303",""};
lookup1("1EBE") ->	{"0","00CA 0301","1EBF"};
lookup1("1EBF") ->	{"0","00EA 0301",""};
lookup1("1EC0") ->	{"0","00CA 0300","1EC1"};
lookup1("1EC1") ->	{"0","00EA 0300",""};
lookup1("1EC2") ->	{"0","00CA 0309","1EC3"};
lookup1("1EC3") ->	{"0","00EA 0309",""};
lookup1("1EC4") ->	{"0","00CA 0303","1EC5"};
lookup1("1EC5") ->	{"0","00EA 0303",""};
lookup1("1EC6") ->	{"0","1EB8 0302","1EC7"};
lookup1("1EC7") ->	{"0","1EB9 0302",""};
lookup1("1EC8") ->	{"0","0049 0309","1EC9"};
lookup1("1EC9") ->	{"0","0069 0309",""};
lookup1("1ECA") ->	{"0","0049 0323","1ECB"};
lookup1("1ECB") ->	{"0","0069 0323",""};
lookup1("1ECC") ->	{"0","004F 0323","1ECD"};
lookup1("1ECD") ->	{"0","006F 0323",""};
lookup1("1ECE") ->	{"0","004F 0309","1ECF"};
lookup1("1ECF") ->	{"0","006F 0309",""};
lookup1("1ED0") ->	{"0","00D4 0301","1ED1"};
lookup1("1ED1") ->	{"0","00F4 0301",""};
lookup1("1ED2") ->	{"0","00D4 0300","1ED3"};
lookup1("1ED3") ->	{"0","00F4 0300",""};
lookup1("1ED4") ->	{"0","00D4 0309","1ED5"};
lookup1("1ED5") ->	{"0","00F4 0309",""};
lookup1("1ED6") ->	{"0","00D4 0303","1ED7"};
lookup1("1ED7") ->	{"0","00F4 0303",""};
lookup1("1ED8") ->	{"0","1ECC 0302","1ED9"};
lookup1("1ED9") ->	{"0","1ECD 0302",""};
lookup1("1EDA") ->	{"0","01A0 0301","1EDB"};
lookup1("1EDB") ->	{"0","01A1 0301",""};
lookup1("1EDC") ->	{"0","01A0 0300","1EDD"};
lookup1("1EDD") ->	{"0","01A1 0300",""};
lookup1("1EDE") ->	{"0","01A0 0309","1EDF"};
lookup1("1EDF") ->	{"0","01A1 0309",""};
lookup1("1EE0") ->	{"0","01A0 0303","1EE1"};
lookup1("1EE00") ->	{"0","<font> 0627",""};
lookup1("1EE01") ->	{"0","<font> 0628",""};
lookup1("1EE02") ->	{"0","<font> 062C",""};
lookup1("1EE03") ->	{"0","<font> 062F",""};
lookup1("1EE05") ->	{"0","<font> 0648",""};
lookup1("1EE06") ->	{"0","<font> 0632",""};
lookup1("1EE07") ->	{"0","<font> 062D",""};
lookup1("1EE08") ->	{"0","<font> 0637",""};
lookup1("1EE09") ->	{"0","<font> 064A",""};
lookup1("1EE0A") ->	{"0","<font> 0643",""};
lookup1("1EE0B") ->	{"0","<font> 0644",""};
lookup1("1EE0C") ->	{"0","<font> 0645",""};
lookup1("1EE0D") ->	{"0","<font> 0646",""};
lookup1("1EE0E") ->	{"0","<font> 0633",""};
lookup1("1EE0F") ->	{"0","<font> 0639",""};
lookup1("1EE1") ->	{"0","01A1 0303",""};
lookup1("1EE10") ->	{"0","<font> 0641",""};
lookup1("1EE11") ->	{"0","<font> 0635",""};
lookup1("1EE12") ->	{"0","<font> 0642",""};
lookup1("1EE13") ->	{"0","<font> 0631",""};
lookup1("1EE14") ->	{"0","<font> 0634",""};
lookup1("1EE15") ->	{"0","<font> 062A",""};
lookup1("1EE16") ->	{"0","<font> 062B",""};
lookup1("1EE17") ->	{"0","<font> 062E",""};
lookup1("1EE18") ->	{"0","<font> 0630",""};
lookup1("1EE19") ->	{"0","<font> 0636",""};
lookup1("1EE1A") ->	{"0","<font> 0638",""};
lookup1("1EE1B") ->	{"0","<font> 063A",""};
lookup1("1EE1C") ->	{"0","<font> 066E",""};
lookup1("1EE1D") ->	{"0","<font> 06BA",""};
lookup1("1EE1E") ->	{"0","<font> 06A1",""};
lookup1("1EE1F") ->	{"0","<font> 066F",""};
lookup1("1EE2") ->	{"0","01A0 0323","1EE3"};
lookup1("1EE21") ->	{"0","<font> 0628",""};
lookup1("1EE22") ->	{"0","<font> 062C",""};
lookup1("1EE24") ->	{"0","<font> 0647",""};
lookup1("1EE27") ->	{"0","<font> 062D",""};
lookup1("1EE29") ->	{"0","<font> 064A",""};
lookup1("1EE2A") ->	{"0","<font> 0643",""};
lookup1("1EE2B") ->	{"0","<font> 0644",""};
lookup1("1EE2C") ->	{"0","<font> 0645",""};
lookup1("1EE2D") ->	{"0","<font> 0646",""};
lookup1("1EE2E") ->	{"0","<font> 0633",""};
lookup1("1EE2F") ->	{"0","<font> 0639",""};
lookup1("1EE3") ->	{"0","01A1 0323",""};
lookup1("1EE30") ->	{"0","<font> 0641",""};
lookup1("1EE31") ->	{"0","<font> 0635",""};
lookup1("1EE32") ->	{"0","<font> 0642",""};
lookup1("1EE34") ->	{"0","<font> 0634",""};
lookup1("1EE35") ->	{"0","<font> 062A",""};
lookup1("1EE36") ->	{"0","<font> 062B",""};
lookup1("1EE37") ->	{"0","<font> 062E",""};
lookup1("1EE39") ->	{"0","<font> 0636",""};
lookup1("1EE3B") ->	{"0","<font> 063A",""};
lookup1("1EE4") ->	{"0","0055 0323","1EE5"};
lookup1("1EE42") ->	{"0","<font> 062C",""};
lookup1("1EE47") ->	{"0","<font> 062D",""};
lookup1("1EE49") ->	{"0","<font> 064A",""};
lookup1("1EE4B") ->	{"0","<font> 0644",""};
lookup1("1EE4D") ->	{"0","<font> 0646",""};
lookup1("1EE4E") ->	{"0","<font> 0633",""};
lookup1("1EE4F") ->	{"0","<font> 0639",""};
lookup1("1EE5") ->	{"0","0075 0323",""};
lookup1("1EE51") ->	{"0","<font> 0635",""};
lookup1("1EE52") ->	{"0","<font> 0642",""};
lookup1("1EE54") ->	{"0","<font> 0634",""};
lookup1("1EE57") ->	{"0","<font> 062E",""};
lookup1("1EE59") ->	{"0","<font> 0636",""};
lookup1("1EE5B") ->	{"0","<font> 063A",""};
lookup1("1EE5D") ->	{"0","<font> 06BA",""};
lookup1("1EE5F") ->	{"0","<font> 066F",""};
lookup1("1EE6") ->	{"0","0055 0309","1EE7"};
lookup1("1EE61") ->	{"0","<font> 0628",""};
lookup1("1EE62") ->	{"0","<font> 062C",""};
lookup1("1EE64") ->	{"0","<font> 0647",""};
lookup1("1EE67") ->	{"0","<font> 062D",""};
lookup1("1EE68") ->	{"0","<font> 0637",""};
lookup1("1EE69") ->	{"0","<font> 064A",""};
lookup1("1EE6A") ->	{"0","<font> 0643",""};
lookup1("1EE6C") ->	{"0","<font> 0645",""};
lookup1("1EE6D") ->	{"0","<font> 0646",""};
lookup1("1EE6E") ->	{"0","<font> 0633",""};
lookup1("1EE6F") ->	{"0","<font> 0639",""};
lookup1("1EE7") ->	{"0","0075 0309",""};
lookup1("1EE70") ->	{"0","<font> 0641",""};
lookup1("1EE71") ->	{"0","<font> 0635",""};
lookup1("1EE72") ->	{"0","<font> 0642",""};
lookup1("1EE74") ->	{"0","<font> 0634",""};
lookup1("1EE75") ->	{"0","<font> 062A",""};
lookup1("1EE76") ->	{"0","<font> 062B",""};
lookup1("1EE77") ->	{"0","<font> 062E",""};
lookup1("1EE79") ->	{"0","<font> 0636",""};
lookup1("1EE7A") ->	{"0","<font> 0638",""};
lookup1("1EE7B") ->	{"0","<font> 063A",""};
lookup1("1EE7C") ->	{"0","<font> 066E",""};
lookup1("1EE7E") ->	{"0","<font> 06A1",""};
lookup1("1EE8") ->	{"0","01AF 0301","1EE9"};
lookup1("1EE80") ->	{"0","<font> 0627",""};
lookup1("1EE81") ->	{"0","<font> 0628",""};
lookup1("1EE82") ->	{"0","<font> 062C",""};
lookup1("1EE83") ->	{"0","<font> 062F",""};
lookup1("1EE84") ->	{"0","<font> 0647",""};
lookup1("1EE85") ->	{"0","<font> 0648",""};
lookup1("1EE86") ->	{"0","<font> 0632",""};
lookup1("1EE87") ->	{"0","<font> 062D",""};
lookup1("1EE88") ->	{"0","<font> 0637",""};
lookup1("1EE89") ->	{"0","<font> 064A",""};
lookup1("1EE8B") ->	{"0","<font> 0644",""};
lookup1("1EE8C") ->	{"0","<font> 0645",""};
lookup1("1EE8D") ->	{"0","<font> 0646",""};
lookup1("1EE8E") ->	{"0","<font> 0633",""};
lookup1("1EE8F") ->	{"0","<font> 0639",""};
lookup1("1EE9") ->	{"0","01B0 0301",""};
lookup1("1EE90") ->	{"0","<font> 0641",""};
lookup1("1EE91") ->	{"0","<font> 0635",""};
lookup1("1EE92") ->	{"0","<font> 0642",""};
lookup1("1EE93") ->	{"0","<font> 0631",""};
lookup1("1EE94") ->	{"0","<font> 0634",""};
lookup1("1EE95") ->	{"0","<font> 062A",""};
lookup1("1EE96") ->	{"0","<font> 062B",""};
lookup1("1EE97") ->	{"0","<font> 062E",""};
lookup1("1EE98") ->	{"0","<font> 0630",""};
lookup1("1EE99") ->	{"0","<font> 0636",""};
lookup1("1EE9A") ->	{"0","<font> 0638",""};
lookup1("1EE9B") ->	{"0","<font> 063A",""};
lookup1("1EEA") ->	{"0","01AF 0300","1EEB"};
lookup1("1EEA1") ->	{"0","<font> 0628",""};
lookup1("1EEA2") ->	{"0","<font> 062C",""};
lookup1("1EEA3") ->	{"0","<font> 062F",""};
lookup1("1EEA5") ->	{"0","<font> 0648",""};
lookup1("1EEA6") ->	{"0","<font> 0632",""};
lookup1("1EEA7") ->	{"0","<font> 062D",""};
lookup1("1EEA8") ->	{"0","<font> 0637",""};
lookup1("1EEA9") ->	{"0","<font> 064A",""};
lookup1("1EEAB") ->	{"0","<font> 0644",""};
lookup1("1EEAC") ->	{"0","<font> 0645",""};
lookup1("1EEAD") ->	{"0","<font> 0646",""};
lookup1("1EEAE") ->	{"0","<font> 0633",""};
lookup1("1EEAF") ->	{"0","<font> 0639",""};
lookup1("1EEB") ->	{"0","01B0 0300",""};
lookup1("1EEB0") ->	{"0","<font> 0641",""};
lookup1("1EEB1") ->	{"0","<font> 0635",""};
lookup1("1EEB2") ->	{"0","<font> 0642",""};
lookup1("1EEB3") ->	{"0","<font> 0631",""};
lookup1("1EEB4") ->	{"0","<font> 0634",""};
lookup1("1EEB5") ->	{"0","<font> 062A",""};
lookup1("1EEB6") ->	{"0","<font> 062B",""};
lookup1("1EEB7") ->	{"0","<font> 062E",""};
lookup1("1EEB8") ->	{"0","<font> 0630",""};
lookup1("1EEB9") ->	{"0","<font> 0636",""};
lookup1("1EEBA") ->	{"0","<font> 0638",""};
lookup1("1EEBB") ->	{"0","<font> 063A",""};
lookup1("1EEC") ->	{"0","01AF 0309","1EED"};
lookup1("1EED") ->	{"0","01B0 0309",""};
lookup1("1EEE") ->	{"0","01AF 0303","1EEF"};
lookup1("1EEF") ->	{"0","01B0 0303",""};
lookup1("1EEF0") ->	{"0","",""};
lookup1("1EEF1") ->	{"0","",""};
lookup1("1EF0") ->	{"0","01AF 0323","1EF1"};
lookup1("1EF1") ->	{"0","01B0 0323",""};
lookup1("1EF2") ->	{"0","0059 0300","1EF3"};
lookup1("1EF3") ->	{"0","0079 0300",""};
lookup1("1EF4") ->	{"0","0059 0323","1EF5"};
lookup1("1EF5") ->	{"0","0079 0323",""};
lookup1("1EF6") ->	{"0","0059 0309","1EF7"};
lookup1("1EF7") ->	{"0","0079 0309",""};
lookup1("1EF8") ->	{"0","0059 0303","1EF9"};
lookup1("1EF9") ->	{"0","0079 0303",""};
lookup1("1EFA") ->	{"0","","1EFB"};
lookup1("1EFB") ->	{"0","",""};
lookup1("1EFC") ->	{"0","","1EFD"};
lookup1("1EFD") ->	{"0","",""};
lookup1("1EFE") ->	{"0","","1EFF"};
lookup1("1EFF") ->	{"0","",""};
lookup1("1F00") ->	{"0","03B1 0313",""};
lookup1("1F000") ->	{"0","",""};
lookup1("1F001") ->	{"0","",""};
lookup1("1F002") ->	{"0","",""};
lookup1("1F003") ->	{"0","",""};
lookup1("1F004") ->	{"0","",""};
lookup1("1F005") ->	{"0","",""};
lookup1("1F006") ->	{"0","",""};
lookup1("1F007") ->	{"0","",""};
lookup1("1F008") ->	{"0","",""};
lookup1("1F009") ->	{"0","",""};
lookup1("1F00A") ->	{"0","",""};
lookup1("1F00B") ->	{"0","",""};
lookup1("1F00C") ->	{"0","",""};
lookup1("1F00D") ->	{"0","",""};
lookup1("1F00E") ->	{"0","",""};
lookup1("1F00F") ->	{"0","",""};
lookup1("1F01") ->	{"0","03B1 0314",""};
lookup1("1F010") ->	{"0","",""};
lookup1("1F011") ->	{"0","",""};
lookup1("1F012") ->	{"0","",""};
lookup1("1F013") ->	{"0","",""};
lookup1("1F014") ->	{"0","",""};
lookup1("1F015") ->	{"0","",""};
lookup1("1F016") ->	{"0","",""};
lookup1("1F017") ->	{"0","",""};
lookup1("1F018") ->	{"0","",""};
lookup1("1F019") ->	{"0","",""};
lookup1("1F01A") ->	{"0","",""};
lookup1("1F01B") ->	{"0","",""};
lookup1("1F01C") ->	{"0","",""};
lookup1("1F01D") ->	{"0","",""};
lookup1("1F01E") ->	{"0","",""};
lookup1("1F01F") ->	{"0","",""};
lookup1("1F02") ->	{"0","1F00 0300",""};
lookup1("1F020") ->	{"0","",""};
lookup1("1F021") ->	{"0","",""};
lookup1("1F022") ->	{"0","",""};
lookup1("1F023") ->	{"0","",""};
lookup1("1F024") ->	{"0","",""};
lookup1("1F025") ->	{"0","",""};
lookup1("1F026") ->	{"0","",""};
lookup1("1F027") ->	{"0","",""};
lookup1("1F028") ->	{"0","",""};
lookup1("1F029") ->	{"0","",""};
lookup1("1F02A") ->	{"0","",""};
lookup1("1F02B") ->	{"0","",""};
lookup1("1F03") ->	{"0","1F01 0300",""};
lookup1("1F030") ->	{"0","",""};
lookup1("1F031") ->	{"0","",""};
lookup1("1F032") ->	{"0","",""};
lookup1("1F033") ->	{"0","",""};
lookup1("1F034") ->	{"0","",""};
lookup1("1F035") ->	{"0","",""};
lookup1("1F036") ->	{"0","",""};
lookup1("1F037") ->	{"0","",""};
lookup1("1F038") ->	{"0","",""};
lookup1("1F039") ->	{"0","",""};
lookup1("1F03A") ->	{"0","",""};
lookup1("1F03B") ->	{"0","",""};
lookup1("1F03C") ->	{"0","",""};
lookup1("1F03D") ->	{"0","",""};
lookup1("1F03E") ->	{"0","",""};
lookup1("1F03F") ->	{"0","",""};
lookup1("1F04") ->	{"0","1F00 0301",""};
lookup1("1F040") ->	{"0","",""};
lookup1("1F041") ->	{"0","",""};
lookup1("1F042") ->	{"0","",""};
lookup1("1F043") ->	{"0","",""};
lookup1("1F044") ->	{"0","",""};
lookup1("1F045") ->	{"0","",""};
lookup1("1F046") ->	{"0","",""};
lookup1("1F047") ->	{"0","",""};
lookup1("1F048") ->	{"0","",""};
lookup1("1F049") ->	{"0","",""};
lookup1("1F04A") ->	{"0","",""};
lookup1("1F04B") ->	{"0","",""};
lookup1("1F04C") ->	{"0","",""};
lookup1("1F04D") ->	{"0","",""};
lookup1("1F04E") ->	{"0","",""};
lookup1("1F04F") ->	{"0","",""};
lookup1("1F05") ->	{"0","1F01 0301",""};
lookup1("1F050") ->	{"0","",""};
lookup1("1F051") ->	{"0","",""};
lookup1("1F052") ->	{"0","",""};
lookup1("1F053") ->	{"0","",""};
lookup1("1F054") ->	{"0","",""};
lookup1("1F055") ->	{"0","",""};
lookup1("1F056") ->	{"0","",""};
lookup1("1F057") ->	{"0","",""};
lookup1("1F058") ->	{"0","",""};
lookup1("1F059") ->	{"0","",""};
lookup1("1F05A") ->	{"0","",""};
lookup1("1F05B") ->	{"0","",""};
lookup1("1F05C") ->	{"0","",""};
lookup1("1F05D") ->	{"0","",""};
lookup1("1F05E") ->	{"0","",""};
lookup1("1F05F") ->	{"0","",""};
lookup1("1F06") ->	{"0","1F00 0342",""};
lookup1("1F060") ->	{"0","",""};
lookup1("1F061") ->	{"0","",""};
lookup1("1F062") ->	{"0","",""};
lookup1("1F063") ->	{"0","",""};
lookup1("1F064") ->	{"0","",""};
lookup1("1F065") ->	{"0","",""};
lookup1("1F066") ->	{"0","",""};
lookup1("1F067") ->	{"0","",""};
lookup1("1F068") ->	{"0","",""};
lookup1("1F069") ->	{"0","",""};
lookup1("1F06A") ->	{"0","",""};
lookup1("1F06B") ->	{"0","",""};
lookup1("1F06C") ->	{"0","",""};
lookup1("1F06D") ->	{"0","",""};
lookup1("1F06E") ->	{"0","",""};
lookup1("1F06F") ->	{"0","",""};
lookup1("1F07") ->	{"0","1F01 0342",""};
lookup1("1F070") ->	{"0","",""};
lookup1("1F071") ->	{"0","",""};
lookup1("1F072") ->	{"0","",""};
lookup1("1F073") ->	{"0","",""};
lookup1("1F074") ->	{"0","",""};
lookup1("1F075") ->	{"0","",""};
lookup1("1F076") ->	{"0","",""};
lookup1("1F077") ->	{"0","",""};
lookup1("1F078") ->	{"0","",""};
lookup1("1F079") ->	{"0","",""};
lookup1("1F07A") ->	{"0","",""};
lookup1("1F07B") ->	{"0","",""};
lookup1("1F07C") ->	{"0","",""};
lookup1("1F07D") ->	{"0","",""};
lookup1("1F07E") ->	{"0","",""};
lookup1("1F07F") ->	{"0","",""};
lookup1("1F08") ->	{"0","0391 0313","1F00"};
lookup1("1F080") ->	{"0","",""};
lookup1("1F081") ->	{"0","",""};
lookup1("1F082") ->	{"0","",""};
lookup1("1F083") ->	{"0","",""};
lookup1("1F084") ->	{"0","",""};
lookup1("1F085") ->	{"0","",""};
lookup1("1F086") ->	{"0","",""};
lookup1("1F087") ->	{"0","",""};
lookup1("1F088") ->	{"0","",""};
lookup1("1F089") ->	{"0","",""};
lookup1("1F08A") ->	{"0","",""};
lookup1("1F08B") ->	{"0","",""};
lookup1("1F08C") ->	{"0","",""};
lookup1("1F08D") ->	{"0","",""};
lookup1("1F08E") ->	{"0","",""};
lookup1("1F08F") ->	{"0","",""};
lookup1("1F09") ->	{"0","0391 0314","1F01"};
lookup1("1F090") ->	{"0","",""};
lookup1("1F091") ->	{"0","",""};
lookup1("1F092") ->	{"0","",""};
lookup1("1F093") ->	{"0","",""};
lookup1("1F0A") ->	{"0","1F08 0300","1F02"};
lookup1("1F0A0") ->	{"0","",""};
lookup1("1F0A1") ->	{"0","",""};
lookup1("1F0A2") ->	{"0","",""};
lookup1("1F0A3") ->	{"0","",""};
lookup1("1F0A4") ->	{"0","",""};
lookup1("1F0A5") ->	{"0","",""};
lookup1("1F0A6") ->	{"0","",""};
lookup1("1F0A7") ->	{"0","",""};
lookup1("1F0A8") ->	{"0","",""};
lookup1("1F0A9") ->	{"0","",""};
lookup1("1F0AA") ->	{"0","",""};
lookup1("1F0AB") ->	{"0","",""};
lookup1("1F0AC") ->	{"0","",""};
lookup1("1F0AD") ->	{"0","",""};
lookup1("1F0AE") ->	{"0","",""};
lookup1("1F0B") ->	{"0","1F09 0300","1F03"};
lookup1("1F0B1") ->	{"0","",""};
lookup1("1F0B2") ->	{"0","",""};
lookup1("1F0B3") ->	{"0","",""};
lookup1("1F0B4") ->	{"0","",""};
lookup1("1F0B5") ->	{"0","",""};
lookup1("1F0B6") ->	{"0","",""};
lookup1("1F0B7") ->	{"0","",""};
lookup1("1F0B8") ->	{"0","",""};
lookup1("1F0B9") ->	{"0","",""};
lookup1("1F0BA") ->	{"0","",""};
lookup1("1F0BB") ->	{"0","",""};
lookup1("1F0BC") ->	{"0","",""};
lookup1("1F0BD") ->	{"0","",""};
lookup1("1F0BE") ->	{"0","",""};
lookup1("1F0BF") ->	{"0","",""};
lookup1("1F0C") ->	{"0","1F08 0301","1F04"};
lookup1("1F0C1") ->	{"0","",""};
lookup1("1F0C2") ->	{"0","",""};
lookup1("1F0C3") ->	{"0","",""};
lookup1("1F0C4") ->	{"0","",""};
lookup1("1F0C5") ->	{"0","",""};
lookup1("1F0C6") ->	{"0","",""};
lookup1("1F0C7") ->	{"0","",""};
lookup1("1F0C8") ->	{"0","",""};
lookup1("1F0C9") ->	{"0","",""};
lookup1("1F0CA") ->	{"0","",""};
lookup1("1F0CB") ->	{"0","",""};
lookup1("1F0CC") ->	{"0","",""};
lookup1("1F0CD") ->	{"0","",""};
lookup1("1F0CE") ->	{"0","",""};
lookup1("1F0CF") ->	{"0","",""};
lookup1("1F0D") ->	{"0","1F09 0301","1F05"};
lookup1("1F0D1") ->	{"0","",""};
lookup1("1F0D2") ->	{"0","",""};
lookup1("1F0D3") ->	{"0","",""};
lookup1("1F0D4") ->	{"0","",""};
lookup1("1F0D5") ->	{"0","",""};
lookup1("1F0D6") ->	{"0","",""};
lookup1("1F0D7") ->	{"0","",""};
lookup1("1F0D8") ->	{"0","",""};
lookup1("1F0D9") ->	{"0","",""};
lookup1("1F0DA") ->	{"0","",""};
lookup1("1F0DB") ->	{"0","",""};
lookup1("1F0DC") ->	{"0","",""};
lookup1("1F0DD") ->	{"0","",""};
lookup1("1F0DE") ->	{"0","",""};
lookup1("1F0DF") ->	{"0","",""};
lookup1("1F0E") ->	{"0","1F08 0342","1F06"};
lookup1("1F0E0") ->	{"0","",""};
lookup1("1F0E1") ->	{"0","",""};
lookup1("1F0E2") ->	{"0","",""};
lookup1("1F0E3") ->	{"0","",""};
lookup1("1F0E4") ->	{"0","",""};
lookup1("1F0E5") ->	{"0","",""};
lookup1("1F0E6") ->	{"0","",""};
lookup1("1F0E7") ->	{"0","",""};
lookup1("1F0E8") ->	{"0","",""};
lookup1("1F0E9") ->	{"0","",""};
lookup1("1F0EA") ->	{"0","",""};
lookup1("1F0EB") ->	{"0","",""};
lookup1("1F0EC") ->	{"0","",""};
lookup1("1F0ED") ->	{"0","",""};
lookup1("1F0EE") ->	{"0","",""};
lookup1("1F0EF") ->	{"0","",""};
lookup1("1F0F") ->	{"0","1F09 0342","1F07"};
lookup1("1F0F0") ->	{"0","",""};
lookup1("1F0F1") ->	{"0","",""};
lookup1("1F0F2") ->	{"0","",""};
lookup1("1F0F3") ->	{"0","",""};
lookup1("1F0F4") ->	{"0","",""};
lookup1("1F0F5") ->	{"0","",""};
lookup1("1F10") ->	{"0","03B5 0313",""};
lookup1("1F100") ->	{"0","<compat> 0030 002E",""};
lookup1("1F101") ->	{"0","<compat> 0030 002C",""};
lookup1("1F102") ->	{"0","<compat> 0031 002C",""};
lookup1("1F103") ->	{"0","<compat> 0032 002C",""};
lookup1("1F104") ->	{"0","<compat> 0033 002C",""};
lookup1("1F105") ->	{"0","<compat> 0034 002C",""};
lookup1("1F106") ->	{"0","<compat> 0035 002C",""};
lookup1("1F107") ->	{"0","<compat> 0036 002C",""};
lookup1("1F108") ->	{"0","<compat> 0037 002C",""};
lookup1("1F109") ->	{"0","<compat> 0038 002C",""};
lookup1("1F10A") ->	{"0","<compat> 0039 002C",""};
lookup1("1F10B") ->	{"0","",""};
lookup1("1F10C") ->	{"0","",""};
lookup1("1F11") ->	{"0","03B5 0314",""};
lookup1("1F110") ->	{"0","<compat> 0028 0041 0029",""};
lookup1("1F111") ->	{"0","<compat> 0028 0042 0029",""};
lookup1("1F112") ->	{"0","<compat> 0028 0043 0029",""};
lookup1("1F113") ->	{"0","<compat> 0028 0044 0029",""};
lookup1("1F114") ->	{"0","<compat> 0028 0045 0029",""};
lookup1("1F115") ->	{"0","<compat> 0028 0046 0029",""};
lookup1("1F116") ->	{"0","<compat> 0028 0047 0029",""};
lookup1("1F117") ->	{"0","<compat> 0028 0048 0029",""};
lookup1("1F118") ->	{"0","<compat> 0028 0049 0029",""};
lookup1("1F119") ->	{"0","<compat> 0028 004A 0029",""};
lookup1("1F11A") ->	{"0","<compat> 0028 004B 0029",""};
lookup1("1F11B") ->	{"0","<compat> 0028 004C 0029",""};
lookup1("1F11C") ->	{"0","<compat> 0028 004D 0029",""};
lookup1("1F11D") ->	{"0","<compat> 0028 004E 0029",""};
lookup1("1F11E") ->	{"0","<compat> 0028 004F 0029",""};
lookup1("1F11F") ->	{"0","<compat> 0028 0050 0029",""};
lookup1("1F12") ->	{"0","1F10 0300",""};
lookup1("1F120") ->	{"0","<compat> 0028 0051 0029",""};
lookup1("1F121") ->	{"0","<compat> 0028 0052 0029",""};
lookup1("1F122") ->	{"0","<compat> 0028 0053 0029",""};
lookup1("1F123") ->	{"0","<compat> 0028 0054 0029",""};
lookup1("1F124") ->	{"0","<compat> 0028 0055 0029",""};
lookup1("1F125") ->	{"0","<compat> 0028 0056 0029",""};
lookup1("1F126") ->	{"0","<compat> 0028 0057 0029",""};
lookup1("1F127") ->	{"0","<compat> 0028 0058 0029",""};
lookup1("1F128") ->	{"0","<compat> 0028 0059 0029",""};
lookup1("1F129") ->	{"0","<compat> 0028 005A 0029",""};
lookup1("1F12A") ->	{"0","<compat> 3014 0053 3015",""};
lookup1("1F12B") ->	{"0","<circle> 0043",""};
lookup1("1F12C") ->	{"0","<circle> 0052",""};
lookup1("1F12D") ->	{"0","<circle> 0043 0044",""};
lookup1("1F12E") ->	{"0","<circle> 0057 005A",""};
lookup1("1F13") ->	{"0","1F11 0300",""};
lookup1("1F130") ->	{"0","<square> 0041",""};
lookup1("1F131") ->	{"0","<square> 0042",""};
lookup1("1F132") ->	{"0","<square> 0043",""};
lookup1("1F133") ->	{"0","<square> 0044",""};
lookup1("1F134") ->	{"0","<square> 0045",""};
lookup1("1F135") ->	{"0","<square> 0046",""};
lookup1("1F136") ->	{"0","<square> 0047",""};
lookup1("1F137") ->	{"0","<square> 0048",""};
lookup1("1F138") ->	{"0","<square> 0049",""};
lookup1("1F139") ->	{"0","<square> 004A",""};
lookup1("1F13A") ->	{"0","<square> 004B",""};
lookup1("1F13B") ->	{"0","<square> 004C",""};
lookup1("1F13C") ->	{"0","<square> 004D",""};
lookup1("1F13D") ->	{"0","<square> 004E",""};
lookup1("1F13E") ->	{"0","<square> 004F",""};
lookup1("1F13F") ->	{"0","<square> 0050",""};
lookup1("1F14") ->	{"0","1F10 0301",""};
lookup1("1F140") ->	{"0","<square> 0051",""};
lookup1("1F141") ->	{"0","<square> 0052",""};
lookup1("1F142") ->	{"0","<square> 0053",""};
lookup1("1F143") ->	{"0","<square> 0054",""};
lookup1("1F144") ->	{"0","<square> 0055",""};
lookup1("1F145") ->	{"0","<square> 0056",""};
lookup1("1F146") ->	{"0","<square> 0057",""};
lookup1("1F147") ->	{"0","<square> 0058",""};
lookup1("1F148") ->	{"0","<square> 0059",""};
lookup1("1F149") ->	{"0","<square> 005A",""};
lookup1("1F14A") ->	{"0","<square> 0048 0056",""};
lookup1("1F14B") ->	{"0","<square> 004D 0056",""};
lookup1("1F14C") ->	{"0","<square> 0053 0044",""};
lookup1("1F14D") ->	{"0","<square> 0053 0053",""};
lookup1("1F14E") ->	{"0","<square> 0050 0050 0056",""};
lookup1("1F14F") ->	{"0","<square> 0057 0043",""};
lookup1("1F15") ->	{"0","1F11 0301",""};
lookup1("1F150") ->	{"0","",""};
lookup1("1F151") ->	{"0","",""};
lookup1("1F152") ->	{"0","",""};
lookup1("1F153") ->	{"0","",""};
lookup1("1F154") ->	{"0","",""};
lookup1("1F155") ->	{"0","",""};
lookup1("1F156") ->	{"0","",""};
lookup1("1F157") ->	{"0","",""};
lookup1("1F158") ->	{"0","",""};
lookup1("1F159") ->	{"0","",""};
lookup1("1F15A") ->	{"0","",""};
lookup1("1F15B") ->	{"0","",""};
lookup1("1F15C") ->	{"0","",""};
lookup1("1F15D") ->	{"0","",""};
lookup1("1F15E") ->	{"0","",""};
lookup1("1F15F") ->	{"0","",""};
lookup1("1F160") ->	{"0","",""};
lookup1("1F161") ->	{"0","",""};
lookup1("1F162") ->	{"0","",""};
lookup1("1F163") ->	{"0","",""};
lookup1("1F164") ->	{"0","",""};
lookup1("1F165") ->	{"0","",""};
lookup1("1F166") ->	{"0","",""};
lookup1("1F167") ->	{"0","",""};
lookup1("1F168") ->	{"0","",""};
lookup1("1F169") ->	{"0","",""};
lookup1("1F16A") ->	{"0","<super> 004D 0043",""};
lookup1("1F16B") ->	{"0","<super> 004D 0044",""};
lookup1("1F170") ->	{"0","",""};
lookup1("1F171") ->	{"0","",""};
lookup1("1F172") ->	{"0","",""};
lookup1("1F173") ->	{"0","",""};
lookup1("1F174") ->	{"0","",""};
lookup1("1F175") ->	{"0","",""};
lookup1("1F176") ->	{"0","",""};
lookup1("1F177") ->	{"0","",""};
lookup1("1F178") ->	{"0","",""};
lookup1("1F179") ->	{"0","",""};
lookup1("1F17A") ->	{"0","",""};
lookup1("1F17B") ->	{"0","",""};
lookup1("1F17C") ->	{"0","",""};
lookup1("1F17D") ->	{"0","",""};
lookup1("1F17E") ->	{"0","",""};
lookup1("1F17F") ->	{"0","",""};
lookup1("1F18") ->	{"0","0395 0313","1F10"};
lookup1("1F180") ->	{"0","",""};
lookup1("1F181") ->	{"0","",""};
lookup1("1F182") ->	{"0","",""};
lookup1("1F183") ->	{"0","",""};
lookup1("1F184") ->	{"0","",""};
lookup1("1F185") ->	{"0","",""};
lookup1("1F186") ->	{"0","",""};
lookup1("1F187") ->	{"0","",""};
lookup1("1F188") ->	{"0","",""};
lookup1("1F189") ->	{"0","",""};
lookup1("1F18A") ->	{"0","",""};
lookup1("1F18B") ->	{"0","",""};
lookup1("1F18C") ->	{"0","",""};
lookup1("1F18D") ->	{"0","",""};
lookup1("1F18E") ->	{"0","",""};
lookup1("1F18F") ->	{"0","",""};
lookup1("1F19") ->	{"0","0395 0314","1F11"};
lookup1("1F190") ->	{"0","<square> 0044 004A",""};
lookup1("1F191") ->	{"0","",""};
lookup1("1F192") ->	{"0","",""};
lookup1("1F193") ->	{"0","",""};
lookup1("1F194") ->	{"0","",""};
lookup1("1F195") ->	{"0","",""};
lookup1("1F196") ->	{"0","",""};
lookup1("1F197") ->	{"0","",""};
lookup1("1F198") ->	{"0","",""};
lookup1("1F199") ->	{"0","",""};
lookup1("1F19A") ->	{"0","",""};
lookup1("1F1A") ->	{"0","1F18 0300","1F12"};
lookup1("1F1B") ->	{"0","1F19 0300","1F13"};
lookup1("1F1C") ->	{"0","1F18 0301","1F14"};
lookup1("1F1D") ->	{"0","1F19 0301","1F15"};
lookup1("1F1E6") ->	{"0","",""};
lookup1("1F1E7") ->	{"0","",""};
lookup1("1F1E8") ->	{"0","",""};
lookup1("1F1E9") ->	{"0","",""};
lookup1("1F1EA") ->	{"0","",""};
lookup1("1F1EB") ->	{"0","",""};
lookup1("1F1EC") ->	{"0","",""};
lookup1("1F1ED") ->	{"0","",""};
lookup1("1F1EE") ->	{"0","",""};
lookup1("1F1EF") ->	{"0","",""};
lookup1("1F1F0") ->	{"0","",""};
lookup1("1F1F1") ->	{"0","",""};
lookup1("1F1F2") ->	{"0","",""};
lookup1("1F1F3") ->	{"0","",""};
lookup1("1F1F4") ->	{"0","",""};
lookup1("1F1F5") ->	{"0","",""};
lookup1("1F1F6") ->	{"0","",""};
lookup1("1F1F7") ->	{"0","",""};
lookup1("1F1F8") ->	{"0","",""};
lookup1("1F1F9") ->	{"0","",""};
lookup1("1F1FA") ->	{"0","",""};
lookup1("1F1FB") ->	{"0","",""};
lookup1("1F1FC") ->	{"0","",""};
lookup1("1F1FD") ->	{"0","",""};
lookup1("1F1FE") ->	{"0","",""};
lookup1("1F1FF") ->	{"0","",""};
lookup1("1F20") ->	{"0","03B7 0313",""};
lookup1("1F200") ->	{"0","<square> 307B 304B",""};
lookup1("1F201") ->	{"0","<square> 30B3 30B3",""};
lookup1("1F202") ->	{"0","<square> 30B5",""};
lookup1("1F21") ->	{"0","03B7 0314",""};
lookup1("1F210") ->	{"0","<square> 624B",""};
lookup1("1F211") ->	{"0","<square> 5B57",""};
lookup1("1F212") ->	{"0","<square> 53CC",""};
lookup1("1F213") ->	{"0","<square> 30C7",""};
lookup1("1F214") ->	{"0","<square> 4E8C",""};
lookup1("1F215") ->	{"0","<square> 591A",""};
lookup1("1F216") ->	{"0","<square> 89E3",""};
lookup1("1F217") ->	{"0","<square> 5929",""};
lookup1("1F218") ->	{"0","<square> 4EA4",""};
lookup1("1F219") ->	{"0","<square> 6620",""};
lookup1("1F21A") ->	{"0","<square> 7121",""};
lookup1("1F21B") ->	{"0","<square> 6599",""};
lookup1("1F21C") ->	{"0","<square> 524D",""};
lookup1("1F21D") ->	{"0","<square> 5F8C",""};
lookup1("1F21E") ->	{"0","<square> 518D",""};
lookup1("1F21F") ->	{"0","<square> 65B0",""};
lookup1("1F22") ->	{"0","1F20 0300",""};
lookup1("1F220") ->	{"0","<square> 521D",""};
lookup1("1F221") ->	{"0","<square> 7D42",""};
lookup1("1F222") ->	{"0","<square> 751F",""};
lookup1("1F223") ->	{"0","<square> 8CA9",""};
lookup1("1F224") ->	{"0","<square> 58F0",""};
lookup1("1F225") ->	{"0","<square> 5439",""};
lookup1("1F226") ->	{"0","<square> 6F14",""};
lookup1("1F227") ->	{"0","<square> 6295",""};
lookup1("1F228") ->	{"0","<square> 6355",""};
lookup1("1F229") ->	{"0","<square> 4E00",""};
lookup1("1F22A") ->	{"0","<square> 4E09",""};
lookup1("1F22B") ->	{"0","<square> 904A",""};
lookup1("1F22C") ->	{"0","<square> 5DE6",""};
lookup1("1F22D") ->	{"0","<square> 4E2D",""};
lookup1("1F22E") ->	{"0","<square> 53F3",""};
lookup1("1F22F") ->	{"0","<square> 6307",""};
lookup1("1F23") ->	{"0","1F21 0300",""};
lookup1("1F230") ->	{"0","<square> 8D70",""};
lookup1("1F231") ->	{"0","<square> 6253",""};
lookup1("1F232") ->	{"0","<square> 7981",""};
lookup1("1F233") ->	{"0","<square> 7A7A",""};
lookup1("1F234") ->	{"0","<square> 5408",""};
lookup1("1F235") ->	{"0","<square> 6E80",""};
lookup1("1F236") ->	{"0","<square> 6709",""};
lookup1("1F237") ->	{"0","<square> 6708",""};
lookup1("1F238") ->	{"0","<square> 7533",""};
lookup1("1F239") ->	{"0","<square> 5272",""};
lookup1("1F23A") ->	{"0","<square> 55B6",""};
lookup1("1F24") ->	{"0","1F20 0301",""};
lookup1("1F240") ->	{"0","<compat> 3014 672C 3015",""};
lookup1("1F241") ->	{"0","<compat> 3014 4E09 3015",""};
lookup1("1F242") ->	{"0","<compat> 3014 4E8C 3015",""};
lookup1("1F243") ->	{"0","<compat> 3014 5B89 3015",""};
lookup1("1F244") ->	{"0","<compat> 3014 70B9 3015",""};
lookup1("1F245") ->	{"0","<compat> 3014 6253 3015",""};
lookup1("1F246") ->	{"0","<compat> 3014 76D7 3015",""};
lookup1("1F247") ->	{"0","<compat> 3014 52DD 3015",""};
lookup1("1F248") ->	{"0","<compat> 3014 6557 3015",""};
lookup1("1F25") ->	{"0","1F21 0301",""};
lookup1("1F250") ->	{"0","<circle> 5F97",""};
lookup1("1F251") ->	{"0","<circle> 53EF",""};
lookup1("1F26") ->	{"0","1F20 0342",""};
lookup1("1F27") ->	{"0","1F21 0342",""};
lookup1("1F28") ->	{"0","0397 0313","1F20"};
lookup1("1F29") ->	{"0","0397 0314","1F21"};
lookup1("1F2A") ->	{"0","1F28 0300","1F22"};
lookup1("1F2B") ->	{"0","1F29 0300","1F23"};
lookup1("1F2C") ->	{"0","1F28 0301","1F24"};
lookup1("1F2D") ->	{"0","1F29 0301","1F25"};
lookup1("1F2E") ->	{"0","1F28 0342","1F26"};
lookup1("1F2F") ->	{"0","1F29 0342","1F27"};
lookup1("1F30") ->	{"0","03B9 0313",""};
lookup1("1F300") ->	{"0","",""};
lookup1("1F301") ->	{"0","",""};
lookup1("1F302") ->	{"0","",""};
lookup1("1F303") ->	{"0","",""};
lookup1("1F304") ->	{"0","",""};
lookup1("1F305") ->	{"0","",""};
lookup1("1F306") ->	{"0","",""};
lookup1("1F307") ->	{"0","",""};
lookup1("1F308") ->	{"0","",""};
lookup1("1F309") ->	{"0","",""};
lookup1("1F30A") ->	{"0","",""};
lookup1("1F30B") ->	{"0","",""};
lookup1("1F30C") ->	{"0","",""};
lookup1("1F30D") ->	{"0","",""};
lookup1("1F30E") ->	{"0","",""};
lookup1("1F30F") ->	{"0","",""};
lookup1("1F31") ->	{"0","03B9 0314",""};
lookup1("1F310") ->	{"0","",""};
lookup1("1F311") ->	{"0","",""};
lookup1("1F312") ->	{"0","",""};
lookup1("1F313") ->	{"0","",""};
lookup1("1F314") ->	{"0","",""};
lookup1("1F315") ->	{"0","",""};
lookup1("1F316") ->	{"0","",""};
lookup1("1F317") ->	{"0","",""};
lookup1("1F318") ->	{"0","",""};
lookup1("1F319") ->	{"0","",""};
lookup1("1F31A") ->	{"0","",""};
lookup1("1F31B") ->	{"0","",""};
lookup1("1F31C") ->	{"0","",""};
lookup1("1F31D") ->	{"0","",""};
lookup1("1F31E") ->	{"0","",""};
lookup1("1F31F") ->	{"0","",""};
lookup1("1F32") ->	{"0","1F30 0300",""};
lookup1("1F320") ->	{"0","",""};
lookup1("1F321") ->	{"0","",""};
lookup1("1F322") ->	{"0","",""};
lookup1("1F323") ->	{"0","",""};
lookup1("1F324") ->	{"0","",""};
lookup1("1F325") ->	{"0","",""};
lookup1("1F326") ->	{"0","",""};
lookup1("1F327") ->	{"0","",""};
lookup1("1F328") ->	{"0","",""};
lookup1("1F329") ->	{"0","",""};
lookup1("1F32A") ->	{"0","",""};
lookup1("1F32B") ->	{"0","",""};
lookup1("1F32C") ->	{"0","",""};
lookup1("1F33") ->	{"0","1F31 0300",""};
lookup1("1F330") ->	{"0","",""};
lookup1("1F331") ->	{"0","",""};
lookup1("1F332") ->	{"0","",""};
lookup1("1F333") ->	{"0","",""};
lookup1("1F334") ->	{"0","",""};
lookup1("1F335") ->	{"0","",""};
lookup1("1F336") ->	{"0","",""};
lookup1("1F337") ->	{"0","",""};
lookup1("1F338") ->	{"0","",""};
lookup1("1F339") ->	{"0","",""};
lookup1("1F33A") ->	{"0","",""};
lookup1("1F33B") ->	{"0","",""};
lookup1("1F33C") ->	{"0","",""};
lookup1("1F33D") ->	{"0","",""};
lookup1("1F33E") ->	{"0","",""};
lookup1("1F33F") ->	{"0","",""};
lookup1("1F34") ->	{"0","1F30 0301",""};
lookup1("1F340") ->	{"0","",""};
lookup1("1F341") ->	{"0","",""};
lookup1("1F342") ->	{"0","",""};
lookup1("1F343") ->	{"0","",""};
lookup1("1F344") ->	{"0","",""};
lookup1("1F345") ->	{"0","",""};
lookup1("1F346") ->	{"0","",""};
lookup1("1F347") ->	{"0","",""};
lookup1("1F348") ->	{"0","",""};
lookup1("1F349") ->	{"0","",""};
lookup1("1F34A") ->	{"0","",""};
lookup1("1F34B") ->	{"0","",""};
lookup1("1F34C") ->	{"0","",""};
lookup1("1F34D") ->	{"0","",""};
lookup1("1F34E") ->	{"0","",""};
lookup1("1F34F") ->	{"0","",""};
lookup1("1F35") ->	{"0","1F31 0301",""};
lookup1("1F350") ->	{"0","",""};
lookup1("1F351") ->	{"0","",""};
lookup1("1F352") ->	{"0","",""};
lookup1("1F353") ->	{"0","",""};
lookup1("1F354") ->	{"0","",""};
lookup1("1F355") ->	{"0","",""};
lookup1("1F356") ->	{"0","",""};
lookup1("1F357") ->	{"0","",""};
lookup1("1F358") ->	{"0","",""};
lookup1("1F359") ->	{"0","",""};
lookup1("1F35A") ->	{"0","",""};
lookup1("1F35B") ->	{"0","",""};
lookup1("1F35C") ->	{"0","",""};
lookup1("1F35D") ->	{"0","",""};
lookup1("1F35E") ->	{"0","",""};
lookup1("1F35F") ->	{"0","",""};
lookup1("1F36") ->	{"0","1F30 0342",""};
lookup1("1F360") ->	{"0","",""};
lookup1("1F361") ->	{"0","",""};
lookup1("1F362") ->	{"0","",""};
lookup1("1F363") ->	{"0","",""};
lookup1("1F364") ->	{"0","",""};
lookup1("1F365") ->	{"0","",""};
lookup1("1F366") ->	{"0","",""};
lookup1("1F367") ->	{"0","",""};
lookup1("1F368") ->	{"0","",""};
lookup1("1F369") ->	{"0","",""};
lookup1("1F36A") ->	{"0","",""};
lookup1("1F36B") ->	{"0","",""};
lookup1("1F36C") ->	{"0","",""};
lookup1("1F36D") ->	{"0","",""};
lookup1("1F36E") ->	{"0","",""};
lookup1("1F36F") ->	{"0","",""};
lookup1("1F37") ->	{"0","1F31 0342",""};
lookup1("1F370") ->	{"0","",""};
lookup1("1F371") ->	{"0","",""};
lookup1("1F372") ->	{"0","",""};
lookup1("1F373") ->	{"0","",""};
lookup1("1F374") ->	{"0","",""};
lookup1("1F375") ->	{"0","",""};
lookup1("1F376") ->	{"0","",""};
lookup1("1F377") ->	{"0","",""};
lookup1("1F378") ->	{"0","",""};
lookup1("1F379") ->	{"0","",""};
lookup1("1F37A") ->	{"0","",""};
lookup1("1F37B") ->	{"0","",""};
lookup1("1F37C") ->	{"0","",""};
lookup1("1F37D") ->	{"0","",""};
lookup1("1F38") ->	{"0","0399 0313","1F30"};
lookup1("1F380") ->	{"0","",""};
lookup1("1F381") ->	{"0","",""};
lookup1("1F382") ->	{"0","",""};
lookup1("1F383") ->	{"0","",""};
lookup1("1F384") ->	{"0","",""};
lookup1("1F385") ->	{"0","",""};
lookup1("1F386") ->	{"0","",""};
lookup1("1F387") ->	{"0","",""};
lookup1("1F388") ->	{"0","",""};
lookup1("1F389") ->	{"0","",""};
lookup1("1F38A") ->	{"0","",""};
lookup1("1F38B") ->	{"0","",""};
lookup1("1F38C") ->	{"0","",""};
lookup1("1F38D") ->	{"0","",""};
lookup1("1F38E") ->	{"0","",""};
lookup1("1F38F") ->	{"0","",""};
lookup1("1F39") ->	{"0","0399 0314","1F31"};
lookup1("1F390") ->	{"0","",""};
lookup1("1F391") ->	{"0","",""};
lookup1("1F392") ->	{"0","",""};
lookup1("1F393") ->	{"0","",""};
lookup1("1F394") ->	{"0","",""};
lookup1("1F395") ->	{"0","",""};
lookup1("1F396") ->	{"0","",""};
lookup1("1F397") ->	{"0","",""};
lookup1("1F398") ->	{"0","",""};
lookup1("1F399") ->	{"0","",""};
lookup1("1F39A") ->	{"0","",""};
lookup1("1F39B") ->	{"0","",""};
lookup1("1F39C") ->	{"0","",""};
lookup1("1F39D") ->	{"0","",""};
lookup1("1F39E") ->	{"0","",""};
lookup1("1F39F") ->	{"0","",""};
lookup1("1F3A") ->	{"0","1F38 0300","1F32"};
lookup1("1F3A0") ->	{"0","",""};
lookup1("1F3A1") ->	{"0","",""};
lookup1("1F3A2") ->	{"0","",""};
lookup1("1F3A3") ->	{"0","",""};
lookup1("1F3A4") ->	{"0","",""};
lookup1("1F3A5") ->	{"0","",""};
lookup1("1F3A6") ->	{"0","",""};
lookup1("1F3A7") ->	{"0","",""};
lookup1("1F3A8") ->	{"0","",""};
lookup1("1F3A9") ->	{"0","",""};
lookup1("1F3AA") ->	{"0","",""};
lookup1("1F3AB") ->	{"0","",""};
lookup1("1F3AC") ->	{"0","",""};
lookup1("1F3AD") ->	{"0","",""};
lookup1("1F3AE") ->	{"0","",""};
lookup1("1F3AF") ->	{"0","",""};
lookup1("1F3B") ->	{"0","1F39 0300","1F33"};
lookup1("1F3B0") ->	{"0","",""};
lookup1("1F3B1") ->	{"0","",""};
lookup1("1F3B2") ->	{"0","",""};
lookup1("1F3B3") ->	{"0","",""};
lookup1("1F3B4") ->	{"0","",""};
lookup1("1F3B5") ->	{"0","",""};
lookup1("1F3B6") ->	{"0","",""};
lookup1("1F3B7") ->	{"0","",""};
lookup1("1F3B8") ->	{"0","",""};
lookup1("1F3B9") ->	{"0","",""};
lookup1("1F3BA") ->	{"0","",""};
lookup1("1F3BB") ->	{"0","",""};
lookup1("1F3BC") ->	{"0","",""};
lookup1("1F3BD") ->	{"0","",""};
lookup1("1F3BE") ->	{"0","",""};
lookup1("1F3BF") ->	{"0","",""};
lookup1("1F3C") ->	{"0","1F38 0301","1F34"};
lookup1("1F3C0") ->	{"0","",""};
lookup1("1F3C1") ->	{"0","",""};
lookup1("1F3C2") ->	{"0","",""};
lookup1("1F3C3") ->	{"0","",""};
lookup1("1F3C4") ->	{"0","",""};
lookup1("1F3C5") ->	{"0","",""};
lookup1("1F3C6") ->	{"0","",""};
lookup1("1F3C7") ->	{"0","",""};
lookup1("1F3C8") ->	{"0","",""};
lookup1("1F3C9") ->	{"0","",""};
lookup1("1F3CA") ->	{"0","",""};
lookup1("1F3CB") ->	{"0","",""};
lookup1("1F3CC") ->	{"0","",""};
lookup1("1F3CD") ->	{"0","",""};
lookup1("1F3CE") ->	{"0","",""};
lookup1("1F3D") ->	{"0","1F39 0301","1F35"};
lookup1("1F3D4") ->	{"0","",""};
lookup1("1F3D5") ->	{"0","",""};
lookup1("1F3D6") ->	{"0","",""};
lookup1("1F3D7") ->	{"0","",""};
lookup1("1F3D8") ->	{"0","",""};
lookup1("1F3D9") ->	{"0","",""};
lookup1("1F3DA") ->	{"0","",""};
lookup1("1F3DB") ->	{"0","",""};
lookup1("1F3DC") ->	{"0","",""};
lookup1("1F3DD") ->	{"0","",""};
lookup1("1F3DE") ->	{"0","",""};
lookup1("1F3DF") ->	{"0","",""};
lookup1("1F3E") ->	{"0","1F38 0342","1F36"};
lookup1("1F3E0") ->	{"0","",""};
lookup1("1F3E1") ->	{"0","",""};
lookup1("1F3E2") ->	{"0","",""};
lookup1("1F3E3") ->	{"0","",""};
lookup1("1F3E4") ->	{"0","",""};
lookup1("1F3E5") ->	{"0","",""};
lookup1("1F3E6") ->	{"0","",""};
lookup1("1F3E7") ->	{"0","",""};
lookup1("1F3E8") ->	{"0","",""};
lookup1("1F3E9") ->	{"0","",""};
lookup1("1F3EA") ->	{"0","",""};
lookup1("1F3EB") ->	{"0","",""};
lookup1("1F3EC") ->	{"0","",""};
lookup1("1F3ED") ->	{"0","",""};
lookup1("1F3EE") ->	{"0","",""};
lookup1("1F3EF") ->	{"0","",""};
lookup1("1F3F") ->	{"0","1F39 0342","1F37"};
lookup1("1F3F0") ->	{"0","",""};
lookup1("1F3F1") ->	{"0","",""};
lookup1("1F3F2") ->	{"0","",""};
lookup1("1F3F3") ->	{"0","",""};
lookup1("1F3F4") ->	{"0","",""};
lookup1("1F3F5") ->	{"0","",""};
lookup1("1F3F6") ->	{"0","",""};
lookup1("1F3F7") ->	{"0","",""};
lookup1("1F40") ->	{"0","03BF 0313",""};
lookup1("1F400") ->	{"0","",""};
lookup1("1F401") ->	{"0","",""};
lookup1("1F402") ->	{"0","",""};
lookup1("1F403") ->	{"0","",""};
lookup1("1F404") ->	{"0","",""};
lookup1("1F405") ->	{"0","",""};
lookup1("1F406") ->	{"0","",""};
lookup1("1F407") ->	{"0","",""};
lookup1("1F408") ->	{"0","",""};
lookup1("1F409") ->	{"0","",""};
lookup1("1F40A") ->	{"0","",""};
lookup1("1F40B") ->	{"0","",""};
lookup1("1F40C") ->	{"0","",""};
lookup1("1F40D") ->	{"0","",""};
lookup1("1F40E") ->	{"0","",""};
lookup1("1F40F") ->	{"0","",""};
lookup1("1F41") ->	{"0","03BF 0314",""};
lookup1("1F410") ->	{"0","",""};
lookup1("1F411") ->	{"0","",""};
lookup1("1F412") ->	{"0","",""};
lookup1("1F413") ->	{"0","",""};
lookup1("1F414") ->	{"0","",""};
lookup1("1F415") ->	{"0","",""};
lookup1("1F416") ->	{"0","",""};
lookup1("1F417") ->	{"0","",""};
lookup1("1F418") ->	{"0","",""};
lookup1("1F419") ->	{"0","",""};
lookup1("1F41A") ->	{"0","",""};
lookup1("1F41B") ->	{"0","",""};
lookup1("1F41C") ->	{"0","",""};
lookup1("1F41D") ->	{"0","",""};
lookup1("1F41E") ->	{"0","",""};
lookup1("1F41F") ->	{"0","",""};
lookup1("1F42") ->	{"0","1F40 0300",""};
lookup1("1F420") ->	{"0","",""};
lookup1("1F421") ->	{"0","",""};
lookup1("1F422") ->	{"0","",""};
lookup1("1F423") ->	{"0","",""};
lookup1("1F424") ->	{"0","",""};
lookup1("1F425") ->	{"0","",""};
lookup1("1F426") ->	{"0","",""};
lookup1("1F427") ->	{"0","",""};
lookup1("1F428") ->	{"0","",""};
lookup1("1F429") ->	{"0","",""};
lookup1("1F42A") ->	{"0","",""};
lookup1("1F42B") ->	{"0","",""};
lookup1("1F42C") ->	{"0","",""};
lookup1("1F42D") ->	{"0","",""};
lookup1("1F42E") ->	{"0","",""};
lookup1("1F42F") ->	{"0","",""};
lookup1("1F43") ->	{"0","1F41 0300",""};
lookup1("1F430") ->	{"0","",""};
lookup1("1F431") ->	{"0","",""};
lookup1("1F432") ->	{"0","",""};
lookup1("1F433") ->	{"0","",""};
lookup1("1F434") ->	{"0","",""};
lookup1("1F435") ->	{"0","",""};
lookup1("1F436") ->	{"0","",""};
lookup1("1F437") ->	{"0","",""};
lookup1("1F438") ->	{"0","",""};
lookup1("1F439") ->	{"0","",""};
lookup1("1F43A") ->	{"0","",""};
lookup1("1F43B") ->	{"0","",""};
lookup1("1F43C") ->	{"0","",""};
lookup1("1F43D") ->	{"0","",""};
lookup1("1F43E") ->	{"0","",""};
lookup1("1F43F") ->	{"0","",""};
lookup1("1F44") ->	{"0","1F40 0301",""};
lookup1("1F440") ->	{"0","",""};
lookup1("1F441") ->	{"0","",""};
lookup1("1F442") ->	{"0","",""};
lookup1("1F443") ->	{"0","",""};
lookup1("1F444") ->	{"0","",""};
lookup1("1F445") ->	{"0","",""};
lookup1("1F446") ->	{"0","",""};
lookup1("1F447") ->	{"0","",""};
lookup1("1F448") ->	{"0","",""};
lookup1("1F449") ->	{"0","",""};
lookup1("1F44A") ->	{"0","",""};
lookup1("1F44B") ->	{"0","",""};
lookup1("1F44C") ->	{"0","",""};
lookup1("1F44D") ->	{"0","",""};
lookup1("1F44E") ->	{"0","",""};
lookup1("1F44F") ->	{"0","",""};
lookup1("1F45") ->	{"0","1F41 0301",""};
lookup1("1F450") ->	{"0","",""};
lookup1("1F451") ->	{"0","",""};
lookup1("1F452") ->	{"0","",""};
lookup1("1F453") ->	{"0","",""};
lookup1("1F454") ->	{"0","",""};
lookup1("1F455") ->	{"0","",""};
lookup1("1F456") ->	{"0","",""};
lookup1("1F457") ->	{"0","",""};
lookup1("1F458") ->	{"0","",""};
lookup1("1F459") ->	{"0","",""};
lookup1("1F45A") ->	{"0","",""};
lookup1("1F45B") ->	{"0","",""};
lookup1("1F45C") ->	{"0","",""};
lookup1("1F45D") ->	{"0","",""};
lookup1("1F45E") ->	{"0","",""};
lookup1("1F45F") ->	{"0","",""};
lookup1("1F460") ->	{"0","",""};
lookup1("1F461") ->	{"0","",""};
lookup1("1F462") ->	{"0","",""};
lookup1("1F463") ->	{"0","",""};
lookup1("1F464") ->	{"0","",""};
lookup1("1F465") ->	{"0","",""};
lookup1("1F466") ->	{"0","",""};
lookup1("1F467") ->	{"0","",""};
lookup1("1F468") ->	{"0","",""};
lookup1("1F469") ->	{"0","",""};
lookup1("1F46A") ->	{"0","",""};
lookup1("1F46B") ->	{"0","",""};
lookup1("1F46C") ->	{"0","",""};
lookup1("1F46D") ->	{"0","",""};
lookup1("1F46E") ->	{"0","",""};
lookup1("1F46F") ->	{"0","",""};
lookup1("1F470") ->	{"0","",""};
lookup1("1F471") ->	{"0","",""};
lookup1("1F472") ->	{"0","",""};
lookup1("1F473") ->	{"0","",""};
lookup1("1F474") ->	{"0","",""};
lookup1("1F475") ->	{"0","",""};
lookup1("1F476") ->	{"0","",""};
lookup1("1F477") ->	{"0","",""};
lookup1("1F478") ->	{"0","",""};
lookup1("1F479") ->	{"0","",""};
lookup1("1F47A") ->	{"0","",""};
lookup1("1F47B") ->	{"0","",""};
lookup1("1F47C") ->	{"0","",""};
lookup1("1F47D") ->	{"0","",""};
lookup1("1F47E") ->	{"0","",""};
lookup1("1F47F") ->	{"0","",""};
lookup1("1F48") ->	{"0","039F 0313","1F40"};
lookup1("1F480") ->	{"0","",""};
lookup1("1F481") ->	{"0","",""};
lookup1("1F482") ->	{"0","",""};
lookup1("1F483") ->	{"0","",""};
lookup1("1F484") ->	{"0","",""};
lookup1("1F485") ->	{"0","",""};
lookup1("1F486") ->	{"0","",""};
lookup1("1F487") ->	{"0","",""};
lookup1("1F488") ->	{"0","",""};
lookup1("1F489") ->	{"0","",""};
lookup1("1F48A") ->	{"0","",""};
lookup1("1F48B") ->	{"0","",""};
lookup1("1F48C") ->	{"0","",""};
lookup1("1F48D") ->	{"0","",""};
lookup1("1F48E") ->	{"0","",""};
lookup1("1F48F") ->	{"0","",""};
lookup1("1F49") ->	{"0","039F 0314","1F41"};
lookup1("1F490") ->	{"0","",""};
lookup1("1F491") ->	{"0","",""};
lookup1("1F492") ->	{"0","",""};
lookup1("1F493") ->	{"0","",""};
lookup1("1F494") ->	{"0","",""};
lookup1("1F495") ->	{"0","",""};
lookup1("1F496") ->	{"0","",""};
lookup1("1F497") ->	{"0","",""};
lookup1("1F498") ->	{"0","",""};
lookup1("1F499") ->	{"0","",""};
lookup1("1F49A") ->	{"0","",""};
lookup1("1F49B") ->	{"0","",""};
lookup1("1F49C") ->	{"0","",""};
lookup1("1F49D") ->	{"0","",""};
lookup1("1F49E") ->	{"0","",""};
lookup1("1F49F") ->	{"0","",""};
lookup1("1F4A") ->	{"0","1F48 0300","1F42"};
lookup1("1F4A0") ->	{"0","",""};
lookup1("1F4A1") ->	{"0","",""};
lookup1("1F4A2") ->	{"0","",""};
lookup1("1F4A3") ->	{"0","",""};
lookup1("1F4A4") ->	{"0","",""};
lookup1("1F4A5") ->	{"0","",""};
lookup1("1F4A6") ->	{"0","",""};
lookup1("1F4A7") ->	{"0","",""};
lookup1("1F4A8") ->	{"0","",""};
lookup1("1F4A9") ->	{"0","",""};
lookup1("1F4AA") ->	{"0","",""};
lookup1("1F4AB") ->	{"0","",""};
lookup1("1F4AC") ->	{"0","",""};
lookup1("1F4AD") ->	{"0","",""};
lookup1("1F4AE") ->	{"0","",""};
lookup1("1F4AF") ->	{"0","",""};
lookup1("1F4B") ->	{"0","1F49 0300","1F43"};
lookup1("1F4B0") ->	{"0","",""};
lookup1("1F4B1") ->	{"0","",""};
lookup1("1F4B2") ->	{"0","",""};
lookup1("1F4B3") ->	{"0","",""};
lookup1("1F4B4") ->	{"0","",""};
lookup1("1F4B5") ->	{"0","",""};
lookup1("1F4B6") ->	{"0","",""};
lookup1("1F4B7") ->	{"0","",""};
lookup1("1F4B8") ->	{"0","",""};
lookup1("1F4B9") ->	{"0","",""};
lookup1("1F4BA") ->	{"0","",""};
lookup1("1F4BB") ->	{"0","",""};
lookup1("1F4BC") ->	{"0","",""};
lookup1("1F4BD") ->	{"0","",""};
lookup1("1F4BE") ->	{"0","",""};
lookup1("1F4BF") ->	{"0","",""};
lookup1("1F4C") ->	{"0","1F48 0301","1F44"};
lookup1("1F4C0") ->	{"0","",""};
lookup1("1F4C1") ->	{"0","",""};
lookup1("1F4C2") ->	{"0","",""};
lookup1("1F4C3") ->	{"0","",""};
lookup1("1F4C4") ->	{"0","",""};
lookup1("1F4C5") ->	{"0","",""};
lookup1("1F4C6") ->	{"0","",""};
lookup1("1F4C7") ->	{"0","",""};
lookup1("1F4C8") ->	{"0","",""};
lookup1("1F4C9") ->	{"0","",""};
lookup1("1F4CA") ->	{"0","",""};
lookup1("1F4CB") ->	{"0","",""};
lookup1("1F4CC") ->	{"0","",""};
lookup1("1F4CD") ->	{"0","",""};
lookup1("1F4CE") ->	{"0","",""};
lookup1("1F4CF") ->	{"0","",""};
lookup1("1F4D") ->	{"0","1F49 0301","1F45"};
lookup1("1F4D0") ->	{"0","",""};
lookup1("1F4D1") ->	{"0","",""};
lookup1("1F4D2") ->	{"0","",""};
lookup1("1F4D3") ->	{"0","",""};
lookup1("1F4D4") ->	{"0","",""};
lookup1("1F4D5") ->	{"0","",""};
lookup1("1F4D6") ->	{"0","",""};
lookup1("1F4D7") ->	{"0","",""};
lookup1("1F4D8") ->	{"0","",""};
lookup1("1F4D9") ->	{"0","",""};
lookup1("1F4DA") ->	{"0","",""};
lookup1("1F4DB") ->	{"0","",""};
lookup1("1F4DC") ->	{"0","",""};
lookup1("1F4DD") ->	{"0","",""};
lookup1("1F4DE") ->	{"0","",""};
lookup1("1F4DF") ->	{"0","",""};
lookup1("1F4E0") ->	{"0","",""};
lookup1("1F4E1") ->	{"0","",""};
lookup1("1F4E2") ->	{"0","",""};
lookup1("1F4E3") ->	{"0","",""};
lookup1("1F4E4") ->	{"0","",""};
lookup1("1F4E5") ->	{"0","",""};
lookup1("1F4E6") ->	{"0","",""};
lookup1("1F4E7") ->	{"0","",""};
lookup1("1F4E8") ->	{"0","",""};
lookup1("1F4E9") ->	{"0","",""};
lookup1("1F4EA") ->	{"0","",""};
lookup1("1F4EB") ->	{"0","",""};
lookup1("1F4EC") ->	{"0","",""};
lookup1("1F4ED") ->	{"0","",""};
lookup1("1F4EE") ->	{"0","",""};
lookup1("1F4EF") ->	{"0","",""};
lookup1("1F4F0") ->	{"0","",""};
lookup1("1F4F1") ->	{"0","",""};
lookup1("1F4F2") ->	{"0","",""};
lookup1("1F4F3") ->	{"0","",""};
lookup1("1F4F4") ->	{"0","",""};
lookup1("1F4F5") ->	{"0","",""};
lookup1("1F4F6") ->	{"0","",""};
lookup1("1F4F7") ->	{"0","",""};
lookup1("1F4F8") ->	{"0","",""};
lookup1("1F4F9") ->	{"0","",""};
lookup1("1F4FA") ->	{"0","",""};
lookup1("1F4FB") ->	{"0","",""};
lookup1("1F4FC") ->	{"0","",""};
lookup1("1F4FD") ->	{"0","",""};
lookup1("1F4FE") ->	{"0","",""};
lookup1("1F50") ->	{"0","03C5 0313",""};
lookup1("1F500") ->	{"0","",""};
lookup1("1F501") ->	{"0","",""};
lookup1("1F502") ->	{"0","",""};
lookup1("1F503") ->	{"0","",""};
lookup1("1F504") ->	{"0","",""};
lookup1("1F505") ->	{"0","",""};
lookup1("1F506") ->	{"0","",""};
lookup1("1F507") ->	{"0","",""};
lookup1("1F508") ->	{"0","",""};
lookup1("1F509") ->	{"0","",""};
lookup1("1F50A") ->	{"0","",""};
lookup1("1F50B") ->	{"0","",""};
lookup1("1F50C") ->	{"0","",""};
lookup1("1F50D") ->	{"0","",""};
lookup1("1F50E") ->	{"0","",""};
lookup1("1F50F") ->	{"0","",""};
lookup1("1F51") ->	{"0","03C5 0314",""};
lookup1("1F510") ->	{"0","",""};
lookup1("1F511") ->	{"0","",""};
lookup1("1F512") ->	{"0","",""};
lookup1("1F513") ->	{"0","",""};
lookup1("1F514") ->	{"0","",""};
lookup1("1F515") ->	{"0","",""};
lookup1("1F516") ->	{"0","",""};
lookup1("1F517") ->	{"0","",""};
lookup1("1F518") ->	{"0","",""};
lookup1("1F519") ->	{"0","",""};
lookup1("1F51A") ->	{"0","",""};
lookup1("1F51B") ->	{"0","",""};
lookup1("1F51C") ->	{"0","",""};
lookup1("1F51D") ->	{"0","",""};
lookup1("1F51E") ->	{"0","",""};
lookup1("1F51F") ->	{"0","",""};
lookup1("1F52") ->	{"0","1F50 0300",""};
lookup1("1F520") ->	{"0","",""};
lookup1("1F521") ->	{"0","",""};
lookup1("1F522") ->	{"0","",""};
lookup1("1F523") ->	{"0","",""};
lookup1("1F524") ->	{"0","",""};
lookup1("1F525") ->	{"0","",""};
lookup1("1F526") ->	{"0","",""};
lookup1("1F527") ->	{"0","",""};
lookup1("1F528") ->	{"0","",""};
lookup1("1F529") ->	{"0","",""};
lookup1("1F52A") ->	{"0","",""};
lookup1("1F52B") ->	{"0","",""};
lookup1("1F52C") ->	{"0","",""};
lookup1("1F52D") ->	{"0","",""};
lookup1("1F52E") ->	{"0","",""};
lookup1("1F52F") ->	{"0","",""};
lookup1("1F53") ->	{"0","1F51 0300",""};
lookup1("1F530") ->	{"0","",""};
lookup1("1F531") ->	{"0","",""};
lookup1("1F532") ->	{"0","",""};
lookup1("1F533") ->	{"0","",""};
lookup1("1F534") ->	{"0","",""};
lookup1("1F535") ->	{"0","",""};
lookup1("1F536") ->	{"0","",""};
lookup1("1F537") ->	{"0","",""};
lookup1("1F538") ->	{"0","",""};
lookup1("1F539") ->	{"0","",""};
lookup1("1F53A") ->	{"0","",""};
lookup1("1F53B") ->	{"0","",""};
lookup1("1F53C") ->	{"0","",""};
lookup1("1F53D") ->	{"0","",""};
lookup1("1F53E") ->	{"0","",""};
lookup1("1F53F") ->	{"0","",""};
lookup1("1F54") ->	{"0","1F50 0301",""};
lookup1("1F540") ->	{"0","",""};
lookup1("1F541") ->	{"0","",""};
lookup1("1F542") ->	{"0","",""};
lookup1("1F543") ->	{"0","",""};
lookup1("1F544") ->	{"0","",""};
lookup1("1F545") ->	{"0","",""};
lookup1("1F546") ->	{"0","",""};
lookup1("1F547") ->	{"0","",""};
lookup1("1F548") ->	{"0","",""};
lookup1("1F549") ->	{"0","",""};
lookup1("1F54A") ->	{"0","",""};
lookup1("1F55") ->	{"0","1F51 0301",""};
lookup1("1F550") ->	{"0","",""};
lookup1("1F551") ->	{"0","",""};
lookup1("1F552") ->	{"0","",""};
lookup1("1F553") ->	{"0","",""};
lookup1("1F554") ->	{"0","",""};
lookup1("1F555") ->	{"0","",""};
lookup1("1F556") ->	{"0","",""};
lookup1("1F557") ->	{"0","",""};
lookup1("1F558") ->	{"0","",""};
lookup1("1F559") ->	{"0","",""};
lookup1("1F55A") ->	{"0","",""};
lookup1("1F55B") ->	{"0","",""};
lookup1("1F55C") ->	{"0","",""};
lookup1("1F55D") ->	{"0","",""};
lookup1("1F55E") ->	{"0","",""};
lookup1("1F55F") ->	{"0","",""};
lookup1("1F56") ->	{"0","1F50 0342",""};
lookup1("1F560") ->	{"0","",""};
lookup1("1F561") ->	{"0","",""};
lookup1("1F562") ->	{"0","",""};
lookup1("1F563") ->	{"0","",""};
lookup1("1F564") ->	{"0","",""};
lookup1("1F565") ->	{"0","",""};
lookup1("1F566") ->	{"0","",""};
lookup1("1F567") ->	{"0","",""};
lookup1("1F568") ->	{"0","",""};
lookup1("1F569") ->	{"0","",""};
lookup1("1F56A") ->	{"0","",""};
lookup1("1F56B") ->	{"0","",""};
lookup1("1F56C") ->	{"0","",""};
lookup1("1F56D") ->	{"0","",""};
lookup1("1F56E") ->	{"0","",""};
lookup1("1F56F") ->	{"0","",""};
lookup1("1F57") ->	{"0","1F51 0342",""};
lookup1("1F570") ->	{"0","",""};
lookup1("1F571") ->	{"0","",""};
lookup1("1F572") ->	{"0","",""};
lookup1("1F573") ->	{"0","",""};
lookup1("1F574") ->	{"0","",""};
lookup1("1F575") ->	{"0","",""};
lookup1("1F576") ->	{"0","",""};
lookup1("1F577") ->	{"0","",""};
lookup1("1F578") ->	{"0","",""};
lookup1("1F579") ->	{"0","",""};
lookup1("1F57B") ->	{"0","",""};
lookup1("1F57C") ->	{"0","",""};
lookup1("1F57D") ->	{"0","",""};
lookup1("1F57E") ->	{"0","",""};
lookup1("1F57F") ->	{"0","",""};
lookup1("1F580") ->	{"0","",""};
lookup1("1F581") ->	{"0","",""};
lookup1("1F582") ->	{"0","",""};
lookup1("1F583") ->	{"0","",""};
lookup1("1F584") ->	{"0","",""};
lookup1("1F585") ->	{"0","",""};
lookup1("1F586") ->	{"0","",""};
lookup1("1F587") ->	{"0","",""};
lookup1("1F588") ->	{"0","",""};
lookup1("1F589") ->	{"0","",""};
lookup1("1F58A") ->	{"0","",""};
lookup1("1F58B") ->	{"0","",""};
lookup1("1F58C") ->	{"0","",""};
lookup1("1F58D") ->	{"0","",""};
lookup1("1F58E") ->	{"0","",""};
lookup1("1F58F") ->	{"0","",""};
lookup1("1F59") ->	{"0","03A5 0314","1F51"};
lookup1("1F590") ->	{"0","",""};
lookup1("1F591") ->	{"0","",""};
lookup1("1F592") ->	{"0","",""};
lookup1("1F593") ->	{"0","",""};
lookup1("1F594") ->	{"0","",""};
lookup1("1F595") ->	{"0","",""};
lookup1("1F596") ->	{"0","",""};
lookup1("1F597") ->	{"0","",""};
lookup1("1F598") ->	{"0","",""};
lookup1("1F599") ->	{"0","",""};
lookup1("1F59A") ->	{"0","",""};
lookup1("1F59B") ->	{"0","",""};
lookup1("1F59C") ->	{"0","",""};
lookup1("1F59D") ->	{"0","",""};
lookup1("1F59E") ->	{"0","",""};
lookup1("1F59F") ->	{"0","",""};
lookup1("1F5A0") ->	{"0","",""};
lookup1("1F5A1") ->	{"0","",""};
lookup1("1F5A2") ->	{"0","",""};
lookup1("1F5A3") ->	{"0","",""};
lookup1("1F5A5") ->	{"0","",""};
lookup1("1F5A6") ->	{"0","",""};
lookup1("1F5A7") ->	{"0","",""};
lookup1("1F5A8") ->	{"0","",""};
lookup1("1F5A9") ->	{"0","",""};
lookup1("1F5AA") ->	{"0","",""};
lookup1("1F5AB") ->	{"0","",""};
lookup1("1F5AC") ->	{"0","",""};
lookup1("1F5AD") ->	{"0","",""};
lookup1("1F5AE") ->	{"0","",""};
lookup1("1F5AF") ->	{"0","",""};
lookup1("1F5B") ->	{"0","1F59 0300","1F53"};
lookup1("1F5B0") ->	{"0","",""};
lookup1("1F5B1") ->	{"0","",""};
lookup1("1F5B2") ->	{"0","",""};
lookup1("1F5B3") ->	{"0","",""};
lookup1("1F5B4") ->	{"0","",""};
lookup1("1F5B5") ->	{"0","",""};
lookup1("1F5B6") ->	{"0","",""};
lookup1("1F5B7") ->	{"0","",""};
lookup1("1F5B8") ->	{"0","",""};
lookup1("1F5B9") ->	{"0","",""};
lookup1("1F5BA") ->	{"0","",""};
lookup1("1F5BB") ->	{"0","",""};
lookup1("1F5BC") ->	{"0","",""};
lookup1("1F5BD") ->	{"0","",""};
lookup1("1F5BE") ->	{"0","",""};
lookup1("1F5BF") ->	{"0","",""};
lookup1("1F5C0") ->	{"0","",""};
lookup1("1F5C1") ->	{"0","",""};
lookup1("1F5C2") ->	{"0","",""};
lookup1("1F5C3") ->	{"0","",""};
lookup1("1F5C4") ->	{"0","",""};
lookup1("1F5C5") ->	{"0","",""};
lookup1("1F5C6") ->	{"0","",""};
lookup1("1F5C7") ->	{"0","",""};
lookup1("1F5C8") ->	{"0","",""};
lookup1("1F5C9") ->	{"0","",""};
lookup1("1F5CA") ->	{"0","",""};
lookup1("1F5CB") ->	{"0","",""};
lookup1("1F5CC") ->	{"0","",""};
lookup1("1F5CD") ->	{"0","",""};
lookup1("1F5CE") ->	{"0","",""};
lookup1("1F5CF") ->	{"0","",""};
lookup1("1F5D") ->	{"0","1F59 0301","1F55"};
lookup1("1F5D0") ->	{"0","",""};
lookup1("1F5D1") ->	{"0","",""};
lookup1("1F5D2") ->	{"0","",""};
lookup1("1F5D3") ->	{"0","",""};
lookup1("1F5D4") ->	{"0","",""};
lookup1("1F5D5") ->	{"0","",""};
lookup1("1F5D6") ->	{"0","",""};
lookup1("1F5D7") ->	{"0","",""};
lookup1("1F5D8") ->	{"0","",""};
lookup1("1F5D9") ->	{"0","",""};
lookup1("1F5DA") ->	{"0","",""};
lookup1("1F5DB") ->	{"0","",""};
lookup1("1F5DC") ->	{"0","",""};
lookup1("1F5DD") ->	{"0","",""};
lookup1("1F5DE") ->	{"0","",""};
lookup1("1F5DF") ->	{"0","",""};
lookup1("1F5E0") ->	{"0","",""};
lookup1("1F5E1") ->	{"0","",""};
lookup1("1F5E2") ->	{"0","",""};
lookup1("1F5E3") ->	{"0","",""};
lookup1("1F5E4") ->	{"0","",""};
lookup1("1F5E5") ->	{"0","",""};
lookup1("1F5E6") ->	{"0","",""};
lookup1("1F5E7") ->	{"0","",""};
lookup1("1F5E8") ->	{"0","",""};
lookup1("1F5E9") ->	{"0","",""};
lookup1("1F5EA") ->	{"0","",""};
lookup1("1F5EB") ->	{"0","",""};
lookup1("1F5EC") ->	{"0","",""};
lookup1("1F5ED") ->	{"0","",""};
lookup1("1F5EE") ->	{"0","",""};
lookup1("1F5EF") ->	{"0","",""};
lookup1("1F5F") ->	{"0","1F59 0342","1F57"};
lookup1("1F5F0") ->	{"0","",""};
lookup1("1F5F1") ->	{"0","",""};
lookup1("1F5F2") ->	{"0","",""};
lookup1("1F5F3") ->	{"0","",""};
lookup1("1F5F4") ->	{"0","",""};
lookup1("1F5F5") ->	{"0","",""};
lookup1("1F5F6") ->	{"0","",""};
lookup1("1F5F7") ->	{"0","",""};
lookup1("1F5F8") ->	{"0","",""};
lookup1("1F5F9") ->	{"0","",""};
lookup1("1F5FA") ->	{"0","",""};
lookup1("1F5FB") ->	{"0","",""};
lookup1("1F5FC") ->	{"0","",""};
lookup1("1F5FD") ->	{"0","",""};
lookup1("1F5FE") ->	{"0","",""};
lookup1("1F5FF") ->	{"0","",""};
lookup1("1F60") ->	{"0","03C9 0313",""};
lookup1("1F600") ->	{"0","",""};
lookup1("1F601") ->	{"0","",""};
lookup1("1F602") ->	{"0","",""};
lookup1("1F603") ->	{"0","",""};
lookup1("1F604") ->	{"0","",""};
lookup1("1F605") ->	{"0","",""};
lookup1("1F606") ->	{"0","",""};
lookup1("1F607") ->	{"0","",""};
lookup1("1F608") ->	{"0","",""};
lookup1("1F609") ->	{"0","",""};
lookup1("1F60A") ->	{"0","",""};
lookup1("1F60B") ->	{"0","",""};
lookup1("1F60C") ->	{"0","",""};
lookup1("1F60D") ->	{"0","",""};
lookup1("1F60E") ->	{"0","",""};
lookup1("1F60F") ->	{"0","",""};
lookup1("1F61") ->	{"0","03C9 0314",""};
lookup1("1F610") ->	{"0","",""};
lookup1("1F611") ->	{"0","",""};
lookup1("1F612") ->	{"0","",""};
lookup1("1F613") ->	{"0","",""};
lookup1("1F614") ->	{"0","",""};
lookup1("1F615") ->	{"0","",""};
lookup1("1F616") ->	{"0","",""};
lookup1("1F617") ->	{"0","",""};
lookup1("1F618") ->	{"0","",""};
lookup1("1F619") ->	{"0","",""};
lookup1("1F61A") ->	{"0","",""};
lookup1("1F61B") ->	{"0","",""};
lookup1("1F61C") ->	{"0","",""};
lookup1("1F61D") ->	{"0","",""};
lookup1("1F61E") ->	{"0","",""};
lookup1("1F61F") ->	{"0","",""};
lookup1("1F62") ->	{"0","1F60 0300",""};
lookup1("1F620") ->	{"0","",""};
lookup1("1F621") ->	{"0","",""};
lookup1("1F622") ->	{"0","",""};
lookup1("1F623") ->	{"0","",""};
lookup1("1F624") ->	{"0","",""};
lookup1("1F625") ->	{"0","",""};
lookup1("1F626") ->	{"0","",""};
lookup1("1F627") ->	{"0","",""};
lookup1("1F628") ->	{"0","",""};
lookup1("1F629") ->	{"0","",""};
lookup1("1F62A") ->	{"0","",""};
lookup1("1F62B") ->	{"0","",""};
lookup1("1F62C") ->	{"0","",""};
lookup1("1F62D") ->	{"0","",""};
lookup1("1F62E") ->	{"0","",""};
lookup1("1F62F") ->	{"0","",""};
lookup1("1F63") ->	{"0","1F61 0300",""};
lookup1("1F630") ->	{"0","",""};
lookup1("1F631") ->	{"0","",""};
lookup1("1F632") ->	{"0","",""};
lookup1("1F633") ->	{"0","",""};
lookup1("1F634") ->	{"0","",""};
lookup1("1F635") ->	{"0","",""};
lookup1("1F636") ->	{"0","",""};
lookup1("1F637") ->	{"0","",""};
lookup1("1F638") ->	{"0","",""};
lookup1("1F639") ->	{"0","",""};
lookup1("1F63A") ->	{"0","",""};
lookup1("1F63B") ->	{"0","",""};
lookup1("1F63C") ->	{"0","",""};
lookup1("1F63D") ->	{"0","",""};
lookup1("1F63E") ->	{"0","",""};
lookup1("1F63F") ->	{"0","",""};
lookup1("1F64") ->	{"0","1F60 0301",""};
lookup1("1F640") ->	{"0","",""};
lookup1("1F641") ->	{"0","",""};
lookup1("1F642") ->	{"0","",""};
lookup1("1F645") ->	{"0","",""};
lookup1("1F646") ->	{"0","",""};
lookup1("1F647") ->	{"0","",""};
lookup1("1F648") ->	{"0","",""};
lookup1("1F649") ->	{"0","",""};
lookup1("1F64A") ->	{"0","",""};
lookup1("1F64B") ->	{"0","",""};
lookup1("1F64C") ->	{"0","",""};
lookup1("1F64D") ->	{"0","",""};
lookup1("1F64E") ->	{"0","",""};
lookup1("1F64F") ->	{"0","",""};
lookup1("1F65") ->	{"0","1F61 0301",""};
lookup1("1F650") ->	{"0","",""};
lookup1("1F651") ->	{"0","",""};
lookup1("1F652") ->	{"0","",""};
lookup1("1F653") ->	{"0","",""};
lookup1("1F654") ->	{"0","",""};
lookup1("1F655") ->	{"0","",""};
lookup1("1F656") ->	{"0","",""};
lookup1("1F657") ->	{"0","",""};
lookup1("1F658") ->	{"0","",""};
lookup1("1F659") ->	{"0","",""};
lookup1("1F65A") ->	{"0","",""};
lookup1("1F65B") ->	{"0","",""};
lookup1("1F65C") ->	{"0","",""};
lookup1("1F65D") ->	{"0","",""};
lookup1("1F65E") ->	{"0","",""};
lookup1("1F65F") ->	{"0","",""};
lookup1("1F66") ->	{"0","1F60 0342",""};
lookup1("1F660") ->	{"0","",""};
lookup1("1F661") ->	{"0","",""};
lookup1("1F662") ->	{"0","",""};
lookup1("1F663") ->	{"0","",""};
lookup1("1F664") ->	{"0","",""};
lookup1("1F665") ->	{"0","",""};
lookup1("1F666") ->	{"0","",""};
lookup1("1F667") ->	{"0","",""};
lookup1("1F668") ->	{"0","",""};
lookup1("1F669") ->	{"0","",""};
lookup1("1F66A") ->	{"0","",""};
lookup1("1F66B") ->	{"0","",""};
lookup1("1F66C") ->	{"0","",""};
lookup1("1F66D") ->	{"0","",""};
lookup1("1F66E") ->	{"0","",""};
lookup1("1F66F") ->	{"0","",""};
lookup1("1F67") ->	{"0","1F61 0342",""};
lookup1("1F670") ->	{"0","",""};
lookup1("1F671") ->	{"0","",""};
lookup1("1F672") ->	{"0","",""};
lookup1("1F673") ->	{"0","",""};
lookup1("1F674") ->	{"0","",""};
lookup1("1F675") ->	{"0","",""};
lookup1("1F676") ->	{"0","",""};
lookup1("1F677") ->	{"0","",""};
lookup1("1F678") ->	{"0","",""};
lookup1("1F679") ->	{"0","",""};
lookup1("1F67A") ->	{"0","",""};
lookup1("1F67B") ->	{"0","",""};
lookup1("1F67C") ->	{"0","",""};
lookup1("1F67D") ->	{"0","",""};
lookup1("1F67E") ->	{"0","",""};
lookup1("1F67F") ->	{"0","",""};
lookup1("1F68") ->	{"0","03A9 0313","1F60"};
lookup1("1F680") ->	{"0","",""};
lookup1("1F681") ->	{"0","",""};
lookup1("1F682") ->	{"0","",""};
lookup1("1F683") ->	{"0","",""};
lookup1("1F684") ->	{"0","",""};
lookup1("1F685") ->	{"0","",""};
lookup1("1F686") ->	{"0","",""};
lookup1("1F687") ->	{"0","",""};
lookup1("1F688") ->	{"0","",""};
lookup1("1F689") ->	{"0","",""};
lookup1("1F68A") ->	{"0","",""};
lookup1("1F68B") ->	{"0","",""};
lookup1("1F68C") ->	{"0","",""};
lookup1("1F68D") ->	{"0","",""};
lookup1("1F68E") ->	{"0","",""};
lookup1("1F68F") ->	{"0","",""};
lookup1("1F69") ->	{"0","03A9 0314","1F61"};
lookup1("1F690") ->	{"0","",""};
lookup1("1F691") ->	{"0","",""};
lookup1("1F692") ->	{"0","",""};
lookup1("1F693") ->	{"0","",""};
lookup1("1F694") ->	{"0","",""};
lookup1("1F695") ->	{"0","",""};
lookup1("1F696") ->	{"0","",""};
lookup1("1F697") ->	{"0","",""};
lookup1("1F698") ->	{"0","",""};
lookup1("1F699") ->	{"0","",""};
lookup1("1F69A") ->	{"0","",""};
lookup1("1F69B") ->	{"0","",""};
lookup1("1F69C") ->	{"0","",""};
lookup1("1F69D") ->	{"0","",""};
lookup1("1F69E") ->	{"0","",""};
lookup1("1F69F") ->	{"0","",""};
lookup1("1F6A") ->	{"0","1F68 0300","1F62"};
lookup1("1F6A0") ->	{"0","",""};
lookup1("1F6A1") ->	{"0","",""};
lookup1("1F6A2") ->	{"0","",""};
lookup1("1F6A3") ->	{"0","",""};
lookup1("1F6A4") ->	{"0","",""};
lookup1("1F6A5") ->	{"0","",""};
lookup1("1F6A6") ->	{"0","",""};
lookup1("1F6A7") ->	{"0","",""};
lookup1("1F6A8") ->	{"0","",""};
lookup1("1F6A9") ->	{"0","",""};
lookup1("1F6AA") ->	{"0","",""};
lookup1("1F6AB") ->	{"0","",""};
lookup1("1F6AC") ->	{"0","",""};
lookup1("1F6AD") ->	{"0","",""};
lookup1("1F6AE") ->	{"0","",""};
lookup1("1F6AF") ->	{"0","",""};
lookup1("1F6B") ->	{"0","1F69 0300","1F63"};
lookup1("1F6B0") ->	{"0","",""};
lookup1("1F6B1") ->	{"0","",""};
lookup1("1F6B2") ->	{"0","",""};
lookup1("1F6B3") ->	{"0","",""};
lookup1("1F6B4") ->	{"0","",""};
lookup1("1F6B5") ->	{"0","",""};
lookup1("1F6B6") ->	{"0","",""};
lookup1("1F6B7") ->	{"0","",""};
lookup1("1F6B8") ->	{"0","",""};
lookup1("1F6B9") ->	{"0","",""};
lookup1("1F6BA") ->	{"0","",""};
lookup1("1F6BB") ->	{"0","",""};
lookup1("1F6BC") ->	{"0","",""};
lookup1("1F6BD") ->	{"0","",""};
lookup1("1F6BE") ->	{"0","",""};
lookup1("1F6BF") ->	{"0","",""};
lookup1("1F6C") ->	{"0","1F68 0301","1F64"};
lookup1("1F6C0") ->	{"0","",""};
lookup1("1F6C1") ->	{"0","",""};
lookup1("1F6C2") ->	{"0","",""};
lookup1("1F6C3") ->	{"0","",""};
lookup1("1F6C4") ->	{"0","",""};
lookup1("1F6C5") ->	{"0","",""};
lookup1("1F6C6") ->	{"0","",""};
lookup1("1F6C7") ->	{"0","",""};
lookup1("1F6C8") ->	{"0","",""};
lookup1("1F6C9") ->	{"0","",""};
lookup1("1F6CA") ->	{"0","",""};
lookup1("1F6CB") ->	{"0","",""};
lookup1("1F6CC") ->	{"0","",""};
lookup1("1F6CD") ->	{"0","",""};
lookup1("1F6CE") ->	{"0","",""};
lookup1("1F6CF") ->	{"0","",""};
lookup1("1F6D") ->	{"0","1F69 0301","1F65"};
lookup1("1F6E") ->	{"0","1F68 0342","1F66"};
lookup1("1F6E0") ->	{"0","",""};
lookup1("1F6E1") ->	{"0","",""};
lookup1("1F6E2") ->	{"0","",""};
lookup1("1F6E3") ->	{"0","",""};
lookup1("1F6E4") ->	{"0","",""};
lookup1("1F6E5") ->	{"0","",""};
lookup1("1F6E6") ->	{"0","",""};
lookup1("1F6E7") ->	{"0","",""};
lookup1("1F6E8") ->	{"0","",""};
lookup1("1F6E9") ->	{"0","",""};
lookup1("1F6EA") ->	{"0","",""};
lookup1("1F6EB") ->	{"0","",""};
lookup1("1F6EC") ->	{"0","",""};
lookup1("1F6F") ->	{"0","1F69 0342","1F67"};
lookup1("1F6F0") ->	{"0","",""};
lookup1("1F6F1") ->	{"0","",""};
lookup1("1F6F2") ->	{"0","",""};
lookup1("1F6F3") ->	{"0","",""};
lookup1("1F70") ->	{"0","03B1 0300",""};
lookup1("1F700") ->	{"0","",""};
lookup1("1F701") ->	{"0","",""};
lookup1("1F702") ->	{"0","",""};
lookup1("1F703") ->	{"0","",""};
lookup1("1F704") ->	{"0","",""};
lookup1("1F705") ->	{"0","",""};
lookup1("1F706") ->	{"0","",""};
lookup1("1F707") ->	{"0","",""};
lookup1("1F708") ->	{"0","",""};
lookup1("1F709") ->	{"0","",""};
lookup1("1F70A") ->	{"0","",""};
lookup1("1F70B") ->	{"0","",""};
lookup1("1F70C") ->	{"0","",""};
lookup1("1F70D") ->	{"0","",""};
lookup1("1F70E") ->	{"0","",""};
lookup1("1F70F") ->	{"0","",""};
lookup1("1F71") ->	{"0","03AC",""};
lookup1("1F710") ->	{"0","",""};
lookup1("1F711") ->	{"0","",""};
lookup1("1F712") ->	{"0","",""};
lookup1("1F713") ->	{"0","",""};
lookup1("1F714") ->	{"0","",""};
lookup1("1F715") ->	{"0","",""};
lookup1("1F716") ->	{"0","",""};
lookup1("1F717") ->	{"0","",""};
lookup1("1F718") ->	{"0","",""};
lookup1("1F719") ->	{"0","",""};
lookup1("1F71A") ->	{"0","",""};
lookup1("1F71B") ->	{"0","",""};
lookup1("1F71C") ->	{"0","",""};
lookup1("1F71D") ->	{"0","",""};
lookup1("1F71E") ->	{"0","",""};
lookup1("1F71F") ->	{"0","",""};
lookup1("1F72") ->	{"0","03B5 0300",""};
lookup1("1F720") ->	{"0","",""};
lookup1("1F721") ->	{"0","",""};
lookup1("1F722") ->	{"0","",""};
lookup1("1F723") ->	{"0","",""};
lookup1("1F724") ->	{"0","",""};
lookup1("1F725") ->	{"0","",""};
lookup1("1F726") ->	{"0","",""};
lookup1("1F727") ->	{"0","",""};
lookup1("1F728") ->	{"0","",""};
lookup1("1F729") ->	{"0","",""};
lookup1("1F72A") ->	{"0","",""};
lookup1("1F72B") ->	{"0","",""};
lookup1("1F72C") ->	{"0","",""};
lookup1("1F72D") ->	{"0","",""};
lookup1("1F72E") ->	{"0","",""};
lookup1("1F72F") ->	{"0","",""};
lookup1("1F73") ->	{"0","03AD",""};
lookup1("1F730") ->	{"0","",""};
lookup1("1F731") ->	{"0","",""};
lookup1("1F732") ->	{"0","",""};
lookup1("1F733") ->	{"0","",""};
lookup1("1F734") ->	{"0","",""};
lookup1("1F735") ->	{"0","",""};
lookup1("1F736") ->	{"0","",""};
lookup1("1F737") ->	{"0","",""};
lookup1("1F738") ->	{"0","",""};
lookup1("1F739") ->	{"0","",""};
lookup1("1F73A") ->	{"0","",""};
lookup1("1F73B") ->	{"0","",""};
lookup1("1F73C") ->	{"0","",""};
lookup1("1F73D") ->	{"0","",""};
lookup1("1F73E") ->	{"0","",""};
lookup1("1F73F") ->	{"0","",""};
lookup1("1F74") ->	{"0","03B7 0300",""};
lookup1("1F740") ->	{"0","",""};
lookup1("1F741") ->	{"0","",""};
lookup1("1F742") ->	{"0","",""};
lookup1("1F743") ->	{"0","",""};
lookup1("1F744") ->	{"0","",""};
lookup1("1F745") ->	{"0","",""};
lookup1("1F746") ->	{"0","",""};
lookup1("1F747") ->	{"0","",""};
lookup1("1F748") ->	{"0","",""};
lookup1("1F749") ->	{"0","",""};
lookup1("1F74A") ->	{"0","",""};
lookup1("1F74B") ->	{"0","",""};
lookup1("1F74C") ->	{"0","",""};
lookup1("1F74D") ->	{"0","",""};
lookup1("1F74E") ->	{"0","",""};
lookup1("1F74F") ->	{"0","",""};
lookup1("1F75") ->	{"0","03AE",""};
lookup1("1F750") ->	{"0","",""};
lookup1("1F751") ->	{"0","",""};
lookup1("1F752") ->	{"0","",""};
lookup1("1F753") ->	{"0","",""};
lookup1("1F754") ->	{"0","",""};
lookup1("1F755") ->	{"0","",""};
lookup1("1F756") ->	{"0","",""};
lookup1("1F757") ->	{"0","",""};
lookup1("1F758") ->	{"0","",""};
lookup1("1F759") ->	{"0","",""};
lookup1("1F75A") ->	{"0","",""};
lookup1("1F75B") ->	{"0","",""};
lookup1("1F75C") ->	{"0","",""};
lookup1("1F75D") ->	{"0","",""};
lookup1("1F75E") ->	{"0","",""};
lookup1("1F75F") ->	{"0","",""};
lookup1("1F76") ->	{"0","03B9 0300",""};
lookup1("1F760") ->	{"0","",""};
lookup1("1F761") ->	{"0","",""};
lookup1("1F762") ->	{"0","",""};
lookup1("1F763") ->	{"0","",""};
lookup1("1F764") ->	{"0","",""};
lookup1("1F765") ->	{"0","",""};
lookup1("1F766") ->	{"0","",""};
lookup1("1F767") ->	{"0","",""};
lookup1("1F768") ->	{"0","",""};
lookup1("1F769") ->	{"0","",""};
lookup1("1F76A") ->	{"0","",""};
lookup1("1F76B") ->	{"0","",""};
lookup1("1F76C") ->	{"0","",""};
lookup1("1F76D") ->	{"0","",""};
lookup1("1F76E") ->	{"0","",""};
lookup1("1F76F") ->	{"0","",""};
lookup1("1F77") ->	{"0","03AF",""};
lookup1("1F770") ->	{"0","",""};
lookup1("1F771") ->	{"0","",""};
lookup1("1F772") ->	{"0","",""};
lookup1("1F773") ->	{"0","",""};
lookup1("1F78") ->	{"0","03BF 0300",""};
lookup1("1F780") ->	{"0","",""};
lookup1("1F781") ->	{"0","",""};
lookup1("1F782") ->	{"0","",""};
lookup1("1F783") ->	{"0","",""};
lookup1("1F784") ->	{"0","",""};
lookup1("1F785") ->	{"0","",""};
lookup1("1F786") ->	{"0","",""};
lookup1("1F787") ->	{"0","",""};
lookup1("1F788") ->	{"0","",""};
lookup1("1F789") ->	{"0","",""};
lookup1("1F78A") ->	{"0","",""};
lookup1("1F78B") ->	{"0","",""};
lookup1("1F78C") ->	{"0","",""};
lookup1("1F78D") ->	{"0","",""};
lookup1("1F78E") ->	{"0","",""};
lookup1("1F78F") ->	{"0","",""};
lookup1("1F79") ->	{"0","03CC",""};
lookup1("1F790") ->	{"0","",""};
lookup1("1F791") ->	{"0","",""};
lookup1("1F792") ->	{"0","",""};
lookup1("1F793") ->	{"0","",""};
lookup1("1F794") ->	{"0","",""};
lookup1("1F795") ->	{"0","",""};
lookup1("1F796") ->	{"0","",""};
lookup1("1F797") ->	{"0","",""};
lookup1("1F798") ->	{"0","",""};
lookup1("1F799") ->	{"0","",""};
lookup1("1F79A") ->	{"0","",""};
lookup1("1F79B") ->	{"0","",""};
lookup1("1F79C") ->	{"0","",""};
lookup1("1F79D") ->	{"0","",""};
lookup1("1F79E") ->	{"0","",""};
lookup1("1F79F") ->	{"0","",""};
lookup1("1F7A") ->	{"0","03C5 0300",""};
lookup1("1F7A0") ->	{"0","",""};
lookup1("1F7A1") ->	{"0","",""};
lookup1("1F7A2") ->	{"0","",""};
lookup1("1F7A3") ->	{"0","",""};
lookup1("1F7A4") ->	{"0","",""};
lookup1("1F7A5") ->	{"0","",""};
lookup1("1F7A6") ->	{"0","",""};
lookup1("1F7A7") ->	{"0","",""};
lookup1("1F7A8") ->	{"0","",""};
lookup1("1F7A9") ->	{"0","",""};
lookup1("1F7AA") ->	{"0","",""};
lookup1("1F7AB") ->	{"0","",""};
lookup1("1F7AC") ->	{"0","",""};
lookup1("1F7AD") ->	{"0","",""};
lookup1("1F7AE") ->	{"0","",""};
lookup1("1F7AF") ->	{"0","",""};
lookup1("1F7B") ->	{"0","03CD",""};
lookup1("1F7B0") ->	{"0","",""};
lookup1("1F7B1") ->	{"0","",""};
lookup1("1F7B2") ->	{"0","",""};
lookup1("1F7B3") ->	{"0","",""};
lookup1("1F7B4") ->	{"0","",""};
lookup1("1F7B5") ->	{"0","",""};
lookup1("1F7B6") ->	{"0","",""};
lookup1("1F7B7") ->	{"0","",""};
lookup1("1F7B8") ->	{"0","",""};
lookup1("1F7B9") ->	{"0","",""};
lookup1("1F7BA") ->	{"0","",""};
lookup1("1F7BB") ->	{"0","",""};
lookup1("1F7BC") ->	{"0","",""};
lookup1("1F7BD") ->	{"0","",""};
lookup1("1F7BE") ->	{"0","",""};
lookup1("1F7BF") ->	{"0","",""};
lookup1("1F7C") ->	{"0","03C9 0300",""};
lookup1("1F7C0") ->	{"0","",""};
lookup1("1F7C1") ->	{"0","",""};
lookup1("1F7C2") ->	{"0","",""};
lookup1("1F7C3") ->	{"0","",""};
lookup1("1F7C4") ->	{"0","",""};
lookup1("1F7C5") ->	{"0","",""};
lookup1("1F7C6") ->	{"0","",""};
lookup1("1F7C7") ->	{"0","",""};
lookup1("1F7C8") ->	{"0","",""};
lookup1("1F7C9") ->	{"0","",""};
lookup1("1F7CA") ->	{"0","",""};
lookup1("1F7CB") ->	{"0","",""};
lookup1("1F7CC") ->	{"0","",""};
lookup1("1F7CD") ->	{"0","",""};
lookup1("1F7CE") ->	{"0","",""};
lookup1("1F7CF") ->	{"0","",""};
lookup1("1F7D") ->	{"0","03CE",""};
lookup1("1F7D0") ->	{"0","",""};
lookup1("1F7D1") ->	{"0","",""};
lookup1("1F7D2") ->	{"0","",""};
lookup1("1F7D3") ->	{"0","",""};
lookup1("1F7D4") ->	{"0","",""};
lookup1("1F80") ->	{"0","1F00 0345",""};
lookup1("1F800") ->	{"0","",""};
lookup1("1F801") ->	{"0","",""};
lookup1("1F802") ->	{"0","",""};
lookup1("1F803") ->	{"0","",""};
lookup1("1F804") ->	{"0","",""};
lookup1("1F805") ->	{"0","",""};
lookup1("1F806") ->	{"0","",""};
lookup1("1F807") ->	{"0","",""};
lookup1("1F808") ->	{"0","",""};
lookup1("1F809") ->	{"0","",""};
lookup1("1F80A") ->	{"0","",""};
lookup1("1F80B") ->	{"0","",""};
lookup1("1F81") ->	{"0","1F01 0345",""};
lookup1("1F810") ->	{"0","",""};
lookup1("1F811") ->	{"0","",""};
lookup1("1F812") ->	{"0","",""};
lookup1("1F813") ->	{"0","",""};
lookup1("1F814") ->	{"0","",""};
lookup1("1F815") ->	{"0","",""};
lookup1("1F816") ->	{"0","",""};
lookup1("1F817") ->	{"0","",""};
lookup1("1F818") ->	{"0","",""};
lookup1("1F819") ->	{"0","",""};
lookup1("1F81A") ->	{"0","",""};
lookup1("1F81B") ->	{"0","",""};
lookup1("1F81C") ->	{"0","",""};
lookup1("1F81D") ->	{"0","",""};
lookup1("1F81E") ->	{"0","",""};
lookup1("1F81F") ->	{"0","",""};
lookup1("1F82") ->	{"0","1F02 0345",""};
lookup1("1F820") ->	{"0","",""};
lookup1("1F821") ->	{"0","",""};
lookup1("1F822") ->	{"0","",""};
lookup1("1F823") ->	{"0","",""};
lookup1("1F824") ->	{"0","",""};
lookup1("1F825") ->	{"0","",""};
lookup1("1F826") ->	{"0","",""};
lookup1("1F827") ->	{"0","",""};
lookup1("1F828") ->	{"0","",""};
lookup1("1F829") ->	{"0","",""};
lookup1("1F82A") ->	{"0","",""};
lookup1("1F82B") ->	{"0","",""};
lookup1("1F82C") ->	{"0","",""};
lookup1("1F82D") ->	{"0","",""};
lookup1("1F82E") ->	{"0","",""};
lookup1("1F82F") ->	{"0","",""};
lookup1("1F83") ->	{"0","1F03 0345",""};
lookup1("1F830") ->	{"0","",""};
lookup1("1F831") ->	{"0","",""};
lookup1("1F832") ->	{"0","",""};
lookup1("1F833") ->	{"0","",""};
lookup1("1F834") ->	{"0","",""};
lookup1("1F835") ->	{"0","",""};
lookup1("1F836") ->	{"0","",""};
lookup1("1F837") ->	{"0","",""};
lookup1("1F838") ->	{"0","",""};
lookup1("1F839") ->	{"0","",""};
lookup1("1F83A") ->	{"0","",""};
lookup1("1F83B") ->	{"0","",""};
lookup1("1F83C") ->	{"0","",""};
lookup1("1F83D") ->	{"0","",""};
lookup1("1F83E") ->	{"0","",""};
lookup1("1F83F") ->	{"0","",""};
lookup1("1F84") ->	{"0","1F04 0345",""};
lookup1("1F840") ->	{"0","",""};
lookup1("1F841") ->	{"0","",""};
lookup1("1F842") ->	{"0","",""};
lookup1("1F843") ->	{"0","",""};
lookup1("1F844") ->	{"0","",""};
lookup1("1F845") ->	{"0","",""};
lookup1("1F846") ->	{"0","",""};
lookup1("1F847") ->	{"0","",""};
lookup1("1F85") ->	{"0","1F05 0345",""};
lookup1("1F850") ->	{"0","",""};
lookup1("1F851") ->	{"0","",""};
lookup1("1F852") ->	{"0","",""};
lookup1("1F853") ->	{"0","",""};
lookup1("1F854") ->	{"0","",""};
lookup1("1F855") ->	{"0","",""};
lookup1("1F856") ->	{"0","",""};
lookup1("1F857") ->	{"0","",""};
lookup1("1F858") ->	{"0","",""};
lookup1("1F859") ->	{"0","",""};
lookup1("1F86") ->	{"0","1F06 0345",""};
lookup1("1F860") ->	{"0","",""};
lookup1("1F861") ->	{"0","",""};
lookup1("1F862") ->	{"0","",""};
lookup1("1F863") ->	{"0","",""};
lookup1("1F864") ->	{"0","",""};
lookup1("1F865") ->	{"0","",""};
lookup1("1F866") ->	{"0","",""};
lookup1("1F867") ->	{"0","",""};
lookup1("1F868") ->	{"0","",""};
lookup1("1F869") ->	{"0","",""};
lookup1("1F86A") ->	{"0","",""};
lookup1("1F86B") ->	{"0","",""};
lookup1("1F86C") ->	{"0","",""};
lookup1("1F86D") ->	{"0","",""};
lookup1("1F86E") ->	{"0","",""};
lookup1("1F86F") ->	{"0","",""};
lookup1("1F87") ->	{"0","1F07 0345",""};
lookup1("1F870") ->	{"0","",""};
lookup1("1F871") ->	{"0","",""};
lookup1("1F872") ->	{"0","",""};
lookup1("1F873") ->	{"0","",""};
lookup1("1F874") ->	{"0","",""};
lookup1("1F875") ->	{"0","",""};
lookup1("1F876") ->	{"0","",""};
lookup1("1F877") ->	{"0","",""};
lookup1("1F878") ->	{"0","",""};
lookup1("1F879") ->	{"0","",""};
lookup1("1F87A") ->	{"0","",""};
lookup1("1F87B") ->	{"0","",""};
lookup1("1F87C") ->	{"0","",""};
lookup1("1F87D") ->	{"0","",""};
lookup1("1F87E") ->	{"0","",""};
lookup1("1F87F") ->	{"0","",""};
lookup1("1F88") ->	{"0","1F08 0345","1F80"};
lookup1("1F880") ->	{"0","",""};
lookup1("1F881") ->	{"0","",""};
lookup1("1F882") ->	{"0","",""};
lookup1("1F883") ->	{"0","",""};
lookup1("1F884") ->	{"0","",""};
lookup1("1F885") ->	{"0","",""};
lookup1("1F886") ->	{"0","",""};
lookup1("1F887") ->	{"0","",""};
lookup1("1F89") ->	{"0","1F09 0345","1F81"};
lookup1("1F890") ->	{"0","",""};
lookup1("1F891") ->	{"0","",""};
lookup1("1F892") ->	{"0","",""};
lookup1("1F893") ->	{"0","",""};
lookup1("1F894") ->	{"0","",""};
lookup1("1F895") ->	{"0","",""};
lookup1("1F896") ->	{"0","",""};
lookup1("1F897") ->	{"0","",""};
lookup1("1F898") ->	{"0","",""};
lookup1("1F899") ->	{"0","",""};
lookup1("1F89A") ->	{"0","",""};
lookup1("1F89B") ->	{"0","",""};
lookup1("1F89C") ->	{"0","",""};
lookup1("1F89D") ->	{"0","",""};
lookup1("1F89E") ->	{"0","",""};
lookup1("1F89F") ->	{"0","",""};
lookup1("1F8A") ->	{"0","1F0A 0345","1F82"};
lookup1("1F8A0") ->	{"0","",""};
lookup1("1F8A1") ->	{"0","",""};
lookup1("1F8A2") ->	{"0","",""};
lookup1("1F8A3") ->	{"0","",""};
lookup1("1F8A4") ->	{"0","",""};
lookup1("1F8A5") ->	{"0","",""};
lookup1("1F8A6") ->	{"0","",""};
lookup1("1F8A7") ->	{"0","",""};
lookup1("1F8A8") ->	{"0","",""};
lookup1("1F8A9") ->	{"0","",""};
lookup1("1F8AA") ->	{"0","",""};
lookup1("1F8AB") ->	{"0","",""};
lookup1("1F8AC") ->	{"0","",""};
lookup1("1F8AD") ->	{"0","",""};
lookup1("1F8B") ->	{"0","1F0B 0345","1F83"};
lookup1("1F8C") ->	{"0","1F0C 0345","1F84"};
lookup1("1F8D") ->	{"0","1F0D 0345","1F85"};
lookup1("1F8E") ->	{"0","1F0E 0345","1F86"};
lookup1("1F8F") ->	{"0","1F0F 0345","1F87"};
lookup1("1F90") ->	{"0","1F20 0345",""};
lookup1("1F91") ->	{"0","1F21 0345",""};
lookup1("1F92") ->	{"0","1F22 0345",""};
lookup1("1F93") ->	{"0","1F23 0345",""};
lookup1("1F94") ->	{"0","1F24 0345",""};
lookup1("1F95") ->	{"0","1F25 0345",""};
lookup1("1F96") ->	{"0","1F26 0345",""};
lookup1("1F97") ->	{"0","1F27 0345",""};
lookup1("1F98") ->	{"0","1F28 0345","1F90"};
lookup1("1F99") ->	{"0","1F29 0345","1F91"};
lookup1("1F9A") ->	{"0","1F2A 0345","1F92"};
lookup1("1F9B") ->	{"0","1F2B 0345","1F93"};
lookup1("1F9C") ->	{"0","1F2C 0345","1F94"};
lookup1("1F9D") ->	{"0","1F2D 0345","1F95"};
lookup1("1F9E") ->	{"0","1F2E 0345","1F96"};
lookup1("1F9F") ->	{"0","1F2F 0345","1F97"};
lookup1("1FA0") ->	{"0","1F60 0345",""};
lookup1("1FA1") ->	{"0","1F61 0345",""};
lookup1("1FA2") ->	{"0","1F62 0345",""};
lookup1("1FA3") ->	{"0","1F63 0345",""};
lookup1("1FA4") ->	{"0","1F64 0345",""};
lookup1("1FA5") ->	{"0","1F65 0345",""};
lookup1("1FA6") ->	{"0","1F66 0345",""};
lookup1("1FA7") ->	{"0","1F67 0345",""};
lookup1("1FA8") ->	{"0","1F68 0345","1FA0"};
lookup1("1FA9") ->	{"0","1F69 0345","1FA1"};
lookup1("1FAA") ->	{"0","1F6A 0345","1FA2"};
lookup1("1FAB") ->	{"0","1F6B 0345","1FA3"};
lookup1("1FAC") ->	{"0","1F6C 0345","1FA4"};
lookup1("1FAD") ->	{"0","1F6D 0345","1FA5"};
lookup1("1FAE") ->	{"0","1F6E 0345","1FA6"};
lookup1("1FAF") ->	{"0","1F6F 0345","1FA7"};
lookup1("1FB0") ->	{"0","03B1 0306",""};
lookup1("1FB1") ->	{"0","03B1 0304",""};
lookup1("1FB2") ->	{"0","1F70 0345",""};
lookup1("1FB3") ->	{"0","03B1 0345",""};
lookup1("1FB4") ->	{"0","03AC 0345",""};
lookup1("1FB6") ->	{"0","03B1 0342",""};
lookup1("1FB7") ->	{"0","1FB6 0345",""};
lookup1("1FB8") ->	{"0","0391 0306","1FB0"};
lookup1("1FB9") ->	{"0","0391 0304","1FB1"};
lookup1("1FBA") ->	{"0","0391 0300","1F70"};
lookup1("1FBB") ->	{"0","0386","1F71"};
lookup1("1FBC") ->	{"0","0391 0345","1FB3"};
lookup1("1FBD") ->	{"0","<compat> 0020 0313",""};
lookup1("1FBE") ->	{"0","03B9",""};
lookup1("1FBF") ->	{"0","<compat> 0020 0313",""};
lookup1("1FC0") ->	{"0","<compat> 0020 0342",""};
lookup1("1FC1") ->	{"0","00A8 0342",""};
lookup1("1FC2") ->	{"0","1F74 0345",""};
lookup1("1FC3") ->	{"0","03B7 0345",""};
lookup1("1FC4") ->	{"0","03AE 0345",""};
lookup1("1FC6") ->	{"0","03B7 0342",""};
lookup1("1FC7") ->	{"0","1FC6 0345",""};
lookup1("1FC8") ->	{"0","0395 0300","1F72"};
lookup1("1FC9") ->	{"0","0388","1F73"};
lookup1("1FCA") ->	{"0","0397 0300","1F74"};
lookup1("1FCB") ->	{"0","0389","1F75"};
lookup1("1FCC") ->	{"0","0397 0345","1FC3"};
lookup1("1FCD") ->	{"0","1FBF 0300",""};
lookup1("1FCE") ->	{"0","1FBF 0301",""};
lookup1("1FCF") ->	{"0","1FBF 0342",""};
lookup1("1FD0") ->	{"0","03B9 0306",""};
lookup1("1FD1") ->	{"0","03B9 0304",""};
lookup1("1FD2") ->	{"0","03CA 0300",""};
lookup1("1FD3") ->	{"0","0390",""};
lookup1("1FD6") ->	{"0","03B9 0342",""};
lookup1("1FD7") ->	{"0","03CA 0342",""};
lookup1("1FD8") ->	{"0","0399 0306","1FD0"};
lookup1("1FD9") ->	{"0","0399 0304","1FD1"};
lookup1("1FDA") ->	{"0","0399 0300","1F76"};
lookup1("1FDB") ->	{"0","038A","1F77"};
lookup1("1FDD") ->	{"0","1FFE 0300",""};
lookup1("1FDE") ->	{"0","1FFE 0301",""};
lookup1("1FDF") ->	{"0","1FFE 0342",""};
lookup1("1FE0") ->	{"0","03C5 0306",""};
lookup1("1FE1") ->	{"0","03C5 0304",""};
lookup1("1FE2") ->	{"0","03CB 0300",""};
lookup1("1FE3") ->	{"0","03B0",""};
lookup1("1FE4") ->	{"0","03C1 0313",""};
lookup1("1FE5") ->	{"0","03C1 0314",""};
lookup1("1FE6") ->	{"0","03C5 0342",""};
lookup1("1FE7") ->	{"0","03CB 0342",""};
lookup1("1FE8") ->	{"0","03A5 0306","1FE0"};
lookup1("1FE9") ->	{"0","03A5 0304","1FE1"};
lookup1("1FEA") ->	{"0","03A5 0300","1F7A"};
lookup1("1FEB") ->	{"0","038E","1F7B"};
lookup1("1FEC") ->	{"0","03A1 0314","1FE5"};
lookup1("1FED") ->	{"0","00A8 0300",""};
lookup1("1FEE") ->	{"0","0385",""};
lookup1("1FEF") ->	{"0","0060",""};
lookup1("1FF2") ->	{"0","1F7C 0345",""};
lookup1("1FF3") ->	{"0","03C9 0345",""};
lookup1("1FF4") ->	{"0","03CE 0345",""};
lookup1("1FF6") ->	{"0","03C9 0342",""};
lookup1("1FF7") ->	{"0","1FF6 0345",""};
lookup1("1FF8") ->	{"0","039F 0300","1F78"};
lookup1("1FF9") ->	{"0","038C","1F79"};
lookup1("1FFA") ->	{"0","03A9 0300","1F7C"};
lookup1("1FFB") ->	{"0","038F","1F7D"};
lookup1("1FFC") ->	{"0","03A9 0345","1FF3"};
lookup1("1FFD") ->	{"0","00B4",""};
lookup1("1FFE") ->	{"0","<compat> 0020 0314",""};
lookup1("2000") ->	{"0","2002",""};
lookup1("20000") ->	{"0","",""};
lookup1("2001") ->	{"0","2003",""};
lookup1("2002") ->	{"0","<compat> 0020",""};
lookup1("2003") ->	{"0","<compat> 0020",""};
lookup1("2004") ->	{"0","<compat> 0020",""};
lookup1("2005") ->	{"0","<compat> 0020",""};
lookup1("2006") ->	{"0","<compat> 0020",""};
lookup1("2007") ->	{"0","<noBreak> 0020",""};
lookup1("2008") ->	{"0","<compat> 0020",""};
lookup1("2009") ->	{"0","<compat> 0020",""};
lookup1("200A") ->	{"0","<compat> 0020",""};
lookup1("200B") ->	{"0","",""};
lookup1("200C") ->	{"0","",""};
lookup1("200D") ->	{"0","",""};
lookup1("200E") ->	{"0","",""};
lookup1("200F") ->	{"0","",""};
lookup1("2010") ->	{"0","",""};
lookup1("2011") ->	{"0","<noBreak> 2010",""};
lookup1("2012") ->	{"0","",""};
lookup1("2013") ->	{"0","",""};
lookup1("2014") ->	{"0","",""};
lookup1("2015") ->	{"0","",""};
lookup1("2016") ->	{"0","",""};
lookup1("2017") ->	{"0","<compat> 0020 0333",""};
lookup1("2018") ->	{"0","",""};
lookup1("2019") ->	{"0","",""};
lookup1("201A") ->	{"0","",""};
lookup1("201B") ->	{"0","",""};
lookup1("201C") ->	{"0","",""};
lookup1("201D") ->	{"0","",""};
lookup1("201E") ->	{"0","",""};
lookup1("201F") ->	{"0","",""};
lookup1("2020") ->	{"0","",""};
lookup1("2021") ->	{"0","",""};
lookup1("2022") ->	{"0","",""};
lookup1("2023") ->	{"0","",""};
lookup1("2024") ->	{"0","<compat> 002E",""};
lookup1("2025") ->	{"0","<compat> 002E 002E",""};
lookup1("2026") ->	{"0","<compat> 002E 002E 002E",""};
lookup1("2027") ->	{"0","",""};
lookup1("2028") ->	{"0","",""};
lookup1("2029") ->	{"0","",""};
lookup1("202A") ->	{"0","",""};
lookup1("202B") ->	{"0","",""};
lookup1("202C") ->	{"0","",""};
lookup1("202D") ->	{"0","",""};
lookup1("202E") ->	{"0","",""};
lookup1("202F") ->	{"0","<noBreak> 0020",""};
lookup1("2030") ->	{"0","",""};
lookup1("2031") ->	{"0","",""};
lookup1("2032") ->	{"0","",""};
lookup1("2033") ->	{"0","<compat> 2032 2032",""};
lookup1("2034") ->	{"0","<compat> 2032 2032 2032",""};
lookup1("2035") ->	{"0","",""};
lookup1("2036") ->	{"0","<compat> 2035 2035",""};
lookup1("2037") ->	{"0","<compat> 2035 2035 2035",""};
lookup1("2038") ->	{"0","",""};
lookup1("2039") ->	{"0","",""};
lookup1("203A") ->	{"0","",""};
lookup1("203B") ->	{"0","",""};
lookup1("203C") ->	{"0","<compat> 0021 0021",""};
lookup1("203D") ->	{"0","",""};
lookup1("203E") ->	{"0","<compat> 0020 0305",""};
lookup1("203F") ->	{"0","",""};
lookup1("2040") ->	{"0","",""};
lookup1("2041") ->	{"0","",""};
lookup1("2042") ->	{"0","",""};
lookup1("2043") ->	{"0","",""};
lookup1("2044") ->	{"0","",""};
lookup1("2045") ->	{"0","",""};
lookup1("2046") ->	{"0","",""};
lookup1("2047") ->	{"0","<compat> 003F 003F",""};
lookup1("2048") ->	{"0","<compat> 003F 0021",""};
lookup1("2049") ->	{"0","<compat> 0021 003F",""};
lookup1("204A") ->	{"0","",""};
lookup1("204B") ->	{"0","",""};
lookup1("204C") ->	{"0","",""};
lookup1("204D") ->	{"0","",""};
lookup1("204E") ->	{"0","",""};
lookup1("204F") ->	{"0","",""};
lookup1("2050") ->	{"0","",""};
lookup1("2051") ->	{"0","",""};
lookup1("2052") ->	{"0","",""};
lookup1("2053") ->	{"0","",""};
lookup1("2054") ->	{"0","",""};
lookup1("2055") ->	{"0","",""};
lookup1("2056") ->	{"0","",""};
lookup1("2057") ->	{"0","<compat> 2032 2032 2032 2032",""};
lookup1("2058") ->	{"0","",""};
lookup1("2059") ->	{"0","",""};
lookup1("205A") ->	{"0","",""};
lookup1("205B") ->	{"0","",""};
lookup1("205C") ->	{"0","",""};
lookup1("205D") ->	{"0","",""};
lookup1("205E") ->	{"0","",""};
lookup1("205F") ->	{"0","<compat> 0020",""};
lookup1("2060") ->	{"0","",""};
lookup1("2061") ->	{"0","",""};
lookup1("2062") ->	{"0","",""};
lookup1("2063") ->	{"0","",""};
lookup1("2064") ->	{"0","",""};
lookup1("2066") ->	{"0","",""};
lookup1("2067") ->	{"0","",""};
lookup1("2068") ->	{"0","",""};
lookup1("2069") ->	{"0","",""};
lookup1("206A") ->	{"0","",""};
lookup1("206B") ->	{"0","",""};
lookup1("206C") ->	{"0","",""};
lookup1("206D") ->	{"0","",""};
lookup1("206E") ->	{"0","",""};
lookup1("206F") ->	{"0","",""};
lookup1("2070") ->	{"0","<super> 0030",""};
lookup1("2071") ->	{"0","<super> 0069",""};
lookup1("2074") ->	{"0","<super> 0034",""};
lookup1("2075") ->	{"0","<super> 0035",""};
lookup1("2076") ->	{"0","<super> 0036",""};
lookup1("2077") ->	{"0","<super> 0037",""};
lookup1("2078") ->	{"0","<super> 0038",""};
lookup1("2079") ->	{"0","<super> 0039",""};
lookup1("207A") ->	{"0","<super> 002B",""};
lookup1("207B") ->	{"0","<super> 2212",""};
lookup1("207C") ->	{"0","<super> 003D",""};
lookup1("207D") ->	{"0","<super> 0028",""};
lookup1("207E") ->	{"0","<super> 0029",""};
lookup1("207F") ->	{"0","<super> 006E",""};
lookup1("2080") ->	{"0","<sub> 0030",""};
lookup1("2081") ->	{"0","<sub> 0031",""};
lookup1("2082") ->	{"0","<sub> 0032",""};
lookup1("2083") ->	{"0","<sub> 0033",""};
lookup1("2084") ->	{"0","<sub> 0034",""};
lookup1("2085") ->	{"0","<sub> 0035",""};
lookup1("2086") ->	{"0","<sub> 0036",""};
lookup1("2087") ->	{"0","<sub> 0037",""};
lookup1("2088") ->	{"0","<sub> 0038",""};
lookup1("2089") ->	{"0","<sub> 0039",""};
lookup1("208A") ->	{"0","<sub> 002B",""};
lookup1("208B") ->	{"0","<sub> 2212",""};
lookup1("208C") ->	{"0","<sub> 003D",""};
lookup1("208D") ->	{"0","<sub> 0028",""};
lookup1("208E") ->	{"0","<sub> 0029",""};
lookup1("2090") ->	{"0","<sub> 0061",""};
lookup1("2091") ->	{"0","<sub> 0065",""};
lookup1("2092") ->	{"0","<sub> 006F",""};
lookup1("2093") ->	{"0","<sub> 0078",""};
lookup1("2094") ->	{"0","<sub> 0259",""};
lookup1("2095") ->	{"0","<sub> 0068",""};
lookup1("2096") ->	{"0","<sub> 006B",""};
lookup1("2097") ->	{"0","<sub> 006C",""};
lookup1("2098") ->	{"0","<sub> 006D",""};
lookup1("2099") ->	{"0","<sub> 006E",""};
lookup1("209A") ->	{"0","<sub> 0070",""};
lookup1("209B") ->	{"0","<sub> 0073",""};
lookup1("209C") ->	{"0","<sub> 0074",""};
lookup1("20A0") ->	{"0","",""};
lookup1("20A1") ->	{"0","",""};
lookup1("20A2") ->	{"0","",""};
lookup1("20A3") ->	{"0","",""};
lookup1("20A4") ->	{"0","",""};
lookup1("20A5") ->	{"0","",""};
lookup1("20A6") ->	{"0","",""};
lookup1("20A7") ->	{"0","",""};
lookup1("20A8") ->	{"0","<compat> 0052 0073",""};
lookup1("20A9") ->	{"0","",""};
lookup1("20AA") ->	{"0","",""};
lookup1("20AB") ->	{"0","",""};
lookup1("20AC") ->	{"0","",""};
lookup1("20AD") ->	{"0","",""};
lookup1("20AE") ->	{"0","",""};
lookup1("20AF") ->	{"0","",""};
lookup1("20B0") ->	{"0","",""};
lookup1("20B1") ->	{"0","",""};
lookup1("20B2") ->	{"0","",""};
lookup1("20B3") ->	{"0","",""};
lookup1("20B4") ->	{"0","",""};
lookup1("20B5") ->	{"0","",""};
lookup1("20B6") ->	{"0","",""};
lookup1("20B7") ->	{"0","",""};
lookup1("20B8") ->	{"0","",""};
lookup1("20B9") ->	{"0","",""};
lookup1("20BA") ->	{"0","",""};
lookup1("20BB") ->	{"0","",""};
lookup1("20BC") ->	{"0","",""};
lookup1("20BD") ->	{"0","",""};
lookup1("20D0") ->	{"230","",""};
lookup1("20D1") ->	{"230","",""};
lookup1("20D2") ->	{"1","",""};
lookup1("20D3") ->	{"1","",""};
lookup1("20D4") ->	{"230","",""};
lookup1("20D5") ->	{"230","",""};
lookup1("20D6") ->	{"230","",""};
lookup1("20D7") ->	{"230","",""};
lookup1("20D8") ->	{"1","",""};
lookup1("20D9") ->	{"1","",""};
lookup1("20DA") ->	{"1","",""};
lookup1("20DB") ->	{"230","",""};
lookup1("20DC") ->	{"230","",""};
lookup1("20DD") ->	{"0","",""};
lookup1("20DE") ->	{"0","",""};
lookup1("20DF") ->	{"0","",""};
lookup1("20E0") ->	{"0","",""};
lookup1("20E1") ->	{"230","",""};
lookup1("20E2") ->	{"0","",""};
lookup1("20E3") ->	{"0","",""};
lookup1("20E4") ->	{"0","",""};
lookup1("20E5") ->	{"1","",""};
lookup1("20E6") ->	{"1","",""};
lookup1("20E7") ->	{"230","",""};
lookup1("20E8") ->	{"220","",""};
lookup1("20E9") ->	{"230","",""};
lookup1("20EA") ->	{"1","",""};
lookup1("20EB") ->	{"1","",""};
lookup1("20EC") ->	{"220","",""};
lookup1("20ED") ->	{"220","",""};
lookup1("20EE") ->	{"220","",""};
lookup1("20EF") ->	{"220","",""};
lookup1("20F0") ->	{"230","",""};
lookup1("2100") ->	{"0","<compat> 0061 002F 0063",""};
lookup1("2101") ->	{"0","<compat> 0061 002F 0073",""};
lookup1("2102") ->	{"0","<font> 0043",""};
lookup1("2103") ->	{"0","<compat> 00B0 0043",""};
lookup1("2104") ->	{"0","",""};
lookup1("2105") ->	{"0","<compat> 0063 002F 006F",""};
lookup1("2106") ->	{"0","<compat> 0063 002F 0075",""};
lookup1("2107") ->	{"0","<compat> 0190",""};
lookup1("2108") ->	{"0","",""};
lookup1("2109") ->	{"0","<compat> 00B0 0046",""};
lookup1("210A") ->	{"0","<font> 0067",""};
lookup1("210B") ->	{"0","<font> 0048",""};
lookup1("210C") ->	{"0","<font> 0048",""};
lookup1("210D") ->	{"0","<font> 0048",""};
lookup1("210E") ->	{"0","<font> 0068",""};
lookup1("210F") ->	{"0","<font> 0127",""};
lookup1("2110") ->	{"0","<font> 0049",""};
lookup1("2111") ->	{"0","<font> 0049",""};
lookup1("2112") ->	{"0","<font> 004C",""};
lookup1("2113") ->	{"0","<font> 006C",""};
lookup1("2114") ->	{"0","",""};
lookup1("2115") ->	{"0","<font> 004E",""};
lookup1("2116") ->	{"0","<compat> 004E 006F",""};
lookup1("2117") ->	{"0","",""};
lookup1("2118") ->	{"0","",""};
lookup1("2119") ->	{"0","<font> 0050",""};
lookup1("211A") ->	{"0","<font> 0051",""};
lookup1("211B") ->	{"0","<font> 0052",""};
lookup1("211C") ->	{"0","<font> 0052",""};
lookup1("211D") ->	{"0","<font> 0052",""};
lookup1("211E") ->	{"0","",""};
lookup1("211F") ->	{"0","",""};
lookup1("2120") ->	{"0","<super> 0053 004D",""};
lookup1("2121") ->	{"0","<compat> 0054 0045 004C",""};
lookup1("2122") ->	{"0","<super> 0054 004D",""};
lookup1("2123") ->	{"0","",""};
lookup1("2124") ->	{"0","<font> 005A",""};
lookup1("2125") ->	{"0","",""};
lookup1("2126") ->	{"0","03A9","03C9"};
lookup1("2127") ->	{"0","",""};
lookup1("2128") ->	{"0","<font> 005A",""};
lookup1("2129") ->	{"0","",""};
lookup1("212A") ->	{"0","004B","006B"};
lookup1("212B") ->	{"0","00C5","00E5"};
lookup1("212C") ->	{"0","<font> 0042",""};
lookup1("212D") ->	{"0","<font> 0043",""};
lookup1("212E") ->	{"0","",""};
lookup1("212F") ->	{"0","<font> 0065",""};
lookup1("2130") ->	{"0","<font> 0045",""};
lookup1("2131") ->	{"0","<font> 0046",""};
lookup1("2132") ->	{"0","","214E"};
lookup1("2133") ->	{"0","<font> 004D",""};
lookup1("2134") ->	{"0","<font> 006F",""};
lookup1("2135") ->	{"0","<compat> 05D0",""};
lookup1("2136") ->	{"0","<compat> 05D1",""};
lookup1("2137") ->	{"0","<compat> 05D2",""};
lookup1("2138") ->	{"0","<compat> 05D3",""};
lookup1("2139") ->	{"0","<font> 0069",""};
lookup1("213A") ->	{"0","",""};
lookup1("213B") ->	{"0","<compat> 0046 0041 0058",""};
lookup1("213C") ->	{"0","<font> 03C0",""};
lookup1("213D") ->	{"0","<font> 03B3",""};
lookup1("213E") ->	{"0","<font> 0393",""};
lookup1("213F") ->	{"0","<font> 03A0",""};
lookup1("2140") ->	{"0","<font> 2211",""};
lookup1("2141") ->	{"0","",""};
lookup1("2142") ->	{"0","",""};
lookup1("2143") ->	{"0","",""};
lookup1("2144") ->	{"0","",""};
lookup1("2145") ->	{"0","<font> 0044",""};
lookup1("2146") ->	{"0","<font> 0064",""};
lookup1("2147") ->	{"0","<font> 0065",""};
lookup1("2148") ->	{"0","<font> 0069",""};
lookup1("2149") ->	{"0","<font> 006A",""};
lookup1("214A") ->	{"0","",""};
lookup1("214B") ->	{"0","",""};
lookup1("214C") ->	{"0","",""};
lookup1("214D") ->	{"0","",""};
lookup1("214E") ->	{"0","",""};
lookup1("214F") ->	{"0","",""};
lookup1("2150") ->	{"0","<fraction> 0031 2044 0037",""};
lookup1("2151") ->	{"0","<fraction> 0031 2044 0039",""};
lookup1("2152") ->	{"0","<fraction> 0031 2044 0031 0030",""};
lookup1("2153") ->	{"0","<fraction> 0031 2044 0033",""};
lookup1("2154") ->	{"0","<fraction> 0032 2044 0033",""};
lookup1("2155") ->	{"0","<fraction> 0031 2044 0035",""};
lookup1("2156") ->	{"0","<fraction> 0032 2044 0035",""};
lookup1("2157") ->	{"0","<fraction> 0033 2044 0035",""};
lookup1("2158") ->	{"0","<fraction> 0034 2044 0035",""};
lookup1("2159") ->	{"0","<fraction> 0031 2044 0036",""};
lookup1("215A") ->	{"0","<fraction> 0035 2044 0036",""};
lookup1("215B") ->	{"0","<fraction> 0031 2044 0038",""};
lookup1("215C") ->	{"0","<fraction> 0033 2044 0038",""};
lookup1("215D") ->	{"0","<fraction> 0035 2044 0038",""};
lookup1("215E") ->	{"0","<fraction> 0037 2044 0038",""};
lookup1("215F") ->	{"0","<fraction> 0031 2044",""};
lookup1("2160") ->	{"0","<compat> 0049","2170"};
lookup1("2161") ->	{"0","<compat> 0049 0049","2171"};
lookup1("2162") ->	{"0","<compat> 0049 0049 0049","2172"};
lookup1("2163") ->	{"0","<compat> 0049 0056","2173"};
lookup1("2164") ->	{"0","<compat> 0056","2174"};
lookup1("2165") ->	{"0","<compat> 0056 0049","2175"};
lookup1("2166") ->	{"0","<compat> 0056 0049 0049","2176"};
lookup1("2167") ->	{"0","<compat> 0056 0049 0049 0049","2177"};
lookup1("2168") ->	{"0","<compat> 0049 0058","2178"};
lookup1("2169") ->	{"0","<compat> 0058","2179"};
lookup1("216A") ->	{"0","<compat> 0058 0049","217A"};
lookup1("216B") ->	{"0","<compat> 0058 0049 0049","217B"};
lookup1("216C") ->	{"0","<compat> 004C","217C"};
lookup1("216D") ->	{"0","<compat> 0043","217D"};
lookup1("216E") ->	{"0","<compat> 0044","217E"};
lookup1("216F") ->	{"0","<compat> 004D","217F"};
lookup1("2170") ->	{"0","<compat> 0069",""};
lookup1("2171") ->	{"0","<compat> 0069 0069",""};
lookup1("2172") ->	{"0","<compat> 0069 0069 0069",""};
lookup1("2173") ->	{"0","<compat> 0069 0076",""};
lookup1("2174") ->	{"0","<compat> 0076",""};
lookup1("2175") ->	{"0","<compat> 0076 0069",""};
lookup1("2176") ->	{"0","<compat> 0076 0069 0069",""};
lookup1("2177") ->	{"0","<compat> 0076 0069 0069 0069",""};
lookup1("2178") ->	{"0","<compat> 0069 0078",""};
lookup1("2179") ->	{"0","<compat> 0078",""};
lookup1("217A") ->	{"0","<compat> 0078 0069",""};
lookup1("217B") ->	{"0","<compat> 0078 0069 0069",""};
lookup1("217C") ->	{"0","<compat> 006C",""};
lookup1("217D") ->	{"0","<compat> 0063",""};
lookup1("217E") ->	{"0","<compat> 0064",""};
lookup1("217F") ->	{"0","<compat> 006D",""};
lookup1("2180") ->	{"0","",""};
lookup1("2181") ->	{"0","",""};
lookup1("2182") ->	{"0","",""};
lookup1("2183") ->	{"0","","2184"};
lookup1("2184") ->	{"0","",""};
lookup1("2185") ->	{"0","",""};
lookup1("2186") ->	{"0","",""};
lookup1("2187") ->	{"0","",""};
lookup1("2188") ->	{"0","",""};
lookup1("2189") ->	{"0","<fraction> 0030 2044 0033",""};
lookup1("2190") ->	{"0","",""};
lookup1("2191") ->	{"0","",""};
lookup1("2192") ->	{"0","",""};
lookup1("2193") ->	{"0","",""};
lookup1("2194") ->	{"0","",""};
lookup1("2195") ->	{"0","",""};
lookup1("2196") ->	{"0","",""};
lookup1("2197") ->	{"0","",""};
lookup1("2198") ->	{"0","",""};
lookup1("2199") ->	{"0","",""};
lookup1("219A") ->	{"0","2190 0338",""};
lookup1("219B") ->	{"0","2192 0338",""};
lookup1("219C") ->	{"0","",""};
lookup1("219D") ->	{"0","",""};
lookup1("219E") ->	{"0","",""};
lookup1("219F") ->	{"0","",""};
lookup1("21A0") ->	{"0","",""};
lookup1("21A1") ->	{"0","",""};
lookup1("21A2") ->	{"0","",""};
lookup1("21A3") ->	{"0","",""};
lookup1("21A4") ->	{"0","",""};
lookup1("21A5") ->	{"0","",""};
lookup1("21A6") ->	{"0","",""};
lookup1("21A7") ->	{"0","",""};
lookup1("21A8") ->	{"0","",""};
lookup1("21A9") ->	{"0","",""};
lookup1("21AA") ->	{"0","",""};
lookup1("21AB") ->	{"0","",""};
lookup1("21AC") ->	{"0","",""};
lookup1("21AD") ->	{"0","",""};
lookup1("21AE") ->	{"0","2194 0338",""};
lookup1("21AF") ->	{"0","",""};
lookup1("21B0") ->	{"0","",""};
lookup1("21B1") ->	{"0","",""};
lookup1("21B2") ->	{"0","",""};
lookup1("21B3") ->	{"0","",""};
lookup1("21B4") ->	{"0","",""};
lookup1("21B5") ->	{"0","",""};
lookup1("21B6") ->	{"0","",""};
lookup1("21B7") ->	{"0","",""};
lookup1("21B8") ->	{"0","",""};
lookup1("21B9") ->	{"0","",""};
lookup1("21BA") ->	{"0","",""};
lookup1("21BB") ->	{"0","",""};
lookup1("21BC") ->	{"0","",""};
lookup1("21BD") ->	{"0","",""};
lookup1("21BE") ->	{"0","",""};
lookup1("21BF") ->	{"0","",""};
lookup1("21C0") ->	{"0","",""};
lookup1("21C1") ->	{"0","",""};
lookup1("21C2") ->	{"0","",""};
lookup1("21C3") ->	{"0","",""};
lookup1("21C4") ->	{"0","",""};
lookup1("21C5") ->	{"0","",""};
lookup1("21C6") ->	{"0","",""};
lookup1("21C7") ->	{"0","",""};
lookup1("21C8") ->	{"0","",""};
lookup1("21C9") ->	{"0","",""};
lookup1("21CA") ->	{"0","",""};
lookup1("21CB") ->	{"0","",""};
lookup1("21CC") ->	{"0","",""};
lookup1("21CD") ->	{"0","21D0 0338",""};
lookup1("21CE") ->	{"0","21D4 0338",""};
lookup1("21CF") ->	{"0","21D2 0338",""};
lookup1("21D0") ->	{"0","",""};
lookup1("21D1") ->	{"0","",""};
lookup1("21D2") ->	{"0","",""};
lookup1("21D3") ->	{"0","",""};
lookup1("21D4") ->	{"0","",""};
lookup1("21D5") ->	{"0","",""};
lookup1("21D6") ->	{"0","",""};
lookup1("21D7") ->	{"0","",""};
lookup1("21D8") ->	{"0","",""};
lookup1("21D9") ->	{"0","",""};
lookup1("21DA") ->	{"0","",""};
lookup1("21DB") ->	{"0","",""};
lookup1("21DC") ->	{"0","",""};
lookup1("21DD") ->	{"0","",""};
lookup1("21DE") ->	{"0","",""};
lookup1("21DF") ->	{"0","",""};
lookup1("21E0") ->	{"0","",""};
lookup1("21E1") ->	{"0","",""};
lookup1("21E2") ->	{"0","",""};
lookup1("21E3") ->	{"0","",""};
lookup1("21E4") ->	{"0","",""};
lookup1("21E5") ->	{"0","",""};
lookup1("21E6") ->	{"0","",""};
lookup1("21E7") ->	{"0","",""};
lookup1("21E8") ->	{"0","",""};
lookup1("21E9") ->	{"0","",""};
lookup1("21EA") ->	{"0","",""};
lookup1("21EB") ->	{"0","",""};
lookup1("21EC") ->	{"0","",""};
lookup1("21ED") ->	{"0","",""};
lookup1("21EE") ->	{"0","",""};
lookup1("21EF") ->	{"0","",""};
lookup1("21F0") ->	{"0","",""};
lookup1("21F1") ->	{"0","",""};
lookup1("21F2") ->	{"0","",""};
lookup1("21F3") ->	{"0","",""};
lookup1("21F4") ->	{"0","",""};
lookup1("21F5") ->	{"0","",""};
lookup1("21F6") ->	{"0","",""};
lookup1("21F7") ->	{"0","",""};
lookup1("21F8") ->	{"0","",""};
lookup1("21F9") ->	{"0","",""};
lookup1("21FA") ->	{"0","",""};
lookup1("21FB") ->	{"0","",""};
lookup1("21FC") ->	{"0","",""};
lookup1("21FD") ->	{"0","",""};
lookup1("21FE") ->	{"0","",""};
lookup1("21FF") ->	{"0","",""};
lookup1("2200") ->	{"0","",""};
lookup1("2201") ->	{"0","",""};
lookup1("2202") ->	{"0","",""};
lookup1("2203") ->	{"0","",""};
lookup1("2204") ->	{"0","2203 0338",""};
lookup1("2205") ->	{"0","",""};
lookup1("2206") ->	{"0","",""};
lookup1("2207") ->	{"0","",""};
lookup1("2208") ->	{"0","",""};
lookup1("2209") ->	{"0","2208 0338",""};
lookup1("220A") ->	{"0","",""};
lookup1("220B") ->	{"0","",""};
lookup1("220C") ->	{"0","220B 0338",""};
lookup1("220D") ->	{"0","",""};
lookup1("220E") ->	{"0","",""};
lookup1("220F") ->	{"0","",""};
lookup1("2210") ->	{"0","",""};
lookup1("2211") ->	{"0","",""};
lookup1("2212") ->	{"0","",""};
lookup1("2213") ->	{"0","",""};
lookup1("2214") ->	{"0","",""};
lookup1("2215") ->	{"0","",""};
lookup1("2216") ->	{"0","",""};
lookup1("2217") ->	{"0","",""};
lookup1("2218") ->	{"0","",""};
lookup1("2219") ->	{"0","",""};
lookup1("221A") ->	{"0","",""};
lookup1("221B") ->	{"0","",""};
lookup1("221C") ->	{"0","",""};
lookup1("221D") ->	{"0","",""};
lookup1("221E") ->	{"0","",""};
lookup1("221F") ->	{"0","",""};
lookup1("2220") ->	{"0","",""};
lookup1("2221") ->	{"0","",""};
lookup1("2222") ->	{"0","",""};
lookup1("2223") ->	{"0","",""};
lookup1("2224") ->	{"0","2223 0338",""};
lookup1("2225") ->	{"0","",""};
lookup1("2226") ->	{"0","2225 0338",""};
lookup1("2227") ->	{"0","",""};
lookup1("2228") ->	{"0","",""};
lookup1("2229") ->	{"0","",""};
lookup1("222A") ->	{"0","",""};
lookup1("222B") ->	{"0","",""};
lookup1("222C") ->	{"0","<compat> 222B 222B",""};
lookup1("222D") ->	{"0","<compat> 222B 222B 222B",""};
lookup1("222E") ->	{"0","",""};
lookup1("222F") ->	{"0","<compat> 222E 222E",""};
lookup1("2230") ->	{"0","<compat> 222E 222E 222E",""};
lookup1("2231") ->	{"0","",""};
lookup1("2232") ->	{"0","",""};
lookup1("2233") ->	{"0","",""};
lookup1("2234") ->	{"0","",""};
lookup1("2235") ->	{"0","",""};
lookup1("2236") ->	{"0","",""};
lookup1("2237") ->	{"0","",""};
lookup1("2238") ->	{"0","",""};
lookup1("2239") ->	{"0","",""};
lookup1("223A") ->	{"0","",""};
lookup1("223B") ->	{"0","",""};
lookup1("223C") ->	{"0","",""};
lookup1("223D") ->	{"0","",""};
lookup1("223E") ->	{"0","",""};
lookup1("223F") ->	{"0","",""};
lookup1("2240") ->	{"0","",""};
lookup1("2241") ->	{"0","223C 0338",""};
lookup1("2242") ->	{"0","",""};
lookup1("2243") ->	{"0","",""};
lookup1("2244") ->	{"0","2243 0338",""};
lookup1("2245") ->	{"0","",""};
lookup1("2246") ->	{"0","",""};
lookup1("2247") ->	{"0","2245 0338",""};
lookup1("2248") ->	{"0","",""};
lookup1("2249") ->	{"0","2248 0338",""};
lookup1("224A") ->	{"0","",""};
lookup1("224B") ->	{"0","",""};
lookup1("224C") ->	{"0","",""};
lookup1("224D") ->	{"0","",""};
lookup1("224E") ->	{"0","",""};
lookup1("224F") ->	{"0","",""};
lookup1("2250") ->	{"0","",""};
lookup1("2251") ->	{"0","",""};
lookup1("2252") ->	{"0","",""};
lookup1("2253") ->	{"0","",""};
lookup1("2254") ->	{"0","",""};
lookup1("2255") ->	{"0","",""};
lookup1("2256") ->	{"0","",""};
lookup1("2257") ->	{"0","",""};
lookup1("2258") ->	{"0","",""};
lookup1("2259") ->	{"0","",""};
lookup1("225A") ->	{"0","",""};
lookup1("225B") ->	{"0","",""};
lookup1("225C") ->	{"0","",""};
lookup1("225D") ->	{"0","",""};
lookup1("225E") ->	{"0","",""};
lookup1("225F") ->	{"0","",""};
lookup1("2260") ->	{"0","003D 0338",""};
lookup1("2261") ->	{"0","",""};
lookup1("2262") ->	{"0","2261 0338",""};
lookup1("2263") ->	{"0","",""};
lookup1("2264") ->	{"0","",""};
lookup1("2265") ->	{"0","",""};
lookup1("2266") ->	{"0","",""};
lookup1("2267") ->	{"0","",""};
lookup1("2268") ->	{"0","",""};
lookup1("2269") ->	{"0","",""};
lookup1("226A") ->	{"0","",""};
lookup1("226B") ->	{"0","",""};
lookup1("226C") ->	{"0","",""};
lookup1("226D") ->	{"0","224D 0338",""};
lookup1("226E") ->	{"0","003C 0338",""};
lookup1("226F") ->	{"0","003E 0338",""};
lookup1("2270") ->	{"0","2264 0338",""};
lookup1("2271") ->	{"0","2265 0338",""};
lookup1("2272") ->	{"0","",""};
lookup1("2273") ->	{"0","",""};
lookup1("2274") ->	{"0","2272 0338",""};
lookup1("2275") ->	{"0","2273 0338",""};
lookup1("2276") ->	{"0","",""};
lookup1("2277") ->	{"0","",""};
lookup1("2278") ->	{"0","2276 0338",""};
lookup1("2279") ->	{"0","2277 0338",""};
lookup1("227A") ->	{"0","",""};
lookup1("227B") ->	{"0","",""};
lookup1("227C") ->	{"0","",""};
lookup1("227D") ->	{"0","",""};
lookup1("227E") ->	{"0","",""};
lookup1("227F") ->	{"0","",""};
lookup1("2280") ->	{"0","227A 0338",""};
lookup1("2281") ->	{"0","227B 0338",""};
lookup1("2282") ->	{"0","",""};
lookup1("2283") ->	{"0","",""};
lookup1("2284") ->	{"0","2282 0338",""};
lookup1("2285") ->	{"0","2283 0338",""};
lookup1("2286") ->	{"0","",""};
lookup1("2287") ->	{"0","",""};
lookup1("2288") ->	{"0","2286 0338",""};
lookup1("2289") ->	{"0","2287 0338",""};
lookup1("228A") ->	{"0","",""};
lookup1("228B") ->	{"0","",""};
lookup1("228C") ->	{"0","",""};
lookup1("228D") ->	{"0","",""};
lookup1("228E") ->	{"0","",""};
lookup1("228F") ->	{"0","",""};
lookup1("2290") ->	{"0","",""};
lookup1("2291") ->	{"0","",""};
lookup1("2292") ->	{"0","",""};
lookup1("2293") ->	{"0","",""};
lookup1("2294") ->	{"0","",""};
lookup1("2295") ->	{"0","",""};
lookup1("2296") ->	{"0","",""};
lookup1("2297") ->	{"0","",""};
lookup1("2298") ->	{"0","",""};
lookup1("2299") ->	{"0","",""};
lookup1("229A") ->	{"0","",""};
lookup1("229B") ->	{"0","",""};
lookup1("229C") ->	{"0","",""};
lookup1("229D") ->	{"0","",""};
lookup1("229E") ->	{"0","",""};
lookup1("229F") ->	{"0","",""};
lookup1("22A0") ->	{"0","",""};
lookup1("22A1") ->	{"0","",""};
lookup1("22A2") ->	{"0","",""};
lookup1("22A3") ->	{"0","",""};
lookup1("22A4") ->	{"0","",""};
lookup1("22A5") ->	{"0","",""};
lookup1("22A6") ->	{"0","",""};
lookup1("22A7") ->	{"0","",""};
lookup1("22A8") ->	{"0","",""};
lookup1("22A9") ->	{"0","",""};
lookup1("22AA") ->	{"0","",""};
lookup1("22AB") ->	{"0","",""};
lookup1("22AC") ->	{"0","22A2 0338",""};
lookup1("22AD") ->	{"0","22A8 0338",""};
lookup1("22AE") ->	{"0","22A9 0338",""};
lookup1("22AF") ->	{"0","22AB 0338",""};
lookup1("22B0") ->	{"0","",""};
lookup1("22B1") ->	{"0","",""};
lookup1("22B2") ->	{"0","",""};
lookup1("22B3") ->	{"0","",""};
lookup1("22B4") ->	{"0","",""};
lookup1("22B5") ->	{"0","",""};
lookup1("22B6") ->	{"0","",""};
lookup1("22B7") ->	{"0","",""};
lookup1("22B8") ->	{"0","",""};
lookup1("22B9") ->	{"0","",""};
lookup1("22BA") ->	{"0","",""};
lookup1("22BB") ->	{"0","",""};
lookup1("22BC") ->	{"0","",""};
lookup1("22BD") ->	{"0","",""};
lookup1("22BE") ->	{"0","",""};
lookup1("22BF") ->	{"0","",""};
lookup1("22C0") ->	{"0","",""};
lookup1("22C1") ->	{"0","",""};
lookup1("22C2") ->	{"0","",""};
lookup1("22C3") ->	{"0","",""};
lookup1("22C4") ->	{"0","",""};
lookup1("22C5") ->	{"0","",""};
lookup1("22C6") ->	{"0","",""};
lookup1("22C7") ->	{"0","",""};
lookup1("22C8") ->	{"0","",""};
lookup1("22C9") ->	{"0","",""};
lookup1("22CA") ->	{"0","",""};
lookup1("22CB") ->	{"0","",""};
lookup1("22CC") ->	{"0","",""};
lookup1("22CD") ->	{"0","",""};
lookup1("22CE") ->	{"0","",""};
lookup1("22CF") ->	{"0","",""};
lookup1("22D0") ->	{"0","",""};
lookup1("22D1") ->	{"0","",""};
lookup1("22D2") ->	{"0","",""};
lookup1("22D3") ->	{"0","",""};
lookup1("22D4") ->	{"0","",""};
lookup1("22D5") ->	{"0","",""};
lookup1("22D6") ->	{"0","",""};
lookup1("22D7") ->	{"0","",""};
lookup1("22D8") ->	{"0","",""};
lookup1("22D9") ->	{"0","",""};
lookup1("22DA") ->	{"0","",""};
lookup1("22DB") ->	{"0","",""};
lookup1("22DC") ->	{"0","",""};
lookup1("22DD") ->	{"0","",""};
lookup1("22DE") ->	{"0","",""};
lookup1("22DF") ->	{"0","",""};
lookup1("22E0") ->	{"0","227C 0338",""};
lookup1("22E1") ->	{"0","227D 0338",""};
lookup1("22E2") ->	{"0","2291 0338",""};
lookup1("22E3") ->	{"0","2292 0338",""};
lookup1("22E4") ->	{"0","",""};
lookup1("22E5") ->	{"0","",""};
lookup1("22E6") ->	{"0","",""};
lookup1("22E7") ->	{"0","",""};
lookup1("22E8") ->	{"0","",""};
lookup1("22E9") ->	{"0","",""};
lookup1("22EA") ->	{"0","22B2 0338",""};
lookup1("22EB") ->	{"0","22B3 0338",""};
lookup1("22EC") ->	{"0","22B4 0338",""};
lookup1("22ED") ->	{"0","22B5 0338",""};
lookup1("22EE") ->	{"0","",""};
lookup1("22EF") ->	{"0","",""};
lookup1("22F0") ->	{"0","",""};
lookup1("22F1") ->	{"0","",""};
lookup1("22F2") ->	{"0","",""};
lookup1("22F3") ->	{"0","",""};
lookup1("22F4") ->	{"0","",""};
lookup1("22F5") ->	{"0","",""};
lookup1("22F6") ->	{"0","",""};
lookup1("22F7") ->	{"0","",""};
lookup1("22F8") ->	{"0","",""};
lookup1("22F9") ->	{"0","",""};
lookup1("22FA") ->	{"0","",""};
lookup1("22FB") ->	{"0","",""};
lookup1("22FC") ->	{"0","",""};
lookup1("22FD") ->	{"0","",""};
lookup1("22FE") ->	{"0","",""};
lookup1("22FF") ->	{"0","",""};
lookup1("2300") ->	{"0","",""};
lookup1("2301") ->	{"0","",""};
lookup1("2302") ->	{"0","",""};
lookup1("2303") ->	{"0","",""};
lookup1("2304") ->	{"0","",""};
lookup1("2305") ->	{"0","",""};
lookup1("2306") ->	{"0","",""};
lookup1("2307") ->	{"0","",""};
lookup1("2308") ->	{"0","",""};
lookup1("2309") ->	{"0","",""};
lookup1("230A") ->	{"0","",""};
lookup1("230B") ->	{"0","",""};
lookup1("230C") ->	{"0","",""};
lookup1("230D") ->	{"0","",""};
lookup1("230E") ->	{"0","",""};
lookup1("230F") ->	{"0","",""};
lookup1("2310") ->	{"0","",""};
lookup1("2311") ->	{"0","",""};
lookup1("2312") ->	{"0","",""};
lookup1("2313") ->	{"0","",""};
lookup1("2314") ->	{"0","",""};
lookup1("2315") ->	{"0","",""};
lookup1("2316") ->	{"0","",""};
lookup1("2317") ->	{"0","",""};
lookup1("2318") ->	{"0","",""};
lookup1("2319") ->	{"0","",""};
lookup1("231A") ->	{"0","",""};
lookup1("231B") ->	{"0","",""};
lookup1("231C") ->	{"0","",""};
lookup1("231D") ->	{"0","",""};
lookup1("231E") ->	{"0","",""};
lookup1("231F") ->	{"0","",""};
lookup1("2320") ->	{"0","",""};
lookup1("2321") ->	{"0","",""};
lookup1("2322") ->	{"0","",""};
lookup1("2323") ->	{"0","",""};
lookup1("2324") ->	{"0","",""};
lookup1("2325") ->	{"0","",""};
lookup1("2326") ->	{"0","",""};
lookup1("2327") ->	{"0","",""};
lookup1("2328") ->	{"0","",""};
lookup1("2329") ->	{"0","3008",""};
lookup1("232A") ->	{"0","3009",""};
lookup1("232B") ->	{"0","",""};
lookup1("232C") ->	{"0","",""};
lookup1("232D") ->	{"0","",""};
lookup1("232E") ->	{"0","",""};
lookup1("232F") ->	{"0","",""};
lookup1("2330") ->	{"0","",""};
lookup1("2331") ->	{"0","",""};
lookup1("2332") ->	{"0","",""};
lookup1("2333") ->	{"0","",""};
lookup1("2334") ->	{"0","",""};
lookup1("2335") ->	{"0","",""};
lookup1("2336") ->	{"0","",""};
lookup1("2337") ->	{"0","",""};
lookup1("2338") ->	{"0","",""};
lookup1("2339") ->	{"0","",""};
lookup1("233A") ->	{"0","",""};
lookup1("233B") ->	{"0","",""};
lookup1("233C") ->	{"0","",""};
lookup1("233D") ->	{"0","",""};
lookup1("233E") ->	{"0","",""};
lookup1("233F") ->	{"0","",""};
lookup1("2340") ->	{"0","",""};
lookup1("2341") ->	{"0","",""};
lookup1("2342") ->	{"0","",""};
lookup1("2343") ->	{"0","",""};
lookup1("2344") ->	{"0","",""};
lookup1("2345") ->	{"0","",""};
lookup1("2346") ->	{"0","",""};
lookup1("2347") ->	{"0","",""};
lookup1("2348") ->	{"0","",""};
lookup1("2349") ->	{"0","",""};
lookup1("234A") ->	{"0","",""};
lookup1("234B") ->	{"0","",""};
lookup1("234C") ->	{"0","",""};
lookup1("234D") ->	{"0","",""};
lookup1("234E") ->	{"0","",""};
lookup1("234F") ->	{"0","",""};
lookup1("2350") ->	{"0","",""};
lookup1("2351") ->	{"0","",""};
lookup1("2352") ->	{"0","",""};
lookup1("2353") ->	{"0","",""};
lookup1("2354") ->	{"0","",""};
lookup1("2355") ->	{"0","",""};
lookup1("2356") ->	{"0","",""};
lookup1("2357") ->	{"0","",""};
lookup1("2358") ->	{"0","",""};
lookup1("2359") ->	{"0","",""};
lookup1("235A") ->	{"0","",""};
lookup1("235B") ->	{"0","",""};
lookup1("235C") ->	{"0","",""};
lookup1("235D") ->	{"0","",""};
lookup1("235E") ->	{"0","",""};
lookup1("235F") ->	{"0","",""};
lookup1("2360") ->	{"0","",""};
lookup1("2361") ->	{"0","",""};
lookup1("2362") ->	{"0","",""};
lookup1("2363") ->	{"0","",""};
lookup1("2364") ->	{"0","",""};
lookup1("2365") ->	{"0","",""};
lookup1("2366") ->	{"0","",""};
lookup1("2367") ->	{"0","",""};
lookup1("2368") ->	{"0","",""};
lookup1("2369") ->	{"0","",""};
lookup1("236A") ->	{"0","",""};
lookup1("236B") ->	{"0","",""};
lookup1("236C") ->	{"0","",""};
lookup1("236D") ->	{"0","",""};
lookup1("236E") ->	{"0","",""};
lookup1("236F") ->	{"0","",""};
lookup1("2370") ->	{"0","",""};
lookup1("2371") ->	{"0","",""};
lookup1("2372") ->	{"0","",""};
lookup1("2373") ->	{"0","",""};
lookup1("2374") ->	{"0","",""};
lookup1("2375") ->	{"0","",""};
lookup1("2376") ->	{"0","",""};
lookup1("2377") ->	{"0","",""};
lookup1("2378") ->	{"0","",""};
lookup1("2379") ->	{"0","",""};
lookup1("237A") ->	{"0","",""};
lookup1("237B") ->	{"0","",""};
lookup1("237C") ->	{"0","",""};
lookup1("237D") ->	{"0","",""};
lookup1("237E") ->	{"0","",""};
lookup1("237F") ->	{"0","",""};
lookup1("2380") ->	{"0","",""};
lookup1("2381") ->	{"0","",""};
lookup1("2382") ->	{"0","",""};
lookup1("2383") ->	{"0","",""};
lookup1("2384") ->	{"0","",""};
lookup1("2385") ->	{"0","",""};
lookup1("2386") ->	{"0","",""};
lookup1("2387") ->	{"0","",""};
lookup1("2388") ->	{"0","",""};
lookup1("2389") ->	{"0","",""};
lookup1("238A") ->	{"0","",""};
lookup1("238B") ->	{"0","",""};
lookup1("238C") ->	{"0","",""};
lookup1("238D") ->	{"0","",""};
lookup1("238E") ->	{"0","",""};
lookup1("238F") ->	{"0","",""};
lookup1("2390") ->	{"0","",""};
lookup1("2391") ->	{"0","",""};
lookup1("2392") ->	{"0","",""};
lookup1("2393") ->	{"0","",""};
lookup1("2394") ->	{"0","",""};
lookup1("2395") ->	{"0","",""};
lookup1("2396") ->	{"0","",""};
lookup1("2397") ->	{"0","",""};
lookup1("2398") ->	{"0","",""};
lookup1("2399") ->	{"0","",""};
lookup1("239A") ->	{"0","",""};
lookup1("239B") ->	{"0","",""};
lookup1("239C") ->	{"0","",""};
lookup1("239D") ->	{"0","",""};
lookup1("239E") ->	{"0","",""};
lookup1("239F") ->	{"0","",""};
lookup1("23A0") ->	{"0","",""};
lookup1("23A1") ->	{"0","",""};
lookup1("23A2") ->	{"0","",""};
lookup1("23A3") ->	{"0","",""};
lookup1("23A4") ->	{"0","",""};
lookup1("23A5") ->	{"0","",""};
lookup1("23A6") ->	{"0","",""};
lookup1("23A7") ->	{"0","",""};
lookup1("23A8") ->	{"0","",""};
lookup1("23A9") ->	{"0","",""};
lookup1("23AA") ->	{"0","",""};
lookup1("23AB") ->	{"0","",""};
lookup1("23AC") ->	{"0","",""};
lookup1("23AD") ->	{"0","",""};
lookup1("23AE") ->	{"0","",""};
lookup1("23AF") ->	{"0","",""};
lookup1("23B0") ->	{"0","",""};
lookup1("23B1") ->	{"0","",""};
lookup1("23B2") ->	{"0","",""};
lookup1("23B3") ->	{"0","",""};
lookup1("23B4") ->	{"0","",""};
lookup1("23B5") ->	{"0","",""};
lookup1("23B6") ->	{"0","",""};
lookup1("23B7") ->	{"0","",""};
lookup1("23B8") ->	{"0","",""};
lookup1("23B9") ->	{"0","",""};
lookup1("23BA") ->	{"0","",""};
lookup1("23BB") ->	{"0","",""};
lookup1("23BC") ->	{"0","",""};
lookup1("23BD") ->	{"0","",""};
lookup1("23BE") ->	{"0","",""};
lookup1("23BF") ->	{"0","",""};
lookup1("23C0") ->	{"0","",""};
lookup1("23C1") ->	{"0","",""};
lookup1("23C2") ->	{"0","",""};
lookup1("23C3") ->	{"0","",""};
lookup1("23C4") ->	{"0","",""};
lookup1("23C5") ->	{"0","",""};
lookup1("23C6") ->	{"0","",""};
lookup1("23C7") ->	{"0","",""};
lookup1("23C8") ->	{"0","",""};
lookup1("23C9") ->	{"0","",""};
lookup1("23CA") ->	{"0","",""};
lookup1("23CB") ->	{"0","",""};
lookup1("23CC") ->	{"0","",""};
lookup1("23CD") ->	{"0","",""};
lookup1("23CE") ->	{"0","",""};
lookup1("23CF") ->	{"0","",""};
lookup1("23D0") ->	{"0","",""};
lookup1("23D1") ->	{"0","",""};
lookup1("23D2") ->	{"0","",""};
lookup1("23D3") ->	{"0","",""};
lookup1("23D4") ->	{"0","",""};
lookup1("23D5") ->	{"0","",""};
lookup1("23D6") ->	{"0","",""};
lookup1("23D7") ->	{"0","",""};
lookup1("23D8") ->	{"0","",""};
lookup1("23D9") ->	{"0","",""};
lookup1("23DA") ->	{"0","",""};
lookup1("23DB") ->	{"0","",""};
lookup1("23DC") ->	{"0","",""};
lookup1("23DD") ->	{"0","",""};
lookup1("23DE") ->	{"0","",""};
lookup1("23DF") ->	{"0","",""};
lookup1("23E0") ->	{"0","",""};
lookup1("23E1") ->	{"0","",""};
lookup1("23E2") ->	{"0","",""};
lookup1("23E3") ->	{"0","",""};
lookup1("23E4") ->	{"0","",""};
lookup1("23E5") ->	{"0","",""};
lookup1("23E6") ->	{"0","",""};
lookup1("23E7") ->	{"0","",""};
lookup1("23E8") ->	{"0","",""};
lookup1("23E9") ->	{"0","",""};
lookup1("23EA") ->	{"0","",""};
lookup1("23EB") ->	{"0","",""};
lookup1("23EC") ->	{"0","",""};
lookup1("23ED") ->	{"0","",""};
lookup1("23EE") ->	{"0","",""};
lookup1("23EF") ->	{"0","",""};
lookup1("23F0") ->	{"0","",""};
lookup1("23F1") ->	{"0","",""};
lookup1("23F2") ->	{"0","",""};
lookup1("23F3") ->	{"0","",""};
lookup1("23F4") ->	{"0","",""};
lookup1("23F5") ->	{"0","",""};
lookup1("23F6") ->	{"0","",""};
lookup1("23F7") ->	{"0","",""};
lookup1("23F8") ->	{"0","",""};
lookup1("23F9") ->	{"0","",""};
lookup1("23FA") ->	{"0","",""};
lookup1("2400") ->	{"0","",""};
lookup1("2401") ->	{"0","",""};
lookup1("2402") ->	{"0","",""};
lookup1("2403") ->	{"0","",""};
lookup1("2404") ->	{"0","",""};
lookup1("2405") ->	{"0","",""};
lookup1("2406") ->	{"0","",""};
lookup1("2407") ->	{"0","",""};
lookup1("2408") ->	{"0","",""};
lookup1("2409") ->	{"0","",""};
lookup1("240A") ->	{"0","",""};
lookup1("240B") ->	{"0","",""};
lookup1("240C") ->	{"0","",""};
lookup1("240D") ->	{"0","",""};
lookup1("240E") ->	{"0","",""};
lookup1("240F") ->	{"0","",""};
lookup1("2410") ->	{"0","",""};
lookup1("2411") ->	{"0","",""};
lookup1("2412") ->	{"0","",""};
lookup1("2413") ->	{"0","",""};
lookup1("2414") ->	{"0","",""};
lookup1("2415") ->	{"0","",""};
lookup1("2416") ->	{"0","",""};
lookup1("2417") ->	{"0","",""};
lookup1("2418") ->	{"0","",""};
lookup1("2419") ->	{"0","",""};
lookup1("241A") ->	{"0","",""};
lookup1("241B") ->	{"0","",""};
lookup1("241C") ->	{"0","",""};
lookup1("241D") ->	{"0","",""};
lookup1("241E") ->	{"0","",""};
lookup1("241F") ->	{"0","",""};
lookup1("2420") ->	{"0","",""};
lookup1("2421") ->	{"0","",""};
lookup1("2422") ->	{"0","",""};
lookup1("2423") ->	{"0","",""};
lookup1("2424") ->	{"0","",""};
lookup1("2425") ->	{"0","",""};
lookup1("2426") ->	{"0","",""};
lookup1("2440") ->	{"0","",""};
lookup1("2441") ->	{"0","",""};
lookup1("2442") ->	{"0","",""};
lookup1("2443") ->	{"0","",""};
lookup1("2444") ->	{"0","",""};
lookup1("2445") ->	{"0","",""};
lookup1("2446") ->	{"0","",""};
lookup1("2447") ->	{"0","",""};
lookup1("2448") ->	{"0","",""};
lookup1("2449") ->	{"0","",""};
lookup1("244A") ->	{"0","",""};
lookup1("2460") ->	{"0","<circle> 0031",""};
lookup1("2461") ->	{"0","<circle> 0032",""};
lookup1("2462") ->	{"0","<circle> 0033",""};
lookup1("2463") ->	{"0","<circle> 0034",""};
lookup1("2464") ->	{"0","<circle> 0035",""};
lookup1("2465") ->	{"0","<circle> 0036",""};
lookup1("2466") ->	{"0","<circle> 0037",""};
lookup1("2467") ->	{"0","<circle> 0038",""};
lookup1("2468") ->	{"0","<circle> 0039",""};
lookup1("2469") ->	{"0","<circle> 0031 0030",""};
lookup1("246A") ->	{"0","<circle> 0031 0031",""};
lookup1("246B") ->	{"0","<circle> 0031 0032",""};
lookup1("246C") ->	{"0","<circle> 0031 0033",""};
lookup1("246D") ->	{"0","<circle> 0031 0034",""};
lookup1("246E") ->	{"0","<circle> 0031 0035",""};
lookup1("246F") ->	{"0","<circle> 0031 0036",""};
lookup1("2470") ->	{"0","<circle> 0031 0037",""};
lookup1("2471") ->	{"0","<circle> 0031 0038",""};
lookup1("2472") ->	{"0","<circle> 0031 0039",""};
lookup1("2473") ->	{"0","<circle> 0032 0030",""};
lookup1("2474") ->	{"0","<compat> 0028 0031 0029",""};
lookup1("2475") ->	{"0","<compat> 0028 0032 0029",""};
lookup1("2476") ->	{"0","<compat> 0028 0033 0029",""};
lookup1("2477") ->	{"0","<compat> 0028 0034 0029",""};
lookup1("2478") ->	{"0","<compat> 0028 0035 0029",""};
lookup1("2479") ->	{"0","<compat> 0028 0036 0029",""};
lookup1("247A") ->	{"0","<compat> 0028 0037 0029",""};
lookup1("247B") ->	{"0","<compat> 0028 0038 0029",""};
lookup1("247C") ->	{"0","<compat> 0028 0039 0029",""};
lookup1("247D") ->	{"0","<compat> 0028 0031 0030 0029",""};
lookup1("247E") ->	{"0","<compat> 0028 0031 0031 0029",""};
lookup1("247F") ->	{"0","<compat> 0028 0031 0032 0029",""};
lookup1("2480") ->	{"0","<compat> 0028 0031 0033 0029",""};
lookup1("2481") ->	{"0","<compat> 0028 0031 0034 0029",""};
lookup1("2482") ->	{"0","<compat> 0028 0031 0035 0029",""};
lookup1("2483") ->	{"0","<compat> 0028 0031 0036 0029",""};
lookup1("2484") ->	{"0","<compat> 0028 0031 0037 0029",""};
lookup1("2485") ->	{"0","<compat> 0028 0031 0038 0029",""};
lookup1("2486") ->	{"0","<compat> 0028 0031 0039 0029",""};
lookup1("2487") ->	{"0","<compat> 0028 0032 0030 0029",""};
lookup1("2488") ->	{"0","<compat> 0031 002E",""};
lookup1("2489") ->	{"0","<compat> 0032 002E",""};
lookup1("248A") ->	{"0","<compat> 0033 002E",""};
lookup1("248B") ->	{"0","<compat> 0034 002E",""};
lookup1("248C") ->	{"0","<compat> 0035 002E",""};
lookup1("248D") ->	{"0","<compat> 0036 002E",""};
lookup1("248E") ->	{"0","<compat> 0037 002E",""};
lookup1("248F") ->	{"0","<compat> 0038 002E",""};
lookup1("2490") ->	{"0","<compat> 0039 002E",""};
lookup1("2491") ->	{"0","<compat> 0031 0030 002E",""};
lookup1("2492") ->	{"0","<compat> 0031 0031 002E",""};
lookup1("2493") ->	{"0","<compat> 0031 0032 002E",""};
lookup1("2494") ->	{"0","<compat> 0031 0033 002E",""};
lookup1("2495") ->	{"0","<compat> 0031 0034 002E",""};
lookup1("2496") ->	{"0","<compat> 0031 0035 002E",""};
lookup1("2497") ->	{"0","<compat> 0031 0036 002E",""};
lookup1("2498") ->	{"0","<compat> 0031 0037 002E",""};
lookup1("2499") ->	{"0","<compat> 0031 0038 002E",""};
lookup1("249A") ->	{"0","<compat> 0031 0039 002E",""};
lookup1("249B") ->	{"0","<compat> 0032 0030 002E",""};
lookup1("249C") ->	{"0","<compat> 0028 0061 0029",""};
lookup1("249D") ->	{"0","<compat> 0028 0062 0029",""};
lookup1("249E") ->	{"0","<compat> 0028 0063 0029",""};
lookup1("249F") ->	{"0","<compat> 0028 0064 0029",""};
lookup1("24A0") ->	{"0","<compat> 0028 0065 0029",""};
lookup1("24A1") ->	{"0","<compat> 0028 0066 0029",""};
lookup1("24A2") ->	{"0","<compat> 0028 0067 0029",""};
lookup1("24A3") ->	{"0","<compat> 0028 0068 0029",""};
lookup1("24A4") ->	{"0","<compat> 0028 0069 0029",""};
lookup1("24A5") ->	{"0","<compat> 0028 006A 0029",""};
lookup1("24A6") ->	{"0","<compat> 0028 006B 0029",""};
lookup1("24A7") ->	{"0","<compat> 0028 006C 0029",""};
lookup1("24A8") ->	{"0","<compat> 0028 006D 0029",""};
lookup1("24A9") ->	{"0","<compat> 0028 006E 0029",""};
lookup1("24AA") ->	{"0","<compat> 0028 006F 0029",""};
lookup1("24AB") ->	{"0","<compat> 0028 0070 0029",""};
lookup1("24AC") ->	{"0","<compat> 0028 0071 0029",""};
lookup1("24AD") ->	{"0","<compat> 0028 0072 0029",""};
lookup1("24AE") ->	{"0","<compat> 0028 0073 0029",""};
lookup1("24AF") ->	{"0","<compat> 0028 0074 0029",""};
lookup1("24B0") ->	{"0","<compat> 0028 0075 0029",""};
lookup1("24B1") ->	{"0","<compat> 0028 0076 0029",""};
lookup1("24B2") ->	{"0","<compat> 0028 0077 0029",""};
lookup1("24B3") ->	{"0","<compat> 0028 0078 0029",""};
lookup1("24B4") ->	{"0","<compat> 0028 0079 0029",""};
lookup1("24B5") ->	{"0","<compat> 0028 007A 0029",""};
lookup1("24B6") ->	{"0","<circle> 0041","24D0"};
lookup1("24B7") ->	{"0","<circle> 0042","24D1"};
lookup1("24B8") ->	{"0","<circle> 0043","24D2"};
lookup1("24B9") ->	{"0","<circle> 0044","24D3"};
lookup1("24BA") ->	{"0","<circle> 0045","24D4"};
lookup1("24BB") ->	{"0","<circle> 0046","24D5"};
lookup1("24BC") ->	{"0","<circle> 0047","24D6"};
lookup1("24BD") ->	{"0","<circle> 0048","24D7"};
lookup1("24BE") ->	{"0","<circle> 0049","24D8"};
lookup1("24BF") ->	{"0","<circle> 004A","24D9"};
lookup1("24C0") ->	{"0","<circle> 004B","24DA"};
lookup1("24C1") ->	{"0","<circle> 004C","24DB"};
lookup1("24C2") ->	{"0","<circle> 004D","24DC"};
lookup1("24C3") ->	{"0","<circle> 004E","24DD"};
lookup1("24C4") ->	{"0","<circle> 004F","24DE"};
lookup1("24C5") ->	{"0","<circle> 0050","24DF"};
lookup1("24C6") ->	{"0","<circle> 0051","24E0"};
lookup1("24C7") ->	{"0","<circle> 0052","24E1"};
lookup1("24C8") ->	{"0","<circle> 0053","24E2"};
lookup1("24C9") ->	{"0","<circle> 0054","24E3"};
lookup1("24CA") ->	{"0","<circle> 0055","24E4"};
lookup1("24CB") ->	{"0","<circle> 0056","24E5"};
lookup1("24CC") ->	{"0","<circle> 0057","24E6"};
lookup1("24CD") ->	{"0","<circle> 0058","24E7"};
lookup1("24CE") ->	{"0","<circle> 0059","24E8"};
lookup1("24CF") ->	{"0","<circle> 005A","24E9"};
lookup1("24D0") ->	{"0","<circle> 0061",""};
lookup1("24D1") ->	{"0","<circle> 0062",""};
lookup1("24D2") ->	{"0","<circle> 0063",""};
lookup1("24D3") ->	{"0","<circle> 0064",""};
lookup1("24D4") ->	{"0","<circle> 0065",""};
lookup1("24D5") ->	{"0","<circle> 0066",""};
lookup1("24D6") ->	{"0","<circle> 0067",""};
lookup1("24D7") ->	{"0","<circle> 0068",""};
lookup1("24D8") ->	{"0","<circle> 0069",""};
lookup1("24D9") ->	{"0","<circle> 006A",""};
lookup1("24DA") ->	{"0","<circle> 006B",""};
lookup1("24DB") ->	{"0","<circle> 006C",""};
lookup1("24DC") ->	{"0","<circle> 006D",""};
lookup1("24DD") ->	{"0","<circle> 006E",""};
lookup1("24DE") ->	{"0","<circle> 006F",""};
lookup1("24DF") ->	{"0","<circle> 0070",""};
lookup1("24E0") ->	{"0","<circle> 0071",""};
lookup1("24E1") ->	{"0","<circle> 0072",""};
lookup1("24E2") ->	{"0","<circle> 0073",""};
lookup1("24E3") ->	{"0","<circle> 0074",""};
lookup1("24E4") ->	{"0","<circle> 0075",""};
lookup1("24E5") ->	{"0","<circle> 0076",""};
lookup1("24E6") ->	{"0","<circle> 0077",""};
lookup1("24E7") ->	{"0","<circle> 0078",""};
lookup1("24E8") ->	{"0","<circle> 0079",""};
lookup1("24E9") ->	{"0","<circle> 007A",""};
lookup1("24EA") ->	{"0","<circle> 0030",""};
lookup1("24EB") ->	{"0","",""};
lookup1("24EC") ->	{"0","",""};
lookup1("24ED") ->	{"0","",""};
lookup1("24EE") ->	{"0","",""};
lookup1("24EF") ->	{"0","",""};
lookup1("24F0") ->	{"0","",""};
lookup1("24F1") ->	{"0","",""};
lookup1("24F2") ->	{"0","",""};
lookup1("24F3") ->	{"0","",""};
lookup1("24F4") ->	{"0","",""};
lookup1("24F5") ->	{"0","",""};
lookup1("24F6") ->	{"0","",""};
lookup1("24F7") ->	{"0","",""};
lookup1("24F8") ->	{"0","",""};
lookup1("24F9") ->	{"0","",""};
lookup1("24FA") ->	{"0","",""};
lookup1("24FB") ->	{"0","",""};
lookup1("24FC") ->	{"0","",""};
lookup1("24FD") ->	{"0","",""};
lookup1("24FE") ->	{"0","",""};
lookup1("24FF") ->	{"0","",""};
lookup1("2500") ->	{"0","",""};
lookup1("2501") ->	{"0","",""};
lookup1("2502") ->	{"0","",""};
lookup1("2503") ->	{"0","",""};
lookup1("2504") ->	{"0","",""};
lookup1("2505") ->	{"0","",""};
lookup1("2506") ->	{"0","",""};
lookup1("2507") ->	{"0","",""};
lookup1("2508") ->	{"0","",""};
lookup1("2509") ->	{"0","",""};
lookup1("250A") ->	{"0","",""};
lookup1("250B") ->	{"0","",""};
lookup1("250C") ->	{"0","",""};
lookup1("250D") ->	{"0","",""};
lookup1("250E") ->	{"0","",""};
lookup1("250F") ->	{"0","",""};
lookup1("2510") ->	{"0","",""};
lookup1("2511") ->	{"0","",""};
lookup1("2512") ->	{"0","",""};
lookup1("2513") ->	{"0","",""};
lookup1("2514") ->	{"0","",""};
lookup1("2515") ->	{"0","",""};
lookup1("2516") ->	{"0","",""};
lookup1("2517") ->	{"0","",""};
lookup1("2518") ->	{"0","",""};
lookup1("2519") ->	{"0","",""};
lookup1("251A") ->	{"0","",""};
lookup1("251B") ->	{"0","",""};
lookup1("251C") ->	{"0","",""};
lookup1("251D") ->	{"0","",""};
lookup1("251E") ->	{"0","",""};
lookup1("251F") ->	{"0","",""};
lookup1("2520") ->	{"0","",""};
lookup1("2521") ->	{"0","",""};
lookup1("2522") ->	{"0","",""};
lookup1("2523") ->	{"0","",""};
lookup1("2524") ->	{"0","",""};
lookup1("2525") ->	{"0","",""};
lookup1("2526") ->	{"0","",""};
lookup1("2527") ->	{"0","",""};
lookup1("2528") ->	{"0","",""};
lookup1("2529") ->	{"0","",""};
lookup1("252A") ->	{"0","",""};
lookup1("252B") ->	{"0","",""};
lookup1("252C") ->	{"0","",""};
lookup1("252D") ->	{"0","",""};
lookup1("252E") ->	{"0","",""};
lookup1("252F") ->	{"0","",""};
lookup1("2530") ->	{"0","",""};
lookup1("2531") ->	{"0","",""};
lookup1("2532") ->	{"0","",""};
lookup1("2533") ->	{"0","",""};
lookup1("2534") ->	{"0","",""};
lookup1("2535") ->	{"0","",""};
lookup1("2536") ->	{"0","",""};
lookup1("2537") ->	{"0","",""};
lookup1("2538") ->	{"0","",""};
lookup1("2539") ->	{"0","",""};
lookup1("253A") ->	{"0","",""};
lookup1("253B") ->	{"0","",""};
lookup1("253C") ->	{"0","",""};
lookup1("253D") ->	{"0","",""};
lookup1("253E") ->	{"0","",""};
lookup1("253F") ->	{"0","",""};
lookup1("2540") ->	{"0","",""};
lookup1("2541") ->	{"0","",""};
lookup1("2542") ->	{"0","",""};
lookup1("2543") ->	{"0","",""};
lookup1("2544") ->	{"0","",""};
lookup1("2545") ->	{"0","",""};
lookup1("2546") ->	{"0","",""};
lookup1("2547") ->	{"0","",""};
lookup1("2548") ->	{"0","",""};
lookup1("2549") ->	{"0","",""};
lookup1("254A") ->	{"0","",""};
lookup1("254B") ->	{"0","",""};
lookup1("254C") ->	{"0","",""};
lookup1("254D") ->	{"0","",""};
lookup1("254E") ->	{"0","",""};
lookup1("254F") ->	{"0","",""};
lookup1("2550") ->	{"0","",""};
lookup1("2551") ->	{"0","",""};
lookup1("2552") ->	{"0","",""};
lookup1("2553") ->	{"0","",""};
lookup1("2554") ->	{"0","",""};
lookup1("2555") ->	{"0","",""};
lookup1("2556") ->	{"0","",""};
lookup1("2557") ->	{"0","",""};
lookup1("2558") ->	{"0","",""};
lookup1("2559") ->	{"0","",""};
lookup1("255A") ->	{"0","",""};
lookup1("255B") ->	{"0","",""};
lookup1("255C") ->	{"0","",""};
lookup1("255D") ->	{"0","",""};
lookup1("255E") ->	{"0","",""};
lookup1("255F") ->	{"0","",""};
lookup1("2560") ->	{"0","",""};
lookup1("2561") ->	{"0","",""};
lookup1("2562") ->	{"0","",""};
lookup1("2563") ->	{"0","",""};
lookup1("2564") ->	{"0","",""};
lookup1("2565") ->	{"0","",""};
lookup1("2566") ->	{"0","",""};
lookup1("2567") ->	{"0","",""};
lookup1("2568") ->	{"0","",""};
lookup1("2569") ->	{"0","",""};
lookup1("256A") ->	{"0","",""};
lookup1("256B") ->	{"0","",""};
lookup1("256C") ->	{"0","",""};
lookup1("256D") ->	{"0","",""};
lookup1("256E") ->	{"0","",""};
lookup1("256F") ->	{"0","",""};
lookup1("2570") ->	{"0","",""};
lookup1("2571") ->	{"0","",""};
lookup1("2572") ->	{"0","",""};
lookup1("2573") ->	{"0","",""};
lookup1("2574") ->	{"0","",""};
lookup1("2575") ->	{"0","",""};
lookup1("2576") ->	{"0","",""};
lookup1("2577") ->	{"0","",""};
lookup1("2578") ->	{"0","",""};
lookup1("2579") ->	{"0","",""};
lookup1("257A") ->	{"0","",""};
lookup1("257B") ->	{"0","",""};
lookup1("257C") ->	{"0","",""};
lookup1("257D") ->	{"0","",""};
lookup1("257E") ->	{"0","",""};
lookup1("257F") ->	{"0","",""};
lookup1("2580") ->	{"0","",""};
lookup1("2581") ->	{"0","",""};
lookup1("2582") ->	{"0","",""};
lookup1("2583") ->	{"0","",""};
lookup1("2584") ->	{"0","",""};
lookup1("2585") ->	{"0","",""};
lookup1("2586") ->	{"0","",""};
lookup1("2587") ->	{"0","",""};
lookup1("2588") ->	{"0","",""};
lookup1("2589") ->	{"0","",""};
lookup1("258A") ->	{"0","",""};
lookup1("258B") ->	{"0","",""};
lookup1("258C") ->	{"0","",""};
lookup1("258D") ->	{"0","",""};
lookup1("258E") ->	{"0","",""};
lookup1("258F") ->	{"0","",""};
lookup1("2590") ->	{"0","",""};
lookup1("2591") ->	{"0","",""};
lookup1("2592") ->	{"0","",""};
lookup1("2593") ->	{"0","",""};
lookup1("2594") ->	{"0","",""};
lookup1("2595") ->	{"0","",""};
lookup1("2596") ->	{"0","",""};
lookup1("2597") ->	{"0","",""};
lookup1("2598") ->	{"0","",""};
lookup1("2599") ->	{"0","",""};
lookup1("259A") ->	{"0","",""};
lookup1("259B") ->	{"0","",""};
lookup1("259C") ->	{"0","",""};
lookup1("259D") ->	{"0","",""};
lookup1("259E") ->	{"0","",""};
lookup1("259F") ->	{"0","",""};
lookup1("25A0") ->	{"0","",""};
lookup1("25A1") ->	{"0","",""};
lookup1("25A2") ->	{"0","",""};
lookup1("25A3") ->	{"0","",""};
lookup1("25A4") ->	{"0","",""};
lookup1("25A5") ->	{"0","",""};
lookup1("25A6") ->	{"0","",""};
lookup1("25A7") ->	{"0","",""};
lookup1("25A8") ->	{"0","",""};
lookup1("25A9") ->	{"0","",""};
lookup1("25AA") ->	{"0","",""};
lookup1("25AB") ->	{"0","",""};
lookup1("25AC") ->	{"0","",""};
lookup1("25AD") ->	{"0","",""};
lookup1("25AE") ->	{"0","",""};
lookup1("25AF") ->	{"0","",""};
lookup1("25B0") ->	{"0","",""};
lookup1("25B1") ->	{"0","",""};
lookup1("25B2") ->	{"0","",""};
lookup1("25B3") ->	{"0","",""};
lookup1("25B4") ->	{"0","",""};
lookup1("25B5") ->	{"0","",""};
lookup1("25B6") ->	{"0","",""};
lookup1("25B7") ->	{"0","",""};
lookup1("25B8") ->	{"0","",""};
lookup1("25B9") ->	{"0","",""};
lookup1("25BA") ->	{"0","",""};
lookup1("25BB") ->	{"0","",""};
lookup1("25BC") ->	{"0","",""};
lookup1("25BD") ->	{"0","",""};
lookup1("25BE") ->	{"0","",""};
lookup1("25BF") ->	{"0","",""};
lookup1("25C0") ->	{"0","",""};
lookup1("25C1") ->	{"0","",""};
lookup1("25C2") ->	{"0","",""};
lookup1("25C3") ->	{"0","",""};
lookup1("25C4") ->	{"0","",""};
lookup1("25C5") ->	{"0","",""};
lookup1("25C6") ->	{"0","",""};
lookup1("25C7") ->	{"0","",""};
lookup1("25C8") ->	{"0","",""};
lookup1("25C9") ->	{"0","",""};
lookup1("25CA") ->	{"0","",""};
lookup1("25CB") ->	{"0","",""};
lookup1("25CC") ->	{"0","",""};
lookup1("25CD") ->	{"0","",""};
lookup1("25CE") ->	{"0","",""};
lookup1("25CF") ->	{"0","",""};
lookup1("25D0") ->	{"0","",""};
lookup1("25D1") ->	{"0","",""};
lookup1("25D2") ->	{"0","",""};
lookup1("25D3") ->	{"0","",""};
lookup1("25D4") ->	{"0","",""};
lookup1("25D5") ->	{"0","",""};
lookup1("25D6") ->	{"0","",""};
lookup1("25D7") ->	{"0","",""};
lookup1("25D8") ->	{"0","",""};
lookup1("25D9") ->	{"0","",""};
lookup1("25DA") ->	{"0","",""};
lookup1("25DB") ->	{"0","",""};
lookup1("25DC") ->	{"0","",""};
lookup1("25DD") ->	{"0","",""};
lookup1("25DE") ->	{"0","",""};
lookup1("25DF") ->	{"0","",""};
lookup1("25E0") ->	{"0","",""};
lookup1("25E1") ->	{"0","",""};
lookup1("25E2") ->	{"0","",""};
lookup1("25E3") ->	{"0","",""};
lookup1("25E4") ->	{"0","",""};
lookup1("25E5") ->	{"0","",""};
lookup1("25E6") ->	{"0","",""};
lookup1("25E7") ->	{"0","",""};
lookup1("25E8") ->	{"0","",""};
lookup1("25E9") ->	{"0","",""};
lookup1("25EA") ->	{"0","",""};
lookup1("25EB") ->	{"0","",""};
lookup1("25EC") ->	{"0","",""};
lookup1("25ED") ->	{"0","",""};
lookup1("25EE") ->	{"0","",""};
lookup1("25EF") ->	{"0","",""};
lookup1("25F0") ->	{"0","",""};
lookup1("25F1") ->	{"0","",""};
lookup1("25F2") ->	{"0","",""};
lookup1("25F3") ->	{"0","",""};
lookup1("25F4") ->	{"0","",""};
lookup1("25F5") ->	{"0","",""};
lookup1("25F6") ->	{"0","",""};
lookup1("25F7") ->	{"0","",""};
lookup1("25F8") ->	{"0","",""};
lookup1("25F9") ->	{"0","",""};
lookup1("25FA") ->	{"0","",""};
lookup1("25FB") ->	{"0","",""};
lookup1("25FC") ->	{"0","",""};
lookup1("25FD") ->	{"0","",""};
lookup1("25FE") ->	{"0","",""};
lookup1("25FF") ->	{"0","",""};
lookup1("2600") ->	{"0","",""};
lookup1("2601") ->	{"0","",""};
lookup1("2602") ->	{"0","",""};
lookup1("2603") ->	{"0","",""};
lookup1("2604") ->	{"0","",""};
lookup1("2605") ->	{"0","",""};
lookup1("2606") ->	{"0","",""};
lookup1("2607") ->	{"0","",""};
lookup1("2608") ->	{"0","",""};
lookup1("2609") ->	{"0","",""};
lookup1("260A") ->	{"0","",""};
lookup1("260B") ->	{"0","",""};
lookup1("260C") ->	{"0","",""};
lookup1("260D") ->	{"0","",""};
lookup1("260E") ->	{"0","",""};
lookup1("260F") ->	{"0","",""};
lookup1("2610") ->	{"0","",""};
lookup1("2611") ->	{"0","",""};
lookup1("2612") ->	{"0","",""};
lookup1("2613") ->	{"0","",""};
lookup1("2614") ->	{"0","",""};
lookup1("2615") ->	{"0","",""};
lookup1("2616") ->	{"0","",""};
lookup1("2617") ->	{"0","",""};
lookup1("2618") ->	{"0","",""};
lookup1("2619") ->	{"0","",""};
lookup1("261A") ->	{"0","",""};
lookup1("261B") ->	{"0","",""};
lookup1("261C") ->	{"0","",""};
lookup1("261D") ->	{"0","",""};
lookup1("261E") ->	{"0","",""};
lookup1("261F") ->	{"0","",""};
lookup1("2620") ->	{"0","",""};
lookup1("2621") ->	{"0","",""};
lookup1("2622") ->	{"0","",""};
lookup1("2623") ->	{"0","",""};
lookup1("2624") ->	{"0","",""};
lookup1("2625") ->	{"0","",""};
lookup1("2626") ->	{"0","",""};
lookup1("2627") ->	{"0","",""};
lookup1("2628") ->	{"0","",""};
lookup1("2629") ->	{"0","",""};
lookup1("262A") ->	{"0","",""};
lookup1("262B") ->	{"0","",""};
lookup1("262C") ->	{"0","",""};
lookup1("262D") ->	{"0","",""};
lookup1("262E") ->	{"0","",""};
lookup1("262F") ->	{"0","",""};
lookup1("2630") ->	{"0","",""};
lookup1("2631") ->	{"0","",""};
lookup1("2632") ->	{"0","",""};
lookup1("2633") ->	{"0","",""};
lookup1("2634") ->	{"0","",""};
lookup1("2635") ->	{"0","",""};
lookup1("2636") ->	{"0","",""};
lookup1("2637") ->	{"0","",""};
lookup1("2638") ->	{"0","",""};
lookup1("2639") ->	{"0","",""};
lookup1("263A") ->	{"0","",""};
lookup1("263B") ->	{"0","",""};
lookup1("263C") ->	{"0","",""};
lookup1("263D") ->	{"0","",""};
lookup1("263E") ->	{"0","",""};
lookup1("263F") ->	{"0","",""};
lookup1("2640") ->	{"0","",""};
lookup1("2641") ->	{"0","",""};
lookup1("2642") ->	{"0","",""};
lookup1("2643") ->	{"0","",""};
lookup1("2644") ->	{"0","",""};
lookup1("2645") ->	{"0","",""};
lookup1("2646") ->	{"0","",""};
lookup1("2647") ->	{"0","",""};
lookup1("2648") ->	{"0","",""};
lookup1("2649") ->	{"0","",""};
lookup1("264A") ->	{"0","",""};
lookup1("264B") ->	{"0","",""};
lookup1("264C") ->	{"0","",""};
lookup1("264D") ->	{"0","",""};
lookup1("264E") ->	{"0","",""};
lookup1("264F") ->	{"0","",""};
lookup1("2650") ->	{"0","",""};
lookup1("2651") ->	{"0","",""};
lookup1("2652") ->	{"0","",""};
lookup1("2653") ->	{"0","",""};
lookup1("2654") ->	{"0","",""};
lookup1("2655") ->	{"0","",""};
lookup1("2656") ->	{"0","",""};
lookup1("2657") ->	{"0","",""};
lookup1("2658") ->	{"0","",""};
lookup1("2659") ->	{"0","",""};
lookup1("265A") ->	{"0","",""};
lookup1("265B") ->	{"0","",""};
lookup1("265C") ->	{"0","",""};
lookup1("265D") ->	{"0","",""};
lookup1("265E") ->	{"0","",""};
lookup1("265F") ->	{"0","",""};
lookup1("2660") ->	{"0","",""};
lookup1("2661") ->	{"0","",""};
lookup1("2662") ->	{"0","",""};
lookup1("2663") ->	{"0","",""};
lookup1("2664") ->	{"0","",""};
lookup1("2665") ->	{"0","",""};
lookup1("2666") ->	{"0","",""};
lookup1("2667") ->	{"0","",""};
lookup1("2668") ->	{"0","",""};
lookup1("2669") ->	{"0","",""};
lookup1("266A") ->	{"0","",""};
lookup1("266B") ->	{"0","",""};
lookup1("266C") ->	{"0","",""};
lookup1("266D") ->	{"0","",""};
lookup1("266E") ->	{"0","",""};
lookup1("266F") ->	{"0","",""};
lookup1("2670") ->	{"0","",""};
lookup1("2671") ->	{"0","",""};
lookup1("2672") ->	{"0","",""};
lookup1("2673") ->	{"0","",""};
lookup1("2674") ->	{"0","",""};
lookup1("2675") ->	{"0","",""};
lookup1("2676") ->	{"0","",""};
lookup1("2677") ->	{"0","",""};
lookup1("2678") ->	{"0","",""};
lookup1("2679") ->	{"0","",""};
lookup1("267A") ->	{"0","",""};
lookup1("267B") ->	{"0","",""};
lookup1("267C") ->	{"0","",""};
lookup1("267D") ->	{"0","",""};
lookup1("267E") ->	{"0","",""};
lookup1("267F") ->	{"0","",""};
lookup1("2680") ->	{"0","",""};
lookup1("2681") ->	{"0","",""};
lookup1("2682") ->	{"0","",""};
lookup1("2683") ->	{"0","",""};
lookup1("2684") ->	{"0","",""};
lookup1("2685") ->	{"0","",""};
lookup1("2686") ->	{"0","",""};
lookup1("2687") ->	{"0","",""};
lookup1("2688") ->	{"0","",""};
lookup1("2689") ->	{"0","",""};
lookup1("268A") ->	{"0","",""};
lookup1("268B") ->	{"0","",""};
lookup1("268C") ->	{"0","",""};
lookup1("268D") ->	{"0","",""};
lookup1("268E") ->	{"0","",""};
lookup1("268F") ->	{"0","",""};
lookup1("2690") ->	{"0","",""};
lookup1("2691") ->	{"0","",""};
lookup1("2692") ->	{"0","",""};
lookup1("2693") ->	{"0","",""};
lookup1("2694") ->	{"0","",""};
lookup1("2695") ->	{"0","",""};
lookup1("2696") ->	{"0","",""};
lookup1("2697") ->	{"0","",""};
lookup1("2698") ->	{"0","",""};
lookup1("2699") ->	{"0","",""};
lookup1("269A") ->	{"0","",""};
lookup1("269B") ->	{"0","",""};
lookup1("269C") ->	{"0","",""};
lookup1("269D") ->	{"0","",""};
lookup1("269E") ->	{"0","",""};
lookup1("269F") ->	{"0","",""};
lookup1("26A0") ->	{"0","",""};
lookup1("26A1") ->	{"0","",""};
lookup1("26A2") ->	{"0","",""};
lookup1("26A3") ->	{"0","",""};
lookup1("26A4") ->	{"0","",""};
lookup1("26A5") ->	{"0","",""};
lookup1("26A6") ->	{"0","",""};
lookup1("26A7") ->	{"0","",""};
lookup1("26A8") ->	{"0","",""};
lookup1("26A9") ->	{"0","",""};
lookup1("26AA") ->	{"0","",""};
lookup1("26AB") ->	{"0","",""};
lookup1("26AC") ->	{"0","",""};
lookup1("26AD") ->	{"0","",""};
lookup1("26AE") ->	{"0","",""};
lookup1("26AF") ->	{"0","",""};
lookup1("26B0") ->	{"0","",""};
lookup1("26B1") ->	{"0","",""};
lookup1("26B2") ->	{"0","",""};
lookup1("26B3") ->	{"0","",""};
lookup1("26B4") ->	{"0","",""};
lookup1("26B5") ->	{"0","",""};
lookup1("26B6") ->	{"0","",""};
lookup1("26B7") ->	{"0","",""};
lookup1("26B8") ->	{"0","",""};
lookup1("26B9") ->	{"0","",""};
lookup1("26BA") ->	{"0","",""};
lookup1("26BB") ->	{"0","",""};
lookup1("26BC") ->	{"0","",""};
lookup1("26BD") ->	{"0","",""};
lookup1("26BE") ->	{"0","",""};
lookup1("26BF") ->	{"0","",""};
lookup1("26C0") ->	{"0","",""};
lookup1("26C1") ->	{"0","",""};
lookup1("26C2") ->	{"0","",""};
lookup1("26C3") ->	{"0","",""};
lookup1("26C4") ->	{"0","",""};
lookup1("26C5") ->	{"0","",""};
lookup1("26C6") ->	{"0","",""};
lookup1("26C7") ->	{"0","",""};
lookup1("26C8") ->	{"0","",""};
lookup1("26C9") ->	{"0","",""};
lookup1("26CA") ->	{"0","",""};
lookup1("26CB") ->	{"0","",""};
lookup1("26CC") ->	{"0","",""};
lookup1("26CD") ->	{"0","",""};
lookup1("26CE") ->	{"0","",""};
lookup1("26CF") ->	{"0","",""};
lookup1("26D0") ->	{"0","",""};
lookup1("26D1") ->	{"0","",""};
lookup1("26D2") ->	{"0","",""};
lookup1("26D3") ->	{"0","",""};
lookup1("26D4") ->	{"0","",""};
lookup1("26D5") ->	{"0","",""};
lookup1("26D6") ->	{"0","",""};
lookup1("26D7") ->	{"0","",""};
lookup1("26D8") ->	{"0","",""};
lookup1("26D9") ->	{"0","",""};
lookup1("26DA") ->	{"0","",""};
lookup1("26DB") ->	{"0","",""};
lookup1("26DC") ->	{"0","",""};
lookup1("26DD") ->	{"0","",""};
lookup1("26DE") ->	{"0","",""};
lookup1("26DF") ->	{"0","",""};
lookup1("26E0") ->	{"0","",""};
lookup1("26E1") ->	{"0","",""};
lookup1("26E2") ->	{"0","",""};
lookup1("26E3") ->	{"0","",""};
lookup1("26E4") ->	{"0","",""};
lookup1("26E5") ->	{"0","",""};
lookup1("26E6") ->	{"0","",""};
lookup1("26E7") ->	{"0","",""};
lookup1("26E8") ->	{"0","",""};
lookup1("26E9") ->	{"0","",""};
lookup1("26EA") ->	{"0","",""};
lookup1("26EB") ->	{"0","",""};
lookup1("26EC") ->	{"0","",""};
lookup1("26ED") ->	{"0","",""};
lookup1("26EE") ->	{"0","",""};
lookup1("26EF") ->	{"0","",""};
lookup1("26F0") ->	{"0","",""};
lookup1("26F1") ->	{"0","",""};
lookup1("26F2") ->	{"0","",""};
lookup1("26F3") ->	{"0","",""};
lookup1("26F4") ->	{"0","",""};
lookup1("26F5") ->	{"0","",""};
lookup1("26F6") ->	{"0","",""};
lookup1("26F7") ->	{"0","",""};
lookup1("26F8") ->	{"0","",""};
lookup1("26F9") ->	{"0","",""};
lookup1("26FA") ->	{"0","",""};
lookup1("26FB") ->	{"0","",""};
lookup1("26FC") ->	{"0","",""};
lookup1("26FD") ->	{"0","",""};
lookup1("26FE") ->	{"0","",""};
lookup1("26FF") ->	{"0","",""};
lookup1("2700") ->	{"0","",""};
lookup1("2701") ->	{"0","",""};
lookup1("2702") ->	{"0","",""};
lookup1("2703") ->	{"0","",""};
lookup1("2704") ->	{"0","",""};
lookup1("2705") ->	{"0","",""};
lookup1("2706") ->	{"0","",""};
lookup1("2707") ->	{"0","",""};
lookup1("2708") ->	{"0","",""};
lookup1("2709") ->	{"0","",""};
lookup1("270A") ->	{"0","",""};
lookup1("270B") ->	{"0","",""};
lookup1("270C") ->	{"0","",""};
lookup1("270D") ->	{"0","",""};
lookup1("270E") ->	{"0","",""};
lookup1("270F") ->	{"0","",""};
lookup1("2710") ->	{"0","",""};
lookup1("2711") ->	{"0","",""};
lookup1("2712") ->	{"0","",""};
lookup1("2713") ->	{"0","",""};
lookup1("2714") ->	{"0","",""};
lookup1("2715") ->	{"0","",""};
lookup1("2716") ->	{"0","",""};
lookup1("2717") ->	{"0","",""};
lookup1("2718") ->	{"0","",""};
lookup1("2719") ->	{"0","",""};
lookup1("271A") ->	{"0","",""};
lookup1("271B") ->	{"0","",""};
lookup1("271C") ->	{"0","",""};
lookup1("271D") ->	{"0","",""};
lookup1("271E") ->	{"0","",""};
lookup1("271F") ->	{"0","",""};
lookup1("2720") ->	{"0","",""};
lookup1("2721") ->	{"0","",""};
lookup1("2722") ->	{"0","",""};
lookup1("2723") ->	{"0","",""};
lookup1("2724") ->	{"0","",""};
lookup1("2725") ->	{"0","",""};
lookup1("2726") ->	{"0","",""};
lookup1("2727") ->	{"0","",""};
lookup1("2728") ->	{"0","",""};
lookup1("2729") ->	{"0","",""};
lookup1("272A") ->	{"0","",""};
lookup1("272B") ->	{"0","",""};
lookup1("272C") ->	{"0","",""};
lookup1("272D") ->	{"0","",""};
lookup1("272E") ->	{"0","",""};
lookup1("272F") ->	{"0","",""};
lookup1("2730") ->	{"0","",""};
lookup1("2731") ->	{"0","",""};
lookup1("2732") ->	{"0","",""};
lookup1("2733") ->	{"0","",""};
lookup1("2734") ->	{"0","",""};
lookup1("2735") ->	{"0","",""};
lookup1("2736") ->	{"0","",""};
lookup1("2737") ->	{"0","",""};
lookup1("2738") ->	{"0","",""};
lookup1("2739") ->	{"0","",""};
lookup1("273A") ->	{"0","",""};
lookup1("273B") ->	{"0","",""};
lookup1("273C") ->	{"0","",""};
lookup1("273D") ->	{"0","",""};
lookup1("273E") ->	{"0","",""};
lookup1("273F") ->	{"0","",""};
lookup1("2740") ->	{"0","",""};
lookup1("2741") ->	{"0","",""};
lookup1("2742") ->	{"0","",""};
lookup1("2743") ->	{"0","",""};
lookup1("2744") ->	{"0","",""};
lookup1("2745") ->	{"0","",""};
lookup1("2746") ->	{"0","",""};
lookup1("2747") ->	{"0","",""};
lookup1("2748") ->	{"0","",""};
lookup1("2749") ->	{"0","",""};
lookup1("274A") ->	{"0","",""};
lookup1("274B") ->	{"0","",""};
lookup1("274C") ->	{"0","",""};
lookup1("274D") ->	{"0","",""};
lookup1("274E") ->	{"0","",""};
lookup1("274F") ->	{"0","",""};
lookup1("2750") ->	{"0","",""};
lookup1("2751") ->	{"0","",""};
lookup1("2752") ->	{"0","",""};
lookup1("2753") ->	{"0","",""};
lookup1("2754") ->	{"0","",""};
lookup1("2755") ->	{"0","",""};
lookup1("2756") ->	{"0","",""};
lookup1("2757") ->	{"0","",""};
lookup1("2758") ->	{"0","",""};
lookup1("2759") ->	{"0","",""};
lookup1("275A") ->	{"0","",""};
lookup1("275B") ->	{"0","",""};
lookup1("275C") ->	{"0","",""};
lookup1("275D") ->	{"0","",""};
lookup1("275E") ->	{"0","",""};
lookup1("275F") ->	{"0","",""};
lookup1("2760") ->	{"0","",""};
lookup1("2761") ->	{"0","",""};
lookup1("2762") ->	{"0","",""};
lookup1("2763") ->	{"0","",""};
lookup1("2764") ->	{"0","",""};
lookup1("2765") ->	{"0","",""};
lookup1("2766") ->	{"0","",""};
lookup1("2767") ->	{"0","",""};
lookup1("2768") ->	{"0","",""};
lookup1("2769") ->	{"0","",""};
lookup1("276A") ->	{"0","",""};
lookup1("276B") ->	{"0","",""};
lookup1("276C") ->	{"0","",""};
lookup1("276D") ->	{"0","",""};
lookup1("276E") ->	{"0","",""};
lookup1("276F") ->	{"0","",""};
lookup1("2770") ->	{"0","",""};
lookup1("2771") ->	{"0","",""};
lookup1("2772") ->	{"0","",""};
lookup1("2773") ->	{"0","",""};
lookup1("2774") ->	{"0","",""};
lookup1("2775") ->	{"0","",""};
lookup1("2776") ->	{"0","",""};
lookup1("2777") ->	{"0","",""};
lookup1("2778") ->	{"0","",""};
lookup1("2779") ->	{"0","",""};
lookup1("277A") ->	{"0","",""};
lookup1("277B") ->	{"0","",""};
lookup1("277C") ->	{"0","",""};
lookup1("277D") ->	{"0","",""};
lookup1("277E") ->	{"0","",""};
lookup1("277F") ->	{"0","",""};
lookup1("2780") ->	{"0","",""};
lookup1("2781") ->	{"0","",""};
lookup1("2782") ->	{"0","",""};
lookup1("2783") ->	{"0","",""};
lookup1("2784") ->	{"0","",""};
lookup1("2785") ->	{"0","",""};
lookup1("2786") ->	{"0","",""};
lookup1("2787") ->	{"0","",""};
lookup1("2788") ->	{"0","",""};
lookup1("2789") ->	{"0","",""};
lookup1("278A") ->	{"0","",""};
lookup1("278B") ->	{"0","",""};
lookup1("278C") ->	{"0","",""};
lookup1("278D") ->	{"0","",""};
lookup1("278E") ->	{"0","",""};
lookup1("278F") ->	{"0","",""};
lookup1("2790") ->	{"0","",""};
lookup1("2791") ->	{"0","",""};
lookup1("2792") ->	{"0","",""};
lookup1("2793") ->	{"0","",""};
lookup1("2794") ->	{"0","",""};
lookup1("2795") ->	{"0","",""};
lookup1("2796") ->	{"0","",""};
lookup1("2797") ->	{"0","",""};
lookup1("2798") ->	{"0","",""};
lookup1("2799") ->	{"0","",""};
lookup1("279A") ->	{"0","",""};
lookup1("279B") ->	{"0","",""};
lookup1("279C") ->	{"0","",""};
lookup1("279D") ->	{"0","",""};
lookup1("279E") ->	{"0","",""};
lookup1("279F") ->	{"0","",""};
lookup1("27A0") ->	{"0","",""};
lookup1("27A1") ->	{"0","",""};
lookup1("27A2") ->	{"0","",""};
lookup1("27A3") ->	{"0","",""};
lookup1("27A4") ->	{"0","",""};
lookup1("27A5") ->	{"0","",""};
lookup1("27A6") ->	{"0","",""};
lookup1("27A7") ->	{"0","",""};
lookup1("27A8") ->	{"0","",""};
lookup1("27A9") ->	{"0","",""};
lookup1("27AA") ->	{"0","",""};
lookup1("27AB") ->	{"0","",""};
lookup1("27AC") ->	{"0","",""};
lookup1("27AD") ->	{"0","",""};
lookup1("27AE") ->	{"0","",""};
lookup1("27AF") ->	{"0","",""};
lookup1("27B0") ->	{"0","",""};
lookup1("27B1") ->	{"0","",""};
lookup1("27B2") ->	{"0","",""};
lookup1("27B3") ->	{"0","",""};
lookup1("27B4") ->	{"0","",""};
lookup1("27B5") ->	{"0","",""};
lookup1("27B6") ->	{"0","",""};
lookup1("27B7") ->	{"0","",""};
lookup1("27B8") ->	{"0","",""};
lookup1("27B9") ->	{"0","",""};
lookup1("27BA") ->	{"0","",""};
lookup1("27BB") ->	{"0","",""};
lookup1("27BC") ->	{"0","",""};
lookup1("27BD") ->	{"0","",""};
lookup1("27BE") ->	{"0","",""};
lookup1("27BF") ->	{"0","",""};
lookup1("27C0") ->	{"0","",""};
lookup1("27C1") ->	{"0","",""};
lookup1("27C2") ->	{"0","",""};
lookup1("27C3") ->	{"0","",""};
lookup1("27C4") ->	{"0","",""};
lookup1("27C5") ->	{"0","",""};
lookup1("27C6") ->	{"0","",""};
lookup1("27C7") ->	{"0","",""};
lookup1("27C8") ->	{"0","",""};
lookup1("27C9") ->	{"0","",""};
lookup1("27CA") ->	{"0","",""};
lookup1("27CB") ->	{"0","",""};
lookup1("27CC") ->	{"0","",""};
lookup1("27CD") ->	{"0","",""};
lookup1("27CE") ->	{"0","",""};
lookup1("27CF") ->	{"0","",""};
lookup1("27D0") ->	{"0","",""};
lookup1("27D1") ->	{"0","",""};
lookup1("27D2") ->	{"0","",""};
lookup1("27D3") ->	{"0","",""};
lookup1("27D4") ->	{"0","",""};
lookup1("27D5") ->	{"0","",""};
lookup1("27D6") ->	{"0","",""};
lookup1("27D7") ->	{"0","",""};
lookup1("27D8") ->	{"0","",""};
lookup1("27D9") ->	{"0","",""};
lookup1("27DA") ->	{"0","",""};
lookup1("27DB") ->	{"0","",""};
lookup1("27DC") ->	{"0","",""};
lookup1("27DD") ->	{"0","",""};
lookup1("27DE") ->	{"0","",""};
lookup1("27DF") ->	{"0","",""};
lookup1("27E0") ->	{"0","",""};
lookup1("27E1") ->	{"0","",""};
lookup1("27E2") ->	{"0","",""};
lookup1("27E3") ->	{"0","",""};
lookup1("27E4") ->	{"0","",""};
lookup1("27E5") ->	{"0","",""};
lookup1("27E6") ->	{"0","",""};
lookup1("27E7") ->	{"0","",""};
lookup1("27E8") ->	{"0","",""};
lookup1("27E9") ->	{"0","",""};
lookup1("27EA") ->	{"0","",""};
lookup1("27EB") ->	{"0","",""};
lookup1("27EC") ->	{"0","",""};
lookup1("27ED") ->	{"0","",""};
lookup1("27EE") ->	{"0","",""};
lookup1("27EF") ->	{"0","",""};
lookup1("27F0") ->	{"0","",""};
lookup1("27F1") ->	{"0","",""};
lookup1("27F2") ->	{"0","",""};
lookup1("27F3") ->	{"0","",""};
lookup1("27F4") ->	{"0","",""};
lookup1("27F5") ->	{"0","",""};
lookup1("27F6") ->	{"0","",""};
lookup1("27F7") ->	{"0","",""};
lookup1("27F8") ->	{"0","",""};
lookup1("27F9") ->	{"0","",""};
lookup1("27FA") ->	{"0","",""};
lookup1("27FB") ->	{"0","",""};
lookup1("27FC") ->	{"0","",""};
lookup1("27FD") ->	{"0","",""};
lookup1("27FE") ->	{"0","",""};
lookup1("27FF") ->	{"0","",""};
lookup1("2800") ->	{"0","",""};
lookup1("2801") ->	{"0","",""};
lookup1("2802") ->	{"0","",""};
lookup1("2803") ->	{"0","",""};
lookup1("2804") ->	{"0","",""};
lookup1("2805") ->	{"0","",""};
lookup1("2806") ->	{"0","",""};
lookup1("2807") ->	{"0","",""};
lookup1("2808") ->	{"0","",""};
lookup1("2809") ->	{"0","",""};
lookup1("280A") ->	{"0","",""};
lookup1("280B") ->	{"0","",""};
lookup1("280C") ->	{"0","",""};
lookup1("280D") ->	{"0","",""};
lookup1("280E") ->	{"0","",""};
lookup1("280F") ->	{"0","",""};
lookup1("2810") ->	{"0","",""};
lookup1("2811") ->	{"0","",""};
lookup1("2812") ->	{"0","",""};
lookup1("2813") ->	{"0","",""};
lookup1("2814") ->	{"0","",""};
lookup1("2815") ->	{"0","",""};
lookup1("2816") ->	{"0","",""};
lookup1("2817") ->	{"0","",""};
lookup1("2818") ->	{"0","",""};
lookup1("2819") ->	{"0","",""};
lookup1("281A") ->	{"0","",""};
lookup1("281B") ->	{"0","",""};
lookup1("281C") ->	{"0","",""};
lookup1("281D") ->	{"0","",""};
lookup1("281E") ->	{"0","",""};
lookup1("281F") ->	{"0","",""};
lookup1("2820") ->	{"0","",""};
lookup1("2821") ->	{"0","",""};
lookup1("2822") ->	{"0","",""};
lookup1("2823") ->	{"0","",""};
lookup1("2824") ->	{"0","",""};
lookup1("2825") ->	{"0","",""};
lookup1("2826") ->	{"0","",""};
lookup1("2827") ->	{"0","",""};
lookup1("2828") ->	{"0","",""};
lookup1("2829") ->	{"0","",""};
lookup1("282A") ->	{"0","",""};
lookup1("282B") ->	{"0","",""};
lookup1("282C") ->	{"0","",""};
lookup1("282D") ->	{"0","",""};
lookup1("282E") ->	{"0","",""};
lookup1("282F") ->	{"0","",""};
lookup1("2830") ->	{"0","",""};
lookup1("2831") ->	{"0","",""};
lookup1("2832") ->	{"0","",""};
lookup1("2833") ->	{"0","",""};
lookup1("2834") ->	{"0","",""};
lookup1("2835") ->	{"0","",""};
lookup1("2836") ->	{"0","",""};
lookup1("2837") ->	{"0","",""};
lookup1("2838") ->	{"0","",""};
lookup1("2839") ->	{"0","",""};
lookup1("283A") ->	{"0","",""};
lookup1("283B") ->	{"0","",""};
lookup1("283C") ->	{"0","",""};
lookup1("283D") ->	{"0","",""};
lookup1("283E") ->	{"0","",""};
lookup1("283F") ->	{"0","",""};
lookup1("2840") ->	{"0","",""};
lookup1("2841") ->	{"0","",""};
lookup1("2842") ->	{"0","",""};
lookup1("2843") ->	{"0","",""};
lookup1("2844") ->	{"0","",""};
lookup1("2845") ->	{"0","",""};
lookup1("2846") ->	{"0","",""};
lookup1("2847") ->	{"0","",""};
lookup1("2848") ->	{"0","",""};
lookup1("2849") ->	{"0","",""};
lookup1("284A") ->	{"0","",""};
lookup1("284B") ->	{"0","",""};
lookup1("284C") ->	{"0","",""};
lookup1("284D") ->	{"0","",""};
lookup1("284E") ->	{"0","",""};
lookup1("284F") ->	{"0","",""};
lookup1("2850") ->	{"0","",""};
lookup1("2851") ->	{"0","",""};
lookup1("2852") ->	{"0","",""};
lookup1("2853") ->	{"0","",""};
lookup1("2854") ->	{"0","",""};
lookup1("2855") ->	{"0","",""};
lookup1("2856") ->	{"0","",""};
lookup1("2857") ->	{"0","",""};
lookup1("2858") ->	{"0","",""};
lookup1("2859") ->	{"0","",""};
lookup1("285A") ->	{"0","",""};
lookup1("285B") ->	{"0","",""};
lookup1("285C") ->	{"0","",""};
lookup1("285D") ->	{"0","",""};
lookup1("285E") ->	{"0","",""};
lookup1("285F") ->	{"0","",""};
lookup1("2860") ->	{"0","",""};
lookup1("2861") ->	{"0","",""};
lookup1("2862") ->	{"0","",""};
lookup1("2863") ->	{"0","",""};
lookup1("2864") ->	{"0","",""};
lookup1("2865") ->	{"0","",""};
lookup1("2866") ->	{"0","",""};
lookup1("2867") ->	{"0","",""};
lookup1("2868") ->	{"0","",""};
lookup1("2869") ->	{"0","",""};
lookup1("286A") ->	{"0","",""};
lookup1("286B") ->	{"0","",""};
lookup1("286C") ->	{"0","",""};
lookup1("286D") ->	{"0","",""};
lookup1("286E") ->	{"0","",""};
lookup1("286F") ->	{"0","",""};
lookup1("2870") ->	{"0","",""};
lookup1("2871") ->	{"0","",""};
lookup1("2872") ->	{"0","",""};
lookup1("2873") ->	{"0","",""};
lookup1("2874") ->	{"0","",""};
lookup1("2875") ->	{"0","",""};
lookup1("2876") ->	{"0","",""};
lookup1("2877") ->	{"0","",""};
lookup1("2878") ->	{"0","",""};
lookup1("2879") ->	{"0","",""};
lookup1("287A") ->	{"0","",""};
lookup1("287B") ->	{"0","",""};
lookup1("287C") ->	{"0","",""};
lookup1("287D") ->	{"0","",""};
lookup1("287E") ->	{"0","",""};
lookup1("287F") ->	{"0","",""};
lookup1("2880") ->	{"0","",""};
lookup1("2881") ->	{"0","",""};
lookup1("2882") ->	{"0","",""};
lookup1("2883") ->	{"0","",""};
lookup1("2884") ->	{"0","",""};
lookup1("2885") ->	{"0","",""};
lookup1("2886") ->	{"0","",""};
lookup1("2887") ->	{"0","",""};
lookup1("2888") ->	{"0","",""};
lookup1("2889") ->	{"0","",""};
lookup1("288A") ->	{"0","",""};
lookup1("288B") ->	{"0","",""};
lookup1("288C") ->	{"0","",""};
lookup1("288D") ->	{"0","",""};
lookup1("288E") ->	{"0","",""};
lookup1("288F") ->	{"0","",""};
lookup1("2890") ->	{"0","",""};
lookup1("2891") ->	{"0","",""};
lookup1("2892") ->	{"0","",""};
lookup1("2893") ->	{"0","",""};
lookup1("2894") ->	{"0","",""};
lookup1("2895") ->	{"0","",""};
lookup1("2896") ->	{"0","",""};
lookup1("2897") ->	{"0","",""};
lookup1("2898") ->	{"0","",""};
lookup1("2899") ->	{"0","",""};
lookup1("289A") ->	{"0","",""};
lookup1("289B") ->	{"0","",""};
lookup1("289C") ->	{"0","",""};
lookup1("289D") ->	{"0","",""};
lookup1("289E") ->	{"0","",""};
lookup1("289F") ->	{"0","",""};
lookup1("28A0") ->	{"0","",""};
lookup1("28A1") ->	{"0","",""};
lookup1("28A2") ->	{"0","",""};
lookup1("28A3") ->	{"0","",""};
lookup1("28A4") ->	{"0","",""};
lookup1("28A5") ->	{"0","",""};
lookup1("28A6") ->	{"0","",""};
lookup1("28A7") ->	{"0","",""};
lookup1("28A8") ->	{"0","",""};
lookup1("28A9") ->	{"0","",""};
lookup1("28AA") ->	{"0","",""};
lookup1("28AB") ->	{"0","",""};
lookup1("28AC") ->	{"0","",""};
lookup1("28AD") ->	{"0","",""};
lookup1("28AE") ->	{"0","",""};
lookup1("28AF") ->	{"0","",""};
lookup1("28B0") ->	{"0","",""};
lookup1("28B1") ->	{"0","",""};
lookup1("28B2") ->	{"0","",""};
lookup1("28B3") ->	{"0","",""};
lookup1("28B4") ->	{"0","",""};
lookup1("28B5") ->	{"0","",""};
lookup1("28B6") ->	{"0","",""};
lookup1("28B7") ->	{"0","",""};
lookup1("28B8") ->	{"0","",""};
lookup1("28B9") ->	{"0","",""};
lookup1("28BA") ->	{"0","",""};
lookup1("28BB") ->	{"0","",""};
lookup1("28BC") ->	{"0","",""};
lookup1("28BD") ->	{"0","",""};
lookup1("28BE") ->	{"0","",""};
lookup1("28BF") ->	{"0","",""};
lookup1("28C0") ->	{"0","",""};
lookup1("28C1") ->	{"0","",""};
lookup1("28C2") ->	{"0","",""};
lookup1("28C3") ->	{"0","",""};
lookup1("28C4") ->	{"0","",""};
lookup1("28C5") ->	{"0","",""};
lookup1("28C6") ->	{"0","",""};
lookup1("28C7") ->	{"0","",""};
lookup1("28C8") ->	{"0","",""};
lookup1("28C9") ->	{"0","",""};
lookup1("28CA") ->	{"0","",""};
lookup1("28CB") ->	{"0","",""};
lookup1("28CC") ->	{"0","",""};
lookup1("28CD") ->	{"0","",""};
lookup1("28CE") ->	{"0","",""};
lookup1("28CF") ->	{"0","",""};
lookup1("28D0") ->	{"0","",""};
lookup1("28D1") ->	{"0","",""};
lookup1("28D2") ->	{"0","",""};
lookup1("28D3") ->	{"0","",""};
lookup1("28D4") ->	{"0","",""};
lookup1("28D5") ->	{"0","",""};
lookup1("28D6") ->	{"0","",""};
lookup1("28D7") ->	{"0","",""};
lookup1("28D8") ->	{"0","",""};
lookup1("28D9") ->	{"0","",""};
lookup1("28DA") ->	{"0","",""};
lookup1("28DB") ->	{"0","",""};
lookup1("28DC") ->	{"0","",""};
lookup1("28DD") ->	{"0","",""};
lookup1("28DE") ->	{"0","",""};
lookup1("28DF") ->	{"0","",""};
lookup1("28E0") ->	{"0","",""};
lookup1("28E1") ->	{"0","",""};
lookup1("28E2") ->	{"0","",""};
lookup1("28E3") ->	{"0","",""};
lookup1("28E4") ->	{"0","",""};
lookup1("28E5") ->	{"0","",""};
lookup1("28E6") ->	{"0","",""};
lookup1("28E7") ->	{"0","",""};
lookup1("28E8") ->	{"0","",""};
lookup1("28E9") ->	{"0","",""};
lookup1("28EA") ->	{"0","",""};
lookup1("28EB") ->	{"0","",""};
lookup1("28EC") ->	{"0","",""};
lookup1("28ED") ->	{"0","",""};
lookup1("28EE") ->	{"0","",""};
lookup1("28EF") ->	{"0","",""};
lookup1("28F0") ->	{"0","",""};
lookup1("28F1") ->	{"0","",""};
lookup1("28F2") ->	{"0","",""};
lookup1("28F3") ->	{"0","",""};
lookup1("28F4") ->	{"0","",""};
lookup1("28F5") ->	{"0","",""};
lookup1("28F6") ->	{"0","",""};
lookup1("28F7") ->	{"0","",""};
lookup1("28F8") ->	{"0","",""};
lookup1("28F9") ->	{"0","",""};
lookup1("28FA") ->	{"0","",""};
lookup1("28FB") ->	{"0","",""};
lookup1("28FC") ->	{"0","",""};
lookup1("28FD") ->	{"0","",""};
lookup1("28FE") ->	{"0","",""};
lookup1("28FF") ->	{"0","",""};
lookup1("2900") ->	{"0","",""};
lookup1("2901") ->	{"0","",""};
lookup1("2902") ->	{"0","",""};
lookup1("2903") ->	{"0","",""};
lookup1("2904") ->	{"0","",""};
lookup1("2905") ->	{"0","",""};
lookup1("2906") ->	{"0","",""};
lookup1("2907") ->	{"0","",""};
lookup1("2908") ->	{"0","",""};
lookup1("2909") ->	{"0","",""};
lookup1("290A") ->	{"0","",""};
lookup1("290B") ->	{"0","",""};
lookup1("290C") ->	{"0","",""};
lookup1("290D") ->	{"0","",""};
lookup1("290E") ->	{"0","",""};
lookup1("290F") ->	{"0","",""};
lookup1("2910") ->	{"0","",""};
lookup1("2911") ->	{"0","",""};
lookup1("2912") ->	{"0","",""};
lookup1("2913") ->	{"0","",""};
lookup1("2914") ->	{"0","",""};
lookup1("2915") ->	{"0","",""};
lookup1("2916") ->	{"0","",""};
lookup1("2917") ->	{"0","",""};
lookup1("2918") ->	{"0","",""};
lookup1("2919") ->	{"0","",""};
lookup1("291A") ->	{"0","",""};
lookup1("291B") ->	{"0","",""};
lookup1("291C") ->	{"0","",""};
lookup1("291D") ->	{"0","",""};
lookup1("291E") ->	{"0","",""};
lookup1("291F") ->	{"0","",""};
lookup1("2920") ->	{"0","",""};
lookup1("2921") ->	{"0","",""};
lookup1("2922") ->	{"0","",""};
lookup1("2923") ->	{"0","",""};
lookup1("2924") ->	{"0","",""};
lookup1("2925") ->	{"0","",""};
lookup1("2926") ->	{"0","",""};
lookup1("2927") ->	{"0","",""};
lookup1("2928") ->	{"0","",""};
lookup1("2929") ->	{"0","",""};
lookup1("292A") ->	{"0","",""};
lookup1("292B") ->	{"0","",""};
lookup1("292C") ->	{"0","",""};
lookup1("292D") ->	{"0","",""};
lookup1("292E") ->	{"0","",""};
lookup1("292F") ->	{"0","",""};
lookup1("2930") ->	{"0","",""};
lookup1("2931") ->	{"0","",""};
lookup1("2932") ->	{"0","",""};
lookup1("2933") ->	{"0","",""};
lookup1("2934") ->	{"0","",""};
lookup1("2935") ->	{"0","",""};
lookup1("2936") ->	{"0","",""};
lookup1("2937") ->	{"0","",""};
lookup1("2938") ->	{"0","",""};
lookup1("2939") ->	{"0","",""};
lookup1("293A") ->	{"0","",""};
lookup1("293B") ->	{"0","",""};
lookup1("293C") ->	{"0","",""};
lookup1("293D") ->	{"0","",""};
lookup1("293E") ->	{"0","",""};
lookup1("293F") ->	{"0","",""};
lookup1("2940") ->	{"0","",""};
lookup1("2941") ->	{"0","",""};
lookup1("2942") ->	{"0","",""};
lookup1("2943") ->	{"0","",""};
lookup1("2944") ->	{"0","",""};
lookup1("2945") ->	{"0","",""};
lookup1("2946") ->	{"0","",""};
lookup1("2947") ->	{"0","",""};
lookup1("2948") ->	{"0","",""};
lookup1("2949") ->	{"0","",""};
lookup1("294A") ->	{"0","",""};
lookup1("294B") ->	{"0","",""};
lookup1("294C") ->	{"0","",""};
lookup1("294D") ->	{"0","",""};
lookup1("294E") ->	{"0","",""};
lookup1("294F") ->	{"0","",""};
lookup1("2950") ->	{"0","",""};
lookup1("2951") ->	{"0","",""};
lookup1("2952") ->	{"0","",""};
lookup1("2953") ->	{"0","",""};
lookup1("2954") ->	{"0","",""};
lookup1("2955") ->	{"0","",""};
lookup1("2956") ->	{"0","",""};
lookup1("2957") ->	{"0","",""};
lookup1("2958") ->	{"0","",""};
lookup1("2959") ->	{"0","",""};
lookup1("295A") ->	{"0","",""};
lookup1("295B") ->	{"0","",""};
lookup1("295C") ->	{"0","",""};
lookup1("295D") ->	{"0","",""};
lookup1("295E") ->	{"0","",""};
lookup1("295F") ->	{"0","",""};
lookup1("2960") ->	{"0","",""};
lookup1("2961") ->	{"0","",""};
lookup1("2962") ->	{"0","",""};
lookup1("2963") ->	{"0","",""};
lookup1("2964") ->	{"0","",""};
lookup1("2965") ->	{"0","",""};
lookup1("2966") ->	{"0","",""};
lookup1("2967") ->	{"0","",""};
lookup1("2968") ->	{"0","",""};
lookup1("2969") ->	{"0","",""};
lookup1("296A") ->	{"0","",""};
lookup1("296B") ->	{"0","",""};
lookup1("296C") ->	{"0","",""};
lookup1("296D") ->	{"0","",""};
lookup1("296E") ->	{"0","",""};
lookup1("296F") ->	{"0","",""};
lookup1("2970") ->	{"0","",""};
lookup1("2971") ->	{"0","",""};
lookup1("2972") ->	{"0","",""};
lookup1("2973") ->	{"0","",""};
lookup1("2974") ->	{"0","",""};
lookup1("2975") ->	{"0","",""};
lookup1("2976") ->	{"0","",""};
lookup1("2977") ->	{"0","",""};
lookup1("2978") ->	{"0","",""};
lookup1("2979") ->	{"0","",""};
lookup1("297A") ->	{"0","",""};
lookup1("297B") ->	{"0","",""};
lookup1("297C") ->	{"0","",""};
lookup1("297D") ->	{"0","",""};
lookup1("297E") ->	{"0","",""};
lookup1("297F") ->	{"0","",""};
lookup1("2980") ->	{"0","",""};
lookup1("2981") ->	{"0","",""};
lookup1("2982") ->	{"0","",""};
lookup1("2983") ->	{"0","",""};
lookup1("2984") ->	{"0","",""};
lookup1("2985") ->	{"0","",""};
lookup1("2986") ->	{"0","",""};
lookup1("2987") ->	{"0","",""};
lookup1("2988") ->	{"0","",""};
lookup1("2989") ->	{"0","",""};
lookup1("298A") ->	{"0","",""};
lookup1("298B") ->	{"0","",""};
lookup1("298C") ->	{"0","",""};
lookup1("298D") ->	{"0","",""};
lookup1("298E") ->	{"0","",""};
lookup1("298F") ->	{"0","",""};
lookup1("2990") ->	{"0","",""};
lookup1("2991") ->	{"0","",""};
lookup1("2992") ->	{"0","",""};
lookup1("2993") ->	{"0","",""};
lookup1("2994") ->	{"0","",""};
lookup1("2995") ->	{"0","",""};
lookup1("2996") ->	{"0","",""};
lookup1("2997") ->	{"0","",""};
lookup1("2998") ->	{"0","",""};
lookup1("2999") ->	{"0","",""};
lookup1("299A") ->	{"0","",""};
lookup1("299B") ->	{"0","",""};
lookup1("299C") ->	{"0","",""};
lookup1("299D") ->	{"0","",""};
lookup1("299E") ->	{"0","",""};
lookup1("299F") ->	{"0","",""};
lookup1("29A0") ->	{"0","",""};
lookup1("29A1") ->	{"0","",""};
lookup1("29A2") ->	{"0","",""};
lookup1("29A3") ->	{"0","",""};
lookup1("29A4") ->	{"0","",""};
lookup1("29A5") ->	{"0","",""};
lookup1("29A6") ->	{"0","",""};
lookup1("29A7") ->	{"0","",""};
lookup1("29A8") ->	{"0","",""};
lookup1("29A9") ->	{"0","",""};
lookup1("29AA") ->	{"0","",""};
lookup1("29AB") ->	{"0","",""};
lookup1("29AC") ->	{"0","",""};
lookup1("29AD") ->	{"0","",""};
lookup1("29AE") ->	{"0","",""};
lookup1("29AF") ->	{"0","",""};
lookup1("29B0") ->	{"0","",""};
lookup1("29B1") ->	{"0","",""};
lookup1("29B2") ->	{"0","",""};
lookup1("29B3") ->	{"0","",""};
lookup1("29B4") ->	{"0","",""};
lookup1("29B5") ->	{"0","",""};
lookup1("29B6") ->	{"0","",""};
lookup1("29B7") ->	{"0","",""};
lookup1("29B8") ->	{"0","",""};
lookup1("29B9") ->	{"0","",""};
lookup1("29BA") ->	{"0","",""};
lookup1("29BB") ->	{"0","",""};
lookup1("29BC") ->	{"0","",""};
lookup1("29BD") ->	{"0","",""};
lookup1("29BE") ->	{"0","",""};
lookup1("29BF") ->	{"0","",""};
lookup1("29C0") ->	{"0","",""};
lookup1("29C1") ->	{"0","",""};
lookup1("29C2") ->	{"0","",""};
lookup1("29C3") ->	{"0","",""};
lookup1("29C4") ->	{"0","",""};
lookup1("29C5") ->	{"0","",""};
lookup1("29C6") ->	{"0","",""};
lookup1("29C7") ->	{"0","",""};
lookup1("29C8") ->	{"0","",""};
lookup1("29C9") ->	{"0","",""};
lookup1("29CA") ->	{"0","",""};
lookup1("29CB") ->	{"0","",""};
lookup1("29CC") ->	{"0","",""};
lookup1("29CD") ->	{"0","",""};
lookup1("29CE") ->	{"0","",""};
lookup1("29CF") ->	{"0","",""};
lookup1("29D0") ->	{"0","",""};
lookup1("29D1") ->	{"0","",""};
lookup1("29D2") ->	{"0","",""};
lookup1("29D3") ->	{"0","",""};
lookup1("29D4") ->	{"0","",""};
lookup1("29D5") ->	{"0","",""};
lookup1("29D6") ->	{"0","",""};
lookup1("29D7") ->	{"0","",""};
lookup1("29D8") ->	{"0","",""};
lookup1("29D9") ->	{"0","",""};
lookup1("29DA") ->	{"0","",""};
lookup1("29DB") ->	{"0","",""};
lookup1("29DC") ->	{"0","",""};
lookup1("29DD") ->	{"0","",""};
lookup1("29DE") ->	{"0","",""};
lookup1("29DF") ->	{"0","",""};
lookup1("29E0") ->	{"0","",""};
lookup1("29E1") ->	{"0","",""};
lookup1("29E2") ->	{"0","",""};
lookup1("29E3") ->	{"0","",""};
lookup1("29E4") ->	{"0","",""};
lookup1("29E5") ->	{"0","",""};
lookup1("29E6") ->	{"0","",""};
lookup1("29E7") ->	{"0","",""};
lookup1("29E8") ->	{"0","",""};
lookup1("29E9") ->	{"0","",""};
lookup1("29EA") ->	{"0","",""};
lookup1("29EB") ->	{"0","",""};
lookup1("29EC") ->	{"0","",""};
lookup1("29ED") ->	{"0","",""};
lookup1("29EE") ->	{"0","",""};
lookup1("29EF") ->	{"0","",""};
lookup1("29F0") ->	{"0","",""};
lookup1("29F1") ->	{"0","",""};
lookup1("29F2") ->	{"0","",""};
lookup1("29F3") ->	{"0","",""};
lookup1("29F4") ->	{"0","",""};
lookup1("29F5") ->	{"0","",""};
lookup1("29F6") ->	{"0","",""};
lookup1("29F7") ->	{"0","",""};
lookup1("29F8") ->	{"0","",""};
lookup1("29F9") ->	{"0","",""};
lookup1("29FA") ->	{"0","",""};
lookup1("29FB") ->	{"0","",""};
lookup1("29FC") ->	{"0","",""};
lookup1("29FD") ->	{"0","",""};
lookup1("29FE") ->	{"0","",""};
lookup1("29FF") ->	{"0","",""};
lookup1("2A00") ->	{"0","",""};
lookup1("2A01") ->	{"0","",""};
lookup1("2A02") ->	{"0","",""};
lookup1("2A03") ->	{"0","",""};
lookup1("2A04") ->	{"0","",""};
lookup1("2A05") ->	{"0","",""};
lookup1("2A06") ->	{"0","",""};
lookup1("2A07") ->	{"0","",""};
lookup1("2A08") ->	{"0","",""};
lookup1("2A09") ->	{"0","",""};
lookup1("2A0A") ->	{"0","",""};
lookup1("2A0B") ->	{"0","",""};
lookup1("2A0C") ->	{"0","<compat> 222B 222B 222B 222B",""};
lookup1("2A0D") ->	{"0","",""};
lookup1("2A0E") ->	{"0","",""};
lookup1("2A0F") ->	{"0","",""};
lookup1("2A10") ->	{"0","",""};
lookup1("2A11") ->	{"0","",""};
lookup1("2A12") ->	{"0","",""};
lookup1("2A13") ->	{"0","",""};
lookup1("2A14") ->	{"0","",""};
lookup1("2A15") ->	{"0","",""};
lookup1("2A16") ->	{"0","",""};
lookup1("2A17") ->	{"0","",""};
lookup1("2A18") ->	{"0","",""};
lookup1("2A19") ->	{"0","",""};
lookup1("2A1A") ->	{"0","",""};
lookup1("2A1B") ->	{"0","",""};
lookup1("2A1C") ->	{"0","",""};
lookup1("2A1D") ->	{"0","",""};
lookup1("2A1E") ->	{"0","",""};
lookup1("2A1F") ->	{"0","",""};
lookup1("2A20") ->	{"0","",""};
lookup1("2A21") ->	{"0","",""};
lookup1("2A22") ->	{"0","",""};
lookup1("2A23") ->	{"0","",""};
lookup1("2A24") ->	{"0","",""};
lookup1("2A25") ->	{"0","",""};
lookup1("2A26") ->	{"0","",""};
lookup1("2A27") ->	{"0","",""};
lookup1("2A28") ->	{"0","",""};
lookup1("2A29") ->	{"0","",""};
lookup1("2A2A") ->	{"0","",""};
lookup1("2A2B") ->	{"0","",""};
lookup1("2A2C") ->	{"0","",""};
lookup1("2A2D") ->	{"0","",""};
lookup1("2A2E") ->	{"0","",""};
lookup1("2A2F") ->	{"0","",""};
lookup1("2A30") ->	{"0","",""};
lookup1("2A31") ->	{"0","",""};
lookup1("2A32") ->	{"0","",""};
lookup1("2A33") ->	{"0","",""};
lookup1("2A34") ->	{"0","",""};
lookup1("2A35") ->	{"0","",""};
lookup1("2A36") ->	{"0","",""};
lookup1("2A37") ->	{"0","",""};
lookup1("2A38") ->	{"0","",""};
lookup1("2A39") ->	{"0","",""};
lookup1("2A3A") ->	{"0","",""};
lookup1("2A3B") ->	{"0","",""};
lookup1("2A3C") ->	{"0","",""};
lookup1("2A3D") ->	{"0","",""};
lookup1("2A3E") ->	{"0","",""};
lookup1("2A3F") ->	{"0","",""};
lookup1("2A40") ->	{"0","",""};
lookup1("2A41") ->	{"0","",""};
lookup1("2A42") ->	{"0","",""};
lookup1("2A43") ->	{"0","",""};
lookup1("2A44") ->	{"0","",""};
lookup1("2A45") ->	{"0","",""};
lookup1("2A46") ->	{"0","",""};
lookup1("2A47") ->	{"0","",""};
lookup1("2A48") ->	{"0","",""};
lookup1("2A49") ->	{"0","",""};
lookup1("2A4A") ->	{"0","",""};
lookup1("2A4B") ->	{"0","",""};
lookup1("2A4C") ->	{"0","",""};
lookup1("2A4D") ->	{"0","",""};
lookup1("2A4E") ->	{"0","",""};
lookup1("2A4F") ->	{"0","",""};
lookup1("2A50") ->	{"0","",""};
lookup1("2A51") ->	{"0","",""};
lookup1("2A52") ->	{"0","",""};
lookup1("2A53") ->	{"0","",""};
lookup1("2A54") ->	{"0","",""};
lookup1("2A55") ->	{"0","",""};
lookup1("2A56") ->	{"0","",""};
lookup1("2A57") ->	{"0","",""};
lookup1("2A58") ->	{"0","",""};
lookup1("2A59") ->	{"0","",""};
lookup1("2A5A") ->	{"0","",""};
lookup1("2A5B") ->	{"0","",""};
lookup1("2A5C") ->	{"0","",""};
lookup1("2A5D") ->	{"0","",""};
lookup1("2A5E") ->	{"0","",""};
lookup1("2A5F") ->	{"0","",""};
lookup1("2A60") ->	{"0","",""};
lookup1("2A61") ->	{"0","",""};
lookup1("2A62") ->	{"0","",""};
lookup1("2A63") ->	{"0","",""};
lookup1("2A64") ->	{"0","",""};
lookup1("2A65") ->	{"0","",""};
lookup1("2A66") ->	{"0","",""};
lookup1("2A67") ->	{"0","",""};
lookup1("2A68") ->	{"0","",""};
lookup1("2A69") ->	{"0","",""};
lookup1("2A6A") ->	{"0","",""};
lookup1("2A6B") ->	{"0","",""};
lookup1("2A6C") ->	{"0","",""};
lookup1("2A6D") ->	{"0","",""};
lookup1("2A6D6") ->	{"0","",""};
lookup1("2A6E") ->	{"0","",""};
lookup1("2A6F") ->	{"0","",""};
lookup1("2A70") ->	{"0","",""};
lookup1("2A700") ->	{"0","",""};
lookup1("2A71") ->	{"0","",""};
lookup1("2A72") ->	{"0","",""};
lookup1("2A73") ->	{"0","",""};
lookup1("2A74") ->	{"0","<compat> 003A 003A 003D",""};
lookup1("2A75") ->	{"0","<compat> 003D 003D",""};
lookup1("2A76") ->	{"0","<compat> 003D 003D 003D",""};
lookup1("2A77") ->	{"0","",""};
lookup1("2A78") ->	{"0","",""};
lookup1("2A79") ->	{"0","",""};
lookup1("2A7A") ->	{"0","",""};
lookup1("2A7B") ->	{"0","",""};
lookup1("2A7C") ->	{"0","",""};
lookup1("2A7D") ->	{"0","",""};
lookup1("2A7E") ->	{"0","",""};
lookup1("2A7F") ->	{"0","",""};
lookup1("2A80") ->	{"0","",""};
lookup1("2A81") ->	{"0","",""};
lookup1("2A82") ->	{"0","",""};
lookup1("2A83") ->	{"0","",""};
lookup1("2A84") ->	{"0","",""};
lookup1("2A85") ->	{"0","",""};
lookup1("2A86") ->	{"0","",""};
lookup1("2A87") ->	{"0","",""};
lookup1("2A88") ->	{"0","",""};
lookup1("2A89") ->	{"0","",""};
lookup1("2A8A") ->	{"0","",""};
lookup1("2A8B") ->	{"0","",""};
lookup1("2A8C") ->	{"0","",""};
lookup1("2A8D") ->	{"0","",""};
lookup1("2A8E") ->	{"0","",""};
lookup1("2A8F") ->	{"0","",""};
lookup1("2A90") ->	{"0","",""};
lookup1("2A91") ->	{"0","",""};
lookup1("2A92") ->	{"0","",""};
lookup1("2A93") ->	{"0","",""};
lookup1("2A94") ->	{"0","",""};
lookup1("2A95") ->	{"0","",""};
lookup1("2A96") ->	{"0","",""};
lookup1("2A97") ->	{"0","",""};
lookup1("2A98") ->	{"0","",""};
lookup1("2A99") ->	{"0","",""};
lookup1("2A9A") ->	{"0","",""};
lookup1("2A9B") ->	{"0","",""};
lookup1("2A9C") ->	{"0","",""};
lookup1("2A9D") ->	{"0","",""};
lookup1("2A9E") ->	{"0","",""};
lookup1("2A9F") ->	{"0","",""};
lookup1("2AA0") ->	{"0","",""};
lookup1("2AA1") ->	{"0","",""};
lookup1("2AA2") ->	{"0","",""};
lookup1("2AA3") ->	{"0","",""};
lookup1("2AA4") ->	{"0","",""};
lookup1("2AA5") ->	{"0","",""};
lookup1("2AA6") ->	{"0","",""};
lookup1("2AA7") ->	{"0","",""};
lookup1("2AA8") ->	{"0","",""};
lookup1("2AA9") ->	{"0","",""};
lookup1("2AAA") ->	{"0","",""};
lookup1("2AAB") ->	{"0","",""};
lookup1("2AAC") ->	{"0","",""};
lookup1("2AAD") ->	{"0","",""};
lookup1("2AAE") ->	{"0","",""};
lookup1("2AAF") ->	{"0","",""};
lookup1("2AB0") ->	{"0","",""};
lookup1("2AB1") ->	{"0","",""};
lookup1("2AB2") ->	{"0","",""};
lookup1("2AB3") ->	{"0","",""};
lookup1("2AB4") ->	{"0","",""};
lookup1("2AB5") ->	{"0","",""};
lookup1("2AB6") ->	{"0","",""};
lookup1("2AB7") ->	{"0","",""};
lookup1("2AB8") ->	{"0","",""};
lookup1("2AB9") ->	{"0","",""};
lookup1("2ABA") ->	{"0","",""};
lookup1("2ABB") ->	{"0","",""};
lookup1("2ABC") ->	{"0","",""};
lookup1("2ABD") ->	{"0","",""};
lookup1("2ABE") ->	{"0","",""};
lookup1("2ABF") ->	{"0","",""};
lookup1("2AC0") ->	{"0","",""};
lookup1("2AC1") ->	{"0","",""};
lookup1("2AC2") ->	{"0","",""};
lookup1("2AC3") ->	{"0","",""};
lookup1("2AC4") ->	{"0","",""};
lookup1("2AC5") ->	{"0","",""};
lookup1("2AC6") ->	{"0","",""};
lookup1("2AC7") ->	{"0","",""};
lookup1("2AC8") ->	{"0","",""};
lookup1("2AC9") ->	{"0","",""};
lookup1("2ACA") ->	{"0","",""};
lookup1("2ACB") ->	{"0","",""};
lookup1("2ACC") ->	{"0","",""};
lookup1("2ACD") ->	{"0","",""};
lookup1("2ACE") ->	{"0","",""};
lookup1("2ACF") ->	{"0","",""};
lookup1("2AD0") ->	{"0","",""};
lookup1("2AD1") ->	{"0","",""};
lookup1("2AD2") ->	{"0","",""};
lookup1("2AD3") ->	{"0","",""};
lookup1("2AD4") ->	{"0","",""};
lookup1("2AD5") ->	{"0","",""};
lookup1("2AD6") ->	{"0","",""};
lookup1("2AD7") ->	{"0","",""};
lookup1("2AD8") ->	{"0","",""};
lookup1("2AD9") ->	{"0","",""};
lookup1("2ADA") ->	{"0","",""};
lookup1("2ADB") ->	{"0","",""};
lookup1("2ADC") ->	{"0","2ADD 0338",""};
lookup1("2ADD") ->	{"0","",""};
lookup1("2ADE") ->	{"0","",""};
lookup1("2ADF") ->	{"0","",""};
lookup1("2AE0") ->	{"0","",""};
lookup1("2AE1") ->	{"0","",""};
lookup1("2AE2") ->	{"0","",""};
lookup1("2AE3") ->	{"0","",""};
lookup1("2AE4") ->	{"0","",""};
lookup1("2AE5") ->	{"0","",""};
lookup1("2AE6") ->	{"0","",""};
lookup1("2AE7") ->	{"0","",""};
lookup1("2AE8") ->	{"0","",""};
lookup1("2AE9") ->	{"0","",""};
lookup1("2AEA") ->	{"0","",""};
lookup1("2AEB") ->	{"0","",""};
lookup1("2AEC") ->	{"0","",""};
lookup1("2AED") ->	{"0","",""};
lookup1("2AEE") ->	{"0","",""};
lookup1("2AEF") ->	{"0","",""};
lookup1("2AF0") ->	{"0","",""};
lookup1("2AF1") ->	{"0","",""};
lookup1("2AF2") ->	{"0","",""};
lookup1("2AF3") ->	{"0","",""};
lookup1("2AF4") ->	{"0","",""};
lookup1("2AF5") ->	{"0","",""};
lookup1("2AF6") ->	{"0","",""};
lookup1("2AF7") ->	{"0","",""};
lookup1("2AF8") ->	{"0","",""};
lookup1("2AF9") ->	{"0","",""};
lookup1("2AFA") ->	{"0","",""};
lookup1("2AFB") ->	{"0","",""};
lookup1("2AFC") ->	{"0","",""};
lookup1("2AFD") ->	{"0","",""};
lookup1("2AFE") ->	{"0","",""};
lookup1("2AFF") ->	{"0","",""};
lookup1("2B00") ->	{"0","",""};
lookup1("2B01") ->	{"0","",""};
lookup1("2B02") ->	{"0","",""};
lookup1("2B03") ->	{"0","",""};
lookup1("2B04") ->	{"0","",""};
lookup1("2B05") ->	{"0","",""};
lookup1("2B06") ->	{"0","",""};
lookup1("2B07") ->	{"0","",""};
lookup1("2B08") ->	{"0","",""};
lookup1("2B09") ->	{"0","",""};
lookup1("2B0A") ->	{"0","",""};
lookup1("2B0B") ->	{"0","",""};
lookup1("2B0C") ->	{"0","",""};
lookup1("2B0D") ->	{"0","",""};
lookup1("2B0E") ->	{"0","",""};
lookup1("2B0F") ->	{"0","",""};
lookup1("2B10") ->	{"0","",""};
lookup1("2B11") ->	{"0","",""};
lookup1("2B12") ->	{"0","",""};
lookup1("2B13") ->	{"0","",""};
lookup1("2B14") ->	{"0","",""};
lookup1("2B15") ->	{"0","",""};
lookup1("2B16") ->	{"0","",""};
lookup1("2B17") ->	{"0","",""};
lookup1("2B18") ->	{"0","",""};
lookup1("2B19") ->	{"0","",""};
lookup1("2B1A") ->	{"0","",""};
lookup1("2B1B") ->	{"0","",""};
lookup1("2B1C") ->	{"0","",""};
lookup1("2B1D") ->	{"0","",""};
lookup1("2B1E") ->	{"0","",""};
lookup1("2B1F") ->	{"0","",""};
lookup1("2B20") ->	{"0","",""};
lookup1("2B21") ->	{"0","",""};
lookup1("2B22") ->	{"0","",""};
lookup1("2B23") ->	{"0","",""};
lookup1("2B24") ->	{"0","",""};
lookup1("2B25") ->	{"0","",""};
lookup1("2B26") ->	{"0","",""};
lookup1("2B27") ->	{"0","",""};
lookup1("2B28") ->	{"0","",""};
lookup1("2B29") ->	{"0","",""};
lookup1("2B2A") ->	{"0","",""};
lookup1("2B2B") ->	{"0","",""};
lookup1("2B2C") ->	{"0","",""};
lookup1("2B2D") ->	{"0","",""};
lookup1("2B2E") ->	{"0","",""};
lookup1("2B2F") ->	{"0","",""};
lookup1("2B30") ->	{"0","",""};
lookup1("2B31") ->	{"0","",""};
lookup1("2B32") ->	{"0","",""};
lookup1("2B33") ->	{"0","",""};
lookup1("2B34") ->	{"0","",""};
lookup1("2B35") ->	{"0","",""};
lookup1("2B36") ->	{"0","",""};
lookup1("2B37") ->	{"0","",""};
lookup1("2B38") ->	{"0","",""};
lookup1("2B39") ->	{"0","",""};
lookup1("2B3A") ->	{"0","",""};
lookup1("2B3B") ->	{"0","",""};
lookup1("2B3C") ->	{"0","",""};
lookup1("2B3D") ->	{"0","",""};
lookup1("2B3E") ->	{"0","",""};
lookup1("2B3F") ->	{"0","",""};
lookup1("2B40") ->	{"0","",""};
lookup1("2B41") ->	{"0","",""};
lookup1("2B42") ->	{"0","",""};
lookup1("2B43") ->	{"0","",""};
lookup1("2B44") ->	{"0","",""};
lookup1("2B45") ->	{"0","",""};
lookup1("2B46") ->	{"0","",""};
lookup1("2B47") ->	{"0","",""};
lookup1("2B48") ->	{"0","",""};
lookup1("2B49") ->	{"0","",""};
lookup1("2B4A") ->	{"0","",""};
lookup1("2B4B") ->	{"0","",""};
lookup1("2B4C") ->	{"0","",""};
lookup1("2B4D") ->	{"0","",""};
lookup1("2B4E") ->	{"0","",""};
lookup1("2B4F") ->	{"0","",""};
lookup1("2B50") ->	{"0","",""};
lookup1("2B51") ->	{"0","",""};
lookup1("2B52") ->	{"0","",""};
lookup1("2B53") ->	{"0","",""};
lookup1("2B54") ->	{"0","",""};
lookup1("2B55") ->	{"0","",""};
lookup1("2B56") ->	{"0","",""};
lookup1("2B57") ->	{"0","",""};
lookup1("2B58") ->	{"0","",""};
lookup1("2B59") ->	{"0","",""};
lookup1("2B5A") ->	{"0","",""};
lookup1("2B5B") ->	{"0","",""};
lookup1("2B5C") ->	{"0","",""};
lookup1("2B5D") ->	{"0","",""};
lookup1("2B5E") ->	{"0","",""};
lookup1("2B5F") ->	{"0","",""};
lookup1("2B60") ->	{"0","",""};
lookup1("2B61") ->	{"0","",""};
lookup1("2B62") ->	{"0","",""};
lookup1("2B63") ->	{"0","",""};
lookup1("2B64") ->	{"0","",""};
lookup1("2B65") ->	{"0","",""};
lookup1("2B66") ->	{"0","",""};
lookup1("2B67") ->	{"0","",""};
lookup1("2B68") ->	{"0","",""};
lookup1("2B69") ->	{"0","",""};
lookup1("2B6A") ->	{"0","",""};
lookup1("2B6B") ->	{"0","",""};
lookup1("2B6C") ->	{"0","",""};
lookup1("2B6D") ->	{"0","",""};
lookup1("2B6E") ->	{"0","",""};
lookup1("2B6F") ->	{"0","",""};
lookup1("2B70") ->	{"0","",""};
lookup1("2B71") ->	{"0","",""};
lookup1("2B72") ->	{"0","",""};
lookup1("2B73") ->	{"0","",""};
lookup1("2B734") ->	{"0","",""};
lookup1("2B740") ->	{"0","",""};
lookup1("2B76") ->	{"0","",""};
lookup1("2B77") ->	{"0","",""};
lookup1("2B78") ->	{"0","",""};
lookup1("2B79") ->	{"0","",""};
lookup1("2B7A") ->	{"0","",""};
lookup1("2B7B") ->	{"0","",""};
lookup1("2B7C") ->	{"0","",""};
lookup1("2B7D") ->	{"0","",""};
lookup1("2B7E") ->	{"0","",""};
lookup1("2B7F") ->	{"0","",""};
lookup1("2B80") ->	{"0","",""};
lookup1("2B81") ->	{"0","",""};
lookup1("2B81D") ->	{"0","",""};
lookup1("2B82") ->	{"0","",""};
lookup1("2B83") ->	{"0","",""};
lookup1("2B84") ->	{"0","",""};
lookup1("2B85") ->	{"0","",""};
lookup1("2B86") ->	{"0","",""};
lookup1("2B87") ->	{"0","",""};
lookup1("2B88") ->	{"0","",""};
lookup1("2B89") ->	{"0","",""};
lookup1("2B8A") ->	{"0","",""};
lookup1("2B8B") ->	{"0","",""};
lookup1("2B8C") ->	{"0","",""};
lookup1("2B8D") ->	{"0","",""};
lookup1("2B8E") ->	{"0","",""};
lookup1("2B8F") ->	{"0","",""};
lookup1("2B90") ->	{"0","",""};
lookup1("2B91") ->	{"0","",""};
lookup1("2B92") ->	{"0","",""};
lookup1("2B93") ->	{"0","",""};
lookup1("2B94") ->	{"0","",""};
lookup1("2B95") ->	{"0","",""};
lookup1("2B98") ->	{"0","",""};
lookup1("2B99") ->	{"0","",""};
lookup1("2B9A") ->	{"0","",""};
lookup1("2B9B") ->	{"0","",""};
lookup1("2B9C") ->	{"0","",""};
lookup1("2B9D") ->	{"0","",""};
lookup1("2B9E") ->	{"0","",""};
lookup1("2B9F") ->	{"0","",""};
lookup1("2BA0") ->	{"0","",""};
lookup1("2BA1") ->	{"0","",""};
lookup1("2BA2") ->	{"0","",""};
lookup1("2BA3") ->	{"0","",""};
lookup1("2BA4") ->	{"0","",""};
lookup1("2BA5") ->	{"0","",""};
lookup1("2BA6") ->	{"0","",""};
lookup1("2BA7") ->	{"0","",""};
lookup1("2BA8") ->	{"0","",""};
lookup1("2BA9") ->	{"0","",""};
lookup1("2BAA") ->	{"0","",""};
lookup1("2BAB") ->	{"0","",""};
lookup1("2BAC") ->	{"0","",""};
lookup1("2BAD") ->	{"0","",""};
lookup1("2BAE") ->	{"0","",""};
lookup1("2BAF") ->	{"0","",""};
lookup1("2BB0") ->	{"0","",""};
lookup1("2BB1") ->	{"0","",""};
lookup1("2BB2") ->	{"0","",""};
lookup1("2BB3") ->	{"0","",""};
lookup1("2BB4") ->	{"0","",""};
lookup1("2BB5") ->	{"0","",""};
lookup1("2BB6") ->	{"0","",""};
lookup1("2BB7") ->	{"0","",""};
lookup1("2BB8") ->	{"0","",""};
lookup1("2BB9") ->	{"0","",""};
lookup1("2BBD") ->	{"0","",""};
lookup1("2BBE") ->	{"0","",""};
lookup1("2BBF") ->	{"0","",""};
lookup1("2BC0") ->	{"0","",""};
lookup1("2BC1") ->	{"0","",""};
lookup1("2BC2") ->	{"0","",""};
lookup1("2BC3") ->	{"0","",""};
lookup1("2BC4") ->	{"0","",""};
lookup1("2BC5") ->	{"0","",""};
lookup1("2BC6") ->	{"0","",""};
lookup1("2BC7") ->	{"0","",""};
lookup1("2BC8") ->	{"0","",""};
lookup1("2BCA") ->	{"0","",""};
lookup1("2BCB") ->	{"0","",""};
lookup1("2BCC") ->	{"0","",""};
lookup1("2BCD") ->	{"0","",""};
lookup1("2BCE") ->	{"0","",""};
lookup1("2BCF") ->	{"0","",""};
lookup1("2BD0") ->	{"0","",""};
lookup1("2BD1") ->	{"0","",""};
lookup1("2C00") ->	{"0","","2C30"};
lookup1("2C01") ->	{"0","","2C31"};
lookup1("2C02") ->	{"0","","2C32"};
lookup1("2C03") ->	{"0","","2C33"};
lookup1("2C04") ->	{"0","","2C34"};
lookup1("2C05") ->	{"0","","2C35"};
lookup1("2C06") ->	{"0","","2C36"};
lookup1("2C07") ->	{"0","","2C37"};
lookup1("2C08") ->	{"0","","2C38"};
lookup1("2C09") ->	{"0","","2C39"};
lookup1("2C0A") ->	{"0","","2C3A"};
lookup1("2C0B") ->	{"0","","2C3B"};
lookup1("2C0C") ->	{"0","","2C3C"};
lookup1("2C0D") ->	{"0","","2C3D"};
lookup1("2C0E") ->	{"0","","2C3E"};
lookup1("2C0F") ->	{"0","","2C3F"};
lookup1("2C10") ->	{"0","","2C40"};
lookup1("2C11") ->	{"0","","2C41"};
lookup1("2C12") ->	{"0","","2C42"};
lookup1("2C13") ->	{"0","","2C43"};
lookup1("2C14") ->	{"0","","2C44"};
lookup1("2C15") ->	{"0","","2C45"};
lookup1("2C16") ->	{"0","","2C46"};
lookup1("2C17") ->	{"0","","2C47"};
lookup1("2C18") ->	{"0","","2C48"};
lookup1("2C19") ->	{"0","","2C49"};
lookup1("2C1A") ->	{"0","","2C4A"};
lookup1("2C1B") ->	{"0","","2C4B"};
lookup1("2C1C") ->	{"0","","2C4C"};
lookup1("2C1D") ->	{"0","","2C4D"};
lookup1("2C1E") ->	{"0","","2C4E"};
lookup1("2C1F") ->	{"0","","2C4F"};
lookup1("2C20") ->	{"0","","2C50"};
lookup1("2C21") ->	{"0","","2C51"};
lookup1("2C22") ->	{"0","","2C52"};
lookup1("2C23") ->	{"0","","2C53"};
lookup1("2C24") ->	{"0","","2C54"};
lookup1("2C25") ->	{"0","","2C55"};
lookup1("2C26") ->	{"0","","2C56"};
lookup1("2C27") ->	{"0","","2C57"};
lookup1("2C28") ->	{"0","","2C58"};
lookup1("2C29") ->	{"0","","2C59"};
lookup1("2C2A") ->	{"0","","2C5A"};
lookup1("2C2B") ->	{"0","","2C5B"};
lookup1("2C2C") ->	{"0","","2C5C"};
lookup1("2C2D") ->	{"0","","2C5D"};
lookup1("2C2E") ->	{"0","","2C5E"};
lookup1("2C30") ->	{"0","",""};
lookup1("2C31") ->	{"0","",""};
lookup1("2C32") ->	{"0","",""};
lookup1("2C33") ->	{"0","",""};
lookup1("2C34") ->	{"0","",""};
lookup1("2C35") ->	{"0","",""};
lookup1("2C36") ->	{"0","",""};
lookup1("2C37") ->	{"0","",""};
lookup1("2C38") ->	{"0","",""};
lookup1("2C39") ->	{"0","",""};
lookup1("2C3A") ->	{"0","",""};
lookup1("2C3B") ->	{"0","",""};
lookup1("2C3C") ->	{"0","",""};
lookup1("2C3D") ->	{"0","",""};
lookup1("2C3E") ->	{"0","",""};
lookup1("2C3F") ->	{"0","",""};
lookup1("2C40") ->	{"0","",""};
lookup1("2C41") ->	{"0","",""};
lookup1("2C42") ->	{"0","",""};
lookup1("2C43") ->	{"0","",""};
lookup1("2C44") ->	{"0","",""};
lookup1("2C45") ->	{"0","",""};
lookup1("2C46") ->	{"0","",""};
lookup1("2C47") ->	{"0","",""};
lookup1("2C48") ->	{"0","",""};
lookup1("2C49") ->	{"0","",""};
lookup1("2C4A") ->	{"0","",""};
lookup1("2C4B") ->	{"0","",""};
lookup1("2C4C") ->	{"0","",""};
lookup1("2C4D") ->	{"0","",""};
lookup1("2C4E") ->	{"0","",""};
lookup1("2C4F") ->	{"0","",""};
lookup1("2C50") ->	{"0","",""};
lookup1("2C51") ->	{"0","",""};
lookup1("2C52") ->	{"0","",""};
lookup1("2C53") ->	{"0","",""};
lookup1("2C54") ->	{"0","",""};
lookup1("2C55") ->	{"0","",""};
lookup1("2C56") ->	{"0","",""};
lookup1("2C57") ->	{"0","",""};
lookup1("2C58") ->	{"0","",""};
lookup1("2C59") ->	{"0","",""};
lookup1("2C5A") ->	{"0","",""};
lookup1("2C5B") ->	{"0","",""};
lookup1("2C5C") ->	{"0","",""};
lookup1("2C5D") ->	{"0","",""};
lookup1("2C5E") ->	{"0","",""};
lookup1("2C60") ->	{"0","","2C61"};
lookup1("2C61") ->	{"0","",""};
lookup1("2C62") ->	{"0","","026B"};
lookup1("2C63") ->	{"0","","1D7D"};
lookup1("2C64") ->	{"0","","027D"};
lookup1("2C65") ->	{"0","",""};
lookup1("2C66") ->	{"0","",""};
lookup1("2C67") ->	{"0","","2C68"};
lookup1("2C68") ->	{"0","",""};
lookup1("2C69") ->	{"0","","2C6A"};
lookup1("2C6A") ->	{"0","",""};
lookup1("2C6B") ->	{"0","","2C6C"};
lookup1("2C6C") ->	{"0","",""};
lookup1("2C6D") ->	{"0","","0251"};
lookup1("2C6E") ->	{"0","","0271"};
lookup1("2C6F") ->	{"0","","0250"};
lookup1("2C70") ->	{"0","","0252"};
lookup1("2C71") ->	{"0","",""};
lookup1("2C72") ->	{"0","","2C73"};
lookup1("2C73") ->	{"0","",""};
lookup1("2C74") ->	{"0","",""};
lookup1("2C75") ->	{"0","","2C76"};
lookup1("2C76") ->	{"0","",""};
lookup1("2C77") ->	{"0","",""};
lookup1("2C78") ->	{"0","",""};
lookup1("2C79") ->	{"0","",""};
lookup1("2C7A") ->	{"0","",""};
lookup1("2C7B") ->	{"0","",""};
lookup1("2C7C") ->	{"0","<sub> 006A",""};
lookup1("2C7D") ->	{"0","<super> 0056",""};
lookup1("2C7E") ->	{"0","","023F"};
lookup1("2C7F") ->	{"0","","0240"};
lookup1("2C80") ->	{"0","","2C81"};
lookup1("2C81") ->	{"0","",""};
lookup1("2C82") ->	{"0","","2C83"};
lookup1("2C83") ->	{"0","",""};
lookup1("2C84") ->	{"0","","2C85"};
lookup1("2C85") ->	{"0","",""};
lookup1("2C86") ->	{"0","","2C87"};
lookup1("2C87") ->	{"0","",""};
lookup1("2C88") ->	{"0","","2C89"};
lookup1("2C89") ->	{"0","",""};
lookup1("2C8A") ->	{"0","","2C8B"};
lookup1("2C8B") ->	{"0","",""};
lookup1("2C8C") ->	{"0","","2C8D"};
lookup1("2C8D") ->	{"0","",""};
lookup1("2C8E") ->	{"0","","2C8F"};
lookup1("2C8F") ->	{"0","",""};
lookup1("2C90") ->	{"0","","2C91"};
lookup1("2C91") ->	{"0","",""};
lookup1("2C92") ->	{"0","","2C93"};
lookup1("2C93") ->	{"0","",""};
lookup1("2C94") ->	{"0","","2C95"};
lookup1("2C95") ->	{"0","",""};
lookup1("2C96") ->	{"0","","2C97"};
lookup1("2C97") ->	{"0","",""};
lookup1("2C98") ->	{"0","","2C99"};
lookup1("2C99") ->	{"0","",""};
lookup1("2C9A") ->	{"0","","2C9B"};
lookup1("2C9B") ->	{"0","",""};
lookup1("2C9C") ->	{"0","","2C9D"};
lookup1("2C9D") ->	{"0","",""};
lookup1("2C9E") ->	{"0","","2C9F"};
lookup1("2C9F") ->	{"0","",""};
lookup1("2CA0") ->	{"0","","2CA1"};
lookup1("2CA1") ->	{"0","",""};
lookup1("2CA2") ->	{"0","","2CA3"};
lookup1("2CA3") ->	{"0","",""};
lookup1("2CA4") ->	{"0","","2CA5"};
lookup1("2CA5") ->	{"0","",""};
lookup1("2CA6") ->	{"0","","2CA7"};
lookup1("2CA7") ->	{"0","",""};
lookup1("2CA8") ->	{"0","","2CA9"};
lookup1("2CA9") ->	{"0","",""};
lookup1("2CAA") ->	{"0","","2CAB"};
lookup1("2CAB") ->	{"0","",""};
lookup1("2CAC") ->	{"0","","2CAD"};
lookup1("2CAD") ->	{"0","",""};
lookup1("2CAE") ->	{"0","","2CAF"};
lookup1("2CAF") ->	{"0","",""};
lookup1("2CB0") ->	{"0","","2CB1"};
lookup1("2CB1") ->	{"0","",""};
lookup1("2CB2") ->	{"0","","2CB3"};
lookup1("2CB3") ->	{"0","",""};
lookup1("2CB4") ->	{"0","","2CB5"};
lookup1("2CB5") ->	{"0","",""};
lookup1("2CB6") ->	{"0","","2CB7"};
lookup1("2CB7") ->	{"0","",""};
lookup1("2CB8") ->	{"0","","2CB9"};
lookup1("2CB9") ->	{"0","",""};
lookup1("2CBA") ->	{"0","","2CBB"};
lookup1("2CBB") ->	{"0","",""};
lookup1("2CBC") ->	{"0","","2CBD"};
lookup1("2CBD") ->	{"0","",""};
lookup1("2CBE") ->	{"0","","2CBF"};
lookup1("2CBF") ->	{"0","",""};
lookup1("2CC0") ->	{"0","","2CC1"};
lookup1("2CC1") ->	{"0","",""};
lookup1("2CC2") ->	{"0","","2CC3"};
lookup1("2CC3") ->	{"0","",""};
lookup1("2CC4") ->	{"0","","2CC5"};
lookup1("2CC5") ->	{"0","",""};
lookup1("2CC6") ->	{"0","","2CC7"};
lookup1("2CC7") ->	{"0","",""};
lookup1("2CC8") ->	{"0","","2CC9"};
lookup1("2CC9") ->	{"0","",""};
lookup1("2CCA") ->	{"0","","2CCB"};
lookup1("2CCB") ->	{"0","",""};
lookup1("2CCC") ->	{"0","","2CCD"};
lookup1("2CCD") ->	{"0","",""};
lookup1("2CCE") ->	{"0","","2CCF"};
lookup1("2CCF") ->	{"0","",""};
lookup1("2CD0") ->	{"0","","2CD1"};
lookup1("2CD1") ->	{"0","",""};
lookup1("2CD2") ->	{"0","","2CD3"};
lookup1("2CD3") ->	{"0","",""};
lookup1("2CD4") ->	{"0","","2CD5"};
lookup1("2CD5") ->	{"0","",""};
lookup1("2CD6") ->	{"0","","2CD7"};
lookup1("2CD7") ->	{"0","",""};
lookup1("2CD8") ->	{"0","","2CD9"};
lookup1("2CD9") ->	{"0","",""};
lookup1("2CDA") ->	{"0","","2CDB"};
lookup1("2CDB") ->	{"0","",""};
lookup1("2CDC") ->	{"0","","2CDD"};
lookup1("2CDD") ->	{"0","",""};
lookup1("2CDE") ->	{"0","","2CDF"};
lookup1("2CDF") ->	{"0","",""};
lookup1("2CE0") ->	{"0","","2CE1"};
lookup1("2CE1") ->	{"0","",""};
lookup1("2CE2") ->	{"0","","2CE3"};
lookup1("2CE3") ->	{"0","",""};
lookup1("2CE4") ->	{"0","",""};
lookup1("2CE5") ->	{"0","",""};
lookup1("2CE6") ->	{"0","",""};
lookup1("2CE7") ->	{"0","",""};
lookup1("2CE8") ->	{"0","",""};
lookup1("2CE9") ->	{"0","",""};
lookup1("2CEA") ->	{"0","",""};
lookup1("2CEB") ->	{"0","","2CEC"};
lookup1("2CEC") ->	{"0","",""};
lookup1("2CED") ->	{"0","","2CEE"};
lookup1("2CEE") ->	{"0","",""};
lookup1("2CEF") ->	{"230","",""};
lookup1("2CF0") ->	{"230","",""};
lookup1("2CF1") ->	{"230","",""};
lookup1("2CF2") ->	{"0","","2CF3"};
lookup1("2CF3") ->	{"0","",""};
lookup1("2CF9") ->	{"0","",""};
lookup1("2CFA") ->	{"0","",""};
lookup1("2CFB") ->	{"0","",""};
lookup1("2CFC") ->	{"0","",""};
lookup1("2CFD") ->	{"0","",""};
lookup1("2CFE") ->	{"0","",""};
lookup1("2CFF") ->	{"0","",""};
lookup1("2D00") ->	{"0","",""};
lookup1("2D01") ->	{"0","",""};
lookup1("2D02") ->	{"0","",""};
lookup1("2D03") ->	{"0","",""};
lookup1("2D04") ->	{"0","",""};
lookup1("2D05") ->	{"0","",""};
lookup1("2D06") ->	{"0","",""};
lookup1("2D07") ->	{"0","",""};
lookup1("2D08") ->	{"0","",""};
lookup1("2D09") ->	{"0","",""};
lookup1("2D0A") ->	{"0","",""};
lookup1("2D0B") ->	{"0","",""};
lookup1("2D0C") ->	{"0","",""};
lookup1("2D0D") ->	{"0","",""};
lookup1("2D0E") ->	{"0","",""};
lookup1("2D0F") ->	{"0","",""};
lookup1("2D10") ->	{"0","",""};
lookup1("2D11") ->	{"0","",""};
lookup1("2D12") ->	{"0","",""};
lookup1("2D13") ->	{"0","",""};
lookup1("2D14") ->	{"0","",""};
lookup1("2D15") ->	{"0","",""};
lookup1("2D16") ->	{"0","",""};
lookup1("2D17") ->	{"0","",""};
lookup1("2D18") ->	{"0","",""};
lookup1("2D19") ->	{"0","",""};
lookup1("2D1A") ->	{"0","",""};
lookup1("2D1B") ->	{"0","",""};
lookup1("2D1C") ->	{"0","",""};
lookup1("2D1D") ->	{"0","",""};
lookup1("2D1E") ->	{"0","",""};
lookup1("2D1F") ->	{"0","",""};
lookup1("2D20") ->	{"0","",""};
lookup1("2D21") ->	{"0","",""};
lookup1("2D22") ->	{"0","",""};
lookup1("2D23") ->	{"0","",""};
lookup1("2D24") ->	{"0","",""};
lookup1("2D25") ->	{"0","",""};
lookup1("2D27") ->	{"0","",""};
lookup1("2D2D") ->	{"0","",""};
lookup1("2D30") ->	{"0","",""};
lookup1("2D31") ->	{"0","",""};
lookup1("2D32") ->	{"0","",""};
lookup1("2D33") ->	{"0","",""};
lookup1("2D34") ->	{"0","",""};
lookup1("2D35") ->	{"0","",""};
lookup1("2D36") ->	{"0","",""};
lookup1("2D37") ->	{"0","",""};
lookup1("2D38") ->	{"0","",""};
lookup1("2D39") ->	{"0","",""};
lookup1("2D3A") ->	{"0","",""};
lookup1("2D3B") ->	{"0","",""};
lookup1("2D3C") ->	{"0","",""};
lookup1("2D3D") ->	{"0","",""};
lookup1("2D3E") ->	{"0","",""};
lookup1("2D3F") ->	{"0","",""};
lookup1("2D40") ->	{"0","",""};
lookup1("2D41") ->	{"0","",""};
lookup1("2D42") ->	{"0","",""};
lookup1("2D43") ->	{"0","",""};
lookup1("2D44") ->	{"0","",""};
lookup1("2D45") ->	{"0","",""};
lookup1("2D46") ->	{"0","",""};
lookup1("2D47") ->	{"0","",""};
lookup1("2D48") ->	{"0","",""};
lookup1("2D49") ->	{"0","",""};
lookup1("2D4A") ->	{"0","",""};
lookup1("2D4B") ->	{"0","",""};
lookup1("2D4C") ->	{"0","",""};
lookup1("2D4D") ->	{"0","",""};
lookup1("2D4E") ->	{"0","",""};
lookup1("2D4F") ->	{"0","",""};
lookup1("2D50") ->	{"0","",""};
lookup1("2D51") ->	{"0","",""};
lookup1("2D52") ->	{"0","",""};
lookup1("2D53") ->	{"0","",""};
lookup1("2D54") ->	{"0","",""};
lookup1("2D55") ->	{"0","",""};
lookup1("2D56") ->	{"0","",""};
lookup1("2D57") ->	{"0","",""};
lookup1("2D58") ->	{"0","",""};
lookup1("2D59") ->	{"0","",""};
lookup1("2D5A") ->	{"0","",""};
lookup1("2D5B") ->	{"0","",""};
lookup1("2D5C") ->	{"0","",""};
lookup1("2D5D") ->	{"0","",""};
lookup1("2D5E") ->	{"0","",""};
lookup1("2D5F") ->	{"0","",""};
lookup1("2D60") ->	{"0","",""};
lookup1("2D61") ->	{"0","",""};
lookup1("2D62") ->	{"0","",""};
lookup1("2D63") ->	{"0","",""};
lookup1("2D64") ->	{"0","",""};
lookup1("2D65") ->	{"0","",""};
lookup1("2D66") ->	{"0","",""};
lookup1("2D67") ->	{"0","",""};
lookup1("2D6F") ->	{"0","<super> 2D61",""};
lookup1("2D70") ->	{"0","",""};
lookup1("2D7F") ->	{"9","",""};
lookup1("2D80") ->	{"0","",""};
lookup1("2D81") ->	{"0","",""};
lookup1("2D82") ->	{"0","",""};
lookup1("2D83") ->	{"0","",""};
lookup1("2D84") ->	{"0","",""};
lookup1("2D85") ->	{"0","",""};
lookup1("2D86") ->	{"0","",""};
lookup1("2D87") ->	{"0","",""};
lookup1("2D88") ->	{"0","",""};
lookup1("2D89") ->	{"0","",""};
lookup1("2D8A") ->	{"0","",""};
lookup1("2D8B") ->	{"0","",""};
lookup1("2D8C") ->	{"0","",""};
lookup1("2D8D") ->	{"0","",""};
lookup1("2D8E") ->	{"0","",""};
lookup1("2D8F") ->	{"0","",""};
lookup1("2D90") ->	{"0","",""};
lookup1("2D91") ->	{"0","",""};
lookup1("2D92") ->	{"0","",""};
lookup1("2D93") ->	{"0","",""};
lookup1("2D94") ->	{"0","",""};
lookup1("2D95") ->	{"0","",""};
lookup1("2D96") ->	{"0","",""};
lookup1("2DA0") ->	{"0","",""};
lookup1("2DA1") ->	{"0","",""};
lookup1("2DA2") ->	{"0","",""};
lookup1("2DA3") ->	{"0","",""};
lookup1("2DA4") ->	{"0","",""};
lookup1("2DA5") ->	{"0","",""};
lookup1("2DA6") ->	{"0","",""};
lookup1("2DA8") ->	{"0","",""};
lookup1("2DA9") ->	{"0","",""};
lookup1("2DAA") ->	{"0","",""};
lookup1("2DAB") ->	{"0","",""};
lookup1("2DAC") ->	{"0","",""};
lookup1("2DAD") ->	{"0","",""};
lookup1("2DAE") ->	{"0","",""};
lookup1("2DB0") ->	{"0","",""};
lookup1("2DB1") ->	{"0","",""};
lookup1("2DB2") ->	{"0","",""};
lookup1("2DB3") ->	{"0","",""};
lookup1("2DB4") ->	{"0","",""};
lookup1("2DB5") ->	{"0","",""};
lookup1("2DB6") ->	{"0","",""};
lookup1("2DB8") ->	{"0","",""};
lookup1("2DB9") ->	{"0","",""};
lookup1("2DBA") ->	{"0","",""};
lookup1("2DBB") ->	{"0","",""};
lookup1("2DBC") ->	{"0","",""};
lookup1("2DBD") ->	{"0","",""};
lookup1("2DBE") ->	{"0","",""};
lookup1("2DC0") ->	{"0","",""};
lookup1("2DC1") ->	{"0","",""};
lookup1("2DC2") ->	{"0","",""};
lookup1("2DC3") ->	{"0","",""};
lookup1("2DC4") ->	{"0","",""};
lookup1("2DC5") ->	{"0","",""};
lookup1("2DC6") ->	{"0","",""};
lookup1("2DC8") ->	{"0","",""};
lookup1("2DC9") ->	{"0","",""};
lookup1("2DCA") ->	{"0","",""};
lookup1("2DCB") ->	{"0","",""};
lookup1("2DCC") ->	{"0","",""};
lookup1("2DCD") ->	{"0","",""};
lookup1("2DCE") ->	{"0","",""};
lookup1("2DD0") ->	{"0","",""};
lookup1("2DD1") ->	{"0","",""};
lookup1("2DD2") ->	{"0","",""};
lookup1("2DD3") ->	{"0","",""};
lookup1("2DD4") ->	{"0","",""};
lookup1("2DD5") ->	{"0","",""};
lookup1("2DD6") ->	{"0","",""};
lookup1("2DD8") ->	{"0","",""};
lookup1("2DD9") ->	{"0","",""};
lookup1("2DDA") ->	{"0","",""};
lookup1("2DDB") ->	{"0","",""};
lookup1("2DDC") ->	{"0","",""};
lookup1("2DDD") ->	{"0","",""};
lookup1("2DDE") ->	{"0","",""};
lookup1("2DE0") ->	{"230","",""};
lookup1("2DE1") ->	{"230","",""};
lookup1("2DE2") ->	{"230","",""};
lookup1("2DE3") ->	{"230","",""};
lookup1("2DE4") ->	{"230","",""};
lookup1("2DE5") ->	{"230","",""};
lookup1("2DE6") ->	{"230","",""};
lookup1("2DE7") ->	{"230","",""};
lookup1("2DE8") ->	{"230","",""};
lookup1("2DE9") ->	{"230","",""};
lookup1("2DEA") ->	{"230","",""};
lookup1("2DEB") ->	{"230","",""};
lookup1("2DEC") ->	{"230","",""};
lookup1("2DED") ->	{"230","",""};
lookup1("2DEE") ->	{"230","",""};
lookup1("2DEF") ->	{"230","",""};
lookup1("2DF0") ->	{"230","",""};
lookup1("2DF1") ->	{"230","",""};
lookup1("2DF2") ->	{"230","",""};
lookup1("2DF3") ->	{"230","",""};
lookup1("2DF4") ->	{"230","",""};
lookup1("2DF5") ->	{"230","",""};
lookup1("2DF6") ->	{"230","",""};
lookup1("2DF7") ->	{"230","",""};
lookup1("2DF8") ->	{"230","",""};
lookup1("2DF9") ->	{"230","",""};
lookup1("2DFA") ->	{"230","",""};
lookup1("2DFB") ->	{"230","",""};
lookup1("2DFC") ->	{"230","",""};
lookup1("2DFD") ->	{"230","",""};
lookup1("2DFE") ->	{"230","",""};
lookup1("2DFF") ->	{"230","",""};
lookup1("2E00") ->	{"0","",""};
lookup1("2E01") ->	{"0","",""};
lookup1("2E02") ->	{"0","",""};
lookup1("2E03") ->	{"0","",""};
lookup1("2E04") ->	{"0","",""};
lookup1("2E05") ->	{"0","",""};
lookup1("2E06") ->	{"0","",""};
lookup1("2E07") ->	{"0","",""};
lookup1("2E08") ->	{"0","",""};
lookup1("2E09") ->	{"0","",""};
lookup1("2E0A") ->	{"0","",""};
lookup1("2E0B") ->	{"0","",""};
lookup1("2E0C") ->	{"0","",""};
lookup1("2E0D") ->	{"0","",""};
lookup1("2E0E") ->	{"0","",""};
lookup1("2E0F") ->	{"0","",""};
lookup1("2E10") ->	{"0","",""};
lookup1("2E11") ->	{"0","",""};
lookup1("2E12") ->	{"0","",""};
lookup1("2E13") ->	{"0","",""};
lookup1("2E14") ->	{"0","",""};
lookup1("2E15") ->	{"0","",""};
lookup1("2E16") ->	{"0","",""};
lookup1("2E17") ->	{"0","",""};
lookup1("2E18") ->	{"0","",""};
lookup1("2E19") ->	{"0","",""};
lookup1("2E1A") ->	{"0","",""};
lookup1("2E1B") ->	{"0","",""};
lookup1("2E1C") ->	{"0","",""};
lookup1("2E1D") ->	{"0","",""};
lookup1("2E1E") ->	{"0","",""};
lookup1("2E1F") ->	{"0","",""};
lookup1("2E20") ->	{"0","",""};
lookup1("2E21") ->	{"0","",""};
lookup1("2E22") ->	{"0","",""};
lookup1("2E23") ->	{"0","",""};
lookup1("2E24") ->	{"0","",""};
lookup1("2E25") ->	{"0","",""};
lookup1("2E26") ->	{"0","",""};
lookup1("2E27") ->	{"0","",""};
lookup1("2E28") ->	{"0","",""};
lookup1("2E29") ->	{"0","",""};
lookup1("2E2A") ->	{"0","",""};
lookup1("2E2B") ->	{"0","",""};
lookup1("2E2C") ->	{"0","",""};
lookup1("2E2D") ->	{"0","",""};
lookup1("2E2E") ->	{"0","",""};
lookup1("2E2F") ->	{"0","",""};
lookup1("2E30") ->	{"0","",""};
lookup1("2E31") ->	{"0","",""};
lookup1("2E32") ->	{"0","",""};
lookup1("2E33") ->	{"0","",""};
lookup1("2E34") ->	{"0","",""};
lookup1("2E35") ->	{"0","",""};
lookup1("2E36") ->	{"0","",""};
lookup1("2E37") ->	{"0","",""};
lookup1("2E38") ->	{"0","",""};
lookup1("2E39") ->	{"0","",""};
lookup1("2E3A") ->	{"0","",""};
lookup1("2E3B") ->	{"0","",""};
lookup1("2E3C") ->	{"0","",""};
lookup1("2E3D") ->	{"0","",""};
lookup1("2E3E") ->	{"0","",""};
lookup1("2E3F") ->	{"0","",""};
lookup1("2E40") ->	{"0","",""};
lookup1("2E41") ->	{"0","",""};
lookup1("2E42") ->	{"0","",""};
lookup1("2E80") ->	{"0","",""};
lookup1("2E81") ->	{"0","",""};
lookup1("2E82") ->	{"0","",""};
lookup1("2E83") ->	{"0","",""};
lookup1("2E84") ->	{"0","",""};
lookup1("2E85") ->	{"0","",""};
lookup1("2E86") ->	{"0","",""};
lookup1("2E87") ->	{"0","",""};
lookup1("2E88") ->	{"0","",""};
lookup1("2E89") ->	{"0","",""};
lookup1("2E8A") ->	{"0","",""};
lookup1("2E8B") ->	{"0","",""};
lookup1("2E8C") ->	{"0","",""};
lookup1("2E8D") ->	{"0","",""};
lookup1("2E8E") ->	{"0","",""};
lookup1("2E8F") ->	{"0","",""};
lookup1("2E90") ->	{"0","",""};
lookup1("2E91") ->	{"0","",""};
lookup1("2E92") ->	{"0","",""};
lookup1("2E93") ->	{"0","",""};
lookup1("2E94") ->	{"0","",""};
lookup1("2E95") ->	{"0","",""};
lookup1("2E96") ->	{"0","",""};
lookup1("2E97") ->	{"0","",""};
lookup1("2E98") ->	{"0","",""};
lookup1("2E99") ->	{"0","",""};
lookup1("2E9B") ->	{"0","",""};
lookup1("2E9C") ->	{"0","",""};
lookup1("2E9D") ->	{"0","",""};
lookup1("2E9E") ->	{"0","",""};
lookup1("2E9F") ->	{"0","<compat> 6BCD",""};
lookup1("2EA0") ->	{"0","",""};
lookup1("2EA1") ->	{"0","",""};
lookup1("2EA2") ->	{"0","",""};
lookup1("2EA3") ->	{"0","",""};
lookup1("2EA4") ->	{"0","",""};
lookup1("2EA5") ->	{"0","",""};
lookup1("2EA6") ->	{"0","",""};
lookup1("2EA7") ->	{"0","",""};
lookup1("2EA8") ->	{"0","",""};
lookup1("2EA9") ->	{"0","",""};
lookup1("2EAA") ->	{"0","",""};
lookup1("2EAB") ->	{"0","",""};
lookup1("2EAC") ->	{"0","",""};
lookup1("2EAD") ->	{"0","",""};
lookup1("2EAE") ->	{"0","",""};
lookup1("2EAF") ->	{"0","",""};
lookup1("2EB0") ->	{"0","",""};
lookup1("2EB1") ->	{"0","",""};
lookup1("2EB2") ->	{"0","",""};
lookup1("2EB3") ->	{"0","",""};
lookup1("2EB4") ->	{"0","",""};
lookup1("2EB5") ->	{"0","",""};
lookup1("2EB6") ->	{"0","",""};
lookup1("2EB7") ->	{"0","",""};
lookup1("2EB8") ->	{"0","",""};
lookup1("2EB9") ->	{"0","",""};
lookup1("2EBA") ->	{"0","",""};
lookup1("2EBB") ->	{"0","",""};
lookup1("2EBC") ->	{"0","",""};
lookup1("2EBD") ->	{"0","",""};
lookup1("2EBE") ->	{"0","",""};
lookup1("2EBF") ->	{"0","",""};
lookup1("2EC0") ->	{"0","",""};
lookup1("2EC1") ->	{"0","",""};
lookup1("2EC2") ->	{"0","",""};
lookup1("2EC3") ->	{"0","",""};
lookup1("2EC4") ->	{"0","",""};
lookup1("2EC5") ->	{"0","",""};
lookup1("2EC6") ->	{"0","",""};
lookup1("2EC7") ->	{"0","",""};
lookup1("2EC8") ->	{"0","",""};
lookup1("2EC9") ->	{"0","",""};
lookup1("2ECA") ->	{"0","",""};
lookup1("2ECB") ->	{"0","",""};
lookup1("2ECC") ->	{"0","",""};
lookup1("2ECD") ->	{"0","",""};
lookup1("2ECE") ->	{"0","",""};
lookup1("2ECF") ->	{"0","",""};
lookup1("2ED0") ->	{"0","",""};
lookup1("2ED1") ->	{"0","",""};
lookup1("2ED2") ->	{"0","",""};
lookup1("2ED3") ->	{"0","",""};
lookup1("2ED4") ->	{"0","",""};
lookup1("2ED5") ->	{"0","",""};
lookup1("2ED6") ->	{"0","",""};
lookup1("2ED7") ->	{"0","",""};
lookup1("2ED8") ->	{"0","",""};
lookup1("2ED9") ->	{"0","",""};
lookup1("2EDA") ->	{"0","",""};
lookup1("2EDB") ->	{"0","",""};
lookup1("2EDC") ->	{"0","",""};
lookup1("2EDD") ->	{"0","",""};
lookup1("2EDE") ->	{"0","",""};
lookup1("2EDF") ->	{"0","",""};
lookup1("2EE0") ->	{"0","",""};
lookup1("2EE1") ->	{"0","",""};
lookup1("2EE2") ->	{"0","",""};
lookup1("2EE3") ->	{"0","",""};
lookup1("2EE4") ->	{"0","",""};
lookup1("2EE5") ->	{"0","",""};
lookup1("2EE6") ->	{"0","",""};
lookup1("2EE7") ->	{"0","",""};
lookup1("2EE8") ->	{"0","",""};
lookup1("2EE9") ->	{"0","",""};
lookup1("2EEA") ->	{"0","",""};
lookup1("2EEB") ->	{"0","",""};
lookup1("2EEC") ->	{"0","",""};
lookup1("2EED") ->	{"0","",""};
lookup1("2EEE") ->	{"0","",""};
lookup1("2EEF") ->	{"0","",""};
lookup1("2EF0") ->	{"0","",""};
lookup1("2EF1") ->	{"0","",""};
lookup1("2EF2") ->	{"0","",""};
lookup1("2EF3") ->	{"0","<compat> 9F9F",""};
lookup1("2F00") ->	{"0","<compat> 4E00",""};
lookup1("2F01") ->	{"0","<compat> 4E28",""};
lookup1("2F02") ->	{"0","<compat> 4E36",""};
lookup1("2F03") ->	{"0","<compat> 4E3F",""};
lookup1("2F04") ->	{"0","<compat> 4E59",""};
lookup1("2F05") ->	{"0","<compat> 4E85",""};
lookup1("2F06") ->	{"0","<compat> 4E8C",""};
lookup1("2F07") ->	{"0","<compat> 4EA0",""};
lookup1("2F08") ->	{"0","<compat> 4EBA",""};
lookup1("2F09") ->	{"0","<compat> 513F",""};
lookup1("2F0A") ->	{"0","<compat> 5165",""};
lookup1("2F0B") ->	{"0","<compat> 516B",""};
lookup1("2F0C") ->	{"0","<compat> 5182",""};
lookup1("2F0D") ->	{"0","<compat> 5196",""};
lookup1("2F0E") ->	{"0","<compat> 51AB",""};
lookup1("2F0F") ->	{"0","<compat> 51E0",""};
lookup1("2F10") ->	{"0","<compat> 51F5",""};
lookup1("2F11") ->	{"0","<compat> 5200",""};
lookup1("2F12") ->	{"0","<compat> 529B",""};
lookup1("2F13") ->	{"0","<compat> 52F9",""};
lookup1("2F14") ->	{"0","<compat> 5315",""};
lookup1("2F15") ->	{"0","<compat> 531A",""};
lookup1("2F16") ->	{"0","<compat> 5338",""};
lookup1("2F17") ->	{"0","<compat> 5341",""};
lookup1("2F18") ->	{"0","<compat> 535C",""};
lookup1("2F19") ->	{"0","<compat> 5369",""};
lookup1("2F1A") ->	{"0","<compat> 5382",""};
lookup1("2F1B") ->	{"0","<compat> 53B6",""};
lookup1("2F1C") ->	{"0","<compat> 53C8",""};
lookup1("2F1D") ->	{"0","<compat> 53E3",""};
lookup1("2F1E") ->	{"0","<compat> 56D7",""};
lookup1("2F1F") ->	{"0","<compat> 571F",""};
lookup1("2F20") ->	{"0","<compat> 58EB",""};
lookup1("2F21") ->	{"0","<compat> 5902",""};
lookup1("2F22") ->	{"0","<compat> 590A",""};
lookup1("2F23") ->	{"0","<compat> 5915",""};
lookup1("2F24") ->	{"0","<compat> 5927",""};
lookup1("2F25") ->	{"0","<compat> 5973",""};
lookup1("2F26") ->	{"0","<compat> 5B50",""};
lookup1("2F27") ->	{"0","<compat> 5B80",""};
lookup1("2F28") ->	{"0","<compat> 5BF8",""};
lookup1("2F29") ->	{"0","<compat> 5C0F",""};
lookup1("2F2A") ->	{"0","<compat> 5C22",""};
lookup1("2F2B") ->	{"0","<compat> 5C38",""};
lookup1("2F2C") ->	{"0","<compat> 5C6E",""};
lookup1("2F2D") ->	{"0","<compat> 5C71",""};
lookup1("2F2E") ->	{"0","<compat> 5DDB",""};
lookup1("2F2F") ->	{"0","<compat> 5DE5",""};
lookup1("2F30") ->	{"0","<compat> 5DF1",""};
lookup1("2F31") ->	{"0","<compat> 5DFE",""};
lookup1("2F32") ->	{"0","<compat> 5E72",""};
lookup1("2F33") ->	{"0","<compat> 5E7A",""};
lookup1("2F34") ->	{"0","<compat> 5E7F",""};
lookup1("2F35") ->	{"0","<compat> 5EF4",""};
lookup1("2F36") ->	{"0","<compat> 5EFE",""};
lookup1("2F37") ->	{"0","<compat> 5F0B",""};
lookup1("2F38") ->	{"0","<compat> 5F13",""};
lookup1("2F39") ->	{"0","<compat> 5F50",""};
lookup1("2F3A") ->	{"0","<compat> 5F61",""};
lookup1("2F3B") ->	{"0","<compat> 5F73",""};
lookup1("2F3C") ->	{"0","<compat> 5FC3",""};
lookup1("2F3D") ->	{"0","<compat> 6208",""};
lookup1("2F3E") ->	{"0","<compat> 6236",""};
lookup1("2F3F") ->	{"0","<compat> 624B",""};
lookup1("2F40") ->	{"0","<compat> 652F",""};
lookup1("2F41") ->	{"0","<compat> 6534",""};
lookup1("2F42") ->	{"0","<compat> 6587",""};
lookup1("2F43") ->	{"0","<compat> 6597",""};
lookup1("2F44") ->	{"0","<compat> 65A4",""};
lookup1("2F45") ->	{"0","<compat> 65B9",""};
lookup1("2F46") ->	{"0","<compat> 65E0",""};
lookup1("2F47") ->	{"0","<compat> 65E5",""};
lookup1("2F48") ->	{"0","<compat> 66F0",""};
lookup1("2F49") ->	{"0","<compat> 6708",""};
lookup1("2F4A") ->	{"0","<compat> 6728",""};
lookup1("2F4B") ->	{"0","<compat> 6B20",""};
lookup1("2F4C") ->	{"0","<compat> 6B62",""};
lookup1("2F4D") ->	{"0","<compat> 6B79",""};
lookup1("2F4E") ->	{"0","<compat> 6BB3",""};
lookup1("2F4F") ->	{"0","<compat> 6BCB",""};
lookup1("2F50") ->	{"0","<compat> 6BD4",""};
lookup1("2F51") ->	{"0","<compat> 6BDB",""};
lookup1("2F52") ->	{"0","<compat> 6C0F",""};
lookup1("2F53") ->	{"0","<compat> 6C14",""};
lookup1("2F54") ->	{"0","<compat> 6C34",""};
lookup1("2F55") ->	{"0","<compat> 706B",""};
lookup1("2F56") ->	{"0","<compat> 722A",""};
lookup1("2F57") ->	{"0","<compat> 7236",""};
lookup1("2F58") ->	{"0","<compat> 723B",""};
lookup1("2F59") ->	{"0","<compat> 723F",""};
lookup1("2F5A") ->	{"0","<compat> 7247",""};
lookup1("2F5B") ->	{"0","<compat> 7259",""};
lookup1("2F5C") ->	{"0","<compat> 725B",""};
lookup1("2F5D") ->	{"0","<compat> 72AC",""};
lookup1("2F5E") ->	{"0","<compat> 7384",""};
lookup1("2F5F") ->	{"0","<compat> 7389",""};
lookup1("2F60") ->	{"0","<compat> 74DC",""};
lookup1("2F61") ->	{"0","<compat> 74E6",""};
lookup1("2F62") ->	{"0","<compat> 7518",""};
lookup1("2F63") ->	{"0","<compat> 751F",""};
lookup1("2F64") ->	{"0","<compat> 7528",""};
lookup1("2F65") ->	{"0","<compat> 7530",""};
lookup1("2F66") ->	{"0","<compat> 758B",""};
lookup1("2F67") ->	{"0","<compat> 7592",""};
lookup1("2F68") ->	{"0","<compat> 7676",""};
lookup1("2F69") ->	{"0","<compat> 767D",""};
lookup1("2F6A") ->	{"0","<compat> 76AE",""};
lookup1("2F6B") ->	{"0","<compat> 76BF",""};
lookup1("2F6C") ->	{"0","<compat> 76EE",""};
lookup1("2F6D") ->	{"0","<compat> 77DB",""};
lookup1("2F6E") ->	{"0","<compat> 77E2",""};
lookup1("2F6F") ->	{"0","<compat> 77F3",""};
lookup1("2F70") ->	{"0","<compat> 793A",""};
lookup1("2F71") ->	{"0","<compat> 79B8",""};
lookup1("2F72") ->	{"0","<compat> 79BE",""};
lookup1("2F73") ->	{"0","<compat> 7A74",""};
lookup1("2F74") ->	{"0","<compat> 7ACB",""};
lookup1("2F75") ->	{"0","<compat> 7AF9",""};
lookup1("2F76") ->	{"0","<compat> 7C73",""};
lookup1("2F77") ->	{"0","<compat> 7CF8",""};
lookup1("2F78") ->	{"0","<compat> 7F36",""};
lookup1("2F79") ->	{"0","<compat> 7F51",""};
lookup1("2F7A") ->	{"0","<compat> 7F8A",""};
lookup1("2F7B") ->	{"0","<compat> 7FBD",""};
lookup1("2F7C") ->	{"0","<compat> 8001",""};
lookup1("2F7D") ->	{"0","<compat> 800C",""};
lookup1("2F7E") ->	{"0","<compat> 8012",""};
lookup1("2F7F") ->	{"0","<compat> 8033",""};
lookup1("2F80") ->	{"0","<compat> 807F",""};
lookup1("2F800") ->	{"0","4E3D",""};
lookup1("2F801") ->	{"0","4E38",""};
lookup1("2F802") ->	{"0","4E41",""};
lookup1("2F803") ->	{"0","20122",""};
lookup1("2F804") ->	{"0","4F60",""};
lookup1("2F805") ->	{"0","4FAE",""};
lookup1("2F806") ->	{"0","4FBB",""};
lookup1("2F807") ->	{"0","5002",""};
lookup1("2F808") ->	{"0","507A",""};
lookup1("2F809") ->	{"0","5099",""};
lookup1("2F80A") ->	{"0","50E7",""};
lookup1("2F80B") ->	{"0","50CF",""};
lookup1("2F80C") ->	{"0","349E",""};
lookup1("2F80D") ->	{"0","2063A",""};
lookup1("2F80E") ->	{"0","514D",""};
lookup1("2F80F") ->	{"0","5154",""};
lookup1("2F81") ->	{"0","<compat> 8089",""};
lookup1("2F810") ->	{"0","5164",""};
lookup1("2F811") ->	{"0","5177",""};
lookup1("2F812") ->	{"0","2051C",""};
lookup1("2F813") ->	{"0","34B9",""};
lookup1("2F814") ->	{"0","5167",""};
lookup1("2F815") ->	{"0","518D",""};
lookup1("2F816") ->	{"0","2054B",""};
lookup1("2F817") ->	{"0","5197",""};
lookup1("2F818") ->	{"0","51A4",""};
lookup1("2F819") ->	{"0","4ECC",""};
lookup1("2F81A") ->	{"0","51AC",""};
lookup1("2F81B") ->	{"0","51B5",""};
lookup1("2F81C") ->	{"0","291DF",""};
lookup1("2F81D") ->	{"0","51F5",""};
lookup1("2F81E") ->	{"0","5203",""};
lookup1("2F81F") ->	{"0","34DF",""};
lookup1("2F82") ->	{"0","<compat> 81E3",""};
lookup1("2F820") ->	{"0","523B",""};
lookup1("2F821") ->	{"0","5246",""};
lookup1("2F822") ->	{"0","5272",""};
lookup1("2F823") ->	{"0","5277",""};
lookup1("2F824") ->	{"0","3515",""};
lookup1("2F825") ->	{"0","52C7",""};
lookup1("2F826") ->	{"0","52C9",""};
lookup1("2F827") ->	{"0","52E4",""};
lookup1("2F828") ->	{"0","52FA",""};
lookup1("2F829") ->	{"0","5305",""};
lookup1("2F82A") ->	{"0","5306",""};
lookup1("2F82B") ->	{"0","5317",""};
lookup1("2F82C") ->	{"0","5349",""};
lookup1("2F82D") ->	{"0","5351",""};
lookup1("2F82E") ->	{"0","535A",""};
lookup1("2F82F") ->	{"0","5373",""};
lookup1("2F83") ->	{"0","<compat> 81EA",""};
lookup1("2F830") ->	{"0","537D",""};
lookup1("2F831") ->	{"0","537F",""};
lookup1("2F832") ->	{"0","537F",""};
lookup1("2F833") ->	{"0","537F",""};
lookup1("2F834") ->	{"0","20A2C",""};
lookup1("2F835") ->	{"0","7070",""};
lookup1("2F836") ->	{"0","53CA",""};
lookup1("2F837") ->	{"0","53DF",""};
lookup1("2F838") ->	{"0","20B63",""};
lookup1("2F839") ->	{"0","53EB",""};
lookup1("2F83A") ->	{"0","53F1",""};
lookup1("2F83B") ->	{"0","5406",""};
lookup1("2F83C") ->	{"0","549E",""};
lookup1("2F83D") ->	{"0","5438",""};
lookup1("2F83E") ->	{"0","5448",""};
lookup1("2F83F") ->	{"0","5468",""};
lookup1("2F84") ->	{"0","<compat> 81F3",""};
lookup1("2F840") ->	{"0","54A2",""};
lookup1("2F841") ->	{"0","54F6",""};
lookup1("2F842") ->	{"0","5510",""};
lookup1("2F843") ->	{"0","5553",""};
lookup1("2F844") ->	{"0","5563",""};
lookup1("2F845") ->	{"0","5584",""};
lookup1("2F846") ->	{"0","5584",""};
lookup1("2F847") ->	{"0","5599",""};
lookup1("2F848") ->	{"0","55AB",""};
lookup1("2F849") ->	{"0","55B3",""};
lookup1("2F84A") ->	{"0","55C2",""};
lookup1("2F84B") ->	{"0","5716",""};
lookup1("2F84C") ->	{"0","5606",""};
lookup1("2F84D") ->	{"0","5717",""};
lookup1("2F84E") ->	{"0","5651",""};
lookup1("2F84F") ->	{"0","5674",""};
lookup1("2F85") ->	{"0","<compat> 81FC",""};
lookup1("2F850") ->	{"0","5207",""};
lookup1("2F851") ->	{"0","58EE",""};
lookup1("2F852") ->	{"0","57CE",""};
lookup1("2F853") ->	{"0","57F4",""};
lookup1("2F854") ->	{"0","580D",""};
lookup1("2F855") ->	{"0","578B",""};
lookup1("2F856") ->	{"0","5832",""};
lookup1("2F857") ->	{"0","5831",""};
lookup1("2F858") ->	{"0","58AC",""};
lookup1("2F859") ->	{"0","214E4",""};
lookup1("2F85A") ->	{"0","58F2",""};
lookup1("2F85B") ->	{"0","58F7",""};
lookup1("2F85C") ->	{"0","5906",""};
lookup1("2F85D") ->	{"0","591A",""};
lookup1("2F85E") ->	{"0","5922",""};
lookup1("2F85F") ->	{"0","5962",""};
lookup1("2F86") ->	{"0","<compat> 820C",""};
lookup1("2F860") ->	{"0","216A8",""};
lookup1("2F861") ->	{"0","216EA",""};
lookup1("2F862") ->	{"0","59EC",""};
lookup1("2F863") ->	{"0","5A1B",""};
lookup1("2F864") ->	{"0","5A27",""};
lookup1("2F865") ->	{"0","59D8",""};
lookup1("2F866") ->	{"0","5A66",""};
lookup1("2F867") ->	{"0","36EE",""};
lookup1("2F868") ->	{"0","36FC",""};
lookup1("2F869") ->	{"0","5B08",""};
lookup1("2F86A") ->	{"0","5B3E",""};
lookup1("2F86B") ->	{"0","5B3E",""};
lookup1("2F86C") ->	{"0","219C8",""};
lookup1("2F86D") ->	{"0","5BC3",""};
lookup1("2F86E") ->	{"0","5BD8",""};
lookup1("2F86F") ->	{"0","5BE7",""};
lookup1("2F87") ->	{"0","<compat> 821B",""};
lookup1("2F870") ->	{"0","5BF3",""};
lookup1("2F871") ->	{"0","21B18",""};
lookup1("2F872") ->	{"0","5BFF",""};
lookup1("2F873") ->	{"0","5C06",""};
lookup1("2F874") ->	{"0","5F53",""};
lookup1("2F875") ->	{"0","5C22",""};
lookup1("2F876") ->	{"0","3781",""};
lookup1("2F877") ->	{"0","5C60",""};
lookup1("2F878") ->	{"0","5C6E",""};
lookup1("2F879") ->	{"0","5CC0",""};
lookup1("2F87A") ->	{"0","5C8D",""};
lookup1("2F87B") ->	{"0","21DE4",""};
lookup1("2F87C") ->	{"0","5D43",""};
lookup1("2F87D") ->	{"0","21DE6",""};
lookup1("2F87E") ->	{"0","5D6E",""};
lookup1("2F87F") ->	{"0","5D6B",""};
lookup1("2F88") ->	{"0","<compat> 821F",""};
lookup1("2F880") ->	{"0","5D7C",""};
lookup1("2F881") ->	{"0","5DE1",""};
lookup1("2F882") ->	{"0","5DE2",""};
lookup1("2F883") ->	{"0","382F",""};
lookup1("2F884") ->	{"0","5DFD",""};
lookup1("2F885") ->	{"0","5E28",""};
lookup1("2F886") ->	{"0","5E3D",""};
lookup1("2F887") ->	{"0","5E69",""};
lookup1("2F888") ->	{"0","3862",""};
lookup1("2F889") ->	{"0","22183",""};
lookup1("2F88A") ->	{"0","387C",""};
lookup1("2F88B") ->	{"0","5EB0",""};
lookup1("2F88C") ->	{"0","5EB3",""};
lookup1("2F88D") ->	{"0","5EB6",""};
lookup1("2F88E") ->	{"0","5ECA",""};
lookup1("2F88F") ->	{"0","2A392",""};
lookup1("2F89") ->	{"0","<compat> 826E",""};
lookup1("2F890") ->	{"0","5EFE",""};
lookup1("2F891") ->	{"0","22331",""};
lookup1("2F892") ->	{"0","22331",""};
lookup1("2F893") ->	{"0","8201",""};
lookup1("2F894") ->	{"0","5F22",""};
lookup1("2F895") ->	{"0","5F22",""};
lookup1("2F896") ->	{"0","38C7",""};
lookup1("2F897") ->	{"0","232B8",""};
lookup1("2F898") ->	{"0","261DA",""};
lookup1("2F899") ->	{"0","5F62",""};
lookup1("2F89A") ->	{"0","5F6B",""};
lookup1("2F89B") ->	{"0","38E3",""};
lookup1("2F89C") ->	{"0","5F9A",""};
lookup1("2F89D") ->	{"0","5FCD",""};
lookup1("2F89E") ->	{"0","5FD7",""};
lookup1("2F89F") ->	{"0","5FF9",""};
lookup1("2F8A") ->	{"0","<compat> 8272",""};
lookup1("2F8A0") ->	{"0","6081",""};
lookup1("2F8A1") ->	{"0","393A",""};
lookup1("2F8A2") ->	{"0","391C",""};
lookup1("2F8A3") ->	{"0","6094",""};
lookup1("2F8A4") ->	{"0","226D4",""};
lookup1("2F8A5") ->	{"0","60C7",""};
lookup1("2F8A6") ->	{"0","6148",""};
lookup1("2F8A7") ->	{"0","614C",""};
lookup1("2F8A8") ->	{"0","614E",""};
lookup1("2F8A9") ->	{"0","614C",""};
lookup1("2F8AA") ->	{"0","617A",""};
lookup1("2F8AB") ->	{"0","618E",""};
lookup1("2F8AC") ->	{"0","61B2",""};
lookup1("2F8AD") ->	{"0","61A4",""};
lookup1("2F8AE") ->	{"0","61AF",""};
lookup1("2F8AF") ->	{"0","61DE",""};
lookup1("2F8B") ->	{"0","<compat> 8278",""};
lookup1("2F8B0") ->	{"0","61F2",""};
lookup1("2F8B1") ->	{"0","61F6",""};
lookup1("2F8B2") ->	{"0","6210",""};
lookup1("2F8B3") ->	{"0","621B",""};
lookup1("2F8B4") ->	{"0","625D",""};
lookup1("2F8B5") ->	{"0","62B1",""};
lookup1("2F8B6") ->	{"0","62D4",""};
lookup1("2F8B7") ->	{"0","6350",""};
lookup1("2F8B8") ->	{"0","22B0C",""};
lookup1("2F8B9") ->	{"0","633D",""};
lookup1("2F8BA") ->	{"0","62FC",""};
lookup1("2F8BB") ->	{"0","6368",""};
lookup1("2F8BC") ->	{"0","6383",""};
lookup1("2F8BD") ->	{"0","63E4",""};
lookup1("2F8BE") ->	{"0","22BF1",""};
lookup1("2F8BF") ->	{"0","6422",""};
lookup1("2F8C") ->	{"0","<compat> 864D",""};
lookup1("2F8C0") ->	{"0","63C5",""};
lookup1("2F8C1") ->	{"0","63A9",""};
lookup1("2F8C2") ->	{"0","3A2E",""};
lookup1("2F8C3") ->	{"0","6469",""};
lookup1("2F8C4") ->	{"0","647E",""};
lookup1("2F8C5") ->	{"0","649D",""};
lookup1("2F8C6") ->	{"0","6477",""};
lookup1("2F8C7") ->	{"0","3A6C",""};
lookup1("2F8C8") ->	{"0","654F",""};
lookup1("2F8C9") ->	{"0","656C",""};
lookup1("2F8CA") ->	{"0","2300A",""};
lookup1("2F8CB") ->	{"0","65E3",""};
lookup1("2F8CC") ->	{"0","66F8",""};
lookup1("2F8CD") ->	{"0","6649",""};
lookup1("2F8CE") ->	{"0","3B19",""};
lookup1("2F8CF") ->	{"0","6691",""};
lookup1("2F8D") ->	{"0","<compat> 866B",""};
lookup1("2F8D0") ->	{"0","3B08",""};
lookup1("2F8D1") ->	{"0","3AE4",""};
lookup1("2F8D2") ->	{"0","5192",""};
lookup1("2F8D3") ->	{"0","5195",""};
lookup1("2F8D4") ->	{"0","6700",""};
lookup1("2F8D5") ->	{"0","669C",""};
lookup1("2F8D6") ->	{"0","80AD",""};
lookup1("2F8D7") ->	{"0","43D9",""};
lookup1("2F8D8") ->	{"0","6717",""};
lookup1("2F8D9") ->	{"0","671B",""};
lookup1("2F8DA") ->	{"0","6721",""};
lookup1("2F8DB") ->	{"0","675E",""};
lookup1("2F8DC") ->	{"0","6753",""};
lookup1("2F8DD") ->	{"0","233C3",""};
lookup1("2F8DE") ->	{"0","3B49",""};
lookup1("2F8DF") ->	{"0","67FA",""};
lookup1("2F8E") ->	{"0","<compat> 8840",""};
lookup1("2F8E0") ->	{"0","6785",""};
lookup1("2F8E1") ->	{"0","6852",""};
lookup1("2F8E2") ->	{"0","6885",""};
lookup1("2F8E3") ->	{"0","2346D",""};
lookup1("2F8E4") ->	{"0","688E",""};
lookup1("2F8E5") ->	{"0","681F",""};
lookup1("2F8E6") ->	{"0","6914",""};
lookup1("2F8E7") ->	{"0","3B9D",""};
lookup1("2F8E8") ->	{"0","6942",""};
lookup1("2F8E9") ->	{"0","69A3",""};
lookup1("2F8EA") ->	{"0","69EA",""};
lookup1("2F8EB") ->	{"0","6AA8",""};
lookup1("2F8EC") ->	{"0","236A3",""};
lookup1("2F8ED") ->	{"0","6ADB",""};
lookup1("2F8EE") ->	{"0","3C18",""};
lookup1("2F8EF") ->	{"0","6B21",""};
lookup1("2F8F") ->	{"0","<compat> 884C",""};
lookup1("2F8F0") ->	{"0","238A7",""};
lookup1("2F8F1") ->	{"0","6B54",""};
lookup1("2F8F2") ->	{"0","3C4E",""};
lookup1("2F8F3") ->	{"0","6B72",""};
lookup1("2F8F4") ->	{"0","6B9F",""};
lookup1("2F8F5") ->	{"0","6BBA",""};
lookup1("2F8F6") ->	{"0","6BBB",""};
lookup1("2F8F7") ->	{"0","23A8D",""};
lookup1("2F8F8") ->	{"0","21D0B",""};
lookup1("2F8F9") ->	{"0","23AFA",""};
lookup1("2F8FA") ->	{"0","6C4E",""};
lookup1("2F8FB") ->	{"0","23CBC",""};
lookup1("2F8FC") ->	{"0","6CBF",""};
lookup1("2F8FD") ->	{"0","6CCD",""};
lookup1("2F8FE") ->	{"0","6C67",""};
lookup1("2F8FF") ->	{"0","6D16",""};
lookup1("2F90") ->	{"0","<compat> 8863",""};
lookup1("2F900") ->	{"0","6D3E",""};
lookup1("2F901") ->	{"0","6D77",""};
lookup1("2F902") ->	{"0","6D41",""};
lookup1("2F903") ->	{"0","6D69",""};
lookup1("2F904") ->	{"0","6D78",""};
lookup1("2F905") ->	{"0","6D85",""};
lookup1("2F906") ->	{"0","23D1E",""};
lookup1("2F907") ->	{"0","6D34",""};
lookup1("2F908") ->	{"0","6E2F",""};
lookup1("2F909") ->	{"0","6E6E",""};
lookup1("2F90A") ->	{"0","3D33",""};
lookup1("2F90B") ->	{"0","6ECB",""};
lookup1("2F90C") ->	{"0","6EC7",""};
lookup1("2F90D") ->	{"0","23ED1",""};
lookup1("2F90E") ->	{"0","6DF9",""};
lookup1("2F90F") ->	{"0","6F6E",""};
lookup1("2F91") ->	{"0","<compat> 897E",""};
lookup1("2F910") ->	{"0","23F5E",""};
lookup1("2F911") ->	{"0","23F8E",""};
lookup1("2F912") ->	{"0","6FC6",""};
lookup1("2F913") ->	{"0","7039",""};
lookup1("2F914") ->	{"0","701E",""};
lookup1("2F915") ->	{"0","701B",""};
lookup1("2F916") ->	{"0","3D96",""};
lookup1("2F917") ->	{"0","704A",""};
lookup1("2F918") ->	{"0","707D",""};
lookup1("2F919") ->	{"0","7077",""};
lookup1("2F91A") ->	{"0","70AD",""};
lookup1("2F91B") ->	{"0","20525",""};
lookup1("2F91C") ->	{"0","7145",""};
lookup1("2F91D") ->	{"0","24263",""};
lookup1("2F91E") ->	{"0","719C",""};
lookup1("2F91F") ->	{"0","243AB",""};
lookup1("2F92") ->	{"0","<compat> 898B",""};
lookup1("2F920") ->	{"0","7228",""};
lookup1("2F921") ->	{"0","7235",""};
lookup1("2F922") ->	{"0","7250",""};
lookup1("2F923") ->	{"0","24608",""};
lookup1("2F924") ->	{"0","7280",""};
lookup1("2F925") ->	{"0","7295",""};
lookup1("2F926") ->	{"0","24735",""};
lookup1("2F927") ->	{"0","24814",""};
lookup1("2F928") ->	{"0","737A",""};
lookup1("2F929") ->	{"0","738B",""};
lookup1("2F92A") ->	{"0","3EAC",""};
lookup1("2F92B") ->	{"0","73A5",""};
lookup1("2F92C") ->	{"0","3EB8",""};
lookup1("2F92D") ->	{"0","3EB8",""};
lookup1("2F92E") ->	{"0","7447",""};
lookup1("2F92F") ->	{"0","745C",""};
lookup1("2F93") ->	{"0","<compat> 89D2",""};
lookup1("2F930") ->	{"0","7471",""};
lookup1("2F931") ->	{"0","7485",""};
lookup1("2F932") ->	{"0","74CA",""};
lookup1("2F933") ->	{"0","3F1B",""};
lookup1("2F934") ->	{"0","7524",""};
lookup1("2F935") ->	{"0","24C36",""};
lookup1("2F936") ->	{"0","753E",""};
lookup1("2F937") ->	{"0","24C92",""};
lookup1("2F938") ->	{"0","7570",""};
lookup1("2F939") ->	{"0","2219F",""};
lookup1("2F93A") ->	{"0","7610",""};
lookup1("2F93B") ->	{"0","24FA1",""};
lookup1("2F93C") ->	{"0","24FB8",""};
lookup1("2F93D") ->	{"0","25044",""};
lookup1("2F93E") ->	{"0","3FFC",""};
lookup1("2F93F") ->	{"0","4008",""};
lookup1("2F94") ->	{"0","<compat> 8A00",""};
lookup1("2F940") ->	{"0","76F4",""};
lookup1("2F941") ->	{"0","250F3",""};
lookup1("2F942") ->	{"0","250F2",""};
lookup1("2F943") ->	{"0","25119",""};
lookup1("2F944") ->	{"0","25133",""};
lookup1("2F945") ->	{"0","771E",""};
lookup1("2F946") ->	{"0","771F",""};
lookup1("2F947") ->	{"0","771F",""};
lookup1("2F948") ->	{"0","774A",""};
lookup1("2F949") ->	{"0","4039",""};
lookup1("2F94A") ->	{"0","778B",""};
lookup1("2F94B") ->	{"0","4046",""};
lookup1("2F94C") ->	{"0","4096",""};
lookup1("2F94D") ->	{"0","2541D",""};
lookup1("2F94E") ->	{"0","784E",""};
lookup1("2F94F") ->	{"0","788C",""};
lookup1("2F95") ->	{"0","<compat> 8C37",""};
lookup1("2F950") ->	{"0","78CC",""};
lookup1("2F951") ->	{"0","40E3",""};
lookup1("2F952") ->	{"0","25626",""};
lookup1("2F953") ->	{"0","7956",""};
lookup1("2F954") ->	{"0","2569A",""};
lookup1("2F955") ->	{"0","256C5",""};
lookup1("2F956") ->	{"0","798F",""};
lookup1("2F957") ->	{"0","79EB",""};
lookup1("2F958") ->	{"0","412F",""};
lookup1("2F959") ->	{"0","7A40",""};
lookup1("2F95A") ->	{"0","7A4A",""};
lookup1("2F95B") ->	{"0","7A4F",""};
lookup1("2F95C") ->	{"0","2597C",""};
lookup1("2F95D") ->	{"0","25AA7",""};
lookup1("2F95E") ->	{"0","25AA7",""};
lookup1("2F95F") ->	{"0","7AEE",""};
lookup1("2F96") ->	{"0","<compat> 8C46",""};
lookup1("2F960") ->	{"0","4202",""};
lookup1("2F961") ->	{"0","25BAB",""};
lookup1("2F962") ->	{"0","7BC6",""};
lookup1("2F963") ->	{"0","7BC9",""};
lookup1("2F964") ->	{"0","4227",""};
lookup1("2F965") ->	{"0","25C80",""};
lookup1("2F966") ->	{"0","7CD2",""};
lookup1("2F967") ->	{"0","42A0",""};
lookup1("2F968") ->	{"0","7CE8",""};
lookup1("2F969") ->	{"0","7CE3",""};
lookup1("2F96A") ->	{"0","7D00",""};
lookup1("2F96B") ->	{"0","25F86",""};
lookup1("2F96C") ->	{"0","7D63",""};
lookup1("2F96D") ->	{"0","4301",""};
lookup1("2F96E") ->	{"0","7DC7",""};
lookup1("2F96F") ->	{"0","7E02",""};
lookup1("2F97") ->	{"0","<compat> 8C55",""};
lookup1("2F970") ->	{"0","7E45",""};
lookup1("2F971") ->	{"0","4334",""};
lookup1("2F972") ->	{"0","26228",""};
lookup1("2F973") ->	{"0","26247",""};
lookup1("2F974") ->	{"0","4359",""};
lookup1("2F975") ->	{"0","262D9",""};
lookup1("2F976") ->	{"0","7F7A",""};
lookup1("2F977") ->	{"0","2633E",""};
lookup1("2F978") ->	{"0","7F95",""};
lookup1("2F979") ->	{"0","7FFA",""};
lookup1("2F97A") ->	{"0","8005",""};
lookup1("2F97B") ->	{"0","264DA",""};
lookup1("2F97C") ->	{"0","26523",""};
lookup1("2F97D") ->	{"0","8060",""};
lookup1("2F97E") ->	{"0","265A8",""};
lookup1("2F97F") ->	{"0","8070",""};
lookup1("2F98") ->	{"0","<compat> 8C78",""};
lookup1("2F980") ->	{"0","2335F",""};
lookup1("2F981") ->	{"0","43D5",""};
lookup1("2F982") ->	{"0","80B2",""};
lookup1("2F983") ->	{"0","8103",""};
lookup1("2F984") ->	{"0","440B",""};
lookup1("2F985") ->	{"0","813E",""};
lookup1("2F986") ->	{"0","5AB5",""};
lookup1("2F987") ->	{"0","267A7",""};
lookup1("2F988") ->	{"0","267B5",""};
lookup1("2F989") ->	{"0","23393",""};
lookup1("2F98A") ->	{"0","2339C",""};
lookup1("2F98B") ->	{"0","8201",""};
lookup1("2F98C") ->	{"0","8204",""};
lookup1("2F98D") ->	{"0","8F9E",""};
lookup1("2F98E") ->	{"0","446B",""};
lookup1("2F98F") ->	{"0","8291",""};
lookup1("2F99") ->	{"0","<compat> 8C9D",""};
lookup1("2F990") ->	{"0","828B",""};
lookup1("2F991") ->	{"0","829D",""};
lookup1("2F992") ->	{"0","52B3",""};
lookup1("2F993") ->	{"0","82B1",""};
lookup1("2F994") ->	{"0","82B3",""};
lookup1("2F995") ->	{"0","82BD",""};
lookup1("2F996") ->	{"0","82E6",""};
lookup1("2F997") ->	{"0","26B3C",""};
lookup1("2F998") ->	{"0","82E5",""};
lookup1("2F999") ->	{"0","831D",""};
lookup1("2F99A") ->	{"0","8363",""};
lookup1("2F99B") ->	{"0","83AD",""};
lookup1("2F99C") ->	{"0","8323",""};
lookup1("2F99D") ->	{"0","83BD",""};
lookup1("2F99E") ->	{"0","83E7",""};
lookup1("2F99F") ->	{"0","8457",""};
lookup1("2F9A") ->	{"0","<compat> 8D64",""};
lookup1("2F9A0") ->	{"0","8353",""};
lookup1("2F9A1") ->	{"0","83CA",""};
lookup1("2F9A2") ->	{"0","83CC",""};
lookup1("2F9A3") ->	{"0","83DC",""};
lookup1("2F9A4") ->	{"0","26C36",""};
lookup1("2F9A5") ->	{"0","26D6B",""};
lookup1("2F9A6") ->	{"0","26CD5",""};
lookup1("2F9A7") ->	{"0","452B",""};
lookup1("2F9A8") ->	{"0","84F1",""};
lookup1("2F9A9") ->	{"0","84F3",""};
lookup1("2F9AA") ->	{"0","8516",""};
lookup1("2F9AB") ->	{"0","273CA",""};
lookup1("2F9AC") ->	{"0","8564",""};
lookup1("2F9AD") ->	{"0","26F2C",""};
lookup1("2F9AE") ->	{"0","455D",""};
lookup1("2F9AF") ->	{"0","4561",""};
lookup1("2F9B") ->	{"0","<compat> 8D70",""};
lookup1("2F9B0") ->	{"0","26FB1",""};
lookup1("2F9B1") ->	{"0","270D2",""};
lookup1("2F9B2") ->	{"0","456B",""};
lookup1("2F9B3") ->	{"0","8650",""};
lookup1("2F9B4") ->	{"0","865C",""};
lookup1("2F9B5") ->	{"0","8667",""};
lookup1("2F9B6") ->	{"0","8669",""};
lookup1("2F9B7") ->	{"0","86A9",""};
lookup1("2F9B8") ->	{"0","8688",""};
lookup1("2F9B9") ->	{"0","870E",""};
lookup1("2F9BA") ->	{"0","86E2",""};
lookup1("2F9BB") ->	{"0","8779",""};
lookup1("2F9BC") ->	{"0","8728",""};
lookup1("2F9BD") ->	{"0","876B",""};
lookup1("2F9BE") ->	{"0","8786",""};
lookup1("2F9BF") ->	{"0","45D7",""};
lookup1("2F9C") ->	{"0","<compat> 8DB3",""};
lookup1("2F9C0") ->	{"0","87E1",""};
lookup1("2F9C1") ->	{"0","8801",""};
lookup1("2F9C2") ->	{"0","45F9",""};
lookup1("2F9C3") ->	{"0","8860",""};
lookup1("2F9C4") ->	{"0","8863",""};
lookup1("2F9C5") ->	{"0","27667",""};
lookup1("2F9C6") ->	{"0","88D7",""};
lookup1("2F9C7") ->	{"0","88DE",""};
lookup1("2F9C8") ->	{"0","4635",""};
lookup1("2F9C9") ->	{"0","88FA",""};
lookup1("2F9CA") ->	{"0","34BB",""};
lookup1("2F9CB") ->	{"0","278AE",""};
lookup1("2F9CC") ->	{"0","27966",""};
lookup1("2F9CD") ->	{"0","46BE",""};
lookup1("2F9CE") ->	{"0","46C7",""};
lookup1("2F9CF") ->	{"0","8AA0",""};
lookup1("2F9D") ->	{"0","<compat> 8EAB",""};
lookup1("2F9D0") ->	{"0","8AED",""};
lookup1("2F9D1") ->	{"0","8B8A",""};
lookup1("2F9D2") ->	{"0","8C55",""};
lookup1("2F9D3") ->	{"0","27CA8",""};
lookup1("2F9D4") ->	{"0","8CAB",""};
lookup1("2F9D5") ->	{"0","8CC1",""};
lookup1("2F9D6") ->	{"0","8D1B",""};
lookup1("2F9D7") ->	{"0","8D77",""};
lookup1("2F9D8") ->	{"0","27F2F",""};
lookup1("2F9D9") ->	{"0","20804",""};
lookup1("2F9DA") ->	{"0","8DCB",""};
lookup1("2F9DB") ->	{"0","8DBC",""};
lookup1("2F9DC") ->	{"0","8DF0",""};
lookup1("2F9DD") ->	{"0","208DE",""};
lookup1("2F9DE") ->	{"0","8ED4",""};
lookup1("2F9DF") ->	{"0","8F38",""};
lookup1("2F9E") ->	{"0","<compat> 8ECA",""};
lookup1("2F9E0") ->	{"0","285D2",""};
lookup1("2F9E1") ->	{"0","285ED",""};
lookup1("2F9E2") ->	{"0","9094",""};
lookup1("2F9E3") ->	{"0","90F1",""};
lookup1("2F9E4") ->	{"0","9111",""};
lookup1("2F9E5") ->	{"0","2872E",""};
lookup1("2F9E6") ->	{"0","911B",""};
lookup1("2F9E7") ->	{"0","9238",""};
lookup1("2F9E8") ->	{"0","92D7",""};
lookup1("2F9E9") ->	{"0","92D8",""};
lookup1("2F9EA") ->	{"0","927C",""};
lookup1("2F9EB") ->	{"0","93F9",""};
lookup1("2F9EC") ->	{"0","9415",""};
lookup1("2F9ED") ->	{"0","28BFA",""};
lookup1("2F9EE") ->	{"0","958B",""};
lookup1("2F9EF") ->	{"0","4995",""};
lookup1("2F9F") ->	{"0","<compat> 8F9B",""};
lookup1("2F9F0") ->	{"0","95B7",""};
lookup1("2F9F1") ->	{"0","28D77",""};
lookup1("2F9F2") ->	{"0","49E6",""};
lookup1("2F9F3") ->	{"0","96C3",""};
lookup1("2F9F4") ->	{"0","5DB2",""};
lookup1("2F9F5") ->	{"0","9723",""};
lookup1("2F9F6") ->	{"0","29145",""};
lookup1("2F9F7") ->	{"0","2921A",""};
lookup1("2F9F8") ->	{"0","4A6E",""};
lookup1("2F9F9") ->	{"0","4A76",""};
lookup1("2F9FA") ->	{"0","97E0",""};
lookup1("2F9FB") ->	{"0","2940A",""};
lookup1("2F9FC") ->	{"0","4AB2",""};
lookup1("2F9FD") ->	{"0","29496",""};
lookup1("2F9FE") ->	{"0","980B",""};
lookup1("2F9FF") ->	{"0","980B",""};
lookup1("2FA0") ->	{"0","<compat> 8FB0",""};
lookup1("2FA00") ->	{"0","9829",""};
lookup1("2FA01") ->	{"0","295B6",""};
lookup1("2FA02") ->	{"0","98E2",""};
lookup1("2FA03") ->	{"0","4B33",""};
lookup1("2FA04") ->	{"0","9929",""};
lookup1("2FA05") ->	{"0","99A7",""};
lookup1("2FA06") ->	{"0","99C2",""};
lookup1("2FA07") ->	{"0","99FE",""};
lookup1("2FA08") ->	{"0","4BCE",""};
lookup1("2FA09") ->	{"0","29B30",""};
lookup1("2FA0A") ->	{"0","9B12",""};
lookup1("2FA0B") ->	{"0","9C40",""};
lookup1("2FA0C") ->	{"0","9CFD",""};
lookup1("2FA0D") ->	{"0","4CCE",""};
lookup1("2FA0E") ->	{"0","4CED",""};
lookup1("2FA0F") ->	{"0","9D67",""};
lookup1("2FA1") ->	{"0","<compat> 8FB5",""};
lookup1("2FA10") ->	{"0","2A0CE",""};
lookup1("2FA11") ->	{"0","4CF8",""};
lookup1("2FA12") ->	{"0","2A105",""};
lookup1("2FA13") ->	{"0","2A20E",""};
lookup1("2FA14") ->	{"0","2A291",""};
lookup1("2FA15") ->	{"0","9EBB",""};
lookup1("2FA16") ->	{"0","4D56",""};
lookup1("2FA17") ->	{"0","9EF9",""};
lookup1("2FA18") ->	{"0","9EFE",""};
lookup1("2FA19") ->	{"0","9F05",""};
lookup1("2FA1A") ->	{"0","9F0F",""};
lookup1("2FA1B") ->	{"0","9F16",""};
lookup1("2FA1C") ->	{"0","9F3B",""};
lookup1("2FA1D") ->	{"0","2A600",""};
lookup1("2FA2") ->	{"0","<compat> 9091",""};
lookup1("2FA3") ->	{"0","<compat> 9149",""};
lookup1("2FA4") ->	{"0","<compat> 91C6",""};
lookup1("2FA5") ->	{"0","<compat> 91CC",""};
lookup1("2FA6") ->	{"0","<compat> 91D1",""};
lookup1("2FA7") ->	{"0","<compat> 9577",""};
lookup1("2FA8") ->	{"0","<compat> 9580",""};
lookup1("2FA9") ->	{"0","<compat> 961C",""};
lookup1("2FAA") ->	{"0","<compat> 96B6",""};
lookup1("2FAB") ->	{"0","<compat> 96B9",""};
lookup1("2FAC") ->	{"0","<compat> 96E8",""};
lookup1("2FAD") ->	{"0","<compat> 9751",""};
lookup1("2FAE") ->	{"0","<compat> 975E",""};
lookup1("2FAF") ->	{"0","<compat> 9762",""};
lookup1("2FB0") ->	{"0","<compat> 9769",""};
lookup1("2FB1") ->	{"0","<compat> 97CB",""};
lookup1("2FB2") ->	{"0","<compat> 97ED",""};
lookup1("2FB3") ->	{"0","<compat> 97F3",""};
lookup1("2FB4") ->	{"0","<compat> 9801",""};
lookup1("2FB5") ->	{"0","<compat> 98A8",""};
lookup1("2FB6") ->	{"0","<compat> 98DB",""};
lookup1("2FB7") ->	{"0","<compat> 98DF",""};
lookup1("2FB8") ->	{"0","<compat> 9996",""};
lookup1("2FB9") ->	{"0","<compat> 9999",""};
lookup1("2FBA") ->	{"0","<compat> 99AC",""};
lookup1("2FBB") ->	{"0","<compat> 9AA8",""};
lookup1("2FBC") ->	{"0","<compat> 9AD8",""};
lookup1("2FBD") ->	{"0","<compat> 9ADF",""};
lookup1("2FBE") ->	{"0","<compat> 9B25",""};
lookup1("2FBF") ->	{"0","<compat> 9B2F",""};
lookup1("2FC0") ->	{"0","<compat> 9B32",""};
lookup1("2FC1") ->	{"0","<compat> 9B3C",""};
lookup1("2FC2") ->	{"0","<compat> 9B5A",""};
lookup1("2FC3") ->	{"0","<compat> 9CE5",""};
lookup1("2FC4") ->	{"0","<compat> 9E75",""};
lookup1("2FC5") ->	{"0","<compat> 9E7F",""};
lookup1("2FC6") ->	{"0","<compat> 9EA5",""};
lookup1("2FC7") ->	{"0","<compat> 9EBB",""};
lookup1("2FC8") ->	{"0","<compat> 9EC3",""};
lookup1("2FC9") ->	{"0","<compat> 9ECD",""};
lookup1("2FCA") ->	{"0","<compat> 9ED1",""};
lookup1("2FCB") ->	{"0","<compat> 9EF9",""};
lookup1("2FCC") ->	{"0","<compat> 9EFD",""};
lookup1("2FCD") ->	{"0","<compat> 9F0E",""};
lookup1("2FCE") ->	{"0","<compat> 9F13",""};
lookup1("2FCF") ->	{"0","<compat> 9F20",""};
lookup1("2FD0") ->	{"0","<compat> 9F3B",""};
lookup1("2FD1") ->	{"0","<compat> 9F4A",""};
lookup1("2FD2") ->	{"0","<compat> 9F52",""};
lookup1("2FD3") ->	{"0","<compat> 9F8D",""};
lookup1("2FD4") ->	{"0","<compat> 9F9C",""};
lookup1("2FD5") ->	{"0","<compat> 9FA0",""};
lookup1("2FF0") ->	{"0","",""};
lookup1("2FF1") ->	{"0","",""};
lookup1("2FF2") ->	{"0","",""};
lookup1("2FF3") ->	{"0","",""};
lookup1("2FF4") ->	{"0","",""};
lookup1("2FF5") ->	{"0","",""};
lookup1("2FF6") ->	{"0","",""};
lookup1("2FF7") ->	{"0","",""};
lookup1("2FF8") ->	{"0","",""};
lookup1("2FF9") ->	{"0","",""};
lookup1("2FFA") ->	{"0","",""};
lookup1("2FFB") ->	{"0","",""};
lookup1("3000") ->	{"0","<wide> 0020",""};
lookup1("3001") ->	{"0","",""};
lookup1("3002") ->	{"0","",""};
lookup1("3003") ->	{"0","",""};
lookup1("3004") ->	{"0","",""};
lookup1("3005") ->	{"0","",""};
lookup1("3006") ->	{"0","",""};
lookup1("3007") ->	{"0","",""};
lookup1("3008") ->	{"0","",""};
lookup1("3009") ->	{"0","",""};
lookup1("300A") ->	{"0","",""};
lookup1("300B") ->	{"0","",""};
lookup1("300C") ->	{"0","",""};
lookup1("300D") ->	{"0","",""};
lookup1("300E") ->	{"0","",""};
lookup1("300F") ->	{"0","",""};
lookup1("3010") ->	{"0","",""};
lookup1("3011") ->	{"0","",""};
lookup1("3012") ->	{"0","",""};
lookup1("3013") ->	{"0","",""};
lookup1("3014") ->	{"0","",""};
lookup1("3015") ->	{"0","",""};
lookup1("3016") ->	{"0","",""};
lookup1("3017") ->	{"0","",""};
lookup1("3018") ->	{"0","",""};
lookup1("3019") ->	{"0","",""};
lookup1("301A") ->	{"0","",""};
lookup1("301B") ->	{"0","",""};
lookup1("301C") ->	{"0","",""};
lookup1("301D") ->	{"0","",""};
lookup1("301E") ->	{"0","",""};
lookup1("301F") ->	{"0","",""};
lookup1("3020") ->	{"0","",""};
lookup1("3021") ->	{"0","",""};
lookup1("3022") ->	{"0","",""};
lookup1("3023") ->	{"0","",""};
lookup1("3024") ->	{"0","",""};
lookup1("3025") ->	{"0","",""};
lookup1("3026") ->	{"0","",""};
lookup1("3027") ->	{"0","",""};
lookup1("3028") ->	{"0","",""};
lookup1("3029") ->	{"0","",""};
lookup1("302A") ->	{"218","",""};
lookup1("302B") ->	{"228","",""};
lookup1("302C") ->	{"232","",""};
lookup1("302D") ->	{"222","",""};
lookup1("302E") ->	{"224","",""};
lookup1("302F") ->	{"224","",""};
lookup1("3030") ->	{"0","",""};
lookup1("3031") ->	{"0","",""};
lookup1("3032") ->	{"0","",""};
lookup1("3033") ->	{"0","",""};
lookup1("3034") ->	{"0","",""};
lookup1("3035") ->	{"0","",""};
lookup1("3036") ->	{"0","<compat> 3012",""};
lookup1("3037") ->	{"0","",""};
lookup1("3038") ->	{"0","<compat> 5341",""};
lookup1("3039") ->	{"0","<compat> 5344",""};
lookup1("303A") ->	{"0","<compat> 5345",""};
lookup1("303B") ->	{"0","",""};
lookup1("303C") ->	{"0","",""};
lookup1("303D") ->	{"0","",""};
lookup1("303E") ->	{"0","",""};
lookup1("303F") ->	{"0","",""};
lookup1("3041") ->	{"0","",""};
lookup1("3042") ->	{"0","",""};
lookup1("3043") ->	{"0","",""};
lookup1("3044") ->	{"0","",""};
lookup1("3045") ->	{"0","",""};
lookup1("3046") ->	{"0","",""};
lookup1("3047") ->	{"0","",""};
lookup1("3048") ->	{"0","",""};
lookup1("3049") ->	{"0","",""};
lookup1("304A") ->	{"0","",""};
lookup1("304B") ->	{"0","",""};
lookup1("304C") ->	{"0","304B 3099",""};
lookup1("304D") ->	{"0","",""};
lookup1("304E") ->	{"0","304D 3099",""};
lookup1("304F") ->	{"0","",""};
lookup1("3050") ->	{"0","304F 3099",""};
lookup1("3051") ->	{"0","",""};
lookup1("3052") ->	{"0","3051 3099",""};
lookup1("3053") ->	{"0","",""};
lookup1("3054") ->	{"0","3053 3099",""};
lookup1("3055") ->	{"0","",""};
lookup1("3056") ->	{"0","3055 3099",""};
lookup1("3057") ->	{"0","",""};
lookup1("3058") ->	{"0","3057 3099",""};
lookup1("3059") ->	{"0","",""};
lookup1("305A") ->	{"0","3059 3099",""};
lookup1("305B") ->	{"0","",""};
lookup1("305C") ->	{"0","305B 3099",""};
lookup1("305D") ->	{"0","",""};
lookup1("305E") ->	{"0","305D 3099",""};
lookup1("305F") ->	{"0","",""};
lookup1("3060") ->	{"0","305F 3099",""};
lookup1("3061") ->	{"0","",""};
lookup1("3062") ->	{"0","3061 3099",""};
lookup1("3063") ->	{"0","",""};
lookup1("3064") ->	{"0","",""};
lookup1("3065") ->	{"0","3064 3099",""};
lookup1("3066") ->	{"0","",""};
lookup1("3067") ->	{"0","3066 3099",""};
lookup1("3068") ->	{"0","",""};
lookup1("3069") ->	{"0","3068 3099",""};
lookup1("306A") ->	{"0","",""};
lookup1("306B") ->	{"0","",""};
lookup1("306C") ->	{"0","",""};
lookup1("306D") ->	{"0","",""};
lookup1("306E") ->	{"0","",""};
lookup1("306F") ->	{"0","",""};
lookup1("3070") ->	{"0","306F 3099",""};
lookup1("3071") ->	{"0","306F 309A",""};
lookup1("3072") ->	{"0","",""};
lookup1("3073") ->	{"0","3072 3099",""};
lookup1("3074") ->	{"0","3072 309A",""};
lookup1("3075") ->	{"0","",""};
lookup1("3076") ->	{"0","3075 3099",""};
lookup1("3077") ->	{"0","3075 309A",""};
lookup1("3078") ->	{"0","",""};
lookup1("3079") ->	{"0","3078 3099",""};
lookup1("307A") ->	{"0","3078 309A",""};
lookup1("307B") ->	{"0","",""};
lookup1("307C") ->	{"0","307B 3099",""};
lookup1("307D") ->	{"0","307B 309A",""};
lookup1("307E") ->	{"0","",""};
lookup1("307F") ->	{"0","",""};
lookup1("3080") ->	{"0","",""};
lookup1("3081") ->	{"0","",""};
lookup1("3082") ->	{"0","",""};
lookup1("3083") ->	{"0","",""};
lookup1("3084") ->	{"0","",""};
lookup1("3085") ->	{"0","",""};
lookup1("3086") ->	{"0","",""};
lookup1("3087") ->	{"0","",""};
lookup1("3088") ->	{"0","",""};
lookup1("3089") ->	{"0","",""};
lookup1("308A") ->	{"0","",""};
lookup1("308B") ->	{"0","",""};
lookup1("308C") ->	{"0","",""};
lookup1("308D") ->	{"0","",""};
lookup1("308E") ->	{"0","",""};
lookup1("308F") ->	{"0","",""};
lookup1("3090") ->	{"0","",""};
lookup1("3091") ->	{"0","",""};
lookup1("3092") ->	{"0","",""};
lookup1("3093") ->	{"0","",""};
lookup1("3094") ->	{"0","3046 3099",""};
lookup1("3095") ->	{"0","",""};
lookup1("3096") ->	{"0","",""};
lookup1("3099") ->	{"8","",""};
lookup1("309A") ->	{"8","",""};
lookup1("309B") ->	{"0","<compat> 0020 3099",""};
lookup1("309C") ->	{"0","<compat> 0020 309A",""};
lookup1("309D") ->	{"0","",""};
lookup1("309E") ->	{"0","309D 3099",""};
lookup1("309F") ->	{"0","<vertical> 3088 308A",""};
lookup1("30A0") ->	{"0","",""};
lookup1("30A1") ->	{"0","",""};
lookup1("30A2") ->	{"0","",""};
lookup1("30A3") ->	{"0","",""};
lookup1("30A4") ->	{"0","",""};
lookup1("30A5") ->	{"0","",""};
lookup1("30A6") ->	{"0","",""};
lookup1("30A7") ->	{"0","",""};
lookup1("30A8") ->	{"0","",""};
lookup1("30A9") ->	{"0","",""};
lookup1("30AA") ->	{"0","",""};
lookup1("30AB") ->	{"0","",""};
lookup1("30AC") ->	{"0","30AB 3099",""};
lookup1("30AD") ->	{"0","",""};
lookup1("30AE") ->	{"0","30AD 3099",""};
lookup1("30AF") ->	{"0","",""};
lookup1("30B0") ->	{"0","30AF 3099",""};
lookup1("30B1") ->	{"0","",""};
lookup1("30B2") ->	{"0","30B1 3099",""};
lookup1("30B3") ->	{"0","",""};
lookup1("30B4") ->	{"0","30B3 3099",""};
lookup1("30B5") ->	{"0","",""};
lookup1("30B6") ->	{"0","30B5 3099",""};
lookup1("30B7") ->	{"0","",""};
lookup1("30B8") ->	{"0","30B7 3099",""};
lookup1("30B9") ->	{"0","",""};
lookup1("30BA") ->	{"0","30B9 3099",""};
lookup1("30BB") ->	{"0","",""};
lookup1("30BC") ->	{"0","30BB 3099",""};
lookup1("30BD") ->	{"0","",""};
lookup1("30BE") ->	{"0","30BD 3099",""};
lookup1("30BF") ->	{"0","",""};
lookup1("30C0") ->	{"0","30BF 3099",""};
lookup1("30C1") ->	{"0","",""};
lookup1("30C2") ->	{"0","30C1 3099",""};
lookup1("30C3") ->	{"0","",""};
lookup1("30C4") ->	{"0","",""};
lookup1("30C5") ->	{"0","30C4 3099",""};
lookup1("30C6") ->	{"0","",""};
lookup1("30C7") ->	{"0","30C6 3099",""};
lookup1("30C8") ->	{"0","",""};
lookup1("30C9") ->	{"0","30C8 3099",""};
lookup1("30CA") ->	{"0","",""};
lookup1("30CB") ->	{"0","",""};
lookup1("30CC") ->	{"0","",""};
lookup1("30CD") ->	{"0","",""};
lookup1("30CE") ->	{"0","",""};
lookup1("30CF") ->	{"0","",""};
lookup1("30D0") ->	{"0","30CF 3099",""};
lookup1("30D1") ->	{"0","30CF 309A",""};
lookup1("30D2") ->	{"0","",""};
lookup1("30D3") ->	{"0","30D2 3099",""};
lookup1("30D4") ->	{"0","30D2 309A",""};
lookup1("30D5") ->	{"0","",""};
lookup1("30D6") ->	{"0","30D5 3099",""};
lookup1("30D7") ->	{"0","30D5 309A",""};
lookup1("30D8") ->	{"0","",""};
lookup1("30D9") ->	{"0","30D8 3099",""};
lookup1("30DA") ->	{"0","30D8 309A",""};
lookup1("30DB") ->	{"0","",""};
lookup1("30DC") ->	{"0","30DB 3099",""};
lookup1("30DD") ->	{"0","30DB 309A",""};
lookup1("30DE") ->	{"0","",""};
lookup1("30DF") ->	{"0","",""};
lookup1("30E0") ->	{"0","",""};
lookup1("30E1") ->	{"0","",""};
lookup1("30E2") ->	{"0","",""};
lookup1("30E3") ->	{"0","",""};
lookup1("30E4") ->	{"0","",""};
lookup1("30E5") ->	{"0","",""};
lookup1("30E6") ->	{"0","",""};
lookup1("30E7") ->	{"0","",""};
lookup1("30E8") ->	{"0","",""};
lookup1("30E9") ->	{"0","",""};
lookup1("30EA") ->	{"0","",""};
lookup1("30EB") ->	{"0","",""};
lookup1("30EC") ->	{"0","",""};
lookup1("30ED") ->	{"0","",""};
lookup1("30EE") ->	{"0","",""};
lookup1("30EF") ->	{"0","",""};
lookup1("30F0") ->	{"0","",""};
lookup1("30F1") ->	{"0","",""};
lookup1("30F2") ->	{"0","",""};
lookup1("30F3") ->	{"0","",""};
lookup1("30F4") ->	{"0","30A6 3099",""};
lookup1("30F5") ->	{"0","",""};
lookup1("30F6") ->	{"0","",""};
lookup1("30F7") ->	{"0","30EF 3099",""};
lookup1("30F8") ->	{"0","30F0 3099",""};
lookup1("30F9") ->	{"0","30F1 3099",""};
lookup1("30FA") ->	{"0","30F2 3099",""};
lookup1("30FB") ->	{"0","",""};
lookup1("30FC") ->	{"0","",""};
lookup1("30FD") ->	{"0","",""};
lookup1("30FE") ->	{"0","30FD 3099",""};
lookup1("30FF") ->	{"0","<vertical> 30B3 30C8",""};
lookup1("3105") ->	{"0","",""};
lookup1("3106") ->	{"0","",""};
lookup1("3107") ->	{"0","",""};
lookup1("3108") ->	{"0","",""};
lookup1("3109") ->	{"0","",""};
lookup1("310A") ->	{"0","",""};
lookup1("310B") ->	{"0","",""};
lookup1("310C") ->	{"0","",""};
lookup1("310D") ->	{"0","",""};
lookup1("310E") ->	{"0","",""};
lookup1("310F") ->	{"0","",""};
lookup1("3110") ->	{"0","",""};
lookup1("3111") ->	{"0","",""};
lookup1("3112") ->	{"0","",""};
lookup1("3113") ->	{"0","",""};
lookup1("3114") ->	{"0","",""};
lookup1("3115") ->	{"0","",""};
lookup1("3116") ->	{"0","",""};
lookup1("3117") ->	{"0","",""};
lookup1("3118") ->	{"0","",""};
lookup1("3119") ->	{"0","",""};
lookup1("311A") ->	{"0","",""};
lookup1("311B") ->	{"0","",""};
lookup1("311C") ->	{"0","",""};
lookup1("311D") ->	{"0","",""};
lookup1("311E") ->	{"0","",""};
lookup1("311F") ->	{"0","",""};
lookup1("3120") ->	{"0","",""};
lookup1("3121") ->	{"0","",""};
lookup1("3122") ->	{"0","",""};
lookup1("3123") ->	{"0","",""};
lookup1("3124") ->	{"0","",""};
lookup1("3125") ->	{"0","",""};
lookup1("3126") ->	{"0","",""};
lookup1("3127") ->	{"0","",""};
lookup1("3128") ->	{"0","",""};
lookup1("3129") ->	{"0","",""};
lookup1("312A") ->	{"0","",""};
lookup1("312B") ->	{"0","",""};
lookup1("312C") ->	{"0","",""};
lookup1("312D") ->	{"0","",""};
lookup1("3131") ->	{"0","<compat> 1100",""};
lookup1("3132") ->	{"0","<compat> 1101",""};
lookup1("3133") ->	{"0","<compat> 11AA",""};
lookup1("3134") ->	{"0","<compat> 1102",""};
lookup1("3135") ->	{"0","<compat> 11AC",""};
lookup1("3136") ->	{"0","<compat> 11AD",""};
lookup1("3137") ->	{"0","<compat> 1103",""};
lookup1("3138") ->	{"0","<compat> 1104",""};
lookup1("3139") ->	{"0","<compat> 1105",""};
lookup1("313A") ->	{"0","<compat> 11B0",""};
lookup1("313B") ->	{"0","<compat> 11B1",""};
lookup1("313C") ->	{"0","<compat> 11B2",""};
lookup1("313D") ->	{"0","<compat> 11B3",""};
lookup1("313E") ->	{"0","<compat> 11B4",""};
lookup1("313F") ->	{"0","<compat> 11B5",""};
lookup1("3140") ->	{"0","<compat> 111A",""};
lookup1("3141") ->	{"0","<compat> 1106",""};
lookup1("3142") ->	{"0","<compat> 1107",""};
lookup1("3143") ->	{"0","<compat> 1108",""};
lookup1("3144") ->	{"0","<compat> 1121",""};
lookup1("3145") ->	{"0","<compat> 1109",""};
lookup1("3146") ->	{"0","<compat> 110A",""};
lookup1("3147") ->	{"0","<compat> 110B",""};
lookup1("3148") ->	{"0","<compat> 110C",""};
lookup1("3149") ->	{"0","<compat> 110D",""};
lookup1("314A") ->	{"0","<compat> 110E",""};
lookup1("314B") ->	{"0","<compat> 110F",""};
lookup1("314C") ->	{"0","<compat> 1110",""};
lookup1("314D") ->	{"0","<compat> 1111",""};
lookup1("314E") ->	{"0","<compat> 1112",""};
lookup1("314F") ->	{"0","<compat> 1161",""};
lookup1("3150") ->	{"0","<compat> 1162",""};
lookup1("3151") ->	{"0","<compat> 1163",""};
lookup1("3152") ->	{"0","<compat> 1164",""};
lookup1("3153") ->	{"0","<compat> 1165",""};
lookup1("3154") ->	{"0","<compat> 1166",""};
lookup1("3155") ->	{"0","<compat> 1167",""};
lookup1("3156") ->	{"0","<compat> 1168",""};
lookup1("3157") ->	{"0","<compat> 1169",""};
lookup1("3158") ->	{"0","<compat> 116A",""};
lookup1("3159") ->	{"0","<compat> 116B",""};
lookup1("315A") ->	{"0","<compat> 116C",""};
lookup1("315B") ->	{"0","<compat> 116D",""};
lookup1("315C") ->	{"0","<compat> 116E",""};
lookup1("315D") ->	{"0","<compat> 116F",""};
lookup1("315E") ->	{"0","<compat> 1170",""};
lookup1("315F") ->	{"0","<compat> 1171",""};
lookup1("3160") ->	{"0","<compat> 1172",""};
lookup1("3161") ->	{"0","<compat> 1173",""};
lookup1("3162") ->	{"0","<compat> 1174",""};
lookup1("3163") ->	{"0","<compat> 1175",""};
lookup1("3164") ->	{"0","<compat> 1160",""};
lookup1("3165") ->	{"0","<compat> 1114",""};
lookup1("3166") ->	{"0","<compat> 1115",""};
lookup1("3167") ->	{"0","<compat> 11C7",""};
lookup1("3168") ->	{"0","<compat> 11C8",""};
lookup1("3169") ->	{"0","<compat> 11CC",""};
lookup1("316A") ->	{"0","<compat> 11CE",""};
lookup1("316B") ->	{"0","<compat> 11D3",""};
lookup1("316C") ->	{"0","<compat> 11D7",""};
lookup1("316D") ->	{"0","<compat> 11D9",""};
lookup1("316E") ->	{"0","<compat> 111C",""};
lookup1("316F") ->	{"0","<compat> 11DD",""};
lookup1("3170") ->	{"0","<compat> 11DF",""};
lookup1("3171") ->	{"0","<compat> 111D",""};
lookup1("3172") ->	{"0","<compat> 111E",""};
lookup1("3173") ->	{"0","<compat> 1120",""};
lookup1("3174") ->	{"0","<compat> 1122",""};
lookup1("3175") ->	{"0","<compat> 1123",""};
lookup1("3176") ->	{"0","<compat> 1127",""};
lookup1("3177") ->	{"0","<compat> 1129",""};
lookup1("3178") ->	{"0","<compat> 112B",""};
lookup1("3179") ->	{"0","<compat> 112C",""};
lookup1("317A") ->	{"0","<compat> 112D",""};
lookup1("317B") ->	{"0","<compat> 112E",""};
lookup1("317C") ->	{"0","<compat> 112F",""};
lookup1("317D") ->	{"0","<compat> 1132",""};
lookup1("317E") ->	{"0","<compat> 1136",""};
lookup1("317F") ->	{"0","<compat> 1140",""};
lookup1("3180") ->	{"0","<compat> 1147",""};
lookup1("3181") ->	{"0","<compat> 114C",""};
lookup1("3182") ->	{"0","<compat> 11F1",""};
lookup1("3183") ->	{"0","<compat> 11F2",""};
lookup1("3184") ->	{"0","<compat> 1157",""};
lookup1("3185") ->	{"0","<compat> 1158",""};
lookup1("3186") ->	{"0","<compat> 1159",""};
lookup1("3187") ->	{"0","<compat> 1184",""};
lookup1("3188") ->	{"0","<compat> 1185",""};
lookup1("3189") ->	{"0","<compat> 1188",""};
lookup1("318A") ->	{"0","<compat> 1191",""};
lookup1("318B") ->	{"0","<compat> 1192",""};
lookup1("318C") ->	{"0","<compat> 1194",""};
lookup1("318D") ->	{"0","<compat> 119E",""};
lookup1("318E") ->	{"0","<compat> 11A1",""};
lookup1("3190") ->	{"0","",""};
lookup1("3191") ->	{"0","",""};
lookup1("3192") ->	{"0","<super> 4E00",""};
lookup1("3193") ->	{"0","<super> 4E8C",""};
lookup1("3194") ->	{"0","<super> 4E09",""};
lookup1("3195") ->	{"0","<super> 56DB",""};
lookup1("3196") ->	{"0","<super> 4E0A",""};
lookup1("3197") ->	{"0","<super> 4E2D",""};
lookup1("3198") ->	{"0","<super> 4E0B",""};
lookup1("3199") ->	{"0","<super> 7532",""};
lookup1("319A") ->	{"0","<super> 4E59",""};
lookup1("319B") ->	{"0","<super> 4E19",""};
lookup1("319C") ->	{"0","<super> 4E01",""};
lookup1("319D") ->	{"0","<super> 5929",""};
lookup1("319E") ->	{"0","<super> 5730",""};
lookup1("319F") ->	{"0","<super> 4EBA",""};
lookup1("31A0") ->	{"0","",""};
lookup1("31A1") ->	{"0","",""};
lookup1("31A2") ->	{"0","",""};
lookup1("31A3") ->	{"0","",""};
lookup1("31A4") ->	{"0","",""};
lookup1("31A5") ->	{"0","",""};
lookup1("31A6") ->	{"0","",""};
lookup1("31A7") ->	{"0","",""};
lookup1("31A8") ->	{"0","",""};
lookup1("31A9") ->	{"0","",""};
lookup1("31AA") ->	{"0","",""};
lookup1("31AB") ->	{"0","",""};
lookup1("31AC") ->	{"0","",""};
lookup1("31AD") ->	{"0","",""};
lookup1("31AE") ->	{"0","",""};
lookup1("31AF") ->	{"0","",""};
lookup1("31B0") ->	{"0","",""};
lookup1("31B1") ->	{"0","",""};
lookup1("31B2") ->	{"0","",""};
lookup1("31B3") ->	{"0","",""};
lookup1("31B4") ->	{"0","",""};
lookup1("31B5") ->	{"0","",""};
lookup1("31B6") ->	{"0","",""};
lookup1("31B7") ->	{"0","",""};
lookup1("31B8") ->	{"0","",""};
lookup1("31B9") ->	{"0","",""};
lookup1("31BA") ->	{"0","",""};
lookup1("31C0") ->	{"0","",""};
lookup1("31C1") ->	{"0","",""};
lookup1("31C2") ->	{"0","",""};
lookup1("31C3") ->	{"0","",""};
lookup1("31C4") ->	{"0","",""};
lookup1("31C5") ->	{"0","",""};
lookup1("31C6") ->	{"0","",""};
lookup1("31C7") ->	{"0","",""};
lookup1("31C8") ->	{"0","",""};
lookup1("31C9") ->	{"0","",""};
lookup1("31CA") ->	{"0","",""};
lookup1("31CB") ->	{"0","",""};
lookup1("31CC") ->	{"0","",""};
lookup1("31CD") ->	{"0","",""};
lookup1("31CE") ->	{"0","",""};
lookup1("31CF") ->	{"0","",""};
lookup1("31D0") ->	{"0","",""};
lookup1("31D1") ->	{"0","",""};
lookup1("31D2") ->	{"0","",""};
lookup1("31D3") ->	{"0","",""};
lookup1("31D4") ->	{"0","",""};
lookup1("31D5") ->	{"0","",""};
lookup1("31D6") ->	{"0","",""};
lookup1("31D7") ->	{"0","",""};
lookup1("31D8") ->	{"0","",""};
lookup1("31D9") ->	{"0","",""};
lookup1("31DA") ->	{"0","",""};
lookup1("31DB") ->	{"0","",""};
lookup1("31DC") ->	{"0","",""};
lookup1("31DD") ->	{"0","",""};
lookup1("31DE") ->	{"0","",""};
lookup1("31DF") ->	{"0","",""};
lookup1("31E0") ->	{"0","",""};
lookup1("31E1") ->	{"0","",""};
lookup1("31E2") ->	{"0","",""};
lookup1("31E3") ->	{"0","",""};
lookup1("31F0") ->	{"0","",""};
lookup1("31F1") ->	{"0","",""};
lookup1("31F2") ->	{"0","",""};
lookup1("31F3") ->	{"0","",""};
lookup1("31F4") ->	{"0","",""};
lookup1("31F5") ->	{"0","",""};
lookup1("31F6") ->	{"0","",""};
lookup1("31F7") ->	{"0","",""};
lookup1("31F8") ->	{"0","",""};
lookup1("31F9") ->	{"0","",""};
lookup1("31FA") ->	{"0","",""};
lookup1("31FB") ->	{"0","",""};
lookup1("31FC") ->	{"0","",""};
lookup1("31FD") ->	{"0","",""};
lookup1("31FE") ->	{"0","",""};
lookup1("31FF") ->	{"0","",""};
lookup1("3200") ->	{"0","<compat> 0028 1100 0029",""};
lookup1("3201") ->	{"0","<compat> 0028 1102 0029",""};
lookup1("3202") ->	{"0","<compat> 0028 1103 0029",""};
lookup1("3203") ->	{"0","<compat> 0028 1105 0029",""};
lookup1("3204") ->	{"0","<compat> 0028 1106 0029",""};
lookup1("3205") ->	{"0","<compat> 0028 1107 0029",""};
lookup1("3206") ->	{"0","<compat> 0028 1109 0029",""};
lookup1("3207") ->	{"0","<compat> 0028 110B 0029",""};
lookup1("3208") ->	{"0","<compat> 0028 110C 0029",""};
lookup1("3209") ->	{"0","<compat> 0028 110E 0029",""};
lookup1("320A") ->	{"0","<compat> 0028 110F 0029",""};
lookup1("320B") ->	{"0","<compat> 0028 1110 0029",""};
lookup1("320C") ->	{"0","<compat> 0028 1111 0029",""};
lookup1("320D") ->	{"0","<compat> 0028 1112 0029",""};
lookup1("320E") ->	{"0","<compat> 0028 1100 1161 0029",""};
lookup1("320F") ->	{"0","<compat> 0028 1102 1161 0029",""};
lookup1("3210") ->	{"0","<compat> 0028 1103 1161 0029",""};
lookup1("3211") ->	{"0","<compat> 0028 1105 1161 0029",""};
lookup1("3212") ->	{"0","<compat> 0028 1106 1161 0029",""};
lookup1("3213") ->	{"0","<compat> 0028 1107 1161 0029",""};
lookup1("3214") ->	{"0","<compat> 0028 1109 1161 0029",""};
lookup1("3215") ->	{"0","<compat> 0028 110B 1161 0029",""};
lookup1("3216") ->	{"0","<compat> 0028 110C 1161 0029",""};
lookup1("3217") ->	{"0","<compat> 0028 110E 1161 0029",""};
lookup1("3218") ->	{"0","<compat> 0028 110F 1161 0029",""};
lookup1("3219") ->	{"0","<compat> 0028 1110 1161 0029",""};
lookup1("321A") ->	{"0","<compat> 0028 1111 1161 0029",""};
lookup1("321B") ->	{"0","<compat> 0028 1112 1161 0029",""};
lookup1("321C") ->	{"0","<compat> 0028 110C 116E 0029",""};
lookup1("321D") ->	{"0","<compat> 0028 110B 1169 110C 1165 11AB 0029",""};
lookup1("321E") ->	{"0","<compat> 0028 110B 1169 1112 116E 0029",""};
lookup1("3220") ->	{"0","<compat> 0028 4E00 0029",""};
lookup1("3221") ->	{"0","<compat> 0028 4E8C 0029",""};
lookup1("3222") ->	{"0","<compat> 0028 4E09 0029",""};
lookup1("3223") ->	{"0","<compat> 0028 56DB 0029",""};
lookup1("3224") ->	{"0","<compat> 0028 4E94 0029",""};
lookup1("3225") ->	{"0","<compat> 0028 516D 0029",""};
lookup1("3226") ->	{"0","<compat> 0028 4E03 0029",""};
lookup1("3227") ->	{"0","<compat> 0028 516B 0029",""};
lookup1("3228") ->	{"0","<compat> 0028 4E5D 0029",""};
lookup1("3229") ->	{"0","<compat> 0028 5341 0029",""};
lookup1("322A") ->	{"0","<compat> 0028 6708 0029",""};
lookup1("322B") ->	{"0","<compat> 0028 706B 0029",""};
lookup1("322C") ->	{"0","<compat> 0028 6C34 0029",""};
lookup1("322D") ->	{"0","<compat> 0028 6728 0029",""};
lookup1("322E") ->	{"0","<compat> 0028 91D1 0029",""};
lookup1("322F") ->	{"0","<compat> 0028 571F 0029",""};
lookup1("3230") ->	{"0","<compat> 0028 65E5 0029",""};
lookup1("3231") ->	{"0","<compat> 0028 682A 0029",""};
lookup1("3232") ->	{"0","<compat> 0028 6709 0029",""};
lookup1("3233") ->	{"0","<compat> 0028 793E 0029",""};
lookup1("3234") ->	{"0","<compat> 0028 540D 0029",""};
lookup1("3235") ->	{"0","<compat> 0028 7279 0029",""};
lookup1("3236") ->	{"0","<compat> 0028 8CA1 0029",""};
lookup1("3237") ->	{"0","<compat> 0028 795D 0029",""};
lookup1("3238") ->	{"0","<compat> 0028 52B4 0029",""};
lookup1("3239") ->	{"0","<compat> 0028 4EE3 0029",""};
lookup1("323A") ->	{"0","<compat> 0028 547C 0029",""};
lookup1("323B") ->	{"0","<compat> 0028 5B66 0029",""};
lookup1("323C") ->	{"0","<compat> 0028 76E3 0029",""};
lookup1("323D") ->	{"0","<compat> 0028 4F01 0029",""};
lookup1("323E") ->	{"0","<compat> 0028 8CC7 0029",""};
lookup1("323F") ->	{"0","<compat> 0028 5354 0029",""};
lookup1("3240") ->	{"0","<compat> 0028 796D 0029",""};
lookup1("3241") ->	{"0","<compat> 0028 4F11 0029",""};
lookup1("3242") ->	{"0","<compat> 0028 81EA 0029",""};
lookup1("3243") ->	{"0","<compat> 0028 81F3 0029",""};
lookup1("3244") ->	{"0","<circle> 554F",""};
lookup1("3245") ->	{"0","<circle> 5E7C",""};
lookup1("3246") ->	{"0","<circle> 6587",""};
lookup1("3247") ->	{"0","<circle> 7B8F",""};
lookup1("3248") ->	{"0","",""};
lookup1("3249") ->	{"0","",""};
lookup1("324A") ->	{"0","",""};
lookup1("324B") ->	{"0","",""};
lookup1("324C") ->	{"0","",""};
lookup1("324D") ->	{"0","",""};
lookup1("324E") ->	{"0","",""};
lookup1("324F") ->	{"0","",""};
lookup1("3250") ->	{"0","<square> 0050 0054 0045",""};
lookup1("3251") ->	{"0","<circle> 0032 0031",""};
lookup1("3252") ->	{"0","<circle> 0032 0032",""};
lookup1("3253") ->	{"0","<circle> 0032 0033",""};
lookup1("3254") ->	{"0","<circle> 0032 0034",""};
lookup1("3255") ->	{"0","<circle> 0032 0035",""};
lookup1("3256") ->	{"0","<circle> 0032 0036",""};
lookup1("3257") ->	{"0","<circle> 0032 0037",""};
lookup1("3258") ->	{"0","<circle> 0032 0038",""};
lookup1("3259") ->	{"0","<circle> 0032 0039",""};
lookup1("325A") ->	{"0","<circle> 0033 0030",""};
lookup1("325B") ->	{"0","<circle> 0033 0031",""};
lookup1("325C") ->	{"0","<circle> 0033 0032",""};
lookup1("325D") ->	{"0","<circle> 0033 0033",""};
lookup1("325E") ->	{"0","<circle> 0033 0034",""};
lookup1("325F") ->	{"0","<circle> 0033 0035",""};
lookup1("3260") ->	{"0","<circle> 1100",""};
lookup1("3261") ->	{"0","<circle> 1102",""};
lookup1("3262") ->	{"0","<circle> 1103",""};
lookup1("3263") ->	{"0","<circle> 1105",""};
lookup1("3264") ->	{"0","<circle> 1106",""};
lookup1("3265") ->	{"0","<circle> 1107",""};
lookup1("3266") ->	{"0","<circle> 1109",""};
lookup1("3267") ->	{"0","<circle> 110B",""};
lookup1("3268") ->	{"0","<circle> 110C",""};
lookup1("3269") ->	{"0","<circle> 110E",""};
lookup1("326A") ->	{"0","<circle> 110F",""};
lookup1("326B") ->	{"0","<circle> 1110",""};
lookup1("326C") ->	{"0","<circle> 1111",""};
lookup1("326D") ->	{"0","<circle> 1112",""};
lookup1("326E") ->	{"0","<circle> 1100 1161",""};
lookup1("326F") ->	{"0","<circle> 1102 1161",""};
lookup1("3270") ->	{"0","<circle> 1103 1161",""};
lookup1("3271") ->	{"0","<circle> 1105 1161",""};
lookup1("3272") ->	{"0","<circle> 1106 1161",""};
lookup1("3273") ->	{"0","<circle> 1107 1161",""};
lookup1("3274") ->	{"0","<circle> 1109 1161",""};
lookup1("3275") ->	{"0","<circle> 110B 1161",""};
lookup1("3276") ->	{"0","<circle> 110C 1161",""};
lookup1("3277") ->	{"0","<circle> 110E 1161",""};
lookup1("3278") ->	{"0","<circle> 110F 1161",""};
lookup1("3279") ->	{"0","<circle> 1110 1161",""};
lookup1("327A") ->	{"0","<circle> 1111 1161",""};
lookup1("327B") ->	{"0","<circle> 1112 1161",""};
lookup1("327C") ->	{"0","<circle> 110E 1161 11B7 1100 1169",""};
lookup1("327D") ->	{"0","<circle> 110C 116E 110B 1174",""};
lookup1("327E") ->	{"0","<circle> 110B 116E",""};
lookup1("327F") ->	{"0","",""};
lookup1("3280") ->	{"0","<circle> 4E00",""};
lookup1("3281") ->	{"0","<circle> 4E8C",""};
lookup1("3282") ->	{"0","<circle> 4E09",""};
lookup1("3283") ->	{"0","<circle> 56DB",""};
lookup1("3284") ->	{"0","<circle> 4E94",""};
lookup1("3285") ->	{"0","<circle> 516D",""};
lookup1("3286") ->	{"0","<circle> 4E03",""};
lookup1("3287") ->	{"0","<circle> 516B",""};
lookup1("3288") ->	{"0","<circle> 4E5D",""};
lookup1("3289") ->	{"0","<circle> 5341",""};
lookup1("328A") ->	{"0","<circle> 6708",""};
lookup1("328B") ->	{"0","<circle> 706B",""};
lookup1("328C") ->	{"0","<circle> 6C34",""};
lookup1("328D") ->	{"0","<circle> 6728",""};
lookup1("328E") ->	{"0","<circle> 91D1",""};
lookup1("328F") ->	{"0","<circle> 571F",""};
lookup1("3290") ->	{"0","<circle> 65E5",""};
lookup1("3291") ->	{"0","<circle> 682A",""};
lookup1("3292") ->	{"0","<circle> 6709",""};
lookup1("3293") ->	{"0","<circle> 793E",""};
lookup1("3294") ->	{"0","<circle> 540D",""};
lookup1("3295") ->	{"0","<circle> 7279",""};
lookup1("3296") ->	{"0","<circle> 8CA1",""};
lookup1("3297") ->	{"0","<circle> 795D",""};
lookup1("3298") ->	{"0","<circle> 52B4",""};
lookup1("3299") ->	{"0","<circle> 79D8",""};
lookup1("329A") ->	{"0","<circle> 7537",""};
lookup1("329B") ->	{"0","<circle> 5973",""};
lookup1("329C") ->	{"0","<circle> 9069",""};
lookup1("329D") ->	{"0","<circle> 512A",""};
lookup1("329E") ->	{"0","<circle> 5370",""};
lookup1("329F") ->	{"0","<circle> 6CE8",""};
lookup1("32A0") ->	{"0","<circle> 9805",""};
lookup1("32A1") ->	{"0","<circle> 4F11",""};
lookup1("32A2") ->	{"0","<circle> 5199",""};
lookup1("32A3") ->	{"0","<circle> 6B63",""};
lookup1("32A4") ->	{"0","<circle> 4E0A",""};
lookup1("32A5") ->	{"0","<circle> 4E2D",""};
lookup1("32A6") ->	{"0","<circle> 4E0B",""};
lookup1("32A7") ->	{"0","<circle> 5DE6",""};
lookup1("32A8") ->	{"0","<circle> 53F3",""};
lookup1("32A9") ->	{"0","<circle> 533B",""};
lookup1("32AA") ->	{"0","<circle> 5B97",""};
lookup1("32AB") ->	{"0","<circle> 5B66",""};
lookup1("32AC") ->	{"0","<circle> 76E3",""};
lookup1("32AD") ->	{"0","<circle> 4F01",""};
lookup1("32AE") ->	{"0","<circle> 8CC7",""};
lookup1("32AF") ->	{"0","<circle> 5354",""};
lookup1("32B0") ->	{"0","<circle> 591C",""};
lookup1("32B1") ->	{"0","<circle> 0033 0036",""};
lookup1("32B2") ->	{"0","<circle> 0033 0037",""};
lookup1("32B3") ->	{"0","<circle> 0033 0038",""};
lookup1("32B4") ->	{"0","<circle> 0033 0039",""};
lookup1("32B5") ->	{"0","<circle> 0034 0030",""};
lookup1("32B6") ->	{"0","<circle> 0034 0031",""};
lookup1("32B7") ->	{"0","<circle> 0034 0032",""};
lookup1("32B8") ->	{"0","<circle> 0034 0033",""};
lookup1("32B9") ->	{"0","<circle> 0034 0034",""};
lookup1("32BA") ->	{"0","<circle> 0034 0035",""};
lookup1("32BB") ->	{"0","<circle> 0034 0036",""};
lookup1("32BC") ->	{"0","<circle> 0034 0037",""};
lookup1("32BD") ->	{"0","<circle> 0034 0038",""};
lookup1("32BE") ->	{"0","<circle> 0034 0039",""};
lookup1("32BF") ->	{"0","<circle> 0035 0030",""};
lookup1("32C0") ->	{"0","<compat> 0031 6708",""};
lookup1("32C1") ->	{"0","<compat> 0032 6708",""};
lookup1("32C2") ->	{"0","<compat> 0033 6708",""};
lookup1("32C3") ->	{"0","<compat> 0034 6708",""};
lookup1("32C4") ->	{"0","<compat> 0035 6708",""};
lookup1("32C5") ->	{"0","<compat> 0036 6708",""};
lookup1("32C6") ->	{"0","<compat> 0037 6708",""};
lookup1("32C7") ->	{"0","<compat> 0038 6708",""};
lookup1("32C8") ->	{"0","<compat> 0039 6708",""};
lookup1("32C9") ->	{"0","<compat> 0031 0030 6708",""};
lookup1("32CA") ->	{"0","<compat> 0031 0031 6708",""};
lookup1("32CB") ->	{"0","<compat> 0031 0032 6708",""};
lookup1("32CC") ->	{"0","<square> 0048 0067",""};
lookup1("32CD") ->	{"0","<square> 0065 0072 0067",""};
lookup1("32CE") ->	{"0","<square> 0065 0056",""};
lookup1("32CF") ->	{"0","<square> 004C 0054 0044",""};
lookup1("32D0") ->	{"0","<circle> 30A2",""};
lookup1("32D1") ->	{"0","<circle> 30A4",""};
lookup1("32D2") ->	{"0","<circle> 30A6",""};
lookup1("32D3") ->	{"0","<circle> 30A8",""};
lookup1("32D4") ->	{"0","<circle> 30AA",""};
lookup1("32D5") ->	{"0","<circle> 30AB",""};
lookup1("32D6") ->	{"0","<circle> 30AD",""};
lookup1("32D7") ->	{"0","<circle> 30AF",""};
lookup1("32D8") ->	{"0","<circle> 30B1",""};
lookup1("32D9") ->	{"0","<circle> 30B3",""};
lookup1("32DA") ->	{"0","<circle> 30B5",""};
lookup1("32DB") ->	{"0","<circle> 30B7",""};
lookup1("32DC") ->	{"0","<circle> 30B9",""};
lookup1("32DD") ->	{"0","<circle> 30BB",""};
lookup1("32DE") ->	{"0","<circle> 30BD",""};
lookup1("32DF") ->	{"0","<circle> 30BF",""};
lookup1("32E0") ->	{"0","<circle> 30C1",""};
lookup1("32E1") ->	{"0","<circle> 30C4",""};
lookup1("32E2") ->	{"0","<circle> 30C6",""};
lookup1("32E3") ->	{"0","<circle> 30C8",""};
lookup1("32E4") ->	{"0","<circle> 30CA",""};
lookup1("32E5") ->	{"0","<circle> 30CB",""};
lookup1("32E6") ->	{"0","<circle> 30CC",""};
lookup1("32E7") ->	{"0","<circle> 30CD",""};
lookup1("32E8") ->	{"0","<circle> 30CE",""};
lookup1("32E9") ->	{"0","<circle> 30CF",""};
lookup1("32EA") ->	{"0","<circle> 30D2",""};
lookup1("32EB") ->	{"0","<circle> 30D5",""};
lookup1("32EC") ->	{"0","<circle> 30D8",""};
lookup1("32ED") ->	{"0","<circle> 30DB",""};
lookup1("32EE") ->	{"0","<circle> 30DE",""};
lookup1("32EF") ->	{"0","<circle> 30DF",""};
lookup1("32F0") ->	{"0","<circle> 30E0",""};
lookup1("32F1") ->	{"0","<circle> 30E1",""};
lookup1("32F2") ->	{"0","<circle> 30E2",""};
lookup1("32F3") ->	{"0","<circle> 30E4",""};
lookup1("32F4") ->	{"0","<circle> 30E6",""};
lookup1("32F5") ->	{"0","<circle> 30E8",""};
lookup1("32F6") ->	{"0","<circle> 30E9",""};
lookup1("32F7") ->	{"0","<circle> 30EA",""};
lookup1("32F8") ->	{"0","<circle> 30EB",""};
lookup1("32F9") ->	{"0","<circle> 30EC",""};
lookup1("32FA") ->	{"0","<circle> 30ED",""};
lookup1("32FB") ->	{"0","<circle> 30EF",""};
lookup1("32FC") ->	{"0","<circle> 30F0",""};
lookup1("32FD") ->	{"0","<circle> 30F1",""};
lookup1("32FE") ->	{"0","<circle> 30F2",""};
lookup1("3300") ->	{"0","<square> 30A2 30D1 30FC 30C8",""};
lookup1("3301") ->	{"0","<square> 30A2 30EB 30D5 30A1",""};
lookup1("3302") ->	{"0","<square> 30A2 30F3 30DA 30A2",""};
lookup1("3303") ->	{"0","<square> 30A2 30FC 30EB",""};
lookup1("3304") ->	{"0","<square> 30A4 30CB 30F3 30B0",""};
lookup1("3305") ->	{"0","<square> 30A4 30F3 30C1",""};
lookup1("3306") ->	{"0","<square> 30A6 30A9 30F3",""};
lookup1("3307") ->	{"0","<square> 30A8 30B9 30AF 30FC 30C9",""};
lookup1("3308") ->	{"0","<square> 30A8 30FC 30AB 30FC",""};
lookup1("3309") ->	{"0","<square> 30AA 30F3 30B9",""};
lookup1("330A") ->	{"0","<square> 30AA 30FC 30E0",""};
lookup1("330B") ->	{"0","<square> 30AB 30A4 30EA",""};
lookup1("330C") ->	{"0","<square> 30AB 30E9 30C3 30C8",""};
lookup1("330D") ->	{"0","<square> 30AB 30ED 30EA 30FC",""};
lookup1("330E") ->	{"0","<square> 30AC 30ED 30F3",""};
lookup1("330F") ->	{"0","<square> 30AC 30F3 30DE",""};
lookup1("3310") ->	{"0","<square> 30AE 30AC",""};
lookup1("3311") ->	{"0","<square> 30AE 30CB 30FC",""};
lookup1("3312") ->	{"0","<square> 30AD 30E5 30EA 30FC",""};
lookup1("3313") ->	{"0","<square> 30AE 30EB 30C0 30FC",""};
lookup1("3314") ->	{"0","<square> 30AD 30ED",""};
lookup1("3315") ->	{"0","<square> 30AD 30ED 30B0 30E9 30E0",""};
lookup1("3316") ->	{"0","<square> 30AD 30ED 30E1 30FC 30C8 30EB",""};
lookup1("3317") ->	{"0","<square> 30AD 30ED 30EF 30C3 30C8",""};
lookup1("3318") ->	{"0","<square> 30B0 30E9 30E0",""};
lookup1("3319") ->	{"0","<square> 30B0 30E9 30E0 30C8 30F3",""};
lookup1("331A") ->	{"0","<square> 30AF 30EB 30BC 30A4 30ED",""};
lookup1("331B") ->	{"0","<square> 30AF 30ED 30FC 30CD",""};
lookup1("331C") ->	{"0","<square> 30B1 30FC 30B9",""};
lookup1("331D") ->	{"0","<square> 30B3 30EB 30CA",""};
lookup1("331E") ->	{"0","<square> 30B3 30FC 30DD",""};
lookup1("331F") ->	{"0","<square> 30B5 30A4 30AF 30EB",""};
lookup1("3320") ->	{"0","<square> 30B5 30F3 30C1 30FC 30E0",""};
lookup1("3321") ->	{"0","<square> 30B7 30EA 30F3 30B0",""};
lookup1("3322") ->	{"0","<square> 30BB 30F3 30C1",""};
lookup1("3323") ->	{"0","<square> 30BB 30F3 30C8",""};
lookup1("3324") ->	{"0","<square> 30C0 30FC 30B9",""};
lookup1("3325") ->	{"0","<square> 30C7 30B7",""};
lookup1("3326") ->	{"0","<square> 30C9 30EB",""};
lookup1("3327") ->	{"0","<square> 30C8 30F3",""};
lookup1("3328") ->	{"0","<square> 30CA 30CE",""};
lookup1("3329") ->	{"0","<square> 30CE 30C3 30C8",""};
lookup1("332A") ->	{"0","<square> 30CF 30A4 30C4",""};
lookup1("332B") ->	{"0","<square> 30D1 30FC 30BB 30F3 30C8",""};
lookup1("332C") ->	{"0","<square> 30D1 30FC 30C4",""};
lookup1("332D") ->	{"0","<square> 30D0 30FC 30EC 30EB",""};
lookup1("332E") ->	{"0","<square> 30D4 30A2 30B9 30C8 30EB",""};
lookup1("332F") ->	{"0","<square> 30D4 30AF 30EB",""};
lookup1("3330") ->	{"0","<square> 30D4 30B3",""};
lookup1("3331") ->	{"0","<square> 30D3 30EB",""};
lookup1("3332") ->	{"0","<square> 30D5 30A1 30E9 30C3 30C9",""};
lookup1("3333") ->	{"0","<square> 30D5 30A3 30FC 30C8",""};
lookup1("3334") ->	{"0","<square> 30D6 30C3 30B7 30A7 30EB",""};
lookup1("3335") ->	{"0","<square> 30D5 30E9 30F3",""};
lookup1("3336") ->	{"0","<square> 30D8 30AF 30BF 30FC 30EB",""};
lookup1("3337") ->	{"0","<square> 30DA 30BD",""};
lookup1("3338") ->	{"0","<square> 30DA 30CB 30D2",""};
lookup1("3339") ->	{"0","<square> 30D8 30EB 30C4",""};
lookup1("333A") ->	{"0","<square> 30DA 30F3 30B9",""};
lookup1("333B") ->	{"0","<square> 30DA 30FC 30B8",""};
lookup1("333C") ->	{"0","<square> 30D9 30FC 30BF",""};
lookup1("333D") ->	{"0","<square> 30DD 30A4 30F3 30C8",""};
lookup1("333E") ->	{"0","<square> 30DC 30EB 30C8",""};
lookup1("333F") ->	{"0","<square> 30DB 30F3",""};
lookup1("3340") ->	{"0","<square> 30DD 30F3 30C9",""};
lookup1("3341") ->	{"0","<square> 30DB 30FC 30EB",""};
lookup1("3342") ->	{"0","<square> 30DB 30FC 30F3",""};
lookup1("3343") ->	{"0","<square> 30DE 30A4 30AF 30ED",""};
lookup1("3344") ->	{"0","<square> 30DE 30A4 30EB",""};
lookup1("3345") ->	{"0","<square> 30DE 30C3 30CF",""};
lookup1("3346") ->	{"0","<square> 30DE 30EB 30AF",""};
lookup1("3347") ->	{"0","<square> 30DE 30F3 30B7 30E7 30F3",""};
lookup1("3348") ->	{"0","<square> 30DF 30AF 30ED 30F3",""};
lookup1("3349") ->	{"0","<square> 30DF 30EA",""};
lookup1("334A") ->	{"0","<square> 30DF 30EA 30D0 30FC 30EB",""};
lookup1("334B") ->	{"0","<square> 30E1 30AC",""};
lookup1("334C") ->	{"0","<square> 30E1 30AC 30C8 30F3",""};
lookup1("334D") ->	{"0","<square> 30E1 30FC 30C8 30EB",""};
lookup1("334E") ->	{"0","<square> 30E4 30FC 30C9",""};
lookup1("334F") ->	{"0","<square> 30E4 30FC 30EB",""};
lookup1("3350") ->	{"0","<square> 30E6 30A2 30F3",""};
lookup1("3351") ->	{"0","<square> 30EA 30C3 30C8 30EB",""};
lookup1("3352") ->	{"0","<square> 30EA 30E9",""};
lookup1("3353") ->	{"0","<square> 30EB 30D4 30FC",""};
lookup1("3354") ->	{"0","<square> 30EB 30FC 30D6 30EB",""};
lookup1("3355") ->	{"0","<square> 30EC 30E0",""};
lookup1("3356") ->	{"0","<square> 30EC 30F3 30C8 30B2 30F3",""};
lookup1("3357") ->	{"0","<square> 30EF 30C3 30C8",""};
lookup1("3358") ->	{"0","<compat> 0030 70B9",""};
lookup1("3359") ->	{"0","<compat> 0031 70B9",""};
lookup1("335A") ->	{"0","<compat> 0032 70B9",""};
lookup1("335B") ->	{"0","<compat> 0033 70B9",""};
lookup1("335C") ->	{"0","<compat> 0034 70B9",""};
lookup1("335D") ->	{"0","<compat> 0035 70B9",""};
lookup1("335E") ->	{"0","<compat> 0036 70B9",""};
lookup1("335F") ->	{"0","<compat> 0037 70B9",""};
lookup1("3360") ->	{"0","<compat> 0038 70B9",""};
lookup1("3361") ->	{"0","<compat> 0039 70B9",""};
lookup1("3362") ->	{"0","<compat> 0031 0030 70B9",""};
lookup1("3363") ->	{"0","<compat> 0031 0031 70B9",""};
lookup1("3364") ->	{"0","<compat> 0031 0032 70B9",""};
lookup1("3365") ->	{"0","<compat> 0031 0033 70B9",""};
lookup1("3366") ->	{"0","<compat> 0031 0034 70B9",""};
lookup1("3367") ->	{"0","<compat> 0031 0035 70B9",""};
lookup1("3368") ->	{"0","<compat> 0031 0036 70B9",""};
lookup1("3369") ->	{"0","<compat> 0031 0037 70B9",""};
lookup1("336A") ->	{"0","<compat> 0031 0038 70B9",""};
lookup1("336B") ->	{"0","<compat> 0031 0039 70B9",""};
lookup1("336C") ->	{"0","<compat> 0032 0030 70B9",""};
lookup1("336D") ->	{"0","<compat> 0032 0031 70B9",""};
lookup1("336E") ->	{"0","<compat> 0032 0032 70B9",""};
lookup1("336F") ->	{"0","<compat> 0032 0033 70B9",""};
lookup1("3370") ->	{"0","<compat> 0032 0034 70B9",""};
lookup1("3371") ->	{"0","<square> 0068 0050 0061",""};
lookup1("3372") ->	{"0","<square> 0064 0061",""};
lookup1("3373") ->	{"0","<square> 0041 0055",""};
lookup1("3374") ->	{"0","<square> 0062 0061 0072",""};
lookup1("3375") ->	{"0","<square> 006F 0056",""};
lookup1("3376") ->	{"0","<square> 0070 0063",""};
lookup1("3377") ->	{"0","<square> 0064 006D",""};
lookup1("3378") ->	{"0","<square> 0064 006D 00B2",""};
lookup1("3379") ->	{"0","<square> 0064 006D 00B3",""};
lookup1("337A") ->	{"0","<square> 0049 0055",""};
lookup1("337B") ->	{"0","<square> 5E73 6210",""};
lookup1("337C") ->	{"0","<square> 662D 548C",""};
lookup1("337D") ->	{"0","<square> 5927 6B63",""};
lookup1("337E") ->	{"0","<square> 660E 6CBB",""};
lookup1("337F") ->	{"0","<square> 682A 5F0F 4F1A 793E",""};
lookup1("3380") ->	{"0","<square> 0070 0041",""};
lookup1("3381") ->	{"0","<square> 006E 0041",""};
lookup1("3382") ->	{"0","<square> 03BC 0041",""};
lookup1("3383") ->	{"0","<square> 006D 0041",""};
lookup1("3384") ->	{"0","<square> 006B 0041",""};
lookup1("3385") ->	{"0","<square> 004B 0042",""};
lookup1("3386") ->	{"0","<square> 004D 0042",""};
lookup1("3387") ->	{"0","<square> 0047 0042",""};
lookup1("3388") ->	{"0","<square> 0063 0061 006C",""};
lookup1("3389") ->	{"0","<square> 006B 0063 0061 006C",""};
lookup1("338A") ->	{"0","<square> 0070 0046",""};
lookup1("338B") ->	{"0","<square> 006E 0046",""};
lookup1("338C") ->	{"0","<square> 03BC 0046",""};
lookup1("338D") ->	{"0","<square> 03BC 0067",""};
lookup1("338E") ->	{"0","<square> 006D 0067",""};
lookup1("338F") ->	{"0","<square> 006B 0067",""};
lookup1("3390") ->	{"0","<square> 0048 007A",""};
lookup1("3391") ->	{"0","<square> 006B 0048 007A",""};
lookup1("3392") ->	{"0","<square> 004D 0048 007A",""};
lookup1("3393") ->	{"0","<square> 0047 0048 007A",""};
lookup1("3394") ->	{"0","<square> 0054 0048 007A",""};
lookup1("3395") ->	{"0","<square> 03BC 2113",""};
lookup1("3396") ->	{"0","<square> 006D 2113",""};
lookup1("3397") ->	{"0","<square> 0064 2113",""};
lookup1("3398") ->	{"0","<square> 006B 2113",""};
lookup1("3399") ->	{"0","<square> 0066 006D",""};
lookup1("339A") ->	{"0","<square> 006E 006D",""};
lookup1("339B") ->	{"0","<square> 03BC 006D",""};
lookup1("339C") ->	{"0","<square> 006D 006D",""};
lookup1("339D") ->	{"0","<square> 0063 006D",""};
lookup1("339E") ->	{"0","<square> 006B 006D",""};
lookup1("339F") ->	{"0","<square> 006D 006D 00B2",""};
lookup1("33A0") ->	{"0","<square> 0063 006D 00B2",""};
lookup1("33A1") ->	{"0","<square> 006D 00B2",""};
lookup1("33A2") ->	{"0","<square> 006B 006D 00B2",""};
lookup1("33A3") ->	{"0","<square> 006D 006D 00B3",""};
lookup1("33A4") ->	{"0","<square> 0063 006D 00B3",""};
lookup1("33A5") ->	{"0","<square> 006D 00B3",""};
lookup1("33A6") ->	{"0","<square> 006B 006D 00B3",""};
lookup1("33A7") ->	{"0","<square> 006D 2215 0073",""};
lookup1("33A8") ->	{"0","<square> 006D 2215 0073 00B2",""};
lookup1("33A9") ->	{"0","<square> 0050 0061",""};
lookup1("33AA") ->	{"0","<square> 006B 0050 0061",""};
lookup1("33AB") ->	{"0","<square> 004D 0050 0061",""};
lookup1("33AC") ->	{"0","<square> 0047 0050 0061",""};
lookup1("33AD") ->	{"0","<square> 0072 0061 0064",""};
lookup1("33AE") ->	{"0","<square> 0072 0061 0064 2215 0073",""};
lookup1("33AF") ->	{"0","<square> 0072 0061 0064 2215 0073 00B2",""};
lookup1("33B0") ->	{"0","<square> 0070 0073",""};
lookup1("33B1") ->	{"0","<square> 006E 0073",""};
lookup1("33B2") ->	{"0","<square> 03BC 0073",""};
lookup1("33B3") ->	{"0","<square> 006D 0073",""};
lookup1("33B4") ->	{"0","<square> 0070 0056",""};
lookup1("33B5") ->	{"0","<square> 006E 0056",""};
lookup1("33B6") ->	{"0","<square> 03BC 0056",""};
lookup1("33B7") ->	{"0","<square> 006D 0056",""};
lookup1("33B8") ->	{"0","<square> 006B 0056",""};
lookup1("33B9") ->	{"0","<square> 004D 0056",""};
lookup1("33BA") ->	{"0","<square> 0070 0057",""};
lookup1("33BB") ->	{"0","<square> 006E 0057",""};
lookup1("33BC") ->	{"0","<square> 03BC 0057",""};
lookup1("33BD") ->	{"0","<square> 006D 0057",""};
lookup1("33BE") ->	{"0","<square> 006B 0057",""};
lookup1("33BF") ->	{"0","<square> 004D 0057",""};
lookup1("33C0") ->	{"0","<square> 006B 03A9",""};
lookup1("33C1") ->	{"0","<square> 004D 03A9",""};
lookup1("33C2") ->	{"0","<square> 0061 002E 006D 002E",""};
lookup1("33C3") ->	{"0","<square> 0042 0071",""};
lookup1("33C4") ->	{"0","<square> 0063 0063",""};
lookup1("33C5") ->	{"0","<square> 0063 0064",""};
lookup1("33C6") ->	{"0","<square> 0043 2215 006B 0067",""};
lookup1("33C7") ->	{"0","<square> 0043 006F 002E",""};
lookup1("33C8") ->	{"0","<square> 0064 0042",""};
lookup1("33C9") ->	{"0","<square> 0047 0079",""};
lookup1("33CA") ->	{"0","<square> 0068 0061",""};
lookup1("33CB") ->	{"0","<square> 0048 0050",""};
lookup1("33CC") ->	{"0","<square> 0069 006E",""};
lookup1("33CD") ->	{"0","<square> 004B 004B",""};
lookup1("33CE") ->	{"0","<square> 004B 004D",""};
lookup1("33CF") ->	{"0","<square> 006B 0074",""};
lookup1("33D0") ->	{"0","<square> 006C 006D",""};
lookup1("33D1") ->	{"0","<square> 006C 006E",""};
lookup1("33D2") ->	{"0","<square> 006C 006F 0067",""};
lookup1("33D3") ->	{"0","<square> 006C 0078",""};
lookup1("33D4") ->	{"0","<square> 006D 0062",""};
lookup1("33D5") ->	{"0","<square> 006D 0069 006C",""};
lookup1("33D6") ->	{"0","<square> 006D 006F 006C",""};
lookup1("33D7") ->	{"0","<square> 0050 0048",""};
lookup1("33D8") ->	{"0","<square> 0070 002E 006D 002E",""};
lookup1("33D9") ->	{"0","<square> 0050 0050 004D",""};
lookup1("33DA") ->	{"0","<square> 0050 0052",""};
lookup1("33DB") ->	{"0","<square> 0073 0072",""};
lookup1("33DC") ->	{"0","<square> 0053 0076",""};
lookup1("33DD") ->	{"0","<square> 0057 0062",""};
lookup1("33DE") ->	{"0","<square> 0056 2215 006D",""};
lookup1("33DF") ->	{"0","<square> 0041 2215 006D",""};
lookup1("33E0") ->	{"0","<compat> 0031 65E5",""};
lookup1("33E1") ->	{"0","<compat> 0032 65E5",""};
lookup1("33E2") ->	{"0","<compat> 0033 65E5",""};
lookup1("33E3") ->	{"0","<compat> 0034 65E5",""};
lookup1("33E4") ->	{"0","<compat> 0035 65E5",""};
lookup1("33E5") ->	{"0","<compat> 0036 65E5",""};
lookup1("33E6") ->	{"0","<compat> 0037 65E5",""};
lookup1("33E7") ->	{"0","<compat> 0038 65E5",""};
lookup1("33E8") ->	{"0","<compat> 0039 65E5",""};
lookup1("33E9") ->	{"0","<compat> 0031 0030 65E5",""};
lookup1("33EA") ->	{"0","<compat> 0031 0031 65E5",""};
lookup1("33EB") ->	{"0","<compat> 0031 0032 65E5",""};
lookup1("33EC") ->	{"0","<compat> 0031 0033 65E5",""};
lookup1("33ED") ->	{"0","<compat> 0031 0034 65E5",""};
lookup1("33EE") ->	{"0","<compat> 0031 0035 65E5",""};
lookup1("33EF") ->	{"0","<compat> 0031 0036 65E5",""};
lookup1("33F0") ->	{"0","<compat> 0031 0037 65E5",""};
lookup1("33F1") ->	{"0","<compat> 0031 0038 65E5",""};
lookup1("33F2") ->	{"0","<compat> 0031 0039 65E5",""};
lookup1("33F3") ->	{"0","<compat> 0032 0030 65E5",""};
lookup1("33F4") ->	{"0","<compat> 0032 0031 65E5",""};
lookup1("33F5") ->	{"0","<compat> 0032 0032 65E5",""};
lookup1("33F6") ->	{"0","<compat> 0032 0033 65E5",""};
lookup1("33F7") ->	{"0","<compat> 0032 0034 65E5",""};
lookup1("33F8") ->	{"0","<compat> 0032 0035 65E5",""};
lookup1("33F9") ->	{"0","<compat> 0032 0036 65E5",""};
lookup1("33FA") ->	{"0","<compat> 0032 0037 65E5",""};
lookup1("33FB") ->	{"0","<compat> 0032 0038 65E5",""};
lookup1("33FC") ->	{"0","<compat> 0032 0039 65E5",""};
lookup1("33FD") ->	{"0","<compat> 0033 0030 65E5",""};
lookup1("33FE") ->	{"0","<compat> 0033 0031 65E5",""};
lookup1("33FF") ->	{"0","<square> 0067 0061 006C",""};
lookup1("3400") ->	{"0","",""};
lookup1("4DB5") ->	{"0","",""};
lookup1("4DC0") ->	{"0","",""};
lookup1("4DC1") ->	{"0","",""};
lookup1("4DC2") ->	{"0","",""};
lookup1("4DC3") ->	{"0","",""};
lookup1("4DC4") ->	{"0","",""};
lookup1("4DC5") ->	{"0","",""};
lookup1("4DC6") ->	{"0","",""};
lookup1("4DC7") ->	{"0","",""};
lookup1("4DC8") ->	{"0","",""};
lookup1("4DC9") ->	{"0","",""};
lookup1("4DCA") ->	{"0","",""};
lookup1("4DCB") ->	{"0","",""};
lookup1("4DCC") ->	{"0","",""};
lookup1("4DCD") ->	{"0","",""};
lookup1("4DCE") ->	{"0","",""};
lookup1("4DCF") ->	{"0","",""};
lookup1("4DD0") ->	{"0","",""};
lookup1("4DD1") ->	{"0","",""};
lookup1("4DD2") ->	{"0","",""};
lookup1("4DD3") ->	{"0","",""};
lookup1("4DD4") ->	{"0","",""};
lookup1("4DD5") ->	{"0","",""};
lookup1("4DD6") ->	{"0","",""};
lookup1("4DD7") ->	{"0","",""};
lookup1("4DD8") ->	{"0","",""};
lookup1("4DD9") ->	{"0","",""};
lookup1("4DDA") ->	{"0","",""};
lookup1("4DDB") ->	{"0","",""};
lookup1("4DDC") ->	{"0","",""};
lookup1("4DDD") ->	{"0","",""};
lookup1("4DDE") ->	{"0","",""};
lookup1("4DDF") ->	{"0","",""};
lookup1("4DE0") ->	{"0","",""};
lookup1("4DE1") ->	{"0","",""};
lookup1("4DE2") ->	{"0","",""};
lookup1("4DE3") ->	{"0","",""};
lookup1("4DE4") ->	{"0","",""};
lookup1("4DE5") ->	{"0","",""};
lookup1("4DE6") ->	{"0","",""};
lookup1("4DE7") ->	{"0","",""};
lookup1("4DE8") ->	{"0","",""};
lookup1("4DE9") ->	{"0","",""};
lookup1("4DEA") ->	{"0","",""};
lookup1("4DEB") ->	{"0","",""};
lookup1("4DEC") ->	{"0","",""};
lookup1("4DED") ->	{"0","",""};
lookup1("4DEE") ->	{"0","",""};
lookup1("4DEF") ->	{"0","",""};
lookup1("4DF0") ->	{"0","",""};
lookup1("4DF1") ->	{"0","",""};
lookup1("4DF2") ->	{"0","",""};
lookup1("4DF3") ->	{"0","",""};
lookup1("4DF4") ->	{"0","",""};
lookup1("4DF5") ->	{"0","",""};
lookup1("4DF6") ->	{"0","",""};
lookup1("4DF7") ->	{"0","",""};
lookup1("4DF8") ->	{"0","",""};
lookup1("4DF9") ->	{"0","",""};
lookup1("4DFA") ->	{"0","",""};
lookup1("4DFB") ->	{"0","",""};
lookup1("4DFC") ->	{"0","",""};
lookup1("4DFD") ->	{"0","",""};
lookup1("4DFE") ->	{"0","",""};
lookup1("4DFF") ->	{"0","",""};
lookup1("4E00") ->	{"0","",""};
lookup1("9FCC") ->	{"0","",""};
lookup1("A000") ->	{"0","",""};
lookup1("A001") ->	{"0","",""};
lookup1("A002") ->	{"0","",""};
lookup1("A003") ->	{"0","",""};
lookup1("A004") ->	{"0","",""};
lookup1("A005") ->	{"0","",""};
lookup1("A006") ->	{"0","",""};
lookup1("A007") ->	{"0","",""};
lookup1("A008") ->	{"0","",""};
lookup1("A009") ->	{"0","",""};
lookup1("A00A") ->	{"0","",""};
lookup1("A00B") ->	{"0","",""};
lookup1("A00C") ->	{"0","",""};
lookup1("A00D") ->	{"0","",""};
lookup1("A00E") ->	{"0","",""};
lookup1("A00F") ->	{"0","",""};
lookup1("A010") ->	{"0","",""};
lookup1("A011") ->	{"0","",""};
lookup1("A012") ->	{"0","",""};
lookup1("A013") ->	{"0","",""};
lookup1("A014") ->	{"0","",""};
lookup1("A015") ->	{"0","",""};
lookup1("A016") ->	{"0","",""};
lookup1("A017") ->	{"0","",""};
lookup1("A018") ->	{"0","",""};
lookup1("A019") ->	{"0","",""};
lookup1("A01A") ->	{"0","",""};
lookup1("A01B") ->	{"0","",""};
lookup1("A01C") ->	{"0","",""};
lookup1("A01D") ->	{"0","",""};
lookup1("A01E") ->	{"0","",""};
lookup1("A01F") ->	{"0","",""};
lookup1("A020") ->	{"0","",""};
lookup1("A021") ->	{"0","",""};
lookup1("A022") ->	{"0","",""};
lookup1("A023") ->	{"0","",""};
lookup1("A024") ->	{"0","",""};
lookup1("A025") ->	{"0","",""};
lookup1("A026") ->	{"0","",""};
lookup1("A027") ->	{"0","",""};
lookup1("A028") ->	{"0","",""};
lookup1("A029") ->	{"0","",""};
lookup1("A02A") ->	{"0","",""};
lookup1("A02B") ->	{"0","",""};
lookup1("A02C") ->	{"0","",""};
lookup1("A02D") ->	{"0","",""};
lookup1("A02E") ->	{"0","",""};
lookup1("A02F") ->	{"0","",""};
lookup1("A030") ->	{"0","",""};
lookup1("A031") ->	{"0","",""};
lookup1("A032") ->	{"0","",""};
lookup1("A033") ->	{"0","",""};
lookup1("A034") ->	{"0","",""};
lookup1("A035") ->	{"0","",""};
lookup1("A036") ->	{"0","",""};
lookup1("A037") ->	{"0","",""};
lookup1("A038") ->	{"0","",""};
lookup1("A039") ->	{"0","",""};
lookup1("A03A") ->	{"0","",""};
lookup1("A03B") ->	{"0","",""};
lookup1("A03C") ->	{"0","",""};
lookup1("A03D") ->	{"0","",""};
lookup1("A03E") ->	{"0","",""};
lookup1("A03F") ->	{"0","",""};
lookup1("A040") ->	{"0","",""};
lookup1("A041") ->	{"0","",""};
lookup1("A042") ->	{"0","",""};
lookup1("A043") ->	{"0","",""};
lookup1("A044") ->	{"0","",""};
lookup1("A045") ->	{"0","",""};
lookup1("A046") ->	{"0","",""};
lookup1("A047") ->	{"0","",""};
lookup1("A048") ->	{"0","",""};
lookup1("A049") ->	{"0","",""};
lookup1("A04A") ->	{"0","",""};
lookup1("A04B") ->	{"0","",""};
lookup1("A04C") ->	{"0","",""};
lookup1("A04D") ->	{"0","",""};
lookup1("A04E") ->	{"0","",""};
lookup1("A04F") ->	{"0","",""};
lookup1("A050") ->	{"0","",""};
lookup1("A051") ->	{"0","",""};
lookup1("A052") ->	{"0","",""};
lookup1("A053") ->	{"0","",""};
lookup1("A054") ->	{"0","",""};
lookup1("A055") ->	{"0","",""};
lookup1("A056") ->	{"0","",""};
lookup1("A057") ->	{"0","",""};
lookup1("A058") ->	{"0","",""};
lookup1("A059") ->	{"0","",""};
lookup1("A05A") ->	{"0","",""};
lookup1("A05B") ->	{"0","",""};
lookup1("A05C") ->	{"0","",""};
lookup1("A05D") ->	{"0","",""};
lookup1("A05E") ->	{"0","",""};
lookup1("A05F") ->	{"0","",""};
lookup1("A060") ->	{"0","",""};
lookup1("A061") ->	{"0","",""};
lookup1("A062") ->	{"0","",""};
lookup1("A063") ->	{"0","",""};
lookup1("A064") ->	{"0","",""};
lookup1("A065") ->	{"0","",""};
lookup1("A066") ->	{"0","",""};
lookup1("A067") ->	{"0","",""};
lookup1("A068") ->	{"0","",""};
lookup1("A069") ->	{"0","",""};
lookup1("A06A") ->	{"0","",""};
lookup1("A06B") ->	{"0","",""};
lookup1("A06C") ->	{"0","",""};
lookup1("A06D") ->	{"0","",""};
lookup1("A06E") ->	{"0","",""};
lookup1("A06F") ->	{"0","",""};
lookup1("A070") ->	{"0","",""};
lookup1("A071") ->	{"0","",""};
lookup1("A072") ->	{"0","",""};
lookup1("A073") ->	{"0","",""};
lookup1("A074") ->	{"0","",""};
lookup1("A075") ->	{"0","",""};
lookup1("A076") ->	{"0","",""};
lookup1("A077") ->	{"0","",""};
lookup1("A078") ->	{"0","",""};
lookup1("A079") ->	{"0","",""};
lookup1("A07A") ->	{"0","",""};
lookup1("A07B") ->	{"0","",""};
lookup1("A07C") ->	{"0","",""};
lookup1("A07D") ->	{"0","",""};
lookup1("A07E") ->	{"0","",""};
lookup1("A07F") ->	{"0","",""};
lookup1("A080") ->	{"0","",""};
lookup1("A081") ->	{"0","",""};
lookup1("A082") ->	{"0","",""};
lookup1("A083") ->	{"0","",""};
lookup1("A084") ->	{"0","",""};
lookup1("A085") ->	{"0","",""};
lookup1("A086") ->	{"0","",""};
lookup1("A087") ->	{"0","",""};
lookup1("A088") ->	{"0","",""};
lookup1("A089") ->	{"0","",""};
lookup1("A08A") ->	{"0","",""};
lookup1("A08B") ->	{"0","",""};
lookup1("A08C") ->	{"0","",""};
lookup1("A08D") ->	{"0","",""};
lookup1("A08E") ->	{"0","",""};
lookup1("A08F") ->	{"0","",""};
lookup1("A090") ->	{"0","",""};
lookup1("A091") ->	{"0","",""};
lookup1("A092") ->	{"0","",""};
lookup1("A093") ->	{"0","",""};
lookup1("A094") ->	{"0","",""};
lookup1("A095") ->	{"0","",""};
lookup1("A096") ->	{"0","",""};
lookup1("A097") ->	{"0","",""};
lookup1("A098") ->	{"0","",""};
lookup1("A099") ->	{"0","",""};
lookup1("A09A") ->	{"0","",""};
lookup1("A09B") ->	{"0","",""};
lookup1("A09C") ->	{"0","",""};
lookup1("A09D") ->	{"0","",""};
lookup1("A09E") ->	{"0","",""};
lookup1("A09F") ->	{"0","",""};
lookup1("A0A0") ->	{"0","",""};
lookup1("A0A1") ->	{"0","",""};
lookup1("A0A2") ->	{"0","",""};
lookup1("A0A3") ->	{"0","",""};
lookup1("A0A4") ->	{"0","",""};
lookup1("A0A5") ->	{"0","",""};
lookup1("A0A6") ->	{"0","",""};
lookup1("A0A7") ->	{"0","",""};
lookup1("A0A8") ->	{"0","",""};
lookup1("A0A9") ->	{"0","",""};
lookup1("A0AA") ->	{"0","",""};
lookup1("A0AB") ->	{"0","",""};
lookup1("A0AC") ->	{"0","",""};
lookup1("A0AD") ->	{"0","",""};
lookup1("A0AE") ->	{"0","",""};
lookup1("A0AF") ->	{"0","",""};
lookup1("A0B0") ->	{"0","",""};
lookup1("A0B1") ->	{"0","",""};
lookup1("A0B2") ->	{"0","",""};
lookup1("A0B3") ->	{"0","",""};
lookup1("A0B4") ->	{"0","",""};
lookup1("A0B5") ->	{"0","",""};
lookup1("A0B6") ->	{"0","",""};
lookup1("A0B7") ->	{"0","",""};
lookup1("A0B8") ->	{"0","",""};
lookup1("A0B9") ->	{"0","",""};
lookup1("A0BA") ->	{"0","",""};
lookup1("A0BB") ->	{"0","",""};
lookup1("A0BC") ->	{"0","",""};
lookup1("A0BD") ->	{"0","",""};
lookup1("A0BE") ->	{"0","",""};
lookup1("A0BF") ->	{"0","",""};
lookup1("A0C0") ->	{"0","",""};
lookup1("A0C1") ->	{"0","",""};
lookup1("A0C2") ->	{"0","",""};
lookup1("A0C3") ->	{"0","",""};
lookup1("A0C4") ->	{"0","",""};
lookup1("A0C5") ->	{"0","",""};
lookup1("A0C6") ->	{"0","",""};
lookup1("A0C7") ->	{"0","",""};
lookup1("A0C8") ->	{"0","",""};
lookup1("A0C9") ->	{"0","",""};
lookup1("A0CA") ->	{"0","",""};
lookup1("A0CB") ->	{"0","",""};
lookup1("A0CC") ->	{"0","",""};
lookup1("A0CD") ->	{"0","",""};
lookup1("A0CE") ->	{"0","",""};
lookup1("A0CF") ->	{"0","",""};
lookup1("A0D0") ->	{"0","",""};
lookup1("A0D1") ->	{"0","",""};
lookup1("A0D2") ->	{"0","",""};
lookup1("A0D3") ->	{"0","",""};
lookup1("A0D4") ->	{"0","",""};
lookup1("A0D5") ->	{"0","",""};
lookup1("A0D6") ->	{"0","",""};
lookup1("A0D7") ->	{"0","",""};
lookup1("A0D8") ->	{"0","",""};
lookup1("A0D9") ->	{"0","",""};
lookup1("A0DA") ->	{"0","",""};
lookup1("A0DB") ->	{"0","",""};
lookup1("A0DC") ->	{"0","",""};
lookup1("A0DD") ->	{"0","",""};
lookup1("A0DE") ->	{"0","",""};
lookup1("A0DF") ->	{"0","",""};
lookup1("A0E0") ->	{"0","",""};
lookup1("A0E1") ->	{"0","",""};
lookup1("A0E2") ->	{"0","",""};
lookup1("A0E3") ->	{"0","",""};
lookup1("A0E4") ->	{"0","",""};
lookup1("A0E5") ->	{"0","",""};
lookup1("A0E6") ->	{"0","",""};
lookup1("A0E7") ->	{"0","",""};
lookup1("A0E8") ->	{"0","",""};
lookup1("A0E9") ->	{"0","",""};
lookup1("A0EA") ->	{"0","",""};
lookup1("A0EB") ->	{"0","",""};
lookup1("A0EC") ->	{"0","",""};
lookup1("A0ED") ->	{"0","",""};
lookup1("A0EE") ->	{"0","",""};
lookup1("A0EF") ->	{"0","",""};
lookup1("A0F0") ->	{"0","",""};
lookup1("A0F1") ->	{"0","",""};
lookup1("A0F2") ->	{"0","",""};
lookup1("A0F3") ->	{"0","",""};
lookup1("A0F4") ->	{"0","",""};
lookup1("A0F5") ->	{"0","",""};
lookup1("A0F6") ->	{"0","",""};
lookup1("A0F7") ->	{"0","",""};
lookup1("A0F8") ->	{"0","",""};
lookup1("A0F9") ->	{"0","",""};
lookup1("A0FA") ->	{"0","",""};
lookup1("A0FB") ->	{"0","",""};
lookup1("A0FC") ->	{"0","",""};
lookup1("A0FD") ->	{"0","",""};
lookup1("A0FE") ->	{"0","",""};
lookup1("A0FF") ->	{"0","",""};
lookup1("A100") ->	{"0","",""};
lookup1("A101") ->	{"0","",""};
lookup1("A102") ->	{"0","",""};
lookup1("A103") ->	{"0","",""};
lookup1("A104") ->	{"0","",""};
lookup1("A105") ->	{"0","",""};
lookup1("A106") ->	{"0","",""};
lookup1("A107") ->	{"0","",""};
lookup1("A108") ->	{"0","",""};
lookup1("A109") ->	{"0","",""};
lookup1("A10A") ->	{"0","",""};
lookup1("A10B") ->	{"0","",""};
lookup1("A10C") ->	{"0","",""};
lookup1("A10D") ->	{"0","",""};
lookup1("A10E") ->	{"0","",""};
lookup1("A10F") ->	{"0","",""};
lookup1("A110") ->	{"0","",""};
lookup1("A111") ->	{"0","",""};
lookup1("A112") ->	{"0","",""};
lookup1("A113") ->	{"0","",""};
lookup1("A114") ->	{"0","",""};
lookup1("A115") ->	{"0","",""};
lookup1("A116") ->	{"0","",""};
lookup1("A117") ->	{"0","",""};
lookup1("A118") ->	{"0","",""};
lookup1("A119") ->	{"0","",""};
lookup1("A11A") ->	{"0","",""};
lookup1("A11B") ->	{"0","",""};
lookup1("A11C") ->	{"0","",""};
lookup1("A11D") ->	{"0","",""};
lookup1("A11E") ->	{"0","",""};
lookup1("A11F") ->	{"0","",""};
lookup1("A120") ->	{"0","",""};
lookup1("A121") ->	{"0","",""};
lookup1("A122") ->	{"0","",""};
lookup1("A123") ->	{"0","",""};
lookup1("A124") ->	{"0","",""};
lookup1("A125") ->	{"0","",""};
lookup1("A126") ->	{"0","",""};
lookup1("A127") ->	{"0","",""};
lookup1("A128") ->	{"0","",""};
lookup1("A129") ->	{"0","",""};
lookup1("A12A") ->	{"0","",""};
lookup1("A12B") ->	{"0","",""};
lookup1("A12C") ->	{"0","",""};
lookup1("A12D") ->	{"0","",""};
lookup1("A12E") ->	{"0","",""};
lookup1("A12F") ->	{"0","",""};
lookup1("A130") ->	{"0","",""};
lookup1("A131") ->	{"0","",""};
lookup1("A132") ->	{"0","",""};
lookup1("A133") ->	{"0","",""};
lookup1("A134") ->	{"0","",""};
lookup1("A135") ->	{"0","",""};
lookup1("A136") ->	{"0","",""};
lookup1("A137") ->	{"0","",""};
lookup1("A138") ->	{"0","",""};
lookup1("A139") ->	{"0","",""};
lookup1("A13A") ->	{"0","",""};
lookup1("A13B") ->	{"0","",""};
lookup1("A13C") ->	{"0","",""};
lookup1("A13D") ->	{"0","",""};
lookup1("A13E") ->	{"0","",""};
lookup1("A13F") ->	{"0","",""};
lookup1("A140") ->	{"0","",""};
lookup1("A141") ->	{"0","",""};
lookup1("A142") ->	{"0","",""};
lookup1("A143") ->	{"0","",""};
lookup1("A144") ->	{"0","",""};
lookup1("A145") ->	{"0","",""};
lookup1("A146") ->	{"0","",""};
lookup1("A147") ->	{"0","",""};
lookup1("A148") ->	{"0","",""};
lookup1("A149") ->	{"0","",""};
lookup1("A14A") ->	{"0","",""};
lookup1("A14B") ->	{"0","",""};
lookup1("A14C") ->	{"0","",""};
lookup1("A14D") ->	{"0","",""};
lookup1("A14E") ->	{"0","",""};
lookup1("A14F") ->	{"0","",""};
lookup1("A150") ->	{"0","",""};
lookup1("A151") ->	{"0","",""};
lookup1("A152") ->	{"0","",""};
lookup1("A153") ->	{"0","",""};
lookup1("A154") ->	{"0","",""};
lookup1("A155") ->	{"0","",""};
lookup1("A156") ->	{"0","",""};
lookup1("A157") ->	{"0","",""};
lookup1("A158") ->	{"0","",""};
lookup1("A159") ->	{"0","",""};
lookup1("A15A") ->	{"0","",""};
lookup1("A15B") ->	{"0","",""};
lookup1("A15C") ->	{"0","",""};
lookup1("A15D") ->	{"0","",""};
lookup1("A15E") ->	{"0","",""};
lookup1("A15F") ->	{"0","",""};
lookup1("A160") ->	{"0","",""};
lookup1("A161") ->	{"0","",""};
lookup1("A162") ->	{"0","",""};
lookup1("A163") ->	{"0","",""};
lookup1("A164") ->	{"0","",""};
lookup1("A165") ->	{"0","",""};
lookup1("A166") ->	{"0","",""};
lookup1("A167") ->	{"0","",""};
lookup1("A168") ->	{"0","",""};
lookup1("A169") ->	{"0","",""};
lookup1("A16A") ->	{"0","",""};
lookup1("A16B") ->	{"0","",""};
lookup1("A16C") ->	{"0","",""};
lookup1("A16D") ->	{"0","",""};
lookup1("A16E") ->	{"0","",""};
lookup1("A16F") ->	{"0","",""};
lookup1("A170") ->	{"0","",""};
lookup1("A171") ->	{"0","",""};
lookup1("A172") ->	{"0","",""};
lookup1("A173") ->	{"0","",""};
lookup1("A174") ->	{"0","",""};
lookup1("A175") ->	{"0","",""};
lookup1("A176") ->	{"0","",""};
lookup1("A177") ->	{"0","",""};
lookup1("A178") ->	{"0","",""};
lookup1("A179") ->	{"0","",""};
lookup1("A17A") ->	{"0","",""};
lookup1("A17B") ->	{"0","",""};
lookup1("A17C") ->	{"0","",""};
lookup1("A17D") ->	{"0","",""};
lookup1("A17E") ->	{"0","",""};
lookup1("A17F") ->	{"0","",""};
lookup1("A180") ->	{"0","",""};
lookup1("A181") ->	{"0","",""};
lookup1("A182") ->	{"0","",""};
lookup1("A183") ->	{"0","",""};
lookup1("A184") ->	{"0","",""};
lookup1("A185") ->	{"0","",""};
lookup1("A186") ->	{"0","",""};
lookup1("A187") ->	{"0","",""};
lookup1("A188") ->	{"0","",""};
lookup1("A189") ->	{"0","",""};
lookup1("A18A") ->	{"0","",""};
lookup1("A18B") ->	{"0","",""};
lookup1("A18C") ->	{"0","",""};
lookup1("A18D") ->	{"0","",""};
lookup1("A18E") ->	{"0","",""};
lookup1("A18F") ->	{"0","",""};
lookup1("A190") ->	{"0","",""};
lookup1("A191") ->	{"0","",""};
lookup1("A192") ->	{"0","",""};
lookup1("A193") ->	{"0","",""};
lookup1("A194") ->	{"0","",""};
lookup1("A195") ->	{"0","",""};
lookup1("A196") ->	{"0","",""};
lookup1("A197") ->	{"0","",""};
lookup1("A198") ->	{"0","",""};
lookup1("A199") ->	{"0","",""};
lookup1("A19A") ->	{"0","",""};
lookup1("A19B") ->	{"0","",""};
lookup1("A19C") ->	{"0","",""};
lookup1("A19D") ->	{"0","",""};
lookup1("A19E") ->	{"0","",""};
lookup1("A19F") ->	{"0","",""};
lookup1("A1A0") ->	{"0","",""};
lookup1("A1A1") ->	{"0","",""};
lookup1("A1A2") ->	{"0","",""};
lookup1("A1A3") ->	{"0","",""};
lookup1("A1A4") ->	{"0","",""};
lookup1("A1A5") ->	{"0","",""};
lookup1("A1A6") ->	{"0","",""};
lookup1("A1A7") ->	{"0","",""};
lookup1("A1A8") ->	{"0","",""};
lookup1("A1A9") ->	{"0","",""};
lookup1("A1AA") ->	{"0","",""};
lookup1("A1AB") ->	{"0","",""};
lookup1("A1AC") ->	{"0","",""};
lookup1("A1AD") ->	{"0","",""};
lookup1("A1AE") ->	{"0","",""};
lookup1("A1AF") ->	{"0","",""};
lookup1("A1B0") ->	{"0","",""};
lookup1("A1B1") ->	{"0","",""};
lookup1("A1B2") ->	{"0","",""};
lookup1("A1B3") ->	{"0","",""};
lookup1("A1B4") ->	{"0","",""};
lookup1("A1B5") ->	{"0","",""};
lookup1("A1B6") ->	{"0","",""};
lookup1("A1B7") ->	{"0","",""};
lookup1("A1B8") ->	{"0","",""};
lookup1("A1B9") ->	{"0","",""};
lookup1("A1BA") ->	{"0","",""};
lookup1("A1BB") ->	{"0","",""};
lookup1("A1BC") ->	{"0","",""};
lookup1("A1BD") ->	{"0","",""};
lookup1("A1BE") ->	{"0","",""};
lookup1("A1BF") ->	{"0","",""};
lookup1("A1C0") ->	{"0","",""};
lookup1("A1C1") ->	{"0","",""};
lookup1("A1C2") ->	{"0","",""};
lookup1("A1C3") ->	{"0","",""};
lookup1("A1C4") ->	{"0","",""};
lookup1("A1C5") ->	{"0","",""};
lookup1("A1C6") ->	{"0","",""};
lookup1("A1C7") ->	{"0","",""};
lookup1("A1C8") ->	{"0","",""};
lookup1("A1C9") ->	{"0","",""};
lookup1("A1CA") ->	{"0","",""};
lookup1("A1CB") ->	{"0","",""};
lookup1("A1CC") ->	{"0","",""};
lookup1("A1CD") ->	{"0","",""};
lookup1("A1CE") ->	{"0","",""};
lookup1("A1CF") ->	{"0","",""};
lookup1("A1D0") ->	{"0","",""};
lookup1("A1D1") ->	{"0","",""};
lookup1("A1D2") ->	{"0","",""};
lookup1("A1D3") ->	{"0","",""};
lookup1("A1D4") ->	{"0","",""};
lookup1("A1D5") ->	{"0","",""};
lookup1("A1D6") ->	{"0","",""};
lookup1("A1D7") ->	{"0","",""};
lookup1("A1D8") ->	{"0","",""};
lookup1("A1D9") ->	{"0","",""};
lookup1("A1DA") ->	{"0","",""};
lookup1("A1DB") ->	{"0","",""};
lookup1("A1DC") ->	{"0","",""};
lookup1("A1DD") ->	{"0","",""};
lookup1("A1DE") ->	{"0","",""};
lookup1("A1DF") ->	{"0","",""};
lookup1("A1E0") ->	{"0","",""};
lookup1("A1E1") ->	{"0","",""};
lookup1("A1E2") ->	{"0","",""};
lookup1("A1E3") ->	{"0","",""};
lookup1("A1E4") ->	{"0","",""};
lookup1("A1E5") ->	{"0","",""};
lookup1("A1E6") ->	{"0","",""};
lookup1("A1E7") ->	{"0","",""};
lookup1("A1E8") ->	{"0","",""};
lookup1("A1E9") ->	{"0","",""};
lookup1("A1EA") ->	{"0","",""};
lookup1("A1EB") ->	{"0","",""};
lookup1("A1EC") ->	{"0","",""};
lookup1("A1ED") ->	{"0","",""};
lookup1("A1EE") ->	{"0","",""};
lookup1("A1EF") ->	{"0","",""};
lookup1("A1F0") ->	{"0","",""};
lookup1("A1F1") ->	{"0","",""};
lookup1("A1F2") ->	{"0","",""};
lookup1("A1F3") ->	{"0","",""};
lookup1("A1F4") ->	{"0","",""};
lookup1("A1F5") ->	{"0","",""};
lookup1("A1F6") ->	{"0","",""};
lookup1("A1F7") ->	{"0","",""};
lookup1("A1F8") ->	{"0","",""};
lookup1("A1F9") ->	{"0","",""};
lookup1("A1FA") ->	{"0","",""};
lookup1("A1FB") ->	{"0","",""};
lookup1("A1FC") ->	{"0","",""};
lookup1("A1FD") ->	{"0","",""};
lookup1("A1FE") ->	{"0","",""};
lookup1("A1FF") ->	{"0","",""};
lookup1("A200") ->	{"0","",""};
lookup1("A201") ->	{"0","",""};
lookup1("A202") ->	{"0","",""};
lookup1("A203") ->	{"0","",""};
lookup1("A204") ->	{"0","",""};
lookup1("A205") ->	{"0","",""};
lookup1("A206") ->	{"0","",""};
lookup1("A207") ->	{"0","",""};
lookup1("A208") ->	{"0","",""};
lookup1("A209") ->	{"0","",""};
lookup1("A20A") ->	{"0","",""};
lookup1("A20B") ->	{"0","",""};
lookup1("A20C") ->	{"0","",""};
lookup1("A20D") ->	{"0","",""};
lookup1("A20E") ->	{"0","",""};
lookup1("A20F") ->	{"0","",""};
lookup1("A210") ->	{"0","",""};
lookup1("A211") ->	{"0","",""};
lookup1("A212") ->	{"0","",""};
lookup1("A213") ->	{"0","",""};
lookup1("A214") ->	{"0","",""};
lookup1("A215") ->	{"0","",""};
lookup1("A216") ->	{"0","",""};
lookup1("A217") ->	{"0","",""};
lookup1("A218") ->	{"0","",""};
lookup1("A219") ->	{"0","",""};
lookup1("A21A") ->	{"0","",""};
lookup1("A21B") ->	{"0","",""};
lookup1("A21C") ->	{"0","",""};
lookup1("A21D") ->	{"0","",""};
lookup1("A21E") ->	{"0","",""};
lookup1("A21F") ->	{"0","",""};
lookup1("A220") ->	{"0","",""};
lookup1("A221") ->	{"0","",""};
lookup1("A222") ->	{"0","",""};
lookup1("A223") ->	{"0","",""};
lookup1("A224") ->	{"0","",""};
lookup1("A225") ->	{"0","",""};
lookup1("A226") ->	{"0","",""};
lookup1("A227") ->	{"0","",""};
lookup1("A228") ->	{"0","",""};
lookup1("A229") ->	{"0","",""};
lookup1("A22A") ->	{"0","",""};
lookup1("A22B") ->	{"0","",""};
lookup1("A22C") ->	{"0","",""};
lookup1("A22D") ->	{"0","",""};
lookup1("A22E") ->	{"0","",""};
lookup1("A22F") ->	{"0","",""};
lookup1("A230") ->	{"0","",""};
lookup1("A231") ->	{"0","",""};
lookup1("A232") ->	{"0","",""};
lookup1("A233") ->	{"0","",""};
lookup1("A234") ->	{"0","",""};
lookup1("A235") ->	{"0","",""};
lookup1("A236") ->	{"0","",""};
lookup1("A237") ->	{"0","",""};
lookup1("A238") ->	{"0","",""};
lookup1("A239") ->	{"0","",""};
lookup1("A23A") ->	{"0","",""};
lookup1("A23B") ->	{"0","",""};
lookup1("A23C") ->	{"0","",""};
lookup1("A23D") ->	{"0","",""};
lookup1("A23E") ->	{"0","",""};
lookup1("A23F") ->	{"0","",""};
lookup1("A240") ->	{"0","",""};
lookup1("A241") ->	{"0","",""};
lookup1("A242") ->	{"0","",""};
lookup1("A243") ->	{"0","",""};
lookup1("A244") ->	{"0","",""};
lookup1("A245") ->	{"0","",""};
lookup1("A246") ->	{"0","",""};
lookup1("A247") ->	{"0","",""};
lookup1("A248") ->	{"0","",""};
lookup1("A249") ->	{"0","",""};
lookup1("A24A") ->	{"0","",""};
lookup1("A24B") ->	{"0","",""};
lookup1("A24C") ->	{"0","",""};
lookup1("A24D") ->	{"0","",""};
lookup1("A24E") ->	{"0","",""};
lookup1("A24F") ->	{"0","",""};
lookup1("A250") ->	{"0","",""};
lookup1("A251") ->	{"0","",""};
lookup1("A252") ->	{"0","",""};
lookup1("A253") ->	{"0","",""};
lookup1("A254") ->	{"0","",""};
lookup1("A255") ->	{"0","",""};
lookup1("A256") ->	{"0","",""};
lookup1("A257") ->	{"0","",""};
lookup1("A258") ->	{"0","",""};
lookup1("A259") ->	{"0","",""};
lookup1("A25A") ->	{"0","",""};
lookup1("A25B") ->	{"0","",""};
lookup1("A25C") ->	{"0","",""};
lookup1("A25D") ->	{"0","",""};
lookup1("A25E") ->	{"0","",""};
lookup1("A25F") ->	{"0","",""};
lookup1("A260") ->	{"0","",""};
lookup1("A261") ->	{"0","",""};
lookup1("A262") ->	{"0","",""};
lookup1("A263") ->	{"0","",""};
lookup1("A264") ->	{"0","",""};
lookup1("A265") ->	{"0","",""};
lookup1("A266") ->	{"0","",""};
lookup1("A267") ->	{"0","",""};
lookup1("A268") ->	{"0","",""};
lookup1("A269") ->	{"0","",""};
lookup1("A26A") ->	{"0","",""};
lookup1("A26B") ->	{"0","",""};
lookup1("A26C") ->	{"0","",""};
lookup1("A26D") ->	{"0","",""};
lookup1("A26E") ->	{"0","",""};
lookup1("A26F") ->	{"0","",""};
lookup1("A270") ->	{"0","",""};
lookup1("A271") ->	{"0","",""};
lookup1("A272") ->	{"0","",""};
lookup1("A273") ->	{"0","",""};
lookup1("A274") ->	{"0","",""};
lookup1("A275") ->	{"0","",""};
lookup1("A276") ->	{"0","",""};
lookup1("A277") ->	{"0","",""};
lookup1("A278") ->	{"0","",""};
lookup1("A279") ->	{"0","",""};
lookup1("A27A") ->	{"0","",""};
lookup1("A27B") ->	{"0","",""};
lookup1("A27C") ->	{"0","",""};
lookup1("A27D") ->	{"0","",""};
lookup1("A27E") ->	{"0","",""};
lookup1("A27F") ->	{"0","",""};
lookup1("A280") ->	{"0","",""};
lookup1("A281") ->	{"0","",""};
lookup1("A282") ->	{"0","",""};
lookup1("A283") ->	{"0","",""};
lookup1("A284") ->	{"0","",""};
lookup1("A285") ->	{"0","",""};
lookup1("A286") ->	{"0","",""};
lookup1("A287") ->	{"0","",""};
lookup1("A288") ->	{"0","",""};
lookup1("A289") ->	{"0","",""};
lookup1("A28A") ->	{"0","",""};
lookup1("A28B") ->	{"0","",""};
lookup1("A28C") ->	{"0","",""};
lookup1("A28D") ->	{"0","",""};
lookup1("A28E") ->	{"0","",""};
lookup1("A28F") ->	{"0","",""};
lookup1("A290") ->	{"0","",""};
lookup1("A291") ->	{"0","",""};
lookup1("A292") ->	{"0","",""};
lookup1("A293") ->	{"0","",""};
lookup1("A294") ->	{"0","",""};
lookup1("A295") ->	{"0","",""};
lookup1("A296") ->	{"0","",""};
lookup1("A297") ->	{"0","",""};
lookup1("A298") ->	{"0","",""};
lookup1("A299") ->	{"0","",""};
lookup1("A29A") ->	{"0","",""};
lookup1("A29B") ->	{"0","",""};
lookup1("A29C") ->	{"0","",""};
lookup1("A29D") ->	{"0","",""};
lookup1("A29E") ->	{"0","",""};
lookup1("A29F") ->	{"0","",""};
lookup1("A2A0") ->	{"0","",""};
lookup1("A2A1") ->	{"0","",""};
lookup1("A2A2") ->	{"0","",""};
lookup1("A2A3") ->	{"0","",""};
lookup1("A2A4") ->	{"0","",""};
lookup1("A2A5") ->	{"0","",""};
lookup1("A2A6") ->	{"0","",""};
lookup1("A2A7") ->	{"0","",""};
lookup1("A2A8") ->	{"0","",""};
lookup1("A2A9") ->	{"0","",""};
lookup1("A2AA") ->	{"0","",""};
lookup1("A2AB") ->	{"0","",""};
lookup1("A2AC") ->	{"0","",""};
lookup1("A2AD") ->	{"0","",""};
lookup1("A2AE") ->	{"0","",""};
lookup1("A2AF") ->	{"0","",""};
lookup1("A2B0") ->	{"0","",""};
lookup1("A2B1") ->	{"0","",""};
lookup1("A2B2") ->	{"0","",""};
lookup1("A2B3") ->	{"0","",""};
lookup1("A2B4") ->	{"0","",""};
lookup1("A2B5") ->	{"0","",""};
lookup1("A2B6") ->	{"0","",""};
lookup1("A2B7") ->	{"0","",""};
lookup1("A2B8") ->	{"0","",""};
lookup1("A2B9") ->	{"0","",""};
lookup1("A2BA") ->	{"0","",""};
lookup1("A2BB") ->	{"0","",""};
lookup1("A2BC") ->	{"0","",""};
lookup1("A2BD") ->	{"0","",""};
lookup1("A2BE") ->	{"0","",""};
lookup1("A2BF") ->	{"0","",""};
lookup1("A2C0") ->	{"0","",""};
lookup1("A2C1") ->	{"0","",""};
lookup1("A2C2") ->	{"0","",""};
lookup1("A2C3") ->	{"0","",""};
lookup1("A2C4") ->	{"0","",""};
lookup1("A2C5") ->	{"0","",""};
lookup1("A2C6") ->	{"0","",""};
lookup1("A2C7") ->	{"0","",""};
lookup1("A2C8") ->	{"0","",""};
lookup1("A2C9") ->	{"0","",""};
lookup1("A2CA") ->	{"0","",""};
lookup1("A2CB") ->	{"0","",""};
lookup1("A2CC") ->	{"0","",""};
lookup1("A2CD") ->	{"0","",""};
lookup1("A2CE") ->	{"0","",""};
lookup1("A2CF") ->	{"0","",""};
lookup1("A2D0") ->	{"0","",""};
lookup1("A2D1") ->	{"0","",""};
lookup1("A2D2") ->	{"0","",""};
lookup1("A2D3") ->	{"0","",""};
lookup1("A2D4") ->	{"0","",""};
lookup1("A2D5") ->	{"0","",""};
lookup1("A2D6") ->	{"0","",""};
lookup1("A2D7") ->	{"0","",""};
lookup1("A2D8") ->	{"0","",""};
lookup1("A2D9") ->	{"0","",""};
lookup1("A2DA") ->	{"0","",""};
lookup1("A2DB") ->	{"0","",""};
lookup1("A2DC") ->	{"0","",""};
lookup1("A2DD") ->	{"0","",""};
lookup1("A2DE") ->	{"0","",""};
lookup1("A2DF") ->	{"0","",""};
lookup1("A2E0") ->	{"0","",""};
lookup1("A2E1") ->	{"0","",""};
lookup1("A2E2") ->	{"0","",""};
lookup1("A2E3") ->	{"0","",""};
lookup1("A2E4") ->	{"0","",""};
lookup1("A2E5") ->	{"0","",""};
lookup1("A2E6") ->	{"0","",""};
lookup1("A2E7") ->	{"0","",""};
lookup1("A2E8") ->	{"0","",""};
lookup1("A2E9") ->	{"0","",""};
lookup1("A2EA") ->	{"0","",""};
lookup1("A2EB") ->	{"0","",""};
lookup1("A2EC") ->	{"0","",""};
lookup1("A2ED") ->	{"0","",""};
lookup1("A2EE") ->	{"0","",""};
lookup1("A2EF") ->	{"0","",""};
lookup1("A2F0") ->	{"0","",""};
lookup1("A2F1") ->	{"0","",""};
lookup1("A2F2") ->	{"0","",""};
lookup1("A2F3") ->	{"0","",""};
lookup1("A2F4") ->	{"0","",""};
lookup1("A2F5") ->	{"0","",""};
lookup1("A2F6") ->	{"0","",""};
lookup1("A2F7") ->	{"0","",""};
lookup1("A2F8") ->	{"0","",""};
lookup1("A2F9") ->	{"0","",""};
lookup1("A2FA") ->	{"0","",""};
lookup1("A2FB") ->	{"0","",""};
lookup1("A2FC") ->	{"0","",""};
lookup1("A2FD") ->	{"0","",""};
lookup1("A2FE") ->	{"0","",""};
lookup1("A2FF") ->	{"0","",""};
lookup1("A300") ->	{"0","",""};
lookup1("A301") ->	{"0","",""};
lookup1("A302") ->	{"0","",""};
lookup1("A303") ->	{"0","",""};
lookup1("A304") ->	{"0","",""};
lookup1("A305") ->	{"0","",""};
lookup1("A306") ->	{"0","",""};
lookup1("A307") ->	{"0","",""};
lookup1("A308") ->	{"0","",""};
lookup1("A309") ->	{"0","",""};
lookup1("A30A") ->	{"0","",""};
lookup1("A30B") ->	{"0","",""};
lookup1("A30C") ->	{"0","",""};
lookup1("A30D") ->	{"0","",""};
lookup1("A30E") ->	{"0","",""};
lookup1("A30F") ->	{"0","",""};
lookup1("A310") ->	{"0","",""};
lookup1("A311") ->	{"0","",""};
lookup1("A312") ->	{"0","",""};
lookup1("A313") ->	{"0","",""};
lookup1("A314") ->	{"0","",""};
lookup1("A315") ->	{"0","",""};
lookup1("A316") ->	{"0","",""};
lookup1("A317") ->	{"0","",""};
lookup1("A318") ->	{"0","",""};
lookup1("A319") ->	{"0","",""};
lookup1("A31A") ->	{"0","",""};
lookup1("A31B") ->	{"0","",""};
lookup1("A31C") ->	{"0","",""};
lookup1("A31D") ->	{"0","",""};
lookup1("A31E") ->	{"0","",""};
lookup1("A31F") ->	{"0","",""};
lookup1("A320") ->	{"0","",""};
lookup1("A321") ->	{"0","",""};
lookup1("A322") ->	{"0","",""};
lookup1("A323") ->	{"0","",""};
lookup1("A324") ->	{"0","",""};
lookup1("A325") ->	{"0","",""};
lookup1("A326") ->	{"0","",""};
lookup1("A327") ->	{"0","",""};
lookup1("A328") ->	{"0","",""};
lookup1("A329") ->	{"0","",""};
lookup1("A32A") ->	{"0","",""};
lookup1("A32B") ->	{"0","",""};
lookup1("A32C") ->	{"0","",""};
lookup1("A32D") ->	{"0","",""};
lookup1("A32E") ->	{"0","",""};
lookup1("A32F") ->	{"0","",""};
lookup1("A330") ->	{"0","",""};
lookup1("A331") ->	{"0","",""};
lookup1("A332") ->	{"0","",""};
lookup1("A333") ->	{"0","",""};
lookup1("A334") ->	{"0","",""};
lookup1("A335") ->	{"0","",""};
lookup1("A336") ->	{"0","",""};
lookup1("A337") ->	{"0","",""};
lookup1("A338") ->	{"0","",""};
lookup1("A339") ->	{"0","",""};
lookup1("A33A") ->	{"0","",""};
lookup1("A33B") ->	{"0","",""};
lookup1("A33C") ->	{"0","",""};
lookup1("A33D") ->	{"0","",""};
lookup1("A33E") ->	{"0","",""};
lookup1("A33F") ->	{"0","",""};
lookup1("A340") ->	{"0","",""};
lookup1("A341") ->	{"0","",""};
lookup1("A342") ->	{"0","",""};
lookup1("A343") ->	{"0","",""};
lookup1("A344") ->	{"0","",""};
lookup1("A345") ->	{"0","",""};
lookup1("A346") ->	{"0","",""};
lookup1("A347") ->	{"0","",""};
lookup1("A348") ->	{"0","",""};
lookup1("A349") ->	{"0","",""};
lookup1("A34A") ->	{"0","",""};
lookup1("A34B") ->	{"0","",""};
lookup1("A34C") ->	{"0","",""};
lookup1("A34D") ->	{"0","",""};
lookup1("A34E") ->	{"0","",""};
lookup1("A34F") ->	{"0","",""};
lookup1("A350") ->	{"0","",""};
lookup1("A351") ->	{"0","",""};
lookup1("A352") ->	{"0","",""};
lookup1("A353") ->	{"0","",""};
lookup1("A354") ->	{"0","",""};
lookup1("A355") ->	{"0","",""};
lookup1("A356") ->	{"0","",""};
lookup1("A357") ->	{"0","",""};
lookup1("A358") ->	{"0","",""};
lookup1("A359") ->	{"0","",""};
lookup1("A35A") ->	{"0","",""};
lookup1("A35B") ->	{"0","",""};
lookup1("A35C") ->	{"0","",""};
lookup1("A35D") ->	{"0","",""};
lookup1("A35E") ->	{"0","",""};
lookup1("A35F") ->	{"0","",""};
lookup1("A360") ->	{"0","",""};
lookup1("A361") ->	{"0","",""};
lookup1("A362") ->	{"0","",""};
lookup1("A363") ->	{"0","",""};
lookup1("A364") ->	{"0","",""};
lookup1("A365") ->	{"0","",""};
lookup1("A366") ->	{"0","",""};
lookup1("A367") ->	{"0","",""};
lookup1("A368") ->	{"0","",""};
lookup1("A369") ->	{"0","",""};
lookup1("A36A") ->	{"0","",""};
lookup1("A36B") ->	{"0","",""};
lookup1("A36C") ->	{"0","",""};
lookup1("A36D") ->	{"0","",""};
lookup1("A36E") ->	{"0","",""};
lookup1("A36F") ->	{"0","",""};
lookup1("A370") ->	{"0","",""};
lookup1("A371") ->	{"0","",""};
lookup1("A372") ->	{"0","",""};
lookup1("A373") ->	{"0","",""};
lookup1("A374") ->	{"0","",""};
lookup1("A375") ->	{"0","",""};
lookup1("A376") ->	{"0","",""};
lookup1("A377") ->	{"0","",""};
lookup1("A378") ->	{"0","",""};
lookup1("A379") ->	{"0","",""};
lookup1("A37A") ->	{"0","",""};
lookup1("A37B") ->	{"0","",""};
lookup1("A37C") ->	{"0","",""};
lookup1("A37D") ->	{"0","",""};
lookup1("A37E") ->	{"0","",""};
lookup1("A37F") ->	{"0","",""};
lookup1("A380") ->	{"0","",""};
lookup1("A381") ->	{"0","",""};
lookup1("A382") ->	{"0","",""};
lookup1("A383") ->	{"0","",""};
lookup1("A384") ->	{"0","",""};
lookup1("A385") ->	{"0","",""};
lookup1("A386") ->	{"0","",""};
lookup1("A387") ->	{"0","",""};
lookup1("A388") ->	{"0","",""};
lookup1("A389") ->	{"0","",""};
lookup1("A38A") ->	{"0","",""};
lookup1("A38B") ->	{"0","",""};
lookup1("A38C") ->	{"0","",""};
lookup1("A38D") ->	{"0","",""};
lookup1("A38E") ->	{"0","",""};
lookup1("A38F") ->	{"0","",""};
lookup1("A390") ->	{"0","",""};
lookup1("A391") ->	{"0","",""};
lookup1("A392") ->	{"0","",""};
lookup1("A393") ->	{"0","",""};
lookup1("A394") ->	{"0","",""};
lookup1("A395") ->	{"0","",""};
lookup1("A396") ->	{"0","",""};
lookup1("A397") ->	{"0","",""};
lookup1("A398") ->	{"0","",""};
lookup1("A399") ->	{"0","",""};
lookup1("A39A") ->	{"0","",""};
lookup1("A39B") ->	{"0","",""};
lookup1("A39C") ->	{"0","",""};
lookup1("A39D") ->	{"0","",""};
lookup1("A39E") ->	{"0","",""};
lookup1("A39F") ->	{"0","",""};
lookup1("A3A0") ->	{"0","",""};
lookup1("A3A1") ->	{"0","",""};
lookup1("A3A2") ->	{"0","",""};
lookup1("A3A3") ->	{"0","",""};
lookup1("A3A4") ->	{"0","",""};
lookup1("A3A5") ->	{"0","",""};
lookup1("A3A6") ->	{"0","",""};
lookup1("A3A7") ->	{"0","",""};
lookup1("A3A8") ->	{"0","",""};
lookup1("A3A9") ->	{"0","",""};
lookup1("A3AA") ->	{"0","",""};
lookup1("A3AB") ->	{"0","",""};
lookup1("A3AC") ->	{"0","",""};
lookup1("A3AD") ->	{"0","",""};
lookup1("A3AE") ->	{"0","",""};
lookup1("A3AF") ->	{"0","",""};
lookup1("A3B0") ->	{"0","",""};
lookup1("A3B1") ->	{"0","",""};
lookup1("A3B2") ->	{"0","",""};
lookup1("A3B3") ->	{"0","",""};
lookup1("A3B4") ->	{"0","",""};
lookup1("A3B5") ->	{"0","",""};
lookup1("A3B6") ->	{"0","",""};
lookup1("A3B7") ->	{"0","",""};
lookup1("A3B8") ->	{"0","",""};
lookup1("A3B9") ->	{"0","",""};
lookup1("A3BA") ->	{"0","",""};
lookup1("A3BB") ->	{"0","",""};
lookup1("A3BC") ->	{"0","",""};
lookup1("A3BD") ->	{"0","",""};
lookup1("A3BE") ->	{"0","",""};
lookup1("A3BF") ->	{"0","",""};
lookup1("A3C0") ->	{"0","",""};
lookup1("A3C1") ->	{"0","",""};
lookup1("A3C2") ->	{"0","",""};
lookup1("A3C3") ->	{"0","",""};
lookup1("A3C4") ->	{"0","",""};
lookup1("A3C5") ->	{"0","",""};
lookup1("A3C6") ->	{"0","",""};
lookup1("A3C7") ->	{"0","",""};
lookup1("A3C8") ->	{"0","",""};
lookup1("A3C9") ->	{"0","",""};
lookup1("A3CA") ->	{"0","",""};
lookup1("A3CB") ->	{"0","",""};
lookup1("A3CC") ->	{"0","",""};
lookup1("A3CD") ->	{"0","",""};
lookup1("A3CE") ->	{"0","",""};
lookup1("A3CF") ->	{"0","",""};
lookup1("A3D0") ->	{"0","",""};
lookup1("A3D1") ->	{"0","",""};
lookup1("A3D2") ->	{"0","",""};
lookup1("A3D3") ->	{"0","",""};
lookup1("A3D4") ->	{"0","",""};
lookup1("A3D5") ->	{"0","",""};
lookup1("A3D6") ->	{"0","",""};
lookup1("A3D7") ->	{"0","",""};
lookup1("A3D8") ->	{"0","",""};
lookup1("A3D9") ->	{"0","",""};
lookup1("A3DA") ->	{"0","",""};
lookup1("A3DB") ->	{"0","",""};
lookup1("A3DC") ->	{"0","",""};
lookup1("A3DD") ->	{"0","",""};
lookup1("A3DE") ->	{"0","",""};
lookup1("A3DF") ->	{"0","",""};
lookup1("A3E0") ->	{"0","",""};
lookup1("A3E1") ->	{"0","",""};
lookup1("A3E2") ->	{"0","",""};
lookup1("A3E3") ->	{"0","",""};
lookup1("A3E4") ->	{"0","",""};
lookup1("A3E5") ->	{"0","",""};
lookup1("A3E6") ->	{"0","",""};
lookup1("A3E7") ->	{"0","",""};
lookup1("A3E8") ->	{"0","",""};
lookup1("A3E9") ->	{"0","",""};
lookup1("A3EA") ->	{"0","",""};
lookup1("A3EB") ->	{"0","",""};
lookup1("A3EC") ->	{"0","",""};
lookup1("A3ED") ->	{"0","",""};
lookup1("A3EE") ->	{"0","",""};
lookup1("A3EF") ->	{"0","",""};
lookup1("A3F0") ->	{"0","",""};
lookup1("A3F1") ->	{"0","",""};
lookup1("A3F2") ->	{"0","",""};
lookup1("A3F3") ->	{"0","",""};
lookup1("A3F4") ->	{"0","",""};
lookup1("A3F5") ->	{"0","",""};
lookup1("A3F6") ->	{"0","",""};
lookup1("A3F7") ->	{"0","",""};
lookup1("A3F8") ->	{"0","",""};
lookup1("A3F9") ->	{"0","",""};
lookup1("A3FA") ->	{"0","",""};
lookup1("A3FB") ->	{"0","",""};
lookup1("A3FC") ->	{"0","",""};
lookup1("A3FD") ->	{"0","",""};
lookup1("A3FE") ->	{"0","",""};
lookup1("A3FF") ->	{"0","",""};
lookup1("A400") ->	{"0","",""};
lookup1("A401") ->	{"0","",""};
lookup1("A402") ->	{"0","",""};
lookup1("A403") ->	{"0","",""};
lookup1("A404") ->	{"0","",""};
lookup1("A405") ->	{"0","",""};
lookup1("A406") ->	{"0","",""};
lookup1("A407") ->	{"0","",""};
lookup1("A408") ->	{"0","",""};
lookup1("A409") ->	{"0","",""};
lookup1("A40A") ->	{"0","",""};
lookup1("A40B") ->	{"0","",""};
lookup1("A40C") ->	{"0","",""};
lookup1("A40D") ->	{"0","",""};
lookup1("A40E") ->	{"0","",""};
lookup1("A40F") ->	{"0","",""};
lookup1("A410") ->	{"0","",""};
lookup1("A411") ->	{"0","",""};
lookup1("A412") ->	{"0","",""};
lookup1("A413") ->	{"0","",""};
lookup1("A414") ->	{"0","",""};
lookup1("A415") ->	{"0","",""};
lookup1("A416") ->	{"0","",""};
lookup1("A417") ->	{"0","",""};
lookup1("A418") ->	{"0","",""};
lookup1("A419") ->	{"0","",""};
lookup1("A41A") ->	{"0","",""};
lookup1("A41B") ->	{"0","",""};
lookup1("A41C") ->	{"0","",""};
lookup1("A41D") ->	{"0","",""};
lookup1("A41E") ->	{"0","",""};
lookup1("A41F") ->	{"0","",""};
lookup1("A420") ->	{"0","",""};
lookup1("A421") ->	{"0","",""};
lookup1("A422") ->	{"0","",""};
lookup1("A423") ->	{"0","",""};
lookup1("A424") ->	{"0","",""};
lookup1("A425") ->	{"0","",""};
lookup1("A426") ->	{"0","",""};
lookup1("A427") ->	{"0","",""};
lookup1("A428") ->	{"0","",""};
lookup1("A429") ->	{"0","",""};
lookup1("A42A") ->	{"0","",""};
lookup1("A42B") ->	{"0","",""};
lookup1("A42C") ->	{"0","",""};
lookup1("A42D") ->	{"0","",""};
lookup1("A42E") ->	{"0","",""};
lookup1("A42F") ->	{"0","",""};
lookup1("A430") ->	{"0","",""};
lookup1("A431") ->	{"0","",""};
lookup1("A432") ->	{"0","",""};
lookup1("A433") ->	{"0","",""};
lookup1("A434") ->	{"0","",""};
lookup1("A435") ->	{"0","",""};
lookup1("A436") ->	{"0","",""};
lookup1("A437") ->	{"0","",""};
lookup1("A438") ->	{"0","",""};
lookup1("A439") ->	{"0","",""};
lookup1("A43A") ->	{"0","",""};
lookup1("A43B") ->	{"0","",""};
lookup1("A43C") ->	{"0","",""};
lookup1("A43D") ->	{"0","",""};
lookup1("A43E") ->	{"0","",""};
lookup1("A43F") ->	{"0","",""};
lookup1("A440") ->	{"0","",""};
lookup1("A441") ->	{"0","",""};
lookup1("A442") ->	{"0","",""};
lookup1("A443") ->	{"0","",""};
lookup1("A444") ->	{"0","",""};
lookup1("A445") ->	{"0","",""};
lookup1("A446") ->	{"0","",""};
lookup1("A447") ->	{"0","",""};
lookup1("A448") ->	{"0","",""};
lookup1("A449") ->	{"0","",""};
lookup1("A44A") ->	{"0","",""};
lookup1("A44B") ->	{"0","",""};
lookup1("A44C") ->	{"0","",""};
lookup1("A44D") ->	{"0","",""};
lookup1("A44E") ->	{"0","",""};
lookup1("A44F") ->	{"0","",""};
lookup1("A450") ->	{"0","",""};
lookup1("A451") ->	{"0","",""};
lookup1("A452") ->	{"0","",""};
lookup1("A453") ->	{"0","",""};
lookup1("A454") ->	{"0","",""};
lookup1("A455") ->	{"0","",""};
lookup1("A456") ->	{"0","",""};
lookup1("A457") ->	{"0","",""};
lookup1("A458") ->	{"0","",""};
lookup1("A459") ->	{"0","",""};
lookup1("A45A") ->	{"0","",""};
lookup1("A45B") ->	{"0","",""};
lookup1("A45C") ->	{"0","",""};
lookup1("A45D") ->	{"0","",""};
lookup1("A45E") ->	{"0","",""};
lookup1("A45F") ->	{"0","",""};
lookup1("A460") ->	{"0","",""};
lookup1("A461") ->	{"0","",""};
lookup1("A462") ->	{"0","",""};
lookup1("A463") ->	{"0","",""};
lookup1("A464") ->	{"0","",""};
lookup1("A465") ->	{"0","",""};
lookup1("A466") ->	{"0","",""};
lookup1("A467") ->	{"0","",""};
lookup1("A468") ->	{"0","",""};
lookup1("A469") ->	{"0","",""};
lookup1("A46A") ->	{"0","",""};
lookup1("A46B") ->	{"0","",""};
lookup1("A46C") ->	{"0","",""};
lookup1("A46D") ->	{"0","",""};
lookup1("A46E") ->	{"0","",""};
lookup1("A46F") ->	{"0","",""};
lookup1("A470") ->	{"0","",""};
lookup1("A471") ->	{"0","",""};
lookup1("A472") ->	{"0","",""};
lookup1("A473") ->	{"0","",""};
lookup1("A474") ->	{"0","",""};
lookup1("A475") ->	{"0","",""};
lookup1("A476") ->	{"0","",""};
lookup1("A477") ->	{"0","",""};
lookup1("A478") ->	{"0","",""};
lookup1("A479") ->	{"0","",""};
lookup1("A47A") ->	{"0","",""};
lookup1("A47B") ->	{"0","",""};
lookup1("A47C") ->	{"0","",""};
lookup1("A47D") ->	{"0","",""};
lookup1("A47E") ->	{"0","",""};
lookup1("A47F") ->	{"0","",""};
lookup1("A480") ->	{"0","",""};
lookup1("A481") ->	{"0","",""};
lookup1("A482") ->	{"0","",""};
lookup1("A483") ->	{"0","",""};
lookup1("A484") ->	{"0","",""};
lookup1("A485") ->	{"0","",""};
lookup1("A486") ->	{"0","",""};
lookup1("A487") ->	{"0","",""};
lookup1("A488") ->	{"0","",""};
lookup1("A489") ->	{"0","",""};
lookup1("A48A") ->	{"0","",""};
lookup1("A48B") ->	{"0","",""};
lookup1("A48C") ->	{"0","",""};
lookup1("A490") ->	{"0","",""};
lookup1("A491") ->	{"0","",""};
lookup1("A492") ->	{"0","",""};
lookup1("A493") ->	{"0","",""};
lookup1("A494") ->	{"0","",""};
lookup1("A495") ->	{"0","",""};
lookup1("A496") ->	{"0","",""};
lookup1("A497") ->	{"0","",""};
lookup1("A498") ->	{"0","",""};
lookup1("A499") ->	{"0","",""};
lookup1("A49A") ->	{"0","",""};
lookup1("A49B") ->	{"0","",""};
lookup1("A49C") ->	{"0","",""};
lookup1("A49D") ->	{"0","",""};
lookup1("A49E") ->	{"0","",""};
lookup1("A49F") ->	{"0","",""};
lookup1("A4A0") ->	{"0","",""};
lookup1("A4A1") ->	{"0","",""};
lookup1("A4A2") ->	{"0","",""};
lookup1("A4A3") ->	{"0","",""};
lookup1("A4A4") ->	{"0","",""};
lookup1("A4A5") ->	{"0","",""};
lookup1("A4A6") ->	{"0","",""};
lookup1("A4A7") ->	{"0","",""};
lookup1("A4A8") ->	{"0","",""};
lookup1("A4A9") ->	{"0","",""};
lookup1("A4AA") ->	{"0","",""};
lookup1("A4AB") ->	{"0","",""};
lookup1("A4AC") ->	{"0","",""};
lookup1("A4AD") ->	{"0","",""};
lookup1("A4AE") ->	{"0","",""};
lookup1("A4AF") ->	{"0","",""};
lookup1("A4B0") ->	{"0","",""};
lookup1("A4B1") ->	{"0","",""};
lookup1("A4B2") ->	{"0","",""};
lookup1("A4B3") ->	{"0","",""};
lookup1("A4B4") ->	{"0","",""};
lookup1("A4B5") ->	{"0","",""};
lookup1("A4B6") ->	{"0","",""};
lookup1("A4B7") ->	{"0","",""};
lookup1("A4B8") ->	{"0","",""};
lookup1("A4B9") ->	{"0","",""};
lookup1("A4BA") ->	{"0","",""};
lookup1("A4BB") ->	{"0","",""};
lookup1("A4BC") ->	{"0","",""};
lookup1("A4BD") ->	{"0","",""};
lookup1("A4BE") ->	{"0","",""};
lookup1("A4BF") ->	{"0","",""};
lookup1("A4C0") ->	{"0","",""};
lookup1("A4C1") ->	{"0","",""};
lookup1("A4C2") ->	{"0","",""};
lookup1("A4C3") ->	{"0","",""};
lookup1("A4C4") ->	{"0","",""};
lookup1("A4C5") ->	{"0","",""};
lookup1("A4C6") ->	{"0","",""};
lookup1("A4D0") ->	{"0","",""};
lookup1("A4D1") ->	{"0","",""};
lookup1("A4D2") ->	{"0","",""};
lookup1("A4D3") ->	{"0","",""};
lookup1("A4D4") ->	{"0","",""};
lookup1("A4D5") ->	{"0","",""};
lookup1("A4D6") ->	{"0","",""};
lookup1("A4D7") ->	{"0","",""};
lookup1("A4D8") ->	{"0","",""};
lookup1("A4D9") ->	{"0","",""};
lookup1("A4DA") ->	{"0","",""};
lookup1("A4DB") ->	{"0","",""};
lookup1("A4DC") ->	{"0","",""};
lookup1("A4DD") ->	{"0","",""};
lookup1("A4DE") ->	{"0","",""};
lookup1("A4DF") ->	{"0","",""};
lookup1("A4E0") ->	{"0","",""};
lookup1("A4E1") ->	{"0","",""};
lookup1("A4E2") ->	{"0","",""};
lookup1("A4E3") ->	{"0","",""};
lookup1("A4E4") ->	{"0","",""};
lookup1("A4E5") ->	{"0","",""};
lookup1("A4E6") ->	{"0","",""};
lookup1("A4E7") ->	{"0","",""};
lookup1("A4E8") ->	{"0","",""};
lookup1("A4E9") ->	{"0","",""};
lookup1("A4EA") ->	{"0","",""};
lookup1("A4EB") ->	{"0","",""};
lookup1("A4EC") ->	{"0","",""};
lookup1("A4ED") ->	{"0","",""};
lookup1("A4EE") ->	{"0","",""};
lookup1("A4EF") ->	{"0","",""};
lookup1("A4F0") ->	{"0","",""};
lookup1("A4F1") ->	{"0","",""};
lookup1("A4F2") ->	{"0","",""};
lookup1("A4F3") ->	{"0","",""};
lookup1("A4F4") ->	{"0","",""};
lookup1("A4F5") ->	{"0","",""};
lookup1("A4F6") ->	{"0","",""};
lookup1("A4F7") ->	{"0","",""};
lookup1("A4F8") ->	{"0","",""};
lookup1("A4F9") ->	{"0","",""};
lookup1("A4FA") ->	{"0","",""};
lookup1("A4FB") ->	{"0","",""};
lookup1("A4FC") ->	{"0","",""};
lookup1("A4FD") ->	{"0","",""};
lookup1("A4FE") ->	{"0","",""};
lookup1("A4FF") ->	{"0","",""};
lookup1("A500") ->	{"0","",""};
lookup1("A501") ->	{"0","",""};
lookup1("A502") ->	{"0","",""};
lookup1("A503") ->	{"0","",""};
lookup1("A504") ->	{"0","",""};
lookup1("A505") ->	{"0","",""};
lookup1("A506") ->	{"0","",""};
lookup1("A507") ->	{"0","",""};
lookup1("A508") ->	{"0","",""};
lookup1("A509") ->	{"0","",""};
lookup1("A50A") ->	{"0","",""};
lookup1("A50B") ->	{"0","",""};
lookup1("A50C") ->	{"0","",""};
lookup1("A50D") ->	{"0","",""};
lookup1("A50E") ->	{"0","",""};
lookup1("A50F") ->	{"0","",""};
lookup1("A510") ->	{"0","",""};
lookup1("A511") ->	{"0","",""};
lookup1("A512") ->	{"0","",""};
lookup1("A513") ->	{"0","",""};
lookup1("A514") ->	{"0","",""};
lookup1("A515") ->	{"0","",""};
lookup1("A516") ->	{"0","",""};
lookup1("A517") ->	{"0","",""};
lookup1("A518") ->	{"0","",""};
lookup1("A519") ->	{"0","",""};
lookup1("A51A") ->	{"0","",""};
lookup1("A51B") ->	{"0","",""};
lookup1("A51C") ->	{"0","",""};
lookup1("A51D") ->	{"0","",""};
lookup1("A51E") ->	{"0","",""};
lookup1("A51F") ->	{"0","",""};
lookup1("A520") ->	{"0","",""};
lookup1("A521") ->	{"0","",""};
lookup1("A522") ->	{"0","",""};
lookup1("A523") ->	{"0","",""};
lookup1("A524") ->	{"0","",""};
lookup1("A525") ->	{"0","",""};
lookup1("A526") ->	{"0","",""};
lookup1("A527") ->	{"0","",""};
lookup1("A528") ->	{"0","",""};
lookup1("A529") ->	{"0","",""};
lookup1("A52A") ->	{"0","",""};
lookup1("A52B") ->	{"0","",""};
lookup1("A52C") ->	{"0","",""};
lookup1("A52D") ->	{"0","",""};
lookup1("A52E") ->	{"0","",""};
lookup1("A52F") ->	{"0","",""};
lookup1("A530") ->	{"0","",""};
lookup1("A531") ->	{"0","",""};
lookup1("A532") ->	{"0","",""};
lookup1("A533") ->	{"0","",""};
lookup1("A534") ->	{"0","",""};
lookup1("A535") ->	{"0","",""};
lookup1("A536") ->	{"0","",""};
lookup1("A537") ->	{"0","",""};
lookup1("A538") ->	{"0","",""};
lookup1("A539") ->	{"0","",""};
lookup1("A53A") ->	{"0","",""};
lookup1("A53B") ->	{"0","",""};
lookup1("A53C") ->	{"0","",""};
lookup1("A53D") ->	{"0","",""};
lookup1("A53E") ->	{"0","",""};
lookup1("A53F") ->	{"0","",""};
lookup1("A540") ->	{"0","",""};
lookup1("A541") ->	{"0","",""};
lookup1("A542") ->	{"0","",""};
lookup1("A543") ->	{"0","",""};
lookup1("A544") ->	{"0","",""};
lookup1("A545") ->	{"0","",""};
lookup1("A546") ->	{"0","",""};
lookup1("A547") ->	{"0","",""};
lookup1("A548") ->	{"0","",""};
lookup1("A549") ->	{"0","",""};
lookup1("A54A") ->	{"0","",""};
lookup1("A54B") ->	{"0","",""};
lookup1("A54C") ->	{"0","",""};
lookup1("A54D") ->	{"0","",""};
lookup1("A54E") ->	{"0","",""};
lookup1("A54F") ->	{"0","",""};
lookup1("A550") ->	{"0","",""};
lookup1("A551") ->	{"0","",""};
lookup1("A552") ->	{"0","",""};
lookup1("A553") ->	{"0","",""};
lookup1("A554") ->	{"0","",""};
lookup1("A555") ->	{"0","",""};
lookup1("A556") ->	{"0","",""};
lookup1("A557") ->	{"0","",""};
lookup1("A558") ->	{"0","",""};
lookup1("A559") ->	{"0","",""};
lookup1("A55A") ->	{"0","",""};
lookup1("A55B") ->	{"0","",""};
lookup1("A55C") ->	{"0","",""};
lookup1("A55D") ->	{"0","",""};
lookup1("A55E") ->	{"0","",""};
lookup1("A55F") ->	{"0","",""};
lookup1("A560") ->	{"0","",""};
lookup1("A561") ->	{"0","",""};
lookup1("A562") ->	{"0","",""};
lookup1("A563") ->	{"0","",""};
lookup1("A564") ->	{"0","",""};
lookup1("A565") ->	{"0","",""};
lookup1("A566") ->	{"0","",""};
lookup1("A567") ->	{"0","",""};
lookup1("A568") ->	{"0","",""};
lookup1("A569") ->	{"0","",""};
lookup1("A56A") ->	{"0","",""};
lookup1("A56B") ->	{"0","",""};
lookup1("A56C") ->	{"0","",""};
lookup1("A56D") ->	{"0","",""};
lookup1("A56E") ->	{"0","",""};
lookup1("A56F") ->	{"0","",""};
lookup1("A570") ->	{"0","",""};
lookup1("A571") ->	{"0","",""};
lookup1("A572") ->	{"0","",""};
lookup1("A573") ->	{"0","",""};
lookup1("A574") ->	{"0","",""};
lookup1("A575") ->	{"0","",""};
lookup1("A576") ->	{"0","",""};
lookup1("A577") ->	{"0","",""};
lookup1("A578") ->	{"0","",""};
lookup1("A579") ->	{"0","",""};
lookup1("A57A") ->	{"0","",""};
lookup1("A57B") ->	{"0","",""};
lookup1("A57C") ->	{"0","",""};
lookup1("A57D") ->	{"0","",""};
lookup1("A57E") ->	{"0","",""};
lookup1("A57F") ->	{"0","",""};
lookup1("A580") ->	{"0","",""};
lookup1("A581") ->	{"0","",""};
lookup1("A582") ->	{"0","",""};
lookup1("A583") ->	{"0","",""};
lookup1("A584") ->	{"0","",""};
lookup1("A585") ->	{"0","",""};
lookup1("A586") ->	{"0","",""};
lookup1("A587") ->	{"0","",""};
lookup1("A588") ->	{"0","",""};
lookup1("A589") ->	{"0","",""};
lookup1("A58A") ->	{"0","",""};
lookup1("A58B") ->	{"0","",""};
lookup1("A58C") ->	{"0","",""};
lookup1("A58D") ->	{"0","",""};
lookup1("A58E") ->	{"0","",""};
lookup1("A58F") ->	{"0","",""};
lookup1("A590") ->	{"0","",""};
lookup1("A591") ->	{"0","",""};
lookup1("A592") ->	{"0","",""};
lookup1("A593") ->	{"0","",""};
lookup1("A594") ->	{"0","",""};
lookup1("A595") ->	{"0","",""};
lookup1("A596") ->	{"0","",""};
lookup1("A597") ->	{"0","",""};
lookup1("A598") ->	{"0","",""};
lookup1("A599") ->	{"0","",""};
lookup1("A59A") ->	{"0","",""};
lookup1("A59B") ->	{"0","",""};
lookup1("A59C") ->	{"0","",""};
lookup1("A59D") ->	{"0","",""};
lookup1("A59E") ->	{"0","",""};
lookup1("A59F") ->	{"0","",""};
lookup1("A5A0") ->	{"0","",""};
lookup1("A5A1") ->	{"0","",""};
lookup1("A5A2") ->	{"0","",""};
lookup1("A5A3") ->	{"0","",""};
lookup1("A5A4") ->	{"0","",""};
lookup1("A5A5") ->	{"0","",""};
lookup1("A5A6") ->	{"0","",""};
lookup1("A5A7") ->	{"0","",""};
lookup1("A5A8") ->	{"0","",""};
lookup1("A5A9") ->	{"0","",""};
lookup1("A5AA") ->	{"0","",""};
lookup1("A5AB") ->	{"0","",""};
lookup1("A5AC") ->	{"0","",""};
lookup1("A5AD") ->	{"0","",""};
lookup1("A5AE") ->	{"0","",""};
lookup1("A5AF") ->	{"0","",""};
lookup1("A5B0") ->	{"0","",""};
lookup1("A5B1") ->	{"0","",""};
lookup1("A5B2") ->	{"0","",""};
lookup1("A5B3") ->	{"0","",""};
lookup1("A5B4") ->	{"0","",""};
lookup1("A5B5") ->	{"0","",""};
lookup1("A5B6") ->	{"0","",""};
lookup1("A5B7") ->	{"0","",""};
lookup1("A5B8") ->	{"0","",""};
lookup1("A5B9") ->	{"0","",""};
lookup1("A5BA") ->	{"0","",""};
lookup1("A5BB") ->	{"0","",""};
lookup1("A5BC") ->	{"0","",""};
lookup1("A5BD") ->	{"0","",""};
lookup1("A5BE") ->	{"0","",""};
lookup1("A5BF") ->	{"0","",""};
lookup1("A5C0") ->	{"0","",""};
lookup1("A5C1") ->	{"0","",""};
lookup1("A5C2") ->	{"0","",""};
lookup1("A5C3") ->	{"0","",""};
lookup1("A5C4") ->	{"0","",""};
lookup1("A5C5") ->	{"0","",""};
lookup1("A5C6") ->	{"0","",""};
lookup1("A5C7") ->	{"0","",""};
lookup1("A5C8") ->	{"0","",""};
lookup1("A5C9") ->	{"0","",""};
lookup1("A5CA") ->	{"0","",""};
lookup1("A5CB") ->	{"0","",""};
lookup1("A5CC") ->	{"0","",""};
lookup1("A5CD") ->	{"0","",""};
lookup1("A5CE") ->	{"0","",""};
lookup1("A5CF") ->	{"0","",""};
lookup1("A5D0") ->	{"0","",""};
lookup1("A5D1") ->	{"0","",""};
lookup1("A5D2") ->	{"0","",""};
lookup1("A5D3") ->	{"0","",""};
lookup1("A5D4") ->	{"0","",""};
lookup1("A5D5") ->	{"0","",""};
lookup1("A5D6") ->	{"0","",""};
lookup1("A5D7") ->	{"0","",""};
lookup1("A5D8") ->	{"0","",""};
lookup1("A5D9") ->	{"0","",""};
lookup1("A5DA") ->	{"0","",""};
lookup1("A5DB") ->	{"0","",""};
lookup1("A5DC") ->	{"0","",""};
lookup1("A5DD") ->	{"0","",""};
lookup1("A5DE") ->	{"0","",""};
lookup1("A5DF") ->	{"0","",""};
lookup1("A5E0") ->	{"0","",""};
lookup1("A5E1") ->	{"0","",""};
lookup1("A5E2") ->	{"0","",""};
lookup1("A5E3") ->	{"0","",""};
lookup1("A5E4") ->	{"0","",""};
lookup1("A5E5") ->	{"0","",""};
lookup1("A5E6") ->	{"0","",""};
lookup1("A5E7") ->	{"0","",""};
lookup1("A5E8") ->	{"0","",""};
lookup1("A5E9") ->	{"0","",""};
lookup1("A5EA") ->	{"0","",""};
lookup1("A5EB") ->	{"0","",""};
lookup1("A5EC") ->	{"0","",""};
lookup1("A5ED") ->	{"0","",""};
lookup1("A5EE") ->	{"0","",""};
lookup1("A5EF") ->	{"0","",""};
lookup1("A5F0") ->	{"0","",""};
lookup1("A5F1") ->	{"0","",""};
lookup1("A5F2") ->	{"0","",""};
lookup1("A5F3") ->	{"0","",""};
lookup1("A5F4") ->	{"0","",""};
lookup1("A5F5") ->	{"0","",""};
lookup1("A5F6") ->	{"0","",""};
lookup1("A5F7") ->	{"0","",""};
lookup1("A5F8") ->	{"0","",""};
lookup1("A5F9") ->	{"0","",""};
lookup1("A5FA") ->	{"0","",""};
lookup1("A5FB") ->	{"0","",""};
lookup1("A5FC") ->	{"0","",""};
lookup1("A5FD") ->	{"0","",""};
lookup1("A5FE") ->	{"0","",""};
lookup1("A5FF") ->	{"0","",""};
lookup1("A600") ->	{"0","",""};
lookup1("A601") ->	{"0","",""};
lookup1("A602") ->	{"0","",""};
lookup1("A603") ->	{"0","",""};
lookup1("A604") ->	{"0","",""};
lookup1("A605") ->	{"0","",""};
lookup1("A606") ->	{"0","",""};
lookup1("A607") ->	{"0","",""};
lookup1("A608") ->	{"0","",""};
lookup1("A609") ->	{"0","",""};
lookup1("A60A") ->	{"0","",""};
lookup1("A60B") ->	{"0","",""};
lookup1("A60C") ->	{"0","",""};
lookup1("A60D") ->	{"0","",""};
lookup1("A60E") ->	{"0","",""};
lookup1("A60F") ->	{"0","",""};
lookup1("A610") ->	{"0","",""};
lookup1("A611") ->	{"0","",""};
lookup1("A612") ->	{"0","",""};
lookup1("A613") ->	{"0","",""};
lookup1("A614") ->	{"0","",""};
lookup1("A615") ->	{"0","",""};
lookup1("A616") ->	{"0","",""};
lookup1("A617") ->	{"0","",""};
lookup1("A618") ->	{"0","",""};
lookup1("A619") ->	{"0","",""};
lookup1("A61A") ->	{"0","",""};
lookup1("A61B") ->	{"0","",""};
lookup1("A61C") ->	{"0","",""};
lookup1("A61D") ->	{"0","",""};
lookup1("A61E") ->	{"0","",""};
lookup1("A61F") ->	{"0","",""};
lookup1("A620") ->	{"0","",""};
lookup1("A621") ->	{"0","",""};
lookup1("A622") ->	{"0","",""};
lookup1("A623") ->	{"0","",""};
lookup1("A624") ->	{"0","",""};
lookup1("A625") ->	{"0","",""};
lookup1("A626") ->	{"0","",""};
lookup1("A627") ->	{"0","",""};
lookup1("A628") ->	{"0","",""};
lookup1("A629") ->	{"0","",""};
lookup1("A62A") ->	{"0","",""};
lookup1("A62B") ->	{"0","",""};
lookup1("A640") ->	{"0","","A641"};
lookup1("A641") ->	{"0","",""};
lookup1("A642") ->	{"0","","A643"};
lookup1("A643") ->	{"0","",""};
lookup1("A644") ->	{"0","","A645"};
lookup1("A645") ->	{"0","",""};
lookup1("A646") ->	{"0","","A647"};
lookup1("A647") ->	{"0","",""};
lookup1("A648") ->	{"0","","A649"};
lookup1("A649") ->	{"0","",""};
lookup1("A64A") ->	{"0","","A64B"};
lookup1("A64B") ->	{"0","",""};
lookup1("A64C") ->	{"0","","A64D"};
lookup1("A64D") ->	{"0","",""};
lookup1("A64E") ->	{"0","","A64F"};
lookup1("A64F") ->	{"0","",""};
lookup1("A650") ->	{"0","","A651"};
lookup1("A651") ->	{"0","",""};
lookup1("A652") ->	{"0","","A653"};
lookup1("A653") ->	{"0","",""};
lookup1("A654") ->	{"0","","A655"};
lookup1("A655") ->	{"0","",""};
lookup1("A656") ->	{"0","","A657"};
lookup1("A657") ->	{"0","",""};
lookup1("A658") ->	{"0","","A659"};
lookup1("A659") ->	{"0","",""};
lookup1("A65A") ->	{"0","","A65B"};
lookup1("A65B") ->	{"0","",""};
lookup1("A65C") ->	{"0","","A65D"};
lookup1("A65D") ->	{"0","",""};
lookup1("A65E") ->	{"0","","A65F"};
lookup1("A65F") ->	{"0","",""};
lookup1("A660") ->	{"0","","A661"};
lookup1("A661") ->	{"0","",""};
lookup1("A662") ->	{"0","","A663"};
lookup1("A663") ->	{"0","",""};
lookup1("A664") ->	{"0","","A665"};
lookup1("A665") ->	{"0","",""};
lookup1("A666") ->	{"0","","A667"};
lookup1("A667") ->	{"0","",""};
lookup1("A668") ->	{"0","","A669"};
lookup1("A669") ->	{"0","",""};
lookup1("A66A") ->	{"0","","A66B"};
lookup1("A66B") ->	{"0","",""};
lookup1("A66C") ->	{"0","","A66D"};
lookup1("A66D") ->	{"0","",""};
lookup1("A66E") ->	{"0","",""};
lookup1("A66F") ->	{"230","",""};
lookup1("A670") ->	{"0","",""};
lookup1("A671") ->	{"0","",""};
lookup1("A672") ->	{"0","",""};
lookup1("A673") ->	{"0","",""};
lookup1("A674") ->	{"230","",""};
lookup1("A675") ->	{"230","",""};
lookup1("A676") ->	{"230","",""};
lookup1("A677") ->	{"230","",""};
lookup1("A678") ->	{"230","",""};
lookup1("A679") ->	{"230","",""};
lookup1("A67A") ->	{"230","",""};
lookup1("A67B") ->	{"230","",""};
lookup1("A67C") ->	{"230","",""};
lookup1("A67D") ->	{"230","",""};
lookup1("A67E") ->	{"0","",""};
lookup1("A67F") ->	{"0","",""};
lookup1("A680") ->	{"0","","A681"};
lookup1("A681") ->	{"0","",""};
lookup1("A682") ->	{"0","","A683"};
lookup1("A683") ->	{"0","",""};
lookup1("A684") ->	{"0","","A685"};
lookup1("A685") ->	{"0","",""};
lookup1("A686") ->	{"0","","A687"};
lookup1("A687") ->	{"0","",""};
lookup1("A688") ->	{"0","","A689"};
lookup1("A689") ->	{"0","",""};
lookup1("A68A") ->	{"0","","A68B"};
lookup1("A68B") ->	{"0","",""};
lookup1("A68C") ->	{"0","","A68D"};
lookup1("A68D") ->	{"0","",""};
lookup1("A68E") ->	{"0","","A68F"};
lookup1("A68F") ->	{"0","",""};
lookup1("A690") ->	{"0","","A691"};
lookup1("A691") ->	{"0","",""};
lookup1("A692") ->	{"0","","A693"};
lookup1("A693") ->	{"0","",""};
lookup1("A694") ->	{"0","","A695"};
lookup1("A695") ->	{"0","",""};
lookup1("A696") ->	{"0","","A697"};
lookup1("A697") ->	{"0","",""};
lookup1("A698") ->	{"0","","A699"};
lookup1("A699") ->	{"0","",""};
lookup1("A69A") ->	{"0","","A69B"};
lookup1("A69B") ->	{"0","",""};
lookup1("A69C") ->	{"0","<super> 044A",""};
lookup1("A69D") ->	{"0","<super> 044C",""};
lookup1("A69F") ->	{"230","",""};
lookup1("A6A0") ->	{"0","",""};
lookup1("A6A1") ->	{"0","",""};
lookup1("A6A2") ->	{"0","",""};
lookup1("A6A3") ->	{"0","",""};
lookup1("A6A4") ->	{"0","",""};
lookup1("A6A5") ->	{"0","",""};
lookup1("A6A6") ->	{"0","",""};
lookup1("A6A7") ->	{"0","",""};
lookup1("A6A8") ->	{"0","",""};
lookup1("A6A9") ->	{"0","",""};
lookup1("A6AA") ->	{"0","",""};
lookup1("A6AB") ->	{"0","",""};
lookup1("A6AC") ->	{"0","",""};
lookup1("A6AD") ->	{"0","",""};
lookup1("A6AE") ->	{"0","",""};
lookup1("A6AF") ->	{"0","",""};
lookup1("A6B0") ->	{"0","",""};
lookup1("A6B1") ->	{"0","",""};
lookup1("A6B2") ->	{"0","",""};
lookup1("A6B3") ->	{"0","",""};
lookup1("A6B4") ->	{"0","",""};
lookup1("A6B5") ->	{"0","",""};
lookup1("A6B6") ->	{"0","",""};
lookup1("A6B7") ->	{"0","",""};
lookup1("A6B8") ->	{"0","",""};
lookup1("A6B9") ->	{"0","",""};
lookup1("A6BA") ->	{"0","",""};
lookup1("A6BB") ->	{"0","",""};
lookup1("A6BC") ->	{"0","",""};
lookup1("A6BD") ->	{"0","",""};
lookup1("A6BE") ->	{"0","",""};
lookup1("A6BF") ->	{"0","",""};
lookup1("A6C0") ->	{"0","",""};
lookup1("A6C1") ->	{"0","",""};
lookup1("A6C2") ->	{"0","",""};
lookup1("A6C3") ->	{"0","",""};
lookup1("A6C4") ->	{"0","",""};
lookup1("A6C5") ->	{"0","",""};
lookup1("A6C6") ->	{"0","",""};
lookup1("A6C7") ->	{"0","",""};
lookup1("A6C8") ->	{"0","",""};
lookup1("A6C9") ->	{"0","",""};
lookup1("A6CA") ->	{"0","",""};
lookup1("A6CB") ->	{"0","",""};
lookup1("A6CC") ->	{"0","",""};
lookup1("A6CD") ->	{"0","",""};
lookup1("A6CE") ->	{"0","",""};
lookup1("A6CF") ->	{"0","",""};
lookup1("A6D0") ->	{"0","",""};
lookup1("A6D1") ->	{"0","",""};
lookup1("A6D2") ->	{"0","",""};
lookup1("A6D3") ->	{"0","",""};
lookup1("A6D4") ->	{"0","",""};
lookup1("A6D5") ->	{"0","",""};
lookup1("A6D6") ->	{"0","",""};
lookup1("A6D7") ->	{"0","",""};
lookup1("A6D8") ->	{"0","",""};
lookup1("A6D9") ->	{"0","",""};
lookup1("A6DA") ->	{"0","",""};
lookup1("A6DB") ->	{"0","",""};
lookup1("A6DC") ->	{"0","",""};
lookup1("A6DD") ->	{"0","",""};
lookup1("A6DE") ->	{"0","",""};
lookup1("A6DF") ->	{"0","",""};
lookup1("A6E0") ->	{"0","",""};
lookup1("A6E1") ->	{"0","",""};
lookup1("A6E2") ->	{"0","",""};
lookup1("A6E3") ->	{"0","",""};
lookup1("A6E4") ->	{"0","",""};
lookup1("A6E5") ->	{"0","",""};
lookup1("A6E6") ->	{"0","",""};
lookup1("A6E7") ->	{"0","",""};
lookup1("A6E8") ->	{"0","",""};
lookup1("A6E9") ->	{"0","",""};
lookup1("A6EA") ->	{"0","",""};
lookup1("A6EB") ->	{"0","",""};
lookup1("A6EC") ->	{"0","",""};
lookup1("A6ED") ->	{"0","",""};
lookup1("A6EE") ->	{"0","",""};
lookup1("A6EF") ->	{"0","",""};
lookup1("A6F0") ->	{"230","",""};
lookup1("A6F1") ->	{"230","",""};
lookup1("A6F2") ->	{"0","",""};
lookup1("A6F3") ->	{"0","",""};
lookup1("A6F4") ->	{"0","",""};
lookup1("A6F5") ->	{"0","",""};
lookup1("A6F6") ->	{"0","",""};
lookup1("A6F7") ->	{"0","",""};
lookup1("A700") ->	{"0","",""};
lookup1("A701") ->	{"0","",""};
lookup1("A702") ->	{"0","",""};
lookup1("A703") ->	{"0","",""};
lookup1("A704") ->	{"0","",""};
lookup1("A705") ->	{"0","",""};
lookup1("A706") ->	{"0","",""};
lookup1("A707") ->	{"0","",""};
lookup1("A708") ->	{"0","",""};
lookup1("A709") ->	{"0","",""};
lookup1("A70A") ->	{"0","",""};
lookup1("A70B") ->	{"0","",""};
lookup1("A70C") ->	{"0","",""};
lookup1("A70D") ->	{"0","",""};
lookup1("A70E") ->	{"0","",""};
lookup1("A70F") ->	{"0","",""};
lookup1("A710") ->	{"0","",""};
lookup1("A711") ->	{"0","",""};
lookup1("A712") ->	{"0","",""};
lookup1("A713") ->	{"0","",""};
lookup1("A714") ->	{"0","",""};
lookup1("A715") ->	{"0","",""};
lookup1("A716") ->	{"0","",""};
lookup1("A717") ->	{"0","",""};
lookup1("A718") ->	{"0","",""};
lookup1("A719") ->	{"0","",""};
lookup1("A71A") ->	{"0","",""};
lookup1("A71B") ->	{"0","",""};
lookup1("A71C") ->	{"0","",""};
lookup1("A71D") ->	{"0","",""};
lookup1("A71E") ->	{"0","",""};
lookup1("A71F") ->	{"0","",""};
lookup1("A720") ->	{"0","",""};
lookup1("A721") ->	{"0","",""};
lookup1("A722") ->	{"0","","A723"};
lookup1("A723") ->	{"0","",""};
lookup1("A724") ->	{"0","","A725"};
lookup1("A725") ->	{"0","",""};
lookup1("A726") ->	{"0","","A727"};
lookup1("A727") ->	{"0","",""};
lookup1("A728") ->	{"0","","A729"};
lookup1("A729") ->	{"0","",""};
lookup1("A72A") ->	{"0","","A72B"};
lookup1("A72B") ->	{"0","",""};
lookup1("A72C") ->	{"0","","A72D"};
lookup1("A72D") ->	{"0","",""};
lookup1("A72E") ->	{"0","","A72F"};
lookup1("A72F") ->	{"0","",""};
lookup1("A730") ->	{"0","",""};
lookup1("A731") ->	{"0","",""};
lookup1("A732") ->	{"0","","A733"};
lookup1("A733") ->	{"0","",""};
lookup1("A734") ->	{"0","","A735"};
lookup1("A735") ->	{"0","",""};
lookup1("A736") ->	{"0","","A737"};
lookup1("A737") ->	{"0","",""};
lookup1("A738") ->	{"0","","A739"};
lookup1("A739") ->	{"0","",""};
lookup1("A73A") ->	{"0","","A73B"};
lookup1("A73B") ->	{"0","",""};
lookup1("A73C") ->	{"0","","A73D"};
lookup1("A73D") ->	{"0","",""};
lookup1("A73E") ->	{"0","","A73F"};
lookup1("A73F") ->	{"0","",""};
lookup1("A740") ->	{"0","","A741"};
lookup1("A741") ->	{"0","",""};
lookup1("A742") ->	{"0","","A743"};
lookup1("A743") ->	{"0","",""};
lookup1("A744") ->	{"0","","A745"};
lookup1("A745") ->	{"0","",""};
lookup1("A746") ->	{"0","","A747"};
lookup1("A747") ->	{"0","",""};
lookup1("A748") ->	{"0","","A749"};
lookup1("A749") ->	{"0","",""};
lookup1("A74A") ->	{"0","","A74B"};
lookup1("A74B") ->	{"0","",""};
lookup1("A74C") ->	{"0","","A74D"};
lookup1("A74D") ->	{"0","",""};
lookup1("A74E") ->	{"0","","A74F"};
lookup1("A74F") ->	{"0","",""};
lookup1("A750") ->	{"0","","A751"};
lookup1("A751") ->	{"0","",""};
lookup1("A752") ->	{"0","","A753"};
lookup1("A753") ->	{"0","",""};
lookup1("A754") ->	{"0","","A755"};
lookup1("A755") ->	{"0","",""};
lookup1("A756") ->	{"0","","A757"};
lookup1("A757") ->	{"0","",""};
lookup1("A758") ->	{"0","","A759"};
lookup1("A759") ->	{"0","",""};
lookup1("A75A") ->	{"0","","A75B"};
lookup1("A75B") ->	{"0","",""};
lookup1("A75C") ->	{"0","","A75D"};
lookup1("A75D") ->	{"0","",""};
lookup1("A75E") ->	{"0","","A75F"};
lookup1("A75F") ->	{"0","",""};
lookup1("A760") ->	{"0","","A761"};
lookup1("A761") ->	{"0","",""};
lookup1("A762") ->	{"0","","A763"};
lookup1("A763") ->	{"0","",""};
lookup1("A764") ->	{"0","","A765"};
lookup1("A765") ->	{"0","",""};
lookup1("A766") ->	{"0","","A767"};
lookup1("A767") ->	{"0","",""};
lookup1("A768") ->	{"0","","A769"};
lookup1("A769") ->	{"0","",""};
lookup1("A76A") ->	{"0","","A76B"};
lookup1("A76B") ->	{"0","",""};
lookup1("A76C") ->	{"0","","A76D"};
lookup1("A76D") ->	{"0","",""};
lookup1("A76E") ->	{"0","","A76F"};
lookup1("A76F") ->	{"0","",""};
lookup1("A770") ->	{"0","<super> A76F",""};
lookup1("A771") ->	{"0","",""};
lookup1("A772") ->	{"0","",""};
lookup1("A773") ->	{"0","",""};
lookup1("A774") ->	{"0","",""};
lookup1("A775") ->	{"0","",""};
lookup1("A776") ->	{"0","",""};
lookup1("A777") ->	{"0","",""};
lookup1("A778") ->	{"0","",""};
lookup1("A779") ->	{"0","","A77A"};
lookup1("A77A") ->	{"0","",""};
lookup1("A77B") ->	{"0","","A77C"};
lookup1("A77C") ->	{"0","",""};
lookup1("A77D") ->	{"0","","1D79"};
lookup1("A77E") ->	{"0","","A77F"};
lookup1("A77F") ->	{"0","",""};
lookup1("A780") ->	{"0","","A781"};
lookup1("A781") ->	{"0","",""};
lookup1("A782") ->	{"0","","A783"};
lookup1("A783") ->	{"0","",""};
lookup1("A784") ->	{"0","","A785"};
lookup1("A785") ->	{"0","",""};
lookup1("A786") ->	{"0","","A787"};
lookup1("A787") ->	{"0","",""};
lookup1("A788") ->	{"0","",""};
lookup1("A789") ->	{"0","",""};
lookup1("A78A") ->	{"0","",""};
lookup1("A78B") ->	{"0","","A78C"};
lookup1("A78C") ->	{"0","",""};
lookup1("A78D") ->	{"0","","0265"};
lookup1("A78E") ->	{"0","",""};
lookup1("A790") ->	{"0","","A791"};
lookup1("A791") ->	{"0","",""};
lookup1("A792") ->	{"0","","A793"};
lookup1("A793") ->	{"0","",""};
lookup1("A794") ->	{"0","",""};
lookup1("A795") ->	{"0","",""};
lookup1("A796") ->	{"0","","A797"};
lookup1("A797") ->	{"0","",""};
lookup1("A798") ->	{"0","","A799"};
lookup1("A799") ->	{"0","",""};
lookup1("A79A") ->	{"0","","A79B"};
lookup1("A79B") ->	{"0","",""};
lookup1("A79C") ->	{"0","","A79D"};
lookup1("A79D") ->	{"0","",""};
lookup1("A79E") ->	{"0","","A79F"};
lookup1("A79F") ->	{"0","",""};
lookup1("A7A0") ->	{"0","","A7A1"};
lookup1("A7A1") ->	{"0","",""};
lookup1("A7A2") ->	{"0","","A7A3"};
lookup1("A7A3") ->	{"0","",""};
lookup1("A7A4") ->	{"0","","A7A5"};
lookup1("A7A5") ->	{"0","",""};
lookup1("A7A6") ->	{"0","","A7A7"};
lookup1("A7A7") ->	{"0","",""};
lookup1("A7A8") ->	{"0","","A7A9"};
lookup1("A7A9") ->	{"0","",""};
lookup1("A7AA") ->	{"0","","0266"};
lookup1("A7AB") ->	{"0","","025C"};
lookup1("A7AC") ->	{"0","","0261"};
lookup1("A7AD") ->	{"0","","026C"};
lookup1("A7B0") ->	{"0","","029E"};
lookup1("A7B1") ->	{"0","","0287"};
lookup1("A7F7") ->	{"0","",""};
lookup1("A7F8") ->	{"0","<super> 0126",""};
lookup1("A7F9") ->	{"0","<super> 0153",""};
lookup1("A7FA") ->	{"0","",""};
lookup1("A7FB") ->	{"0","",""};
lookup1("A7FC") ->	{"0","",""};
lookup1("A7FD") ->	{"0","",""};
lookup1("A7FE") ->	{"0","",""};
lookup1("A7FF") ->	{"0","",""};
lookup1("A800") ->	{"0","",""};
lookup1("A801") ->	{"0","",""};
lookup1("A802") ->	{"0","",""};
lookup1("A803") ->	{"0","",""};
lookup1("A804") ->	{"0","",""};
lookup1("A805") ->	{"0","",""};
lookup1("A806") ->	{"9","",""};
lookup1("A807") ->	{"0","",""};
lookup1("A808") ->	{"0","",""};
lookup1("A809") ->	{"0","",""};
lookup1("A80A") ->	{"0","",""};
lookup1("A80B") ->	{"0","",""};
lookup1("A80C") ->	{"0","",""};
lookup1("A80D") ->	{"0","",""};
lookup1("A80E") ->	{"0","",""};
lookup1("A80F") ->	{"0","",""};
lookup1("A810") ->	{"0","",""};
lookup1("A811") ->	{"0","",""};
lookup1("A812") ->	{"0","",""};
lookup1("A813") ->	{"0","",""};
lookup1("A814") ->	{"0","",""};
lookup1("A815") ->	{"0","",""};
lookup1("A816") ->	{"0","",""};
lookup1("A817") ->	{"0","",""};
lookup1("A818") ->	{"0","",""};
lookup1("A819") ->	{"0","",""};
lookup1("A81A") ->	{"0","",""};
lookup1("A81B") ->	{"0","",""};
lookup1("A81C") ->	{"0","",""};
lookup1("A81D") ->	{"0","",""};
lookup1("A81E") ->	{"0","",""};
lookup1("A81F") ->	{"0","",""};
lookup1("A820") ->	{"0","",""};
lookup1("A821") ->	{"0","",""};
lookup1("A822") ->	{"0","",""};
lookup1("A823") ->	{"0","",""};
lookup1("A824") ->	{"0","",""};
lookup1("A825") ->	{"0","",""};
lookup1("A826") ->	{"0","",""};
lookup1("A827") ->	{"0","",""};
lookup1("A828") ->	{"0","",""};
lookup1("A829") ->	{"0","",""};
lookup1("A82A") ->	{"0","",""};
lookup1("A82B") ->	{"0","",""};
lookup1("A830") ->	{"0","",""};
lookup1("A831") ->	{"0","",""};
lookup1("A832") ->	{"0","",""};
lookup1("A833") ->	{"0","",""};
lookup1("A834") ->	{"0","",""};
lookup1("A835") ->	{"0","",""};
lookup1("A836") ->	{"0","",""};
lookup1("A837") ->	{"0","",""};
lookup1("A838") ->	{"0","",""};
lookup1("A839") ->	{"0","",""};
lookup1("A840") ->	{"0","",""};
lookup1("A841") ->	{"0","",""};
lookup1("A842") ->	{"0","",""};
lookup1("A843") ->	{"0","",""};
lookup1("A844") ->	{"0","",""};
lookup1("A845") ->	{"0","",""};
lookup1("A846") ->	{"0","",""};
lookup1("A847") ->	{"0","",""};
lookup1("A848") ->	{"0","",""};
lookup1("A849") ->	{"0","",""};
lookup1("A84A") ->	{"0","",""};
lookup1("A84B") ->	{"0","",""};
lookup1("A84C") ->	{"0","",""};
lookup1("A84D") ->	{"0","",""};
lookup1("A84E") ->	{"0","",""};
lookup1("A84F") ->	{"0","",""};
lookup1("A850") ->	{"0","",""};
lookup1("A851") ->	{"0","",""};
lookup1("A852") ->	{"0","",""};
lookup1("A853") ->	{"0","",""};
lookup1("A854") ->	{"0","",""};
lookup1("A855") ->	{"0","",""};
lookup1("A856") ->	{"0","",""};
lookup1("A857") ->	{"0","",""};
lookup1("A858") ->	{"0","",""};
lookup1("A859") ->	{"0","",""};
lookup1("A85A") ->	{"0","",""};
lookup1("A85B") ->	{"0","",""};
lookup1("A85C") ->	{"0","",""};
lookup1("A85D") ->	{"0","",""};
lookup1("A85E") ->	{"0","",""};
lookup1("A85F") ->	{"0","",""};
lookup1("A860") ->	{"0","",""};
lookup1("A861") ->	{"0","",""};
lookup1("A862") ->	{"0","",""};
lookup1("A863") ->	{"0","",""};
lookup1("A864") ->	{"0","",""};
lookup1("A865") ->	{"0","",""};
lookup1("A866") ->	{"0","",""};
lookup1("A867") ->	{"0","",""};
lookup1("A868") ->	{"0","",""};
lookup1("A869") ->	{"0","",""};
lookup1("A86A") ->	{"0","",""};
lookup1("A86B") ->	{"0","",""};
lookup1("A86C") ->	{"0","",""};
lookup1("A86D") ->	{"0","",""};
lookup1("A86E") ->	{"0","",""};
lookup1("A86F") ->	{"0","",""};
lookup1("A870") ->	{"0","",""};
lookup1("A871") ->	{"0","",""};
lookup1("A872") ->	{"0","",""};
lookup1("A873") ->	{"0","",""};
lookup1("A874") ->	{"0","",""};
lookup1("A875") ->	{"0","",""};
lookup1("A876") ->	{"0","",""};
lookup1("A877") ->	{"0","",""};
lookup1("A880") ->	{"0","",""};
lookup1("A881") ->	{"0","",""};
lookup1("A882") ->	{"0","",""};
lookup1("A883") ->	{"0","",""};
lookup1("A884") ->	{"0","",""};
lookup1("A885") ->	{"0","",""};
lookup1("A886") ->	{"0","",""};
lookup1("A887") ->	{"0","",""};
lookup1("A888") ->	{"0","",""};
lookup1("A889") ->	{"0","",""};
lookup1("A88A") ->	{"0","",""};
lookup1("A88B") ->	{"0","",""};
lookup1("A88C") ->	{"0","",""};
lookup1("A88D") ->	{"0","",""};
lookup1("A88E") ->	{"0","",""};
lookup1("A88F") ->	{"0","",""};
lookup1("A890") ->	{"0","",""};
lookup1("A891") ->	{"0","",""};
lookup1("A892") ->	{"0","",""};
lookup1("A893") ->	{"0","",""};
lookup1("A894") ->	{"0","",""};
lookup1("A895") ->	{"0","",""};
lookup1("A896") ->	{"0","",""};
lookup1("A897") ->	{"0","",""};
lookup1("A898") ->	{"0","",""};
lookup1("A899") ->	{"0","",""};
lookup1("A89A") ->	{"0","",""};
lookup1("A89B") ->	{"0","",""};
lookup1("A89C") ->	{"0","",""};
lookup1("A89D") ->	{"0","",""};
lookup1("A89E") ->	{"0","",""};
lookup1("A89F") ->	{"0","",""};
lookup1("A8A0") ->	{"0","",""};
lookup1("A8A1") ->	{"0","",""};
lookup1("A8A2") ->	{"0","",""};
lookup1("A8A3") ->	{"0","",""};
lookup1("A8A4") ->	{"0","",""};
lookup1("A8A5") ->	{"0","",""};
lookup1("A8A6") ->	{"0","",""};
lookup1("A8A7") ->	{"0","",""};
lookup1("A8A8") ->	{"0","",""};
lookup1("A8A9") ->	{"0","",""};
lookup1("A8AA") ->	{"0","",""};
lookup1("A8AB") ->	{"0","",""};
lookup1("A8AC") ->	{"0","",""};
lookup1("A8AD") ->	{"0","",""};
lookup1("A8AE") ->	{"0","",""};
lookup1("A8AF") ->	{"0","",""};
lookup1("A8B0") ->	{"0","",""};
lookup1("A8B1") ->	{"0","",""};
lookup1("A8B2") ->	{"0","",""};
lookup1("A8B3") ->	{"0","",""};
lookup1("A8B4") ->	{"0","",""};
lookup1("A8B5") ->	{"0","",""};
lookup1("A8B6") ->	{"0","",""};
lookup1("A8B7") ->	{"0","",""};
lookup1("A8B8") ->	{"0","",""};
lookup1("A8B9") ->	{"0","",""};
lookup1("A8BA") ->	{"0","",""};
lookup1("A8BB") ->	{"0","",""};
lookup1("A8BC") ->	{"0","",""};
lookup1("A8BD") ->	{"0","",""};
lookup1("A8BE") ->	{"0","",""};
lookup1("A8BF") ->	{"0","",""};
lookup1("A8C0") ->	{"0","",""};
lookup1("A8C1") ->	{"0","",""};
lookup1("A8C2") ->	{"0","",""};
lookup1("A8C3") ->	{"0","",""};
lookup1("A8C4") ->	{"9","",""};
lookup1("A8CE") ->	{"0","",""};
lookup1("A8CF") ->	{"0","",""};
lookup1("A8D0") ->	{"0","",""};
lookup1("A8D1") ->	{"0","",""};
lookup1("A8D2") ->	{"0","",""};
lookup1("A8D3") ->	{"0","",""};
lookup1("A8D4") ->	{"0","",""};
lookup1("A8D5") ->	{"0","",""};
lookup1("A8D6") ->	{"0","",""};
lookup1("A8D7") ->	{"0","",""};
lookup1("A8D8") ->	{"0","",""};
lookup1("A8D9") ->	{"0","",""};
lookup1("A8E0") ->	{"230","",""};
lookup1("A8E1") ->	{"230","",""};
lookup1("A8E2") ->	{"230","",""};
lookup1("A8E3") ->	{"230","",""};
lookup1("A8E4") ->	{"230","",""};
lookup1("A8E5") ->	{"230","",""};
lookup1("A8E6") ->	{"230","",""};
lookup1("A8E7") ->	{"230","",""};
lookup1("A8E8") ->	{"230","",""};
lookup1("A8E9") ->	{"230","",""};
lookup1("A8EA") ->	{"230","",""};
lookup1("A8EB") ->	{"230","",""};
lookup1("A8EC") ->	{"230","",""};
lookup1("A8ED") ->	{"230","",""};
lookup1("A8EE") ->	{"230","",""};
lookup1("A8EF") ->	{"230","",""};
lookup1("A8F0") ->	{"230","",""};
lookup1("A8F1") ->	{"230","",""};
lookup1("A8F2") ->	{"0","",""};
lookup1("A8F3") ->	{"0","",""};
lookup1("A8F4") ->	{"0","",""};
lookup1("A8F5") ->	{"0","",""};
lookup1("A8F6") ->	{"0","",""};
lookup1("A8F7") ->	{"0","",""};
lookup1("A8F8") ->	{"0","",""};
lookup1("A8F9") ->	{"0","",""};
lookup1("A8FA") ->	{"0","",""};
lookup1("A8FB") ->	{"0","",""};
lookup1("A900") ->	{"0","",""};
lookup1("A901") ->	{"0","",""};
lookup1("A902") ->	{"0","",""};
lookup1("A903") ->	{"0","",""};
lookup1("A904") ->	{"0","",""};
lookup1("A905") ->	{"0","",""};
lookup1("A906") ->	{"0","",""};
lookup1("A907") ->	{"0","",""};
lookup1("A908") ->	{"0","",""};
lookup1("A909") ->	{"0","",""};
lookup1("A90A") ->	{"0","",""};
lookup1("A90B") ->	{"0","",""};
lookup1("A90C") ->	{"0","",""};
lookup1("A90D") ->	{"0","",""};
lookup1("A90E") ->	{"0","",""};
lookup1("A90F") ->	{"0","",""};
lookup1("A910") ->	{"0","",""};
lookup1("A911") ->	{"0","",""};
lookup1("A912") ->	{"0","",""};
lookup1("A913") ->	{"0","",""};
lookup1("A914") ->	{"0","",""};
lookup1("A915") ->	{"0","",""};
lookup1("A916") ->	{"0","",""};
lookup1("A917") ->	{"0","",""};
lookup1("A918") ->	{"0","",""};
lookup1("A919") ->	{"0","",""};
lookup1("A91A") ->	{"0","",""};
lookup1("A91B") ->	{"0","",""};
lookup1("A91C") ->	{"0","",""};
lookup1("A91D") ->	{"0","",""};
lookup1("A91E") ->	{"0","",""};
lookup1("A91F") ->	{"0","",""};
lookup1("A920") ->	{"0","",""};
lookup1("A921") ->	{"0","",""};
lookup1("A922") ->	{"0","",""};
lookup1("A923") ->	{"0","",""};
lookup1("A924") ->	{"0","",""};
lookup1("A925") ->	{"0","",""};
lookup1("A926") ->	{"0","",""};
lookup1("A927") ->	{"0","",""};
lookup1("A928") ->	{"0","",""};
lookup1("A929") ->	{"0","",""};
lookup1("A92A") ->	{"0","",""};
lookup1("A92B") ->	{"220","",""};
lookup1("A92C") ->	{"220","",""};
lookup1("A92D") ->	{"220","",""};
lookup1("A92E") ->	{"0","",""};
lookup1("A92F") ->	{"0","",""};
lookup1("A930") ->	{"0","",""};
lookup1("A931") ->	{"0","",""};
lookup1("A932") ->	{"0","",""};
lookup1("A933") ->	{"0","",""};
lookup1("A934") ->	{"0","",""};
lookup1("A935") ->	{"0","",""};
lookup1("A936") ->	{"0","",""};
lookup1("A937") ->	{"0","",""};
lookup1("A938") ->	{"0","",""};
lookup1("A939") ->	{"0","",""};
lookup1("A93A") ->	{"0","",""};
lookup1("A93B") ->	{"0","",""};
lookup1("A93C") ->	{"0","",""};
lookup1("A93D") ->	{"0","",""};
lookup1("A93E") ->	{"0","",""};
lookup1("A93F") ->	{"0","",""};
lookup1("A940") ->	{"0","",""};
lookup1("A941") ->	{"0","",""};
lookup1("A942") ->	{"0","",""};
lookup1("A943") ->	{"0","",""};
lookup1("A944") ->	{"0","",""};
lookup1("A945") ->	{"0","",""};
lookup1("A946") ->	{"0","",""};
lookup1("A947") ->	{"0","",""};
lookup1("A948") ->	{"0","",""};
lookup1("A949") ->	{"0","",""};
lookup1("A94A") ->	{"0","",""};
lookup1("A94B") ->	{"0","",""};
lookup1("A94C") ->	{"0","",""};
lookup1("A94D") ->	{"0","",""};
lookup1("A94E") ->	{"0","",""};
lookup1("A94F") ->	{"0","",""};
lookup1("A950") ->	{"0","",""};
lookup1("A951") ->	{"0","",""};
lookup1("A952") ->	{"0","",""};
lookup1("A953") ->	{"9","",""};
lookup1("A95F") ->	{"0","",""};
lookup1("A960") ->	{"0","",""};
lookup1("A961") ->	{"0","",""};
lookup1("A962") ->	{"0","",""};
lookup1("A963") ->	{"0","",""};
lookup1("A964") ->	{"0","",""};
lookup1("A965") ->	{"0","",""};
lookup1("A966") ->	{"0","",""};
lookup1("A967") ->	{"0","",""};
lookup1("A968") ->	{"0","",""};
lookup1("A969") ->	{"0","",""};
lookup1("A96A") ->	{"0","",""};
lookup1("A96B") ->	{"0","",""};
lookup1("A96C") ->	{"0","",""};
lookup1("A96D") ->	{"0","",""};
lookup1("A96E") ->	{"0","",""};
lookup1("A96F") ->	{"0","",""};
lookup1("A970") ->	{"0","",""};
lookup1("A971") ->	{"0","",""};
lookup1("A972") ->	{"0","",""};
lookup1("A973") ->	{"0","",""};
lookup1("A974") ->	{"0","",""};
lookup1("A975") ->	{"0","",""};
lookup1("A976") ->	{"0","",""};
lookup1("A977") ->	{"0","",""};
lookup1("A978") ->	{"0","",""};
lookup1("A979") ->	{"0","",""};
lookup1("A97A") ->	{"0","",""};
lookup1("A97B") ->	{"0","",""};
lookup1("A97C") ->	{"0","",""};
lookup1("A980") ->	{"0","",""};
lookup1("A981") ->	{"0","",""};
lookup1("A982") ->	{"0","",""};
lookup1("A983") ->	{"0","",""};
lookup1("A984") ->	{"0","",""};
lookup1("A985") ->	{"0","",""};
lookup1("A986") ->	{"0","",""};
lookup1("A987") ->	{"0","",""};
lookup1("A988") ->	{"0","",""};
lookup1("A989") ->	{"0","",""};
lookup1("A98A") ->	{"0","",""};
lookup1("A98B") ->	{"0","",""};
lookup1("A98C") ->	{"0","",""};
lookup1("A98D") ->	{"0","",""};
lookup1("A98E") ->	{"0","",""};
lookup1("A98F") ->	{"0","",""};
lookup1("A990") ->	{"0","",""};
lookup1("A991") ->	{"0","",""};
lookup1("A992") ->	{"0","",""};
lookup1("A993") ->	{"0","",""};
lookup1("A994") ->	{"0","",""};
lookup1("A995") ->	{"0","",""};
lookup1("A996") ->	{"0","",""};
lookup1("A997") ->	{"0","",""};
lookup1("A998") ->	{"0","",""};
lookup1("A999") ->	{"0","",""};
lookup1("A99A") ->	{"0","",""};
lookup1("A99B") ->	{"0","",""};
lookup1("A99C") ->	{"0","",""};
lookup1("A99D") ->	{"0","",""};
lookup1("A99E") ->	{"0","",""};
lookup1("A99F") ->	{"0","",""};
lookup1("A9A0") ->	{"0","",""};
lookup1("A9A1") ->	{"0","",""};
lookup1("A9A2") ->	{"0","",""};
lookup1("A9A3") ->	{"0","",""};
lookup1("A9A4") ->	{"0","",""};
lookup1("A9A5") ->	{"0","",""};
lookup1("A9A6") ->	{"0","",""};
lookup1("A9A7") ->	{"0","",""};
lookup1("A9A8") ->	{"0","",""};
lookup1("A9A9") ->	{"0","",""};
lookup1("A9AA") ->	{"0","",""};
lookup1("A9AB") ->	{"0","",""};
lookup1("A9AC") ->	{"0","",""};
lookup1("A9AD") ->	{"0","",""};
lookup1("A9AE") ->	{"0","",""};
lookup1("A9AF") ->	{"0","",""};
lookup1("A9B0") ->	{"0","",""};
lookup1("A9B1") ->	{"0","",""};
lookup1("A9B2") ->	{"0","",""};
lookup1("A9B3") ->	{"7","",""};
lookup1("A9B4") ->	{"0","",""};
lookup1("A9B5") ->	{"0","",""};
lookup1("A9B6") ->	{"0","",""};
lookup1("A9B7") ->	{"0","",""};
lookup1("A9B8") ->	{"0","",""};
lookup1("A9B9") ->	{"0","",""};
lookup1("A9BA") ->	{"0","",""};
lookup1("A9BB") ->	{"0","",""};
lookup1("A9BC") ->	{"0","",""};
lookup1("A9BD") ->	{"0","",""};
lookup1("A9BE") ->	{"0","",""};
lookup1("A9BF") ->	{"0","",""};
lookup1("A9C0") ->	{"9","",""};
lookup1("A9C1") ->	{"0","",""};
lookup1("A9C2") ->	{"0","",""};
lookup1("A9C3") ->	{"0","",""};
lookup1("A9C4") ->	{"0","",""};
lookup1("A9C5") ->	{"0","",""};
lookup1("A9C6") ->	{"0","",""};
lookup1("A9C7") ->	{"0","",""};
lookup1("A9C8") ->	{"0","",""};
lookup1("A9C9") ->	{"0","",""};
lookup1("A9CA") ->	{"0","",""};
lookup1("A9CB") ->	{"0","",""};
lookup1("A9CC") ->	{"0","",""};
lookup1("A9CD") ->	{"0","",""};
lookup1("A9CF") ->	{"0","",""};
lookup1("A9D0") ->	{"0","",""};
lookup1("A9D1") ->	{"0","",""};
lookup1("A9D2") ->	{"0","",""};
lookup1("A9D3") ->	{"0","",""};
lookup1("A9D4") ->	{"0","",""};
lookup1("A9D5") ->	{"0","",""};
lookup1("A9D6") ->	{"0","",""};
lookup1("A9D7") ->	{"0","",""};
lookup1("A9D8") ->	{"0","",""};
lookup1("A9D9") ->	{"0","",""};
lookup1("A9DE") ->	{"0","",""};
lookup1("A9DF") ->	{"0","",""};
lookup1("A9E0") ->	{"0","",""};
lookup1("A9E1") ->	{"0","",""};
lookup1("A9E2") ->	{"0","",""};
lookup1("A9E3") ->	{"0","",""};
lookup1("A9E4") ->	{"0","",""};
lookup1("A9E5") ->	{"0","",""};
lookup1("A9E6") ->	{"0","",""};
lookup1("A9E7") ->	{"0","",""};
lookup1("A9E8") ->	{"0","",""};
lookup1("A9E9") ->	{"0","",""};
lookup1("A9EA") ->	{"0","",""};
lookup1("A9EB") ->	{"0","",""};
lookup1("A9EC") ->	{"0","",""};
lookup1("A9ED") ->	{"0","",""};
lookup1("A9EE") ->	{"0","",""};
lookup1("A9EF") ->	{"0","",""};
lookup1("A9F0") ->	{"0","",""};
lookup1("A9F1") ->	{"0","",""};
lookup1("A9F2") ->	{"0","",""};
lookup1("A9F3") ->	{"0","",""};
lookup1("A9F4") ->	{"0","",""};
lookup1("A9F5") ->	{"0","",""};
lookup1("A9F6") ->	{"0","",""};
lookup1("A9F7") ->	{"0","",""};
lookup1("A9F8") ->	{"0","",""};
lookup1("A9F9") ->	{"0","",""};
lookup1("A9FA") ->	{"0","",""};
lookup1("A9FB") ->	{"0","",""};
lookup1("A9FC") ->	{"0","",""};
lookup1("A9FD") ->	{"0","",""};
lookup1("A9FE") ->	{"0","",""};
lookup1("AA00") ->	{"0","",""};
lookup1("AA01") ->	{"0","",""};
lookup1("AA02") ->	{"0","",""};
lookup1("AA03") ->	{"0","",""};
lookup1("AA04") ->	{"0","",""};
lookup1("AA05") ->	{"0","",""};
lookup1("AA06") ->	{"0","",""};
lookup1("AA07") ->	{"0","",""};
lookup1("AA08") ->	{"0","",""};
lookup1("AA09") ->	{"0","",""};
lookup1("AA0A") ->	{"0","",""};
lookup1("AA0B") ->	{"0","",""};
lookup1("AA0C") ->	{"0","",""};
lookup1("AA0D") ->	{"0","",""};
lookup1("AA0E") ->	{"0","",""};
lookup1("AA0F") ->	{"0","",""};
lookup1("AA10") ->	{"0","",""};
lookup1("AA11") ->	{"0","",""};
lookup1("AA12") ->	{"0","",""};
lookup1("AA13") ->	{"0","",""};
lookup1("AA14") ->	{"0","",""};
lookup1("AA15") ->	{"0","",""};
lookup1("AA16") ->	{"0","",""};
lookup1("AA17") ->	{"0","",""};
lookup1("AA18") ->	{"0","",""};
lookup1("AA19") ->	{"0","",""};
lookup1("AA1A") ->	{"0","",""};
lookup1("AA1B") ->	{"0","",""};
lookup1("AA1C") ->	{"0","",""};
lookup1("AA1D") ->	{"0","",""};
lookup1("AA1E") ->	{"0","",""};
lookup1("AA1F") ->	{"0","",""};
lookup1("AA20") ->	{"0","",""};
lookup1("AA21") ->	{"0","",""};
lookup1("AA22") ->	{"0","",""};
lookup1("AA23") ->	{"0","",""};
lookup1("AA24") ->	{"0","",""};
lookup1("AA25") ->	{"0","",""};
lookup1("AA26") ->	{"0","",""};
lookup1("AA27") ->	{"0","",""};
lookup1("AA28") ->	{"0","",""};
lookup1("AA29") ->	{"0","",""};
lookup1("AA2A") ->	{"0","",""};
lookup1("AA2B") ->	{"0","",""};
lookup1("AA2C") ->	{"0","",""};
lookup1("AA2D") ->	{"0","",""};
lookup1("AA2E") ->	{"0","",""};
lookup1("AA2F") ->	{"0","",""};
lookup1("AA30") ->	{"0","",""};
lookup1("AA31") ->	{"0","",""};
lookup1("AA32") ->	{"0","",""};
lookup1("AA33") ->	{"0","",""};
lookup1("AA34") ->	{"0","",""};
lookup1("AA35") ->	{"0","",""};
lookup1("AA36") ->	{"0","",""};
lookup1("AA40") ->	{"0","",""};
lookup1("AA41") ->	{"0","",""};
lookup1("AA42") ->	{"0","",""};
lookup1("AA43") ->	{"0","",""};
lookup1("AA44") ->	{"0","",""};
lookup1("AA45") ->	{"0","",""};
lookup1("AA46") ->	{"0","",""};
lookup1("AA47") ->	{"0","",""};
lookup1("AA48") ->	{"0","",""};
lookup1("AA49") ->	{"0","",""};
lookup1("AA4A") ->	{"0","",""};
lookup1("AA4B") ->	{"0","",""};
lookup1("AA4C") ->	{"0","",""};
lookup1("AA4D") ->	{"0","",""};
lookup1("AA50") ->	{"0","",""};
lookup1("AA51") ->	{"0","",""};
lookup1("AA52") ->	{"0","",""};
lookup1("AA53") ->	{"0","",""};
lookup1("AA54") ->	{"0","",""};
lookup1("AA55") ->	{"0","",""};
lookup1("AA56") ->	{"0","",""};
lookup1("AA57") ->	{"0","",""};
lookup1("AA58") ->	{"0","",""};
lookup1("AA59") ->	{"0","",""};
lookup1("AA5C") ->	{"0","",""};
lookup1("AA5D") ->	{"0","",""};
lookup1("AA5E") ->	{"0","",""};
lookup1("AA5F") ->	{"0","",""};
lookup1("AA60") ->	{"0","",""};
lookup1("AA61") ->	{"0","",""};
lookup1("AA62") ->	{"0","",""};
lookup1("AA63") ->	{"0","",""};
lookup1("AA64") ->	{"0","",""};
lookup1("AA65") ->	{"0","",""};
lookup1("AA66") ->	{"0","",""};
lookup1("AA67") ->	{"0","",""};
lookup1("AA68") ->	{"0","",""};
lookup1("AA69") ->	{"0","",""};
lookup1("AA6A") ->	{"0","",""};
lookup1("AA6B") ->	{"0","",""};
lookup1("AA6C") ->	{"0","",""};
lookup1("AA6D") ->	{"0","",""};
lookup1("AA6E") ->	{"0","",""};
lookup1("AA6F") ->	{"0","",""};
lookup1("AA70") ->	{"0","",""};
lookup1("AA71") ->	{"0","",""};
lookup1("AA72") ->	{"0","",""};
lookup1("AA73") ->	{"0","",""};
lookup1("AA74") ->	{"0","",""};
lookup1("AA75") ->	{"0","",""};
lookup1("AA76") ->	{"0","",""};
lookup1("AA77") ->	{"0","",""};
lookup1("AA78") ->	{"0","",""};
lookup1("AA79") ->	{"0","",""};
lookup1("AA7A") ->	{"0","",""};
lookup1("AA7B") ->	{"0","",""};
lookup1("AA7C") ->	{"0","",""};
lookup1("AA7D") ->	{"0","",""};
lookup1("AA7E") ->	{"0","",""};
lookup1("AA7F") ->	{"0","",""};
lookup1("AA80") ->	{"0","",""};
lookup1("AA81") ->	{"0","",""};
lookup1("AA82") ->	{"0","",""};
lookup1("AA83") ->	{"0","",""};
lookup1("AA84") ->	{"0","",""};
lookup1("AA85") ->	{"0","",""};
lookup1("AA86") ->	{"0","",""};
lookup1("AA87") ->	{"0","",""};
lookup1("AA88") ->	{"0","",""};
lookup1("AA89") ->	{"0","",""};
lookup1("AA8A") ->	{"0","",""};
lookup1("AA8B") ->	{"0","",""};
lookup1("AA8C") ->	{"0","",""};
lookup1("AA8D") ->	{"0","",""};
lookup1("AA8E") ->	{"0","",""};
lookup1("AA8F") ->	{"0","",""};
lookup1("AA90") ->	{"0","",""};
lookup1("AA91") ->	{"0","",""};
lookup1("AA92") ->	{"0","",""};
lookup1("AA93") ->	{"0","",""};
lookup1("AA94") ->	{"0","",""};
lookup1("AA95") ->	{"0","",""};
lookup1("AA96") ->	{"0","",""};
lookup1("AA97") ->	{"0","",""};
lookup1("AA98") ->	{"0","",""};
lookup1("AA99") ->	{"0","",""};
lookup1("AA9A") ->	{"0","",""};
lookup1("AA9B") ->	{"0","",""};
lookup1("AA9C") ->	{"0","",""};
lookup1("AA9D") ->	{"0","",""};
lookup1("AA9E") ->	{"0","",""};
lookup1("AA9F") ->	{"0","",""};
lookup1("AAA0") ->	{"0","",""};
lookup1("AAA1") ->	{"0","",""};
lookup1("AAA2") ->	{"0","",""};
lookup1("AAA3") ->	{"0","",""};
lookup1("AAA4") ->	{"0","",""};
lookup1("AAA5") ->	{"0","",""};
lookup1("AAA6") ->	{"0","",""};
lookup1("AAA7") ->	{"0","",""};
lookup1("AAA8") ->	{"0","",""};
lookup1("AAA9") ->	{"0","",""};
lookup1("AAAA") ->	{"0","",""};
lookup1("AAAB") ->	{"0","",""};
lookup1("AAAC") ->	{"0","",""};
lookup1("AAAD") ->	{"0","",""};
lookup1("AAAE") ->	{"0","",""};
lookup1("AAAF") ->	{"0","",""};
lookup1("AAB0") ->	{"230","",""};
lookup1("AAB1") ->	{"0","",""};
lookup1("AAB2") ->	{"230","",""};
lookup1("AAB3") ->	{"230","",""};
lookup1("AAB4") ->	{"220","",""};
lookup1("AAB5") ->	{"0","",""};
lookup1("AAB6") ->	{"0","",""};
lookup1("AAB7") ->	{"230","",""};
lookup1("AAB8") ->	{"230","",""};
lookup1("AAB9") ->	{"0","",""};
lookup1("AABA") ->	{"0","",""};
lookup1("AABB") ->	{"0","",""};
lookup1("AABC") ->	{"0","",""};
lookup1("AABD") ->	{"0","",""};
lookup1("AABE") ->	{"230","",""};
lookup1("AABF") ->	{"230","",""};
lookup1("AAC0") ->	{"0","",""};
lookup1("AAC1") ->	{"230","",""};
lookup1("AAC2") ->	{"0","",""};
lookup1("AADB") ->	{"0","",""};
lookup1("AADC") ->	{"0","",""};
lookup1("AADD") ->	{"0","",""};
lookup1("AADE") ->	{"0","",""};
lookup1("AADF") ->	{"0","",""};
lookup1("AAE0") ->	{"0","",""};
lookup1("AAE1") ->	{"0","",""};
lookup1("AAE2") ->	{"0","",""};
lookup1("AAE3") ->	{"0","",""};
lookup1("AAE4") ->	{"0","",""};
lookup1("AAE5") ->	{"0","",""};
lookup1("AAE6") ->	{"0","",""};
lookup1("AAE7") ->	{"0","",""};
lookup1("AAE8") ->	{"0","",""};
lookup1("AAE9") ->	{"0","",""};
lookup1("AAEA") ->	{"0","",""};
lookup1("AAEB") ->	{"0","",""};
lookup1("AAEC") ->	{"0","",""};
lookup1("AAED") ->	{"0","",""};
lookup1("AAEE") ->	{"0","",""};
lookup1("AAEF") ->	{"0","",""};
lookup1("AAF0") ->	{"0","",""};
lookup1("AAF1") ->	{"0","",""};
lookup1("AAF2") ->	{"0","",""};
lookup1("AAF3") ->	{"0","",""};
lookup1("AAF4") ->	{"0","",""};
lookup1("AAF5") ->	{"0","",""};
lookup1("AAF6") ->	{"9","",""};
lookup1("AB01") ->	{"0","",""};
lookup1("AB02") ->	{"0","",""};
lookup1("AB03") ->	{"0","",""};
lookup1("AB04") ->	{"0","",""};
lookup1("AB05") ->	{"0","",""};
lookup1("AB06") ->	{"0","",""};
lookup1("AB09") ->	{"0","",""};
lookup1("AB0A") ->	{"0","",""};
lookup1("AB0B") ->	{"0","",""};
lookup1("AB0C") ->	{"0","",""};
lookup1("AB0D") ->	{"0","",""};
lookup1("AB0E") ->	{"0","",""};
lookup1("AB11") ->	{"0","",""};
lookup1("AB12") ->	{"0","",""};
lookup1("AB13") ->	{"0","",""};
lookup1("AB14") ->	{"0","",""};
lookup1("AB15") ->	{"0","",""};
lookup1("AB16") ->	{"0","",""};
lookup1("AB20") ->	{"0","",""};
lookup1("AB21") ->	{"0","",""};
lookup1("AB22") ->	{"0","",""};
lookup1("AB23") ->	{"0","",""};
lookup1("AB24") ->	{"0","",""};
lookup1("AB25") ->	{"0","",""};
lookup1("AB26") ->	{"0","",""};
lookup1("AB28") ->	{"0","",""};
lookup1("AB29") ->	{"0","",""};
lookup1("AB2A") ->	{"0","",""};
lookup1("AB2B") ->	{"0","",""};
lookup1("AB2C") ->	{"0","",""};
lookup1("AB2D") ->	{"0","",""};
lookup1("AB2E") ->	{"0","",""};
lookup1("AB30") ->	{"0","",""};
lookup1("AB31") ->	{"0","",""};
lookup1("AB32") ->	{"0","",""};
lookup1("AB33") ->	{"0","",""};
lookup1("AB34") ->	{"0","",""};
lookup1("AB35") ->	{"0","",""};
lookup1("AB36") ->	{"0","",""};
lookup1("AB37") ->	{"0","",""};
lookup1("AB38") ->	{"0","",""};
lookup1("AB39") ->	{"0","",""};
lookup1("AB3A") ->	{"0","",""};
lookup1("AB3B") ->	{"0","",""};
lookup1("AB3C") ->	{"0","",""};
lookup1("AB3D") ->	{"0","",""};
lookup1("AB3E") ->	{"0","",""};
lookup1("AB3F") ->	{"0","",""};
lookup1("AB40") ->	{"0","",""};
lookup1("AB41") ->	{"0","",""};
lookup1("AB42") ->	{"0","",""};
lookup1("AB43") ->	{"0","",""};
lookup1("AB44") ->	{"0","",""};
lookup1("AB45") ->	{"0","",""};
lookup1("AB46") ->	{"0","",""};
lookup1("AB47") ->	{"0","",""};
lookup1("AB48") ->	{"0","",""};
lookup1("AB49") ->	{"0","",""};
lookup1("AB4A") ->	{"0","",""};
lookup1("AB4B") ->	{"0","",""};
lookup1("AB4C") ->	{"0","",""};
lookup1("AB4D") ->	{"0","",""};
lookup1("AB4E") ->	{"0","",""};
lookup1("AB4F") ->	{"0","",""};
lookup1("AB50") ->	{"0","",""};
lookup1("AB51") ->	{"0","",""};
lookup1("AB52") ->	{"0","",""};
lookup1("AB53") ->	{"0","",""};
lookup1("AB54") ->	{"0","",""};
lookup1("AB55") ->	{"0","",""};
lookup1("AB56") ->	{"0","",""};
lookup1("AB57") ->	{"0","",""};
lookup1("AB58") ->	{"0","",""};
lookup1("AB59") ->	{"0","",""};
lookup1("AB5A") ->	{"0","",""};
lookup1("AB5B") ->	{"0","",""};
lookup1("AB5C") ->	{"0","<super> A727",""};
lookup1("AB5D") ->	{"0","<super> AB37",""};
lookup1("AB5E") ->	{"0","<super> 026B",""};
lookup1("AB5F") ->	{"0","<super> AB52",""};
lookup1("AB64") ->	{"0","",""};
lookup1("AB65") ->	{"0","",""};
lookup1("ABC0") ->	{"0","",""};
lookup1("ABC1") ->	{"0","",""};
lookup1("ABC2") ->	{"0","",""};
lookup1("ABC3") ->	{"0","",""};
lookup1("ABC4") ->	{"0","",""};
lookup1("ABC5") ->	{"0","",""};
lookup1("ABC6") ->	{"0","",""};
lookup1("ABC7") ->	{"0","",""};
lookup1("ABC8") ->	{"0","",""};
lookup1("ABC9") ->	{"0","",""};
lookup1("ABCA") ->	{"0","",""};
lookup1("ABCB") ->	{"0","",""};
lookup1("ABCC") ->	{"0","",""};
lookup1("ABCD") ->	{"0","",""};
lookup1("ABCE") ->	{"0","",""};
lookup1("ABCF") ->	{"0","",""};
lookup1("ABD0") ->	{"0","",""};
lookup1("ABD1") ->	{"0","",""};
lookup1("ABD2") ->	{"0","",""};
lookup1("ABD3") ->	{"0","",""};
lookup1("ABD4") ->	{"0","",""};
lookup1("ABD5") ->	{"0","",""};
lookup1("ABD6") ->	{"0","",""};
lookup1("ABD7") ->	{"0","",""};
lookup1("ABD8") ->	{"0","",""};
lookup1("ABD9") ->	{"0","",""};
lookup1("ABDA") ->	{"0","",""};
lookup1("ABDB") ->	{"0","",""};
lookup1("ABDC") ->	{"0","",""};
lookup1("ABDD") ->	{"0","",""};
lookup1("ABDE") ->	{"0","",""};
lookup1("ABDF") ->	{"0","",""};
lookup1("ABE0") ->	{"0","",""};
lookup1("ABE1") ->	{"0","",""};
lookup1("ABE2") ->	{"0","",""};
lookup1("ABE3") ->	{"0","",""};
lookup1("ABE4") ->	{"0","",""};
lookup1("ABE5") ->	{"0","",""};
lookup1("ABE6") ->	{"0","",""};
lookup1("ABE7") ->	{"0","",""};
lookup1("ABE8") ->	{"0","",""};
lookup1("ABE9") ->	{"0","",""};
lookup1("ABEA") ->	{"0","",""};
lookup1("ABEB") ->	{"0","",""};
lookup1("ABEC") ->	{"0","",""};
lookup1("ABED") ->	{"9","",""};
lookup1("ABF0") ->	{"0","",""};
lookup1("ABF1") ->	{"0","",""};
lookup1("ABF2") ->	{"0","",""};
lookup1("ABF3") ->	{"0","",""};
lookup1("ABF4") ->	{"0","",""};
lookup1("ABF5") ->	{"0","",""};
lookup1("ABF6") ->	{"0","",""};
lookup1("ABF7") ->	{"0","",""};
lookup1("ABF8") ->	{"0","",""};
lookup1("ABF9") ->	{"0","",""};
lookup1("AC00") ->	{"0","",""};
lookup1("D7A3") ->	{"0","",""};
lookup1("D7B0") ->	{"0","",""};
lookup1("D7B1") ->	{"0","",""};
lookup1("D7B2") ->	{"0","",""};
lookup1("D7B3") ->	{"0","",""};
lookup1("D7B4") ->	{"0","",""};
lookup1("D7B5") ->	{"0","",""};
lookup1("D7B6") ->	{"0","",""};
lookup1("D7B7") ->	{"0","",""};
lookup1("D7B8") ->	{"0","",""};
lookup1("D7B9") ->	{"0","",""};
lookup1("D7BA") ->	{"0","",""};
lookup1("D7BB") ->	{"0","",""};
lookup1("D7BC") ->	{"0","",""};
lookup1("D7BD") ->	{"0","",""};
lookup1("D7BE") ->	{"0","",""};
lookup1("D7BF") ->	{"0","",""};
lookup1("D7C0") ->	{"0","",""};
lookup1("D7C1") ->	{"0","",""};
lookup1("D7C2") ->	{"0","",""};
lookup1("D7C3") ->	{"0","",""};
lookup1("D7C4") ->	{"0","",""};
lookup1("D7C5") ->	{"0","",""};
lookup1("D7C6") ->	{"0","",""};
lookup1("D7CB") ->	{"0","",""};
lookup1("D7CC") ->	{"0","",""};
lookup1("D7CD") ->	{"0","",""};
lookup1("D7CE") ->	{"0","",""};
lookup1("D7CF") ->	{"0","",""};
lookup1("D7D0") ->	{"0","",""};
lookup1("D7D1") ->	{"0","",""};
lookup1("D7D2") ->	{"0","",""};
lookup1("D7D3") ->	{"0","",""};
lookup1("D7D4") ->	{"0","",""};
lookup1("D7D5") ->	{"0","",""};
lookup1("D7D6") ->	{"0","",""};
lookup1("D7D7") ->	{"0","",""};
lookup1("D7D8") ->	{"0","",""};
lookup1("D7D9") ->	{"0","",""};
lookup1("D7DA") ->	{"0","",""};
lookup1("D7DB") ->	{"0","",""};
lookup1("D7DC") ->	{"0","",""};
lookup1("D7DD") ->	{"0","",""};
lookup1("D7DE") ->	{"0","",""};
lookup1("D7DF") ->	{"0","",""};
lookup1("D7E0") ->	{"0","",""};
lookup1("D7E1") ->	{"0","",""};
lookup1("D7E2") ->	{"0","",""};
lookup1("D7E3") ->	{"0","",""};
lookup1("D7E4") ->	{"0","",""};
lookup1("D7E5") ->	{"0","",""};
lookup1("D7E6") ->	{"0","",""};
lookup1("D7E7") ->	{"0","",""};
lookup1("D7E8") ->	{"0","",""};
lookup1("D7E9") ->	{"0","",""};
lookup1("D7EA") ->	{"0","",""};
lookup1("D7EB") ->	{"0","",""};
lookup1("D7EC") ->	{"0","",""};
lookup1("D7ED") ->	{"0","",""};
lookup1("D7EE") ->	{"0","",""};
lookup1("D7EF") ->	{"0","",""};
lookup1("D7F0") ->	{"0","",""};
lookup1("D7F1") ->	{"0","",""};
lookup1("D7F2") ->	{"0","",""};
lookup1("D7F3") ->	{"0","",""};
lookup1("D7F4") ->	{"0","",""};
lookup1("D7F5") ->	{"0","",""};
lookup1("D7F6") ->	{"0","",""};
lookup1("D7F7") ->	{"0","",""};
lookup1("D7F8") ->	{"0","",""};
lookup1("D7F9") ->	{"0","",""};
lookup1("D7FA") ->	{"0","",""};
lookup1("D7FB") ->	{"0","",""};
lookup1("D800") ->	{"0","",""};
lookup1("DB7F") ->	{"0","",""};
lookup1("DB80") ->	{"0","",""};
lookup1("DBFF") ->	{"0","",""};
lookup1("DC00") ->	{"0","",""};
lookup1("DFFF") ->	{"0","",""};
lookup1("E000") ->	{"0","",""};
lookup1("E0001") ->	{"0","",""};
lookup1("E0020") ->	{"0","",""};
lookup1("E0021") ->	{"0","",""};
lookup1("E0022") ->	{"0","",""};
lookup1("E0023") ->	{"0","",""};
lookup1("E0024") ->	{"0","",""};
lookup1("E0025") ->	{"0","",""};
lookup1("E0026") ->	{"0","",""};
lookup1("E0027") ->	{"0","",""};
lookup1("E0028") ->	{"0","",""};
lookup1("E0029") ->	{"0","",""};
lookup1("E002A") ->	{"0","",""};
lookup1("E002B") ->	{"0","",""};
lookup1("E002C") ->	{"0","",""};
lookup1("E002D") ->	{"0","",""};
lookup1("E002E") ->	{"0","",""};
lookup1("E002F") ->	{"0","",""};
lookup1("E0030") ->	{"0","",""};
lookup1("E0031") ->	{"0","",""};
lookup1("E0032") ->	{"0","",""};
lookup1("E0033") ->	{"0","",""};
lookup1("E0034") ->	{"0","",""};
lookup1("E0035") ->	{"0","",""};
lookup1("E0036") ->	{"0","",""};
lookup1("E0037") ->	{"0","",""};
lookup1("E0038") ->	{"0","",""};
lookup1("E0039") ->	{"0","",""};
lookup1("E003A") ->	{"0","",""};
lookup1("E003B") ->	{"0","",""};
lookup1("E003C") ->	{"0","",""};
lookup1("E003D") ->	{"0","",""};
lookup1("E003E") ->	{"0","",""};
lookup1("E003F") ->	{"0","",""};
lookup1("E0040") ->	{"0","",""};
lookup1("E0041") ->	{"0","",""};
lookup1("E0042") ->	{"0","",""};
lookup1("E0043") ->	{"0","",""};
lookup1("E0044") ->	{"0","",""};
lookup1("E0045") ->	{"0","",""};
lookup1("E0046") ->	{"0","",""};
lookup1("E0047") ->	{"0","",""};
lookup1("E0048") ->	{"0","",""};
lookup1("E0049") ->	{"0","",""};
lookup1("E004A") ->	{"0","",""};
lookup1("E004B") ->	{"0","",""};
lookup1("E004C") ->	{"0","",""};
lookup1("E004D") ->	{"0","",""};
lookup1("E004E") ->	{"0","",""};
lookup1("E004F") ->	{"0","",""};
lookup1("E0050") ->	{"0","",""};
lookup1("E0051") ->	{"0","",""};
lookup1("E0052") ->	{"0","",""};
lookup1("E0053") ->	{"0","",""};
lookup1("E0054") ->	{"0","",""};
lookup1("E0055") ->	{"0","",""};
lookup1("E0056") ->	{"0","",""};
lookup1("E0057") ->	{"0","",""};
lookup1("E0058") ->	{"0","",""};
lookup1("E0059") ->	{"0","",""};
lookup1("E005A") ->	{"0","",""};
lookup1("E005B") ->	{"0","",""};
lookup1("E005C") ->	{"0","",""};
lookup1("E005D") ->	{"0","",""};
lookup1("E005E") ->	{"0","",""};
lookup1("E005F") ->	{"0","",""};
lookup1("E0060") ->	{"0","",""};
lookup1("E0061") ->	{"0","",""};
lookup1("E0062") ->	{"0","",""};
lookup1("E0063") ->	{"0","",""};
lookup1("E0064") ->	{"0","",""};
lookup1("E0065") ->	{"0","",""};
lookup1("E0066") ->	{"0","",""};
lookup1("E0067") ->	{"0","",""};
lookup1("E0068") ->	{"0","",""};
lookup1("E0069") ->	{"0","",""};
lookup1("E006A") ->	{"0","",""};
lookup1("E006B") ->	{"0","",""};
lookup1("E006C") ->	{"0","",""};
lookup1("E006D") ->	{"0","",""};
lookup1("E006E") ->	{"0","",""};
lookup1("E006F") ->	{"0","",""};
lookup1("E0070") ->	{"0","",""};
lookup1("E0071") ->	{"0","",""};
lookup1("E0072") ->	{"0","",""};
lookup1("E0073") ->	{"0","",""};
lookup1("E0074") ->	{"0","",""};
lookup1("E0075") ->	{"0","",""};
lookup1("E0076") ->	{"0","",""};
lookup1("E0077") ->	{"0","",""};
lookup1("E0078") ->	{"0","",""};
lookup1("E0079") ->	{"0","",""};
lookup1("E007A") ->	{"0","",""};
lookup1("E007B") ->	{"0","",""};
lookup1("E007C") ->	{"0","",""};
lookup1("E007D") ->	{"0","",""};
lookup1("E007E") ->	{"0","",""};
lookup1("E007F") ->	{"0","",""};
lookup1("E0100") ->	{"0","",""};
lookup1("E0101") ->	{"0","",""};
lookup1("E0102") ->	{"0","",""};
lookup1("E0103") ->	{"0","",""};
lookup1("E0104") ->	{"0","",""};
lookup1("E0105") ->	{"0","",""};
lookup1("E0106") ->	{"0","",""};
lookup1("E0107") ->	{"0","",""};
lookup1("E0108") ->	{"0","",""};
lookup1("E0109") ->	{"0","",""};
lookup1("E010A") ->	{"0","",""};
lookup1("E010B") ->	{"0","",""};
lookup1("E010C") ->	{"0","",""};
lookup1("E010D") ->	{"0","",""};
lookup1("E010E") ->	{"0","",""};
lookup1("E010F") ->	{"0","",""};
lookup1("E0110") ->	{"0","",""};
lookup1("E0111") ->	{"0","",""};
lookup1("E0112") ->	{"0","",""};
lookup1("E0113") ->	{"0","",""};
lookup1("E0114") ->	{"0","",""};
lookup1("E0115") ->	{"0","",""};
lookup1("E0116") ->	{"0","",""};
lookup1("E0117") ->	{"0","",""};
lookup1("E0118") ->	{"0","",""};
lookup1("E0119") ->	{"0","",""};
lookup1("E011A") ->	{"0","",""};
lookup1("E011B") ->	{"0","",""};
lookup1("E011C") ->	{"0","",""};
lookup1("E011D") ->	{"0","",""};
lookup1("E011E") ->	{"0","",""};
lookup1("E011F") ->	{"0","",""};
lookup1("E0120") ->	{"0","",""};
lookup1("E0121") ->	{"0","",""};
lookup1("E0122") ->	{"0","",""};
lookup1("E0123") ->	{"0","",""};
lookup1("E0124") ->	{"0","",""};
lookup1("E0125") ->	{"0","",""};
lookup1("E0126") ->	{"0","",""};
lookup1("E0127") ->	{"0","",""};
lookup1("E0128") ->	{"0","",""};
lookup1("E0129") ->	{"0","",""};
lookup1("E012A") ->	{"0","",""};
lookup1("E012B") ->	{"0","",""};
lookup1("E012C") ->	{"0","",""};
lookup1("E012D") ->	{"0","",""};
lookup1("E012E") ->	{"0","",""};
lookup1("E012F") ->	{"0","",""};
lookup1("E0130") ->	{"0","",""};
lookup1("E0131") ->	{"0","",""};
lookup1("E0132") ->	{"0","",""};
lookup1("E0133") ->	{"0","",""};
lookup1("E0134") ->	{"0","",""};
lookup1("E0135") ->	{"0","",""};
lookup1("E0136") ->	{"0","",""};
lookup1("E0137") ->	{"0","",""};
lookup1("E0138") ->	{"0","",""};
lookup1("E0139") ->	{"0","",""};
lookup1("E013A") ->	{"0","",""};
lookup1("E013B") ->	{"0","",""};
lookup1("E013C") ->	{"0","",""};
lookup1("E013D") ->	{"0","",""};
lookup1("E013E") ->	{"0","",""};
lookup1("E013F") ->	{"0","",""};
lookup1("E0140") ->	{"0","",""};
lookup1("E0141") ->	{"0","",""};
lookup1("E0142") ->	{"0","",""};
lookup1("E0143") ->	{"0","",""};
lookup1("E0144") ->	{"0","",""};
lookup1("E0145") ->	{"0","",""};
lookup1("E0146") ->	{"0","",""};
lookup1("E0147") ->	{"0","",""};
lookup1("E0148") ->	{"0","",""};
lookup1("E0149") ->	{"0","",""};
lookup1("E014A") ->	{"0","",""};
lookup1("E014B") ->	{"0","",""};
lookup1("E014C") ->	{"0","",""};
lookup1("E014D") ->	{"0","",""};
lookup1("E014E") ->	{"0","",""};
lookup1("E014F") ->	{"0","",""};
lookup1("E0150") ->	{"0","",""};
lookup1("E0151") ->	{"0","",""};
lookup1("E0152") ->	{"0","",""};
lookup1("E0153") ->	{"0","",""};
lookup1("E0154") ->	{"0","",""};
lookup1("E0155") ->	{"0","",""};
lookup1("E0156") ->	{"0","",""};
lookup1("E0157") ->	{"0","",""};
lookup1("E0158") ->	{"0","",""};
lookup1("E0159") ->	{"0","",""};
lookup1("E015A") ->	{"0","",""};
lookup1("E015B") ->	{"0","",""};
lookup1("E015C") ->	{"0","",""};
lookup1("E015D") ->	{"0","",""};
lookup1("E015E") ->	{"0","",""};
lookup1("E015F") ->	{"0","",""};
lookup1("E0160") ->	{"0","",""};
lookup1("E0161") ->	{"0","",""};
lookup1("E0162") ->	{"0","",""};
lookup1("E0163") ->	{"0","",""};
lookup1("E0164") ->	{"0","",""};
lookup1("E0165") ->	{"0","",""};
lookup1("E0166") ->	{"0","",""};
lookup1("E0167") ->	{"0","",""};
lookup1("E0168") ->	{"0","",""};
lookup1("E0169") ->	{"0","",""};
lookup1("E016A") ->	{"0","",""};
lookup1("E016B") ->	{"0","",""};
lookup1("E016C") ->	{"0","",""};
lookup1("E016D") ->	{"0","",""};
lookup1("E016E") ->	{"0","",""};
lookup1("E016F") ->	{"0","",""};
lookup1("E0170") ->	{"0","",""};
lookup1("E0171") ->	{"0","",""};
lookup1("E0172") ->	{"0","",""};
lookup1("E0173") ->	{"0","",""};
lookup1("E0174") ->	{"0","",""};
lookup1("E0175") ->	{"0","",""};
lookup1("E0176") ->	{"0","",""};
lookup1("E0177") ->	{"0","",""};
lookup1("E0178") ->	{"0","",""};
lookup1("E0179") ->	{"0","",""};
lookup1("E017A") ->	{"0","",""};
lookup1("E017B") ->	{"0","",""};
lookup1("E017C") ->	{"0","",""};
lookup1("E017D") ->	{"0","",""};
lookup1("E017E") ->	{"0","",""};
lookup1("E017F") ->	{"0","",""};
lookup1("E0180") ->	{"0","",""};
lookup1("E0181") ->	{"0","",""};
lookup1("E0182") ->	{"0","",""};
lookup1("E0183") ->	{"0","",""};
lookup1("E0184") ->	{"0","",""};
lookup1("E0185") ->	{"0","",""};
lookup1("E0186") ->	{"0","",""};
lookup1("E0187") ->	{"0","",""};
lookup1("E0188") ->	{"0","",""};
lookup1("E0189") ->	{"0","",""};
lookup1("E018A") ->	{"0","",""};
lookup1("E018B") ->	{"0","",""};
lookup1("E018C") ->	{"0","",""};
lookup1("E018D") ->	{"0","",""};
lookup1("E018E") ->	{"0","",""};
lookup1("E018F") ->	{"0","",""};
lookup1("E0190") ->	{"0","",""};
lookup1("E0191") ->	{"0","",""};
lookup1("E0192") ->	{"0","",""};
lookup1("E0193") ->	{"0","",""};
lookup1("E0194") ->	{"0","",""};
lookup1("E0195") ->	{"0","",""};
lookup1("E0196") ->	{"0","",""};
lookup1("E0197") ->	{"0","",""};
lookup1("E0198") ->	{"0","",""};
lookup1("E0199") ->	{"0","",""};
lookup1("E019A") ->	{"0","",""};
lookup1("E019B") ->	{"0","",""};
lookup1("E019C") ->	{"0","",""};
lookup1("E019D") ->	{"0","",""};
lookup1("E019E") ->	{"0","",""};
lookup1("E019F") ->	{"0","",""};
lookup1("E01A0") ->	{"0","",""};
lookup1("E01A1") ->	{"0","",""};
lookup1("E01A2") ->	{"0","",""};
lookup1("E01A3") ->	{"0","",""};
lookup1("E01A4") ->	{"0","",""};
lookup1("E01A5") ->	{"0","",""};
lookup1("E01A6") ->	{"0","",""};
lookup1("E01A7") ->	{"0","",""};
lookup1("E01A8") ->	{"0","",""};
lookup1("E01A9") ->	{"0","",""};
lookup1("E01AA") ->	{"0","",""};
lookup1("E01AB") ->	{"0","",""};
lookup1("E01AC") ->	{"0","",""};
lookup1("E01AD") ->	{"0","",""};
lookup1("E01AE") ->	{"0","",""};
lookup1("E01AF") ->	{"0","",""};
lookup1("E01B0") ->	{"0","",""};
lookup1("E01B1") ->	{"0","",""};
lookup1("E01B2") ->	{"0","",""};
lookup1("E01B3") ->	{"0","",""};
lookup1("E01B4") ->	{"0","",""};
lookup1("E01B5") ->	{"0","",""};
lookup1("E01B6") ->	{"0","",""};
lookup1("E01B7") ->	{"0","",""};
lookup1("E01B8") ->	{"0","",""};
lookup1("E01B9") ->	{"0","",""};
lookup1("E01BA") ->	{"0","",""};
lookup1("E01BB") ->	{"0","",""};
lookup1("E01BC") ->	{"0","",""};
lookup1("E01BD") ->	{"0","",""};
lookup1("E01BE") ->	{"0","",""};
lookup1("E01BF") ->	{"0","",""};
lookup1("E01C0") ->	{"0","",""};
lookup1("E01C1") ->	{"0","",""};
lookup1("E01C2") ->	{"0","",""};
lookup1("E01C3") ->	{"0","",""};
lookup1("E01C4") ->	{"0","",""};
lookup1("E01C5") ->	{"0","",""};
lookup1("E01C6") ->	{"0","",""};
lookup1("E01C7") ->	{"0","",""};
lookup1("E01C8") ->	{"0","",""};
lookup1("E01C9") ->	{"0","",""};
lookup1("E01CA") ->	{"0","",""};
lookup1("E01CB") ->	{"0","",""};
lookup1("E01CC") ->	{"0","",""};
lookup1("E01CD") ->	{"0","",""};
lookup1("E01CE") ->	{"0","",""};
lookup1("E01CF") ->	{"0","",""};
lookup1("E01D0") ->	{"0","",""};
lookup1("E01D1") ->	{"0","",""};
lookup1("E01D2") ->	{"0","",""};
lookup1("E01D3") ->	{"0","",""};
lookup1("E01D4") ->	{"0","",""};
lookup1("E01D5") ->	{"0","",""};
lookup1("E01D6") ->	{"0","",""};
lookup1("E01D7") ->	{"0","",""};
lookup1("E01D8") ->	{"0","",""};
lookup1("E01D9") ->	{"0","",""};
lookup1("E01DA") ->	{"0","",""};
lookup1("E01DB") ->	{"0","",""};
lookup1("E01DC") ->	{"0","",""};
lookup1("E01DD") ->	{"0","",""};
lookup1("E01DE") ->	{"0","",""};
lookup1("E01DF") ->	{"0","",""};
lookup1("E01E0") ->	{"0","",""};
lookup1("E01E1") ->	{"0","",""};
lookup1("E01E2") ->	{"0","",""};
lookup1("E01E3") ->	{"0","",""};
lookup1("E01E4") ->	{"0","",""};
lookup1("E01E5") ->	{"0","",""};
lookup1("E01E6") ->	{"0","",""};
lookup1("E01E7") ->	{"0","",""};
lookup1("E01E8") ->	{"0","",""};
lookup1("E01E9") ->	{"0","",""};
lookup1("E01EA") ->	{"0","",""};
lookup1("E01EB") ->	{"0","",""};
lookup1("E01EC") ->	{"0","",""};
lookup1("E01ED") ->	{"0","",""};
lookup1("E01EE") ->	{"0","",""};
lookup1("E01EF") ->	{"0","",""};
lookup1("F0000") ->	{"0","",""};
lookup1("F8FF") ->	{"0","",""};
lookup1("F900") ->	{"0","8C48",""};
lookup1("F901") ->	{"0","66F4",""};
lookup1("F902") ->	{"0","8ECA",""};
lookup1("F903") ->	{"0","8CC8",""};
lookup1("F904") ->	{"0","6ED1",""};
lookup1("F905") ->	{"0","4E32",""};
lookup1("F906") ->	{"0","53E5",""};
lookup1("F907") ->	{"0","9F9C",""};
lookup1("F908") ->	{"0","9F9C",""};
lookup1("F909") ->	{"0","5951",""};
lookup1("F90A") ->	{"0","91D1",""};
lookup1("F90B") ->	{"0","5587",""};
lookup1("F90C") ->	{"0","5948",""};
lookup1("F90D") ->	{"0","61F6",""};
lookup1("F90E") ->	{"0","7669",""};
lookup1("F90F") ->	{"0","7F85",""};
lookup1("F910") ->	{"0","863F",""};
lookup1("F911") ->	{"0","87BA",""};
lookup1("F912") ->	{"0","88F8",""};
lookup1("F913") ->	{"0","908F",""};
lookup1("F914") ->	{"0","6A02",""};
lookup1("F915") ->	{"0","6D1B",""};
lookup1("F916") ->	{"0","70D9",""};
lookup1("F917") ->	{"0","73DE",""};
lookup1("F918") ->	{"0","843D",""};
lookup1("F919") ->	{"0","916A",""};
lookup1("F91A") ->	{"0","99F1",""};
lookup1("F91B") ->	{"0","4E82",""};
lookup1("F91C") ->	{"0","5375",""};
lookup1("F91D") ->	{"0","6B04",""};
lookup1("F91E") ->	{"0","721B",""};
lookup1("F91F") ->	{"0","862D",""};
lookup1("F920") ->	{"0","9E1E",""};
lookup1("F921") ->	{"0","5D50",""};
lookup1("F922") ->	{"0","6FEB",""};
lookup1("F923") ->	{"0","85CD",""};
lookup1("F924") ->	{"0","8964",""};
lookup1("F925") ->	{"0","62C9",""};
lookup1("F926") ->	{"0","81D8",""};
lookup1("F927") ->	{"0","881F",""};
lookup1("F928") ->	{"0","5ECA",""};
lookup1("F929") ->	{"0","6717",""};
lookup1("F92A") ->	{"0","6D6A",""};
lookup1("F92B") ->	{"0","72FC",""};
lookup1("F92C") ->	{"0","90CE",""};
lookup1("F92D") ->	{"0","4F86",""};
lookup1("F92E") ->	{"0","51B7",""};
lookup1("F92F") ->	{"0","52DE",""};
lookup1("F930") ->	{"0","64C4",""};
lookup1("F931") ->	{"0","6AD3",""};
lookup1("F932") ->	{"0","7210",""};
lookup1("F933") ->	{"0","76E7",""};
lookup1("F934") ->	{"0","8001",""};
lookup1("F935") ->	{"0","8606",""};
lookup1("F936") ->	{"0","865C",""};
lookup1("F937") ->	{"0","8DEF",""};
lookup1("F938") ->	{"0","9732",""};
lookup1("F939") ->	{"0","9B6F",""};
lookup1("F93A") ->	{"0","9DFA",""};
lookup1("F93B") ->	{"0","788C",""};
lookup1("F93C") ->	{"0","797F",""};
lookup1("F93D") ->	{"0","7DA0",""};
lookup1("F93E") ->	{"0","83C9",""};
lookup1("F93F") ->	{"0","9304",""};
lookup1("F940") ->	{"0","9E7F",""};
lookup1("F941") ->	{"0","8AD6",""};
lookup1("F942") ->	{"0","58DF",""};
lookup1("F943") ->	{"0","5F04",""};
lookup1("F944") ->	{"0","7C60",""};
lookup1("F945") ->	{"0","807E",""};
lookup1("F946") ->	{"0","7262",""};
lookup1("F947") ->	{"0","78CA",""};
lookup1("F948") ->	{"0","8CC2",""};
lookup1("F949") ->	{"0","96F7",""};
lookup1("F94A") ->	{"0","58D8",""};
lookup1("F94B") ->	{"0","5C62",""};
lookup1("F94C") ->	{"0","6A13",""};
lookup1("F94D") ->	{"0","6DDA",""};
lookup1("F94E") ->	{"0","6F0F",""};
lookup1("F94F") ->	{"0","7D2F",""};
lookup1("F950") ->	{"0","7E37",""};
lookup1("F951") ->	{"0","964B",""};
lookup1("F952") ->	{"0","52D2",""};
lookup1("F953") ->	{"0","808B",""};
lookup1("F954") ->	{"0","51DC",""};
lookup1("F955") ->	{"0","51CC",""};
lookup1("F956") ->	{"0","7A1C",""};
lookup1("F957") ->	{"0","7DBE",""};
lookup1("F958") ->	{"0","83F1",""};
lookup1("F959") ->	{"0","9675",""};
lookup1("F95A") ->	{"0","8B80",""};
lookup1("F95B") ->	{"0","62CF",""};
lookup1("F95C") ->	{"0","6A02",""};
lookup1("F95D") ->	{"0","8AFE",""};
lookup1("F95E") ->	{"0","4E39",""};
lookup1("F95F") ->	{"0","5BE7",""};
lookup1("F960") ->	{"0","6012",""};
lookup1("F961") ->	{"0","7387",""};
lookup1("F962") ->	{"0","7570",""};
lookup1("F963") ->	{"0","5317",""};
lookup1("F964") ->	{"0","78FB",""};
lookup1("F965") ->	{"0","4FBF",""};
lookup1("F966") ->	{"0","5FA9",""};
lookup1("F967") ->	{"0","4E0D",""};
lookup1("F968") ->	{"0","6CCC",""};
lookup1("F969") ->	{"0","6578",""};
lookup1("F96A") ->	{"0","7D22",""};
lookup1("F96B") ->	{"0","53C3",""};
lookup1("F96C") ->	{"0","585E",""};
lookup1("F96D") ->	{"0","7701",""};
lookup1("F96E") ->	{"0","8449",""};
lookup1("F96F") ->	{"0","8AAA",""};
lookup1("F970") ->	{"0","6BBA",""};
lookup1("F971") ->	{"0","8FB0",""};
lookup1("F972") ->	{"0","6C88",""};
lookup1("F973") ->	{"0","62FE",""};
lookup1("F974") ->	{"0","82E5",""};
lookup1("F975") ->	{"0","63A0",""};
lookup1("F976") ->	{"0","7565",""};
lookup1("F977") ->	{"0","4EAE",""};
lookup1("F978") ->	{"0","5169",""};
lookup1("F979") ->	{"0","51C9",""};
lookup1("F97A") ->	{"0","6881",""};
lookup1("F97B") ->	{"0","7CE7",""};
lookup1("F97C") ->	{"0","826F",""};
lookup1("F97D") ->	{"0","8AD2",""};
lookup1("F97E") ->	{"0","91CF",""};
lookup1("F97F") ->	{"0","52F5",""};
lookup1("F980") ->	{"0","5442",""};
lookup1("F981") ->	{"0","5973",""};
lookup1("F982") ->	{"0","5EEC",""};
lookup1("F983") ->	{"0","65C5",""};
lookup1("F984") ->	{"0","6FFE",""};
lookup1("F985") ->	{"0","792A",""};
lookup1("F986") ->	{"0","95AD",""};
lookup1("F987") ->	{"0","9A6A",""};
lookup1("F988") ->	{"0","9E97",""};
lookup1("F989") ->	{"0","9ECE",""};
lookup1("F98A") ->	{"0","529B",""};
lookup1("F98B") ->	{"0","66C6",""};
lookup1("F98C") ->	{"0","6B77",""};
lookup1("F98D") ->	{"0","8F62",""};
lookup1("F98E") ->	{"0","5E74",""};
lookup1("F98F") ->	{"0","6190",""};
lookup1("F990") ->	{"0","6200",""};
lookup1("F991") ->	{"0","649A",""};
lookup1("F992") ->	{"0","6F23",""};
lookup1("F993") ->	{"0","7149",""};
lookup1("F994") ->	{"0","7489",""};
lookup1("F995") ->	{"0","79CA",""};
lookup1("F996") ->	{"0","7DF4",""};
lookup1("F997") ->	{"0","806F",""};
lookup1("F998") ->	{"0","8F26",""};
lookup1("F999") ->	{"0","84EE",""};
lookup1("F99A") ->	{"0","9023",""};
lookup1("F99B") ->	{"0","934A",""};
lookup1("F99C") ->	{"0","5217",""};
lookup1("F99D") ->	{"0","52A3",""};
lookup1("F99E") ->	{"0","54BD",""};
lookup1("F99F") ->	{"0","70C8",""};
lookup1("F9A0") ->	{"0","88C2",""};
lookup1("F9A1") ->	{"0","8AAA",""};
lookup1("F9A2") ->	{"0","5EC9",""};
lookup1("F9A3") ->	{"0","5FF5",""};
lookup1("F9A4") ->	{"0","637B",""};
lookup1("F9A5") ->	{"0","6BAE",""};
lookup1("F9A6") ->	{"0","7C3E",""};
lookup1("F9A7") ->	{"0","7375",""};
lookup1("F9A8") ->	{"0","4EE4",""};
lookup1("F9A9") ->	{"0","56F9",""};
lookup1("F9AA") ->	{"0","5BE7",""};
lookup1("F9AB") ->	{"0","5DBA",""};
lookup1("F9AC") ->	{"0","601C",""};
lookup1("F9AD") ->	{"0","73B2",""};
lookup1("F9AE") ->	{"0","7469",""};
lookup1("F9AF") ->	{"0","7F9A",""};
lookup1("F9B0") ->	{"0","8046",""};
lookup1("F9B1") ->	{"0","9234",""};
lookup1("F9B2") ->	{"0","96F6",""};
lookup1("F9B3") ->	{"0","9748",""};
lookup1("F9B4") ->	{"0","9818",""};
lookup1("F9B5") ->	{"0","4F8B",""};
lookup1("F9B6") ->	{"0","79AE",""};
lookup1("F9B7") ->	{"0","91B4",""};
lookup1("F9B8") ->	{"0","96B8",""};
lookup1("F9B9") ->	{"0","60E1",""};
lookup1("F9BA") ->	{"0","4E86",""};
lookup1("F9BB") ->	{"0","50DA",""};
lookup1("F9BC") ->	{"0","5BEE",""};
lookup1("F9BD") ->	{"0","5C3F",""};
lookup1("F9BE") ->	{"0","6599",""};
lookup1("F9BF") ->	{"0","6A02",""};
lookup1("F9C0") ->	{"0","71CE",""};
lookup1("F9C1") ->	{"0","7642",""};
lookup1("F9C2") ->	{"0","84FC",""};
lookup1("F9C3") ->	{"0","907C",""};
lookup1("F9C4") ->	{"0","9F8D",""};
lookup1("F9C5") ->	{"0","6688",""};
lookup1("F9C6") ->	{"0","962E",""};
lookup1("F9C7") ->	{"0","5289",""};
lookup1("F9C8") ->	{"0","677B",""};
lookup1("F9C9") ->	{"0","67F3",""};
lookup1("F9CA") ->	{"0","6D41",""};
lookup1("F9CB") ->	{"0","6E9C",""};
lookup1("F9CC") ->	{"0","7409",""};
lookup1("F9CD") ->	{"0","7559",""};
lookup1("F9CE") ->	{"0","786B",""};
lookup1("F9CF") ->	{"0","7D10",""};
lookup1("F9D0") ->	{"0","985E",""};
lookup1("F9D1") ->	{"0","516D",""};
lookup1("F9D2") ->	{"0","622E",""};
lookup1("F9D3") ->	{"0","9678",""};
lookup1("F9D4") ->	{"0","502B",""};
lookup1("F9D5") ->	{"0","5D19",""};
lookup1("F9D6") ->	{"0","6DEA",""};
lookup1("F9D7") ->	{"0","8F2A",""};
lookup1("F9D8") ->	{"0","5F8B",""};
lookup1("F9D9") ->	{"0","6144",""};
lookup1("F9DA") ->	{"0","6817",""};
lookup1("F9DB") ->	{"0","7387",""};
lookup1("F9DC") ->	{"0","9686",""};
lookup1("F9DD") ->	{"0","5229",""};
lookup1("F9DE") ->	{"0","540F",""};
lookup1("F9DF") ->	{"0","5C65",""};
lookup1("F9E0") ->	{"0","6613",""};
lookup1("F9E1") ->	{"0","674E",""};
lookup1("F9E2") ->	{"0","68A8",""};
lookup1("F9E3") ->	{"0","6CE5",""};
lookup1("F9E4") ->	{"0","7406",""};
lookup1("F9E5") ->	{"0","75E2",""};
lookup1("F9E6") ->	{"0","7F79",""};
lookup1("F9E7") ->	{"0","88CF",""};
lookup1("F9E8") ->	{"0","88E1",""};
lookup1("F9E9") ->	{"0","91CC",""};
lookup1("F9EA") ->	{"0","96E2",""};
lookup1("F9EB") ->	{"0","533F",""};
lookup1("F9EC") ->	{"0","6EBA",""};
lookup1("F9ED") ->	{"0","541D",""};
lookup1("F9EE") ->	{"0","71D0",""};
lookup1("F9EF") ->	{"0","7498",""};
lookup1("F9F0") ->	{"0","85FA",""};
lookup1("F9F1") ->	{"0","96A3",""};
lookup1("F9F2") ->	{"0","9C57",""};
lookup1("F9F3") ->	{"0","9E9F",""};
lookup1("F9F4") ->	{"0","6797",""};
lookup1("F9F5") ->	{"0","6DCB",""};
lookup1("F9F6") ->	{"0","81E8",""};
lookup1("F9F7") ->	{"0","7ACB",""};
lookup1("F9F8") ->	{"0","7B20",""};
lookup1("F9F9") ->	{"0","7C92",""};
lookup1("F9FA") ->	{"0","72C0",""};
lookup1("F9FB") ->	{"0","7099",""};
lookup1("F9FC") ->	{"0","8B58",""};
lookup1("F9FD") ->	{"0","4EC0",""};
lookup1("F9FE") ->	{"0","8336",""};
lookup1("F9FF") ->	{"0","523A",""};
lookup1("FA00") ->	{"0","5207",""};
lookup1("FA01") ->	{"0","5EA6",""};
lookup1("FA02") ->	{"0","62D3",""};
lookup1("FA03") ->	{"0","7CD6",""};
lookup1("FA04") ->	{"0","5B85",""};
lookup1("FA05") ->	{"0","6D1E",""};
lookup1("FA06") ->	{"0","66B4",""};
lookup1("FA07") ->	{"0","8F3B",""};
lookup1("FA08") ->	{"0","884C",""};
lookup1("FA09") ->	{"0","964D",""};
lookup1("FA0A") ->	{"0","898B",""};
lookup1("FA0B") ->	{"0","5ED3",""};
lookup1("FA0C") ->	{"0","5140",""};
lookup1("FA0D") ->	{"0","55C0",""};
lookup1("FA0E") ->	{"0","",""};
lookup1("FA0F") ->	{"0","",""};
lookup1("FA10") ->	{"0","585A",""};
lookup1("FA11") ->	{"0","",""};
lookup1("FA12") ->	{"0","6674",""};
lookup1("FA13") ->	{"0","",""};
lookup1("FA14") ->	{"0","",""};
lookup1("FA15") ->	{"0","51DE",""};
lookup1("FA16") ->	{"0","732A",""};
lookup1("FA17") ->	{"0","76CA",""};
lookup1("FA18") ->	{"0","793C",""};
lookup1("FA19") ->	{"0","795E",""};
lookup1("FA1A") ->	{"0","7965",""};
lookup1("FA1B") ->	{"0","798F",""};
lookup1("FA1C") ->	{"0","9756",""};
lookup1("FA1D") ->	{"0","7CBE",""};
lookup1("FA1E") ->	{"0","7FBD",""};
lookup1("FA1F") ->	{"0","",""};
lookup1("FA20") ->	{"0","8612",""};
lookup1("FA21") ->	{"0","",""};
lookup1("FA22") ->	{"0","8AF8",""};
lookup1("FA23") ->	{"0","",""};
lookup1("FA24") ->	{"0","",""};
lookup1("FA25") ->	{"0","9038",""};
lookup1("FA26") ->	{"0","90FD",""};
lookup1("FA27") ->	{"0","",""};
lookup1("FA28") ->	{"0","",""};
lookup1("FA29") ->	{"0","",""};
lookup1("FA2A") ->	{"0","98EF",""};
lookup1("FA2B") ->	{"0","98FC",""};
lookup1("FA2C") ->	{"0","9928",""};
lookup1("FA2D") ->	{"0","9DB4",""};
lookup1("FA2E") ->	{"0","90DE",""};
lookup1("FA2F") ->	{"0","96B7",""};
lookup1("FA30") ->	{"0","4FAE",""};
lookup1("FA31") ->	{"0","50E7",""};
lookup1("FA32") ->	{"0","514D",""};
lookup1("FA33") ->	{"0","52C9",""};
lookup1("FA34") ->	{"0","52E4",""};
lookup1("FA35") ->	{"0","5351",""};
lookup1("FA36") ->	{"0","559D",""};
lookup1("FA37") ->	{"0","5606",""};
lookup1("FA38") ->	{"0","5668",""};
lookup1("FA39") ->	{"0","5840",""};
lookup1("FA3A") ->	{"0","58A8",""};
lookup1("FA3B") ->	{"0","5C64",""};
lookup1("FA3C") ->	{"0","5C6E",""};
lookup1("FA3D") ->	{"0","6094",""};
lookup1("FA3E") ->	{"0","6168",""};
lookup1("FA3F") ->	{"0","618E",""};
lookup1("FA40") ->	{"0","61F2",""};
lookup1("FA41") ->	{"0","654F",""};
lookup1("FA42") ->	{"0","65E2",""};
lookup1("FA43") ->	{"0","6691",""};
lookup1("FA44") ->	{"0","6885",""};
lookup1("FA45") ->	{"0","6D77",""};
lookup1("FA46") ->	{"0","6E1A",""};
lookup1("FA47") ->	{"0","6F22",""};
lookup1("FA48") ->	{"0","716E",""};
lookup1("FA49") ->	{"0","722B",""};
lookup1("FA4A") ->	{"0","7422",""};
lookup1("FA4B") ->	{"0","7891",""};
lookup1("FA4C") ->	{"0","793E",""};
lookup1("FA4D") ->	{"0","7949",""};
lookup1("FA4E") ->	{"0","7948",""};
lookup1("FA4F") ->	{"0","7950",""};
lookup1("FA50") ->	{"0","7956",""};
lookup1("FA51") ->	{"0","795D",""};
lookup1("FA52") ->	{"0","798D",""};
lookup1("FA53") ->	{"0","798E",""};
lookup1("FA54") ->	{"0","7A40",""};
lookup1("FA55") ->	{"0","7A81",""};
lookup1("FA56") ->	{"0","7BC0",""};
lookup1("FA57") ->	{"0","7DF4",""};
lookup1("FA58") ->	{"0","7E09",""};
lookup1("FA59") ->	{"0","7E41",""};
lookup1("FA5A") ->	{"0","7F72",""};
lookup1("FA5B") ->	{"0","8005",""};
lookup1("FA5C") ->	{"0","81ED",""};
lookup1("FA5D") ->	{"0","8279",""};
lookup1("FA5E") ->	{"0","8279",""};
lookup1("FA5F") ->	{"0","8457",""};
lookup1("FA60") ->	{"0","8910",""};
lookup1("FA61") ->	{"0","8996",""};
lookup1("FA62") ->	{"0","8B01",""};
lookup1("FA63") ->	{"0","8B39",""};
lookup1("FA64") ->	{"0","8CD3",""};
lookup1("FA65") ->	{"0","8D08",""};
lookup1("FA66") ->	{"0","8FB6",""};
lookup1("FA67") ->	{"0","9038",""};
lookup1("FA68") ->	{"0","96E3",""};
lookup1("FA69") ->	{"0","97FF",""};
lookup1("FA6A") ->	{"0","983B",""};
lookup1("FA6B") ->	{"0","6075",""};
lookup1("FA6C") ->	{"0","242EE",""};
lookup1("FA6D") ->	{"0","8218",""};
lookup1("FA70") ->	{"0","4E26",""};
lookup1("FA71") ->	{"0","51B5",""};
lookup1("FA72") ->	{"0","5168",""};
lookup1("FA73") ->	{"0","4F80",""};
lookup1("FA74") ->	{"0","5145",""};
lookup1("FA75") ->	{"0","5180",""};
lookup1("FA76") ->	{"0","52C7",""};
lookup1("FA77") ->	{"0","52FA",""};
lookup1("FA78") ->	{"0","559D",""};
lookup1("FA79") ->	{"0","5555",""};
lookup1("FA7A") ->	{"0","5599",""};
lookup1("FA7B") ->	{"0","55E2",""};
lookup1("FA7C") ->	{"0","585A",""};
lookup1("FA7D") ->	{"0","58B3",""};
lookup1("FA7E") ->	{"0","5944",""};
lookup1("FA7F") ->	{"0","5954",""};
lookup1("FA80") ->	{"0","5A62",""};
lookup1("FA81") ->	{"0","5B28",""};
lookup1("FA82") ->	{"0","5ED2",""};
lookup1("FA83") ->	{"0","5ED9",""};
lookup1("FA84") ->	{"0","5F69",""};
lookup1("FA85") ->	{"0","5FAD",""};
lookup1("FA86") ->	{"0","60D8",""};
lookup1("FA87") ->	{"0","614E",""};
lookup1("FA88") ->	{"0","6108",""};
lookup1("FA89") ->	{"0","618E",""};
lookup1("FA8A") ->	{"0","6160",""};
lookup1("FA8B") ->	{"0","61F2",""};
lookup1("FA8C") ->	{"0","6234",""};
lookup1("FA8D") ->	{"0","63C4",""};
lookup1("FA8E") ->	{"0","641C",""};
lookup1("FA8F") ->	{"0","6452",""};
lookup1("FA90") ->	{"0","6556",""};
lookup1("FA91") ->	{"0","6674",""};
lookup1("FA92") ->	{"0","6717",""};
lookup1("FA93") ->	{"0","671B",""};
lookup1("FA94") ->	{"0","6756",""};
lookup1("FA95") ->	{"0","6B79",""};
lookup1("FA96") ->	{"0","6BBA",""};
lookup1("FA97") ->	{"0","6D41",""};
lookup1("FA98") ->	{"0","6EDB",""};
lookup1("FA99") ->	{"0","6ECB",""};
lookup1("FA9A") ->	{"0","6F22",""};
lookup1("FA9B") ->	{"0","701E",""};
lookup1("FA9C") ->	{"0","716E",""};
lookup1("FA9D") ->	{"0","77A7",""};
lookup1("FA9E") ->	{"0","7235",""};
lookup1("FA9F") ->	{"0","72AF",""};
lookup1("FAA0") ->	{"0","732A",""};
lookup1("FAA1") ->	{"0","7471",""};
lookup1("FAA2") ->	{"0","7506",""};
lookup1("FAA3") ->	{"0","753B",""};
lookup1("FAA4") ->	{"0","761D",""};
lookup1("FAA5") ->	{"0","761F",""};
lookup1("FAA6") ->	{"0","76CA",""};
lookup1("FAA7") ->	{"0","76DB",""};
lookup1("FAA8") ->	{"0","76F4",""};
lookup1("FAA9") ->	{"0","774A",""};
lookup1("FAAA") ->	{"0","7740",""};
lookup1("FAAB") ->	{"0","78CC",""};
lookup1("FAAC") ->	{"0","7AB1",""};
lookup1("FAAD") ->	{"0","7BC0",""};
lookup1("FAAE") ->	{"0","7C7B",""};
lookup1("FAAF") ->	{"0","7D5B",""};
lookup1("FAB0") ->	{"0","7DF4",""};
lookup1("FAB1") ->	{"0","7F3E",""};
lookup1("FAB2") ->	{"0","8005",""};
lookup1("FAB3") ->	{"0","8352",""};
lookup1("FAB4") ->	{"0","83EF",""};
lookup1("FAB5") ->	{"0","8779",""};
lookup1("FAB6") ->	{"0","8941",""};
lookup1("FAB7") ->	{"0","8986",""};
lookup1("FAB8") ->	{"0","8996",""};
lookup1("FAB9") ->	{"0","8ABF",""};
lookup1("FABA") ->	{"0","8AF8",""};
lookup1("FABB") ->	{"0","8ACB",""};
lookup1("FABC") ->	{"0","8B01",""};
lookup1("FABD") ->	{"0","8AFE",""};
lookup1("FABE") ->	{"0","8AED",""};
lookup1("FABF") ->	{"0","8B39",""};
lookup1("FAC0") ->	{"0","8B8A",""};
lookup1("FAC1") ->	{"0","8D08",""};
lookup1("FAC2") ->	{"0","8F38",""};
lookup1("FAC3") ->	{"0","9072",""};
lookup1("FAC4") ->	{"0","9199",""};
lookup1("FAC5") ->	{"0","9276",""};
lookup1("FAC6") ->	{"0","967C",""};
lookup1("FAC7") ->	{"0","96E3",""};
lookup1("FAC8") ->	{"0","9756",""};
lookup1("FAC9") ->	{"0","97DB",""};
lookup1("FACA") ->	{"0","97FF",""};
lookup1("FACB") ->	{"0","980B",""};
lookup1("FACC") ->	{"0","983B",""};
lookup1("FACD") ->	{"0","9B12",""};
lookup1("FACE") ->	{"0","9F9C",""};
lookup1("FACF") ->	{"0","2284A",""};
lookup1("FAD0") ->	{"0","22844",""};
lookup1("FAD1") ->	{"0","233D5",""};
lookup1("FAD2") ->	{"0","3B9D",""};
lookup1("FAD3") ->	{"0","4018",""};
lookup1("FAD4") ->	{"0","4039",""};
lookup1("FAD5") ->	{"0","25249",""};
lookup1("FAD6") ->	{"0","25CD0",""};
lookup1("FAD7") ->	{"0","27ED3",""};
lookup1("FAD8") ->	{"0","9F43",""};
lookup1("FAD9") ->	{"0","9F8E",""};
lookup1("FB00") ->	{"0","<compat> 0066 0066",""};
lookup1("FB01") ->	{"0","<compat> 0066 0069",""};
lookup1("FB02") ->	{"0","<compat> 0066 006C",""};
lookup1("FB03") ->	{"0","<compat> 0066 0066 0069",""};
lookup1("FB04") ->	{"0","<compat> 0066 0066 006C",""};
lookup1("FB05") ->	{"0","<compat> 017F 0074",""};
lookup1("FB06") ->	{"0","<compat> 0073 0074",""};
lookup1("FB13") ->	{"0","<compat> 0574 0576",""};
lookup1("FB14") ->	{"0","<compat> 0574 0565",""};
lookup1("FB15") ->	{"0","<compat> 0574 056B",""};
lookup1("FB16") ->	{"0","<compat> 057E 0576",""};
lookup1("FB17") ->	{"0","<compat> 0574 056D",""};
lookup1("FB1D") ->	{"0","05D9 05B4",""};
lookup1("FB1E") ->	{"26","",""};
lookup1("FB1F") ->	{"0","05F2 05B7",""};
lookup1("FB20") ->	{"0","<font> 05E2",""};
lookup1("FB21") ->	{"0","<font> 05D0",""};
lookup1("FB22") ->	{"0","<font> 05D3",""};
lookup1("FB23") ->	{"0","<font> 05D4",""};
lookup1("FB24") ->	{"0","<font> 05DB",""};
lookup1("FB25") ->	{"0","<font> 05DC",""};
lookup1("FB26") ->	{"0","<font> 05DD",""};
lookup1("FB27") ->	{"0","<font> 05E8",""};
lookup1("FB28") ->	{"0","<font> 05EA",""};
lookup1("FB29") ->	{"0","<font> 002B",""};
lookup1("FB2A") ->	{"0","05E9 05C1",""};
lookup1("FB2B") ->	{"0","05E9 05C2",""};
lookup1("FB2C") ->	{"0","FB49 05C1",""};
lookup1("FB2D") ->	{"0","FB49 05C2",""};
lookup1("FB2E") ->	{"0","05D0 05B7",""};
lookup1("FB2F") ->	{"0","05D0 05B8",""};
lookup1("FB30") ->	{"0","05D0 05BC",""};
lookup1("FB31") ->	{"0","05D1 05BC",""};
lookup1("FB32") ->	{"0","05D2 05BC",""};
lookup1("FB33") ->	{"0","05D3 05BC",""};
lookup1("FB34") ->	{"0","05D4 05BC",""};
lookup1("FB35") ->	{"0","05D5 05BC",""};
lookup1("FB36") ->	{"0","05D6 05BC",""};
lookup1("FB38") ->	{"0","05D8 05BC",""};
lookup1("FB39") ->	{"0","05D9 05BC",""};
lookup1("FB3A") ->	{"0","05DA 05BC",""};
lookup1("FB3B") ->	{"0","05DB 05BC",""};
lookup1("FB3C") ->	{"0","05DC 05BC",""};
lookup1("FB3E") ->	{"0","05DE 05BC",""};
lookup1("FB40") ->	{"0","05E0 05BC",""};
lookup1("FB41") ->	{"0","05E1 05BC",""};
lookup1("FB43") ->	{"0","05E3 05BC",""};
lookup1("FB44") ->	{"0","05E4 05BC",""};
lookup1("FB46") ->	{"0","05E6 05BC",""};
lookup1("FB47") ->	{"0","05E7 05BC",""};
lookup1("FB48") ->	{"0","05E8 05BC",""};
lookup1("FB49") ->	{"0","05E9 05BC",""};
lookup1("FB4A") ->	{"0","05EA 05BC",""};
lookup1("FB4B") ->	{"0","05D5 05B9",""};
lookup1("FB4C") ->	{"0","05D1 05BF",""};
lookup1("FB4D") ->	{"0","05DB 05BF",""};
lookup1("FB4E") ->	{"0","05E4 05BF",""};
lookup1("FB4F") ->	{"0","<compat> 05D0 05DC",""};
lookup1("FB50") ->	{"0","<isolated> 0671",""};
lookup1("FB51") ->	{"0","<final> 0671",""};
lookup1("FB52") ->	{"0","<isolated> 067B",""};
lookup1("FB53") ->	{"0","<final> 067B",""};
lookup1("FB54") ->	{"0","<initial> 067B",""};
lookup1("FB55") ->	{"0","<medial> 067B",""};
lookup1("FB56") ->	{"0","<isolated> 067E",""};
lookup1("FB57") ->	{"0","<final> 067E",""};
lookup1("FB58") ->	{"0","<initial> 067E",""};
lookup1("FB59") ->	{"0","<medial> 067E",""};
lookup1("FB5A") ->	{"0","<isolated> 0680",""};
lookup1("FB5B") ->	{"0","<final> 0680",""};
lookup1("FB5C") ->	{"0","<initial> 0680",""};
lookup1("FB5D") ->	{"0","<medial> 0680",""};
lookup1("FB5E") ->	{"0","<isolated> 067A",""};
lookup1("FB5F") ->	{"0","<final> 067A",""};
lookup1("FB60") ->	{"0","<initial> 067A",""};
lookup1("FB61") ->	{"0","<medial> 067A",""};
lookup1("FB62") ->	{"0","<isolated> 067F",""};
lookup1("FB63") ->	{"0","<final> 067F",""};
lookup1("FB64") ->	{"0","<initial> 067F",""};
lookup1("FB65") ->	{"0","<medial> 067F",""};
lookup1("FB66") ->	{"0","<isolated> 0679",""};
lookup1("FB67") ->	{"0","<final> 0679",""};
lookup1("FB68") ->	{"0","<initial> 0679",""};
lookup1("FB69") ->	{"0","<medial> 0679",""};
lookup1("FB6A") ->	{"0","<isolated> 06A4",""};
lookup1("FB6B") ->	{"0","<final> 06A4",""};
lookup1("FB6C") ->	{"0","<initial> 06A4",""};
lookup1("FB6D") ->	{"0","<medial> 06A4",""};
lookup1("FB6E") ->	{"0","<isolated> 06A6",""};
lookup1("FB6F") ->	{"0","<final> 06A6",""};
lookup1("FB70") ->	{"0","<initial> 06A6",""};
lookup1("FB71") ->	{"0","<medial> 06A6",""};
lookup1("FB72") ->	{"0","<isolated> 0684",""};
lookup1("FB73") ->	{"0","<final> 0684",""};
lookup1("FB74") ->	{"0","<initial> 0684",""};
lookup1("FB75") ->	{"0","<medial> 0684",""};
lookup1("FB76") ->	{"0","<isolated> 0683",""};
lookup1("FB77") ->	{"0","<final> 0683",""};
lookup1("FB78") ->	{"0","<initial> 0683",""};
lookup1("FB79") ->	{"0","<medial> 0683",""};
lookup1("FB7A") ->	{"0","<isolated> 0686",""};
lookup1("FB7B") ->	{"0","<final> 0686",""};
lookup1("FB7C") ->	{"0","<initial> 0686",""};
lookup1("FB7D") ->	{"0","<medial> 0686",""};
lookup1("FB7E") ->	{"0","<isolated> 0687",""};
lookup1("FB7F") ->	{"0","<final> 0687",""};
lookup1("FB80") ->	{"0","<initial> 0687",""};
lookup1("FB81") ->	{"0","<medial> 0687",""};
lookup1("FB82") ->	{"0","<isolated> 068D",""};
lookup1("FB83") ->	{"0","<final> 068D",""};
lookup1("FB84") ->	{"0","<isolated> 068C",""};
lookup1("FB85") ->	{"0","<final> 068C",""};
lookup1("FB86") ->	{"0","<isolated> 068E",""};
lookup1("FB87") ->	{"0","<final> 068E",""};
lookup1("FB88") ->	{"0","<isolated> 0688",""};
lookup1("FB89") ->	{"0","<final> 0688",""};
lookup1("FB8A") ->	{"0","<isolated> 0698",""};
lookup1("FB8B") ->	{"0","<final> 0698",""};
lookup1("FB8C") ->	{"0","<isolated> 0691",""};
lookup1("FB8D") ->	{"0","<final> 0691",""};
lookup1("FB8E") ->	{"0","<isolated> 06A9",""};
lookup1("FB8F") ->	{"0","<final> 06A9",""};
lookup1("FB90") ->	{"0","<initial> 06A9",""};
lookup1("FB91") ->	{"0","<medial> 06A9",""};
lookup1("FB92") ->	{"0","<isolated> 06AF",""};
lookup1("FB93") ->	{"0","<final> 06AF",""};
lookup1("FB94") ->	{"0","<initial> 06AF",""};
lookup1("FB95") ->	{"0","<medial> 06AF",""};
lookup1("FB96") ->	{"0","<isolated> 06B3",""};
lookup1("FB97") ->	{"0","<final> 06B3",""};
lookup1("FB98") ->	{"0","<initial> 06B3",""};
lookup1("FB99") ->	{"0","<medial> 06B3",""};
lookup1("FB9A") ->	{"0","<isolated> 06B1",""};
lookup1("FB9B") ->	{"0","<final> 06B1",""};
lookup1("FB9C") ->	{"0","<initial> 06B1",""};
lookup1("FB9D") ->	{"0","<medial> 06B1",""};
lookup1("FB9E") ->	{"0","<isolated> 06BA",""};
lookup1("FB9F") ->	{"0","<final> 06BA",""};
lookup1("FBA0") ->	{"0","<isolated> 06BB",""};
lookup1("FBA1") ->	{"0","<final> 06BB",""};
lookup1("FBA2") ->	{"0","<initial> 06BB",""};
lookup1("FBA3") ->	{"0","<medial> 06BB",""};
lookup1("FBA4") ->	{"0","<isolated> 06C0",""};
lookup1("FBA5") ->	{"0","<final> 06C0",""};
lookup1("FBA6") ->	{"0","<isolated> 06C1",""};
lookup1("FBA7") ->	{"0","<final> 06C1",""};
lookup1("FBA8") ->	{"0","<initial> 06C1",""};
lookup1("FBA9") ->	{"0","<medial> 06C1",""};
lookup1("FBAA") ->	{"0","<isolated> 06BE",""};
lookup1("FBAB") ->	{"0","<final> 06BE",""};
lookup1("FBAC") ->	{"0","<initial> 06BE",""};
lookup1("FBAD") ->	{"0","<medial> 06BE",""};
lookup1("FBAE") ->	{"0","<isolated> 06D2",""};
lookup1("FBAF") ->	{"0","<final> 06D2",""};
lookup1("FBB0") ->	{"0","<isolated> 06D3",""};
lookup1("FBB1") ->	{"0","<final> 06D3",""};
lookup1("FBB2") ->	{"0","",""};
lookup1("FBB3") ->	{"0","",""};
lookup1("FBB4") ->	{"0","",""};
lookup1("FBB5") ->	{"0","",""};
lookup1("FBB6") ->	{"0","",""};
lookup1("FBB7") ->	{"0","",""};
lookup1("FBB8") ->	{"0","",""};
lookup1("FBB9") ->	{"0","",""};
lookup1("FBBA") ->	{"0","",""};
lookup1("FBBB") ->	{"0","",""};
lookup1("FBBC") ->	{"0","",""};
lookup1("FBBD") ->	{"0","",""};
lookup1("FBBE") ->	{"0","",""};
lookup1("FBBF") ->	{"0","",""};
lookup1("FBC0") ->	{"0","",""};
lookup1("FBC1") ->	{"0","",""};
lookup1("FBD3") ->	{"0","<isolated> 06AD",""};
lookup1("FBD4") ->	{"0","<final> 06AD",""};
lookup1("FBD5") ->	{"0","<initial> 06AD",""};
lookup1("FBD6") ->	{"0","<medial> 06AD",""};
lookup1("FBD7") ->	{"0","<isolated> 06C7",""};
lookup1("FBD8") ->	{"0","<final> 06C7",""};
lookup1("FBD9") ->	{"0","<isolated> 06C6",""};
lookup1("FBDA") ->	{"0","<final> 06C6",""};
lookup1("FBDB") ->	{"0","<isolated> 06C8",""};
lookup1("FBDC") ->	{"0","<final> 06C8",""};
lookup1("FBDD") ->	{"0","<isolated> 0677",""};
lookup1("FBDE") ->	{"0","<isolated> 06CB",""};
lookup1("FBDF") ->	{"0","<final> 06CB",""};
lookup1("FBE0") ->	{"0","<isolated> 06C5",""};
lookup1("FBE1") ->	{"0","<final> 06C5",""};
lookup1("FBE2") ->	{"0","<isolated> 06C9",""};
lookup1("FBE3") ->	{"0","<final> 06C9",""};
lookup1("FBE4") ->	{"0","<isolated> 06D0",""};
lookup1("FBE5") ->	{"0","<final> 06D0",""};
lookup1("FBE6") ->	{"0","<initial> 06D0",""};
lookup1("FBE7") ->	{"0","<medial> 06D0",""};
lookup1("FBE8") ->	{"0","<initial> 0649",""};
lookup1("FBE9") ->	{"0","<medial> 0649",""};
lookup1("FBEA") ->	{"0","<isolated> 0626 0627",""};
lookup1("FBEB") ->	{"0","<final> 0626 0627",""};
lookup1("FBEC") ->	{"0","<isolated> 0626 06D5",""};
lookup1("FBED") ->	{"0","<final> 0626 06D5",""};
lookup1("FBEE") ->	{"0","<isolated> 0626 0648",""};
lookup1("FBEF") ->	{"0","<final> 0626 0648",""};
lookup1("FBF0") ->	{"0","<isolated> 0626 06C7",""};
lookup1("FBF1") ->	{"0","<final> 0626 06C7",""};
lookup1("FBF2") ->	{"0","<isolated> 0626 06C6",""};
lookup1("FBF3") ->	{"0","<final> 0626 06C6",""};
lookup1("FBF4") ->	{"0","<isolated> 0626 06C8",""};
lookup1("FBF5") ->	{"0","<final> 0626 06C8",""};
lookup1("FBF6") ->	{"0","<isolated> 0626 06D0",""};
lookup1("FBF7") ->	{"0","<final> 0626 06D0",""};
lookup1("FBF8") ->	{"0","<initial> 0626 06D0",""};
lookup1("FBF9") ->	{"0","<isolated> 0626 0649",""};
lookup1("FBFA") ->	{"0","<final> 0626 0649",""};
lookup1("FBFB") ->	{"0","<initial> 0626 0649",""};
lookup1("FBFC") ->	{"0","<isolated> 06CC",""};
lookup1("FBFD") ->	{"0","<final> 06CC",""};
lookup1("FBFE") ->	{"0","<initial> 06CC",""};
lookup1("FBFF") ->	{"0","<medial> 06CC",""};
lookup1("FC00") ->	{"0","<isolated> 0626 062C",""};
lookup1("FC01") ->	{"0","<isolated> 0626 062D",""};
lookup1("FC02") ->	{"0","<isolated> 0626 0645",""};
lookup1("FC03") ->	{"0","<isolated> 0626 0649",""};
lookup1("FC04") ->	{"0","<isolated> 0626 064A",""};
lookup1("FC05") ->	{"0","<isolated> 0628 062C",""};
lookup1("FC06") ->	{"0","<isolated> 0628 062D",""};
lookup1("FC07") ->	{"0","<isolated> 0628 062E",""};
lookup1("FC08") ->	{"0","<isolated> 0628 0645",""};
lookup1("FC09") ->	{"0","<isolated> 0628 0649",""};
lookup1("FC0A") ->	{"0","<isolated> 0628 064A",""};
lookup1("FC0B") ->	{"0","<isolated> 062A 062C",""};
lookup1("FC0C") ->	{"0","<isolated> 062A 062D",""};
lookup1("FC0D") ->	{"0","<isolated> 062A 062E",""};
lookup1("FC0E") ->	{"0","<isolated> 062A 0645",""};
lookup1("FC0F") ->	{"0","<isolated> 062A 0649",""};
lookup1("FC10") ->	{"0","<isolated> 062A 064A",""};
lookup1("FC11") ->	{"0","<isolated> 062B 062C",""};
lookup1("FC12") ->	{"0","<isolated> 062B 0645",""};
lookup1("FC13") ->	{"0","<isolated> 062B 0649",""};
lookup1("FC14") ->	{"0","<isolated> 062B 064A",""};
lookup1("FC15") ->	{"0","<isolated> 062C 062D",""};
lookup1("FC16") ->	{"0","<isolated> 062C 0645",""};
lookup1("FC17") ->	{"0","<isolated> 062D 062C",""};
lookup1("FC18") ->	{"0","<isolated> 062D 0645",""};
lookup1("FC19") ->	{"0","<isolated> 062E 062C",""};
lookup1("FC1A") ->	{"0","<isolated> 062E 062D",""};
lookup1("FC1B") ->	{"0","<isolated> 062E 0645",""};
lookup1("FC1C") ->	{"0","<isolated> 0633 062C",""};
lookup1("FC1D") ->	{"0","<isolated> 0633 062D",""};
lookup1("FC1E") ->	{"0","<isolated> 0633 062E",""};
lookup1("FC1F") ->	{"0","<isolated> 0633 0645",""};
lookup1("FC20") ->	{"0","<isolated> 0635 062D",""};
lookup1("FC21") ->	{"0","<isolated> 0635 0645",""};
lookup1("FC22") ->	{"0","<isolated> 0636 062C",""};
lookup1("FC23") ->	{"0","<isolated> 0636 062D",""};
lookup1("FC24") ->	{"0","<isolated> 0636 062E",""};
lookup1("FC25") ->	{"0","<isolated> 0636 0645",""};
lookup1("FC26") ->	{"0","<isolated> 0637 062D",""};
lookup1("FC27") ->	{"0","<isolated> 0637 0645",""};
lookup1("FC28") ->	{"0","<isolated> 0638 0645",""};
lookup1("FC29") ->	{"0","<isolated> 0639 062C",""};
lookup1("FC2A") ->	{"0","<isolated> 0639 0645",""};
lookup1("FC2B") ->	{"0","<isolated> 063A 062C",""};
lookup1("FC2C") ->	{"0","<isolated> 063A 0645",""};
lookup1("FC2D") ->	{"0","<isolated> 0641 062C",""};
lookup1("FC2E") ->	{"0","<isolated> 0641 062D",""};
lookup1("FC2F") ->	{"0","<isolated> 0641 062E",""};
lookup1("FC30") ->	{"0","<isolated> 0641 0645",""};
lookup1("FC31") ->	{"0","<isolated> 0641 0649",""};
lookup1("FC32") ->	{"0","<isolated> 0641 064A",""};
lookup1("FC33") ->	{"0","<isolated> 0642 062D",""};
lookup1("FC34") ->	{"0","<isolated> 0642 0645",""};
lookup1("FC35") ->	{"0","<isolated> 0642 0649",""};
lookup1("FC36") ->	{"0","<isolated> 0642 064A",""};
lookup1("FC37") ->	{"0","<isolated> 0643 0627",""};
lookup1("FC38") ->	{"0","<isolated> 0643 062C",""};
lookup1("FC39") ->	{"0","<isolated> 0643 062D",""};
lookup1("FC3A") ->	{"0","<isolated> 0643 062E",""};
lookup1("FC3B") ->	{"0","<isolated> 0643 0644",""};
lookup1("FC3C") ->	{"0","<isolated> 0643 0645",""};
lookup1("FC3D") ->	{"0","<isolated> 0643 0649",""};
lookup1("FC3E") ->	{"0","<isolated> 0643 064A",""};
lookup1("FC3F") ->	{"0","<isolated> 0644 062C",""};
lookup1("FC40") ->	{"0","<isolated> 0644 062D",""};
lookup1("FC41") ->	{"0","<isolated> 0644 062E",""};
lookup1("FC42") ->	{"0","<isolated> 0644 0645",""};
lookup1("FC43") ->	{"0","<isolated> 0644 0649",""};
lookup1("FC44") ->	{"0","<isolated> 0644 064A",""};
lookup1("FC45") ->	{"0","<isolated> 0645 062C",""};
lookup1("FC46") ->	{"0","<isolated> 0645 062D",""};
lookup1("FC47") ->	{"0","<isolated> 0645 062E",""};
lookup1("FC48") ->	{"0","<isolated> 0645 0645",""};
lookup1("FC49") ->	{"0","<isolated> 0645 0649",""};
lookup1("FC4A") ->	{"0","<isolated> 0645 064A",""};
lookup1("FC4B") ->	{"0","<isolated> 0646 062C",""};
lookup1("FC4C") ->	{"0","<isolated> 0646 062D",""};
lookup1("FC4D") ->	{"0","<isolated> 0646 062E",""};
lookup1("FC4E") ->	{"0","<isolated> 0646 0645",""};
lookup1("FC4F") ->	{"0","<isolated> 0646 0649",""};
lookup1("FC50") ->	{"0","<isolated> 0646 064A",""};
lookup1("FC51") ->	{"0","<isolated> 0647 062C",""};
lookup1("FC52") ->	{"0","<isolated> 0647 0645",""};
lookup1("FC53") ->	{"0","<isolated> 0647 0649",""};
lookup1("FC54") ->	{"0","<isolated> 0647 064A",""};
lookup1("FC55") ->	{"0","<isolated> 064A 062C",""};
lookup1("FC56") ->	{"0","<isolated> 064A 062D",""};
lookup1("FC57") ->	{"0","<isolated> 064A 062E",""};
lookup1("FC58") ->	{"0","<isolated> 064A 0645",""};
lookup1("FC59") ->	{"0","<isolated> 064A 0649",""};
lookup1("FC5A") ->	{"0","<isolated> 064A 064A",""};
lookup1("FC5B") ->	{"0","<isolated> 0630 0670",""};
lookup1("FC5C") ->	{"0","<isolated> 0631 0670",""};
lookup1("FC5D") ->	{"0","<isolated> 0649 0670",""};
lookup1("FC5E") ->	{"0","<isolated> 0020 064C 0651",""};
lookup1("FC5F") ->	{"0","<isolated> 0020 064D 0651",""};
lookup1("FC60") ->	{"0","<isolated> 0020 064E 0651",""};
lookup1("FC61") ->	{"0","<isolated> 0020 064F 0651",""};
lookup1("FC62") ->	{"0","<isolated> 0020 0650 0651",""};
lookup1("FC63") ->	{"0","<isolated> 0020 0651 0670",""};
lookup1("FC64") ->	{"0","<final> 0626 0631",""};
lookup1("FC65") ->	{"0","<final> 0626 0632",""};
lookup1("FC66") ->	{"0","<final> 0626 0645",""};
lookup1("FC67") ->	{"0","<final> 0626 0646",""};
lookup1("FC68") ->	{"0","<final> 0626 0649",""};
lookup1("FC69") ->	{"0","<final> 0626 064A",""};
lookup1("FC6A") ->	{"0","<final> 0628 0631",""};
lookup1("FC6B") ->	{"0","<final> 0628 0632",""};
lookup1("FC6C") ->	{"0","<final> 0628 0645",""};
lookup1("FC6D") ->	{"0","<final> 0628 0646",""};
lookup1("FC6E") ->	{"0","<final> 0628 0649",""};
lookup1("FC6F") ->	{"0","<final> 0628 064A",""};
lookup1("FC70") ->	{"0","<final> 062A 0631",""};
lookup1("FC71") ->	{"0","<final> 062A 0632",""};
lookup1("FC72") ->	{"0","<final> 062A 0645",""};
lookup1("FC73") ->	{"0","<final> 062A 0646",""};
lookup1("FC74") ->	{"0","<final> 062A 0649",""};
lookup1("FC75") ->	{"0","<final> 062A 064A",""};
lookup1("FC76") ->	{"0","<final> 062B 0631",""};
lookup1("FC77") ->	{"0","<final> 062B 0632",""};
lookup1("FC78") ->	{"0","<final> 062B 0645",""};
lookup1("FC79") ->	{"0","<final> 062B 0646",""};
lookup1("FC7A") ->	{"0","<final> 062B 0649",""};
lookup1("FC7B") ->	{"0","<final> 062B 064A",""};
lookup1("FC7C") ->	{"0","<final> 0641 0649",""};
lookup1("FC7D") ->	{"0","<final> 0641 064A",""};
lookup1("FC7E") ->	{"0","<final> 0642 0649",""};
lookup1("FC7F") ->	{"0","<final> 0642 064A",""};
lookup1("FC80") ->	{"0","<final> 0643 0627",""};
lookup1("FC81") ->	{"0","<final> 0643 0644",""};
lookup1("FC82") ->	{"0","<final> 0643 0645",""};
lookup1("FC83") ->	{"0","<final> 0643 0649",""};
lookup1("FC84") ->	{"0","<final> 0643 064A",""};
lookup1("FC85") ->	{"0","<final> 0644 0645",""};
lookup1("FC86") ->	{"0","<final> 0644 0649",""};
lookup1("FC87") ->	{"0","<final> 0644 064A",""};
lookup1("FC88") ->	{"0","<final> 0645 0627",""};
lookup1("FC89") ->	{"0","<final> 0645 0645",""};
lookup1("FC8A") ->	{"0","<final> 0646 0631",""};
lookup1("FC8B") ->	{"0","<final> 0646 0632",""};
lookup1("FC8C") ->	{"0","<final> 0646 0645",""};
lookup1("FC8D") ->	{"0","<final> 0646 0646",""};
lookup1("FC8E") ->	{"0","<final> 0646 0649",""};
lookup1("FC8F") ->	{"0","<final> 0646 064A",""};
lookup1("FC90") ->	{"0","<final> 0649 0670",""};
lookup1("FC91") ->	{"0","<final> 064A 0631",""};
lookup1("FC92") ->	{"0","<final> 064A 0632",""};
lookup1("FC93") ->	{"0","<final> 064A 0645",""};
lookup1("FC94") ->	{"0","<final> 064A 0646",""};
lookup1("FC95") ->	{"0","<final> 064A 0649",""};
lookup1("FC96") ->	{"0","<final> 064A 064A",""};
lookup1("FC97") ->	{"0","<initial> 0626 062C",""};
lookup1("FC98") ->	{"0","<initial> 0626 062D",""};
lookup1("FC99") ->	{"0","<initial> 0626 062E",""};
lookup1("FC9A") ->	{"0","<initial> 0626 0645",""};
lookup1("FC9B") ->	{"0","<initial> 0626 0647",""};
lookup1("FC9C") ->	{"0","<initial> 0628 062C",""};
lookup1("FC9D") ->	{"0","<initial> 0628 062D",""};
lookup1("FC9E") ->	{"0","<initial> 0628 062E",""};
lookup1("FC9F") ->	{"0","<initial> 0628 0645",""};
lookup1("FCA0") ->	{"0","<initial> 0628 0647",""};
lookup1("FCA1") ->	{"0","<initial> 062A 062C",""};
lookup1("FCA2") ->	{"0","<initia