#pragma once

#include "clientBootstrap.h"
#include "clientDispatcher.hpp"
#include "fetchManager.hpp"
#include "handleCache.hpp"
#include "periodicHandler.hpp"
#include "proto/rtransfer_messages.pb.h"
#include "protoHandler.hpp"

#include <folly/FBVector.h>
#include <folly/SpinLock.h>
#include <folly/executors/SerialExecutor.h>
#include <wangle/acceptor/ConnectionManager.h>
#include <wangle/acceptor/ManagedConnection.h>
#include <wangle/channel/StaticPipeline.h>
#include <wangle/codec/LengthFieldBasedFrameDecoder.h>
#include <wangle/codec/LengthFieldPrepender.h>

#include <atomic>
#include <memory>

namespace rtransfer {
namespace client {

using MsgPtr = std::unique_ptr<proto::LinkMessage>;

using Pipeline = wangle::Pipeline<folly::IOBufQueue &, MsgPtr>;
using DataPipeline =
    wangle::Pipeline<folly::IOBufQueue &, std::unique_ptr<folly::IOBuf>>;
using DataBootstrapPtr =
    std::shared_ptr<rtransfer::ClientBootstrap<DataPipeline>>;

class Client;
class ConnectionCloseHandler;

class CtrlIdHandler : public wangle::HandlerAdapter<folly::IOBufQueue &,
                          std::unique_ptr<folly::IOBuf>> {
public:
    CtrlIdHandler(folly::fbstring addr);

    void read(Context *ctx, folly::IOBufQueue &buf) override;

    folly::Future<folly::fbstring> ctrlId();

private:
    folly::fbstring peerAddr_;
    folly::Promise<folly::fbstring> ctrlIdPromise_;
};

class HttpResponseHandler : public wangle::HandlerAdapter<folly::IOBufQueue &,
                                std::unique_ptr<folly::IOBuf>> {
public:
    HttpResponseHandler(folly::fbstring addr);

    void read(Context *ctx, folly::IOBufQueue &buf) override;

    folly::Future<folly::Unit> done();

private:
    folly::fbstring peerAddr_;
    folly::Promise<folly::Unit> promise_;
};

class PeerSecretHandler : public wangle::HandlerAdapter<folly::IOBufQueue &,
                              std::unique_ptr<folly::IOBuf>> {
public:
    PeerSecretHandler(folly::fbstring addr, folly::fbstring peerSecret)
        : peerSecret_{std::move(peerSecret)}
        , peerAddr_{std::move(addr)}
    {
    }

    void read(Context *ctx, folly::IOBufQueue &buf) override;

    folly::Future<folly::Unit> done();

private:
    folly::fbstring peerSecret_;
    folly::fbstring peerAddr_;
    folly::Promise<folly::Unit> promise_;
};

class PipelineFactory : public wangle::PipelineFactory<Pipeline> {
public:
    PipelineFactory(
        std::shared_ptr<ConnectionCloseHandler> connectionCloseHandler,
        folly::fbstring peerSecret, folly::SocketAddress addr);

    Pipeline::Ptr newPipeline(
        std::shared_ptr<folly::AsyncTransportWrapper> sock) override;

private:
    std::shared_ptr<ConnectionCloseHandler> connectionCloseHandler_;
    folly::fbstring peerSecret_;
    folly::fbstring peerAddr_;
};

class ConnectionCloseHandler
    : public wangle::HandlerAdapter<folly::IOBufQueue &,
          std::unique_ptr<folly::IOBuf>> {
public:
    folly::Future<folly::Unit> getCloseFuture() { return promise_.getFuture(); }

    void readEOF(Context *) override { setPromise(); }

    void readException(Context *, folly::exception_wrapper) override
    {
        setPromise();
    }

private:
    void setPromise()
    {
        if (!promiseFulfilled_.test_and_set())
            promise_.setValue();
    }

    std::atomic_flag promiseFulfilled_ = ATOMIC_FLAG_INIT;
    folly::Promise<folly::Unit> promise_;
};

class DataPipelineFactory : public wangle::PipelineFactory<DataPipeline> {
public:
    DataPipelineFactory(std::shared_ptr<FetchManager> fetchManager,
        std::shared_ptr<ConnectionCloseHandler> connectionCloseHandler,
        folly::fbstring peerSecret, folly::SocketAddress addr);

    DataPipeline::Ptr newPipeline(
        std::shared_ptr<folly::AsyncTransportWrapper> sock) override;

private:
    std::shared_ptr<FetchManager> fetchManager_;
    std::shared_ptr<ConnectionCloseHandler> connectionCloseHandler_;
    folly::fbstring peerSecret_;
    folly::fbstring peerAddr_;
};

class Client : public std::enable_shared_from_this<Client> {
public:
    using ClientBootstrapPtr =
        std::shared_ptr<rtransfer::ClientBootstrap<Pipeline>>;

    Client(folly::SocketAddress addr,
        std::shared_ptr<folly::IOThreadPoolExecutor> executor,
        StoragesMap &storages, folly::fbstring mySecret,
        folly::fbstring peerSecret);

    ~Client();

    folly::Future<folly::Unit> connect();

    folly::Future<folly::Unit> ping();

    folly::Future<std::size_t> fetch(folly::StringPiece srcStorageId,
        folly::StringPiece srcFileId, folly::StringPiece destStorageId,
        folly::StringPiece destFileId, folly::StringPiece fileGuid,
        std::uint64_t offset, std::size_t size, std::uint8_t priority,
        std::uint64_t reqId, folly::StringPiece transferData,
        folly::Function<void(std::uint64_t, std::size_t)> notifyCb);

    folly::Future<folly::Unit> cancel(std::uint64_t reqId,
        folly::StringPiece srcStorageId, folly::StringPiece dstStorageId);

    void ack(std::uint64_t reqId, std::uint64_t offset);

    static folly::Future<MsgPtr> ensureResponse(
        MsgPtr msg, proto::LinkMessage::PayloadCase expected);

    folly::Future<folly::Unit> newDataConn();

    ClientBootstrapPtr ctrl();

    folly::fbvector<DataBootstrapPtr> &data();

    void closeNow();

    folly::Future<folly::Unit> getCloseFuture()
    {
        return connectionCloseHandler_->getCloseFuture();
    };

private:
    std::unique_ptr<folly::IOBuf> makeHandshake(folly::StringPiece ctrlId);
    void ackLoop();

    std::unique_ptr<PeriodicHandler> pinger_;

    StoragesMap &storages_;
    folly::fbstring mySecret_;
    folly::fbstring peerSecret_;
    folly::SocketAddress addr_;
    HandleCache handleCache_;
    std::shared_ptr<FetchManager> fetchManager_ =
        std::make_shared<FetchManager>(*this, storages_, handleCache_);
    folly::fbstring ctrlId_;
    ClientBootstrapPtr ctrl_{
        std::make_shared<rtransfer::ClientBootstrap<Pipeline>>()};
    folly::fbvector<DataBootstrapPtr> data_;
    ClientDispatcher dispatcher_;
    wangle::Service<MsgPtr, MsgPtr> &service_{dispatcher_};

    std::shared_ptr<ConnectionCloseHandler> connectionCloseHandler_ =
        std::make_shared<ConnectionCloseHandler>();

    std::shared_ptr<folly::IOThreadPoolExecutor> clientExecutor_;
    folly::SerialExecutor executor_;

    folly::SpinLock acksLock_;
    folly::fbvector<std::pair<std::uint64_t, std::uint64_t>> pendingAcks_;
    folly::fbvector<std::pair<std::uint64_t, std::uint64_t>> usedAcks_;
    bool someoneHandlingAcks_ = false;

    std::atomic_flag closeNowFlag_ = ATOMIC_FLAG_INIT;
};

}  // namespace client
}  // namespace rtransfer
