#pragma once

#include "proto/rtransfer_messages.pb.h"
#include "uuid.hpp"

#include <folly/concurrency/ConcurrentHashMap.h>
#include <glog/logging.h>
#include <monitoring/monitoring.h>
#include <wangle/service/ClientDispatcher.h>

#include <unordered_map>

namespace rtransfer {
using MsgPtr = std::unique_ptr<proto::LinkMessage>;

class ClientDispatcher
    : public wangle::ClientDispatcherBase<
          wangle::Pipeline<folly::IOBufQueue &, MsgPtr>, MsgPtr> {
public:
    void read(Context *, MsgPtr in) override
    {
        auto it = promises_.find(in->req_id());
        if (it == promises_.cend()) {
            LOG(WARNING) << "No promise for " << in->req_id();
            return;
        }

        ONE_METRIC_COUNTER_DEC("requests.remote.outgoing.concurrent");
        it->second->setValue(std::move(in));
        promises_.erase(it);
    }

    folly::Future<MsgPtr> operator()(MsgPtr arg) override
    {
        DCHECK(this->pipeline_);
        ONE_METRIC_COUNTER_INC("requests.remote.outgoing.concurrent");

        auto reqId = genUUID();
        arg->set_req_id(reqId.data(), reqId.size());
        auto result = promises_.emplace(
            reqId, std::make_shared<folly::Promise<MsgPtr>>());

        auto promise = result.first->second;
        promise->setInterruptHandler([reqId, this](folly::exception_wrapper e) {
            auto it = promises_.find(reqId);
            if (it != promises_.cend()) {
                it->second->setException(std::move(e));
                promises_.erase(it);
            }
        });

        auto transport = this->pipeline_->getTransport();
        DCHECK(transport);
        auto base = transport->getEventBase();
        DCHECK(base);

        base->runInEventBaseThread(
            [pipeline = this->pipeline_, arg = std::move(arg)]() mutable {
                pipeline->write(std::move(arg));
            });

        return promise->getFuture();
    }

private:
    folly::ConcurrentHashMap<folly::fbstring,
        std::shared_ptr<folly::Promise<MsgPtr>>>
        promises_;
};

}  // namespace rtransfer
