#include "controlService.hpp"
#include "storage.hpp"

#include "jsonMessageHandler.hpp"
#include "secretManager.hpp"
#include "stdIOHandler.hpp"
#include "stopBatonHandler.hpp"
#include "uuid.hpp"

#include <jemalloc/jemalloc.h>
#include <monitoring/monitoring.h>
#include <wangle/channel/StaticPipeline.h>
#include <wangle/codec/LineBasedFrameDecoder.h>
#include <wangle/codec/StringCodec.h>
#include <wangle/service/ServerDispatcher.h>

#include <chrono>

namespace rtransfer {
namespace link_control {

ControlService::~ControlService()
{
    auto c = clientLinks_.cbegin();
    while (c != clientLinks_.cend()) {
        c->second->closeNow();
        ++c;
    }
    clientLinks_.clear();
}

folly::Future<proto::ResponsePtr> ControlService::operator()(
    proto::RequestPtr request)
{
    VLOG(1) << "Control request: " << request->DebugString();

    auto &reqId = request->req_id();
    if (request->is_answer()) {
        auto questionIt = questions_.find(reqId);
        if (questionIt == questions_.cend())
            return error("bad answer");

        questionIt->second->setValue(std::move(request));
        questions_.erase(questionIt);
        return folly::makeFuture<proto::ResponsePtr>(proto::ResponsePtr{});
    }

    return doAction(std::move(request))
        .onError([](const std::exception &e) { return error(e.what()); })
        .then([reqId = std::move(reqId)](proto::ResponsePtr response) {
            response->set_req_id(reqId);
            if (!response->is_update())
                VLOG(1) << "Control reply: " << response->DebugString();
            return response;
        });
}

folly::Future<proto::RequestPtr> ControlService::ask(proto::ResponsePtr request)
{
    auto reqId = genUUID();
    auto promise = std::make_shared<folly::Promise<proto::RequestPtr>>();
    auto future = promise->getFuture();
    request->set_req_id(reqId.data(), reqId.size());
    request->set_is_question(true);
    questions_.emplace(std::move(reqId), std::move(promise));
    pipeline_->write(std::move(request));
    return future;
}

folly::Future<proto::ResponsePtr> ControlService::doAction(
    proto::RequestPtr request)
{
    try {
        ONE_METRIC_COUNTER_INC("requests.control.total");
        switch (request->payload_case()) {
            case proto::Request::kCreateHelper:
                ONE_METRIC_COUNTER_INC("requests.control.create_helper");
                return createHelper(std::move(request));
            case proto::Request::kConnect:
                ONE_METRIC_COUNTER_INC("requests.control.connect");
                return connect(std::move(request));
            case proto::Request::kFetch:
                ONE_METRIC_COUNTER_INC("requests.control.fetch");
                return fetch(std::move(request));
            case proto::Request::kCancel:
                ONE_METRIC_COUNTER_INC("requests.control.cancel");
                return cancel(std::move(request));
            case proto::Request::kAllowConnection:
                ONE_METRIC_COUNTER_INC("requests.control.allow_connection");
                return allowConnection(std::move(request));
            case proto::Request::kGetMemoryStats:
                return memoryStats(std::move(request));
            default:
                ONE_METRIC_COUNTER_INC("requests.control.bad_request");
                return error("bad request");
        }
    }
    catch (const std::exception &e) {
        return error(e.what());
    }
}

folly::Future<proto::ResponsePtr> ControlService::createHelper(
    proto::RequestPtr req)
{
    auto helper = helperFactory_(req->create_helper());
    storages_.insert_or_assign(req->create_helper().storage_id(), helper);
    return done();
}

folly::Future<proto::ResponsePtr> ControlService::error(
    folly::StringPiece description)
{
    auto msg = std::make_unique<proto::Response>();
    msg->mutable_error()->set_description(
        description.data(), description.size());
    return folly::makeFuture(std::move(msg));
}

folly::Future<proto::ResponsePtr> ControlService::done()
{
    auto msg = std::make_unique<proto::Response>();
    msg->set_done(true);
    return folly::makeFuture(std::move(msg));
}

folly::Future<proto::ResponsePtr> ControlService::connect(proto::RequestPtr req)
{
    auto &msg = req->connect();
    if (msg.peer_port() > 65535)
        return error("bad port number");

    folly::SocketAddress addr{
        msg.peer_host(), static_cast<std::uint16_t>(msg.peer_port()), true};

    VLOG(1) << "Connecting to " << addr.describe();
    auto client = std::make_shared<client::Client>(
        addr, clientExecutor_, storages_, msg.my_secret(), msg.peer_secret());
    return client->connect()
        .then([this, client, addr] {
            VLOG(1) << "Connected to " << addr.describe();
            clientLinks_.insert_or_assign(addr.describe(), client);
            client->getCloseFuture().then([=] {
                auto it = clientLinks_.find(addr.describe());
                if (it == clientLinks_.cend())
                    return;

                LOG(INFO) << "Erasing client " << addr.describe();
                it->second->closeNow();
                clientLinks_.erase(it);

                auto msg = folly::make_unique<proto::Response>();
                msg->mutable_disconnected()->set_connection_id(addr.describe());
                pipeline_->write(std::move(msg));
            });
        })
        .then([addr] {
            auto resp = std::make_unique<proto::Response>();
            resp->set_connection_id(addr.describe());
            return resp;
        });
}

folly::Future<proto::ResponsePtr> ControlService::fetch(proto::RequestPtr req)
{
    auto &msg = req->fetch();
    auto it = clientLinks_.find(msg.connection_id());
    if (it == clientLinks_.cend())
        return error("no such connection: " + msg.connection_id());

    auto notifyCb = std::bind(&NotifyAggregator::notify, &notifyAggregator_,
        req->req_id(), std::placeholders::_1, std::placeholders::_2);

    return it->second
        ->fetch(msg.src_storage_id(), msg.src_file_id(), msg.dest_storage_id(),
            msg.dest_file_id(), msg.file_guid(), msg.offset(), msg.size(),
            msg.priority(), msg.req_id(), msg.transfer_data(),
            std::move(notifyCb))
        .then([this, reqId = req->req_id()](folly::Try<std::size_t> wrote) {
            notifyAggregator_.flush(reqId);
            return folly::makeFuture<std::size_t>(std::move(wrote));
        })
        .then([](std::size_t wrote) {
            auto resp = std::make_unique<proto::Response>();
            resp->set_wrote(wrote);
            return resp;
        });
}

void ControlService::notify(
    folly::fbstring reqId, std::uint64_t offset, std::size_t size)
{
    auto resp = std::make_unique<proto::Response>();
    resp->set_req_id(reqId.data());
    resp->set_is_update(true);
    auto progress = resp->mutable_progress();
    progress->set_offset(offset);
    progress->set_size(size);
    pipeline_->write(std::move(resp));
}

folly::Future<proto::ResponsePtr> ControlService::cancel(proto::RequestPtr req)
{
    auto &msg = req->cancel();
    auto it = clientLinks_.find(msg.connection_id());
    if (it == clientLinks_.cend())
        return error("no such connection: " + msg.connection_id());

    return it->second
        ->cancel(msg.req_id(), msg.src_storage_id(), msg.dest_storage_id())
        .then(&ControlService::done);
}

folly::Future<proto::ResponsePtr> ControlService::allowConnection(
    proto::RequestPtr req)
{
    auto &msg = req->allow_connection();
    auto expirationTime = std::chrono::steady_clock::now() +
                          std::chrono::milliseconds{msg.expiration()};
    secretManager_.allowConnection(
        msg.my_secret(), msg.peer_secret(), msg.provider_id(), expirationTime);
    return done();
}

folly::Future<proto::ResponsePtr> ControlService::memoryStats(
    proto::RequestPtr req)
{
    static auto makeResponse = [](void *m, const char *stats) {
        static_cast<proto::Response *>(m)->set_memory_stats(stats);
    };

    auto resp = std::make_unique<proto::Response>();
    malloc_stats_print(makeResponse, resp.get(), "J");
    return folly::makeFuture(std::move(resp));
}

Pipeline::Ptr newPipeline(ControlService &service, folly::Baton<> &stopBaton)
{
    auto stdIOHandler = std::make_shared<StdIOHandler>();
    auto pipeline = wangle::StaticPipeline<folly::IOBufQueue &,
        proto::ResponsePtr, StdIOHandler, StopBatonHandler,
        wangle::LineBasedFrameDecoder, wangle::StringCodec,
        JSONMessageHandler<proto::Request, proto::Response>,
        wangle::MultiplexServerDispatcher<proto::RequestPtr,
            proto::ResponsePtr>>::create(stdIOHandler,
        StopBatonHandler{stopBaton},
        wangle::LineBasedFrameDecoder{UINT_MAX, true,
            wangle::LineBasedFrameDecoder::TerminatorType::NEWLINE},
        wangle::StringCodec{},
        JSONMessageHandler<proto::Request, proto::Response>{},
        wangle::MultiplexServerDispatcher<proto::RequestPtr,
            proto::ResponsePtr>{&service});

    service.setPipeline(pipeline.get());

    return pipeline;
}

}  // namespace link_control
}  // namespace rtransfer
