#include "fetchManager.hpp"

#include "client.hpp"
#include "shaper.hpp"

#include <gflags/gflags.h>

DEFINE_uint64(max_incoming_buffered_size, 1024 * 1024 * 200,
    "maximum number of buffered incoming bytes (pending write to storage) per "
    "link");

namespace rtransfer {

FetchManager::Header::Header(std::unique_ptr<folly::IOBuf> &buf)
{
    buf->gather(21);
    std::memcpy(&reqId, buf->data(), sizeof(reqId));
    buf->trimStart(sizeof(reqId));
    std::memcpy(&part, buf->data(), sizeof(part));
    buf->trimStart(sizeof(part));
    std::memcpy(&offset, buf->data(), sizeof(offset));
    buf->trimStart(sizeof(offset) + 4);

    isLastPart = part & proto::last_part_mask;
    if (isLastPart)
        part &= ~proto::last_part_mask;

    reqId = folly::Endian::big(reqId);
    offset = folly::Endian::big(offset);
}

FetchManager::FetchManager(
    client::Client &client, StoragesMap &storages, HandleCache &handleCache)
    : client_{client}
    , storages_{storages}
    , handleCache_{handleCache}
{
}

folly::Future<std::size_t> FetchManager::newFetch(std::uint64_t requestId,
    folly::fbstring destStorageId, folly::fbstring destFileId,
    folly::fbstring fileGuid, std::uint64_t offset, std::size_t size,
    folly::Function<void(std::uint64_t, std::size_t)> notifyCb)
{
    VLOG(1) << "New fetch: " << requestId << " " << offset << " " << size;

    auto fi = std::make_shared<FetchInfo>();
    fi->storageId = destStorageId;
    fi->fileId = destFileId;
    fi->fileGuid = fileGuid;
    fi->size = size;
    fi->promise.emplace();
    fi->notifyCb = std::move(notifyCb);

    auto result = fetchInfo_.emplace(requestId, std::move(fi));
    DCHECK(result.second);
    ONE_METRIC_COUNTER_INC("fetches.concurrent");
    return result.first->second->promise->getFuture().then([=](std::size_t s) {
        auto storageIt = storages_.find(destStorageId);
        if (storageIt == storages_.cend()) {
            LOG(WARNING) << "Storage " << folly::hexlify(destStorageId)
                         << " not found, skipping fsync";
            return folly::makeFuture<std::size_t>(std::move(s));
        }
        return handleCache_.sync(*storageIt->second, destFileId, fileGuid)
            .then([=] { return s; });
    });
}

void FetchManager::read(Context *ctx, std::unique_ptr<folly::IOBuf> buf)
{
    folly::getCPUExecutor()->addWithPriority(
        [this, buf = std::move(buf)]() mutable { doRead(std::move(buf)); },
        SHAPER_OPS_PRIO);
}

void FetchManager::doRead(std::unique_ptr<folly::IOBuf> buf)
{
    Header header{buf};

    auto it = fetchInfo_.find(header.reqId);
    if (it == fetchInfo_.cend()) {
        LOG(INFO) << "Discarding canceled data for reqId = " << header.reqId;
        return;
    }

    auto &req = *it->second;
    auto storageIt = storages_.find(req.storageId);
    if (storageIt == storages_.cend())
        return cancelFetchStr(
            it, "no such destination storage " + folly::hexlify(req.storageId));

    auto queue = tallyParts(req, header, std::move(buf));
    if (!queue)
        return;

    client_.ack(header.reqId, header.offset);

    std::size_t toWrite = queue->chainLength();
    VLOG(2) << "Received data about " << header.reqId << " offset "
            << header.offset << " size " << toWrite;

    handleCache_
        .write(*storageIt->second, header.reqId, req.fileId, req.fileGuid,
            header.offset, std::move(*queue), 0)
        .then([this, offset = header.offset, toWrite, reqId = header.reqId] {
            auto it = fetchInfo_.find(reqId);
            if (it == fetchInfo_.cend())
                return;

            {
                folly::SpinLockGuard guard{it->second->lock};
                it->second->notifyCb(offset, toWrite);
                it->second->wrote += toWrite;
            }

            maybeFulfillPromise(it);
        })
        .onError([this, reqId = header.reqId](folly::exception_wrapper e) {
            cancelFetch(reqId, std::move(e));
        });
}

void FetchManager::cancelFetch(std::uint64_t requestId)
{
    cancelFetchStr(requestId, "canceled");
}

void FetchManager::cancelFetch(
    std::uint64_t requestId, folly::exception_wrapper exception)
{
    auto it = fetchInfo_.find(requestId);
    if (it == fetchInfo_.cend()) {
        LOG(WARNING) << "Unable to cancel unknown fetch request " << requestId
                     << " (cancel reason: " << exception.what() << ")";
        return;
    }

    cancelFetch(it, std::move(exception));
}

void FetchManager::setTotalSize(std::uint64_t reqId, std::size_t size)
{
    auto it = fetchInfo_.find(reqId);
    if (it == fetchInfo_.cend())
        return;

    {
        folly::SpinLockGuard guard{it->second->lock};
        if (it->second->size < size) {
            LOG(WARNING) << "Refusing to set size to larger than requested";
            return;
        }
        it->second->size = size;
    }

    maybeFulfillPromise(it);
}

template <typename RefType>
void FetchManager::cancelFetchStr(RefType ref, folly::StringPiece reason)
{
    cancelFetch(
        ref, folly::make_exception_wrapper<std::runtime_error>(reason.str()));
}

void FetchManager::cancelFetch(
    Map::const_iterator it, folly::exception_wrapper exception)
{
    ONE_METRIC_COUNTER_DEC("fetches.concurrent");
    ONE_METRIC_COUNTER_INC("fetches.cancelled");

    folly::Optional<folly::Promise<std::size_t>> promise;

    {
        folly::SpinLockGuard guard{it->second->lock};
        promise = std::move(it->second->promise);
    }

    if (promise) {
        promise->setException(std::move(exception));
        fetchInfo_.erase(it);
    }
}

void FetchManager::maybeFulfillPromise(Map::const_iterator it)
{
    ONE_METRIC_COUNTER_DEC("fetches.concurrent");

    folly::Optional<folly::Promise<std::size_t>> promise;
    std::size_t wrote = 0;

    {
        folly::SpinLockGuard guard{it->second->lock};
        if (it->second->wrote >= it->second->size) {
            promise = std::move(it->second->promise);
            wrote = it->second->wrote;
        }
    }

    if (promise) {
        promise->setValue(wrote);
        fetchInfo_.erase(it);
    }
}

folly::Optional<folly::IOBufQueue> FetchManager::tallyParts(
    FetchInfo &req, Header &header, std::unique_ptr<folly::IOBuf> buf)
{
    folly::SpinLockGuard guard{req.lock};
    auto res = req.partsInfo.emplace(header.offset, PartsInfo{});
    auto &pi = res.first->second;
    pi.bufs.emplace_back(std::make_pair(header.part, std::move(buf)));
    if (header.isLastPart)
        pi.totalParts = header.part + 1;

    DLOG(INFO) << "ReqId: " << header.reqId << ", part "
               << static_cast<int>(header.part) << " of "
               << static_cast<int>(pi.totalParts) << " (" << pi.bufs.size()
               << " so far)";

    if (pi.bufs.size() < pi.totalParts)
        return {};

    std::sort(pi.bufs.begin(), pi.bufs.end());

    folly::IOBufQueue queue{folly::IOBufQueue::cacheChainLength()};
    for (auto &el : pi.bufs)
        queue.append(std::move(el.second));

    req.partsInfo.erase(res.first);

    return {std::move(queue)};
}

}  // namespace rtransfer
