#pragma once

#include "common.hpp"
#include "handleCache.hpp"
#include "storage.hpp"

#include <folly/FBString.h>
#include <folly/SpinLock.h>
#include <folly/String.h>
#include <folly/concurrency/ConcurrentHashMap.h>
#include <folly/futures/Future.h>
#include <monitoring/monitoring.h>
#include <wangle/channel/Handler.h>

#include <cstdint>
#include <unordered_map>

namespace rtransfer {

namespace client {
class Client;
}

class FetchManager
    : public wangle::InboundHandler<std::unique_ptr<folly::IOBuf>,
          folly::Unit> {
public:
    FetchManager(client::Client &client, StoragesMap &storages,
        HandleCache &handleCache);

    folly::Future<std::size_t> newFetch(std::uint64_t requestId,
        folly::fbstring destStorageId, folly::fbstring destFileId,
        folly::fbstring fileGuid, std::uint64_t offset, std::size_t size,
        folly::Function<void(std::uint64_t, std::size_t)> notifyCb);

    void read(Context *ctx, std::unique_ptr<folly::IOBuf> buf) override;

    void cancelFetch(std::uint64_t requestId);

    void cancelFetch(
        std::uint64_t requestId, folly::exception_wrapper exception);

    void setTotalSize(std::uint64_t reqId, std::size_t size);

private:
    struct PartsInfo {
        std::uint8_t totalParts = 255;
        folly::fbvector<std::pair<std::uint8_t, std::unique_ptr<folly::IOBuf>>>
            bufs;
    };

    struct FetchInfo {
        mutable folly::SpinLock lock;
        folly::fbstring storageId;
        folly::fbstring fileId;
        folly::fbstring fileGuid;
        std::unordered_map<std::uint64_t, PartsInfo> partsInfo;
        std::size_t size;
        std::size_t wrote = 0;
        folly::Optional<folly::Promise<std::size_t>> promise;
        folly::Function<void(std::uint64_t, std::size_t)> notifyCb;
    };

    struct Header {
        Header(std::unique_ptr<folly::IOBuf> &);
        std::uint64_t reqId;
        bool isLastPart;
        std::uint8_t part;
        std::uint64_t offset;
    };

    using Map =
        folly::ConcurrentHashMap<std::uint64_t, std::shared_ptr<FetchInfo>>;

    void doRead(std::unique_ptr<folly::IOBuf> buf);

    template <typename Ref>
    void cancelFetchStr(Ref ref, folly::StringPiece reason);

    void cancelFetch(
        Map::const_iterator it, folly::exception_wrapper exception);

    void maybeFulfillPromise(Map::const_iterator it);

    folly::Optional<folly::IOBufQueue> tallyParts(
        FetchInfo &info, Header &header, std::unique_ptr<folly::IOBuf> buf);

    client::Client &client_;
    StoragesMap &storages_;
    HandleCache &handleCache_;
    Map fetchInfo_;
};

}  // namespace rtransfer
